/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CME implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api_debug.h"
#include "decoder_obj.h"
#include "channel_obj.h"
#include "ccache.h"
#include "sms_obj.h"
#include "cid_obj.h"
#include "sms_api.h"
#include "cme.h"
#include "_cme.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CME_hCreate
*
* Creates the Content Monitoring Engine. There is one instance of the
* CME for each DECODER object instantiated. Initally the content monitoring
* engine contains no registered content. The function returns a handle to
* the CME instance owned by the provided DECODER and is used for other CME
* APIs.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object for which to attach
*               or associate this CME instance.
*
* Outputs:
*
*   CME_OBJECT  A valid CME_OBJECT handle on success or CME_INVALID_OBJECT
*               if an error or failure occurred.
*
*****************************************************************************/
CME_OBJECT CME_hCreate ( DECODER_OBJECT hDecoder )
{
    CME_OBJECT hCME = CME_INVALID_OBJECT;
    CME_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    size_t tIndex;

    // Create an instance of the CME object
    psObj = (CME_OBJECT_STRUCT *)
        SMSO_hCreate(
            CME_OBJECT_NAME,
            sizeof(CME_OBJECT_STRUCT),
            (SMS_OBJECT)hDecoder, // This object is a child of the hDecoder
            FALSE);
    if(psObj == NULL)
    {
        // Error!
        return CME_INVALID_OBJECT;
    }

    do
    {
        // Create the master linked list which contains all registered content.

        // Construct an appropriate name for list to be created.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 CME_OBJECT_NAME":Master Content List");

        // We need to create a master content list. Note this list need
        // not have mutually exclusive access protection since it is accessed
        // only by the DECODER object itself. It is a linear list
        // of events kept in CID order. It is important that this list is kept
        // in CID order as we will place markers throughout the list which
        // help us quickly locate groups of CIDs (based on type) without having
        // to maintain multiple linked lists (cheats).
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hMasterContentList,
                &acName[0],
                (OSAL_LL_COMPARE_HANDLER)n16CompareContent,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize the CID type entry handles to help searches by
        // indexing directly to the right CID group.
        for(tIndex = CID_FIRST; tIndex < CID_LAST; tIndex++)
        {
            // Initialize CID entry as empty
            psObj->ahEntry[tIndex] = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Create a list of pending events which need to be triggered. Once
        // an event is triggered it is removed. It is then up to the higher
        // level calls to decide what to do with the event.

        // Construct an appropriate name for list to be created.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 CME_OBJECT_NAME":Pending Event List");

        // This is a simple circular, FIFO list of pending events
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->hPendingEventList,
                &acName[0],
                (OSAL_LL_COMPARE_HANDLER)NULL, // No order, just FIFO
                OSAL_LL_OPTION_CIRCULAR
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize content monitoring as enabled
        psObj->bEnabled = TRUE;

        // Initialize current iterated entry (none)
        psObj->hCurrentEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Allocate a CME update event
        psObj->hUpdateEvent = 
            DECODER_hAllocateCMEUpdateEvent(hDecoder);
        if (psObj->hUpdateEvent == SMS_INVALID_EVENT_HDL)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CME_OBJECT_NAME
                ": Failed to allocate CME update event");
            break;
        }

        // Create CME update timer
        eReturnCode = OSAL.eTimerCreate(
            &psObj->hUpdateEventTimer,
            CME_OBJECT_NAME": Update Timer",
            vPostUpdateEvent,
            (void *)psObj);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CME_OBJECT_NAME
                ": Failed to create CME update timer");
            break;
        }

        psObj->bUpdateTimerActive = FALSE;

        // Assign CME object
        hCME = (CME_OBJECT)psObj;

        // Return CME object
        return hCME;

    } while(FALSE);

    // Something went wrong
    CME_vDestroy((CME_OBJECT)psObj);
    return CME_INVALID_OBJECT;
}

/*****************************************************************************
*
*   CME_vDestroy
*
* Destroys the Content Monitoring Engine. There is one instance of the
* CME for each DECODER object instantiated. Needs to be called when a
* DECODER is released as part of the DECODER clean-up.
*
* Inputs:
*
*   hCME    A handle to a valid CME object for which to destroy and all it's
*           resources.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
void CME_vDestroy ( CME_OBJECT hCME )
{
    BOOLEAN bOwner;
    CME_OBJECT_STRUCT *psObj = (CME_OBJECT_STRUCT *)hCME;

    // Verify input, caller must already own the object.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCME);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // Check if update timer exist
    if (psObj->hUpdateEventTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eTimerDelete(psObj->hUpdateEventTimer);
    }

    // Check if the pending event list exists.
    if(psObj->hPendingEventList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list
        OSAL.eLinkedListRemoveAll(
            psObj->hPendingEventList,
            (OSAL_LL_RELEASE_HANDLER)NULL // No remove handler required
                                          // since pending list is just a list
                                          // of entries which are in the
                                          // content list anyway.
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->hPendingEventList);
        psObj->hPendingEventList = OSAL_INVALID_OBJECT_HDL;
    }

    // Check if the master content list exists.
    if(psObj->hMasterContentList != OSAL_INVALID_OBJECT_HDL)
    {
        // Destroy all content entries first. Anything left in this
        // list evidently never got removed by other means. So since
        // we are kind of pulling the rug out from the upper layers
        // we also need to notify them we removed this.
        OSAL.eLinkedListRemoveAll(
            psObj->hMasterContentList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyContentEntry
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->hMasterContentList);
        psObj->hMasterContentList = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the object itself
    SMSO_vDestroy((SMS_OBJECT)hCME);

    return;
}

/*****************************************************************************
*
*   CME_bUpdateContent
*
* Runs the Content Monitoring Engine updating any pending events which need
* to be triggered. This API essentially queues a list of events which need
* to be triggered but does not actually trigger them. Another API,
* CME_vTriggerEvents() must be called to actually trigger any pending
* or queued events based on content changed.
*
* This API is used whenever we have a CDO which needs to have one of it's
* CID's either updated or created for the first time. This API updates or
* creates the CID as needed and indicates if this resulted in a change of
* content.
*
* Inputs:
*
*   hCDO        A handle to a valid CDO object for which to inform the CME
*               about content which is either new or no longer being broadcast.
*   eNewType    A CID type which described the new CID. This is used in case
*               a CID handle is not provided and a new CID needs to be
*               created.
*   pvNewObjectData A pointer to the new object data which defines the CID.
*               This is the CID specific data used to create or update the CID.
*   phCID       A CID handle pointer to use as a CID to update with the new
*               content or if it points to an INVALID handle, to be populated
*               with a new CID to be created within this API.
*
* Outputs:
*
*   BOOLEAN     FALSE if no update was performed or TRUE if an existing CID's
*               content was updated.
*
*****************************************************************************/
BOOLEAN CME_bUpdateContent (
    CD_OBJECT hCDO,
    CID_ENUM eNewType,
    const void *pvNewObjectData,
    CID_OBJECT *phCID
        )
{
    BOOLEAN bOwner, bChanged = FALSE;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CME_OBJECT hCME = CME_INVALID_OBJECT;

    do
    {
        // Determine if the CDO provided  is valid and the caller
        // already owns this resource.
        bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);
        if(bOwner == FALSE)
        {
            // Error! Caller must provide an owned CDO
            break;
        }

        // Provided CID must be of valid type
        if((eNewType < CID_FIRST) || (eNewType > CID_LAST))
        {
            // Error! Invalid CID type provided
            break;
        }

        // Input CID handle pointer cannot be NULL
        if(phCID == NULL)
        {
            // Error! Handle pointer cannot be NULL, but what it points to
            // can be INVALID.
            break;
        }

        // Input object data cannot be NULL
        if((pvNewObjectData == NULL) && (*phCID != CID_INVALID_OBJECT))
        {
            // remove cid
            CME_eDestroyContent(hCDO, phCID);
            bChanged = TRUE;
        }

        // Extract the CME and CHANNEL handles associated with this CDO object.
        // Not all handles may always be available, so one must check as needed.
        vGetHandles(hCDO, &hCME, &hChannel);

        // Check if CID object does not exist
        if(*phCID == CID_INVALID_OBJECT)
        {
            // It does not exist so create it for the first time using
            // caller's provided handle pointer.
            *phCID = CDO_hCidCreate(hCDO, eNewType, pvNewObjectData);
            if(*phCID != CID_INVALID_OBJECT)
            {
                // Indicate provided CID was created (and thus changed)
                bChanged = TRUE;
            }
        }
        // CID already exists and we have a CME
        else if(hCME != CME_INVALID_OBJECT)
        {
            CID_OBJECT hId;
            CID_ENUM eType;

            // Make a copy of this CID, because
            // we first need to modify it, then update the CME letting it know
            // the content is going away. But we only do that if the content
            // has actually changed (i.e. we received two CID of the same
            // content back to back, which might happen if the song changes
            // but artist does not for example).
            // Anyway once we have a copy of the 'old' CID or content which
            // is going away, we can then modify it with the 'cid' CID and
            // at the same time determine if a change occurred.
            // However, instead of using the hDuplicate method (which invokes
            // a new allocation always) we just use a few friend functions
            // such as CID_hCreate() which first checks to see if there is a
            // CID which can be recycled and uses that. If a CID cannot be
            // recycled (not available) then it simply creates a new CID
            // anyway, but then we just recycled it because we need it only
            // for a short time. It's just more efficient this way.

            // Retrieve current CID type so that we can allocate/acquire one
            // of the same type.
            eType = CID_eType(*phCID);

            // Now create or obtain a recycled CID of this type. Note we
            // don't specify object data here since the CID will already
            // allocate the optimum storage size for the CIDs object data.
            hId = CDO_hCidCreate(hCDO, eType, "???");
            if(hId != CID_INVALID_OBJECT)
            {
                // Now copy current CID into new CID
                CID_bCopy(hId, *phCID);

                // Now since we got here because there already existed a CID
                // which is not going to get modified, we just update what we
                // have with the new object data.
                bChanged =
                    CID_bModify(phCID, pvNewObjectData);
                if((bChanged == TRUE) && (hChannel != CHANNEL_INVALID_OBJECT))
                {
                    // Indicate provided CID was updated (changed) and is
                    // now going away.

                    // Since the CID already exists, we need to make sure
                    // we trigger content updates for the existing content which
                    // has now ended and has changed. This is done by providing
                    // the FALSE argument.
                    vUpdateContent(hCME, hChannel, hId, FALSE);
                }

                // We don't need the 'copy' anymore. Recycle it.
                CID_vDestroy(hId);
            }
            else // Unable to create a 'copy' of the existing CID
            {
                // We couldn't create a copy of the existing CID to compare
                // against the new CID. So we just assume they changed.
                // This might result in a false change notification, but no
                // information will be lost. Annoying but all we can do.

                // Tell CME a change occurred, this CID is going away
                vUpdateContent(hCME, hChannel, *phCID, FALSE);

                // Modify the CID with new data
                bChanged = CID_bModify(phCID, pvNewObjectData);
            }
        }

        // If something changed (new content) and we have a CME handle
        // Otherwise there is nothing to do here.
        if((bChanged == TRUE) && (hCME != CME_INVALID_OBJECT) &&
            (hChannel != CHANNEL_INVALID_OBJECT))
        {
            // Update the CME with the new CID (what it is now)
            // to potentially trigger any events for content which
            // is now occurring. This is done by providing the TRUE
            // argument.
            vUpdateContent(hCME, hChannel, *phCID, TRUE);
        }


    } while(0);

    // Return an indicator which tells the caller if anything changed.
    return bChanged;
}

/*****************************************************************************
*
*   CME_bChangeContent
*
* Runs the Content Monitoring Engine updating any pending events which need
* to be triggered. This API essentially queues a list of events which need
* to be triggered but does not actually trigger them. Another API,
* CME_vTriggerEvents() must be called to actually trigger any pending
* or queued events based on content changed.
*
* This API is used when we already know we have an old or previous CID
* we are simply changing to a new CID with new content. In this case we want
* to make sure we potentially trigger events when the content is one thing
* and then becomes another. This potentially results in two events. The first
* event indicating the content has "ended" and another indicating the new
* content has "started". This also filters out incomming content changes
* which really do not have any content which changed (repeat info)
*
* Inputs:
*
*   hCDO    A handle to a valid CDO object for which to inform about
*           content which is either new or no longer being broadcast.
*   hOldCID A CID for which to update content with. This CID is the
*           CID about to be replaced or the new CID content (old to new).
*   hNewCID A CID for which to update content with. This CID is the
*           CID replacing the old CID content (new one).
*
* Outputs:
*   BOOLEAN  FALSE if no update or TRUE if an existing CID's
*            content was updated.
*
*****************************************************************************/
BOOLEAN CME_bChangeContent (
    CD_OBJECT hCDO,
    CID_OBJECT hOldCID,
    CID_OBJECT hNewCID
        )
{
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CME_OBJECT hCME = CME_INVALID_OBJECT;
    BOOLEAN bOwner, bEqual, bChanged = FALSE;

    do
    {
        // Determine if the CDO provided  is valid and the caller
        // already owns this resource.
        bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);
        if(bOwner == FALSE)
        {
            // Error! Caller must provide an owned CDO
            break;
        }

        // First, check if the two provided CIDs are not equal. If they
        // are equal then there is really no reason to run this.
        bEqual = (CID.n16Equal(hOldCID, hNewCID) == 0) ?
            TRUE : FALSE;
        if(bEqual == TRUE)
        {
            // Indicate nothing changed to caller.
            break;
        }

        // Extract the CME and CHANNEL handles associated with this CDO object.
        // Not all handles may always be available, so one must check as needed.
        vGetHandles(hCDO, &hCME, &hChannel);
        if((hCME != CME_INVALID_OBJECT) && (hChannel != CHANNEL_INVALID_OBJECT))
        {
            // Now, run the CME with the 'previous' (old) CID
            // to potentially trigger any events for content which is no
            // longer occurring.
            vUpdateContent(hCME, hChannel, hOldCID, FALSE);

            // Finally run the CME with the 'current' CID (what it changed to)
            // to potentially trigger any events for content which is now
            // occurring.
            vUpdateContent(hCME, hChannel, hNewCID, TRUE);
        }

        // Indicate content has changed
        bChanged = TRUE;

    } while(0);

    return bChanged;
}

/*****************************************************************************
*
*   CME_eDestroyContent
*
* Runs the Content Monitoring Engine updating any pending events which need
* to be triggered. This API essentially queues a list of events which need
* to be triggered but does not actually trigger them. Another API,
* CME_vTriggerEvents() must be called to actually trigger any pending
* or queued events based on content changed.
*
* Specifically this API handles the case when content is not changed in so
* much as it has been destroyed or no longer exists along with no other
* content to replace it. This might happen if for instance a program
* changes from MUSIC to NEWS or something like that.
*
* Inputs:
*
*   hCDO    A handle to a valid CDO object for which to inform about
*           content which is no longer being broadcast.
*   hCID    A CID for which to remove content from.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM  SMSAPI_RETURN_CODE_SUCCESS on success. Or not on error
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eDestroyContent (
    CD_OBJECT hCDO,
    CID_OBJECT *phCID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_SUCCESS;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CME_OBJECT hCME = CME_INVALID_OBJECT;
    BOOLEAN bOwner;

    do
    {
        // Determine if the CDO provided  is valid and the caller
        // already owns this resource.
        bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);
        if(bOwner == FALSE)
        {
            // Error! Caller must provide an owned CDO
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        // Check if CID provided.
        if(phCID == NULL)
        {
            // Nothing to do!
            break;
        }

        // Check if CID is valid.
        if(*phCID == CID_INVALID_OBJECT)
        {
            // Nothing to do!
            break;
        }

        // Extract the CME and CHANNEL handles associated with this CDO object.
        // Not all handles may always be available, so one must check as needed.
        vGetHandles(hCDO, &hCME, &hChannel);
        if((hCME != CME_INVALID_OBJECT) && (hChannel != CHANNEL_INVALID_OBJECT))
        {
            // Now, run the CME with the previous (old) CID
            // to potentially trigger any events for content which is no
            // longer occurring.
            vUpdateContent(hCME, hChannel, *phCID, FALSE);
        }

        // Destroy this CID
        CID_vDestroy(*phCID);
        *phCID = CID_INVALID_OBJECT;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_vTriggerEvents
*
* Instructs ALL pending Content Monitoring Events to be triggered. This
* function iterates the list of all pending events which were added while
* the DECODER was processing newly received CIDs (the DECODER calls
* to CME_bUpdateContent, CME_bChangeContent and CME_eDestroyContent).
* The iteration (during list removal) triggers each event in the list.
* At the end of this call all pending events will be removed from the list
* leaving the list empty.
*
* Inputs:
*
*   hCME    A handle to a valid CME object for which to trigger pending events.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
void CME_vTriggerEvents(
    CME_OBJECT hCME,
    BOOLEAN bBlockUpdateTimer
        )
{
    BOOLEAN bOwner;
    CME_OBJECT_STRUCT *psObj = (CME_OBJECT_STRUCT *)hCME;

    // Verify input, caller must already own the CME object.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCME);
    if(bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        if (psObj->bUpdateTimerActive == TRUE)
        {
            eReturnCode = OSAL.eTimerStop(psObj->hUpdateEventTimer);
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->bUpdateTimerActive = FALSE;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CME_OBJECT_NAME
                    ": Failed to stop CME update timer: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), 
                    eReturnCode);

                // In case of failure, this will just cause
                // one more update.
            }
        }

       // sort the list, before triggering
       eReturnCode = OSAL.eLinkedListSort(
           psObj->hPendingEventList, n16Sort);

       if (eReturnCode != OSAL_SUCCESS)
       {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               CME_OBJECT_NAME
               ": Failed to sort pedning event list: %s (#%d)",
               OSAL.pacGetReturnCodeName(eReturnCode), 
               eReturnCode);

           // In this case events will be in FIFO order, as is.
           // Not exactly what was desired, but still acceptable.
       }

       // Block update timer (if desired)
       psObj->bBlockUpdateTimer = bBlockUpdateTimer;

        // Remove all entries, meanwhile executing a trigger
        // event callback.
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->hPendingEventList,
            (OSAL_LL_RELEASE_HANDLER)vTriggerEvent
                );

        // Make sure the update timer unlocked
        psObj->bBlockUpdateTimer = FALSE;

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               CME_OBJECT_NAME
               ": Failed to trigger pending events: %s (#%d)",
               OSAL.pacGetReturnCodeName(eReturnCode), 
               eReturnCode);
        }
    }

    return;
}

/*****************************************************************************
*
*   CME_eRegisterContent
*
* Register for a content event by adding it to the master contentlist.
* The API allows caller's to add an entry in the master content list
* for content monitoring. The caller's content and associated parameters
* are stored in the master list in a specific order to fascilitate fast
* look-up, searches and compares.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be registered with.
*   hCID        A content identifier representing content to be monitored.
*   un32Options Options provided by the caller about this registration
*               such as initial, end, etc.
*   vEventCallback A pointer to a function to call when the registered content
*               is detected.
*   pvEventCallbackArg A pointer to an anonymous type (defined by the caller)
*               which will be associated with this registered content and
*               provided to the caller when the content event occurs.
*   phContent   A pointer to a content entry handle which is assigned
*               the the registered content as a reference to this content
*               entry within the CME object. This may be used by callers to
*               reference previously registered content.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM  SMSAPI_RETURN_CODE_SUCCESS on success. Or not on error
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eRegisterContent (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hCID,
    UN32 un32Options,
    CME_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    CME_REGISTERED_ENTRY *phContentEntry
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_ENTRY_STRUCT *psEntry = NULL;
    CME_OBJECT_STRUCT *psObj;
    BOOLEAN bAdded, bValid;

    // Check and condition options provided
    eReturnCode = eVerifyOptions(&un32Options);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error! Invalid options provided
        return eReturnCode;
    }

    // Check if the event callback is valid, if not what is the point?
    if(vEventCallback == NULL)
    {
        // Error! No callback event specified
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object, this only fails if the DECODER
    // handle provided is invalid.
    eReturnCode = eLock(hDecoder, &psObj);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        return eReturnCode;
    }

    // Verify provided CID is valid
    bValid = SMSO_bValid((SMS_OBJECT)hCID);
    if(bValid == FALSE)
    {
        // Error! Caller provided a bad CID
        vUnlock(hDecoder);
        return SMSAPI_RETURN_CODE_CID_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the DECODER
    // object and we have extracted the associated CME instance (psObj).
    // Now we can register the caller's content.

    // Create a new content entry for this registration providing caller's
    // arguments.
    psEntry = psCreateContentEntry(
        psObj, hCID, un32Options,
        vEventCallback, pvEventCallbackArg);
    if(psEntry != NULL)
    {
        // Now that we have the entry structure (instance), we can add it to
        // the master content list and re-populate the content list pointers
        // for different event types (cheats).
        bAdded = bAddContentEntry(psObj, psEntry);
        if(bAdded == TRUE)
        {
            // Content has been successfully added
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            // Something went wrong.
            eReturnCode = SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT;
        }
    }
    else
    {
        // Something went wrong
        eReturnCode = SMSAPI_RETURN_CODE_UNABLE_TO_CREATE_CONTENT;
    }

    // If the return code is anything but SMSAPI_RETURN_CODE_SUCCESS remove
    // and destroy the content entry as appropriate
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Check if entry exists
        if(psEntry != NULL)
        {
            // Destroy entry, but do not notify
            psEntry->sContent.vEventCallback = NULL;
            vDestroyContentEntry(psEntry);
            psEntry = NULL;
        }
    }
    // Check if they want to monitor against any currently on-going
    // program content, rather than just waiting for the next update or
    // occurrence. If this is what they want then we must scan the
    // current channel cache for any content which might match.
    if( (psEntry != NULL)
       && (psEntry->sContent.un32Options &
              SMS_EVENT_REGISTRATION_OPTION_CURRENT)
      )
    {
        BOOLEAN bSuccess;
        CCACHE_OBJECT hCCache;
        CME_CACHE_ITERATOR_STRUCT sIterator;

        // Yes, they do. So we must scan current content for a match to
        // the content which has just been added (registered) with the CME.

        // Learn the DECODER's channel cache handle
        hCCache = DECODER_hCCache(hDecoder);

        // Populate iterator structure to record the CME handle and the
        // CID object we are looking for.
        sIterator.hId = hCID;
        sIterator.hCME = (CME_OBJECT)psObj;
        sIterator.psEntry = psEntry;

        // Look through the channel cache and see if we can find this CID.
        // This iterator will handle any and all channels which have a
        // matching CID.
        bSuccess = CCACHE_bIterateChannels(
            hCCache, FALSE, bChannelIterator, &sIterator);
        if(bSuccess == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
        psEntry = sIterator.psEntry;
    }
    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
       && (psEntry !=NULL))
    {
        // Return a copy of this entry's handle
        *phContentEntry = (CME_REGISTERED_ENTRY)psEntry->hEntry;
    }
    else
    {
        *phContentEntry = NULL;
    }

    // Unlock DECODER object
    vUnlock(hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eIterateContent
*
* This API is used to iterate the master list of all registered content.
* While iterating the callback iterator allows the ability to replace,
* remove, enable and disable each registered content.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be iterated with.
*   hStartEntry A handle to the first entry to iterate with. If this input
*               is OSAL_INVALID_LINKED_LIST_ENTRY then we just start from
*               the beginning.
*   bContentIterator The function to call for each content entry in the
*               master content list until the return value here is FALSE.
*   pvContentIteratorArg A pointer to an anonymous type (defined by the caller)
*               which will passed to the iterator function for each entry
*               that is iterated.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM  SMSAPI_RETURN_CODE_SUCCESS on success. Or not on error
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eIterateContent (
    DECODER_OBJECT hDecoder,
    CME_REGISTERED_ENTRY hStartEntry,
    CME_ITERATOR_CALLBACK bContentIterator,
    void *pvContentIteratorArg
        )
{
    CME_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    CME_ITERATOR_STRUCT sIterator;
    CME_ENTRY_STRUCT *psEntry = NULL, *psNextEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hCurrentEntry, hNextEntry;

    // Check input
    if(bContentIterator == NULL)
    {
        // Error! Invalid input.
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object, this only fails if the DECODER
    // handle provided is invalid.
    eReturnCode = eLock(hDecoder, &psObj);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        return eReturnCode;
    }

    // At this point we have exclusive access to the DECODER
    // object and we have extracted the associated CME instance (psObj).

    // Remember current entry handle to fasciliate nested calls.
    hCurrentEntry = psObj->hCurrentEntry;

    // Set return code before iterating
    eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Initialize current iterated entry. This may be from the beginning
    // of the list or some random access point within the list.
    if((OSAL_LINKED_LIST_ENTRY)hStartEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        BOOLEAN bContinue;

        // Extract entry object from entry handle
        psEntry = OSAL.pvLinkedListThis((OSAL_LINKED_LIST_ENTRY)hStartEntry);
        if(psEntry != NULL)
        {
            // Initialize the current iterated entry
            psObj->hCurrentEntry = (OSAL_LINKED_LIST_ENTRY)hStartEntry;

            // We will iterate the list from an arbitrary start point.
            // First thing we do is find the content structure of the
            // chosen start point.

            // Loop (iterate) list from this point onward until
            // the return-value is FALSE or we reach the end of the list.
            do
            {
				// Get the next one
                hNextEntry =
                    OSAL.hLinkedListNext(psObj->hCurrentEntry, (void*)&psNextEntry);

                // Use the current entry
                bContinue = bContentIterator(
                    hDecoder, &psEntry->sContent, pvContentIteratorArg);

                // Go to the next one
                psObj->hCurrentEntry = hNextEntry;
				psEntry = psNextEntry;

            } while (
                (bContinue == TRUE) &&
                (psObj->hCurrentEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
                (psEntry != NULL)
                    );

            // List was iterated
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }
    else
    {
        // Populate iterator structure
        sIterator.hDecoder = hDecoder;
        sIterator.bIterator = bContentIterator;
        sIterator.pvIteratorArg = pvContentIteratorArg;
        sIterator.phCurrentEntry = &psObj->hCurrentEntry;

        // Iterate all registered content
        eOsalReturnCode =
            OSAL.eLinkedListIterate(
                psObj->hMasterContentList,
                (OSAL_LL_ITERATOR_HANDLER)bIterator, &sIterator);
        if( (eOsalReturnCode == OSAL_SUCCESS) ||
            (eOsalReturnCode == OSAL_NO_OBJECTS))
        {
            // List was iterated
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    // Re-initialize current iterated content since we are no
    // longer iterating. Restore current entry
    psObj->hCurrentEntry = hCurrentEntry;

    // Unlock DECODER object
    vUnlock(hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eReplaceContent
*
*   This API is used to replace the content which is currently being
*   iterated with new content information. It must only be called from within
*   a content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will replace the current content with new content information
*   provided by the caller.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be replaced within.
*   un32Options A set of options to use to modify the event with
*               (always replaced).
*   vEventCallback  A callback function to replace the current content with.
*               NULL if to leave as is.
*   pvEventCallbackArg A callback function argument to replace the current
*               content with (always replaced).
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eReplaceContent(
    DECODER_OBJECT hDecoder,
    UN32 un32Options,
    CME_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_OBJECT_STRUCT *psObj;
    CME_ENTRY_STRUCT *psEntry;
    BOOLEAN bAdded;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Check and condition options provided
        eReturnCode = eVerifyOptions(&un32Options);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error! Invalid options provided
            return eReturnCode;
        }

        // Update remaining content information according to inputs...
        if(vEventCallback != NULL)
        {
            // Replace callback function
            psEntry->sContent.vEventCallback = vEventCallback;
        }

        // Always replace callback argument.
        psEntry->sContent.pvEventCallbackArg = pvEventCallbackArg;

        // Always replace options.
        psEntry->sContent.un32Options = un32Options;

        // Replace content with new information (re-add or replace) as needed.
        bAdded = bAddContentEntry(psObj, psEntry);
        if(bAdded == FALSE)
        {
            // Something went wrong.
            eReturnCode = SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT;
            break;
        }

        // If we made it this far all is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eRemoveContent
*
*   This API is used to remove the current content being iterated from the
*   master content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will remove the current iterated content.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be removed from.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eRemoveContent(
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_OBJECT_STRUCT *psObj;
    CME_ENTRY_STRUCT *psEntry;
    BOOLEAN bRemoved;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, &psObj, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Remove this content from the master list.
        bRemoved = bRemoveContentEntry(psObj, psEntry);
        if(bRemoved == FALSE)
        {
            // Unable to remove content.
            eReturnCode = SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
            break;
        }

        // Destroy this content entry, but don't notify anything
        psEntry->sContent.vEventCallback = NULL;
        vDestroyContentEntry(psEntry);

        // If we made it this far all is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eEnableContent
*
*   This API is used to enable the current content being iterated from the
*   master content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will enable the current iterated content.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be enabled within.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eEnableContent(
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_ENTRY_STRUCT *psEntry;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, NULL, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Enable this content to be monitored and generate events
        psEntry->un32StateFlags &= ~(CONTENT_STATE_DISABLED);
        psEntry->un32StateFlags |= CONTENT_STATE_ENABLED;

        // If we made it this far all is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eDisableContent
*
*   This API is used to disable the current content being iterated from the
*   master content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will disable the current iterated content.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be disabled within.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on
*                               error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eDisableContent(
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_ENTRY_STRUCT *psEntry;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, NULL, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Disable this content from being monitored and generateing events
        psEntry->un32StateFlags &= ~(CONTENT_STATE_ENABLED);
        psEntry->un32StateFlags |= CONTENT_STATE_DISABLED;

        // If we made it this far all is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eContentEnabled
*
*   This API is used to query the current content being iterated from the
*   master content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will retrieve the enabled status of the currently iterated
*   content.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object whose enabled status will be reported.
*   pbEnabled   Pointer to copy the enabled status.
*               TRUE = Enabled. FALSE = Disabled
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on
*                               error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eContentEnabled(
    DECODER_OBJECT hDecoder,
    BOOLEAN *pbEnabled
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_ENTRY_STRUCT *psEntry;

    if (pbEnabled == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, NULL, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Retrieve staus of this content
        if (psEntry->un32StateFlags & CONTENT_STATE_ENABLED)
        {
            *pbEnabled = TRUE;
        }
        else
        {
            *pbEnabled = FALSE;
        }

        // If we made it this far all is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_hCID
*
*   This API is used to query the current content being iterated from the
*   master content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will retrieve the CID of the currently iterated content.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object whose CID will be returned.

*
*   Outputs:
*
*   valid CID OBJECT on success, CID_INVALID_OBJECT on failure
*
*****************************************************************************/
CID_OBJECT CME_hCID(
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_ENTRY_STRUCT *psEntry;
    CID_OBJECT hCID = CID_INVALID_OBJECT;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, NULL, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Retrieve staus of this content
        hCID = psEntry->sContent.hCID;

        // If we made it this far all is well

    } while(0);

    return hCID;
}

/*****************************************************************************
*
*   CME_eBlockAllEvents
*
* Instructs ALL Content Monitoring Events to be disabled. This superceeds
* any individual content flag settings (enabled or disabled) although it
* does not alter individual content flags. Instead this flag is ANDed
* with individual content flags to determine if an event should not be
* triggered.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be blocked for.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eBlockAllEvents (
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_OBJECT_STRUCT *psObj;

    // Verify and lock SMS Object
    eReturnCode = eLock(hDecoder, &psObj);
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // At this point we have exclusive access to the DECODER
        // object and we have extracted the associated CME instance (psObj).

        // Disable all events
        psObj->bEnabled = FALSE;

        // Unlock DECODER object
        vUnlock(hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eUnBlockAllEvents
*
* Instructs ALL Content Monitoring Events to be enabled. This superceeds
* any individual content flag settings (enabled or disabled) although it
* does not alter individual content flags. Instead this flag is ANDed
* with individual content flags to determine if a event should be triggered.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be blocked for.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eUnBlockAllEvents (
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_OBJECT_STRUCT *psObj;

    // Verify and lock SMS Object
    eReturnCode = eLock(hDecoder, &psObj);
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // At this point we have exclusive access to the DECODER
        // object and we have extracted the associated CME instance (psObj).

        // Enable all events
        psObj->bEnabled = TRUE;

        // Unlock DECODER object
        vUnlock(hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_eSearchForCurrent
*
*   This API is used to search if the current content being iterated from the
*   master content list is occurring. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will search for content and trigger events if it is found.
*
*   Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object whose enabled status will be reported.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM
*       SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CME_eSearchForCurrent(
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    CME_ENTRY_STRUCT *psEntry;
    BOOLEAN bSuccess;
    CCACHE_OBJECT hCCache;
    CME_CACHE_ITERATOR_STRUCT sIterator;

    do
    {
        // Retrieve object information
        eReturnCode = eAccessObjectFromDecoder(hDecoder, NULL, &psEntry);
        if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            break;
        }

        // Learn the DECODER's channel cache handle
        hCCache = DECODER_hCCache(hDecoder);

        // Populate iterator structure to record the CME handle and the
        // CID object we are looking for.
        sIterator.hId = psEntry->sContent.hCID;
        sIterator.hCME = DECODER_hCME(hDecoder);
        sIterator.psEntry = psEntry;

        // Look through the channel cache and see if we can find this CID.
        // This iterator will handle any and all channels which have a
        // matching CID.
        bSuccess = CCACHE_bIterateChannels(
            hCCache, FALSE, bChannelIterator, &sIterator);
        if(bSuccess == TRUE)
        {
            // If we made it this far all is well
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    } while(0);

    return eReturnCode;
}

/*****************************************************************************
*
*   CME_vStartTimeoutEventTimer
*
*****************************************************************************/
void CME_vStartTimeoutEventTimer(
    CME_OBJECT hCME
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCME);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // De-reference object
        CME_OBJECT_STRUCT *psObj = 
            (CME_OBJECT_STRUCT *)hCME;

        if (psObj->bBlockUpdateTimer == FALSE)
        {
            eReturnCode = OSAL.eTimerStartRelative(
                psObj->hUpdateEventTimer,
                CME_UPDATE_EVENT_TIMEOUT,
                0); // One-shot

            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->bUpdateTimerActive = TRUE;
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CME_OBJECT_NAME
                    ": Failed to start CME update timer: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), 
                    eReturnCode);
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   CME_DEBUG_vPrintContent
*
* This API is used to print all the registered content belonging to this
* DECODER as part of the associated CME object.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object which owns a CME object.
*               This is the CME object the content will be printed for.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
void CME_DEBUG_vPrintContent (
    DECODER_OBJECT hDecoder,
    FILE *psFile
        )
{
#if SMS_DEBUG == 1
    CME_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify and lock SMS Object, this only fails if the DECODER
    // handle provided is invalid.
    eReturnCode = eLock(hDecoder, &psObj);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        return;
    }

    // At this point we have exclusive access to the DECODER
    // object and we have extracted the associated CME instance (psObj).

    // Iterate all registered content
    OSAL.eLinkedListIterate(
        psObj->hMasterContentList,
        (OSAL_LL_ITERATOR_HANDLER)bPrintContent, psFile);

    // Unlock DECODER object
    vUnlock(hDecoder);
#endif
    return;
}

#if SMS_DEBUG == 1
/*****************************************************************************
*
*   bPrintContent
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. This is used to
* print the verbose contents of and entry in the master content list.
*
*****************************************************************************/
static BOOLEAN bPrintContent (
    CME_ENTRY_STRUCT *psEntry,
    FILE *psFile
        )
{
    // Check inputs
    if((psEntry == NULL) || (psFile == NULL))
    {
        // Cease iterating as something is wrong.
        return FALSE;
    }

    // Just dump it...
    fprintf(psFile,
        "hChannel = 0x%0p\n"
        "hEntry = 0x%0p\n"
        "un32StateFlags = %#0x\n"
        "sContent:\n"
        "\thCID = 0x%0p\n"
        "\tun32Options = %#0x\n"
        "\tvEventCallback = 0x%0p\n"
        "\tpvEventCallbackArg = 0x%0p\n\n"
        ,
        psEntry->hChannel,
        psEntry->hEntry,
        psEntry->un32StateFlags,
        psEntry->sContent.hCID,
        psEntry->sContent.un32Options,
        psEntry->sContent.vEventCallback,
        psEntry->sContent.pvEventCallbackArg
            );

    // Continue to the next one
    return TRUE;
}
#endif

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16CompareContent
*
*       This is the function used to compare two existing
*       CME_ENTRY_STRUCTs. Specifically this function is
*       used to perform a relational comparison for sorting. This sorter keeps
*       the content in order of the assoicated CID.
*
*       Outputs:
*               0   - psEntry1's CID and psEntry2's CID have the same
*                       value (equal)
*               > 0 - psEntry1's CID is greater than (after) psEntry2's CID
*               < 0 - psEntry1's CID is less than (before) psEntry2's CID
*                       (or error)
*
*****************************************************************************/
static N16 n16CompareContent (
    CME_ENTRY_STRUCT *psEntry1,
    CME_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Result = N16_MIN;

    // Verify input
    if((psEntry1 != NULL) && (psEntry2 != NULL))
    {
        // Use the CIDs relational compare method
        n16Result = CID.n16Compare(
            psEntry1->sContent.hCID,
            psEntry2->sContent.hCID
                );
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16Equal
*
*       This is the function used to equate an existing CME_ENTRY_STRUCT's
*       CID and a provided CID. Specifically this function is used to perform
*       a binary comparison for testing a go, no-go situation.
*
*       Outputs:
*               0   - CIDs have the same value (equal)
*               < 0  - CIDs are not equal (or error)
*
*****************************************************************************/
static N16 n16Equal (
    CME_ENTRY_STRUCT *psEntry,
    CID_OBJECT hCID
        )
{
    N16 n16Result;

    // Verify input
    if((psEntry == NULL) || (hCID == CID_INVALID_OBJECT))
    {
        // Error!
        return N16_MIN;
    }

    // Use the CIDs equal compare method (binary)
    n16Result = CID.n16Equal(psEntry->sContent.hCID, hCID);

    return n16Result;
}

/*****************************************************************************
*
*   psCreateContentEntry
*
* Create a content entry and populate it. An event contains the CID to trigger
* on and the caller's callback and argument. Options are specified as to how
* the content monitoring is to behave.
*
*****************************************************************************/
static CME_ENTRY_STRUCT *psCreateContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CID_OBJECT hCID,
    UN32 un32Options,
    CME_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    CME_ENTRY_STRUCT *psEntry;

    do
    {
        // Create an instance of this object
        psEntry = (CME_ENTRY_STRUCT *)
            SMSO_hCreate(
                CME_OBJECT_NAME":Entry",
                sizeof(CME_ENTRY_STRUCT),
                (SMS_OBJECT)psObj, // This object is a child
                FALSE );           // of the hCME provided.

        if(psEntry == NULL)
        {
            // Error!
            break;
        }

        // Initialize entry self-reference (invalid for now)
        psEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add the internal event flags as needed (default as enabled)
        psEntry->un32StateFlags = CONTENT_STATE_ENABLED;

        // Initialize CHANNEL (for pending events)
        psEntry->hChannel = CHANNEL_INVALID_OBJECT;

        //////////////////////////////////////////////////
        // Populate content entry with provided inputs...
        //////////////////////////////////////////////////

        // Make a duplicate copy of the provided CID. This way we can
        // permanently own it since as content changes this CID will go away.
        psEntry->sContent.hCID = CID.hDuplicate(hCID);
        if(psEntry->sContent.hCID == CID_INVALID_OBJECT)
        {
            // Could not duplicate provided CID. Bail.
            break;
        }

        // Record caller's callback and argument.
        psEntry->sContent.vEventCallback = vEventCallback;
        psEntry->sContent.pvEventCallbackArg = pvEventCallbackArg;

        // Record caller's options (previously verified as valid).
        psEntry->sContent.un32Options = un32Options;

        //////////////////////////////////////////

        return psEntry;

    } while( FALSE );

    // Destroy anything this entry created as well as itself.
    vDestroyContentEntry(psEntry);

    return (CME_ENTRY_STRUCT *)NULL;
}

/*****************************************************************************
*
*   vDestroyContentEntry
*
* Destroy a content entry. The entries resources are completely released.
*
*****************************************************************************/
static void vDestroyContentEntry (
    CME_ENTRY_STRUCT *psEntry
        )
{
    BOOLEAN bValid;
    UN32 un32Flags = ACTIVE_EVENT_FLAG_FINAL;

    // Check input
    bValid = SMSO_bValid((SMS_OBJECT)psEntry);
    if(bValid == TRUE)
    {
        // Check if we need to notify anything
        if(psEntry->sContent.vEventCallback != NULL)
        {
            CME_OBJECT hCME;
            DECODER_OBJECT hDecoder;

            // All EVENT objects belong to a CME object (their parent)
            hCME = (CME_OBJECT)SMSO_hParent((SMS_OBJECT)psEntry);
            if(hCME == CME_INVALID_OBJECT)
            {
                // Not a valid handle
                return;
            }

            // All CME objects belong to a DECODER object (their parent)
            hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)hCME);
            if(hDecoder == DECODER_INVALID_OBJECT)
            {
                // Not a valid handle
                return;
            }

            // Prepare flags to provide callback
            if(psEntry->un32StateFlags & CONTENT_STATE_INITIAL)
            {
                un32Flags |= ACTIVE_EVENT_FLAG_INITIAL;
                psEntry->un32StateFlags &= ~CONTENT_STATE_INITIAL;
            }
            if(psEntry->un32StateFlags & CONTENT_STATE_END)
            {
                un32Flags |= ACTIVE_EVENT_FLAG_END;
                psEntry->un32StateFlags &= ~CONTENT_STATE_END;
            }

            // Call provided event callback function
            psEntry->sContent.vEventCallback(
                hDecoder,
                psEntry->hChannel,
                un32Flags,
                psEntry->sContent.pvEventCallbackArg
                    );
        }

        // Initialize content memory
        psEntry->hChannel = CHANNEL_INVALID_OBJECT;
        psEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psEntry->un32StateFlags = CONTENT_STATE_NONE;
        psEntry->sContent.pvEventCallbackArg = NULL;
        psEntry->sContent.un32Options = SMS_EVENT_REGISTRATION_OPTION_NONE;
        psEntry->sContent.vEventCallback = (CME_CALLBACK)NULL;

        // Destroy CID
        if(psEntry->sContent.hCID != CID_INVALID_OBJECT)
        {
            // Destroy our copy of the registered CID.
            CID.vDestroy(psEntry->sContent.hCID);
            psEntry->sContent.hCID = CID_INVALID_OBJECT;
        }

        // Destroy entry object itself
        SMSO_vDestroy((SMS_OBJECT)psEntry);

    }

    return;
}

/*****************************************************************************
*
*   bAddContentEntry
*
* Add/Replace a content entry in the master content list. Also update any
* content "cheats" by pointing to related groups of content.
*
*****************************************************************************/
static BOOLEAN bAddContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CME_ENTRY_STRUCT *psEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY *phCidGroupFirstEntry;
    CID_ENUM eType;

    // Determine the type of CID this content is for
    eType = CID_eType(psEntry->sContent.hCID);
    if(eType == CID_INVALID)
    {
        // Cannot add content to list
        return FALSE;
    }

    // Check if I need to add or replace it.
    if(psEntry->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) // Replace it
    {
        // Replace the given content entry with new contents.
        // Actually the psEntry pointer provided is already populated with
        // the new contents, I just want to force it to re-sort it using the
        // new information. Replacing the entry with itself does just that.
        eReturnCode = OSAL.eLinkedListReplaceEntry(
            psObj->hMasterContentList,
            psEntry->hEntry,
            (void *)psEntry);
    }
    else // Add it
    {
        // Add this entry to the master content list. It is a new entry
        // and has never been added.
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hMasterContentList,
            &psEntry->hEntry,
            (void *)psEntry);
    }

    if(eReturnCode != OSAL_SUCCESS)
    {
        // Cannot add/replace entry to/in list
        return FALSE;
    }

    // Now that the entry is added we want to see if it has
    // replaced the first entry in the CID group to which this belongs.
    // If the existing entry handle is NULL then this is the first and only and
    // can be set as such. If the valid and existing entry handle's previous
    // entry is equal to this one then it was replaced. So make this one the
    // new first in the group. If not, do nothing.

    // First check if the entry handle exists for this CID group
    phCidGroupFirstEntry = &psObj->ahEntry[eType];

    // If existing group entry is NULL, this is the new first (and only)entry
    if(*phCidGroupFirstEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Make this the first (and only) entry in the group
        *phCidGroupFirstEntry = psEntry->hEntry;
    }
    else
    {
        // Not the only entry, is it the new first entry?
        if(psEntry->hEntry ==
            OSAL.hLinkedListPrev(*phCidGroupFirstEntry, NULL))
        {
            // Yep, replace it
            *phCidGroupFirstEntry = psEntry->hEntry;
        }
    }

    // Content has now been successfully added to the CME

    // We're done now
    return TRUE;
}

/*****************************************************************************
*
*   bRemoveContentEntry
*
* Remove an entry from the master list as well as adjust any search enhancing
* indexies or "cheats".
*
*****************************************************************************/
static BOOLEAN bRemoveContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CME_ENTRY_STRUCT *psEntry
        )
{
    OSAL_LINKED_LIST_ENTRY hCidGroupFirstEntry;
    CID_ENUM eType;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hNextEntry;
    CME_ENTRY_STRUCT *psNextEvent;

    // Determine the type of CID this entry is for
    eType = CID_eType(psEntry->sContent.hCID);
    if(eType > CID_LAST)
    {
        // Cannot remove entry from list
        return FALSE;
    }

    // Get the first entry handle for this CID group. We know
    // it must exist or else it could have never been added.
    hCidGroupFirstEntry = psObj->ahEntry[eType];

    // The new first entry 'might' be the next, so let's determine what
    // that is exactly.
    hNextEntry = OSAL.hLinkedListNext(
        psEntry->hEntry, (void **)&psNextEvent);

    // Check if the existing group entry is the same as the one being
    // removed. If so, since this will be removed, the new entry must be the
    // next in the list if it is of the same type. Otherwise, make it NULL
    // since this means this was the last one.
    if(hCidGroupFirstEntry == psEntry->hEntry)
    {
        // Check if there is a next and if it contains a valid entry.
        if((hNextEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
           (psNextEvent != NULL))
        {
            CID_ENUM eNextEntryType;

            // Now we need to check if the next entry (which is valid)
            // is the same type of CID. So determine the type of the
            // next CID.
            eNextEntryType = CID_eType(psNextEvent->sContent.hCID);

            // Check if types match
            // If the next entry is not the same type, then there are no
            // more. So just NULL it out. If they are, then this is the next
            // first entry.
            if(eType != eNextEntryType) // Nope
            {
                hCidGroupFirstEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            }
            else // Yep! the same
            {
                hCidGroupFirstEntry = hNextEntry;
            }
        }
        else
        {
            // There are either no more entries or the next one
            // is invalid or NULL. Either way, initialize the first as empty.
            hCidGroupFirstEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
    }

    // Now we are free to remove the entry completely
    eReturnCode = OSAL.eLinkedListRemove(psEntry->hEntry);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Entry was removed successfully. Invalidate self reference
        psEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Set 'final' flag
        psEntry->un32StateFlags |= CONTENT_STATE_FINAL;

        // Update the new first entry 'cheat' for this group.
        // This does nothing if the one removed was not the first
        // since it will just get replaced with what it already had.
        psObj->ahEntry[eType] = hCidGroupFirstEntry;

        // Also adjust the current entry being iterated (if needed).
        psObj->hCurrentEntry = hNextEntry;

        return TRUE;
    }

    // If the entry could not be removed, then it remains with all
    // "cheats" intact as well.
    return FALSE;
}

/*****************************************************************************
*
*   bIterator
*
* This is the object iterator helper function which allows the use of
* an OSAL Linked List iterator with additional arguments. It simply uses the
* CME iterator structure to call the caller's own iterator function along
* with an argument for each entry in the list.
*
*****************************************************************************/
static BOOLEAN bIterator (
    CME_ENTRY_STRUCT *psEntry,
    CME_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bReturn = FALSE;

    // Check inputs
    if((psEntry == NULL) || (psIterator == NULL))
    {
        // Cease iterating as something is wrong.
        return FALSE;
    }

    // Call provided iterator function if one exists, if not
    // then there really is no reason to continue anyway.
    if(psIterator->bIterator != NULL)
    {
        // Set current iterated entry by using the self-reference.
        // This will be used if the implementation within the callback
        // desires to manipulate the list of entries
        // (i.e. replace, remove, etc.)
        *psIterator->phCurrentEntry = psEntry->hEntry;

        // Return whatever their callback returns back to the iterator
        bReturn = psIterator->bIterator (
            psIterator->hDecoder, &psEntry->sContent,
            psIterator->pvIteratorArg
                );
    }

    return bReturn;
}

/*****************************************************************************
*
*   eAccessObjectFromDecoder
*
*   This helper API is used to un-wrap the required CME object information
*   from a provided DECODER object handle. It is used to leverage common
*   object verification and aquisition for many of the event iteration APIs.
*   It also returns to the caller the pointer to the current entry being
*   iterated.
*
*   Inputs:
*      hDecoder     A handle to the DECODER object for which the
*                    current entry is being accessed.
*      ppsObj      A pointer which (if non-NULL) will be populated with
*                    the CME object pointer.
*      ppsEntry     A pointer which (if non-NULL) will be populated with
*                    the current entry being iterated (if any).
*
*   Outputs:
*      SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAccessObjectFromDecoder(
    DECODER_OBJECT hDecoder,
    CME_OBJECT_STRUCT **ppsObj,
    CME_ENTRY_STRUCT **ppsEntry
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_DECODER;
    CME_OBJECT_STRUCT *psLocalObj = NULL;
    CME_ENTRY_STRUCT *psLocalEvent = NULL;

    // Verify caller is already the owner of the object.
    // If not this function may not be called.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // Check if caller provided a valid ppsObj. If not, that's ok
        // we'll use our own.
        if(ppsObj == NULL)
        {
            // Use ours.
            ppsObj = &psLocalObj;
        }

        // Check if caller provided a valid ppsEntry. If not, that's ok
        // we'll use our own.
        if(ppsEntry == NULL)
        {
            // Use ours.
            ppsEntry = &psLocalEvent;
        }

        do
        {
            CME_OBJECT hCME;

            // Retrieve the DECODER's CME instance (if one exists)
            hCME = DECODER_hCME(hDecoder);
            if(hCME == CME_INVALID_OBJECT)
            {
                // Sorry, if this decoder does not support the ability
                // to monitor events then there is no point in continuing.
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
                break;
            }

            // De-reference the CME handle.
            *ppsObj = (CME_OBJECT_STRUCT *)hCME;

            // At this point we have exclusive access to the DECODER
            // object and we have extracted the associated CME instance.

            // Check to make sure we have a valid content entry being iterated
            if((*ppsObj)->hCurrentEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
                break;
            }

            // Get the currently iterated content information
            *ppsEntry = (CME_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis((*ppsObj)->hCurrentEntry);
            if(*ppsEntry == NULL)
            {
                eReturnCode = SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                break;
            }

            // If we made it this far all is well
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bLock
*
* This helper function locks the provided DECODER object and returns the
* extracted and de-referenced CME object if successful.
* NOTE: If error is returned, the DECODER will NOT be locked.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eLock (
    DECODER_OBJECT hDecoder,
    CME_OBJECT_STRUCT **ppsObj
        )
{
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CME_OBJECT hCME;

    do
    {
        // Verify and lock SMS Object
        bLocked =
            SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == FALSE)
        {
            // Unable to lock the provided object
            return SMSAPI_RETURN_CODE_INVALID_DECODER;
        }

        // Retrieve the DECODER's CME instance (if one exists)
        hCME = DECODER_hCME(hDecoder);
        if(hCME == CME_INVALID_OBJECT)
        {
            // Sorry, if this decoder does not support the ability
            // to monitor events then there is no point in continuing.
            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        // De-reference the CME handle.
        *ppsObj = (CME_OBJECT_STRUCT *)hCME;

        // At this point we have exclusive access to the DECODER
        // object and we have extracted the associated CME instance.

        // If we made it this far all is well
        return SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    // an error occurred. unlock DECODER
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   vUnlock
*
* This helper function unlocks the provided DECODER object. This is just a
* wrapper to make the lock/unlock calls look uniform, but it is necessary
* to properly unlock access to the DECODER object.
*
*****************************************************************************/
static void vUnlock (
    DECODER_OBJECT hDecoder
        )
{
    // Unlock DECODER object
    SMSO_vUnlock((SMS_OBJECT)hDecoder);
    return;
}

/*****************************************************************************
*
*   vTriggerEvent
*
* Trigger an event entry. This callback is called while iterating the list
* of pending content events. This is the function which actually gets called
* to call the registered callback provided when the content was registered.
*
*****************************************************************************/
static void vTriggerEvent (
    CME_ENTRY_STRUCT *psEntry
        )
{
    BOOLEAN bValid;
    UN32 un32Flags = 0;

    // Check input for validity
    bValid = SMSO_bValid((SMS_OBJECT)psEntry);
    if(bValid == TRUE)
    {
        CME_OBJECT hCME;
        DECODER_OBJECT hDecoder;

        // All EVENT objects belong to a CME object (their parent)
        hCME = (CME_OBJECT)SMSO_hParent((SMS_OBJECT)psEntry);
        if(hCME == CME_INVALID_OBJECT)
        {
            // Not a valid handle
            return;
        }

        // All CME objects belong to a DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)hCME);
        if(hDecoder == DECODER_INVALID_OBJECT)
        {
            // Not a valid handle
            return;
        }

        // Determine if this entry pointer is to be destroyed since.
        // This will be the case if the entry handle is invalid due
        // to being removed from the content list.
        if(
            (psEntry->hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            ((psEntry->un32StateFlags & CONTENT_STATE_FINAL) ==
                CONTENT_STATE_FINAL)
                )
        {
            // Notify caller via callback and also destroy it.
            vDestroyContentEntry(psEntry);
        }
        // Entry will not be destroyed, just do notification
        else
        {
            // Prepare flags to provide callback
            if(psEntry->un32StateFlags & CONTENT_STATE_INITIAL)
            {
                un32Flags |= ACTIVE_EVENT_FLAG_INITIAL;
                psEntry->un32StateFlags &= ~CONTENT_STATE_INITIAL;
            }
            if(psEntry->un32StateFlags & CONTENT_STATE_END)
            {
                un32Flags |= ACTIVE_EVENT_FLAG_END;
                psEntry->un32StateFlags &= ~CONTENT_STATE_END;
            }
            if(psEntry->un32StateFlags & CONTENT_STATE_FINAL)
            {
                un32Flags |= ACTIVE_EVENT_FLAG_FINAL;
                psEntry->un32StateFlags &= ~CONTENT_STATE_FINAL;
            }

            // Call provided event callback function
            psEntry->sContent.vEventCallback(
                hDecoder,
                psEntry->hChannel,
                un32Flags,
                psEntry->sContent.pvEventCallbackArg
                    );
        }
    }

    return;
}

/*****************************************************************************
*
*   vUpdateContent
*
* Runs the Content Monitoring Engine updating any pending events which need
* to be triggered. This is the real workhorse of the CEMS. This API essentially
* queues a list of events which need to be triggered but does not actually
* trigger them. Another API, CME_vTriggerEvents() must be called to actually
* trigger any pending or queued events based on content changed when ready.
*
* Inputs:
*
*   hCME    A handle to a valid CME object for which to inform about
*           content which is either new or no longer being broadcast.
*   hChannel A handle to a valid channel which contains the content being
*           examined here.
*   hCID    A CID for which to update content with. The CID is either the
*           CID about to be replaced or the new CID content.
*   bNewContent A flag to indicate if new content is being broadcast
*           (TRUE) or if content is no longer being broadcast (FALSE).
*
* Outputs:
*   None.
*
*****************************************************************************/
static void vUpdateContent (
    CME_OBJECT hCME,
    CHANNEL_OBJECT hChannel,
    CID_OBJECT hCID,
    BOOLEAN bNewContent
        )
{
    CME_OBJECT_STRUCT *psObj = (CME_OBJECT_STRUCT *)hCME;
    CID_ENUM eType;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    CME_ENTRY_STRUCT *psEntry;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    do
    {
        // Determine if all CME events are enabled. If not there is
        // nothing for us to do.
        if(psObj->bEnabled == FALSE)
        {
            // No harm done, just nothing to do
            SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Not Enabled");
            break;
        }

        // Determine the type of CID this event is for
        eType = CID_eType(hCID);
        if(eType > CID_LAST)
        {
            // Error! Cannot process CIDs which are invalid.
            break;
        }

        // Based on the CID type we can focus in on the first entry
        // in our master content list of entries registered with that same
        // CID type.
        hEntry = psObj->ahEntry[eType];
        if(hEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // There are no entries in the list for this CID group
            // so we need not do anything else. We're done here.
            break;
        }

        // This is the first entry in the group, but the Linked List
        // Search API uses the 'next' entry if one is provided. So we
        // need to first back it up. If there is no previous entry then
        // it will be NULL and that's ok too which means, search from
        // the beginning of the list.
        hEntry = OSAL.hLinkedListPrev(hEntry, (void **)NULL);

        // Now search the list, starting from the appropriate CID group's
        // first entry until that group is finished. Keep in mind that more than
        // one event may be registered for the same CID so we need to continue
        // our search even if one is found. However, once a CID has been found
        // and that CID is no longer in the list we know we are done and can
        // finish the search early. We don't stop till there are no
        // more to search. This search keeps searching from either the first one
        // in the group (initialized above) or from the last one found.
        for(;;)
        {
            // Search the list for this CID...
            eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                psObj->hMasterContentList,
                &hEntry,
                (OSAL_LL_COMPARE_HANDLER)n16Equal,
                (void*)hCID
                    );
            if(eOsalReturnCode != OSAL_SUCCESS)
            {
                // No content match found, just exit
                break;
            }

            // Extract event pointer from found content entry
            psEntry = (CME_ENTRY_STRUCT*)OSAL.pvLinkedListThis(hEntry);

            // Verify entry pointer is valid
            if(SMSO_bIsValid((SMS_OBJECT)psEntry) == FALSE)
            {
                // Error! Cannot process null entry. Exit loop.
                break;
            }

            // Ok, now we have found a potential match and it is valid.
            // Now it is time to decide what to do.

            // Check if this content is enabled, if not we just continue our
            // search for the next content (if one exists).
            if(
                  ((psEntry->un32StateFlags & CONTENT_STATE_DISABLED)
                            == CONTENT_STATE_DISABLED)
              )
            {
                // Content events for this entry has been disabled.
                // Search for another one.
                SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Content disabled. Skipping to the next one");
                continue;
            }

            // First we need to handle events where the provided CID matches
            // and it is not new content. This means this content is now going
            // away (ended). Otherwise we check to see if the flag indicates
            // new broadcast content is occurring.
            if(bNewContent == FALSE) // Content leaving
            {
                // Content has ended, over, done with, caput!
                SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Content is ended (options: %d)", psEntry->sContent.un32Options);

                // Content is not new and thus it has ended, check registration
                // options and determine if we need to queue an event. Check if
                // this content is registered to get "END" events.
                if((psEntry->sContent.un32Options &
                    SMS_EVENT_REGISTRATION_OPTION_END)
                        == SMS_EVENT_REGISTRATION_OPTION_END)
                {
                    // Add this event to the pending events list indicating it
                    // has ended.
                    SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: OPTION_END is present. Adding into the Pending list");
                    eOsalReturnCode = OSAL.eLinkedListAdd(
                        psObj->hPendingEventList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                        psEntry
                            );
                    if(eOsalReturnCode == OSAL_SUCCESS)
                    {
#if DEBUG_OBJECT
                        UN32 un32OldFlags = psEntry->un32StateFlags;
#endif
                        // Indicate event has been added to the pending
                        // list and what state it is. Note if the INITIAL
                        // reason was already set we clear it now because it
                        // is not right to report something is starting when
                        // we already know it ended.
                        psEntry->un32StateFlags &=
                            ~(CONTENT_STATE_INITIAL); // Clear state
                        psEntry->un32StateFlags |=
                            CONTENT_STATE_END; // Set end state

#if DEBUG_OBJECT
                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Updating flags %x => %x",
                            un32OldFlags, psEntry->un32StateFlags);

                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Updating channel %d => %d",
                            CHANNEL.tChannelId(psEntry->hChannel), CHANNEL.tChannelId(hChannel));
#endif

                        // Record channel which has this content as part of
                        // this event.
                        psEntry->hChannel = hChannel;

                        // Now, since the END event has occurred we need to
                        // see if this content is registered as a one-shot.
                        // If so, it must be removed from the master content
                        // list.
                        if((psEntry->sContent.un32Options &
                            SMS_EVENT_REGISTRATION_OPTION_ONETIME) ==
                                SMS_EVENT_REGISTRATION_OPTION_ONETIME
                                    )
                        {
                            // Remove this entry from the master content list
                            // since we always remove one-shot events when the
                            // END condition occurs.
                            bRemoveContentEntry(
                                (CME_OBJECT_STRUCT *)hCME, psEntry);
                            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                            SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: It was ONE_TIME. Removing from master list");
                        }
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Cannot add into Pending list (%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: OPTION_END is not present");
                }
            }
            else // New Content
            {
                // Content is new...yea!
                SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Content is started (options: %d)", psEntry->sContent.un32Options);

                // Content is new and thus it has started, check options
                // and determine if we need to trigger an event. Check if
                // registered to get "INITIAL" events.
                SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: OPTION_INITIAL is present. Adding into the Pending list");
                if((psEntry->sContent.un32Options &
                    SMS_EVENT_REGISTRATION_OPTION_INITIAL)
                        == SMS_EVENT_REGISTRATION_OPTION_INITIAL)
                {
                    // Add this event to the pending events list indicating it
                    // has begun.
                    eOsalReturnCode = OSAL.eLinkedListAdd(
                        psObj->hPendingEventList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                        psEntry
                            );
                    if(eOsalReturnCode == OSAL_SUCCESS)
                    {
#if DEBUG_OBJECT
                        UN32 un32OldFlags = psEntry->un32StateFlags;
#endif

                        // Indicate event has been added to the pending
                        // list and the state. Note if the END
                        // reason was already set we clear it now because it
                        // is not right to report something has ended when
                        // we already know it started.
                        psEntry->un32StateFlags &=
                            ~(CONTENT_STATE_END); // Clear state
                        psEntry->un32StateFlags |=
                            CONTENT_STATE_INITIAL; // Set initial state

#if DEBUG_OBJECT
                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Updating flags %x => %x",
                            un32OldFlags, psEntry->un32StateFlags);

                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Updating channel %d => %d",
                            CHANNEL.tChannelId(psEntry->hChannel), CHANNEL.tChannelId(hChannel));
#endif

                        // Record channel which has this content
                        psEntry->hChannel = hChannel;

                        // Now, since the INITIAL event has occurred we need to
                        // see if this content is registered as a one-shot AND
                        // it is set up ONLY for the initial event. If so, it
                        // must be removed from the master content list.
                        // If the options indicate...
                        if( (psEntry->sContent.un32Options &
                            (
                                SMS_EVENT_REGISTRATION_OPTION_ONETIME |
                                SMS_EVENT_REGISTRATION_OPTION_INITIAL |
                                SMS_EVENT_REGISTRATION_OPTION_END
                             ) )
                             ==
                             (
                                 // One-shot AND
                                 SMS_EVENT_REGISTRATION_OPTION_ONETIME |
                                 // Initial AND
                                 SMS_EVENT_REGISTRATION_OPTION_INITIAL
                                 // not END
                              )
                           )
                        {
                            // Remove this entry from the master content list
                            bRemoveContentEntry(
                                (CME_OBJECT_STRUCT *)hCME, psEntry);
                            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                            SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: OPTION_END is present. Removing from master list");
                        }
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: Cannot add into Pending list (%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(CME_OBJECT_NAME, 2, "vUpdateContent: OPTION_INITIAL is not present");
                }
            }
        } // for(;;)

    } while(0);

    return;
}


/*****************************************************************************
*
*   vUpdateContent
*
* For a specific CME entry set events.
*
* Inputs:
*
*   hCME    A handle to a valid CME object for which to inform about
*           content which is either new or no longer being broadcast.
*   hChannel A handle to a valid channel which contains the content being
*           examined here.
*   psEntry The CME Entry to update
*
* Outputs:
*   None.
*
*****************************************************************************/
static void vUpdateSpecificContent (
    CME_OBJECT hCME,
    CHANNEL_OBJECT hChannel,
    CME_ENTRY_STRUCT *psEntry,
    BOOLEAN *pbContentRemoved
        )
{
    CME_OBJECT_STRUCT *psObj = (CME_OBJECT_STRUCT *)hCME;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    *pbContentRemoved = FALSE;
    do
    {
        // Determine if all CME events are enabled. If not there is
        // nothing for us to do.
        if(psObj->bEnabled == FALSE)
        {
            // No harm done, just nothing to do
            break;
        }


        // Verify entry pointer is valid
        if(SMSO_bIsValid((SMS_OBJECT)psEntry) == FALSE)
        {
            // Error! Cannot process null entry. Exit loop.
            break;
        }

        // Ok, now we have found a potential match and it is valid.
        // Now it is time to decide what to do.

        // Check if this content is enabled, if not we just continue our
        // search for the next content (if one exists).
        if(
              ((psEntry->un32StateFlags & CONTENT_STATE_DISABLED)
                        == CONTENT_STATE_DISABLED)
          )
        {
            // Content events for this entry has been disabled.
            // Search for another one.
            break;
        }

        // Content is new and thus it has started, check options
        // and determine if we need to trigger an event. Check if
        // registered to get "INITIAL" events.
        if((psEntry->sContent.un32Options &
            SMS_EVENT_REGISTRATION_OPTION_INITIAL)
                == SMS_EVENT_REGISTRATION_OPTION_INITIAL)
        {
            // Add this event to the pending events list indicating it
            // has begun.
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psObj->hPendingEventList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                psEntry
                    );
            if(eOsalReturnCode == OSAL_SUCCESS)
            {
                // Indicate event has been added to the pending
                // list and the state. Note if the END
                // reason was already set we clear it now because it
                // is not right to report something has ended when
                // we already know it started.
                psEntry->un32StateFlags &=
                    ~(CONTENT_STATE_END); // Clear state
                psEntry->un32StateFlags |=
                    CONTENT_STATE_INITIAL; // Set initial state

                // Record channel which has this content
                psEntry->hChannel = hChannel;

                // Now, since the INITIAL event has occurred we need to
                // see if this content is registered as a one-shot AND
                // it is set up ONLY for the initial event. If so, it
                // must be removed from the master content list.
                // If the options indicate...
                if( (psEntry->sContent.un32Options &
                    (
                        SMS_EVENT_REGISTRATION_OPTION_ONETIME |
                        SMS_EVENT_REGISTRATION_OPTION_INITIAL |
                        SMS_EVENT_REGISTRATION_OPTION_END
                     ) )
                     ==
                     (
                         // One-shot AND
                         SMS_EVENT_REGISTRATION_OPTION_ONETIME |
                         // Initial AND
                         SMS_EVENT_REGISTRATION_OPTION_INITIAL
                         // not END
                      )
                   )
                {
                    // Remove this entry from the master content list
                    *pbContentRemoved = bRemoveContentEntry(
                        (CME_OBJECT_STRUCT *)hCME, psEntry);
                }

            }
        }

    } while(0);

    return;
}

/*****************************************************************************
*
*   vGetHandles
*
* This helper function simply extracts various handles from a provided CDO.
*
* Inputs:
*
*   hCDO        A handle to a valid CDO object for which to extract the CME
*               which can monitor it.
*   phCME       A pointer to a CME handle to return the CME handle to the
*                caller.
*   phChannel   A pointer to a CHANNEL handle to return the CHANNEL handle
*                to the caller.
*
* Outputs:
*
*   BOOLEAN     TRUE on success or FALSE on failure.
*
*****************************************************************************/
static void vGetHandles (
    CD_OBJECT hCDO,
    CME_OBJECT *phCME,
    CHANNEL_OBJECT *phChannel
        )
{
    CCACHE_OBJECT hCCache = CCACHE_INVALID_OBJECT;
    CCACHE_OBJECT *phCCache = &hCCache;

    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    DECODER_OBJECT *phDecoder = &hDecoder;

    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    CME_OBJECT hCME = CME_INVALID_OBJECT;

    // Determine which handles to use based on inputs
    if(phCME == NULL)
    {
        // Use ours
        phCME = &hCME;
    }

    if(phChannel == NULL)
    {
        // Use ours
        phChannel = &hChannel;
    }

    // All CDO's belong to a CHANNEL object (their parent). So here
    // we extract the CHANNEL object this CDO belongs to.
    *phChannel = (CHANNEL_OBJECT)SMSO_hParent((SMS_OBJECT)hCDO);
    if(*phChannel == CHANNEL_INVALID_OBJECT)
    {
        // We can't go any further
        return;
    }

    // All CHANNEL objects belong to a CCACHE object (their parent). So
    // here we extract the CCACHE handle.
    *phCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)*phChannel);
    if(*phCCache == CCACHE_INVALID_OBJECT)
    {
        // We can't go any further
        return;
    }

    // All CCACHE objects belong to a DECODER object (their parent). So
    // here we extract the DECODER handle.
    *phDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)*phCCache);
    if(*phDecoder == DECODER_INVALID_OBJECT)
    {
        // We can't go any further
        return;
    }

    // Extract the CME handle associated with this DECODER object. Now,
    // finally we have the CME handle.
    *phCME = DECODER_hCME(*phDecoder);
    if(*phCME == CME_INVALID_OBJECT)
    {
        // We can't go any further
        return;
    }

    return;
}

/*****************************************************************************
*
*   eVerifyOptions
*
* This local function simply verifies the input options are correct, and
* makes adjustments (where it should) if they are not.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eVerifyOptions (
    UN32 *pun32Options
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_OPTIONS;

    // Check if options are valid first and that there are no other unknown
    // options selected.
    if( (*pun32Options & ~SMS_EVENT_REGISTRATION_OPTION_ALL) ==
            SMS_EVENT_REGISTRATION_OPTION_NONE)
    {
        // If the 'CURRENT' option is selected, then an 'INITIAL' option
        // must also be selected. We select it for the caller, regardless if
        // they did or not.
        if( (*pun32Options & SMS_EVENT_REGISTRATION_OPTION_CURRENT) ==
            SMS_EVENT_REGISTRATION_OPTION_CURRENT)
        {
            // Always apply 'INITIAL'
            *pun32Options |= SMS_EVENT_REGISTRATION_OPTION_INITIAL;
        }

        // Make sure they have provided either RETRIGGER or ONETIME
        // as a trigger option, but not both or neither. One or the other
        // must be provided.
        if(
            ((*pun32Options & (SMS_EVENT_REGISTRATION_OPTION_RETRIGGER |
                        SMS_EVENT_REGISTRATION_OPTION_ONETIME)) != 0)
                        && // -- AND --
            ((*pun32Options & (SMS_EVENT_REGISTRATION_OPTION_RETRIGGER |
                        SMS_EVENT_REGISTRATION_OPTION_ONETIME)) !=
                            (SMS_EVENT_REGISTRATION_OPTION_RETRIGGER |
                             SMS_EVENT_REGISTRATION_OPTION_ONETIME))
          )
        {
            // Ok! It all looks good.
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bChannelIterator
*
*   This is a simple channel cache iterator function. It just simply
*   allows the channel cache iterator to apply a caller defined callback with
*   associated data, in this case a CID. It is called for each channel
*   in the cache.
*
*****************************************************************************/
static BOOLEAN bChannelIterator (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    BOOLEAN bFound;
    CD_OBJECT hCDO;
    CME_CACHE_ITERATOR_STRUCT *psIterator =
        (CME_CACHE_ITERATOR_STRUCT *)pvArg;

    // Extract this channel's CDO first.
    hCDO = CHANNEL.hCDO(hChannel);

    // Now, check to see if this channel's CDO contains this CID.
    // This call will check the entire CDO and it's sub-type of any
    // CID which matches it.
    bFound = CDO_bHasId(hCDO, psIterator->hId);
    if(bFound == TRUE)
    {
        BOOLEAN bRemoved;
        // Yep! We found one.

        // Run this particular entry through the CME. We only want to generate
        // events for this specific entry not others which may exist.
        vUpdateSpecificContent(psIterator->hCME, hChannel, psIterator->psEntry, &bRemoved);

        // Trigger any CME's we set in the call above.
        CME_vTriggerEvents(psIterator->hCME, FALSE);
        if (bRemoved == TRUE)
        {
            psIterator->psEntry = NULL;
        }
    }

    return TRUE; // keep iterating
}

/*****************************************************************************
*
*   n16Sort
*
*       This is the function used as a sorting funtion by comparing the
*       priorities of 2 events
*
*       Outputs:
*             0 - psEvent1's priority is equal to psEvent's priority
*           > 0 - psEvent1's priority is less than psEvent's priority
*           < 0 - psEvent1's priority is greater than psEvent's priority
*
*****************************************************************************/
static N16 n16Sort (
    void *psEvent1,
    void *psEvent2
        )
{
    CME_ENTRY_STRUCT *psEntry1 = (CME_ENTRY_STRUCT *)psEvent1;
    CME_ENTRY_STRUCT *psEntry2 = (CME_ENTRY_STRUCT *)psEvent2;
    N16 n16MapEvent1, n16MapEvent2;

    // Verify inputs
    if( (psEntry1 == NULL) || (psEntry2 == NULL) )
    {
        // Error
        return N16_MIN;
    }

    n16MapEvent1 = n16MapEvent(psEntry1->un32StateFlags);
    n16MapEvent2 = n16MapEvent(psEntry2->un32StateFlags);

    return n16MapEvent2-n16MapEvent1;
}

/*****************************************************************************
*
*   n16MapEvent
*
*       This is the function used to map an event to a number that can be
*       used to sort.
*
*       Outputs:
*               N16 - the mapped priority of the event. A higher number
*                     indicates a higher priority
*
*****************************************************************************/
static N16 n16MapEvent(UN32 un32StateFlags)
{
    N16 n16MapEvent = CONTENT_STATE_INITIAL_PRIORITY;

    if ( (un32StateFlags & CONTENT_STATE_FINAL) ==
          CONTENT_STATE_FINAL )
    {
        n16MapEvent = CONTENT_STATE_FINAL_PRIORITY;
    }
    if ( (un32StateFlags & CONTENT_STATE_END) ==
          CONTENT_STATE_END )
    {
        n16MapEvent = CONTENT_STATE_END_PRIORITY;
    }
    else if ( (un32StateFlags & CONTENT_STATE_INITIAL) ==
               CONTENT_STATE_INITIAL )
    {
        n16MapEvent = CONTENT_STATE_INITIAL_PRIORITY;
    }

    return n16MapEvent;
}

/*****************************************************************************
*
*   vPostUpdateEvent
*
*****************************************************************************/
void vPostUpdateEvent(
    OSAL_OBJECT_HDL hTimer, 
    void *pvArg
        )
{
    BOOLEAN bPosted;
    CME_OBJECT_STRUCT *psObj = (CME_OBJECT_STRUCT *)pvArg;

    // Timer fired. Set to inactive.
    psObj->bUpdateTimerActive = FALSE;

    bPosted = SMSE_bPostEvent(psObj->hUpdateEvent);
    if (bPosted == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CME_OBJECT_NAME": Failed to post CME update event");
    }

    return;
}
