/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the CID STRING implementation for the
 *  Sirius Module Services (SMS). This module is intended to serve as a
 *  generic CID Object Data implementation for CIDs which use STRINGs
 *  as their basic object data type. Currently Artist, Title and Composer
 *  Text CIDs as well as Low-Band PID and AIDs use STRINGs as their
 *  object type. These methods are made public (as friend functions) to
 *  any CID type implementation which needs to use them.
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"

// Include CID STRING module headers
#include "cid_string.h"
#include "_cid_string.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CIDSTR_pvCreate
*
*   Creates a CID object data object(STRING) from object data which is a null
*   terminated C-string character array.
*
*****************************************************************************/
const void *CIDSTR_pvCreate (
    const void *pvSrcObjectData,
    size_t tMinimumSize,
    SMS_OBJECT hParent,
    BOOLEAN bConstant
        )
{
    size_t tLength;
    // Object data for this type is a STRING object
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    // Source object data (raw) is a null terminate character string
    const char *pacString = (const char*)pvSrcObjectData;

    // Determine input string length
    tLength = strlen(pacString);

    // Only create a STRING if...
    // The provided string is of some size (> 0)
    if(tLength > 0)
    {
        // Choose the STRING object creation method desired. If the
        // flag is TRUE the STRING object must be constant
        // and thus cannot be modified.
        if(bConstant == TRUE)
        {
            // Create a constant STRING object based on the provided
            // null-terminated c-string.
            hString =
                STRING_hCreateConst(pacString, tLength);
        }
        else
        {
            // Create a modifiable STRING object from the provided
            // null-terminated c-string but reserve at least tMinimum Size
            // for it.
            hString =
                STRING_hCreate(
                    hParent,
                    pacString,
                    tLength,
                    tMinimumSize
                        );
        }
    }

    return (const void *)hString;
}

/*****************************************************************************
*
*   CIDSTR_vDestroy
*
*   Destroys a CID object data object(STRING)
*
*****************************************************************************/
void CIDSTR_vDestroy (
    const void *pvObject
        )
{
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;
    STRING_vDestroy(hString);
    return;
}

/*****************************************************************************
*
*   CIDSTR_pvDuplicate
*
*   Duplicates a CID object data object (STRING)
*
*****************************************************************************/
const void *CIDSTR_pvDuplicate (
    const void *pvObject
        )
{
    // These are STRING's
    return (const void *)STRING.hDuplicate((STRING_OBJECT)pvObject);
}

/*****************************************************************************
*
*   CIDSTR_bModify
*
*   Modifies a CID object data object (STRING)
*
*****************************************************************************/
BOOLEAN CIDSTR_bModify (
    const void **ppvObject,
    const void *pvSrcObjectData
        )
{
    // These are STRING objects
    STRING_OBJECT *phString = (STRING_OBJECT *)ppvObject;
    const char *pacString = (const char *)pvSrcObjectData;

    // Use the STRING object to modify it.
    return STRING.bModifyCStr(*phString, pacString);
}

/*****************************************************************************
*
*   CIDSTR_bCopy
*
*   Copies a CID object data object (STRING) into another
*
*****************************************************************************/
BOOLEAN CIDSTR_bCopy (
    void **ppvDstObject,
    const void *pvSrcObject
        )
{
    // These are STRING objects
    STRING_OBJECT *phDstString = (STRING_OBJECT *)ppvDstObject;
    STRING_OBJECT hSrcString = (STRING_OBJECT)pvSrcObject;
    size_t tSize;
    BOOLEAN bRetval = FALSE;

    tSize = STRING.tCopy(hSrcString, *phDstString);
    if(tSize == STRING.tSize(hSrcString))
    {
        // String copied as far as I can tell.
        bRetval = TRUE;
    }

    return bRetval;
}

/*****************************************************************************
*
*   CIDSTR_tSize
*
*   Retrieves the size a CID object data object (STRING) in bytes
*
*****************************************************************************/
size_t CIDSTR_tSize(
    const void *pvObject
        )
{
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;

    // STRING size and add one for NULL terminator
    return (STRING.tSize(hString) + 1);
}

/*****************************************************************************
*
*   CIDSTR_n16Compare
*
*   Compares a CID object data object (STRING) to another CID object data
*   object.
*
*****************************************************************************/
N16 CIDSTR_n16Compare (
    const void *pvObject1,
    const void *pvObject2,
    BOOLEAN bBinary
        )
{
    // These are STRING objects
    STRING_OBJECT hString1 = (STRING_OBJECT)pvObject1;
    STRING_OBJECT hString2 = (STRING_OBJECT)pvObject2;

    // Use the STRING object to compare
    return STRING.n16Compare(hString1, hString2, bBinary);
}

/*****************************************************************************
*
*   CIDSTR_n32FWrite
*
*   Writes a CID object data object (STRING) to a device.
*
*****************************************************************************/
N32 CIDSTR_n32FWrite (
    const void *pvObject,
    FILE *psFile
        )
{
    N32 n32Written = 0, n32RetVal;
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;

    do
    {
        // Write the string
        n32RetVal = STRING.n32FWrite((STRING_OBJECT)hString, psFile);
        if(n32RetVal == 0 && STRING.tSize(hString) > 0)
        {
            // Error!
            n32Written = EOF;
            break;
        }

        // Accumulate # of bytes written
        n32Written += n32RetVal;

        // Add null terminator to identify the end of the string
        n32RetVal = (N32)fwrite("\0", 1, 1, psFile);
        if(n32RetVal < 0)
        {
            // Error!
            n32Written = EOF;
            break;
        }

        // Accumulate # of bytes written
        n32Written += n32RetVal;

    } while(FALSE);

    return n32Written;
}

/*****************************************************************************
*
*   CIDSTR_n32FWriteToMemory
*
*   Writes a CID object data object (STRING) to memory.
*
*****************************************************************************/
N32 CIDSTR_n32FWriteToMemory (
    const void *pvObject,
    void **ppvMemory
        )
{
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;
    size_t tSize;
    N32 n32Return;
    UN8 *pun8Memory = (UN8 *)*ppvMemory; // Byte-wise addressable memory

    // obtain size of the string and add one for null terminator
    tSize = STRING.tSize(hString) + 1;

    // These CIDs are all STRING objects
    n32Return = (N32)STRING.tCopyToCStr(hString, (char *)pun8Memory, tSize);
    if(n32Return > 0)
    {
        // update the memory
        pun8Memory += (UN32)n32Return;
    }

    // Compute size written
    n32Return = pun8Memory - (UN8 *)*ppvMemory;
    *ppvMemory = pun8Memory;

    return n32Return;
}

/*****************************************************************************
*
*   CIDSTR_n32GetValue
*
*****************************************************************************/
N32 CIDSTR_n32GetValue (
    const void *pvObject,
    void **ppvValue
        )
{
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;
    size_t tSize;
    N32 n32Return;
    UN8 *pun8Memory = (UN8 *)*ppvValue; // Byte-wise addressable memory

    // obtain size of the string and add one for null terminator
    tSize = STRING.tSize(hString) + 1;

    // These CIDs are all STRING objects
    n32Return = (N32)STRING.tCopyToCStr(hString, (char *)pun8Memory, tSize);

    return n32Return;
}

/*****************************************************************************
*
*   CIDSTR_pvRead
*
*   Reads a CID object data object (STRING) from a file
*
*****************************************************************************/
const void *CIDSTR_pvRead (
    FILE *psFile
        )
{
    // Return STRING handle as the data object
    return (const void *)STRING_hFRead(SMS_INVALID_OBJECT, psFile);
}

/*****************************************************************************
*
*   CIDSTR_n32FPrintf
*
*   Prints a CID object data object (STRING) to a file
*
*****************************************************************************/
N32 CIDSTR_n32FPrintf (
    const void *pvObject,
    FILE *psFile
        )
{
    // These are STRING objects
    STRING_OBJECT hString = (STRING_OBJECT)pvObject;

    // Print the string
    return STRING.n32FPrintf(hString, psFile);
}

/*****************************************************************************
*
*   CIDSTR_bReadFromMemory
*
*****************************************************************************/
BOOLEAN CIDSTR_bReadFromMemory(
    const void **ppvObjectData,
    void **ppvMemory,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bOk = FALSE;

    if ((ppvObjectData != NULL) && (ppvMemory != NULL))
    {
        // need to create new object data.
        if(*ppvObjectData != NULL)
        {
            bOk = CIDSTR_bModify(ppvObjectData, *ppvMemory);
        }
        else
        {
            *ppvObjectData = CIDSTR_pvCreate(
                                 *ppvMemory,
                                 GsCidStringIntf.tMinSize,
                                 hParent,
                                 FALSE);
            if (*ppvObjectData != NULL)
            {
                bOk = TRUE;
            }
        }
    }
    return bOk;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
