/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CID implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "song_tag_id.h"
#include "league_team_id.h"

#include "cid_obj.h"
#include "_cid_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroy
*
* This method is used by anyone who used CID.hDuplicate() or CID.hFRead()
* to create a new CID. If the CID was not created this way it will not
* be destroyed. Typically only external (Applications) will need to make a
* call to this API. SMS calls which use CID_hCreatex() calls should instead
* use the corresponding CID_vDestroy().
*
* Inputs:
*   hCID - A handle to a valid CID obtained by CID.hDuplicate() or CID.hFread()
*
* Outputs:
*   None.
*
*****************************************************************************/
static void vDestroy (
    CID_OBJECT hCID
        )
{
    BOOLEAN bOwner;

    // Validate we own this CID
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == TRUE)
    {
        CID_OBJECT_STRUCT *psObj = (CID_OBJECT_STRUCT *)hCID;

        // Next check the creator, if it is not application it cannot be
        // destroyed this way.
        if((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_APPLICATION)
        {
            // Destroy the CID since it cannot be recycled.
            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   n16Equal
*
* This interface method compares the equality of two CIDs. This performs a
* simple binary compare (go, no-go) and does not indicate relationship
* (such as lt, gt, or eq).
*
* Outputs:
*   0   - CIDs have the same value (equal)
*   < 0 - CIDs are not equal (or error)
*
*****************************************************************************/
static N16 n16Equal (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2
        )
{
    BOOLEAN bOwner;
    N16 n16Return = N16_MIN;

    // Validate we own these CIDs
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID1);
    if(bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hCID2);
    }

    if(bOwner == TRUE)
    {
        // Compare using binary compare (TRUE)
        n16Return = n16CompareObjects(hCID1, hCID2, TRUE);

        if (n16Return != 0)
        {
            BOOLEAN bEquivMatch;

            // Now check the equivalent cid list
            bEquivMatch = bMatchWithEquivalentCids(
                (CID_OBJECT_STRUCT *)hCID1,
                (CID_OBJECT_STRUCT *)hCID2);
            if (bEquivMatch == TRUE)
            {
                // One of the equivalent cids matched
                n16Return = 0;
            }
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16Compare
*
* This interface method compares the relationship of two CIDs. This performs a
* relationship (lt, gt, eq) compare typically used for sorting.
*
* Outputs:
*   0   - CIDs have the same value (equal)
*   > 0 - hCID1 is greater than (after) hCID2
*   < 0 - hCID1 is less than (before) hCID2 (or error)
*
*****************************************************************************/
static N16 n16Compare (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2
        )
{
    BOOLEAN bOwner;
    N16 n16Return = N16_MIN;

    // Validate we own these CIDs
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID1);
    if(bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hCID2);
    }

    if(bOwner == TRUE)
    {
        // Compare using relationship compare (FALSE)
        n16Return = n16CompareObjects(hCID1, hCID2, FALSE);

        if (n16Return != 0)
        {
            BOOLEAN bEquivMatch;

            // Now check the equivalent cid list
            bEquivMatch = bMatchWithEquivalentCids(
                (CID_OBJECT_STRUCT *)hCID1,
                (CID_OBJECT_STRUCT *)hCID2);
            if (bEquivMatch == TRUE)
            {
                // One of the equivalent cids matched
                n16Return = 0;
            }
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   bMatchWithEquivalentCids
*
* This method compares the relationship of the equivalent cids in the two
* provided CIDs. This performs a relationship (just eq) compare used to
* aid in direct matching of CID objects.
*
*****************************************************************************/
static BOOLEAN bMatchWithEquivalentCids (
    CID_OBJECT_STRUCT *psObj1,
    CID_OBJECT_STRUCT *psObj2
        )
{
    CID_EQUIVALENT_CID_STRUCT sFirstCid1, sFirstCid2, *psEquiv1, *psEquiv2;
    N16 n16Return;

    // Get out quick if there are no equivalent lists to compare
    if ((psObj1->psEquivalent == (CID_EQUIVALENT_CID_STRUCT *)NULL) &&
        (psObj2->psEquivalent == (CID_EQUIVALENT_CID_STRUCT *)NULL))
    {
        return FALSE;
    }

    // Populate this "entry" with info for the first CID
    sFirstCid1.bUseCid = TRUE; // This cid is always in use
    sFirstCid1.hCid = (CID_OBJECT)psObj1; // This is the cid we're comparing
    sFirstCid1.psNext = psObj1->psEquivalent; // the next entry is the actual equivalent list head

    // Populate this "entry" with info for the first CID
    sFirstCid2.bUseCid = TRUE; // This cid is always in use
    sFirstCid2.hCid = (CID_OBJECT)psObj2; // This is the cid we're comparing
    sFirstCid2.psNext = psObj2->psEquivalent; // the next entry is the actual equivalent list head

    // The "equivalent list" we are using here is going
    // to start with the actual cid we have here
    psEquiv1 = &sFirstCid1;

    while (psEquiv1 != (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        // Stop at the first cid we can't use --
        // all the following cids are also
        // unusable
        if (psEquiv1->bUseCid == FALSE)
        {
            break;
        }

        // The "equivalent list" we are using here is going
        // to start with the actual cid we have here
        psEquiv2 = &sFirstCid2;

        while (psEquiv2 != (CID_EQUIVALENT_CID_STRUCT *)NULL)
        {
            // Stop at the first cid we can't use --
            // all the following cids are also
            // unusable
            if (psEquiv2->bUseCid == FALSE)
            {
                break;
            }

            // Compare the two current cids (yes, no)
            n16Return = n16CompareObjects(psEquiv1->hCid, psEquiv2->hCid, TRUE);
            if (n16Return == 0)
            {
                // This matches, so stop comparing now
                return TRUE;
            }

            // Get the next cid from this list
            psEquiv2 = psEquiv2->psNext;
        }

        // Get the next cid from this list
        psEquiv1 = psEquiv1->psNext;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bDuplicateEquivalentCids
*
* This method is used in conjunction with CID.hDuplicate to copy the
* equivalent cid list from the source object into the destination object
*
*****************************************************************************/
static BOOLEAN bDuplicateEquivalentCids (
    CID_OBJECT_STRUCT *psSrcObj,
    CID_OBJECT_STRUCT *psDstObj
        )
{
    CID_EQUIVALENT_CID_STRUCT *psSrcEquiv =
        psSrcObj->psEquivalent;
    BOOLEAN bSuccess = TRUE;

    while (psSrcEquiv != (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        if (psSrcEquiv->bUseCid == TRUE)
        {
            // Add this CID as an equivalent in the
            // destination object
            bSuccess = CID.bSetEquivalent(
                (CID_OBJECT)psDstObj,psSrcEquiv->hCid);
        }

        if (bSuccess == FALSE)
        {
            // Stop since we had an erorr
            break;
        }

        // Move to the next one
        psSrcEquiv = psSrcEquiv->psNext;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   n32FWriteEquivalentCids
*
* This method is used in conjunction with CID.n32FWrite to write the
* equivalent cid list from the source object into the provided file
*
*****************************************************************************/
static N32 n32FWriteEquivalentCids (
    CID_OBJECT_STRUCT *psObj,
    FILE *psFile
        )
{
    UN16 un16NumEquivalentCids = 0, un16Index;
    CID_EQUIVALENT_CID_STRUCT *psEquivalent = psObj->psEquivalent;
    N32 n32Written, n32Return;

    // Count how many we have to write
    while (psEquivalent != (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        if (psEquivalent->bUseCid == FALSE)
        {
            // Stop here since any following cids will
            // unusable as well
            break;
        }

        un16NumEquivalentCids++;
        psEquivalent = psEquivalent->psNext;
    }

    // Write that number (as readable characters)
    n32Written = fprintf(psFile, "%u", un16NumEquivalentCids);

    // Add null terminator to identify the end of the string
    n32Written += (N32)fwrite("\0", 1, 1, psFile);

    // Then, write the cids
    psEquivalent = psObj->psEquivalent;
    for (un16Index = 0; un16Index < un16NumEquivalentCids; un16Index++)
    {
        n32Return = CID.n32FWrite(psEquivalent->hCid, psFile);

        // Only update the number of bytes written if necessary
        if (n32Return > 0)
        {
            n32Written += n32Return;
        }

        psEquivalent = psEquivalent->psNext;
    }

    return n32Written;
}

/*****************************************************************************
*
*   bFReadEquivalentCids
*
* This method is used in conjunction with CID.hFRead to read the
* equivalent cid list from the provided file into the source object
*
*****************************************************************************/
static BOOLEAN bFReadEquivalentCids (
    CID_OBJECT_STRUCT *psObj,
    FILE *psFile
        )
{
    CID_OBJECT hEquivalentCid;
    UN16 un16NumEquivalentCids, un16Index = 0;
    size_t tNumObjsRead;
    BOOLEAN bSuccess = TRUE;
    char acBuffer[8]; // Just a decent sized buffer to incrementally
                       // read in a string.

    // Place a null terminator in the last element of our buffer and
    // leave it there for processing purposes later.
    acBuffer[sizeof(acBuffer) - 1] = '\0';

    // Read a string from the file provided byte by byte until a '\0' is found.
    // A '\0' is used to determine the end of a integer entity.
    do
    {
        // Read a character from the file
        tNumObjsRead = fread(&acBuffer[un16Index], sizeof(acBuffer[0]), 1, psFile);
        if(tNumObjsRead != 1)
        {
            // Error! File error occurred.
            bSuccess = FALSE;
            break;
        }

        if((acBuffer[un16Index] == '\0') ||
            (un16Index == sizeof(acBuffer) - 2))
        {
            break;
        }

        un16Index++;

    } while(TRUE);

    if (bSuccess == FALSE)
    {
        return FALSE;
    }

    // Convert that string
    un16NumEquivalentCids = (size_t)strtoul(&acBuffer[0], NULL, 10);

    // Read all the equivalent cids
    for (un16Index = 0; un16Index < un16NumEquivalentCids; un16Index++)
    {
        // Read the next cid
        hEquivalentCid = CID.hFRead(psFile);

        if (hEquivalentCid != CID_INVALID_OBJECT)
        {
            // Use this CID as an equivalent
            bSuccess = CID.bSetEquivalent((CID_OBJECT)psObj, hEquivalentCid);

            // Don't need this anymore
            CID.vDestroy(hEquivalentCid);
        }
        else
        {
            // That should have worked
            bSuccess = FALSE;
        }

        if (bSuccess == FALSE)
        {
            break;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   hDuplicate
*
* This object interface method is used to duplicate an existing CID.
* The source CID may be one created by SMS or an Application. Either way
* a deep-copy is performed and thus a duplicate of the original is made.
* This new CID may be used by the application independent of SMS however
* it sees fit. Once the caller is done with the CID however it must
* release it by calling CID.vDestroy() at some point.
*
* Duplicated CIDs are not part of the CID pool and thus are never recycled.
*
* Inputs:
*
*   hCID - The CID the caller wishes to duplicate.
*
* Outputs:
*
*   A new CID_OBJECT on success. Otherwise CID_INVALID_OBJECT. This new CID
*   must be destroyed using CID.vDestroy() at some point when the caller is
*   finished with it.
*
*****************************************************************************/
static CID_OBJECT hDuplicate (
    CID_OBJECT hCID
        )
{
    CID_OBJECT hNewCID = CID_INVALID_OBJECT;
    CID_OBJECT_STRUCT *psObj = (CID_OBJECT_STRUCT *)hCID;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Use interface function if one exists
    if(psObj->psIntf->sObjData.pvDuplicate != NULL)
    {
        const void *pvObjectData;

        // Based on the type, duplicate the object data first
        pvObjectData = psObj->psIntf->sObjData.pvDuplicate(psObj->pvObjectData);
        // Check if CID object data was created, if so continue. Otherwise
        // we were unsuccessful.
        if(pvObjectData != NULL)
        {
            CID_OBJECT_STRUCT *psNewObj;

            // Create an instance of the CID object without a parent.
            psNewObj = psCreateObject(NULL, psObj->psIntf->eType, NULL);
            if(psNewObj != NULL)
            {
                BOOLEAN bEquivDuplicated;

                // Initialize object per inputs
                psNewObj->pvObjectData = pvObjectData;
                psNewObj->un8Flags &= ~(CID_CREATOR);
                psNewObj->un8Flags |= CID_CREATOR_APPLICATION;

                // Duplicate all equivalent cids as well
                bEquivDuplicated =
                    bDuplicateEquivalentCids(psObj, psNewObj);
                if (bEquivDuplicated == TRUE)
                {
                    // Initialize handle
                    hNewCID = (CID_OBJECT)psNewObj;
                }
                else
                {
                    // Error!
                    vDestroyObject(psObj);
                }
            }
            else
            {
                // Error!
                // Use interface function if one exists
                if(psObj->psIntf->sObjData.vDestroy != NULL)
                {
                    // Free object data based on type
                    psObj->psIntf->sObjData.vDestroy(pvObjectData);
                }
            }
        }
    }

    return hNewCID;
}

/*****************************************************************************
*
*   bSetEquivalent
*
* This object interface method is used to equate a CID (given by hCID)
* with another CID (hEquivalentCID).  This allows a single CID to contain
* a set of CIDs to which it will match in calls to .n16Compare and
* .n16Equal.  Other than this behavior, this CID behaves in the same
* manner as other CIDS and all equivalent CIDs will be freed when
* this CID is destroyed via CID.vDestroy
*
* Either CID may be one created by SMS or an Application. Either way
* a deep-copy is performed on the equivalent CID and thus a duplicate
* of that object is made and stored in hCID.
*
* Duplicated CIDs are not part of the CID pool and thus are never recycled.
*
* Inputs:
*
*   hCID - The CID the caller wishes to enhance with equivalent CIDs.
*   hEquivalentCID - The CID which this CID should match in comparison APIs
*
* Outputs:
*
*   TRUE on success. Otherwise FALSE.
*
*****************************************************************************/
static BOOLEAN bSetEquivalent (
    CID_OBJECT hCID,
    CID_OBJECT hEquivalentCID
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hEquivalentCID);
    }

    if(bOwner == TRUE)
    {
        CID_OBJECT_STRUCT *psObj = (CID_OBJECT_STRUCT *)hCID;
        CID_EQUIVALENT_CID_STRUCT **ppsEquiv = &psObj->psEquivalent;

        while (*ppsEquiv != (CID_EQUIVALENT_CID_STRUCT *)NULL)
        {
            // Stop when we find an entry with a cid
            // that isn't in use
            if ((*ppsEquiv)->bUseCid == FALSE)
            {
                break;
            }

            // Move to the next block
            ppsEquiv = &((*ppsEquiv)->psNext);
        }

        // Do we need to allocate a new block?
        if (*ppsEquiv == (CID_EQUIVALENT_CID_STRUCT *)NULL)
        {
            SMS_OBJECT hParent;

            // Grab this object's parent
            hParent = SMSO_hParent((SMS_OBJECT)hCID);

            // Create this block
            *ppsEquiv = (CID_EQUIVALENT_CID_STRUCT *)SMSO_hCreate(
                CID_OBJECT_NAME":equiv",
                sizeof(CID_EQUIVALENT_CID_STRUCT), hParent, FALSE);

            if ((*ppsEquiv ) != (CID_EQUIVALENT_CID_STRUCT *)NULL)
            {
                (*ppsEquiv)->bUseCid = FALSE;
                (*ppsEquiv)->hCid = CID_INVALID_OBJECT;
                (*ppsEquiv)->psNext = (CID_EQUIVALENT_CID_STRUCT *)NULL;
            }
        }

        // Do we have a block that we can use now?
        if (*ppsEquiv != (CID_EQUIVALENT_CID_STRUCT *)NULL)
        {
            // Do we need to create a new cid or just modify one?
            if ((*ppsEquiv)->hCid == CID_INVALID_OBJECT)
            {
                // Create a new one
                (*ppsEquiv)->hCid = CID.hDuplicate(hEquivalentCID);

                if ((*ppsEquiv)->hCid != CID_INVALID_OBJECT)
                {
                    bSuccess = TRUE;
                }
            }
            else
            {
                CID_OBJECT_STRUCT *psEquivalentObj =
                    (CID_OBJECT_STRUCT *)hEquivalentCID;

                // We need to modify an existing cid
                bSuccess = CID_bModify(
                    &(*ppsEquiv)->hCid,
                    psEquivalentObj->pvObjectData);
            }

            // We can use this cid if that worked
            (*ppsEquiv)->bUseCid = bSuccess;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   n32FWrite
*
* This object interface method is used to write the contents of a CID
* (performing a deep write) to a device specified by psFile. The intent of
* this method is to effectively store the contents of a CID for later
* retrieval (perhaps after a power cycle). The entire CID contents are written
* to the device, with enough information to recreate the exact CID
* when the hFRead() method is called.
*
* Inputs:
*
*   hCID - The CID handle the caller wishes to write.
*   psFile - The device to write the CID contents to.
*
* Outputs:
*
*   The number of bytes written or EOF on error.
*
*****************************************************************************/
static N32 n32FWrite (
    CID_OBJECT hCID,
    FILE *psFile
        )
{
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    N32 n32Return, n32Written;
    BOOLEAN bOwner;
    const char *pacTypeString;
    size_t tLen;

    // Verify inputs, must be owned and file valid
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if((bOwner == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Write CID type string first
    pacTypeString = pacTypeText(psObj->psIntf->eType, &tLen);
    n32Return = (N32)fwrite(pacTypeString, tLen, 1, psFile);

    // Then we can write the cid value to a file...
    if(n32Return == 1)
    {
        // fwrite returns number of elements, not bytes
        n32Return = tLen;
        tLen = strlen(CID_TYPE_VALUE_DELIMITER_STRING);
        n32Written = (N32)fwrite(
                             CID_TYPE_VALUE_DELIMITER_STRING,
                             tLen,
                             1,
                             psFile
                                 ) ;
        if(n32Written != 1)
        {
            return EOF;
        }

        n32Return += tLen;

        // Use interface function if one exists
        if(psObj->psIntf->sObjData.n32FWrite != NULL)
        {
            N32 n32RetVal;

            // Write contents based on type
            n32RetVal = psObj->psIntf->sObjData.n32FWrite(
                psObj->pvObjectData, psFile);

            if(n32RetVal != EOF)
            {
                // Accumulate what was written
                n32Return += n32RetVal;
            }

            // Now, write the equivalent cid list
            n32RetVal = n32FWriteEquivalentCids(psObj, psFile);

            if (n32RetVal != EOF)
            {
                // Accumulate what was written
                n32Return += n32RetVal;
            }
            else
            {
                // Error!
                return EOF;
            }
        }
    }

    return n32Return;
}

/*****************************************************************************
*
*   hFRead
*
* This object interface method is used to read from a specified device psFile
* and generate a CID from that information. The data read from the device
* must have been previously written by the n32FWrite method. Upon
* successful execution of this method a new CID is created (created by the
* caller) which may be used to register for events or presented to the
* UI for display, etc. This method allows the caller to effectively read
* a previously stored CID regenerating the original CID written(saved) at
* an earlier time.
*
* CIDs which are created using this method do not belong to the CID pool.
*
* Inputs:
*
*   psFile - The device to read the CID contents from.
*
* Outputs:
*
*   A new CID on success, otherwise CID_INVALID_OBJECT on failure.
*
*****************************************************************************/
static CID_OBJECT hFRead (
    FILE *psFile
        )
{
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    CID_ENUM eType = CID_INVALID;
    UN8 un8Index = 0;
    size_t tLen, tLongestCIDypeTextLen = 0;
    CID_OBJECT_STRUCT *psObj;
    BOOLEAN bOk = FALSE;
    BOOLEAN bLocked;
    char *pacBuff;

    // Verify input
    if(psFile == NULL)
    {
        return CID_INVALID_OBJECT;
    }

    // Read CID from file...

    // Read type first
    tLongestCIDypeTextLen = sizeof(CID_TYPE_TEXT);

    pacBuff = (char *) SMSO_hCreate(CID_OBJECT_NAME":RB",
                    tLongestCIDypeTextLen,
                    SMS_INVALID_OBJECT, FALSE // No lock necessary
                    );

    if (pacBuff == NULL)
    {
        return CID_INVALID_OBJECT;
    }

    do
    {
        tLen = fread(&pacBuff[un8Index], sizeof(char), 1, psFile);
        if(tLen != 1)
        {
            // End of file or error occurred
            break;
        }
        else
        {
            if (pacBuff[un8Index] == CID_TYPE_VALUE_DELIMITER_STRING[0])
            {
                // found the delimiter
                bOk = TRUE;
                break;
            }
        }
        un8Index++;
        if (un8Index > tLongestCIDypeTextLen)
        {
            break;
        }

    } while (TRUE);

    if (bOk == FALSE)
    {
        // done with this
        SMSO_vDestroy((SMS_OBJECT)pacBuff);
        return CID_INVALID_OBJECT;
    }

    eType = eTypeFromText(&pacBuff[0]);

    // Lock down the CID object for access.
    bLocked = SMSO_bLock((SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // Create an instance of the CID object of this type without a parent.
        psObj = psCreateObject(NULL, eType, NULL);
        if(psObj != NULL)
        {
            // Initialize object per inputs
            psObj->un8Flags &= ~(CID_CREATOR);
            psObj->un8Flags |= CID_CREATOR_APPLICATION;
            psObj->pvObjectData = NULL;

            // Use interface function if one exists
            if(psObj->psIntf->sObjData.pvRead != NULL)
            {
                // Read contents based on type
                psObj->pvObjectData = psObj->psIntf->sObjData.pvRead(psFile);
            }

            // Check if the CID's new object data was created.
            if(psObj->pvObjectData != NULL)
            {
                BOOLEAN bEquivRead;

                bEquivRead = bFReadEquivalentCids(psObj, psFile);

                if (bEquivRead == TRUE)
                {
                    // Initialize handle
                    hCID = (CID_OBJECT)psObj;
                }
                else
                {
                    // Error!
                    vDestroyObject(psObj);
                }
            }
            else
            {
                // Error!
                vDestroyObject(psObj);
            }
        }

        // Unlock CID object
        SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
    }

    // done with this
    SMSO_vDestroy((SMS_OBJECT)pacBuff);

    return hCID;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a CID's contents to a specified file or device.
* This is mainly helpful during debugging of CID's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CID to a file for the
* purposes of later re-generating the CID (for storage of the object).
* This API instead sends the CID as a verbose formatted output version.
*
* Inputs:
*
*   hCID - The CID handle the caller wishes to write.
*   psFile - The device to write the CID contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    CID_OBJECT hCID,
    FILE *psFile
        )
{
    N32 n32Return = 0;

#if SMS_DEBUG == 1
    N32 n32Temp   = 0;
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    BOOLEAN bOwner;

    // Determine if the CID is owned
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);

    // Verify inputs. Object handle must be owned as well as the file handle.
    if((bOwner == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Print generic CID information
    n32Return += fprintf(psFile, "CID: hCID = 0x%p\n", psObj);
    n32Return += fprintf(psFile, "\tFlags: %s\n",
        (
        ((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_INVALID) ?
                MACRO_TO_STRING(CID_CREATOR_INVALID) :
        ((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_APPLICATION) ?
                MACRO_TO_STRING(CID_CREATOR_APPLICATION) :
        ((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_SMS) ?
                MACRO_TO_STRING(CID_CREATOR_SMS) :
        ((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_UNASSIGNED) ?
                MACRO_TO_STRING(CID_CREATOR_UNASSIGNED) :
        "?" )
                );
    n32Return += fprintf(psFile, "\thEntry: 0x%p\n",
        psObj->hEntry);
    n32Return += fprintf(psFile, "\teType: %s\n",
        pacTypeText(psObj->psIntf->eType, NULL));
    n32Return += fprintf(psFile, "\tpvObjectData = 0x%p\n",
        psObj->pvObjectData);

    // Use interface function if one exists
    if(psObj->psIntf->sObjData.n32FPrintf != NULL)
    {
        // Dump the object data
        n32Temp = psObj->psIntf->sObjData.n32FPrintf(
            psObj->pvObjectData, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    // Print all equivalent cids
    if (psObj->psEquivalent !=  (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        CID_EQUIVALENT_CID_STRUCT *psEquivalent = psObj->psEquivalent;

        n32Return += fprintf(psFile, "\nEquivalent CID List:\n");

        while (psEquivalent != (CID_EQUIVALENT_CID_STRUCT *)NULL)
        {
            if (psEquivalent->bUseCid == FALSE)
            {
                // All following cids will also not be
                // usable so stop here
                break;
            }

            // Write this CID
            n32Return += CID.n32FPrintf(psEquivalent->hCid, psFile);

            // Move to the next
            psEquivalent = psEquivalent->psNext;
        }

    }

    n32Return += fprintf(psFile, "\n");

#endif /* SMS_DEBUG == 1 */

    return n32Return;
}

/*****************************************************************************
*
* pacTypeText
*
* This is a local function which simply maps an enumerated type to
* a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacTypeText(
    CID_ENUM eType,
    size_t *ptLength
        )
{
    const char *pacReturnString;
    size_t tLength;

    if (ptLength == NULL)
    {
        ptLength = &tLength;
    }

    switch (eType)
    {
        case CID_PID:
            pacReturnString = MACRO_TO_STRING(CID_PID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_PID)) - 1;
        break;

        case CID_AID:
            pacReturnString = MACRO_TO_STRING(CID_AID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_AID)) - 1;
        break;

        case CID_SXI_ID:
            pacReturnString = MACRO_TO_STRING(CID_SXI_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SXI_ID)) - 1;
        break;

        case CID_SXI_ARTIST_ID:
            pacReturnString = MACRO_TO_STRING(CID_SXI_ARTIST_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SXI_ARTIST_ID)) - 1;
        break;

        case CID_SXI_SONG_ID:
            pacReturnString = MACRO_TO_STRING(CID_SXI_SONG_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SXI_SONG_ID)) - 1;
        break;

        case CID_SXI_MARKET_ID:
            pacReturnString = MACRO_TO_STRING(CID_SXI_MARKET_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SXI_MARKET_ID)) - 1;
        break;

        case CID_SXI_SPORTS_ID:
            pacReturnString = MACRO_TO_STRING(CID_SXI_SPORTS_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SXI_SPORTS_ID)) - 1;
        break;

        case CID_SONG_TAG_ID:
            pacReturnString = MACRO_TO_STRING(CID_SONG_TAG_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_SONG_TAG_ID)) - 1;
        break;

        case CID_ARTIST_TEXT:
            pacReturnString = MACRO_TO_STRING(CID_ARTIST_TEXT);
            *ptLength = sizeof(MACRO_TO_STRING(CID_ARTIST_TEXT)) - 1;
        break;

        case CID_TITLE_TEXT:
            pacReturnString = MACRO_TO_STRING(CID_TITLE_TEXT);
            *ptLength = sizeof(MACRO_TO_STRING(CID_TITLE_TEXT)) - 1;
        break;

        case CID_COMPOSER_TEXT:
            pacReturnString = MACRO_TO_STRING(CID_COMPOSER_TEXT);
            *ptLength = sizeof(MACRO_TO_STRING(CID_COMPOSER_TEXT)) - 1;
        break;

        case CID_ALBUM_NAME_TEXT:
            pacReturnString = MACRO_TO_STRING(CID_ALBUM_NAME_TEXT);
            *ptLength = sizeof(MACRO_TO_STRING(CID_ALBUM_NAME_TEXT)) - 1;
            break;

        case CID_CONTENTINFO_TEXT:
            pacReturnString = MACRO_TO_STRING(CID_CONTENTINFO_TEXT);
            *ptLength = sizeof(MACRO_TO_STRING(CID_CONTENTINFO_TEXT)) - 1;
        break;

        case CID_LEAGUE_TEAM_ID:
            pacReturnString = MACRO_TO_STRING(CID_LEAGUE_TEAM_ID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_LEAGUE_TEAM_ID)) - 1;
        break;

        case CID_INVALID:
        default:
            pacReturnString = MACRO_TO_STRING(CID_INVALID);
            *ptLength = sizeof(MACRO_TO_STRING(CID_INVALID)) - 1;
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CID_bInstall
*
* Install the CID object which is applicable to all of SMS. The CID object
* contains a list of registered CID types and their interfaces. CID types may
* be registered at any time but cannot be unregistered.
*
* Inputs:
*   hSMS - An SMS object to associate as this registry parent.
*
* Outputs:
*   BOOLEAN - TRUE if successful, otherwise FALSE on error.
*
*****************************************************************************/
BOOLEAN CID_bInstall (
    SMS_OBJECT hSMS
        )
{
    BOOLEAN bRetval = FALSE;
    CID_CTRL_STRUCT *psCidCtrl;
    BOOLEAN bOwner;

    // Check first that we are the owner of the object we are
    // associating this registry with.
    bOwner = SMSO_bOwner(hSMS);
    if(bOwner == FALSE)
    {
        return FALSE;
    }

    // Allocate a chunk of memory to hold the CID control structure.
    psCidCtrl = (CID_CTRL_STRUCT *)
        SMSO_hCreate(
            CID_OBJECT_NAME":Ctrl",
            sizeof(CID_CTRL_STRUCT),
            SMS_INVALID_OBJECT,
            TRUE); // Enable lock feature
    if(psCidCtrl == NULL)
    {
        // Error! We cannot proceed
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CID_OBJECT_NAME": CID Ctrl cannot be created.");
        return FALSE;
    }

    // Check if global pointer is NULL. If so, populate it with
    // the new object. Otherwise there is nothing for us to do.
    if (gpsCidCtrl == NULL)
    {
        // This is the new object, and it is locked already.
        gpsCidCtrl = psCidCtrl;
    }
    else
    {
        // Destroy it an exit
        SMSO_vDestroy((SMS_OBJECT)psCidCtrl);
        return TRUE; // We already have or are creating it
    }

    do
    {
        // Initialize array to hold registered CID types
        OSAL.bMemSet((void*)gpsCidCtrl->apsTypes, 0,
                     sizeof(gpsCidCtrl->apsTypes));

        // Register our own CID types (e.g. Artist, Title, Composer, Album)

        // Artist Text
        bRetval = CID_bRegister(&gsArtistTextIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register Artist Text CID type.");
            break;
        }

        // Title Text
        bRetval = CID_bRegister(&gsTitleTextIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register Title Text CID type.");
            break;
        }

        // Composer Text
        bRetval = CID_bRegister(&gsComposerTextIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register Composer Text CID type.");
            break;
        }

        // Album Name
        bRetval = CID_bRegister(&gsAlbumNameIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register Album Name CID type.");
            break;
        }

        // Song Tag
        bRetval = CID_bRegister(&GsSongTagIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register Song Tag Id CID type.");
            break;
        }

        // ContentInfo Text
        bRetval = CID_bRegister(&gsContentInfoTextIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register ContentInfo Text CID type.");
            break;
        }

        // TeamId Text
        bRetval = CID_bRegister(&GsLeagueTeamIdIntf);
        if(bRetval == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot register TeamId CID type.");
            break;
        }

        // Unlock this object
        SMSO_vUnlock((SMS_OBJECT)psCidCtrl);

        // If we made it this far the CID object has been created

    } while(FALSE);

    if (bRetval == FALSE)
    {
        CID_vUninstall();
    }

    return bRetval;
}

/*****************************************************************************
*
*   CID_vUninstall
*
* Destroy an existing CID object which belongs to all of SMS
*
*****************************************************************************/
void CID_vUninstall (
    void
        )
{
    BOOLEAN bLocked;

    // Lock down the CID object for access.
    bLocked = SMSO_bLock((SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CID_CTRL_STRUCT *psCidCtrl;
        const CID_TYPE_INTERFACE_STRUCT **ppsIntf;
        CID_ENUM eCidType;

        // Release CID resources...
        ppsIntf = gpsCidCtrl->apsTypes;
        for (eCidType = CID_FIRST;
             eCidType <= CID_LAST;
             ++eCidType, ++ppsIntf)
        {
            // Use interface function if one exists
            if(((*ppsIntf) != NULL) &&
               ((*ppsIntf)->vUnInitialize != NULL))
            {
                // Based on the type, un-initialize it
                (*ppsIntf)->vUnInitialize();
            }
        }

        // Prepare local var to destroy the object
        psCidCtrl = gpsCidCtrl;

        // Clear CID ctrl global pointer
        gpsCidCtrl = NULL;

        // Destroy the CID-ctrl object itself
        SMSO_vDestroy((SMS_OBJECT)psCidCtrl);
    }

    return;
}

/*****************************************************************************
*
*   CID_bRegister
*
* Allows the registration of a CID type interface structure which defines
* how to identify, parse, create, destroy and manipulate object data associated
* with that type.
*
*****************************************************************************/
BOOLEAN CID_bRegister (
    const CID_TYPE_INTERFACE_STRUCT *psIntf
        )
{
    BOOLEAN bRegistered = FALSE;

    // Verify input
    if((psIntf != NULL) &&
       (psIntf->eType >= CID_FIRST) && (psIntf->eType <= CID_LAST))
    {
        BOOLEAN bLocked;

        // Lock the CID object
        bLocked =
            SMSO_bLock((SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Add interface to our list of types if it isn't already
            // in our list (unique).
            if (gpsCidCtrl->apsTypes[psIntf->eType] == NULL)
            {
                // Reference the interface from the array
                gpsCidCtrl->apsTypes[psIntf->eType] = psIntf;

                // Type added, run initialization method if it exists
                // Use interface function if one exists
                if(psIntf->bInitialize != NULL)
                {
                    // Based on CID type, run initialization of object data
                    bRegistered = psIntf->bInitialize();
                    if(bRegistered == FALSE)
                    {
                        // Remove entry we added
                        gpsCidCtrl->apsTypes[psIntf->eType] = NULL;
                    }
                }
                else
                {
                    // All is well
                    bRegistered = TRUE;
                }
            }

            // Unlock CID object
            SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
        }
    }

    return bRegistered;
}

/*****************************************************************************
*
*   CID_hCreate
*
* Create a CID which is modifiable, meaning it may be created, modified and
* later destroyed. Recycling of non-constant CIDs will also be handled
* here as well.
*
* Inputs:
*   hCidPool - The pool from which to recycle/create CIDs
*   eType - The type of CID object to create.
*   pvObjectDataPtr - The raw source data from which to create this object.
*
* Outputs:
*   CID_OBJECT - a new or recycled CID which represents the content provided.
*
*****************************************************************************/
CID_OBJECT CID_hCreate (
    CID_POOL hCidPool,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        )
{
    CID_OBJECT_STRUCT *psObj = NULL;

    // Input object data ptr cannot be NULL
    if(pvObjectDataPtr != NULL)
    {
        BOOLEAN bLocked;

        // Lock the CID-ctrl object
        bLocked = SMSO_bLock((SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            BOOLEAN bValid;
            CID_POOL_STRUCT *psCidPool = (CID_POOL_STRUCT*) hCidPool;
            CID_SUB_POOL_STRUCT *psSPool = NULL;

            // Determine the CID pool we are working with (if any)
            bValid = SMSO_bValid((SMS_OBJECT)hCidPool);
            if(bValid == TRUE)
            {
                // You also must be the owner
                BOOLEAN bOwner;

                bOwner = SMSO_bOwner((SMS_OBJECT)hCidPool);
                if(bOwner == TRUE)
                {
                    // First, we are to create a non-constant CID, check if
                    // we have one available in our CID pool. If so, we can use a recycled
                    // one. There is no point in putting constant CIDs into a CID-pool
                    // since they cannot be reused anyway. So constant CIDs are not in the
                    // CID-pool. If a recycled CID of this type can be found, return it.
                    // If so the recycled CID obtained will be re-inserted as unavailable.
                    psObj = psFindReusable(psCidPool, eType, &psSPool);
                }
            }

            // Check if we have a CID
            if (psObj == NULL)
            {
                // Could not obtain a recycled CID, so let's make a new one. This
                // one's parent is the CID pool since it belongs to it. Also
                // this new CID will be added to the CID pool.
                psObj = psCreateObject(psCidPool, eType, psSPool);
                if(psObj == NULL)
                {
                    // Error!
                    // Something is wrong, we cannot find a recycled CID
                    // nor can we create one.
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CID_OBJECT_NAME": Cannot get any CID.");
                }
            }

            // Verify we have a CID
            if(psObj != NULL)
            {
                // Check if this CID has object data we can reuse.
                // If so, we can just modify it. Otherwise we
                // need to create new object data.
                if(psObj->pvObjectData != NULL)
                {
                    // We have existing object data
                    CID_OBJECT hId = (CID_OBJECT)psObj;

                    // Modify the existing object data, replacing it with the new
                    // object data provided by the caller.
                    CID_bModify(&hId, pvObjectDataPtr);
                    psObj = (CID_OBJECT_STRUCT *)hId;

                    // We don't check the return value because we don't care
                    // that anything has changed. We just want it modified.
                }
                else
                {
                    // Nope, we need to create the object data so we do that now.

                    // Assign object data
                    // Use interface function if one exists
                    if(psObj->psIntf->sObjData.pvCreate != NULL)
                    {
                        // Based on CID type, create object data which is
                        // Modifiable (not constant).
                        psObj->pvObjectData = psObj->psIntf->sObjData.pvCreate(
                            pvObjectDataPtr, psObj->psIntf->sObjData.tMinSize,
                            (SMS_OBJECT)psObj, FALSE
                                );
                    }

                    // Check if we have or could create object data
                    if(psObj->pvObjectData == NULL)
                    {
                        // Something went wrong as we could not create the
                        // object data or they didn't provide us anything
                        // interesting to create. So we have to release our CID
                        // and bail out of here. It's ok though. It just goes
                        // back into the pool.
                        CID_vDestroy((CID_OBJECT)psObj);
                        psObj = NULL;
                    }
                }
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
        }
    }

    return (CID_OBJECT)psObj;
}

/*****************************************************************************
*
*   CID_hCreateConst
*
* Create a constant CID which cannot be modified. Constant CIDs are not
* part of the CID pool and connot benefit from or use recycled CIDs.
*
* Inputs:
*   eType - The type of object to create.
*   pvObjectDataPtr - The raw source data from which to create this object.
*
* Outputs:
*   CID_OBJECT - a new CID which represents the content provided.
*
*****************************************************************************/
CID_OBJECT CID_hCreateConst (
    CID_ENUM eType,
    const void *pvObjectDataPtr
        )
{
    CID_OBJECT_STRUCT *psObj = NULL;

    // Input object data ptr cannot be NULL
    if(pvObjectDataPtr != NULL)
    {
        BOOLEAN bLocked;

        // Lock the CID object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Create an instance of the CID object without a parent.
            psObj = psCreateObject(NULL, eType, NULL);
            if(psObj != NULL)
            {
                // Assign object data by creating it
                // Use interface function if one exists
                if(psObj->psIntf->sObjData.pvCreate != NULL)
                {
                    // Based on CID type, create object data
                    psObj->pvObjectData = psObj->psIntf->sObjData.pvCreate(
                        pvObjectDataPtr, psObj->psIntf->sObjData.tMinSize,
                        (SMS_OBJECT)psObj, TRUE);
                }
                // Check if we have or could create object data
                if(psObj->pvObjectData == NULL)
                {
                    // Something went wrong as we could not create the
                    // object data. So we have to release or CID and
                    // bail out of here.
                    CID_vDestroy((CID_OBJECT)psObj);
                    psObj = NULL;
                }
            }

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
        }
    }

    return (CID_OBJECT)psObj;
}

/*****************************************************************************
*
*   CID_vDestroy
*
* Destroy the provided CID given it's handle. This API is only called by
* SMS and thus will destroy Application or SMS CIDs
*
* Inputs:
*   hCID - A valid CID created by SMS
*
* Outputs:
*   None.
*
*****************************************************************************/
void CID_vDestroy (
    CID_OBJECT hCID
        )
{
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == TRUE)
    {
        CID_OBJECT_STRUCT *psObj = (CID_OBJECT_STRUCT *)hCID;

        // Check if this CID belongs to the CID pool. If it does, then
        // we can recycle it for future use. So, we skip destroying the
        // object data and actually just re-initialize it later when we need it.
        // This essentially puts this CID back into the CID pool.
        if(psObj->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Recycle CID
            vRecycle(psObj);
        }
        else // Not in CID-pool so we just destroy it.
        {
            // Destroy the CID since it cannot be recycled.
            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   CID_pvObjectData
*
* Retrieves the object data associated with this CID. The object data
* is returned as a void * and must be cast appropriately to the
* type used when it was created. Each CID's object data has particular meaning
* to each type.
*
* Inputs:
*   hCID - Any valid CID.
*
* Outputs:
*   void * - A pointer to the CID's object data
*
*****************************************************************************/
const void *CID_pvObjectData (
    CID_OBJECT hCID
        )
{
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == FALSE)
    {
        // Error!
        return NULL;
    }

    return psObj->pvObjectData;
}

/*****************************************************************************
*
*   CID_eType
*
* Retrieves which type of CID this is.
*
* Inputs:
*
*   hCID - A valid CID object for which to determine the type.
*
* Outputs:
*
*   The CID's type (CID_ENUM).
*
*****************************************************************************/
CID_ENUM CID_eType (
    CID_OBJECT hCID
        )
{
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == FALSE)
    {
        // Error!
        return CID_INVALID;
    }

    return psObj->psIntf->eType;
}

/*****************************************************************************
*
*   CID_tSize
*
* Inputs:
*
*   hCID - A valid CID object for which to determine the size of.
*
* Outputs:
*
*   size of the contents of this CID in bytes.
*
*****************************************************************************/
size_t CID_tSize (
    CID_OBJECT hCID
        )
{
    size_t tSize = 0;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == TRUE)
    {
        CID_OBJECT_STRUCT *psObj =
            (CID_OBJECT_STRUCT *)hCID;

        // All CIDs have a type
        pacTypeText(psObj->psIntf->eType, &tSize);

        tSize += sizeof(char); // all have the comma delimiter

        // Use interface function if one exists
        if(psObj->psIntf->sObjData.tSize != NULL)
        {
            // Determine size of this object data
            tSize += psObj->psIntf->sObjData.tSize(psObj->pvObjectData);
        }
    }

    return tSize;
}

/*****************************************************************************
*
*   CID_bModify
*
* Modify an existing CID object. The CID must exist already and the caller
* must have exclusive access to the object. The caller may provide new
* object data for which to update the CID with. If a replacement CID
* is created to accommodate the new data, one is created and the old one
* will be destroyed. The resulting CID is always returned via the provided
* pointer to a CID handle in either case.
*
* Inputs:
*
*   phCID - A pointer to a valid CID handle which represents the CID
*   to update. It also is used as a return value in case the CID provided
*   is replaced with another.
*
*   pvObjectDataPtr - The object data required to update the CID with.
*
* Outputs:
*
*   TRUE on success and if the CID was updated (something changed), otherwise
*   FALSE is returned indicating nothing has changed or error.
*
*****************************************************************************/
BOOLEAN CID_bModify (
    CID_OBJECT *phCID,
    const void *pvObjectDataPtr
        )
{
    BOOLEAN bRetval = FALSE;
    CID_OBJECT_STRUCT **ppsObj =
        (CID_OBJECT_STRUCT **)phCID;
    CID_EQUIVALENT_CID_STRUCT *psEquiv =
        (CID_EQUIVALENT_CID_STRUCT *)NULL;
    BOOLEAN bOwner;

    // Verify inputs. Object handle pointer and the handle itself as
    // well as the source pointer must all be valid.
    if( (phCID == NULL) || (pvObjectDataPtr == NULL) )
    {
        // Error!
        return FALSE;
    }

    // Determine if this is valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)*phCID);
    if(bOwner == FALSE)
    {
        // Error!
        return FALSE;
    }

    // Mark all equivalent cid entries as available
    psEquiv = (*ppsObj)->psEquivalent;

    while (psEquiv != (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        psEquiv->bUseCid = FALSE;
        psEquiv = psEquiv->psNext;
    }

    // Use interface function if one exists
    if((*ppsObj)->psIntf->sObjData.bModify != NULL)
    {
        // Modify object according to type
        bRetval = (*ppsObj)->psIntf->sObjData.bModify(
            &(*ppsObj)->pvObjectData, pvObjectDataPtr);
    }

    // Object modified
    return bRetval;
}

/*****************************************************************************
*
*   CID_bCopy
*
* This object interface method is used to copy an existing CID into another.
* The source CID may be one created by SMS or an Application. Either way
* a deep-copy is performed and thus a duplicate of the original is made into
* another existing CID, but a new CID is not created. For a copy to occur
* both CIDs must be of the same type.
*
* This destination CID may be used by the application as it was originally
* intended. Once the caller is done with the CID however it must be released
* the same way it was originally created.
*
* Inputs:
*
*   hDst - The destination CID the caller wishes to copy into.
*   hSrc - The source CID the caller wishes to copy from.
*
* Outputs:
*
*   TRUE on success, otherwise FALSE.
*
*****************************************************************************/
BOOLEAN CID_bCopy (
    CID_OBJECT hDst,
    CID_OBJECT hSrc
        )
{
    BOOLEAN bOwner, bRetval = FALSE;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDst);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hSrc);
    }

    if (bOwner == TRUE)
    {
        CID_OBJECT_STRUCT
            *psDstObj = (CID_OBJECT_STRUCT *)hDst,
            *psSrcObj = (CID_OBJECT_STRUCT *)hSrc;

        // Both Src and Dst CID types must match.
        if(psDstObj->psIntf->eType == psSrcObj->psIntf->eType)
        {
            // Use interface function if one exists
            if(psDstObj->psIntf->sObjData.bCopy != NULL)
            {
                // Copy object data according to type
                bRetval = psDstObj->psIntf->sObjData.bCopy(
                    (void *)&psDstObj->pvObjectData, psSrcObj->pvObjectData);
            }
        }
    }

    return bRetval;
}

/*****************************************************************************
*
*   CID_n32FWriteToMemory
*
* This function is called to write the copy the contents of a CID to
* a memory location
*
* Inputs:
*
*   hCID - The CID handle the caller wishes to write.
*   ppvMemory - The address to where the CID should be copied.
*               This parameter will be updated to the address just after the
*               CID we just copied. This is done to allow this function to
*               make it easy to copy a series of CIDs to a memory region.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
N32 CID_n32FWriteToMemory (
    CID_OBJECT hCID,
    void **ppvMemory
        )
{
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    N32 n32Return;
    UN8 *pun8Memory; // Byte-wise addressable memory
    const char *pacType;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);
    if(bOwner == FALSE)
    {
        // Error!
        return EOF;
    }

    // Verify inputs
    if(ppvMemory == NULL)
    {
        // Error!
        return EOF;
    }

    // Assign byte-wise addressable memory
    pun8Memory = (UN8 *)*ppvMemory;

    // Verify input
    if(pun8Memory == NULL)
    {
        // Error!
        return EOF;
    }

    // Write CID to memory...

    // copy type
    pacType = pacTypeText(psObj->psIntf->eType, NULL);
    while(*pacType != '\0')
    {
        *pun8Memory++ = *pacType++;
    }

    *pun8Memory++ = CID_TYPE_VALUE_DELIMITER_STRING[0];

    // Use interface function if one exists
    if(psObj->psIntf->sObjData.n32FWriteToMemory != NULL)
    {
        // Write contents
        n32Return = psObj->psIntf->sObjData.n32FWriteToMemory(
            psObj->pvObjectData, (void **)&pun8Memory);
        if(n32Return < 0)
        {
            return EOF;
        }
    }

    // Compute size written
    n32Return = pun8Memory - (UN8 *)*ppvMemory;
    *ppvMemory = pun8Memory;

    return n32Return;
}

/*****************************************************************************
*
*   CID_hReadFromMemory
*
* This function is called to read data from memory and create a CID from it
*
* Inputs:
*
*   hCID - The CID handle the caller wishes to write.
*   ppvMemory - The address to where the CID should be copied.
*               This parameter will be updated to the address just after the
*               CID we just copied. This is done to allow this function to
*               make it easy to copy a series of CIDs to a memory region.
*
* Outputs:
*
*   A handle to a valid CID or CID_INVALID_OBJECT
*
*****************************************************************************/
CID_OBJECT CID_hReadFromMemory (
    const void **ppvMemory
        )
{
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    CID_ENUM eType = CID_INVALID;
    UN8 *pun8Memory, *pun8Ptr = NULL; // Byte-wise addressable memory
    CID_OBJECT_STRUCT *psObj = NULL;

    // Verify inputs
    if(ppvMemory == NULL)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Assign byte-wise addressable memory
    pun8Memory = (UN8 *)*ppvMemory;

    // Verify input
    if(pun8Memory == NULL)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // read CID type...
    // scan for the ,

    // then strncmp

    // based on the text, figure out the type

    eType = eTypeFromText((char*)pun8Memory);

    // Provided CID must be of valid type
    if((eType < CID_FIRST) || (eType > CID_LAST))
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    pun8Ptr = (UN8 *)strpbrk(
        (char *)pun8Memory, CID_TYPE_VALUE_DELIMITER_STRING);
    if (pun8Ptr != NULL)
    {
        BOOLEAN bOk = FALSE;
        BOOLEAN bLocked;

        // point to just after the comma
        pun8Ptr++;

        // Lock the CID-ctrl object
        bLocked = SMSO_bLock((SMS_OBJECT)gpsCidCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Create an instance of the CID object without a parent.
            psObj = psCreateObject(NULL, eType, NULL);
            if(psObj == NULL)
            {
                // Error!
                // Something is wrong, we cannot find a recycled CID
                // nor can we create one.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CID_OBJECT_NAME": Cannot get any CID.");

                SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
                return CID_INVALID_OBJECT;
            }

            if (psObj->psIntf->sObjData.bReadFromMemory != NULL)
            {
                bOk = psObj->psIntf->sObjData.bReadFromMemory(
                    &psObj->pvObjectData,
                    (void **)&pun8Ptr, (SMS_OBJECT)psObj);
                if (bOk == TRUE)
                {
                    hCID = (CID_OBJECT)psObj;
                }
            }

            if (bOk == FALSE)
            {
                CID_vDestroy((CID_OBJECT)psObj);
            }

            if (hCID != CID_INVALID_OBJECT)
            {
                size_t tSize;

                // update memory pointer
                tSize = CID_tSize(hCID);

                // Assign byte-wise addressable memory
                pun8Memory = (UN8 *)*ppvMemory;
                pun8Memory += tSize;
                *ppvMemory = pun8Memory;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)gpsCidCtrl);
    }

    return hCID;
}

/*****************************************************************************
*
*   CID_n32GetValue
*
*****************************************************************************/
N32 CID_n32GetValue (
    CID_OBJECT hCID,
    void **ppvValue
        )
{
    CID_OBJECT_STRUCT *psObj =
        (CID_OBJECT_STRUCT *)hCID;
    N32 n32Return = 0;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCID);

    // Verify inputs
    if((bOwner == FALSE) || (ppvValue == NULL))
    {
        // Error!
        return EOF;
    }

    // Use interface function if one exists
    if(psObj->psIntf->sObjData.n32GetValue != NULL)
    {
        // Write value to provided location
        n32Return = psObj->psIntf->sObjData.n32GetValue(
            psObj->pvObjectData, ppvValue);
        if(n32Return < 0)
        {
            return EOF;
        }
    }

    return n32Return;
}

/*****************************************************************************
*
* CID_hCreatePool
*
* Create a CID pool which belongs to the SMS object provided.
*
* So what exactly is this CID pool...
*
* A CID pool is used to create and release CIDs without having to re-allocate
* and free new CIDs as content changes. As it so happens as we receive
* channel information about a broadcast, the CDO parses this information
* and applies it to the CHANNEL object information. Part of this information
* are CIDs which uniquely identify specific aspects of this content such
* as artist id's, program id's and textual representations. As content changes
* so do the CIDs. Normally as new content arrives the previous CIDs are
* destroyed and new ones are created. This is somewhat inefficient as it
* taxes the memory allocation of the system quite often. Instead a CID pool
* allows CIDs to be recycled. So when CIDs are created they have hooks into
* the CID pool but are marked as unavailable. When the CID needs to be
* destroyed it is simply returned to the CID pool and marked as available.
* At that point it is available for anyone else who needs to create a new
* CID of that same type. The idea is that it is more efficient to simply
* reallocate an existing CID rather than destroying and creating new ones all
* the time. In practice it is found that CIDs are changing all the time and
* CID recycling eliminates the need to repetitively allocate and free memory.
*
* Not all CIDs belong to the CID pool. Constant CIDs, those created by
* hDuplicate() or hFRead() are constant and application owned and thus
* cannot be recycled. So they never appear in the pool. This is ok since there
* are usually very few of these and of those which exist they are usually
* created and not destroyed until SMS is uninitialized anyway, so being non-
* dynamic they do not suffer from the same reallocation issues the non-constant
* CIDs do.
*
* In addition to the recycling properties of the CID pool and it's CIDs, the
* CID pool is owned by a single DECODER and thus all CHANNELs within it's
* channel cache. This improves overall efficiency by allowing CIDs to
* be recycled among all active channels within a channel cache and not just
* within a single CHANNEL.
*
*****************************************************************************/
CID_POOL CID_hCreatePool (
    SMS_OBJECT hSMS
        )
{
    CID_POOL_STRUCT *psCidPool;

    // Allocate a chunk of memory to hold the CID control structure.
    psCidPool = (CID_POOL_STRUCT *)
        SMSO_hCreate(
            CID_OBJECT_NAME":Pool",
            sizeof(CID_POOL_STRUCT),
            hSMS, // Inherit lock feature
            FALSE);
    if(psCidPool == NULL)
    {
        // Error! We cannot proceed
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CID_OBJECT_NAME": CID pool cannot be created.");
        return CID_POOL_INVALID_OBJECT;
    }

    // Create an array to associate with the CID pool. This array will
    // maintain a list of CID Sub-pool of objects which have been allocated
    // for this particular CID pool. As CIDs are created they are added to
    // the list, as they are destroyed they are returned to the list.
    // Creating a CID will first look to this list for any
    // already allocated and available CID before creating and adding a
    // new one.
    OSAL.bMemSet(psCidPool->apsSubPool, 0, sizeof(psCidPool->apsSubPool));

    return (CID_POOL)psCidPool;
}

/*****************************************************************************
*
*   CID_vDestroyPool
*
*****************************************************************************/
void CID_vDestroyPool (
    CID_POOL hCidPool
        )
{
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be owned.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCidPool);
    if(bOwner == TRUE)
    {
        CID_POOL_STRUCT *psCidPool = (CID_POOL_STRUCT *)hCidPool;
        CID_SUB_POOL_STRUCT **ppsSPool;
        CID_ENUM eCidType;

        // Remove all entries from the list and destroy CID which are
        // in the pool.
        ppsSPool = psCidPool->apsSubPool;
        for (eCidType = CID_FIRST;
             eCidType <= CID_LAST;
             ++eCidType, ++ppsSPool)
        {
            vSubPoolDestroy(*ppsSPool);
            *ppsSPool = NULL;
        }

        // Now we can destroy the pool itself
        SMSO_vDestroy((SMS_OBJECT)psCidPool);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16CompareObjects
*
* This is the real function which compares two CIDs. It cannot be used
* directly since the OSAL compare interface does not have the binary option.
* This function performs either a relationship or binary compare based on
* the input parameter bBinary.
*
* Outputs:
*   0   - CIDs have the same value (equal)
*   > 0 - hCID1 is greater than (after) hCID2
*   < 0 - hCID1 is less than (before) hCID2 or error
*
*****************************************************************************/
static N16 n16CompareObjects (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2,
    BOOLEAN bBinary
        )
{
    N16 n16Result = N16_MIN;
    CID_OBJECT_STRUCT *psObj1 = (CID_OBJECT_STRUCT *)hCID1,
                      *psObj2 = (CID_OBJECT_STRUCT *)hCID2;

    // Both CIDs must exist
    if((psObj1 == NULL) || (psObj2 == NULL))
    {
        // Error!
        return n16Result;
    }

    // Both CIDs must be of the same type
    n16Result = psObj1->psIntf->eType - psObj2->psIntf->eType;
    if(n16Result == 0) // Equal
    {
        // Use interface function if one exists
        if(psObj1->psIntf->sObjData.n16Compare != NULL)
        {
            // Based on CID type, perform the compare
            n16Result = psObj1->psIntf->sObjData.n16Compare(
                psObj1->pvObjectData, psObj2->pvObjectData, bBinary);
        }
    }

    // Re-condition return for strict binary compare.
    // STRING object returns 0 or 1 for binary compare.
    // We need -1 when not equal or 0 for equal.
    if((bBinary == TRUE) && (n16Result != 0))
    {
        // Re-condition output
        n16Result = -1;
    }

    return n16Result;
}

/*****************************************************************************
*
*   psGetTypeIntf
*
*****************************************************************************/
static const CID_TYPE_INTERFACE_STRUCT *psGetTypeIntf(
    CID_CTRL_STRUCT *psCidCtrl,
    CID_ENUM eType
        )
{
    const CID_TYPE_INTERFACE_STRUCT *psResult;

    // Looking for the interface
    if ((CID_FIRST <= eType) && (eType <= CID_LAST))
    {
        psResult = psCidCtrl->apsTypes[eType];
    }
    else
    {
        psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   psSubPoolCreate
*
*****************************************************************************/
static CID_SUB_POOL_STRUCT *psSubPoolCreate (
    CID_POOL_STRUCT *psPool,
    CID_ENUM eType
        )
{
    CID_SUB_POOL_STRUCT *psResult = NULL;
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    const CID_TYPE_INTERFACE_STRUCT *psIntf;

    do
    {
        // Get the interface
        psIntf = psGetTypeIntf(gpsCidCtrl, eType);
        if (psIntf == NULL)
        {
            printf(CID_OBJECT_NAME": unknown type #%d", (int)eType);
            break;
        }

        // Create the sub-pool structure
        psResult =
            (CID_SUB_POOL_STRUCT*)SMSO_hCreate(CID_OBJECT_NAME":SPool",
                                               sizeof(CID_SUB_POOL_STRUCT),
                                               (SMS_OBJECT)psPool,
                                               FALSE);
        if (psResult == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to allocate memory for sub-pool #%d",
                (int) eType);
            break;
        }

        // Fill in the object's data
        psResult->hPool = OSAL_INVALID_OBJECT_HDL;
        psResult->psIntf = psIntf;

        // Create the pool inside
        eReturnCode =
            OSAL.eLinkedListCreate(&psResult->hPool,
                    CID_OBJECT_NAME":SPool:List", NULL,
                    OSAL_LL_OPTION_LINEAR |
                    OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                  CID_OBJECT_NAME": failed to create sub-pool list #%d (%s)",
                  (int) eType, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Register pool with in the sub-pool list
        psPool->apsSubPool[eType] = psResult;

        // We've done
        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        vSubPoolDestroy(psResult);
        psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   vSubPoolDestroy
*
*****************************************************************************/
static void vSubPoolDestroy(
    CID_SUB_POOL_STRUCT *psSPool
        )
{
    // Clean up the controlled objects if exists
    if ((psSPool != NULL) &&
        (psSPool->hPool != OSAL_INVALID_OBJECT_HDL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Remove all items from the list
        eReturnCode = OSAL.eLinkedListRemoveAll(psSPool->hPool,
                                (OSAL_LL_RELEASE_HANDLER)vSubPoolItemDestroy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to clean up the sub-pool list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        // Delete the list
        eReturnCode = OSAL.eLinkedListDelete(psSPool->hPool);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to delete the sub-pool list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        psSPool->hPool = OSAL_INVALID_OBJECT_HDL;

        // Destroy the objects
        SMSO_vDestroy((SMS_OBJECT) psSPool);
    }

    return;
}

/*****************************************************************************
*
*   vSubPoolItemDestroy
*
*****************************************************************************/
static void vSubPoolItemDestroy (
    CID_SUB_POOL_ITEM_STRUCT *psItem
        )
{
    if (psItem->psObj != NULL)
    {
        // Destroy controlled object
        vDestroyObject(psItem->psObj);
    }

    // Release the item's memory
    OSAL.vLinkedListMemoryFree((void*) psItem);

    return;
}

/*****************************************************************************
*
*   bSubPoolRearrange
*
*****************************************************************************/
static BOOLEAN bSubPoolReArrange(
    CID_SUB_POOL_STRUCT *psSPool,
    CID_SUB_POOL_ITEM_STRUCT *psItem,
    CID_SUB_POOL_REARRANGEMENT_ACTION_ENUM eAction
        )
{
    BOOLEAN bResult = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CID_OBJECT_STRUCT *psObj = psItem->psObj;

    do
    {
        if (eAction != CID_SUB_POOL_REARRANGEMENT_ACTION_ADD)
        {
            // Remove the node from its current position because there is
            // a hit it is in the pool now
            eReturnCode = OSAL.eLinkedListRemove(psObj->hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CID_OBJECT_NAME": failed to remve entry from the list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            // The list not longer has the item
            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }

        // Based on the action perform rearrangement
        if (eAction == CID_SUB_POOL_REARRANGEMENT_ACTION_RECYCLE)
        {
            OSAL_LINKED_LIST_ENTRY hTopEntry;

            // In case of the release the item has to be put on the list top
            // In other words, it must become the first entry in the list

            // Simply add the node to the not-sorted list - in this case
            // the item will be put at the end of the list
            hTopEntry = OSAL.hLinkedListFirst(psSPool->hPool, NULL);
            if (hTopEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                eReturnCode =
                    OSAL.eLinkedListAdd(psSPool->hPool, &psObj->hEntry,
                        (void*)psItem);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CID_OBJECT_NAME": failed to add entry to the list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }
            }
            else
            {
                // Add this entry right before the first one
                eReturnCode =
                    OSAL.eLinkedListAddBeforeEntry(psSPool->hPool, &hTopEntry,
                        (void*)psItem);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CID_OBJECT_NAME
                        ": failed to add entry before the first to the list (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }

                // Populate new position of the entry
                psObj->hEntry = hTopEntry;
            }
        }
        else
        {
            // Simply add the node to the not-sorted list - in this case
            // the item will be put at the end of the list
            eReturnCode =
                OSAL.eLinkedListAdd(psSPool->hPool, &psObj->hEntry,
                                    (void*)psItem);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CID_OBJECT_NAME": failed to remove entry from the list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        // We're ok
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   psFindSubPool
*
*****************************************************************************/
static CID_SUB_POOL_STRUCT *psFindSubPool(
    CID_POOL_STRUCT *psPool,
    CID_ENUM eType,
    BOOLEAN *pbCreated
        )
{
    CID_SUB_POOL_STRUCT *psResult = NULL;

    // Find the sub-pool by type
    psResult = psPool->apsSubPool[eType];
    if (psResult == (CID_SUB_POOL_STRUCT*) NULL)
    {
        // Create the sub-pool if asked
        psResult = psSubPoolCreate(psPool, eType);
        if (psResult == (CID_SUB_POOL_STRUCT*) NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to create the sub-pool #%d",
                (int) eType);
        }
        else if (pbCreated != (BOOLEAN*) NULL)
        {
            // Just created pool
            *pbCreated = TRUE;
        }
    }

    return psResult;
}

/*****************************************************************************
*
*   psFindReusable
*
*   This function attempts to locate an available CID in the CID pool which
*   matches the type provided.
*
*****************************************************************************/
static CID_OBJECT_STRUCT *psFindReusable (
    CID_POOL_STRUCT *psCidPool,
    CID_ENUM eType,
    CID_SUB_POOL_STRUCT **ppsSPool
        )
{
    CID_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bIsSPoolCreated = FALSE;

    // Search for an available CID with this same type as one we want
    // to create.
    *ppsSPool = psFindSubPool(psCidPool, eType, &bIsSPoolCreated);

    // There is a sense looking for some reusable object if pools exists
    // and it has not been created from scratch while the searching request
    if ((*ppsSPool != NULL) && (bIsSPoolCreated == FALSE))
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        CID_SUB_POOL_ITEM_STRUCT *psItem;

        // Check the first item in the sub-pool
        hEntry = OSAL.hLinkedListFirst((*ppsSPool)->hPool, (void**)&psItem);
        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            BOOLEAN bReused = FALSE;

            // Assign for fast usage
            psObj = psItem->psObj;

            // Check its free
            if ((psObj->un8Flags & CID_CREATOR) == CID_CREATOR_UNASSIGNED)
            {
                // Mark as allocated to SMS by modifying the flags
                psObj->un8Flags &= ~(CID_CREATOR);
                psObj->un8Flags |= CID_CREATOR_SMS;

                // Re-arrange the item in the list
                bReused = bSubPoolReArrange(*ppsSPool, psItem,
                    CID_SUB_POOL_REARRANGEMENT_ACTION_REUSE);
            }

            if (bReused != TRUE)
            {
                // At this point there isn't much we can do, so we just
                // go ahead and leave it alone, but don't use it.
                psObj = NULL;
            }
#if DEBUG_OBJECT != 0
            else
            {
                (*ppsSPool)->un32Available--;
                (*ppsSPool)->un32Unavailable++;

                printf(CID_OBJECT_NAME": Reusing CID from the pool #%d (%u/%u/%u)\n",
                    (int) (*ppsSPool)->psIntf->eType,
                    (*ppsSPool)->un32Available,
                    (*ppsSPool)->un32Unavailable,
                    (*ppsSPool)->un32Available + (*ppsSPool)->un32Unavailable
                        );
            }
#endif
        }
    }

    return psObj;
}

/*****************************************************************************
*
*   psCreateObject
*
*   This function is used to create a new CID object and put it into
*   the CID pool if the SMS_OBJECT handle provided is valid.
*
*****************************************************************************/
static CID_OBJECT_STRUCT *psCreateObject (
    CID_POOL_STRUCT *psCidPool,
    CID_ENUM eType,
    CID_SUB_POOL_STRUCT *psSPool
        )
{
    CID_OBJECT_STRUCT *psObj = NULL;
    CID_SUB_POOL_ITEM_STRUCT *psItem = NULL;
    SMS_OBJECT hParent;
    const CID_TYPE_INTERFACE_STRUCT *psIntf = NULL;

    do
    {
        // Search for an available CID type interface for the one we want
        // to create.
        if (psSPool == NULL)
        {
            // Get type from the CID Control Structure
            psIntf = psGetTypeIntf(gpsCidCtrl, eType);
            if (psIntf == NULL)
            {
                // Simply stop since requested unknown type
                printf(CID_OBJECT_NAME": there is no interface for #%d\n",
                    (int) eType );
                break;
            }

            // The CID pool is a parent from here
            hParent = (SMS_OBJECT) psCidPool;
        }
        else
        {
            // Grab the interface
            psIntf = psSPool->psIntf;

            // Create sub-pool item to handle newly created CIDC
            psItem = (CID_SUB_POOL_ITEM_STRUCT*)
                OSAL.pvLinkedListMemoryAllocate(CID_OBJECT_NAME":SPoolItem",
                    sizeof(CID_SUB_POOL_ITEM_STRUCT), TRUE);
            if (psItem == NULL)
            {
                printf(CID_OBJECT_NAME
                    ": failed to allocate sub-pool #%d item, "
                    "but keep creating the CID as non-reusable",
                    (int) eType );

                // The CID pool is a parent from here
                hParent = (SMS_OBJECT) psCidPool;
            }
            else
            {
                // The CID sub-pool is a parent from here
                hParent = (SMS_OBJECT) psSPool;
            }
        }

        // Create an instance of the CID object.
        // If a valid handle was provided, the SMS object create will
        // simply inherit any locking ability that objects has. However
        // if the handle is invalid, we would never use any locking. This
        // is due to the fact that this happens only when making a duplicate.
        // Therefore providing FALSE for the lock feature is always right.
        psObj = (CID_OBJECT_STRUCT *)
            SMSO_hCreate(
                CID_OBJECT_NAME,
                sizeof(CID_OBJECT_STRUCT),
                hParent, // Child-of this
                FALSE
                    );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to cerate CID object");
            break;
        }

        // Initialize object to defaults and assign interface
        psObj->pvObjectData = NULL;
        psObj->psEquivalent = (CID_EQUIVALENT_CID_STRUCT *)NULL;
        psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psObj->un8Flags = CID_CREATOR_SMS;
        psObj->psIntf = psIntf;

        // Is this CID to be added to the CID sub-pool?
        if (psItem != NULL)
        {
            BOOLEAN bOk;

            // Make new object part of the pool
            psItem->psObj = psObj;

            // Place the item to the sub-pool
            bOk = bSubPoolReArrange(psSPool, psItem,
                        CID_SUB_POOL_REARRANGEMENT_ACTION_ADD);
            if (bOk == FALSE)
            {
                printf(CID_OBJECT_NAME
                    ": failed to put new object to the sub-pool #%d, "
                    " but it is ok to make CID as not-reusable",
                    (int) eType);

                // The item is not needed since here
                psItem->psObj = NULL;
            }
#if DEBUG_OBJECT != 0
            else
            {
                psSPool->un32Unavailable++;

                printf(CID_OBJECT_NAME": Added new CID to sub-pool #%d (%u/%u/%u)\n",
                    (int) psSPool->psIntf->eType,
                    psSPool->un32Available,
                    psSPool->un32Unavailable,
                    psSPool->un32Available + psSPool->un32Unavailable
                        );
            }
#endif
        }
    } while (FALSE);

    // Clean up if needed
    if ((psItem != NULL) && (psItem->psObj == NULL))
    {
        vSubPoolItemDestroy(psItem);
    }

    return psObj;
}

/*****************************************************************************
*
*   vRecycle
*
*   This function replaces a CID back into the CID pool for future
*   allocation when needed, effectively recycling it.
*
*****************************************************************************/
static void vRecycle (
    CID_OBJECT_STRUCT *psObj
        )
{
    CID_SUB_POOL_STRUCT *psSPool;
    CID_SUB_POOL_ITEM_STRUCT *psItem;
    BOOLEAN bOk = FALSE;

    // Determine the CID pool this CID belongs to
    psSPool = (CID_SUB_POOL_STRUCT *)SMSO_hParent((SMS_OBJECT)psObj);
    if(psSPool == NULL)
    {
        // Error!
        return;
    }

    // Update CID flags, make it an unassigned CID in the pool.
    psObj->un8Flags &= ~(CID_CREATOR);
    psObj->un8Flags |= CID_CREATOR_UNASSIGNED;

    // Re-insert this entry as an unused CID using it's new flags,
    // but leaving everything else alone.
    psItem = (CID_SUB_POOL_ITEM_STRUCT*)OSAL.pvLinkedListThis(psObj->hEntry);
    if (psItem != NULL)
    {
        bOk = bSubPoolReArrange(psSPool, psItem,
                        CID_SUB_POOL_REARRANGEMENT_ACTION_RECYCLE);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": failed to rearrange the item in the sub-pool #%d",
                psSPool->psIntf->eType);
        }
    }
#if SMS_DEBUG == 1
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CID_OBJECT_NAME": failed to get item for the sub-pool #%d",
            psSPool->psIntf->eType);
    }
#endif

    if (bOk == FALSE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Remove CID from pool
        eReturnCode = OSAL.eLinkedListRemove(
            psObj->hEntry
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Destroy this CID now that it is not part of the pool.
            vDestroyObject(psObj);

#if DEBUG_OBJECT != 0
            psSPool->un32Unavailable--;

            printf(
                CID_OBJECT_NAME": Removed CID from sub-pool #%d (%u/%u/%u).",
                (int) psSPool->psIntf->eType,
                psSPool->un32Available,
                psSPool->un32Unavailable,
                psSPool->un32Available + psSPool->un32Unavailable
                    );
#endif
        }
        else
        {
            // Error! But don't destroy it. I'd rather leave it in
            // as a dangling CID than destroy it which might cause
            // corruption when iterating the list.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CID_OBJECT_NAME": Cannot remove CID from sub-pool -- "
                "dangling CID exists.");
        }
    }
#if DEBUG_OBJECT != 0
    else
    {
        psSPool->un32Available++;
        psSPool->un32Unavailable--;

        printf(CID_OBJECT_NAME": Recycling CID back into sub-pool #%d (%u/%u/%u)\n",
            (int) psSPool->psIntf->eType,
            psSPool->un32Available,
            psSPool->un32Unavailable,
            psSPool->un32Available + psSPool->un32Unavailable
                );
    }
#endif

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*   This function destroys a CID object.
*
*****************************************************************************/
static void vDestroyObject (
    CID_OBJECT_STRUCT *psObj
        )
{
    CID_EQUIVALENT_CID_STRUCT *psEquiv = psObj->psEquivalent,
                              *psNextEquiv = (CID_EQUIVALENT_CID_STRUCT *)NULL;

    // Clear this pointer
    psObj->psEquivalent = (CID_EQUIVALENT_CID_STRUCT *)NULL;

    // Destroy all equivalent cids and their list entries
    while (psEquiv != (CID_EQUIVALENT_CID_STRUCT *)NULL)
    {
        // Grab the next entry
        psNextEquiv = psEquiv->psNext;

        // Clear this one
        if (psEquiv->hCid != CID_INVALID_OBJECT)
        {
            CID.vDestroy(psEquiv->hCid);
            psEquiv->hCid = CID_INVALID_OBJECT;
        }

        psEquiv->bUseCid = FALSE;
        psEquiv->psNext = (CID_EQUIVALENT_CID_STRUCT *)NULL;
        SMSO_vDestroy((SMS_OBJECT)psEquiv);

        psEquiv = psNextEquiv;
    }

    // Use interface function if one exists
    if(psObj->psIntf->sObjData.vDestroy != NULL)
    {
        // Based on the type, destroy object data
        psObj->psIntf->sObjData.vDestroy(psObj->pvObjectData);
    }

    // Re-Initialize object
    psObj->pvObjectData = NULL;
    psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->un8Flags = CID_CREATOR_INVALID;
    psObj->psIntf = NULL;

    // Free object instance itself
    SMSO_vDestroy((SMS_OBJECT)psObj);

    return;
}

/*****************************************************************************
*
*   eTypeFromText
*
*   This function attempts to map type text to the corresponding cid type
*   enum
*
*   If the text doesn't match any cid type, CID_INVALID will be returned.
*
*****************************************************************************/
static CID_ENUM eTypeFromText (
    const char *pacText
        )
{
    CID_ENUM eType = CID_INVALID;
    if (pacText != NULL)
    {
        CID_ENUM eIndex;

        ////////// find out the type!!!!
        for(eIndex = CID_FIRST; eIndex <= CID_LAST; ++eIndex)
        {
            size_t tNum;
            const char *pacType;

            pacType = pacTypeText(eIndex, &tNum);
            if (strncmp(pacType, pacText, tNum) == 0)
            {
                eType = eIndex;
                break;
            }
        }
    }

    return eType;
}
