/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the CID INTEGER implementation for the
 *  Sirius Module Services (SMS). This module is intended to serve as a
 *  generic CID Object Data implementation for CIDs which use INTEGERs
 *  as their basic object data type. These methods are made public
 *  (as friend functions) to any CID type implementation which needs to use
 *  them.
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

// Include things I need from SMS
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"

// Include CID INTEGER module headers
#include "cid_integer.h"
#include "_cid_integer.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CIDINT_pvCreate
*
*   Creates a CID object data object(INTEGER) from object data which is an
*   integer.
*
*****************************************************************************/
const void *CIDINT_pvCreate (
    const void *pvSrcObjectData,
    size_t tMinimumSize,
    SMS_OBJECT hParent,
    BOOLEAN bConstant
        )
{
    // Object data for this type is a INTEGER object
    CID_INTEGER_STRUCT *psInteger = NULL;

    // Source object data (raw) is a CID_INTEGER_VALUE_TYPE type
    const CID_INTEGER_VALUE_TYPE *ptValue =
        (const CID_INTEGER_VALUE_TYPE *)(size_t)pvSrcObjectData;

    // If a valid handle was provided, the SMS object create will
    // simply inherit any locking ability that objects has. However
    // if the handle is invalid, we would never use any locking. This
    // is due to the fact that this happens only when making a duplicate.
    // Therefore providing FALSE for the lock feature is always right.

    // Create an INTEGER object
    psInteger = (CID_INTEGER_STRUCT *)SMSO_hCreate(
                               "CID_Int",
                               sizeof(CID_INTEGER_STRUCT),
                               hParent,
                               FALSE);

    if (psInteger != NULL)
    {
        if(ptValue != NULL)
        {
            // set the value
            psInteger->tValue = *ptValue;
        }
        else
        {
           //otherwise set zero
           psInteger->tValue = 0;
        }

        psInteger->bConstant = bConstant;
    }

    return (const void *)psInteger;
}

/*****************************************************************************
*
*   CIDINT_vDestroy
*
*   Destroys a CID object data object(INTEGER)
*
*****************************************************************************/
void CIDINT_vDestroy (
    const void *pvObject
        )
{
    // These are SMSO objects
    SMSO_vDestroy((SMS_OBJECT)pvObject);
    return;
}

/*****************************************************************************
*
*   CIDINT_pvDuplicate
*
*   Duplicates a CID object data object (INTEGER)
*
*****************************************************************************/
const void *CIDINT_pvDuplicate (
    const void *pvObject
        )
{
    // These are INTEGER's
    CID_INTEGER_OBJECT hInteger = (CID_INTEGER_OBJECT)pvObject;
    CID_INTEGER_STRUCT *psInteger = (CID_INTEGER_STRUCT *)hInteger;
    CID_INTEGER_STRUCT const *psDuplicateInteger;

    psDuplicateInteger = CIDINT_pvCreate((void const *)&psInteger->tValue,
                                         sizeof(CID_INTEGER_STRUCT),
                                         SMS_INVALID_OBJECT,
                                         psInteger->bConstant);

    return (const void *)psDuplicateInteger;
}

/*****************************************************************************
*
*   CIDINT_bModify
*
*   Modifies a CID object data object (INTEGER)
*
*****************************************************************************/
BOOLEAN CIDINT_bModify (
    const void **ppvObject,
    const void *pvSrcObjectData
        )
{
    // These are integers
    CID_INTEGER_OBJECT *phInteger = (CID_INTEGER_OBJECT *)ppvObject;
    CID_INTEGER_STRUCT *psInteger;
    const CID_INTEGER_VALUE_TYPE *ptValue =
        (const CID_INTEGER_VALUE_TYPE *)pvSrcObjectData;

    // modify it.
    psInteger = (CID_INTEGER_STRUCT *)*phInteger;

    if (psInteger->bConstant == FALSE)
    {
        if (psInteger->tValue != *ptValue)
        {
            psInteger->tValue = *ptValue;

            return TRUE;
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   CIDINT_bCopy
*
*   Copies a CID object data object (INTEGER) into another
*
*****************************************************************************/
BOOLEAN CIDINT_bCopy (
    void **ppvDstObject,
    const void *pvSrcObject
        )
{
    // These are CID_INTEGER_OBJECTs
    CID_INTEGER_OBJECT *phInteger = (CID_INTEGER_OBJECT *)ppvDstObject;
    CID_INTEGER_OBJECT hInteger = (CID_INTEGER_OBJECT)pvSrcObject;

    CID_INTEGER_STRUCT *psIntegerSrc,  *psIntegerDst;

    psIntegerDst = (CID_INTEGER_STRUCT *)*phInteger;
    psIntegerSrc = (CID_INTEGER_STRUCT *)hInteger;

    psIntegerDst->tValue = psIntegerSrc->tValue;
    psIntegerDst->bConstant = psIntegerSrc->bConstant;

    return TRUE;
}

/*****************************************************************************
*
*   CIDINT_tSize
*
*   Retrieves the size a CID object data object (INTEGER) in bytes
*
*****************************************************************************/
size_t CIDINT_tSize(
    const void *pvObject
        )
{
    return (size_t)MAX_INTEGER_AS_CHARS;
}

/*****************************************************************************
*
*   CIDINT_n16Compare
*
*   Compares a CID object data object (INTEGER) to another CID object data
*   object.
*
*****************************************************************************/
N16 CIDINT_n16Compare (
    const void *pvObject1,
    const void *pvObject2,
    BOOLEAN bBinary
        )
{
    // These are INTEGER objects
    CID_INTEGER_STRUCT *psInteger1, *psInteger2;
    CID_INTEGER_VALUE_TYPE tNum1, tNum2;
    N16 n16Return;

    psInteger1 = (CID_INTEGER_STRUCT *)pvObject1;
    psInteger2 = (CID_INTEGER_STRUCT *)pvObject2;

    tNum1 = psInteger1->tValue;
    tNum2 = psInteger2->tValue;

    if (tNum1 == tNum2)
    {
        n16Return = 0;
    }
    else if (bBinary == TRUE)
    {
        n16Return = 1;
    }
    else if (tNum1 < tNum2)
    {
        n16Return = -1;
    }
    else
    {
        n16Return = 1;
    }

    return n16Return;
}

/*****************************************************************************
*
*   CIDINT_n32FWrite
*
*   Writes a CID object data object (INTEGER) to a device.
*
*****************************************************************************/
N32 CIDINT_n32FWrite (
    const void *pvObject,
    FILE *psFile
        )
{

    N32 n32Written = 0, n32RetVal;
    // These are INTEGER objects
    CID_INTEGER_STRUCT *psInteger = (CID_INTEGER_STRUCT *)pvObject;

    do
    {
        // Write the string
        n32RetVal = fprintf(psFile, "%u", psInteger->tValue);
        if(n32RetVal < 0)
        {
            // Error!
            n32Written = EOF;
            break;
        }

        // Accumulate # of bytes written
        n32Written += n32RetVal;

        // Add null terminator to identify the end of the string
        n32RetVal = (N32)fwrite("\0", 1, 1, psFile);
        if(n32RetVal < 0)
        {
            // Error!
            n32Written = EOF;
            break;
        }

        // Accumulate # of bytes written
        n32Written += n32RetVal;

    } while(FALSE);

    return n32Written;
}

/*****************************************************************************
*
*   CIDINT_n32FWriteToMemory
*
*   Writes a CID object data object (INTEGER) to memory.
*
*****************************************************************************/
N32 CIDINT_n32FWriteToMemory (
    const void *pvObject,
    void **ppvMemory
        )
{
    CID_INTEGER_OBJECT hInteger = (CID_INTEGER_OBJECT)pvObject;
    CID_INTEGER_STRUCT *psInteger = (CID_INTEGER_STRUCT *)hInteger;
    char *pacMemory = (char *)*ppvMemory; // Byte-wise addressable memory
    N32 n32Return = 0;

    n32Return = snprintf(pacMemory, MAX_INTEGER_AS_CHARS+1, "%u", psInteger->tValue);

    if(n32Return > 0)
    {
        // update the memory
        pacMemory += (UN32)n32Return;
    }

    // Compute size written
    n32Return = pacMemory - (char *)*ppvMemory;
    *ppvMemory = pacMemory;

    return n32Return;
}

/*****************************************************************************
*
*   CIDINT_n32GetValue
*
*   Retreives CID object data value (INTEGER)
*
*****************************************************************************/
N32 CIDINT_n32GetValue (
    const void *pvObject,
    void **ppvValue
        )
{

    CID_INTEGER_OBJECT hInteger = (CID_INTEGER_OBJECT)pvObject;
    CID_INTEGER_STRUCT *psInteger = (CID_INTEGER_STRUCT *)hInteger;
    CID_INTEGER_VALUE_TYPE *ptValue = (CID_INTEGER_VALUE_TYPE *)*ppvValue;
    N32 n32Return = EOF;

    if (ptValue != NULL)
    {
        // write the value
        *ptValue =  psInteger->tValue;

        n32Return = sizeof(CID_INTEGER_VALUE_TYPE);
    }

    return n32Return;
}

/*****************************************************************************
*
*   CIDINT_pvRead
*
*   Reads a CID object data object (INTEGER) from a file
*
*****************************************************************************/
const void *CIDINT_pvRead (
    FILE *psFile
        )
{
    CID_INTEGER_OBJECT hInteger = CID_INTEGER_INVALID_OBJECT;
    char acBuffer[32]; // Just a decent sized buffer to incrementally
                       // read in a string.
    UN8 un8Index = 0;
    size_t tNum;
    BOOLEAN bError = FALSE;

    // Verify input
    if(psFile == NULL)
    {
        // Error!
        return CID_INTEGER_INVALID_OBJECT;
    }

    // Place a null terminator in the last element of our buffer and
    // leave it there for processing purposes later.
    acBuffer[sizeof(acBuffer) - 1] = '\0';

    // Read a string from the file provide byte by byte until a '\0' is found.
    // A '\0' is used to determine the end of a integer entity.
    do
    {
        // Read a character from the file
        tNum = fread(&acBuffer[un8Index], sizeof(acBuffer[0]), 1, psFile);
        if(tNum != 1)
        {
            // Error! File error occurred.
            bError = TRUE;
            break;
        }

        if((acBuffer[un8Index] == '\0') ||
            (un8Index == sizeof(acBuffer) - 2))
        {
            break;
        }

        un8Index++;

    } while(TRUE);

    // If we have found either the end of string
    if(acBuffer[un8Index] == '\0')
    {
        CID_INTEGER_VALUE_TYPE tValue;

        tValue = (CID_INTEGER_VALUE_TYPE)atoi(&acBuffer[0]);
        hInteger = (CID_INTEGER_OBJECT)CIDINT_pvCreate(&tValue,
                        sizeof(CID_INTEGER_VALUE_TYPE),
                        SMS_INVALID_OBJECT, FALSE);
    }
    else
    {
        bError = TRUE;
    }

    // Check if there was an error. If so destroy anything we created
    // and return an invalid handle.
    if(bError == TRUE)
    {
        if(hInteger != CID_INTEGER_INVALID_OBJECT)
        {
            // Destroy object we created
            CIDINT_vDestroy(hInteger);
            hInteger = CID_INTEGER_INVALID_OBJECT;
        }
    }

    return hInteger;
}

/*****************************************************************************
*
*   CIDINT_hReadFromMemory
*
*   Create a CID object (INTEGER) from data in memory (similar to create, but
*   the data from which the cid is formed is a char string, not an integer.
*
*****************************************************************************/
BOOLEAN CIDINT_bReadFromMemory(
    const void **ppvObjectData,
    void **ppvMemory,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bOk = FALSE;

    // Verify we have a CID
    if ( (ppvObjectData != NULL) && (ppvMemory != NULL) )
    {
        CID_INTEGER_STRUCT *psInteger;

        // Check if this CID has object data we can reuse.
        // If so, we can just modify it. Otherwise we
        // need to create new object data.
        if(*ppvObjectData != NULL)
        {
            psInteger = (CID_INTEGER_STRUCT *)*ppvObjectData;

            // Modify the existing object data, replacing it with the new
            // object data provided by the caller.
            psInteger->tValue = atoi(*ppvMemory);
            psInteger->bConstant = FALSE;

            bOk = TRUE;
        }
        else
        {
            // If a valid handle was provided, the SMS object create will
            // simply inherit any locking ability that objects has. However
            // if the handle is invalid, we would never use any locking. This
            // is due to the fact that this happens only when making a duplicate.
            // Therefore providing FALSE for the lock feature is always right.

            // Nope, we need to create the object data so we do that now.
            psInteger = (CID_INTEGER_STRUCT *)SMSO_hCreate(
                       "CID_Int",
                       sizeof(CID_INTEGER_STRUCT),
                       hParent, FALSE
                       );

            if (psInteger != NULL)
            {
                // set the value
                psInteger->tValue = atoi(*ppvMemory);
                psInteger->bConstant = FALSE;

                *ppvObjectData = (void *)psInteger;

                bOk = TRUE;
            }
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   CIDINT_n32FPrintf
*
*   Prints a CID object data object (INTEGER) to a file
*
*****************************************************************************/
N32 CIDINT_n32FPrintf (
    const void *pvObject,
    FILE *psFile
        )
{
    N32 n32NumWritten = EOF;
    if (pvObject != NULL)
    {
        CID_INTEGER_STRUCT *psInteger = (CID_INTEGER_STRUCT *)pvObject;

        n32NumWritten = fprintf(psFile, "Integer: hInteger = %p", psInteger);
        n32NumWritten += fprintf(psFile, "\tValue = %u", psInteger->tValue);
    }

    return n32NumWritten;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
