/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CHANNELLIST implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "radio.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "module_obj.h"
#include "decoder_obj.h"
#include "string_obj.h"
#include "category_obj.h"
#include "channel_obj.h"
#include "ccache.h"
#include "browse_obj.h"
#include "presets_obj.h"
#include "channellist_obj.h"
#include "_channellist_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
*
*   hCreate
*
*   This API is used to create a CHANNELLIST object. If the API is able to
*   create the list, a handle to this CHANNELLIST is returned. This handle
*   is used for all subsequent APIs which require a CHANNELLIST handle. If
*   the call to this API is unsuccessful an invalid handle is returned.
*   The call of this API owns the created CHANNELLIST and is responsible
*   for eventually destroying it when it is no longer needed, the associated
*   DECODER is released or the system is shutdown.
*
*****************************************************************************/
static CHANNELLIST_OBJECT hCreate (
    DECODER_OBJECT hDecoder,
    BROWSE_TYPE_ENUM eBrowseType,
    CHANNEL_ID tReferenceChannelId,
    CATEGORY_ID tReferenceCategoryId,
    UN16 un16RequestedBefore,
    UN16 un16RequestedAfter,
    CHANNEL_EVENT_MASK tEventRequestMask,
    CHANNELLIST_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        )
{
    static UN32 un32Instance = 0;
    BOOLEAN bValid, bInitOk;
    CHANNELLIST_OBJECT_STRUCT *psObj =
        (CHANNELLIST_OBJECT_STRUCT *)CHANNELLIST_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BROWSE_CHANNEL_COMPARE_HANDLER bInitialBrowseChannelCompareHandler;
    void *pvInitialBrowseChannelCompareArg;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    UN32 un32LLOptions;
    OSAL_LL_COMPARE_HANDLER n16LLCompareHandler;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == FALSE)
    {
        // Error!
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Construct a unique name for this channel list.
    snprintf(&acName[0],
            sizeof(acName),
            CHANNELLIST_OBJECT_NAME":%u",
            un32Instance++);

    // Create an instance of this object. CHANNELLIST objects are owned
    // by the creator and are the parent object and have a lock associated
    // with them. The lock is required because channel lists may be updated
    // in the context of the DECODER via CHANNEL object callbacks.
    psObj = (CHANNELLIST_OBJECT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(CHANNELLIST_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, // Parent
            TRUE ); // Lock
    if(psObj == NULL)
    {
        // Error!
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Copy object name into CHANNELLIST object
    strncpy(&psObj->acName[0], &acName[0], sizeof(psObj->acName));

    // Initialize object with inputs
    psObj->hDecoder = hDecoder;
    psObj->un16BeforeActual = 0;
    psObj->un16AfterActual = 0;
    psObj->un16RequestedBefore = un16RequestedBefore;
    psObj->un16RequestedAfter = un16RequestedAfter;

    // Initialize reference channel/category info
    psObj->hReferenceChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hTopChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hBottomChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hReferenceCategory = CATEGORY_INVALID_OBJECT;

    if (bBrowseChannelCompareHandler == NULL)
    {
        // Caller didn't provide a browse channel compare handler
        // so we'll set it to the default handler -- all channels
        bInitialBrowseChannelCompareHandler = bBrowseAllChannelsCompareHandler;
        pvInitialBrowseChannelCompareArg = NULL;
    }
    else
    {
        // Caller provided a browse channel compare handler
        bInitialBrowseChannelCompareHandler = bBrowseChannelCompareHandler;
        pvInitialBrowseChannelCompareArg = pvBrowseChannelCompareArg;
    }

    // Create the browse object
    psObj->hBrowse = BROWSE_hCreate(
        (SMS_OBJECT)psObj,
        psObj->hDecoder );

    if (psObj->hBrowse == BROWSE_INVALID_OBJECT)
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Set the channel browse style
    bInitOk = BROWSE_bSetChannelBrowseStyle(
        psObj->hBrowse,
        bInitialBrowseChannelCompareHandler,
        pvInitialBrowseChannelCompareArg);

    if (bInitOk != TRUE)
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        CHANNEL_OBJECT_EVENT_INITIAL,
        tEventRequestMask,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
        pvEventCallbackArg);

    psObj->tEventRequestMask = tEventRequestMask;

    // Initialize channel list
    psObj->hChannelList = OSAL_INVALID_OBJECT_HDL;

    if(eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
    {
        // When a list of type "all channels" it must be unique.
        un32LLOptions = (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE);
        n16LLCompareHandler = (OSAL_LL_COMPARE_HANDLER)n16CompareChannelIDs;
    }
    else if(eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        // When a list is of type "category" the members of the list are not
        // required to be unique (like in virtual or user cats).
        un32LLOptions = OSAL_LL_OPTION_CIRCULAR;
        n16LLCompareHandler = (OSAL_LL_COMPARE_HANDLER)NULL;
    }
    else
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Create channel list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hChannelList,
        &psObj->acName[0],
        n16LLCompareHandler,
        un32LLOptions
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // create LL for old channel set
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hChannelListOld,
        CHANNELLIST_OBJECT_NAME": Old Channel Set",
        n16LLCompareHandler,
        un32LLOptions
            );

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Initialize cursor for old channel set
    psObj->hCursor = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Initialize this channel list
    bInitOk = bInitChannelList(
        psObj, eBrowseType,
        tReferenceChannelId, tReferenceCategoryId );

    if(bInitOk == FALSE)
    {
        // Error!
        vDestroy((CHANNELLIST_OBJECT)psObj);
        return CHANNELLIST_INVALID_OBJECT;
    }

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT)psObj);

    return (CHANNELLIST_OBJECT)psObj;
}

/*****************************************************************************
*
*   vDestroy
*
*   This API is used to destroy an existing CHANNELLIST object. This API
*   will perform any removal of CHANNEL objects from the list as required
*   and releases all resources it consumed. When this API returns to the
*   caller it should be assumed the CHANNELLIST object handle is no longer
*   valid. Only the original creator/owner of the CHANNELLIST may destroy
*   a CHANNELLIST.
*
*****************************************************************************/
static void vDestroy (
    CHANNELLIST_OBJECT hChannelList
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DECODER_OBJECT hDecoder;
    CCACHE_OBJECT hCCache;

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );

    if (psObj == NULL)
    {
        // Error!
        return;
    }

    // Filter out all further channel updates
    SMSU_tFilter(&psObj->sEvent, CHANNEL_OBJECT_EVENT_ALL);

    // Unregister our category notification if necessary
    if (psObj->hReferenceCategory != CATEGORY_INVALID_OBJECT)
    {
        // Unregister with the previous reference category
        CATEGORY_vUnregisterNotification(
            psObj->hReferenceCategory,
            vCategoryEventCallback,
            (void *)(size_t)psObj );

        psObj->hReferenceCategory = CATEGORY_INVALID_OBJECT;
    }

    // get ccache handle
    hCCache = DECODER_hCCache(psObj->hDecoder);

    // We no longer need to be notified of ccache events
    CCACHE_bUnRegisterNotification (
        hCCache,
        vCCacheEventCallback,
        (CHANNELLIST_OBJECT)psObj
            );

    CCACHE_bUnRegisterAllChannelsNotification (
        hCCache,
        vChannelInfoEventCallback,
        (void*)psObj
            );

    // Clear this channellist 
    // (clears both, current and old channel sets)
    vClearChannellist( psObj );

    // Destroy channel list if one exists
    if(psObj->hChannelList != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->hChannelList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hChannelList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    if (psObj->hChannelListOld != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete old channel set linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->hChannelListOld);
        if (eReturnCode == OSAL_SUCCESS)
        {
            psObj->hChannelListOld = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy the browse object if it exists
    if (psObj->hBrowse != BROWSE_INVALID_OBJECT)
    {
        BROWSE_vDestroy( psObj->hBrowse );
        psObj->hBrowse = BROWSE_INVALID_OBJECT;
    }

    // Clear channel list parameters...
    hDecoder = psObj->hDecoder;
    psObj->hDecoder = DECODER_INVALID_OBJECT;
    psObj->un16BeforeActual = 0;
    psObj->un16AfterActual = 0;
    psObj->un16RequestedBefore = 0;
    psObj->un16RequestedAfter = 0;

    // Destroy the SMS Update object
    SMSU_vDestroy(&psObj->sEvent);

    // Unlock the decoder only
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hChannelList);

    return;
}


/*****************************************************************************
*
*   bUpdate
*
*****************************************************************************/
static BOOLEAN bUpdate (
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tMask
        )
{
    BOOLEAN bSuccess = FALSE;
    CHANNELLIST_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );
    if (psObj == NULL)
    {
        return FALSE;
    }

    // Condition the mask so that only appropriate
    // bits can be set
    tMask &= CHANNEL_OBJECT_EVENT_INITIAL;

    // OR the mask into each channel in the channel list
    eReturnCode = OSAL.eLinkedListIterate (
        psObj->hChannelList,
        (OSAL_LL_ITERATOR_HANDLER)bUpdateChannelMask,
        (void *)(size_t)tMask);

    // If we were successful, update the channel list's mask
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Update the SMSU object with the specified mask
        SMSU_tUpdate(&psObj->sEvent, tMask);

        // Notify if something changed
        SMSU_bNotify(&psObj->sEvent);

        bSuccess = TRUE;
    }

    vUnlockSystem( psObj );

    return bSuccess;
}

/*****************************************************************************
*
*   eModifyRegisteredEventMask
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    if ( eModification < SMSAPI_MODIFY_EVENT_MASK_START ||
         eModification >= SMSAPI_MODIFY_EVENT_MASK_INVALID )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hChannelList);
    if(bValid == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj;

        psObj = (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
        eReturnCode = DECODER_eModifyChanListRegisteredEventMask (
            psObj->hDecoder,
            hChannelList,
            tEventMask,
            eModification
        );
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hChannel
*
*   This API is used to index into a CHANNELLIST much like an array, using
*   a provided signed offset with respect to the reference channel. This
*   offset represents the direction and index into the array or list of
*   channels this CHANNELLIST object possesses. The offset provided is
*   signed and will wrap around the list if the magnitude of the offset is
*   larger than the list itself. This API may only be called from the context
*   of a DECODER object (e.g. CHANNELLIST callback function). It cannot be
*   called outside of a DECODER.
*
*   NOTE: THIS API CAN ONLY BE DIRECTLY CALLED FROM THE CONTEXT
*   OF A DECODER OBJECT!
*
*****************************************************************************/
static CHANNEL_OBJECT hChannel (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        )
{
    BOOLEAN bLocked;
    CHANNEL_OBJECT hChannel =
        CHANNEL_INVALID_OBJECT;
    int iOffset = (int)n16Offset;

    // Verify and Lock the channellist
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CHANNELLIST_ENTRY_STRUCT *psEntry = NULL;
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
        BOOLEAN bOwner;

        // Check caller ownership. Caller may only call
        // this API if it is in the context of the DECODER
        // or it has the DECODER mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if(bOwner == TRUE)
        {
            int iOffsetOrig;
            UN32 un32NumItems = 0;

            // Determine how big our list is and adjust offset...
            OSAL.eLinkedListItems(psObj->hChannelList, &un32NumItems);
            if(un32NumItems == 0)
            {
                iOffset = 0;
            }
            else
            {
                N8 n8Sign;
                n8Sign = iOffset < 0 ? -1 : 1;
                iOffset = abs(iOffset) % un32NumItems;
                iOffset *= n8Sign;
            }

            // Save the offset we calculated
            iOffsetOrig = iOffset;

            psEntry = psGetChannelEntry(psObj, (N16)iOffset);

            if (psEntry != NULL)
            {
                hChannel = psEntry->hChannel;
            }

            // Now, update this channel's preset handle
            vUpdateChannelPresetHandle( psObj, hChannel, (N16)iOffsetOrig, FALSE );
        }
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return hChannel;
}

/*****************************************************************************
*
*   tChannelId
*
*   This API is similar to CHANNELLIST.hChannel() described above in that it
*   is used to index into a CHANNELLIST much like an array, using a provided
*   signed offset with respect to the reference channel. The difference is
*   that this API may be called from any context and returns a channel id
*   rather than a handle. The offset provided represents the direction and
*   index into the array or list of channels this CHANNELLIST object
*   possesses. The offset provided is signed and will wrap around the list
*   if the magnitude of the offset is larger than the list itself.
*
*****************************************************************************/
static CHANNEL_ID tChannelId (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    CHANNEL_OBJECT hChannelAtOffset;

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );

    if (psObj != NULL)
    {
        // First get the channel object at the
        // requested offset
        hChannelAtOffset = hChannel(hChannelList, n16Offset);

        // Extract channel id
        tChannelId = CHANNEL.tChannelId(hChannelAtOffset);

        // Unlock the system
        vUnlockSystem( psObj );
    }

    return tChannelId;
}

/*****************************************************************************
*
*   tChannelEventMask
*
*   NOTE: THIS API CAN ONLY BE DIRECTLY CALLED FROM THE CONTEXT
*   OF A DECODER OBJECT!
*****************************************************************************/
static CHANNEL_EVENT_MASK tChannelEventMask (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        )
{
    CHANNEL_EVENT_MASK tMask = CHANNEL_OBJECT_EVENT_NONE;
    BOOLEAN bLocked;

    // Verify and Lock the channellist
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CHANNELLIST_ENTRY_STRUCT *psEntry = NULL;
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
        BOOLEAN bOwner;

        // Check caller ownership. Caller may only call
        // this API if it is in the context of the DECODER
        // or it has the DECODER mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if(bOwner == TRUE)
        {
            psEntry = psGetChannelEntry(psObj, n16Offset);

            if (psEntry != NULL)
            {
                tMask = psEntry->tMask & psObj->tEventRequestMask;
                psEntry->tMask &= ~tMask;
            }
        }
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return tMask;
}

/*****************************************************************************
*
*   hCategory
*
*   This API is similar to CHANNELLIST.hChannel() described above in that it
*   is used to index into a CHANNELLIST much like an array, using a provided
*   signed offset with respect to the reference channel. The difference is
*   that this API returns the category assocated with the channel found
*   at n16Offset. The offset provided represents the direction and
*   index into the array or list of channels this CHANNELLIST object
*   possesses. The offset provided is signed and will wrap around the list
*   if the magnitude of the offset is larger than the list itself.
*
*   NOTE: THIS API CAN ONLY BE DIRECTLY CALLED FROM THE CONTEXT
*   OF A DECODER OBJECT!
*
*****************************************************************************/
static CATEGORY_OBJECT hCategory (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        )
{
    BOOLEAN bLocked;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;

    // Verify and Lock the channellist
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
        BOOLEAN bOwner;

        // Check caller ownership. Caller may only call
        // this API if it is in the context of the DECODER
        // or it has the DECODER mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if(bOwner == TRUE)
        {
            CCACHE_OBJECT hCCache;

            // Get the cache handle
            hCCache = DECODER_hCCache( psObj->hDecoder );
            if (hCCache != CCACHE_INVALID_OBJECT)
            {
                BROWSE_TYPE_ENUM eBrowseType;

                // Get our current browse type
                eBrowseType = BROWSE_eGetMode(
                    psObj->hBrowse);

                // Check the channel list browse type
                if (eBrowseType == BROWSE_TYPE_CATEGORY)
                {
                    // we have the ref cat handle
                    hCategory = psObj->hReferenceCategory;

                    if (hCategory != CATEGORY_INVALID_OBJECT)
                    {
                        // see if this is still a valid category
                        CATEGORY_ID tReferenceCategoryId;

                        tReferenceCategoryId = CATEGORY.tGetCategoryId(hCategory);

                        hCategory = CCACHE_hCategory(
                            hCCache, &tReferenceCategoryId, 0);

                        if (hCategory != psObj->hReferenceCategory)
                        {
                            // our ref cat is no good
                            hCategory = CATEGORY_INVALID_OBJECT;
                            psObj->hReferenceCategory = CATEGORY_INVALID_OBJECT;
                        }
                    }
                    /*
                    CATEGORY_ID tReferenceCategoryId =
                        CATEGORY_INVALID_ID;

                    // Get the reference category Id
                    BROWSE_bGetBrowsed(
                        psObj->hBrowse,
                        NULL,
                        &tReferenceCategoryId );

                    // All of the channels in this list
                    // belong to the reference category
                    hCategory = CCACHE_hCategory(
                        hCCache, &tReferenceCategoryId, 0);
                    */
                }
                else
                {
                    // Otherwise, just report the requested
                    // channel's broadcast category
                    CHANNEL_OBJECT hRequestedChannel;

                    // Use our API to get the requested channel
                    hRequestedChannel = hChannel(
                        hChannelList, n16Offset);

                    // Now extract the broadcast category
                    hCategory = CHANNEL.hCategory(
                        hRequestedChannel, 0 );
                }
            }
        }
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return hCategory;
}

/*****************************************************************************
*
*   tCategoryId
*
*   This API is similar to CHANNELLIST.hChannel() described above in that it
*   is used to index into a CHANNELLIST much like an array, using a provided
*   signed offset with respect to the reference channel. The difference is
*   that this API returns the category Id assocated with the channel found
*   at n16Offset and may be called from any context. The offset provided
*   represents the direction and index into the array or list of channels
*   this CHANNELLIST object possesses. The offset provided is signed and
*   will wrap around the list if the magnitude of the offset is larger than
*   the list itself.
*
*****************************************************************************/
static CATEGORY_ID tCategoryId (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );

    if (psObj != NULL)
    {
        CATEGORY_OBJECT hRequestedCategory;

        // Use our API to get the category
        hRequestedCategory = hCategory( hChannelList, n16Offset);

        // Extract the category Id
        tCategoryId = CATEGORY.tGetCategoryId( hRequestedCategory );

        vUnlockSystem( psObj );
    }

    return tCategoryId;
}

/*****************************************************************************
*
*   eBrowseList
*
*   This API is used displace the current reference channel within an
*   existing CHANNELLIST object. Effectively this browses the list by moving
*   the reference channel a direction and distance provided by the signed
*   offset parameter. This API may be called when a valid CHANNELLIST handle
*   is available.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseList (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16BrowseOffset
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj;
    CCACHE_OBJECT hCCache;
    BOOLEAN bOk, bSuccess = FALSE, bFinalPass = FALSE, bSimpleCleanUp = FALSE;
    N16 n16BrowseDirection = SMSAPI_DIRECTION_NEXT;
    UN16 un16BrowseMagnitude;
    CATEGORY_ID tBrowsedCategoryId;
    CHANNEL_ID tBrowsedChannelId;
    N16 n16NumAvailableChannels;
    BROWSE_TYPE_ENUM eBrowseType =
        BROWSE_TYPE_INVALID;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );
    if (psObj == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get our current browse type
    eBrowseType = BROWSE_eGetMode(
        psObj->hBrowse);

    // Verify channel list configuration status
    if  ( ( eBrowseType != BROWSE_TYPE_ALL_CHANNELS ) &&
          ( eBrowseType != BROWSE_TYPE_CATEGORY ) )

    {
        // Error!

        // Unlock
        vUnlockSystem( psObj );

        return SMSAPI_RETURN_CODE_ERROR;
    }


    // From the DECODER handle, determine the channel cache handle
    hCCache = DECODER_hCCache(psObj->hDecoder);

    // Verify channel cache
    if ( hCCache == CCACHE_INVALID_OBJECT)
    {
        // Error!
        vUnlockSystem( psObj );

        // We're done
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Get the currently browsed Ids
    bOk = BROWSE_bGetBrowsed(
        psObj->hBrowse,
        &tBrowsedChannelId,
        &tBrowsedCategoryId );

    if (bOk == FALSE)
    {
        // Error!
        vUnlockSystem( psObj );

        // We're done
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Get the number of available channels
    // based on how we are operating
    if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
    {
        CCACHE_vStats( hCCache, &n16NumAvailableChannels, NULL );
    }
    else
    {
        n16NumAvailableChannels = (N16)
            CATEGORY.tSize( psObj->hDecoder, tBrowsedCategoryId);
        if (n16NumAvailableChannels == 0)
        {
            tBrowsedChannelId = CHANNEL_INVALID_ID;

            // Right here we're finally ensured that browsed category
            // is empty and all what we can do - clean up it w/ no attempt of
            // recreation. The clean procedure performed later in this function
            // and we're interested in calling vClearChannellist() will
            // following clients notification in a proper way.
            bSimpleCleanUp = TRUE;
        }
    }

    // Ensure we are browsing in the correct direction
    if (n16BrowseOffset < 0)
    {
        n16BrowseDirection = SMSAPI_DIRECTION_PREVIOUS;
    }

    // Compute the browse magnitude
    un16BrowseMagnitude = n16BrowseOffset * n16BrowseDirection;

    // if we're a category chanlist we don't need to search all avaliable
    // channels, but we need to handle the wraparound case where the
    // requested offset is larger than the number of channels in the
    // category
    if (eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        if (n16NumAvailableChannels != 0)
        {
            if (un16BrowseMagnitude > n16NumAvailableChannels)
            {
                un16BrowseMagnitude =
                    un16BrowseMagnitude % (UN16)n16NumAvailableChannels;
            }
        }
    }

    // If our browse magnitude is zero, it
    // means we're supposed to just do this
    // one browse, so set it to 1 if
    // that is the case
    if (un16BrowseMagnitude == 0)
    {
        n16BrowseDirection = SMSAPI_DIRECTION_DIRECT;
        un16BrowseMagnitude = 1;
    }
    else
    {
        // We're not trying to do a direct browse so we can only do at most
        // one pass through the BROWSE_bBrowse function
        bFinalPass = TRUE;
    }

    // Cap browsing to the number of channels available
    while (n16NumAvailableChannels-- >= 0)
    {
        // Perform the browse
        bOk = BROWSE_bBrowse(
            psObj->hBrowse,
            (SMSAPI_DIRECTION_ENUM)n16BrowseDirection,
            TRUE,
            &tBrowsedChannelId,
            &tBrowsedCategoryId,
            TRUE);

        if ((bOk == TRUE) && (bSimpleCleanUp == FALSE))
        {
            // We're one step closer to
            // meeting the requested browse
            un16BrowseMagnitude--;

            if (un16BrowseMagnitude == 0)
            {
                // All done!
                bSuccess = TRUE;
                break;
            }
        }
        else
        {
            if ((bFinalPass == FALSE) && (bSimpleCleanUp == FALSE))
            {
                // The channel didn't meet the requested browse.
                if (n16BrowseDirection == SMSAPI_DIRECTION_DIRECT)
                {
                    // And it was direct. That means to browse up until we find
                    // a suitable channel.
                    n16BrowseDirection = SMSAPI_DIRECTION_NEXT;
                }

                // The next pass be our last time through BROWSE_bBrowse
                bFinalPass = TRUE;
            }
            else
            {
                // Couldn't find a reference channel that met the browse style.
                // but that's ok, the list can exist with 0 channels for now.
                // Maybe later some channels will be able to satisfy the browse
                // style

                // clear out the list
                vClearChannellist( psObj );

                // Indicate the channel list has been modified
                SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_INITIAL);

                // Notify callback if registered and change occurred
                SMSU_bNotify(&psObj->sEvent);

                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                vUnlockSystem( psObj );
                return eReturnCode;
            }
        }
    }

    // Don't need to validate category size here --
    // If this is a category channel list, that category
    // has already been verified.  If this is a regular channel
    // list, the category is a broadcast category

    if (bSuccess == TRUE)
    {
        // Create the new channel list based
        // on the current references
        bSuccess = bCreateList( psObj, hCCache );

        if (bSuccess == TRUE)
        {
            // Notify callback if registered and change occurred
            SMSU_bNotify(&psObj->sEvent);

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    vUnlockSystem( psObj );

    return eReturnCode;
}

/*****************************************************************************
*
*   eBrowseStyleSet
*
*   This API is used set the current category browsing style for this channel
*   list.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseStyleSet(
    CHANNELLIST_OBJECT hChannelList,
    BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
    void *pvBrowseCategoryCompareArg,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eErrorCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    BROWSE_CHANNEL_COMPARE_HANDLER bNewBrowseChannelCompareHandler;

    // Lock object for exclusive access
    bLocked =
        SMSO_bLock((SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
        BOOLEAN bOk = TRUE;
        BROWSE_TYPE_ENUM eBrowseType;
        void *pvNewBrowseChannelCompareArg;

        // Get our current browse type
        eBrowseType = BROWSE_eGetMode(
            psObj->hBrowse);

        // Only configure category handlers if
        // this channel list is of the correct type
        if (eBrowseType == BROWSE_TYPE_CATEGORY)
        {
            BROWSE_CATEGORY_COMPARE_HANDLER
                bNewBrowseCategoryCompareHandler;
            void *pvNewBrowseCategoryCompareArg;

            // Set the callback and handler into the object
            if (bBrowseCategoryCompareHandler != NULL)
            {
                bNewBrowseCategoryCompareHandler =
                    bBrowseCategoryCompareHandler;
                pvNewBrowseCategoryCompareArg =
                    pvBrowseCategoryCompareArg;
            }
            else
            {
                // use the default compare handler
                bNewBrowseCategoryCompareHandler =
                    bBrowseAllCategoriesCompareHandler;
                pvNewBrowseCategoryCompareArg = NULL;
            }

            // Set the category browse style
            bOk = BROWSE_bSetCategoryBrowseStyle (
                psObj->hBrowse,
                bNewBrowseCategoryCompareHandler,
                pvNewBrowseCategoryCompareArg );
        }

        // Set the channel compare handler and argument
        // regardless of type
        if (bBrowseChannelCompareHandler != NULL)
        {
            bNewBrowseChannelCompareHandler =
                bBrowseChannelCompareHandler;
            pvNewBrowseChannelCompareArg =
                pvBrowseChannelCompareArg;
        }
        else
        {
            // use the default compare handler
            bNewBrowseChannelCompareHandler =
                bBrowseAllChannelsCompareHandler;
            pvNewBrowseChannelCompareArg = NULL;
        }

        // Set the channel browse style
        bOk |= BROWSE_bSetChannelBrowseStyle(
            psObj->hBrowse,
            bNewBrowseChannelCompareHandler,
            pvNewBrowseChannelCompareArg );

        if (bOk == TRUE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   eBrowseCategory
*
*   This API is used to browse an existing channel category list through
*   categories. A browse to the next category causes the browsed category
*   numbers to be increased based on the current category browsing style.
*   Likewise a browse to the previous category causes the browsed category
*   numbers to be decreased based on the current category browsing style.
*   This API may be called when a valid CHANNELLIST handle is available.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseCategory (
    CHANNELLIST_OBJECT hChannelList,
    SMSAPI_DIRECTION_ENUM eDirection
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    CHANNELLIST_OBJECT_STRUCT *psObj;

    // If the caller did not specify to browse in the
    // up or down direction, we cannot browse using
    // this API
    if ( ( eDirection != SMSAPI_DIRECTION_NEXT ) &&
         ( eDirection != SMSAPI_DIRECTION_PREVIOUS ) )
    {
        // Error!
        return SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );

    if (psObj != NULL)
    {
        // Handle the browse request
        eReturnCode = eBrowseCategories(
            psObj,
            CATEGORY_INVALID_ID,
            eDirection );

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Notify callback if registered and change occurred
            SMSU_bNotify(&psObj->sEvent);
        }

        vUnlockSystem( psObj );
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eBrowseToCategory
*
*   This API is used to browse an existing channel category list to a direct
*   category identified by its category id. A direct browse causes the browsed
*   category list to have a reference channel based on the current category
*   browsing style. This API may be called when a valid CHANNELLIST handle
*   is available.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseToCategory (
    CHANNELLIST_OBJECT hChannelList,
    CATEGORY_ID tCategoryId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    CHANNELLIST_OBJECT_STRUCT *psObj;

    if( tCategoryId == CATEGORY_INVALID_ID )
    {
        // Error!
        return SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
    }

    // Lock the system and get our pointer
    psObj = psLockSystem( hChannelList );

    if(psObj != NULL)
    {
        // Handle the browse request
        eReturnCode = eBrowseCategories(
            psObj,
            tCategoryId,
            SMSAPI_DIRECTION_DIRECT );

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Notify callback if registered and change occurred
            SMSU_bNotify(&psObj->sEvent);
        }

        vUnlockSystem( psObj );
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   tSize
*
*   This API is used to report a channel list size to the caller. It reports
*   the current size of the list based on parameters of how it was created
*   and the number of channels actually available in the list. This API
*   helps applications define the range of offset values the channel list
*   has with respect to the reference channel within it. This API may be
*   called when a valid CHANNELLIST handle is available.
*
*****************************************************************************/
static size_t tSize (
    CHANNELLIST_OBJECT hChannelList,
    UN16 *pun16Before,
    UN16 *pun16After
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj =
        (CHANNELLIST_OBJECT_STRUCT *)hChannelList;
    UN32 un32Items = 0;
    BOOLEAN bLocked;

    // Verify and lock the channel list object
    bLocked = SMSO_bLock((SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL.eLinkedListItems(
            psObj->hChannelList, &un32Items);

        if(pun16Before != NULL)
        {
            *pun16Before = psObj->un16BeforeActual;
        }

        if(pun16After != NULL)
        {
            *pun16After = psObj->un16AfterActual;
        }

        // Unlock CHANNELLIST
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return (size_t)un32Items;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*******************************************************************************
*
*   CHANNELLIST_bProcessChannelListEventMaskChange
*
********************************************************************************/
BOOLEAN CHANNELLIST_bProcessChannelListEventMaskChange(
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // Verify and lock the channel list object
    bLocked = SMSO_bLock((SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj;

        psObj = (CHANNELLIST_OBJECT_STRUCT *)hChannelList;

        // Make the change to the event request mask
        // and see if it worked
        bSuccess = SMSU_bModifyRequestMask(  &psObj->sEvent,
                                            NULL,
                                            NULL,
                                            tEventMask,
                                            eModification
                                         );
        if (bSuccess == TRUE)
        {
            switch (eModification)
            {
                case SMSAPI_MODIFY_EVENT_MASK_REPLACE:
                {
                    psObj->tEventRequestMask = tEventMask;
                }
                break;
                case SMSAPI_MODIFY_EVENT_MASK_ENABLE:
                {
                    psObj->tEventRequestMask |= tEventMask;
                }
                break;
                case SMSAPI_MODIFY_EVENT_MASK_DISABLE:
                {
                    psObj->tEventRequestMask &= ~tEventMask;
                }
                break;
                default:
                {
                    printf(CHANNELLIST_OBJECT_NAME": eModification is out of bounds (%u) \n",eModification );
                }
                break;
            }
        }
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return bSuccess;
}

/*****************************************************************************
                             LOCAL STATIC FUNCTIONS
*****************************************************************************/

/*******************************************************************************
*
*   psLockSystem
*
*   This function is called in order to lock the system in a specific manner.
*   In order to prevent the possibility of a deadlock, the channellist must
*   first lock its decoder.  In order to do this, the channellist must
*   validate the channellist handle, extract the decoder, and then lock the
*   decoder, and then it can lock itself.
*
*   Whew.
*
*******************************************************************************/
static CHANNELLIST_OBJECT_STRUCT *psLockSystem (
    CHANNELLIST_OBJECT hChannelList
        )
{
    BOOLEAN bOk;

    // First validate the channel list
    bOk = SMSO_bValid((SMS_OBJECT)hChannelList);
    if (bOk == TRUE)
    {
        // Extract the object from the handle
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *) hChannelList;

        // Now, lock the decoder
        bOk = SMSO_bLock(
            (SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bOk == TRUE)
        {
            // Now, lock the channellist
            bOk = SMSO_bLock(
                (SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
            if (bOk == TRUE)
            {
                // All locked up!  Give 'em the pointer
                return psObj;
            }
            else
            {
                // Unlock the DECODER
                SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
            }
        }
    }

    return NULL;
}

/*******************************************************************************
*
*   vUnlockSystem
*
*   This function is called in order to unlock the system in a specific manner.
*   In order to prevent the possibility of a deadlock, we must first unlock the
*   channelist and then the decoder
*
*   Whew.
*
*******************************************************************************/
static void vUnlockSystem (
    CHANNELLIST_OBJECT_STRUCT *psObj
        )
{
    // Extract the decoder (probably a little
    // too conservative, but we're all bush fans here, right?)
    DECODER_OBJECT hDecoder = psObj->hDecoder;

    // Unlock the channel list
    SMSO_vUnlock((SMS_OBJECT)psObj);

    // Unlock the decoder
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    return;
}

/*******************************************************************************
*
*   bInitChannelList
*
*   This function is called whenever a new channel list object is created. It
*   populates basic channel list attributes and utilizes bCreateList to
*   ensure the list is populated with channels based upon its initialization
*   inputs.
*
*******************************************************************************/
static BOOLEAN bInitChannelList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    BROWSE_TYPE_ENUM eBrowseType,
    CHANNEL_ID tReferenceChannelId,
    CATEGORY_ID tReferenceCategoryId
        )
{
    BOOLEAN bLocked;
    BOOLEAN bSuccess = FALSE, bEmptyCatChanList = FALSE;

    // Lock our DECODER
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);

    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;
        CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
        CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
        SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

        do
        {
            // Get the cache handle
            hCCache = DECODER_hCCache(psObj->hDecoder);

            if (hCCache == CCACHE_INVALID_OBJECT)
            {
                // Error! CCache unavailable
                break;
            }

            if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
            {
                // Grab the reference channel handle
                hChannel = CCACHE_hChannelFromIds(
                    hCCache,
                    SERVICE_INVALID_ID,
                    tReferenceChannelId,
                    FALSE
                        );

                // Grab the broadcast category associated
                // with the reference channel
                hCategory = CHANNEL.hCategory(hChannel, 0);

                // Get this category's Id
                tReferenceCategoryId =
                    CATEGORY.tGetCategoryId( hCategory );
            }
            else if (eBrowseType == BROWSE_TYPE_CATEGORY)
            {
                size_t tCategorySize;

                // Grab the reference category handle
                hCategory = CCACHE_hCategory(
                    hCCache, &tReferenceCategoryId, 0 );

                // Verify we have a valid category
                if (hCategory == CATEGORY_INVALID_OBJECT)
                {
                    // Error -- Invalid category
                    break;
                }

                // Verify the category has some channels in it
                tCategorySize = CATEGORY.tSize(
                    psObj->hDecoder, tReferenceCategoryId );

                if (tCategorySize == 0)
                {
                    bEmptyCatChanList = TRUE;
                }

                // Get the requested channel
                hChannel = CCACHE_hChannelFromIds(
                    hCCache,
                    SERVICE_INVALID_ID,
                    tReferenceChannelId,
                    FALSE
                        );

                // is that a valid channel?
                if (hChannel != CHANNEL_INVALID_OBJECT)
                {
                    // yes. it is valid, but is it in the reference category?
                    N16 n16Offset;

                    eReturn = CHANNEL.eCategoryOffset(hChannel,
                                  tReferenceCategoryId, &n16Offset);
                }

                // check if we have a good reference channel
                // (valid and in reference category)
                if ((eReturn != SMSAPI_RETURN_CODE_SUCCESS) ||
                    (hChannel == CHANNEL_INVALID_OBJECT))
                {
                    // either the supplied channel id wasn't a valid channel
                    // or the channel isn't in the reference category
                    // so instead we'll get the first channel in the category
                    hChannel = CATEGORY_hGetChanHdlByOffset(
                                   hCategory, 0);

                    tReferenceChannelId = CHANNEL.tChannelId(hChannel);
                }
            }

            // Double check we have a valid channel
            if ((hChannel == CHANNEL_INVALID_OBJECT) && (bEmptyCatChanList == FALSE))
            {
                // Error -- Invalid channel
                break;
            }

            // Set the browse mode and reference Ids
            bSuccess = BROWSE_bSetMode(
                    psObj->hBrowse,
                    eBrowseType,
                    tReferenceChannelId,
                    tReferenceCategoryId );

            if (bSuccess == FALSE)
            {
                // Error! Browse object failure
                break;
            }

            // We need to get notified of ccache events
            bSuccess = CCACHE_bRegisterNotification (
                              hCCache,
                              CCACHE_OBJECT_EVENT_UPDATE,
                              vCCacheEventCallback,
                              (CHANNELLIST_OBJECT)psObj
                                  );

            if (bSuccess != TRUE)
            {
                // Error -- registration failed
                break;
            }

            // Register the channel list with the ccache for all channels
            bSuccess = CCACHE_bRegisterAllChannelsNotification(
                hCCache, CHANNEL_OBJECT_EVENT_CHANNEL_INFO, TRUE,
                vChannelInfoEventCallback, (void*)psObj);

            if (bSuccess != TRUE)
            {
                // Error -- registration failed
                break;
            }

            // Create a list...
            bSuccess = bCreateList(psObj, hCCache);

            if(bSuccess != TRUE)
            {
                // Error -- list creation failed
                break;
            }

            // Notify callback if registered and change occurred
            SMSU_bNotify(&psObj->sEvent);

        } while (0);

        // unlock decoder
        SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
    }

    return bSuccess;
}

/*******************************************************************************
*
*   vChannelInfoEventCallback
*
*   This callback function is used to register with all CHANNEL objects in
*   the CCACHE. This function will handle any changes that result from any of
*   the channels in the CCACHE having their channel information change

*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*******************************************************************************/
static void vChannelInfoEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    CHANNELLIST_OBJECT_STRUCT *psObj =
        (CHANNELLIST_OBJECT_STRUCT *)pvEventCallbackArg;
    BOOLEAN bLocked, bOwner, bRebuildList = FALSE;
    CHANNELLIST_ENTRY_STRUCT *psEntry;
    CHANNEL_ID tChannelId;

    do
    {
        // Lock CHANNELLIST object since this call is always made from
        // the context of the DECODER.
        bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            break;
        }

        if (psObj->bReady == FALSE)
        {
            break;
        }

        // Check caller ownership. Caller may only call this
        // API if it is in the context of the DECODER or it has the DECODER
        // mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);

        if (bOwner == FALSE)
        {
            break;
        }

        // Refresh our Browse mode.  The offset is saved in the
        // BROWSE_OBJECT_STRUCT.  I'm not sure why this is,
        // but it gets out of sync with channel line-up changes.
        BROWSE_bRefresh(psObj->hBrowse);

        // Fetch the channel ID for this channel,
        // used for the various try-browse scenerios
        tChannelId = CHANNEL.tChannelId(hChannel);

        // Fetch the entry for this channel in our list, if it exists
        psEntry = psGetChannelEntryByHandle(psObj, hChannel);

        if (psEntry != NULL)
        {
            if ((tEventMask & CHANNEL_OBJECT_EVENT_REMOVED) != 0)
            {
                // A channel was removed, make sure to record this
                SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_REMOVED);

                printf("Channellist member channel (%u) removed. Rebuild List\n", tChannelId);
                // Flag that the list should be re-built
                bRebuildList = TRUE;
            }
            else if ((tEventMask & CHANNEL_OBJECT_EVENT_CHANNEL_ID) != 0)
            {
                // The channel ID changed for a channel in our channel list
                // changed. The offsets are probably whacked now in our browse
                // object.  Re-build it
                printf("Channellist member channel (%u) remapped. Rebuild List\n", tChannelId);
                bRebuildList = TRUE;
            }
            else
            {
                BOOLEAN bBrowseDirect;

                // Try a direct browse to see if the channel still meets the
                // list critera
                bBrowseDirect =
                    BROWSE_bTryBrowse(
                        psObj->hBrowse,
                        tChannelId,
                        CATEGORY_INVALID_ID
                            );

                if (bBrowseDirect == FALSE)
                {
                    // We can no longer browse directly to the channel in our
                    // list using the specified compare handlers.  We need to
                    // re-build the list so that this channel is no longer a
                    // member
                    printf("Channellist member channel (%u) no longer eligible. Rebuild List\n", tChannelId);
                    bRebuildList = TRUE;
                }
                else
                {
                    // The structure of the channel list didn't change, so
                    // notify the application
                    psEntry->tMask |= tEventMask;

                    // Copy in the event mask to our update object
                    SMSU_tUpdate(&psObj->sEvent, tEventMask);

                    // Notify callback if registered and change occurred
                    SMSU_bNotify(&psObj->sEvent);
                }
            }
        }
        else
        {
            bRebuildList =
                BROWSE_bTryBrowse(
                    psObj->hBrowse,
                    tChannelId,
                    CATEGORY_INVALID_ID
                        );
            if (bRebuildList == TRUE)
        {
                 printf("channel (%u) is eligible. Rebuild List\n", tChannelId);
            }
        }

        if (bRebuildList == TRUE)
        {
            // I guess we can just do a re-browse for now since this
            // change affected our channel list in some fashion?
            // The re-browse process doesn't give the event masks a clean look,
            // but at least it indicates that a lot has changed and they should
            // re-evaluate it.
            // Need to figure out a better method here to rebuild the list
            printf("Rebuild List\n");
            eBrowseList((CHANNELLIST_OBJECT)psObj, 0);
        }

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Unlock CHANNELLIST
        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return;
}

/*******************************************************************************
*
*   vChannelMetaEventCallback
*
*   This callback function is used to register with all CHANNEL objects in
*   that are in the CHANNELLIST.  Since metadata doesn't affect the makeup of the
*   CHANNELLIST, this function simply records the event changes and notifies the
*   Application if any changes they care about were made.
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*******************************************************************************/
static void vChannelMetadataEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    CHANNELLIST_OBJECT hChannelList =
        (CHANNELLIST_OBJECT)pvEventCallbackArg;
    BOOLEAN bLocked, bOwner;

    // Lock CHANNELLIST object since this call is always made from
    // the context of the DECODER.
    bLocked = SMSO_bLock((SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;

        if (psObj->bReady == FALSE)
        {
            SMSO_vUnlock((SMS_OBJECT)hChannelList);
            return;
        }

        // Check caller ownership. Caller may only call this
        // API if it is in the context of the DECODER or it has the DECODER mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if(bOwner == TRUE)
        {
            CHANNELLIST_ENTRY_STRUCT *psEntry;

            // Find this channel in our channel list
            psEntry = psGetChannelEntryByHandle(psObj, hChannel);

            if (psEntry != NULL)
            {
                // Save the mask in this channel's entry, but make sure
                // we are setting values
                psEntry->tMask |= tEventMask;

                // Copy in the event mask to our update object
                SMSU_tUpdate(&psObj->sEvent, tEventMask);

                // Notify callback if registered and change occurred
                SMSU_bNotify(&psObj->sEvent);
            }
        }

        // Unlock CHANNELLIST
        SMSO_vUnlock((SMS_OBJECT)hChannelList);
    }

    return;
}

/*******************************************************************************
*
*   vCCacheEventCallback
*
*   This callback function is used to registered with the CCACHE object which
*   is associated with a specific CHANNELLIST. The purpose of this function is
*   to simply apply any CCACHE object event to the entire CHANNELLIST object.
*
*   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
*
*****************************************************************************/
static void vCCacheEventCallback (
    CCACHE_OBJECT hCCache,
    CCACHE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    CHANNELLIST_OBJECT hChannelList =
        (CHANNELLIST_OBJECT)pvEventCallbackArg;
    BOOLEAN bLocked, bOwner;

    // Lock CHANNELLIST object since this call is always made from
    // the context of the DECODER.
    bLocked = SMSO_bLock((SMS_OBJECT)hChannelList, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNELLIST_OBJECT_STRUCT *psObj =
            (CHANNELLIST_OBJECT_STRUCT *)hChannelList;

        // Check caller ownership. Caller may only call this
        // API if it is in the context of the DECODER or it has the DECODER mutex.
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if(bOwner == TRUE)
        {
            if (tEventMask & CCACHE_OBJECT_EVENT_UPDATE)
            {
                // Force the list to re-browse itself.
                eBrowseList((CHANNELLIST_OBJECT)psObj, 0);
            }
        }

        // Unlock CHANNELLIST
        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return;
}

/*******************************************************************************
*
*   vCategoryEventCallback
*
*   This callback is invoked when our reference category has been updated.
*   If the channellist is operating in all channels mode, we'll browse to
*   0 no matter what the event is (update or destroy), and if the channellist
*   is operating in category mode, we'll browse to 0 if an update occurs,
*   or we'll browse to the first available category if this category has
*   been destroyed.
*
********************************************************************************/
static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CHANNELLIST_OBJECT_STRUCT *psObj =
        (CHANNELLIST_OBJECT_STRUCT *)pvEventCallbackArg;
    BOOLEAN bOk;

    // Validate the channellist object
    bOk = SMSO_bValid((SMS_OBJECT)psObj);

    if (bOk != TRUE)
    {
        // Invalid handle
        return;
    }

    // Check that we have ownership of the decoder
    bOk = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);

    if (bOk != TRUE)
    {
        // In the wrong context
        return;
    }

    // Lock the channel list object since this call is always made from
    // the context of the DECODER.
    bOk = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);

    // Verify lock
    if (bOk == TRUE)
    {
        BROWSE_TYPE_ENUM eBrowseType;

        // How are we operating?
        eBrowseType = BROWSE_eGetMode( psObj->hBrowse );

        // Determine how to continue by our operating mode
        if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
        {
            if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                            == CATEGORY_OBJECT_EVENT_REMOVED)
            {
                // Force the list to browse forward because
                // we know the current reference channel no
                // longer has a broadcast category
                eBrowseList((CHANNELLIST_OBJECT)psObj, 1);
            }
            else if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS)
                                 == CATEGORY_OBJECT_EVENT_CONTENTS)
            {
                // Force the list to re-browse itself
                eBrowseList((CHANNELLIST_OBJECT)psObj, 0);
            }
        }
        else if (eBrowseType == BROWSE_TYPE_CATEGORY)
        {
            if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                            == CATEGORY_OBJECT_EVENT_REMOVED)
            {
                CCACHE_OBJECT hCCache;

                // Get the channel cache
                hCCache = DECODER_hCCache( psObj->hDecoder );

                if (hCCache != CCACHE_INVALID_OBJECT)
                {
                    CATEGORY_ID tFirstCategoryId =
                        CATEGORY_INVALID_ID;

                    // Get the first category's Id
                    CCACHE_hCategory(
                        hCCache, &tFirstCategoryId, 1 );

                    // Browse to the first category
                    // (it that fails, we'll browse
                    // to the first category we can)
                    eBrowseToCategory(
                        (CHANNELLIST_OBJECT)psObj, tFirstCategoryId);
                }
            }
            else if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS)
                                 == CATEGORY_OBJECT_EVENT_CONTENTS)
            {
                // Force the list to re-browse itself
                // (within this category)
                eReturnCode = eBrowseList((CHANNELLIST_OBJECT)psObj, 0);

                // Did that work out?
                if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    // Browsing within this category failed.
                    // Try the next one
                    eBrowseCategory(
                        (CHANNELLIST_OBJECT)psObj, SMSAPI_DIRECTION_NEXT);
                }

            }
        }

        SMSO_vUnlock((SMS_OBJECT)psObj);
    }
    return;
}

/*****************************************************************************
*
*   bCreateList
*
*   This function prepares a channel list for creation, and then
*   invokes bPopulateChannelList to get the work done.
*
*****************************************************************************/
static BOOLEAN bCreateList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CCACHE_OBJECT hCCache
        )
{
    size_t tAvailable = 0;
    BROWSE_TYPE_ENUM eBrowseType;
    CHANNEL_ID tReferenceChannelId;
    CATEGORY_ID tReferenceCategoryId;
    BOOLEAN bOk;

    psObj->bReady = FALSE;

    // Get the currently browsed Ids
    bOk = BROWSE_bGetBrowsed(
        psObj->hBrowse,
        &tReferenceChannelId,
        &tReferenceCategoryId );

    // Get the current browse type
    eBrowseType = BROWSE_eGetMode( psObj->hBrowse );

    // Determine which type of list this is and configure list create
    // parameters based on that.
    if (eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        // Determine how many channels there are in this category
        tAvailable =
            CATEGORY.tSize( psObj->hDecoder, tReferenceCategoryId );
    }
    else if(eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
    {
        N16 n16Available = 0;
        // Determine the total number of channels possible
        CCACHE_vStats(hCCache, &n16Available, NULL);
        tAvailable = (size_t)n16Available;
    }
    else
    {
        // Error! Unknown type
        return FALSE;
    }

    if ( (tAvailable == 0) &&
         (eBrowseType == BROWSE_TYPE_ALL_CHANNELS) )
    {
        // Error! No channels available
        return FALSE;
    }

    // Browse to the reference channel
    bOk = BROWSE_bBrowse(
        psObj->hBrowse,
        SMSAPI_DIRECTION_DIRECT,
        TRUE,
        &tReferenceChannelId,
        &tReferenceCategoryId,
        TRUE);

    if (bOk == FALSE)
    {
        // The reference channel doesn't meet our browse style, look for the
        // first channel that does, in the forward direction.
        bOk = BROWSE_bBrowse(
            psObj->hBrowse,
            SMSAPI_DIRECTION_NEXT,
            TRUE,
            &tReferenceChannelId,
            &tReferenceCategoryId,
            TRUE);

        if (bOk == FALSE)
        {
            // we couln't find any suitable reference channel. But that's ok.
            // there may be channels which will later satisfy the list, for
            // example the list could be for skipped channels, and there aren't
            // any now, but later there may be.
            psObj->bReady = TRUE;
            return TRUE;
        }
    }

    // Are we in category mode?
    if (eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        // Only handle category registrations when we are in
        // category mode

        // Unregister notification of previous reference category
        if (psObj->hReferenceCategory != CATEGORY_INVALID_OBJECT)
        {
            // Unregister with the previous reference category
            CATEGORY_vUnregisterNotification(
                psObj->hReferenceCategory,
                vCategoryEventCallback,
                (void *)(size_t)psObj );
        }

        // Get the new reference category's object handle
        psObj->hReferenceCategory =
            CCACHE_hCategory( hCCache, &tReferenceCategoryId, 0 );

        if (psObj->hReferenceCategory == CATEGORY_INVALID_OBJECT)
        {
            // Error! New reference category is invalid
            return FALSE;
        }

        // Now, register the use of the current reference category
        bOk = CATEGORY_bRegisterNotification(
            psObj->hReferenceCategory,
            CATEGORY_OBJECT_EVENT_ALL,
            vCategoryEventCallback,
            (void *)(size_t)psObj );

        // Were we able to register?
        if (bOk != TRUE)
        {
            return FALSE;
        }
    }

    // Now, generate the requested channel list...
    bOk = bPopulateChannelList(
        psObj, hCCache, tAvailable);
    if (bOk == FALSE)
    {
        if (eBrowseType == BROWSE_TYPE_CATEGORY)
        {
            if (tAvailable == 0)
            {
                psObj->bReady = TRUE;

                // Indicate the channel list has been modified
                SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_INITIAL);

                bOk = TRUE;
            }
            else
            {
                // Unregister the category
                CATEGORY_vUnregisterNotification(
                    psObj->hReferenceCategory,
                    vCategoryEventCallback, (void *)(size_t)psObj );
            }
        }
    }
    else
    {
        psObj->bReady = TRUE;
    }
#if DEBUG_OBJECT == 1

    // DEBUG
    {
        CHANNEL_ID tChannelId;
        CHANNELLIST_ENTRY_STRUCT *psEntry;
        OSAL_LINKED_LIST_ENTRY hChannelEntry =
            psObj->hReferenceChannelEntry, hFirstChannelEntry;
        UN16 un16BeforeActual = psObj->un16BeforeActual;

        printf("\nChannel List:\n\n");
        psEntry = (CHANNELLIST_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(psObj->hTopChannelEntry);
        tChannelId = CHANNEL.tChannelId(psEntry->hChannel);
        printf("Top Channel: \t\t%u\n", tChannelId);
        psEntry = (CHANNELLIST_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(psObj->hReferenceChannelEntry);
        tChannelId = CHANNEL.tChannelId(psEntry->hChannel);
        printf("Reference Channel: \t%u\n", tChannelId);
        psEntry = (CHANNELLIST_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(psObj->hBottomChannelEntry);
        tChannelId = CHANNEL.tChannelId(psEntry->hChannel);
        printf("Bottom Channel: \t%u\n\n", tChannelId);
        while(un16BeforeActual-- > 0) // find first one
        {
            hChannelEntry = OSAL.hLinkedListPrev(hChannelEntry, (void**)NULL);
        };
        psEntry = (CHANNELLIST_ENTRY_STRUCT *)
            OSAL.pvLinkedListThis(hChannelEntry);
        hFirstChannelEntry = hChannelEntry;
        do
        {
            tChannelId = CHANNEL.tChannelId(psEntry->hChannel);
            if(tChannelId == tReferenceChannelId) printf("\n*");
            printf("Channel #%u\n", tChannelId);
            if(tChannelId == tReferenceChannelId) printf("\n");
            hChannelEntry =
                OSAL.hLinkedListNext(hChannelEntry, (void**)&psEntry);

        } while(hChannelEntry != hFirstChannelEntry);
        printf("\n");
    }

#endif // DEBUG_OBJECT == 1

    // Done!
    return bOk;
}

/*****************************************************************************
*
*   bPopulateChannelList
*
*   This function utilizes the browse object in order to populate the
*   channel list based upon the filtering rules given to us previously.
*   The reference channel is first added to the list, followed by the
*   channels which appear before the reference, then the channels which
*   appear after the reference are added.
*
*****************************************************************************/
static BOOLEAN bPopulateChannelList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CCACHE_OBJECT hCCache,
    size_t tAvailable
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_OBJECT hChannel;
    OSAL_LINKED_LIST_ENTRY hChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hBottom = OSAL_INVALID_LINKED_LIST_ENTRY,
                           hTop = OSAL_INVALID_LINKED_LIST_ENTRY;
    UN16 un16RequestedChannelsInList;
    UN16 un16MaxChannelsBefore, un16MaxChannelsAfter;
    BOOLEAN bInclude, bOk;
    BOOLEAN bReduced;
    UN16 un16CurrentChan;
    CATEGORY_ID tReferenceCategoryId;
    CATEGORY_ID tBrowsedCategoryId;
    CHANNEL_ID tReferenceChannelId;
    CHANNEL_ID tBrowsedChannelId;
    size_t tNumberBrowses = 0;
    BOOLEAN bSuccess;
    CATEGORY_OBJECT hCategory;
    OSAL_OBJECT_HDL hLL;

    // Get the currently browsed Ids
    bOk = BROWSE_bGetBrowsed(
        psObj->hBrowse,
        &tReferenceChannelId,
        &tReferenceCategoryId );

    if (bOk == FALSE)
    {
        return FALSE;
    }

    // Reset channel list state
    psObj->un16AfterActual = 0;
    psObj->un16BeforeActual = 0;

    // Initialize our list creation variables
    tBrowsedChannelId = tReferenceChannelId;
    tBrowsedCategoryId = tReferenceCategoryId;

    // Now, determine how large to make the list
    un16RequestedChannelsInList = psObj->un16RequestedAfter
        + psObj->un16RequestedBefore + 1;
    un16MaxChannelsBefore = psObj->un16RequestedBefore;
    un16MaxChannelsAfter = psObj->un16RequestedAfter;

    // If we need to reduce the number
    // of channels in this list then get to it
    while (un16RequestedChannelsInList > tAvailable)
    {
        bReduced = FALSE;

        // Determine if we can reduce the
        // number of channels which appear
        // before the reference
        if (un16MaxChannelsBefore > 0)
        {
            un16MaxChannelsBefore--;
            un16RequestedChannelsInList--;
            bReduced = TRUE;
        }

        // Determine if we can reduce the
        // number of channels which appear
        // after the reference
        if ((un16RequestedChannelsInList > tAvailable) &&
            (un16MaxChannelsAfter > 0))
        {
            un16MaxChannelsAfter--;
            un16RequestedChannelsInList--;
            bReduced = TRUE;
        }

        // Did we reduce anything?
        if (bReduced == FALSE)
        {
            // Nope! We can go no further
            break;
        }
    }

    // Now check that things worked out
    if (un16RequestedChannelsInList > tAvailable)
    {
        // We can reduce no further!
        return FALSE;
    }

    // Make sure we don't have something in the 'old' channel 
    // set and destroy all remaining entries in the old channel set.

    // Since channel list assembly based on the sequential
    // algorithm, there could be some unused entries remaining
    // from the previous time.
    // Later, this list will be swapped with the original one,
    // so channel list will be built on what is currently in 
    // this set.
    vClearChannelSet(psObj, psObj->hChannelListOld);

    // Unregister all notifications from the current channel set
    // to avoid increment of usage counter.
    vUnregisterChannelNotifications(psObj, psObj->hChannelList);

    // Swap channel sets.
    hLL = psObj->hChannelList;
    psObj->hChannelList = psObj->hChannelListOld;
    psObj->hChannelListOld = hLL;

    // At this point, the 'original' channel set will 
    // be treated as 'old', and the 'old' one (empty at 
    // this moment) will be treated as 'new'.

    // Set cursor position at the reference entry 
    // in the old channel set.
    psObj->hCursor = psObj->hReferenceChannelEntry;

    // The reference channel has already been browsed, so
    // add it to the list of channels
    hChannel = CCACHE_hChannelFromIds(
        hCCache, SERVICE_INVALID_ID, tReferenceChannelId, FALSE );
    if (hChannel == NULL)
    {
        // Error! Reference channel handle invalid
        return FALSE;
    }

    // Add the reference channel to the list
    eReturnCode = eAddChannel(
        psObj, hChannel, &hChannelEntry, FALSE);

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error! Unable to update list
        return FALSE;
    }

    // Register a notification telling the user when
    // updates occur for this channel
    bSuccess =
        CHANNEL_bRegisterNotification(
            hChannel,
            CHANNEL_OBJECT_EVENT_METADATA,
            vChannelMetadataEventCallback,
            psObj, 
            FALSE);

    // If we couldn't register the notification
    if(bSuccess == FALSE)
    {
        // Error!
        eRemoveChannelEntry(hChannelEntry);
        return FALSE;
    }

    // We have successfully added a reference channel 
    // entry in the new channel set. Need to set top and bottom
    // at the reference entry since there might be no after 
    // and before.

    // Set reference, top and bottom channel entries
    psObj->hReferenceChannelEntry = hBottom = hTop = hChannelEntry;

    // Start working on the "after" channels
    if (un16MaxChannelsAfter > 0)
    {
        for ( un16CurrentChan = 0;
              un16CurrentChan < tAvailable;
              un16CurrentChan++)
        {
            // Browse to the previous channel
            bInclude = BROWSE_bBrowse(
                psObj->hBrowse,
                SMSAPI_DIRECTION_NEXT,
                TRUE,
                &tBrowsedChannelId,
                &tBrowsedCategoryId,
                TRUE);

            hCategory = CCACHE_hCategory(hCCache, &tBrowsedCategoryId, 0);
            // Keep track of the number of times
            // we browse around
            tNumberBrowses++;

            if (bInclude == TRUE)
            {
                BROWSE_TYPE_ENUM eType;

                // Grab the channel handle
                hChannel = CCACHE_hChannelFromIds(
                    hCCache, SERVICE_INVALID_ID, tBrowsedChannelId, FALSE );
                if (hChannel == NULL)
                {
                    // Error! Reference channel handle invalid
                    return FALSE;
                }

                eType = BROWSE_eGetMode(psObj->hBrowse);
                if (eType == BROWSE_TYPE_CATEGORY)
                {
                    // we need to find out how many times this channel is already in the LL
                    CHANNELLIST_ITERATOR_STRUCT sIterator;
                    N16 n16OccurrenceIndex;

                    sIterator.hChannel = hChannel;
                    sIterator.un16Count = 0;

                    eReturnCode = OSAL.eLinkedListIterate (
                                    psObj->hChannelList,
                                    bChannelCountIterator,
                                    (void *)&sIterator);

                    if ((eReturnCode != OSAL_SUCCESS) &&
                        (eReturnCode != OSAL_NO_OBJECTS))
                    {
                        printf(CHANNELLIST_OBJECT_NAME
                            ": failed to iterate the list %p (%s)",
                            psObj->hChannelList,
                            OSAL.pacGetReturnCodeName(eReturnCode));
                    }

                    // we need to find out how many times this channel is in the category.
                    // if it is in multiple times we can include in the list multiple times
                    // otherwise we can't
                    n16OccurrenceIndex =
                       CATEGORY_n16ChannelOccurrence(hCategory, hChannel, N16_MAX);
                    if (n16OccurrenceIndex <= sIterator.un16Count)
                    {
                        // can't add more times than the channel occurs in the
                        // category
                        break;
                    }
                }

                hChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                // Add this channel to the channel list
                eReturnCode = eAddChannel(
                    psObj, hChannel, &hChannelEntry, FALSE);
                if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
                {
                    // we must have wrapped around and found a channel already
                    // in the list.
                    break;
                }
                else if(eReturnCode != OSAL_SUCCESS)
                {
                    // Error!
                    return FALSE;
                }

                // Bottom position should be adjusted only using
                // unique entries.
                hBottom = hChannelEntry;

                // Make sure to register a notification telling the user when
                // updates occur for this channel
                CHANNEL_bRegisterNotification(
                        hChannel,
                        CHANNEL_OBJECT_EVENT_METADATA,
                        vChannelMetadataEventCallback,
                        psObj, 
                        FALSE);

                psObj->un16AfterActual++;
                if (psObj->un16AfterActual == un16MaxChannelsAfter)
                {
                    // All done with channels
                    // which appear before the reference
                    break;
                }

            }
        }
    }

    // Get back to where we started
    while (tNumberBrowses > 0)
    {
        // Now, get back to the reference channel
        bOk = BROWSE_bBrowse(
            psObj->hBrowse,
            SMSAPI_DIRECTION_PREVIOUS,
            TRUE,
            &tBrowsedChannelId,
            &tBrowsedCategoryId,
            TRUE);

        if (bOk == FALSE)
        {
            return FALSE;
        }

        tNumberBrowses--;
    }

    // Adjust cursor position.
    if (psObj->hCursor == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Since cursor originally was set at the reference 
        // entry, at this moment it should pass entries which
        // is 'after' the reference.
        // If a new channel set is bigger than the original one,
        // a new channel entry will be allocated and cursor will
        // be set to nothing. In this case, need to set it's position
        // to the current 'bottom' position in the old channel 
        // set (the nearest entry before the old reference position).
        psObj->hCursor = psObj->hBottomChannelEntry;

#if DEBUG_OBJECT==1
        {
            CHANNEL_OBJECT hChannelCursor = CHANNEL_INVALID_OBJECT;
            CHANNELLIST_ENTRY_STRUCT *psCursor;

            psCursor = OSAL.pvLinkedListThis(psObj->hCursor);
            if (psCursor != NULL)
            {
                hChannelCursor = psCursor->hChannel;
            }

            printf(CHANNELLIST_OBJECT_NAME
                ": Setting cursor at the Channel ID: %d\n",
                CHANNEL.tChannelId(hChannelCursor));
        }
#endif // DEBUG_OBJECT==1
            
    }
    else
    {
        // If cursor points to something, then it means
        // that the new channel set is smaller and the nearest
        // entry before the old reference position is 
        // right behind cursor.
        psObj->hCursor = OSAL.hLinkedListPrev(
            psObj->hCursor, (void **)NULL);
    }

    // We have done with 'after' entries. So we may set a 
    // bottom position in the new channel set.
    psObj->hBottomChannelEntry = hBottom;

#if DEBUG_OBJECT==1
    {
        CHANNELLIST_ENTRY_STRUCT *psEntry;

        if (psObj->hBottomChannelEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            psEntry = (CHANNELLIST_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(psObj->hBottomChannelEntry);

            if (psEntry != NULL)
            {
                printf(CHANNELLIST_OBJECT_NAME
                    ": Setting new Bottom at the Channel ID: %d\n",
                    CHANNEL.tChannelId(psEntry->hChannel));
            }
        }
    }
#endif // DEBUG_OBJECT==1

    hCategory = CCACHE_hCategory(hCCache, &tBrowsedCategoryId, 0);
    // We'll be adding channels before the reference
    // now, so initialize it as our entry
    hChannelEntry = psObj->hReferenceChannelEntry;

    // do the "before" channels
    if (un16MaxChannelsBefore > 0)
    {
        for ( un16CurrentChan = 0;
              un16CurrentChan < tAvailable;
              un16CurrentChan++)
        {
            // Browse to the previous channel
            bInclude = BROWSE_bBrowse(
                psObj->hBrowse,
                SMSAPI_DIRECTION_PREVIOUS,
                TRUE,
                &tBrowsedChannelId,
                &tBrowsedCategoryId,
                TRUE);

            // Keep track of the number of times
            // we browse around
            tNumberBrowses++;

            if (bInclude == TRUE)
            {
                BROWSE_TYPE_ENUM eType;

                // Grab the channel handle
                hChannel = CCACHE_hChannelFromIds(
                    hCCache, SERVICE_INVALID_ID, tBrowsedChannelId, FALSE );
                if (hChannel == NULL)
                {
                    // Error! Reference channel handle invalid
                    return FALSE;
                }

                eType = BROWSE_eGetMode(psObj->hBrowse);
                if (eType == BROWSE_TYPE_CATEGORY)
                {
                    // we need to find out how many times this channel is already in the LL
                    CHANNELLIST_ITERATOR_STRUCT sIterator;
                    N16 n16OccurrenceIndex;

                    sIterator.hChannel = hChannel;
                    sIterator.un16Count = 0;

                    eReturnCode = OSAL.eLinkedListIterate (
                                    psObj->hChannelList,
                                    bChannelCountIterator,
                                    (void *)&sIterator);

                    if ((eReturnCode != OSAL_SUCCESS) &&
                        (eReturnCode != OSAL_NO_OBJECTS))
                    {
                        printf(CHANNELLIST_OBJECT_NAME
                            ": failed to iterate the list %p (%s)",
                            psObj->hChannelList,
                            OSAL.pacGetReturnCodeName(eReturnCode));
                    }

                    // we need to find out how many times this channel is in the category.
                    // if it is in multiple times we can include in the list multiple times
                    // otherwise we can't
                    n16OccurrenceIndex =
                       CATEGORY_n16ChannelOccurrence (
                        hCategory, hChannel, N16_MAX );
                    if (n16OccurrenceIndex <= sIterator.un16Count)
                    {
                        // can't add more times than the channel occurs in the
                        // category
                        break;
                    }
                }

                // Add this channel to the channel list
                eReturnCode = eAddChannel(
                    psObj, hChannel, &hChannelEntry, TRUE);
                if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
                {
                    // we must have wrapped around and found a channel already
                    // in the list.
                    break;
                }
                else if(eReturnCode != OSAL_SUCCESS)
                {
                    // Error!
                    return FALSE;
                }

                // Should set top entry only for unique enties.
                hTop = hChannelEntry;

                // Make sure to register a notification telling the user when
                // updates occur for this channel
                CHANNEL_bRegisterNotification(
                        hChannel,
                        CHANNEL_OBJECT_EVENT_METADATA,
                        vChannelMetadataEventCallback,
                        psObj, 
                        FALSE);

                psObj->un16BeforeActual++;

                if (psObj->un16BeforeActual == un16MaxChannelsBefore)
                {
                    // All done with channels
                    // which appear before the reference
                    break;
                }
            }
        }
    }

    // Get back to where we started
    while (tNumberBrowses > 0)
    {
        // Now, get back to the reference channel
        bOk = BROWSE_bBrowse(
            psObj->hBrowse,
            SMSAPI_DIRECTION_NEXT,
            TRUE,
            &tBrowsedChannelId,
            &tBrowsedCategoryId,
            TRUE);

        if (bOk == FALSE)
        {
            return FALSE;
        }

        tNumberBrowses--;
    }

    // We have done with 'before' channels. So we can 
    // set top position in the new channel set.
    psObj->hTopChannelEntry = hTop;

#if DEBUG_OBJECT==1
    {
        CHANNELLIST_ENTRY_STRUCT *psEntry;

        if (psObj->hTopChannelEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            psEntry = (CHANNELLIST_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(psObj->hTopChannelEntry);

            if (psEntry != NULL)
            {
                printf(CHANNELLIST_OBJECT_NAME
                    ": Setting new Top at the Channel ID: %d\n",
                    CHANNEL.tChannelId(psEntry->hChannel));
            }
        }
    }
#endif // DEBUG_OBJECT==1

#if DEBUG_OBJECT==1

    OSAL.eLinkedListIterate(
        psObj->hChannelList,
        bPrintList,
        (void *)NULL);

#endif // DEBUG_OBJECT==1

    // Unregister and destroy all remaining channel entries
    vClearChannelSet(psObj, psObj->hChannelListOld);

    // Done
    return TRUE;
}

#if DEBUG_OBJECT==1

/*****************************************************************************
*
*   bPrintList
*
*****************************************************************************/
static BOOLEAN bPrintList(
    void *pvData,
    void *pvArg
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry = 
        (CHANNELLIST_ENTRY_STRUCT *)pvData;

    printf(CHANNELLIST_OBJECT_NAME": Channel ID: %3d, Mask: %8.8X\n", 
        CHANNEL.tChannelId(psEntry->hChannel), 
        psEntry->tMask);

    return TRUE;
}

#endif // DEBUG_OBJECT==1

/*****************************************************************************
*
*   bUnregisterChannelNotificationIterator
*
*****************************************************************************/
static BOOLEAN bUnregisterChannelNotificationIterator(
    void *pvData,
    void *pvArg
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry =
        (CHANNELLIST_ENTRY_STRUCT *)pvData;
    CHANNELLIST_OBJECT_STRUCT *psObj =
        (CHANNELLIST_OBJECT_STRUCT *)pvArg;

    if (psEntry != NULL)
    {
        CHANNEL_vUnregisterNotification(
            psEntry->hChannel, vChannelMetadataEventCallback, pvArg);

        // Clear the preset handle
        vUpdateChannelPresetHandle(psObj, psEntry->hChannel, 0, TRUE);
    }
    return TRUE;
}

/*****************************************************************************
*
*   bChannelCountIterator
*
*****************************************************************************/
static BOOLEAN bChannelCountIterator(
    void *pvData,
    void *pvArg
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry =
        (CHANNELLIST_ENTRY_STRUCT *)pvData;
    CHANNELLIST_ITERATOR_STRUCT *psIterator =
        (CHANNELLIST_ITERATOR_STRUCT *)pvArg;

    if (psEntry != NULL)
    {
        if (psEntry->hChannel == psIterator->hChannel)
        {
            (psIterator->un16Count)++;
        }
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bUpdateChannelMask
*
*****************************************************************************/
static BOOLEAN bUpdateChannelMask(
    CHANNELLIST_ENTRY_STRUCT *psEntry,
    void *pvArg
        )
{
    CHANNEL_EVENT_MASK tMask =
        (CHANNEL_EVENT_MASK)(size_t)pvArg;

    if (psEntry != NULL)
    {
        psEntry->tMask |= tMask;
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   eBrowseCategories
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseCategories (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId,
    SMSAPI_DIRECTION_ENUM eDirection )
{
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    CCACHE_OBJECT hCCache = CCACHE_INVALID_OBJECT;
    BROWSE_TYPE_ENUM eBrowseType;
    BOOLEAN bCreated;
    BOOLEAN bOk;
    CHANNEL_ID tBrowsedChannelId = CHANNEL_INVALID_ID;
    CATEGORY_ID tBrowsedCategoryId = tCategoryId;
    size_t tBrowsedCategorySize;
    N16 n16NumCategories = 0;

    // Determine our browsing mode
    eBrowseType = BROWSE_eGetMode( psObj->hBrowse );

    // Verify inputs
    if (eBrowseType != BROWSE_TYPE_CATEGORY )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // From the DECODER handle, determine the channel cache handle
    hCCache = DECODER_hCCache( psObj->hDecoder );

    if ( hCCache == CCACHE_INVALID_OBJECT )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Get the number of categories
    CCACHE_vStats( hCCache, NULL, &n16NumCategories );
    if (n16NumCategories <= 0)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // We may need to do some extra browsing
    // to handle error cases... but
    // make sure we cap it off at n16NumCategories
    while (n16NumCategories-- != 0)
    {
        // Perform the requested category browse
        bOk = BROWSE_bBrowse(
            psObj->hBrowse, eDirection,
            FALSE, &tBrowsedChannelId,
            &tBrowsedCategoryId,
            TRUE);
        if (bOk == FALSE)
        {
            psObj->hReferenceCategory = CCACHE_hCategory(hCCache, &tBrowsedCategoryId, 0);

            // clear out the list
            vClearChannellist( psObj );

            // Now, register the use of the current reference category
            bOk = CATEGORY_bRegisterNotification(
                psObj->hReferenceCategory,
                CATEGORY_OBJECT_EVENT_ALL,
                vCategoryEventCallback,
                (void *)(size_t)psObj );

            // Were we able to register?
            if (bOk != TRUE)
            {
                return SMSAPI_RETURN_CODE_ERROR;
            }

            // Indicate the channel list has been modified
            SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_INITIAL);

            // Notify callback if registered and change occurred
            SMSU_bNotify(&psObj->sEvent);

            return SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Get the browsed category's handle
        hCategory = CCACHE_hCategory(
            hCCache, &tBrowsedCategoryId, 0 );

        // Verify category handle first
        if ( hCategory == CATEGORY_INVALID_OBJECT)
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }

        // Get the category's size
        tBrowsedCategorySize = CATEGORY.tSize(
            psObj->hDecoder, tBrowsedCategoryId );

        // If this category is of size 0,
        // then we can't make a channel list for it
        if (tBrowsedCategorySize > 0)
        {
            // Category has some channels in
            // it -- we're done browsing
            break;
        }

        // If this was a direct browse, we'll
        // need to change it to a relative browse
        if (eDirection == SMSAPI_DIRECTION_DIRECT)
        {
            eDirection = SMSAPI_DIRECTION_NEXT;
        }
    }

    // Re-create a category list based upon the reference...
    bCreated = bCreateList( psObj, hCCache );

    if( bCreated == FALSE )
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   vClearChannellist
*
*****************************************************************************/
static void vClearChannellist(
    CHANNELLIST_OBJECT_STRUCT *psObj
        )
{
    // Initialize reference channel info
    psObj->hReferenceChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hTopChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hBottomChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    vClearChannelSet(psObj, psObj->hChannelList);
    vClearChannelSet(psObj, psObj->hChannelListOld);

    return;
}

/*****************************************************************************
*
*   vUnregisterChannelNotifications
*
*****************************************************************************/
static void vUnregisterChannelNotifications(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hChannelList
        )
{
    if (hChannelList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(
            hChannelList,
            bUnregisterChannelNotificationIterator,
            psObj);

        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            // Error! 
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNELLIST_OBJECT_NAME
                ": Failed to unregister current channel "
                "set notifications: %s (#%d)",
                OSAL.pacGetReturnCodeName(eReturnCode), 
                eReturnCode);
        }
    }

    return;
}

/*****************************************************************************
*
*   vClearChannelSet
*
*****************************************************************************/
static void vClearChannelSet(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hChannelList
        )
{
    if (hChannelList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        vUnregisterChannelNotifications(psObj, hChannelList);

        eReturnCode = OSAL.eLinkedListRemoveAll( hChannelList,
            (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! 
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNELLIST_OBJECT_NAME
                ": Failed to remove entries from the old "
                "channel set: %s (#%d)",
                OSAL.pacGetReturnCodeName(eReturnCode), 
                eReturnCode);
        }
    }

    return;
}

/*****************************************************************************
*
*   vUpdateChannelPresetHandle
*
*   This function utilizes the preset service in order to ensure the
*   channel found at n16ChannelOffset has its preset handle set correctly
*   based upon where that channel is found in the presets category
*
*****************************************************************************/
static void vUpdateChannelPresetHandle(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    N16 n16ChannelOffset,
    BOOLEAN bResetHandle
        )
{
    PRESETS_OBJECT hPresets;
    BROWSE_TYPE_ENUM eBrowseType;
    int iChannelOffset = (int)n16ChannelOffset;

    // Get the presets handle
    hPresets = DECODER_hPresets(psObj->hDecoder);

    // Validate inputs
    if ((hChannel == CHANNEL_INVALID_OBJECT) ||
        (hPresets == PRESETS_INVALID_OBJECT))
    {
        return;
    }

    // Presets is active.  We need to update
    // the channel's preset handle, but only
    // if we are operating in category mode

    // Get our current browsing mode
    eBrowseType = BROWSE_eGetMode(psObj->hBrowse);

    // Are we browsing categories
    if (eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        CATEGORY_ID tReferenceCategoryId =
            CATEGORY_INVALID_ID;
        CATEGORY_ID tPresetsCategoryId;

        // Get the reference category id
        BROWSE_bGetBrowsed(
            psObj->hBrowse,
            NULL,
            &tReferenceCategoryId );

        // Get the presets category id
        tPresetsCategoryId = PRESETS.tCategoryId( hPresets );

        // Are we operating in the presets category?
        if (tReferenceCategoryId == tPresetsCategoryId)
        {
            size_t tPresetIndex = 0;

            // Are we resetting the handle or are
            // we trying to get a real index?
            if (bResetHandle == FALSE)
            {
                size_t tCategorySize;
                N8 n8Sign;

                // Get the size of this category
                tCategorySize = CATEGORY.tSize(
                    psObj->hDecoder,
                    tReferenceCategoryId );

                if (tCategorySize > 0)
                {
                    N16 n16ChannelOccurrence;

                    // Calculate the actual offset of the
                    // channel provided in the category
                    n8Sign = iChannelOffset < 0 ? -1 : 1;
                    iChannelOffset = abs(iChannelOffset) % tCategorySize;

                    if (n8Sign < 0)
                    {
                        // Get the actual offset
                        iChannelOffset = (tCategorySize - iChannelOffset);
                    }

                    // Get the number of times this channel occurs
                    // in this category by this index
                    n16ChannelOccurrence = CATEGORY_n16ChannelOccurrence(
                        psObj->hReferenceCategory,
                        hChannel, (N16)iChannelOffset );
                    if (n16ChannelOccurrence >= 0)
                    {
                        tPresetIndex = (size_t)n16ChannelOccurrence;
                    }
                }
            }

            // Update the channel's preset
            // handle using the index we calculated
            PRESETS_vUpdateHdl (
                hPresets,
                hChannel,
                tPresetIndex
                    );
        }
    }

    return;
}

/*******************************************************************************
*
*   bBrowseAllChannelsCompareHandler
*   (This is the default channel compare handler)
*
*****************************************************************************/
static BOOLEAN bBrowseAllChannelsCompareHandler(
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    // Verify inputs
    if ( ( hDecoder == DECODER_INVALID_OBJECT ) ||
         ( hChannel == CHANNEL_INVALID_OBJECT ) )
    {
        return FALSE;
    }

    // Include this
    return TRUE;
}

/*******************************************************************************
*
*   bBrowseAllCategoriesCompareHandler
*   (This is the default category compare handler)
*
********************************************************************************/
static BOOLEAN bBrowseAllCategoriesCompareHandler(
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT hCategory,
    void *pvIterateArg,
    N16 *pn16Offset
        )
{

    if ( ( hDecoder == DECODER_INVALID_OBJECT ) ||
         ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pn16Offset == NULL ) )
    {
        return FALSE;
    }

    // Otherwise, always return true with an offset of 0
    *pn16Offset = 0;

    return TRUE;
}

/*******************************************************************************
*
*   n16CompareChannelIDs
*
********************************************************************************/
static N16 n16CompareChannelIDs (
    CHANNELLIST_ENTRY_STRUCT *psEntry1,
    CHANNELLIST_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Return = N16_MIN;

    if (psEntry1 != NULL && psEntry2 != NULL)
    {
        n16Return = CHANNEL_n16CompareChannelIds(
            psEntry1->hChannel, psEntry2->hChannel);
    }

    return n16Return;
}

/*******************************************************************************
*
*   n16CompareChannelHandles
*
********************************************************************************/
static N16 n16CompareChannelHandles (
    CHANNELLIST_ENTRY_STRUCT *psEntry1,
    CHANNELLIST_ENTRY_STRUCT *psEntry2
        )
{
    N16 n16Return = N16_MIN;

    if (psEntry1 != NULL && psEntry2 != NULL)
    {
        if (psEntry1->hChannel < psEntry2->hChannel)
        {
            n16Return = -1;
        }
        else if (psEntry1->hChannel > psEntry2->hChannel)
        {
            n16Return = 1;
        }
        else
        {
            n16Return = 0;
        }
    }

    return n16Return;
}


/*******************************************************************************
*
*   psGetChannelEntry
*
*   Assumes that psObj is already locked and in the valid context
*
********************************************************************************/
static CHANNELLIST_ENTRY_STRUCT *psGetChannelEntry(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    N16 n16Offset
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry = NULL;
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;
    UN32 un32NumItems = 0;
    int iOffset = (int)n16Offset;

    // Determine how big our list is and adjust offset...
    OSAL.eLinkedListItems(psObj->hChannelList, &un32NumItems);
    if(un32NumItems == 0)
    {
        iOffset = 0;
    }
    else
    {
        N8 n8Sign;
        n8Sign = iOffset < 0 ? -1 : 1;
        iOffset = abs(iOffset) % un32NumItems;
        iOffset *= n8Sign;
    }

    // Current entry handle starts out as the reference entry
    hCurrentEntry = psObj->hReferenceChannelEntry;
    psEntry = (CHANNELLIST_ENTRY_STRUCT *)
        OSAL.pvLinkedListThis(hCurrentEntry);

    // Based on offset provided, find the channel
    do
    {
        if(iOffset > 0)
        {
            hCurrentEntry =
                OSAL.hLinkedListNext(hCurrentEntry,
                    (void **)((void *)&psEntry));
            iOffset--;
        }
        else if(iOffset < 0)
        {
            hCurrentEntry =
                OSAL.hLinkedListPrev(hCurrentEntry,
                    (void **)((void *)&psEntry));
            iOffset++;
        }

    } while(iOffset != 0);


    return psEntry;
}

/*******************************************************************************
*
*   psGetChannelEntryByHandle
*
********************************************************************************/
static CHANNELLIST_ENTRY_STRUCT *psGetChannelEntryByHandle(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_LINKED_LIST_ENTRY hLLEntry =
       OSAL_INVALID_LINKED_LIST_ENTRY;
   CHANNELLIST_ENTRY_STRUCT *psEntry = NULL;
   CHANNELLIST_ENTRY_STRUCT sSearchEntry;
   OSAL_RETURN_CODE_ENUM eReturnCode;

   sSearchEntry.hChannel = hChannel;

   eReturnCode = OSAL.eLinkedListLinearSearch(
       psObj->hChannelList, &hLLEntry,
       (OSAL_LL_COMPARE_HANDLER)n16CompareChannelHandles,
       (void *)&sSearchEntry);

   if (eReturnCode == OSAL_SUCCESS)
   {
       psEntry = (CHANNELLIST_ENTRY_STRUCT *)OSAL.pvLinkedListThis(hLLEntry);
   }

   return psEntry;
}

/*******************************************************************************
*
*   eAddChannel
*
*   Assumes that psObj is already locked and valid
*
********************************************************************************/
static OSAL_RETURN_CODE_ENUM eAddChannel (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    OSAL_LINKED_LIST_ENTRY *phChannelEntry,
    BOOLEAN bBefore
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        if (psObj->hCursor == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            psEntry = (CHANNELLIST_ENTRY_STRUCT *)
                SMSO_hCreate(
                    CHANNELLIST_OBJECT_NAME":ChEntry",
                    sizeof(CHANNELLIST_ENTRY_STRUCT),
                    (SMS_OBJECT)psObj,
                    FALSE);

            puts(CHANNELLIST_OBJECT_NAME": Allocated new entry");
        }
        else
        {
            OSAL_LINKED_LIST_ENTRY hRemove;

            // We can re-use entry from the old channel 
            // set and perform an update.
            psEntry = (CHANNELLIST_ENTRY_STRUCT *)
                OSAL.pvLinkedListThis(psObj->hCursor);

            printf(CHANNELLIST_OBJECT_NAME
                ": Extracted entry with Channel ID: %3d\n",
                CHANNEL.tChannelId(psEntry->hChannel));

            // When done, remove this entry from the 'old' channel set.
            hRemove = psObj->hCursor;

            if (bBefore == TRUE)
            {
                if (psObj->hCursor == psObj->hTopChannelEntry)
                {
                    // We have just reached the top element in the old
                    // channel set. That means, that we don't have
                    // more entries to update and we should allocate 
                    // a new entry next time.
                    psObj->hTopChannelEntry = OSAL.hLinkedListNext(
                        psObj->hTopChannelEntry, (void **)NULL);

                    // The LL configured as 'circular', so make sure, 
                    // that the 'old' top position is actually set 
                    // to nothing.
                    if (psObj->hTopChannelEntry == psObj->hCursor)
                    {
                        psObj->hTopChannelEntry = 
                            OSAL_INVALID_LINKED_LIST_ENTRY;
                    }

#if DEBUG_OBJECT==1
                    if (psObj->hTopChannelEntry != 
                        OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        CHANNELLIST_ENTRY_STRUCT *psTop;

                        psTop = (CHANNELLIST_ENTRY_STRUCT *)
                            OSAL.pvLinkedListThis(psObj->hTopChannelEntry);

                        if (psTop != NULL)
                        {
                            printf(CHANNELLIST_OBJECT_NAME
                                ": Setting Top at the Channel ID: %d\n",
                                CHANNEL.tChannelId(psTop->hChannel));
                        }
                    }
#endif // DEBUG_OBJECT==1

                    puts(CHANNELLIST_OBJECT_NAME": Resetting cursor");

                    // Reset cursor position. Since we don't have 
                    // more entries to update, that will cause a new
                    // entry to be allocated next time.
                    psObj->hCursor = OSAL_INVALID_LINKED_LIST_ENTRY;
                }
                else
                {
                    // Adjust cursor position at the previous entry.
                    psObj->hCursor = 
                        OSAL.hLinkedListPrev(psObj->hCursor, (void **)NULL);
                }
            }
            else
            {
                if (psObj->hCursor == psObj->hBottomChannelEntry)
                {
                    // We have just reached the bottom element in the old
                    // channel set. That means, that we don't have
                    // more entries to update and we should allocate 
                    // a new entry next time.
                    psObj->hBottomChannelEntry = OSAL.hLinkedListPrev(
                        psObj->hBottomChannelEntry, (void **)NULL);

                    // The LL configured as 'circular', so make sure, 
                    // that the 'old' bottom position is actually set 
                    // to nothing.
                    if (psObj->hBottomChannelEntry == psObj->hCursor)
                    {
                        psObj->hBottomChannelEntry = 
                            OSAL_INVALID_LINKED_LIST_ENTRY;
                    }

#if DEBUG_OBJECT==1
                    if (psObj->hBottomChannelEntry !=
                        OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        CHANNELLIST_ENTRY_STRUCT *psBottom;

                        psBottom = (CHANNELLIST_ENTRY_STRUCT *)
                            OSAL.pvLinkedListThis(psObj->hBottomChannelEntry);

                        if (psBottom != NULL)
                        {
                            printf(CHANNELLIST_OBJECT_NAME
                                ": Setting Bottom at the Channel ID: %d\n",
                                CHANNEL.tChannelId(psBottom->hChannel));
                        }
                    }
#endif // DEBUG_OBJECT==1

                    puts(CHANNELLIST_OBJECT_NAME": Resetting cursor");

                    // Reset cursor position. Since we don't have 
                    // more entries to update, that will cause a new
                    // entry to be allocated next time.
                    psObj->hCursor = OSAL_INVALID_LINKED_LIST_ENTRY;
                }
                else
                {
                    // Adjust cursor position at the next entry.
                    psObj->hCursor = 
                        OSAL.hLinkedListNext(psObj->hCursor, (void **)NULL);
                }
            }

            // Remove re-used entry from the old channel set.
            OSAL.eLinkedListRemove(hRemove);
        }

        if (psEntry == NULL)
        {
            break;
        }

        // Compare old and new channels on the same 
        // position to calculate differences.
        psEntry->tMask = CHANNEL_tCompareContent(psEntry->hChannel, hChannel);
        psEntry->hChannel = hChannel;

        if (bBefore == TRUE)
        {
            eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                psObj->hChannelList,
                phChannelEntry,
                (void *)psEntry );
        }
        else
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->hChannelList,
                phChannelEntry,
                (void *)psEntry );
        }

        printf(CHANNELLIST_OBJECT_NAME
            ": Inserting entry with Channel ID: %3d%s, %s (#%d)\n",
                CHANNEL.tChannelId(psEntry->hChannel), 
                (bBefore == TRUE) ? " Before" : "",
                OSAL.pacGetReturnCodeName(eReturnCode), 
                eReturnCode);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Update Channel List
        SMSU_tUpdate(&psObj->sEvent, psEntry->tMask);

        return OSAL_SUCCESS;
    }
    while (FALSE);

    if (psEntry != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   eRemoveChannelEntry
*
*   Assumes that psObj is already locked and valid
*
********************************************************************************/
static OSAL_RETURN_CODE_ENUM eRemoveChannelEntry (
    OSAL_LINKED_LIST_ENTRY hChannelEntry
        )
{
    CHANNELLIST_ENTRY_STRUCT *psEntry =
        OSAL.pvLinkedListThis(hChannelEntry);

    if (psEntry != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return OSAL.eLinkedListRemove(hChannelEntry);
}
