/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CHANNEL implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "radio.h"
#include "sms_api_debug.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "module_obj.h"
#include "decoder_obj.h"
#include "string_obj.h"
#include "category_obj.h"
#include "cdo_obj.h"
#include "dataservice_mgr_obj.h"
#include "channel_art_mgr_obj.h"
#include "channel_art_obj.h"
#include "epg_mgr_obj.h"
#include "presets_obj.h"

#include "channel_obj.h"
#include "_channel_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tEventMask
*
*****************************************************************************/
static CHANNEL_EVENT_MASK tEventMask (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    CHANNEL_EVENT_MASK tMask = CHANNEL_OBJECT_EVENT_NONE;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == TRUE)
    {
        tMask = SMSU_tMask(&psObj->sEvent);
        DECODER_vChannelEventMask(hChannel, &tMask);
    }

    return tMask;
}

/*****************************************************************************
*
*   tServiceId
*
*****************************************************************************/
static SERVICE_ID tServiceId (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return SERVICE_INVALID_ID;
    }

    return psObj->tServiceId;
}

/*****************************************************************************
*
*   tChannelId
*
*****************************************************************************/
static CHANNEL_ID tChannelId (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return CHANNEL_INVALID_ID;
    }

    return psObj->tId;
}

/*****************************************************************************
*
*   hShortName
*
*****************************************************************************/
static STRING_OBJECT hShortName (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    return psObj->hShortName;
}

/*****************************************************************************
*
*   hMediumName
*
*****************************************************************************/
static STRING_OBJECT hMediumName (
    CHANNEL_OBJECT hChannel
    )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    return psObj->hMediumName;
}

/*****************************************************************************
*
*   hLongName
*
*****************************************************************************/
static STRING_OBJECT hLongName (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    return psObj->hLongName;
}

/*****************************************************************************
*
*   hShortDescription
*
*****************************************************************************/
static STRING_OBJECT hShortDescription (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    return psObj->hShortDesc;
}

/*****************************************************************************
*
*   hLongDescription
*
*****************************************************************************/
static STRING_OBJECT hLongDescription (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return STRING_INVALID_OBJECT;
    }

    return psObj->hLongDesc;
}

/*****************************************************************************
*
*   eCategoryOffset
*
*   This function gets the offset into the channel's list of categories for the
*   specified CATEGORY_ID.
*
*       Inputs:
*           hChannel        Handle to the channel to search for the category
*                           id in
*           tCategoryId     Category id to search for in the channel
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success and
*                                       SMSAPI_RETURN_CODE_ERROR on failure.
*           pn16Offset      This function will write the offset into the
*                           channel's category list upon success.  The value
*                           will not be modified upon a failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCategoryOffset(
    CHANNEL_OBJECT hChannel,
    CATEGORY_ID tCategoryId,
    N16 *pn16Offset
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;
    UN32 un32NumCategories = 0;
    UN16 un16Return = 0;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if(  ( bValid == FALSE ) ||
         ( tCategoryId == CATEGORY_INVALID_ID ) ||
         ( pn16Offset == NULL ) )
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Determine the number of categories in our list
    eOsalReturnCode = OSAL.eLinkedListItems(
                        psObj->hCategoryList,
                        &un32NumCategories );

    if ( eOsalReturnCode != OSAL_SUCCESS )
    {
        // Error
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Find this category's offset within this channel

    // Determine first category in the channel's list of categories
    hCurrentEntry = OSAL.hLinkedListFirst(
        psObj->hCategoryList, NULL );

    if ( hCurrentEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        // Error
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Extract category handle of the first entry
    hCategory = (CATEGORY_OBJECT)OSAL.pvLinkedListThis( hCurrentEntry );

    for ( un16Return = 0; un16Return < un32NumCategories; un16Return++ )
    {
        // Check if current entry is the category we are looking for
        if( CATEGORY.tGetCategoryId( hCategory ) == tCategoryId )
        {
            // Copy over the offset into the contents of the pointer
            *pn16Offset = (N16)un16Return;

            // Found it!
            break;
        }

        // Try the next one
        hCurrentEntry =
            OSAL.hLinkedListNext(hCurrentEntry, (void *)((void *)&hCategory));
    }

    // Check if it was not found
    if ( un16Return >= un32NumCategories )
    {
        // Error
        return SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    // We found it
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eNumCategories
*
*   This function gets the number of categories that a given channel is assigned
*   to.
*
*       Inputs:
*           hChannel - Handle to get the number of categories for
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM - SMSAPI_RETURN_CODE_SUCCESS on success and
*               SMSAPI_RETURN_CODE_ERROR on failure.
*           pn16NumCategories - This function will write the number of
*               categories associated with the provided channel into the
*               contents of this pointer upon success.  The value will not be
*               modified upon a failure.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNumCategories(
    CHANNEL_OBJECT hChannel,
    N16 *pn16NumCategories
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    UN32 un32NumCategories = 0;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if(  ( bValid == FALSE ) ||
         ( pn16NumCategories == NULL ) )
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Determine the number of categories in our list
    eOsalReturnCode = OSAL.eLinkedListItems(
                        psObj->hCategoryList,
                        &un32NumCategories );

    // If we failed
    if ( eOsalReturnCode != OSAL_SUCCESS )
    {
        // Error
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // If we were sucessful, copy the number of channels into the pointer
    *pn16NumCategories = (N16)un32NumCategories;

    return SMSAPI_RETURN_CODE_SUCCESS;
}


/*****************************************************************************
*
*   hCategory
*
*****************************************************************************/
static CATEGORY_OBJECT hCategory (
    CHANNEL_OBJECT hChannel,
    N16 n16Offset
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hCurrentEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    UN32 un32NumItems = 0;
    int iOffset = (int)n16Offset;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if(FALSE == bValid)
    {
        // Error!
        return CATEGORY_INVALID_OBJECT;
    }

    // Verify inputs
    if(psObj->hCategoryList != OSAL_INVALID_OBJECT_HDL)
    {
        // Assign current entry handle to the reference entry
        hCurrentEntry = psObj->hReferenceCategoryEntry;
        if(hCurrentEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            hCategory = (CATEGORY_OBJECT)OSAL.pvLinkedListThis(hCurrentEntry);

            // Determine how big our list is and adjust offset...
            OSAL.eLinkedListItems(psObj->hCategoryList, &un32NumItems);
            if(un32NumItems == 0)
            {
                iOffset = 0;
            }
            else
            {
                N8 n8Sign;
                n8Sign = iOffset < 0 ? -1 : 1;
                iOffset = abs(iOffset) % un32NumItems;
                iOffset *= n8Sign;
            }

            // Based on offset provided, find the category
            do
            {
                if(iOffset > 0)
                {
                    hCurrentEntry =
                        OSAL.hLinkedListNext(hCurrentEntry,
                            (void *)((void *)&hCategory));
                    iOffset--;
                }
                else if(iOffset < 0)
                {
                    hCurrentEntry =
                        OSAL.hLinkedListPrev(hCurrentEntry,
                            (void *)((void *)&hCategory));
                    iOffset++;
                }

            } while(iOffset != 0);
        }
    }

    return hCategory;
}

/*****************************************************************************
*
*   hCDO
*
*****************************************************************************/
static CD_OBJECT hCDO (
    CHANNEL_OBJECT hChannel
        )
{
    // Return CHANNEL's CDO without overriding subscription state
    return CHANNEL_hCDO(hChannel, FALSE);
}

/*****************************************************************************
*
*   hArt
*
*****************************************************************************/
static CHANNEL_ART_OBJECT hArt (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == FALSE)
    {
        // Error!
        return CHANNEL_ART_INVALID_OBJECT;
    }

    return psObj->hArt;
}

/*****************************************************************************
*
*   ePreset
*
*   This API is used to retrieve the current preset to which this channel
*   belongs.  This must only be called from within a channel list or
*   category iterator, as the current preset is only maintained within
*   those iterators.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePreset (
    CHANNEL_OBJECT hChannel,
    PRESET_BAND_OBJECT *phBand,
    STRING_OBJECT *phPresetName,
    size_t *ptPresetIndex
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_NOT_OWNER;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;
        BOOLEAN bOk;

        // Get the data for this preset
        bOk = PRESETS_bThisPreset(
            hChannel,
            psObj->hCurrentPresetHdl,
            phBand,
            phPresetName,
            ptPresetIndex );

        // Condition the return code
        if (bOk == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eIsLocked
*
*       This API is used to check if a specific channel is locked.  This API may
*       only be used in the context of the Decoder's Event Callback.
*
*       Inputs:
*           hChannel - A handle to a valid CHANNEL object to check the lock
*               attribute for.
*
*       Outputs:
*           pbLocked - pointer to a BOOLEAN that will hold the value of TRUE if
*               the channel is locked or FALSE if the channel is not locked.
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*               SMSAPI_RETURN_CODE_ERROR on error if the channel id cannot be
*               found/invalid.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsLocked (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbLocked
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ( pbLocked == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Check if the attributes contain a channel lock
        if (( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_LOCKED ) ==
            CHANNEL_OBJECT_ATTRIBUTE_LOCKED )
        {
            *pbLocked = TRUE;
        }
        else
        {
            *pbLocked = FALSE;
        }
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eIsSkipped
*
*       This API is used to check if a specific channel is skipped. This API may
*       only be used in the context of the Decoder's Event Callback.
*
*       Inputs:
*           hChannel - A handle to a valid CHANNEL object to check the skip
*               attribute for.
*
*       Outputs:
*           pbSkipped - pointer to a BOOLEAN that will hold the value of TRUE if
*               the channel is skipped or FALSE if the channel is not skipped.
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*               SMSAPI_RETURN_CODE_ERROR on error if the channel id cannot be
*               found/invalid.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsSkipped (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbSkipped
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ( pbSkipped == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Check if the attributes contain a channel lock
        if (( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_SKIPPED ) ==
            CHANNEL_OBJECT_ATTRIBUTE_SKIPPED )
        {
            *pbSkipped = TRUE;
        }
        else
        {
            *pbSkipped = FALSE;
        }
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eIsSubscribed
*
*       This API is used to check if a specific channel is subscribed. This API
*       may only be used in the context of the Decoder's Event Callback.
*
*       Inputs:
*       hChannel - A handle to a valid CHANNEL object to check the subscribed
*           attribute for.
*
*       Outputs:
*       pbSubscribed - pointer to a BOOLEAN that will hold the value of TRUE if
*           the channel is subscribed or FALSE if the channel is not
*           subscribed.
*
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*              SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsSubscribed (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbSubscribed
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ( pbSubscribed == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

         // Check if the attributes include subscribed
         if (( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED ) ==
               CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED)
         {
             *pbSubscribed = TRUE;
         }
         else
         {
             *pbSubscribed = FALSE;
         }
         return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eIsMature
*
*       This API is used to check if a specific channel has been determined to
*       be broadcasting "Extreme Languange" content. This API may only be used
*       in the context of the Decoder's Event Callback.
*
*       Inputs:
*       hChannel - A handle to a valid CHANNEL object to check the mature
*           attribute for.
*
*       Outputs:
*       pbMature - pointer to a BOOLEAN that will hold the value of TRUE if
*           the channel is broadcasting mature content or FALSE if the channel
*           is not.
*
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*              SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsMature (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbMature
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ( pbMature == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

         // Check if the attributes include mature
         if (( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_MATURE ) ==
               CHANNEL_OBJECT_ATTRIBUTE_MATURE)
         {
             *pbMature = TRUE;
         }
         else
         {
             *pbMature = FALSE;
         }
         return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eIsFreeToAir
*
*       This API is used to check if a specific channel is always available
*       regardless of the current subscription state. This API may only be used
*       in the context of the Decoder's Event Callback.
*
*       Inputs:
*       hChannel - A handle to a valid CHANNEL object to check the mature
*           attribute for.
*
*       Outputs:
*       pbFreeToAir - pointer to a BOOLEAN that will hold the value of TRUE if
*           the channel is free-to-air or FALSE if the channel is not.
*
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*              SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsFreeToAir (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbFreeToAir
    )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ( pbFreeToAir == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;
        // Check if the attributes include the free-to-air bit
        if (( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR ) ==
            CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR)
        {
            *pbFreeToAir = TRUE;
        }
        else
        {
            *pbFreeToAir = FALSE;
        }
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}


/*****************************************************************************
*
*  eIterateSimilarChannels
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateSimilarChannels (
    CHANNEL_OBJECT hChannel,
    CHANNEL_SIMILAR_ITERATOR_CALLBACK bSimilarIterator,
    void *pvIteratorArg)
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;

    if (NULL == bSimilarIterator)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        UN32 un32Index;
        CCACHE_OBJECT hCCache;
        CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;

        hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
        if (hCCache != CCACHE_INVALID_OBJECT)
        {
            for (un32Index = 0; un32Index < psObj->sSimilarChannels.un32Items; un32Index++)
            {
                CHANNEL_OBJECT hSimilarChannel;

                hSimilarChannel = CCACHE_hChannelFromIds(
                    hCCache,
                    psObj->sSimilarChannels.ptServiceIds[un32Index],
                    CHANNEL_INVALID_ID,
                    FALSE);

                if (hSimilarChannel != CHANNEL_INVALID_OBJECT)
                {
                    BOOLEAN bContinue;

                    bContinue = bSimilarIterator(hSimilarChannel, pvIteratorArg);
                    if (bContinue == FALSE)
                    {
                        break;
                    }
                }
            }

            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
        	eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *  eIteratePrograms
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIteratePrograms (
        CHANNEL_OBJECT hChannel,
        PROGRAM_ITERATOR_CALLBACK bEpgIterator,
        void *pvIteratorArg)
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (NULL == bEpgIterator)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *) hChannel;
        EPG_CHANNEL_OBJECT_STRUCT *psChanObj =
                (EPG_CHANNEL_OBJECT_STRUCT*) psObj->hEpgObjectsList;
        EPG_FILTER_STRUCT sEpgFilter;

        OSAL.bMemSet(&sEpgFilter, 0, sizeof(EPG_FILTER_STRUCT));
        sEpgFilter.u8NumChannels = 1;
        sEpgFilter.ptChannels = &(psObj->tServiceId);

        if (psChanObj == NULL)
        {
            return SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        if ((SMS_OBJECT)psChanObj->hEpgService == SMS_INVALID_OBJECT)
        {
            printf("EPG_MGR handler is not provided.\n");
            return SMSAPI_RETURN_CODE_NOT_OWNER;
        }

        eReturnCode = EPG.eIteratePrograms(psChanObj->hEpgService,
                                           bEpgIterator,
                                           pvIteratorArg,
                                           &sEpgFilter);
        if (eReturnCode == SMSAPI_RETURN_CODE_NO_OBJECTS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eNotifyOnChange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNotifyOnChange (
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        hCCache = DECODER_hCCache(hDecoder);

        // Now that we have the CCACHE handle, register this function
        // for the application
        eErrorCode = CCACHE_eHandleAppChannelNotifyOnChange(
            hCCache, vNotifierCallback, pvNotifierCallbackArg);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if(eErrorCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bSuccess;

            // Tell the decoder to start processing the initial dump of notifications
            bSuccess = DECODER_bStartAllChanCatNotifications(hDecoder, TRUE);
            if(bSuccess == FALSE)
            {
                eErrorCode  = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    }

    return eErrorCode;
}

/*****************************************************************************
*
*       eIterateAll
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateAll (
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT_ITERATOR bIterator,
    CHANNEL_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Verify inputs
    if(bIterator == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    if(bInclude == NULL)
    {
        // Accept all
        bInclude = bAlwaysInclude;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        hCCache = DECODER_hCCache(hDecoder);
        if(hCCache != CCACHE_INVALID_OBJECT)
        {
            CHANNEL_ITERATOR_STRUCT sIterator;
            BOOLEAN bOk;

            // Populate local iterator
            sIterator.bIterator = bIterator;
            sIterator.bInclude = bInclude;
            sIterator.pvArg = pvArg;

            bOk = CCACHE_bIterateChannels(
                hCCache, FALSE, bIteratorShim, &sIterator);
            if(bOk == TRUE)
            {
                eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*       tACO
*
*****************************************************************************/
static CHANNEL_ACO tACO (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bValid;
    CHANNEL_ACO tACO = CHANNEL_ACO_DEFAULT;

    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if (bValid == TRUE)
    {
        // De-reference object
        CHANNEL_OBJECT_STRUCT *psObj = 
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        tACO = psObj->tACO;
    }

    return tACO;
}

/************************************************************************
*
*       bQualifiedForTuneMix
*
*************************************************************************/
static BOOLEAN bQualifiedForTuneMix (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bValid, bResult = FALSE;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == TRUE)
    {
        // Checking if  content type is MUSIC or UNKNOWN (not yet acquired)
        if ((psObj->eContentType == CONTENT_TYPE_UNKNOWN) ||
            (psObj->eContentType == CONTENT_TYPE_MUSIC_PID))
        {
            bResult = TRUE;
        }
    }

    return bResult;
}

/*****************************************************************************
*
*       ePlayOnSelect
*
*       This API is used to request Play On Select Method value. 
*       This API may only be used in the context of the Decoder's Event 
*       Callback.
*
*       Inputs:
*       hChannel - A handle to a valid CHANNEL object to get Play On
*                  Select Method value.
*
*       Outputs:
*       pePlayOnSelect - pointer to a CHANNEL_PLAY_ON_SELECT_METHOD_ENUM that will 
*                  hold the value of the method.
*
*       Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*              SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePlayOnSelect (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM *pePlayOnSelect
        )
{
    BOOLEAN bOwner;

    // Verify inputs and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if ((bOwner == TRUE) && (pePlayOnSelect != NULL))
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        *pePlayOnSelect = psObj->ePlayOnSelectMethod;
        return SMSAPI_RETURN_CODE_SUCCESS;
    }
    
    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CHANNEL_hCreate
*
*****************************************************************************/
CHANNEL_OBJECT CHANNEL_hCreate (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)CHANNEL_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Verify input
    // It may seem strange to allow a channel to be created with an invalid id
    // but it is required when creating a channel for an unmapped service id
    // "normal" means of getting a channel based on channel id already check
    // for CHANNEL_INVALID_ID anyway

    if( !(tChannelId <= CHANNEL_INVALID_ID) )
    {
         // There are input parameter errors!
        return CHANNEL_INVALID_OBJECT;
    }

    if( !(tServiceId <= SERVICE_INVALID_ID) )
    {
         // There are input parameter errors!
        return CHANNEL_INVALID_OBJECT;
    }

    // Construct a unique name for this channel.
    snprintf(&acName[0],
            sizeof(acName),
            CHANNEL_OBJECT_NAME":%d-%d",
            tChannelId, tServiceId);

    // Create an instance of this object
    psObj = (CHANNEL_OBJECT_STRUCT *)
        SMSO_hCreate(
            acName,
            sizeof(CHANNEL_OBJECT_STRUCT),
            (SMS_OBJECT)hCCache, // Child
            FALSE); // No Lock (ignored)
    if(psObj == NULL)
    {
        // Error!
        return CHANNEL_INVALID_OBJECT;
    }

    // Initialize object...

    // Initialize channel object info
    psObj->tId = CHANNEL_INVALID_ID;
    psObj->tServiceId = SERVICE_INVALID_ID;
    psObj->eType = CHANNEL_TYPE_UNKNOWN;
    psObj->hLongName = STRING_INVALID_OBJECT;
    psObj->hShortName = STRING_INVALID_OBJECT;
    psObj->hMediumName = STRING_INVALID_OBJECT;
    psObj->hLongDesc = STRING_INVALID_OBJECT;
    psObj->hShortDesc = STRING_INVALID_OBJECT;
    psObj->hChannelArtService = CHANNEL_ART_SERVICE_INVALID_OBJECT;
    psObj->hArt = CHANNEL_ART_INVALID_OBJECT;
    psObj->hEpgObjectsList = EPG_INVALID_CHANNEL_OBJECT;
    psObj->tAttributes = CHANNEL_OBJECT_ATTRIBUTE_NONE;
    psObj->tUnconfirmedAttributes = CHANNEL_OBJECT_ATTRIBUTE_NONE;
    psObj->tUsageCounter = 0;
    psObj->tFilter = CHANNEL_OBJECT_EVENT_ALL;
    psObj->hBasePresetHdl = CHANNEL_PRESET_INVALID_HDL;
    psObj->hCurrentPresetHdl = CHANNEL_PRESET_INVALID_HDL;
    psObj->sSimilarChannels.ptServiceIds = NULL;
    psObj->sSimilarChannels.un32Items = 0;
    psObj->sSimilarChannels.un32Size = 0;
    psObj->hReferenceCategoryEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->ePlayOnSelectMethod = CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;
    psObj->eContentType = CONTENT_TYPE_UNKNOWN;
    psObj->eIRNavigationClass = IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;
    psObj->tACO = CHANNEL_ACO_DEFAULT;

    // Assign a CDO (blank)
    psObj->hCDO = CDO_hCreate((CHANNEL_OBJECT)psObj);
    if(psObj->hCDO == CD_INVALID_OBJECT)
    {
        // Error!
        CHANNEL_vDestroy((CHANNEL_OBJECT)psObj);
        return CHANNEL_INVALID_OBJECT;
    }

    // Initialize asynchronous update configuration
    // Note: We use 'CHANNEL_OBJECT_EVENT_INITIAL' as
    // the initial update-event mask because we want
    // to indicate that all events associated with a
    // CHANNEL object are indeed updated/modified as a
    // result of creating the object itself. This is to
    // specifically mean all CHANNEL object updateable events
    // are initially marked as 'updated'. Note further that our
    // update event request mask is 'NONE'. This is because nobody
    // has yet asked for any updates to a new CHANNEL_OBJECT.
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        CHANNEL_OBJECT_EVENT_INITIAL,
        SMS_OBJECT_EVENT_NONE,
        (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
        NULL);

    // Create a category linked list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hCategoryList,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16CompareCategoryIds,
        OSAL_LL_OPTION_CIRCULAR|OSAL_LL_OPTION_UNIQUE
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CHANNEL_vDestroy((CHANNEL_OBJECT)psObj);
        return CHANNEL_INVALID_OBJECT;
    }

    // Initialize reference category.
    // The first category in the list (index 0) is to be broadcast category
    // Until it assigned (if ever assigned at all), we put CATEGORY_INVALID_OBJECT
    // there. See GHE issue #736 for requirements.
    eReturnCode = OSAL.eLinkedListAdd(
        psObj->hCategoryList,
        &psObj->hReferenceCategoryEntry,
        (void *)CATEGORY_INVALID_OBJECT
            );

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CHANNEL_vDestroy((CHANNEL_OBJECT)psObj);
        return CHANNEL_INVALID_OBJECT;
    }

    // Initialize (clear) all channel information. This brings the channel
    // to a normalized initial state. This is the same state a channel will
    // be in after an channel-line-up change.

    // Clear the contents of this channel sans service id,
    // and any application provided attributes.
    CHANNEL_vClearBroadcast((CHANNEL_OBJECT)psObj);

    // Now, record service and channel id provided by caller. This is
    // really all we know about the channel right now.
    CHANNEL_bUpdateIds((CHANNEL_OBJECT)psObj, tServiceId, tChannelId);

    return (CHANNEL_OBJECT)psObj;
}

/*****************************************************************************
*
*   CHANNEL_vDestroy
*
*****************************************************************************/
void CHANNEL_vDestroy (
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Destroy the SMS Update object. This will also
    // remove any remaining event notification callback nodes.
    // It will also NOT make any further event callbacks.
    SMSU_vDestroy(&psObj->sEvent);

    // Destroy category list if one exists
    if(psObj->hCategoryList != OSAL_INVALID_OBJECT_HDL)
    {
        // Iterate the category list this channel has and
        // remove any category associations from this channel
        OSAL.eLinkedListIterate(
            psObj->hCategoryList, bRemoveChannelFromCategory,
            (void *)hChannel
                );

        // Remove all entries in the linked list
        OSAL.eLinkedListRemoveAll(
            psObj->hCategoryList, (OSAL_LL_RELEASE_HANDLER)NULL);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->hCategoryList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hCategoryList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy similar chan list list if one exists
    if( psObj->sSimilarChannels.ptServiceIds != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->sSimilarChannels.ptServiceIds);
        psObj->sSimilarChannels.ptServiceIds = NULL;
    }

    // Un-Initialize channel parameters...
    psObj->hReferenceCategoryEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->tServiceId =
        SERVICE_INVALID_ID;
    psObj->tId =
        CHANNEL_INVALID_ID;
    psObj->eType =
        CHANNEL_TYPE_UNKNOWN;
    psObj->tAttributes = CHANNEL_OBJECT_ATTRIBUTE_NONE;
    psObj->tUnconfirmedAttributes = CHANNEL_OBJECT_ATTRIBUTE_NONE;

    // Destroy CHANNEL art handles
    psObj->hArt = CHANNEL_ART_INVALID_OBJECT;
    psObj->hChannelArtService = CHANNEL_ART_SERVICE_INVALID_OBJECT;

    // Destroy channel objects (if they exist)...

    if(psObj->hLongName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hLongName);
        psObj->hLongName = STRING_INVALID_OBJECT;
    }

    if(psObj->hMediumName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hMediumName);
        psObj->hMediumName = STRING_INVALID_OBJECT;
    }

    if(psObj->hShortName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hShortName);
        psObj->hShortName = STRING_INVALID_OBJECT;
    }

    if(psObj->hLongDesc != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hLongDesc);
        psObj->hLongDesc = STRING_INVALID_OBJECT;
    }

    if(psObj->hShortDesc != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hShortDesc);
        psObj->hShortDesc = STRING_INVALID_OBJECT;
    }

    // Destroy its CDO (if one exists)
    if(psObj->hCDO != CD_INVALID_OBJECT)
    {
        CDO_vDestroy(psObj->hCDO);
        psObj->hCDO = CD_INVALID_OBJECT;
    }

    // At this point the CHANNEL should not be in use by anyone.
    // If it is this means someone still has a CHANNEL in use
    // and did not clean up properly. So I think at this point
    // all I can do is leave the CHANNEL object intact but it is
    // an error not to destroy it.
    if(psObj->tUsageCounter == 0)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hChannel);
    }
    else
    {
        // Error!
        printf(CHANNEL_OBJECT_NAME":CHANNEL_vDestroy(): "
            "Usage counter is not zero, channel will not be destroyed!\n");
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_eType
*
*****************************************************************************/
CHANNEL_TYPE_ENUM CHANNEL_eType (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return CHANNEL_TYPE_UNKNOWN;
    }

    return psObj->eType;
}

/*****************************************************************************
*
*   CHANNEL_bRegisterNotification
*
* When working with a CHANNEL object, if the caller needs to be notified
* of CHANNEL attribute or content changes then they should use this API
* to register for those updates. It is also used to mark this channel
* as 'used' meaning someone was used the CHANNEL object handle (by reference)
* in their own object. Thus this channel must remain persistant until no
* other entities continue to 'use' this channel. If the caller does not
* wish to actually get notifications, but instead just mark the channel as
* used, then they can supply an event mask of CHANNEL_OBJECT_EVENT_NONE
* and need not provide a callback function or arguments.
*
*****************************************************************************/
BOOLEAN CHANNEL_bRegisterNotification (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventRequestMask,
    ...
        )
{
    va_list tRegisterArgs;
    BOOLEAN bRegistered = TRUE;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback =
        (CHANNEL_OBJECT_EVENT_CALLBACK)NULL;
    void *pvEventCallbackArg = (void *)(0);

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return FALSE;
    }

    // Extract variable arguments (as needed) otherwise use defaults.
    if(tEventRequestMask != CHANNEL_OBJECT_EVENT_NONE)
    {
        BOOLEAN bSetInitialEvents;

        va_start( tRegisterArgs, tEventRequestMask );

        vEventCallback =
            va_arg( tRegisterArgs, CHANNEL_OBJECT_EVENT_CALLBACK );
        pvEventCallbackArg =
            va_arg( tRegisterArgs, void * );
        bSetInitialEvents =
            (BOOLEAN)va_arg( tRegisterArgs, int );

        va_end( tRegisterArgs );

        // Initialize registration indication
        bRegistered = FALSE;

        // Check if a valid callback was provided.
        if(vEventCallback != NULL)
        {
            // Add the callback to the
            // asynchronous update configuration
            bRegistered = SMSU_bRegisterCallback(
                &psObj->sEvent,
                tEventRequestMask,
                (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
                pvEventCallbackArg,
                bSetInitialEvents
                    );
        }
    }

    // Check if we should increment the usage counter
    if(bRegistered == TRUE)
    {
        // Mark this channel as "in-use"
        psObj->tUsageCounter++;
    }

    //printf(CHANNEL_OBJECT_NAME":CHANNEL_bRegisterNotification():"
    //    " #%u(=%u)\n", psObj->tId, psObj->tUsageCounter);

    return bRegistered;
}

/*****************************************************************************
*
*   CHANNEL_bNotifyIfPending
*
*****************************************************************************/
BOOLEAN CHANNEL_bNotifyIfPending (
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvCallbackArg
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bSuccess;

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return FALSE;
    }

    bSuccess = SMSU_bIsCallbackPending(&psObj->sEvent,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback, pvCallbackArg);

    if (bSuccess == TRUE)
    {
        bSuccess = SMSU_bNotifyCallback(&psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback, pvCallbackArg);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CHANNEL_vUnregisterNotification
*
* This API is used to unregister a previously registered notification or
* to simply unmark a channel as used. If the original registration did
* not involve a callback notification function or argument then they can
* simply supply NULL for the callback parameter and need not provide
* an argument. Otherwise they must specify the callback function and argument
* to properly unregister for notifications.
*
*****************************************************************************/
void CHANNEL_vUnregisterNotification (
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    ...
        )
{
    BOOLEAN bUnRegistered = TRUE;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Extract variable arguments (as needed) otherwise use defaults.
    if(vEventCallback != (CHANNEL_OBJECT_EVENT_CALLBACK)NULL)
    {
        va_list tRegisterArgs;
        void *pvEventCallbackArg;
    
        va_start( tRegisterArgs, vEventCallback );

        pvEventCallbackArg =
            va_arg( tRegisterArgs, void * );

        va_end( tRegisterArgs );

        // Remove the callback from the
        // asynchronous update configuration
        bUnRegistered = SMSU_bUnregisterCallback(
            &psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg);
    }

    // Check if we satisfied unregistration, or are not unregistering a callback
    if(bUnRegistered == TRUE)
    {
        // Mark this channel as "not-used" by the caller
        if(psObj->tUsageCounter > 0)
        {
            psObj->tUsageCounter--;
        }
    }

    //printf(CHANNEL_OBJECT_NAME":CHANNEL_vUnregisterNotification():"
    //    " #%u(=%u)\n", psObj->tId, psObj->tUsageCounter);

    return;
}

/*****************************************************************************
*
*   CHANNEL_bAddCategory
*
*****************************************************************************/
BOOLEAN CHANNEL_bAddCategory (
    CHANNEL_OBJECT hChannel,
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bAdded = FALSE, bNotify = FALSE;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_TYPE_ENUM eType;
    CATEGORY_OBJECT hOldCategory = CATEGORY_INVALID_OBJECT;

    // Verify inputs
    if( (hChannel == CHANNEL_INVALID_OBJECT) ||
        (hCategory == CATEGORY_INVALID_OBJECT)
            )
    {
        // Error!
        return FALSE;
    }

    // Get a category type
    eType = CATEGORY.eType(hCategory);

    // Check if this category is a broadcast category. If so, it
    // needs to be made the "reference entry", or in other words
    // the first one in the list.
    if (eType == CATEGORY_TYPE_BROADCAST)
    {
        CATEGORY_ID tReferenceCategoryId, tCategoryId;

        // This is a broadcast category. So we now check to see if
        // the category id is different than the current reference
        // we have.
        tCategoryId = CATEGORY.tGetCategoryId(hCategory);

        // Get current reference category
        hOldCategory =
            (CATEGORY_OBJECT)OSAL.pvLinkedListThis(
                psObj->hReferenceCategoryEntry);

        // Get reference category id
        tReferenceCategoryId = CATEGORY.tGetCategoryId(hOldCategory);

        // Compare them
        if(tReferenceCategoryId != tCategoryId)
        {
            // This broadcast category is a new reference category
            eReturnCode = OSAL.eLinkedListReplaceEntry(
                psObj->hCategoryList,
                psObj->hReferenceCategoryEntry,
                (void *)hCategory);

            // Remove ourselves from the category
            CATEGORY_vRemoveChannel(hOldCategory, hChannel, FALSE);

            if (eReturnCode == OSAL_SUCCESS)
            {
                bNotify = TRUE;
            }
        }
    }
    else
    {
        // Add it to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hCategoryList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hCategory
                );

        if(OSAL_SUCCESS == eReturnCode)
        {
            // OSAL_SUCCESS code means that category was successfully added
            // to the channel's category list. It means that this category
            // is really new for the channel.
            bNotify = TRUE;
        }
    }

    if (bNotify == TRUE)
    {
        // Effectively the channel category has been modified. So note
        // that now and send notification if required.
        SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_CATEGORY);
        SMSU_bNotify(&psObj->sEvent);

        // Verify category type is broadcast
        if (eType == CATEGORY_TYPE_BROADCAST)
        {
            // Add this channel to the provided category to make sure they
            // have references to each other.
            bAdded = CATEGORY_bAddChannel(hCategory, hChannel);
            if (bAdded == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_OBJECT_NAME": Failed to add Channel Id: %d into "
                        "the Category Id: %d",
                        CHANNEL.tChannelId(hChannel),
                        CATEGORY.tGetCategoryId(hCategory));

                // Try to rollback.
                eReturnCode = OSAL.eLinkedListReplaceEntry(
                    psObj->hCategoryList,
                    psObj->hReferenceCategoryEntry,
                    (void *)hOldCategory);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_OBJECT_NAME": Cannot rollback category "
                            "update: %s",
                            OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
        }
    }

    return bAdded;
}

/*****************************************************************************
*
*   CHANNEL_vRemoveCategory
*
*****************************************************************************/
void CHANNEL_vRemoveCategory (
    CHANNEL_OBJECT hChannel,
    CATEGORY_OBJECT hCategory,
    BOOLEAN bSuppressCatEvents
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bCatRemoved;

    // Verify inputs
    if( hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Remove the category
    bCatRemoved = bRemoveCategory(psObj, hCategory, bSuppressCatEvents);

    if (bCatRemoved == TRUE)
    {
        // Effectively the channel category has been modified. So note
        // that now and send notification if required.
        SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_CATEGORY);
        SMSU_bNotify(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_vUpdateArtService
*
*   Update's this CHANNEL_OBJECT's channel art service handle.  If this
*   handle is valid, the CHANNEL_OBJECT will attempt to retrieve any channel
*   art associated with it.  If art was found, indicate to the App.
*
*****************************************************************************/
void CHANNEL_vUpdateArtService (
    CHANNEL_OBJECT hChannel,
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Store the channel art service handle if it has changed
    if(psObj->hChannelArtService != hChannelArtService)
    {
        CD_OBJECT hCDO;
        SMSAPI_RETURN_CODE_ENUM eReturn;
        BOOLEAN bIsSubscribed = FALSE;

        // It has changed, update with new service
        psObj->hChannelArtService = hChannelArtService;

        // Get the channel art for this channel now that the service is
        // up and running and make any notifications required.
        CHANNEL_vUpdateArt(hChannel, TRUE);

        // Now update the album art

        hCDO = CHANNEL.hCDO( hChannel );
        CDO_vUpdateArt( hCDO, hChannel );

        // As a workaround for all unsubscribed channels sharing the same
        // CDO, we have to manually issue updates if this channel is
        // unsubscribed. This way each channel will get one art event
        // at product startup (which is fine, as the unsubscribed
        // CDO art will never change.)

        eReturn = CHANNEL.eIsSubscribed( hChannel, &bIsSubscribed );

        // Make sure that worked

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_OBJECT_NAME": Couldn't get channel subscription status!"
                );
            return;
        }
        else if ( FALSE == bIsSubscribed )
        {
            // Update the channel's event mask with the art event
            CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_ART);
        }
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_hGetArtService
*
*   Friend function used CHANNEL_OBJECT's channel art service handle.
*   Used by the CDO "children" of this CHANNEL_OBJECT to grab the art
*   service handle for use in updating their art.
*
*****************************************************************************/
CHANNEL_ART_SERVICE_OBJECT CHANNEL_hGetArtService (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Return an invalid object on for an invalid
    // channel.
    if( CHANNEL_INVALID_OBJECT == hChannel )
    {
        return CHANNEL_ART_SERVICE_INVALID_OBJECT;
    }

    // Otherwise, return the art service handle
    return psObj->hChannelArtService;
}

/*****************************************************************************
*
*   CHANNEL_vUpdateArt
*
*   Update this CHANNEL_OBJECT's channel art handle.  If bNotify is TRUE,
*   (and the art has changed, indicate this update to the application. If
*   bNofity is FALSE, the caller of this function is responsible for
*   updating the channel with the proper event.
*
*****************************************************************************/
void CHANNEL_vUpdateArt (
    CHANNEL_OBJECT hChannel,
    BOOLEAN bNotify
        )
{
    CHANNEL_ART_OBJECT hOldArt;
    CHANNEL_ART_VERSION tArtVersion;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify input
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Check the state of the channel object
    if (psObj->tServiceId == SERVICE_INVALID_ID)
    {
        // The channel is not in a state
        // in which to acquire channel art.
        // Ensure the art handle has been cleared,
        // and get outta here
        psObj->hArt = CHANNEL_ART_INVALID_OBJECT;

        return;
    }

    // Keep current art object
    hOldArt = psObj->hArt;

    // Update the channel art object handle for this channel
    psObj->hArt = CHANNEL_ART_MGR_hGetArtForServiceId(
        psObj->hChannelArtService,
        psObj->tServiceId,
        &tArtVersion
            );

    // Check to see that the handle actually changed
    if (psObj->hArt == hOldArt)
    {
        if (psObj->tArtVersion == tArtVersion)
        {
            bNotify = FALSE;
        }
        else
        {
            psObj->tArtVersion = tArtVersion;
        }
    }

    if (bNotify == TRUE)
    {
        // Set event mask & notify
        CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_ART);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_vSetEvents
*
*****************************************************************************/
void CHANNEL_vSetEvents (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        // Set event mask per input
        SMSU_tUpdate(&psObj->sEvent, tEventMask);
        // Notify callback if registered and change occurred
        SMSU_bNotify(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_n16CompareChannelIds
*
*   This function is used to compare the 2 channels.
*   This is used when searching the LL for a specific channel.
*
*   Inputs:
*       *pvArg1 - pointer to one of the channels being compared
*       *pvArg2 - pointer to the other channel being compared
*
*   Outputs:
*       n16Result - The comparison between the channel passed in
*           and the channel of the struct currently being searched.
*          0 - Objects have the same value (equal, error )
*       > 0 - Object1 is greater than (after) Object2
*       < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
N16 CHANNEL_n16CompareChannelIds( void *pvArg1, void *pvArg2 )
{
    N16 n16Result = N16_MIN;
    CHANNEL_OBJECT_STRUCT *psObj1 = (CHANNEL_OBJECT_STRUCT *)pvArg1,
                          *psObj2 = (CHANNEL_OBJECT_STRUCT *)pvArg2;

    // Check pointers
    if((psObj1 != NULL) && (psObj2 != NULL))
    {
        // A channel is considered equal if the channel ids match.
        // We will never have multiple channels with the same channel id.
        n16Result = psObj1->tId - psObj2->tId;
    }

    return n16Result;
}

/*****************************************************************************
*
*   CHANNEL_n16CompareServiceIds
*
*   This function is used to compare the 2 channels.
*   This is used when searching the LL for a specific channel.
*
*   Inputs:
*       *pvArg1 - pointer to one of the channels being compared
*       *pvArg2 - pointer to the other channel being compared
*
*   Outputs:
*       n16Result - The comparison between the channel passed in
*           and the channel of the struct currently being searched.
*          0 - Objects have the same value (equal, error )
*       > 0 - Object1 is greater than (after) Object2
*       < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
N16 CHANNEL_n16CompareServiceIds( void *pvArg1, void *pvArg2 )
{
    N16 n16Result = N16_MIN;
    CHANNEL_OBJECT_STRUCT *psObj1 = (CHANNEL_OBJECT_STRUCT *)pvArg1,
                          *psObj2 = (CHANNEL_OBJECT_STRUCT *)pvArg2;

    // Check pointers
    if((psObj1 != NULL) && (psObj2 != NULL))
    {
        // A channel is considered equal if the service ids match.
        // We will never have multiple channels with the same service id.
        n16Result = psObj1->tServiceId - psObj2->tServiceId;
    }

    return n16Result;
}

/*****************************************************************************
*
*   CHANNEL_n16CompareAlternateIds
*
*   This function is used to compare the 2 channels.
*   This is used when searching the LL for a specific channel.
*
*   Inputs:
*       hChannel1 - one of the channels being compared
*       hChannel2 - other channel being compared
*
*   Outputs:
*       n16Result - The comparison between the channel passed in
*           and the channel of the struct currently being searched.
*
*         0 - Objects have the same value (equal, error )
*       > 0 - Object1 is greater than (after) Object2
*       < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
N16 CHANNEL_n16CompareAlternateIds(
    CHANNEL_OBJECT hChannel1,
    CHANNEL_OBJECT hChannel2
        )
{
    CHANNEL_OBJECT_STRUCT *psObj1 = (CHANNEL_OBJECT_STRUCT *)hChannel1,
                          *psObj2 = (CHANNEL_OBJECT_STRUCT *)hChannel2;

    if ((psObj1 != NULL) && (psObj2 != NULL))
    {
        if ((psObj1->tACO != CHANNEL_ACO_DEFAULT) &&
            (psObj2->tACO != CHANNEL_ACO_DEFAULT))
        {
            if (psObj1->tACO != psObj2->tACO)
            {
                return COMPARE( psObj1->tACO, psObj2->tACO );
            }
        }

        return COMPARE( psObj1->tId, psObj2->tId );
    }

    return N16_MIN;
}

/*******************************************************************************
*
*   CHANNEL_eSetAttribute
*
*****************************************************************************/
CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eSetAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner, bUpdate = FALSE;
    CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus = CHANNEL_ATTRIBUTE_STATUS_ERROR;
    CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        // Check if attribute already set, if so don't do anything
        if(!(psObj->tAttributes & tAttribute))
        {
            // Set this attribute
            psObj->tAttributes |= tAttribute;

            if (tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT)
            {
                // Set event mask for the unsubscribed CDO text change
                tEventMask = tGetEventsForSubscriptionAlertCDOChange(psObj);
            }
            else if (tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED)
            {
                // Set event mask for the unsubscribed CDO text change
                tEventMask = tGetEventsForUnsubscribedCDOChange(psObj);
            }

            // we don't want to set the attributes event for sub alert
            if ((tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT) ==
                CHANNEL_OBJECT_ATTRIBUTE_NONE)
            {
                // Set event mask for attribute change
                tEventMask |= CHANNEL_OBJECT_EVENT_ATTRIBUTES;
            }

            SMSU_tUpdate(&psObj->sEvent, tEventMask);

            // Notify callback if registered and change occurred
            bUpdate = SMSU_bNotify(&psObj->sEvent);

            if (bUpdate == TRUE)
            {
                // attribute changed
                eStatus = CHANNEL_ATTRIBUTE_STATUS_CHANGED;
            }
        }
        else
        {
            // It's already set.
            eStatus = CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE;
        }
    }

    return eStatus;
}

/*******************************************************************************
*
*   CHANNEL_eClearAttribute
*
*****************************************************************************/
CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eClearAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner, bUpdate = FALSE;
    CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus = CHANNEL_ATTRIBUTE_STATUS_ERROR;
    CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        // Check if attribute is already cleared, if so don't do anything
        if(psObj->tAttributes & tAttribute)
        {
            // Remove this attribute
            psObj->tAttributes &= (~tAttribute);

            if (tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT)
            {
                // Set event mask for the unsubscribed CDO text change
                tEventMask = tGetEventsForUnsubscribedCDOChange(psObj);
            }
            else if (tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED)
            {
                // Set event mask for the unsubscribed CDO text change
                tEventMask = tGetEventsForUnsubscribedCDOChange(psObj);
            }


            if ((tAttribute & CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT) ==
                CHANNEL_OBJECT_ATTRIBUTE_NONE)
            {
                // Set event mask for attribute change
                tEventMask |= CHANNEL_OBJECT_EVENT_ATTRIBUTES;
            }

            SMSU_tUpdate(&psObj->sEvent, tEventMask);

            // Notify callback if registered and change occurred
            bUpdate = SMSU_bNotify(&psObj->sEvent);

            if (bUpdate == TRUE)
            {
                // attribute changed
                eStatus = CHANNEL_ATTRIBUTE_STATUS_CHANGED;
            }
        }
        else
        {
            // It's already cleared.
            eStatus = CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE;
        }
    }

    return eStatus;
}

/*******************************************************************************
*
*   CHANNEL_eSetUnconfirmedAttribute
*
*****************************************************************************/
CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eSetUnconfirmedAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner;
    CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus = CHANNEL_ATTRIBUTE_STATUS_ERROR;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        // Check if attribute already set, if so don't do anything
        if(!(psObj->tUnconfirmedAttributes & tAttribute))
        {
            // Set this attribute
            psObj->tUnconfirmedAttributes |= tAttribute;

            // attribute changed
            eStatus = CHANNEL_ATTRIBUTE_STATUS_CHANGED;
        }
        else
        {
            // It's already set.
            eStatus = CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE;
        }
    }

    return eStatus;
}

/*******************************************************************************
*
*   CHANNEL_eClearUnconfirmedAttribute
*
*****************************************************************************/
CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eClearUnconfirmedAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner;
    CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus = CHANNEL_ATTRIBUTE_STATUS_ERROR;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        // Check if attribute is already cleared, if so don't do anything
        if(psObj->tUnconfirmedAttributes & tAttribute)
        {
            // Remove this attribute
            psObj->tUnconfirmedAttributes &= (~tAttribute);

            // attribute changed
            eStatus = CHANNEL_ATTRIBUTE_STATUS_CHANGED;
        }
        else
        {
            // It's already cleared.
            eStatus = CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE;
        }
    }

    return eStatus;
}

/*****************************************************************************
*
*   CHANNEL_bCompareAttributes
*
*****************************************************************************/
BOOLEAN CHANNEL_bCompareAttributes(
    CHANNEL_OBJECT hChan1,
    CHANNEL_OBJECT hChan2
        )
{
    BOOLEAN bOwner, bDifferent;

    bOwner = SMSO_bOwner((SMS_OBJECT)hChan1);
    bOwner &= SMSO_bOwner((SMS_OBJECT)hChan2);

    if (bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj1, *psObj2;
        psObj1 = (CHANNEL_OBJECT_STRUCT *)hChan1;
        psObj2 = (CHANNEL_OBJECT_STRUCT *)hChan2;

        if (psObj1->tAttributes != psObj2->tAttributes)
        {
            bDifferent = TRUE;
        }
        else
        {
            bDifferent = FALSE;
        }
    }
    else
    {
        bDifferent = TRUE;
    }

    return bDifferent;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateChannelId
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateChannelId(
    CHANNEL_OBJECT hChannel,
    CHANNEL_ID tChannelId
        )
{
    CCACHE_OBJECT hCCache;
    BOOLEAN bUpdate = FALSE;

    // Extract the channel's ccache handle
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
    if(hCCache != CCACHE_INVALID_OBJECT)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Different than the past?
        if(psObj->tId != tChannelId)
        {
            // Call upon CCACHE to perform update and list maintenance
            bUpdate = CCACHE_bUpdateChannelId(
                hChannel, tChannelId);
            // anything updated?
            if(bUpdate == TRUE)
            {
                // If both service id and channel id are good we can
                // consider this as updated.
                if( (psObj->tServiceId != SERVICE_INVALID_ID) &&
                        (psObj->tId != CHANNEL_INVALID_ID))
                {
                    // Set update attributes.
                    psObj->tAttributes |= CHANNEL_OBJECT_ATTRIBUTE_UPDATED;
                }
                else
                {
                    // clear update attributes.
                    psObj->tAttributes &= ~CHANNEL_OBJECT_ATTRIBUTE_UPDATED;
                }

                // Set event mask
                SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_CHANNEL_ID);

                // Notify callback if registered that change occurred
                SMSU_bNotify(&psObj->sEvent);
            }
        }
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateIds
*
*   CAUTION! YOU MUST KNOW WHAT YOU ARE DOING IF YOU CALL THIS API.
*   PLEASE NOTE THAT THIS API DIRECTLY MODIFIES THE SERVICE AND/OR CHANNEL
*   ID OF A CHANNEL_OBJECT PROVIDED. IT DOES SO WITHOUT INTERACTION FROM
*   OR KNOWLEDGE OF THE CHANNEL CCACHE. IF YOU THINK YOU WANT TO CALL THIS
*   YOU PROBABLY DONT. INSTEAD CALL CHANNEL_bUpdateChannelId().
*
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateIds (
    CHANNEL_OBJECT hChannel,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        )
{
    BOOLEAN bOwner, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Update ids. Never change the service id
        // to invalid however.
        if(tServiceId != SERVICE_INVALID_ID)
        {
            psObj->tServiceId = tServiceId;
        }
        psObj->tId = tChannelId;

        // If both service id and channel id are good we can
        // consider this as updated.
        if( (psObj->tServiceId != SERVICE_INVALID_ID) &&
                (psObj->tId != CHANNEL_INVALID_ID))
        {
            // Set update attributes.
            psObj->tAttributes |= CHANNEL_OBJECT_ATTRIBUTE_UPDATED;
        }
        else
        {
            // clear update attributes.
            psObj->tAttributes &= ~CHANNEL_OBJECT_ATTRIBUTE_UPDATED;
        }

        // Indicate an update occurred
        bUpdate = TRUE;
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateType
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateType(
    CHANNEL_OBJECT hChannel,
    CHANNEL_TYPE_ENUM eType
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        if(psObj->eType != eType)
        {
            // set type
            psObj->eType = eType;

            // Indicate an update occurred
            bUpdate = TRUE;
        }
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateShortName
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateShortName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {

        bUpdate = bUpdateStringLocal(
            psObj,
            &psObj->hShortName,
            GsRadio.sChannel.un16ShortNameMaxLen,
            pacName,
            CHANNEL_OBJECT_EVENT_NAME
        );
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateMediumName
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateMediumName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {

        bUpdate = bUpdateStringLocal(
            psObj,
            &psObj->hMediumName,
            GsRadio.sChannel.un16MediumNameMaxLen,
            pacName,
            CHANNEL_OBJECT_EVENT_NAME
        );
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateLongName
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateLongName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {

        bUpdate = bUpdateStringLocal(
            psObj,
            &psObj->hLongName,
            GsRadio.sChannel.un16LongNameMaxLen,
            pacName,
            CHANNEL_OBJECT_EVENT_NAME
        );
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateLongChanDesc
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateLongChanDesc(
    CHANNEL_OBJECT hChannel,
    const char *pacLongDesc
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {

        bUpdate = bUpdateStringLocal(
            psObj,
            &psObj->hLongDesc,
            GsRadio.sChannel.un16LongDescMaxLen,
            pacLongDesc,
            CHANNEL_OBJECT_EVENT_DESCRIPTION
        );
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateShortChanDesc
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateShortChanDesc(
    CHANNEL_OBJECT hChannel,
    const char *pacShortDesc
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {

        bUpdate = bUpdateStringLocal(
            psObj,
            &psObj->hShortDesc,
            GsRadio.sChannel.un16ShortDescMaxLen,
            pacShortDesc,
            CHANNEL_OBJECT_EVENT_DESCRIPTION
        );
    }

    return bUpdate;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateSimilarChannels
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateSimilarChannels(
    CHANNEL_OBJECT hChannel,
    UN32 un32NumSimilarChannels,
    SERVICE_ID *patSimilarChannels
        )
{
    CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
    BOOLEAN bUpdated = FALSE;

    do
    {
        BOOLEAN bOwner, bCompare = TRUE;

        bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
        if (bOwner == FALSE)
        {
            break;
        }

        if (psObj->sSimilarChannels.un32Items != un32NumSimilarChannels)
        {
            bCompare = FALSE;
            bUpdated = TRUE;
        }

        // Sorting incoming array for a simple binary compare
        if (un32NumSimilarChannels > 1)
        {
            BOOLEAN bSwapped = TRUE;
            UN32 un32Outer = un32NumSimilarChannels;
            SERVICE_ID tTempId;

            while (bSwapped == TRUE)
            {
                UN32 un32Inner;
                bSwapped = FALSE;

                un32Outer--;

                for (un32Inner = 0; un32Inner < un32Outer; un32Inner++)
                {
                    if (patSimilarChannels[un32Inner] > patSimilarChannels[un32Inner + 1])
                    {
                        tTempId = patSimilarChannels[un32Inner];
                        patSimilarChannels[un32Inner] = patSimilarChannels[un32Inner + 1];
                        patSimilarChannels[un32Inner + 1] = tTempId;

                        bSwapped = TRUE;
                    }
                }
            }
        }

        if (bCompare == TRUE)
        {
            int iResult;

            iResult = memcmp(
                psObj->sSimilarChannels.ptServiceIds,
                patSimilarChannels,
                sizeof(SERVICE_ID) * un32NumSimilarChannels
                    );

            if (iResult != 0)
            {
                bUpdated = TRUE;
            }
        }

        if (bUpdated == TRUE)
        {
            // If the current array size doesn't fit new data, the array
            // should be re-allocated
            if (psObj->sSimilarChannels.un32Size < un32NumSimilarChannels)
            {
                if (psObj->sSimilarChannels.ptServiceIds != NULL)
                {
                    SMSO_vDestroy((SMS_OBJECT)psObj->sSimilarChannels.ptServiceIds);

                    psObj->sSimilarChannels.un32Size = 0;
                    psObj->sSimilarChannels.un32Items = 0;
                }

                psObj->sSimilarChannels.ptServiceIds =
                    (SERVICE_ID *)SMSO_hCreate(
                        CHANNEL_OBJECT_NAME": Similar Channels",
                        sizeof(SERVICE_ID) * un32NumSimilarChannels,
                        SMS_INVALID_OBJECT,
                        FALSE);

                if (psObj->sSimilarChannels.ptServiceIds == NULL)
                {
                    break;
                }

                psObj->sSimilarChannels.un32Size = un32NumSimilarChannels;
            }

            if (un32NumSimilarChannels > 0)
            {
                BOOLEAN bSuccess;

                bSuccess = OSAL.bMemCpy(
                    psObj->sSimilarChannels.ptServiceIds,
                    patSimilarChannels,
                    sizeof(SERVICE_ID) * un32NumSimilarChannels
                        );

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_OBJECT_NAME": Failed to copy new similar channels");
                }
            }

            psObj->sSimilarChannels.un32Items = un32NumSimilarChannels;
        }

    } while (FALSE);

    if (bUpdated == TRUE)
    {
        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS);

        // Notify callback if registered that change occurred
        SMSU_bNotify(&psObj->sEvent);
    }

    return bUpdated;
}

/*******************************************************************************
*
*   CHANNEL_bUpdatePlayOnSelectMethod
*
********************************************************************************/
BOOLEAN CHANNEL_bUpdatePlayOnSelectMethod(
    CHANNEL_OBJECT hChannel,
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM ePlayOnSelectMethod
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // De-reference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        if (ePlayOnSelectMethod < CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B)
        {
            // Set new Play On Select Method value
            psObj->ePlayOnSelectMethod = ePlayOnSelectMethod;
        }
        else
        {
            psObj->ePlayOnSelectMethod = CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;
        }

        // Indicate an update occurred
        bUpdated = TRUE;
    }

    return bUpdated;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateContentType
*
********************************************************************************/
BOOLEAN CHANNEL_bUpdateContentType(
    CHANNEL_OBJECT hChannel,
    CONTENT_TYPE_ENUM eContentType
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        if (eContentType < CONTENT_TYPE_UNKNOWN)
        {
            // Set new Content Type value
            psObj->eContentType = eContentType;
        }
        else
        {
            psObj->eContentType = CONTENT_TYPE_UNKNOWN;
        }

        // Indicate an update occurred
        bUpdated = TRUE;
    }

    return bUpdated;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateIRNavigationClass
*
********************************************************************************/
BOOLEAN CHANNEL_bUpdateIRNavigationClass(
    CHANNEL_OBJECT hChannel,
    IR_NAVIGATION_CLASS_ENUM eIRNavigationClass
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        if (eIRNavigationClass < IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B)
        {
            // Set new IR Navigation CLass value
            psObj->eIRNavigationClass = eIRNavigationClass;
        }
        else
        {
            psObj->eIRNavigationClass = IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;
        }

        // Indicate an update occurred
        bUpdated = TRUE;
    }

    return bUpdated;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateACO
*
********************************************************************************/
BOOLEAN CHANNEL_bUpdateACO(
    CHANNEL_OBJECT hChannel,
    CHANNEL_ACO tACO
        )
{
    BOOLEAN bOwner, bUpdated = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        // De-reference object
        CHANNEL_OBJECT_STRUCT *psObj = 
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // This API should not know anything about protocol specific 
        // values and limitations. In general, this feature could be 
        // protocol independent (defaulted to 0xFFFFFFFF if not supported).
        // Radio should already preprocess this value and make it 
        // protocol independent.

        if (psObj->tACO != tACO)
        {
            // Set new list order
            psObj->tACO = tACO;

            // Extract the network category
            hCategory = (CATEGORY_OBJECT)
                OSAL.pvLinkedListThis(psObj->hReferenceCategoryEntry);

            // This function can be called to restore ACO values from NVM. 
            // However, in this case, since the whole procedure performed 
            // at the very initial point of the CCACHE's lifecycle, a network
            // category will not be assigned yet.

            // Update the network category
            if (hCategory != CATEGORY_INVALID_OBJECT)
            {
                CATEGORY_eSort(hCategory, CHANNEL_n16CompareAlternateIds);
            }

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, CHANNEL_OBJECT_EVENT_ACO);

            // Notify callback if registered that change occurred
            SMSU_bNotify(&psObj->sEvent);

            // Indicate that update occurred
            bUpdated = TRUE;
        }
    }

    return bUpdated;
}

/*******************************************************************************
*
*   CHANNEL_ePlayOnSelectMethod
*
********************************************************************************/
CHANNEL_PLAY_ON_SELECT_METHOD_ENUM CHANNEL_ePlayOnSelectMethod(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM ePlayOnSelectMethod =
        CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        ePlayOnSelectMethod = psObj->ePlayOnSelectMethod;
    }

    return ePlayOnSelectMethod;
}

/*******************************************************************************
*
*   CHANNEL_eContentType
*
********************************************************************************/
CONTENT_TYPE_ENUM CHANNEL_eContentType(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;
    CONTENT_TYPE_ENUM eContentType =
        CONTENT_TYPE_LIVE_AT;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        eContentType = psObj->eContentType;
    }

    return eContentType;
}

/*******************************************************************************
*
*   CHANNEL_eIRNavigationClass
*
********************************************************************************/
IR_NAVIGATION_CLASS_ENUM CHANNEL_eIRNavigationClass(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;
    IR_NAVIGATION_CLASS_ENUM eIRNavigationClass =
        IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // Dereference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        eIRNavigationClass = psObj->eIRNavigationClass;
    }

    return eIRNavigationClass;
}

/*******************************************************************************
*
*   CHANNEL_bUpdateCategoryId
*
*****************************************************************************/
BOOLEAN CHANNEL_bUpdateCategoryId(
    CHANNEL_OBJECT hChannel,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bOwner = FALSE, bUpdate = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CATEGORY_ID tCurrentCatgory = CATEGORY_INVALID_ID;
        CATEGORY_OBJECT hCurrentCategory;
        CCACHE_OBJECT hCCache;

        // get broadcast category (if any)
        hCurrentCategory = CHANNEL.hCategory(hChannel, 0);
        if(hCurrentCategory != CATEGORY_INVALID_OBJECT)
        {
            tCurrentCatgory = CATEGORY.tGetCategoryId(hCurrentCategory);
        }

        // Extract the channel's ccache handle
        hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
        if(hCCache != CCACHE_INVALID_OBJECT)
        {
            // Identify the category associated with this channel, so we need to
            // find one or create it and then associate it. If we do
            // not have a valid category id, then we should remove any
            // associated category object.
            if(tCategoryId == CATEGORY_INVALID_ID)
            {
                // A reference (broadcast) category is no longer associated
                // with this channel.

                if (hCurrentCategory != CATEGORY_INVALID_OBJECT)
                {
                    CHANNEL_vRemoveCategory(hChannel, hCurrentCategory, FALSE);
                }
            }
            else
            {
                if (tCategoryId != tCurrentCatgory)
                {
                    CATEGORY_OBJECT hCategory;

                    // We do have a category
                    hCategory = CCACHE_hCategory(hCCache, &tCategoryId, 0);
                    if(hCategory != CATEGORY_INVALID_OBJECT)
                    {
                        // Associate this category with this channel. This will
                        // also make it the reference category as well. It will also
                        // update any required events (CATEGORY) based on change.
                        // If it is already associated then no harm is done.
                        CHANNEL_bAddCategory(hChannel, hCategory);
                    }
                }
            }
        }
    }

    return bUpdate;
}

/*****************************************************************************
*
*   CHANNEL_vClearBroadcast
*
*****************************************************************************/
void CHANNEL_vClearBroadcast (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    CD_OBJECT hCDO;

    // clear out all channel info (set to defaults) except for
    // tServiceId which is never set to default.
    CHANNEL_bUpdateChannelId(hChannel, CHANNEL_INVALID_ID);
    CHANNEL_bUpdateType(hChannel, CHANNEL_TYPE_UNKNOWN);

    // Do not clear all attributes, since some are set by
    // the application and NOT broadcast. only clear out
    // broadcast attributes.
    psObj->tAttributes &= ~CHANNEL_OBJECT_ATTRIBUTE_BROADCAST;

    CHANNEL_bUpdateShortName(hChannel, "");
    CHANNEL_bUpdateLongName(hChannel, "");
    CHANNEL_bUpdateCategoryId(hChannel, CATEGORY_INVALID_ID);

    // tAttributes such as locked, skipped and subscribed
    // will remain since those aspects are not related to broadcast.
    psObj->tAttributes &= ~CHANNEL_OBJECT_ATTRIBUTE_UPDATED;

    // Set extended channel metadata attributes to default
    psObj->ePlayOnSelectMethod = CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B;
    psObj->eContentType = CONTENT_TYPE_UNKNOWN;
    psObj->eIRNavigationClass = IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B;
    psObj->tACO = CHANNEL_ACO_DEFAULT;

    // Clear out the CDO (make it unknown)

    // Clear out all the common CDO fields.
    CDO_vUpdate(hChannel, "", CDO_FIELD_ALL);

    // Clear out the CDO (make it unknown)
    hCDO = CHANNEL_hCDO(hChannel, TRUE);
    CDO_bAssignType(hCDO, CDO_UNKNOWN);

    // hCategoryList will remain since all that remains is this
    // channel's membership in user/virtual categories.

    // hChannelArtManager will remain since the art manager never changes
    // hArt will remain since it is managed by the art manager

    // tUsageCounter will not be modified since this channel object
    // remains in existence.

    // Signal all events for this channel but service id which has not
    // changed and channel removed.
    SMSU_tUpdate(&psObj->sEvent,
        CHANNEL_OBJECT_EVENT_ALL &
            ~(CHANNEL_OBJECT_EVENT_SERVICE_ID | CHANNEL_OBJECT_EVENT_REMOVED)
            );

    return;
}

/*******************************************************************************
*
*   CHANNEL_bInUse
*
* This API is used to determine if a CHANNEL object is currently being used
* by another object. In use could bean that any of the channel attributes
* are set by the application or another SMS object is using the channel by
* reference (as indicated by notification registration). Currently this
* API is used by the Channel Cache to determine if a CHANNEL reference
* (handle) can be re-assigned (destroyed or otherwise) or if it must
* remain.
*
*****************************************************************************/
BOOLEAN CHANNEL_bInUse(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner, bInUse = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj = (CHANNEL_OBJECT_STRUCT *)hChannel;
        bInUse = ((
            (psObj->tUsageCounter == 0) && // Nobody using
            !(psObj->tAttributes & // No attributes other than default
                (CHANNEL_OBJECT_ATTRIBUTE_LOCKED |
                 CHANNEL_OBJECT_ATTRIBUTE_SKIPPED))
        ) ?
        FALSE : TRUE
            );
     }

    return bInUse;
}

/*****************************************************************************
*
* CHANNEL_bNeedsUpdated
*
* This API is used to determine if a CHANNEL object needs to be updated
* or reshreshed with complete channel information from the hardware.
*
*****************************************************************************/
BOOLEAN CHANNEL_bNeedsUpdated (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner, bNeedsUpdated = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Check if the attributes contain a updated flag
        if (!(( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_UPDATED ) ==
            CHANNEL_OBJECT_ATTRIBUTE_UPDATED ))
        {
            bNeedsUpdated = TRUE;
        }
    }

    return bNeedsUpdated;
}

/*****************************************************************************
*
*   CHANNEL_bBlockNotifications
*
* Note! This function is not-recurisve, it must be used in conjunction with
* vReleaseNotifications within the same CHANNEL object. So this means you
* should only call this API to block notifications, do your thing and then
* release notifications.
* This is to be used when the caller wishes to make multiple CHANNEL
* object updates but wants to block notifications until they are all finished.
*
* Inputs:
*   hChannel - A valid channel for which to block (filter) all future
*   channel update notifications.
*
* Returns:
*   BOOLEAN
*
*****************************************************************************/
BOOLEAN CHANNEL_bBlockNotifications (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner, bBlocked = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // We can only block further channel updates (notifications)
        // if our filter holding element is ALL. Otherwise this means
        // someone else has already filtered something in which case
        // this wont need to do anything anyway.
        if(psObj->tFilter == CHANNEL_OBJECT_EVENT_ALL)
        {
            // Record current filter (if any) and filter all further
            // notifications. We record it here so we can restore it later.
            psObj->tFilter =
                SMSU_tFilter(&psObj->sEvent, CHANNEL_OBJECT_EVENT_ALL);

            // Indicate channel notifications are blocked
            bBlocked = TRUE;
        }
    }

    return bBlocked;
}

/*****************************************************************************
*
*   CHANNEL_vReleaseNotifications
*
* Processes any pending notifications and restores the CHANNEL object's
* filter setting before notifications we blocked.
*
* Inputs:
*   hChannel - A valid channel for which to release (un-filter) all future
*   channel update notifications which were previously blocked.
*
* Returns:
*   None.
*
*****************************************************************************/
void CHANNEL_vReleaseNotifications (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Restore previous filter and process any pending notifications.

        // We will only release notifications if we actually have some
        // previously stored filter (other than ALL) otherwise dont
        // do anything.
        if(psObj->tFilter != CHANNEL_OBJECT_EVENT_ALL)
        {
            SMSAPI_EVENT_MASK tRestoreFilter = psObj->tFilter;

            // First thing we do is un-filter everything and record the
            // previous filter mask (should be ALL)
            // This will also return our "filter hold" to ALL
            psObj->tFilter =
                SMSU_tUnFilter(&psObj->sEvent, CHANNEL_OBJECT_EVENT_ALL);

            // Now we filter what was filtered before we got to it.
            SMSU_tFilter(&psObj->sEvent, tRestoreFilter);
        }

        // Process any pending notifications
        SMSU_bNotify(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_hCDO
*
* This API is similar to the public API CHANNEL.hCDO. However the public
* CHANNEL.hCDO may automatically substitute the "unsubscibed" channel
* information (shared by all channels) rather than the actual channel CDO
* handle when the channel is unsubscibed. When attempting to update a
* channel's CDO contents this is not desirable since it is not the
* reserved unsubscribed channel's attributes we want to modify, but rather
* the actual CHANNEL's attributes. So we can provide an argument to indicate
* exactly which one we want.
*
* Inputs:
*   hChannel - A valid channel handle for which to extract the CDO handle from.
*   bOverrideSubscription - A flag to indicate where the CDO should come from
*       if a channel is presently unsubscribed. If TRUE the CDO returned will
*       be that of the provided CHANNEL_OBJECT regardless of subscription
*       status. If FALSE, the CDO returned is the special "unsubscribed"
*       CDO which is shared by all unsubscribed channels. If the provided
*       channel is subscribed the behavior is the same regardless of this
*       flag where the CDO of the provided CHANNEL_OBJECT is always provided.
*
* Returns:
*   CD_OBJECT - A valid CD_OBJECT is one can be extracted. Otherwise
*       CD_INVALID_OBJECT.
*
*
*****************************************************************************/
CD_OBJECT CHANNEL_hCDO (
    CHANNEL_OBJECT hChannel,
    BOOLEAN bOverrideSubscription
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    CD_OBJECT hCDO = CD_INVALID_OBJECT;
    BOOLEAN bValid = FALSE;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);
    if( bValid == TRUE)
    {
        CCACHE_OBJECT hCCache;

        // Extract CCache Handle from the parent
        hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);

        if ((( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT ) ==
                   CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT) &&
            // We don't want to override subscription
            (bOverrideSubscription == FALSE))
        {
            // this channel isn't subscribed - return the
            // unsubscribed CDO instead

            CHANNEL_OBJECT_STRUCT *psSubAlertChannelObj;

            // Extract special unsubscribed channel from the channel cache
            // and de-reference the pointer
            psSubAlertChannelObj = (CHANNEL_OBJECT_STRUCT *)
                CCACHE_hSubscriptionAlertChannel(hCCache);
            if(psSubAlertChannelObj != NULL)
            {
                // Return unsubscribed CHANNEL's CDO handle
                hCDO = psSubAlertChannelObj->hCDO;
            }
        }
        // Is this channel un-subscribed?
        else if ((( psObj->tAttributes & CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED ) !=
                   CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED) &&
            // We don't want to override subscription
            (bOverrideSubscription == FALSE))
        {
            // this channel isn't subscribed - return the
            // unsubscribed CDO instead

            CHANNEL_OBJECT_STRUCT *psUnsubscribedChannelObj;

            // Extract special unsubscribed channel from the channel cache
            // and de-reference the pointer
            psUnsubscribedChannelObj = (CHANNEL_OBJECT_STRUCT *)
                CCACHE_hUnsubscribedChannel(hCCache);
            if(psUnsubscribedChannelObj != NULL)
            {
                // Return unsubscribed CHANNEL's CDO handle
                hCDO = psUnsubscribedChannelObj->hCDO;
            }
        }
        else
        {
            // Use this channel's CDO
            hCDO = psObj->hCDO;
        }
    }

    // Return CHANNEL's CDO handle
    return hCDO;
}

/*****************************************************************************
*
*   CHANNEL_vSetBasePresetHdl
*
* This API is invoked by the presets service in order to provide this
* channel with its first (or base) preset entry.  This allows the channel
* to have a starting point when iterating through its assigned presets.
*
* Inputs:
*   hChannel - A valid channel handle for which to set the preset handle.
*   hPresetHdl - A A handle to a preset entry which has been assigned to
*       this channel.
*
* Returns:
*   None
*
*****************************************************************************/
void CHANNEL_vSetBasePresetHdl (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        // Set the preset handles (whenever the base
        // is updated we need to update the current as well)
        psObj->hBasePresetHdl = hPresetHdl;
        psObj->hCurrentPresetHdl = hPresetHdl;
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_hGetBasePresetHdl
*
* This API is invoked by the presets service in order to get the base preset
* handle currently associated with this channel.
*
* Inputs:
*   hChannel - A valid channel handle for which to set the preset handle.
*
* Returns:
*   A CHANNEL_PRESET_HDL on success, CHANNEL_PRESET_INVALID_HDL on error.
*
*****************************************************************************/
CHANNEL_PRESET_HDL CHANNEL_hGetBasePresetHdl (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_PRESET_HDL hPresetHdl =
        CHANNEL_PRESET_INVALID_HDL;
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        hPresetHdl = psObj->hBasePresetHdl;
    }

    return hPresetHdl;
}

/*****************************************************************************
*
*   CHANNEL_vSetCurrentPresetHdl
*
* This API is invoked by the presets service in order to set the current
* preset handle for this channel.
*
* Inputs:
*   hChannel - A valid channel handle for which to set the preset handle.
*
* Returns:
*   None
*
*****************************************************************************/
void CHANNEL_vSetCurrentPresetHdl (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;

    // Verify inputs
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        // Set the current preset handle
        psObj->hCurrentPresetHdl = hPresetHdl;
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_un32NumSimilarChannels
*
*****************************************************************************/
UN32 CHANNEL_un32NumSimilarChannels (
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT_STRUCT *psObj =
        (CHANNEL_OBJECT_STRUCT *)hChannel;
    UN32 un32NumSimilarChannels = 0;

    // Verify inputs
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        // Get number of similar channels
        un32NumSimilarChannels = psObj->sSimilarChannels.un32Items;
    }

    return un32NumSimilarChannels;
}

/*******************************************************************************
 *
 *   CHANNEL_vSetProgramsList
 *
 *****************************************************************************/
void CHANNEL_vSetProgramsList (
        CHANNEL_OBJECT hChannel,
        EPG_CHANNEL_OBJECT hEpgChannel
)
{
    CHANNEL_OBJECT_STRUCT *psChannel = NULL;
    psChannel = (CHANNEL_OBJECT_STRUCT*) hChannel;
    if (psChannel != NULL)
    {
        printf( "Changing hEpgObjectsList from %p to %p\n",
                psChannel->hEpgObjectsList,
                hEpgChannel);
        psChannel->hEpgObjectsList = hEpgChannel;
        // Set event mask & notify
        CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_EPG);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_hCidCreate
*
* Create a CID which is modifiable, meaning it may be created, modified and
* later destroyed. Recycling of non-constant CIDs will also be handled
* here as well. This is done via the CHANNEL object.
*
* Inputs:
*   hChannel - CHANNEL associated with this CID
*   eType - The type of CID object to create.
*   pvObjectDataPtr - The raw source data from which to create this object.
*
* Outputs:
*   CID_OBJECT - a new or recycled CID which represents the content provided.
*
*****************************************************************************/
CID_OBJECT CHANNEL_hCidCreate (
    CHANNEL_OBJECT hChannel,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        )
{
    CID_OBJECT hCid = CID_INVALID_OBJECT;
    CCACHE_OBJECT hCCache;

    // The parent of a CDO is a CCACHE
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
    if(hCCache != CCACHE_INVALID_OBJECT)
    {
        hCid = CCACHE_hCidCreate(hCCache, eType, pvObjectDataPtr);
    }

    return hCid;
}

/*****************************************************************************
*
*   CHANNEL_pacPlayOnSelectMethod
*
*****************************************************************************/
const char *CHANNEL_pacPlayOnSelectMethod(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;
    char *pacPlayOnSelectMethod = "Default";

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // De-reference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        switch (psObj->ePlayOnSelectMethod)
        {
            case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_A:
            {
                pacPlayOnSelectMethod = "Newest (0)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_B:
            {
                pacPlayOnSelectMethod = "Newest (1)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_C:
            {
                pacPlayOnSelectMethod = "Newest (2)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_D:
            {
                pacPlayOnSelectMethod = "Newest (3)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_A:
            {
                pacPlayOnSelectMethod = "Constrained (4)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_B:
            {
                pacPlayOnSelectMethod = "Constrained (5)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_A:
            {
                pacPlayOnSelectMethod = "Realtime (6)";
            }
            break;

            case CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B:
            {
                pacPlayOnSelectMethod = "Realtime (7)";
            }
            break;
        }
    }

    return pacPlayOnSelectMethod;
}

/*****************************************************************************
*
*   CHANNEL_pacContentType
*
*****************************************************************************/
const char *CHANNEL_pacContentType(
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner;
    char *pacContentType = "Default";

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if (bOwner == TRUE)
    {
        // De-reference object
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        switch (psObj->eContentType)
        {
            case CONTENT_TYPE_MUSIC_PID:
            {
                pacContentType = "Music/PID (0)";
            }
            break;

            case CONTENT_TYPE_MUSIC_AT:
            {
                pacContentType = "Music/PAD (1)";
            }
            break;

            case CONTENT_TYPE_TALK_PID:
            {
                pacContentType = "Talk/PID (2)";
            }
            break;

            case CONTENT_TYPE_TALK_AT:
            {
                pacContentType = "Talk/PAD (3)";
            }
            break;

            case CONTENT_TYPE_LIVE_PID:
            {
                pacContentType = "Live/PID (4)";
            }
            break;

            case CONTENT_TYPE_LIVE_AT:
            {
                pacContentType = "Live/PAD (5)";
            }
            break;

            case CONTENT_TYPE_UNKNOWN:
            {
                pacContentType = "Unknown";
            }
            break;
        }
    }

    return pacContentType;
}

/*****************************************************************************
*
*       CHANNEL_eIsLockedUnconfirmed
*
*       This API is used to check if a specific channel is locked.  This API may
*       only be used in the context of the Decoder's Event Callback.
*
*       Inputs:
*           hChannel - A handle to a valid CHANNEL object to check the lock
*               attribute for.
*
*       Outputs:
*           pbLocked - pointer to a BOOLEAN that will hold the value of TRUE if
*               the channel is locked or FALSE if the channel is not locked.
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*               SMSAPI_RETURN_CODE_ERROR on error if the channel id cannot be
*               found/invalid.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CHANNEL_eIsLockedUnconfirmed (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbLocked
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if (pbLocked == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Check if the attributes contain a channel lock
        if ((psObj->tUnconfirmedAttributes & CHANNEL_OBJECT_ATTRIBUTE_LOCKED) ==
             CHANNEL_OBJECT_ATTRIBUTE_LOCKED)
        {
            *pbLocked = TRUE;
        }
        else
        {
            *pbLocked = FALSE;
        }

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       CHANNEL_eIsSkippedUnconfirmed
*
*       This API is used to check if a specific channel is skipped. This API may
*       only be used in the context of the Decoder's Event Callback.
*
*       Inputs:
*           hChannel - A handle to a valid CHANNEL object to check the skip
*               attribute for.
*
*       Outputs:
*           pbSkipped - pointer to a BOOLEAN that will hold the value of TRUE if
*               the channel is skipped or FALSE if the channel is not skipped.
*           Returns - SMSAPI_RETURN_CODE_SUCCESS on success,
*               SMSAPI_RETURN_CODE_ERROR on error if the channel id cannot be
*               found/invalid.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CHANNEL_eIsSkippedUnconfirmed (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbSkipped
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if (pbSkipped == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT_STRUCT *psObj =
            (CHANNEL_OBJECT_STRUCT *)hChannel;

        // Check if the attributes contain a channel lock
        if ((psObj->tUnconfirmedAttributes & CHANNEL_OBJECT_ATTRIBUTE_SKIPPED) ==
             CHANNEL_OBJECT_ATTRIBUTE_SKIPPED)
        {
            *pbSkipped = TRUE;
        }
        else
        {
            *pbSkipped = FALSE;
        }

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
 *
 *   CHANNEL_tCompareContent
 *
 *****************************************************************************/
CHANNEL_EVENT_MASK CHANNEL_tCompareContent(
    CHANNEL_OBJECT hOldChannel,
    CHANNEL_OBJECT hNewChannel
        )
{
    CHANNEL_EVENT_MASK tMask = CHANNEL_OBJECT_EVENT_NONE;
    CHANNEL_ID tOldChannelId, tNewChannelId;
    SERVICE_ID tOldServiceId, tNewServiceId;
    CATEGORY_OBJECT hOldCategory, hNewCategory;
    STRING_OBJECT hOldString, hNewString;
    CHANNEL_ART_OBJECT hOldArt, hNewArt;
    CD_OBJECT hOldCDO, hNewCDO;
    N16 n16Diff;
    BOOLEAN bDifferent;
    CHANNEL_SIMILAR_COMPARE_STRUCT sCompare;
    UN32 un32OldNumSimChans, un32NewNumSimChans;
    CHANNEL_ACO tACO1, tACO2;

    // Compare SIDs
    tOldServiceId = CHANNEL.tServiceId(hOldChannel);
    tNewServiceId = CHANNEL.tServiceId(hNewChannel);
    if (tOldServiceId != tNewServiceId)
    {
        tMask |= CHANNEL_OBJECT_EVENT_SERVICE_ID;
    }

    // Compare Channel IDs
    tOldChannelId = CHANNEL.tChannelId(hOldChannel);
    tNewChannelId = CHANNEL.tChannelId(hNewChannel);
    if (tOldChannelId != tNewChannelId)
    {
        tMask |= CHANNEL_OBJECT_EVENT_CHANNEL_ID;
    }

    // Compare channel attributes
    bDifferent = CHANNEL_bCompareAttributes(hOldChannel, hNewChannel);
    if (bDifferent == TRUE)
    {
        tMask |= CHANNEL_OBJECT_EVENT_ATTRIBUTES;
    }

    // Compare names
    // Match short names first
    hOldString = CHANNEL.hShortName(hOldChannel);
    hNewString = CHANNEL.hShortName(hNewChannel);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_NAME;
        }
        else
        {
            // Now match medium names
            hOldString = CHANNEL.hMediumName(hOldChannel);
            hNewString = CHANNEL.hMediumName(hNewChannel);

            if ((hOldString != STRING_INVALID_OBJECT) &&
                (hNewString != STRING_INVALID_OBJECT))
            {
                n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
                if (n16Diff != 0)
                {
                    tMask |= CHANNEL_OBJECT_EVENT_NAME;
                }
                else
                {
                    // Now match long names
                    hOldString = CHANNEL.hLongName(hOldChannel);
                    hNewString = CHANNEL.hLongName(hNewChannel);

                    if ((hOldString != STRING_INVALID_OBJECT) &&
                        (hNewString != STRING_INVALID_OBJECT))
                    {
                        n16Diff = STRING.n16Compare(hOldString, hNewString, 
                            TRUE);
                        if (n16Diff != 0)
                        {
                            tMask |= CHANNEL_OBJECT_EVENT_NAME;
                        }
                    }
                    else if (hOldString != hNewString)
                    {
                        tMask |= CHANNEL_OBJECT_EVENT_NAME;
                    }
                }
            }
            else if (hOldString != hNewString)
            {
                tMask |= CHANNEL_OBJECT_EVENT_NAME;
            }
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_NAME;
    }

    // Compare channel categories
    hOldCategory = CHANNEL.hCategory(hOldChannel, 0);
    hNewCategory = CHANNEL.hCategory(hNewChannel, 0);
    if (hOldCategory != hNewCategory)
    {
        tMask |= CHANNEL_OBJECT_EVENT_CATEGORY;
    }

    // Compare broadcasted content
    hOldCDO = CHANNEL.hCDO(hOldChannel);
    hNewCDO = CHANNEL.hCDO(hNewChannel);

    // Compare content titles
    hOldString = CDO.hTitle(hOldCDO);
    hNewString = CDO.hTitle(hNewCDO);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_TITLE;
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_TITLE;
    }

    // Compare content album names
    hOldString = CDO.hAlbum(hOldCDO);
    hNewString = CDO.hAlbum(hNewCDO);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_ALBUM;
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_ALBUM;
    }

    // Compare content artists
    hOldString = CDO.hArtist(hOldCDO);
    hNewString = CDO.hArtist(hNewCDO);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_ARTIST;
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_ARTIST;
    }

    // Compare composers
    hOldString = CDO.hComposer(hOldCDO);
    hNewString = CDO.hComposer(hNewCDO);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_COMPOSER;
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_COMPOSER;
    }

    // Compare content info
    hOldString = CDO.hContentInfo(hOldCDO);
    hNewString = CDO.hContentInfo(hNewCDO);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_CONTENTINFO;
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_CONTENTINFO;
    }

    // Compare descriptions
    hOldString = CHANNEL.hShortDescription(hOldChannel);
    hNewString = CHANNEL.hShortDescription(hNewChannel);

    if ((hOldString != STRING_INVALID_OBJECT) &&
        (hNewString != STRING_INVALID_OBJECT))
    {
        n16Diff = STRING.n16Compare(hOldString, hNewString, TRUE);
        if (n16Diff != 0)
        {
            tMask |= CHANNEL_OBJECT_EVENT_DESCRIPTION;
        }
        else
        {
            hOldString = CHANNEL.hLongDescription(hOldChannel);
            hNewString = CHANNEL.hLongDescription(hNewChannel);

            if ((hOldString != STRING_INVALID_OBJECT) &&
                (hNewString != STRING_INVALID_OBJECT))
            {
                n16Diff = STRING.n16Compare(hOldString, hNewString, 
                    TRUE);
                if (n16Diff != 0)
                {
                    tMask |= CHANNEL_OBJECT_EVENT_DESCRIPTION;
                }
            }
            else if (hOldString != hNewString)
            {
                tMask |= CHANNEL_OBJECT_EVENT_DESCRIPTION;
            }
        }
    }
    else if (hOldString != hNewString)
    {
        tMask |= CHANNEL_OBJECT_EVENT_DESCRIPTION;
    }

    // Compare channel arts
    hOldArt = CHANNEL.hArt(hOldChannel);
    hNewArt = CHANNEL.hArt(hNewChannel);
    if (hOldArt != hNewArt)
    {
        tMask |= CHANNEL_OBJECT_EVENT_ART;
    }

    // get number of similar channels on each channel.
    un32OldNumSimChans = CHANNEL_un32NumSimilarChannels(hOldChannel);
    un32NewNumSimChans = CHANNEL_un32NumSimilarChannels(hNewChannel);

    if (un32OldNumSimChans != un32NewNumSimChans)
    {
        // if different set event
        tMask |= CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS;
    }
    else
    {
        // otherwise dig into the list
        // iterate old channel's similar channel list
        // pass in new channel
        // if any channels different
        sCompare.hChannel = hNewChannel;
        sCompare.bDifferent = FALSE;

        CHANNEL.eIterateSimilarChannels(
            hOldChannel,
            bCompareSimilarChannelsOuter,
            &sCompare);

        if (sCompare.bDifferent == TRUE)
        {
            tMask |= CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS;
        }
    }

    // Compare ACO values
    tACO1 = CHANNEL.tACO(hOldChannel);
    tACO2 = CHANNEL.tACO(hNewChannel);

    if (tACO1 != tACO2)
    {
        tMask |= CHANNEL_OBJECT_EVENT_ACO;
    }

    // Compare content types
    n16Diff = CDO.n16Compare(hOldCDO, hNewCDO);
    if (n16Diff != 0)
    {
        tMask |= CHANNEL_OBJECT_EVENT_TYPE_INFO;
    }

    return tMask;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bRemoveChannelFromCategory
*
*   The purpose of this function is used as a linked list iterator
*   to remove a category from a channel
*
*****************************************************************************/
static BOOLEAN bRemoveChannelFromCategory(
    void *pvData,
    void *pvArg
        )
{
    // remove the channel from the category
    CATEGORY_vRemoveChannel((CATEGORY_OBJECT)pvData, (CHANNEL_OBJECT)pvArg, FALSE);

    // keep iterating
    return TRUE;
}

/*******************************************************************************
*
*   bUpdateStringLocal
*
*****************************************************************************/
static BOOLEAN bUpdateStringLocal(
    CHANNEL_OBJECT_STRUCT *psObj,
    STRING_OBJECT *phString,
    UN16 un16MaxLength,
    const char *pacName,
    SMSAPI_EVENT_MASK tMask
        )
{
    BOOLEAN bUpdate = FALSE;

    // Update channel long name...
    if(*phString == STRING_INVALID_OBJECT)
    {
        // Create a STRING object to hold the long name
        *phString =
            STRING_hCreate(
                (SMS_OBJECT)psObj,
                pacName,
                strlen(pacName),
                un16MaxLength
                    );

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, tMask);

        // Notify callback if registered that change occurred
        SMSU_bNotify(&psObj->sEvent);

        // Indicate an update occurred
        bUpdate = TRUE;
    }
    else
    {
        // Update long name (if different than current)
        bUpdate =
            STRING.bModifyCStr(
                *phString,
                pacName
                    );

        if(bUpdate == TRUE)
        {
            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, tMask);

            // Notify callback if registered that change occurred
            SMSU_bNotify(&psObj->sEvent);
        }
    }


    return bUpdate;
}

/*******************************************************************************
*
*   tGetEventsForUnsubscribedCDOChange
*
*****************************************************************************/
static CHANNEL_EVENT_MASK tGetEventsForUnsubscribedCDOChange (
    CHANNEL_OBJECT_STRUCT *psObj
        )
{
    CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT_STRUCT *psUnsubscribedChannelObj;

    // Extract CCache Handle from the parent
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    // Extract special unsubscribed channel from the channel cache
    // and de-reference the pointer
    psUnsubscribedChannelObj = (CHANNEL_OBJECT_STRUCT *)
        CCACHE_hUnsubscribedChannel(hCCache);

    // get the mask
    tEventMask = tGetEventsForCDOChange(psObj, psUnsubscribedChannelObj);

    return tEventMask;
}

/*******************************************************************************
*
*   tGetEventsForSubscriptionAlertCDOChange
*
*****************************************************************************/
static CHANNEL_EVENT_MASK tGetEventsForSubscriptionAlertCDOChange (
    CHANNEL_OBJECT_STRUCT *psObj
        )
{
    CHANNEL_EVENT_MASK tEventMask;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT_STRUCT *psSubAlertChannelObj;

    // Extract CCache Handle from the parent
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

    // Extract special unsubscribed channel from the channel cache
    // and de-reference the pointer
    psSubAlertChannelObj = (CHANNEL_OBJECT_STRUCT *)
        CCACHE_hSubscriptionAlertChannel(hCCache);

    // get the mask
    tEventMask = tGetEventsForCDOChange(psObj, psSubAlertChannelObj);

    return tEventMask;
}

/*****************************************************************************
*
*   bAlwaysInclude
*
*****************************************************************************/
static BOOLEAN bAlwaysInclude (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    return TRUE;
}

/*****************************************************************************
*
*   bIteratorShim
*
*****************************************************************************/
static BOOLEAN bIteratorShim (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    CHANNEL_ITERATOR_STRUCT *psIterator = (CHANNEL_ITERATOR_STRUCT *)pvArg;
    BOOLEAN bInclude, bContinue = TRUE;

    // For this object, run it's comparator first to decide if it
    // is to be included.
    bInclude = psIterator->bInclude(hChannel, psIterator->pvArg);
    if(bInclude == TRUE)
    {
        // Go ahead...include it
        bContinue = psIterator->bIterator(hChannel, psIterator->pvArg);
    }

    return bContinue; // Keep iterating?
}

/*****************************************************************************
*
*   bRemoveCategory
*
*****************************************************************************/
static BOOLEAN bRemoveCategory (
    CHANNEL_OBJECT_STRUCT *psObj,
    CATEGORY_OBJECT hCategory,
    BOOLEAN bSuppressCatEvents
        )
{
    BOOLEAN bRemoved = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (hCategory == CATEGORY_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Start from first entry
    hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Search category list to see if this category is a member
    eReturnCode =
        OSAL.eLinkedListSearch(
            psObj->hCategoryList,
            &hEntry,
            (void *)hCategory );

    if(eReturnCode == OSAL_SUCCESS) // found
    {
        // If we removed the reference entry, be sure to NULL it now
        if(hEntry == psObj->hReferenceCategoryEntry)
        {
            eReturnCode = OSAL.eLinkedListReplaceEntry(
                psObj->hCategoryList,
                psObj->hReferenceCategoryEntry,
                (void *)CATEGORY_INVALID_OBJECT
                    );
        }
        else
        {
            // Remove entry
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
        }

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Flag that we removed the category
            bRemoved = TRUE;
        }
    }

    // Remove the provided channel from the category provided
    CATEGORY_vRemoveChannel(hCategory, (CHANNEL_OBJECT)psObj, bSuppressCatEvents);

    return bRemoved;
}

/*******************************************************************************
*
*   tGetEventsForSubscriptionAlertCDOChange
*
*****************************************************************************/
static CHANNEL_EVENT_MASK tGetEventsForCDOChange (
    CHANNEL_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT_STRUCT *psDummyObj
        )
{
    CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;
    N16 n16Compare;
    STRING_OBJECT hChanString, hReplacementString;

    // Extract special unsubscribed channel from the channel cache
    // and de-reference the pointer

    if(psDummyObj != NULL)
    {
        // Compare Artist
        hChanString = CDO.hArtist(psObj->hCDO);
        hReplacementString = CDO.hArtist(psDummyObj->hCDO);
        n16Compare = STRING.n16Compare(hChanString, hReplacementString, TRUE);

        if (n16Compare != 0)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_ARTIST;
        }

        // Compare Title
        hChanString = CDO.hTitle(psObj->hCDO);
        hReplacementString = CDO.hTitle(psDummyObj->hCDO);
        n16Compare = STRING.n16Compare(hChanString, hReplacementString, TRUE);

        if (n16Compare != 0)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_TITLE;
        }

        // Compare Composer
        hChanString = CDO.hComposer(psObj->hCDO);
        hReplacementString = CDO.hComposer(psDummyObj->hCDO);
        n16Compare = STRING.n16Compare(hChanString, hReplacementString, TRUE);

        if (n16Compare != 0)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_COMPOSER;
        }

        // Compare ContentInfo
        hChanString = CDO.hContentInfo(psObj->hCDO);
        hReplacementString = CDO.hContentInfo(psDummyObj->hCDO);
        n16Compare = STRING.n16Compare(hChanString, hReplacementString, TRUE);

        if (n16Compare != 0)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_CONTENTINFO;
        }

        // Compare CDO
        n16Compare = CDO.n16Compare(psObj->hCDO, psDummyObj->hCDO);
        if (n16Compare != 0)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_TYPE_INFO;
        }
    }

    return tEventMask;
}

/*****************************************************************************
*
*   n16CompareCategoryIds
*
*   This function is used to compare 2 categories.
*   This is used when searching the LL for a specific category.
*
*****************************************************************************/
static N16 n16CompareCategoryIds(
    CATEGORY_OBJECT hCategory1,
    CATEGORY_OBJECT hCategory2
        )
{
    CATEGORY_ID tCategoryId1, tCategoryId2;

    if (hCategory1 == CATEGORY_INVALID_OBJECT)
    {
        return -1;
    }

    if (hCategory2 == CATEGORY_INVALID_OBJECT)
    {
        return 1;
    }

    tCategoryId1 = CATEGORY.tGetCategoryId(hCategory1);
    tCategoryId2 = CATEGORY.tGetCategoryId(hCategory2);

    return tCategoryId1 - tCategoryId2;
}

/*****************************************************************************
 *
 *   bCompareSimilarChannelsOuter
 *
 *****************************************************************************/
static BOOLEAN bCompareSimilarChannelsOuter(
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
        )
{
    CHANNEL_SIMILAR_COMPARE_STRUCT *psCompare, sInner;
    psCompare = (CHANNEL_SIMILAR_COMPARE_STRUCT *)pvEventCallbackArg;

    sInner.hChannel = hChannel;
    sInner.bDifferent = TRUE;

    CHANNEL.eIterateSimilarChannels(
        psCompare->hChannel,
        bCompareSimilarChannelsNewerInner,
        &sInner);

    if (sInner.bDifferent == FALSE)
    {
        // found a match
        // keep on iteratin'
        return TRUE;
    }

    // we found a difference,
    psCompare->bDifferent = TRUE;

    // so we can stop iterating
    return FALSE;
}

/*****************************************************************************
 *
 *   bCompareSimilarChannelsNewerInner
 *
 *****************************************************************************/
static BOOLEAN bCompareSimilarChannelsNewerInner(
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
        )
{
    CHANNEL_SIMILAR_COMPARE_STRUCT *psCompare;
    psCompare = (CHANNEL_SIMILAR_COMPARE_STRUCT *)pvEventCallbackArg;

    if (hChannel == psCompare->hChannel)
    {
        // found a match. the channels are NOT different
        psCompare->bDifferent = FALSE;

        // we found a difference, so we can stop iterating
        return FALSE;
    }

    // keep on iteratin'
    return TRUE;
}
