/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CHANNEL_ART implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sms_update.h"
#include "string_obj.h"
#include "channel_art_db_constants.h"
#include "channel_art_mgr_obj.h"
#include "channel_art_obj.h"
#include "_channel_art_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CHANNEL_ART_hCreate
*
*   Create a channel art object using the given owner object and the file
*   path.
*
*****************************************************************************/
CHANNEL_ART_OBJECT CHANNEL_ART_hCreate (
    SMS_OBJECT hOwner,
    const char *pacFilePath
        )
{
    // Just create the channel art object
    // and return it to the caller
    return hCreateChannelArtObject( hOwner, pacFilePath );
}

/*****************************************************************************
*
*   CHANNEL_ART_hCopy
*
*   This function creates an exact (deep) copy CHANNEL_ART_OBJECT based
*   upon the provided object.
*
*****************************************************************************/
CHANNEL_ART_OBJECT CHANNEL_ART_hCopy (
    SMS_OBJECT hOwner,
    CHANNEL_ART_OBJECT hChannelArt
        )
{
    CHANNEL_ART_OBJECT hCopy =
        CHANNEL_ART_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner(hOwner);
    bOwner &= SMSO_bOwner((SMS_OBJECT)hChannelArt);

    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Create the new channel art object
        hCopy = hCreateChannelArtObject(
            hOwner, psObj->pacFilePath );

        // If we were able to create a new channel art object
        // attempt to copy the contents
        if (hCopy != CHANNEL_ART_INVALID_OBJECT)
        {
            BOOLEAN bOk;

            // Copy all images from the provided object into the copy
            bOk = CHANNEL_ART_bCopyAllImages(hCopy, hChannelArt);

            // Did we encounter any issues?
            if (bOk == FALSE)
            {
                // Yeah, we don't know what is good
                // and what didn't work out --
                // destroy the new copy
                CHANNEL_ART_vDestroy(hCopy);
                hCopy = CHANNEL_ART_INVALID_OBJECT;

                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_OBJECT_NAME
                    ": CHANNEL_ART_bCopyAllImages failed");
            }
        }
    }

    return hCopy;
}

/*****************************************************************************
*
*   CHANNEL_ART_hImage
*
*   This function is used solely by .eUseArt -- so always attempt to
*   provide the content-based art first.
*
*****************************************************************************/
IMAGE_OBJECT CHANNEL_ART_hImage (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        )
{
    BOOLEAN bOwner;
    IMAGE_OBJECT hImage = IMAGE_INVALID_OBJECT;

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry;
        BOOLEAN bSecondaryRequested = FALSE;

        psEntry = psFindEntry(psObj, eImageType, &bSecondaryRequested);

        if (psEntry != NULL)
        {
            CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc = NULL;

            // Always attempt the content-based
            // art first
            if (psEntry->psContent != NULL)
            {
                psDesc = psEntry->psContent;
            }

            // Then the static if necessary
            if (psDesc == NULL)
            {
                psDesc = psEntry->psStatic;
            }

            // Now, grab the requested image
            if (psDesc != NULL)
            {
                if (bSecondaryRequested == FALSE)
                {
                    hImage = psDesc->hImage;
                }
                else
                {
                    hImage = psDesc->hSecondaryImage;
                }
            }
        }
    }

    return hImage;
}

/*****************************************************************************
*
*   CHANNEL_ART_vDestroy
*
*****************************************************************************/
void CHANNEL_ART_vDestroy (
    CHANNEL_ART_OBJECT hChannelArt
        )
{
    BOOLEAN bOwner;

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_IMAGETYPE_ENUM eCurrentImage =
            (CHANNEL_ART_IMAGETYPE_ENUM)0;

        // Iterate through all entries
        do
        {
            // Clear this entry
            bClearArtEntry(psObj, eCurrentImage, FALSE);

        } while ((++eCurrentImage) < CHANNEL_ART_IMAGETYPE_MAX);

        // Clear the file path if it exists
        if (psObj->pacFilePath != NULL)
        {
            psObj->pacFilePath = NULL;
        }

        // Clear the image mask
        psObj->tAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_NONE;

        // Destroy this object
        SMSO_vDestroy((SMS_OBJECT)hChannelArt);
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_ART_bSetChannelImage
*
*   This API adds a new channel art image file to a channel art object.
*   The arguments passed in are the raw values pulled from the channel art
*   database.
*
*   This API also is the point at which the secondary logo starts to
*   come into use, and it is where we break it out into a separate
*   association.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bSetChannelImage (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageUpdated
        )
{
    BOOLEAN bOwner;
    BOOLEAN bResult = FALSE;

    // Validate inputs
    if ((psImageAttribs == NULL) ||
        (pbImageUpdated == NULL) ||
        (psAssocRow == NULL))
    {
        return FALSE;
    }

    // Nothing has been updated yet
    *pbImageUpdated = FALSE;

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Validate inputs
    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Set this image, use the type & options from the
        // provided attribute row
        bResult = bSetChannelImage(
            psObj, psAssocRow, psImageAttribs, pbImageUpdated);
    }

    return bResult;
}


/*****************************************************************************
*
*   CHANNEL_ART_bSetAlbumImage
*
*   This API adds a new album art image file to a channel art object.
*   The argument passed in is an album art association row detailing
*   the image specifics (SID/PID). This association can be generated
*   from the album art DB or by OTA broadcast.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bSetAlbumImage (
    CHANNEL_ART_OBJECT hChannelArt,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    STRING_OBJECT *phCaption,
    BOOLEAN *pbImageUpdated
        )
{
    BOOLEAN bOwner;
    BOOLEAN bResult = FALSE;

    // Validate inputs; note that the caption
    // can be NULL,
    if ( (pbImageUpdated == NULL) ||
         (psAssocRow == NULL) )
    {
        return FALSE;
    }

    // Nothing has been updated yet
    *pbImageUpdated = FALSE;

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Validate inputs
    if ( bOwner == TRUE )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Set this image, use the type & options from the
        // provided attribute row
        bResult = bSetAlbumImage(
            psObj, psAssocRow, phCaption, pbImageUpdated);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": SMSO_bOwner(hChannelArt) failed!");
    }

    return bResult;
}

/*****************************************************************************
*
*   CHANNEL_ART_bUpdateImageAttributes
*
*   This API is used to update the attributes of an image which is already
*   a part of this object.  Search the matching image and update it
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bUpdateImageAttributes (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs
        )
{
    BOOLEAN bOwner;
    CHANNEL_ART_UPDATE_IMAGE_ITERATOR_STRUCT sUpdate;

    // Initialize structure
    sUpdate.bUpdated = FALSE;
    sUpdate.psUpdatedAttribs = psImageAttribs;

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Validate inputs
    if (( bOwner == FALSE ) || ( psImageAttribs == NULL ))
    {
        return FALSE;
    }
    else
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Iterate all matching image descriptors
        vIterateMatchingDescriptors(
            psObj,
            psImageAttribs->un16ImageId,
            psImageAttribs->un8ImageType,
            bUpdateImageIterator, &sUpdate);
    }

    return sUpdate.bUpdated;
}

/*****************************************************************************
*
*   CHANNEL_ART_bRemoveImage
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bRemoveImage (
    CHANNEL_ART_OBJECT hChannelArt,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        )
{
    BOOLEAN bOwner, bRemoved = FALSE;

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Validate input
    if ( bOwner == TRUE )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Clear the indicated art entry
        bRemoved = bClearArtEntry(
            psObj, eImageType, bContentAssociation);
        if (bRemoved == TRUE)
        {
            // If we remove a content-based association we shouldn't
            // alter the available image mask since we still
            // have the static association
            BOOLEAN bUpdateAvailable = !bContentAssociation;

            // Update our event control structure
            // based on the image we just updated
            vUpdateEventForImageType(
                psObj, eImageType, FALSE, bUpdateAvailable );
        }
    }

    return bRemoved;
}

/*****************************************************************************
*
*   CHANNEL_ART_bCopyAllImages
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bCopyAllImages (
    CHANNEL_ART_OBJECT hChannelArtDst,
    CHANNEL_ART_OBJECT hChannelArtSrc
        )
{
    BOOLEAN bOwner, bImagesCopied = FALSE;

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArtDst);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArtSrc);
    }

    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psDstObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArtDst;
        CHANNEL_ART_OBJECT_STRUCT *psSrcObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArtSrc;
        CHANNEL_ART_IMAGETYPE_ENUM eCurrentImage =
            (CHANNEL_ART_IMAGETYPE_ENUM)0;
        BOOLEAN bOk, bCopied, bAlreadyDone;

        // Copy all the entries from the provided
        // channel art object
        do
        {
            // Nothing has been copied yet
            bCopied = FALSE;

            // First, check the image source.  Is the destination
            // object already using the image from the source object?
            bAlreadyDone = CHANNEL_ART_bImageSource(
                hChannelArtDst, hChannelArtSrc, FALSE,
                eCurrentImage, NULL);

            if (bAlreadyDone == FALSE)
            {
                // Copy the currently indexed entry
                bOk = bCopyArtEntry(
                    psSrcObj, psDstObj, FALSE, eCurrentImage, &bCopied);
                if (bOk == FALSE)
                {
                    // Creation failed!  Stop and indicate error
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_OBJECT_NAME
                        ": Unable to copy art");
                    break;
                }
            }

            // Did we actually copy an image with this entry?
            if (bCopied == TRUE)
            {
                // Yes -- Indicate this image was updated
                vUpdateEventForImageType(
                    psDstObj, eCurrentImage, TRUE, TRUE );

                if (bCopied == TRUE)
                {
                    bImagesCopied = TRUE;
                }
            }
        } while ((++eCurrentImage) < CHANNEL_ART_IMAGETYPE_MAX);
    }

    return bImagesCopied;
}

/*****************************************************************************
*
*   CHANNEL_ART_bCopyImage
*
*     This function copies the image (of the given type) from the source
*     channel art object into the destination object, even if the source
*     image is invalid.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bCopyImage (
    CHANNEL_ART_OBJECT hChannelArtDst,
    CHANNEL_ART_OBJECT hChannelArtSrc,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify we own the channel art objects
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArtDst);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArtSrc);
    }

    if ((bOwner == TRUE) && ( eImageType < CHANNEL_ART_IMAGETYPE_MAX ))
    {
        CHANNEL_ART_OBJECT_STRUCT *psDstObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArtDst;
        CHANNEL_ART_OBJECT_STRUCT *psSrcObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArtSrc;
        BOOLEAN bImageCopied = FALSE;

        // Copy the entry for this type
        bSuccess = bCopyArtEntry(
            psSrcObj, psDstObj, bContentAssociation, eImageType, &bImageCopied);
        if (bSuccess == FALSE)
        {
            // Creation failed!  Stop and indicate error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": Unable to copy image");
        }

        // Did we actually copy an image with this entry?
        if ((bSuccess == TRUE) && (bImageCopied == TRUE))
        {
            // Yes -- Indicate this image was updated
            vUpdateEventForImageType(
                psDstObj, eImageType, TRUE, TRUE );
        }

        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CHANNEL_ART_bReset
*
*   "Resets" the object such that all images (either content-related only
*   or all) are removed from the object, and updates the event mask to
*   indicate any changes.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bReset (
    CHANNEL_ART_OBJECT hChannelArt,
    BOOLEAN bContentArtOnly
        )
{
    BOOLEAN bOwner, bObjectUpdated = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_IMAGETYPE_ENUM eCurrentImage =
            (CHANNEL_ART_IMAGETYPE_ENUM)0;

        do
        {
            // Remove all images matching the reset criteria --
            // Keep track of any updates
            bObjectUpdated |= CHANNEL_ART_bRemoveImage(
                hChannelArt, bContentArtOnly, eCurrentImage);

        } while ((++eCurrentImage) < CHANNEL_ART_IMAGETYPE_MAX);
    }

    return bObjectUpdated;
}

/*****************************************************************************
*
*   CHANNEL_ART_bImageSource
*
*   This function is used in order to determine if the image with the given
*   image type was created by the channel art object given via
*   hChannelArtQuery
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bImageSource (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_OBJECT hChannelArtQuery,
    BOOLEAN bContentArt,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN *pbImageTypePresent
        )
{
    BOOLEAN bOwner, bMatch = FALSE,
            bImageTypePresent = FALSE;

    // Did the caller care to populate pbImageTypePresent?
    if (NULL == pbImageTypePresent)
    {
        // No big deal -- use our local variable
        pbImageTypePresent = &bImageTypePresent;
    }

    // Image type is not present as far as we know right now
    *pbImageTypePresent = FALSE;

    // Verify we own the channel art objects
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArtQuery);
    }

    if (bOwner == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc;

        // Find the appropriate image descriptor, but don't create
        // it if it doesn't exist
        psDesc = psFindImageDesc(psObj, bContentArt, eImageType, FALSE);
        if (psDesc != NULL)
        {
            // We know this image type is now present
            *pbImageTypePresent = TRUE;

            // Was this image provided by the channel art
            // object hChannelArtQuery?
            if (psDesc->hSourceObject == hChannelArtQuery)
            {
                // Yes
                bMatch = TRUE;
            }
        }
    }

    return bMatch;
}

/*****************************************************************************
*
*   CHANNEL_ART_bHasImage
*
*     This API is used to query the channel art object for a specific
*     image type & Id
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bHasImage (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttrib
        )
{
    BOOLEAN bOwner;
    BOOLEAN bFound = FALSE;

    // Verify input
    if (psAttrib == NULL)
    {
        return FALSE;
    }

    // Verify we own the SMS Object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Validate inputs
    if (bOwner == TRUE )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc;

        // Find this descriptor using both the type & Id since the caller
        // doesn't know if they're looking for a static or content-based image
        psDesc = psFindImageDescByTypeAndId(
            psObj, psAttrib->un16ImageId, psAttrib->un8ImageType);

        if (psDesc != NULL)
        {
            bFound = TRUE;
        }
    }

    return bFound;
}

/*****************************************************************************
*
*   CHANNEL_ART_bAssociationTypesForImage
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bAssociationTypesForImage(
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    CHANNEL_ART_ASSOC_TYPE_ITERATOR bIterator,
    void *pvArg
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if ((psImageAttribs == NULL) ||
        (bIterator == NULL))
    {
        return FALSE;
    }

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);

    if (bOwner == TRUE )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_ASSOC_TYPE_ITERATOR_STRUCT sIterator;
        BOOLEAN bContinue = TRUE;
        UN8 un8Index;

        // Initialize iterator
        sIterator.un8NumTypesToReport = 0;

        // Iterate all descriptors which match this id/type and
        // report their type to the caller
        vIterateMatchingDescriptors(
            psObj,
            psImageAttribs->un16ImageId, psImageAttribs->un8ImageType,
            bReportAssocTypeIterator, &sIterator);

        // Now that we know which types we need to report, it is
        // time to report them
        for (un8Index = 0;
             (un8Index < sIterator.un8NumTypesToReport) && (bContinue == TRUE);
             un8Index++)
        {
            // Report this type to the callback
            bContinue = bIterator(sIterator.aeTypesToReport[un8Index], pvArg);
        }
    }

    return bOwner;
}

/*****************************************************************************
*
*   CHANNEL_ART_bGetImageAttributes
*
*   Basically just a wrapper around the private bGetImageAttributes.  This
*   provides other objects within SMS the capability to query an IMAGE
*   object via its parent CHANNEL_ART_OBJECT.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bGetImageAttributes (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageCopied
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Verify inputs (pun8AssociationVersion may be null)
    if (   (bOwner == TRUE )
        && (psImageAttribs != NULL)
       )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Retrieve the requested info
        bSuccess = bGetImageAttributes(
            psObj, eImageAssocType,
            psImageAttribs, pbImageCopied );
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CHANNEL_ART_bGetAssocAttributes
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_bGetAssocAttributes (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType,
    UN8 un8ImageType,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // You have to know what you're looking for!
    if ((eImageAssocType != CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC) &&
        (eImageAssocType != CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT))
    {
        return FALSE;
    }

    if (psAssocRow == NULL)
    {
        return FALSE;
    }

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Verify inputs (pun8AssociationVersion may be null)
    if (bOwner == TRUE )
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc = NULL;
        BOOLEAN bContentAssociation = FALSE;

        if (eImageAssocType == CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT)
        {
            bContentAssociation = TRUE;
        }

        // Get the appropriate descriptor in the array
        psDesc = psFindImageDesc(
            psObj, bContentAssociation,
            (CHANNEL_ART_IMAGETYPE_ENUM)un8ImageType, FALSE);
        if (psDesc != NULL)
        {
            // The association is only valid if we didn't
            // copy this info from another object
            if (psDesc->hSourceObject == (CHANNEL_ART_OBJECT)psObj)
            {
                psAssocRow->un8AssocVer = psDesc->un8AssociationVersion;
                psAssocRow->bAssocVerConfirmed  = psDesc->bAssocVerConfirmed;
                psAssocRow->bHighPriority = psDesc->bHighPriority;

                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CHANNEL_ART_vConfirmAssociation
*
*****************************************************************************/
void CHANNEL_ART_vConfirmAssociation (
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType,
    UN8 un8ImageType
        )
{
    BOOLEAN bOwner;

    // You have to know what you're confirming!
    if ((eImageAssocType != CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC) &&
        (eImageAssocType != CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT))
    {
        return;
    }

    // Verify we own the SMS Object
    bOwner = SMSO_bOwner((SMS_OBJECT)hChannelArt);

    // Verify inputs (pun8AssociationVersion may be null)
    if (TRUE == bOwner)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc = NULL;
        BOOLEAN bContentAssociation = FALSE;

        if (CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT == eImageAssocType)
        {
            bContentAssociation = TRUE;
        }

        // Get the appropriate descriptor in the array
        psDesc = psFindImageDesc(
            psObj, bContentAssociation,
            (CHANNEL_ART_IMAGETYPE_ENUM)un8ImageType, FALSE);
        if (psDesc != NULL)
        {
            // The association is only valid if we didn't
            // copy this info from another object
            if (psDesc->hSourceObject == (CHANNEL_ART_OBJECT)psObj)
            {
                psDesc->bAssocVerConfirmed = TRUE;
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_ART_tImagesAvailable
*
*   This function provides a mask indicating which images are available
*   via this object to the caller.
*
*   This call locks the channel art service.
*
*****************************************************************************/
CHANNEL_ART_AVAILABLE_IMAGE_MASK CHANNEL_ART_tImagesAvailable (
    CHANNEL_ART_OBJECT hChannelArt
        )
{
    BOOLEAN bLocked;
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableMask =
        CHANNEL_ART_AVAILABLE_IMAGE_NONE;

    // Lock the SMS object
    bLocked = SMSO_bLock((SMS_OBJECT)hChannelArt, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Copy the available image mask
        tAvailableMask = psObj->tAvailableMask;

        SMSO_vUnlock((SMS_OBJECT)hChannelArt);
    }

    return tAvailableMask;
}

/*****************************************************************************
*
*   CHANNEL_ART_tVersion
*
*   This function provides a version of given CHANNEL_ART_OBJECT
*
*****************************************************************************/
CHANNEL_ART_VERSION CHANNEL_ART_tVersion (
    CHANNEL_ART_OBJECT hChannelArt
        )
{
    BOOLEAN bValid;
    CHANNEL_ART_VERSION tVersion = 0;

    // Check object validity
    bValid = SMSO_bIsValid((SMS_OBJECT)hChannelArt);
    if(bValid == TRUE)
    {
        CHANNEL_ART_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_OBJECT_STRUCT *)hChannelArt;

        // Copy the available image mask
        tVersion = psObj->tVersion;
    }

    return tVersion;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateChannelArtObject
*
*   Performs the common tasks required for CHANNEL_ART_hCreate &
*   CHANNEL_ART_hCreateCopy.
*
*****************************************************************************/
static CHANNEL_ART_OBJECT hCreateChannelArtObject(
    SMS_OBJECT hOwner,
    const char *pacFilePath
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static UN32 un32InitInstance = 0;
    CHANNEL_ART_OBJECT_STRUCT *psObj;
    BOOLEAN bOk;

    snprintf( &acName[0], sizeof(acName),
              CHANNEL_ART_OBJECT_NAME":%u",
              un32InitInstance );

    // Create an instance of this object
    psObj = (CHANNEL_ART_OBJECT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(CHANNEL_ART_OBJECT_STRUCT),
            hOwner,
            FALSE );
    if(psObj == NULL)
    {
        // Error!
        return CHANNEL_ART_INVALID_OBJECT;
    }

    // Clear out the image structures
    bOk = OSAL.bMemSet(
        &psObj->sLogoEntry, 0, sizeof(CHANNEL_ART_TABLE_ENTRY_STRUCT));
    bOk &= OSAL.bMemSet(
        &psObj->sBackgroundEntry, 0, sizeof(CHANNEL_ART_TABLE_ENTRY_STRUCT));
    bOk &= OSAL.bMemSet(
        &psObj->sAlbumEntry, 0, sizeof(CHANNEL_ART_TABLE_ENTRY_STRUCT));

    if (bOk == FALSE)
    {
        // Error!  Couldn't access memory
        SMSO_vDestroy((SMS_OBJECT)psObj);
        return CHANNEL_ART_INVALID_OBJECT;
    }

    // Initialize the channel art image mask
    psObj->tAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_NONE;

    // Store the path pointer
    psObj->pacFilePath = pacFilePath;

    // Increment the init instance
    un32InitInstance++;

    puts("Channel Art object created\n");
    return (CHANNEL_ART_OBJECT)psObj;

}

/*****************************************************************************
*
*   bGetImageAttributes
*
*   Provides the caller with all the information related to a particular
*   image in the form of a CHANNEL_ART_ATTRIB_ROW_STRUCT *
*
*****************************************************************************/
static BOOLEAN bGetImageAttributes (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageCopied
        )
{
    BOOLEAN bResult = FALSE;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc = NULL;

    if ((eImageAssocType == CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC) ||
        (eImageAssocType == CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT))
    {
        BOOLEAN bContentAssociation = FALSE;

        if (eImageAssocType == CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT)
        {
            bContentAssociation = TRUE;
        }

        // Get the appropriate descriptor in the array
        psDesc = psFindImageDesc(
            psObj, bContentAssociation,
            (CHANNEL_ART_IMAGETYPE_ENUM)psImageAttribs->un8ImageType, FALSE);
    }
    else
    {
        // Find this descriptor using both the type & Id since the caller
        // doesn't know if they're looking for a static or content-based image
        psDesc = psFindImageDescByTypeAndId(
            psObj, psImageAttribs->un16ImageId, psImageAttribs->un8ImageType);
    }

    // Ensure we found something
    if ( psDesc != NULL )
    {
        // Extract the data if the IMAGE handle is not invalid
        if (psDesc->hImage != IMAGE_INVALID_OBJECT)
        {
            BOOLEAN bUseBackgroundColor,
                    bUseLineBitmap;
            size_t tLineBitmapEntrySize;
            RGB_STRUCT sBackgroundColor;
            SMSAPI_RETURN_CODE_ENUM eSmsApiReturnCode;
            IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sPropertyValue;

            // We have found the requested image
            bResult = TRUE;

            // Get the main attributes (id & ver)
            eSmsApiReturnCode = IMAGE_eProperty(
                psDesc->hImage,
                IMAGE_PROPERTY_INTERNAL_ID,
                &sPropertyValue);
            if (eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                psImageAttribs->un16ImageId =
                    (UN16)sPropertyValue.uData.un32Value;
            }
            else
            {
                bResult = FALSE;    
            }

            eSmsApiReturnCode = IMAGE_eProperty(
                psDesc->hImage,
                IMAGE_PROPERTY_INTERNAL_VERSION,
                &sPropertyValue);
            if (eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                psImageAttribs->un16ImageVer =
                    (UN16)sPropertyValue.uData.un32Value;
            }
            else
            {
                bResult = FALSE;
            }

            psImageAttribs->bSecondaryAvailable = psDesc->bSecondaryAvailable;

            // Get the background info as well
            // and generate useful data
            // for the caller
            psImageAttribs->sBackgroundGfx.tBackgroundOptions =
                CHANNEL_ART_BACKGROUND_OPTION_NONE;
            psImageAttribs->sBackgroundGfx.sLineBitmap.tLineBitmapIndex = 0;

            // Get the bulk of the background info
            IMAGE.eImageBackgroundDisplayRules(
                psDesc->hImage,
                &bUseBackgroundColor,
                &bUseLineBitmap,
                &tLineBitmapEntrySize,
                &sBackgroundColor );

            // If the image uses the background
            // color then set the option flag and
            // get the color data to the caller
            if (bUseBackgroundColor == TRUE)
            {
                // Set the option
                psImageAttribs->sBackgroundGfx.tBackgroundOptions |=
                    CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR;

                // Generate the color data
                psImageAttribs->sBackgroundGfx.un32BackgroundColor =
                    CHANNEL_ART_GET_BACKGROUND_COLOR(
                        sBackgroundColor.un8Red,
                        sBackgroundColor.un8Green,
                        sBackgroundColor.un8Blue );
            }

            // If the image uses a line bitmap
            // then set the option flag and
            // get the line bitmap data to the caller
            if (bUseLineBitmap == TRUE)
            {
                // Set the option
                psImageAttribs->sBackgroundGfx.tBackgroundOptions |=
                    CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP;

                // Get the line bitmap
                psImageAttribs->sBackgroundGfx.sLineBitmap.pun8LineBitmap =
                    (const UN8 *)IMAGE_psGetLineBitmap(
                        psDesc->hImage );
            }

            // Pull out the attributes to populate the
            // provided structure pointer
            psImageAttribs->un8CodeType = (UN8)IMAGE.eFormat( psDesc->hImage );

            // Retrieve the "copied" flag
            if (pbImageCopied != NULL)
            {
                // An image has been copied if this
                // image's source object is not
                // equal to the provided object handle
                *pbImageCopied =
                    (psDesc->hSourceObject != (CHANNEL_ART_OBJECT)psObj);
            }
        }
    }

    return bResult;
}

/*****************************************************************************
*
*   vUpdateEventForImageType
*
*   This function updates the objects event mask to reflect the change
*   to a given image type
*
*****************************************************************************/
static void vUpdateEventForImageType (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bAdd,
    BOOLEAN bUpdateAvailable
        )
{
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tNewAvailableMask =
        CHANNEL_ART_AVAILABLE_IMAGE_NONE;

    switch (eImageType)
    {
        case CHANNEL_ART_IMAGETYPE_LOGO:
        {
            tNewAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_LOGO;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO:
        {
            tNewAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_BKGRND:
        {
            tNewAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_ALBUM:
        {
            tNewAvailableMask = CHANNEL_ART_AVAILABLE_IMAGE_ALBUM;
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": Unknown event mask provided to vUpdateEventForImageType (%u)",
                eImageType);
            return;
        }
    }

    // Only update the available image mask if we
    // were told to do so
    if (bUpdateAvailable == TRUE)
    {
        // Update the reset mask
        if (bAdd == TRUE)
        {
            // We now have this image type in
            // our list, so remember that
            psObj->tAvailableMask |= tNewAvailableMask;
        }
        else
        {
            // We no longer have this image
            psObj->tAvailableMask &= ~tNewAvailableMask;
        }

        // Increase the version
        psObj->tVersion++;

        printf("Available is now %u:\n", psObj->tAvailableMask);
    }
    else
    {
        puts("Available not updated");
    }

    return;
}

/*****************************************************************************
*
*   bCopyArtEntry
*
*****************************************************************************/
static BOOLEAN bCopyArtEntry (
    CHANNEL_ART_OBJECT_STRUCT *psSrcObj,
    CHANNEL_ART_OBJECT_STRUCT *psDstObj,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN *pbImageCopied
        )
{
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psSrcEntry;
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psDstEntry;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT **ppsSrcDesc;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT **ppsDstDesc;
    BOOLEAN bSecondaryImage = FALSE;

    // Initialize this always
    *pbImageCopied = FALSE;

    // Bounds-check
    if (eImageType >= CHANNEL_ART_IMAGETYPE_MAX)
    {
        return FALSE;
    }

    // Get a hold of the correct entries
    psSrcEntry = psFindEntry(psSrcObj, eImageType, &bSecondaryImage);
    psDstEntry = psFindEntry(psDstObj, eImageType, NULL);

    // The secondary image is just an additional
    // image handle which is a part of a
    // primary image type.  So, that is taken care
    // of when the primary image is copied, and we
    // don't need to worry about it here
    if (bSecondaryImage == TRUE)
    {
        return TRUE;
    }

    if (bContentAssociation == FALSE)
    {
        ppsSrcDesc = &psSrcEntry->psStatic;
        ppsDstDesc = &psDstEntry->psStatic;
    }
    else
    {
        ppsSrcDesc = &psSrcEntry->psContent;
        ppsDstDesc = &psDstEntry->psContent;
    }

    /***************************************************
    * First, ensure the state of the image descriptors *
    ****************************************************/

    // Is this descriptor present in the source?
    if (*ppsSrcDesc != NULL)
    {
        // Yes!
        if (*ppsDstDesc == NULL)
        {
            // Allocate a new descriptor
            *ppsDstDesc = (CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *)
                SMSO_hCreate("ImgDesc",
                    sizeof(CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT),
                    SMS_INVALID_OBJECT, FALSE);

            if (*ppsDstDesc == NULL)
            {
                return FALSE;
            }
        }
    }
    else // Descriptor not present
    {
        BOOLEAN bCleared;

        // Ensure this entry is cleared
        bCleared = bClearArtEntry(psDstObj, eImageType, bContentAssociation);
        if (bCleared == TRUE)
        {
            // We "copied" the NULL image to this entry
            *pbImageCopied = TRUE;
        }
    }

    /*************************
    * Then, perform the copy *
    **************************/

    // Do we have something to work with?
    if (*ppsSrcDesc != NULL)
    {
        // Clear any pre-existing image at the destination
        if ((*ppsDstDesc)->hImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy((*ppsDstDesc)->hImage);
            (*ppsDstDesc)->hImage = IMAGE_INVALID_OBJECT;
        }

        // Copy the image's descriptor info
        (*ppsDstDesc)->un8AssociationVersion =
            (*ppsSrcDesc)->un8AssociationVersion;
        (*ppsDstDesc)->hSourceObject = (*ppsSrcDesc)->hSourceObject;

        // This association has not been confirmed
        (*ppsDstDesc)->bAssocVerConfirmed = FALSE;

        // Create a copy of that image
        (*ppsDstDesc)->hImage =
            IMAGE_hCreateCopy( (*ppsSrcDesc)->hImage );

        if ((*ppsDstDesc)->hImage == IMAGE_INVALID_OBJECT)
        {
            // Creation failed!  Stop and indicate error
            return FALSE;
        }

        // We successfully copied an image
        *pbImageCopied = TRUE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bClearArtEntry
*
*****************************************************************************/
static BOOLEAN bClearArtEntry (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bContentArtOnly
        )
{
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry = NULL;
    BOOLEAN bEntryRemoved = FALSE, bSecondaryImage = FALSE;

    // Grab the relevant entry
    psEntry = psFindEntry(psObj, eImageType, &bSecondaryImage);
    if (psEntry == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": bClearArtEntry given bad image type: %u\n",
            eImageType);
        return FALSE;
    }

    if ((bContentArtOnly == FALSE) && (psEntry->psStatic != NULL))
    {
        // If the image type being cleared is a secondary image,
        // then just clear that image handle.  Otherwise, clear it all

        // Always clear the secondary image, since both cases
        // include clearing the secondary image
        if (psEntry->psStatic->hSecondaryImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psEntry->psStatic->hSecondaryImage);
            psEntry->psStatic->hSecondaryImage = IMAGE_INVALID_OBJECT;
        }

        if (bSecondaryImage == FALSE)
        {
            // Free the IMAGE object
            if (psEntry->psStatic->hImage != IMAGE_INVALID_OBJECT)
            {
                IMAGE_vDestroy(psEntry->psStatic->hImage);
                psEntry->psStatic->hImage = IMAGE_INVALID_OBJECT;
            }

            // Clear the entry
            OSAL.bMemSet(psEntry->psStatic, 0, sizeof(CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT));

            // Destroy the entry
            SMSO_vDestroy((SMS_OBJECT)psEntry->psStatic);

            // Clear the pointer
            psEntry->psStatic = NULL;
        }

        bEntryRemoved = TRUE;
    }

    if (psEntry->psContent != NULL)
    {
        // Always clear the secondary image, since both cases
        // include clearing the secondary image
        if (psEntry->psContent->hSecondaryImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psEntry->psContent->hSecondaryImage);
            psEntry->psContent->hSecondaryImage = IMAGE_INVALID_OBJECT;
        }

        if (bSecondaryImage == FALSE)
        {
            if (psEntry->psContent->hImage != IMAGE_INVALID_OBJECT)
            {
                // Free the IMAGE object
                IMAGE_vDestroy(psEntry->psContent->hImage);
                psEntry->psContent->hImage = IMAGE_INVALID_OBJECT;
            }

            // Clear the entry
            OSAL.bMemSet(psEntry->psContent, 0, sizeof(CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT));

            // Destroy the entry
            SMSO_vDestroy((SMS_OBJECT)psEntry->psContent);

            // Clear the pointer
            psEntry->psContent = NULL;
        }
        bEntryRemoved = TRUE;
    }

    return bEntryRemoved;
}

/*****************************************************************************
*
*   bSetChannelImage
*
*****************************************************************************/
static BOOLEAN bSetChannelImage (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageUpdated
        )
{
    BOOLEAN bResult = FALSE;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc;
    IMAGE_OBJECT hNewImage;
    BOOLEAN bUpdateMasks = TRUE;
    SMSAPI_RETURN_CODE_ENUM eSmsApiReturnCode;

    // Get the appropriate descriptor in the array
    // which we are going to populate now
    psDesc = psFindImageDesc(
        psObj, psAssocRow->bContent,
        (CHANNEL_ART_IMAGETYPE_ENUM)psImageAttribs->un8ImageType, TRUE);
    if (psDesc == NULL)
    {
        // Can't find an entry for this image type
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": CHANNEL_ART_bUpdateImage: Unable to locate image");
        return FALSE;
    }

    // Determine if there is anything new here
    do
    {
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sCurImageId;
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sCurImageVersion;

        OSAL.bMemSet(&sCurImageId, 0, sizeof(sCurImageId));
        OSAL.bMemSet(&sCurImageVersion, 0, sizeof(sCurImageVersion));

        if (psDesc->hImage == IMAGE_INVALID_OBJECT)
        {
            // Yeah, we don't have an image, so this is new
            break;
        }

        eSmsApiReturnCode =
            IMAGE_eProperty(psDesc->hImage,
                    IMAGE_PROPERTY_INTERNAL_ID, &sCurImageId);
        if (eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            eSmsApiReturnCode =
                IMAGE_eProperty(psDesc->hImage,
                    IMAGE_PROPERTY_INTERNAL_VERSION, &sCurImageVersion);
        }

        if (eSmsApiReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": CHANNEL_ART_bUpdateImage: Unable to query image");
            return FALSE;
        }

        // Has any of our image data changed?
        if ((sCurImageId.uData.un32Value != (UN32)psImageAttribs->un16ImageId) ||
            (sCurImageVersion.uData.un32Value != (UN32)psImageAttribs->un16ImageVer))
        {
            // Yep, this is new to us
            break;
        }

        // Nope, this is just an assoc version update or we
        // have just confirmed an existing assoc
        psDesc->un8AssociationVersion = psAssocRow->un8AssocVer;
        psDesc->bAssocVerConfirmed = psAssocRow->bAssocVerConfirmed;

        // No problems, but we didn't update anything
        return TRUE;

    } while (FALSE);

    // We will mark this as updated for now, and tweak in the
    // block below if needed
    *pbImageUpdated = TRUE;

    // Don't mess with the masks or tell the caller this
    // call resulted in an art update if we're updating
    // a static association on an object with an active
    // content association
    if (psAssocRow->bContent == FALSE)
    {
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psContentDesc;

        // Get the content descriptor in the array
        // for this type
        psContentDesc = psFindImageDesc(
            psObj, TRUE,
            (CHANNEL_ART_IMAGETYPE_ENUM)psImageAttribs->un8ImageType, FALSE);

        // Does it exist with a valid image?
        if ((psContentDesc != NULL) &&
            (psContentDesc->hImage != IMAGE_INVALID_OBJECT))
        {
            // Yeah, this art object has content art associated
            // with it for this type -- so don't update the masks
            // and don't tell the caller this call resulted in an update
            *pbImageUpdated = FALSE;
            bUpdateMasks = FALSE;
        }
    }

    // Create & configure a new IMAGE_OBJECT
    hNewImage = hCreateImage(
        psObj, psImageAttribs,
        (CHANNEL_ART_IMAGETYPE_ENUM)psImageAttribs->un8ImageType,
        psImageAttribs->sBackgroundGfx.tBackgroundOptions);

    if (hNewImage != IMAGE_INVALID_OBJECT)
    {
        BOOLEAN bSecondaryAdded = FALSE;
        BOOLEAN bSecondaryUpdated = FALSE;

        // If there is a pre-existing, valid IMAGE handle
        // in the desired location destroy it to make
        // room for the new IMAGE
        if (psDesc->hImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psDesc->hImage);
        }

        // Now we can copy over the new image
        // descriptor information
        psDesc->hImage = hNewImage;
        psDesc->un8AssociationVersion = psAssocRow->un8AssocVer;
        psDesc->bAssocVerConfirmed = psAssocRow->bAssocVerConfirmed;
        psDesc->bHighPriority = psAssocRow->bHighPriority;
        psDesc->bSecondaryAvailable = psImageAttribs->bSecondaryAvailable;
        psDesc->hSourceObject = (CHANNEL_ART_OBJECT)psObj;

        if (bUpdateMasks == TRUE)
        {
            // Update our event control structure
            // based on the image we just updated
            vUpdateEventForImageType(
                psObj,
                (CHANNEL_ART_IMAGETYPE_ENUM)
                    psImageAttribs->un8ImageType, TRUE, TRUE );
        }

        // Now, handle the secondary image, if necessary
        // (if the primary image changed then we can guarantee
        // that the secondary image has changed as well)

        // Always clear the secondary image, since we'll
        // either replace it with the new version
        // or we'll clear it if the new version indicates
        // there is no secondary image now
        if (psDesc->hSecondaryImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psDesc->hSecondaryImage);
            psDesc->hSecondaryImage = IMAGE_INVALID_OBJECT;
            bSecondaryUpdated = TRUE;
        }

        if (psDesc->bSecondaryAvailable == TRUE)
        {
            // Create the secondary image
            psDesc->hSecondaryImage = hCreateImage(
                psObj, psImageAttribs,
                CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO,
                CHANNEL_ART_SECONDARY_LOGO_BK_OPTIONS);
            if (psDesc->hSecondaryImage != IMAGE_INVALID_OBJECT)
            {
                bSecondaryAdded = TRUE;
                bSecondaryUpdated = TRUE;
            }
        }

        if ((bUpdateMasks == TRUE) && (bSecondaryUpdated == TRUE))
        {
            // Update our event control structure
            // based on the image we just updated
            vUpdateEventForImageType(
                psObj, CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO,
                bSecondaryAdded, TRUE );
        }

        // Success!
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   bSetAlbumImage
*
*****************************************************************************/
static BOOLEAN bSetAlbumImage (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    STRING_OBJECT *phCaption,
    BOOLEAN *pbImageUpdated
        )
{
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc;
    CHANNEL_ART_IMAGE_DATA_STRUCT sImageData;
    IMAGE_OBJECT hImage;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // By default, we'll say nothing has been updated.
    *pbImageUpdated = FALSE;

    // Get the appropriate descriptor in the array
    // which we are going to populate now; for album art,
    // we store images in the content
    // description as a convention.
    psDesc = psFindImageDesc(  psObj,
                           TRUE, // Content Association
                           CHANNEL_ART_IMAGETYPE_ALBUM,
                           TRUE // Create if needed
                       );

    if ( psDesc == NULL )
    {
        // Can't find an entry for this image type
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": bSetAlbumImage: Unable to locate album image" );
        return FALSE;
    }

    do
    {
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sVersion, sID;

        OSAL.bMemSet(&sVersion, 0, sizeof(sVersion));
        OSAL.bMemSet(&sID, 0, sizeof(sID));

        if ( psDesc->hImage == IMAGE_INVALID_OBJECT )
        {
            // Yeah, we don't have an image, so this is new
            break;
        }

        eReturn = IMAGE_eProperty(psDesc->hImage, IMAGE_PROPERTY_INTERNAL_ID, &sID);
        if ( SMSAPI_RETURN_CODE_SUCCESS == eReturn )
        {
            eReturn = IMAGE_eProperty(psDesc->hImage, IMAGE_PROPERTY_INTERNAL_VERSION, &sVersion);
        }

        // If we failed to get either the ID or the version, bail out
        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": CHANNEL_ART_bUpdateAlbumImage: Unable to query image properties: %s",
                SMSAPI_DEBUG_pacReturnCodeText(eReturn) );
            return FALSE;
        }

        // Has any of our image data changed?
        if ( ((U16)sID.uData.un32Value != psAssocRow->un16ImageId) ||
             ((U16)sVersion.uData.un32Value != psAssocRow->un16ImageVer) )
        {
            // Yep, this is new to us
            break;
        }

        // No problems, but we didn't update anything
        return TRUE;

    } while ( FALSE );

    // Initialize the new image data
    OSAL.bMemSet(&sImageData, 0, sizeof(sImageData));

    sImageData.eImageType = CHANNEL_ART_IMAGETYPE_ALBUM;
    sImageData.un16ImageId = psAssocRow->un16ImageId;
    sImageData.un16ImageVer = psAssocRow->un16ImageVer;
    sImageData.hCaption =
        (NULL != phCaption) ? (*phCaption) : STRING_INVALID_OBJECT;

    // Create a new IMAGE_OBJECT; Note: Album art is JPEG-only
    hImage = IMAGE_hCreate(
            (SMS_OBJECT)psObj,
            psObj->pacFilePath,
            IMAGE_FORMAT_JPEG,
            &GsChannelArtImageIntf,
            (void*) &sImageData,
            TRUE
         );

    if ( hImage == IMAGE_INVALID_OBJECT )
    {
        // We can't do any more
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": IMAGE_hCreate failed!");
        return FALSE;
    }

    // If we made it here, we'll be updating the image
    *pbImageUpdated = TRUE;

    if ( IMAGE_INVALID_OBJECT != psDesc->hImage )
    {
        // Delete the existing image if it exists
        IMAGE_vDestroy( psDesc->hImage );
    }

    // Now that we're here, set the new image
    psDesc->hImage = hImage;
    psDesc->hSourceObject = (CHANNEL_ART_OBJECT)psObj;

    // Update our event control structure
    // based on the image we just updated
    vUpdateEventForImageType( 
                psObj,
                CHANNEL_ART_IMAGETYPE_ALBUM,
                TRUE, // bAdd
                TRUE  // bUpdateAvailable
             );

    return TRUE;
}

/*****************************************************************************
*
*   psFindEntry
*
*****************************************************************************/
static CHANNEL_ART_TABLE_ENTRY_STRUCT *psFindEntry (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN *pbSecondaryImageRequested
        )
{
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry = NULL;
    BOOLEAN bSecondaryImageRequested;

    if (pbSecondaryImageRequested == NULL)
    {
        // Use local variable if necessary
        pbSecondaryImageRequested = &bSecondaryImageRequested;
    }

    // Initialize
    *pbSecondaryImageRequested = FALSE;

    switch (eImageType)
    {
        case CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO:
        {
            *pbSecondaryImageRequested = TRUE;
        }
        // Don't break -- this comes from the logo entry as well

        case CHANNEL_ART_IMAGETYPE_LOGO:
        {
            // Grab the logo entry
            psEntry = &psObj->sLogoEntry;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_BKGRND:
        {
            // Grab the background entry
            psEntry = &psObj->sBackgroundEntry;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_ALBUM:
        {
            // Grab the album art entry
            psEntry = &psObj->sAlbumEntry;
        }
        break;

        default:
        {
        }
        break;
    }

    return psEntry;
}

/*****************************************************************************
*
*   psFindImageDescByTypeAndId
*
*****************************************************************************/
static CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psFindImageDescByTypeAndId (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN8 un8ImageType
        )
{
    SMSAPI_RETURN_CODE_ENUM eSmsApiReturnCode;
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc = NULL;
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sCurrentImageId;

    // Extract the entry for this type -- we
    // don't care if we're looking for a primary
    // or secondary image here since they both have
    // the same metadata
    psEntry = psFindEntry(psObj, (CHANNEL_ART_IMAGETYPE_ENUM)un8ImageType, NULL);
    if (psEntry == NULL)
    {
        // We got nothin'
        return NULL;
    }

    // Check the static entry first
    if (psEntry->psStatic != NULL)
    {
        if (psEntry->psStatic->hImage != IMAGE_INVALID_OBJECT)
        {
            // Get this image's Id
            eSmsApiReturnCode =
                IMAGE_eProperty(
                    psEntry->psStatic->hImage, IMAGE_PROPERTY_INTERNAL_ID,
                    &sCurrentImageId);

            // If that went well and we have a match
            // then we're done
            if ((eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS ) &&
                (sCurrentImageId.uData.un32Value == (UN32)un16ImageId))
            {
                // We have this image
                psDesc = psEntry->psStatic;
            }
        }
    }

    // Then check the content (if necessary)
    if ((psDesc == NULL) && (psEntry->psContent != NULL))
    {
        if (psEntry->psContent->hImage != IMAGE_INVALID_OBJECT)
        {
            // Get this image's Id
            eSmsApiReturnCode =
                IMAGE_eProperty(
                    psEntry->psContent->hImage, IMAGE_PROPERTY_INTERNAL_ID,
                    &sCurrentImageId);

            // If that went well and we have a match
            // then we're done
            if ((eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS ) &&
                (sCurrentImageId.uData.un32Value == (UN32)un16ImageId))
            {
                // We have this image
                psDesc = psEntry->psContent;
            }
        }
    }

    return psDesc;
}

/*****************************************************************************
*
*   psFindImageDesc
*
*****************************************************************************/
static CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psFindImageDesc (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bCreate
        )
{
    CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc =
        (CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *)NULL;

    // Don't need to know if this is a secondary image or not
    psEntry = psFindEntry(psObj, eImageType, NULL);

    if ( psEntry != NULL)
    {
        CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT **ppsDesc;

        if (bContentAssociation == TRUE)
        {
            ppsDesc = &psEntry->psContent;
        }
        else
        {
            ppsDesc = &psEntry->psStatic;
        }

        if ((bCreate == TRUE) && (*ppsDesc == NULL))
        {
            // Create a a new descriptor
            *ppsDesc = (CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *)
                SMSO_hCreate(CHANNEL_ART_OBJECT_NAME": Image Descriptor",
                    sizeof(CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT),
                    SMS_INVALID_OBJECT, FALSE);
        }

        // Provide it to the caller
        psDesc = *ppsDesc;
    }

    return psDesc;
}

/*****************************************************************************
*
*   vIterateMatchingDescriptors
*
*   Reports every descriptor which contains an image with the matching
*   id / type.  Useful when updating attributes of an image when we all we
*   know are the details of that image
*
*****************************************************************************/
static void vIterateMatchingDescriptors (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN8 un8ImageType,
    ART_DESC_ITERATOR bIterator,
    void *pvArg
        )
{
    do
    {
        BOOLEAN bContinue = TRUE;
        SMSAPI_RETURN_CODE_ENUM eSmsApiReturnCode;
        CHANNEL_ART_TABLE_ENTRY_STRUCT *psEntry;
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sCurrentImageId;

        // Extract the entry for this type -- we
        // don't care if we're looking for a primary
        // or secondary image here since they both have
        // the same metadata
        psEntry = psFindEntry(psObj, (CHANNEL_ART_IMAGETYPE_ENUM)un8ImageType, NULL);
        if (psEntry == NULL)
        {
            // We got nothin'
            break;
        }

        // Check the static entry first
        if (psEntry->psStatic != NULL)
        {
            if (psEntry->psStatic->hImage != IMAGE_INVALID_OBJECT)
            {
                // Get this image's Id
                eSmsApiReturnCode = IMAGE_eProperty(
                    psEntry->psStatic->hImage, IMAGE_PROPERTY_INTERNAL_ID,
                    &sCurrentImageId);

                // If that went well and we have a match
                // then we're done
                if ((eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
                    (sCurrentImageId.uData.un32Value == (UN32)un16ImageId))
                {
                    // We have this image
                    bContinue = bIterator(psObj, psEntry->psStatic, TRUE, pvArg);
                }
            }
        }

        if (bContinue == FALSE)
        {
            // Stop here
            break;
        }

        // Then check the content (if necessary)
        if (psEntry->psContent != NULL)
        {
            if (psEntry->psContent->hImage != IMAGE_INVALID_OBJECT)
            {
                // Get this image's Id
                eSmsApiReturnCode = IMAGE_eProperty(
                    psEntry->psContent->hImage,
                    IMAGE_PROPERTY_INTERNAL_ID,
                    &sCurrentImageId);

                // If that went well and we have a match
                // then we're done
                if ((eSmsApiReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
                    (sCurrentImageId.uData.un32Value == (UN32)un16ImageId))
                {
                    // We have this image
                    bContinue = bIterator(psObj, psEntry->psContent, FALSE, pvArg);
                }
            }
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bReportAssocTypeIterator
*
*   Iterator which reports all assoc types an image is a part of
*
*****************************************************************************/
static BOOLEAN bReportAssocTypeIterator (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc,
    BOOLEAN bStatic,
    void *pvArg
        )
{
    CHANNEL_ART_ASSOC_TYPE_ITERATOR_STRUCT *psIterator =
        (CHANNEL_ART_ASSOC_TYPE_ITERATOR_STRUCT *)pvArg;

    if (pvArg == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": NULL provided as argument");
        return FALSE;
    }

    if (bStatic == TRUE)
    {
        psIterator->aeTypesToReport[psIterator->un8NumTypesToReport++] =
            CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC;
    }
    else
    {
        psIterator->aeTypesToReport[psIterator->un8NumTypesToReport++] =
            CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT;
    }

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bUpdateImageIterator
*
*   Iterator which updates all matches images with a given set of attributes
*
*****************************************************************************/
static BOOLEAN bUpdateImageIterator (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc,
    BOOLEAN bStatic,
    void *pvArg
        )
{
    CHANNEL_ART_UPDATE_IMAGE_ITERATOR_STRUCT *psUpdate =
        (CHANNEL_ART_UPDATE_IMAGE_ITERATOR_STRUCT *)pvArg;
    BOOLEAN bPropertiesUpdated = FALSE,
            bSecondaryUpdated = FALSE;

    if (pvArg == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_OBJECT_NAME
            ": NULL provided as argument");
        return FALSE;
    }

    do
    {
        BOOLEAN bSecondaryAdded = FALSE;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sPropertyValue;

        if (psDesc->hImage == IMAGE_INVALID_OBJECT)
        {
            // Can't find an image for this image type
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME
                ": bUpdateImageAttrs: Unable to locate image");

            // Skip this entry, this didn't go well
            break;
        }

        // Update this IMAGE's attributes
        sPropertyValue.eType = IMAGE_PROPERTY_INTERNAL_TYPE_UN32;
        sPropertyValue.uData.un32Value =
            psUpdate->psUpdatedAttribs->un16ImageVer;
        eReturnCode = IMAGE_eUpdateProperty(psDesc->hImage,
                                IMAGE_PROPERTY_INTERNAL_VERSION,
                                &sPropertyValue,
                                &bPropertiesUpdated
                                    );
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME": failed to update version (%s)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode)
                    );
            break;
        }
        else if (bPropertiesUpdated == FALSE)
        {
            // Nothing to do here
            break;
        }

        // Update our event control structure
        // based on the image we just updated
        vUpdateEventForImageType(
            psObj,
            (CHANNEL_ART_IMAGETYPE_ENUM)
                psUpdate->psUpdatedAttribs->un8ImageType,
                TRUE, TRUE );

        // Now, do we need to update the secondary image?
        if (psDesc->bSecondaryAvailable != psUpdate->psUpdatedAttribs->bSecondaryAvailable)
        {
            // Destroy the secondary image
            if (psUpdate->psUpdatedAttribs->bSecondaryAvailable == FALSE)
            {
                // Do we have anything to do?
                if (psDesc->hSecondaryImage != IMAGE_INVALID_OBJECT)
                {
                    // Yeah, do it and indicate we've updated the image
                    bSecondaryUpdated = TRUE;
                    IMAGE_vDestroy(psDesc->hSecondaryImage);
                    psDesc->hSecondaryImage = IMAGE_INVALID_OBJECT;
                }
            }
            else
            {
                // Create a new secondary image
                psDesc->hSecondaryImage = hCreateImage(
                    psObj,
                    psUpdate->psUpdatedAttribs,
                    CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO,
                    CHANNEL_ART_SECONDARY_LOGO_BK_OPTIONS);
                if (psDesc->hSecondaryImage != IMAGE_INVALID_OBJECT)
                {
                    bSecondaryUpdated = TRUE;
                    bSecondaryAdded = TRUE;
                }
            }
        }
        else if (psDesc->bSecondaryAvailable == TRUE)
        {
            // Update the secondary IMAGE's attributes
            sPropertyValue.eType = IMAGE_PROPERTY_INTERNAL_TYPE_UN32;
            sPropertyValue.uData.un32Value =
                psUpdate->psUpdatedAttribs->un16ImageVer;
            eReturnCode =
                IMAGE_eUpdateProperty(psDesc->hSecondaryImage,
                    IMAGE_PROPERTY_INTERNAL_VERSION,
                    &sPropertyValue, &bSecondaryUpdated);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_OBJECT_NAME
                    ": failed to update property for the secondary image (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode)
                        );
                break;
            }

            if (bSecondaryUpdated == TRUE)
            {
                bSecondaryAdded = TRUE;
            }
        }

        // Update secondary flag to match the current status
        psDesc->bSecondaryAvailable = psUpdate->psUpdatedAttribs->bSecondaryAvailable;

        if (bSecondaryUpdated == FALSE)
        {
            // We're done now if we didn't update the secondary image
            break;
        }

        // Update our event control structure
        // based on the image we just updated
        vUpdateEventForImageType(
            psObj,
            CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO,
            bSecondaryAdded, TRUE );

    } while (FALSE);

    // An update is flagged if either one of these happened
    psUpdate->bUpdated =
        psUpdate->bUpdated || bPropertiesUpdated || bSecondaryUpdated;

    return TRUE;
}

/*****************************************************************************
*
*   hCreateImage
*
*****************************************************************************/
static IMAGE_OBJECT hCreateImage(
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    CHANNEL_ART_BACKGROUND_OPTION_MASK tBackgroundOptions
        )
{
    IMAGE_OBJECT hImage = IMAGE_INVALID_OBJECT;

    do
    {
        CHANNEL_ART_IMAGE_DATA_STRUCT sImageData;

        // Initialize image data
        OSAL.bMemSet(&sImageData, 0, sizeof(sImageData));

        sImageData.eImageType = eImageType;
        sImageData.un16ImageId = psImageAttribs->un16ImageId;
        sImageData.un16ImageVer = psImageAttribs->un16ImageVer;
        sImageData.tFileSizeInBytes = 0;

        // Do we need to do anything else?
        if (tBackgroundOptions != CHANNEL_ART_BACKGROUND_OPTION_NONE)
        {
            IMAGE_BACKGROUND_STRUCT *psBackground;

            psBackground = &sImageData.sBackground;

            // Now, we need to configure the background based
            // on the provided option mask

            // Does this image have a background color?
            if ((tBackgroundOptions &
                    CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR) !=
                        CHANNEL_ART_BACKGROUND_OPTION_NONE)
            {
                psBackground->bUseBackgroundColor = TRUE;

                // Populate the color struct
                psBackground->sBackgroundColor.un8Red =
                    CHANNEL_ART_BACKGROUNDCOLOR_TO_RED(
                        psImageAttribs->sBackgroundGfx.un32BackgroundColor);
                psBackground->sBackgroundColor.un8Green =
                    CHANNEL_ART_BACKGROUNDCOLOR_TO_GREEN(
                        psImageAttribs->sBackgroundGfx.un32BackgroundColor);
                psBackground->sBackgroundColor.un8Blue =
                    CHANNEL_ART_BACKGROUNDCOLOR_TO_BLUE(
                        psImageAttribs->sBackgroundGfx.un32BackgroundColor);
            }

            // Does this image have a line bitmap associated with it?
            if ((tBackgroundOptions &
                    CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP) !=
                        CHANNEL_ART_BACKGROUND_OPTION_NONE)
            {
                psBackground->bUseLineBitmap = TRUE;

                // Calculate the number of entries in this bitmap
                psBackground->tNumLineBitmapEntries =
                    psImageAttribs->sBackgroundGfx.sLineBitmap.tLineBitmapByteSize
                        / sizeof(RGB_STRUCT);
            }

            psBackground->psLineBitmap =
                (const RGB_STRUCT *)
                    psImageAttribs->sBackgroundGfx.sLineBitmap.pun8LineBitmap;
        }

        // Create a new IMAGE_OBJECT
        hImage = IMAGE_hCreate(
            (SMS_OBJECT)psObj,
            psObj->pacFilePath,
            (IMAGE_FORMAT_ENUM)psImageAttribs->un8CodeType,
            &GsChannelArtImageIntf,
            (void*) &sImageData,
            TRUE
                );
        if (hImage == IMAGE_INVALID_OBJECT)
        {
            // We can't do any more
            break;
        }

        // We're good
    } while (FALSE);

    return hImage;
}

/*****************************************************************************
*
*   bImageInitSpecificData
*
*****************************************************************************/
static BOOLEAN bImageInitSpecificData (
    IMAGE_OBJECT hImage,
    void *pvSpecificData,
    void *pvArg
        )
{
    CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
        (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;
    CHANNEL_ART_IMAGE_DATA_STRUCT *psArg =
        (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvArg;

    // Assign data
    *psImageData = *psArg;

    return TRUE; 
}

/*****************************************************************************
*
*   vImageUninitSpecificData
*
*****************************************************************************/
static void vImageUninitSpecificData (
    IMAGE_OBJECT hImage,
    void *pvSpecificData
        )
{
    CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
        (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;

    // Remove Caption if exists
    if (psImageData->hCaption != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psImageData->hCaption);
        psImageData->hCaption = STRING_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bImageCalculateFileNameLen
*
*****************************************************************************/
static BOOLEAN bImageCalculateFileNameLen (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        )
{
    BOOLEAN bResult = FALSE;

    if ((ptLength != NULL) && (pacFilePath != NULL) &&
        (pvSpecificData != NULL))
    {
        CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
            (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;
        IMAGE_FORMAT_ENUM eFormat;

        // Get format
        eFormat = IMAGE.eFormat(hImage);

        // calculate the real length
        if ( CHANNEL_ART_IMAGETYPE_ALBUM == psImageData->eImageType )
        {
            *ptLength =
                CHANNEL_ART_MGR_tAlbumArtImageFilenameLen(eFormat,
                    psImageData->eImageType, pacFilePath);
        }
        else
        {
            *ptLength =
                CHANNEL_ART_MGR_tChannelArtImageFilenameLen(eFormat,
                    psImageData->eImageType, pacFilePath);
        }

        if (*ptLength > 0)
        {
            bResult = TRUE;
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bImageCreateImageFileName
 *
 *****************************************************************************/
static BOOLEAN bImageCreateImageFileName (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bOk = FALSE;

    if ((pacFilePath != NULL) && (pvSpecificData != NULL) &&
        (pacBuffer != NULL))
    {
        CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
            (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;
        IMAGE_FORMAT_ENUM eFormat;

        // Get format from the object
        eFormat = IMAGE.eFormat(hImage);

        // Do the generation
        if ( CHANNEL_ART_IMAGETYPE_ALBUM == psImageData->eImageType )
        {
            bOk = CHANNEL_ART_MGR_bAlbumArtCreateImageFilename(psImageData->un16ImageId,
                                 psImageData->un16ImageVer, pacFilePath,
                                 pacBuffer, tBufferSize );
        }
        else
        {
            bOk = CHANNEL_ART_MGR_bChannelArtCreateImageFilename(psImageData->un16ImageId,
                                 psImageData->un16ImageVer, psImageData->eImageType,
                                 eFormat, pacFilePath, pacBuffer, tBufferSize);
        }
    }
    
    return bOk;
}

/*****************************************************************************
*
*   eImageProperty
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eImageProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    void *pvSpecificData,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    if ((pvSpecificData != NULL) && (psValue != NULL))
    {
        CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
            (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;

        // Process properties the object knows at this point
        switch (eProperty)
        {
            case IMAGE_PROPERTY_INTERNAL_ID:
            {
                psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_UN32;
                psValue->uData.un32Value = psImageData->un16ImageId;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_VERSION:
            {
                psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_UN32;
                psValue->uData.un32Value = psImageData->un16ImageVer;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_CAPTION:
            {
                psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_STRING;
                psValue->uData.hStringValue = psImageData->hCaption;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_BACKGROUND:
            {
                psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_BACKGROUND;
                psValue->uData.psBackground = &psImageData->sBackground;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            default:
            {
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            }
            break;
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eImageUpdateProperty
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eImageUpdateProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    void *pvSpecificData,
    const IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue,
    BOOLEAN *pbUpdated
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    if ( pvSpecificData != NULL )
    {
        CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
            (CHANNEL_ART_IMAGE_DATA_STRUCT*)pvSpecificData;

        // Process properties the object knows at this point
        switch (eProperty)
        {
            case IMAGE_PROPERTY_INTERNAL_ID:
            {
                if (psImageData->un16ImageId != (UN16)psValue->uData.un32Value)
                {
                    psImageData->un16ImageId = (UN16)psValue->uData.un32Value;
                    *pbUpdated = TRUE;
                }
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_VERSION:
            {
                if (psImageData->un16ImageVer != (UN16)psValue->uData.un32Value)
                {
                    psImageData->un16ImageVer = (UN16)psValue->uData.un32Value;
                    *pbUpdated = TRUE;
                }
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_CAPTION:
            {
                if ( psImageData->hCaption != (STRING_OBJECT)psValue->uData.hStringValue )
                {
                    // The channel art image owns the captions; if we don't destroy it,
                    // nobody else will. Note: this is also used to clear our our
                    // caption when destroying an album art object.
                    if ( STRING_INVALID_OBJECT != psImageData->hCaption )
                    {
                        STRING_vDestroy( psImageData->hCaption );
                    }
                    psImageData->hCaption = (STRING_OBJECT)psValue->uData.hStringValue;
                    *pbUpdated = TRUE;
                }
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            default:
            {
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            }
            break;
        }

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bOk;
            size_t tPrevSize;

            // Store previous size
            tPrevSize = psImageData->tFileSizeInBytes;

            // Get new size
            bOk = bImageDetermineFileSize(hImage, psImageData);
            if (bOk == FALSE)
            {
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }
            else if (tPrevSize != psImageData->tFileSizeInBytes)
            {
                *pbUpdated = TRUE;
            }
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bImageDetermineFileSize
*
*****************************************************************************/
static BOOLEAN bImageDetermineFileSize (
    IMAGE_OBJECT hImage,
    void *pvSpecificData
        )
{
    FILE *psImageFile = NULL;
    BOOLEAN bOk = FALSE;
    CHANNEL_ART_IMAGE_DATA_STRUCT *psImageData =
        (CHANNEL_ART_IMAGE_DATA_STRUCT *)pvSpecificData;

    do
    {
        STRING_OBJECT hFileName;
        const char *pacFileName;

        // Get image filename
        hFileName = IMAGE.hFileName(hImage);
        if (hFileName == STRING_INVALID_OBJECT)
        {
            break;
        }

        // Extract C-like file name
        pacFileName = STRING.pacCStr(hFileName);
        if (pacFileName == NULL)
        {
            break;
        }

        // Open the Image file
        psImageFile = fopen(pacFileName, CHANNEL_ART_IMAGE_FILE_READ_MODE);
        if (psImageFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_OBJECT_NAME": failed to open file %s",
                pacFileName);
            break;
        }

        // Get OSAL to tell us the file size
        bOk = OSAL.bFileSystemGetFileSize(
                psImageFile, &psImageData->tFileSizeInBytes);
        if (bOk == FALSE)
        {
            psImageData->tFileSizeInBytes = 0;
            break;
        }
    } while (FALSE);

    // Close the file of it has been opened.
    if (psImageFile != NULL)
    {
        fclose(psImageFile);
    }

    return bOk;
}
