/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CHANNELARTMGR implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "srm_obj.h"
#include "sms_update.h"

#include "dataservice_mgr_impl.h"
#include "channel_art_mgr_obj.h"
#include "_channel_art_mgr_obj.h"
#include "channel_art_db_constants.h"
#include "image_obj.h"
#include "decoder_obj.h"
#include "channel_obj.h"
#include "category_obj.h"
#include "ccache.h"
#include "sms_event.h"

#include "db_util.h"
#include "sql_interface_obj.h"
#include "channel_art_interface.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static CHANNEL_ART_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    CHANNEL_ART_SERVICE_EVENT_MASK tEventRequestMask,
    CHANNEL_ART_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tSelectedImageTypes,
    CHANNEL_ART_OPTIONS_MASK tOptions,
    ...
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    DATASERVICE_CREATE_STRUCT sCreate;
    BOOLEAN bOk;
    va_list tList; // variable arguments list

    // Validate selected images
    if (CHANNEL_ART_SELECTED_IS_VALID(tSelectedImageTypes) != TRUE)
    {
        return CHANNEL_ART_SERVICE_INVALID_OBJECT;
    }

    // Populate our data service creation structure
    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = CHANNEL_ART_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(CHANNEL_ART_MGR_OBJECT_STRUCT);
    sCreate.tDataID = DATASERVICE_ID_ART;
    sCreate.bRequiresDeviceGroup = TRUE;

    // We need multiple DSI support to differentiate between channel and album
    // art data
    sCreate.bMultiDSIRequired = TRUE;

    // Set up our product info for Album Art.
    sCreate.sProductsInfo.peProducts = GasArtProducts;
    sCreate.sProductsInfo.un8Count = sizeof(GasArtProducts)/sizeof(GasArtProducts[0]);
    sCreate.sProductsInfo.bGetDSIForProduct = bGetDSIInfo;
    sCreate.sProductsInfo.eGetNextProductState = eGetNextState;

    // This service does not need time information
    sCreate.bTimeRequired = FALSE;

    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (
        DATASERVICE_EVENT_STATE |
        DATASERVICE_EVENT_NEW_DATA |
        DATASERVICE_EVENT_TIMEOUT |
        DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE |
        DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED |
        DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED |
        DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE);

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psObj = (CHANNEL_ART_MGR_OBJECT_STRUCT *)
        DATASERVICE_IMPL_hCreateNewService( &sCreate );
    if (psObj == NULL)
    {
        // Can't create the service, fail out!
        return CHANNEL_ART_SERVICE_INVALID_OBJECT;
    }

    // Initialize the channel art DBMS connection handles
    psObj->hChanSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    psObj->hAlbumPerSQLConnection = SQL_INTERFACE_INVALID_OBJECT;

    // Initialize the flag indicating that we don't need to
    // reset art objects at this time
    psObj->bResetArtObjects = FALSE;

    // Save the image types requested by the application
    psObj->tSelectedImageTypes = tSelectedImageTypes;
    psObj->bSecondarySupported =
        ((tSelectedImageTypes & CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO)
            == CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO);

    // Init association processing attributes
    OSAL.bMemSet( &psObj->sAssocCtrl, 0,
        sizeof(CHANNEL_ART_MGR_ASSOC_CTRL_STRUCT));

    // Init OTA processing attributes
    psObj->sOTACtrl.bEventsPending = FALSE;

    // Initialize the channel art path attributes
    psObj->pacBasePath = NULL;
    psObj->pacChannelArtServiceFilePath = NULL;
    psObj->pacEphemeralPath = NULL;
    psObj->pacFullyQualifiedChanDatabaseFilePath = NULL;
    psObj->pacFullyQualifiedAlbumPerDatabaseFilePath = NULL;
    psObj->pacFullyQualifiedAlbumRefDatabaseFilePath = NULL;

    // Initialize book-keeping attributes; as the service default images
    // typically have the same ID as their service ID; thus, we'll start
    // the ephemeral image IDs after any possible service ID number.
    psObj->un32NextCleanupSecs = 0;
    psObj->un32NextEphemeralID = (UN32)GsAlbumArtIntf.tEndServiceID + 1;

    // Process the options provided to us

    // grab variable arguments (pop)
    va_start(tList, tOptions);
    bOk = bProcessOptions(psObj, tOptions, &tList);
    // restore stack (push)
    va_end(tList);

    if (bOk == FALSE)
    {
        // Error!
        vUninitObject( psObj );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

        return CHANNEL_ART_SERVICE_INVALID_OBJECT;
    }

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        CHANNEL_ART_SERVICE_EVENT_NONE,
        tEventRequestMask,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
        pvEventCallbackArg);

    // Initialize the art owner object
    bOk = bInitializeArtOwner(psObj);

    if ( bOk == TRUE )
    {
        // Construct the paths needed by both Channel Art and 
        // Album Art data services
        bOk = bBuildCommonFilePaths( psObj );
    }

    if ( bOk == TRUE )
    {
        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);
    }

    if ( bOk == FALSE )
    {
        // Error!
        vUninitObject( psObj );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);

        return CHANNEL_ART_SERVICE_INVALID_OBJECT;
    }

    return (CHANNEL_ART_SERVICE_OBJECT)psObj;
}

/*****************************************************************************
*
*   eUseArt
*
*   This API is used to perform any type of action on the images of
*   a CHANNEL_ART_OBJECT. It allows a caller to provide a function and
*   argument which allows the direct manipulation of IMAGE objects using
*   a provided CHANNEL_ART_OBJECT handle. Manipulation of the IMAGE object
*   is performed while safely observing exclusive access rules.
*
*   Inputs:
*       hArt - A handle to a valid CHANNEL_ART_OBJECT which
*           holds the desired IMAGE objects.
*       vChannelArtAccessCallback - The callback which will be
*           invoked if the requested image type is located within the
*           provided art object.
*       *pvCallbackArg - The caller-provided argument which will be
*       given as a paramter to vChannelArtAccessCallback
*
*   Outputs:
*       SMSAPI_RETURN_CODE_SUCCESS on success or
*       SMSAPI_RETURN_CODE_INVALID_INPUT if any parameter check fails.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUseArt (
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_ACCESS_CALLBACK vChannelArtAccessCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bLocked;
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages;

    // Verify input pointers
    if  (vChannelArtAccessCallback == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hArt, OSAL_OBJ_TIMEOUT_INFINITE);

    if ( bLocked == FALSE )
    {
        // We were provided a bad channel art handle
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // We need to provide the available image type to the
    // application's callback; get that here ...
    tAvailableImages = CHANNEL_ART_tImagesAvailable( hArt );

    // Call function
    vChannelArtAccessCallback( hArt, tAvailableImages, pvCallbackArg );

    // Unlock and return success
    SMSO_vUnlock( (SMS_OBJECT)hArt );

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eImage
*
*   This API is used to perform any type of action on an IMAGE object.
*   It allows a caller to provide a function and argument which
*   allows the direct manipulation of an IMAGE object by first providing
*   a CHANNEL_ART_OBJECT handle and an image type. As this method is intended
*   to be called from within an eUseArt callback, manipulation of the
*   IMAGE object is performed while safely observing exclusive access rules.
*
*   Inputs:
*       hArt - A handle to a valid CHANNEL_ART_OBJECT from which
*           to derive the IMAGE object.
*       eImageType - Channel art identifier for the IMAGE which is to be
*           queried.
*       phImage - A pointer to an IMAGE handle; this API will return
*           the requested IMAGE (if found) via this pointer.
*       pbIsDefault - A pointer to a BOOLEAN (the pointer may be null).
*           The boolean will be set TRUE if the requested image is a
*           default, or FALSE otherwise. If the pointer is NULL, the
*           default status check is skipped.
*
*   Outputs:
*       SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on
*           error, SMSAPI_RETURN_CODE_INVALID_INPUT if any parameter
*           check fails, or SMSAPI_RETURN_CODE_NO_OBJECTS if the requested
*           image type is not available.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eImage (
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    IMAGE_OBJECT *phImage,
    BOOLEAN *pbIsDefault
        )
{
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sImageId;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify the image input pointer; the pointer for
    // bIsDefault can be NULL (in which case it won't be
    // populated), so we don't bother checking  that here.
    if  ( phImage == NULL )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get the requested IMAGE from the CHANNEL_ART handle ...
    *phImage = CHANNEL_ART_hImage( hArt, eImageType );

    // ... and make sure it's available
    if ( IMAGE_INVALID_OBJECT == *phImage )
    {
        // This image is not available; null out the
        // image in case the app didn't do that itself
        *phImage = IMAGE_INVALID_OBJECT;
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    // Check to see if the user is interested in the
    // defaultitude of the image. If not, our work
    // is done.
    if ( NULL == pbIsDefault )
    {
        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    // If we made it here, let he caller know
    // if the image is default or not.

    // Get the image ID from the image object
    eReturnCode =
        IMAGE_eProperty(*phImage, IMAGE_PROPERTY_INTERNAL_ID, &sImageId);
    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode )
    {
        // We have the image but can't get it's ID?
        // Something's serious amiss here.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "Failed to retrieve the ID for a valid Image."
                "This should not happen! (%s)",
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));

        return eReturnCode;
    }

    // We now need to figure out if this is
    // a "default" image.  Luckily, the default ID
    // is always the same.  So, just check that.
    if ( CHANNEL_ART_DEFAULT_IMAGE_ID == sImageId.uData.un32Value )
    {
        *pbIsDefault = TRUE;
    }
    else
    {
        *pbIsDefault = FALSE;
    }

    // All is well
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CHANNEL_ART_MGR_hGetArtForServiceId
*
*     Channel objects will make use of this API in order to get their
*     channel art handle.  This call is made in the context of an
*     SMS callback provided to the application.
*
*****************************************************************************/
CHANNEL_ART_OBJECT CHANNEL_ART_MGR_hGetArtForServiceId (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SERVICE_ID tServiceId,
    CHANNEL_ART_VERSION* ptArtVersion
        )
{
    BOOLEAN bValid;
    CHANNEL_ART_OBJECT hChannelArt = CHANNEL_ART_INVALID_OBJECT;
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;

    // Verify the channel art service
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hChannelArtService);
    if ( TRUE == bValid )
    {
        UN16 un16SourceId = (UN16)tServiceId;

        // Get the channel art object for this service id
        hChannelArt =
            hChanGetArtForSourceAndType( psObj, un16SourceId, FALSE );

        if (ptArtVersion != NULL)
        {
            *ptArtVersion = CHANNEL_ART_tVersion( hChannelArt );
        }
    }

    return hChannelArt;
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_hGetArtForCategoryId
*
*   Category objects will make use of this API in order to get their
*   channel art handle.  This call is made in the context of an
*   SMS task.
*
*****************************************************************************/
CHANNEL_ART_OBJECT CHANNEL_ART_MGR_hGetArtForCategoryId (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CATEGORY_ID tCategoryId
        )
{
    CHANNEL_ART_OBJECT hChannelArt = CHANNEL_ART_INVALID_OBJECT;

    // Do we have an object here at all?
    if (hChannelArtService != CHANNEL_ART_SERVICE_INVALID_OBJECT)
    {
        BOOLEAN bValid;

        // Yes, verify the channel art service object
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hChannelArtService);
        if ( TRUE == bValid )
        {
            CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
                (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
            UN16 un16SourceId = (UN16)tCategoryId;

            // Get the channel art object for this category id
            hChannelArt =
                hChanGetArtForSourceAndType( psObj, un16SourceId, TRUE );
        }
    }

    return hChannelArt;
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_hGetArtForCDO
*
*   CDOs will make use of this API in order to get their
*   channel art handle.  This call is made in the context of an
*   SMS task. This f'n basically does some validation and locking
*   before calling hGetAlbumArt, which is the "heart" of the album
*   art decision algorithm.
*
*****************************************************************************/
CHANNEL_ART_OBJECT CHANNEL_ART_MGR_hGetArtForCDO (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    SERVICE_ID tServiceId,
    PROGRAM_ID tProgramId,
    CDO_TYPE_ENUM eType
        )
{
    CHANNEL_ART_OBJECT hChannelArt = CHANNEL_ART_INVALID_OBJECT;
    BOOLEAN bValid, bLocked;
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;

    // Verify the service object is valid and that album art is READY
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hChannelArtService);
    psObj = (CHANNEL_ART_MGR_OBJECT_STRUCT*)hChannelArtService;

    // If our dataservice isn't valid, or our album art product isn't ready, then
    // just return an invalid album art object.
    if ( TRUE != bValid )
    {
        return hChannelArt;
    }

    // Now block the art owner so we can look for the right
    // association without the list changing underneath us.
    bLocked = SMSO_bLock( (SMS_OBJECT)psObj->psArtOwner, 
                            OSAL_OBJ_TIMEOUT_INFINITE 
                        );        

    if (bLocked == FALSE)
    {
        return hChannelArt;
    }

    // Look for the association
    hChannelArt =
        hAlbumGetArt(psObj->psArtOwner, tServiceId, tProgramId, eType);

    // Release the art owner
    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return hChannelArt;
}

/*****************************************************************************
 *
 *  CHANNEL_ART_vDecoderEventHandler
 *
 *****************************************************************************/
void CHANNEL_ART_vDecoderEventHandler (
    DECODER_OBJECT hDecoder,
    SMS_EVENT_ART_STRUCT const *psArt
        )
{
    static BOOLEAN bNeedToSendDirectedUnsub = FALSE;
    BOOLEAN bOwner;
    CCACHE_OBJECT hCCache;

    // Verify ownership of the decoder
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            ": CHANNEL_ART_vDecoderEventHandler: called in wrong context");
        return;
    }

    // Get this decoder's channel cache
    hCCache = DECODER_hCCache(hDecoder);
    if (hCCache == CCACHE_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            ": CHANNEL_ART_vDecoderEventHandler: Unable to find channel cache");
        return;
    }

    switch (psArt->eStatus)
    {
        case SMS_EVENT_ART_STATUS_START:
        {
            // Whenever we start the service we
            // know we'll eventually need to send the unsub
            bNeedToSendDirectedUnsub = TRUE;

            // We need to update the service handle
            CCACHE_vUpdateChannelArt(
                hCCache, psArt->hArtService,
                CCACHE_UPDATE_TYPE_ART_SERVICE );
        }
        break;

        case SMS_EVENT_ART_STATUS_STOP:
        {
            // Clear out our handle and wipe our art when
            // shutting down the service.

            CCACHE_vUpdateChannelArt(
                hCCache, CHANNEL_ART_SERVICE_INVALID_OBJECT,
                CCACHE_UPDATE_TYPE_ART_SERVICE );

            // Do we need to send a "directed unsubscribe"
            // to the manager?
            if ( TRUE == bNeedToSendDirectedUnsub )
            {
                // Yes, send it now
                BOOLEAN bPosted;

                // Send a "directed unsubscribe" to this manager
                bPosted = DATASERVICE_IMPL_bPostEvent(
                    (DATASERVICE_IMPL_HDL)psArt->hArtService,
                    DATASERVICE_FW_EVENT_DECODER_UNSUBSCRIBED,
                    hDecoder);

                // If the event was sent then we mark
                // this as false since we don't want to send it again
                bNeedToSendDirectedUnsub = (bPosted == TRUE) ? FALSE : TRUE;
            }
        }
        break;

        case SMS_EVENT_ART_STATUS_ALBUM_START:
        case SMS_EVENT_ART_STATUS_ALBUM_STOP:
        {
            // Update the album art for all channels in the CCache
            // update art as needed
            CCACHE_vUpdateChannelArt(
                hCCache, psArt->hArtService,
                CCACHE_UPDATE_TYPE_ALBUM_ART );
        }
        break;

        case SMS_EVENT_ART_STATUS_CHAN_START:
        case SMS_EVENT_ART_STATUS_CHAN_STOP:
        {
            // Update the channel art for all channels in the CCache
            CCACHE_vUpdateChannelArt(
                hCCache, psArt->hArtService,
                CCACHE_UPDATE_TYPE_CHAN_ART );
        }
        break;

        case SMS_EVENT_ART_STATUS_UPDATE:
        {
            // Update the art for this decoder now
            vUpdateDecoderArt(
                hCCache,
                psArt->hArtService,
                psArt->eType,
                psArt->hArtList);

            // We're all done with this list!
            vDestroyDecoderUpdateList(psArt->hArtList);
        }
        break;

        default:
        {
        }
        break;
    }

    return;
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_tChannelArtImageFilenameLen
*
*   This function is used by all interested parties in order to determine
*   the size of the filenames that are used by the channel art data service
*   provided a given path image format (ie. PNG), and image type (ie. logo).
*   Defers to tImageFilenameLen.
*
*****************************************************************************/
size_t CHANNEL_ART_MGR_tChannelArtImageFilenameLen (
    IMAGE_FORMAT_ENUM eFormat,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    const char *pacFilePath
        )
{
    return tImageFilenameLen (
        eFormat, eImageType, pacFilePath,
        CHANNEL_ART_FILE_NAME_ID_LABEL_LEN );
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_tAlbumArtImageFilenameLen
*
*   This function is used by all interested parties in order to determine
*   the size of the filenames that are used by the album art product.
*   Defers to tImageFilenameLen.
*
*****************************************************************************/
size_t CHANNEL_ART_MGR_tAlbumArtImageFilenameLen (
    IMAGE_FORMAT_ENUM eFormat,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    const char *pacFilePath
        )
{
    return tImageFilenameLen (
        eFormat, eImageType, pacFilePath,
        ALBUM_ART_FILE_NAME_ID_LABEL_LEN );
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_bChannelArtCreateImageFilename
*
*   This function is used by the channel art manager in order to generate
*   a filename for a newly received image file (files are received OTA).  In
*   addition, the IMAGE object which provides an abstract representation of
*   an image file must use the same naming conventions for locating files
*   in NV storage.  In order to ensure a common naming approach, the channel
*   art manager provides this as a friend function.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_MGR_bChannelArtCreateImageFilename (
    UN16 un16ImageId,
    UN16 un16ImageVer,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    IMAGE_FORMAT_ENUM eFormat,
    const char *pacFilePath,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    size_t tFilenameLen;
    const char *pacFileExtension;

    // Check the image type
    if ( eImageType >= CHANNEL_ART_IMAGETYPE_MAX )
    {
        // This is an invalid image type
        return FALSE;
    }

    // Calculate the name of the file name
    tFilenameLen =
        tImageFilenameLen( eFormat, eImageType,
                pacFilePath, CHANNEL_ART_FILE_NAME_ID_LABEL_LEN );

    if (tFilenameLen == 0)
    {
        return FALSE;
    }

    // Determine which file extension to use
    switch (eFormat)
    {
        case IMAGE_FORMAT_PNG:
        {
            pacFileExtension = CHANNEL_ART_PNG_FILE_EXTENSION;
        }
        break;

        case IMAGE_FORMAT_JPEG:
        {
            pacFileExtension = CHANNEL_ART_JPEG_FILE_EXTENSION;
        }
        break;

        // No extension, no file!
        default:
        {
            return FALSE;
        }
    }

    // Ensure the destination buffer size can hold the
    // filename which will be generated
    if (tBufferSize < tFilenameLen)
    {
        return FALSE;
    }

    // Generate the filename
    snprintf( pacBuffer,
              tFilenameLen,
              "%s"CHANNEL_ART_FILE_NAME_ID_LABEL"%s",
              pacFilePath,
              un16ImageId,
              eImageType,
              pacFileExtension );

    return TRUE;
}

/*****************************************************************************
*
*   CHANNEL_ART_MGR_bAlbumArtCreateImageFilename
*
*   This function is used by the channel art manager in order to generate
*   a filename for a newly received image file (files are received OTA).  In
*   addition, the IMAGE object which provides an abstract representation of
*   an image file must use the same naming conventions for locating files
*   in NV storage.  In order to ensure a common naming approach, the channel
*   art manager provides this as a friend function.
*
*****************************************************************************/
BOOLEAN CHANNEL_ART_MGR_bAlbumArtCreateImageFilename (
    UN16 un16ImageId,
    UN16 un16ImageVer,
    const char *pacFilePath,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    size_t tFilenameLen;

    // Calculate the name of the file name; album art only uses
    // JPEG.
    tFilenameLen
        = tImageFilenameLen( IMAGE_FORMAT_JPEG, 0,
                pacFilePath, ALBUM_ART_FILE_NAME_ID_LABEL_LEN );

    // Ensure the destination buffer size can hold the
    // filename which will be generated
    if ( ( tFilenameLen == 0 ) ||
         ( tBufferSize < tFilenameLen ) )
    {
        return FALSE;
    }

    // Generate the filename
    snprintf( pacBuffer, tFilenameLen, "%s"ALBUM_ART_FILE_NAME_ID_LABEL,
            pacFilePath, un16ImageId, un16ImageVer );

    return TRUE;
}

#if 0
/*****************************************************************************
*
*   CHANNEL_ART_MGR_vAlbumArtRemoveAssoc
*
*   Reserved for future "safe-to-delete" functionality; at some point in the
*   future, we're supposed to get some sort of 'safe-to-delete art'
*   notification telling us that a song fell out of the playback buffer.
*
*   Removes an album association given a source ID and program ID. Called by
*   the CDO object to clean up old album art associations that are no longer
*   needed. Defers to the static function pvRemoveAssocEntry.
*
*****************************************************************************/
void CHANNEL_ART_MGR_vAlbumArtRemoveAssoc (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    UN16 un16Source,
    UN32 un32ProgramId
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
    BOOLEAN bValid;

    // Verify the channel art service is up and running. If not, it isn't
    // really an error, as CDOs may try to get their art before the channel
    // art service is ready.
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bValid == TRUE)
    {
        BOOLEAN bLocked;

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            return;
        }

        vAlbumRemoveAssocEntry ( psObj->psArtOwner, un16Source, un32ProgramId );

        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
    }

    return;
}
#endif

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tValidateOptionString
*
*****************************************************************************/
static size_t tValidateOptionString (
    const char *pacOptionString,
    const char *pacOptionName
        )
{
    size_t tPathLen = 0;

    if (pacOptionString == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": No argument provided for option %s",
                pacOptionName);
        return tPathLen;
    }

    tPathLen = strlen(pacOptionString);

    if (tPathLen == 0)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Invalid argument provided for option %s",
                pacOptionName);
    }

    return tPathLen;
}

/*****************************************************************************
*
*   bProcessOptions
*
*****************************************************************************/
static BOOLEAN bProcessOptions (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_OPTIONS_MASK tOptions,
    va_list *ptArgList
        )
{
    if ((ptArgList == NULL) || (psObj == NULL))
    {
        // No arguments?
        return FALSE;
    }

    // Verify we have been provided only valid options
    // for this service
    if( (tOptions & ~CHANNEL_ART_OPTION_ALL) != CHANNEL_ART_OPTION_NONE )
    {
        return FALSE;
    }

    // Process the arguments in order (by option value)

    // File Path
    if ((tOptions & CHANNEL_ART_OPTION_FILE_PATH)
            == CHANNEL_ART_OPTION_FILE_PATH)
    {
        BOOLEAN bSuccess = FALSE;

        do
        {
            const char *pcFilePath;
            size_t tPathLen;

            // Grab the provided path
            pcFilePath = (const char *)va_arg( *ptArgList, const char * );

            tPathLen = tValidateOptionString( pcFilePath,
                    "CHANNEL_ART_OPTION_FILE_PATH" );
            if ( (size_t)0 == tPathLen )
            {
                break;
            }

            // Allocate memory to store this path
            psObj->pacBasePath =
                   (char *) SMSO_hCreate(
                        CHANNEL_ART_MGR_OBJECT_NAME":FilePath",
                        tPathLen + 1, SMS_INVALID_OBJECT, FALSE);
            if (psObj->pacBasePath == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME
                        ": unable to allocate memory");
                break;
            }

            // Copy the string
            snprintf(&psObj->pacBasePath[0],
                tPathLen + 1, "%s", &pcFilePath[0]);

            bSuccess = TRUE;
        } while (FALSE);

        if (bSuccess == FALSE)
        {
            if (psObj->pacBasePath != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psObj->pacBasePath);
                psObj->pacBasePath = NULL;
            }

            return FALSE;
        }
    }

    if ( (tOptions & CHANNEL_ART_OPTION_EPHEMERAL_FILE_PATH)
            == CHANNEL_ART_OPTION_EPHEMERAL_FILE_PATH )
    {
        BOOLEAN bSuccess = FALSE;

        // We start with one because of the NULL terminator at
        // the end of the string
        UN8 un8ExtraBytes = 1;

        do
        {
            const char *pcFilePath;
            size_t tPathLen;

            // Grab the provided path
            pcFilePath = (const char *)va_arg( *ptArgList, const char * );

            tPathLen = tValidateOptionString( pcFilePath,
                    "CHANNEL_ART_OPTION_EPHEMERAL_FILE_PATH" );
            if ( (size_t)0 == tPathLen )
            {
                break;
            }

            // Check to see if we need a delimiter; we append this conditionally
            // here, although we could just blindly append a '/' when creating 
            // image filenames. Doing this here saves us a small amount of time vs
            // doing the append on every Image object creation, and also makes
            // the CLI filename displays look "cleaner", as there's no chance
            // to have doubled delimiters.

            tPathLen = strlen(pcFilePath);
            if ( (pcFilePath[tPathLen-1] != '\\') &&
                 (pcFilePath[tPathLen-1] != '/') )
            {
                // We need an extra byte for a delimiter
                un8ExtraBytes++;
            }

            // Allocate memory to store this path
            psObj->pacEphemeralPath =
                   (char *) SMSO_hCreate(
                        CHANNEL_ART_MGR_OBJECT_NAME":EphemeralPath", 
                        tPathLen + un8ExtraBytes, SMS_INVALID_OBJECT, FALSE);
            if ( NULL == psObj->pacEphemeralPath )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME
                        ": unable to allocate memory");
                break;
            }

            // Copy the string
            snprintf(&psObj->pacEphemeralPath[0],
                tPathLen + un8ExtraBytes, "%s%s", &pcFilePath[0],
                (un8ExtraBytes == 2) ? "/" : "");

            bSuccess = TRUE;
        } while ( FALSE );

        if ( FALSE == bSuccess )
        {
            if ( NULL != psObj->pacEphemeralPath )
            {
                SMSO_vDestroy((SMS_OBJECT)psObj->pacEphemeralPath);
                psObj->pacEphemeralPath = NULL;
            }

            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid;
    SMSAPI_EVENT_MASK tEventMask =
        CHANNEL_ART_SERVICE_EVENT_NONE;

    // Get our channel art handle from the callback argument
    psObj = (CHANNEL_ART_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // Handle Channel Art Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;
                BOOLEAN bStateChanged;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsChannelArtStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;
                }
            }
            break;

            // A decoder has subscribed to this service
            case DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED:
            {
                DECODER_OBJECT hDecoder =
                    (DECODER_OBJECT)pvEventArg;

                if (hDecoder != DECODER_INVALID_OBJECT)
                {
                    // Process the subscription
                    vHandleDecoderSubscribed( psObj, hDecoder );
                }
            }
            break;

            case DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED:
            {
                // Event argument is the decoder wishing to
                // be unsubscribed from this service
                DECODER_OBJECT hDecoder = (DECODER_OBJECT)pvEventArg;

                // Send an "art stop" event to the decoder now
                // Don't worry about return value
                bSendUpdate(psObj,  CHANNEL_ART_UPDATE_TYPE_STOP, hDecoder);
            }
            break;

            // This service has a message to process
            case DATASERVICE_EVENT_NEW_DATA:
            {   
                BOOLEAN bOk;

                // Event argument is a message payload
                OSAL_BUFFER_HDL hPayload =
                    (OSAL_BUFFER_HDL)pvEventArg;

                // Handle the message reception
                bOk = bHandleMessageReception( psObj, hPayload );

                if (bOk != TRUE)
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= CHANNEL_ART_SERVICE_EVENT_SERVICE_STATE;

                    DATASERVICE_IMPL_vLog(
                        CHANNEL_ART_MGR_OBJECT_NAME": Failed to process message\n");
                }
            }
            break;

            case DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE:
            {
                DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psProdEvent;
                psProdEvent = (DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT*)pvEventArg;

                // Double-check this is one of the channel art service's
                // products that we're getting before we process the state change.
                if ( ( DATA_PRODUCT_TYPE_ALBUM_ART != psProdEvent->eType ) &&
                     ( DATA_PRODUCT_TYPE_CHANNEL_ART != psProdEvent->eType ) )
                {
                    DATASERVICE_IMPL_vLog(
                        CHANNEL_ART_MGR_OBJECT_NAME": Received state for unknown service: %u\n",
                            psProdEvent->eType );
                }
                else
                {
                    BOOLEAN bStateChanged;

                    // Now let the state machine do what it needs to (including
                    // object creation / deletion. )
                    bStateChanged = bProductFSM( psProdEvent->eType,
                            psProdEvent->ePreviousState, psProdEvent->eState, psObj );

                    if ( TRUE == bStateChanged )
                    {
                        // Notify that a state change has occurred to the application
                        if ( DATA_PRODUCT_TYPE_ALBUM_ART == psProdEvent->eType )
                        {
                            tEventMask |= CHANNEL_ART_SERVICE_EVENT_ALBUM_PROD_STATE;
                        }
                        else if ( DATA_PRODUCT_TYPE_CHANNEL_ART == psProdEvent->eType )
                        {
                            tEventMask |= CHANNEL_ART_SERVICE_EVENT_CHAN_PROD_STATE;
                        }

                        DATASERVICE_IMPL_vLog(
                                CHANNEL_ART_MGR_OBJECT_NAME
                                ": Product %u transition to state %u\n",
                                psProdEvent->eType,
                                psProdEvent->eState );
                    }
                }
            }
            break;

            case DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE:
            {
                BOOLEAN bLocked;
                const SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION *puArg =
                    (const SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION*)pvEventArg;
                const SXM_EVENT_DATASERVICE_SXI_CONTENT_BUFFERED_STRUCT *psCB =
                    &puArg->sContentBuffered;

                DATASERVICE_IMPL_vLog(CHANNEL_ART_MGR_OBJECT_NAME
                    ": Processing SXi Messages (Content Buffered SID=%d, CID=%d, PID Count=%d)\n",
                    (int)psCB->tServiceId, (int)psCB->tChannleId,
                    (int)psCB->un8PIDCount);

                // Lock the channel art owner object
                bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner,
                    OSAL_OBJ_TIMEOUT_INFINITE);
                if (bLocked == TRUE)
                {
                    vAlbumPidAssocListScrub(psObj->psArtOwner,
                        psCB->tServiceId, psCB->tChannleId,
                        psCB->atPIDList, psCB->un8PIDCount);

                    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME
                        ": failed to lock owner object");
                }
            }
            break;

            // We only have one timeout at the moment, and it's the
            // album art association cleanup event.
            case DATASERVICE_EVENT_TIMEOUT:
            {
                vAlbumAssocCleanup( (CHANNEL_ART_SERVICE_OBJECT)psObj );
            }
            break;

            default:
            break;
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (tCurrentEvent == DATASERVICE_EVENT_STATE)
        {
            DATASERVICE_STATE_ENUM eFinalState;

            // If we're stopped, don't allow any more events to be generated
            eFinalState = DATASERVICE_IMPL_eState((DATASERVICE_IMPL_HDL)psObj);
            if (eFinalState == DATASERVICE_STATE_STOPPED)
            {
                // Filter out all further channel art mgr updates
                SMSU_tFilter(&psObj->sEvent, CHANNEL_ART_SERVICE_EVENT_ALL);

                // Clean up memory allocations
                vUninitObject( psObj );
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   bInitializeArtOwner
*
*****************************************************************************/
static BOOLEAN bInitializeArtOwner (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Name the channel art owner object
    snprintf( &acName[0], sizeof(acName),
              CHANNEL_ART_MGR_OBJECT_NAME":ArtOwner");

    // Create it
    psObj->psArtOwner = (CHANNEL_ART_OWNER_OBJECT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(CHANNEL_ART_OWNER_OBJECT_STRUCT),
            SMS_INVALID_OBJECT, TRUE);
    if (psObj->psArtOwner == NULL)
    {
        return FALSE;
    }

    // Relinquish ownership
    SMSO_vUnlock((SMS_OBJECT) psObj->psArtOwner);
    return TRUE;
}


/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;

    if (psObj->psArtOwner == NULL)
    {
        return;
    }

    // Lock the channel art owner object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": Unable to lock art owner");
        return;
    }

    vChanCleanUp(psObj->psArtOwner, TRUE);

    vAlbumCleanUp(psObj->psArtOwner, TRUE);

    // Destroy the channel art owner
    SMSO_vDestroy((SMS_OBJECT)psObj->psArtOwner);
    psObj->psArtOwner = NULL;

    // Destroy path allocations
    if (psObj->pacBasePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacBasePath);
        psObj->pacBasePath = NULL;
    }

    if ( psObj->pacEphemeralPath != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacEphemeralPath);
        psObj->pacEphemeralPath = NULL;
    }

    if (psObj->pacChannelArtServiceFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacChannelArtServiceFilePath);
        psObj->pacChannelArtServiceFilePath = NULL;
    }

    if (psObj->pacFullyQualifiedChanDatabaseFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacFullyQualifiedChanDatabaseFilePath);
        psObj->pacFullyQualifiedChanDatabaseFilePath = NULL;
    }

    if ( psObj->pacFullyQualifiedAlbumPerDatabaseFilePath != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacFullyQualifiedAlbumPerDatabaseFilePath);
        psObj->pacFullyQualifiedAlbumPerDatabaseFilePath = NULL;
    }

    if ( psObj->pacFullyQualifiedAlbumRefDatabaseFilePath != NULL )
    {
        SMSO_vDestroy((SMS_OBJECT)
            psObj->pacFullyQualifiedAlbumRefDatabaseFilePath);
        psObj->pacFullyQualifiedAlbumRefDatabaseFilePath = NULL;
    }

    // Clear our handle to the cleanup event
    psObj->hCleanupEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

    // Destroy the SMS Update object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   bInitDefaultAssociations
*
*   This function initializes the default associations (category & channel)
*   by creating channel art objects for them and additionally sets some
*   static attributes in the association structures themselves.
*
*   The association row structures utilized for delayed processing of updates
*   to the default associations are also initialized here.
*
*****************************************************************************/
static BOOLEAN bInitDefaultAssociations (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return FALSE;
    }

    // Set the default category association attributes
    psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt = CHANNEL_ART_INVALID_OBJECT;
    psObj->psArtOwner->sDefaultCategoryAssoc.bCategory = TRUE;
    psObj->psArtOwner->sDefaultCategoryAssoc.bDefault = TRUE;
    psObj->psArtOwner->sDefaultCategoryAssoc.un16Source = CHANNEL_ART_INVALID_SOURCE_ID;

    // Set the default channel association attributes
    psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt = CHANNEL_ART_INVALID_OBJECT;
    psObj->psArtOwner->sDefaultChannelAssoc.bCategory = FALSE;
    psObj->psArtOwner->sDefaultChannelAssoc.bDefault = TRUE;
    psObj->psArtOwner->sDefaultChannelAssoc.un16Source = CHANNEL_ART_INVALID_SOURCE_ID;

    do
    {
        // Create the default category art object only if necessary
        if ((psObj->tSelectedImageTypes &  CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND)
                == CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND)
        {
            // Create the channel art object for the
            // default category association
            psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt =
                CHANNEL_ART_hCreate(
                    (SMS_OBJECT)psObj->psArtOwner,
                    &psObj->pacChannelArtServiceFilePath[0]);

            if (psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt
                    == CHANNEL_ART_INVALID_OBJECT)
            {
                break;
            }
        }

        // Create the channel art object for the
        // default channel association (always necessary)
        psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt =
            CHANNEL_ART_hCreate(
                (SMS_OBJECT)psObj->psArtOwner,
                &psObj->pacChannelArtServiceFilePath[0] );

        if (psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt
                == CHANNEL_ART_INVALID_OBJECT)
        {
            break;
        }

        bSuccess = TRUE;
    } while (FALSE);

    if (bSuccess == FALSE)
    {
        // Make sure we've cleaned up
        if (psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt !=
                CHANNEL_ART_INVALID_OBJECT)
        {
            CHANNEL_ART_vDestroy(
                psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt);
            psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt =
                CHANNEL_ART_INVALID_OBJECT;
        }

        if (psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt !=
                CHANNEL_ART_INVALID_OBJECT)
        {
            CHANNEL_ART_vDestroy(
                psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt);
            psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt =
                CHANNEL_ART_INVALID_OBJECT;
        }
    }

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return bSuccess;
}

/*****************************************************************************
*
*   vHandleDecoderSubscribed
*
*   This function is called when a decoder subscribes to this service
*   after it has already started.  This information is provided to give
*   us a chance to handle any late-comers.
*
*****************************************************************************/
static void vHandleDecoderSubscribed (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bUpdateSent;

    // Send an update to the subscribed decoder now
    bUpdateSent = bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_START,
        hDecoder );

    // Did the update go out correctly?
    if ( TRUE == bUpdateSent )
    {
        // Yes, tell the DSM that this decoder is now subscribed
        DATASERVICE_IMPL_vDecoderSubscribed(
            (DATASERVICE_IMPL_HDL)psObj, hDecoder);
    }

    return;
}

/*****************************************************************************
*
*   bHandleMessageReception
*
*   This function is called when a new message is received by the channel
*   art service.
*
*****************************************************************************/
static BOOLEAN bHandleMessageReception (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk = TRUE;

    // Ensure the payload handle is valid.
    // If it isn't, whatevs, we'll just
    // ignore it.
    if ( hPayload == OSAL_INVALID_BUFFER_HDL )
    {
        puts(CHANNEL_ART_MGR_OBJECT_NAME": Empty payload");
    }
    else
    {
        // Process the channel art message and update the system; we give
        // the message processor a pointer it can update; if it is set to
        // NULL, the message handler has taken ownership of the  payload.
        bOk = bProcessMessage( psObj, &hPayload );

        // The message was successfully processed --
        // issue the event list that we generated
        if ( bOk == TRUE )
        {
            puts(CHANNEL_ART_MGR_OBJECT_NAME": Completed Processing Message");
        }
    }

    // Free the payload (assuming we still own it)
    if ( hPayload != OSAL_INVALID_BUFFER_HDL )
    {
        DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    }

    return bOk;
}

/*****************************************************************************
*
*   bProductFSM
*
*   This function implements a finite state machine for art
*   product transitions.
*
*****************************************************************************/
static BOOLEAN bProductFSM (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_STATE_ENUM eOldState,
    DATA_PRODUCT_STATE_ENUM eNewState,
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bStateChanged = FALSE,
            bActionNeeded = FALSE;

    // Make sure the state actually changed; if not, there's
    // nothing for us to do.
    if ( eOldState == eNewState )
    {
        return bStateChanged;
    }

    // On the other hand, if something changed, see if we need to act
    // on the state transition.
    switch (eOldState)
    {
        case DATA_PRODUCT_STATE_DISABLED:
        {
            switch(eNewState)
            {
                case DATA_PRODUCT_STATE_INITIAL:
                {
                    bActionNeeded = TRUE;
                }

                // Transitions that don't matter; note: intentional
                // fall-through ahead.
                case DATA_PRODUCT_STATE_READY:
                case DATA_PRODUCT_STATE_UNSUBSCRIBED:
                case DATA_PRODUCT_STATE_ERROR:

                default:
                    break;
            }
        }
        break;

        case DATA_PRODUCT_STATE_INITIAL:
        {
            switch (eNewState)
            {
                // Call a state transition function; note: intentional
                // fall-through ahead.
                case DATA_PRODUCT_STATE_READY:
                case DATA_PRODUCT_STATE_UNSUBSCRIBED:
                case DATA_PRODUCT_STATE_DISABLED:
                case DATA_PRODUCT_STATE_ERROR:
                {
                    bActionNeeded = TRUE;
                }
                break;

                // Transitions that don't matter
                default:
                    break;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            switch (eNewState)
            {
                // Call a state transition function; note: intentional
                // fall-through ahead.
                case DATA_PRODUCT_STATE_READY:
                case DATA_PRODUCT_STATE_ERROR:
                case DATA_PRODUCT_STATE_DISABLED:
                {
                    bActionNeeded = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATA_PRODUCT_STATE_INITIAL:

                default:
                    break;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            switch (eNewState)
            {
                // Call a state transition function; note: intentional
                // fall-through ahead.
                case DATA_PRODUCT_STATE_DISABLED:
                case DATA_PRODUCT_STATE_ERROR:
                case DATA_PRODUCT_STATE_UNSUBSCRIBED:
                {
                    bActionNeeded = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATA_PRODUCT_STATE_INITIAL:

                default:
                    break;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            switch (eNewState)
            {
                // Call a state transition function; note: intentional
                // fall-through ahead.
                case DATA_PRODUCT_STATE_READY:
                case DATA_PRODUCT_STATE_UNSUBSCRIBED:
                case DATA_PRODUCT_STATE_DISABLED:
                {
                    bActionNeeded = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATA_PRODUCT_STATE_INITIAL:

                default:
                    break;
            }
        }
        break;

        default:
            break;
    }

    // If we need to take action, do so here
    if ( TRUE == bActionNeeded )
    {
        ART_PRODUCT_STATE_HANDLERS const *psHandlers = NULL;

        if ( eType == DATA_PRODUCT_TYPE_ALBUM_ART )
        {
            psHandlers = &GsAlbumStateHandlers;
        }
        else if ( eType == DATA_PRODUCT_TYPE_CHANNEL_ART )
        {
            psHandlers = &GsChanStateHandlers;
        }

        if (psHandlers != NULL)
        {
            switch ( eNewState )
            {
                case DATA_PRODUCT_STATE_READY:
                {
                    bStateChanged = psHandlers->bReadyHandler( psObj );
                }
                break;

                case DATA_PRODUCT_STATE_DISABLED:
                {
                    bStateChanged = psHandlers->bDisabledHandler( psObj );
                }
                break;

                case DATA_PRODUCT_STATE_ERROR:
                {
                    bStateChanged = psHandlers->bErrorHandler( psObj );
                }
                break;

                case DATA_PRODUCT_STATE_INITIAL:
                {
                    bStateChanged = psHandlers->bInitHandler( psObj );
                }
                break;

                case DATA_PRODUCT_STATE_UNSUBSCRIBED:
                {
                    bStateChanged = TRUE;
                }
                break;
            }
        }
    }

    return bStateChanged;
}

/*****************************************************************************
*
*   vAlbumCleanUp
*
*****************************************************************************/
static void vAlbumCleanUp (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bFullCleanUp
        )
{
    if (bFullCleanUp == FALSE)
    {
        // Just in case clean at the backups
        if (psArtOwner->sBackup.hAlbumAssociations != OSAL_INVALID_OBJECT_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": backup is busy. Clean it up");
            vAlbumCleanUp(psArtOwner, TRUE);
        }
        psArtOwner->sBackup.hAlbumAssociations = psArtOwner->hAlbumAssociations;
        psArtOwner->hAlbumAssociations = OSAL_INVALID_OBJECT_HDL;
    }
    else if (psArtOwner->sBackup.hAlbumAssociations != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Clear linked list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psArtOwner->sBackup.hAlbumAssociations,
            vAlbumReleaseAssociations );

        // Only finish clean up if that succeeded.  This allows
        // us to more easily track this kind of issue
        if (OSAL_SUCCESS  == eReturnCode)
        {
            // Delete linked list
            OSAL.eLinkedListDelete(
                psArtOwner->sBackup.hAlbumAssociations );

            // Clear the handle
            psArtOwner->sBackup.hAlbumAssociations =
                OSAL_INVALID_OBJECT_HDL;
        }
    }
    else
    {
        printf(CHANNEL_ART_MGR_OBJECT_NAME": nothing to cleanup for Album");
    }

    return;
}

/*****************************************************************************
*
*   bAlbumHandleInitState
*
*   This function is called when the album art product is transitioning to
*   the INIT state. We prep for going to the start state by creating
*   any data structures that will be needed later.
*
*****************************************************************************/
static BOOLEAN bAlbumHandleInitState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Name the channel art object linked list
    snprintf( &acName[0], sizeof(acName),
                CHANNEL_ART_MGR_OBJECT_NAME":AlbumAssociations" );

    // Build the album art association list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->psArtOwner->hAlbumAssociations,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16AlbumCompareAssociations,
        OSAL_LL_OPTION_BINARY_SEARCH );

    if( OSAL_SUCCESS != eReturnCode )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Unable to create Album Art Association List" );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bAlbumHandleReadyState
*
*   This function is called when the album art product is transitioning to
*   the ready state. Upon the transition to the state, we create our Album
*   Art data interface, initialize the DB, and send an update to subscribed
*   decoders to start the process of updating their album art.
*
*****************************************************************************/
static BOOLEAN bAlbumHandleReadyState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOTADataAvailable;
    BOOLEAN bLocked;
    BOOLEAN bOk = FALSE;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;

    // Are OTA updates available to us?
    bOTADataAvailable =
        DATASERVICE_IMPL_bOTADataAvailable((DATASERVICE_IMPL_HDL)psObj);

    do
    {
        // Now block the art owner so we can look for the right
        // association without the list changing underneath us.
        bLocked = SMSO_bLock( (SMS_OBJECT)psObj->psArtOwner,
                OSAL_OBJ_TIMEOUT_INFINITE );

        if ( bLocked == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME": Cannot lock art owner.");
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // Check if Album Art resources are already initialized
        if ( TRUE == psObj->psArtOwner->bAlbumArtInitialized )
        {
            // OK. Already initialized.
            bOk = TRUE;
            break;
        }

        // Try to clean up backups
        vAlbumCleanUp(psObj->psArtOwner, TRUE);

        // Create file paths
        bOk = bBuildAlbumArtFilePaths( psObj );
        if (FALSE == bOk)
        {
            break;
        }

        // Each break below will indicate error by default
        bOk = FALSE;

        // If updates are available, then we'll have to initialize the album
        // art interface to handle the incoming message stream
        if ( ( TRUE == bOTADataAvailable )&&
             ( ALBUM_ART_INTERFACE_INVALID_OBJECT == psObj->hAlbInterface ) )
        {
            // Initialize the Album Art interface
            psObj->hAlbInterface =
                GsAlbumArtIntf.hInit(
                    (CHANNEL_ART_SERVICE_OBJECT)psObj,
                    DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

            if ( ALBUM_ART_INTERFACE_INVALID_OBJECT == psObj->hAlbInterface )
            {
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }
        }

        if ( (psObj->tSelectedImageTypes & CHANNEL_ART_AVAILABLE_IMAGE_ALBUM)
                == CHANNEL_ART_AVAILABLE_IMAGE_NONE )
        {
            // This image type was not requested during startup, so considering
            // as error
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        if ( SQL_INTERFACE_INVALID_OBJECT == psObj->hAlbumPerSQLConnection )
        {
            // Connect to the album art persistent database
            psObj->hAlbumPerSQLConnection =
                DB_UTIL_hConnectToPersistent(
                    &psObj->pacFullyQualifiedAlbumPerDatabaseFilePath[0],
                    (DB_UTIL_CREATE_DB_HANDLER)bAlbumCreatePersistDBTables,
                    psObj,
                    (DB_UTIL_CHECK_VERSION_HANDLER)bAlbumInspectDBInfo,
                    NULL,
                    &eErrorCode);

            if ( SQL_INTERFACE_INVALID_OBJECT == psObj->hAlbumPerSQLConnection )
            {
                // Make sure that file doesn't exists at this point to let
                // service recreate it from scratch during next startup
                remove(&psObj->pacFullyQualifiedAlbumPerDatabaseFilePath[0]);
                break;
            }

            // Inspect the database info table, retrieve some values
            bOk = bAlbumInspectDBInfo( psObj->hAlbumPerSQLConnection );

            if ( bOk == FALSE )
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH;
                break;
            }

            // Process the database associations and
            // generate all needed objects
            bOk = bAlbumProcessDatabase( psObj );

            if ( bOk == FALSE )
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
                break;
            }

            // Prepare statements
            psObj->hAlbumSelectImageAttrsStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                psObj->hAlbumPerSQLConnection,
                AA_SELECT_IMAGE_ATTRIBUTES);

            if (psObj->hAlbumSelectImageAttrsStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
                break;
            }
        }

        // Album Art resources are now initialized
        psObj->psArtOwner->bAlbumArtInitialized = TRUE;

        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
        bLocked = FALSE;

        // Register for our cleanup event (if we haven't created one
        // already; if art has been brought up and down before, we'll
        // just re-use the same event handle (as it's actually owned and
        // destroyed by the "umbrella" art service.
        if ( DATASERVICE_TIMED_EVENT_INVALID_HDL == psObj->hCleanupEvent )
        {
            psObj->hCleanupEvent = DATASERVICE_IMPL_hRegisterTimedEvent(
                (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

            if ( DATASERVICE_TIMED_EVENT_INVALID_HDL ==  psObj->hCleanupEvent )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME
                        ": Cannot create cleanup timer.");
                bOk = FALSE;
                eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }
        }

        // Send an update to the subscribed decoders now
        bOk = bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_START,
                DECODER_INVALID_OBJECT );

        if ( FALSE == bOk )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME
                    ": Cannot post Album Art start event.");
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
            break;
        }

        // All done!
        puts(CHANNEL_ART_MGR_OBJECT_NAME": Album Art Ready");

    } while ( FALSE );

    // Make sure we unlock the art owner, even if something goes wrong
    if ( TRUE == bLocked )
    {
        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
    }

    if ( FALSE == bOk )
    {
        vSetError(psObj, eErrorCode);
    }

    return bOk;
}

/*****************************************************************************
*
*   bAlbumCreatePersistDBTables
*
*****************************************************************************/
static BOOLEAN bAlbumCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hPerSQLConnection,
    void *pvArg
        )
{
    BOOLEAN bSuccess = FALSE, bTransactionStarted = FALSE,
            bAttached = FALSE;
    char acBuffer[CHANNEL_ART_MAX_SQL_STRING_LENGTH];
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj = (CHANNEL_ART_MGR_OBJECT_STRUCT*)pvArg;

    do
    {
        DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_CODE_NONE;

        bAttached = SQL_INTERFACE.bAttachDBToConnection( hPerSQLConnection,
                        &psObj->pacFullyQualifiedAlbumRefDatabaseFilePath[0],
                        ALBUM_ART_PER_DB_ATTACH_NAME, &eErrorCode );

        if ( FALSE == bAttached )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": failed to attach reference DB");
            vSetError(psObj, eErrorCode);
            break;
        }

        bTransactionStarted = SQL_INTERFACE.bStartTransaction(hPerSQLConnection);

        if ( FALSE == bTransactionStarted )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": failed to start transaction to recreate persistent storage");
            break;
        }

        // Create the association table

        bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection,
                AA_CREATE_ASSOCIATION_TABLE);

        if ( FALSE == bSuccess )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_ASSOC_TABLE_NAME
                ": failed to create association table in persistent storage");
            break;
        }

        bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection,
                AA_CREATE_ATTRIBUTE_TABLE);

        if ( FALSE == bSuccess )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_ASSOC_TABLE_NAME
                ": failed to create attribute table in persistent storage");
            break;
        }

        // Copy the association table

        bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection,
                    AA_COPY_ASSOC );

        if ( FALSE == bSuccess )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_ASSOC_TABLE_NAME
                ": failed to copy association table in persistent storage");
            break;
        }

        // Copy the attribute table as well

        bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection,
                    AA_COPY_ATTRIB );

        if ( FALSE == bSuccess )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                IMAGE_ATTRIBUTE_TABLE_NAME
                ": failed to copy attribute table in persistent storage");
            break;
        }

        // Create version table

        bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection,
                       AA_CREATE_VERSION_TABLE);

       if ( FALSE == bSuccess )
       {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               CHANNEL_ART_MGR_OBJECT_NAME
               ": failed to create version table in persistent storage");
           break;
       }

       // Build the version string

       snprintf( &acBuffer[0], sizeof(acBuffer), AA_INSERT_VERSION_ROW,
           ALBUM_ART_DATABASE_FILE_VERSION, GsAlbumArtIntf.tDSI );

       bSuccess = SQL_INTERFACE.bExecuteCommand(hPerSQLConnection, &acBuffer[0]);

       if ( FALSE == bSuccess )
       {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               CHANNEL_ART_MGR_OBJECT_NAME
               ": failed to set db version in persistent storage");
       }
    }
    while (FALSE);

    // End the transaction if necessary
    if ( TRUE == bTransactionStarted )
    {
        BOOLEAN bTransactionEnded;

        bTransactionEnded = SQL_INTERFACE.bEndTransaction(hPerSQLConnection);

        if ( FALSE == bTransactionEnded )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME
                ": failed to end transaction");

            bSuccess = FALSE;
        }
    }

    // Once the transaction (if any) is completed, we're safe to
    // detach the reference table from the persistent table (again,
    // if necessary)
    if ( TRUE == bAttached )
    {
        SQL_INTERFACE.vDetachDBFromConnection(hPerSQLConnection,
                ALBUM_ART_PER_DB_ATTACH_NAME);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bAlbumHandleDisabledState
*
*   This function is called when the album art product is transitioning to
*   the disabled state. Upon the transition to the state, we clean up our
*   objects, close out the DB link, and send an update to subscribed
*   decoders to NULL out their album art.
*
*****************************************************************************/
static BOOLEAN bAlbumHandleDisabledState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked, bOk;

    // Lock the channel art owner object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if ( FALSE == bLocked )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": Unable to lock art owner");
        return FALSE;
    }

    // Do the preliminary album's stuff cleaning up.
    vAlbumCleanUp(psObj->psArtOwner, FALSE);

    // Destroy the album art interface object if it exists
    if ( ALBUM_ART_INTERFACE_INVALID_OBJECT != psObj->hAlbInterface )
    {
        GsAlbumArtIntf.vUnInit( psObj->hAlbInterface );
        psObj->hAlbInterface = ALBUM_ART_INTERFACE_INVALID_OBJECT;
    }

    if ( SQL_INTERFACE_INVALID_OBJECT != psObj->hAlbumPerSQLConnection )
    {

        // Destroy prepared statements
        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hAlbumPerSQLConnection,
            psObj->hAlbumSelectImageAttrsStmt);
        
        psObj->hAlbumSelectImageAttrsStmt = SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect(psObj->hAlbumPerSQLConnection);
        psObj->hAlbumPerSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Now Album Art resources are uninitialized
    psObj->psArtOwner->bAlbumArtInitialized = FALSE;

    // Make sure let go of the art owner at this point
    SMSO_vUnlock( (SMS_OBJECT)psObj->psArtOwner );

    // We now have the album art clients remove their art (which will
    // be updated to NULL, as our product is no longer ready.)
    bOk = bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_STOP,
                DECODER_INVALID_OBJECT );

    if ( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": Cannot post album art stop"
                    "event.");
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bAlbumHandleErrorState
*
*   This function is called when the album art product enters the error
*   state (which means something has gone seriously wrong.) We stop our
*   timed event and print an error (which is about all we can do.)
*
*****************************************************************************/
static BOOLEAN bAlbumHandleErrorState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    // Stop the timed event (this will still return TRUE even if
    // the timed event wasn't active.)
    bOk = DATASERVICE_IMPL_bStopTimedEvent( psObj->hCleanupEvent );

    if ( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Unable to stop the album art timed event!" );
    }

    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        CHANNEL_ART_MGR_OBJECT_NAME
        ": Album Art Product Transition to ERROR state!" );

    return TRUE;
}

/*****************************************************************************
*
*   vChanCleanUp
*
*****************************************************************************/
static void vChanCleanUp (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bFullCleanUp
        )
{
    if (bFullCleanUp == FALSE)
    {
        // Clean up all backups
        vChanCleanUp(psArtOwner, TRUE);

        psArtOwner->sBackup.hChanAssociations = psArtOwner->hChanAssociations;
        psArtOwner->hChanAssociations = OSAL_INVALID_OBJECT_HDL;
        psArtOwner->sBackup.hLineBitmaps = psArtOwner->hLineBitmaps;
        psArtOwner->hLineBitmaps = OSAL_INVALID_OBJECT_HDL;
        psArtOwner->sBackup.sDefaultCategoryAssoc = psArtOwner->sDefaultCategoryAssoc;
        psArtOwner->sDefaultCategoryAssoc.hChannelArt = CHANNEL_ART_INVALID_OBJECT;
        psArtOwner->sBackup.sDefaultChannelAssoc = psArtOwner->sDefaultChannelAssoc;
        psArtOwner->sDefaultChannelAssoc.hChannelArt = CHANNEL_ART_INVALID_OBJECT;
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

       // Destroy association list if one exists
       if ( OSAL_INVALID_OBJECT_HDL != psArtOwner->sBackup.hChanAssociations )
       {
           // Clear linked list
           eReturnCode = OSAL.eLinkedListRemoveAll(
               psArtOwner->sBackup.hChanAssociations,
               vChanReleaseAssociations );

           // Only finish clean up if that succeeded.  This allows
           // us to more easily track this kind of issue
           if ( OSAL_SUCCESS == eReturnCode )
           {
               // Delete linked list
               eReturnCode = OSAL.eLinkedListDelete(
                   psArtOwner->sBackup.hChanAssociations );

               if ( OSAL_SUCCESS != eReturnCode )
               {
                   SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                       CHANNEL_ART_MGR_OBJECT_NAME
                       ": Cannot destroy channel associations list (%s)",
                       OSAL.pacGetReturnCodeName(eReturnCode));
               }

               // Clear the handle
               psArtOwner->sBackup.hChanAssociations =
                   OSAL_INVALID_OBJECT_HDL;
           }
       }

       // Destroy line bitmap list if it exists
       if ( OSAL_INVALID_OBJECT_HDL != psArtOwner->sBackup.hLineBitmaps )
       {
           // Clear the linked list
           eReturnCode = OSAL.eLinkedListRemoveAll(
               psArtOwner->sBackup.hLineBitmaps,
               vReleaseLineBitmaps );

           // Only finish clean up if that succeeded.  This allows
           // us to more easily track this kind of issue
           if ( OSAL_SUCCESS == eReturnCode )
           {
               // Delete linked list
               eReturnCode = OSAL.eLinkedListDelete(
                   psArtOwner->sBackup.hLineBitmaps );

               if ( OSAL_SUCCESS != eReturnCode )
               {
                   SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                       CHANNEL_ART_MGR_OBJECT_NAME
                       ": Cannot destroy line bitmaps list (%s)",
                       OSAL.pacGetReturnCodeName(eReturnCode));
               }

               // Clear the handle
               psArtOwner->sBackup.hLineBitmaps = OSAL_INVALID_OBJECT_HDL;
           }
       }

       // Destroy the default category association, if necessary
       if ( psArtOwner->sBackup.sDefaultCategoryAssoc.hChannelArt
               != CHANNEL_ART_INVALID_OBJECT )
       {
           CHANNEL_ART_vDestroy(
               psArtOwner->sBackup.sDefaultCategoryAssoc.hChannelArt);
           psArtOwner->sBackup.sDefaultCategoryAssoc.hChannelArt =
               CHANNEL_ART_INVALID_OBJECT;
       }

       // Destroy the default channel association, if necessary
       if ( psArtOwner->sBackup.sDefaultChannelAssoc.hChannelArt
               != CHANNEL_ART_INVALID_OBJECT )
       {
           CHANNEL_ART_vDestroy(
               psArtOwner->sBackup.sDefaultChannelAssoc.hChannelArt);
           psArtOwner->sBackup.sDefaultChannelAssoc.hChannelArt =
               CHANNEL_ART_INVALID_OBJECT;
       }
    }

    return;
}

/*****************************************************************************
*
*   bChanHandleInitState
*
*   This function is called when the album art product is transitioning to
*   the INIT state. We prep for going to the start state by creating
*   any data structures that will be needed later.
*
*****************************************************************************/
static BOOLEAN bChanHandleInitState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Name the channel art object linked list
    snprintf( &acName[0], sizeof(acName),
              CHANNEL_ART_MGR_OBJECT_NAME":ChannelAssociations" );

    // Build the channel art association list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->psArtOwner->hChanAssociations,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16ChanCompareAssociations,
        OSAL_LL_OPTION_BINARY_SEARCH
            );

    if ( OSAL_SUCCESS != eReturnCode )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Unable to create Channel Art Association List" );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bChanHandleReadyState
*
*   This function is called by the product state machine when the channel
*   art product is about to go ready, and the product will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bChanHandleReadyState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bLocked;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode =
        DATASERVICE_ERROR_CODE_GENERAL;

    do
    {
        BOOLEAN bChanAssocsAvailable = FALSE,
                bOverlayChanAssocsAvailable = FALSE,
                bCatAssocsAvailable = FALSE,
                bOTADataAvailable;

        // Are OTA updates available to us?
        bOTADataAvailable =
            DATASERVICE_IMPL_bOTADataAvailable((DATASERVICE_IMPL_HDL)psObj);

        // Now block the art owner so we're not rudely interrupted while
        // building our associations from the DB
        bLocked = SMSO_bLock( (SMS_OBJECT)psObj->psArtOwner,
                OSAL_OBJ_TIMEOUT_INFINITE );

        if ( FALSE == bLocked )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME": Cannot lock art owner.");
            break;
        }

        // Check if Channel Art resources are already initialized
        if ( TRUE == psObj->psArtOwner->bChannelArtInitialized )
        {
            // OK. Already initialized.
            bOk = TRUE;
            break;
        }

        // Try to clean up backups
        vChanCleanUp(psObj->psArtOwner, TRUE);

        // Create file paths
        bOk = bBuildChannelArtFilePaths( psObj );
        if (FALSE == bOk)
        {
            break;
        }

        // If updates are available, then we'll have to
        // initialize the channel art interface to handle
        // the incoming message stream
        if (bOTADataAvailable == TRUE &&
            psObj->hChanInterface == CHANNEL_ART_INTERFACE_INVALID_OBJECT)
        {
            // Initialize the Channel Art interface
            psObj->hChanInterface =
                GsChannelArtIntf.hInit(
                    (CHANNEL_ART_SERVICE_OBJECT)psObj,
                    DATASERVICE_IMPL_hSMSObj((DATASERVICE_IMPL_HDL)psObj));

            if (psObj->hChanInterface == CHANNEL_ART_INTERFACE_INVALID_OBJECT)
            {
                bOk = FALSE;
                break;
            }
        }

        if (psObj->hChanSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
        {
            // Now, initialize our default associations
            bOk = bInitDefaultAssociations(psObj);
            if (bOk == FALSE)
            {
                break;
            }

            // Each break below will indicate error by default
            bOk = FALSE;

            // Connect to the channel art database
            psObj->hChanSQLConnection =
                SQL_INTERFACE.hConnect(
                    &psObj->pacFullyQualifiedChanDatabaseFilePath[0],
                    FALSE,
                    &eErrorCode
                        );

            if (psObj->hChanSQLConnection == SQL_INTERFACE_INVALID_OBJECT)
            {
                break;
            }

            // Resetting to General error code
            eErrorCode = DATASERVICE_ERROR_CODE_GENERAL;

            // Prepare statements
            psObj->hSelectImageAttrsStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                    psObj->hChanSQLConnection,
                    CA_SELECT_IMAGE_ATTRIBUTES);

            if (psObj->hSelectImageAttrsStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                break;
            }

            psObj->hUpdateAssocRowStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                    psObj->hChanSQLConnection,
                    CA_UPDATE_ASSOCIATION_ROW);

            if (psObj->hUpdateAssocRowStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                break;
            }

            psObj->hInsertAssocRowStmt =
                SQL_INTERFACE.hCreatePreparedStatement(
                    psObj->hChanSQLConnection,
                    CA_INSERT_ASSOCIATION_ROW);

            if (psObj->hInsertAssocRowStmt ==
                SQL_PREPARED_STATEMENT_INVALID_HANDLE)
            {
                break;
            }

            // Inspect the database info table, retrieve some values
            bOk = bChanInspectDBInfo( psObj->hChanSQLConnection, &bChanAssocsAvailable,
                &bOverlayChanAssocsAvailable, &bCatAssocsAvailable );

            if (bOk == FALSE)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH;
                break;
            }

            // Process the database associations and
            // generate all needed objects
            bOk = bChanProcessArtDatabase( psObj, bChanAssocsAvailable,
                    bOverlayChanAssocsAvailable, bCatAssocsAvailable );

            if (bOk == FALSE)
            {
                eErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
                break;
            }

            // Clear the updated flags now since they're not
            // useful just yet
            bOk = bChanClearAllUpdatedFlags(psObj);
            if (bOk == FALSE)
            {
                break;
            }
        }

        // Channel Art resources are now initialized
        psObj->psArtOwner->bChannelArtInitialized = TRUE;

        // Now unlock before we send our update
        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
        bLocked = FALSE;

        // Send an update to the subscribed decoders now
        bOk = bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_CHAN_ART_START,
                DECODER_INVALID_OBJECT );

        if ( FALSE == bOk )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Cannot post Channel Art start event.");
            break;
        }

        // All done!
        puts(CHANNEL_ART_MGR_OBJECT_NAME": Channel Art Ready");
    } while (FALSE);

    // Make sure we unlock the art owner, even if something goes wrong
    if ( TRUE == bLocked )
    {
        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
    }

    if ( FALSE == bOk )
    {
        vSetError(psObj, eErrorCode);
    }

    return bOk;
}

/*****************************************************************************
*
*   bChanHandleDisabledState
*
*****************************************************************************/
static BOOLEAN bChanHandleDisabledState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk, bLocked;

    // Lock the channel art owner objects
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if ( FALSE == bLocked )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": Unable to lock art owner");
        return FALSE;
    }

    // Destroy the channel art interface object if it exists
    if ( CHANNEL_ART_INTERFACE_INVALID_OBJECT != psObj->hChanInterface )
    {
        GsChannelArtIntf.vUnInit( psObj->hChanInterface );
        psObj->hChanInterface = CHANNEL_ART_INTERFACE_INVALID_OBJECT;
    }

    // Do the preliminary chan's stuff cleaning up.
    vChanCleanUp(psObj->psArtOwner, FALSE);

    // Disconnect from the database application
    if ( psObj->hChanSQLConnection
            != SQL_INTERFACE_INVALID_OBJECT )
    {
        // Destroy prepared statements
        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hChanSQLConnection,
            psObj->hSelectImageAttrsStmt);

        psObj->hSelectImageAttrsStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hChanSQLConnection,
            psObj->hUpdateAssocRowStmt);

        psObj->hUpdateAssocRowStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDestroyPreparedStatement(
            psObj->hChanSQLConnection,
            psObj->hInsertAssocRowStmt);

        psObj->hInsertAssocRowStmt =
            SQL_PREPARED_STATEMENT_INVALID_HANDLE;

        SQL_INTERFACE.vDisconnect( psObj->hChanSQLConnection );
        psObj->hChanSQLConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    // Now Channel Art resources are uninitialized
    psObj->psArtOwner->bChannelArtInitialized = FALSE;

    // Make sure to let go of the art owner at this point
    SMSO_vUnlock( (SMS_OBJECT)psObj->psArtOwner );

    // Have the channel art clients remove their art
    // (which will be updated to NULL, as our product is no longer
    // ready.)
    bOk = bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_CHAN_ART_STOP,
                DECODER_INVALID_OBJECT );

    if( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Cannot post Channel Art stop event.");
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL );
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bChanHandleErrorState
*
*   This function is called when the album art product enters the error
*   state (which means something has gone seriously wrong.) We print an error
*   (as there's not much else we can do.)
*
*****************************************************************************/
static BOOLEAN bChanHandleErrorState(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        CHANNEL_ART_MGR_OBJECT_NAME
        ": Channel Art Product Transition to ERROR state!" );

    return TRUE;
}

/*****************************************************************************
*
*   bGetDSIInfo
*
*   A callback used by the product interface to populate information about
*   art service products, including DSI and required buffer size.
*
*****************************************************************************/
static BOOLEAN bGetDSIInfo (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        )
{
    switch ( eType )
    {
        case DATA_PRODUCT_TYPE_ALBUM_ART:
        {
            // Otherwise, just set up our DSI Info struct appropriately
            psDSIInfo->tDSI = GsAlbumArtIntf.tDSI;
            psDSIInfo->tSuggestedOTABufferByteSize = GsAlbumArtIntf.tOTABufferByteSize;
            psDSIInfo->bEnableAllDMIs = TRUE;
        }
        break;

        case  DATA_PRODUCT_TYPE_CHANNEL_ART:
        {
            // Otherwise, just set up our DSI Info struct appropriately
            psDSIInfo->tDSI = GsChannelArtIntf.tDSI;
            psDSIInfo->tSuggestedOTABufferByteSize = GsChannelArtIntf.tOTABufferByteSize;
            psDSIInfo->bEnableAllDMIs = TRUE;
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "Unknown product %u provided to bGetDSIInfo", eType);
            return FALSE;
        }
        break;
    }

    return TRUE;
}

/*****************************************************************************
*
*   eGetNextState
*
*   A callback used by the product framework translate DSI states into
*   product states.
*
*****************************************************************************/
static DATA_PRODUCT_STATE_ENUM eGetNextState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        )
{
    DATA_PRODUCT_STATE_ENUM eProductState;

    switch( eDSIState )
    {
        case DATASERVICE_STATE_INITIAL:
        {
            eProductState = DATA_PRODUCT_STATE_INITIAL;
        }
        break;

        case DATASERVICE_STATE_READY:
        {
            eProductState = DATA_PRODUCT_STATE_READY;
        }
        break;

        case DATASERVICE_STATE_STOPPED:
        {
            eProductState = DATA_PRODUCT_STATE_DISABLED;
        }
        break;

        case DATASERVICE_STATE_UNSUBSCRIBED:
        {
            eProductState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
        }
        break;

        // As art is a free service, there's quite a
        // few DSI states that we should never get. If we do,
        // then flag that as an error.
        case DATASERVICE_STATE_UNAVAILABLE:
        case DATASERVICE_STATE_POI_UPDATES_ONLY:
        case DATASERVICE_STATE_ERROR:
        case DATASERVICE_STATE_INVALID:
        default:
        {
            eProductState = DATA_PRODUCT_STATE_ERROR;
        }
    }

    return eProductState;
}

/*****************************************************************************
*
*   bBuildCommonFilePaths
*
*   This function generates file paths needed by both Channel Art  and Album
*   Art data services based upon a base file path.
*
*****************************************************************************/
static BOOLEAN bBuildCommonFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    size_t tChannelArtPathLen, tIndex;
    BOOLEAN bInsertServiceDir = TRUE;

    // Initialize the base path to whatever
    // the manager is currently configured to use
    const char *pacBasePath = psObj->pacBasePath;

    // Do we already have initialized service path?
    if (NULL != psObj->pacChannelArtServiceFilePath)
    {
        // Nothing to do any more
        return TRUE;
    }

    // Do we have a usable base path?
    if (pacBasePath != NULL)
    {
        bInsertServiceDir = FALSE;
    }
    else
    {
        // No, just use the default SMS path
        pacBasePath = SMS_pacGetPath();
    }

    // Verify we have something to use now
    if (NULL == pacBasePath)
    {
        return FALSE;
    }

    // Base path len is +1 for delimiter
    tChannelArtPathLen = strlen(pacBasePath) + 1;

    // Do we need to insert the service directory name here?
    if (TRUE == bInsertServiceDir)
    {
        // Yeah, add it into the equation now
        tChannelArtPathLen += sizeof(CHANNEL_ART_DIRECTORY_NAME) + 1;
    }

    // Now we know the size of the path, so we can now
    // allocate the proper amount of memory

    psObj->pacChannelArtServiceFilePath =
        (char *) SMSO_hCreate(
        CHANNEL_ART_MGR_OBJECT_NAME":FilePath",
        tChannelArtPathLen + 1, // Trailing NUL
        SMS_INVALID_OBJECT,
        FALSE );

    // Ensure allocation succeeded
    if (psObj->pacChannelArtServiceFilePath == NULL)
    {
        return FALSE;
    }

    // Begin writing the path now

    // Just the base path and the delimiter
    tIndex = snprintf( (&psObj->pacChannelArtServiceFilePath[0]),
        tChannelArtPathLen + 1, // Trailing NULL
        "%s/", pacBasePath);

    if (TRUE == bInsertServiceDir)
    {
        // Now, insert the service directory
        snprintf( (&psObj->pacChannelArtServiceFilePath[tIndex]),
            sizeof(CHANNEL_ART_DIRECTORY_NAME) + 1 + 1, // Delimiter and NULL
            CHANNEL_ART_DIRECTORY_NAME"/" );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bBuildChannelArtFilePaths
*
*   This function generates the channel art file path based upon
*   a base file path.  This information is used in order to locate the
*   channel art database and image files.
*
*****************************************************************************/
static BOOLEAN bBuildChannelArtFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE;

    if (NULL == psObj->pacChannelArtServiceFilePath)
    {
        // Error! Common part is not initialized.
        return FALSE;
    }

    if (NULL == psObj->pacFullyQualifiedChanDatabaseFilePath)
    {
        // Construct the channel art database path now
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacBasePath,
            CHANNEL_ART_DIRECTORY_NAME,
            CHANNEL_ART_DATABASE_FILENAME,
            &psObj->pacFullyQualifiedChanDatabaseFilePath);
    }

    return bOk;
}

/*****************************************************************************
*
*   bBuildAlbumArtFilePaths
*
*   This function generates the album art file path based upon a base file 
*   path. This information is used in order to locate the album art database.
*   Album art images will be placed in the ephemeral path, which *may* be 
*   the same as the base file path depending on which options were passed 
*   to hStart.
*
*****************************************************************************/
static BOOLEAN bBuildAlbumArtFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE;

    if (NULL == psObj->pacChannelArtServiceFilePath)
    {
        // Error! Common part is not initialized.
        return FALSE;
    }

    // Check to see if the ephemeral path has been initialized (which will
    // have happened already if the user specified an ephemeral path).
    // If not, we'll just store our ephemeral images in the OSAL temp
    // directory
    if ( NULL == psObj->pacEphemeralPath )
    {
        BOOLEAN bDirPresent;
        size_t tTempPathSize, tEphemeralPathSize;
        UN8 un8Attrs;

        // Determine the size needed for the temp path
        tTempPathSize = OSAL.tFileSystemGetTempPath(0, NULL);

        // Create album art folder path, re-using tPathSize
        tEphemeralPathSize = tTempPathSize +
                        strlen(ALBUM_ART_DIRECTORY_NAME) +
                        3 /* 2 separators + trailing 0 */;

        psObj->pacEphemeralPath =
               (char *) SMSO_hCreate(
                    CHANNEL_ART_MGR_OBJECT_NAME":EphemeralPath",
                    tEphemeralPathSize, SMS_INVALID_OBJECT, FALSE);

        if ( NULL == psObj->pacEphemeralPath  )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "Could not create ephemeral path object!");
            return FALSE;
        }

        // Write out the ephemeral path to our str object; the +1s below
        // are for the NULL terminators
        OSAL.tFileSystemGetTempPath(tTempPathSize + 1,
            psObj->pacEphemeralPath );
        strncat(psObj->pacEphemeralPath, "/"ALBUM_ART_DIRECTORY_NAME"/",
            ( tEphemeralPathSize - tTempPathSize ));

        // Grab the attributes of the directory on the filesystem
        bDirPresent = OSAL.bFileSystemGetFileAttributes(
            psObj->pacEphemeralPath, &un8Attrs );

        // Now create the directory if needed.
        if ( FALSE == bDirPresent )
        {
            bDirPresent = OSAL.bFileSystemMakeDir( psObj->pacEphemeralPath );
        }

        // Now we *have* to have the directory, or something's wrong.
        if ( TRUE != bDirPresent )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": Call to "
                "OSAL.bFileSystemMakeDir failed." );
            
            bOk = FALSE;
        }
    }

    if ( (TRUE == bOk) &&
         (NULL == psObj->pacFullyQualifiedAlbumRefDatabaseFilePath) )
    {
        // Construct the Album Art reference database path
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacBasePath,
            CHANNEL_ART_DIRECTORY_NAME,
            ALBUM_ART_REF_DATABASE_FILENAME,
            &psObj->pacFullyQualifiedAlbumRefDatabaseFilePath );
    }

    if ( (TRUE == bOk) &&
         (NULL == psObj->pacFullyQualifiedAlbumPerDatabaseFilePath) )
    {
        // Construct the Album Art persistent database path
        bOk = DB_UTIL_bCreateFilePath(
            psObj->pacBasePath,
            CHANNEL_ART_DIRECTORY_NAME,
            ALBUM_ART_PER_DATABASE_FILENAME,
            &psObj->pacFullyQualifiedAlbumPerDatabaseFilePath );
    }
    
    if ( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
            "Could not construct Album Art DB path(s)" );

        // Cleaning up strings, if these were allocated
        if (psObj->pacEphemeralPath != NULL)
        {
            SMSO_vDestroy(
                (SMS_OBJECT)psObj->pacEphemeralPath );
            psObj->pacEphemeralPath = NULL;
        }

        if (psObj->pacFullyQualifiedAlbumRefDatabaseFilePath != NULL)
        {
            SMSO_vDestroy(
                (SMS_OBJECT)psObj->pacFullyQualifiedAlbumRefDatabaseFilePath );
            psObj->pacFullyQualifiedAlbumRefDatabaseFilePath = NULL;
        }

        if (psObj->pacFullyQualifiedAlbumPerDatabaseFilePath != NULL)
        {
            SMSO_vDestroy(
                (SMS_OBJECT)psObj->pacFullyQualifiedAlbumPerDatabaseFilePath );
            psObj->pacFullyQualifiedAlbumPerDatabaseFilePath = NULL;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   hChanGetArtForSourceAndType
*
*   This function provides the main functionality for locating a channel art
*   object file for either the CHANNEL object or the CATEGORY object.
*
*****************************************************************************/
static CHANNEL_ART_OBJECT hChanGetArtForSourceAndType (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16SourceId,
    BOOLEAN bCategory
        )
{
    BOOLEAN bLocked;
    CHANNEL_ART_OBJECT hChannelArt =
        CHANNEL_ART_INVALID_OBJECT;

    // Verify and lock art Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return CHANNEL_ART_INVALID_OBJECT;
    }

    do
    {
        CHANNEL_ART_ASSOC_STRUCT *psAssoc = NULL,
                                 *psDefaultAssoc = NULL;

        printf(CHANNEL_ART_MGR_OBJECT_NAME": Get Art for %s: %u",
            (bCategory == TRUE)?"Category":"Channel",
            un16SourceId);

        // Look for this association
        psAssoc = psChanGetArtAssoc(psObj->psArtOwner, FALSE,
                bCategory, un16SourceId);

        if (psAssoc != NULL)
        {
            // We found the association.
            puts(CHANNEL_ART_MGR_OBJECT_NAME": Art Found");

            // Extract the handle and provide it to the caller
            hChannelArt = psAssoc->hChannelArt;

            break;
        }

        puts(CHANNEL_ART_MGR_OBJECT_NAME": Using default");

        // Use the default assoc (category or service)
        if (bCategory == TRUE)
        {
            psDefaultAssoc = &psObj->psArtOwner->sDefaultCategoryAssoc;
        }
        else
        {
            psDefaultAssoc = &psObj->psArtOwner->sDefaultChannelAssoc;
        }

        if (psDefaultAssoc->hChannelArt == CHANNEL_ART_INVALID_OBJECT)
        {
            // We have nothing to work with here
            break;
        }

        // Add a new channel art association entry
        // to the association linked list
        psAssoc = psChanAddNewAssocEntry(
            psObj->psArtOwner,
            bCategory,
            un16SourceId );

        if (psAssoc == NULL)
        {
            break;
        }

        // Copy the default association's
        // channel art object to use for this source
        psAssoc->hChannelArt =
            CHANNEL_ART_hCopy(
                (SMS_OBJECT)psObj->psArtOwner,
                psDefaultAssoc->hChannelArt );

        if (psAssoc->hChannelArt == CHANNEL_ART_INVALID_OBJECT)
        {
            // Remove the linked list entry
            OSAL.eLinkedListRemove( psAssoc->hEntry );
            psAssoc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Destroy the association node
            SMSO_vDestroy( (SMS_OBJECT) psAssoc );

            break;
        }

        // We have a handle for the caller
        hChannelArt = psAssoc->hChannelArt;

    } while (FALSE);

    // Release the art owner
    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return hChannelArt;
}

/*****************************************************************************
*
*   bChanProcessArtDatabase
*
*   This function processes the associations found within the database
*   in order to create all necessary channel art objects and associations.
*   Processing is done by extracting all associations and attributes from
*   the database via a "natural join" between the image attribute and
*   association relations.  When this operation has completed, all necessary
*   channel art objects and associations have been made based upon the
*   database's contents.
*
*****************************************************************************/
static BOOLEAN bChanProcessArtDatabase (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bChanAssocsAvailable,
    BOOLEAN bOverlayChanAssocsAvailable,
    BOOLEAN bCatAssocsAvailable
        )
{
    BOOLEAN bOk, bWantLogos = TRUE, bWantBackgrounds = TRUE;
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT sResult;

    // We may need to re-create the association table
    bOk = bChanRegenerateArtAssocTable(
        psObj, psObj->hChanSQLConnection, bChanAssocsAvailable,
        bOverlayChanAssocsAvailable, bCatAssocsAvailable);

    if (bOk == FALSE)
    {
        return FALSE;
    }

    // We have validated the selected image types earlier.  We know
    // the application has specified at least one image type, and
    // if the application chose secondary logos it also chose primary logos.
    if ( (psObj->tSelectedImageTypes & CHANNEL_ART_AVAILABLE_IMAGE_LOGO)
            != CHANNEL_ART_AVAILABLE_IMAGE_LOGO )
    {
        // We don't want logos
        bWantLogos = FALSE;
    }

    if ( (psObj->tSelectedImageTypes & CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND)
            != CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND )
    {
        // We don't want backgrounds
        bWantBackgrounds = FALSE;
    }

    if ( (bWantLogos == FALSE) && (bWantBackgrounds == FALSE) )
    {
        // We validated the mask before, but as far as well can tell we
        // don't want any image types. Odd.
        return FALSE;
    }

    // Format the SQL query
    snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
        CA_SELECT_DB_JOIN_BY_TYPE,
        (bWantLogos == TRUE)?CHANNEL_ART_IMAGETYPE_LOGO:CHANNEL_ART_IMAGETYPE_BKGRND,
        (bWantBackgrounds == TRUE)?CHANNEL_ART_IMAGETYPE_BKGRND:CHANNEL_ART_IMAGETYPE_LOGO);

    // Initialize the result structure
    OSAL.bMemSet(&sResult, 0, sizeof(CHANNEL_ART_DB_JOIN_RESULT_STRUCT));
    sResult.psObj = psObj;

    // Initialize the portion of the assoc ctrl stucture we're going to use
    psObj->sAssocCtrl.psChanLastArtAssoc = NULL;

    // Grab a default to start the process
    psObj->sAssocCtrl.psChanLastArtAssoc = psChanGetArtAssoc(
        psObj->psArtOwner, TRUE, TRUE, CHANNEL_ART_INVALID_SOURCE_ID);
    if ( NULL == psObj->sAssocCtrl.psChanLastArtAssoc )
    {
        // Should never happen!
        return FALSE;
    }

    bOk = SQL_INTERFACE.bQuery(
        psObj->hChanSQLConnection, &psObj->acBuffer[0],
        (SQL_QUERY_RESULT_HANDLER)bChanProcessJoinDB, &sResult ) ;

    if ((bOk == TRUE) && (sResult.bSuccess == TRUE))
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bChanRegenerateArtAssocTable
*
*   Populates the association table using values from the default association
*   table.  Additionally, indicate in the DB that these associations are
*   available for use.
*
*****************************************************************************/
static BOOLEAN bChanRegenerateArtAssocTable (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN bChanAssocsAvailable,
    BOOLEAN bOverlayChanAssocsAvailable,
    BOOLEAN bCatAssocsAvailable
        )
{
    BOOLEAN bOk = TRUE;

    do
    {
        // Do we need to regenerate the channel associations?
        if (bChanAssocsAvailable == FALSE)
        {
            vStartTransaction( hSQLConnection,
                    &psObj->bChanInTransaction );

            // Build the update string to ensure the channel association
            // entries are in a known state (empty)
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_REMOVE_ASSOCS, FALSE,
                GsChannelArtIntf.tStartServiceID,
                GsChannelArtIntf.tEndServiceID);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Copy the associations from the default table to the assoc table
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_COPY_DEFAULT_ASSOCS, FALSE,
                GsChannelArtIntf.tStartServiceID,
                GsChannelArtIntf.tEndServiceID);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Update the DB info table to indicate channel associations are
            // now available
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_UPDATE_CHAN_ASSOC_AVAIL, TRUE );
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }
        }

        if (bOverlayChanAssocsAvailable == FALSE)
        {
            vStartTransaction( hSQLConnection,
                    &psObj->bChanInTransaction );

            // Build the update string to ensure the overlay channel
            // association entries are in a known state (empty)
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_REMOVE_ASSOCS, FALSE,
                GsChannelArtIntf.tOverlayStartServiceID,
                GsChannelArtIntf.tOverlayEndServiceID);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Copy the overlay associations from the default table to the
            // assoc table
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_COPY_DEFAULT_ASSOCS, FALSE,
                GsChannelArtIntf.tOverlayStartServiceID,
                GsChannelArtIntf.tOverlayEndServiceID);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Update the DB info table to indicate channel associations are
            // now available
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_UPDATE_OVERLAY_CHAN_ASSOC_AVAIL, TRUE );
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,  &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }
        }

        // Do we need to regenerate the category associations?
        if (bCatAssocsAvailable == FALSE)
        {
            vStartTransaction( hSQLConnection,
                    &psObj->bChanInTransaction );

            // Build the update string to ensure the category association
            // entries are in a known state
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_REMOVE_ASSOCS, TRUE,
                GsRadio.sCategory.un16BroadcastIdMin,
                GsRadio.sCategory.un16BroadcastIdMax);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Copy the associations from the default table to the assoc table
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_COPY_DEFAULT_ASSOCS, TRUE,
                GsRadio.sCategory.un16BroadcastIdMin,
                GsRadio.sCategory.un16BroadcastIdMax);
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }

            // Update the DB info table to indicate category associations are
            // now available
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_UPDATE_CAT_ASSOC_AVAIL, TRUE );
            bOk = SQL_INTERFACE.bExecuteCommand(hSQLConnection,
                    &psObj->acBuffer[0]);
            if (bOk == FALSE)
            {
                break;
            }
        }

        vEndTransaction(hSQLConnection, &psObj->bChanInTransaction);

        return TRUE;

    } while (FALSE);

    vEndTransaction(hSQLConnection, &psObj->bChanInTransaction);

    // Error! DB access failure
    vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );

    return FALSE;
}

/*****************************************************************************
*
*   bImageTypeSelected
*
*****************************************************************************/
static BOOLEAN bImageTypeSelected (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        )
{
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tType;

    tType = tImageTypeToMask(eImageType);

    return ((psObj->tSelectedImageTypes & tType) == tType);
}

/*****************************************************************************
*
*   tImageTypeToMask
*
*****************************************************************************/
static CHANNEL_ART_AVAILABLE_IMAGE_MASK tImageTypeToMask (
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        )
{
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tMask;

    switch (eImageType)
    {
        case CHANNEL_ART_IMAGETYPE_LOGO:
        {
            tMask = CHANNEL_ART_AVAILABLE_IMAGE_LOGO;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO:
        {
            tMask = CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_BKGRND:
        {
            tMask = CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND;
        }
        break;

        case CHANNEL_ART_IMAGETYPE_MAX:
        default:
        {
            tMask = CHANNEL_ART_AVAILABLE_IMAGE_NONE;
        }
        break;
    }

    return tMask;
}

/*****************************************************************************
*
*   tImageFilenameLen
*
*   This function is used by all interested parties in order to determine
*   the size of the filenames that are used by the channel art data service
*   provided a given path image format (ie. PNG), and image type (ie. logo)
*   This is used by shim friend functions to get both album and channel
*   art filename lengths.
*
*****************************************************************************/
static size_t tImageFilenameLen (
    IMAGE_FORMAT_ENUM eFormat,
    UN16 un16ServiceSpecificImageType,
    const char *pacFilePath,
    size_t tLabelLength
        )
{
    size_t tImageFilenameLen = 0;

    // Verify input pointer and the image type
    if ( ( pacFilePath == NULL ) ||
         ( un16ServiceSpecificImageType >= CHANNEL_ART_IMAGETYPE_MAX ) )
    {
        return 0;
    }

    // Determine which file extension to use
    switch ( eFormat )
    {
        case IMAGE_FORMAT_PNG:
        {
            // Add the length of the extension
            tImageFilenameLen +=
                sizeof(CHANNEL_ART_PNG_FILE_EXTENSION);
        }
        break;

        case IMAGE_FORMAT_JPEG:
        {
            // Add the length of the extension
            tImageFilenameLen +=
                sizeof(CHANNEL_ART_JPEG_FILE_EXTENSION);
        }
        break;

        // No extension, no file len!
        default:
        {
            return 0;
        }
    }

    // Add the length of the provided path
    tImageFilenameLen += strlen(pacFilePath);

    // Add the length of the file name
    tImageFilenameLen += tLabelLength;

    // Add one byte for the trailing NULL character
    tImageFilenameLen += 1;

    return tImageFilenameLen;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   This is the top-level channel art message processor function.  An
*   incoming message is routed to the appropriate message type processor
*   (within the channel art interface) by this function.
*
*****************************************************************************/
static BOOLEAN bProcessMessage (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bSuccess = FALSE,
            bAlbum = FALSE;
    DSI tDSI = (DSI)0;

    // No events pending just yet
    psObj->sOTACtrl.bEventsPending = FALSE;

    // The DSI is written to the head of the buffer in the DSM using the
    // non-endian-correcting tBufferWriteHead, so we read it back the
    // same way.
    (void) OSAL.tBufferReadHead (
            *phPayload, &tDSI, sizeof(tDSI));

    if ( GsChannelArtIntf.tDSI == tDSI )
    {
        // Always say we want the secondary image
        // until we know we don't (conditioned by
        // the service's support of secondary images)
        psObj->sOTACtrl.bSecondaryImageNeeded = psObj->bSecondarySupported;

        // Ask the interface to process this message
        bSuccess = GsChannelArtIntf.bProcessMessage(
            psObj->hChanInterface, phPayload );
    }
    else if ( GsAlbumArtIntf.tDSI == tDSI )
    {
        // Mark that we're actually handling album art
        bAlbum = TRUE;

        // Ask the interface to process this message
        bSuccess = GsAlbumArtIntf.bProcessMessage(
            psObj->hAlbInterface, phPayload );
    }
    else
    {
        // We don't know what to do with this, so just return
        // TRUE as this isn't necessarily an error.

        printf(
            CHANNEL_ART_MGR_OBJECT_NAME
            ": Received unknown DSI:%u", (UN16)tDSI
            );
        return TRUE;

    }

    if ( ( TRUE == bSuccess )                       &&
         ( psObj->sOTACtrl.bEventsPending == TRUE ) )
    {
        // Issue the event list generated
        bSuccess = bIssueUpdate( psObj, bAlbum );
    }

    psObj->sOTACtrl.bEventsPending = FALSE;

    return bSuccess;
}

/*******************************************************************************
*
*   bChanProcessImageUpdate
*
*   This function is called after the channel art interface has successfully
*   processed a portion of an image message.  Processing the results of that
*   operation will indicate if any updates need to be applied to memory or
*   the database and if any events need to be generated.
*
*   In addition, the channel art interface has left the image data in
*   hPayload for us.
*
*   Note: this function operates under the assumption that an SQL transaction
*   is open and will be closed by the caller.
*
*******************************************************************************/
static BOOLEAN bChanProcessImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t tImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        )
{
    FILE *psImageFile;
    BOOLEAN bOk;

    // Condition the secondary image field based on
    // what the service currently supports
    psAttribRow->bSecondaryAvailable &= psObj->bSecondarySupported;

    // Secondary Images aren't explicitly stored in the database,
    // they are indicated via a flag in the database for
    // their "primary" counterpart.  If we are processing
    // a secondary image, just write the file
    if (psAttribRow->un8ImageType != CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO)
    {
        CHANNEL_ART_ATTRIB_ROW_STRUCT sStoredImageAttrs;

        // Store the primary image attrs & status
        psObj->sOTACtrl.sPrimaryImageAttrs = *psAttribRow;
        psObj->sOTACtrl.bPrimaryImageUpdated = FALSE;

        // We are looking for the attributes matching the
        // Image Id and type specified in psAttribs
        sStoredImageAttrs.un16ImageId = psAttribRow->un16ImageId;
        sStoredImageAttrs.un8ImageType = psAttribRow->un8ImageType;

        // Look for a row that has the same image id/type as that
        // which was just received -- this will tell us if a pre-existing
        // image attribute row has been updated by this message
        bOk = bGetAttribRow( psObj, NULL, &sStoredImageAttrs );

        if (bOk == TRUE)
        {
            // The image has been found!
            BOOLEAN bImageVersionEqual;

            // Get the interface to compare these
            // two versions
            bImageVersionEqual = GsChannelArtIntf.bCompareImageVersions(
                &sStoredImageAttrs, psAttribRow, NULL );

            // Verify the version in the database
            // is older than the new image version
            if ( bImageVersionEqual == FALSE)
            {
                printf(
                    CHANNEL_ART_MGR_OBJECT_NAME
                    ": Update Received For Image:I%-5.5uT%-2.2u.%s"
                    "\n\tOld Version: %u \n\tNew Version: %u \n\tHas Secondary: %s\n",
                    sStoredImageAttrs.un16ImageId, sStoredImageAttrs.un8ImageType,
                    (sStoredImageAttrs.un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
                    sStoredImageAttrs.un16ImageVer, psAttribRow->un16ImageVer,
                    (psAttribRow->bSecondaryAvailable == TRUE)?"Yes":"No");

                DATASERVICE_IMPL_vLog(
                    CHANNEL_ART_MGR_OBJECT_NAME
                    ": Update Received For Image:I%-5.5uT%-2.2u.%s"
                    "\n\tOld Version: %u \n\tNew Version: %u \n\tHas Secondary: %s\n",
                    sStoredImageAttrs.un16ImageId, sStoredImageAttrs.un8ImageType,
                    (sStoredImageAttrs.un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
                    sStoredImageAttrs.un16ImageVer, psAttribRow->un16ImageVer,
                    (psAttribRow->bSecondaryAvailable == TRUE)?"Yes":"No");

                // A pre-existing image has been updated -- we may
                // need to indicate the change to our subscribed devices
                psObj->sOTACtrl.bPrimaryImageUpdated = TRUE;
            }
            else
            {
                // Nothing to do - we're all up to date with this
                // image since we have an image stored with the
                // same id and version (this goes for
                // any secondary image as well)
                psObj->sOTACtrl.bSecondaryImageNeeded = FALSE;
                return TRUE;
            }
        }
        else
        {
            // There were no results, which means we need to add
            // a new row in the database for this image
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": New Received Image:I%-5.5uT%-2.2u.%s"
                "\n\tVersion: %u\n\tHas Secondary: %s\n",
                psAttribRow->un16ImageId, psAttribRow->un8ImageType,
                (psAttribRow->un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
                psAttribRow->un16ImageVer,
                (psAttribRow->bSecondaryAvailable == TRUE)?"Yes":"No");

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": New Received Image:I%-5.5uT%-2.2u.%s"
                "\n\tVersion: %u\n\tHas Secondary: %s\n",
                psAttribRow->un16ImageId, psAttribRow->un8ImageType,
                (psAttribRow->un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
                psAttribRow->un16ImageVer,
                (psAttribRow->bSecondaryAvailable == TRUE)?"Yes":"No");
        }
    }
    else if (psObj->sOTACtrl.bSecondaryImageNeeded == FALSE)
    {
        // All done!
        return TRUE;
    }

    // Now, create/update the file (hold off on the
    // database update until we're done)

    // Generate the file name
    bOk = CHANNEL_ART_MGR_bChannelArtCreateImageFilename (
        psAttribRow->un16ImageId,
        psAttribRow->un16ImageVer,
        (CHANNEL_ART_IMAGETYPE_ENUM)psAttribRow->un8ImageType,
        (IMAGE_FORMAT_ENUM)psAttribRow->un8CodeType,
        &psObj->pacChannelArtServiceFilePath[0],
        &psObj->acBuffer[0],
        sizeof(psObj->acBuffer)
            );

    if (bOk == FALSE)
    {
        vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );

        return FALSE;
    }

    // Open the file, creating it if necessary
    psImageFile = fopen( psObj->acBuffer, "wb");

    if (psImageFile != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eBufferWriteToFile(hPayload, FALSE, psImageFile);

        // Close the file
        fclose(psImageFile);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // We are unable to write the buffer
            // for some reason
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );

            return FALSE;
        }
    }
    else
    {
        // File accesses shouldn't fail -- we have to
        // have access to storage for this service to be
        // useful.  Consider this an error
        vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );

        return FALSE;
    }

    // Perform the database command and update it
    // now that the file is intact, but only after
    // we've handled all necessary images (primary & secondary)
    if (psAttribRow->bSecondaryAvailable == FALSE)
    {
        // Generate the command
        if (psObj->sOTACtrl.bPrimaryImageUpdated == TRUE)
        {
            // This is a pre-existing image which
            // has been updated by this message

            // Generate the SQL command to update the data row
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_UPDATE_ATTRIB_ROW,
                psObj->sOTACtrl.sPrimaryImageAttrs.un16ImageVer,
                psObj->sOTACtrl.sPrimaryImageAttrs.un8CodeType,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.tBackgroundOptions,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.sLineBitmap.tLineBitmapIndex,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.un32BackgroundColor,
                psObj->sOTACtrl.sPrimaryImageAttrs.bSecondaryAvailable,
                psObj->sOTACtrl.sPrimaryImageAttrs.un16ImageId,
                psObj->sOTACtrl.sPrimaryImageAttrs.un8ImageType
                    );
        }
        else
        {
            // This is a new image that we have
            // no history of

            // Generate the SQL command to insert the new data row
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_INSERT_ATTRIB_ROW,
                psObj->sOTACtrl.sPrimaryImageAttrs.un16ImageId,
                psObj->sOTACtrl.sPrimaryImageAttrs.un8ImageType,
                psObj->sOTACtrl.sPrimaryImageAttrs.un16ImageVer,
                psObj->sOTACtrl.sPrimaryImageAttrs.un8CodeType,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.tBackgroundOptions,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.sLineBitmap.tLineBitmapIndex,
                psObj->sOTACtrl.sPrimaryImageAttrs.sBackgroundGfx.un32BackgroundColor,
                psObj->sOTACtrl.sPrimaryImageAttrs.bSecondaryAvailable
                    );
        }

        // Start a transaction for this update
        vStartTransaction( psObj->hChanSQLConnection,
                &psObj->bChanInTransaction );

        bOk = SQL_INTERFACE.bExecuteCommand(
            psObj->hChanSQLConnection, &psObj->acBuffer[0] );

        if (bOk == FALSE)
        {
            // Error! DB access failure
            vSetError( psObj, DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE );

            return FALSE;
        }

        // There may be associations which are affected by
        // this update.  Inform the interested parties
        if (psObj->sOTACtrl.bPrimaryImageUpdated == TRUE)
        {
            BOOLEAN bEventGenerated = FALSE;

            // Attempt to update art objects now
            bOk = bUpdateArtObjectsForImageUpdate(
                psObj, &psObj->sOTACtrl.sPrimaryImageAttrs, &bEventGenerated);
            if (bOk == FALSE)
            {
                // Error!
                vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL);
                return FALSE;
            }

            // Determine if we added at least one event
            if (bEventGenerated == TRUE)
            {
                // We have event(s) to issue
                psObj->sOTACtrl.bEventsPending = TRUE;
            }
        }
    }

    return TRUE;
}

/*******************************************************************************
*
*   bAlbumProcessImageUpdate
*
*   This function is called after the album art interface has successfully
*   processed a portion of an image message.  Processing the results of that
*   operation will indicate if any updates need to be applied to memory or
*   the database and if any events need to be generated.
*
*   In addition, the channel art interface has left the image data in
*   hPayload for us.
*
*******************************************************************************/
static BOOLEAN bAlbumProcessImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload,
    STRING_OBJECT *phCaption,
    CHANNEL_ART_OBJECT *phArtToUpdate
        )
{
    char *pacFilePath;
    BOOLEAN bOk, bEventGenerated = FALSE;

    // What kind of image is this? Ephemeral or a service default? If the
    // caller wants us to update an art object, we know this is ephemeral,
    // as service defaults just get written to disk. (We also need to know
    // as we use this to determine where the file will be stored: service
    // default art goes in the channel art service default path, while
    // everything else goes to the ephemeral path.

    if ( ( NULL == phArtToUpdate ) ||
         ( CHANNEL_ART_INVALID_OBJECT == *phArtToUpdate ) )
    {
        pacFilePath = psObj->pacChannelArtServiceFilePath;
    }
    else
    {
        pacFilePath = psObj->pacEphemeralPath;
    }

    // If this is a service default we need to save the
    // version info of the image we're writing to disk to
    // the attribute DB table.
    if ( ( NULL == phArtToUpdate ) ||
         ( CHANNEL_ART_INVALID_OBJECT == *phArtToUpdate ) )
    {
        BOOLEAN bInTransaction = FALSE;
        UN16 un16Version;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // We have to write the incoming image to disk
        eReturnCode =
            eAlbumProbeDefaultImage(psObj, psAssocRow->un16ImageId, &un16Version);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            printf("Default image already exists for Id %d",
                   (int)psAssocRow->un16ImageId);
            if (psAssocRow->un16ImageVer == un16Version)
            {
                printf("Default image is up-to-date. Pretend we're ok for the caller");
                return TRUE;
            }
            else
            {
                printf("Will update from version %d to %d",
                    (int)un16Version, (int)psAssocRow->un16ImageVer);
            }
        }

        (void) bWriteAlbumArtToFile( psObj,
                    psAssocRow->un16ImageId, psAssocRow->un16ImageVer,
                    pacFilePath, hPayload );

        vStartTransaction( psObj->hAlbumPerSQLConnection,
                &bInTransaction );

        if ( FALSE == bInTransaction )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": Error starting album"
                "art attribute table transaction." );

            return FALSE;
        }

        // Generate the update command based using the new
        // attribute info
        snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                AA_UPDATE_ATTRIBUTE_ROW, psAssocRow->un16ImageId,
                psAssocRow->un16ImageVer );

        // Update or insert the DB entry
        bOk = SQL_INTERFACE.bExecuteCommand(
                psObj->hAlbumPerSQLConnection,
                &psObj->acBuffer[0] );

        vEndTransaction( psObj->hAlbumPerSQLConnection,
                &bInTransaction );

        if ( ( FALSE == bOk ) ||
             ( TRUE == bInTransaction ) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": Error updating album art"
                "DB attribute." );

            return FALSE;
        }
    }
    // If we received an channel art handle (meaning we have an ephemeral image),
    // try and update the channel art now.
    else
    {
        BOOLEAN bLocked;

        (void) bWriteAlbumArtToFile( psObj,
                    psAssocRow->un16ImageId, psAssocRow->un16ImageVer,
                    pacFilePath, hPayload );

        // Lock object for exclusive access
        bLocked = SMSO_bLock((SMS_OBJECT)*phArtToUpdate, OSAL_OBJ_TIMEOUT_INFINITE);

        if ( bLocked == FALSE )
        {
            // We were provided a bad channel art handle
            return FALSE;
        }

        bOk = CHANNEL_ART_bSetAlbumImage( *phArtToUpdate, psAssocRow,
                phCaption, &bEventGenerated );

        // Now unlock and make sure everything went okay
        SMSO_vUnlock( (SMS_OBJECT)*phArtToUpdate );

        if ( bOk == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Error trying to set an album art image." );

            return FALSE;
        }
    }

    // Determine if we added at least one event
    if ( TRUE == bEventGenerated )
    {
        // We have event(s) to issue
        psObj->sOTACtrl.bEventsPending = TRUE;
    }

    // If we're here, everything went okay.
    return TRUE;
}

/*******************************************************************************
*
*   bWriteAlbumArtToFile
*
*   This function write album art data (from a buffer) into a file in the
*   application's storage area. This uses the image data in
*   hPayload.
*
*******************************************************************************/
static BOOLEAN bWriteAlbumArtToFile(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16Id, UN16 un16Ver,
    char const * const pacFilePath,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOk;
    FILE *psImageFile;
    OSAL_RETURN_CODE_ENUM eReturn;

    // Generate the file name
    bOk = CHANNEL_ART_MGR_bAlbumArtCreateImageFilename (
        un16Id,
        un16Ver,
        pacFilePath,
        &psObj->acBuffer[0],
        sizeof(psObj->acBuffer)
            );

    // Make sure that went okay ...
    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "Can not create an album art image filename.");

        vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
        return FALSE;
    }

    // Now open the file, creating it if necessary
    psImageFile = fopen( psObj->acBuffer, "wb");

    if ( NULL == psImageFile )
    {
        // File accesses shouldn't fail -- we have to
        // have access to storage for this service to be
        // useful.  Consider this an error

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "Can not perform an fopen on a new album "
                "art image.");

        vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
        return FALSE;
    }

    eReturn = OSAL.eBufferWriteToFile( hPayload, TRUE, psImageFile );

    // Close the file
    fclose(psImageFile);

    if ( OSAL_SUCCESS != eReturn )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
               "Error writing buffer to file: %s",
               OSAL.pacGetReturnCodeName(eReturn));

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateArtObjectsForImageUpdate
*
*****************************************************************************/
static BOOLEAN bUpdateArtObjectsForImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    BOOLEAN *pbEventGenerated
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // Lock the art owner
    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT sUpdate;

        // Populate the update struct for linked list iteration
        sUpdate.psObj = psObj;

        // Update the image attributes of all
        // image data using the new attributes
        sUpdate.psImageAttrs = psAttribRow;

        // No events have been added yet
        sUpdate.bEventAdded = FALSE;

        do
        {
            // Iterate over the linked list of associations
            // to see if there are any active associations with
            // this Image Id and update them
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->psArtOwner->hChanAssociations,
                bUpdateChannelArtWithMatchingImageAttrs,
                (void *)&sUpdate
                    );

            // Indicate an error occurred if the iteration failed (but
            // not if the list is empty)
            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                // Error! Can't access the linked list
                vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
                break;
            }

            // Determine if we added at least one event
            if (sUpdate.bEventAdded == TRUE)
            {
                // We have event(s) to issue
                *pbEventGenerated = TRUE;
            }

            bSuccess = TRUE;

        } while (FALSE);

        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
    }

    return bSuccess;
}

/*******************************************************************************
*
*   eChanProcessAssocUpdate
*
*******************************************************************************/
static CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eChanProcessAssocUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow
        )
{
    BOOLEAN bNewerVersion, bAssocUpdated;
    CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eResult =
            CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;

    printf(CHANNEL_ART_MGR_OBJECT_NAME
            ": Found Association in Message S:%u, I:%u, T:%u %s\n",
            psAssocRow->un16Source,
            psAssocRow->un16ImageId,
            psAssocRow->un8ImageType,
            (psAssocRow->bRemove)?"(REMOVE)":"");

    // Update this association in the system
    bAssocUpdated = bChanUpdateAssociation( psObj, psAssocRow, &bNewerVersion );

    // If a new association object was created due to the
    // update, then we need to add this event to our
    // event list to inform all subscribed devices
    if (bAssocUpdated == TRUE)
    {
        printf(
            CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u updated (Image:I%-5.5uT%-2.2u)\n",
            (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
            (psAssocRow->bCategory == TRUE)?"category":"service id",
            psAssocRow->un16Source, psAssocRow->un16ImageId,
            psAssocRow->un8ImageType
                );

        DATASERVICE_IMPL_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u updated (Image:I%-5.5uT%-2.2u)\n",
            (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
            (psAssocRow->bCategory == TRUE)?"category":"service id",
            psAssocRow->un16Source, psAssocRow->un16ImageId,
            psAssocRow->un8ImageType
                );

        // This association was updated
        eResult = CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED;
    }
    else // Association not updated
    {
        if (bNewerVersion == TRUE)
        {
            // The new association is a
            // newer version that what we
            // have now. So, indicate that
            // this association message has at least
            // one pending entry.
            psObj->sAssocCtrl.bCurrentAssociationHasPendingEntries = TRUE;

            if (psAssocRow->bHighPriority == FALSE)
            {
                // This association needs to be forced to use the default
                // until we're able to service it.  Why? well, we only process the
                // low-priority association if the high-priority association failed.  So,
                // if this association failed it means we got nothing for this channel/category
                // and we better make sure it's just using the default 'cause god knows
                // what image is associated with it right now.
                eResult = CHANNEL_ART_ASSOC_PROCESS_RESULT_PENDING_FORCE_DEFAULT;
            }
            else
            {
                // We are pending on some image(s)
                eResult = CHANNEL_ART_ASSOC_PROCESS_RESULT_PENDING;
            }

            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u waiting for new version of Image:I%-5.5uT%-2.2u\n",
                (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
                (psAssocRow->bCategory == TRUE)?"category":"service id",
                psAssocRow->un16Source, psAssocRow->un16ImageId,
                psAssocRow->un8ImageType
                    );

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u waiting for new version of Image: I%-5.5uT%-2.2u\n",
                (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
                (psAssocRow->bCategory == TRUE)?"category":"service id",
                psAssocRow->un16Source, psAssocRow->un16ImageId,
                psAssocRow->un8ImageType
                    );
        }
        else
        {
            // We didn't update anything...didn't need to
            eResult = CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
        }
    }

    return eResult;
}

/*******************************************************************************
*
*   eAlbumProcessAssocUpdate
*
*   Given an album association row, we find the appropriate record in our
*   album associations table and update it appropriately. As part of the
*   process, we pass an appropriate channel art handle back up the chain
*   which will be used to receive our image data. Note: this function
*   *must* be called from a context where the art owner is already
*   locked.
*
*******************************************************************************/
static CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eAlbumProcessAssocUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_OBJECT *phArtToUpdate,
    BOOLEAN bFromDB
        )
{
    BOOLEAN bAddedAssoc = FALSE;
    ALBUM_ART_ASSOC_STRUCT *psAssoc;

    printf(CHANNEL_ART_MGR_OBJECT_NAME
      ": Found Association in Message S:%u, I:%u, V:%u \n",
      psAssocRow->un16ServiceId, psAssocRow->un16ImageId,
      psAssocRow->un16ImageVer );

    psAssoc = psAlbumGetAssocByService ( psObj->psArtOwner,
            psAssocRow->un16ServiceId );

    if (NULL == psAssoc)
    {
        bAddedAssoc = TRUE;
        psAssoc = psAlbumAddNewAssocEntry( psObj->psArtOwner,
            (SERVICE_ID)psAssocRow->un16ServiceId );

        if ( NULL == psAssoc )
        {
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
        }
    }

    // Take action depending on what time of information is in the association;
    // We take care of the easy cases first: if we're commanded to either default
    // or blank the art for a SID, do that now.
    if ( TRUE == psAssocRow->bDefault )
    {
        // Make sure we're not already defaulted.
        if ( psAssoc->bDefault == TRUE )
        {
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
        }

        psAssoc->bDefault = TRUE;
        psAssoc->bBlank = FALSE;

        // Per SX-9845-0231, 5.1.3.3.1, clear out any PID association that we're
        // currently using
        vAlbumPidAssocListClean(psAssoc);
    }
    else if ( TRUE == psAssocRow->bBlank )
    {
        // Make sure we're not already blank
        if ( psAssoc->bBlank == TRUE )
        {
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
        }

        psAssoc->bDefault = FALSE;
        psAssoc->bBlank = TRUE;

        // Per SX-9845-0231, 5.1.3.4.1, clear out any PID association that we're
        // currently using
        vAlbumPidAssocListClean(psAssoc);
    }
    // Only timed associations will have a populated un32ExpirationDelta field.
    // If we have one of those, we know this is a timed association.
    else if ( psAssocRow->un32Expiration > 0 )
    {
        UN32 un32NowSec = 0;
        OSAL_RETURN_CODE_ENUM eReturn;

        // This counts as a command to turn off any defaulting or blanking
        // currently on the channel, per SX-9845-0231 5.1.3.3.1
        // and 5.1.3.4.1
        psAssoc->bDefault = FALSE;
        psAssoc->bBlank = FALSE;

        // No matter what, we're going to need the current time
        eReturn = OSAL.eTimeGet( &un32NowSec );

        if ( eReturn != OSAL_SUCCESS )
        {
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
        }

        // If the timer was inactive, or if our new expiration happens
        // sooner than the next one in the system, make sure to start
        // the timer (again, if necessary)
        if ( ( psObj->un32NextCleanupSecs > psAssocRow->un32Expiration ) ||
             ( 0 == psObj->un32NextCleanupSecs ) )
        {
            BOOLEAN bOk;
            UN32 un32NewExpirationSecs;

            un32NewExpirationSecs = psAssocRow->un32Expiration - un32NowSec;

            bOk = DATASERVICE_IMPL_bSetTimedEvent(
                psObj->hCleanupEvent, TRUE, FALSE, un32NewExpirationSecs);

            if ( FALSE == bOk )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Unable to set up dataservice timed event!" );
            }

            psObj->un32NextCleanupSecs = psAssocRow->un32Expiration;
        }

        // Update the expiration in the association.
        psAssoc->un32ExpirationDelta = psAssocRow->un32Expiration;

        // Create new timed art if necessary
        if ( CHANNEL_ART_INVALID_OBJECT == psAssoc->hSidArtTimed )
        {
            psAssoc->hSidArtTimed = CHANNEL_ART_hCreate((SMS_OBJECT)psObj->psArtOwner,
                        &psObj->pacEphemeralPath[0]);
            psAssocRow->un16ImageId = psObj->un32NextEphemeralID++;
            printf("Assigning new ID: %u\n", psAssocRow->un16ImageId);
        }
        else
        {
            UN32 un32ImageID = 0;
            eGetAlbumArtProperties ( psAssoc->hSidArtTimed, &un32ImageID, NULL );
            printf("Re-using ID: %u\n", un32ImageID);
            psAssocRow->un16ImageId = un32ImageID;
        }

        // We *have* to have a valid phArtToUpdate pointer for this to
        // work.
        if ( NULL != phArtToUpdate )
        {
            *phArtToUpdate = psAssoc->hSidArtTimed;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Received a NULL phArtToUpdate on a non-default"
                    "association!");
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
        }

    }
    // A PID will only be present in the association row if this association
    // is for a PID/SID combo.
    //
    // Note: per Leslie, if we have a SID/PID-related association and we
    // receive an association for the same SID/PID combo, it's to be
    // treated as a duplicate.
    else if ( psAssocRow->un32ProgramId > 0 )
    {
        ALBUM_ART_PID_TO_ART_MAP *psAssocMap;

        // See if the channel art needs a tweak
        psAssocMap = psAlbumPidAssocFindArt(psAssoc, psAssocRow->un32ProgramId);
        if ((psAssocMap != NULL) && (psAssocMap->hArt != CHANNEL_ART_INVALID_OBJECT))
        {
            // We already have this association, so just skip over this.
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
        }

        if (psAssocMap == NULL)
        {
            // Create new SID-PID mapping
            psAssocMap =
                psAlbumPidAssocAllocate(psAssoc,
                    (PROGRAM_ID)psAssocRow->un32ProgramId);
            if (psAssocMap == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME": failed to allocate assoc");
                return CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
            }
        }

        if (CHANNEL_ART_INVALID_OBJECT == psAssocMap->hArt)
        {
            // New association detected
            psAssocMap->hArt =
                CHANNEL_ART_hCreate((SMS_OBJECT)psObj->psArtOwner,
                                    &psObj->pacEphemeralPath[0]);
            psAssocRow->un16ImageId = psObj->un32NextEphemeralID++;
            psAssocMap->bInUse = FALSE;
            printf("Assigning new ID: %u\n", psAssocRow->un16ImageId);
        }
        else
        {
            // Re-used association detected
            UN32 un32ImageID = 0;
            eGetAlbumArtProperties(psAssocMap->hArt, &un32ImageID, NULL);
            printf("Re-using ID: %u\n", un32ImageID);
            psAssocRow->un16ImageId = un32ImageID;
        }

        // We have to have a valid phArtToUpdate pointer for this to
        // work.
        if ( NULL != phArtToUpdate )
        {
            *phArtToUpdate = psAssocMap->hArt;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Received a NULL phArtToUpdate on a non-default"
                    "association!");
            return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
        }
    }
    // If we got here, we're adding or updating a default
    else
    {
        BOOLEAN bOk, bArtUpdated;

        // See if we can determine that we don't need this update
        // right off the bat
        if ( CHANNEL_ART_INVALID_OBJECT != psAssoc->hDefaultArt )
        {
            UN32 un32Id = 0, un32Version = 0;
            SMSAPI_RETURN_CODE_ENUM eReturn;

            // Get the ID and version; errors are printed in
            // eGetAlbumArtProperties, so just return if things go south
            eReturn = eGetAlbumArtProperties(psAssoc->hDefaultArt, &un32Id,
                    &un32Version );
            if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
            {
                return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
            }

            if ( ( un32Id == (UN32)psAssocRow->un16ImageId ) &&
                 ( FALSE == bAddedAssoc ) )
            {
                // We already have this ...
                return CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED;
            }
        }

        // This function is called from two contexts:
        // a) The read from the DB on start, and
        // b) OTA updates
        // As only case b) is really "new" to us, we
        // don't write these back out to the DB unless
        // they're from OTA.

        if ( FALSE == bFromDB )
        {
            // Update the association entry in the database
            // Generate the update command based upon the new association
            snprintf( &psObj->acBuffer[0],
                      sizeof(psObj->acBuffer),
                      AA_UPDATE_ASSOCIATION_ROW,
                      psAssocRow->un16ImageId,
                      psAssocRow->un16ServiceId,
                      (UN8)psAssocRow->eType
                );

            // Update or insert the DB entry
            bOk = SQL_INTERFACE.bExecuteCommand(
                psObj->hAlbumPerSQLConnection,
                    &psObj->acBuffer[0] );

            if ( FALSE == bOk )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME": Error updating album art"
                        "DB assocation.");

                return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
            }
        }
        else
        {
            // Create the art if need be; default art
            // always goes in the service file path
            if ( CHANNEL_ART_INVALID_OBJECT == psAssoc->hDefaultArt ) 
            {
                psAssoc->hDefaultArt =
                    CHANNEL_ART_hCreate(
                        (SMS_OBJECT)psObj->psArtOwner,
                    &psObj->pacChannelArtServiceFilePath[0] );

                // Make sure it's valid now
                if ( CHANNEL_ART_INVALID_OBJECT == psAssoc->hDefaultArt )
                {
                    return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
                }
            }

            // Make sure to set the type mask (as default images can be limited
            // to certain CDO types
            psAssoc->eType = psAssocRow->eType;

            // Add the image to the channel art object; the NULL is for the
            // caption, which will only ever be present for ephemeral images.
            bOk = CHANNEL_ART_bSetAlbumImage( psAssoc->hDefaultArt,
                psAssocRow, NULL, &bArtUpdated );

            if ( TRUE != ( bOk & bArtUpdated) )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME": Call to "
                        "CHANNEL_ART_bSetAlbumImage failed.");
                return CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;
            }
        }

        // The interface may not provide us an art handle pointer if this
        // is merely an default image association update, so this
        // is not flagged as an error.
        if ( NULL != phArtToUpdate )
        {
            *phArtToUpdate = psAssoc->hDefaultArt;
        }
    }

    // Any paths through the logic where we didn't update the
    // association would have returned by now, so let's mark this
    // as updated before returning.
    psAssoc->bUpdated = TRUE;
    return CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED;
}

/*******************************************************************************
*
*   bChanFinalizeAssocProcessing
*
*   This function is called when the interface indicates the channel art
*   association list has been completely reported.
*
*******************************************************************************/
static BOOLEAN bChanFinalizeAssocProcessing (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN *pbEventGenerated
        )
{
    BOOLEAN bSuccess = FALSE;

    // Were all entries handled within this association message?
    if (psObj->sAssocCtrl.bCurrentAssociationHasPendingEntries == FALSE)
    {
        // Yes -- indicate that this message has been
        // completed so we can skip it later
        puts(CHANNEL_ART_MGR_OBJECT_NAME": All associations completed");

        // Tell the interface we were able to complete all the associations
        // provided to us by the interface
        GsChannelArtIntf.vAllAssociationsCompleted( psObj->hChanInterface );
    }

    do
    {
        // Do we need to verify the lineup & default the
        // unreferenced channels/categories?
        if (psObj->sAssocCtrl.bVerifyLineupAtCompletion == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            CHANNEL_ART_DEFAULT_LEFTOVER_STRUCT sLeftover;
            CHANNEL_ART_ASSOC_STRUCT *psAssoc;
            BOOLEAN bOk;

            // Just use the last flag here since they are all the same
            sLeftover.bCategory = psObj->sAssocCtrl.bCategory;
            sLeftover.bOverlay = psObj->sAssocCtrl.bOverlay;
            sLeftover.pbEventGenerated = pbEventGenerated;

            // Get the appropriate default art handle
            psAssoc = psChanGetArtAssoc(
                psObj->psArtOwner,
                TRUE,
                sLeftover.bCategory,
                CHANNEL_ART_INVALID_SOURCE_ID);

            if (psAssoc == NULL)
            {
                break;
            }

            sLeftover.hDefaultArt = psAssoc->hChannelArt;

            // Lock the art owner
            bOk = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
            if (bOk == FALSE)
            {
                break;
            }

            // Iterate our assocaitions and see if we come across any that haven't
            // been referenced.  We need to default any that haven't been referenced.
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->psArtOwner->hChanAssociations,
                (OSAL_LL_ITERATOR_HANDLER)bDefaultLeftoverAssociations, (void *)&sLeftover);

            // Always unlock regardless of what just happened
            SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*******************************************************************************
*
*   eChanAssocBegin
*
*   The interface uses this function to indicate a new group of associations
*   are under way.
*
*******************************************************************************/
static CHANNEL_ART_MGR_PROCESS_ASSOC_ENUM eChanAssocBegin (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_IMAGETYPE_ENUM eBaseImageType,
    BOOLEAN bVerifyLineup,
    BOOLEAN bCategory,
    BOOLEAN bOverlay
        )
{
    CHANNEL_ART_MGR_PROCESS_ASSOC_ENUM eResult =
        CHANNEL_ART_MGR_PROCESS_ASSOC_ERROR;
    BOOLEAN bOwner;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
        BOOLEAN bSelected = TRUE;

        // Attempt to filter categories -- channel associations
        // are always wanted
        if (bCategory == TRUE)
        {
            // Do we want associations of this type?
            bSelected = bImageTypeSelected(psObj, eBaseImageType);
        }

        if (bSelected == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
            puts(CHANNEL_ART_MGR_OBJECT_NAME": Begin Association Processing");
            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME": Begin Association Processing\n");

            // Flag the last association as not succeeding
            // since we're starting a new assoc message
            psObj->sAssocCtrl.bLastAssocSucceeded = FALSE;

            // We don't have any previous associations
            psObj->sAssocCtrl.psChanLastArtAssoc = NULL;

            // Indicate that this association message does not
            // have entries which have not yet been processed
            psObj->sAssocCtrl.bCurrentAssociationHasPendingEntries = FALSE;

            // Track the status of these flags
            psObj->sAssocCtrl.bVerifyLineupAtCompletion = bVerifyLineup;
            psObj->sAssocCtrl.bCategory = bCategory;
            psObj->sAssocCtrl.bOverlay = bOverlay;

            // Do we need to verify the lineup after this is done?
            if (bVerifyLineup == TRUE)
            {
                // Yes - Iterate to clear all referenced flags
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->psArtOwner->hChanAssociations,
                    (OSAL_LL_ITERATOR_HANDLER)bClearReferencedFlags, NULL);

                if (eReturnCode == OSAL_NO_OBJECTS)
                {
                    // Not a problem
                    eReturnCode = OSAL_SUCCESS;
                }
            }

            if (eReturnCode == OSAL_SUCCESS)
            {
                // We will begin processing this message
                eResult = CHANNEL_ART_MGR_PROCESS_ASSOC_BEGIN;
            }
        }
        else
        {
            // Ignore this message
            eResult = CHANNEL_ART_MGR_PROCESS_ASSOC_IGNORE;
        }
    }

    return eResult;
}

/*******************************************************************************
*
*   bChanAssocEnd
*
*   The interface uses this function to indicate it has completed a group of
*   associations.
*
*******************************************************************************/
static BOOLEAN bChanAssocEnd (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
        BOOLEAN bAssocUpdated = FALSE;

        DATASERVICE_IMPL_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": End Association Processing\n");

        puts(CHANNEL_ART_MGR_OBJECT_NAME": End Association Processing");

        bSuccess = bChanFinalizeAssocProcessing( psObj, &bAssocUpdated );

        if (bSuccess == TRUE)
        {
            // If we updated some associations while finalizing this message
            // then make sure we flag that there are events pending
            psObj->sOTACtrl.bEventsPending |= bAssocUpdated;
        }

        // End the SQL transaction
        vEndTransaction(psObj->hChanSQLConnection,
                &psObj->bChanInTransaction);
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bAlbumAssocBegin
*
*   The album art interface uses this function to indicate it is starting
*   a group of associations.
*
*******************************************************************************/
static BOOLEAN bAlbumAssocBegin (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        )
{
    BOOLEAN bOwner;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if ( TRUE == bOwner )
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
            = (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;

        DATASERVICE_IMPL_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": Start Album Association Processing\n");

        // Start the SQL transaction
        vStartTransaction( psObj->hAlbumPerSQLConnection,
                &psObj->bAlbumInTransaction );
    }

    return bOwner;
}

/*******************************************************************************
*
*   bAlbumAssocEnd
*
*   The album art interface uses this function to indicate it has completed
*   a group of associations.
*
*******************************************************************************/
static BOOLEAN bAlbumAssocEnd (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        )
{
    BOOLEAN bOwner;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if ( TRUE == bOwner )
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
            = (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;

        DATASERVICE_IMPL_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": End Album Association Processing\n");

        // End the SQL transaction
        vEndTransaction(psObj->hAlbumPerSQLConnection,
                &psObj->bAlbumInTransaction );
    }

    return bOwner;
}

/*******************************************************************************
*
*   bChanAssociationUpdate
*
*   The channel art interface uses this function to provide an association
*   update to the manager.
*
*******************************************************************************/
static BOOLEAN bChanAssociationUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    BOOLEAN *pbAssociationPending
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
    BOOLEAN bOwner, bLocked, bSelected, bSuccess = TRUE;
    CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eResult =
        CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;

    if (pbAssociationPending == NULL)
    {
        return FALSE;
    }

    // Initialize this to true
    *pbAssociationPending = TRUE;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bOwner == FALSE)
    {
        // Error! This is broken! BROKEN I say!
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        return FALSE;
    }

    // Do we want this image type?
    bSelected = bImageTypeSelected(
        psObj,
        (CHANNEL_ART_IMAGETYPE_ENUM)psAssocRow->un8ImageType);
    if (bSelected == FALSE)
    {
        // Not pending since we don't care about it
        *pbAssociationPending = FALSE;

        // All done!
        return TRUE;
    }

    // Now, lock the art owner
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == FALSE)
    {
        // Error! Can't get a hold of the art owner
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        return FALSE;
    }

    // Process this association update
    eResult = eChanProcessAssocUpdate(psObj, psAssocRow);

    // Okay, so what happened?
    if ((eResult == CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED) ||
        (eResult == CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED))
    {
        // We can only complete an association if it
        // is the "high priority version"
        if (psAssocRow->bHighPriority == TRUE)
        {
            // Tell the interface that this association
            // is no longer pending since we were just
            // able to apply it or it wasn't new to us
            *pbAssociationPending = FALSE;
        }

        if (eResult == CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED)
        {
            // An association was updated, so we have an event
            psObj->sOTACtrl.bEventsPending = TRUE;
        }
    }
    else if (eResult == CHANNEL_ART_ASSOC_PROCESS_RESULT_PENDING_FORCE_DEFAULT)
    {
        // In this situation, the assocation has failed all attempts,
        // so that means we'll update this assocation to use the default,
        // but we don't want say we've succefully applied an association
        // since we still need to apply it for real
        BOOLEAN bUpdated;

        // Make this association indicate the default image
        // update the database to use the default image here
        bUpdated = bChanDefaultAssoc (
            psObj, psObj->sAssocCtrl.psChanLastArtAssoc, 
            psAssocRow, TRUE, FALSE);
        if (bUpdated == TRUE)
        {
            psObj->sOTACtrl.bEventsPending = TRUE;
        }
    }
    else if (eResult == CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR)
    {
        bSuccess = FALSE;
    }

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return bSuccess;
}

/*******************************************************************************
*
*   bAlbumAssociationUpdate
*
*   The album art interface uses this function to provide an association update
*   to the manager.
*
*******************************************************************************/
static BOOLEAN bAlbumAssociationUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_OBJECT *phArtToUpdate
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
    BOOLEAN bOwner, bLocked, bSelected, bSuccess = TRUE;
    CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eResult =
        CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if ( FALSE == bOwner )
    {
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        // Error! This is broken! BROKEN I say!
        return FALSE;
    }

    // Do we want this image type?
    bSelected = bImageTypeSelected( psObj, CHANNEL_ART_IMAGETYPE_ALBUM );
    if ( FALSE == bSelected )
    {
        // All done!
        return TRUE;
    }

    // Now, lock the art owner
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);

    if ( FALSE == bLocked )
    {
        // Error! Can't get a hold of the art owner
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        return FALSE;
    }

    // Process this association update
    eResult = eAlbumProcessAssocUpdate(psObj, psAssocRow, phArtToUpdate, 
        FALSE);
    
    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    if ( CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED == eResult )
    {
        // If we updated some associations while finalizing this message
        // then make sure we flag that there are events pending
        psObj->sOTACtrl.bEventsPending = TRUE;
    }

    return bSuccess;
}


/*****************************************************************************
*
*   bChanResetAssociations
*
*   The interface uses this function to indicate to the manager that an art
*   "reset" is required and which portion of the art on which to operate.
*
*****************************************************************************/
static BOOLEAN bChanResetAssociations (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    BOOLEAN bContentOnly,
    BOOLEAN bOverlay,
    BOOLEAN bCategory
        )
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
    BOOLEAN bOwner, bSuccess = TRUE;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // Now, lock the art owner
    bOwner = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bOwner == FALSE)
    {
        // Error! Can't get a hold of the art owner
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

        return FALSE;
    }

    printf(CHANNEL_ART_MGR_OBJECT_NAME": Reset all %s %s associations %s\n",
        (bContentOnly == TRUE)?"dynamic":"static",
        (bCategory == TRUE)?"category":"channel",
        (bOverlay == TRUE)?"for overlay":" "
            );

    DATASERVICE_IMPL_vLog(
        CHANNEL_ART_MGR_OBJECT_NAME": Reset all %s %s associations %s\n",
        (bContentOnly == TRUE)?"dynamic":"static",
        (bCategory == TRUE)?"category":"channel",
        (bOverlay == TRUE)?"for overlay":" ");

    // Do we just need to remove content-based associations?
    if (bContentOnly == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        CHANNEL_ART_MGR_RESET_ASSOC_STRUCT sReset;

        // Populate the iterator structure
        sReset.psObj = psObj;
        sReset.bCategory = bCategory;
        sReset.bAssocUpdated = FALSE;
        sReset.bOverlay = bOverlay;

        // Iterate the list of association objects
        // to reset all affected entries
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psArtOwner->hChanAssociations,
            bResetContentArtWithMatchingType,
            (void *)&sReset );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! Can't iterate the list!
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

            bSuccess = FALSE;
        }

        if (sReset.bAssocUpdated == TRUE)
        {
            // Flag that we have events pending if
            // at least one association was updated
            psObj->sOTACtrl.bEventsPending = TRUE;
        }
    }
    else // Reset all art objects of a given type (channel / category)
    {
        BOOLEAN bChanAssocsAvailable,
                bOverlayChanAssocsAvailable,
                bCatAssocsAvailable;

        vStartTransaction( psObj->hChanSQLConnection,
                &psObj->bChanInTransaction );

        if (bCategory == TRUE)
        {
            // Indicate the category associations are no longer available
            snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                CA_UPDATE_CAT_ASSOC_AVAIL, FALSE );
            SQL_INTERFACE.bExecuteCommand(psObj->hChanSQLConnection,  &psObj->acBuffer[0]);
        }
        else
        {
            if (bOverlay == TRUE)
            {
                // Indicate the overlay channel associations are no longer available
                snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                    CA_UPDATE_OVERLAY_CHAN_ASSOC_AVAIL, FALSE );
                SQL_INTERFACE.bExecuteCommand(psObj->hChanSQLConnection,  &psObj->acBuffer[0]);
            }
            else
            {
                // Indicate the channel associations are no longer available
                snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
                    CA_UPDATE_CHAN_ASSOC_AVAIL, FALSE );
                SQL_INTERFACE.bExecuteCommand(psObj->hChanSQLConnection,  &psObj->acBuffer[0]);
            }

        }

        vEndTransaction(psObj->hChanSQLConnection,
                &psObj->bChanInTransaction);

        // Set the reset flag to ensure we have clean art objects 
        // before we start configuring them
        psObj->bResetArtObjects = TRUE;

        // Category associations are available in the database if
        // we are not resetting category associations now
        bCatAssocsAvailable = (bCategory == FALSE);

        // Channel associations are available if
        // category or we are resetting overlay
        bChanAssocsAvailable = (bCategory == TRUE) || (bOverlay == TRUE);

        // Overlay associations are available if we are resetting categories
        // or not reseting overlay
        bOverlayChanAssocsAvailable = (bCategory == TRUE) || (bOverlay == FALSE);

        // Process the database as we would when we start
        bSuccess = bChanProcessArtDatabase(psObj, bChanAssocsAvailable,
            bOverlayChanAssocsAvailable, bCatAssocsAvailable );

        if (bSuccess == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            CHANNEL_ART_MGR_DEFAULT_BY_TYPE_STRUCT sDefault;

            // Now that we've processed the database, we
            // no longer to reset the art objects
            psObj->bResetArtObjects = FALSE;

            // We may be "defaulting" some pre-existing associations
            // so we must populate our "set default" struct in order
            // to prepare for that situation
            sDefault.psObj = psObj;
            sDefault.bCategory = bCategory;
            sDefault.bOverlay = bOverlay;
            sDefault.bEventAdded = FALSE;

            // Iterate the list of association objects
            // now (we may have a bunch of associations
            // that don't exist in the default database)
            // to set all untouched associations of this
            // type to the default
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->psArtOwner->hChanAssociations,
                bDefaultAssociationsByType,
                (void *)&sDefault );
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error! Can't iterate the list!
                vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);

                bSuccess = FALSE;
            }
        }
    }

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return bSuccess;
}

/*****************************************************************************
*
*   bChannelImageUpdate
*
*   The interface uses this function to indicate an image update needs to
*   be processed.
*
*****************************************************************************/
static BOOLEAN bChannelImageUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t tImageDataByteLen,
    OSAL_BUFFER_HDL hImageData
        )
{
    BOOLEAN bOwner;
    BOOLEAN bSuccess = FALSE;

    // We should already be in the correct context...
    bOwner = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
    if (bOwner == TRUE)
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
            (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;
        BOOLEAN bSelected;

        // Do we want this image?
        bSelected = bImageTypeSelected(
            psObj,
            (CHANNEL_ART_IMAGETYPE_ENUM)psAttribRow->un8ImageType);
        if (bSelected == TRUE)
        {
            puts(CHANNEL_ART_MGR_OBJECT_NAME": Image Update");

            // Process the message
            bSuccess = bChanProcessImageUpdate(
                psObj, psAttribRow, tImageDataByteLen, hImageData );

            // End the transaction for this update
            vEndTransaction(psObj->hChanSQLConnection,
                    &psObj->bChanInTransaction);
        }
        else
        {
            // Not a problem
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}


/*****************************************************************************
*
*   bAlbumImageUpdate
*
*   The interface uses this function to indicate an image update needs to
*   be processed.
*
*****************************************************************************/
static BOOLEAN bAlbumImageUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hImageData,
    STRING_OBJECT *phCaption,
    CHANNEL_ART_OBJECT *phArtToUpdate
        )
{
    BOOLEAN bOk = FALSE;

    do 
    {
        CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;

        // We should already be in the correct context...
        bOk = DATASERVICE_IMPL_bOwner((DATASERVICE_IMPL_HDL)hChannelArtService);
        if ( FALSE == bOk )
        {
            break;
        }

        // Get the art service
        psObj = (CHANNEL_ART_MGR_OBJECT_STRUCT *)hChannelArtService;

        // Do we want album images ?
        bOk = bImageTypeSelected( psObj, CHANNEL_ART_IMAGETYPE_ALBUM );
        if ( FALSE == bOk )
        {
            // Note that we'll actually return TRUE in this case, as
            // this isn't a failure, per se. We just don't want album
            // art.
            bOk = TRUE;
            break;
        }

        // Process the message
        bOk = bAlbumProcessImageUpdate(
            psObj, psAssocRow, hImageData, phCaption, phArtToUpdate );

        if ( FALSE == bOk )
        {
            // We have the image but can't get it's ID?
            // Something's serious amiss here.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Album image update process failed!");
        }

    } while ( FALSE );

    return bOk;
}

/*****************************************************************************
*
*   bResetContentArtWithMatchingType
*
*   This linked list iterator updates all channel art objects utilizing
*   content art in order to switch them back to "static" art.
*
*****************************************************************************/
static BOOLEAN bResetContentArtWithMatchingType (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvData;
    CHANNEL_ART_MGR_RESET_ASSOC_STRUCT *psReset =
        (CHANNEL_ART_MGR_RESET_ASSOC_STRUCT *)pvArg;

    if (psReset->bOverlay == TRUE)
    {
        if ( (psAssoc->un16Source < GsChannelArtIntf.tOverlayStartServiceID) ||
             (psAssoc->un16Source > GsChannelArtIntf.tOverlayEndServiceID) )
        {
            // Keep iterating, not an overlay service ID
            return TRUE;
        }
    }
    else
    {
        // Not an overlay channel, if it is a base service ID (not category), check the range
        if ((psReset->bCategory == FALSE) &&
             (  psAssoc->un16Source > GsChannelArtIntf.tEndServiceID ||
                psAssoc->un16Source < GsChannelArtIntf.tStartServiceID))
        {
            // Keep iterating, not a base service ID
            return TRUE;
        }
    }

    if ((psAssoc->bCategory == psReset->bCategory))
    {
        BOOLEAN bAssocUpdated;

        // Reset this art object's contents
        bAssocUpdated = CHANNEL_ART_bReset(psAssoc->hChannelArt, TRUE);
        if (bAssocUpdated == TRUE)
        {
            // Mark this specific entry as updated
            psAssoc->bUpdated = TRUE;

            // Note that we have updated at least one
            // entry
            psReset->bAssocUpdated = TRUE;
        }
    }

    // Iterate all entries
    return TRUE;
}

/*****************************************************************************
*
*   bUpdateChannelArtWithMatchingImageAttrs
*
*   This linked list iterator updates all channel art objects which have
*   an image with matching attributes.
*
*****************************************************************************/
static BOOLEAN bUpdateChannelArtWithMatchingImageAttrs (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvData;
    CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT *psUpdate =
        (CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT *)pvArg;
    BOOLEAN bMatch;

    // Does this association utilize this image?
    bMatch = CHANNEL_ART_bHasImage(
        psAssoc->hChannelArt,
        psUpdate->psImageAttrs );

    // If this image matches the image in this association,
    // then update the event list
    if (bMatch == TRUE)
    {
        BOOLEAN bOk;
        BOOLEAN bImageCompatible = FALSE;
        CHANNEL_ART_ATTRIB_ROW_STRUCT sAssocImageAttribs;

        // Does the image version still match what the
        // association identifies?
        sAssocImageAttribs.un16ImageId = psUpdate->psImageAttrs->un16ImageId;
        sAssocImageAttribs.un8ImageType = psUpdate->psImageAttrs->un8ImageType;

        bOk = CHANNEL_ART_bGetImageAttributes(
            psAssoc->hChannelArt,
            CHANNEL_ART_IMAGE_ASSOC_TYPE_UNKNOWN,
            &sAssocImageAttribs, NULL);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": unable to query channel art object for image attribs");

            // Skip this entry -- keep looking at others
            return TRUE;
        }

        // Get the interface to compare these
        // two versions and check to see if this
        // association can utilize the image we found
        GsChannelArtIntf.bCompareImageVersions(
            psUpdate->psImageAttrs, &sAssocImageAttribs, &bImageCompatible);

        // Is the updated image still compatible with this association ?
        if (bImageCompatible == TRUE)
        {
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": %s %u Update Image: I%-5.5uT%-2.2u.%s\n",
                (psAssoc->bCategory == TRUE)?"Category ID":"Service ID",
                psAssoc->un16Source,
                psUpdate->psImageAttrs->un16ImageId,
                psUpdate->psImageAttrs->un8ImageType,
                (psUpdate->psImageAttrs->un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg");

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME": %s %u Update Image: I%-5.5uT%-2.2u.%s\n",
                (psAssoc->bCategory == TRUE)?"Category ID":"Service ID",
                psAssoc->un16Source,
                psUpdate->psImageAttrs->un16ImageId,
                psUpdate->psImageAttrs->un8ImageType,
                (psUpdate->psImageAttrs->un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg");

            // Update the image attributes to match
            // the new attribute row
            bOk = CHANNEL_ART_bUpdateImageAttributes (
                psAssoc->hChannelArt,
                psUpdate->psImageAttrs );
            if (bOk == TRUE)
            {
                psAssoc->bUpdated = TRUE;
                psUpdate->bEventAdded = TRUE;
            }
        }
        else
        {
            // This image is no longer compatible with this association.
            // We need to remove it from all entries in the
            // channel art object

            // Use this association structure for the following
            // iteration within the channel art object
            psUpdate->psAssoc = psAssoc;

            // Iterate the assocation types used by this
            // art object for this image
            CHANNEL_ART_bAssociationTypesForImage (
                psAssoc->hChannelArt,
                psUpdate->psImageAttrs,
                bIterateAssocTypesForImage, psUpdate);
        }

        if (bOk == TRUE)
        {
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": Art updated for %s: %u\n",
                (psAssoc->bCategory == TRUE)?"Category":"Channel",
                psAssoc->un16Source
                    );
        }
    }

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bIterateAssocTypesForImage
*
*   Iterates the associations used by a channel art object which reference
*   a given id/type
*
*****************************************************************************/
static BOOLEAN bIterateAssocTypesForImage (
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eAssocType,
    void *pvCallbackArg
        )
{
    CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT *psUpdate =
        (CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT *)pvCallbackArg;
    CHANNEL_ART_ASSOC_ROW_STRUCT sAssocRow;
    BOOLEAN bUpdated;

    // Populate the assoc row with pertinent information
    sAssocRow.bCategory = psUpdate->psAssoc->bCategory;
    sAssocRow.un16Source = psUpdate->psAssoc->un16Source;
    sAssocRow.un8ImageType = psUpdate->psImageAttrs->un8ImageType;

    // Which association do we update now?
    if (eAssocType == CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC)
    {
        sAssocRow.bContent = FALSE;
    }
    else
    {
        sAssocRow.bContent = TRUE;
    }

    // Default this association now
    bUpdated = bChanDefaultAssoc(
        psUpdate->psObj, psUpdate->psAssoc, &sAssocRow, TRUE, TRUE);
    if (bUpdated == TRUE)
    {
        psUpdate->psAssoc->bUpdated = TRUE;
        psUpdate->bEventAdded = TRUE;
    }

    // We must make sure that we are given the chance to process any
    // associations that use this image again because we may have
    // more work to do now (we don't want to leave them with the defult
    // in place)
    GsChannelArtIntf.vResetAssociationTracking(
            psUpdate->psObj->hChanInterface );

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bDefaultAssociationsByType
*
*   This linked list iterator is used to aid in the reset of the association
*   table.  All associations which have not already been updated are reset
*   to utilize the appropriate default.
*
*****************************************************************************/
static BOOLEAN bDefaultAssociationsByType (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvData;
    CHANNEL_ART_MGR_DEFAULT_BY_TYPE_STRUCT *psDefault =
        (CHANNEL_ART_MGR_DEFAULT_BY_TYPE_STRUCT *)pvArg;

    // Only update associations which have not yet
    // been affected
    if (psAssoc->bUpdated == TRUE)
    {
        // Move along...nothing to see here
        return TRUE;
    }

    // Only update assocations for the range of service IDs we are worried about
    if (psDefault->bOverlay == TRUE)
    {
        if ( (psAssoc->un16Source < GsChannelArtIntf.tOverlayStartServiceID) ||
             (psAssoc->un16Source > GsChannelArtIntf.tOverlayEndServiceID) )
        {
            // Keep iterating, not an overlay service ID
            return TRUE;
        }
    }
    else
    {
        // Not an overlay channel, if it is a base service ID (not category), check the range
        if ((psDefault->bCategory == FALSE) &&
             (  psAssoc->un16Source > GsChannelArtIntf.tEndServiceID ||
                psAssoc->un16Source < GsChannelArtIntf.tStartServiceID))
        {
            // Keep iterating, not a base service ID
            return TRUE;
        }
    }


    // Only default associations of the same type
    if (psAssoc->bCategory == psDefault->bCategory)
    {
        BOOLEAN bUpdated;
        CHANNEL_ART_OBJECT hDefaultArt;

        if (psAssoc->bCategory == TRUE)
        {
            hDefaultArt =
                psDefault->psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt;
        }
        else
        {
            hDefaultArt =
                psDefault->psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt;
        }

        // Update this art object with the default image(s)
        bUpdated = CHANNEL_ART_bCopyAllImages(
            psAssoc->hChannelArt, hDefaultArt);
        if (bUpdated == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Unable to copy images from default assoc");
        }
        psAssoc->bUpdated = bUpdated;
    }

    // Keep iteration going
    return TRUE;
}

/*****************************************************************************
*
*   bClearReferencedFlags
*
*   This linked list iterator is used to ensure that all associations which
*   weren't specified in an association message are forced to use the default
*
*****************************************************************************/
static BOOLEAN bClearReferencedFlags (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    void *pvUnused
        )
{
    if (psAssoc != NULL)
    {
        psAssoc->bReferenced = FALSE;
    }

    // Iterate the entire list
    return TRUE;
}

/*****************************************************************************
*
*   bDefaultLeftoverAssociations
*
*   This linked list iterator is used to ensure that all associations which
*   weren't specified in an association message are forced to use the default
*
*****************************************************************************/
static BOOLEAN bDefaultLeftoverAssociations (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_DEFAULT_LEFTOVER_STRUCT *psLeftover
        )
{
    BOOLEAN bUpdated;

    if (psLeftover == NULL)
    {
        // Error! Just Stop
        return FALSE;
    }

    // Odd, but just skip it
    if (psAssoc == NULL)
    {
        return TRUE;
    }

    // We don't need to default an association if:
    // 1) It was mentioned at in the association message, regardless
    // of whether or not it succeeded
    // 2) It doesn't match the association type (bCategory)
    if ((psAssoc->bCategory != psLeftover->bCategory) ||
        ((psAssoc->bReferenced == TRUE)))
    {
        return TRUE;
    }

    // Check source Ids now, but only for channels
    if (psLeftover->bCategory == FALSE)
    {
        // Verify we only update appropriate sources
        if (psLeftover->bOverlay == FALSE)
        {
            // Are we in range?
            if ((psAssoc->un16Source < GsChannelArtIntf.tStartServiceID) ||
                (psAssoc->un16Source > GsChannelArtIntf.tEndServiceID))
            {
                return TRUE;
            }
        }
        else
        {
            // Are we in range?
            if ((psAssoc->un16Source < GsChannelArtIntf.tOverlayStartServiceID) ||
                (psAssoc->un16Source > GsChannelArtIntf.tOverlayEndServiceID))
            {
                return TRUE;
            }
        }
    }

    // Now, set the image to the default appropriate
    // for its type
    bUpdated = CHANNEL_ART_bCopyAllImages(
        psAssoc->hChannelArt, psLeftover->hDefaultArt);
    if (bUpdated == TRUE)
    {
        // This association has been updated
        psAssoc->bUpdated = TRUE;
        (*psLeftover->pbEventGenerated) = TRUE;
    }

    // Keep iteration going
    return TRUE;
}

/*******************************************************************************
*
*   n16ChanCompareAssociations
*
*   This function compares two CHANNEL_ART_ASSOC_STRUCT and orders
*   them by type & source id.
*
*   Inputs:
*       psAssoc1 (in list), psAssoc2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16ChanCompareAssociations (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc1,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc2
        )
{
    if ((psAssoc1 == NULL) || (psAssoc2 == NULL))
    {
        return N16_MIN;
    }

    // Sort by type (channel, category) if they are different
    if (psAssoc1->bCategory != psAssoc2->bCategory)
    {
        // Channels come first
        if (psAssoc1->bCategory == FALSE)
        {
            return -1;
        }

        // Then categories
        return 1;
    }

    // Lower source Ids come first
    if (psAssoc1->un16Source < psAssoc2->un16Source)
    {
        return -1;
    }

    else if (psAssoc1->un16Source > psAssoc2->un16Source)
    {
        return 1;
    }

    return 0;
}

/*******************************************************************************
*
*   n16AlbumCompareAssociations
*
*   This function compares two ALBUM_ART_ASSOC_STRUCTs and orders
*   them by source id.
*
*   Inputs:
*       psAssoc1 (in list), psAssoc2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16AlbumCompareAssociations (
    ALBUM_ART_ASSOC_STRUCT *psAssoc1,
    ALBUM_ART_ASSOC_STRUCT *psAssoc2
        )
{
    // We want to make sure we're not dereferencing NULL pointers,
    // which is a little too exciting for the likes of us.
    if ( (psAssoc1 == NULL) ||
         (psAssoc2 == NULL) )
    {
        return N16_MIN;
    }

    // First, we sort by service ID association.
    return COMPARE(psAssoc1->tSID, psAssoc2->tSID);
}

/*******************************************************************************
*
*   n16CompareLineBitmaps
*
*   This function compares two IMAGE_BACKGROUND_GRAPHICS_STRUCT and orders
*   them by their line bitmap index.
*
*   Inputs:
*       pvArg1 (in list), pvArg2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareLineBitmaps (
    LINE_BITMAP_INFO_STRUCT *psLineBitmapInfo1,
    LINE_BITMAP_INFO_STRUCT *psLineBitmapInfo2
        )
{
    if (psLineBitmapInfo1->tLineBitmapIndex < psLineBitmapInfo2->tLineBitmapIndex)
    {
        // Lower indices come first
        return -1;
    }

    if (psLineBitmapInfo1->tLineBitmapIndex > psLineBitmapInfo2->tLineBitmapIndex)
    {
        // Higher indices come later
        return 1;
    }

    // Match
    return 0;
}

/*******************************************************************************
*
*   n16FindAssociationWithImageAttrs
*
*   Attempts to locate the first association with a given image id and type
*
*******************************************************************************/
static N16 n16FindAssociationWithImageAttrs (
    void *pvArg1,
    void *pvArg2
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvArg1;
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttrib =
        (CHANNEL_ART_ATTRIB_ROW_STRUCT *)(size_t)pvArg2;
    N16 n16Result = -1;
    BOOLEAN bMatch;

    // Ask the channel art object if this id
    // is a match for any of its images
    bMatch = CHANNEL_ART_bHasImage(
        psAssoc->hChannelArt,
        psAttrib );

    if (bMatch == TRUE)
    {
        n16Result = 0; // Found it!
    }

    return n16Result;
}

/*******************************************************************************
*
*   bChanInspectDBInfo
*
*     This function inspects the DB info table to ensure that the database is
*   the correct version for this version of the art manager code as well as
*   the status of the association tables
*
*******************************************************************************/
static BOOLEAN bChanInspectDBInfo (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbChanAssocsAvailable,
    BOOLEAN *pbOverlayChanAssocsAvailable,
    BOOLEAN *pbCatAssocsAvailable
        )
{
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    // Initialize result struct
    OSAL.bMemSet(&sQueryResult, 0, sizeof(CHANNEL_ART_DB_QUERY_RESULT_STRUCT));
    sQueryResult.psObj = NULL;
    sQueryResult.bResultantRows = FALSE;

    // Perform the SQL query and process the result
       bOk = SQL_INTERFACE.bQuery(
           hSQLConnection, CA_SELECT_DB_INFO,
           (SQL_QUERY_RESULT_HANDLER)bProcessSelectDBInfoResult,
           &sQueryResult ) ;

    if (bOk == TRUE)
    {
        if (sQueryResult.bResultantRows == TRUE)
        {
            CHANNEL_ART_DB_INFO_ROW_STRUCT *psInfoRow =
                &sQueryResult.uDbRow.sDBInfo;
            UN8 un8ActualDBVersion;

            // Grab the actual database version
            un8ActualDBVersion = (UN8)atoi(CHANNEL_ART_DATABASE_FILE_VERSION);

            // Verify all the fields
            if (   ( psInfoRow->un8DBVer == un8ActualDBVersion )
                && ( psInfoRow->un16DSI == GsChannelArtIntf.tDSI )
                )
            {
                // Version match! -- Report the status of the assoc table
                *pbChanAssocsAvailable = psInfoRow->bChanAssocsAvailable;
                *pbOverlayChanAssocsAvailable = psInfoRow->bOverlayChanAssocsAvailable;
                *pbCatAssocsAvailable = psInfoRow->bCatAssocsAvailable;

                return TRUE;
            }
            else
            {
                // Version mismatch!
                return FALSE;
            }
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   bChanUpdateAssociation
*
*   This function updates all aspects of a channel art association ( objects,
*   files, database entries) based upon the association row provided.
*
*******************************************************************************/
static BOOLEAN bChanUpdateAssociation (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psNewAssocRow,
    BOOLEAN *pbNewerVersion
        )
{
    CHANNEL_ART_ASSOC_ROW_STRUCT sStoredAssocRow;
    CHANNEL_ART_ATTRIB_ROW_STRUCT sStoredAttribRow;
    BOOLEAN bOk,
            bCreate = FALSE,
            bUpdate = FALSE,
            bAssocUpdated = FALSE;

    // Verify input
    if (pbNewerVersion == NULL)
    {
        return FALSE;
    }

    // Initialize the structures
    OSAL.bMemSet(&sStoredAssocRow, 0, sizeof(CHANNEL_ART_ASSOC_ROW_STRUCT));
    OSAL.bMemSet(&sStoredAttribRow, 0, sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT));

    // Initially indicate the provided
    // row is not a newer version
    *pbNewerVersion = FALSE;

    // Is this a lower-priority association
    // in a successive list for a particular chan/cat?
    if (psObj->sAssocCtrl.bLastAssocSucceeded == TRUE)
    {
        // Determine if this association is of the same
        // source, source type, and image type as the last one
        // we successfully processed
        BOOLEAN bAssocsEqual =
            CHANNEL_ART_ASSOC_TYPE_EQUAL(
                psNewAssocRow, &psObj->sAssocCtrl.sChanLastAssocRow );

        // Is this association attempting to do the
        // same thing as the last association (trying
        // to associate the same channel / category and
        // image type as the last entry did?
        if (bAssocsEqual == TRUE)
        {
            // Yes it is.  This means that the interface
            // gave us an association row with a
            // "lower priority" that what came before.
            // That association succeeded, so we don't
            // have to do this

            // This association was cut
            psObj->sAssocCtrl.bLastAssocSucceeded = FALSE;

            // We're all done -- nothing happened
            return FALSE;
        }
    }

    // Do we need to find the association in our list, or do we already have it?
    if ( (psObj->sAssocCtrl.psChanLastArtAssoc == NULL) ||
         ( ( (psObj->sAssocCtrl.psChanLastArtAssoc->bCategory !=
                psNewAssocRow->bCategory) ||
             (psObj->sAssocCtrl.psChanLastArtAssoc->un16Source !=
                psNewAssocRow->un16Source) ) ) )
    {
        // We need to find the association
        psObj->sAssocCtrl.psChanLastArtAssoc = psChanGetArtAssoc(
            psObj->psArtOwner, psNewAssocRow->bDefault,
            psNewAssocRow->bCategory, psNewAssocRow->un16Source);
        if (psObj->sAssocCtrl.psChanLastArtAssoc != NULL)
        {
            // If we found something, note that this
            // association has been referenced by the association message
            psObj->sAssocCtrl.psChanLastArtAssoc->bReferenced = TRUE;
        }
    }

    // Initialize this as not succeeded now
    psObj->sAssocCtrl.bLastAssocSucceeded = FALSE;

    // Were we told to remove this association?
    if (psNewAssocRow->bRemove == TRUE)
    {
        bAssocUpdated = bChanRemoveAssociation(psObj,
                psObj->sAssocCtrl.psChanLastArtAssoc, psNewAssocRow);

        // Save this association as our last
        psObj->sAssocCtrl.sChanLastAssocRow.bCategory =
            psNewAssocRow->bCategory;
        psObj->sAssocCtrl.sChanLastAssocRow.un16Source =
            psNewAssocRow->un16Source;
        psObj->sAssocCtrl.sChanLastAssocRow.un8ImageType =
            psNewAssocRow->un8ImageType;

        // This succeeded
        psObj->sAssocCtrl.bLastAssocSucceeded = TRUE;

        // Tell the caller if we did anything here
        return bAssocUpdated;
    }

    // Attempt to get an association row from
    // the database that matches the parameters
    // we just received
    bOk = bGetAssocRow (
        psObj, psObj->sAssocCtrl.psChanLastArtAssoc,
        &sStoredAssocRow,
        psNewAssocRow );

    if (bOk == TRUE)
    {
        BOOLEAN bAssocVersionEqual,
                bStoredAssocNeedsConfirmation = FALSE;

        // Get the interface to compare these
        // two versions
        bAssocVersionEqual =
            GsChannelArtIntf.bCompareAssociationVersions(
                &sStoredAssocRow, psNewAssocRow,
                &bStoredAssocNeedsConfirmation );

        // Verify the version in the database
        // is older than the new association version
        if (bAssocVersionEqual == FALSE)
        {
            // The new association is an updated version
            // of what we had previously -- so it needs to
            // be updated
            bUpdate = TRUE;
            *pbNewerVersion = TRUE;
        }
        // The version number is equal, but we need to
        // check that the association is confirmed as equal
        else if (TRUE == bStoredAssocNeedsConfirmation)
        {
            BOOLEAN bAssocsEqual =
                CHANNEL_ART_ASSOC_EQUAL(&sStoredAssocRow, psNewAssocRow);

            if (FALSE == bAssocsEqual)
            {
                // The association is an update of what we have already,
                // even though the associations appear to be equal
                // on the surface
                bUpdate = TRUE;
                *pbNewerVersion = TRUE;
            }
            else
            {
                CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType =
                    CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC;

                if (TRUE == psNewAssocRow->bContent)
                {
                    eImageAssocType = CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT;
                }

                // Mark this association as confirmed
                CHANNEL_ART_vConfirmAssociation(
                    psObj->sAssocCtrl.psChanLastArtAssoc->hChannelArt,
                    eImageAssocType,
                    psNewAssocRow->un8ImageType);

                DATASERVICE_IMPL_vLog(
                    CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u confirmed (Image:I%-5.5uT%-2.2u)\n",
                    (psNewAssocRow->bContent == TRUE)?"Dynamic":"Static",
                    (psNewAssocRow->bCategory == TRUE)?"category":"service id",
                    psNewAssocRow->un16Source, psNewAssocRow->un16ImageId,
                    psNewAssocRow->un8ImageType
                        );
            }
        }
    }
    // If the query had no results, then we need to store
    // a new row for this association
    else
    {
        bCreate = TRUE;

        // This is obviously a newer version if
        // none existed before
        *pbNewerVersion = TRUE;
    }

    // Only update the association if necessary
    if ((bUpdate == TRUE) || (bCreate == TRUE))
    {
        BOOLEAN bImageCompatible = FALSE;
        CHANNEL_ART_ATTRIB_ROW_STRUCT sAssocImageAttribs;

        // Search for the image in the database to ensure
        // the image identified in the association exists

        // Get the attribute row for this image id and type
        sStoredAttribRow.un16ImageId = psNewAssocRow->un16ImageId;
        sStoredAttribRow.un8ImageType = psNewAssocRow->un8ImageType;

        bOk = bGetAttribRow( psObj, psNewAssocRow, &sStoredAttribRow );
        if (bOk == FALSE)
        {
            // The image was not found --
            // we can not service this association
            // at this time
            return FALSE;
        }

        // Build an image row based upon the information
        // in the association row
        sAssocImageAttribs.un16ImageId = psNewAssocRow->un16ImageId;
        sAssocImageAttribs.un8ImageType = psNewAssocRow->un8ImageType;
        sAssocImageAttribs.un16ImageVer = psNewAssocRow->un16ImageVer;

        // Get the interface to compare these
        // two versions and check to see if this
        // association can utilize the image we found
        GsChannelArtIntf.bCompareImageVersions(
            &sStoredAttribRow, &sAssocImageAttribs, &bImageCompatible );

        if (bImageCompatible == FALSE)
        {
            // If these two aren't compatible, it means
            // we don't have the correct version of
            // this image with which to associate
            *pbNewerVersion = TRUE;

            return FALSE;
        }

        // Only update the database if
        // this is not a content-related association
        if (psNewAssocRow->bContent == FALSE)
        {
            SQL_PREPARED_STATEMENT_HANDLE hStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
            size_t tNumParams = 0;

            // We are updating a stored association
            if (bUpdate == TRUE)
            {
                hStmt = psObj->hUpdateAssocRowStmt;

                vSetUpdateAssocRowStmtParams(
                    psNewAssocRow, psObj->asBindParams);

                tNumParams = CA_UPDATE_ASSOC_ROW_STMT_PARAMS_COUNT;
            }
            else
            {
                hStmt = psObj->hInsertAssocRowStmt;

                vSetInsertAssocRowStmtParams(
                    psNewAssocRow, psObj->asBindParams);

                tNumParams = CA_INSERT_ASSOC_ROW_STMT_PARAMS_COUNT;
            }

            // Start an SQL transaction (if we haven't already)
            vStartTransaction( psObj->hChanSQLConnection,
                    &psObj->bChanInTransaction );

            // Update or insert the DB entry
            bOk = SQL_INTERFACE.bExecutePreparedStatement(
                psObj->hChanSQLConnection, hStmt, NULL, NULL,
                tNumParams, psObj->asBindParams);
        }

        if (bOk == TRUE)
        {
            // Update the association entry
            bOk = bChanUpdateAssociationEntry(
                psObj, &psObj->sAssocCtrl.psChanLastArtAssoc,
                psNewAssocRow, &sStoredAttribRow,
                &bAssocUpdated);

            if (bOk == TRUE)
            {
                // Save this association as our last
                psObj->sAssocCtrl.sChanLastAssocRow.bCategory =
                    psNewAssocRow->bCategory;
                psObj->sAssocCtrl.sChanLastAssocRow.un16Source =
                    psNewAssocRow->un16Source;
                psObj->sAssocCtrl.sChanLastAssocRow.un8ImageType =
                    psNewAssocRow->un8ImageType;

                if (bAssocUpdated == FALSE)
                {
                    // This isn't a newer version that
                    // anybody cares about
                    *pbNewerVersion = FALSE;
                }

                // This succeeded
                psObj->sAssocCtrl.bLastAssocSucceeded = TRUE;

                return bAssocUpdated;
            }
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   vSetUpdateAssocRowStmtParams
*
*******************************************************************************/
static void vSetUpdateAssocRowStmtParams (
    CHANNEL_ART_ASSOC_ROW_STRUCT const *psAssocRow,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        )
{
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_ID_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_ID_PARAM].pvData =
        (void*)(size_t)psAssocRow->un16ImageId;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_VER_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_VER_PARAM].pvData =
        (void*)(size_t)psAssocRow->un16ImageVer;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_ASSOC_VER_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_ASSOC_VER_PARAM].pvData =
        (void*)(size_t)psAssocRow->un8AssocVer;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_HI_PRIO_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_HI_PRIO_PARAM].pvData =
        (void*)(size_t)psAssocRow->bHighPriority;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_SRC_TYPE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_SRC_TYPE_PARAM].pvData =
        (void*)(size_t)psAssocRow->bCategory;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_SOURCE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_SOURCE_PARAM].pvData =
        (void*)((TRUE == psAssocRow->bDefault)
            ? (size_t)CHANNEL_ART_INVALID_SOURCE_ID
            : (size_t)psAssocRow->un16Source);

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM].pvData =
        (void*)(size_t)psAssocRow->un8ImageType;

    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IS_DEFAULT_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_UPDATE_ASSOC_ROW_STMT_IS_DEFAULT_PARAM].pvData =
        (void*)(size_t)psAssocRow->bDefault;

    return;
}

/*******************************************************************************
*
*   vSetInsertAssocRowStmtParams
*
*******************************************************************************/
static void vSetInsertAssocRowStmtParams (
    CHANNEL_ART_ASSOC_ROW_STRUCT const *psAssocRow,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        )
{
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_SRC_TYPE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_SRC_TYPE_PARAM].pvData =
        (void*)(size_t)psAssocRow->bCategory;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_SOURCE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_SOURCE_PARAM].pvData =
        (void*)(size_t)psAssocRow->un16Source;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM].pvData =
        (void*)(size_t)psAssocRow->un8ImageType;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_ID_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_ID_PARAM].pvData =
        (void*)(size_t)psAssocRow->un16ImageId;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_VER_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IMAGE_VER_PARAM].pvData =
        (void*)(size_t)psAssocRow->un16ImageVer;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IS_DEFAULT_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_IS_DEFAULT_PARAM].pvData =
        (void*)(size_t)psAssocRow->bDefault;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_ASSOC_VER_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_ASSOC_VER_PARAM].pvData =
        (void*)(size_t)psAssocRow->un8AssocVer;

    psBindParams[CA_INSERT_ASSOC_ROW_STMT_HI_PRIO_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psBindParams[CA_INSERT_ASSOC_ROW_STMT_HI_PRIO_PARAM].pvData =
        (void*)(size_t)psAssocRow->bHighPriority;

    return;
}

/*******************************************************************************
*
*   bChanUpdateAssociationEntry
*
*   This function updates an association entry found within the association
*   linked list based upon the provided arguments.  It will either update
*   a previously existing association entry or create a new one.
*
*******************************************************************************/
static BOOLEAN bChanUpdateAssociationEntry (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT **ppsAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    BOOLEAN *pbAssociationEntryUpdated
        )
{
    BOOLEAN bLocked, bSuccess = FALSE, bEntryUpdated = FALSE;

    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return FALSE;
    }

    if (pbAssociationEntryUpdated == NULL)
    {
        // Use local variable
        pbAssociationEntryUpdated = &bEntryUpdated;
    }

    // Nothing's been updated yet
    *pbAssociationEntryUpdated = FALSE;

    // If the association pointer is invalid and we are
    // working on a non-default assoc, we need to create a new
    // association and place it in the list
    if ((*ppsAssoc == NULL) && (psAssocRow->bDefault == FALSE))
    {
        // Add a new channel art association entry
        // to the association linked list
        *ppsAssoc = psChanAddNewAssocEntry(
                psObj->psArtOwner,
                psAssocRow->bCategory,
                psAssocRow->un16Source );

        if (*ppsAssoc != NULL)
        {
            // Create a channel art object for this association
            (*ppsAssoc)->hChannelArt =
                CHANNEL_ART_hCreate(
                    (SMS_OBJECT)psObj->psArtOwner,
                    &psObj->pacChannelArtServiceFilePath[0] );

            if ((*ppsAssoc)->hChannelArt == CHANNEL_ART_INVALID_OBJECT)
            {
                // Remove the linked list entry
                OSAL.eLinkedListRemove( (*ppsAssoc)->hEntry );
                (*ppsAssoc)->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                // Destroy the association node
                SMSO_vDestroy( (SMS_OBJECT) *ppsAssoc );
                *ppsAssoc = NULL;

                SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

                return FALSE;
            }

            // A new entry means we have an update
            *pbAssociationEntryUpdated = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                " bUpdateAssociationEntry(): Failed to create new assoc entry");

            SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
            return FALSE;
        }
    }

    // We should now have a valid association pointer.
    // If we don't there is an error
    if (*ppsAssoc != NULL)
    {
        BOOLEAN bArtUpdatedForReset = FALSE,
                bArtUpdated = FALSE;

        // If we're resetting our associations (or initializing them),
        // ensure that the channel art object is in an initial state
        if ((psObj->bResetArtObjects == TRUE) && ((*ppsAssoc)->bUpdated == FALSE))
        {
            // This function's return value indicates if anything
            // changed in the object due to this call.
            bArtUpdatedForReset = CHANNEL_ART_bReset((*ppsAssoc)->hChannelArt, FALSE);

            if (bArtUpdatedForReset == TRUE)
            {
                (*ppsAssoc)->bUpdated = TRUE;
            }
        }

        // Add the image to the channel art object
        bSuccess = CHANNEL_ART_bSetChannelImage(
            (*ppsAssoc)->hChannelArt,
            psAssocRow, psAttribRow, &bArtUpdated );

        if (bSuccess == TRUE)
        {
            if (bArtUpdated == TRUE)
            {
                (*ppsAssoc)->bUpdated = TRUE;
            }
        }

        if ((bArtUpdated == TRUE) || (bArtUpdatedForReset == TRUE))
        {
            *pbAssociationEntryUpdated = TRUE;
        }
    }

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    return bSuccess;
}

/*******************************************************************************
*
*   bChanRemoveAssociation
*
*   This function updates an association by removing a particular image type,
*   leaving this portion of the association empty.
*
*******************************************************************************/
static BOOLEAN bChanRemoveAssociation (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow
        )
{
    BOOLEAN bAssociationUpdated = FALSE;

    // Ensure we haven't been told to delete
    // a default association
    if (psAssocRow->bDefault == TRUE)
    {
        return FALSE;
    }

    if (psAssoc == NULL)
    {
        // Nothing to do
        return FALSE;
    }

    // Just make the association the default if it isn't already
    // (don't update the db here because we're gonna do it ourselves)
    bAssociationUpdated = bChanDefaultAssoc(
        psObj, psAssoc, psAssocRow, FALSE, TRUE);

    if ((bAssociationUpdated == TRUE) && (psAssocRow->bContent == FALSE))
    {
        // Remove the assoc line from the db

        // Generate the delete command
        snprintf( &psObj->acBuffer[0],
                  sizeof(psObj->acBuffer),
                  CA_DELETE_ASSOCIATION_ROW,
                  psAssocRow->bCategory,
                  psAssocRow->un16Source,
                  psAssocRow->un8ImageType );

        // Start an SQL transaction (if we haven't already)
        vStartTransaction( psObj->hChanSQLConnection,
                &psObj->bChanInTransaction );

        // Delete the DB entry
        SQL_INTERFACE.bExecuteCommand(
            psObj->hChanSQLConnection,
            &psObj->acBuffer[0] );
    }

    return bAssociationUpdated;
}

/*******************************************************************************
*
*   bChanDefaultAssoc
*
*******************************************************************************/
static BOOLEAN bChanDefaultAssoc (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    BOOLEAN bUpdateDB,
    BOOLEAN bOnlyReplaceExisting
        )
{
    BOOLEAN bUpdated = FALSE;

    do
    {
        CHANNEL_ART_OBJECT hDefaultArt;
        BOOLEAN bUsingDefault, bOk, bHasImage = FALSE;

        if (psAssoc == NULL)
        {
            break;
        }

        if (psAssocRow->bCategory == TRUE)
        {
            hDefaultArt =
                psObj->psArtOwner->sDefaultCategoryAssoc.hChannelArt;
        }
        else
        {
            hDefaultArt =
                psObj->psArtOwner->sDefaultChannelAssoc.hChannelArt;
        }

        // Is this channel art object already using the default?
        bUsingDefault = CHANNEL_ART_bImageSource(
            psAssoc->hChannelArt, hDefaultArt,
            psAssocRow->bContent,
            (CHANNEL_ART_IMAGETYPE_ENUM)psAssocRow->un8ImageType,
            &bHasImage);

        if (bUsingDefault == TRUE)
        {
            // Nothing to do
            break;
        }

        // Were we told to only replace an existing
        // association with the default?
        if (TRUE == bOnlyReplaceExisting)
        {
            // Yes! -- is there anything here?
            if (FALSE == bHasImage)
            {
                if (FALSE == psAssocRow->bContent)
                {
                    // Nope, turns out we have nothing to do here
                    break;
                }
                else
                {
                    bUsingDefault = CHANNEL_ART_bImageSource(
                        psAssoc->hChannelArt, hDefaultArt,
                        FALSE,
                        (CHANNEL_ART_IMAGETYPE_ENUM)psAssocRow->un8ImageType,
                        &bHasImage);
                    if ((TRUE == bUsingDefault) || (FALSE == bHasImage))
                    {
                        // Nothing to do
                        break;
                    }
                }
            }
        }

        // Make sure the default is placed in this spot
        bOk = CHANNEL_ART_bCopyImage(
            psAssoc->hChannelArt, hDefaultArt,
            psAssocRow->bContent,
            (CHANNEL_ART_IMAGETYPE_ENUM)
            psAssocRow->un8ImageType);
        if (bOk == FALSE)
        {
            break;
        }

        psAssoc->bUpdated = TRUE;
        bUpdated = TRUE;

        printf(
            CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u set to default for type %u\n",
            (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
            (psAssocRow->bCategory == TRUE)?"category":"service id",
            psAssocRow->un16Source, psAssocRow->un8ImageType
                );

        DATASERVICE_MGR_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": %s association for %s %u set to default for type %u\n",
            (psAssocRow->bContent == TRUE)?"Dynamic":"Static",
            (psAssocRow->bCategory == TRUE)?"category":"service id",
            psAssocRow->un16Source, psAssocRow->un8ImageType
                );

        // Only update the database if requested, and only if
        // this isn't a content-based association
        if ((bUpdateDB == TRUE) && (psAssocRow->bContent == FALSE))
        {
            // We need to update the DB to insert a direct
            // association between this source id and
            // the default
            CHANNEL_ART_ATTRIB_ROW_STRUCT sDefaultAttribs;
            CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eAssocType =
                CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC;
            CHANNEL_ART_ASSOC_ROW_STRUCT sUpdateAssocRow;

            OSAL.bMemSet(&sDefaultAttribs, 0,
                sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT));

            bOk = CHANNEL_ART_bGetImageAttributes(
                hDefaultArt,
                eAssocType,
                &sDefaultAttribs, NULL);
            if (bOk == FALSE)
            {
                break;
            }

            // Generate the update command to
            // have this source use the default
            // until we are able to actually
            // associate with something
            sUpdateAssocRow = *psAssocRow;
            sUpdateAssocRow.un16ImageId = sDefaultAttribs.un16ImageId;
            sUpdateAssocRow.un16ImageVer = sDefaultAttribs.un16ImageVer;
            sUpdateAssocRow.bHighPriority = FALSE;
            sUpdateAssocRow.bDefault = FALSE;
            vSetUpdateAssocRowStmtParams(
                &sUpdateAssocRow, psObj->asBindParams);

            // Start an SQL transaction (if we haven't already)
            vStartTransaction( psObj->hChanSQLConnection,
                    &psObj->bChanInTransaction );

            SQL_INTERFACE.bExecutePreparedStatement(
                psObj->hChanSQLConnection, psObj->hUpdateAssocRowStmt,
                NULL, NULL,
                CA_UPDATE_ASSOC_ROW_STMT_PARAMS_COUNT, psObj->asBindParams);
        }

    } while (FALSE);

    return bUpdated;
}

/*******************************************************************************
*
*   bProcessSelectDBInfoResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the database info relation.  It populates a
*   pointer to a CHANNEL_ART_DB_INFO_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectDBInfoResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    CHAN_DB_INFO_FIELDS_ENUM eCurrentField =
        (CHAN_DB_INFO_FIELDS_ENUM)0;
    CHANNEL_ART_DB_INFO_ROW_STRUCT *psInfoRow;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == CHAN_DB_INFO_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psInfoRow = &psResult->uDbRow.sDBInfo;

    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case CHAN_DB_INFO_FIELD_DB_VER:
            {
                psInfoRow->un8DBVer =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_INFO_FIELD_DSI:
            {
                psInfoRow->un16DSI =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_INFO_CHAN_ASSOC_AVAILABLE:
            {
                psInfoRow->bChanAssocsAvailable =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_INFO_OVERLAY_CHAN_ASSOC_AVAILABLE:
            {
                psInfoRow->bOverlayChanAssocsAvailable =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_INFO_CAT_ASSOC_AVAILABLE:
            {
                psInfoRow->bCatAssocsAvailable =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_INFO_DATA_CONTENT_VERSION:
            {
                psInfoRow->un16DataContentVersion =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < CHAN_DB_INFO_MAX_FIELDS);

    // We only want one row of data
    return FALSE;
}

// This function may be useful at some later date...
#if 0
/*******************************************************************************
*
*   bProcessSelectImageAssoc
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on image association relation.  It populates a
*   CHANNEL_ART_ASSOC_ROW_STRUCT based upon the information retrieved
*   from the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectImageAssoc (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    IMAGE_ASSOCIATION_FIELDS_ENUM eCurrentField =
        (IMAGE_ASSOCIATION_FIELDS_ENUM)0;
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssoc;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there is the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == IMAGE_ASSOC_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psAssoc = (CHANNEL_ART_ASSOC_ROW_STRUCT *)&psResult->uDbRow.sAssoc;

    // Clear the association structure
    OSAL.bMemSet(psAssoc, 0, sizeof(CHANNEL_ART_ASSOC_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the aspects of a particular image association.
    // Process the results (just the first row returned)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case IMAGE_ASSOC_FIELD_SRCTYPE:
            {
                psAssoc->bCategory =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_SRC:
            {
                psAssoc->un16Source =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_IMAGE_TYPE:
            {
                psAssoc->un8ImageType =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }

            case IMAGE_ASSOC_FIELD_IMAGE_ID:
            {
                psAssoc->un16ImageId =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_IMAGE_VER:
            {
                psAssoc->un16ImageVer =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_IMAGE_DEFAULT:
            {
                psAssoc->bDefault =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_ASSOC_VER:
            {
                psAssoc->un8AssocVer =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case IMAGE_ASSOC_FIELD_HIGH_PRIO:
            {
                psAssoc->bHighPriority =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < IMAGE_ASSOC_MAX_FIELDS);

    // We only want one row
    return FALSE;
}
#endif

/*******************************************************************************
*
*   bProcessSelectImageAttrs
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on the image attribute relation.  It populates a
*   pointer to a CHANNEL_ART_ATTRIB_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectImageAttrs (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    CHAN_DB_IMAGE_ATTRIB_FIELDS_ENUM eCurrentField =
        (CHAN_DB_IMAGE_ATTRIB_FIELDS_ENUM)0;
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttrs;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there is the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == CHAN_DB_IMAGE_ATTRIB_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psImageAttrs = (CHANNEL_ART_ATTRIB_ROW_STRUCT *)&psResult->uDbRow.sAttrib;

    // Clear the attribute structure
    OSAL.bMemSet(psImageAttrs, 0, sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT));

    // We have just performed an SQL query in order to
    // determine the aspects of a particular image's attributes.
    // Process the results (just the first row returned)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_ID:
            {
                psImageAttrs->un16ImageId =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_TYPE:
            {
                psImageAttrs->un8ImageType =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_VER:
            {
                psImageAttrs->un16ImageVer =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_FIELD_CODE_TYPE:
            {
                psImageAttrs->un8CodeType =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_BACKGROUND_OPTIONS:
            {
                psImageAttrs->sBackgroundGfx.tBackgroundOptions =
                    (CHANNEL_ART_BACKGROUND_OPTION_MASK)
                        psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_LINE_BITMAP_INDEX:
            {
                psImageAttrs->sBackgroundGfx.sLineBitmap.tLineBitmapIndex =
                    (size_t)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_BACKGROUND_COLOR:
            {
                psImageAttrs->sBackgroundGfx.un32BackgroundColor =
                    (UN32)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_ATTRIB_SECONDARY_PRESENT:
            {
                // Supported if indicated and the service currently supports it
                psImageAttrs->bSecondaryAvailable =
                    ((BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data)
                        & psResult->psObj->bSecondarySupported;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < CHAN_DB_IMAGE_ATTRIB_MAX_FIELDS);

    // We only want one row
    return FALSE;
}

/*******************************************************************************
*
*   bProcessSelectLineBitmap
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on the line bitmap relation.  It populates a
*   pointer to a LINE_BITMAP_INFO_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bProcessSelectLineBitmap (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    LINE_BITMAP_FIELDS_ENUM eCurrentField =
        (LINE_BITMAP_FIELDS_ENUM)0;
    LINE_BITMAP_INFO_STRUCT *psLineBitmapInfo;

    // Verify input
    if (psResult == NULL)
    {
        return FALSE;
    }

    // If there the correct
    // number of columns, then we have good results
    if ( n32NumberOfColumns == CHAN_DB_LINE_BITMAP_MAX_FIELDS )
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psLineBitmapInfo = &psResult->uDbRow.sAttrib.sBackgroundGfx.sLineBitmap;

    // Clear the line bitmap fields
    psLineBitmapInfo->pun8LineBitmap = NULL;
    psLineBitmapInfo->tLineBitmapIndex = 0;
    psLineBitmapInfo->tLineBitmapByteSize = 0;

    // We have just performed an SQL query in order to
    // determine the aspects of a particular line bitmap.
    // Process the results (just the first row returned)
    do
    {
        // Decode the current field based on it's type
        switch (eCurrentField)
        {
            case CHAN_DB_LINE_BITMAP_FIELD_ID:
            {
                psLineBitmapInfo->tLineBitmapIndex =
                    (size_t)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_LINE_BITMAP_FIELD_SIZE:
            {
                psLineBitmapInfo->tLineBitmapByteSize =
                    (size_t)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_LINE_BITMAP_FIELD_DATA:
            {
                char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

                // Create a name for this line bitmap
                snprintf( &acName[0], sizeof(acName),
                          "%s line bitmap %u",
                          CHANNEL_ART_MGR_OBJECT_NAME,
                          psLineBitmapInfo->tLineBitmapIndex );

                // Allocate space for the line bitmap
                psLineBitmapInfo->pun8LineBitmap = (UN8 *)
                     SMSO_hCreate(
                        &acName[0],
                        psLineBitmapInfo->tLineBitmapByteSize,
                        SMS_INVALID_OBJECT, FALSE );

                if (psLineBitmapInfo->pun8LineBitmap == NULL)
                {
                    psResult->bResultantRows = FALSE;
                    break;
                }

                // Copy the bitmap data
                psResult->bResultantRows = OSAL.bMemCpy(
                    (void *)&psLineBitmapInfo->pun8LineBitmap[0],
                    psColumn[eCurrentField].uData.sBLOB.pvData,
                    psColumn[eCurrentField].uData.sBLOB.tSize );

                if (psResult->bResultantRows == FALSE)
                {
                    SMSO_vDestroy((SMS_OBJECT)psLineBitmapInfo->pun8LineBitmap);
                    psLineBitmapInfo->pun8LineBitmap = NULL;
                }
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < CHAN_DB_LINE_BITMAP_MAX_FIELDS);

    // We only want one row
    return FALSE;
}

/*******************************************************************************
*
*   bChanProcessJoinDB
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on a natural join of the image association and the
*   the image attribute relations.  The results are used to create all the
*   initial associations & object required for this service to operate.
*
*******************************************************************************/
static BOOLEAN bChanProcessJoinDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT *psResult
        )
{
    CHANNEL_ART_ASSOC_ROW_STRUCT sStoredAssoc;
    CHANNEL_ART_ATTRIB_ROW_STRUCT sStoredAttribs;
    CHAN_DB_IMAGE_JOIN_FIELDS_ENUM eCurrentField;

    OSAL.bMemSet(&sStoredAssoc, 0, sizeof(CHANNEL_ART_ASSOC_ROW_STRUCT) );
    OSAL.bMemSet(&sStoredAttribs, 0, sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT) );

    // Each of these association versions must be confirmed by OTA processing
    sStoredAssoc.bAssocVerConfirmed = FALSE;

    if ( n32NumberOfColumns == CHAN_DB_IMAGE_JOIN_MAX_FIELDS )
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;

        return FALSE;
    }

    eCurrentField = (CHAN_DB_IMAGE_JOIN_FIELDS_ENUM)0;

    do
    {
        // Decode the current field based on its type
        switch (eCurrentField)
        {
            case CHAN_DB_IMAGE_JOIN_FIELD_SRCTYPE:
            {
                sStoredAssoc.bCategory =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_SRC:
            {
                sStoredAssoc.un16Source =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_TYPE:
            {
                sStoredAttribs.un8ImageType =
                  sStoredAssoc.un8ImageType =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_ID:
            {
                sStoredAssoc.un16ImageId =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;

                sStoredAttribs.un16ImageId =
                    sStoredAssoc.un16ImageId;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_ASSOC_IMAGE_VER:
            {
                sStoredAssoc.un16ImageVer =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_DEFAULT:
            {
                sStoredAssoc.bDefault =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_ASSOC_VER:
            {
                sStoredAssoc.un8AssocVer =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_HIGH_PRIO_ASSOC:
            {
                sStoredAssoc.bHighPriority =
                    (BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_VER:
            {
                sStoredAttribs.un16ImageVer =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_FIELD_CODE_TYPE:
            {
                sStoredAttribs.un8CodeType =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_BACKGROUND_OPTIONS:
            {
                sStoredAttribs.sBackgroundGfx.tBackgroundOptions =
                    (CHANNEL_ART_BACKGROUND_OPTION_MASK)
                        psColumn[eCurrentField].uData.sUN32.un32Data;

                // Do the background options indicate that we
                // need to retrieve a line bitmap?
                if ((sStoredAttribs.sBackgroundGfx.tBackgroundOptions
                         & CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP) ==
                             CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP)
                {
                    // Yes, get the line bitmap
                    bGetLineBitmap(
                        psResult->psObj,
                        &sStoredAttribs.sBackgroundGfx.sLineBitmap);
                }
            }
            break;

            case CHAN_DB_IMAGE_JOIN_LINE_BITMAP_INDEX:
            {
                sStoredAttribs.sBackgroundGfx.sLineBitmap.tLineBitmapIndex =
                    (size_t)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_BACKGROUND_COLOR:
            {
                sStoredAttribs.sBackgroundGfx.un32BackgroundColor =
                    (UN32)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case CHAN_DB_IMAGE_JOIN_SECONDARY_PRESENT:
            {
                // Supported if indicated and the service currently supports it
                sStoredAttribs.bSecondaryAvailable =
                    ((BOOLEAN)psColumn[eCurrentField].uData.sUN32.un32Data) & psResult->psObj->bSecondarySupported;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < CHAN_DB_IMAGE_JOIN_MAX_FIELDS );

    // We're building our associations now, so whenever we get a new channel
    // or category or default it means we're learning about something new
    if ( (psResult->psObj->sAssocCtrl.psChanLastArtAssoc->bDefault
            != sStoredAssoc.bDefault) ||
         (psResult->psObj->sAssocCtrl.psChanLastArtAssoc->bCategory
            != sStoredAssoc.bCategory) ||
         (psResult->psObj->sAssocCtrl.psChanLastArtAssoc->un16Source
            != sStoredAssoc.un16Source) )
    {
        if (sStoredAssoc.bDefault == TRUE)
        {
            // Get the default entry since we always have these
            psResult->psObj->sAssocCtrl.psChanLastArtAssoc = 
                psChanGetArtAssoc(psResult->psObj->psArtOwner, TRUE,
                    sStoredAssoc.bCategory, sStoredAssoc.un16Source);
        }
        else
        {
            // We know our last assoc won't match this one, so clear it now
            psResult->psObj->sAssocCtrl.psChanLastArtAssoc = NULL;
        }
    }

    // Update the channel art's association entry
    // for the association provided here
    psResult->bSuccess = bChanUpdateAssociationEntry (
        psResult->psObj, 
        &psResult->psObj->sAssocCtrl.psChanLastArtAssoc, 
        &sStoredAssoc, &sStoredAttribs, NULL);

    // If that worked and we're reading the default channel info,
    // make sure we add it to that channel art object's default
    // content-association as well
    if ((psResult->bSuccess == TRUE) &&
        (sStoredAssoc.bDefault == TRUE) &&
        (sStoredAssoc.bCategory == FALSE))
    {
        // Add a content-based default image as well
        sStoredAssoc.bContent = TRUE;

        psResult->bSuccess = bChanUpdateAssociationEntry (
            psResult->psObj, 
            &psResult->psObj->sAssocCtrl.psChanLastArtAssoc, 
            &sStoredAssoc, &sStoredAttribs, NULL);
    }

    // If we couldn't do that, stop cause there was an error
    if (psResult->bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            " Pulling assocs from DB unsuccessful.");
        return FALSE;
    }

    printf(CHANNEL_ART_MGR_OBJECT_NAME
        ": Assoc Found in DB: %s %s ID:%u Image I%-5.5uT%-2.2u.%s (ver:%u)\n",
        (sStoredAssoc.bDefault == TRUE)?"*Default*":"",
        (sStoredAssoc.bCategory == TRUE)?"Category":"Service",
        sStoredAssoc.un16Source,
        sStoredAssoc.un16ImageId,
        sStoredAssoc.un8ImageType,
        (sStoredAttribs.un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
        sStoredAttribs.un16ImageVer);

    DATASERVICE_IMPL_vLog(CHANNEL_ART_MGR_OBJECT_NAME
        ": Assoc Found in DB: %s %s ID:%u Image I%-5.5uT%-2.2u.%s (ver:%u)\n",
        (sStoredAssoc.bDefault == TRUE)?"*Default*":"",
        (sStoredAssoc.bCategory == TRUE)?"Category":"Service",
        sStoredAssoc.un16Source,
        sStoredAssoc.un16ImageId,
        sStoredAssoc.un8ImageType,
        (sStoredAttribs.un8CodeType == IMAGE_FORMAT_PNG)?"png":"jpg",
        sStoredAttribs.un16ImageVer);

    return TRUE;
}

/*******************************************************************************
*
*   bGetAssocRowFromList
*
*   This function retrieves an association row from memory based
*   upon the parameters provided and populates a CHANNEL_ART_ASSOC_ROW_STRUCT
*   with the result.
*
*******************************************************************************/
static BOOLEAN bGetAssocRowFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psSearchParameters
        )
{
    BOOLEAN bLocked;
    BOOLEAN bOk = FALSE;

    if (psAssoc == NULL)
    {
        // Nothing to do here
        return FALSE;
    }

    // Lock the art owner
    bLocked =
        SMSO_bLock((SMS_OBJECT)psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType =
            CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC;

        if (psSearchParameters->bContent == TRUE)
        {
            eImageAssocType = CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT;
        }

        // Get the association version data
        bOk = CHANNEL_ART_bGetAssocAttributes(
            psAssoc->hChannelArt,
            eImageAssocType,
            psSearchParameters->un8ImageType,
            psAssocRow);

        // If we were able to get the image attributes
        // from the channel art object then then we can
        // give this data to the caller
        if (bOk == TRUE)
        {
            // This image exists so now copy the
            // results out to the caller's pointer

            // These are the parameters we matched
            // in our search
            psAssocRow->bCategory = psSearchParameters->bCategory;
            psAssocRow->bDefault = psSearchParameters->bDefault;
            psAssocRow->un16Source = psSearchParameters->un16Source;
            psAssocRow->un16ImageId = psSearchParameters->un16ImageId;
            psAssocRow->un8ImageType = psSearchParameters->un8ImageType;
        }

        SMSO_vUnlock((SMS_OBJECT)psArtOwner);
    }

    return bOk;
}

/*******************************************************************************
*
*   psChanGetArtAssoc
*
*******************************************************************************/
static CHANNEL_ART_ASSOC_STRUCT *psChanGetArtAssoc (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bDefault,
    BOOLEAN bCategory,
    UN16 un16Source
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    // Just grab the default association if we're
    // looking for a default
    if (bDefault == TRUE)
    {
        if (bCategory == TRUE)
        {
            psAssoc = &psArtOwner->sDefaultCategoryAssoc;
        }
        else
        {
            psAssoc = &psArtOwner->sDefaultChannelAssoc;
        }
    }
    else // Otherwise, search for the association
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        CHANNEL_ART_ASSOC_STRUCT sSearch;

        sSearch.bCategory = bCategory;
        sSearch.un16Source = un16Source;

        // Search the linked list for this association
        // using the sorted search
        eReturnCode =
            OSAL.eLinkedListSearch (
                psArtOwner->hChanAssociations,
                &hEntry,
                (void *)&sSearch );

        if (eReturnCode == OSAL_SUCCESS)
        {
            // Extract the association from memory
            psAssoc = (CHANNEL_ART_ASSOC_STRUCT *)
                OSAL.pvLinkedListThis( hEntry );
        }
    }

    return psAssoc;
}

/*******************************************************************************
*
*   bGetAssocRow
*
*******************************************************************************/
static BOOLEAN bGetAssocRow (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psSearchParameters
        )
{
    BOOLEAN bFound;

    // Attempt to pull the assoc row from
    // the association list
    bFound = bGetAssocRowFromList(psObj->psArtOwner, psAssoc, psAssocRow, psSearchParameters);
    if (bFound == TRUE)
    {
        // Indicate that we found this in memory
        printf(
            CHANNEL_ART_MGR_OBJECT_NAME": Pulled assoc (s:%u i:%u) from memory\n",
            psSearchParameters->un16Source,
            psSearchParameters->un16ImageId);
    }

    /*
       Useful to keep this here for posterity.  But why take this out?
       Because at startup we read all associations and copied them into
       memory.  So, we already have all associations available to us.
       Invoking the code below is just a waste of time.
    */
#if 0
    else if (psSearchParameters->bContent == FALSE)
    {
        // Now we have to take a look at the database (but only if we're
        // looking for a "static" association
        bFound = bGetAssocRowFromDB(psObj, psAssocRow, psSearchParameters);
        if (bFound == TRUE)
        {
            // Indicate that we found this in the DB
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": Pulled assoc (s:%u i:%u) from DB\n",
                psSearchParameters->un16Source, psSearchParameters->un16ImageId);
        }
    }
#endif

    return bFound;
}

/*******************************************************************************
*
*   bGetAttribRowFromList
*
*******************************************************************************/
static BOOLEAN bGetAttribRowFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psRelatedAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        )
{
    BOOLEAN bLocked;
    BOOLEAN bImageCopied = FALSE, bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    CHANNEL_ART_ASSOC_STRUCT *psAssoc;
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eAssocType =
        CHANNEL_ART_IMAGE_ASSOC_TYPE_UNKNOWN;

    bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        return FALSE;
    }

    // Is there an associated assoc row to help our search?
    if (psRelatedAssocRow != NULL)
    {
        // Yeah, determine if this is a content assoc or not
        if (psRelatedAssocRow->bContent == TRUE)
        {
            eAssocType = CHANNEL_ART_IMAGE_ASSOC_TYPE_CONTENT;
        }
        else
        {
            eAssocType = CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC;
        }
    }

    // Search the linked list for this image's attributes
    // via any association that uses it
    eReturnCode =
        OSAL.eLinkedListLinearSearch (
            psObj->hChanAssociations,
            &hEntry,
            n16FindAssociationWithImageAttrs,
            (void *)(size_t)psAttribRow );

    if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        // Check the defaults
        psAssoc = &psObj->sDefaultCategoryAssoc;

        // Is this image a part of the default category assoc?
        bOk = CHANNEL_ART_bHasImage(
            psAssoc->hChannelArt,
            psAttribRow );

        if (bOk == TRUE)
        {
            // Okay, get this image's attributes
            bOk = CHANNEL_ART_bGetImageAttributes(
                psAssoc->hChannelArt,
                CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC,
                psAttribRow, &bImageCopied );
        }
        else
        {
            // Check the default channel assoc
            psAssoc = &psObj->sDefaultChannelAssoc;

            // Is this image a part of the default channel assoc?
            bOk = CHANNEL_ART_bHasImage(
                psAssoc->hChannelArt,
                psAttribRow );

            if (bOk == TRUE)
            {
                // Okay, get this image's attributes
                bOk = CHANNEL_ART_bGetImageAttributes(
                    psAssoc->hChannelArt,
                    CHANNEL_ART_IMAGE_ASSOC_TYPE_STATIC,
                    psAttribRow, &bImageCopied );
            }
        }
    }
    // If an association was found, attempt
    // to extract the image data
    else if (eReturnCode == OSAL_SUCCESS)
    {
        // Extract the association structure
        psAssoc = (CHANNEL_ART_ASSOC_STRUCT *)
            OSAL.pvLinkedListThis( hEntry );

        // Get the image attributes
        bOk = CHANNEL_ART_bGetImageAttributes(
            psAssoc->hChannelArt,
            eAssocType, psAttribRow,
            &bImageCopied );
    }

    // If the attributes we received came not
    // from the database but from another
    // channel art object (a copy), then we don't
    // want to use this data for lookups --
    // force us to attempt the database
    if (bImageCopied == TRUE)
    {
        bOk = FALSE;
    }

    SMSO_vUnlock((SMS_OBJECT)psObj);

    return bOk;
}

/*******************************************************************************
*
*   bGetAttribRowFromDB
*
*******************************************************************************/
static BOOLEAN bGetAttribRowFromDB (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        )
{
    BOOLEAN bOk;
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT sQueryResult;

    // Initialize result struct
    OSAL.bMemSet(&sQueryResult, 0, sizeof(CHANNEL_ART_DB_QUERY_RESULT_STRUCT));
    sQueryResult.psObj = psObj;
    sQueryResult.bResultantRows = FALSE;

    // Generate our query request based upon the association
    // row passed in as an argument

    psObj->asBindParams[CA_SELECT_IMAGE_ATTR_STMT_IMAGE_ID_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psObj->asBindParams[CA_SELECT_IMAGE_ATTR_STMT_IMAGE_ID_PARAM].pvData =
        (void*)(size_t)psAttribRow->un16ImageId;

    psObj->asBindParams[CA_SELECT_IMAGE_ATTR_STMT_IMAGE_TYPE_PARAM].eType =
        SQL_BIND_TYPE_UN32;
    psObj->asBindParams[CA_SELECT_IMAGE_ATTR_STMT_IMAGE_TYPE_PARAM].pvData =
        (void*)(size_t)psAttribRow->un8ImageType;

    // Perform the SQL query and process the result
    bOk = SQL_INTERFACE.bExecutePreparedStatement(
        psObj->hChanSQLConnection,
        psObj->hSelectImageAttrsStmt,
        (SQL_QUERY_RESULT_HANDLER)bProcessSelectImageAttrs,
        &sQueryResult,
        CA_SELECT_IMAGE_ATTR_STMT_PARAMS_COUNT,
        psObj->asBindParams);

    if (bOk == TRUE)
    {
        // If the query yielded results, then we
        // found our row -- this means the image
        // is present in the system
        if (sQueryResult.bResultantRows == TRUE)
        {
            // Grab the line bitmap if necessary
            if ((sQueryResult.uDbRow.sAttrib.sBackgroundGfx.tBackgroundOptions
                    & CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP) != 0)
            {
                // We need to get the line bitmap as well
                bOk = bGetLineBitmap( psObj,
                    &sQueryResult.uDbRow.sAttrib.sBackgroundGfx.sLineBitmap);
            }

            if (bOk == TRUE)
            {
                // Copy the results out to the caller's
                // pointer argument
                bOk = OSAL.bMemCpy(
                    psAttribRow,
                    &sQueryResult.uDbRow.sAttrib,
                    sizeof(CHANNEL_ART_ATTRIB_ROW_STRUCT) );
            }
        }
        else
        {
            bOk = FALSE;
        }
    }

    return bOk;
}

/*******************************************************************************
*
*   bGetAttribRow
*
*******************************************************************************/
static BOOLEAN bGetAttribRow (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psRelatedAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        )
{
    BOOLEAN bFound;

    // Attempt to pull the attrib row from
    // the association list
    bFound = bGetAttribRowFromList(psObj->psArtOwner, psRelatedAssocRow, psAttribRow);
    if (bFound == TRUE)
    {
        // Indicate that we found this in memory
        printf(
            CHANNEL_ART_MGR_OBJECT_NAME": Pulled attrib (i:%u t:%u) from memory\n",
            psAttribRow->un16ImageId,
            psAttribRow->un8ImageType);
    }
    else
    {
        // Now we have to take a look at the database
        bFound = bGetAttribRowFromDB(psObj, psAttribRow);
        if (bFound == TRUE)
        {
            // Indicate that we found this in the DB
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": Pulled attrib (i:%u, t:%u) from DB\n",
                psAttribRow->un16ImageId,
                psAttribRow->un8ImageType);
        }
    }

    return bFound;
}

/*******************************************************************************
*
*   bGetLineBitmapFromList
*
*******************************************************************************/
static BOOLEAN bGetLineBitmapFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // Lock the art owner
    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        LINE_BITMAP_INFO_STRUCT *psStoredLineInfo = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Build the line bitmap list if it doesn't exist
        if (psObj->hLineBitmaps == OSAL_INVALID_OBJECT_HDL)
        {
            // Build the line bitmap list
            eReturnCode = OSAL.eLinkedListCreate (
                &psObj->hLineBitmaps,
                CHANNEL_ART_MGR_OBJECT_NAME": Linebitmaps",
                (OSAL_LL_COMPARE_HANDLER)n16CompareLineBitmaps, // Sort by bitmap Id
                OSAL_LL_OPTION_NONE
                    );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Ensure the handle is invalid
                psObj->hLineBitmaps = OSAL_INVALID_OBJECT_HDL;
            }
        }

        if (psObj->hLineBitmaps != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_LINKED_LIST_ENTRY hEntry =
                OSAL_INVALID_LINKED_LIST_ENTRY;

            // Use configured search
            eReturnCode = OSAL.eLinkedListSearch(
                psObj->hLineBitmaps,
                &hEntry,
                (void *)psLineInfo);

            if (eReturnCode == OSAL_SUCCESS)
            {
                // Grab the background info from the linked list
                psStoredLineInfo = (LINE_BITMAP_INFO_STRUCT *)
                    OSAL.pvLinkedListThis(hEntry);
            }
        }

        if (psStoredLineInfo != NULL)
        {
            psLineInfo->pun8LineBitmap = psStoredLineInfo->pun8LineBitmap;
            psLineInfo->tLineBitmapByteSize = psStoredLineInfo->tLineBitmapByteSize;
            bSuccess = TRUE;
        }

        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bGetLineBitmapFromDB
*
*******************************************************************************/
static BOOLEAN bGetLineBitmapFromDB (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        )
{
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    // Initialize result struct
    OSAL.bMemSet(&sQueryResult, 0, sizeof(CHANNEL_ART_DB_QUERY_RESULT_STRUCT));
    sQueryResult.psObj = psObj;
    sQueryResult.bResultantRows = FALSE;

    // Generate our query request
    snprintf( &psObj->acBuffer[0], sizeof(psObj->acBuffer),
              CA_SELECT_LINE_BITMAP,
              psLineInfo->tLineBitmapIndex );

    // Perform the SQL query and process the result
    bOk = SQL_INTERFACE.bQuery(
            psObj->hChanSQLConnection, &psObj->acBuffer[0],
            (SQL_QUERY_RESULT_HANDLER)bProcessSelectLineBitmap,
            &sQueryResult ) ;

    if (bOk == TRUE)
    {
        // If the query yielded results, then we
        // found our row -- this means the line
        // bitmap is present in the system
        if (sQueryResult.bResultantRows == TRUE)
        {
            // Add the entry to the linked list
            vAddNewLineBitmapEntry(psObj,
                &sQueryResult.uDbRow.sAttrib.sBackgroundGfx.sLineBitmap);

            psLineInfo->pun8LineBitmap =
                sQueryResult.uDbRow.sAttrib.sBackgroundGfx.sLineBitmap.pun8LineBitmap;
            psLineInfo->tLineBitmapByteSize =
                sQueryResult.uDbRow.sAttrib.sBackgroundGfx.sLineBitmap.tLineBitmapByteSize;

            return TRUE;
        }
    }

    return FALSE;
}

/*******************************************************************************
*
*   bGetLineBitmap
*
*   This function retrieves an line bitmap either from memory or from
*   in the database based upon the parameters provided and populates a
*   LINE_BITMAP_INFO_STRUCT *with the result.
*
*******************************************************************************/
static BOOLEAN bGetLineBitmap (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        )
{
    BOOLEAN bFound;

    // Attempt to pull the line bitmap from
    // the association list
    bFound = bGetLineBitmapFromList(psObj->psArtOwner, psLineInfo);
    if (bFound == TRUE)
    {
        // Indicate that we found this in memory
        printf(
            CHANNEL_ART_MGR_OBJECT_NAME": Pulled line bitmap (i:%u) from memory\n",
            psLineInfo->tLineBitmapIndex);
    }
    else
    {
        // Now we have to take a look at the database
        bFound = bGetLineBitmapFromDB(psObj, psLineInfo);
        if (bFound == TRUE)
        {
            // Indicate that we found this in the DB
            printf(
                CHANNEL_ART_MGR_OBJECT_NAME": Pulled line bitmap (i:%u) from DB\n",
                psLineInfo->tLineBitmapIndex);
        }
    }

    return bFound;
}

/*******************************************************************************
*
*   vAddNewLineBitmapEntry
*
*   This function adds a line bitmap, which has just been pulled from the DB,
*   into the line bitmap list which resides in memory, allowing for easier
*   retrieval of the line bitmap data for the next time it is requested.
*
*******************************************************************************/
static void vAddNewLineBitmapEntry (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        )
{
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        LINE_BITMAP_INFO_STRUCT *psLineInfoCopy;

        // Create the name for the line bitmap entry
        snprintf( &psObj->acBuffer[0],
                  OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                  CHANNEL_ART_MGR_OBJECT_NAME"bitmap entry %u",
                  psLineInfo->tLineBitmapIndex);

        // Allocate the memory
        psLineInfoCopy = (LINE_BITMAP_INFO_STRUCT *)
            SMSO_hCreate(
                &psObj->acBuffer[0],
                sizeof(LINE_BITMAP_INFO_STRUCT),
                (SMS_OBJECT)psObj->psArtOwner,
                FALSE );

        if (psLineInfoCopy != NULL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Copy the line bitmap's attributes
            psLineInfoCopy->pun8LineBitmap = psLineInfo->pun8LineBitmap;
            psLineInfoCopy->tLineBitmapByteSize = psLineInfo->tLineBitmapByteSize;
            psLineInfoCopy->tLineBitmapIndex = psLineInfo->tLineBitmapIndex;

            // Add the node to the list
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->psArtOwner->hLineBitmaps,
                NULL, psLineInfoCopy );

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSO_vDestroy((SMS_OBJECT)psLineInfoCopy);
            }
        }

        SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);
    }

    return;
}

/*******************************************************************************
*
*   psChanAddNewAssocEntry
*
*   Create a new association structure and insert it into the association
*   list.  Provide the caller with the newly created association.
*
*******************************************************************************/
static CHANNEL_ART_ASSOC_STRUCT * psChanAddNewAssocEntry (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bCategory,
    UN16 un16Source
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create the name for this association entry
    snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
              CHANNEL_ART_MGR_OBJECT_NAME"Assoc(S:%u C:%u)",
              un16Source,
              bCategory );

    // Allocate space for the list node
    psAssoc = (CHANNEL_ART_ASSOC_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(CHANNEL_ART_ASSOC_STRUCT),
                    (SMS_OBJECT)psArtOwner,
                    FALSE );

    if (psAssoc != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // The associations created here are
        // NEVER default associations
        psAssoc->bDefault = FALSE;

        // They have been referenced by
        // an association as well
        psAssoc->bReferenced = TRUE;

        // Use the caller-provided parameters
        psAssoc->bCategory = bCategory;
        psAssoc->un16Source = un16Source;

        // Add the node to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psArtOwner->hChanAssociations,
            &psAssoc->hEntry, psAssoc );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Destroy the association node
            SMSO_vDestroy( (SMS_OBJECT) psAssoc );
            psAssoc = (CHANNEL_ART_ASSOC_STRUCT *)NULL;
        }
    }

    return psAssoc;
}

/*******************************************************************************
*
*   bAlbumProcessSelectDBInfoResult
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select all" query made on the AA database info relation.  It populates a
*   pointer to a ALBUM_ART_DB_INFO_ROW_STRUCT with the information found
*   in the database.
*
*******************************************************************************/
static BOOLEAN bAlbumProcessSelectDBInfoResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    ALBUM_ART_DB_QUERY_RESULT_STRUCT *psResult
        )
{
    ALBUM_DB_INFO_FIELDS_ENUM eCurrentField =
        (ALBUM_DB_INFO_FIELDS_ENUM)0;
    ALBUM_ART_DB_INFO_ROW_STRUCT *psInfoRow;

    // Verify input
    if ( NULL == psResult )
    {
        return FALSE;
    }

    // If there are the correct
    // number of columns, then we have good results
    if (n32NumberOfColumns == ALBUM_DB_INFO_MAX_FIELDS)
    {
        psResult->bResultantRows = TRUE;
    }
    else
    {
        psResult->bResultantRows = FALSE;

        return FALSE;
    }

    // Get a more convenient pointer
    psInfoRow = &psResult->uDbRow.sDBInfo;

    do
    {
        // Decode the current field based on it's type
        switch ( eCurrentField )
        {
            case ALBUM_DB_INFO_FIELD_DB_VER:
            {
                psInfoRow->un8DBVer =
                    (UN8)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case ALBUM_DB_INFO_FIELD_DSI:
            {
                psInfoRow->un16DSI =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case ALBUM_DB_INFO_DATA_CONTENT_VERSION:
            {
                psInfoRow->un16DataContentVersion =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default: // Shouldn't happen
            break;
        }
    } while ( ++eCurrentField < ALBUM_DB_INFO_MAX_FIELDS );

    // We only want one row of data
    return FALSE;
}

/*******************************************************************************
*
*   bAlbumInspectDBInfo
*
*   This function inspects the DB info table to ensure that the AA database is
*   the correct version for this version of the art manager code as well as
*   the status of the association table
*
*******************************************************************************/
static BOOLEAN bAlbumInspectDBInfo (
    SQL_INTERFACE_OBJECT hSQLConnection
        )
{
    ALBUM_ART_DB_QUERY_RESULT_STRUCT sQueryResult;
    BOOLEAN bOk;

    // Initialize result struct
    OSAL.bMemSet(&sQueryResult, 0, sizeof(ALBUM_ART_DB_QUERY_RESULT_STRUCT));
    sQueryResult.psObj = NULL;
    sQueryResult.bResultantRows = FALSE;

    // Perform the SQL query and process the result
    bOk = SQL_INTERFACE.bQuery( hSQLConnection, CA_SELECT_DB_INFO,
            (SQL_QUERY_RESULT_HANDLER)bAlbumProcessSelectDBInfoResult,
            &sQueryResult ) ;

    if ( ( TRUE == bOk ) &&
         ( sQueryResult.bResultantRows == TRUE) )
    {
        // Grab the actual database version
        UN8 un8ActualDBVersion = (UN8)atoi(ALBUM_ART_DATABASE_FILE_VERSION);
        ALBUM_ART_DB_INFO_ROW_STRUCT *psInfoRow = &sQueryResult.uDbRow.sDBInfo;

        // Verify all the fields
        if ( ( psInfoRow->un8DBVer == un8ActualDBVersion )  &&
             ( psInfoRow->un16DSI == GsAlbumArtIntf.tDSI ) )
        {
            // Version matach
            return TRUE;
        }
        else
        {
            // Version mismatch
            return FALSE;
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   bAlbumProcessDatabase
*
*   This function processes the associations found within the database
*   in order to create all necessary channel art objects and associations.
*   When this operation has completed, all necessary channel art objects
*   and associations have been made based upon the database's contents.
*
*****************************************************************************/
static BOOLEAN bAlbumProcessDatabase (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
    )
{
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT sResult;
    BOOLEAN bOk = TRUE, bLocked;

    // Initialize the result structure; the channel art db's result
    // structure works fine for album art too, so we'll just borrow theirs.
    OSAL.bMemSet(&sResult, 0, sizeof(CHANNEL_ART_DB_JOIN_RESULT_STRUCT));
    sResult.psObj = psObj;

    // Initialize the portion of the assoc ctrl structure we're going to use
    psObj->sAssocCtrl.psChanLastArtAssoc = NULL;

    // Now, lock the art owner
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);

    if ( FALSE == bLocked )
    {
        // Error! Can't get a hold of the art owner
        vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
        return FALSE;
    }

    // Handle the Persistent DB
    bOk = SQL_INTERFACE.bQuery(
        psObj->hAlbumPerSQLConnection, AA_SELECT_DB_JOIN_BY_ID,
        (SQL_QUERY_RESULT_HANDLER)bAlbumProcessJoinDB, &sResult ) ;

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    if ( ( bOk != TRUE ) &&
         ( sResult.bSuccess != TRUE ) )
    {
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
*
*   bAlbumProcessJoinDB
*
*   This function is provided to the SQL_INTERFACE_OBJECT in order to process
*   a "select" query made on the album art image associations. The results
*   are used to create all the initial associations & object required for this
*   service to operate.
*
*******************************************************************************/
static BOOLEAN bAlbumProcessJoinDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT *psResult
        )
{
    ALBUM_ART_ASSOC_ROW_STRUCT sStoredAssoc;
    ALBUM_DB_IMAGE_JOIN_FIELDS_ENUM eCurrentField;
    CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eResult;

    OSAL.bMemSet(&sStoredAssoc, 0, sizeof(ALBUM_ART_ASSOC_ROW_STRUCT) );

    if ( ALBUM_DB_IMAGE_MAX_FIELDS == n32NumberOfColumns )
    {
        psResult->bSuccess = TRUE;
    }
    else
    {
        psResult->bSuccess = FALSE;
        return FALSE;
    }

    eCurrentField = (ALBUM_DB_IMAGE_JOIN_FIELDS_ENUM)0;

    do
    {
        // Decode the current field based on its type
        switch (eCurrentField)
        {
            case ALBUM_DB_IMAGE_FIELD_SRC:
            {
                sStoredAssoc.un16ServiceId =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case ALBUM_DB_IMAGE_FIELD_IMAGE_ID:
            {
                sStoredAssoc.un16ImageId =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case ALBUM_DB_IMAGE_FIELD_IMAGE_TYPEMASK:
            {
                sStoredAssoc.eType =
                    (CDO_TYPE_ENUM)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            case ALBUM_DB_IMAGE_FIELD_IMAGE_VER:
            {
                sStoredAssoc.un16ImageVer =
                    (UN16)psColumn[eCurrentField].uData.sUN32.un32Data;
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                          "Received an unknown field; this should not happen!"
                        );
            }
            break;
        }
    } while ( ++eCurrentField < ALBUM_DB_IMAGE_MAX_FIELDS);

    // Process this DB row into an association in our list;
    // note that we pass TRUE for the last param to let
    // the processor know we're doing assoc population
    // from an existing database (and, thus, nothing needs
    // to be written back out again.
    eResult = eAlbumProcessAssocUpdate ( psResult->psObj,
            &sStoredAssoc, NULL, TRUE );

    // If we couldn't do that, stop cause there was an error
    if ( CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR == eResult )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            " Pulling album assocs from DB unsuccessful.");
        return FALSE;
    }

    DATASERVICE_IMPL_vLog(CHANNEL_ART_MGR_OBJECT_NAME
        ": Assoc Found in DB: %s SID:%u PID:%u Image I%-5.5u\n",
        (sStoredAssoc.bDefault == TRUE)?"*Default*":"",
        sStoredAssoc.un16ServiceId,
        sStoredAssoc.un32ProgramId,
        sStoredAssoc.un16ImageId);

    return TRUE;
}

/*******************************************************************************
*
*   psAlbumGetAssocByService
*
*   A helper function that takes care of searching the album art association
*   table. As the table is ordered by SID, this is all we need to grab the
*   right row. NULL is returned if the row can not be found.
*
*******************************************************************************/
static ALBUM_ART_ASSOC_STRUCT* psAlbumGetAssocByService (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    SERVICE_ID tSID
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    ALBUM_ART_ASSOC_STRUCT sSearch;

    // If there's no association list, album art hasn't been
    // started yet.
    if ( OSAL_INVALID_OBJECT_HDL == psArtOwner->hAlbumAssociations )
    {
        return psAssoc;
    }

    // Search the linked list for this association
    sSearch.tSID = tSID;
    eReturnCode =
        OSAL.eLinkedListSearch (
            psArtOwner->hAlbumAssociations,
            &hEntry,
            (void *)&sSearch );

    if ( OSAL_SUCCESS == eReturnCode )
    {
        // Extract the association from memory
        psAssoc = (ALBUM_ART_ASSOC_STRUCT *) OSAL.pvLinkedListThis( hEntry );
    }
    else if ( OSAL_OBJECT_NOT_FOUND != eReturnCode )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            " psGetAssocStructByService(): code %u returned from eLinkedListSearch",
            eReturnCode);
    }

    return psAssoc;
}

/*******************************************************************************
*
*   hAlbumGetArt
*
*   This function contains the 'core' logic of the album art algorithm
*
*******************************************************************************/
static CHANNEL_ART_OBJECT hAlbumGetArt (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    SERVICE_ID tSID,
    PROGRAM_ID tPID,
    CDO_TYPE_ENUM eType
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc = NULL;
    CHANNEL_ART_OBJECT hArt = CHANNEL_ART_INVALID_OBJECT;

    // Before we do anything, make sure that album art is running; if not
    // just return invalid art for now. if there's no association list,
    // album art hasn't been started yet.
    if ( OSAL_INVALID_OBJECT_HDL == psArtOwner->hAlbumAssociations )
    {
        return CHANNEL_ART_INVALID_OBJECT;
    }

    // This if/else structure lays out the hierarchy for selecting which album
    // art image to use; it basically amounts to
    // 1) Use the service default (if we can't find the assoc)
    // 2) Check against the newer PID-associated image
    // 3) Check against the older PID-associated image
    // 4) Use the SID default (if told to via bDefault)
    // 5) Use a blank (NULL) image (if told to via bBlank)
    // 6) Check if we have a timed association for this SID
    // 7) If all else fails, use the default image for this SID

    psAssoc = psAlbumGetAssocByService(psArtOwner, tSID);

    // Can't find the assoc, or just want the service default?
    if ( psAssoc == NULL )
    {
        // This association was not found; we'll just hand over the SXM
        // service default art.
        psAssoc = psAlbumGetAssocByService(psArtOwner,
                ALBUM_ART_SERVICE_DEFAULT_ID);
        if ( psAssoc == NULL )
        {
            // We don't actually want to flag this as an error, as we can get
            // an invalid service row if the product has started, we haven't
            // completely finished processing the DB, and we get an a CDO
            // trying to grab art. This is actually okay, as we update all
            // album art after the table is fully loaded. For now, just
            // return an invalid art object.
            return CHANNEL_ART_INVALID_OBJECT;
        }

        return psAssoc->hDefaultArt;
    }

    // Try to find PID associated image
    if (tPID > 0)
    {
        ALBUM_ART_PID_TO_ART_MAP *psAssocMap;

        psAssocMap = psAlbumPidAssocFindArt(psAssoc, tPID);
        if ((psAssocMap != NULL) &&
            (psAssocMap->hArt != CHANNEL_ART_INVALID_OBJECT))
        {
            hArt = psAssocMap->hArt;
            psAssocMap->bInUse = TRUE;

            // Per Section 5.1.3.3.1 and 5.1.3.4.1, matching a trigger
            // means we stop blanking / defaulting
            psAssoc->bBlank = FALSE;
            psAssoc->bDefault = FALSE;
        }
    }

    if (hArt == CHANNEL_ART_INVALID_OBJECT)
    {
        // Just want the SID default?
        if ( ( psAssoc->bDefault == TRUE ) &&
                  ( psAssoc->hDefaultArt != CHANNEL_ART_INVALID_OBJECT  ) )
        {
            return psAssoc->hDefaultArt;
        }
        // Want a blank? Easy enough ...
        else if ( psAssoc->bBlank == TRUE )
        {
            return CHANNEL_ART_INVALID_OBJECT;
        }
        // See if there's a timed association for this SID
        else if ( ( psAssoc->un32ExpirationDelta > 0 ) &&
                  ( psAssoc->hSidArtTimed != CHANNEL_ART_INVALID_OBJECT ) )
        {
            hArt = psAssoc->hSidArtTimed;
        }
        // Just grab this SID's default then (assuming the eType
        // mask will let us)
        else
        {
            // They want this channel's default; double-check that our album
            // art assoc is compatible with this CDO object. An empty
            // eType means "anything goes."
            if ( ( ( eType          == ALBUM_ART_NULL_CDO_ETYPE ) ||
                   ( psAssoc->eType == ALBUM_ART_NULL_CDO_ETYPE ) ||
                   ( psAssoc->eType == eType )                     ) &&
                  psAssoc->hDefaultArt != CHANNEL_ART_INVALID_OBJECT  )
            {
                hArt = psAssoc->hDefaultArt;
            }
            else
            {
                // That didn't work, so grab the service default; if that's
                // not available, we don't flag it as an error for the
                // reasons stated above.
                psAssoc = psAlbumGetAssocByService(psArtOwner,
                        ALBUM_ART_SERVICE_DEFAULT_ID);

                // Make sure we didn't get a NULL assoc; if we did, it's indicative
                // of a DB problem.
                if ( NULL == psAssoc )
                {
                    hArt = CHANNEL_ART_INVALID_OBJECT;
                }
                else
                {
                    hArt = psAssoc->hDefaultArt;
                }
            }
        }
    }

    return hArt;
}

/*****************************************************************************
*
*   eGetAlbumArtProperties
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetAlbumArtProperties (
    CHANNEL_ART_OBJECT hArt,
    UN32 *pun32ImageId,
    UN32 *pun32ImageVer
        )
{
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT sProperty;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    IMAGE_OBJECT hImage;

    // Verify the art handle, and make sure the caller
    // wants at least one property
    if ( ( CHANNEL_ART_INVALID_OBJECT == hArt) ||
         ( ( NULL == pun32ImageId ) &&
           ( NULL == pun32ImageVer ) ) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get the requested IMAGE from the CHANNEL_ART handle ...
    hImage = CHANNEL_ART_hImage( hArt, CHANNEL_ART_IMAGETYPE_ALBUM );

    // ... and make sure it's available
    if ( IMAGE_INVALID_OBJECT == hImage )
    {
        return SMSAPI_RETURN_CODE_NO_OBJECTS;
    }

    if ( NULL != pun32ImageId )
    {
        // Get the image ID from the image object
        eReturnCode = IMAGE_eProperty (hImage, IMAGE_PROPERTY_INTERNAL_ID,
                &sProperty);

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode )
        {
            // We have the image but can't get it's ID?
            // Something's serious amiss here.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to retrieve the ID for a valid Image."
                    "Returned: (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));

            return eReturnCode;
        }

        *pun32ImageId = sProperty.uData.un32Value;
    }

    if ( NULL != pun32ImageVer )
    {
        // Get the image ID from the image object
        eReturnCode = IMAGE_eProperty (hImage, IMAGE_PROPERTY_INTERNAL_VERSION,
                &sProperty);

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturnCode )
        {
            // We have the image but can't get it's ID?
            // Something's serious amiss here.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to retrieve the version for a valid Image."
                    "Returned: (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));

            return eReturnCode;
        }

        *pun32ImageVer = sProperty.uData.un32Value;
    }

    // All is well
    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   bAlbumProbeDefaultImageResult
*
*****************************************************************************/
static BOOLEAN bAlbumProbeDefaultImageResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    ALBUM_ART_IMAGE_ATTRIB_STRUCT *psImage
        )
{
    if ((psColumn != NULL) &&
        (n32NumberOfColumns == (N32)ALBUM_DB_ATTRIB_MAX_FIELDS))
    {
        psImage->un16Id =
            psColumn[ALBUM_DB_ATTRIB_FIELD_IMAGE_ID].uData.sUN32.un32Data;
        psImage->un16Version =
            psColumn[ALBUM_DB_ATTRIB_FIELD_IMAGE_VER].uData.sUN32.un32Data;
        psImage->bValid = TRUE;
    }
    
    return FALSE; /* per DB schema up-to one row can meet the query */
}

/*****************************************************************************
*
*   eAlbumProbeDefaultImage
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAlbumProbeDefaultImage(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN16 *pun16ImageVersion
        )
{
    BOOLEAN bOk;
    ALBUM_ART_IMAGE_ATTRIB_STRUCT sImage;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Generate our query request based upon the association
    // row passed in as an argument

    psObj->asBindParams->eType = SQL_BIND_TYPE_UN32;
    psObj->asBindParams->pvData = (void*)(size_t)un16ImageId;

    memset(&sImage, 0, sizeof(sImage));
    sImage.bValid = FALSE;

    bOk = SQL_INTERFACE.bExecutePreparedStatement(
            psObj->hAlbumPerSQLConnection,
            psObj->hAlbumSelectImageAttrsStmt,
            (SQL_QUERY_RESULT_HANDLER)bAlbumProbeDefaultImageResult,
            (void*)&sImage,
            AA_SELECT_IMAGE_ATTRIB_STM_COUNT,
            &psObj->asBindParams[0]);

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "Failed to execute prepared statement for Id %d",
            (int)un16ImageId);
        eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
    }
    else if (sImage.bValid == TRUE)
    {
        *pun16ImageVersion = sImage.un16Version;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*   psAddNewAlbumAssocEntry
*
*   Create a new album art association structure and insert it into the
*   AA association list. Provide the caller with the newly created association.
*
*******************************************************************************/
static ALBUM_ART_ASSOC_STRUCT * psAlbumAddNewAssocEntry (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    SERVICE_ID tSID
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create the name for this association entry
    snprintf( &acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
              CHANNEL_ART_MGR_OBJECT_NAME"AlbumAssoc(S:%d)",
              (int)tSID);

    // Allocate space for the list node
    psAssoc = (ALBUM_ART_ASSOC_STRUCT *) SMSO_hCreate( &acName[0],
                sizeof(ALBUM_ART_ASSOC_STRUCT), (SMS_OBJECT)psArtOwner,
                FALSE );

    if ( psAssoc != NULL )
    {
        BOOLEAN bLocked;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // The associations created here are
        // NEVER default associations
        psAssoc->bDefault = FALSE;

        // Set the service ID
        psAssoc->tSID = tSID;

        // By default the number of simultaneously kept images
        // within the PidAssoc list is restricted by service developer
        // Later, this number can be changed due to ContentBuffered indication
        // provided by the Module.
        psAssoc->un32PidListCapacity = ALBUM_ART_PERSISTENT_IMAGES_COUNT;

        // Create the PIDs' list
        strcat(&acName[0], ":PIDs");
        eReturnCode = OSAL.eLinkedListCreate(
            &psAssoc->hPidArtList, &acName[0], NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS | OSAL_LL_OPTION_LINEAR);
        if (eReturnCode == OSAL_SUCCESS)
        {
            bLocked = SMSO_bLock((SMS_OBJECT)psArtOwner,
                    OSAL_OBJ_TIMEOUT_INFINITE);

            if (FALSE == bLocked)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        "Could not obtain art owner lock!");
                eReturnCode = OSAL_ERROR;
            }
            else
            {
                // Add the node to the list
                eReturnCode = OSAL.eLinkedListAdd(
                    psArtOwner->hAlbumAssociations,
                    &psAssoc->hEntry, psAssoc);

                SMSO_vUnlock((SMS_OBJECT)psArtOwner);
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to create list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Destroy the association node
            vAlbumAssocDestroy( psAssoc );
            psAssoc = (ALBUM_ART_ASSOC_STRUCT *)NULL;
        }
    }

    return psAssoc;
}

/*******************************************************************************
*
*   n16AlbumPidAssocFindCallback
*
*******************************************************************************/
static N16 n16AlbumPidAssocFindCallback(
    ALBUM_ART_PID_TO_ART_MAP *psObj,
    PROGRAM_ID *ptPID
        )
{
    return (psObj->tPID != *ptPID) ? -1 : 0;
}

/*******************************************************************************
*
*   psAlbumPidAssocFindArt
*
*******************************************************************************/
static ALBUM_ART_PID_TO_ART_MAP *psAlbumPidAssocFindArt (
    ALBUM_ART_ASSOC_STRUCT *psEntry,
    PROGRAM_ID tPID
        )
{
    ALBUM_ART_PID_TO_ART_MAP *psResult;
    OSAL_RETURN_CODE_ENUM eOsalCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    eOsalCode = OSAL.eLinkedListLinearSearch(
        psEntry->hPidArtList, &hEntry,
        (OSAL_LL_COMPARE_HANDLER)n16AlbumPidAssocFindCallback,
        (void*)&tPID);
    if (eOsalCode == OSAL_SUCCESS)
    {
        psResult = (ALBUM_ART_PID_TO_ART_MAP*)OSAL.pvLinkedListThis(hEntry);
    }
    else
    {
        psResult = NULL;
    }

    return psResult;
}

/*******************************************************************************
*
*   psAlbumPidAssocAllocate
*
*******************************************************************************/
static ALBUM_ART_PID_TO_ART_MAP *psAlbumPidAssocAllocate (
    ALBUM_ART_ASSOC_STRUCT *psEntry,
    PROGRAM_ID tPID
        )
{
    ALBUM_ART_PID_TO_ART_MAP *psResult = NULL;
    OSAL_RETURN_CODE_ENUM eOsalCode = OSAL_ERROR;
    UN32 un32Items = 0;

    // Find out the size of the pid assoc list
    eOsalCode = OSAL.eLinkedListItems(psEntry->hPidArtList, &un32Items);
    if (eOsalCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": failed to retrieve size of the list (%s)",
            OSAL.pacGetReturnCodeName(eOsalCode));
    }

    if (un32Items >= psEntry->un32PidListCapacity)
    {
        // Thus, in this case we shall do re-use of the oldest entry
        OSAL_LINKED_LIST_ENTRY hEntry;

        hEntry = OSAL.hLinkedListFirst(psEntry->hPidArtList, (void**)&psResult);
        if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (psResult == NULL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": failed to reuse entry (%p, %p)",
                hEntry, psResult);
        }
        else
        {
            OSAL_RETURN_CODE_ENUM eOsalCode;
            eOsalCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME": failed to remove entry (%s)",
                    OSAL.pacGetReturnCodeName(eOsalCode));
            }
        }
    }
    else
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        // Prepare name
        snprintf(&acName[0], sizeof(acName),
            CHANNEL_ART_MGR_OBJECT_NAME":AlbumAssoc(S=%d,PID=%d)",
            (int)psEntry->tSID, (int)tPID);

        // Allocate the memory
        psResult = (ALBUM_ART_PID_TO_ART_MAP*)
                OSAL.pvLinkedListMemoryAllocate(&acName[0],
                    sizeof(ALBUM_ART_PID_TO_ART_MAP), FALSE);
        if (psResult == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CHANNEL_ART_MGR_OBJECT_NAME
                        ": failed to allocate memory for %s", &acName[0]);
        }
        else
        {
            psResult->hArt = CHANNEL_ART_INVALID_OBJECT;
        }
    }
    if (psResult != NULL)
    {
        // Fill up fields
        psResult->tPID = tPID;
        psResult->bInUse = FALSE;

        // Append to the list
        eOsalCode = OSAL.eLinkedListAdd(psEntry->hPidArtList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void*)psResult);
        if (eOsalCode != OSAL_SUCCESS)
        {
            OSAL.vLinkedListMemoryFree((void*)psResult);
            psResult = NULL;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": failed to add assoc PID %d",
                (int)tPID);
        }
        else
        {
            printf(CHANNEL_ART_MGR_OBJECT_NAME": added PID %d to SID %d",
                (int)tPID, (int)psEntry->tSID);
        }
    }

    return psResult;
}

/*******************************************************************************
*
*   vAlbumPidAssocDestroy
*
*******************************************************************************/
static void vAlbumPidAssocDestroy (
    ALBUM_ART_PID_TO_ART_MAP *psObj
        )
{
    if (psObj->hArt != CHANNEL_ART_INVALID_OBJECT)
    {
        IMAGE_OBJECT hImage;

        hImage = CHANNEL_ART_hImage(psObj->hArt, CHANNEL_ART_IMAGETYPE_ALBUM);
        if (hImage != IMAGE_INVALID_OBJECT)
        {
            STRING_OBJECT hFileName;

            hFileName = IMAGE.hFileName(hImage);
            if (hFileName != STRING_INVALID_OBJECT)
            {
                int iResult;
                iResult = remove(STRING.pacCStr(hFileName));
                printf(CHANNEL_ART_MGR_OBJECT_NAME": removed file %s (%d)\n",
                    STRING.pacCStr(hFileName), iResult);
            }
        }
        CHANNEL_ART_vDestroy(psObj->hArt);
    }

    OSAL.vLinkedListMemoryFree((void*)psObj);

    return;
}

/*******************************************************************************
*
*   vAlbumPidAssocListClean
*
*******************************************************************************/
static void vAlbumPidAssocListClean (
    ALBUM_ART_ASSOC_STRUCT *psEntry
        )
{
    OSAL_RETURN_CODE_ENUM eOsalCode;

    eOsalCode = OSAL.eLinkedListRemoveAll(psEntry->hPidArtList,
        (OSAL_LL_RELEASE_HANDLER)vAlbumPidAssocDestroy);
    if ((eOsalCode != OSAL_SUCCESS) && (eOsalCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": failed to clean up the list SID %d",
            (int)psEntry->tSID);
    }

    return;
}

/*******************************************************************************
*
*   vAlbumPidAssocListScrub
*
*   This routine is used to scrub the list of album art images associated with
*   certain channel.
*
*   Each PID-IMAGE associated which doesn't meat the PID list has to be removed.
*   However, despite of the PID list the channel has to have at least
*   ALBUM_ART_PERSISTENT_IMAGES_COUNT recently received images if there
*   are more than ALBUM_ART_PERSISTENT_IMAGES_COUNT of them, or if less or equal -
*   ignore the scrub procedure.
*
*******************************************************************************/
static void vAlbumPidAssocListScrub (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    SERVICE_ID tSID,
    CHANNEL_ID tChannelID,
    PROGRAM_ID *atPIDList,
    UN8 un8PIDListSize
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc;

    // Check out the association
    psAssoc = psAlbumGetAssocByService(psArtOwner, tSID);
    if (psAssoc != NULL)
    {
        UN32 un32Items = 0;
        OSAL_RETURN_CODE_ENUM eOsalCode;

        // Update images control path. If the PID list is empty
        // we consider that we should keep restricted amount of associated
        // images.
        psAssoc->un32PidListCapacity =
            (un8PIDListSize == 0) ? ALBUM_ART_PERSISTENT_IMAGES_COUNT :
                                    ALBUM_ART_PERSISTENT_IMAGES_COUNT_MAX;

        // Check the size. In case of error the size will be less or equal to
        // which ALBUM_ART_PERSISTENT_IMAGES_COUNT prevents us from scrub
        // procedure.
        eOsalCode = OSAL.eLinkedListItems(psAssoc->hPidArtList, &un32Items);
        if (eOsalCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": failed to get list size");
        }

        if (un32Items > ALBUM_ART_PERSISTENT_IMAGES_COUNT)
        {
            OSAL_LINKED_LIST_ENTRY hEntry;
            ALBUM_ART_PID_TO_ART_MAP *psMap;

            // Remove all except the recent one(s)
            hEntry = OSAL.hLinkedListFirst(psAssoc->hPidArtList, (void**)&psMap);
            while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
                   (un32Items > ALBUM_ART_PERSISTENT_IMAGES_COUNT))
            {
                // Find it within the PID list
                UN8 un8Idx;
                for (un8Idx = 0;
                     (un8Idx < un8PIDListSize) && (psMap->tPID != atPIDList[un8Idx]);
                     ++un8Idx)
                { }

                // The PID is out of the SID associations. We can remove it
                if (un8Idx == un8PIDListSize)
                {
                    OSAL_LINKED_LIST_ENTRY hEntryToRemove = hEntry;

                    //Get next
                    hEntry = OSAL.hLinkedListNext(hEntry, (void**)NULL);

                    // Remove from the list and release corresponded resources
                    eOsalCode = OSAL.eLinkedListRemove(hEntryToRemove);
                    if (eOsalCode != OSAL_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            CHANNEL_ART_MGR_OBJECT_NAME": failed to remove assoc from list");
                    }
                    else
                    {
                        // Clean up the PID-IMAGE association
                        vAlbumPidAssocDestroy(psMap);
                        psMap = (ALBUM_ART_PID_TO_ART_MAP *)OSAL.pvLinkedListThis(hEntry);

                        // Size if the list reduced by 1
                        --un32Items;
                    }
                }
                else
                {
                    //Get next and keep the current one alive
                    hEntry = OSAL.hLinkedListNext(hEntry, (void**)&psMap);
                }
            }
        }
        else
        {
            printf(CHANNEL_ART_MGR_OBJECT_NAME": nothing to scrub. %d items",
                (int)un32Items);
        }
    }
    else
    {
        printf(CHANNEL_ART_MGR_OBJECT_NAME
            ": there is not Album Assoc for SID %d", (int)tSID);
    }

    return;
}

/*******************************************************************************
*
*   vAlbumAssocDestroy
*
*******************************************************************************/
static void vAlbumAssocDestroy(
    ALBUM_ART_ASSOC_STRUCT *psEntry
        )
{
    if (psEntry != NULL)
    {
        if (psEntry->hPidArtList != OSAL_INVALID_OBJECT_HDL)
        {
            vAlbumPidAssocListClean(psEntry);
            (void)OSAL.eLinkedListDelete(psEntry->hPidArtList);
        }

        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return;
}

#if 0
/*******************************************************************************
*
*   vAlbumRemoveAssocEntry
*
*   Remove an existing association from the association list and
*   destroy it. Used to remove transient / program-based
*   associations.
*
*   Reserved for future safe-to-delete functionality
*
*******************************************************************************/
static void vAlbumRemoveAssocEntry (
        CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
        UN16 un16ServiceId,
        UN32 un32ProgramId
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc;
    psAssoc = psAlbumGetAssocByService ( psObj, un16ServiceId );

    if ( NULL == psAssoc )
    {
        // Didn't find the entry; happens at startup when CDOs
        // are trying to grab art before the art service
        // is running. Just return.
        return;
    }

    if ( ( psAssoc->sPidArtNew.un32ProgramId != 0 ) &&
         ( psAssoc->sPidArtNew.un32ProgramId == un32ProgramId ) )
    {
        psAssoc->sPidArtNew.un32ProgramId = 0;
        psAssoc->sPidArtNew.bInUse = FALSE;
    }
    else if ( ( psAssoc->sPidArtOld.un32ProgramId != 0 ) &&
              ( psAssoc->sPidArtOld.un32ProgramId == un32ProgramId ) )
    {
        psAssoc->sPidArtOld.un32ProgramId = 0;
        psAssoc->sPidArtOld.bInUse = FALSE;
    }

    return;
}
#endif

/*****************************************************************************
*
*   bChanClearUpdatedFlag
*
*****************************************************************************/
static BOOLEAN bChanClearUpdatedFlag (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvData;
    if (psAssoc != NULL)
    {
        psAssoc->bUpdated = FALSE;
    }
    return TRUE;
}

/*****************************************************************************
*
*   bAlbumClearUpdatedFlag
*
*****************************************************************************/
static BOOLEAN bAlbumClearUpdatedFlag (
    void *pvData,
    void *pvArg
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc =
        (ALBUM_ART_ASSOC_STRUCT *)pvData;
    if ( NULL != psAssoc )
    {
        psAssoc->bUpdated = FALSE;
    }
    return TRUE;
}

/*******************************************************************************
*
*   vChanReleaseAssociations
*
*******************************************************************************/
static void vChanReleaseAssociations (
    void *pvData
        )
{
    CHANNEL_ART_ASSOC_STRUCT *psAssoc =
        (CHANNEL_ART_ASSOC_STRUCT *)pvData;

    if (psAssoc != NULL)
    {
        CHANNEL_ART_vDestroy( psAssoc->hChannelArt );
        psAssoc->hChannelArt = CHANNEL_ART_INVALID_OBJECT;

        SMSO_vDestroy((SMS_OBJECT)psAssoc);
    }

    return;
}

/*******************************************************************************
*
*   vAlbumReleaseAssociations
*
*******************************************************************************/
static void vAlbumReleaseAssociations (
    void *pvData
        )
{
    ALBUM_ART_ASSOC_STRUCT *psAssoc =
        (ALBUM_ART_ASSOC_STRUCT *)pvData;

    if (psAssoc != NULL)
    {
        CHANNEL_ART_vDestroy( psAssoc->hSidArtTimed );
        psAssoc->hSidArtTimed = CHANNEL_ART_INVALID_OBJECT;

        CHANNEL_ART_vDestroy( psAssoc->hDefaultArt );
        psAssoc->hDefaultArt = CHANNEL_ART_INVALID_OBJECT;

        vAlbumAssocDestroy(psAssoc);
    }

    return;
}

/*******************************************************************************
*
*   vReleaseLineBitmaps
*
********************************************************************************/
static void vReleaseLineBitmaps (
    void *pvData
        )
{
    LINE_BITMAP_INFO_STRUCT *psInfo =
        (LINE_BITMAP_INFO_STRUCT *)pvData;

    if (psInfo != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psInfo->pun8LineBitmap);
        psInfo->pun8LineBitmap = NULL;

        SMSO_vDestroy((SMS_OBJECT)psInfo);
    }

    return;
}

/*******************************************************************************
*
*   bIssueUpdate
*
*   This function issues an event list to the subscribed devices if there
*   are any events present and if there are any subscribed devices.
*
*   This function returns whether or not an event list went out at all.
*
*******************************************************************************/
static BOOLEAN bIssueUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bAlbum
        )
{
    BOOLEAN bSuccess;
    CHANNEL_ART_UPDATE_TYPE_ENUM eUpdateType;
    BOOLEAN (*bUpdateClearFn)(CHANNEL_ART_MGR_OBJECT_STRUCT*);

    DATASERVICE_IMPL_vLog(
        CHANNEL_ART_MGR_OBJECT_NAME": Updating subscribed decoders\n");
    
    if ( bAlbum == TRUE )
    {
        // Working with album art
        eUpdateType = CHANNEL_ART_UPDATE_TYPE_ALBUM_ASSOC_CHANGE;
        bUpdateClearFn = bAlbumClearAllUpdatedFlags;
    }
    else
    {
        // Working with channel art
        eUpdateType = CHANNEL_ART_UPDATE_TYPE_CHAN_ASSOC_CHANGE;
        bUpdateClearFn = bChanClearAllUpdatedFlags;
    }

    // Send an update to the subscribed decoders now
    bSuccess = bSendUpdate( psObj, eUpdateType,
        DECODER_INVALID_OBJECT );

    if (TRUE == bSuccess)
    {
        bSuccess = bUpdateClearFn(psObj);
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bChanClearAllUpdatedFlags
*
*******************************************************************************/
static BOOLEAN bChanClearAllUpdatedFlags (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Clear the privately-held flag in our defaults
    psObj->psArtOwner->sDefaultCategoryAssoc.bUpdated = FALSE;
    psObj->psArtOwner->sDefaultChannelAssoc.bUpdated = FALSE;

    // Clear the updated flag in all associations
    // we keep in the list
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->psArtOwner->hChanAssociations,
        bChanClearUpdatedFlag, NULL);

    return (eReturnCode == OSAL_SUCCESS);
}

/*******************************************************************************
*
*   bAlbumClearAllUpdatedFlags
*
*******************************************************************************/
static BOOLEAN bAlbumClearAllUpdatedFlags (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Clear the updated flag in all associations
    // we keep in the list
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->psArtOwner->hAlbumAssociations,
        bAlbumClearUpdatedFlag, NULL);

    return (eReturnCode == OSAL_SUCCESS);
}

/*******************************************************************************
*
*   vStartTransaction
*
*   Starts a transaction on the specific SQL interface. Populates
*   pbInTransaction with a BOOLEAN indicating if we're in a transaction or not.
*
*******************************************************************************/
static void vStartTransaction (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbInTransaction
        )
{
    // Only start a transaction if we're not already
    // in the middle of one
    if (*pbInTransaction == FALSE)
    {
        puts(CHANNEL_ART_MGR_OBJECT_NAME": Starting transaction");

        // Start a transaction for all the upcoming
        // station updates
        *pbInTransaction =
            SQL_INTERFACE.bStartTransaction(hSQLConnection);
    }

    return;
}

/*******************************************************************************
*
*   vEndTransaction
*
*******************************************************************************/
static void vEndTransaction (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbInTransaction
        )
{
    if (*pbInTransaction == TRUE)
    {
        puts(CHANNEL_ART_MGR_OBJECT_NAME": Ending transaction");
        *pbInTransaction =
            !SQL_INTERFACE.bEndTransaction(hSQLConnection);
    }
    return;
}

/*******************************************************************************
*
*   bSendUpdate
*
*   Common function used to provide updates to subscribed decoders
*
*******************************************************************************/
static BOOLEAN bSendUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_UPDATE_TYPE_ENUM eType,
    DECODER_OBJECT hDecoder
        )
{
    CHANNEL_ART_SUB_UPDATE_STRUCT sUpdate;
    BOOLEAN bSuccess;

    // Populate the update structure; note that un32ActiveProductMask
    // will remain unused for most messages.
    sUpdate.psObj = psObj;
    sUpdate.bSuccess = TRUE;
    sUpdate.eUpdateType = eType;

    // When starting the art service, we also need
    // to send the state of the products (used to determine
    // which handlers to call later)

    if ( CHANNEL_ART_UPDATE_TYPE_START == eType )
    {
        DATA_PRODUCT_STATE_ENUM eAlbumState = DATA_PRODUCT_STATE_INITIAL;
        DATA_PRODUCT_STATE_ENUM eChanState = DATA_PRODUCT_STATE_INITIAL;
        DATA_PRODUCT_MASK eProductMask;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        sUpdate.un32ActiveProductMask = 0;

        // Get the state of the album art product
        eReturn = DATA.eProductState(
                    (DATASERVICE_MGR_OBJECT)psObj,
                    DATA_PRODUCT_TYPE_ALBUM_ART, &eProductMask,
                    &eAlbumState );

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to retrieve album art product "
                    "state: (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturn));
            return FALSE;
        }

        // Get the state of the album art product
        eReturn = DATA.eProductState(
                    (DATASERVICE_MGR_OBJECT)psObj,
                    DATA_PRODUCT_TYPE_CHANNEL_ART, &eProductMask,
                    &eChanState );

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    "Failed to retrieve channel art product "
                    "state: (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturn));
            return FALSE;
        }

        if ( DATA_PRODUCT_STATE_READY == eAlbumState )
        {
            sUpdate.un32ActiveProductMask |= CHANNEL_ART_PRODUCT_MASK_ALBUM;
        }

        if ( DATA_PRODUCT_STATE_READY == eChanState )
        {
            sUpdate.un32ActiveProductMask |= CHANNEL_ART_PRODUCT_MASK_CHAN;
        }
    }

    // Do we have a specific destination for this event?
    if (DECODER_INVALID_OBJECT != hDecoder)
    {
        // Send the event direct to this decoder.  Saves
        // a bit of overhead and also is useful when we are
        // starting a new subscription because the destination
        // decoder would not yet be in our subscribed list, which
        // means that DATASERVICE_IMPL_bUpdateSubscribed is
        // not going to help us in that situation.
        bSuccess = bIterateSubscribedForEvent(hDecoder, &sUpdate);
    }
    else // Go ahead and use the iterator
    {
        // Send the subscribed devices the event list
        bSuccess = DATASERVICE_IMPL_bUpdateSubscribed(
            (DATASERVICE_IMPL_HDL)psObj,
            hDecoder,
            (DATASERVICE_IMPL_UPDATE_SUBSCRIBED_ITERATOR)
                bIterateSubscribedForEvent,
            &sUpdate );
    }

    // Put 'em together and what have you got?
    bSuccess &= sUpdate.bSuccess;

    if (FALSE == bSuccess)
    {
        DATASERVICE_IMPL_vLog(
            CHANNEL_ART_MGR_OBJECT_NAME": Failed to update subscribed decoders\n");
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bIterateSubscribedForEvent
*
*******************************************************************************/
static BOOLEAN bIterateSubscribedForEvent (
    DECODER_OBJECT hDecoder,
    CHANNEL_ART_SUB_UPDATE_STRUCT *psUpdate
        )
{
    SMS_EVENT_ART_STRUCT sUpdateData;

    // Initialize the event data now
    sUpdateData.eStatus = SMS_EVENT_ART_STATUS_UPDATE;
    sUpdateData.hArtList = OSAL_INVALID_OBJECT_HDL;
    sUpdateData.eType = SMS_EVENT_ART_TYPE_CHANNEL;
    sUpdateData.hArtService =
        (CHANNEL_ART_SERVICE_OBJECT)psUpdate->psObj;

    switch (psUpdate->eUpdateType)
    {
        case CHANNEL_ART_UPDATE_TYPE_START:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_START;
            sUpdateData.un32ActiveProductMask
                = psUpdate->un32ActiveProductMask;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending start event to subscribed decoders\n");
        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_STOP:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_STOP;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending stop event to subscribed decoders\n");
        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_START:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_ALBUM_START;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending album art start event to subscribed decoders\n");

        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_STOP:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_ALBUM_STOP;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending album art stop event to subscribed decoders\n");

        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_CHAN_ART_START:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_CHAN_START;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending channel art start event to subscribed decoders\n");

        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_CHAN_ART_STOP:
        {
            sUpdateData.eStatus = SMS_EVENT_ART_STATUS_CHAN_STOP;

            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending channel art stop event to subscribed decoders\n");

        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_CHAN_ASSOC_CHANGE:
        {
            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending art update event to subscribed decoders");

            // All updates are for the same type.  Learn what type of
            // update this is from the association control structure
            // Note: we default to channel above.
            if ( TRUE == psUpdate->psObj->sAssocCtrl.bCategory )
            {
                sUpdateData.eType = SMS_EVENT_ART_TYPE_CATEGORY;
            }

            // Create the list now
            sUpdateData.hArtList = hChanCreateUpdateListForDecoder(
                psUpdate->psObj,
                sUpdateData.eType);

            if (sUpdateData.hArtList == OSAL_INVALID_OBJECT_HDL)
            {
                psUpdate->bSuccess = FALSE;
            }
        }
        break;

        case CHANNEL_ART_UPDATE_TYPE_ALBUM_ASSOC_CHANGE:
        {
            DATASERVICE_IMPL_vLog(
                CHANNEL_ART_MGR_OBJECT_NAME
                ": Sending art update event to subscribed decoders");

            sUpdateData.eType = SMS_EVENT_ART_TYPE_ALBUM;

            // Create the list now
            sUpdateData.hArtList = hAlbumCreateUpdateListForDecoder(
                psUpdate->psObj,
                SMS_EVENT_ART_TYPE_ALBUM );

            if (sUpdateData.hArtList == OSAL_INVALID_OBJECT_HDL)
            {
                psUpdate->bSuccess = FALSE;
            }
        }
        break;

        case CHANNEL_ART_INVALID_UPDATE_TYPE:
        default:
        {
            // Error!
            psUpdate->bSuccess = FALSE;
        }
    }

    if (psUpdate->bSuccess == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        BOOLEAN bOk = FALSE;

        // Allocate a decoder event for this update
        hEvent = DECODER_hAllocateEvent(
            hDecoder, SMS_EVENT_ART, SMS_EVENT_OPTION_NONE, &puEventData);
        if (hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Copy the update data to the event argument
            puEventData->uDecoder.sArt = sUpdateData;

            // Post the event now (nothing we really can do if this fails)
            bOk = SMSE_bPostEvent(hEvent);
        }

        if (FALSE == bOk)
        {
            // Destroy any memory we may have allocated
            vDestroyDecoderUpdateList(sUpdateData.hArtList);

            // Can't post to the decoder!
            psUpdate->bSuccess = FALSE;
        }
    }

    return psUpdate->bSuccess;
}

/*****************************************************************************
*
*   bCopyUpdatedAssocsToDecoderList
*
*   Places an entry in an update list if that entry has been marked as
*   updated by the channel art manager. Needs a channel or album art
*   shim function to "feed" it update info.
*
*****************************************************************************/
static BOOLEAN bCopyUpdatedAssocsToDecoderList (
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate,
    CHANNEL_ART_OBJECT hArt,
    UN16 un16Source,
    BOOLEAN bUpdated,
    BOOLEAN bCategory
        )
{
    // Translate can't be NULL
    if ( NULL == psTranslate )
    {
        // Error!
        return FALSE;
    }

    // Only utilize associations which have
    // been updated since we only want to tell
    // decoders about stuff that has changed
    if ( TRUE == bUpdated )
    {
        ART_UPDATE_STRUCT *psUpdate;

        // Create an update for this memory
        psUpdate = (ART_UPDATE_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
            CHANNEL_ART_MGR_OBJECT_NAME":UpdateEntry",
            sizeof(ART_UPDATE_STRUCT), TRUE);
        if (psUpdate == NULL)
        {
            psTranslate->bSuccess = FALSE;
        }

        if (psTranslate->bSuccess == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Populate the entry
            psUpdate->hArt = hArt;
            psUpdate->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            if ( TRUE == bCategory )
            {
                psUpdate->uId.tCategoryId = (CATEGORY_ID)un16Source;
            }
            else
            {
                psUpdate->uId.tServiceId = (SERVICE_ID)un16Source;
            }

            // Add it to the list
            eReturnCode = OSAL.eLinkedListAdd(
                psTranslate->hList, &psUpdate->hEntry, (void *)psUpdate);
            if (eReturnCode != OSAL_SUCCESS)
            {
                psTranslate->bSuccess = FALSE;
            }
        }
    }

    return psTranslate->bSuccess;
}

/*****************************************************************************
*
*   bChanCopyUpdatedAssocsToDecoderList
*
*   Places a channel art updated entry in an update list by calling
*   bCopyUpdatedAssocsToDecoderLit.
*
*****************************************************************************/
static BOOLEAN bChanCopyUpdatedAssocsToDecoderList (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate
        )
{
    return bCopyUpdatedAssocsToDecoderList (
        psTranslate, psAssoc->hChannelArt, psAssoc->un16Source,
         psAssoc->bUpdated, psAssoc->bCategory );
}

/*****************************************************************************
*
*   bAlbumCopyUpdatedAssocsToDecoderList
*
*   Places an album art updated entry in an update list by calling
*   bCopyUpdatedAssocsToDecoderLit.
*
*****************************************************************************/
static BOOLEAN bAlbumCopyUpdatedAssocsToDecoderList (
    ALBUM_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate
        )
{
    // The specific piece of art doesn't matter for
    // album art (hence the NULL), while we use FALSE
    // as the last parameter to indicate that album art
    // is always for a service, not a category.
    return bCopyUpdatedAssocsToDecoderList (
        psTranslate, NULL, psAssoc->tSID,
          psAssoc->bUpdated, FALSE );
}

/*****************************************************************************
*
*   hCreateUpdateListForDecoder
*
*****************************************************************************/
static OSAL_OBJECT_HDL hCreateUpdateListForDecoder (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType,
    OSAL_OBJECT_HDL hAssociations,
    OSAL_LL_ITERATOR_HANDLER pfIteratorHandler
        )
{
    CHANNEL_ART_TRANSLATE_STRUCT sTranslate;

    // Initialize the list handle
    sTranslate.hList = OSAL_INVALID_OBJECT_HDL;

    sTranslate.bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LL_COMPARE_HANDLER n16Compare =
            (OSAL_LL_COMPARE_HANDLER)n16CompareEntriesForDecodersBySvcId;

        // Channel art and Album Art are both need the channels sorted
        // by service ID; if we're updating categories, sort by cat ID instead.
        if ( SMS_EVENT_ART_TYPE_CATEGORY == eType )
        {
            n16Compare = (OSAL_LL_COMPARE_HANDLER)n16CompareEntriesForDecodersByCatId;
        }

        // Create the list for this decoder (pre-allocate elements)
        eReturnCode = OSAL.eLinkedListCreate(
            &sTranslate.hList, CHANNEL_ART_MGR_OBJECT_NAME":UpdateList",
            n16Compare,
            OSAL_LL_OPTION_LINEAR |
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                " unable to create update list");
            break;
        }

        sTranslate.bSuccess = TRUE;
        eReturnCode = OSAL.eLinkedListIterate(
            hAssociations,
            pfIteratorHandler,
            (void *)&sTranslate);

        if ((eReturnCode != OSAL_SUCCESS) ||
            (sTranslate.bSuccess == FALSE))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                " unable to populate update list");
            break;
        }

        return sTranslate.hList;

    } while (FALSE);

    if (sTranslate.bSuccess == FALSE)
    {
        vDestroyDecoderUpdateList(sTranslate.hList);
    }

    return OSAL_INVALID_OBJECT_HDL;
}

/*****************************************************************************
*
*   hChanCreateUpdateListForDecoder
*
*****************************************************************************/
static OSAL_OBJECT_HDL hChanCreateUpdateListForDecoder(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType
        )
{
    return hCreateUpdateListForDecoder (
        psObj, eType, psObj->psArtOwner->hChanAssociations,
        (OSAL_LL_ITERATOR_HANDLER)bChanCopyUpdatedAssocsToDecoderList);
}

/*****************************************************************************
*
*   hAlbumCreateUpdateListForDecoder
*
*****************************************************************************/
static OSAL_OBJECT_HDL hAlbumCreateUpdateListForDecoder(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType
        )
{
    return hCreateUpdateListForDecoder (
        psObj, eType, psObj->psArtOwner->hAlbumAssociations,
        (OSAL_LL_ITERATOR_HANDLER)bAlbumCopyUpdatedAssocsToDecoderList);
}

/*****************************************************************************
*
*   vDestroyDecoderUpdateList
*
*****************************************************************************/
static void vDestroyDecoderUpdateList (
    OSAL_OBJECT_HDL hUpdateList
        )
{
    if (hUpdateList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(
            hUpdateList,
            (OSAL_LL_ITERATOR_HANDLER)bDestroyEntryForDecoder,
            NULL);
        if ((eReturnCode == OSAL_SUCCESS) ||
            (eReturnCode == OSAL_NO_OBJECTS))
        {
            OSAL.eLinkedListDelete(hUpdateList);
        }
    }

    return;
}

/*******************************************************************************
*
*   n16CompareEntriesForDecodersBySvcId
*
*   This function compares two ART_UPDATE_STRUCT and orders
*   them by service id.
*
*   Inputs:
*       psUpdate1 (in list), psUpdate2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareEntriesForDecodersBySvcId (
    ART_UPDATE_STRUCT *psUpdate1,
    ART_UPDATE_STRUCT *psUpdate2
        )
{
    N16 n16Result;
    if ((psUpdate1 == NULL) || (psUpdate2 == NULL))
    {
        return N16_MIN;
    }

    // Compare these two
    n16Result = (N16)(psUpdate1->uId.tServiceId - psUpdate2->uId.tServiceId);

    return n16Result;
}

/*******************************************************************************
*
*   n16CompareEntriesForDecodersByCatId
*
*   This function compares two ART_UPDATE_STRUCT and orders
*   them by category id.
*
*   Inputs:
*       psUpdate1 (in list), psUpdate2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareEntriesForDecodersByCatId (
    ART_UPDATE_STRUCT *psUpdate1,
    ART_UPDATE_STRUCT *psUpdate2
        )
{
    N16 n16Result;
    if ((psUpdate1 == NULL) || (psUpdate2 == NULL))
    {
        return N16_MIN;
    }

    // Compare these two
    n16Result = (N16)(psUpdate1->uId.tCategoryId - psUpdate2->uId.tCategoryId);

    return n16Result;
}

/*******************************************************************************
*
*   bDestroyEntryForDecoder
*
*******************************************************************************/
static BOOLEAN bDestroyEntryForDecoder (
    ART_UPDATE_STRUCT *psUpdate,
    void *pvUnused
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Attempt the remove now
    eReturnCode = OSAL.eLinkedListRemove(psUpdate->hEntry);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Clear handles
        psUpdate->hArt = CHANNEL_ART_INVALID_OBJECT;
        psUpdate->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        OSAL.vLinkedListMemoryFree((void *)psUpdate);
    }

    return TRUE;
}

/*****************************************************************************
 *
 *  vUpdateDecoderArt
 *
 *****************************************************************************/
static void vUpdateDecoderArt (
    CCACHE_OBJECT hCCache,
    CHANNEL_ART_SERVICE_OBJECT hArtService,
    SMS_EVENT_ART_TYPE_ENUM eType,
    OSAL_OBJECT_HDL hArtList
        )
{
    DECODER_ART_UPDATE_PROCESS_STRUCT sArtUpdate;
    OSAL_LINKED_LIST_ENTRY hFirstEntry;

    // Grab the first entry
    hFirstEntry = OSAL.hLinkedListFirst(hArtList, (void **)&sArtUpdate.psCurUpdate);

    do
    {
        BOOLEAN bOk;


        if ((hFirstEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (sArtUpdate.psCurUpdate == NULL))
        {
            break;
        }

        if ( SMS_EVENT_ART_TYPE_CHANNEL == eType )
        {
            // Iterate through our channels and
            // apply these changes as needed
            bOk = CCACHE_bIterateChannels(
                hCCache, TRUE,
                (CCACHE_CHANNEL_ITERATOR_HANDLER)bIterateChannelsForArt,
                (void *)&sArtUpdate );
        }
        else if ( SMS_EVENT_ART_TYPE_CATEGORY == eType )
        {
            // Iterate through our categories and
            // apply these changes as needed
            bOk = CCACHE_bIterateCategories(
                hCCache,
                (CCACHE_CATEGORY_ITERATOR_HANDLER)bIterateCategoriesForArt,
                (void *)&sArtUpdate );
        }
        else if ( SMS_EVENT_ART_TYPE_ALBUM == eType )
        {
            // Iterate through our categories and
            // apply these changes as needed
            bOk = CCACHE_bIterateChannels(
                hCCache, TRUE,
                (CCACHE_CHANNEL_ITERATOR_HANDLER)bIterateCDOsForArt,
                (void *)&sArtUpdate );
        }
        else
        {
            // Unrecognized art type!
            bOk = FALSE;
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME": Process Art update failed.");
            break;
        }
    } while (FALSE);

    return;
}

/*****************************************************************************
 *
 *   bIterateChannelsForArt
 *
 *****************************************************************************/
static BOOLEAN bIterateChannelsForArt (
    CHANNEL_OBJECT hChannel,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        )
{
    SERVICE_ID tThisServiceId;

    if (psArtUpdate == NULL)
    {
        // Error!
        return FALSE;
    }

    if (psArtUpdate->psCurUpdate == NULL)
    {
        // We're all out of entries to process!
        // But this really shouldn't happen since
        // we should have detected this in an earlier
        // invocation of this iterator
        return FALSE;
    }

    // Get this channel's service id
    tThisServiceId = CHANNEL.tServiceId(hChannel);

    // Catch up to the current id if necessary
    while (psArtUpdate->psCurUpdate->uId.tServiceId < tThisServiceId)
    {
        // Get the next entry
        OSAL.hLinkedListNext(
            psArtUpdate->psCurUpdate->hEntry,
            (void **)&psArtUpdate->psCurUpdate);
        if (psArtUpdate->psCurUpdate == NULL)
        {
            // Nothing left in the list, so we're done now
            break;
        }
    }

    // Do we have a valid pointer now?
    if (psArtUpdate->psCurUpdate != NULL)
    {
        // Does this id match the current id?
        if (tThisServiceId == psArtUpdate->psCurUpdate->uId.tServiceId)
        {
            printf(CHANNEL_ART_MGR_OBJECT_NAME": Service ID %u has updated art\n",
                tThisServiceId);

            // Tell the channel object that its channel art has been
            // updated and send any required notifications.
            CHANNEL_vUpdateArt(hChannel, TRUE);

            // Move the entry along now
            OSAL.hLinkedListNext(psArtUpdate->psCurUpdate->hEntry,
                (void **)&psArtUpdate->psCurUpdate);
        }
    }

    // More work to do if we have a valid pointer
    return (psArtUpdate->psCurUpdate != NULL);
}

/*****************************************************************************
 *
 *   bIterateCategoriesForArt
 *
 *****************************************************************************/
static BOOLEAN bIterateCategoriesForArt (
    CATEGORY_OBJECT hCategory,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        )
{
    CATEGORY_ID tThisCategoryId;

    if (psArtUpdate == NULL)
    {
        // Error!
        return FALSE;
    }

    if (psArtUpdate->psCurUpdate == NULL)
    {
        // We're all out of entries to process!
        // But this really shouldn't happen since
        // we should have detected this in an earlier
        // invocation of this iterator
        return FALSE;
    }

    // Get this category's id
    tThisCategoryId = CATEGORY.tGetCategoryId(hCategory);

    // Catch up to the current id if necessary
    while (psArtUpdate->psCurUpdate->uId.tCategoryId < tThisCategoryId)
    {
        // Get the next entry
        (void) OSAL.hLinkedListNext(
            psArtUpdate->psCurUpdate->hEntry,
            (void **)&psArtUpdate->psCurUpdate);
        if (psArtUpdate->psCurUpdate == NULL)
        {
            // Nothing left in the list, so we're done now
            break;
        }
    }

    // Do we have a valid pointer now?
    if (psArtUpdate->psCurUpdate != NULL)
    {
        // Does this id match the current id?
        if (tThisCategoryId == psArtUpdate->psCurUpdate->uId.tCategoryId)
        {
            printf(CHANNEL_ART_MGR_OBJECT_NAME": Category ID %u has updated art\n",
                tThisCategoryId);

            // Tell the category object that its art has been
            // updated and send any required notifications
            CATEGORY_vUpdateArt(hCategory, FALSE);

            // Move the entry along now
            (void) OSAL.hLinkedListNext(psArtUpdate->psCurUpdate->hEntry,
                (void **)&psArtUpdate->psCurUpdate);
        }
    }

    // More work to do if we have a valid pointer
    return (psArtUpdate->psCurUpdate != NULL);
}


/*****************************************************************************
 *
 *   bIterateCDOsForArt
 *
 *   During the process of updating a decoders art, this function will be
 *   called to get iterate over each channel's CDO and call CDO_vUpdateArt
 *   on it to populate the CDO with album art.
 *
 *****************************************************************************/
static BOOLEAN bIterateCDOsForArt (
    CHANNEL_OBJECT hChannel,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        )
{
    SERVICE_ID tThisServiceId;

    if ( psArtUpdate == NULL )
    {
        // Error!
        return FALSE;
    }

    if ( psArtUpdate->psCurUpdate == NULL )
    {
        // We're all out of entries to process!
        // But this really shouldn't happen since
        // we should have detected this in an earlier
        // invocation of this iterator
        return FALSE;
    }

    // Get this channel's service id
    tThisServiceId = CHANNEL.tServiceId(hChannel);

    // Catch up to the current id if necessary
    while ( psArtUpdate->psCurUpdate->uId.tServiceId < tThisServiceId )
    {
        // Get the next entry
        OSAL.hLinkedListNext(
            psArtUpdate->psCurUpdate->hEntry,
            (void **)&psArtUpdate->psCurUpdate);
        if ( psArtUpdate->psCurUpdate == NULL )
        {
            // Nothing left in the list, so we're done now
            break;
        }
    }

    // Do we have a valid pointer now?
    if ( psArtUpdate->psCurUpdate != NULL )
    {
        // Does this id match the current id?
        if ( tThisServiceId == psArtUpdate->psCurUpdate->uId.tServiceId )
        {
            CD_OBJECT hCDO;
            
            printf(CHANNEL_ART_MGR_OBJECT_NAME": Service ID %u / Channel ID %u has updated art\n",
                tThisServiceId, CHANNEL.tChannelId(hChannel));

            hCDO = CHANNEL.hCDO( hChannel );

            if ( CD_INVALID_OBJECT == hCDO )
            {
                // Well, that was ... unexpected.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CHANNEL_ART_MGR_OBJECT_NAME
                    ": bIterateCDOsForArt: invalid CDO passed!");
                return FALSE;
            }

            // Tell the channel object that its channel art has been
            // updated and send any required notifications.
            CDO_vUpdateArt( hCDO, hChannel );

            // Move the entry along now
            OSAL.hLinkedListNext(psArtUpdate->psCurUpdate->hEntry,
                (void **)&psArtUpdate->psCurUpdate);
        }
    }

    // More work to do if we have a valid pointer
    return ( psArtUpdate->psCurUpdate != NULL );
}

/*****************************************************************************
*
*   bDeleteExpiredTimedAssociations
*
*   This is a callback used during iteration to find any associations that
*   have expired and remove them.
*
*****************************************************************************/
static BOOLEAN bDeleteExpiredTimedAssociations (
    ALBUM_ART_ASSOC_STRUCT *psAssoc,
    ALBUM_ART_CLEANUP_STRUCT *psCleanup
        )
{
    // If this association doesn't have a timed component, just move on ...
    if ( 0 == psAssoc->un32ExpirationDelta )
    {
        // Keep iteration going
        return TRUE;
    }

    // Check to see if the association has expired; note that a zero
    // means that there's currently no timed association
    if ( ( 0 != psAssoc->un32ExpirationDelta ) &&
         ( psAssoc->un32ExpirationDelta < psCleanup->un32Now ) )
    {
        // Remove the association, save the timed art for re-use later.
        psAssoc->un32ExpirationDelta = 0;
        psAssoc->bUpdated = TRUE;
    }
    // Otherwise, see if it's eligible to be the next timed
    // association to be removed
    else if ( psAssoc->un32ExpirationDelta > psCleanup->un32NextTimeout )
    {
        psCleanup->un32NextTimeout = psAssoc->un32ExpirationDelta;
    }

    // Keep iteration going
    return TRUE;
}
/*****************************************************************************
*
*   vAlbumAssocCleanup
*
*****************************************************************************/
static void vAlbumAssocCleanup( CHANNEL_ART_SERVICE_OBJECT hArtService )
{
    UN32 un32Seconds;
    OSAL_RETURN_CODE_ENUM eReturn;
    ALBUM_ART_CLEANUP_STRUCT sCleanup;
    BOOLEAN bLocked;

    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj =
        (CHANNEL_ART_MGR_OBJECT_STRUCT *)hArtService;

    // Init association processing attributes
    OSAL.bMemSet( &sCleanup, 0,
        sizeof(ALBUM_ART_CLEANUP_STRUCT));

    eReturn = OSAL.eTimeGet(&un32Seconds);

    if ( OSAL_SUCCESS != eReturn )
    {
        // Well, that was ... unexpected.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            ": vAlbumAssocCleanup: problem calling OSAL.eTimeGet()!");
        return;
    }

    // Add one hundred milliseconds, so we'll delete anything that's
    // *about* to expire as well. No human is going to notice if their
    // album art changes 100ms in advance.
    sCleanup.un32Now = un32Seconds + ALBUM_ART_CLEANUP_WINDOW_MS;

    // Lock the channel art owner object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psArtOwner,
            OSAL_OBJ_TIMEOUT_INFINITE);
    if ( FALSE == bLocked )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME": Unable to lock art owner");
        return;
    }

    // Iterate over the linked list of associations
    // to see if there are any active associations with
    // that have expired
    eReturn = OSAL.eLinkedListIterate(
        psObj->psArtOwner->hAlbumAssociations,
        (OSAL_LL_ITERATOR_HANDLER)bDeleteExpiredTimedAssociations,
        (void *)&sCleanup
            );

    SMSO_vUnlock((SMS_OBJECT)psObj->psArtOwner);

    // Indicate an error occurred if the iteration failed (but
    // not if the list is empty)
    if ( (eReturn != OSAL_SUCCESS) &&
         (eReturn != OSAL_NO_OBJECTS) )
    {
        // Error! Can't access the linked list
        vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CHANNEL_ART_MGR_OBJECT_NAME
            ": vAlbumArtCleanup: problem iterating album art assocs!");
        return;
    }

    // If we have another association that's going to time-out eventually,
    // make sure we set our timer appropriately.

    if ( 0 != sCleanup.un32NextTimeout )
    {
        UN32 un32OffsetFromNowSecs;

        // Grab the time again, as it may have changed slightly since we grabbed it before
        eReturn = OSAL.eTimeGet( &un32Seconds );

        if ( OSAL_SUCCESS != eReturn )
        {
            // Well, that was ... unexpected.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CHANNEL_ART_MGR_OBJECT_NAME
                ": vAlbumArtCleanup: problem calling OSAL.eTimeGet()!");
            return;
        }

        if ( sCleanup.un32NextTimeout < un32Seconds )
        {
            // This really shouldn't happen, as it means the next timeout
            // would have occurred while we were processing this one. Just
            // set the timer to go off again right away.
            un32OffsetFromNowSecs = 0;
        }
        else
        {
            un32OffsetFromNowSecs = sCleanup.un32NextTimeout - un32Seconds;
            psObj->un32NextCleanupSecs = sCleanup.un32NextTimeout;
        }

        // Start the timed event. We want this to be a one-shot, so we
        // pass FALSE for bRepeatEvent
        (void) DATASERVICE_IMPL_bSetTimedEvent(
            psObj->hCleanupEvent, FALSE, FALSE, un32OffsetFromNowSecs);

    }
    else
    {
        psObj->un32NextCleanupSecs = 0;
    }

    // We wouldn't be here unless we cleaned up *something*; go ahead and iterate
    // our art for update.

    // Send an update to the subscribed decoders now
    (void) bSendUpdate( psObj, CHANNEL_ART_UPDATE_TYPE_ALBUM_ASSOC_CHANGE,
                    DECODER_INVALID_OBJECT );

    // No need to stop the timer, as the channel art manager only ever runs it in
    // one-shot mode.

    return;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

#if 0
/*****************************************************************************
*
*   pacSourceTypeToString
*
*   Not used at present, but may come in handy in the future
*
*****************************************************************************/
static const char* pacSourceTypeToString (
    CHANNEL_ART_SOURCE_TYPE_ENUM eType
        )
{
    const char *pacReturnString;

    switch (eType)
    {
        case CHANNEL_ART_SOURCE_TYPE_CHANNEL:
        {
            pacReturnString = MACRO_TO_STRING(CHANNEL_ART_SOURCE_TYPE_CHANNEL);
            break;
        }
        case CHANNEL_ART_SOURCE_TYPE_CATEGORY:
        {
            pacReturnString = MACRO_TO_STRING(CHANNEL_ART_SOURCE_TYPE_CATEGORY);
            break;
        }
        case CHANNEL_ART_SOURCE_TYPE_ALBUM:
        {
            pacReturnString = MACRO_TO_STRING(CHANNEL_ART_SOURCE_TYPE_ALBUM);
            break;
        }
        default:
        {
            pacReturnString = MACRO_TO_STRING(SOURCE_TYPE_UNKNOWN);
            break;
        }
        break;
    }

    return pacReturnString;
}
#endif
