/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the channel art
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef CHANNEL_ART_DB_CONSTANTS_H_
#define CHANNEL_ART_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database File Name */
#define CHANNEL_ART_DATABASE_FILENAME "channelart.db"
#define ALBUM_ART_REF_DATABASE_FILENAME "albumart_ref.db"
#define ALBUM_ART_PER_DATABASE_FILENAME "albumart.db"

/* Database Table Names; note that we use the same
 * db_info and image_assoc names for both the
 * channel and album art tables.
 */
#define DEFAULT_ASSOC_TABLE_NAME "default_assoc"
#define DATABASE_INFO_TABLE_NAME "db_info"
#define IMAGE_ATTRIBUTE_TABLE_NAME "image_attrib"
#define IMAGE_ASSOC_TABLE_NAME "image_assoc"
#define LINE_BITMAP_TABLE_NAME "line_bitmap"

/* Database Table Attributes */
#define CHANNEL_ART_DATABASE_FILE_VERSION "10"

/* Database Table Attributes */
#define ALBUM_ART_DATABASE_FILE_VERSION "1"

/* Database Queries */

// We can control SQLite's quoting behavior with a few more options
// than the standard c printf. We define commonly-used option sets
// below.

// Select all from info
#define CA_SELECT_DB_INFO \
    "select * from "DATABASE_INFO_TABLE_NAME";"

// Select image association
#define CA_SELECT_IMAGE_ASSOCIATION \
    "select * from "IMAGE_ASSOC_TABLE_NAME \
    " where srcType=%1u and source=%4u and imageType=%2u and isdefault=0;"

// Select image attribute
#define CA_SELECT_IMAGE_ATTR_STMT_IMAGE_ID_PARAM    0
#define CA_SELECT_IMAGE_ATTR_STMT_IMAGE_TYPE_PARAM  1
#define CA_SELECT_IMAGE_ATTR_STMT_PARAMS_COUNT      2

#define CA_SELECT_IMAGE_ATTRIBUTES \
    "select * from "IMAGE_ATTRIBUTE_TABLE_NAME \
    " where imageId=? and imageType=?;"

// Select line bitmap
#define CA_SELECT_LINE_BITMAP \
    "select * from "LINE_BITMAP_TABLE_NAME \
    " where bitmapId=%2u;"

// Natural join the association & attribute tables.
// Result columns are all the columns from the
// association table, then all the columns from the
// attribute table minus the joined attribute
#define CA_SELECT_DB_JOIN_BY_TYPE \
    "select * from "IMAGE_ASSOC_TABLE_NAME \
    " natural join "IMAGE_ATTRIBUTE_TABLE_NAME \
    " where "IMAGE_ASSOC_TABLE_NAME".imageType in (%1u, %1u) order by image_assoc.srcType, image_assoc.source desc;"

// Natural join the association & attribute tables.
// Result columns are all the columns from the
// association table, then all the columns from the
// attribute table minus the joined attribute
#define AA_SELECT_DB_JOIN_BY_ID \
    "select * from "IMAGE_ASSOC_TABLE_NAME \
    " natural join "IMAGE_ATTRIBUTE_TABLE_NAME \
    " order by image_assoc.serviceId desc;"

// Copies all columns from the attached reference
// album art association table into the persistent
// association table.
#define AA_COPY_ASSOC \
    "insert into "IMAGE_ASSOC_TABLE_NAME \
    " select * from ref."IMAGE_ASSOC_TABLE_NAME";"

// Copies all columns from the attached reference
// album art attribute table into the persistent
// attribute table.
#define AA_COPY_ATTRIB \
    "insert into "IMAGE_ATTRIBUTE_TABLE_NAME \
    " select * from ref."IMAGE_ATTRIBUTE_TABLE_NAME";"

#define AA_SELECT_IMAGE_ATTRIB_STM_COUNT 1

// Counts the number of rows with passed image id and version
#define AA_SELECT_IMAGE_ATTRIBUTES \
    "select * from "IMAGE_ATTRIBUTE_TABLE_NAME \
    " where imageId=?;"

/* Database Commands */

// Update the DB info table for channel assoc availability
#define CA_UPDATE_CHAN_ASSOC_AVAIL \
    "update db_info set chanAssocsAvailable=%1u;"

#define CA_UPDATE_OVERLAY_CHAN_ASSOC_AVAIL \
    "update db_info set overlayChanAssocsAvailable=%1u;"

// Update the DB info table for category assoc availability
#define CA_UPDATE_CAT_ASSOC_AVAIL \
    "update db_info set catAssocsAvailable=%1u;"

// Remove all associations of a specific type
#define CA_REMOVE_ASSOCS \
    "delete from "IMAGE_ASSOC_TABLE_NAME \
    " where srcType=%1u and source >= %4u and source <= %4u;"

// Copy default associations by type
#define CA_COPY_DEFAULT_ASSOCS \
    "insert into "IMAGE_ASSOC_TABLE_NAME" select * from "DEFAULT_ASSOC_TABLE_NAME \
    " where srcType=%1u and source >= %4u and source <= %4u;"

// Attribute Row alterations
#define CA_INSERT_ATTRIB_ROW \
    "insert into "IMAGE_ATTRIBUTE_TABLE_NAME \
    " (imageId, imageType, imageVer, codeType," \
    "  backgroundOptions, lineBitmapIndex, backgroundColor, secondaryPresent)" \
    " values ( %5u, %1u, %5u, %1u, %2u, %2u, %8u, %1u );"

#define CA_REMOVE_ATTRIB_ROW \
    "delete from "IMAGE_ATTRIBUTE_TABLE_NAME \
    " where imageId=%5u and imageType=%1u;"

#define CA_UPDATE_ATTRIB_ROW \
    "update "IMAGE_ATTRIBUTE_TABLE_NAME \
    " set imageVer=%5u, codeType=%1u," \
    " backgroundOptions=%2u, lineBitmapIndex=%2u," \
    " backgroundColor=%8u, secondaryPresent=%1u" \
    " where imageId=%5u and imageType=%1u;"

// Association Row alterations
#define CA_INSERT_ASSOC_ROW_STMT_SRC_TYPE_PARAM     0
#define CA_INSERT_ASSOC_ROW_STMT_SOURCE_PARAM       1
#define CA_INSERT_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM   2
#define CA_INSERT_ASSOC_ROW_STMT_IMAGE_ID_PARAM     3
#define CA_INSERT_ASSOC_ROW_STMT_IMAGE_VER_PARAM    4
#define CA_INSERT_ASSOC_ROW_STMT_IS_DEFAULT_PARAM   5
#define CA_INSERT_ASSOC_ROW_STMT_ASSOC_VER_PARAM    6
#define CA_INSERT_ASSOC_ROW_STMT_HI_PRIO_PARAM      7
#define CA_INSERT_ASSOC_ROW_STMT_PARAMS_COUNT       8

#define CA_INSERT_ASSOCIATION_ROW \
    "insert or replace into "IMAGE_ASSOC_TABLE_NAME \
    " (srcType, source, imageType, imageId, imageVersion, isdefault, assocVer, highPrio)" \
    " values ( ?, ?, ?, ?, ?, ?, ?, ? );"

#define CA_UPDATE_ASSOC_ROW_STMT_IMAGE_ID_PARAM     0
#define CA_UPDATE_ASSOC_ROW_STMT_IMAGE_VER_PARAM    1
#define CA_UPDATE_ASSOC_ROW_STMT_ASSOC_VER_PARAM    2
#define CA_UPDATE_ASSOC_ROW_STMT_HI_PRIO_PARAM      3
#define CA_UPDATE_ASSOC_ROW_STMT_SRC_TYPE_PARAM     4
#define CA_UPDATE_ASSOC_ROW_STMT_SOURCE_PARAM       5
#define CA_UPDATE_ASSOC_ROW_STMT_IMAGE_TYPE_PARAM   6
#define CA_UPDATE_ASSOC_ROW_STMT_IS_DEFAULT_PARAM   7
#define CA_UPDATE_ASSOC_ROW_STMT_PARAMS_COUNT       8

#define CA_UPDATE_ASSOCIATION_ROW \
    "update "IMAGE_ASSOC_TABLE_NAME \
    " set imageId=?, imageVersion=?, assocVer=?, highPrio=? " \
    "where srcType=? and source=? and imageType=? and isdefault=?;"

#define CA_DELETE_ASSOCIATION_ROW \
    "delete from "IMAGE_ASSOC_TABLE_NAME \
    " where srcType=%1u and source=%4u and imageType=%2u and isdefault=0;"

#define AA_UPDATE_ASSOCIATION_ROW \
    "replace into "IMAGE_ASSOC_TABLE_NAME \
    "(imageId, serviceId, typeMask) " \
    "values (%5u, %2u, %1u) ;"

#define AA_UPDATE_ATTRIBUTE_ROW \
    "replace into "IMAGE_ATTRIBUTE_TABLE_NAME \
    "(imageId, imageVersion) " \
    "values (%5u, %2u) ;"

#define AA_INSERT_VERSION_ROW \
    "insert into "DATABASE_INFO_TABLE_NAME" ( dbVer, DSI, dataContentVer ) " \
    "values ( %s, %3u, 1 ); "

#define AA_CREATE_VERSION_TABLE \
    "create table "DATABASE_INFO_TABLE_NAME"( " \
    "dbVer tinyint, " \
    "DSI smallint, " \
    "dataContentVer smallint " \
    "); "

#define AA_CREATE_ASSOCIATION_TABLE \
    "create table "IMAGE_ASSOC_TABLE_NAME" ( " \
    "serviceId smallint, " \
    "imageId smallint, " \
    "typeMask tinyint, " \
    "primary key (serviceId) " \
    "); "

#define AA_CREATE_ATTRIBUTE_TABLE \
    "create table "IMAGE_ATTRIBUTE_TABLE_NAME" ( " \
    "imageId smallint, " \
    "imageVersion smallint, " \
    "primary key (imageId) " \
    "); "

// Constant used to size buffers used to perform
// DB queries.  Size is based on largest string
// size (all defines above specify field lengths)
// CA_INSERT_ATTRIB_ROW is the longest at this time
#define CHANNEL_ART_MAX_SQL_STRING_LENGTH (256)

// Image Background Options
#define CHANNEL_ART_BACKGROUND_OPTION_NONE                (0x00)
#define CHANNEL_ART_BACKGROUND_OPTION_BACKGROUND_COLOR    (0x01)
#define CHANNEL_ART_BACKGROUND_OPTION_LINE_BITMAP         (0x02)

#define CHANNEL_ART_INVALID_SOURCE_ID (65535)

// Just a slightly more descriptive #define for an
// emtpy etype
#define ALBUM_ART_NULL_CDO_ETYPE (0)

// Color converters
#define CHANNEL_ART_GET_BACKGROUND_COLOR(red, green, blue) \
    ( ( ((red) & 0xFF) << 16) | ( ((green) & 0xFF) << 8) | ((blue) & 0xFF))
#define CHANNEL_ART_BACKGROUNDCOLOR_TO_RED(color) \
    (((color) >> 16) & 0xFF)
#define CHANNEL_ART_BACKGROUNDCOLOR_TO_GREEN(color) \
    (((color) >> 8) & 0xFF)
#define CHANNEL_ART_BACKGROUNDCOLOR_TO_BLUE(color) \
    ((color) & 0xFF)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Enumeration specifying all the available fields
// in the db_version database relation
typedef enum chan_db_info_fields_enum {
    CHAN_DB_INFO_FIELD_DB_VER =0,
    CHAN_DB_INFO_FIELD_DSI,
    CHAN_DB_INFO_CHAN_ASSOC_AVAILABLE,
    CHAN_DB_INFO_OVERLAY_CHAN_ASSOC_AVAILABLE,
    CHAN_DB_INFO_CAT_ASSOC_AVAILABLE,
    CHAN_DB_INFO_DATA_CONTENT_VERSION,
    CHAN_DB_INFO_MAX_FIELDS

} CHAN_DB_INFO_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the image_assoc database relation
typedef enum chan_db_image_assoc_fields_enum {
    CHAN_DB_IMAGE_ASSOC_FIELD_SRCTYPE = 0,
    CHAN_DB_IMAGE_ASSOC_FIELD_SRC,
    CHAN_DB_IMAGE_ASSOC_FIELD_IMAGE_TYPE,
    CHAN_DB_IMAGE_ASSOC_FIELD_IMAGE_ID,
    CHAN_DB_IMAGE_ASSOC_FIELD_IMAGE_VER,
    CHAN_DB_IMAGE_ASSOC_FIELD_IMAGE_DEFAULT,
    CHAN_DB_IMAGE_ASSOC_FIELD_ASSOC_VER,
    CHAN_DB_IMAGE_ASSOC_FIELD_HIGH_PRIO,
    CHAN_DB_IMAGE_ASSOC_MAX_FIELDS

} CHAN_DB_IMAGE_ASSOCIATION_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the image_attrib database relation
typedef enum chan_db_image_attrib_fields_enum {
    CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_ID = 0,
    CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_TYPE,
    CHAN_DB_IMAGE_ATTRIB_FIELD_IMAGE_VER,
    CHAN_DB_IMAGE_ATTRIB_FIELD_CODE_TYPE,
    CHAN_DB_IMAGE_ATTRIB_BACKGROUND_OPTIONS,
    CHAN_DB_IMAGE_ATTRIB_LINE_BITMAP_INDEX,
    CHAN_DB_IMAGE_ATTRIB_BACKGROUND_COLOR,
    CHAN_DB_IMAGE_ATTRIB_SECONDARY_PRESENT,
    CHAN_DB_IMAGE_ATTRIB_MAX_FIELDS

} CHAN_DB_IMAGE_ATTRIB_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the line_bitmap database relation
typedef enum line_bitmap_fields_enum {
    CHAN_DB_LINE_BITMAP_FIELD_ID = 0,
    CHAN_DB_LINE_BITMAP_FIELD_SIZE,
    CHAN_DB_LINE_BITMAP_FIELD_DATA,
    CHAN_DB_LINE_BITMAP_MAX_FIELDS

} LINE_BITMAP_FIELDS_ENUM;

typedef enum image_join_fields_enum {
    CHAN_DB_IMAGE_JOIN_FIELD_SRCTYPE = 0,
    CHAN_DB_IMAGE_JOIN_FIELD_SRC,
    CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_TYPE,
    CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_ID,
    CHAN_DB_IMAGE_JOIN_FIELD_ASSOC_IMAGE_VER,
    CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_DEFAULT,
    CHAN_DB_IMAGE_JOIN_FIELD_ASSOC_VER,
    CHAN_DB_IMAGE_JOIN_FIELD_HIGH_PRIO_ASSOC,
    CHAN_DB_IMAGE_JOIN_FIELD_IMAGE_VER,
    CHAN_DB_IMAGE_JOIN_FIELD_CODE_TYPE,
    CHAN_DB_IMAGE_JOIN_BACKGROUND_OPTIONS,
    CHAN_DB_IMAGE_JOIN_LINE_BITMAP_INDEX,
    CHAN_DB_IMAGE_JOIN_BACKGROUND_COLOR,
    CHAN_DB_IMAGE_JOIN_SECONDARY_PRESENT,
    CHAN_DB_IMAGE_JOIN_MAX_FIELDS,

} CHAN_DB_IMAGE_JOIN_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the db_version database relation
typedef enum album_db_info_fields_enum {
    ALBUM_DB_INFO_FIELD_DB_VER =0,
    ALBUM_DB_INFO_FIELD_DSI,
    ALBUM_DB_INFO_DATA_CONTENT_VERSION,
    ALBUM_DB_INFO_MAX_FIELDS

} ALBUM_DB_INFO_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the album art image_assoc database relation
typedef enum album_db_image_join_fields_enum {
    ALBUM_DB_IMAGE_FIELD_SRC =0,
    ALBUM_DB_IMAGE_FIELD_IMAGE_ID,
    ALBUM_DB_IMAGE_FIELD_IMAGE_TYPEMASK,
    ALBUM_DB_IMAGE_FIELD_IMAGE_VER,
    ALBUM_DB_IMAGE_MAX_FIELDS,

} ALBUM_DB_IMAGE_JOIN_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the album art image_assoc database relation
typedef enum album_db_assoc_fields_enum {
    ALBUM_DB_ASSOC_FIELD_SERVICE_ID=0,
    ALBUM_DB_ASSOC_FIELD_IMAGE_ID,
    ALBUM_DB_ASSOC_FIELD_TYPE_MASK,
    ALBUM_DB_ASSOC_MAX_FIELDS,

} ALBUM_DB_ASSOC_FIELDS_ENUM;

// Enumeration specifying all the available fields
// in the album art image_attrib database relation
typedef enum album_db_attrib_fields_enum {
    ALBUM_DB_ATTRIB_FIELD_IMAGE_ID=0,
    ALBUM_DB_ATTRIB_FIELD_IMAGE_VER,
    ALBUM_DB_ATTRIB_MAX_FIELDS,

} ALBUM_DB_ATTRIB_FIELDS_ENUM;


/*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Structure which specifies all attributes
// of a single database version row that is
// available within the channel art database
// These are the raw values that are retrieved from the database
typedef struct channel_art_db_info_row_struct
{
    UN8 un8DBVer;           // Database schema version
    UN16 un16DSI;           // DSI
    BOOLEAN bChanAssocsAvailable;
    BOOLEAN bOverlayChanAssocsAvailable;
    BOOLEAN bCatAssocsAvailable;
    UN16 un16DataContentVersion;

} CHANNEL_ART_DB_INFO_ROW_STRUCT;

typedef UN8 CHANNEL_ART_BACKGROUND_OPTION_MASK;

typedef struct line_bitmap_info_struct
{
    size_t tLineBitmapByteSize;
    size_t tLineBitmapIndex;
    const UN8 *pun8LineBitmap;

} LINE_BITMAP_INFO_STRUCT;

typedef struct image_background_graphics_struct
{
    UN32 un32BackgroundColor;
    CHANNEL_ART_BACKGROUND_OPTION_MASK tBackgroundOptions;
    LINE_BITMAP_INFO_STRUCT sLineBitmap;

} IMAGE_BACKGROUND_GRAPHICS_STRUCT;

// Structure which specifies all attributes
// of a single channel art row that is
// available within the channel art database
// These are the raw values that are retrieved from the database
typedef struct channel_art_attrib_row_struct
{
    UN16 un16ImageId;                // Image Id
    UN16 un16ImageVer;                // Image version
    UN8 un8ImageType;                // Image Type
    UN8 un8CodeType;                // File coding type
    BOOLEAN bSecondaryAvailable;    // Flag indicating if this image has a secondary
    IMAGE_BACKGROUND_GRAPHICS_STRUCT sBackgroundGfx; // Background info/rules

} CHANNEL_ART_ATTRIB_ROW_STRUCT;

// Structure which specifies all attributes
// of a single channel art association row
// that is available within the channel art database
// These are the raw values that are retrieved from the database
typedef struct channel_art_assoc_row_struct
{
    BOOLEAN bRemove;    // Flag indicating if this association should be deleted
    BOOLEAN bContent;   // Flag indicating a content-related assoc
    BOOLEAN bCategory;  // The type of service for this association
    UN16 un16Source;    // The source identifier
    UN16 un16ImageId;   // The image Id
    UN16 un16ImageVer;  // The image version
    UN8 un8ImageType;   // The image type
    BOOLEAN bDefault;   // The default flag
    UN8 un8AssocVer;    // The version of this association
    BOOLEAN bAssocVerConfirmed;  // Assoc ver has been confirmed by OTA processing
    BOOLEAN bHighPriority; // Flag indicates if this a "high-priority" association

} CHANNEL_ART_ASSOC_ROW_STRUCT;

// Union to group all the row definitions
// in a convenient manner
typedef union channel_art_db_row_union
{
    CHANNEL_ART_DB_INFO_ROW_STRUCT sDBInfo;
    CHANNEL_ART_ATTRIB_ROW_STRUCT sAttrib;
    CHANNEL_ART_ASSOC_ROW_STRUCT sAssoc;

} CHANNEL_ART_DB_ROW_UNION;

// Structure which specifies all attributes
// of a single database version row that is
// available within the channel art database
// These are the raw values that are retrieved from the database
typedef struct album_art_db_info_row_struct
{
    UN8 un8DBVer;           // Database schema version
    UN16 un16DSI;           // DSI
    UN16 un16DataContentVersion;

} ALBUM_ART_DB_INFO_ROW_STRUCT;

// Structure which specifies all attributes of a single album art association
// row that is available within the album art database. This structure is also
// used to pass association info back and forth between the album art interface
// and the channel art manager.

typedef struct album_art_assoc_row_struct
{
    UN16 un16ServiceId;         // The source identifier
    UN32 un32ProgramId;         // The program ID (00 for non-PID-specific images)
    UN16 un16ImageId;           // The image Id
    UN16 un16ImageVer;          // The image version
    CDO_TYPE_ENUM eType;        // The type of content this association applies to
    UN32 un32Expiration;        // If non-zero, the Unix epoch (seconds) when the
                                // association expires
    BOOLEAN bBlank;             // The blank flag
    BOOLEAN bDefault;           // The default flag
} ALBUM_ART_ASSOC_ROW_STRUCT;

typedef struct album_art_image_attrib_struct
{
    UN16 un16Id;               // Image id
    UN16 un16Version;          // Image version
    BOOLEAN bValid;            // The structure has valid data 

} ALBUM_ART_IMAGE_ATTRIB_STRUCT;

// Union to group all the row definitions
// in a convenient manner
typedef union album_art_db_row_union
{
    ALBUM_ART_DB_INFO_ROW_STRUCT sDBInfo;
    ALBUM_ART_ASSOC_ROW_STRUCT sAssoc;
    ALBUM_ART_IMAGE_ATTRIB_STRUCT sImage;

} ALBUM_ART_DB_ROW_UNION;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif // CHANNEL_ART_DB_CONSTANTS_H_
