/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CEML implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "ccache.h"
#include "decoder_obj.h"
#include "channel_obj.h"
#include "cid_obj.h"
#include "cme.h"
#include "cel.h"
#include "_cel.h"

#include "sms_api_debug.h"
/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateList
*
* Creates a content monitoring event list (CEML). This list may be used to add,
* remove, replace, enable and disable content to be monitored. It also is
* used to maintain and acknoweledge active events when they occur.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object for which to attach
*               or associate this content event monitoring list with.
*   n16CompareHandler
*               A compare/sort handler to associate with this list object. This
*               handler is used to maintain a sorted list of registered content.
*               The implementation of this handler is application specific
*               and applies to all registered content in the CEML.
*   vEventCallback
*               A function which is called when an event concerning
*               any content registered to this list is fired.
*   vEventCallbackArg
*               An argument which is anonymous and application specific. It
*               is provided to the caller when the provided vEventCallback
*               function is called.
*
* Outputs:
*
*   CEML_OBJECT  A valid CEML_OBJECT handle on success or CEML_INVALID_OBJECT
*                   if an error or failure occured.
*
*****************************************************************************/
static CEML_OBJECT hCreateList (
    DECODER_OBJECT hDecoder,
    CONTENT_COMPARE_HANDLER n16CompareHandler,
    ACTIVE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    CEML_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_LL_COMPARE_HANDLER n16RegisteredCompareHandler =
        (OSAL_LL_COMPARE_HANDLER)NULL;
    static UN32 un32Instance = 0;

    do
    {
        // Verify inputs are correct
        if(
            (vEventCallback == NULL) || // Must have a callback
            (SMSO_bValid((SMS_OBJECT)hDecoder) == FALSE) // Valid DECODER
            )
        {
            // Error!
            break;
        }

        // Construct a unique name for this list.
        snprintf(&acName[0],
            sizeof(acName),
            CEML_OBJECT_NAME":Obj:%u",
            un32Instance++);

        // Create a CEML object
        psObj = (CEML_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(CEML_OBJECT_STRUCT),
                SMS_INVALID_OBJECT, // This object is a parent, and lock
                TRUE              // feature is desired
                    );
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Populate the DECODER object handle
        psObj->hDecoder = hDecoder;

        // Populate the compare/sort handler based on what was provided.
        // Notice we actually use our own compare handler as the primary
        // handler for adding and iterating content. However we save any
        // handler provided by the caller for use within our own "shim"
        // handler.
        psObj->sRegistered.n16CompareHandler = n16CompareHandler;
        if(psObj->sRegistered.n16CompareHandler !=
            (CONTENT_COMPARE_HANDLER)NULL)
        {
            // Use caller's compare handler is provided , otherwise don't
            // just use ours only.
            n16RegisteredCompareHandler =
                (OSAL_LL_COMPARE_HANDLER)n16ContentCompareHandler;
        }

        // Create a linked list for registered content
        // RCL (Registered Content List).

        // Construct an appropriate name for list to be created.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 CEML_OBJECT_NAME":RCL");

        // Create the registered content list
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->sRegistered.hContentList,
                &acName[0],
                n16RegisteredCompareHandler,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize currently iterated content entry
        psObj->sRegistered.hCurrentContent =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Initialize blocked/unblocked flag
        psObj->sRegistered.bBlocked = FALSE;

        // Create a linked list for active events
        // AEL (Active Event List).

        // Construct an appropriate name for list to be created.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 CEML_OBJECT_NAME":AEL");

        // Create the active list (FIFO)
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psObj->sActive.hEventList,
                &acName[0],
                (OSAL_LL_COMPARE_HANDLER)NULL,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize currently iterated active event
        psObj->sActive.hCurrentEvent =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Initialize callback and arg
        psObj->sActive.vEventCallback = vEventCallback;
        psObj->sActive.pvEventCallbackArg = pvEventCallbackArg;

        // Create the ended list (FIFO)
        eReturnCode =
            OSAL.eLinkedListCreate(
            &psObj->hEndedEventList,
                &acName[0],
                (OSAL_LL_COMPARE_HANDLER)NULL,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Relinquish ownership of the CEML object
        SMSO_vUnlock((SMS_OBJECT)psObj);

        // Return CEML object to caller
        return (CEML_OBJECT)psObj;

    } while(0);

    // Something went wrong
    vDestroyList((CEML_OBJECT)psObj);
    return CEML_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroyList
*
* Destroys a Content Event Monitoring List (CEML).
*
* Inputs:
*
*   hCEML   A handle to a valid Content Event Monitoring List for which to
*           destroy and release all it's resources.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
static void vDestroyList ( CEML_OBJECT hCEML )
{
    BOOLEAN bLocked, bValid;
    DECODER_OBJECT hDecoder;
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;

    bValid = SMSO_bValid((SMS_OBJECT)hCEML);
    if (bValid == FALSE)
    {
        // Don't mess with invalid pointers
        return;
    }

    // Extract decoder's handle
    hDecoder = psObj->hDecoder;

    // Decoder is to be locked before CEML to avoid deadlock
    bLocked =
        SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        // Unable to lock decoder.
        return;
    }

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock CEML object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
        return;
    }

    // Check if the active event list exists.
    if(psObj->sActive.hEventList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list and destroy each event in the
        // list.
        OSAL.eLinkedListRemoveAll(
            psObj->sActive.hEventList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyEvent
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->sActive.hEventList);
        psObj->sActive.hEventList = OSAL_INVALID_OBJECT_HDL;
    }

    // Check if the registered content list exists.
    if(psObj->sRegistered.hContentList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list and destroy each content entry
        OSAL.eLinkedListRemoveAll(
            psObj->sRegistered.hContentList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyContent
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->sRegistered.hContentList);
        psObj->sRegistered.hContentList = OSAL_INVALID_OBJECT_HDL;
    }

    // Check if the active event list exists.
    if(psObj->hEndedEventList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list and destroy each event in the
        // list.
        OSAL.eLinkedListRemoveAll(
            psObj->hEndedEventList,
            (OSAL_LL_RELEASE_HANDLER)vDestroyEndedEvent
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->hEndedEventList);
        psObj->hEndedEventList = OSAL_INVALID_OBJECT_HDL;
    }

    // Initialize object elements
    psObj->hDecoder = DECODER_INVALID_OBJECT;
    psObj->sRegistered.n16CompareHandler = (CONTENT_COMPARE_HANDLER)NULL;
    psObj->sRegistered.bBlocked = FALSE;
    psObj->sRegistered.hCurrentContent = OSAL_INVALID_LINKED_LIST_ENTRY;

    psObj->sActive.hCurrentEvent = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->sActive.pvEventCallbackArg = NULL;
    psObj->sActive.vEventCallback = (ACTIVE_EVENT_CALLBACK)NULL;

    // Destroy the object itself
    SMSO_vDestroy((SMS_OBJECT)psObj);

    // Unlock DECODER
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    return;
}

/*****************************************************************************
*
*   tContentListItems
*
* This method retrieves the number of items in the CEML's content list.
*
* Inputs:
*
*   hCEML   A handle to a valid Content Event Monitoring List for which to
*           get the number of items in the contentlist.
*
* Outputs:
*
*   The number of items in this CEML object's contentlist.
*
*****************************************************************************/
static size_t tContentListItems (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;
    UN32 un32NumItemsInLL = 0;

    // Verify and lock CEML Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // At this point we have exclusive access to the CEML

        // to how many registered items there are
        OSAL.eLinkedListItems(psObj->sRegistered.hContentList,
                              &un32NumItemsInLL);

        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    return (size_t)(un32NumItemsInLL);
}

/*****************************************************************************
*
*   eIterateContentList
*
* This method allows iteration of the RCL.
*
* Inputs:
*
*   hCEML   A handle to a valid Content Event Monitoring List for which to
*           iterate over.
*   bContentIterator
*           A function to call for each content entry in the RCL.
*   pvContentIteratorArg
*           An anonymous pointer which is caller specific and provided
*           each time bContentIterator is called.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateContentList (
    CEML_OBJECT hCEML,
    CONTENT_ITERATOR_CALLBACK bContentIterator,
    void *pvContentIteratorArg
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    REGISTERED_CONTENT_LIST_ITERATOR_STRUCT sIterator;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    BOOLEAN bLocked, bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hCEML);
    if ((bContentIterator == NULL) || (bValid == FALSE))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        // Unable to lock the decoder
        return SMSAPI_RETURN_CODE_INVALID_DECODER;
    }

    // Verify and lock CEML Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // At this point we have exclusive access to the CEML

        OSAL_LINKED_LIST_ENTRY hSavedCurrentEntry;
        // save the current entry in case there is nested iteration
        hSavedCurrentEntry = psObj->sRegistered.hCurrentContent;

        // Populate iterator structure to iterate the list using our
        // own iterator function and argument to confirm to OSAL's linked list
        // prototypes.
        sIterator.psObj = psObj;
        sIterator.bContentIterator = bContentIterator;
        sIterator.pvContentIteratorArg = pvContentIteratorArg;
        sIterator.phCurrentEntry = &psObj->sRegistered.hCurrentContent;
        sIterator.bContinue = FALSE;

        // Iterate all registered content
        eOsalReturnCode =
            OSAL.eLinkedListIterate(
                psObj->sRegistered.hContentList,
                (OSAL_LL_ITERATOR_HANDLER)bRegisteredContentIterator,
                &sIterator
                    );
        if( (eOsalReturnCode == OSAL_SUCCESS) ||
            (eOsalReturnCode == OSAL_NO_OBJECTS))
        {
            // List was iterated
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // restore current entry
        psObj->sRegistered.hCurrentContent = hSavedCurrentEntry;

        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    // Unlock DECODER
    SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateActiveList
*
* This method allows iteration of the AEL.
*
* Inputs:
*
*   hCEML   A handle to a valid Content Event Monitoring List for which to
*           iterate over.
*   bEventIterator
*           A function to call for each active entry in the AEL.
*   pvEventIteratorArg
*           An anonymous pointer which is caller specific and provided
*           each time bEventIterator is called.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateActiveList (
    CEML_OBJECT hCEML,
    ACTIVE_EVENT_ITERATOR_CALLBACK bEventIterator,
    void *pvEventIteratorArg
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    ACTIVE_EVENT_LIST_ITERATOR_STRUCT sIterator;
    BOOLEAN bLocked, bValid;

    // Verify bEventIterator and hCEML for validness
    bValid = SMSO_bValid((SMS_OBJECT)hCEML);
    if ((bEventIterator == NULL) || (bValid == FALSE))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock the DECODER object. We first lock the associated
    // decoder because we do not want the DECODER to manipulate the active
    // list while we are iterating it.
    bLocked =
        SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_DECODER;
    }

    // At this point we have exclusive access to the DECODER.

    // Verify and lock CEML Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // At this point we have exclusive access to the CEML.

        // Initialize current iterated event (no event presently)
        psObj->sActive.hCurrentEvent =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Populate iterator structure to iterate the list using our
        // own iterator function and argument to confirm to OSAL's
        // linked list prototypes.
        sIterator.psObj = psObj;
        sIterator.bEventIterator = bEventIterator;
        sIterator.pvEventIteratorArg = pvEventIteratorArg;
        sIterator.phCurrentEntry = &psObj->sActive.hCurrentEvent;

        // Iterate all active events
        eOsalReturnCode =
            OSAL.eLinkedListIterate(
            psObj->sActive.hEventList,
            (OSAL_LL_ITERATOR_HANDLER)bActiveEventIterator,
            (void *)&sIterator);
        if( (eOsalReturnCode == OSAL_SUCCESS) ||
            (eOsalReturnCode == OSAL_NO_OBJECTS))
        {
            // List was iterated
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Re-initialize current iterated event since we are no
        // longer iterating.
        psObj->sActive.hCurrentEvent =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    // Unlock DECODER object
    SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   eRegister
*
* Register for a content event by adding it to the content event list (CEML)
* object. Specifically this goes into the RCL.
*
* Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               register this content with.
*   hCID        A content identifier representing content to be monitored.
*   pvContentArg A pointer to an anonymous type (defined by the caller)
*               which will be associated with this registered content and
*               provided to the caller when the content event occurs or when
*               this content is iterated.
*   un32Options Options provided by the caller about this registration
*               such as initial, end, etc.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM  SMSAPI_RETURN_CODE_SUCCESS on success. Or not on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegister (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked, bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hCEML);
    if (bValid == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    bLocked = SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        // Unable to lock the decoder
        return SMSAPI_RETURN_CODE_INVALID_DECODER;
    }

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        BOOLEAN bRegistered;

        // At this point we have exclusive access to the CEML.

        // Only register this content if we do not already have content
        // with this CID currently registered. If we do, then
        // this is considered a duplicate registration and not allowed.
        bRegistered = bIdRegistered(hCEML, hCID);
        if(bRegistered == FALSE)
        {
            REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry;
            CME_REGISTERED_ENTRY hCMEContent =
                CME_INVALID_REGISTERED_ENTRY;

            // Create an instance of this content registration entry
            psContentEntry = (REGISTERED_CONTENT_ENTRY_STRUCT *)
                SMSO_hCreate(
                    CEML_OBJECT_NAME":Content",
                    sizeof(REGISTERED_CONTENT_ENTRY_STRUCT),
                    (SMS_OBJECT)psObj,
                    FALSE
                        );
                    // This object is a child of the hCEML provided.
            if(psContentEntry != NULL)
            {
                // Populate content structure with what I can
                psContentEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                psContentEntry->psObj = psObj;
                psContentEntry->hCMEContent = CME_INVALID_REGISTERED_ENTRY;
                psContentEntry->pvContentArg = pvContentArg;

                // Register content with the CME and retrieve entry handle
                eReturnCode =
                    CME_eRegisterContent(
                        psObj->hDecoder, hCID, un32Options,
                        vActiveEventCallback, (void *)psContentEntry,
                        &hCMEContent
                            );
                if( (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
                    (hCMEContent != CME_INVALID_REGISTERED_ENTRY) )
                {
                    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

                    // Populate content structure with a valid handle now
                    psContentEntry->hCMEContent = hCMEContent;

                    // Add this to our list as well
                    eOsalReturnCode =
                        OSAL.eLinkedListAdd(
                            psObj->sRegistered.hContentList,
                            (OSAL_LINKED_LIST_ENTRY *)&psContentEntry->hEntry,
                            (void *)psContentEntry
                                );
                    if(eOsalReturnCode == OSAL_SUCCESS)
                    {
                        // All is well
                        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                    }
                    else
                    {
                        // Destroy content object we created and
                        // remove entry we just added to the CME
                        vDestroyContent(psContentEntry);
                    }
                }
                else
                {
                    // Destroy content object we created and
                    // remove entry if we just added it to the CME
                    vDestroyContent(psContentEntry);
                }
            }
            else
            {
                // Error! Cannot allocate memory for content
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }
        }
        else
        {
            // Error! Cannot add duplicate content
            eReturnCode = SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
        }

        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    // Unlock DECODER
    SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   hCID
*
*   This API is used to retrive the CID from the content which is currently
*   being iterated. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               retrieve the CID from the currently iterated content obj.
*
*   Outputs:
*
*   valid CID OBJECT on success, CID_INVALID_OBJECT on failure
*
*****************************************************************************/
static CID_OBJECT hCID (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;
    CID_OBJECT hCID;

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        return CID_INVALID_OBJECT;
    }

    // At this point we have exclusive access to the CEML
    hCID =
        CME_hCID(psObj->hDecoder);

    // Unlock CEML object
    SMSO_vUnlock((SMS_OBJECT)hCEML);

    return hCID;
}

/*****************************************************************************
*
*   eReplace
*
*   This API is used to replace the content which is currently being
*   iterated with new content information. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will replace the current content with new content information
*   provided by the caller.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               replace this content.
*   pvContentArg A pointer to an anonymous type (defined by the caller)
*               which will be associated with this registered content and
*               provided to the caller when the content event occurs or when
*               this content is iterated.
*   un32Options A set of options to use to modify the event with
*               (always replaced).
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eReplace (
    CEML_OBJECT hCEML,
    UN32 un32Options
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML

    // Find contents of the current entry
    psContentEntry = OSAL.pvLinkedListThis(
        psObj->sRegistered.hCurrentContent);
    if(psContentEntry != NULL)
    {
        // Replace content in the CME
        eReturnCode = CME_eReplaceContent(
            psObj->hDecoder, un32Options,
            vActiveEventCallback, (void *)psContentEntry
                );
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Now all we need to do is re-insert the existing content entry
            // if the replacement in the CME was successful.
            OSAL.eLinkedListReplaceEntry(
                psObj->sRegistered.hContentList,
                psObj->sRegistered.hCurrentContent,
                (void *)psContentEntry
                    );
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eRemove
*
*   This API is used to remove the current content being iterated from the
*   registered content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will remove the current iterated content.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               remove this content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemove(
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bOwner;
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry;
    OSAL_LINKED_LIST_ENTRY hNextEntry;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner of the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML

    // remove our content entry which contains this content.

    // Find the next entry in our list
    hNextEntry = OSAL.hLinkedListNext(
            psObj->sRegistered.hCurrentContent, (void *)NULL);

    // Extract what it points to
    psContentEntry = OSAL.pvLinkedListThis(
        psObj->sRegistered.hCurrentContent
            );
    if(psContentEntry != NULL)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Remove current entry in list.
        eOsalReturnCode = OSAL.eLinkedListRemove(
            psObj->sRegistered.hCurrentContent);
        if(eOsalReturnCode == OSAL_SUCCESS)
        {
            // Destroy the content entry object itself
            vDestroyContent(psContentEntry);

            // Point to the next entry
            psObj->sRegistered.hCurrentContent = hNextEntry;
        }
    }


    return eReturnCode;
}

/*****************************************************************************
*
*   eRemoveAll
*
*   This API is used to remove all content from the
*   registered content list.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               remove all content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveAll(
    CEML_OBJECT hCEML
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = CEML.eIterateContentList(
                      hCEML,
                      bRemoveAllContentIterator,
                      (void *)NULL
                          );

    return eReturnCode;
}

/*****************************************************************************
*
*   eEnable
*
*   This API is used to enable the current content being iterated from the
*   registered content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will enable the current iterated content.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               enable this content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable(
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML

    // Enable content in the CME
    eReturnCode = CME_eEnableContent(psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDisable
*
*   This API is used to disable the current content being iterated from the
*   registered content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will disable the current iterated content.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               disable this content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML

    // Disable content in the CME
    eReturnCode = CME_eDisableContent(psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   eContentEnabled
*
*   This API is used to query the current content being iterated from the
*   registered content list. It must only be called from within
*   an content object iterator function otherwise the call returns an error
*   code indicating this API cannot be called. When this API is invoked,
*   the API will retrieve the enabled status of the current iterated content.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               query if the currently iterated content is enabled.
*   pbEnabled   Pointer to copy the enabled status.
*               TRUE = Enabled. FALSE = Disabled
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eContentEnabled (
    CEML_OBJECT hCEML,
    BOOLEAN *pbEnabled
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    if (pbEnabled == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        *pbEnabled = FALSE;

        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // find out if content is enabled
    eReturnCode = CME_eContentEnabled(psObj->hDecoder, pbEnabled);

    return eReturnCode;
}

/*****************************************************************************
*
*   eSearchForCurrent
*
*   This API is used to search for occurring content matching the current
*   content being iterated from the registered content list. It must only be
*   called from within an content object iterator function otherwise the
*   call returns an error code indicating this API cannot be called.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               search for currunt content
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM
*       SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSearchForCurrent(
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML

    // Enable content in the CME
    eReturnCode = CME_eSearchForCurrent(psObj->hDecoder);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAcknowledge
*
*   This API is used to acknowledge the current event being iterated from the
*   event list. It must only be called from within an event object iterator
*   function otherwise the call returns an error code indicating this API
*   cannot be called. When this API is invoked, the API will remove
*   the current iterated event.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               acknowledge this event.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAcknowledge (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
    ACTIVE_EVENT_ENTRY_STRUCT *psEvent;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hNextEvent;
    BOOLEAN bOwner;

    // Verify ownership of SMS Object (CEML)
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCEML);
    if(bOwner == FALSE)
    {
        // Not owner the provided object
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Extract current event
    psEvent = OSAL.pvLinkedListThis(psObj->sActive.hCurrentEvent);

    // Extract next event
    hNextEvent = OSAL.hLinkedListNext(psObj->sActive.hCurrentEvent, NULL);

    // Acknowledge this event (which means to remove it)
    eOsalReturnCode = OSAL.eLinkedListRemove(
        psObj->sActive.hCurrentEvent
            );
    if(eOsalReturnCode == OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Removed active event %p (CURRENT) for channel %d flags %d",
            psObj->sActive.hCurrentEvent, CHANNEL.tChannelId(psEvent->hChannel), psEvent->un32Flags);

        // Mark as no longer current, but point to the next one
        psObj->sActive.hCurrentEvent = hNextEvent;

        // Free event
        vDestroyEvent(psEvent);

        // All is well
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }
    else
    {
        SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Cannot remove active event for channel %d flags %d",
            CHANNEL.tChannelId(psEvent->hChannel), psEvent->un32Flags);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eBlockAllEvents
*
* This API is used to disable all events for registered content given
* the provided content event monitoring service handle.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               disable monitoring.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBlockAllEvents (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // At this point we have exclusive access to the CEML
    psObj->sRegistered.bBlocked = TRUE;

    // Unlock CEML object
    SMSO_vUnlock((SMS_OBJECT)hCEML);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eUnBlockAllEvents
*
* This API is used to enable all events for registered content given
* the provided content event monitoring service handle. Individual content
* which has been disabled will remain disabled however.
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               re-enable monitoring.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUnBlockAllEvents (
    CEML_OBJECT hCEML
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // At this point we have exclusive access to the CEML
    psObj->sRegistered.bBlocked = FALSE;

    // Unlock CEML object
    SMSO_vUnlock((SMS_OBJECT)hCEML);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eEventsBlocked
*
* This API is used to determine if a ceml is blocked
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               query if it is blocked.
*   pbBlocked   Pointer to the loation to which the events blocked status
*               will be copied. TRUE - the CEML is blocked.
*                               FALSE - the CEML is unblocked
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEventsBlocked (
    CEML_OBJECT hCEML,
    BOOLEAN *pbBlocked
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;

    if (pbBlocked == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        *pbBlocked = FALSE;
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // At this point we have exclusive access to the CEML
    *pbBlocked = psObj->sRegistered.bBlocked;

    // Unlock CEML object
    SMSO_vUnlock((SMS_OBJECT)hCEML);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eGetEndedEventAttributes
*
* This API is used to get attributes (Channel Id, Artist name and Title)
* of the first event from ceml's list of ended events
*
*   Inputs:
*
*   hCEML       A handle to a valid Content Event Monitoring List for which to
*               query if it is blocked.
*   ptChannelId   Pointer to the ChannelId to which the event's ChannelId
*                 will be copied.
*   phArtist   Pointer to the Artist to which the event's Artist name
*                 will be copied.
*   ptChannelId   Pointer to the Title to which the event's Title
*                 will be copied.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetEndedEventAttributes (
    CEML_OBJECT hCEML,
    CHANNEL_ID *ptChannelId,
    STRING_OBJECT *phArtist,
    STRING_OBJECT *phTitle
        )
{
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)hCEML;
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if ((ptChannelId == NULL) ||
        (phArtist == NULL) ||
        (phTitle == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock SMS Object (CEML)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        ENDED_EVENT_ENTRY_STRUCT *psEndedEvent;

        hEntry = OSAL.hLinkedListFirst(
            psObj->hEndedEventList, (void **)&psEndedEvent);

        if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            if (psEndedEvent != NULL)
            {
                *ptChannelId = psEndedEvent->tChannelId;
                *phArtist = psEndedEvent->hArtist;
                *phTitle = psEndedEvent->hTitle;

                // Destroy the object. Application is responsible for
                // STRING_OBJECTs now.
                SMSO_vDestroy((SMS_OBJECT)psEndedEvent);

                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            // Remove entry.
            OSAL.eLinkedListRemove(hEntry);
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CEL_eAddEndedEvent
*
*   Adds entry with ended event's info to the list. This makes it possible
*   to provide applicaton with the details about ended and destroyed event
*
*   Inputs:
*
*   hCEML        A handle to a valid Content Event Monitoring List to which to
*                store event's details.
*   tChannelId   Id of channel on which event was active.
*   hArtist      STRING_OBJECT containing event's Artist text.
*   hTitle       STRING_OBJECT containing event's Title text.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CEL_eAddEndedEvent(
    CEML_OBJECT hCEML,
    CHANNEL_ID tChannelId,
    STRING_OBJECT hArtist,
    STRING_OBJECT hTitle
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    ENDED_EVENT_ENTRY_STRUCT *psEndedEvent;
    BOOLEAN bLocked;

    do
    {
        CEML_OBJECT_STRUCT *psObj =
            (CEML_OBJECT_STRUCT *)hCEML;

        // Verify and lock SMS Object (CEML)
        bLocked =
            SMSO_bLock((SMS_OBJECT)hCEML, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == FALSE)
        {
            break;
        }

        psEndedEvent = (ENDED_EVENT_ENTRY_STRUCT *)
            SMSO_hCreate(
                CEML_OBJECT_NAME":Ended Event",
                sizeof(ENDED_EVENT_ENTRY_STRUCT),
                (SMS_OBJECT)psObj,
                FALSE);
                // This object is a child of the hCEML provided.
        if(psEndedEvent == NULL)
        {
            // Error!
            break;
        }

        psEndedEvent->tChannelId = tChannelId;
        psEndedEvent->hArtist = STRING.hDuplicate(hArtist);
        psEndedEvent->hTitle = STRING.hDuplicate(hTitle);

        // Add this event to our ended list.
        eOsalReturnCode = OSAL.eLinkedListAdd(
            psObj->hEndedEventList,
            (OSAL_LINKED_LIST_ENTRY *)NULL,
            (void *)psEndedEvent
                );
        if(eOsalReturnCode != OSAL_SUCCESS)
        {
            // Entry could not be added, free it then.
            vDestroyEndedEvent(psEndedEvent);
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while(0);

    if(bLocked == TRUE)
    {
        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)hCEML);
    }

    return eReturnCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bRegisteredContentIterator
*
* An OSAL Linked-list shim used to interface a registered content iterator
* function with an OSAL linked-list iterator.
*
*****************************************************************************/
static BOOLEAN bRegisteredContentIterator (
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry,
    void *pvIteratorCallbackArg
        )
{
    REGISTERED_CONTENT_LIST_ITERATOR_STRUCT *psIterator =
        (REGISTERED_CONTENT_LIST_ITERATOR_STRUCT *)pvIteratorCallbackArg;
    BOOLEAN bContinue = FALSE;

    // Set current iterated content by using the self-reference.
    // This will be used if the implementation within the callback if it
    // desires to manipulate the list of registered content.
    // (i.e. replace, remove, etc.)
    *psIterator->phCurrentEntry = psContentEntry->hEntry;

    // If entry is not registered in master list, there is no reason
    // to look for it in the list
    if (CME_INVALID_REGISTERED_ENTRY != psContentEntry->hCMEContent)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        // Use the CME to iterate (actually just use) one entry in it's
        // list and stop.
        eReturnCode = CME_eIterateContent (
            psIterator->psObj->hDecoder,
            psContentEntry->hCMEContent,
            bContentIterator,
            pvIteratorCallbackArg
                );
        if( (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (psIterator->bContinue == TRUE)
                )
        {
            // Continue to iterate the list
            bContinue = TRUE;
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   bContentIterator
*
* Although not really an "iterator" it does do something similar. It
* allows the use of CME_eIterateContent with a start entry and only
* iterates a single entry and returns. This way the CEML is driving the actual
* iteration.
*
*****************************************************************************/
static BOOLEAN bContentIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void *pvIteratorCallbackArg
        )
{
    // Verify input
    if(psContent != NULL)
    {
        REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry;

        // Extract 'content entry'
        psContentEntry =
            (REGISTERED_CONTENT_ENTRY_STRUCT *)psContent->pvEventCallbackArg;

        // Verify content entry
        if(psContentEntry != NULL)
        {
            REGISTERED_CONTENT_LIST_ITERATOR_STRUCT *psIterator =
                (REGISTERED_CONTENT_LIST_ITERATOR_STRUCT *)pvIteratorCallbackArg;

            // Finally execute caller's iterator callback function
            psIterator->bContinue =
                psIterator->bContentIterator (
                    (CEML_OBJECT)psIterator->psObj,
                    psContent->hCID,
                    psContentEntry->pvContentArg,
                    psContent->un32Options,
                    psIterator->pvContentIteratorArg
                        );
        }
    }

    // Always return FALSE to the CME as we intend to
    // iterate one and only one entry.
    return FALSE;
}

/*****************************************************************************
*
*   bContentRemoveIterator
*
* Although not really an "iterator" it does do something similar. It
* allows the use of CME_eIterateContent with a start entry and only
* iterates a single entry and returns. This way the CEML is driving the actual
* iteration.
*
* Specifically this iterator-shim removes the content from the CME.
*
*****************************************************************************/
static BOOLEAN bContentRemoveIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void *pvIteratorCallbackArg
        )
{
    // Simply remove the content
    CME_eRemoveContent(hDecoder);

    // Always return FALSE to the CME as we intend to
    // iterate one and only one entry.
    return FALSE;
}

/*****************************************************************************
*
*   bContentArgIterator
*
* Although not really an "iterator" it does do something similar. It
* allows the use of CME_eIterateContent with a start entry and only
* iterates a single entry and returns. This way the CEML is driving the actual
* iteration.
*
* Specifically this iterator-shim extracts the registered content's
* pvContentArg which was provided when the content was registered.
*
*****************************************************************************/
static BOOLEAN bContentArgIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void **ppvContentArg
        )
{
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry;

    // Verify input
    if(psContent != NULL)
    {
        // Extract 'content entry'
        psContentEntry =
            (REGISTERED_CONTENT_ENTRY_STRUCT *)psContent->pvEventCallbackArg;

        // Verify content entry
        if(psContentEntry != NULL)
        {
            // Extract content argument
            *ppvContentArg = psContentEntry->pvContentArg;
        }
    }

    // Always return FALSE to the CME as we intend to
    // iterate one and only one entry.
    return FALSE;
}

/*****************************************************************************
*
*   vActiveEventCallback
*
* Called by the CEMS in the context of the associated DECODER object. This
* is an event callback shim which first allows us to manage our own active
* event list based on events which have occured. When we are done with
* our own handling of the event we eventually call the user's callback
* function.
*
* As valid events for discovered content is found we add these events to
* an active event list for the application to acknowledge when necessary.
*
*****************************************************************************/
static void vActiveEventCallback (
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    UN32 un32Flags,
    void *pvEventCallbackArg
        )
{
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry =
        (REGISTERED_CONTENT_ENTRY_STRUCT *)pvEventCallbackArg;
    CEML_OBJECT_STRUCT *psObj = (CEML_OBJECT_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bLocked, bValid;
    ACTIVE_EVENT_ENTRY_STRUCT *psEvent;


    // Verify and lock SMS Object provided, this locks the parent CEML object
    bLocked =
        SMSO_bLock((SMS_OBJECT)psContentEntry, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return;
    }

    do
    {
        // Verify CEML is valid
        bValid = SMSO_bValid((SMS_OBJECT)psContentEntry->psObj);
        if(bValid == FALSE)
        {
            // Invalid object
            break;
        }

        // Extract CEML from the provided callback argument
        psObj = psContentEntry->psObj;

        // At this point we have exclusive access to the CEML object.

        // Intercept events which occurred while the list's state is
        // disabled. Otherwise allow the event to occur as normal.
        if(psObj->sRegistered.bBlocked == TRUE)
        {
            // Events are disabled for this content list. However
            // if the FINAL flag is set it must be processed regardless.
            // So we mask off any other flags and leave the FINAL
            // flag intact and allow the callback to process it.
            // Note: Regardless if content monitoring is enable or disabled for
            // this list, we must always process the "FINAL" flag since the CME
            // is trying to tell us this content is no longer registered. For
            // example the CME may be shutting down or something.
            un32Flags &= ~(ACTIVE_EVENT_FLAG_INITIAL |
                ACTIVE_EVENT_FLAG_END);
        }

        // Now, we can begin to process the event. If the event does
        // not have any flags set then there is no need
        // to process further as we do not queue empty flag events.
        if(un32Flags != 0)
        {
            BOOLEAN bRegistered;

            // The INITIAL, END or FINAL flag is set...process event.

            // Create an instance of this event object
            psEvent = (ACTIVE_EVENT_ENTRY_STRUCT *)
                SMSO_hCreate(
                    CEML_OBJECT_NAME":Event",
                    sizeof(ACTIVE_EVENT_ENTRY_STRUCT),
                    (SMS_OBJECT)psObj,
                    FALSE);
                    // This object is a child of the hCEML provided.
            if(psEvent == NULL)
            {
                // Error!
                break;
            }

            // Populate event structure with event information
            bRegistered = CHANNEL_bRegisterNotification(
                hChannel, CHANNEL_OBJECT_EVENT_NONE);
            if(bRegistered == TRUE)
            {
                // Channel is marked as in-use for us.
                // Now we retain this handle.
                psEvent->hChannel = hChannel;
            }
            else
            {
                // Either handle is invalid or we could not mark as used.
                psEvent->hChannel = CHANNEL_INVALID_OBJECT;
            }

            psEvent->pvEventCallbackArg = pvEventCallbackArg;
            psEvent->un32Flags = un32Flags;
            psEvent->pvContentArg = psContentEntry->pvContentArg;

            // Initialize event self-reference (invalid for now)
            psEvent->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Add this event to our active list. It can only be
            // removed once it has been acknowledged.
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sActive.hEventList,
                (OSAL_LINKED_LIST_ENTRY *)&psEvent->hEntry,
                (void *)psEvent
                    );
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Entry could not be added, free it then.
                SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Cannot add active event for channel %d flags %x (%s)",
                    CHANNEL.tChannelId(hChannel), un32Flags, OSAL.pacGetReturnCodeName(eReturnCode));
                vDestroyEvent(psEvent);
                break;
            }
            else
            {
                SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Added active event %p for channel %d flags %x",
                    psEvent->hEntry, CHANNEL.tChannelId(hChannel), un32Flags);
            }

            // Initialize current event handle so list manipulators like
            // acknowledge work within the callback.
            psObj->sActive.hCurrentEvent = psEvent->hEntry;

            // Call, user callback
            psObj->sActive.vEventCallback(
                (CEML_OBJECT)psObj,
                psEvent->hChannel,
                psEvent->pvContentArg,
                psEvent->un32Flags,
                psObj->sActive.pvEventCallbackArg
                    );

            // Re-init currently active event
            psObj->sActive.hCurrentEvent = OSAL_INVALID_LINKED_LIST_ENTRY;

            // We're done processing this event

            // Check if this entry is being removed by the CME which means
            // the FINAL flag is set. If it is, we need to remove it
            // and destroy it.
            if( (un32Flags & ACTIVE_EVENT_FLAG_FINAL) ==
                ACTIVE_EVENT_FLAG_FINAL)
            {
                OSAL_RETURN_CODE_ENUM eOsalReturnCode;

                // Un-register this content from the CEML.

                // Remove entry from our content list.
                eOsalReturnCode = OSAL.eLinkedListRemove(
                    psContentEntry->hEntry);
                if(eOsalReturnCode == OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Removed active event %p (FINAL) for channel %d flags %x",
                        psContentEntry->hEntry, CHANNEL.tChannelId(hChannel), un32Flags);

                    // Content is already going to be destroyed by the CME when
                    // this callback returns. So no need to destroy it twice as
                    // part of the vDestroyContent() call below.
                    psContentEntry->hCMEContent =
                        CME_INVALID_REGISTERED_ENTRY;

                    // Invalidate entry handle (since it is no longer registered
                    // as part of the list).
                    psContentEntry->hEntry =
                        OSAL_INVALID_LINKED_LIST_ENTRY;

                    // Destroy entry
                    vDestroyContent(psContentEntry);
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(CEML_OBJECT_NAME, 2, "Cannot remove active event for channel %d flags %x",
                        CHANNEL.tChannelId(hChannel), un32Flags);
                }
            }
        }
        else
        {
            // There are no remaining flags set. So we just bail.
        }

    } while(0);

    if (NULL != psObj)
    {
        // Unlock CEML object
        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bActiveEventIterator
*
* An OSAL Linked-list shim used to interface an active event iterator
* function with an OSAL linked-list iterator.
*
*****************************************************************************/
static BOOLEAN bActiveEventIterator (
    ACTIVE_EVENT_ENTRY_STRUCT *psEvent,
    void *pvIteratorArg
        )
{
    ACTIVE_EVENT_LIST_ITERATOR_STRUCT *psIterator =
        (ACTIVE_EVENT_LIST_ITERATOR_STRUCT *)pvIteratorArg;
    CEML_OBJECT_STRUCT *psObj =
        (CEML_OBJECT_STRUCT *)psIterator->psObj;
    BOOLEAN bContinue;

    // Set current iterated event by using the self-reference.
    // This will be used if the implementation within the callback
    // desires to manipulate the list of events
    // (i.e. acknowledge)
    *psIterator->phCurrentEntry = psEvent->hEntry;

    // This event is in the active list. Provide it to the caller.
    bContinue = psIterator->bEventIterator(
        (CEML_OBJECT)psObj,
        psEvent->hChannel,
        psEvent->pvContentArg,
        psEvent->un32Flags,
        psIterator->pvEventIteratorArg
            );

    return bContinue;
}

/*****************************************************************************
*
*   vDestroyContent
*
* Destroy a content entry. The entry's resources are completely released.
*
*****************************************************************************/
static void vDestroyContent (
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry
        )
{
    // Check input
    if(psContentEntry != NULL)
    {
        CEML_OBJECT_STRUCT *psObj;

        // Extract CEML from the provided callback argument
        psObj = psContentEntry->psObj;

        // Remove content from the CME
        if(psContentEntry->hCMEContent != CME_INVALID_REGISTERED_ENTRY)
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            // We need to tell the CME to get rid of this content first
            eReturnCode = CME_eIterateContent (
                psContentEntry->psObj->hDecoder,
                psContentEntry->hCMEContent,
                bContentRemoveIterator,
                NULL
                    );
            if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // Content removed (unregistered)
                psContentEntry->hCMEContent =
                    CME_INVALID_REGISTERED_ENTRY;

                // Check if we need to notify anything
                if(psObj->sActive.vEventCallback != NULL)
                {
                    // Call, user callback
                    psObj->sActive.vEventCallback(
                        (CEML_OBJECT)psObj,
                        CHANNEL_INVALID_OBJECT,
                        psContentEntry->pvContentArg,
                        ACTIVE_EVENT_FLAG_FINAL,
                        psObj->sActive.pvEventCallbackArg
                            );
                }
            }
        }

        // Initialize object pointer
        psContentEntry->psObj = NULL;

        // Initialize entry handle
        psContentEntry->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Destroy the content entry object itself
        SMSO_vDestroy((SMS_OBJECT)psContentEntry);

    }

    return;
}

/*****************************************************************************
*
*   vDestroyEvent
*
* Destroy an event entry. The entry's resources are completely released.
*
*****************************************************************************/
static void vDestroyEvent ( ACTIVE_EVENT_ENTRY_STRUCT *psEvent )
{
    // Check input
    if(psEvent != NULL)
    {
        // Un-reserve CHANNEL object
        CHANNEL_vUnregisterNotification(psEvent->hChannel, NULL);
        psEvent->hChannel = CHANNEL_INVALID_OBJECT;

        // Initialize event memory
        psEvent->un32Flags = 0;
        psEvent->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psEvent->pvContentArg = NULL;
        psEvent->pvEventCallbackArg = NULL;

        // Destroy event object itself
        SMSO_vDestroy((SMS_OBJECT)psEvent);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyEndedEvent
*
* Destroy an ended event entry. The entry's resources are completely released.
*
*****************************************************************************/
static void vDestroyEndedEvent (
    ENDED_EVENT_ENTRY_STRUCT *psEvent
        )
{
    // Check input
    if(psEvent != NULL)
    {
        // Initialize event memory
        psEvent->tChannelId = 0;
        if (psEvent->hArtist != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psEvent->hArtist);
            psEvent->hArtist = STRING_INVALID_OBJECT;
        }
        
        if (psEvent->hTitle != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psEvent->hTitle);
            psEvent->hTitle = STRING_INVALID_OBJECT;
        }

        // Destroy event object itself
        SMSO_vDestroy((SMS_OBJECT)psEvent);
    }

    return;
}

/*****************************************************************************
*
*   bIdRegistered
*
* Find a registered content entry with the provided CID. If it exists return
* TRUE, otherwise return FALSE.
*
*****************************************************************************/
static BOOLEAN bIdRegistered (
    CEML_OBJECT hCEML,
    CID_OBJECT hId
        )
{
    REGISTERED_CONTENT_LIST_SEARCH_ITERATOR_STRUCT sIterator;

    // Search our registered content (belonging to this list) and determine
    // if content with this CID already exists.
    sIterator.bFound = FALSE;
    sIterator.hId = hId;

    // Iterate all registered content...
    CEML.eIterateContentList(
        hCEML,
        bEqualCIDs,
        &sIterator
            );

    return sIterator.bFound;
}

/*******************************************************************************
*
*   bEqualCIDs
*
* An iterator callback function which determins if two CIDs are equal or not.
* If not FALSE is returned. Otherwise if they are TRUE is returned.
*
*****************************************************************************/
static BOOLEAN bEqualCIDs (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvContentIteratorArg
        )
{
    REGISTERED_CONTENT_LIST_SEARCH_ITERATOR_STRUCT *psIterator =
        (REGISTERED_CONTENT_LIST_SEARCH_ITERATOR_STRUCT *)pvContentIteratorArg;
    N16 n16Result;

    // Check if id's match. Use CID's equality compare method.
    n16Result = CID.n16Equal(hCID, psIterator->hId);
    if(n16Result == 0)
    {
        // Match!
        psIterator->bFound = TRUE;

        // Stop looking.
        return FALSE;
    }

    // Keep looking
    return TRUE;
}

/*******************************************************************************
*
*   n16ContentCompareHandler
*
*       This is the function used to compare two existing
*       REGISTERED_CONTENT_ENTRY_STRUCTs. Specifically this function
*       is used to perform a relational comparison for sorting. This sorter
*       keeps the content in order of the assoicated pvContentArg.
*       Since pvContentArg is an application defined function it is up to
*       thier implementation to define this order.
*
*       Outputs:
*               0   - pvContentArg1 and pvContentArg2 have the same value
*                       (equal, error)
*               > 0 - pvContentArg1 is greater than (after) pvContentArg2
*               < 0 - pvContentArg1 is less than (before) pvContentArg2
*
*****************************************************************************/
static N16 n16ContentCompareHandler (
    const REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry1,
    const REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry2
        )
{
    N16 n16Result = 0; // Error
    void *pvContentArg1, *pvContentArg2;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Use the CME to iterate (actually just use) one entry
    // in it's list and stop. This is done to extract the content
    // argument for object 1
    eReturnCode = CME_eIterateContent (
        psContentEntry1->psObj->hDecoder,
        psContentEntry1->hCMEContent,
        (CME_ITERATOR_CALLBACK)bContentArgIterator,
        &pvContentArg1
            );
    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Use the CME to iterate (actually just use) one entry
        // in it's list and stop. This is done to extract the content
        // argument for object 2
        eReturnCode = CME_eIterateContent (
            psContentEntry2->psObj->hDecoder,
            psContentEntry2->hCMEContent,
            (CME_ITERATOR_CALLBACK)bContentArgIterator,
            &pvContentArg2
                );
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Now that we have extracted both arguments, we can call the
            // registered compare handler
            n16Result =
                psContentEntry1->psObj->sRegistered.n16CompareHandler(
                    pvContentArg1, pvContentArg2);
        }
    }

    return n16Result;
}

/*******************************************************************************
*
*   bRemoveAllContentIterator
*
* An iterator callback function which removes content
*
*****************************************************************************/
static BOOLEAN bRemoveAllContentIterator (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        )
{
    CEML.eRemove(hCEML);

    // Keep looking
    return TRUE;
}


