/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CCache implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms.h"
#include "sms_event.h"
#include "sms_event_types.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "channel_obj.h"
#include "category_obj.h"
#include "channel_art_mgr_obj.h"
#include "cdo_obj.h"
#include "radio.h"
#include "report_obj.h"

#include "ccache.h"
#include "_ccache.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CCACHE_hCreate
*
*****************************************************************************/
CCACHE_OBJECT CCACHE_hCreate (
    DECODER_OBJECT hDecoder,
    const char *pacName
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)CCACHE_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bOwner;
    CD_OBJECT hCDO;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == FALSE)
    {
        // Error!
        return CCACHE_INVALID_OBJECT;
    }

    // Create an instance of this object (child of the specified DECODER)
    psObj = (CCACHE_OBJECT_STRUCT *)
        SMSO_hCreate(
            CCACHE_OBJECT_NAME,
            sizeof(CCACHE_OBJECT_STRUCT),
            (SMS_OBJECT)hDecoder, // Child
            FALSE); // No Lock (ignored)
    if(psObj == NULL)
    {
        // Error!
        return CCACHE_INVALID_OBJECT;
    }

    // Initialize object...

    // Extract & Initialize decoder information
    psObj->hDecoder = hDecoder;

    // Initialize the number of derived categories
    psObj->sCategory.n16NumDerivedCategories = 0;

    // Create a CID pool to use with this CCACHE
    psObj->hCidPool = CID_hCreatePool((SMS_OBJECT)psObj);
    if(psObj->hCidPool == NULL)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Construct a unique name for this channel list.
    snprintf(&acName[0],
            sizeof(acName),
            CCACHE_OBJECT_NAME":%s:ChanListByChannelId",
            pacName);

    // Create default channel list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sChannel.hChannelListByChannelId,
        &acName[0],
        CHANNEL_n16CompareChannelIds,
        (OSAL_LL_OPTION_BINARY_SEARCH | OSAL_LL_OPTION_UNIQUE)
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // The channel list sorted by channel id is the default
    // list used for navigation activities.
    psObj->sChannel.hCurrentNavigationChannelList =
        psObj->sChannel.hChannelListByChannelId;

    // Construct a unique name for this channel list.
    snprintf(&acName[0],
            sizeof(acName),
            CCACHE_OBJECT_NAME":%s:ChanListByServiceId",
            pacName);

    // Create channel list (by service id)
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sChannel.hChannelListByServiceId,
        &acName[0],
        CHANNEL_n16CompareServiceIds,
        (OSAL_LL_OPTION_BINARY_SEARCH | OSAL_LL_OPTION_UNIQUE)
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Construct a unique name for this all channel notifier list.
    snprintf(&acName[0],
            sizeof(acName),
            CCACHE_OBJECT_NAME":%s:AllChanNotifyList",
            pacName);

    // Create the "All Channels" Notification list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sChannel.hAllChanNotificationList,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16CompareAllChannelNotifers,
        (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE)
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Construct a unique name for this category list.
    snprintf(&acName[0],
            sizeof(acName),
            CCACHE_OBJECT_NAME":%s:CatList",
            pacName);

    // Create category list
    psObj->sCategory.n16NumInvalidCategories = 0;
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sCategory.hCategoryList,
        &acName[0],
        CATEGORY_n16CompareCategoryIds,
        (OSAL_LL_OPTION_BINARY_SEARCH | OSAL_LL_OPTION_UNIQUE)
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Construct a unique name for this all category notifier list.
    snprintf(&acName[0],
            sizeof(acName),
            CCACHE_OBJECT_NAME":%s:AllCatNotifyList",
            pacName);

    // Create the "All categories" Notification list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sCategory.hAllCatNotificationList,
        &acName[0],
        (OSAL_LL_COMPARE_HANDLER)n16CompareAllCategoryNotifers,
        (OSAL_LL_OPTION_CIRCULAR | OSAL_LL_OPTION_UNIQUE)
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Initialize category dummy information
    psObj->hDummyCategory =
        CATEGORY_hCreateCategory(
            psObj->hDecoder, (SMS_OBJECT)psObj,
            CATEGORY_TYPE_BROADCAST, GsRadio.sCategory.un16BroadcastIdMin,
            "dummy", "dummy", "dummy", 0, FALSE, FALSE);
    if(psObj->hDummyCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Initialize channel dummy information
    psObj->hUnsubscribedChannel =
        CHANNEL_hCreate(
            (CCACHE_OBJECT)psObj, SERVICE_INVALID_ID, CHANNEL_INVALID_ID);
    if(psObj->hUnsubscribedChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Update this channel's  info using default un-subscribed attributes
    CDO_vUpdate(psObj->hUnsubscribedChannel,
        GacUnSubscribedTextDefault, CDO_FIELD_ALL);

    // Clear out the CDO (make it non-program)
    hCDO = CHANNEL.hCDO(psObj->hUnsubscribedChannel);
    CDO_bAssignType(hCDO, CDO_NON_PROGRAM);

    // Use the unsubscribed channel as a dummy channel as well
    psObj->phDummyChannel = &psObj->hUnsubscribedChannel;

    // Initialize channel dummy information
    psObj->hSubscriptionAlertChannel =
        CHANNEL_hCreate(
            (CCACHE_OBJECT)psObj, SERVICE_INVALID_ID, CHANNEL_INVALID_ID);
    if(psObj->hSubscriptionAlertChannel == CHANNEL_INVALID_OBJECT)
    {
        // Error!
        CCACHE_vDestroy((CCACHE_OBJECT)psObj);
        return CCACHE_INVALID_OBJECT;
    }

    // Set channel attribute to indicate it is subscribed.
    CHANNEL_eSetAttribute(
        psObj->hSubscriptionAlertChannel,
        CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED
            );

    // Clear out the CDO (make it non-program)
    hCDO = CHANNEL.hCDO(psObj->hSubscriptionAlertChannel);
    CDO_bAssignType(hCDO, CDO_NON_PROGRAM);

    // Initialize the channel art service handle
    psObj->hChannelArtService = CHANNEL_ART_SERVICE_INVALID_OBJECT;

    // Initialize cache as available
    psObj->bAvailable = TRUE;

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        CCACHE_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
        NULL);

    // Restore CCACHE persistent data
    vRestoreACOData(psObj);

    return (CCACHE_OBJECT)psObj;
}

/*****************************************************************************
*
*   CCACHE_vDestroy
*
*****************************************************************************/
void CCACHE_vDestroy (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // Filter out all further ccache updates
    SMSU_tFilter(&psObj->sEvent, CCACHE_OBJECT_EVENT_ALL);

    // Destroy unsubscribed channel
    if(psObj->hUnsubscribedChannel != CHANNEL_INVALID_OBJECT)
    {
        CHANNEL_vDestroy(psObj->hUnsubscribedChannel);
        psObj->hUnsubscribedChannel = CHANNEL_INVALID_OBJECT;
    }

    // Destroy subscription alert channel
    if(psObj->hSubscriptionAlertChannel != CHANNEL_INVALID_OBJECT)
    {
        CHANNEL_vDestroy(psObj->hSubscriptionAlertChannel);
        psObj->hSubscriptionAlertChannel = CHANNEL_INVALID_OBJECT;
    }

    // Destroy dummy channel
    if(*psObj->phDummyChannel != CHANNEL_INVALID_OBJECT)
    {
        CHANNEL_vDestroy(*psObj->phDummyChannel);
        *psObj->phDummyChannel = CHANNEL_INVALID_OBJECT;
    }

    // Destroy dummy category
    if(psObj->hDummyCategory != CATEGORY_INVALID_OBJECT)
    {
        CATEGORY_vDestroy(psObj->hDummyCategory);
        psObj->hDummyCategory = CATEGORY_INVALID_OBJECT;
    }

    // Check if category list exists
    if(psObj->sCategory.hCategoryList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove(unregister) any remaining "all categories" notifications
        OSAL.eLinkedListIterate(
            psObj->sCategory.hCategoryList,
            bRemoveCategoryNotifierIterator,
            (void *)psObj );
    }

    // Check if default channel list (by service id) exists
    if(psObj->sChannel.hChannelListByServiceId != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove(unregister) any remaining "all channels" notifications
        OSAL.eLinkedListIterate(
            psObj->sChannel.hChannelListByServiceId,
            bRemoveChannelNotifierIterator,
            (void *)psObj );
    }

    // Destroy "all categories" notification list if one exists
    if(psObj->sCategory.hAllCatNotificationList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list and destroy each
        // channel notifiction structure
        OSAL.eLinkedListRemoveAll(
            psObj->sCategory.hAllCatNotificationList, vDestroyCategoryNotifier);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->sCategory.hAllCatNotificationList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->sCategory.hAllCatNotificationList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy "all channels" notification list if one exists
    if(psObj->sChannel.hAllChanNotificationList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list and destroy each
        // channel notification structure
        OSAL.eLinkedListRemoveAll(
            psObj->sChannel.hAllChanNotificationList, vDestroyChannelNotifier);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->sChannel.hAllChanNotificationList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->sChannel.hAllChanNotificationList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy category list if one exists
    if(psObj->sCategory.hCategoryList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list and destroy each category
        OSAL.eLinkedListRemoveAll(
            psObj->sCategory.hCategoryList, vDestroyCategory);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->sCategory.hCategoryList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->sCategory.hCategoryList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // ***********************************************************************
    // TODO: Note we will do multiple list removals here.
    // ***********************************************************************

    // Destroy channel-id channel list if one exists
    if(psObj->sChannel.hChannelListByChannelId != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list
        OSAL.eLinkedListRemoveAll(
            psObj->sChannel.hChannelListByChannelId, NULL);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->sChannel.hChannelListByChannelId);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->sChannel.hChannelListByChannelId = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Destroy service-id channel list if one exists
    if(psObj->sChannel.hChannelListByServiceId != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries in the linked list and destroy each channel
        OSAL.eLinkedListRemoveAll(
            psObj->sChannel.hChannelListByServiceId, vDestroyChannel);
        psObj->sCategory.n16NumInvalidCategories = 0;

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->sChannel.hChannelListByServiceId);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->sChannel.hChannelListByServiceId = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // ***********************************************************************

    // Destroy the SMS Update object
    SMSU_vDestroy(&psObj->sEvent);

    // Destroy the CID-pool if one exists
    if(psObj->hCidPool != CID_POOL_INVALID_OBJECT)
    {
        CID_vDestroyPool(psObj->hCidPool);
        psObj->hCidPool = CID_POOL_INVALID_OBJECT;
    }

    // Initialize structure
    psObj->hDecoder = DECODER_INVALID_OBJECT;

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hCCache);

    return;
}

/*****************************************************************************
*
*   CCACHE_vUpdateChannelArt
*
*****************************************************************************/
void CCACHE_vUpdateChannelArt (
    CCACHE_OBJECT hCCache,
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CCACHE_UPDATE_TYPE_ENUM eUpdateType
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if(bOwner == TRUE)
    {
        CCACHE_ART_UPDATE_STRUCT sUpdateInfo;
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        // Set up the update info struct
        sUpdateInfo.eUpdateType = eUpdateType;
        sUpdateInfo.hChannelArtService = hChannelArtService;

        // copy the updated channel art service handle
        psObj->hChannelArtService = hChannelArtService;

        // Iterate over the channels (and their CDOs) and update
        // art as necessary. Note that channels will always need
        // some sort of update (whether the service, channel art
        // or album art products changed.) For channel art, this
        // will not set channel events; we manually do that later.

        OSAL.eLinkedListIterate(
            psObj->sChannel.hChannelListByServiceId,
            bUpdateAllChannelArt,
            (void *)&sUpdateInfo
                );

        // There's no need to iterate categories if this is an
        // album art-only update.

        if ( ( CCACHE_UPDATE_TYPE_ART_SERVICE == eUpdateType ) ||
             ( CCACHE_UPDATE_TYPE_CHAN_ART == eUpdateType ) )
        {
            // Iterate through all the instantiated categories and have
            // them update their CATEGORY art handles
            OSAL.eLinkedListIterate(
                psObj->sCategory.hCategoryList,
                bUpdateAllCategoryArt,
                (void *)&sUpdateInfo
                    );

            if ( CCACHE_UPDATE_TYPE_CHAN_ART == eUpdateType )
            {
                // Now that both the channel art and category art has been
                // updated, we can send out a channel art update event for
                // our channels. On product stop, certain callbacks may try
                // to use BOTH types of art (chan & category) when only
                // once set of art handles has been updated, leading to
                // segfaults. By manually forcing the channel event to
                // occur *AFTER* all updates have been done, we avoid any
                // possible race conditions.
                OSAL.eLinkedListIterate(
                    psObj->sChannel.hChannelListByServiceId,
                    bSetChannelEvent,
                    (void *)&sUpdateInfo
                        );
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   CCACHE_vCleanChannelsEpgData
*
*****************************************************************************/
void CCACHE_vCleanChannelsEpgData (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        // Iterate through all instantiated channels
        // and clean their EPG CHANNEL handles.
        OSAL.eLinkedListIterate(
            psObj->sChannel.hChannelListByServiceId,
            bUpdateChannelEpgData,
            (void *) EPG_INVALID_CHANNEL_OBJECT);
    }

    return;
}

/*****************************************************************************
*
*   CCACHE_hChannel
*
*   When ptChannelId == NULL, pn16Offset must != NULL
*   If the ptChannelId is NULL then the pn16Offset is used to identify the
*   position from which we will search from the first channel in the
*   cache (lowest channel number) looking for and returning the *pn16Offset
*   channel's handle from the beginning. Return the normalized offset value
*   via the provided pn16Offset pointer.
*
*   When ptChannelId != NULL, pn16Offset may == NULL
*   If the ptChannelId is Non-NULL this means the caller would like to return
*   the channel handle of the channel which is *pn16Offset from the provided
*   channel id if provided.	If pn16Offset is not provided (NULL) we assume
*   this offset is 0 but the behavior is the same. ptChannelId will also be
*   updated with the channel located at the offset from the originally
*   provided ptChannelId (if provided).
*
*   NOTE! This is the one and only way a CHANNEL handle can be obtained from
*   the cache. If the caller wishes to obtain this CHANNEL handle and keep it
*   it is permitted, but the caller must indicate their desire using the
*   CHANNEL object API bRegisterNotification/vUnregisterNotification. This
*   is regardless if they actually want to be asynchronously notified of
*   CHANNEL content changes. Any CHANNEL object which has NOT been registered
*   for should be considered volatile and may be purged from the cache
*   without any notification or indication this has occurred.
*
*   Be aware that this does not mean that even if you register for a CHANNEL
*   notification you can freely access the CHANNEL object contents without
*   being in the owing DECODER->CCACHE context. This is still the case since
*   even though the value of the handle (it's reference) is guaranteed not
*   to change, the contents of a CHANNEL may and probably will change
*   outside the context of the DECODER.
*
*   Additionally note that this look-up uses channel id, which is the native
*   from the channels are stored in. Thus we use binary search-trees in
*   this routine to find the channel and offsets. It is preferred to use this
*   API whenever the caller needs to know information about a channel id
*   and it's relative placement (offset) within the channel-id list.
*
*****************************************************************************/
CHANNEL_OBJECT CCACHE_hChannel (
    CCACHE_OBJECT hCCache,
    CHANNEL_ID *ptChannelId,
    N16 *pn16Offset
        )
{
    BOOLEAN bAvailable;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CHANNEL_ID tChannelId = 0;
    N16 n16Offset = 0;

    // Re-condition inputs as needed.

    // If offset pointer is NULL we just use our own internal value
    // which is the start the offset at 0. Otherwise, use caller
    // provided offset
    if(pn16Offset == NULL)
    {
        pn16Offset = &n16Offset;
    }

    // At this point pn16Offset is NEVER NULL.

    // If they provided a reference to a 'current' channel then use it.
    // Otherwise we just use the default (0) and they want the nth
    // channel in the list from it.
    if(ptChannelId == NULL)
    {
        // If offset pointer is NULL we just use our own internal value
        // which is channel 0. Otherwise, use caller provided channel id
        ptChannelId = &tChannelId;
    }

    // Use this channel id, if it is valid, otherwise bail
    if(*ptChannelId >= CHANNEL_INVALID_ID)
    {
        // Error!
        return CHANNEL_INVALID_OBJECT;
    }

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;
        OSAL_LINKED_LIST_ENTRY hFirstEntry, hThisEntry;
        CHANNEL_OBJECT hThisChannel;
        N16 n16NumTotalValidChannels = 0;

        // Capture cache states and retrieve number of valid channels
        CCACHE_vStats(hCCache, &n16NumTotalValidChannels, NULL);

        if ( n16NumTotalValidChannels == 0 )
        {
            // don't want to divide by zero, so just stop
            return CHANNEL_INVALID_OBJECT;
        }

        // Offset cannot be larger than the number of channels we have.
        // So we just truncate it now and re-apply sign.
        *pn16Offset = ((abs(*pn16Offset) % n16NumTotalValidChannels) *
            ((*pn16Offset < 0) ? -1 : 1));

        // They want to find the nth channel from the provided channel id
        // in either direction. First we need to locate the base channel.
        // Note: This is where we use the default channel list for navigation.
        hThisChannel = hChannelFromIds(
            psObj, &hThisEntry, SERVICE_INVALID_ID, *ptChannelId);
        if(CHANNEL_INVALID_OBJECT != hThisChannel) // found
        {
            // Found the reference channel, now find the nth
            // channel from it.
            N16 n16TargetOffset = *pn16Offset;
            N16 n16OffsetFromReference = 0;

            // Initialize first entry
            hFirstEntry = hThisEntry;

            do
            {
                BOOLEAN bNeedsUpdated;

                // Now look for target channel based on
                // direction and offset
                if(n16TargetOffset > 0) // forward
                {
                    hThisEntry = OSAL.hLinkedListNext(
                        hThisEntry, (void **)&hThisChannel);

                    // Modify offset only if this is a valid channel.
                    // Meaning it must have both a valid channel
                    // and service id
                    bNeedsUpdated = CHANNEL_bNeedsUpdated(hThisChannel);
                    if(bNeedsUpdated == FALSE)
                    {
                        n16TargetOffset--;
                        n16OffsetFromReference++;
                    }
                }
                else if(n16TargetOffset < 0) // backward
                {
                    hThisEntry = OSAL.hLinkedListPrev(
                        hThisEntry, (void **)&hThisChannel);

                    // Modify offset only if this is a valid channel.
                    // Meaning it must have both a valid channel
                    // and service id
                    bNeedsUpdated = CHANNEL_bNeedsUpdated(hThisChannel);
                    if(bNeedsUpdated == FALSE)
                    {
                        n16TargetOffset++;
                        n16OffsetFromReference--;
                    }
                }

                if(0 == n16TargetOffset) // right here
                {
                    // This is the channel we are looking for
                    hChannel = hThisChannel;

                    // This is the offset it is at from the reference
                    *pn16Offset = n16OffsetFromReference;

                    // Return channel id of target channel to caller
                    *ptChannelId = CHANNEL.tChannelId(hChannel);

                    // break out of outer loop
                    break;
                }

            } while(hFirstEntry != hThisEntry);

        } // Did not find base-channel

    } // CCache is not available

    // Return channel handle found
    return hChannel;
}

/*****************************************************************************
*
*   CCACHE_tChannelIdFromServiceId
*
*****************************************************************************/
CHANNEL_ID CCACHE_tChannelIdFromServiceId (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId
        )
{
    BOOLEAN bAvailable;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        CHANNEL_OBJECT hChannel;

        // based on the service id, get the channel, but don't
        // create it if it doesn't already exist
        hChannel = CCACHE_hChannelFromIds(
            hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            // found the channel, get the channel id
            tChannelId = CHANNEL.tChannelId(hChannel);
        }
    }

    // Return channel ID found
    return tChannelId;
}

/*****************************************************************************
*
*   CCACHE_hChannelFromIds
*
*   Locates (and optionally creates) a CHANNEL_OBJECT based first on
*   ther SERVICE_ID provided, then on the CHANNEL_ID provided.
*
*****************************************************************************/
CHANNEL_OBJECT CCACHE_hChannelFromIds (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bCreateIfNotFound
        )
{
    BOOLEAN bAvailable;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;
        OSAL_LINKED_LIST_ENTRY hEntry;

        // Try to find this channel
        hChannel = hChannelFromIds(psObj, &hEntry, tServiceId, tChannelId);

        // Check if no channel was found and the want to create one.
        if ((CHANNEL_INVALID_OBJECT == hChannel) && (TRUE == bCreateIfNotFound))
        {
            // create a channel for this service id
            // and/or chan id provided
            hChannel = hCreateChannel(psObj, tServiceId, tChannelId, TRUE);
        }
    }

    // Return channel found
    return hChannel;
}

/*****************************************************************************
*
*   CCACHE_hCreateChannel
*
*****************************************************************************/
CHANNEL_OBJECT CCACHE_hCreateChannel(
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bNotify
        )
{
    BOOLEAN bAvailable;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if (bAvailable == TRUE)
    {
        // De-reference object
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        // create a channel for this service id
        // and/or chan id provided
        hChannel = hCreateChannel(psObj, tServiceId, tChannelId, bNotify);
    }

    // Return channel created
    return hChannel;
}

/*****************************************************************************
*
*   CCACHE_hCategory
*
*   Locate the category at n16Offset from the reference category id provided
*   or from the first category in the list depending up on input.
*
*   This is the one and only way a CATEGORY handle can be obtained from the
*   cache. Nobody should ever try to "keep" a CATEGORY handle outside the
*   context of the DECODER for which it belongs. CATEGORY object handles can
*   change at any time outside that context. CATEGORY handles can be obtained
*   and used immediately but cannot be "retained". Doing so is in violation
*   of proper CATEGORY handle usage.
*
*   It is possible that a caller might request the next or previous category
*   found from the invalid category.  In this case, the next category would
*   be the first valid category, and the previous category would be the last.
*   Thus an input category id of CATEGORY_INVALID_ID is valid and must be
*   used directly.
*
*****************************************************************************/
CATEGORY_OBJECT CCACHE_hCategory (
    CCACHE_OBJECT hCCache,
    CATEGORY_ID *ptCategoryId,
    N16 n16Offset
        )
{
    BOOLEAN bAvailable;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    CATEGORY_ID tCategoryId;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        N16 n16InputCategoryOffset;
        BOOLEAN bIsBroadcastCategory, bOk;
        CCACHE_DERIVED_CATEGORY_OFFSET_STRUCT sOffset;
        CATEGORY_ID tFirstBroadcastCatId;
        N16 n16NumTotalValidCategories = 0;
        N16 n16NumValidBroadcastCategories = 0;

        // Get the starting category id for broadcast
        bOk = CATEGORY_bCategoryIdRange(
            CATEGORY_TYPE_BROADCAST,
            &tFirstBroadcastCatId, NULL );

        if (bOk == FALSE)
        {
            return CATEGORY_INVALID_OBJECT;
        }

        // Initialize the category ID to the first valid
        // broadcast category Id
        tCategoryId = tFirstBroadcastCatId;

        // If they provided a reference to a 'current' category then use it.
        // Otherwise we just use the default (0)
        if(ptCategoryId != NULL)
        {
            // Sanity check -- can't ask for the invalid
            // category directly
            if ((*ptCategoryId == CATEGORY_INVALID_ID) &&
                (n16Offset == 0))
            {
                // This is the invalid category
                return CATEGORY_INVALID_OBJECT;
            }

            // Use caller's category id, if it is valid
            if(*ptCategoryId < CATEGORY_INVALID_ID)
            {
                // Use this category as a reference
                tCategoryId = *ptCategoryId;
            }
            // If the caller's category Id is INVALID, we'll
            // just use category id 0 as our starting point
            // and adjust the offset accordingly
            else if(*ptCategoryId == CATEGORY_INVALID_ID)
            {
                if (n16Offset > 0)
                {
                    // Reduce the offset by one if
                    // it is positive, since the
                    // category Id is automatically
                    // set to 0 in this case
                    n16Offset--;
                }
            }
        }

        // Capture cache states and compute num broadcast categories
        CCACHE_vStats(hCCache, NULL, &n16NumTotalValidCategories);
        n16NumValidBroadcastCategories =  n16NumTotalValidCategories -
            psObj->sCategory.n16NumDerivedCategories;

        // Find out if the input category is broadcast
        // or if it is something else
        bIsBroadcastCategory = CATEGORY_bIdInRange(
            CATEGORY_TYPE_BROADCAST,
            tCategoryId );
        if (bIsBroadcastCategory == TRUE)
        {
            // Get the input category's offset
            // from the beginning of the broadcast
            // categories
            n16InputCategoryOffset =
                RADIO_n16CategoryOffsetFromId(psObj->hDecoder, tCategoryId);
        }
        else
        {
            // The input category Id is for
            // a derived category, so its offset
            // is at least n16NumValidBroadcastCategories
            sOffset.n16DerivedOffset = 0;
            sOffset.tTargetCategory = tCategoryId;
            sOffset.bSearchByOffset = FALSE;
            sOffset.bFound = FALSE;

            // Calculate the input category's offset
            // from the beginning of the derived
            // categories
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->sCategory.hCategoryList,
                bDerivedCategoryOffsetIterator,
                (void *)(size_t)&sOffset );

            if ((eReturnCode == OSAL_SUCCESS) &&
                (sOffset.bFound == TRUE ))
            {
                n16InputCategoryOffset =
                    n16NumValidBroadcastCategories
                    + sOffset.n16DerivedOffset;
            }
            else
            {
                // We have a problem
                return CATEGORY_INVALID_OBJECT;
            }
        }

        // Now we have the offset for the input category

        // Determine if this is a direct access with no offset
        if (n16Offset != 0)
        {
            N16 n16Direction;

            // Set step direction
            n16Direction = n16Offset > 0 ? +1 : -1;

            // fail gracefully instead of potentially dividing by zero
            if ( n16NumTotalValidCategories == 0 )
            {
                return CATEGORY_INVALID_OBJECT;
            }

            // Offset cannot be larger than the number of categories we have.
            // So we just truncate it now
            n16Offset = abs(n16Offset) % n16NumTotalValidCategories;

            if ((n16Direction < 0) && (n16Offset != 0))
            {
                // If the offset was negative, finish up
                // this calculation (ensures all offsets
                // end up as positive values)
                n16Offset = n16NumTotalValidCategories - n16Offset;
            }

            // Now, compute the final offset (it's a function of the
            // input offset plus the offset at which the provided
            // category resides (always a positive value)
            n16Offset = (n16Offset + n16InputCategoryOffset)
                % n16NumTotalValidCategories;

            if (n16Offset < n16NumValidBroadcastCategories)
            {
                // Search for a broadcast category at this offset using this
                // initial category ID
                tCategoryId =
                    RADIO_tCategoryIdFromOffset(psObj->hDecoder,
                        GsRadio.sCategory.un16BroadcastIdMin, n16Offset);
            }
            else
            {
                // Search for a derived category at this offset
                sOffset.n16DerivedOffset = 0;
                sOffset.n16TargetOffset =
                    n16Offset - n16NumValidBroadcastCategories;
                sOffset.tTargetCategory = CATEGORY_INVALID_ID;
                sOffset.bSearchByOffset = TRUE;
                sOffset.bFound = FALSE;

                // Calculate the input category's offset
                // from the beginning of the derived
                // categories
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->sCategory.hCategoryList,
                    bDerivedCategoryOffsetIterator,
                    (void *)(size_t)&sOffset );

                if ((eReturnCode == OSAL_SUCCESS) &&
                    (sOffset.bFound == TRUE ))
                {
                    tCategoryId = sOffset.tTargetCategory;
                }
                else
                {
                    // Error!
                    return CATEGORY_INVALID_OBJECT;
                }
            }
        }

        // Find this category
        hCategory = hFindCategoryLocal(psObj, tCategoryId);

        // If they provided a way to return the category number, then do so
        if(ptCategoryId != NULL)
        {
            // Return category id to caller
            *ptCategoryId = tCategoryId;
        }
    }

    return hCategory;
}

/*****************************************************************************
*
*   CCACHE_tAddCategory
*
*****************************************************************************/
CATEGORY_ID CCACHE_tAddCategory (
    CCACHE_OBJECT hCCache,
    CATEGORY_TYPE_ENUM eCategoryType,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        )
{
    BOOLEAN bAvailable, bOk;
    // search from top of list
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    CATEGORY_ID tCurrentId;
    CCACHE_OBJECT_STRUCT *psObj = (CCACHE_OBJECT_STRUCT *)hCCache;
    CATEGORY_ID tStartCategoryId;
    CATEGORY_ID tEndCategoryId;

    // Get the ID range for this category type
    bOk = CATEGORY_bCategoryIdRange(
            eCategoryType,
            &tStartCategoryId,
            &tEndCategoryId );

    if (bOk == FALSE)
    {
        // There are input parameter errors!
        return CATEGORY_INVALID_ID;
    }

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        if (psObj->sCategory.hCategoryList == OSAL_INVALID_OBJECT_HDL)
        {
            return CATEGORY_INVALID_ID;
        }

        // At this point we know the start of the specifed
        // section (tStartCategoryId).
        for ( tCurrentId = tStartCategoryId;
              tCurrentId < (tEndCategoryId + 1);
              tCurrentId++ )
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Check if this category exists in our list.
            CATEGORY_bUpdateId(psObj->hDummyCategory, tCurrentId);
            eReturnCode =
                OSAL.eLinkedListSearch(
                    psObj->sCategory.hCategoryList,
                    &hEntry,
                    psObj->hDummyCategory );

            if ( eReturnCode == OSAL_OBJECT_NOT_FOUND )
            {
                // Create the new category using this unused Id
                hCategory = hCreateDerivedCategory(
                    psObj,
                    eCategoryType,
                    tCurrentId,
                    pacLongName,
                    pacMediumName,
                    pacShortName,
                    tInitialNumChannels,
                    bUniqueItems,
                    bReplace);

                // Exit
                break;
            }
            else if ( eReturnCode == OSAL_SUCCESS )
            {
                // No big deal, just continue to iterate
            }
            else
            {
                // There were issues, make sure the category is not created
                return CATEGORY_INVALID_ID;
            }
        }
    }

    // Prepare to return a category Id to the caller
    tCurrentId = CATEGORY_INVALID_ID;

    if (hCategory != CATEGORY_INVALID_OBJECT)
    {
        // If we have a valid category get its Id
        tCurrentId = CATEGORY.tGetCategoryId(hCategory);
    }

    return tCurrentId;
}

/*****************************************************************************
*
*   CCACHE_vRemoveCategory
*
*   This function only removes the specifed category from the hCategoryList
*   kept by the cache.
*****************************************************************************/
void CCACHE_vRemoveCategory (
    CCACHE_OBJECT hCCache,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bAvailable;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;
    OSAL_LINKED_LIST_ENTRY hThisLLEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY; // search from start(top) of list

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        // If the category list exists for the given ccache
        if ( psObj->sCategory.hCategoryList != OSAL_INVALID_OBJECT_HDL )
        {
            // Check if this category exists in our list.
            CATEGORY_bUpdateId(psObj->hDummyCategory, tCategoryId);
            eReturnCode = OSAL.eLinkedListSearch(
                psObj->sCategory.hCategoryList,
                &hThisLLEntry,
                psObj->hDummyCategory );

            // We found the category and the handle to that ll entry now exists
            // in hThisLLEntry
            if ( eReturnCode == OSAL_SUCCESS )
            {
                CATEGORY_OBJECT hCategory;
                CATEGORY_TYPE_ENUM eType;

                // Extract the category object
                hCategory = (CATEGORY_OBJECT)
                    OSAL.pvLinkedListThis( hThisLLEntry );

                // Get this category's type
                eType = CATEGORY.eType( hCategory );

                // If this is not a broadcast category
                // reduce the number of derived categories
                // in the ccache (if possible)
                if ((eType != CATEGORY_TYPE_BROADCAST) &&
                    (psObj->sCategory.n16NumDerivedCategories > 0))
                {
                    psObj->sCategory.n16NumDerivedCategories--;
                }

                eReturnCode = OSAL.eLinkedListRemove( hThisLLEntry );
                if (eReturnCode == OSAL_SUCCESS)
                {
                    if (hThisLLEntry == psObj->sCategory.hCurCatEntry)
                    {
                        psObj->sCategory.hCurCatEntry = 
                            OSAL_INVALID_LINKED_LIST_ENTRY;
                    }
                }
                else
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CCACHE_OBJECT_NAME": Couldn't Remove Category (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
            else
            {
                return;
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   CCACHE_bRemoveChannel
*
*****************************************************************************/
BOOLEAN CCACHE_bRemoveChannel(
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bAvailable, bSuccess = FALSE;

    printf("CCACHE: Remvoing a channel\n");

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            CCACHE_OBJECT_STRUCT *psObj =
                (CCACHE_OBJECT_STRUCT *)hCCache;
            BOOLEAN bBlocked, bRebrowse = FALSE,
                bRemovedTunedChannel = FALSE,
                bRemovedLastTunedChannel = FALSE;
            SERVICE_ID tServiceId;

            // Get service id of the channel to be removed
            tServiceId = CHANNEL.tServiceId(hChannel);

            // Block future CHANNEL object notifications
            bBlocked = CHANNEL_bBlockNotifications(hChannel);
            if(bBlocked == TRUE)
            {
                SERVICE_ID tTunedServiceId;
                SERVICE_ID tLastTuneServiceId;

                tTunedServiceId =
                    DECODER.tCurrentServiceId(psObj->hDecoder);

                tLastTuneServiceId =
                    DECODER_tLastTunedServiceId(psObj->hDecoder);

                if (tTunedServiceId == tServiceId)
                {
                    // the channel being removed IS our tuned channel
                    bRemovedTunedChannel = TRUE;
                }
                else if (tServiceId == tLastTuneServiceId)
                {
                    // the channel being removed IS our last tuned channel
                    bRemovedLastTunedChannel = TRUE;
                }
                else
                {
                    // the tuned channel isn't the one being removed,
                    // what about the browsed channel?
                    SERVICE_ID tBrowsedServiceId;

                    tBrowsedServiceId =
                        DECODER_tBrowsedServiceId(psObj->hDecoder);
                    if (tBrowsedServiceId == tServiceId)
                    {
                        bRebrowse = TRUE;
                    }
                }

                // Clear the contents of this channel sans service id,
                // and any application provided attributes.
                CHANNEL_vClearBroadcast(hChannel);
                CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_REMOVED);

                // Let'em rip!
                CHANNEL_vReleaseNotifications(hChannel);

                if (TRUE == bRemovedTunedChannel)
                {
                    TUNEMIX_OBJECT hActive;

                    // If it is actually the tunemix, we need not  to re-tune
                    // to default or safe channel here
                    hActive = TUNEMIX.hActive(psObj->hDecoder);
                    if (TUNEMIX_INVALID_OBJECT == hActive)
                    {
                        // The tuned channel no longer exists.
                        DECODER_vTunedChanNoLongerAvail(psObj->hDecoder);
                    }
                }

                if (bRemovedLastTunedChannel == TRUE)
                {
                    // The last tuned channel no longer exists.
                    DECODER_vLastTunedChanNoLongerAvail(psObj->hDecoder);
                }

                if (TRUE == bRebrowse)
                {
                    BROWSE_OBJECT hBrowse;

                    printf(CCACHE_OBJECT_NAME": Current Browsed Service Id: %d has been removed\n",
                        tServiceId);

                    hBrowse = DECODER_hBrowse(psObj->hDecoder);
                    if (hBrowse != BROWSE_INVALID_OBJECT)
                    {
                        BROWSE_TYPE_ENUM eBrowseType;

                        eBrowseType = BROWSE_eGetMode(hBrowse);
                        if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
                        {
                            CHANNEL_ID tTunedChannelId;

                            tTunedChannelId =
                                DECODER.tCurrentChannelId(psObj->hDecoder);

                            printf(CCACHE_OBJECT_NAME": Re-Browsing to the Tuned Channel: %d\n",
                                tTunedChannelId);

                            // the channel being removed IS our browsed channel
                            // so browse back to the tuned channel
                            DECODER.eBrowseToChannel(
                                psObj->hDecoder, tTunedChannelId);
                        }
                    }
                }

                // Channel is removed
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}


/*****************************************************************************
*
*   CCACHE_vStats
*
*****************************************************************************/
void CCACHE_vStats (
    CCACHE_OBJECT hCCache,
    N16 *pn16NumValidChannels,
    N16 *pn16NumValidCategories
        )
{
    BOOLEAN bAvailable;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        if(pn16NumValidChannels != NULL)
        {
            UN32 un32Items = 0;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Initialize return value
            *pn16NumValidChannels = 0;

            eReturnCode = OSAL.eLinkedListItems(
                psObj->sChannel.hChannelListByChannelId, &un32Items);
            if(eReturnCode == OSAL_SUCCESS)
            {
                *pn16NumValidChannels = (N16)un32Items;
            }
        }

        if(pn16NumValidCategories != NULL)
        {
            UN32 un32Items = 0;
            N16 n16Items;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Initialize return value
            *pn16NumValidCategories = 0;

            eReturnCode = OSAL.eLinkedListItems(
                psObj->sCategory.hCategoryList, &un32Items);
            if(eReturnCode == OSAL_SUCCESS)
            {
                n16Items = (N16)un32Items;
                if(n16Items >= psObj->sCategory.n16NumInvalidCategories)
                {
                    // Compute number of valid categories
                    *pn16NumValidCategories =
                        n16Items - psObj->sCategory.n16NumInvalidCategories;
                }
                else
                {
                    // Error! Something went wrong, there can never
                    // be more invalid categories than there are categories
                    // in the list. Demand a recount.
                    *pn16NumValidCategories =
                        n16CountNumInvalidCategories(psObj);
                }
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   CCACHE_bSortChannelLists
*
*****************************************************************************/
BOOLEAN CCACHE_bSortChannelLists (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bAvailable, bSuccess = FALSE;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // *******************************************************************
        // TODO: Note we will add multiple list sorting here
        // *******************************************************************

        // Sort lists
        eReturnCode = OSAL.eLinkedListSort(
            psObj->sChannel.hChannelListByServiceId,
            CHANNEL_n16CompareServiceIds
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListSort(
                psObj->sChannel.hChannelListByChannelId,
                CHANNEL_n16CompareChannelIds
                    );
        }
        // *******************************************************************

        if (eReturnCode == OSAL_SUCCESS)
        {
            // All went well.
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CCACHE_bSortCategoryList
*
*****************************************************************************/
BOOLEAN CCACHE_bSortCategoryList (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bAvailable, bSuccess = FALSE;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;

    // Verify and check ownership of object
    bAvailable = bIsAvailable(hCCache);
    if(bAvailable == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListSort(
            psObj->sCategory.hCategoryList,
            CATEGORY_n16CompareCategoryIds
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            // All went well.
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CCACHE_bRegisterNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bRegisterNotification (
    CCACHE_OBJECT hCCache,
    CCACHE_EVENT_MASK tEventRequestMask,
    CCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object (doesn't require cache)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        // Add the callback to the update configuration
        SMSU_bRegisterCallback(
            &psObj->sEvent,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg,
            TRUE);

        bReturn = TRUE;
    }

    return bReturn;
}

/*****************************************************************************
*
*   CCACHE_bUnRegisterNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bUnRegisterNotification (
    CCACHE_OBJECT hCCache,
    CCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object (doesn't require cache)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        // Add the callback to the update configuration
        SMSU_bUnregisterCallback(
            &psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg
                );

        bReturn = TRUE;
    }

    return bReturn;
}

/*******************************************************************************
*
*   CCACHE_bRegisterAllChannelsNotification
*
*******************************************************************************/
BOOLEAN CCACHE_bRegisterAllChannelsNotification (
    CCACHE_OBJECT hCCache,
    CHANNEL_EVENT_MASK tEventRequestMask,
    BOOLEAN bImmediateNotify,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;
        CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create the notification structure
        psNotify = (CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *)
                SMSO_hCreate(
                    CCACHE_OBJECT_NAME":AllChansNotify",
                    sizeof(CCACHE_ALL_CHAN_NOTIFICATION_STRUCT),
                    (SMS_OBJECT)hCCache, // Child
                    FALSE); // No Lock (ignored)

        if (psNotify != NULL)
        {
            psNotify->tEventRequestMask = tEventRequestMask;
            psNotify->vEventCallback = vEventCallback;
            psNotify->pvEventCallbackArg = pvEventCallbackArg;
            psNotify->bImmediateNotify = bImmediateNotify;

            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sChannel.hAllChanNotificationList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                psNotify);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListIterate(
                    psObj->sChannel.hChannelListByServiceId,
                    bUpdateChannelNotificationsIterator,
                    (void *)psNotify
                );

                if(eReturnCode == OSAL_SUCCESS || eReturnCode == OSAL_NO_OBJECTS)
                {
                    // All went well.
                    bReturn = TRUE;
                }
                else
                {
                    vDestroyChannelNotifier((void*)psNotify);
                }
            }
            else
            {
                vDestroyChannelNotifier((void*)psNotify);
            }
        }
    }

    return bReturn;
}

/*******************************************************************************
*
*   CCACHE_bUnRegisterAllChannelsNotification
*
*******************************************************************************/
BOOLEAN CCACHE_bUnRegisterAllChannelsNotification (
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;
        CCACHE_ALL_CHAN_NOTIFICATION_STRUCT sNotifyToFind;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Find the notification structure
        sNotifyToFind.vEventCallback = vEventCallback;
        sNotifyToFind.pvEventCallbackArg = pvEventCallbackArg;

        eReturnCode = OSAL.eLinkedListSearch(
            psObj->sChannel.hAllChanNotificationList, &hEntry, &sNotifyToFind);

        if (eReturnCode == OSAL_SUCCESS)
        {
            CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify;

            psNotify = (CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *)
                   OSAL.pvLinkedListThis(hEntry);

            // Iterate through the channels removing this entry
            psNotify->tEventRequestMask = CHANNEL_OBJECT_EVENT_NONE;
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->sChannel.hChannelListByServiceId,
                bUpdateChannelNotificationsIterator,
                (void *)psNotify);

            if(eReturnCode == OSAL_SUCCESS)
            {
                vDestroyChannelNotifier((void*)psNotify);
                OSAL.eLinkedListRemove(hEntry);

                // All went well.
                bReturn = TRUE;
            }
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   CCACHE_eHandleAppChannelNotifyOnChange
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CCACHE_eHandleAppChannelNotifyOnChange (
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
       BOOLEAN bSuccess = TRUE;
       CCACHE_OBJECT_STRUCT *psObj =
           (CCACHE_OBJECT_STRUCT *)hCCache;


       // If the application already had a callback specied, unregister the
       // previous one before doing anything else
       if (psObj->sChannel.vAppNotifierCallback != NULL)
       {
           bSuccess =
               CCACHE_bUnRegisterAllChannelsNotification(
                   hCCache,
                   psObj->sChannel.vAppNotifierCallback,
                   psObj->sChannel.pvAppNotifierCallbackArg);

           if (bSuccess == TRUE)
           {
               // Clear the handles
               psObj->sChannel.vAppNotifierCallback = NULL;
               psObj->sChannel.pvAppNotifierCallbackArg = NULL;
               psObj->sChannel.hCurChanEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
           }
       }

       // If the application specified a callback, then register it.
       // If all they specified was null, then they jsut wanted to remove
       // the current callback, which we handled above.
       if (vNotifierCallback != NULL && bSuccess == TRUE)
       {

           bSuccess =
               CCACHE_bRegisterAllChannelsNotification(
                   hCCache, CHANNEL_OBJECT_EVENT_ALL, FALSE,
                   vNotifierCallback, pvNotifierCallbackArg);

           if (bSuccess == TRUE)
           {
               // Save the handles
               psObj->sChannel.vAppNotifierCallback = vNotifierCallback;
               psObj->sChannel.pvAppNotifierCallbackArg = pvNotifierCallbackArg;
               psObj->sChannel.hCurChanEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
           }
       }

       if (bSuccess == TRUE)
       {
           eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
       }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CCACHE_bCheckNextChannelForAppNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bCheckNextChannelForAppNotification (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner, bMoreToCheck = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        BOOLEAN bNotified = FALSE;
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;
        CHANNEL_OBJECT hChannel;
        OSAL_LINKED_LIST_ENTRY hFirstEntry, hLastEntry;

        // Grab the first entry, also get the handle to the channel
        // at that entry in case it is needed if
        // psObj->sChannel.hCurChanEntry == OSAL_INVALID_LINKED_LIST_ENTRY
        hFirstEntry = OSAL.hLinkedListFirst(
            psObj->sChannel.hChannelListByServiceId, (void**)&hChannel);

        // Grab the last entry, which is the one before the first
        // since this is a circular list.  OSAL.hLinkedListLast return
        // the first entry for a circular list, not sure if this is right.
        // Perhaps OSAL.hLinkedListLast should return the last item from a
        // circular list as well
        hLastEntry = OSAL.hLinkedListPrev(hFirstEntry, (void**)NULL);

        if (psObj->sChannel.hCurChanEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // We are just starting, so use the first entry in the channel list
            psObj->sChannel.hCurChanEntry = hFirstEntry;
        }
        else
        {
            // grab the next one
            psObj->sChannel.hCurChanEntry = OSAL.hLinkedListNext(
                psObj->sChannel.hCurChanEntry, (void**)&hChannel);
        }

        // Loop until we notify at least one channel or hit the end of the list
        while (bNotified == FALSE)
        {
            CHANNEL_ID tChannelId;

            tChannelId = CHANNEL.tChannelId(hChannel);
            if (tChannelId != CHANNEL_INVALID_ID)
            {
                // Notify the channel object if it is pending
                bNotified = CHANNEL_bNotifyIfPending(hChannel,
                    psObj->sChannel.vAppNotifierCallback,
                    psObj->sChannel.pvAppNotifierCallbackArg);
            }

            if (hLastEntry == psObj->sChannel.hCurChanEntry)
            {
                bMoreToCheck = FALSE;
                break;
            }
            else
            {
                // We aren't at the end yet, so flag that we need to keep going
                bMoreToCheck = TRUE;
            }

            // If we failed to notify in this iteration of the loop, try the next one
            // We should notify at least one channel or relize that we are done in this function
            if (bNotified == FALSE)
            {
                // grab the next one since the current one didn't need notifying
                psObj->sChannel.hCurChanEntry = OSAL.hLinkedListNext(
                    psObj->sChannel.hCurChanEntry, (void**)&hChannel);
            }
        }
    }

    return bMoreToCheck;
}

/*****************************************************************************
*
*   CCACHE_bRegisterAllCategoriesNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bRegisterAllCategoriesNotification (
    CCACHE_OBJECT hCCache,
    CATEGORY_EVENT_MASK tEventRequestMask,
    BOOLEAN bImmediateNotify,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
       CCACHE_OBJECT_STRUCT *psObj =
           (CCACHE_OBJECT_STRUCT *)hCCache;
       CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify;
       OSAL_RETURN_CODE_ENUM eReturnCode;

       // Create the notification structure
       psNotify = (CCACHE_ALL_CAT_NOTIFICATION_STRUCT *)
            SMSO_hCreate(
                CCACHE_OBJECT_NAME":AllCcatsNotify",
                sizeof(CCACHE_ALL_CAT_NOTIFICATION_STRUCT),
                (SMS_OBJECT)hCCache, // Child
                FALSE); // No Lock (ignored)

       if (psNotify != NULL)
       {
           psNotify->tEventRequestMask = tEventRequestMask;
           psNotify->vEventCallback = vEventCallback;
           psNotify->pvEventCallbackArg = pvEventCallbackArg;
           psNotify->bImmediateNotify = bImmediateNotify;

           eReturnCode = OSAL.eLinkedListAdd(
               psObj->sCategory.hAllCatNotificationList,
               OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
               psNotify);

           if (eReturnCode == OSAL_SUCCESS)
           {
               eReturnCode = OSAL.eLinkedListIterate(
                   psObj->sCategory.hCategoryList,
                   bUpdateCategoryNotificationsIterator,
                   (void *)psNotify
                       );

               if(eReturnCode == OSAL_SUCCESS || eReturnCode == OSAL_NO_OBJECTS)
               {
                   // All went well.
                   bReturn = TRUE;
               }
               else
               {
                   vDestroyCategoryNotifier((void*)psNotify);
               }

           }
           else
           {
               vDestroyCategoryNotifier((void*)psNotify);
           }
       }
    }

    return bReturn;
}

/*****************************************************************************
*
*   CCACHE_bUnRegisterAllCategoriesNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bUnRegisterAllCategoriesNotification (
    CCACHE_OBJECT hCCache,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bReturn = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
       CCACHE_OBJECT_STRUCT *psObj =
           (CCACHE_OBJECT_STRUCT *)hCCache;
       CCACHE_ALL_CAT_NOTIFICATION_STRUCT sNotifyToFind;
       OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
       OSAL_RETURN_CODE_ENUM eReturnCode;

       // Find the notification structure
       sNotifyToFind.vEventCallback = vEventCallback;
       sNotifyToFind.pvEventCallbackArg = pvEventCallbackArg;

       eReturnCode = OSAL.eLinkedListSearch(
           psObj->sCategory.hAllCatNotificationList, &hEntry, &sNotifyToFind);

       if (eReturnCode == OSAL_SUCCESS)
       {
            CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify;

            psNotify = (CCACHE_ALL_CAT_NOTIFICATION_STRUCT *)
               OSAL.pvLinkedListThis(hEntry);

            // Iterate through the categories removing this entry
            psNotify->tEventRequestMask = CATEGORY_OBJECT_EVENT_NONE;
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->sCategory.hCategoryList,
                bUpdateCategoryNotificationsIterator,
                (void *)psNotify);

            if((eReturnCode == OSAL_SUCCESS) ||
               (eReturnCode == OSAL_NO_OBJECTS))
            {
                vDestroyCategoryNotifier((void*)psNotify);
                OSAL.eLinkedListRemove(hEntry);

                // All went well.
                bReturn = TRUE;
            }
       }
    }

    return bReturn;
}

/*****************************************************************************
*
*   CCACHE_eHandleAppCategoryNotifyOnChange
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CCACHE_eHandleAppCategoryNotifyOnChange (
    CCACHE_OBJECT hCCache,
    CATEGORY_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
       BOOLEAN bSuccess = TRUE;
       CCACHE_OBJECT_STRUCT *psObj =
           (CCACHE_OBJECT_STRUCT *)hCCache;

       // If the application already had a callback specied, unregister the
       // previous one before doing anything else
       if (psObj->sCategory.vAppNotifierCallback != NULL)
       {
           bSuccess =
               CCACHE_bUnRegisterAllCategoriesNotification(
                   hCCache,
                   psObj->sCategory.vAppNotifierCallback,
                   psObj->sCategory.pvAppNotifierCallbackArg);

           if (bSuccess == TRUE)
           {
               // Clear the handles
               psObj->sCategory.vAppNotifierCallback = NULL;
               psObj->sCategory.pvAppNotifierCallbackArg = NULL;
               psObj->sCategory.hCurCatEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
           }
       }

       // If the application specified a callback, then register it.
       // If all they specified was null, then they just wanted to remove
       // the current callback, which we handled above.
       if (vNotifierCallback != NULL && bSuccess == TRUE)
       {
           bSuccess =
               CCACHE_bRegisterAllCategoriesNotification(
                   hCCache, CATEGORY_OBJECT_EVENT_ALL, FALSE,
                   vNotifierCallback, pvNotifierCallbackArg);

           if (bSuccess == TRUE)
           {
               // Save the handles
               psObj->sCategory.vAppNotifierCallback = vNotifierCallback;
               psObj->sCategory.pvAppNotifierCallbackArg = pvNotifierCallbackArg;
               psObj->sCategory.hCurCatEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
           }
       }

       if (bSuccess == TRUE)
       {
           eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
       }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CCACHE_bCheckNextCategoryForAppNotification
*
*****************************************************************************/
BOOLEAN CCACHE_bCheckNextCategoryForAppNotification (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner, bMoreToCheck = FALSE;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
       BOOLEAN bNotified = FALSE;
       CCACHE_OBJECT_STRUCT *psObj =
           (CCACHE_OBJECT_STRUCT *)hCCache;
       CATEGORY_OBJECT hCategory;
       OSAL_LINKED_LIST_ENTRY hFirstEntry, hLastEntry;

       // Grab the first entry, also get the handle to the category
       // at that entry in case it is needed if
       // psObj->sCategory.hCurCatEntry == OSAL_INVALID_LINKED_LIST_ENTRY
       hFirstEntry = OSAL.hLinkedListFirst(
           psObj->sCategory.hCategoryList, (void**)&hCategory);

       // Grab the last entry, which is the one before the first
       // since this is a circular list.  OSAL.hLinkedListLast return
       // the first entry for a circular list, not sure if this is right.
       // Perhaps OSAL.hLinkedListLast should return the last item from a
       // circular list as well
       hLastEntry = OSAL.hLinkedListPrev(hFirstEntry, (void**)NULL);

       if (psObj->sCategory.hCurCatEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
       {
           // We are just starting, so use the first entry in the channel list
           psObj->sCategory.hCurCatEntry = hFirstEntry;
       }
       else
       {
           // grab the next one
           psObj->sCategory.hCurCatEntry = OSAL.hLinkedListNext(
               psObj->sCategory.hCurCatEntry, (void**)&hCategory);
       }

       // Loop until we notify at least one category or hit the end of the list
       while (bNotified == FALSE)
       {
           // Notify the category object if it is pending
           bNotified = CATEGORY_bNotifyIfPending(hCategory,
               psObj->sCategory.vAppNotifierCallback,
               psObj->sCategory.pvAppNotifierCallbackArg);

           if (hLastEntry == psObj->sCategory.hCurCatEntry)
           {
               bMoreToCheck = FALSE;
               break;
           }
           else
           {
               // We aren't at the end yet, so flag that we need to keep going
               bMoreToCheck = TRUE;
           }

           // If we failed to notify in this iteration of the loop, try the next one
           // We should notify at least one channel or relize that we are done in this function
           if (bNotified == FALSE)
           {
               // grab the next one since the current one didn't need notifying
               psObj->sCategory.hCurCatEntry = OSAL.hLinkedListNext(
                   psObj->sCategory.hCurCatEntry, (void**)&hCategory);
           }
       }
    }

    return bMoreToCheck;
}

/*******************************************************************************
*
*   CCACHE_vEventHandler
*
*   This function runs in the context of the Sirius Services Task
*
*******************************************************************************/
void CCACHE_vEventHandler (
    CCACHE_OBJECT hCCache,
    SMS_EVENT_CCACHE_STRUCT const *psCCache
        )
{
    CCACHE_OBJECT_STRUCT *psObj = (CCACHE_OBJECT_STRUCT *)hCCache;

    if(psCCache == NULL)
    {
        return;
    }

    switch(psCCache->eType)
    {

#if SMS_DEBUG == 1
        case CCACHE_EVENT_TYPE_PRINT:
        {
            if (psCCache->uCache.sPrint.bChannels == TRUE )
            {
                // Dump the channel cache
                SMSAPI_DEBUG_vPrintChannels(
                    psObj->hDecoder, psCCache->uCache.sPrint.bVerbose);
            }

            if (psCCache->uCache.sPrint.bCategories == TRUE)
            {
                // Dump the category cache
                SMSAPI_DEBUG_vPrintCategories(
                    psObj->hDecoder, psCCache->uCache.sPrint.bVerbose);
            }

        }
        break;
#endif /* SMS_DEBUG == 1*/

        case CCACHE_EVENT_TYPE_REMAP:
        {
            vRemapFSM(psObj, &psCCache->uCache.sRemap);
        }
        break;

        case CCACHE_EVENT_TYPE_ACO:
        {
            // TODO: consider to move this logic into the GMD or CM task
            vStoreACOData(psObj);
        }
        break;

        default:
            // Unknown event
        break;
    }

    // Notify callback if registered and change occured
    SMSU_bNotify(&psObj->sEvent);

    return;
}

/*****************************************************************************
*
*   CCACHE_bUpdateChannel
*
*****************************************************************************/
BOOLEAN CCACHE_bUpdateChannel (
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bReturn = FALSE;

    CCACHE_OBJECT hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
    
    if (CCACHE_INVALID_OBJECT != hCCache)
    {
        bReturn = bUpdateChannelLocal((CCACHE_OBJECT_STRUCT *)hCCache, hChannel);
    }

    return bReturn;
}

/*******************************************************************************
*
*   CCACHE_bUpdateChannelId
*
*   This API is and should only be called by the CHANNEL module.
*   It's purpose is to update an existing CHANNEL object's channel-id.
*   Generally this means to simply replace the current channel-id with a new
*   one. However, sometimes the new channel-id may already exist and
*   the previous one needs to be removed first. Lastly, the channel
*   may have been invalid and is now becoming valid.
*
*****************************************************************************/
BOOLEAN CCACHE_bUpdateChannelId(
    CHANNEL_OBJECT hChannelToUpdate,
    CHANNEL_ID tNewChannelId
        )
{
    BOOLEAN bSuccess = FALSE;
    SERVICE_ID tServiceId;
    CCACHE_OBJECT hCCache;

    // At anytime the channel id is updated, we need to re-insert
    // the channel entry in the channel cache. This is because the
    // channel list is sorted, by channel id and does not contain
    // any channel objects with an invalid channel-id.

    // Extract channel's service id
    tServiceId = CHANNEL.tServiceId(hChannelToUpdate);

    // Extract the channel's ccache handle
    hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannelToUpdate);
    if ((hCCache != CCACHE_INVALID_OBJECT) && // valid handle
        (tServiceId != SERVICE_INVALID_ID)) // valid service id
    {
        CCACHE_OBJECT_STRUCT *psObj = (CCACHE_OBJECT_STRUCT *)hCCache;
        OSAL_LINKED_LIST_ENTRY hEntry;
        CHANNEL_ID tExistingChannelId;
        CHANNEL_OBJECT hReplaceChannel;
        BOOLEAN bChanged;

        // PHASE 1 - Prepare

        // Now check if this new channel is going to be in the list.
        // If not, there is nothing to prepare for. However if the new
        // channel already exists, we need to remove it since we
        // are being requested to change the channel-id of one which
        // would be in conflict of this one.
        if(tNewChannelId != CHANNEL_INVALID_ID)
        {
            CHANNEL_OBJECT hExistingChannel;

            // If it already exists in the list, we must remove it
            // since we cannot have two channels with the same id.

            // Find this channel, to get it's entry handle in the list
            // and remove it if it exists.
            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            hExistingChannel = hChannelFromIds(
                psObj, &hEntry, SERVICE_INVALID_ID, tNewChannelId);
            if((hExistingChannel != CHANNEL_INVALID_OBJECT) &&
                (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Now, remove it from the list. If this fails there
                // isn't much I can do. Since the later re-insert will
                // simply fail, we can only just bail.
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    BOOLEAN bChanged;

                    // Remove the channel-id, leave service-id
                    bChanged = CHANNEL_bUpdateIds(
                        hExistingChannel, SERVICE_INVALID_ID,
                        CHANNEL_INVALID_ID);
                    if(bChanged == FALSE)
                    {
                        // Damn! I removed the channel, but can't
                        // update the channel id. Something is
                        // wrong which I can't fix.
                        printf("CCACHE Error! Cannot modify channel "
                            "%d to be an invalid channel.\n", tNewChannelId);
                        return FALSE;
                    }
                }
                else
                {
                    // Error!
                    printf("CCACHE Error! Cannot remove channel "
                        "%d from the ccache.\n", tNewChannelId);
                    return FALSE;
                }
            } // Doesn't exist, no worries
        } // New channel id is not valid

        // PHASE 2 - Remove, Replace, or Add

        // Whatever happened above, at this point we are ready to
        // handle the change from one channel-id to the next.

        // We only searched for this channel because we want
        // to know the existing channel-id, so we can find it
        // in our channel list.
        tExistingChannelId = CHANNEL.tChannelId(hChannelToUpdate);

        // Now find this channel in our list, if it is not invalid
        // If we find one, it will be the actual CHANNEL object we
        // are going to update.
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        if(tExistingChannelId != CHANNEL_INVALID_ID)
        {
            hReplaceChannel = hChannelFromIds(
                psObj, &hEntry, SERVICE_INVALID_ID, tExistingChannelId);
            if((hReplaceChannel != CHANNEL_INVALID_OBJECT) &&
                (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
            {
                // This is the channel we want to update
                hChannelToUpdate = hReplaceChannel;
            }
        }

        // Now change this channel object's id
        bChanged = CHANNEL_bUpdateIds(
            hChannelToUpdate, SERVICE_INVALID_ID, tNewChannelId);
        if(bChanged == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // If new channel-id is invalid, we are done, otherwise
            // we need to either replace it, or add it.
            if(tNewChannelId != CHANNEL_INVALID_ID)
            {
                // Now we can either finally replace or add this
                // channel in the list
                if(hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) // Replace
                {
                    eReturnCode = OSAL.eLinkedListReplaceEntry(
                        psObj->sChannel.hChannelListByChannelId,
                        hEntry, hChannelToUpdate);
                }
                else // Add
                {
                    eReturnCode = OSAL.eLinkedListAdd(
                        psObj->sChannel.hChannelListByChannelId,
                        &hEntry, hChannelToUpdate);
                }
            }
            else if(hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                // Just remove this from our list, if it did exist
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
            }
            else
            {
                // Nothing to do.
                eReturnCode = OSAL_SUCCESS;
            }

            // Check for any problems doing this
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Success.
                bSuccess = TRUE;
            }
            else
            {
                // Error! Put channel back the way it was.
                CHANNEL_bUpdateIds(
                    hChannelToUpdate, SERVICE_INVALID_ID,
                    tExistingChannelId);
                printf("CCACHE Error! Cannot manipulate channel "
                    "%d in the ccache.\n", tNewChannelId);
            }
        }
        else
        {
            printf("CCACHE Error! Cannot modify channel id "
                "%d to be %d.\n", tExistingChannelId, tNewChannelId);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CCACHE_hUnsubscribedChannel
*
*****************************************************************************/
CHANNEL_OBJECT CCACHE_hUnsubscribedChannel (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner;
    CHANNEL_OBJECT hUnsubscribedChannel = CHANNEL_INVALID_OBJECT;

    // Verify and check ownership of object (doesn't require cache)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        // De-reference object
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        hUnsubscribedChannel = psObj->hUnsubscribedChannel;
    }

    return hUnsubscribedChannel;
}

/*****************************************************************************
*
*   CCACHE_hSubscriptionAlertChannel
*
*****************************************************************************/
CHANNEL_OBJECT CCACHE_hSubscriptionAlertChannel (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner;
    CHANNEL_OBJECT hSubscriptionAlertChannel = CHANNEL_INVALID_OBJECT;

    // Verify and check ownership of object (doesn't require cache)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        // De-reference object
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        hSubscriptionAlertChannel = psObj->hSubscriptionAlertChannel;
    }

    return hSubscriptionAlertChannel;
}

/*****************************************************************************
*
*   CCACHE_bUpdateSubscription
*
*****************************************************************************/
BOOLEAN CCACHE_bUpdateSubscription (
    SMS_EVENT_HANDLER hEventHdlr,
    CCACHE_REMAP_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Caller must provide a valid callback or else we cannot do anything.
    if(vCallback == NULL)
    {
        // Error! Bad input.
        return FALSE;
    }

    // Allocate an event to use.
    hEvent = SMSE_hAllocateEvent(
        hEventHdlr, SMS_EVENT_CCACHE, &puEventData, SMS_EVENT_OPTION_NONE);
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        SMS_EVENT_CCACHE_STRUCT *psCCache =
            &puEventData->uDecoder.sCache;

        // Remap event
        psCCache->eType =
            CCACHE_EVENT_TYPE_REMAP;

        // Configure initial state & transition
        psCCache->uCache.sRemap.bInitial = TRUE;
        psCCache->uCache.sRemap.bBlocked = FALSE;
        psCCache->uCache.sRemap.eCemsDisableReturnCode =
            SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        psCCache->uCache.sRemap.eState = CCACHE_REMAP_SUBSCRIPTION;

        // Configure callback
        psCCache->uCache.sRemap.vCallback = vCallback;
        psCCache->uCache.sRemap.pvCallbackArg = pvCallbackArg;

        // Start the subscription update process by posting this event.
        // No need to check return value since this is
        // a re-allocated event provided to us, so we just indicate to the
        // caller what happened.
        bPosted = SMSE_bPostEvent(hEvent);
    }

    return bPosted;
}

/*****************************************************************************
*
*   CCACHE_bRemap
*
* This function is used to instruct the Channel Cache to remap itself
* typically after a receiver has undergone a channel line-up change.
* This process involves clearing the cache of unused channels, remapping
* those which have changed and re-freshing the cache.
*
* Inputs:
*   hEventHdlr - An event handler for messages
*   vCallback - A callback function to call when the remap
*           process is either complete or resulted in an error.
*   pvCallbackArg - An argument provided by the caller provided
*           when the callback function is called.
*
* Outputs:
*   BOOLEAN - TRUE if the remap process has been started or FALSE
*       if the process was unable to start.
*
*****************************************************************************/
BOOLEAN CCACHE_bRemap (
    SMS_EVENT_HANDLER hEventHdlr,
    CCACHE_REMAP_CALLBACK vCallback,
    void *pvCallbackArg
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Caller must provide a valid callback or else we cannot do anything.
    if(vCallback == NULL)
    {
        // Error! Bad input.
        return FALSE;
    }

    // Allocate an event to use.
    hEvent = SMSE_hAllocateEvent(
        hEventHdlr, SMS_EVENT_CCACHE, &puEventData, SMS_EVENT_OPTION_NONE);
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        SMS_EVENT_CCACHE_STRUCT *psCCache =
            &puEventData->uDecoder.sCache;

        // Remap event
        psCCache->eType =
            CCACHE_EVENT_TYPE_REMAP;

        // Configure initial state
        psCCache->uCache.sRemap.bInitial = TRUE;
        psCCache->uCache.sRemap.bBlocked = FALSE;
        psCCache->uCache.sRemap.eCemsDisableReturnCode =
            SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        psCCache->uCache.sRemap.eState = CCACHE_REMAP_INIT;

        // Configure callback
        psCCache->uCache.sRemap.vCallback = vCallback;
        psCCache->uCache.sRemap.pvCallbackArg = pvCallbackArg;

        // Start the remap process by posting this event.
        // No need to check return value since this is
        // a re-allocated event provided to us, so we just indicate to the
        // caller what happened.
        bPosted = SMSE_bPostEvent(hEvent);
    }

    return bPosted;
}

/*****************************************************************************
*
*   CCACHE_bIterateChannels
*
*   This function is used to iterate all the channels in the cache one
*   by one. Based on the iterator function provided iteration starts from
*   the beginning of the cache to the end, if the provider iterator continues
*   to return TRUE. Otherwise, iteration is stopped when this function returns
*   FALSE.
*
*   Inputs:
*       hCCache - A Channel Cache handle.
*       bIterator - An iterator function callback which is called for each
*           channel in the cache.
*       pvArg - A caller defined argument to supply in the iterator callback
*           for each channel in the cache.
*
*   Returns:
*       TRUE if the cache was iterated. FALSE if the cache could not be
*       iterated.
*
*****************************************************************************/
BOOLEAN CCACHE_bIterateChannels (
    CCACHE_OBJECT hCCache,
    BOOLEAN bIterateByServiceId,
    CCACHE_CHANNEL_ITERATOR_HANDLER bIterator,
    void *pvArg
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT_STRUCT *psObj = (CCACHE_OBJECT_STRUCT *)hCCache;
    CCACHE_CHANNEL_ITERATOR_HANDLER_STRUCT sIteratorShim;
    OSAL_OBJECT_HDL hListToIterate;

    // Verify inputs. Must already be a ccache owner and they must provide
    // some type of iterator, otherwise what's the point?
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if((bOwner == FALSE) || (bIterator == NULL))
    {
        // Error!
        return FALSE;
    }

    if (bIterateByServiceId == FALSE)
    {
        hListToIterate = psObj->sChannel.hCurrentNavigationChannelList;
    }
    else
    {
        hListToIterate = psObj->sChannel.hChannelListByServiceId;
    }

    // Populate iterator shim
    sIteratorShim.bIterator = bIterator;
    sIteratorShim.pvArg = pvArg;

    // Iterate all channels in the cache...
    eReturnCode = OSAL.eLinkedListIterate(
        hListToIterate, bChannelIterator, &sIteratorShim);

    if(eReturnCode == OSAL_SUCCESS)
    {
        // All went well.
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CCACHE_bIterateCategories
*
*   This function is used to iterate all the categories in the cache one
*   by one. Based on the iterator function provided iteration starts from
*   the beginning of the cache to the end, if the provider iterator continues
*   to return TRUE. Otherwise, iteration is stopped when this function returns
*   FALSE.
*
*   Inputs:
*       hCCache - A Channel Cache handle.
*       bIterator - An iterator function callback which is called for each
*           category in the cache.
*       pvArg - A caller defined argument to supply in the iterator callback
*           for each category in the cache.
*
*   Returns:
*       TRUE if the cache was iterated. FALSE if the cache could not be
*       iterated.
*
*****************************************************************************/
BOOLEAN CCACHE_bIterateCategories (
    CCACHE_OBJECT hCCache,
    CCACHE_CATEGORY_ITERATOR_HANDLER bIterator,
    void *pvArg
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT_STRUCT *psObj = (CCACHE_OBJECT_STRUCT *)hCCache;
    CCACHE_CATEGORY_ITERATOR_HANDLER_STRUCT sIteratorShim;

    // Verify inputs. Must already be a ccache owner and they must provide
    // some type of iterator, otherwise what's the point?
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if((bOwner == FALSE) || (bIterator == NULL))
    {
        // Error!
        return FALSE;
    }

    // Populate iterator shim
    sIteratorShim.bIterator = bIterator;
    sIteratorShim.pvArg = pvArg;

    // Iterate all categories in the cache...
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->sCategory.hCategoryList,
        bCategoryIterator,
        &sIteratorShim
            );
    if(eReturnCode == OSAL_SUCCESS)
    {
        // All went well.
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CCACHE_hCidCreate
*
* Create a CID which is modifiable, meaning it may be created, modified and
* later destroyed. Recycling of non-constant CIDs will also be handled
* here as well. This is done via the CCACHE object.
*
* Inputs:
*   hCCache - CCACHE associated with this CID
*   eType - The type of CID object to create.
*   pvObjectDataPtr - The raw source data from which to create this object.
*
* Outputs:
*   CID_OBJECT - a new or recycled CID which represents the content provided.
*
*****************************************************************************/
CID_OBJECT CCACHE_hCidCreate (
    CCACHE_OBJECT hCCache,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        )
{
    CID_OBJECT hCid = CID_INVALID_OBJECT;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)hCCache;
    BOOLEAN bOwner;

    // Verify inputs. Must already be a ccache owner.
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if(bOwner == TRUE)
    {
        // We know all about CID-pools!
        hCid = CID_hCreate(psObj->hCidPool, eType, pvObjectDataPtr);
    }

    return hCid;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateChannel
*
*****************************************************************************/
static CHANNEL_OBJECT hCreateChannel (
    CCACHE_OBJECT_STRUCT *psObj,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bNotify
        )
{
    CHANNEL_OBJECT hChannel;

    // Create a new channel object
    hChannel =
        CHANNEL_hCreate(
            (CCACHE_OBJECT)psObj, tServiceId, tChannelId);
    if(hChannel != CHANNEL_INVALID_OBJECT)
    {
        BOOLEAN bAdded = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // *******************************************************************
        // TODO: we will add mutiple list adding here
        // *******************************************************************

        // Add this channel object to the ccache lists...

        // By service id, if one exists
        if(tServiceId != SERVICE_INVALID_ID)
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sChannel.hChannelListByServiceId,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                hChannel
                    );
            if(eReturnCode == OSAL_SUCCESS)
            {
                bAdded = TRUE;
            }
        }

        // By channel id, if one exists
        if(tChannelId != CHANNEL_INVALID_ID)
        {
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sChannel.hChannelListByChannelId,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                hChannel
                    );
            if(eReturnCode == OSAL_SUCCESS)
            {
                bAdded = TRUE;
            }
         }

        // *******************************************************************

        // If we added it to something, finish off addition procedure
        if(bAdded == TRUE)
        {
            CCACHE_CHAN_ADD_NOTIFIER_STRUCT sData;

            // Update this channel with the channel art service handle
            CHANNEL_vUpdateArtService( hChannel, psObj->hChannelArtService );

            sData.bImmediateNotify = bNotify;
            sData.hChannel = hChannel;

            // Register this channel with any of our "all channels" notifiers
            OSAL.eLinkedListIterate(
               psObj->sChannel.hAllChanNotificationList,
               (OSAL_LL_ITERATOR_HANDLER)bAddNotifierToChannelIterator,
               (void *)&sData
                   );
       }
       else
       {
            // We couldnt add to anything, sorry!
            CHANNEL_vDestroy(hChannel);
            hChannel = CHANNEL_INVALID_OBJECT;
       }
    }

    return hChannel;
}

/*****************************************************************************
*
*   hCreateBroadcastCategory
*
*****************************************************************************/
static CATEGORY_OBJECT hCreateBroadcastCategory (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId
        )
{
    CATEGORY_OBJECT hCategory;

    // DEBUG indicating we're getting category info
    //printf(CCACHE_OBJECT_NAME": hCreateBroadcastCategory(%u)\n", tCategoryId);
    hCategory = RADIO_hCreateCategory(psObj->hDecoder, tCategoryId);
    if(hCategory != CATEGORY_INVALID_OBJECT)
    {
        // Add this category object to the cache
        OSAL.eLinkedListAdd(
            psObj->sCategory.hCategoryList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            hCategory);

        // Update this category with the channel art service
        // handle now that channels have been (possibly) added
        // to this category, which would allow for user updates
        CATEGORY_vUpdateArtService( hCategory, psObj->hChannelArtService );

        // Register this category with any of our "all categories" notifiers
        OSAL.eLinkedListIterate(
           psObj->sCategory.hAllCatNotificationList,
           (OSAL_LL_ITERATOR_HANDLER)bAddNotifierToCategoryIterator,
           (void *)hCategory
               );
    }

    // Success
    return hCategory;
}

/*****************************************************************************
*
*   hCreateDerivedCategory
*
*****************************************************************************/
static CATEGORY_OBJECT hCreateDerivedCategory (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_TYPE_ENUM eCategoryType,
    CATEGORY_ID tCategoryId,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        )
{
    CATEGORY_OBJECT hCategory;

    // Create a new category object and populate it with retrieved info
    hCategory = CATEGORY_hCreateCategory(
                    psObj->hDecoder,
                    (SMS_OBJECT)psObj,
                    eCategoryType,
                    tCategoryId,
                    pacLongName,
                    pacMediumName,
                    pacShortName,
                    tInitialNumChannels,
                    bUniqueItems,
                    bReplace );

    if(hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error! If category object cannot be created, just bail
        return CATEGORY_INVALID_OBJECT;
    }

    // Add this category object to the cache
    OSAL.eLinkedListAdd(
        psObj->sCategory.hCategoryList,
        OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
        hCategory);

    // Increment the number of valid derived
    // categories which exist within the cache
    psObj->sCategory.n16NumDerivedCategories++;

    // Update this category with the channel art service
    // handle now that channels have been (possibly) added
    // to this category, which would allow for user updates
    CATEGORY_vUpdateArtService( hCategory, psObj->hChannelArtService );

    // Register this category with any of our "all categories" notifiers
    OSAL.eLinkedListIterate(
       psObj->sCategory.hAllCatNotificationList,
       (OSAL_LL_ITERATOR_HANDLER)bAddNotifierToCategoryIterator,
       (void *)hCategory
           );

    // Success
    return hCategory;
}

/*****************************************************************************
*
*   bUpdateChannelLocal
*
*****************************************************************************/
static BOOLEAN bUpdateChannelLocal (
    CCACHE_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bNeedsUpdated, bUpToDate = TRUE;

    // Check if we really need to do this.
    bNeedsUpdated = CHANNEL_bNeedsUpdated(hChannel);
    if(bNeedsUpdated == TRUE)
    {
        SERVICE_ID tServiceId;

        tServiceId = CHANNEL.tServiceId(hChannel);

        // Update Channel Information from receiver
        if(tServiceId != SERVICE_INVALID_ID)
        {
            // DEBUG indicating we're getting channel info from
            // the receiver.
            // printf(CCACHE_OBJECT_NAME": bUpdateChannelLocal(%u)\n",
            //   tChannelId);
            bUpToDate = RADIO_bRequestChannel(
                psObj->hDecoder, tServiceId
                    );
        }
    }

    return bUpToDate;
}

/*****************************************************************************
*
*   hFindCategoryLocal
*
*****************************************************************************/
static CATEGORY_OBJECT hFindCategoryLocal (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId
        )
{
    OSAL_LINKED_LIST_ENTRY hEntry = // search from top of list
        OSAL_INVALID_LINKED_LIST_ENTRY;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify inputs
    if( !(tCategoryId < CATEGORY_INVALID_ID) ||
        (psObj->sCategory.hCategoryList == OSAL_INVALID_OBJECT_HDL))
    {
         // There are input parameter errors!
        return CATEGORY_INVALID_OBJECT;
    }

    // Check if this category exists in our list.
    CATEGORY_bUpdateId(psObj->hDummyCategory, tCategoryId);
    eReturnCode =
        OSAL.eLinkedListSearch(
            psObj->sCategory.hCategoryList,
            &hEntry,
            psObj->hDummyCategory );

    // if the category was not found, create it,
    // but only if this is a broadcast category.
    // Why?  Because derived categories must
    // be created before someone tries to locate
    // them -- there's no pre-existing notion
    // of those categories, while a broadcast
    // category may have a pre-existing notion
    // thanks to the category info even
    // though there is no corresponding category
    // object associated with it.
    if ( eReturnCode == OSAL_OBJECT_NOT_FOUND )
    {
        BOOLEAN bOk;

        // Check to see if this Id is in range
        // for a broadcast category
        bOk = CATEGORY_bIdInRange (
                CATEGORY_TYPE_BROADCAST,
                tCategoryId );
        if (bOk == TRUE)
        {
            // Create a new category
            hCategory = hCreateBroadcastCategory(psObj, tCategoryId);
        }
    }
    // the category was found
    else if(eReturnCode == OSAL_SUCCESS)
    {
        // Category exists, extract the data from this entry
        hCategory = (CATEGORY_OBJECT)OSAL.pvLinkedListThis(hEntry);
    }
    else
    {
        // The category was not found
        hCategory = CATEGORY_INVALID_OBJECT;
    }

    return hCategory;
}

/*****************************************************************************
*
*   bAddNotifierToChannelIterator
*
*****************************************************************************/
static BOOLEAN bAddNotifierToChannelIterator (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify,
    CCACHE_CHAN_ADD_NOTIFIER_STRUCT *psData
        )
{
    if (psNotify->tEventRequestMask == CHANNEL_OBJECT_EVENT_NONE)
    {
        CHANNEL_vUnregisterNotification(
            psData->hChannel,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);
    }
    else
    {
        CHANNEL_bRegisterNotification(
            psData->hChannel,
            psNotify->tEventRequestMask,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg,
            TRUE);

        if (psData->bImmediateNotify == TRUE)
        {
            // Force a notification since this is a new channel
            CHANNEL_bNotifyIfPending(psData->hChannel,
                psNotify->vEventCallback, psNotify->pvEventCallbackArg);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveNotifierFromChannelIterator
*
*****************************************************************************/
static BOOLEAN bRemoveNotifierFromChannelIterator (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify,
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_vUnregisterNotification(
        hChannel,
        psNotify->vEventCallback,
        psNotify->pvEventCallbackArg);

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateChannelNotificationsIterator
*
*****************************************************************************/
static BOOLEAN bUpdateChannelNotificationsIterator (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify =
        (CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *)pvArg;

    if (psNotify->tEventRequestMask == CHANNEL_OBJECT_EVENT_NONE)
    {
        CHANNEL_vUnregisterNotification(
            hChannel,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);
    }
    else
    {
        CHANNEL_bRegisterNotification(
            hChannel,
            psNotify->tEventRequestMask,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg,
            TRUE);

        if (psNotify->bImmediateNotify == TRUE)
        {
            // Force a notification
            // Notify the channel object if it is pending
            CHANNEL_bNotifyIfPending(hChannel,
                psNotify->vEventCallback, psNotify->pvEventCallbackArg);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bAddNotifierToCategoryIterator
*
*****************************************************************************/
static BOOLEAN bAddNotifierToCategoryIterator (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify,
    CATEGORY_OBJECT hCategory
        )
{
    if (psNotify->tEventRequestMask == CATEGORY_OBJECT_EVENT_NONE)
    {
        CATEGORY_vUnregisterNotification(
            hCategory,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);
    }
    else
    {
        CATEGORY_bRegisterNotification(
            hCategory,
            psNotify->tEventRequestMask,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);

        // Force a notification since this is a new category
        CATEGORY_bNotifyIfPending(hCategory,
            psNotify->vEventCallback, psNotify->pvEventCallbackArg);

    }

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveNotifierFromCategoryIterator
*
*****************************************************************************/
static BOOLEAN bRemoveNotifierFromCategoryIterator (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify,
    CATEGORY_OBJECT hCategory
        )
{
    CATEGORY_vUnregisterNotification(
        hCategory,
        psNotify->vEventCallback,
        psNotify->pvEventCallbackArg);

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateCategoryNotificationsIterator
*
*****************************************************************************/
static BOOLEAN bUpdateCategoryNotificationsIterator (
    void *pvData,
    void *pvArg
        )
{
    CATEGORY_OBJECT hCategory = (CATEGORY_OBJECT)pvData;
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify =
        (CCACHE_ALL_CAT_NOTIFICATION_STRUCT *)pvArg;

    if (psNotify->tEventRequestMask == CATEGORY_OBJECT_EVENT_NONE)
    {
        CATEGORY_vUnregisterNotification(
            hCategory,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);
    }
    else
    {
        CATEGORY_bRegisterNotification(
            hCategory,
            psNotify->tEventRequestMask,
            psNotify->vEventCallback,
            psNotify->pvEventCallbackArg);

        if (psNotify->bImmediateNotify == TRUE)
        {
            // Force a notification
            CATEGORY_bNotifyIfPending(hCategory,
                psNotify->vEventCallback, psNotify->pvEventCallbackArg);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bReleaseChannelIterator
*
*****************************************************************************/
static BOOLEAN bReleaseChannelIterator(
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;

    // Unblock channel notifications
    CHANNEL_vReleaseNotifications(hChannel);

    return TRUE;
}

/*****************************************************************************
*
*   bChannelIterator
*
*   This is a simple linked list iterator shim function for the channel
*   cache. It just simply allows the linked list iterator to apply
*   a caller defined callback with associated data.
*
*****************************************************************************/
static BOOLEAN bChannelIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bRetVal;
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CCACHE_CHANNEL_ITERATOR_HANDLER_STRUCT *psIteratorShim =
        (CCACHE_CHANNEL_ITERATOR_HANDLER_STRUCT *)pvArg;

    // Call iterator provided by the caller
    bRetVal = psIteratorShim->bIterator(hChannel, psIteratorShim->pvArg);

    return bRetVal;
}

/*****************************************************************************
*
*   bCategoryIterator
*
*   This is a simple linked list iterator shim function for the channel
*   cache. It just simply allows the linked list iterator to apply
*   a caller defined callback with associated data.
*
*****************************************************************************/
static BOOLEAN bCategoryIterator (
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bRetVal;
    CATEGORY_OBJECT hCategory = (CATEGORY_OBJECT)pvData;
    CCACHE_CATEGORY_ITERATOR_HANDLER_STRUCT *psIteratorShim =
        (CCACHE_CATEGORY_ITERATOR_HANDLER_STRUCT *)pvArg;

    // Call iterator provided by the caller
    bRetVal = psIteratorShim->bIterator(hCategory, psIteratorShim->pvArg);

    return bRetVal;
}

/*****************************************************************************
*
*   bRemoveCategoryNotifierIterator
*
*****************************************************************************/
static BOOLEAN bRemoveCategoryNotifierIterator(
    void *pvData,
    void *pvArg
        )
{
    CATEGORY_OBJECT hCategory = (CATEGORY_OBJECT)pvData;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)pvArg;

    // We need to iterate the notifiers, and unregister
    OSAL.eLinkedListIterate(
       psObj->sCategory.hAllCatNotificationList,
       (OSAL_LL_ITERATOR_HANDLER)bRemoveNotifierFromCategoryIterator,
       (void *)hCategory
           );

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveChannelNotifierIterator
*
*****************************************************************************/
static BOOLEAN bRemoveChannelNotifierIterator(
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CCACHE_OBJECT_STRUCT *psObj =
        (CCACHE_OBJECT_STRUCT *)pvArg;

    // We need to iterate the notifiers, and unregister
    OSAL.eLinkedListIterate(
       psObj->sChannel.hAllChanNotificationList,
       (OSAL_LL_ITERATOR_HANDLER)bRemoveNotifierFromChannelIterator,
       (void *)hChannel
           );

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateAllChannelArt
*
*   Iterate through the list of channels and either a) provide the channel art
*   manager handle to each one, or b) force them to update their art due
*   to service startup
*
*****************************************************************************/
static BOOLEAN bUpdateAllChannelArt (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CCACHE_ART_UPDATE_STRUCT *psArtUpdate = (CCACHE_ART_UPDATE_STRUCT *)pvArg;

    // If the service handle has changed, update the handle as well as
    // the art
    if ( CCACHE_UPDATE_TYPE_ART_SERVICE == psArtUpdate->eUpdateType )
    {
        // Update the art service for this channel
        CHANNEL_vUpdateArtService( hChannel, psArtUpdate->hChannelArtService );
    }
    // Otherwise, just force the channel to update its
    // art (on a channel art update) ...
    else if ( CCACHE_UPDATE_TYPE_CHAN_ART == psArtUpdate->eUpdateType )
    {
        // Get the art for this category now
        // that the service is up and running;
        // CCACHE__vUpdateChannelArt handles
        // setting the channel notifications after
        // we're done
        CHANNEL_vUpdateArt(hChannel, FALSE);
    }
    // ... or album art (on an album art update)
    else if ( CCACHE_UPDATE_TYPE_ALBUM_ART == psArtUpdate->eUpdateType )
    {
        SMSAPI_RETURN_CODE_ENUM eReturn;
        BOOLEAN bIsSubscribed = FALSE;

        CD_OBJECT hCDO = CHANNEL.hCDO( hChannel );
        CDO_vUpdateArt( hCDO, hChannel );

        // As a workaround for all unsubscribed channels sharing the same
        // CDO, we have to manually issue updates if this channel is
        // unsubscribed. This way each channel will get one art event
        // at product startup (which is fine, as the unsubscribed
        // CDO art will never change.)

        eReturn = CHANNEL.eIsSubscribed( hChannel, &bIsSubscribed );

        // Make sure that worked

        if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Couldn't get channel subscription status!"
                );
            return FALSE;
        }
        else if ( FALSE == bIsSubscribed )
        {
            // Update the channel's event mask with the art event
            CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_ART);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bSetChannelEvent
*
*   During product shutdown, we'll suppress updating channels to avoid
*   race conditions that result from the CLI attempting to print
*   both types of art when only one of the two (channel or category) has
*   been disabled. We use this function to force the channel updates
*   after ALL the art has been removed.
*
*****************************************************************************/
static BOOLEAN bSetChannelEvent (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_ART);

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bUpdateAllCategoryArt
*
*   Iterate through the list of categories and provide the channel art
*   manager handle to each one.
*
*****************************************************************************/
static BOOLEAN bUpdateAllCategoryArt (
    void *pvData,
    void *pvArg
        )
{
    CATEGORY_OBJECT hCategory = (CATEGORY_OBJECT)pvData;
    CCACHE_ART_UPDATE_STRUCT *psArtUpdate = (CCACHE_ART_UPDATE_STRUCT *)pvArg;

    // If the service handle has changed, update the handle as well as
    // the art
    if ( CCACHE_UPDATE_TYPE_ART_SERVICE == psArtUpdate->eUpdateType )
    {
        // Get the category object to update its art handle
        CATEGORY_vUpdateArtService( hCategory,
                psArtUpdate->hChannelArtService );
    }
    // Otherwise, just force the categories to update its art
    // handle
    else if ( CCACHE_UPDATE_TYPE_CHAN_ART == psArtUpdate->eUpdateType )
    {
        // Get the art for this category now
        // that the service is up and running
        CATEGORY_vUpdateArt( hCategory, TRUE );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bUpdateChannelEpgData
*
*****************************************************************************/
static BOOLEAN bUpdateChannelEpgData (
    void *pvData,
    void *pvArg
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT) pvData;
    EPG_CHANNEL_OBJECT hEpgChannel = (EPG_CHANNEL_OBJECT) pvArg;

    CHANNEL_vSetProgramsList( hChannel, hEpgChannel );

    return TRUE;
}

/*****************************************************************************
*
*   bDerivedCategoryOffsetIterator
*
*   Iterate through the list of categories to determine the offset of a
*   derived category within the subset of all derived categories.
*
*****************************************************************************/
static BOOLEAN bDerivedCategoryOffsetIterator (
    void *pvData,
    void *pvArg
        )
{
    CATEGORY_OBJECT hCategory =
        (CATEGORY_OBJECT)pvData;
    CATEGORY_TYPE_ENUM eCategoryType;
    CCACHE_DERIVED_CATEGORY_OFFSET_STRUCT *psOffset =
        (CCACHE_DERIVED_CATEGORY_OFFSET_STRUCT *)(size_t)pvArg;

    // Determine this category's type
    eCategoryType = CATEGORY.eType( hCategory );

    // Only continue if this is not a broadcast
    // category
    if (eCategoryType != CATEGORY_TYPE_BROADCAST)
    {
        CATEGORY_ID tCategoryId;

        // Get the category Id
        tCategoryId = CATEGORY.tGetCategoryId( hCategory );

        // Are we trying to search by offset?
        if (psOffset->bSearchByOffset == TRUE)
        {
            // If the offsets match we have
            // found what the caller is looking for
            if (psOffset->n16DerivedOffset == psOffset->n16TargetOffset)
            {
                // All done!
                psOffset->tTargetCategory = tCategoryId;
                psOffset->bFound = TRUE;
                return FALSE;
            }
        }
        else // We are searching for a given category id
        {
            // Do the category Ids match?
            if (tCategoryId == psOffset->tTargetCategory)
            {
                // Yep! Stop iterating
                psOffset->bFound = TRUE;
                return FALSE;
            }
        }

        // Increase our offset into the
        // land of derived categories
        psOffset->n16DerivedOffset++;
    }

    // Continue iteration
    return TRUE;
}

/*****************************************************************************
*
*   vDestroyChannel
*
*****************************************************************************/
static void vDestroyChannel (
    void *pvData
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;

    CHANNEL_vDestroy(hChannel);

    return;
}

/*****************************************************************************
*
*   vDestroyCategory
*
*****************************************************************************/
static void vDestroyCategory (
    void *pvData
        )
{
    CATEGORY_OBJECT hCategory = (CATEGORY_OBJECT)pvData;

    // Provide it to this category for the destroy
    CATEGORY_vDestroy( hCategory );
    return;
}

/*****************************************************************************
*
*   vDestroyChannelNotifier
*
*****************************************************************************/
static void vDestroyChannelNotifier (
    void *pvData
        )
{
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify =
        (CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *)pvData;

    SMSO_vDestroy((SMS_OBJECT)psNotify);
    return;
}

/*****************************************************************************
*
*   vDestroyCategoryNotifier
*
*****************************************************************************/
static void vDestroyCategoryNotifier (
    void *pvData
        )
{
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify =
        (CCACHE_ALL_CAT_NOTIFICATION_STRUCT *)pvData;

    SMSO_vDestroy((SMS_OBJECT)psNotify);
    return;
}

/*****************************************************************************
*
*   bIsAvailable
*
*****************************************************************************/
static BOOLEAN bIsAvailable (
    CCACHE_OBJECT hCCache
        )
{
    BOOLEAN bOwner, bIsAvailable = FALSE;

    // Verify inputs
    bOwner = SMSO_bOwner((SMS_OBJECT)hCCache);
    if (bOwner == TRUE)
    {
        // De-reference object
        CCACHE_OBJECT_STRUCT *psObj =
            (CCACHE_OBJECT_STRUCT *)hCCache;

        bIsAvailable = psObj->bAvailable;
    }

    return bIsAvailable;
}

/*****************************************************************************
*
*   vRemapFSM
*
* This is a Finite State Machine which control the remapping of the
* Channel Cache.
*
*****************************************************************************/
static void vRemapFSM (
    CCACHE_OBJECT_STRUCT *psObj,
    SMS_EVENT_CCACHE_REMAP_STRUCT const *psInRemap
        )
{
    BOOLEAN bOk = FALSE;
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
    SMS_EVENT_DATA_UNION *puEventData = NULL;
    SMS_EVENT_CCACHE_REMAP_STRUCT *psOutRemap = NULL;

    // Switch on state
    switch(psInRemap->eState)
    {
        case CCACHE_REMAP_INIT:
        {
            // Allocate an event
            hEvent = DECODER_hAllocateEvent(psObj->hDecoder,
                SMS_EVENT_CCACHE, SMS_EVENT_OPTION_NONE, &puEventData);
            if(hEvent == SMS_INVALID_EVENT_HDL)
            {
                // Nothing to do in this case
                break;
            }

            psOutRemap = &puEventData->uDecoder.sCache.uCache.sRemap;

            // Entry actions...

            // Mark cache as unavailable right now
            psObj->bAvailable = FALSE;

            // Disable the Content Monitoring Engine
            psOutRemap->eCemsDisableReturnCode =
                CME_eBlockAllEvents(psObj->hDecoder);

            // We need to invalidate any known markets.
            REPORT_vClear();

            // First prune the channel cache clearing out all contents
            // Prune the ccache leaving only 'in use' channels.
            vPrune(psObj);
            psOutRemap->bBlocked = TRUE; // Channels are blocked

            // We now have a pruned cache with only in use channels
            // and those channels are now empty except for service id
            // and attributes. We have also purged all categories.

            // Clear current channel entry marker, it is invalid anyway
            psObj->sChannel.hCurChanEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // Transition to state
            psOutRemap->bInitial = TRUE;
            psOutRemap->eState = CCACHE_REMAP_UPDATE_CHANNEL_INFO;
            bOk = SMSE_bPostEvent(hEvent);
        }
        break;

        case CCACHE_REMAP_UPDATE_CHANNEL_INFO:
        {
            CHANNEL_OBJECT hChannel;
            CHANNEL_ID tChannelId;

            // All the channels remaining in the cache need their
            // new channel info updated.

            // Initial transition?
            if(psInRemap->bInitial == TRUE)
            {
                BOOLEAN bSorted;

                // The first thing we do is re-sort our channel list
                // since we have no idea anymore if the channels are in
                // channel id order.
                bSorted = CCACHE_bSortChannelLists(
                    (CCACHE_OBJECT)psObj);
                if (FALSE == bSorted) // Error
                {
                    // Sort failed. What are we to do?
                    bOk = FALSE;
                    break;
                }

                // Allocate an event
                hEvent = DECODER_hAllocateEvent(psObj->hDecoder,
                    SMS_EVENT_CCACHE, SMS_EVENT_OPTION_NONE, &puEventData);
                if(hEvent == SMS_INVALID_EVENT_HDL)
                {
                    // Nothing to do in this case
                    break;
                }

                psOutRemap = &puEventData->uDecoder.sCache.uCache.sRemap;

                psOutRemap->hLastEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                // Start from beginning
                psOutRemap->hThisEntry = OSAL.hLinkedListFirst(
                    psObj->sChannel.hChannelListByServiceId,
                    (void **)NULL);

                if (psOutRemap->hThisEntry !=
                        OSAL_INVALID_LINKED_LIST_ENTRY) // Found something
                {
                    // we need to remember what the last entry is so we
                    // can tell when to stop
                    psOutRemap->hLastEntry =
                        OSAL.hLinkedListPrev(
                            psOutRemap->hThisEntry, (void**)NULL);

                    // Initial transition complete
                    psOutRemap->bInitial = FALSE;
                }
                else
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    UN32 un32Items = 0;

                    // How large is the channel list?
                    eReturnCode = OSAL.eLinkedListItems(
                        psObj->sChannel.hChannelListByServiceId,
                        &un32Items );

                    if ( (eReturnCode == OSAL_SUCCESS) &&
                            (un32Items == 0) )
                    {
                        // The list is empty
                        // Initial transition complete
                        psOutRemap->bInitial = FALSE;
                    }
                    else // Error
                    {
                        // Something failed. Try again.
                        bOk = SMSE_bPostEvent(hEvent);
                        break;
                    }
                }
            }

            if (hEvent == SMS_INVALID_EVENT_HDL)
            {
                // Allocate an event
                hEvent = DECODER_hAllocateEvent(psObj->hDecoder,
                    SMS_EVENT_CCACHE, SMS_EVENT_OPTION_NONE, &puEventData);
                if(hEvent == SMS_INVALID_EVENT_HDL)
                {
                    // Nothing to do in this case
                    break;
                }

                psOutRemap = &puEventData->uDecoder.sCache.uCache.sRemap;
            }

            // Get the channel handle from the list entry handle
            hChannel = (CHANNEL_OBJECT)OSAL.pvLinkedListThis(
                psInRemap->hThisEntry );

            // Extract this channel's channel id, if it's valid
            // update it, otherwise we just move on
            tChannelId = CHANNEL.tChannelId(hChannel);
            if(tChannelId != CHANNEL_INVALID_ID)
            {
                // Go update it's info
                bOk = bUpdateChannelLocal(psObj, hChannel);
            }
            else
            {
                // This channel no longer exists, skip it.
                bOk = TRUE;
            }

            // Check if we are done (guard conditions)
            if(psInRemap->hThisEntry != psInRemap->hLastEntry)
            {
                // Advance to next channel if we can, otherwise re-try
                if(bOk == TRUE)
                {
                    // Setup the next entry
                    psOutRemap->hThisEntry = OSAL.hLinkedListNext(
                        psInRemap->hThisEntry, (void**)NULL);
                }
            }
            else
            {
                // We're done.

                // Mark cache as available now.
                psObj->bAvailable = TRUE;

                // Transition to state
                psOutRemap->bInitial = TRUE;
                psOutRemap->eState = CCACHE_REMAP_SUBSCRIPTION;
            }

            // Either if it failed(re-try) or succeeded, post
            bOk = SMSE_bPostEvent(hEvent);
        }
        break;

        case CCACHE_REMAP_SUBSCRIPTION:
        {
            // We're done.

            // Allocate an event
            hEvent = DECODER_hAllocateEvent(psObj->hDecoder,
                SMS_EVENT_CCACHE, SMS_EVENT_OPTION_NONE, &puEventData);
            if(hEvent == SMS_INVALID_EVENT_HDL)
            {
                // Nothing to do in this case
                break;
            }

            psOutRemap = &puEventData->uDecoder.sCache.uCache.sRemap;

            // Transition to state
            psOutRemap->bInitial = TRUE;
            psOutRemap->eState = CCACHE_REMAP_FINAL;
            bOk = SMSE_bPostEvent(hEvent);
        }
        break;

        case CCACHE_REMAP_FINAL:
        {
            // Check if channels are blocked, if so unblock them
            if(psInRemap->bBlocked == TRUE)
            {
                // Iterate all remaining channels in the cache and
                // force an update of anything that changed.
                OSAL.eLinkedListIterate(
                    psObj->sChannel.hChannelListByServiceId,
                    bReleaseChannelIterator,
                    NULL
                        );
            }

            // Check if CME was disabled. If so, enable it
            if(psInRemap->eCemsDisableReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                CME_eUnBlockAllEvents(psObj->hDecoder);
            }

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, CCACHE_OBJECT_EVENT_UPDATE);

            // Call callback as complete
            psInRemap->vCallback(
                (CCACHE_OBJECT)psObj, TRUE, psInRemap->pvCallbackArg);

            // Signal all went well
            bOk = TRUE;
        }
        break;

        case CCACHE_REMAP_ERROR:
        default:
            bOk = FALSE;
        break;
    }

    // Check if an error occurred. If so we are done.
    if(bOk == FALSE)
    {
        // Call callback as incomplete
        psInRemap->vCallback(
            (CCACHE_OBJECT)psObj, FALSE, psInRemap->pvCallbackArg);
    }

    return;
}

/*****************************************************************************
*
*   vPrune
*
* This function is called to iterate through the Channel Cache and pruning
* it down to it's bare minimum required. Channels which are not currently
* in use (i.e. within a CHANNELLIST) can simply be removed and destroyed.
* Channels which are currently in used must remain, however the broadcast
* information contained within it must be cleared out. The remaining cache is
* what will need to be updated.
*
*****************************************************************************/
static void vPrune (
    CCACHE_OBJECT_STRUCT *psObj
        )
{
    OSAL_LINKED_LIST_ENTRY hThisEntry;
    OSAL_LINKED_LIST_ENTRY hLastEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hNextEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Start from the top
    hThisEntry = OSAL.hLinkedListFirst(
        psObj->sCategory.hCategoryList,
        (void **)NULL );

    /*
     * Prune ALL BROADCAST CATEGORY objects first, which removes all
     * channel references from these CATEGORIES
     */

    // Remove all entries in the linked list and destroy each broadcast
    // category.
    if (hThisEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // we need to remember what the last entry is so we can tell
        // when to stop
        hLastEntry = OSAL.hLinkedListPrev(hThisEntry, (void**)NULL);

        // The next entry is this entry
        hNextEntry = hThisEntry;

        do
        {
            CATEGORY_TYPE_ENUM eType;
            CATEGORY_OBJECT hCategory;

            // Move on to the next one
            hThisEntry = hNextEntry;

            // get the next entry
            hNextEntry = OSAL.hLinkedListNext(hThisEntry, (void**)NULL);

            // Get the category handle from the list entry handle
            hCategory = (CATEGORY_OBJECT)OSAL.pvLinkedListThis( hThisEntry );

            // Determine if this category is BROADCAST
            eType = CATEGORY.eType(hCategory);
            if (eType == CATEGORY_TYPE_BROADCAST)
            {
                printf(CCACHE_OBJECT_NAME": Pruning category #%u.\n",
                    CATEGORY.tGetCategoryId(hCategory));

                // Delete this entry in our cache.
                eReturnCode = OSAL.eLinkedListRemove(hThisEntry);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    // Destroy the category object now.
                    vDestroyCategory(hCategory);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CCACHE_OBJECT_NAME": pruning category #%u.",
                        CATEGORY.tGetCategoryId(hCategory));
                }
            } // (eType == CATEGORY_TYPE_BROADCAST)
            else
            {
                // Skip this category
                printf(CCACHE_OBJECT_NAME": Category #%u is not bcast...skip.\n",
                    CATEGORY.tGetCategoryId(hCategory));
            }

        } while ( hThisEntry != hLastEntry );

    } // if (hThisEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    else
    {
        UN32 un32Items = 0;

        // How many categories do we have?
        eReturnCode = OSAL.eLinkedListItems (
            psObj->sCategory.hCategoryList,
            &un32Items );

        if ((eReturnCode == OSAL_SUCCESS) &&
            (un32Items == 0))
        {
            // Nothing to prune
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME":No categories to prune.");
        }
        else
        {
            // Error!
            puts(
                CCACHE_OBJECT_NAME": Error! Unable to prune categories.\n");
        }
    }

    /*
     * Prune all CHANNELS which are currently in our ccache
     *
     */

    // Start from the top
    hThisEntry = OSAL.hLinkedListFirst(
        psObj->sChannel.hChannelListByServiceId,
        (void **)NULL );

    if (hThisEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // we need to remember what the last entry is so we can tell
        // when to stop
        hLastEntry = OSAL.hLinkedListPrev(hThisEntry, (void**)NULL);

        // The next entry is this entry
        hNextEntry = hThisEntry;

        do
        {
            BOOLEAN bInUse;
            CHANNEL_OBJECT hChannel;

            // Move on to the next one
            hThisEntry = hNextEntry;

            // get the next entry
            hNextEntry = OSAL.hLinkedListNext(hThisEntry, (void**)NULL);

            // Get the channel handle from the list entry handle
            hChannel = (CHANNEL_OBJECT)OSAL.pvLinkedListThis( hThisEntry );

            printf(CCACHE_OBJECT_NAME": Pruning channel (%u,%u)...",
                CHANNEL.tServiceId(hChannel), CHANNEL.tChannelId(hChannel));

            // Block channel notifications for now
            CHANNEL_bBlockNotifications(hChannel);

            // Clear the contents of this channel sans service id,
            // and any application provided attributes.
            // This way all channels are cleared regardless of their
            // ability to be removed.
            CHANNEL_vClearBroadcast(hChannel);
            CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_REMOVED);

            // Determine if this channel is currently being used
            bInUse = CHANNEL_bInUse(hChannel);
            if (bInUse == FALSE)
            {
                // Delete this entry in our cache.
                eReturnCode = OSAL.eLinkedListRemove(hThisEntry);
                if (eReturnCode == OSAL_SUCCESS)
                {
                    printf("destroyed.\n");

                    // Destroy the channel object now.
                    vDestroyChannel(hChannel);
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CCACHE_OBJECT_NAME": Couldn't Remove Channel.");
                }
            } // (bInUse == FALSE)
            else
            {
                // Skip this channel
                printf("skipped.\n");
            }

        } while ( hThisEntry != hLastEntry );

    } // if (hThisEntry != OSAL_INVALID_LINKED_LIST_ENTRY
    else
    {
        UN32 un32Items = 0;

        // How many channels do we have?
        eReturnCode = OSAL.eLinkedListItems (
            psObj->sChannel.hChannelListByServiceId,
            &un32Items );

        if ((eReturnCode == OSAL_SUCCESS) &&
            (un32Items == 0))
        {
            // Nothing to prune
            puts( CCACHE_OBJECT_NAME":No channels to prune.\n");
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Unable to prune channels.");
        }
    }

    // At this point all channels have been pruned and cleared of
    // any broadcast information except for service id. This includes
    // category information as well. Thus we are now free to
    // destroy all categories in our cache (except user and virtual).

    printf("\n\n"CCACHE_OBJECT_NAME":vRemapFSM:"
        " Pruning complete...\n\n");
#if SMS_DEBUG == 1
    SMSAPI_DEBUG_vPrintChannels(psObj->hDecoder, FALSE);
    SMSAPI_DEBUG_vPrintCategories(psObj->hDecoder, FALSE);
#endif /* SMS_DEBUG == 1 */

    return;
}

/*****************************************************************************
*
*   n16CompareAllChannelNotifers
*
*****************************************************************************/
static N16 n16CompareAllChannelNotifers (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify1,
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify2
        )
{
    N16 n16Return = N16_MIN;

    if ((psNotify1 != NULL) && (psNotify2 != NULL))
    {
        // At first, we compare callback pointers
        n16Return = COMPARE(psNotify1->vEventCallback, 
            psNotify2->vEventCallback);

        // if callbacks are the same, lets compare argument pointers
        if (n16Return == 0)
        {
            n16Return = COMPARE(psNotify1->pvEventCallbackArg, 
                psNotify2->pvEventCallbackArg);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16CompareAllCategoryNotifers
*
*****************************************************************************/
static N16 n16CompareAllCategoryNotifers (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify1,
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify2
        )
{
    N16 n16Return = N16_MIN;

    if ((psNotify1 != NULL) && (psNotify2 != NULL))
    {
        // At first, we compare callback pointers
        n16Return = COMPARE(psNotify1->vEventCallback, 
            psNotify2->vEventCallback);

        // if callbacks are the same, lets compare argument pointers
        if (n16Return == 0)
        {
            n16Return = COMPARE(psNotify1->pvEventCallbackArg, 
                psNotify2->pvEventCallbackArg);
        }
    }

    return n16Return;
}

/*****************************************************************************
*
*   n16CountNumInvalidCategories
*
*****************************************************************************/
static N16 n16CountNumInvalidCategories(CCACHE_OBJECT_STRUCT *psObj)
{
    N16 n16NumInvalidCategories = 0;
    BOOLEAN bSuccess;

    bSuccess = CCACHE_bSortCategoryList((CCACHE_OBJECT)psObj);
    if(TRUE == bSuccess)
    {
        OSAL_LINKED_LIST_ENTRY hLastEntry, hThisEntry;

        // Start with last entry
        hThisEntry = hLastEntry = OSAL.hLinkedListLast(
            psObj->sCategory.hCategoryList, (void **)NULL);
        if (OSAL_INVALID_LINKED_LIST_ENTRY != hThisEntry)
        {
            // Find invalid categories and count how many there are.
            do
            {
                CATEGORY_OBJECT hCategory;
                CATEGORY_ID tCategoryId;

                hCategory = (CATEGORY_OBJECT)
                    OSAL.pvLinkedListThis(hThisEntry);

                tCategoryId = CATEGORY.tGetCategoryId(hCategory);
                if(CATEGORY_INVALID_ID == tCategoryId)
                {
                    // Accumulate invalid one
                    n16NumInvalidCategories++;

                    // Move to previous entry
                    hThisEntry = OSAL.hLinkedListPrev(hThisEntry, (void **)NULL);
                }
                else
                {
                    // Done looking. First one which is valid is the last
                    // invalid once since the list is sorted by category id.
                    break;
                }

            } while(hLastEntry != hThisEntry);
        }
    }

    // Set number of invalid channels for future use
    psObj->sCategory.n16NumInvalidCategories =
        n16NumInvalidCategories;

    return n16NumInvalidCategories;
}

/*****************************************************************************
*
*   hChannelFromIds
*
*****************************************************************************/
static CHANNEL_OBJECT hChannelFromIds (
    CCACHE_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        )
{
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_OBJECT_NOT_FOUND;

    do
    {
        // search from top of list
        *phEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Update the dummy channel's channel id and service id
        // per inputs from the caller. All this really does is
        // directly modify the 'dummy' channel's ids.
        CHANNEL_bUpdateIds(
            *psObj->phDummyChannel, tServiceId, tChannelId);

        // Search by service-id?
        if(tServiceId != SERVICE_INVALID_ID)
        {
            // Check if this channel exists in our list. Note here
            // we are looking for the CHANNEL using a Service Id.
            eReturnCode =
                OSAL.eLinkedListSearch(
                    psObj->sChannel.hChannelListByServiceId,
                    phEntry,
                    *psObj->phDummyChannel );
            if(eReturnCode == OSAL_SUCCESS) // found
            {
                // Channel exists, extract the data from this entry
                hChannel = (CHANNEL_OBJECT)OSAL.pvLinkedListThis(*phEntry);
                break;
            }
        }

        // Search by channel-id
        if(tChannelId != CHANNEL_INVALID_ID)
        {
            // Check if this channel exists in our list. Note here
            // we are looking for the CHANNEL using a channel Id but
            // we also must match the service id as well.
            eReturnCode =
                OSAL.eLinkedListSearch(
                    psObj->sChannel.hChannelListByChannelId,
                    phEntry,
                    *psObj->phDummyChannel );
            if(eReturnCode == OSAL_SUCCESS) // found
            {
                // Channel exists, extract the data from this entry
                hChannel = (CHANNEL_OBJECT)OSAL.pvLinkedListThis(*phEntry);
                break;
            }
        }

    } while (FALSE);

    // Return channel found, if any
    return hChannel;
}

/*****************************************************************************
*
*   vStoreACOData
*
*****************************************************************************/
static void vStoreACOData(
    CCACHE_OBJECT_STRUCT *psObj
        )
{
    FILE *psFile = NULL;
    BOOLEAN bCorrupted = TRUE;
    const char *pacACOPath = NULL;
    CCACHE_COMMIT_STRUCT sData;

    OSAL_BUFFER_BLOCK_HDL hBlock = OSAL_INVALID_BUFFER_BLOCK_HDL;
    OSAL_OBJECT_HDL hBlockPool = OSAL_INVALID_OBJECT_HDL;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;

    do
    {
        size_t tWritten, tBlockSize;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        
        UN32 un32Channels;
        UN8 *pun8BlockOrigin;

        OSAL_CRC_RESULT tCRC;
        BOOLEAN bSuccess;

        // Determine how many channels we have
        eReturnCode = OSAL.eLinkedListItems(
            psObj->sChannel.hChannelListByChannelId, 
            &un32Channels);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to get number of "
                "channels: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        // nothing to do
        if (un32Channels == 0)
        {
            puts(CCACHE_OBJECT_NAME": CCACHE is empty");
            break;
        }

        // Allocate string and return ACO file path.
        pacACOPath = pacGetACOPath( psObj );
        if (pacACOPath == NULL)
        {
            // Error!
            break;
        }

        // create block pool for CRC calculation
        eReturnCode = OSAL.eBlockPoolCreate(
            &hBlockPool,
            CCACHE_OBJECT_NAME":ACO:Pool",
            // ACO storage struct size x number of channels
            sizeof(CCACHE_ACO_STORAGE_STRUCT) * un32Channels,
            1,
            OSAL_BLOCK_POOL_OPTION_NONE
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to create ACO "
                "Block Pool: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        // allocate buffer in our block pool
        hBuffer = OSAL.hBufferAllocate(
            hBlockPool,
            FALSE,
            FALSE,
            OSAL_BUFFER_ALLOCATE_OPTION_NONE
                );

        if (hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to allocate ACO buffer");
            break;
        }

        hBlock = OSAL.hBufferGetBlock(hBuffer, &pun8BlockOrigin, &tBlockSize);
        if (hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to get block from ACO buffer");
            break;
        }

        // set iterator arguments
        sData.bSuccess = TRUE;
        sData.un32Channels = 0;
        sData.pun8Block = pun8BlockOrigin;

        // iterate thru channels and write them into buffer / file
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->sChannel.hChannelListByChannelId,
            bStoreACODataIterator,
            (void *)&sData);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to iterate CCACHE by "
                "Service IDs: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        if (sData.bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write ACO data into buffer");
            break;
        }

        eReturnCode = OSAL.eBufferWriteBlock(
            hBlock, 
            // Could be less than block size
            sData.pun8Block - pun8BlockOrigin);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write block back to buffer");
            break;
        }

        // Set block to invalid
        hBlock = OSAL_INVALID_BUFFER_BLOCK_HDL;

        if (sData.un32Channels == 0)
        {
            // looks like there are no channels with ACO
            break;
        }

        bSuccess = bCalculateISO3309CRC32(hBuffer, &tCRC);
        if (bSuccess == FALSE)
        {
            // Should have error print outs from inside
            break;
        }

        printf(CCACHE_OBJECT_NAME": Attempting to store %d channels "
            "into file\n", sData.un32Channels);

        // open file
        psFile = fopen(pacACOPath, "wb");
        if (psFile == NULL)
        {
            break;
        }

        // write the number of channels in the beginning
        tWritten = fwrite(&sData.un32Channels, sizeof(UN32), 1, psFile);
        if (tWritten == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write number of "
                "Channels into file");
            break;
        }

        hBlock = OSAL.hBufferReadBlock(
            hBuffer, 
            &sData.pun8Block, 
            &tBlockSize);

        if (hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to get block from buffer");
            break;
        }

        // dump the whole buffer into file
        tWritten = fwrite(sData.pun8Block, tBlockSize, 1, psFile);
        if (tWritten == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write data into file");
            break;
        }

        // write tailing CRC value
        tWritten = fwrite(&tCRC, sizeof(OSAL_CRC_RESULT), 1, psFile);
        if (tWritten == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write CRC value into the file");
            break;
        }

        // Indicate that file is OK
        bCorrupted = FALSE;

        printf(CCACHE_OBJECT_NAME": ACO data has been successfully "
            "stored, CRC: %X\n", tCRC);

    } while (FALSE);

    if (hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(hBuffer);
    }

    if (hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eBlockPoolDelete(hBlockPool);
    }

    if (psFile != NULL)
    {
        fclose(psFile);

        if (bCorrupted == TRUE)
        {
            remove(pacACOPath);
        }
    }

    if (pacACOPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacACOPath);
    }

    return;
}

/*****************************************************************************
*
*   bStoreACODataIterator
*
*****************************************************************************/
static BOOLEAN bStoreACODataIterator ( 
    void *pvData, 
    void *pvArg 
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvData;
    CCACHE_COMMIT_STRUCT *psData = (CCACHE_COMMIT_STRUCT *)pvArg;

    SERVICE_ID tServiceId;
    CHANNEL_ID tChannelId;
    CHANNEL_ACO tACO;

    if ((psData == NULL) || 
        (hChannel == CHANNEL_INVALID_OBJECT))
    {
        return FALSE;
    }

    tServiceId = CHANNEL.tServiceId(hChannel);
    tChannelId = CHANNEL.tChannelId(hChannel);
    tACO = CHANNEL.tACO(hChannel);

    if ((tServiceId != SERVICE_INVALID_ID) &&
        (tChannelId != CHANNEL_INVALID_ID) &&
        (tACO != CHANNEL_ACO_DEFAULT))
    {
        CCACHE_ACO_STORAGE_STRUCT sData;
        BOOLEAN bCopied;

        // convert channel attributes into 4-byte format
        sData.tSID = tServiceId;
        sData.tACO = tACO;

        // write data to the buffer
        bCopied = OSAL.bMemCpy(
            psData->pun8Block, 
            &sData, 
            sizeof(sData));

        if (bCopied == FALSE)
        {
            psData->bSuccess = FALSE;
            return FALSE;
        }

        psData->pun8Block += sizeof(sData);
        psData->un32Channels++;

        printf(CCACHE_OBJECT_NAME": Service ID: %3d, ACO: %5d written "
            "into buffer\n", tServiceId, tACO);
    }

    return TRUE;
}

/*****************************************************************************
*
*   vRestoreACOData
*
*****************************************************************************/
static void vRestoreACOData(
    CCACHE_OBJECT_STRUCT *psObj
        )
{
    const char *pacACOPath = NULL;
    FILE *psFile = NULL;

    OSAL_OBJECT_HDL hBlockPool = OSAL_INVALID_OBJECT_HDL;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    OSAL_BUFFER_BLOCK_HDL hBlock = OSAL_INVALID_BUFFER_BLOCK_HDL;

#if SMS_DEBUG == 0
    // For Debug configuration only, it is allowed to keep 
    // the corrupted file for further analysis. In Release variant,
    // there is a potential corner case which can cause attempts to
    // load such file on every start-up. This file will be overwritten
    // on the nearest carousel cycle (about 3-4 minutes). 
    // If you can pull it, you can have it.
    BOOLEAN bRemove = FALSE;
#endif

    do
    {
        size_t tRead, tIndex;
        UN32 un32Channels;
        BOOLEAN bSuccess = TRUE;
        OSAL_CRC_RESULT tFileCRC, tCRC;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        SERVICE_ID tServiceId;
        CHANNEL_ACO tACO;

        CHANNEL_OBJECT hChannel;

        UN8 *pun8Block;
        size_t tBlockSize;
        CCACHE_ACO_STORAGE_STRUCT *psChannelACO;

        // Allocate string and return ACO file path.
        pacACOPath = pacGetACOPath( psObj );
        if (pacACOPath == NULL)
        {
            break;
        }

        // open file for reading
        psFile = fopen(pacACOPath, "rb");
        if (psFile == NULL)
        {
            break;
        }

        tRead = fread( &un32Channels, sizeof(UN32), 1, psFile );
        if (tRead == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to read number of channels");
            break;
        }

        // Create block pool for CRC calculation. The size of block in 
        // the pool is (ACO storage structure size * number of channels)
        eReturnCode = OSAL.eBlockPoolCreate(
            &hBlockPool,
            CCACHE_OBJECT_NAME":ACO:Pool",
            sizeof(CCACHE_ACO_STORAGE_STRUCT) * un32Channels,
            1,
            OSAL_BLOCK_POOL_OPTION_NONE
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to create ACO "
                "Block Pool: %s (#%d)",
                OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        // allocate buffer in our block pool
        hBuffer = OSAL.hBufferAllocate(
            hBlockPool,
            FALSE,
            FALSE,
            OSAL_BUFFER_ALLOCATE_OPTION_NONE
                );

        if (hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to allocate ACO buffer");
            break;
        }

        hBlock = OSAL.hBufferGetBlock(hBuffer, &pun8Block, &tBlockSize);
        if (hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to get block from buffer");
            break;
        }

        tRead = fread( pun8Block, 
            sizeof(CCACHE_ACO_STORAGE_STRUCT), 
            un32Channels, 
            psFile);

        if (tRead != un32Channels)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to read data from file. "
                "Only %d channels processed", tRead);
            break;
        }

        // write block
        eReturnCode = OSAL.eBufferWriteBlock(hBlock, 
            sizeof(CCACHE_ACO_STORAGE_STRUCT) * un32Channels);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to write block into buffer");
            break;
        }

        // set block to invalid
        hBlock = OSAL_INVALID_BUFFER_BLOCK_HDL;

        // read CRC value from file
        tRead = fread(&tFileCRC, sizeof(OSAL_CRC_RESULT), 1, psFile);
        if (tRead == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to read CRC value from file");
            break;
        }

        // Calculate CRC
        bSuccess = bCalculateISO3309CRC32(hBuffer, &tCRC);
        if (bSuccess == FALSE)
        {
            // Should have error print outs from inside
            break;
        }

        if (tCRC != tFileCRC)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": CRC values don't match!");
#if SMS_DEBUG == 0
            // In release only - kill the file to prevent unnecessary
            // resources utilization
            bRemove = TRUE;
#endif
            break;
        }

        hBlock = OSAL.hBufferReadBlock(hBuffer, &pun8Block, &tBlockSize);

        psChannelACO = (CCACHE_ACO_STORAGE_STRUCT *)pun8Block;

        // data completely read; restore channels now
        for ( tIndex = 0; 
              tIndex < un32Channels; 
              tIndex++ )
        {
            // Read one channel's SID and ACO from buffer
            tServiceId = psChannelACO->tSID;
            tACO = psChannelACO->tACO;
            // Move pointer to next channel
            psChannelACO++;

            // Get channel from cache. Create if not exist
            hChannel = CCACHE_hChannelFromIds(
                (CCACHE_OBJECT)psObj, 
                tServiceId, 
                CHANNEL_INVALID_ID, 
                TRUE);

            if (hChannel == CHANNEL_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CCACHE_OBJECT_NAME": Failed to create Channel SID: %d",
                    tServiceId);

                // Just skip this channel and let others go.
                continue;
            }

            // set ACO value to channel object
            bSuccess = CHANNEL_bUpdateACO(hChannel, tACO);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    CCACHE_OBJECT_NAME": Failed to update ACO value "
                    "for Channel SID: %d\n", tServiceId);

                // Just skip this channel and let others go.
                continue;
            }

            printf(CCACHE_OBJECT_NAME": Channel SID: %3d updated "
                "with ACO: %5d\n", tServiceId, tACO);
        }

    } while(FALSE);

    if (hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(hBuffer);
    }

    if (hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eBlockPoolDelete(hBlockPool);
    }

    if (psFile != NULL)
    {
        fclose(psFile);

#if SMS_DEBUG == 0
        // File corruption has been detected. Remove it in release build.
        if (bRemove == TRUE)
        {
            remove(pacACOPath);
        }
#endif
    }

    if (pacACOPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacACOPath);
    }

    return;
}

/*****************************************************************************
*
*   pacGetACOPath
*
*****************************************************************************/
static const char *pacGetACOPath( 
    CCACHE_OBJECT_STRUCT *psObj 
        )
{
    char *pacACOPath = NULL;

    do
    {
        size_t tPathLength;
        const char *pacSMSPath;

        // Get SMS path
        pacSMSPath = SMS_pacGetPath();
        if (pacSMSPath == NULL)
        {
            break;
        }

        // calculate SMS path length
        tPathLength = strlen(pacSMSPath);

        // allocate ACO path array
        pacACOPath = (char *)SMSO_hCreate(
            CCACHE_OBJECT_NAME":ACO Path",
            tPathLength 
                + sizeof(CCACHE_OBJECT_ACO_FILE_NAME) + 1,
            (SMS_OBJECT)psObj,
            FALSE);

        if (pacACOPath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to allocate ACO path array");
            break;
        }

        // append ACO file name
        snprintf(
            pacACOPath,
            tPathLength + 1 + sizeof(CCACHE_OBJECT_ACO_FILE_NAME),
            "%s/%s",
            pacSMSPath,
            CCACHE_OBJECT_ACO_FILE_NAME
                );
    } while (FALSE);

    return pacACOPath;
}

/*****************************************************************************
*
*   bCalculateISO3309CRC32
*
*****************************************************************************/
static BOOLEAN bCalculateISO3309CRC32(
    OSAL_BUFFER_HDL hBuffer, 
    OSAL_CRC_RESULT *ptCRC
        )
{
    BOOLEAN bReturn = FALSE;
    OSAL_OBJECT_HDL hCRC = OSAL_INVALID_OBJECT_HDL;

    do
    {
        size_t tBytesProcessed;
        BOOLEAN bSuccess;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        size_t tBufferSize;
        OSAL_CRC_RESULT tCRC;

        if (ptCRC == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Invalid CRC pointer provided");
            break;
        }

        eReturnCode = OSAL.eGetCRC(
            &hCRC, 
            OSAL_CRC_TYPE_ISO3309_CRC32
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to get CRC calculator: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            break;
        }

        bSuccess = OSAL.bInitializeCRC(hCRC, &tCRC);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Failed to initialize CRC calculator");
            break;
        }

        // get buffer size
        tBufferSize = OSAL.tBufferGetSize(hBuffer);

        // Compute CRC value
        tCRC = OSAL.tComputeCRC(
            hCRC, 
            tCRC, 
            hBuffer, 
            0, 
            tBufferSize, 
            &tBytesProcessed
                );

        if (tBytesProcessed != tBufferSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CCACHE_OBJECT_NAME": Bytes processed value mismatch");
            break;
        }

        *ptCRC = tCRC;
        bReturn = TRUE;

    } while (FALSE);

    if (hCRC != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eReleaseCRC(hCRC);
    }

    return bReturn;
}
