/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CATEGORY implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_obj.h"
#include "cm.h"
#include "module_obj.h"
#include "decoder_obj.h"
#include "channel_obj.h"
#include "string_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_obj.h"
#include "channel_art_mgr_obj.h"
#include "presets_obj.h"
#include "radio.h"

#include "category_obj.h"
#include "_category_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       tGetCategoryId
*
*       This function is used to get the id for a category via its category
*       handle.  The id returned is the one assigned by the category service.
*       Since the category service assigns these category ids, there is no API
*       available for the application to modify this assigned number.  This
*       will work for all category types.  This function may only be called
*       from the context of the DECODER because it uses the category object
*       handle.  Outside of the DECODER's context, the CATEGORY's handle
*       is not valid.
*
*       Inputs:
*           hCategory           Handle to the category to get the id for
*
*       Outputs:
*           CATEGORY_ID         category identifier for the provided category
*                               object
*
*****************************************************************************/
static CATEGORY_ID tGetCategoryId (
    CATEGORY_OBJECT hCategory
        )
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
    BOOLEAN bOwner;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        tCategoryId = psObj->tId;
    }

    return tCategoryId;
}

/*****************************************************************************
*
*       tGetUserId
*
*       This API is used to get the user id for a category via its category
*       handle.  The id returned is the one the application has assigned for the
*       given channel.  This will work for all category types.
*
*       Inputs:
*           hCategory       Category to get the user id for
*
*       Outputs:
*           USER_ID         user identifier assigned by the application
*                           belonging to the provided category id.
*
*****************************************************************************/
static USER_ID tGetUserId(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    USER_ID tUserId = USER_INVALID_ID;
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if(hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            tUserId = psObj->tUserId;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tUserId;
}

/*****************************************************************************
*
*       eSetUserId
*
*       This function is used to set the user id for a specified category.
*       This will work for all category types.
*
*       Inputs:
*           hCategory       Handle to the category to set the user id for
*           tUserId         The user id to assign to the given category.  This
*                           will overwrite any previous value stored. This id
*                           need not be unique amongst categories.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetUserId(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    USER_ID tUserId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn =
        SMSAPI_RETURN_CODE_ERROR;
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if(hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // Set the user id into the category object
            psObj->tUserId = tUserId;

            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturn;
}

/*****************************************************************************
*
*       tSize
*
*       This API is used to report the size of a category to the caller. The
*       size refers to the number of channel objects in the specified category.
*
*       Inputs:
*           hCategory       A handle to an existing category object.
*
*       Outputs:
*           size_t          The number of channels in this category object.
*
*****************************************************************************/
static size_t tSize(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    UN32 un32NumChannels = 0;
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if(hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // Number of channels
            OSAL.eLinkedListItems(psObj->hChannelList, &un32NumChannels);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return (size_t)un32NumChannels;
}

/*****************************************************************************
*
*   tEventMask
*
*****************************************************************************/
static CATEGORY_EVENT_MASK tEventMask (
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bValid;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Verify inputs
    bValid = SMSO_bIsValid((SMS_OBJECT)hCategory);
    if(bValid == FALSE)
    {
        // Error!
        return CATEGORY_OBJECT_EVENT_NONE;
    }

    return SMSU_tMask(&psObj->sEvent);
}

/*****************************************************************************
*
*       hShortName
*
*       This API is used to get the short name handle for the specified
*       category.  This function may only be called from the context of the
*       DECODER because it uses the category object handle.  Outside of the
*       DECODER's context, the CATEGORY's handle is not valid.
*
*       Inputs:
*           hCategory       A handle to an existing category object.
*
*       Outputs:
*           STRING_OBJECT   Handle to the string object containing the short
*                           name of the category
*
*****************************************************************************/
static STRING_OBJECT hShortName (
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hShortName = STRING_INVALID_OBJECT;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        hShortName = psObj->hShortName;
    }

    return hShortName;
}

/*****************************************************************************
*
*       hMediumName
*
*       This API is used to get the medium name handle for the specified
*       category.  This function may only be called from the context of the
*       DECODER because it uses the category object handle.  Outside of the
*       DECODER's context, the CATEGORY's handle is not valid.
*
*       Inputs:
*           hCategory       A handle to an existing category object.
*
*       Outputs:
*           STRING_OBJECT   Handle to the string object containing
*                           the medium name of the category
*
*****************************************************************************/
static STRING_OBJECT hMediumName (
    CATEGORY_OBJECT hCategory
    )
{
    BOOLEAN bOwner;
    STRING_OBJECT hMediumName = STRING_INVALID_OBJECT;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        hMediumName = psObj->hMediumName;
    }

    return hMediumName;
}

/*****************************************************************************
*
*       hLongName
*
*       This API is used to get the long name handle for the specified
*       category.  This function may only be called from the context of the
*       DECODER because it uses the category object handle.  Outside of the
*       DECODER's context, the CATEGORY's handle is not valid.
*
*       Inputs:
*           hCategory       A handle to an existing category object.
*
*       Outputs:
*           STRING_OBJECT   Handle to the string object containing the long
*                           name of the category
*
*****************************************************************************/
static STRING_OBJECT hLongName (
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bOwner;
    STRING_OBJECT hLongName = STRING_INVALID_OBJECT;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        hLongName = psObj->hLongName;
    }

    return hLongName;
}

/*****************************************************************************
*
*       eRename
*
*       This API is used to change the the specified
*       category's long name and or short name.
*
*       Inputs:
*           tCategoryId     An id of the category object to rename.
*           pacLongName     NULL terminated string containing the long name
*                           of the category.
*           pacShortName    NULL terminated string containing the short name
*                           of the category.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM    SMSAPI_RETURN_CODE_SUCCESS on success or
*                                      not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRename(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    const char *pacNewLongName,
    const char *pacNewShortName
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        // the category handle must be valid and the user must have provided
        // at least one non-NULL name string
        if ( (hCategory != CATEGORY_INVALID_OBJECT) &&
           ( (pacNewLongName != NULL) || (pacNewShortName != NULL) ) )
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // Verify this is not a broadcast category
            if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
            {
                // we don't the application to rename broadcast categories.
                // Unlock object
                SMSO_vUnlock((SMS_OBJECT)hDecoder);
                return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            }

            // Only Long and Short names are supported for user categories
            // so we reuse short name for medium one
            eReturnCode = eRenameLocal(psObj, 
                pacNewLongName, pacNewShortName, pacNewShortName );
        }
        else
        {
            // the caller gave us bad arguments
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eType
*
*       This API is used to get the category type for the specified category.
*
*       Inputs:
*           hCategory       A handle to an existing category object.
*
*       Outputs:
*           CATEGORY_TYPE_ENUM      Broadcast or user category
*
*
*****************************************************************************/
static CATEGORY_TYPE_ENUM eType (
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bOwner;
    CATEGORY_TYPE_ENUM eCategoryType =
        CATEGORY_TYPE_INVALID;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        eCategoryType = psObj->eCategoryType;
    }

    return eCategoryType;
}

/*****************************************************************************
*
*       hArt
*
*       This API is used to get the art for the specified category.
*
*       Inputs:
*           hCategory           A handle to an existing category object.
*
*       Outputs:
*           CHANNEL_ART_OBJECT  Handle to the channel art object containing
*                               the channel art for this category
*
*
*****************************************************************************/
static CHANNEL_ART_OBJECT hArt (
    CATEGORY_OBJECT hCategory
        )
{
    CHANNEL_ART_OBJECT hArt =
        CHANNEL_ART_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        hArt = psObj->hArt;
    }

    return hArt;
}

/*****************************************************************************
*
*       tCreateUserCategory
*
*       This function is used to create a new user category (which contains
*       no valid channels).
*
*       Inputs:
*           hDecoder                Decoder to attach this category to.  This
*                                   value must not be DECODER_INVALID_OBJECT or
*                                   the creation will fail.
*           pacLongName             NULL terminated string containing the long
*                                   name of the category.  If one is not
*                                   provided, the short name will be used.
*           pacShortName            NULL terminated string containing the short
*                                   name of the category.  If one is not
*                                   provided, the long name will be used.
*           tInitialNumChannels     If 0, the list will be created with no
*                                   items.  If tInitialNumChannels >0, the
*                                   category will be created with
*                                   tInitialNumChannels in it all set to
*                                   CHANNEL_INVALID_ID
*
*       Outputs:
*           CATEGORY_OBJECT     Handle to the created category
*
*****************************************************************************/
static CATEGORY_ID tCreateUserCategory(
    DECODER_OBJECT hDecoder,
    const char *pacLongName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels
        )
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
    BOOLEAN bLocked;

    // Verify and Lock the DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        // Get the ccache attached to this decoder
        hCCache = DECODER_hCCache( hDecoder );

        // If we failed, we cannot create the category so just return
        if ( hCCache != CCACHE_INVALID_OBJECT )
        {
            // Get first available CATEGORY_ID from within the category lists
            // that falls into the range of user categories
            tCategoryId =
                CCACHE_tAddCategory(
                            hCCache,
                            CATEGORY_TYPE_USER,
                            pacLongName,
                            pacShortName,  // Reuse short name for medium
                            pacShortName,
                            tInitialNumChannels,
                            FALSE, FALSE);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
*
*   vDestroyDerivedCategory
*
*   This function not only destroys the category, but it also removes the
*   category from the channel cache for the associated decoder.  This function
*   also ensures the provided category really was a user/virtual category.
*
*   Inputs:
*       hDecoder    Handle to the decoder for this category
*       hCategory   Handle to the category to destroy
*
*
*   Outputs:
*       none
*
*****************************************************************************/
static void vDestroyDerivedCategory (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if(hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // Make sure that this is not a broadcast category
            if (psObj->eCategoryType != CATEGORY_TYPE_BROADCAST)
            {
                CCACHE_OBJECT hCCache;
                CHANNEL_OBJECT hChannel;
                BOOLEAN bBlocked = FALSE;

                // Find the cache by using the decoder
                hCCache = DECODER_hCCache( hDecoder );

                // if we found a valid ccache
                if ( hCCache != CCACHE_INVALID_OBJECT )
                {
                    // First remove this category from the ccache
                    CCACHE_vRemoveCategory( hCCache, tCategoryId );
                }

                // Extract browsed channel handle
                hChannel = DECODER_hBrowsedChannel(hDecoder);
                if (hChannel != CHANNEL_INVALID_OBJECT)
                {
                    N16 n16Offset = 0;
                    SMSAPI_RETURN_CODE_ENUM eReturnCode;

                    eReturnCode = CHANNEL.eCategoryOffset(
                        hChannel,
                        tCategoryId,
                        &n16Offset
                            );

                    if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
                        (n16Offset >= 0))
                    {
                        // Browsed channel is part of this category.
                        // Need to block notifications for this channel
                        // while this category is in process of removing.
                        // Due to the logic inside CATEGORY_vDestroy,
                        // channel notification will be generated in the
                        // middle, between CATEGORY_OBJECT_EVENT_CONTENTS
                        // and CATEGORY_OBJECT_EVENT_REMOVED. This makes
                        // difficult to adjust browsed category before
                        // channel notification has occurred.
                        // TODO: probably, category vs. channel hooks
                        // could be reworked, so no additional logic
                        // will be needed.
                        bBlocked = CHANNEL_bBlockNotifications(hChannel);
                    }
                }

                // Complete the cleanup
                CATEGORY_vDestroy( hCategory );

                if (bBlocked == TRUE)
                {
                    CHANNEL_vReleaseNotifications(hChannel);
                }
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
*
*       eIterate
*
*       This API is used to iterate through the channel objects which are
*       members of the specified category. When this API is invoked, the module
*       services will iterate the category's channel list from start to finish,
*       executing the caller provided n16IterateFxn function for each member
*       channel.   The return value from n16IterateFxn will determine how the
*       API traverses or iterates channels within a category.
*
*       Inputs:
*           hDecoder        A handle to the decoder object for the specified
*                           category
*           tCategoryId     An id of the category object in which to iterate.
*           n16IterateFxn   This is a category iteration function or handler
*                           defined by the caller. This handler is called by the
*                           iteration API for each channel object that is a
*                           member of the category.  Refer to the Definitions
*                           section for more information on the
*                           CATEGORY_ITERATION_HANDLER.
*           pvIterateArg    A caller provided pointer which will be provided as
*                           an argument when the n16IterateFxn call is made.
*                           This value is application specific and must be cast
*                           to the appropriate type within the caller provided
*                           iteration function.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     Success or failure
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CATEGORY_ITERATION_HANDLER n16IterateFxn,
    void *pvIterateArg
        )
{
    // Lock object for exclusive access
    BOOLEAN bLocked;

    // Verify the inputs
    if ( n16IterateFxn == NULL )
    {
        // Error!
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;
        CHANNEL_OBJECT hChannel;
        N16 n16CategorySize;
        N16 n16IteratorResult = !0;
        N16 n16CategoryIndex;
        CATEGORY_CHANNEL_INDEX tIndex = 0;
        N8 n8Direction;
        int iNumIterations;

        // Get the category handle for this id
        hCategory = hGetHandle(hDecoder, tCategoryId);

        if (hCategory != CATEGORY_INVALID_OBJECT)
        {
            PRESETS_OBJECT hPresets;
            CATEGORY_ID tPresetsCategoryId =
                CATEGORY_INVALID_ID;
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // Get the presets service handle
            hPresets = DECODER_hPresets( hDecoder );

            // If it is valid, get the category id
            if (hPresets != PRESETS_INVALID_OBJECT)
            {
                // Get the presets category id
                tPresetsCategoryId =
                    PRESETS.tCategoryId( hPresets );
            }

            // Determine first channel in the category
            psObj->sIterationCtrl.hCurrentEntry = OSAL.hLinkedListFirst(
                psObj->hChannelList, NULL );

            if ( psObj->sIterationCtrl.hCurrentEntry
                     == OSAL_INVALID_LINKED_LIST_ENTRY )
            {
                UN32 un32Items = 0;

                // Determine if the search failed
                // because this is an empty category
                OSAL.eLinkedListItems(
                    psObj->hChannelList, &un32Items );

                // Unlock object
                SMSO_vUnlock((SMS_OBJECT)hDecoder);

                if (un32Items == 0)
                {
                    // Search failed because there
                    // are no channels
                    return SMSAPI_RETURN_CODE_NO_OBJECTS;
                }
                else
                {
                    // Search failed for some other reason
                    return SMSAPI_RETURN_CODE_ERROR;
                }
            }

            while ( n16IteratorResult != 0 )
            {
                UN32 un32Items = 0;

                // Initialize iteration attributes for
                // this loop iteration
                psObj->sIterationCtrl.bChannelRemoved = FALSE;
                psObj->sIterationCtrl.un8ChannelsAddedBefore = 0;

                // Extract channel handle
                hChannel = (CHANNEL_OBJECT)OSAL.pvLinkedListThis(
                    psObj->sIterationCtrl.hCurrentEntry);

                // We cannot be sure every channel which belongs to a
                // CATEGORY is up to date. So we make sure by calling this
                // if it is already updated then this doesn't do much.
                CCACHE_bUpdateChannel(hChannel);

                // Is presets active?
                if (hPresets != PRESETS_INVALID_OBJECT)
                {
                    // Default to 0
                    N16 n16OccurrenceIndex = 0;

                    // If we are iterating the presets category
                    if (psObj->tId == tPresetsCategoryId)
                    {
                        // Determine which occurrence of this
                        // channel in the category this is
                        n16OccurrenceIndex =
                            CATEGORY_n16ChannelOccurrence (
                                (CATEGORY_OBJECT)psObj, hChannel, tIndex );
                    }

                    if (n16OccurrenceIndex >= 0)
                    {
                        // Update the channel's current preset handle
                        PRESETS_vUpdateHdl(
                            hPresets,
                            hChannel,
                            n16OccurrenceIndex );
                    }
                }

                // Call the iteration function
                n16IteratorResult = n16IterateFxn(
                    hCategory,
                    tIndex,
                    hChannel,
                    pvIterateArg
                        );

                // If the preset service is active and
                // we're iterating the presets category...
                if ((hPresets != PRESETS_INVALID_OBJECT) &&
                    (psObj->tId == tPresetsCategoryId))
                {
                    // Reset the preset handle
                    // to index 0
                    PRESETS_vUpdateHdl(
                        hPresets,
                        hChannel,
                        0 );
                }

                // Get the size of the category after the
                // callback has done its thing
                OSAL.eLinkedListItems(
                    psObj->hChannelList, &un32Items);
                n16CategorySize = (N16)un32Items;

                // The iterator is finished if the callback
                // indicates it is done or if the category is
                // now empty
                if (( n16IteratorResult == 0 ) || (n16CategorySize == 0))
                {
                    // Clear iteration control attributes
                    psObj->sIterationCtrl.hCurrentEntry
                        = OSAL_INVALID_LINKED_LIST_ENTRY;
                    psObj->sIterationCtrl.bChannelRemoved = FALSE;
                    psObj->sIterationCtrl.un8ChannelsAddedBefore = 0;

                    // Unlock object
                    SMSO_vUnlock((SMS_OBJECT)hDecoder);

                    return SMSAPI_RETURN_CODE_SUCCESS;
                }

                if (n16IteratorResult < 0)
                {
                    n8Direction = -1;
                }
                else // n16IteratorResult > 0
                {
                    n8Direction = 1;
                }

                // Adjust the index based upon what occurred within
                // the callback

                // Was the current channel removed?
                if (psObj->sIterationCtrl.bChannelRemoved == TRUE)
                {
                    // Yep, backtrack the index
                    tIndex += -n8Direction;

                    if (n8Direction == -1)
                    {
                        // When a channel is removed, the current
                        // entry is always set to the previous
                        // entry.  This is useful when we are
                        // iterating "forward", but it isn't
                        // when iterating "backward".  So,
                        // if we're going backward, we need to
                        // set the current entry to the next
                        // to ensure we don't skip an entry
                        psObj->sIterationCtrl.hCurrentEntry =
                            OSAL.hLinkedListNext(
                                psObj->sIterationCtrl.hCurrentEntry,
                                NULL );
                    }
                }

                // The index has increased by the number
                // of channels which have been added before
                // the current entry
                tIndex += psObj->sIterationCtrl.un8ChannelsAddedBefore;

                //------------------------------------------------------------------
                // Only cut out the excess operations if the iterator is not finished
                //------------------------------------------------------------------

                // The returned sign of the modulus operation is MACHINE
                // DEPENDENT so we can't just assume that the remainder of a
                // negative modulus operation is going to be negative

                // Get the absolute value of the return
                iNumIterations = abs( n16IteratorResult );

                // Cap the number of iterations at the size of the category.
                iNumIterations =
                    (iNumIterations % n16CategorySize);

                // Get to the correct position as requested by the iterations return
                for ( n16CategoryIndex = 0;
                      n16CategoryIndex < iNumIterations;
                      n16CategoryIndex++ )
                {

                    // Iterate in the reverse direction, (ABS)n16IterateReturn times
                    if ( n16IteratorResult < 0 )
                    {
                        // Try the previous one
                        psObj->sIterationCtrl.hCurrentEntry =
                                OSAL.hLinkedListPrev(
                                        psObj->sIterationCtrl.hCurrentEntry,
                                        (void *)((void *)&hChannel));

                        // If we are at the start of the list
                        if ( tIndex == 0 )
                        {
                            // Set our current index to the end of the list
                            tIndex = ( n16CategorySize - 1 );
                        }
                        else
                        {
                            // Decrement the index
                            tIndex--;
                        }
                    }
                    else // n16IteratorResult > 0
                    {
                        // Try the next one
                        psObj->sIterationCtrl.hCurrentEntry =
                                OSAL.hLinkedListNext(
                                        psObj->sIterationCtrl.hCurrentEntry,
                                        (void *)((void *)&hChannel));

                        // if we are now at the end of the category, roll over
                        if ( tIndex == ( n16CategorySize - 1 ) )
                        {
                            tIndex = 0;
                        }
                        else
                        {
                            // Increment the index
                            tIndex++;
                        }
                    } // if
                } // for
            } // while ( n16IterateReturn != 0 );

            // Clear iteration control attributes
            psObj->sIterationCtrl.hCurrentEntry
                = OSAL_INVALID_LINKED_LIST_ENTRY;
            psObj->sIterationCtrl.bChannelRemoved = FALSE;
            psObj->sIterationCtrl.un8ChannelsAddedBefore = 0;

        } // hCategory != CATEGORY_INVALID_OBJECT

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

    } // if(bLocked == TRUE)

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eInsertNewChannel
*
*       This API is used to insert a channel within an existing user or virtual
*       category. When this API is invoked, the module services will insert a
*       channel at the end of the category.
*
*       Inputs:
*           hDecoder        A handle to the decoder object in which to insert
*                           the specified channel.
*           hCategory       A handle to the category object in which to insert
*                           the specified channel.
*           tChannelId      The channel id to insert CHANNEL_INVALID_ID is a
*                           valid channel id to use.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInsertNewChannel(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CHANNEL_ID tChannelId
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bLocked;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    do
    {
        // Verify and lock DECODER object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == FALSE)
        {
            break;
        }

        // Access the decoder to get the hCCache handle
        hCCache = DECODER_hCCache( hDecoder );
        if ( hCCache == CCACHE_INVALID_OBJECT )
        {
           eReturnCode = SMSAPI_RETURN_CODE_ERROR;
           break;
        }

        // Do not allow inserting empty(placeholder) channels into
        // category if user did not request it directly.
        // Placeholder can be created only for CHANNEL_INVALID_ID
        if (tChannelId != CHANNEL_INVALID_ID)
        {
            // From the channel id provided, determine the channel object handle.
            hChannel = CCACHE_hChannelFromIds(
                hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);

            if(hChannel == CHANNEL_INVALID_OBJECT)
            {
                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                break;
            }
        }

        // Adding valid hChannel object, or CHANNEL_INVALID_OBJECT if
        // tCategoryId was equal to CHANNEL_INVALID_ID
        eReturnCode = CATEGORY_eInsertNewChannel(hDecoder, tCategoryId, hChannel);

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eInsertAfterChannel
*
*       This API is used to insert a channel within an existing user or virtual
*       category. It must only be called from within a category object iterator
*       function. When this API is invoked, the module services will insert a
*       channel after the current channel being iterated (hCurrentChannel and
*       tCurrentIndex in the CATEGORY_ITERATION_HANDLER).
*
*       Inputs:
*           hDecoder        A handle to the decoder object in which to insert
*                           the specified channel.
*           hCategory       A handle to the category object in which to insert
*                           the specified channel.
*           tChannelId      The channel id to insert the current channel with,
*                           CHANNEL_INVALID_ID is a valid channel id to use.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInsertAfterChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT hCCache;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Access the decoder to get the hCCache handle
    hCCache = DECODER_hCCache( psObj->hDecoder );

    if ( hCCache == CCACHE_INVALID_OBJECT )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Do not allow inserting empty(placeholder) channels into
    // category if user did not request it directly.
    // Placeholder can be created only for CHANNEL_INVALID_ID
    if (tChannelId != CHANNEL_INVALID_ID)
    {
        // From the channel id provided, determine the channel object handle.
        hChannel = CCACHE_hChannelFromIds(
            hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);

        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }
    }

    // Adding valid hChannel object, or CHANNEL_INVALID_OBJECT if
    // tCategoryId was equal to CHANNEL_INVALID_ID
    eReturnCode = eInsertAfterChannelLocal(psObj, hChannel);

    return eReturnCode;
}

/*****************************************************************************
*
*       eInsertBeforeChannel
*
*       This API is used to insert a channel within an existing user or virtual
*       category. It must only be called from within a category object iterator
*       function. When this API is invoked, the module services will insert a
*       channel before the current channel being iterated (hCurrentChannel and
*       tCurrentIndex in the CATEGORY_ITERATION_HANDLER).
*
*       Inputs:
*           hDecoder        A handle to the decoder object in which to insert
*                           the specified channel.
*           hCategory       A handle to the category object in which to insert
*                           the specified channel.
*           tChannelId      The channel id to insert the current channel with,
*                           CHANNEL_INVALID_ID is a valid channel id to use.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInsertBeforeChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT hCCache;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Access the decoder to get the hCCache handle
    hCCache = DECODER_hCCache( psObj->hDecoder );

    if ( hCCache == CCACHE_INVALID_OBJECT )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Do not allow inserting empty(placeholder) channels into
    // category if user did not request it directly.
    // Placeholder can be created only for CHANNEL_INVALID_ID
    if (tChannelId != CHANNEL_INVALID_ID)
    {
        // From the channel id provided, determine the channel object handle.
        hChannel = CCACHE_hChannelFromIds(
            hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);

        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            return SMSAPI_RETURN_CODE_ERROR;
        }
    }

    // Adding valid hChannel object, or CHANNEL_INVALID_OBJECT if
    // tCategoryId was equal to CHANNEL_INVALID_ID
    eReturnCode = eInsertBeforeChannelLocal(psObj, hChannel);

    return eReturnCode;
}

/*****************************************************************************
*
*       eReplaceChannel
*
*       This API is used to replace a channel within an existing user or
*       virtual category. It must only be called from within a category object
*       iterator function. When this API is invoked, the module services will
*       replace a channel with respect to the current channel being iterated
*       (hCurrentChannel and tCurrentIndex in the CATEGORY_ITERATION_HANDLER).
*
*       Inputs:
*           hDecoder        A handle to the decoder object in which to insert
*                           the specified channel.
*           hCategory       A handle to the category object in which to insert
*                           the specified channel.
*           tChannelId      The channel id to replace the current channel with,
*                           CHANNEL_INVALID_ID is a valid channel id to use.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eReplaceChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    CHANNEL_OBJECT hChannel;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CCACHE_OBJECT hCCache;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Access the decoder to get the hCCache handle
    hCCache = DECODER_hCCache( psObj->hDecoder );
    if ( hCCache == CCACHE_INVALID_OBJECT )
    {
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }
    else
    {
        // Only search the cache if we have been provided a valid channel id
        if ( tChannelId != CHANNEL_INVALID_ID )
        {
            // From the channel id provided, determine the channel object handle.
            hChannel = CCACHE_hChannelFromIds(
                hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
            if(hChannel == CHANNEL_INVALID_OBJECT)
            {
                // Error! No Channel!
                return SMSAPI_RETURN_CODE_ERROR;
            }
        }
        else
        {
            // We were provided an invalid id because of a requested "reset"
            // of the info stored in the category
            hChannel = CHANNEL_INVALID_OBJECT;
        }

        eReturnCode = eReplaceChannelLocal(psObj, hChannel);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eRemoveChannel
*
*       This API is used to remove a channel from an existing user or virtual
*       category. It must only be called from within a category object iterator
*       function. When this API is invoked, the module services will remove the
*       current channel being iterated (hCurrentChannel and tCurrentIndex in the
*       CATEGORY_ITERATION_HANDLER).
*
*       Inputs:
*           hCategory       A handle to the category object for which the
*                           current channel is being removed.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveChannel(
    CATEGORY_OBJECT hCategory
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    CHANNEL_OBJECT hChannelBeingRemoved = CHANNEL_INVALID_OBJECT;
    OSAL_LINKED_LIST_ENTRY hPreviousEntry;
    CATEGORY_ITERATION_CTRL_STRUCT sCurrentIterationState;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == FALSE)
    {
        // Error!
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    // Verify this is not a broadcast category
    if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    // Check to make sure we have a valid current entry
    if ( psObj->sIterationCtrl.hCurrentEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        // Error!
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Save the previous entry
    hPreviousEntry = OSAL.hLinkedListPrev(
        psObj->sIterationCtrl.hCurrentEntry, NULL);
    if (hPreviousEntry == psObj->sIterationCtrl.hCurrentEntry)
    {
        hPreviousEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Save the current iteration control structure as well
    sCurrentIterationState = psObj->sIterationCtrl;

    // Grab the channel object from the current entry
    // (this is the channel we're removing)
    hChannelBeingRemoved = (CHANNEL_OBJECT)
        OSAL.pvLinkedListThis(psObj->sIterationCtrl.hCurrentEntry);

    // Remove object from the linked list
    eReturnCode = OSAL.eLinkedListRemove(
        psObj->sIterationCtrl.hCurrentEntry);

    if(eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CATEGORY_OBJECT_NAME": Could not remove entry from LL: %s.",
            OSAL.pacGetReturnCodeName(eReturnCode));

        return SMSAPI_RETURN_CODE_ERROR;
    }

    if (hChannelBeingRemoved != CHANNEL_INVALID_OBJECT)
    {
        // need to see if this was the only entry of this service id in
        // the category. if so, we need to remove it from the channel obj.
        // otherwise we release it
        SERVICE_ID tId;

        // Get the id of the removed channel
        tId = CHANNEL.tServiceId(hChannelBeingRemoved);
        if (tId != SERVICE_INVALID_ID)
        {
            N16 n16Index;

            // Find any other occurrences of this channel
            // in the category. Note: This function utilizes
            // the iterator, which will adjust the
            // psObj->sIterationCtrl.hCurrentEntry handle so we'll have
            // to correct that before we return from this function
            n16Index = CATEGORY_n16GetIndexByServiceId(
                                hCategory,
                                tId);

            if (n16Index < 0)
            {
                // Channel not found -- tell it that
                // it is no longer a part of this category
                CHANNEL_vRemoveCategory(
                    hChannelBeingRemoved, hCategory, TRUE);
            }
        }
    }

    // Now, restore the iteration control structure
    psObj->sIterationCtrl = sCurrentIterationState;

    // And restore the current entry as
    // the previous entry
    psObj->sIterationCtrl.hCurrentEntry = hPreviousEntry;

    // Indicate a channel has been removed
    psObj->sIterationCtrl.bChannelRemoved = TRUE;

    // Notify all registered parties (but don't let them
    // mess up the state of the iterator )
    vNotifyRegisteredAndProtectIterator(
        psObj, CATEGORY_OBJECT_EVENT_CONTENTS );

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*       eSort
*
*       This API is used to sort the specified category's channels according to
*       the provided sort function.  Sorting is an atomic operation (meaning all
*       channels will be sorted accordingly upon return of this API). This API
*       will actually rearrange any existing order the channels are currently
*       in.  Thus any reference to this category will be affected by the result
*       of this sort one it is complete.
*
*       Inputs:
*           hCategory       A handle to the category object to sort
*           n16SortFxn      Sort function that is used when sorting the
*                           category.  If this API is not provided, the sorting
*                           will not occur.  Please refer to the Definitions
*                           section for more information on the n16SortFxn.
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSort(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CATEGORY_SORT_HANDLER n16SortFxn
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;
    // Lock object for exclusive access
    BOOLEAN bLocked = FALSE;

    do
    {
        CATEGORY_OBJECT_STRUCT *psObj;

        // Verify the inputs
        if (n16SortFxn == NULL)
        {
            // Error!
            break;
        }

        // Verify and lock DECODER object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == FALSE)
        {
            break;
        }

        psObj = (CATEGORY_OBJECT_STRUCT *)hGetHandle(hDecoder, tCategoryId);
        if(psObj == NULL)
        {
            break;
        }

        // Callers may not sort a virtual category - the state
        // of virtual categories must be exclusively maintained
        // by SMS
        if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }

        eReturnCode = eSortDirect(psObj, n16SortFxn);

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       eNotifyOnChange
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eNotifyOnChange (
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        hCCache = DECODER_hCCache(hDecoder);

        // Now that we have the CCACHE handle, register this function
        // for the application
        eErrorCode = CCACHE_eHandleAppCategoryNotifyOnChange(
            hCCache, vNotifierCallback, pvNotifierCallbackArg);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if(eErrorCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bSuccess;

            // Tell the decoder to start processing the initial dump of notifications
            bSuccess = DECODER_bStartAllChanCatNotifications(hDecoder, FALSE);
            if(bSuccess == FALSE)
            {
                eErrorCode  = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    }

    return eErrorCode;
}

/*****************************************************************************
*
*       eIterateAll
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateAll (
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT_ITERATOR bIterator,
    CATEGORY_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Verify inputs
    if(bIterator == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    if(bInclude == NULL)
    {
        // Accept all
        bInclude = bAlwaysInclude;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        hCCache = DECODER_hCCache(hDecoder);
        if(hCCache != CCACHE_INVALID_OBJECT)
        {
            CATEGORY_ITERATOR_STRUCT sIterator;
            BOOLEAN bOk;

            // Populate local iterator
            sIterator.bIterator = bIterator;
            sIterator.bInclude = bInclude;
            sIterator.pvArg = pvArg;

            bOk = CCACHE_bIterateCategories(
                hCCache, bIteratorShim, &sIterator);
            if(bOk == TRUE)
            {
                eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       CATEGORY_bUpdateId
*
*****************************************************************************/
BOOLEAN CATEGORY_bUpdateId(
    CATEGORY_OBJECT hCategory,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bUpdated = FALSE;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        if(psObj->tId != tCategoryId)
        {
            psObj->tId = tCategoryId;
            bUpdated = TRUE;
        }
    }

    return bUpdated;
}

/*****************************************************************************
*
*       CATEGORY_tCreateVirtualCategory
*
*       This function is used to create a new virtual category (which contains
*       no valid channels).
*
*       Inputs:
*           hDecoder                Decoder to attach this category to.  This
*                                   value must not be DECODER_INVALID_OBJECT or
*                                   the creation will fail.
*
*           pacLongName             NULL terminated string containing the long
*                                   name of the category.  If one is not
*                                   provided, the short name will be used.
*
*           pacShortName            NULL terminated string containing the short
*                                   name of the category.  If one is not
*                                   provided, the long name will be used.
*
*           tInitialNumChannels     If 0, the list will be created with no
*                                   items.  If tInitialNumChannels >0, the
*                                   category will be created with
*                                   tInitialNumChannels in it all set to
*                                   CHANNEL_INVALID_ID
*
*           bUniqueItems            a BOOLEAN value, defining whenever the
*                                   same channel can be present in the category
*                                   multiple times (default behavior) or it can not.
*
*           bReplace                if bUniqueItems parameter is set to TRUE,
*                                   this boolean value defines whenever duplicate
*                                   channel shall replace already existing one (TRUE),
*                                   or it shall be rejected keeping the original (FALSE).
*
*       Outputs:
*           CATEGORY_ID     Newly created category's ID
*
*****************************************************************************/
CATEGORY_ID  CATEGORY_tCreateVirtualCategory(
    DECODER_OBJECT hDecoder,
    const char *pacLongName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        )
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
    BOOLEAN bLocked;

    // Verify and Lock the DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CCACHE_OBJECT hCCache;

        // Get the ccache attached to this decoder
        hCCache = DECODER_hCCache( hDecoder );

        // If we failed, we cannot create the category so just return
        if ( hCCache != CCACHE_INVALID_OBJECT )
        {
            // Get first available CATEGORY_ID from within the category lists
            // that falls into the range of user categories
            tCategoryId =
                CCACHE_tAddCategory(
                            hCCache,
                            CATEGORY_TYPE_VIRTUAL,
                            pacLongName,
                            pacShortName, // Reuse short name for medium
                            pacShortName,
                            tInitialNumChannels,
                            bUniqueItems,
                            bReplace);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
*
*       CATEGORY_eReplaceChannel
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eReplaceChannel (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner == FALSE)
    {
        // Error!
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        eReturnCode = eReplaceChannelLocal(psObj, hChannel);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       CATEGORY_hCreateCategory
*
*       This API is used to create a category of any type.  It is only called
*       from the CCACHE object when it has determined which Id to use
*       for this particular category.
*
*       NOTE: Nobody except the CCACHE object should call this function!
*
*       Inputs:
*           hDecoder        Handle to the decoder to create the category in
*           eCategoryType   The type of category to create
*           tId             Category Id to use in the creation
*           pacShortName    Short name for the category
*           pacLongName     Long name for the category
*           tInitialNumChannels     Initial number of channels to create in the
*                                   category
*
*       Outputs:
*           CATEGORY_OBJECT     Handle to the created category object.
*
*****************************************************************************/
CATEGORY_OBJECT CATEGORY_hCreateCategory (
    DECODER_OBJECT hDecoder,
    SMS_OBJECT hParent,
    CATEGORY_TYPE_ENUM eCategoryType,
    CATEGORY_ID tId,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        )
{
    CATEGORY_OBJECT_STRUCT *psObj;

    // Can't create a category with this Id
    if (tId == GsRadio.sCategory.un16BroadcastIdNotAssigned)
    {
        return CATEGORY_INVALID_OBJECT;
    }

    // Get the category object created
    psObj = psCreateCategory (
                eCategoryType,
                hDecoder,
                hParent,
                tId,
                pacLongName,
                pacMediumName,
                pacShortName,
                bUniqueItems,
                bReplace);

    // Verify that worked out okay
    if (psObj == NULL)
    {
        return CATEGORY_INVALID_OBJECT;
    }

    // If the category was initialized with a request for any number of
    // channel objects start adding them
    if ( tInitialNumChannels > 0 )
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        CATEGORY_CHANNEL_INDEX tChannelCntr;

        // Take tInitialNumChannels and add that many channels to the
        // channel list
        for ( tChannelCntr = 0;
              tChannelCntr < tInitialNumChannels;
              tChannelCntr++ )
        {
            // Note that having a CHANNEL_OBJECT or CHANNEL_INVALID_OBJECT
            // is perfectly valid for a derived category.  It is also
            // perfectly valid to have more than 1 channel in a user category
            // to contain the same CHANNEL_ID

            // Add it to the list
            eReturnCode = OSAL.eLinkedListAdd(
                    psObj->hChannelList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    CHANNEL_INVALID_OBJECT );

            // Since we are explicitly writing a channel handle as
            // CHANNEL_INVALID_OBJECT we should not make the call to
            // CHANNEL_bAddCategory to connect a specific channel object
            // with this category

            // Failed addition to the linked list
            if ( eReturnCode != OSAL_SUCCESS )
            {
                // Remove all entries in the linked list
                OSAL.eLinkedListRemoveAll(
                    psObj->hChannelList, (OSAL_LL_RELEASE_HANDLER)NULL);

                // Delete linked list
                eReturnCode = OSAL.eLinkedListDelete( psObj->hChannelList);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Reset the handle
                    psObj->hChannelList = OSAL_INVALID_OBJECT_HDL;
                }

                // Destroy the category
                CATEGORY_vDestroy(
                    (CATEGORY_OBJECT)psObj );

                // Return error!
                return CATEGORY_INVALID_OBJECT;
            }
        }
    }

    return (CATEGORY_OBJECT)psObj;
}

/*****************************************************************************
*
*       CATEGORY_vDestroy
*
*       This API is used to destroy the specified category.
*
*       Inputs:
*           hCategory       Handle of the existing category to destroy
*
*       Outputs:
*           none
*
*****************************************************************************/
void CATEGORY_vDestroy (
    CATEGORY_OBJECT hCategory
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Verify inputs
    if(hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Block this category notifications
    bBlockNotifications(hCategory);

    // Destroy channel list if one exists
    if(psObj->hChannelList != OSAL_INVALID_OBJECT_HDL)
    {
        // iterate the list of channels and remove the category
        // from the channels
        OSAL.eLinkedListIterate (
            psObj->hChannelList,
            bRemoveCategoryFromChannel,
            (void *)hCategory);

        // Remove all entries in the linked list
        OSAL.eLinkedListRemoveAll(psObj->hChannelList, NULL);

        // Delete linked list itself
        eReturnCode = OSAL.eLinkedListDelete(
            psObj->hChannelList);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hChannelList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // This category is about to be destroyed. Inform all
    // interested parties.
    SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_REMOVED);

    // Release category notifications
    vReleaseNotifications(hCategory);

    // Destroy the SMS Update object
    SMSU_vDestroy(&psObj->sEvent);

    // Free objects
    if(psObj->hShortName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hShortName);
        psObj->hShortName = STRING_INVALID_OBJECT;
    }
    if(psObj->hMediumName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hMediumName);
        psObj->hMediumName = STRING_INVALID_OBJECT;
    }
    if(psObj->hLongName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hLongName);
        psObj->hLongName = STRING_INVALID_OBJECT;
    }

    // Release CATEGORY ART
    psObj->hChannelArtService = CHANNEL_ART_SERVICE_INVALID_OBJECT;
    psObj->hArt = CHANNEL_ART_INVALID_OBJECT;

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hCategory);

    return;
}

/*****************************************************************************
*
*       CATEGORY_vUpdateArtService
*
*       This API is used to update the channel art service handle for
*       a particular category
*
*****************************************************************************/
void CATEGORY_vUpdateArtService (
    CATEGORY_OBJECT hCategory,
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Verify inputs
    if(hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Store the channel art service handle if it changed
    if(psObj->hChannelArtService != hChannelArtService)
    {
        // It has changed
        psObj->hChannelArtService = hChannelArtService;

        // Get the art for this category now
        // that the service is up and running
        CATEGORY_vUpdateArt( hCategory, TRUE );
    }

    return;
}

/*****************************************************************************
*
*       CATEGORY_vUpdateArt
*
*       This API is used to update the channel art handle for
*       a particular category.  If the update is successful, the
*       CATEGORY will pick a CHANNEL object to use in order to indicate
*       to the Application an update occurred.
*
*       Inputs:
*           hCategory       Handle of the existing category to destroy
*
*       Outputs:
*           None
*
*****************************************************************************/
void CATEGORY_vUpdateArt (
    CATEGORY_OBJECT hCategory,
    BOOLEAN bRefreshArtHandle
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    BOOLEAN bNotify = TRUE;

    // Verify input
    if (hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error!
        return;
    }

    // Check the state of the category object
    if (psObj->tId == CATEGORY_INVALID_ID)
    {
        // The category is not in a state
        // in which to acquire category art.
        // Ensure the art handle has been cleared,
        // and get outta here
        psObj->hArt = CHANNEL_ART_INVALID_OBJECT;

        return;
    }

    // Only grab a new art handle if
    // we were told to do so
    if (bRefreshArtHandle == TRUE)
    {
        CHANNEL_ART_OBJECT hOldArt = psObj->hArt;

        // Update the art object handle for this category
        psObj->hArt =
            CHANNEL_ART_MGR_hGetArtForCategoryId(
                psObj->hChannelArtService,
                psObj->tId );

        // Check to see that the handle actually changed
        if (psObj->hArt == hOldArt)
        {
            bNotify = FALSE;
        }
    }

    if (bNotify == TRUE)
    {
        // Update our event mask and notify our category event listeners
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_ART);

        // Notify callback if registered and change occurred
        SMSU_bNotify(&psObj->sEvent);

        // At this point all CHANNELs in this category need to be
        // notified that their category art has changed. So we
        // need to iterate the entire list of channels belonging
        // to this category.
        OSAL.eLinkedListIterate(
            psObj->hChannelList,
            bNotifyCategoryUpdated,
            NULL);
    }

    return;
}

/*****************************************************************************
*
*       CATEGORY_bAddChannel
*
*       This API is used to add a channel to the specified broadcast category.
*
*       NOTE: This should only be called by the CCache!
*
*       Inputs:
*           hCategory       Handle of the existing category to destroy
*
*       Outputs:
*           none
*
*****************************************************************************/
BOOLEAN CATEGORY_bAddChannel (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel
        )
{
    BOOLEAN bOwner, bOk = FALSE;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify inputs
    if( (hChannel == CHANNEL_INVALID_OBJECT) ||
        (hCategory == CATEGORY_INVALID_OBJECT)
            )
    {
        // Error!
        return FALSE;
    }

    // Verify ownership (the cache owns both
    // the channel & category, so just validate
    // ownership of the category)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner != TRUE)
    {
        return FALSE;
    }

    // Search category list to see if this channel is already a member
    eReturnCode =
        OSAL.eLinkedListSearch(
            psObj->hChannelList,
            &hEntry,
            (void *)hChannel );

    if(eReturnCode == OSAL_SUCCESS) // found
    {
        // Already exists
        bOk = TRUE;
    }
    else // not found
    {
        // Initialize entry handle
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add it to the list
        eReturnCode =
            OSAL.eLinkedListAdd(
                psObj->hChannelList,
                &hEntry,
                hChannel
                    );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            return FALSE;
        }

        // We are going to "use this channel" so we need to make
        // sure we register for notifications, we want to register for service
        // id and channel id changes so that we can see when a remap occurred
        // and resort broadcast category
        bOk = CHANNEL_bRegisterNotification(
            hChannel,
            (CHANNEL_OBJECT_EVENT_SERVICE_ID | CHANNEL_OBJECT_EVENT_CHANNEL_ID),
            vChannelRemappedEventCallback, hCategory, TRUE);

        if (bOk == FALSE)
        {
            // Failed to register notification. Try to rollback.
            OSAL.eLinkedListRemove(hEntry);
        }

        // Update our event mask and notify our category event listeners
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

        // Notify callback if registered and change occurred
        SMSU_bNotify(&psObj->sEvent);
    }

    return bOk;
}

/*****************************************************************************
*
*   CATEGORY_vRemoveChannel
*
*****************************************************************************/
void CATEGORY_vRemoveChannel (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel,
    BOOLEAN bSuppressCatEvents
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify inputs
    if( (hChannel == CHANNEL_INVALID_OBJECT) ||
        (hCategory == CATEGORY_INVALID_OBJECT)
            )
    {
        // Error!
        return;
    }

    // Verify ownership (the cache owns both
    // the channel & category, so just validate
    // ownership of the category)
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner != TRUE)
    {
        return;
    }

    // Search category list to see if this category is already a member
    // don't do a sorted search in case the category became unsorted
    eReturnCode =
        OSAL.eLinkedListLinearSearch(
            psObj->hChannelList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16ChannelCompare,
            (void *)hChannel );
    if(eReturnCode == OSAL_SUCCESS) // found
    {
        // We are no longer going to "use this channel" so we need to make
        // sure we unregister for notifications.
        CHANNEL_vUnregisterNotification(hChannel,
            vChannelRemappedEventCallback, hCategory);

        // Remove entry
        OSAL.eLinkedListRemove(hEntry);
    }

    // are we suppressing events?
    // FYI: we would want to suppress events when removing a channel during iteration
    // only vNotifyRegisteredAndProtectIterator would be used to send notifications
    if (bSuppressCatEvents == FALSE)
    {
        // Update our event mask and notify our category event listeners
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

        // Notify callback if registered and change occurred
        SMSU_bNotify(&psObj->sEvent);
    }

    return;
}

/*****************************************************************************
*
*   CATEGORY_bIdInRange
*
*   This function is used to test if the given CATEGORY_ID is within
*   the range for the given CATEGORY_TYPE_ENUM value.
*
*   Inputs:
*       eCategoryType - The type of category for which to test the Id
*       tCategoryId - The Id to test
*
*   Outputs:
*       TRUE if the Id is in range; FALSE otherwise
*
*****************************************************************************/
BOOLEAN CATEGORY_bIdInRange (
    CATEGORY_TYPE_ENUM eCategoryType,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bInRange = FALSE;
    CATEGORY_ID tRangeMin;
    CATEGORY_ID tRangeMax;

    if (tCategoryId == CATEGORY_INVALID_ID)
    {
        // The invalid Id is not within
        // any category type id range
        return FALSE;
    }

    switch (eCategoryType)
    {
        case CATEGORY_TYPE_BROADCAST:
        {

            tRangeMin = GsRadio.sCategory.un16BroadcastIdMin;
            tRangeMax = GsRadio.sCategory.un16BroadcastIdMax;
        }
        break;

        // These category types share an
        // Id range
        case CATEGORY_TYPE_USER:
        case CATEGORY_TYPE_VIRTUAL:
        {
            tRangeMin = GsRadio.sCategory.un16DerivedMin;
            tRangeMax = GsRadio.sCategory.un16DerivedMax;
        }
        break;

        case CATEGORY_TYPE_INVALID:
        default:
        {
            // Invalid input -- clear values
            tRangeMin = CATEGORY_INVALID_ID;
            tRangeMax = CATEGORY_INVALID_ID;
        }
        break;
    }

    // Check to see if the provided Id
    // in the the appropriate range
    if ( (tCategoryId >= tRangeMin) &&
         (tCategoryId <= tRangeMax) )
    {
        bInRange = TRUE;
    }

    return bInRange;
}

/*****************************************************************************
*
*   CATEGORY_bCategoryIdRange
*
*   This function is used to get the range of Ids which may be assigned
*   to a category of a certain type.
*
*   Inputs:
*       eCategoryType - The type of category for which to get the range
*       *ptStartCategoryId - The start, or "bottom", of the Id range
*           for categories of this type
*       *ptEndCategoryId - The end, or "top", of the Id range for
*           categories of this type
*
*   Outputs:
*       TRUE on success; FALSE on error
*
*****************************************************************************/
BOOLEAN CATEGORY_bCategoryIdRange (
    CATEGORY_TYPE_ENUM eCategoryType,
    CATEGORY_ID *ptStartCategoryId,
    CATEGORY_ID *ptEndCategoryId
        )
{
    BOOLEAN bOk = TRUE;
    CATEGORY_ID tStartId = CATEGORY_INVALID_ID;
    CATEGORY_ID tEndId = CATEGORY_INVALID_ID;

    switch (eCategoryType)
    {
        case CATEGORY_TYPE_BROADCAST:
        {
            tStartId = GsRadio.sCategory.un16BroadcastIdMin;
            tEndId = GsRadio.sCategory.un16BroadcastIdMax;
        }
        break;

        case CATEGORY_TYPE_USER:
        case CATEGORY_TYPE_VIRTUAL:
        {
            tStartId = GsRadio.sCategory.un16DerivedMin;
            tEndId = GsRadio.sCategory.un16DerivedMax;
        }
        break;

        case CATEGORY_TYPE_INVALID:
        default:
        {
            // Indicate failure
            bOk = FALSE;
        }
        break;
    }

    if (ptStartCategoryId != NULL)
    {
        *ptStartCategoryId = tStartId;
    }

    if (ptEndCategoryId != NULL)
    {
        *ptEndCategoryId = tEndId;
    }

    return bOk;
}

/*****************************************************************************
*
*   CATEGORY_n16CompareCategoryIds
*
*   This function is used to compare the 2 categories.
*   This is used when searching the LL for a specific category.
*
*   Inputs:
*       *pvArg1 - pointer to one of the categories being compared
*       *pvArg2 - pointer to the other category being compared
*
*   Outputs:
*       n16Result - The comparison between the category passed in
*           and the category of the struct currently being searched.
*          0 - Objects have the same value (equal, error )
*       > 0 - Object1 is greater than (after) Object2
*       < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
N16 CATEGORY_n16CompareCategoryIds(
    void *pvArg1,
    void *pvArg2
        )
{
    N16 n16Result = N16_MIN;
    CATEGORY_OBJECT_STRUCT *psObj1 = (CATEGORY_OBJECT_STRUCT *)pvArg1,
                           *psObj2 = (CATEGORY_OBJECT_STRUCT *)pvArg2;

    // Check pointers
    if ((psObj1 != NULL) && (psObj2 != NULL))
    {
        // A category is considered equal if the category id's match.
        n16Result = psObj1->tId - psObj2->tId;
    }

    return n16Result;
}

/*****************************************************************************
*
*   CATEGORY_hGetChanHdlByOffset
*
*****************************************************************************/
CHANNEL_OBJECT CATEGORY_hGetChanHdlByOffset(
    CATEGORY_OBJECT hCategory,
    N16 n16Offset
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    CATEGORY_GET_HANDLE_BY_OFFSET_STRUCT sGetHandleIterateStruct;

    // Verify input and ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner != TRUE)
    {
        // Error!
        return CHANNEL_INVALID_OBJECT;
    }

    // Populate the iteration structure
    sGetHandleIterateStruct.hChannel = CHANNEL_INVALID_OBJECT;
    sGetHandleIterateStruct.n16OffsetCntr = n16Offset;

    // Get the channel information from the category
    eReturnCode = CATEGORY.eIterate(
                        psObj->hDecoder,
                        psObj->tId,
                        n16GetHdlByOffsetIterateFxn,
                        (void *)(size_t)&sGetHandleIterateStruct );

    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return CHANNEL_INVALID_OBJECT;
    }

    return sGetHandleIterateStruct.hChannel;
}

/*****************************************************************************
*
*   CATEGORY_n16GetIndexByChanId
*
*****************************************************************************/
N16 CATEGORY_n16GetIndexByChanId(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        )
{
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    CATEGORY_GET_INDEX_BY_ID_STRUCT sGetIndexIterateStruct;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Verify input and ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner != TRUE)
    {
        // Error!
        return N16_MIN;
    }

    // Populate the iteration structure
    sGetIndexIterateStruct.n16Index = -1;
    sGetIndexIterateStruct.tChannelId = tChannelId;
    sGetIndexIterateStruct.un32CategorySize = 0;

    // Get the category size
    eOsalReturnCode = OSAL.eLinkedListItems( psObj->hChannelList,
        &sGetIndexIterateStruct.un32CategorySize );

    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        // Error!
        return N16_MIN;
    }

    // Get the channel information into the category
    eReturnCode = CATEGORY.eIterate(
                        psObj->hDecoder,
                        psObj->tId,
                        n16GetIndexByChanIdIterateFxn,
                        (void *)(size_t)&sGetIndexIterateStruct );

    // Handle explicit failure case by
    // always returning N16_MIN (means not found)
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return N16_MIN;
    }

    return sGetIndexIterateStruct.n16Index;
}

/*****************************************************************************
*
*   CATEGORY_n16GetIndexByServiceId
*
*****************************************************************************/
N16 CATEGORY_n16GetIndexByServiceId(
    CATEGORY_OBJECT hCategory,
    SERVICE_ID tServiceId
        )
{
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    CATEGORY_GET_INDEX_BY_ID_STRUCT sGetIndexIterateStruct;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Verify input and ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if(bOwner != TRUE)
    {
        // Error!
        return N16_MIN;
    }

    // Populate the iteration structure
    sGetIndexIterateStruct.n16Index = -1;
    sGetIndexIterateStruct.tServiceId = tServiceId;
    sGetIndexIterateStruct.un32CategorySize = 0;

    // Get the category size
    eOsalReturnCode = OSAL.eLinkedListItems( psObj->hChannelList,
        &sGetIndexIterateStruct.un32CategorySize );

    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        // Error!
        return N16_MIN;
    }

    // Get the channel information into the category
    eReturnCode = CATEGORY.eIterate(
                        psObj->hDecoder,
                        psObj->tId,
                        n16GetIndexByServiceIdIterateFxn,
                        (void *)(size_t)&sGetIndexIterateStruct );

    // Handle explicit failure case by
    // always returning N16_MIN (means not found)
    if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
    {
        return N16_MIN;
    }

    return sGetIndexIterateStruct.n16Index;
}

/*****************************************************************************
*
*   CATEGORY_n16ChannelOccurrence
*
*   This function utilizes linked list operations to calculate which
*   occurrence of a particular channel Id is found in a portion of the
*   channel list
*
*****************************************************************************/
N16 CATEGORY_n16ChannelOccurrence (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel,
    N16 n16IndexLimit
        )
{
    BOOLEAN bOwner, bValid;
    N16 n16Occurrence = N16_MIN;

    // Verify channel handle
    bValid = SMSO_bValid((SMS_OBJECT)hChannel);

    // Verify category handle and ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);

    // Make sure that all worked out
    if((bOwner == TRUE) && (bValid == TRUE))
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;
        CHANNEL_ID tChannelId;
        tChannelId = CHANNEL.tChannelId( hChannel );
        if (tChannelId != CHANNEL_INVALID_ID)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            CATEGORY_CHAN_OCCURRENCE_STRUCT sOccurrence;

            sOccurrence.tChannelId = tChannelId;
            sOccurrence.n16Limit = n16IndexLimit;
            sOccurrence.n16Index = 0;
            sOccurrence.n16Occurrence = 0;

            eReturnCode = OSAL.eLinkedListIterate(
                psObj->hChannelList,
                bDetermineChanOccurrence,
                (void *)(size_t)&sOccurrence);

            if (eReturnCode == OSAL_SUCCESS)
            {
                n16Occurrence = sOccurrence.n16Occurrence;
            }
        }
    }

    return n16Occurrence;
}

/*****************************************************************************
*
*   CATEGORY_bRegisterNotification
*
*   When working with a CATEGORY object, if the caller needs to be notified
*   of when the object is going to be updated, then they should use this
*   API to register for that event.
*
*****************************************************************************/
BOOLEAN CATEGORY_bRegisterNotification (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventRequestMask,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner, bRegistered = FALSE;

    // Verify inputs
    if((hCategory == CATEGORY_INVALID_OBJECT) ||
       (tEventRequestMask == CATEGORY_OBJECT_EVENT_NONE) ||
       (vEventCallback == NULL)
       )
    {
        // Error!
        return FALSE;
    }

    // Verify we own the category
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        // Add the callback to the
        // asynchronous update configuration
        bRegistered = SMSU_bRegisterCallback(
                &psObj->sEvent,
                tEventRequestMask,
                (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
                (void *)pvEventCallbackArg,
                TRUE);
    }

    return bRegistered;
}

/*****************************************************************************
*
*   CATEGORY_bNotifyIfPending
*
*****************************************************************************/
BOOLEAN CATEGORY_bNotifyIfPending (
    CATEGORY_OBJECT hCategory,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvCallbackArg
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    BOOLEAN bSuccess;

    // Verify inputs
    if(hCategory == CATEGORY_INVALID_OBJECT)
    {
        // Error!
        return FALSE;
    }

    bSuccess = SMSU_bIsCallbackPending(&psObj->sEvent,
        (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback, pvCallbackArg);

    if (bSuccess == TRUE)
    {
        bSuccess = SMSU_bNotifyCallback(&psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback, pvCallbackArg);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   CATEGORY_vUnregisterNotification
*
* This API is used to unregister a previously registered notification.
*
*****************************************************************************/
void CATEGORY_vUnregisterNotification (
    CATEGORY_OBJECT hCategory,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bOwner;

    // Verify inputs
    if((hCategory == CATEGORY_INVALID_OBJECT) ||
       (vEventCallback == NULL)
       )
    {
        // Error!
        return;
    }

    // Verify we own the category
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner == TRUE)
    {
        CATEGORY_OBJECT_STRUCT *psObj =
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        // Remove the callback from the
        // asynchronous update configuration
        SMSU_bUnregisterCallback(
            &psObj->sEvent,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvEventCallbackArg);
    }
    return;
}

/*****************************************************************************
*
*       CATEGORY_eInsertAfterChannel
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eInsertAfterChannel (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel
        )
{
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner == FALSE)
    {
        // Error!
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        eReturnCode = eInsertAfterChannelLocal(psObj, hChannel);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       CATEGORY_eInsertBeforeChannel
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eInsertBeforeChannel (
    CATEGORY_OBJECT hCategory,
    CHANNEL_OBJECT hChannel
        )
{
   CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)hCategory;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOwner;

    // Verify caller is the owner of the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner == FALSE)
    {
        // Error!
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    else
    {
        eReturnCode = eInsertBeforeChannelLocal(psObj, hChannel);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*       CATEGORY_eInsertNewChannel
*
*       This API is used to insert a channel within an existing user or virtual
*       category. When this API is invoked, the module services will insert a
*       channel at the end of the category.
*
*       Inputs:
*           hDecoder        A handle to the decoder object in which to insert
*                           the specified channel.
*           hCategory       A handle to the category object in which to insert
*                           the specified channel.
*           hChannel        The channel obj to insert CHANNEL_INVALID_OBJECT
*                           is valid
*
*       Outputs:
*           SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or
*                                       SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eInsertNewChannel (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CHANNEL_OBJECT hChannel
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_INVALID_INPUT;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    BOOLEAN bLocked;

    // Verify and lock DECODER object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if(hCategory != CATEGORY_INVALID_OBJECT)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
            {
                // Unlock object
                SMSO_vUnlock((SMS_OBJECT)hDecoder);

                return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            }
            else
            {
                // This category enforces to use each channels only once,
                // Duplicate channel would be moved at the end of the list
                if (psObj->bUniqueItems == TRUE)
                {
                    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                    // We should move this channel at the end of the list
                    eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                        psObj->hChannelList,
                        &hEntry,
                        CHANNEL_n16CompareChannelIds,
                        (void *)hChannel);

                    if (eOsalReturnCode == OSAL_SUCCESS)
                    {
                        if (psObj->bReplace == TRUE)
                        {
                            // Remove this channel
                            OSAL.eLinkedListRemove(hEntry);

                            // Insert this channel in the end of list
                            OSAL.eLinkedListAdd(
                                psObj->hChannelList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                (void *)hChannel);

                            // Update the event mask
                            SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

                            // Notify registered parties
                            SMSU_bNotify(&psObj->sEvent);

                            // Relinquish ownership
                            SMSO_vUnlock((SMS_OBJECT)hDecoder);

                            return SMSAPI_RETURN_CODE_SUCCESS;
                        }
                        else
                        {
                            // Duplicate channel should be rejected
                            // Relinquish ownership
                            SMSO_vUnlock((SMS_OBJECT)hDecoder);

                            // We found this channel
                            return SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
                        }
                    }
                }

                // DO NOT check hChannel against CHANNEL_INVALID_OBJECT because
                // it is perfectly valid to use that value as a reset condition
                // for a member of the category. Add it regardless

                eOsalReturnCode =
                    OSAL.eLinkedListAdd(psObj->hChannelList,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hChannel);
                if ( eOsalReturnCode == OSAL_SUCCESS )
                {
                    if (hChannel != CHANNEL_INVALID_OBJECT)
                    {
                        // add this category to the channel
                        CHANNEL_bAddCategory (hChannel, hCategory);
                    }

                    // Inform interested parties of this update

                    // Update the event mask
                    SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

                    // Notify registered parties
                    SMSU_bNotify(&psObj->sEvent);

                    eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                }
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CATEGORY_eRename
*
* This API is used to rename a category
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eRename (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    const char *pacNewLongName,
    const char *pacNewMedName,
    const char *pacNewShortName
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;

    // Verify we own the decoder
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        CATEGORY_OBJECT hCategory;

        hCategory = hGetHandle(hDecoder, tCategoryId);
        // the category handle must be valid and the user must have provided
        // at least one non-NULL name string
        if ( (hCategory != CATEGORY_INVALID_OBJECT) &&
           ( (pacNewLongName != NULL) || 
             (pacNewMedName != NULL) || 
             (pacNewShortName != NULL) ) )
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            eReturnCode = eRenameLocal(psObj, 
                pacNewLongName, pacNewMedName, pacNewShortName );
        }
        else
        {
            // the caller gave us bad arguments
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
*
*   CATEGORY_eFill
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eFill (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CHANNEL_OBJECT *phChannel,
    size_t tArraySize
        )
{
    BOOLEAN bOwner = FALSE, bUpdated = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    CATEGORY_OBJECT_STRUCT *psObj = NULL;

    do
    {
        size_t tIndex;
        CATEGORY_OBJECT hCategory;
        OSAL_RETURN_CODE_ENUM eOsalCode = OSAL_SUCCESS;
        UN32 un32ListSize = 0;

        if ((tCategoryId == CATEGORY_INVALID_ID) ||
            ((tArraySize != 0) && (phChannel == NULL)))
        {
            // Nothing to populate
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Verify we own the decoder
        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOwner == FALSE)
        {
            break;
        }

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if (hCategory == CATEGORY_INVALID_OBJECT)
        {
            break;
        }

        // De-reference object
        psObj = (CATEGORY_OBJECT_STRUCT *)hCategory;

        if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
        {
            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        eOsalCode = OSAL.eLinkedListItems(psObj->hChannelList, &un32ListSize);
        if (eOsalCode != OSAL_SUCCESS)
        {
            // The list is totally broken! Aborting.
            break;
        }

        if (un32ListSize > 0)
        {
            // Remove all channels from category
            eOsalCode = OSAL.eLinkedListRemoveAll(psObj->hChannelList, NULL);
            if (eOsalCode != OSAL_SUCCESS)
            {
                break;
            }

            // List was changed. We'll need to notify application about it
            bUpdated = TRUE;
        }

        // Add new channels in category
        for ( tIndex = 0; tIndex < tArraySize; tIndex++ )
        {
            eOsalCode = OSAL.eLinkedListAdd(
                psObj->hChannelList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                phChannel[tIndex] );

            if (eOsalCode != OSAL_SUCCESS)
            {
                break;
            }

            // If succeeded at least once, marking list as "updated"
            bUpdated = TRUE;
        }

        // Checking if operation went well
        if (eOsalCode != OSAL_SUCCESS)
        {
            // Trying to recover, if at least one element was added
            if (bUpdated == TRUE && tIndex > 0)
            {
                OSAL.eLinkedListRemoveAll(psObj->hChannelList, NULL);
                // Regardless of return code, we are in error state anyway
                // (because some Add failed before)
            }
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (bUpdated == TRUE && psObj != NULL)
    {
        // Update the event mask
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

        // Notify registered parties
        SMSU_bNotify(&psObj->sEvent);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CATEGORY_eSerialize
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eSerialize (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    TAG_OBJECT hParentTag
        )
{
    TAG_OBJECT hCategoryTag = TAG_INVALID_OBJECT;
    BOOLEAN bOwner = FALSE, bCommitMe = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode =
        SMSAPI_RETURN_CODE_ERROR;

    do
    {
        N32 n32Value;
        const char *pacShortName;
        CATEGORY_OBJECT hCategory;
        CATEGORY_OBJECT_STRUCT *psObj;
        SMSAPI_RETURN_CODE_ENUM eResultCode;
        OSAL_RETURN_CODE_ENUM eOsalCode = OSAL_ERROR;
        TAG_OBJECT hLongNameTag, hMediumNameTag, hTypeTag, 
                   hUniqueTag, hReplaceTag, hChannelsTag;
        UN32 un32Index;
        CATEGORY_SERIALIZATION_STRUCT sData;

        // Verify inputs
        if ((hDecoder == DECODER_INVALID_OBJECT) ||
            (tCategoryId == CATEGORY_INVALID_ID) ||
            (hParentTag == TAG_INVALID_OBJECT))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Verify we own the decoder
        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOwner == FALSE)
        {
            break;
        }

        // Get category handle
        hCategory = hGetHandle(hDecoder, tCategoryId);
        if (hCategory == CATEGORY_INVALID_OBJECT)
        {
            break;
        }

        // De-reference object
        psObj = (CATEGORY_OBJECT_STRUCT *)hCategory;

        /* Get all the tags */

        pacShortName = STRING.pacCStr(psObj->hShortName);

        // Get the category tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_NAME,
            hParentTag,
            &hCategoryTag,
            pacShortName,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get medium name tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_MED_NAME_TAG,
            hCategoryTag,
            &hMediumNameTag,
            (char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get long name tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_LONG_NAME_TAG,
            hCategoryTag,
            &hLongNameTag,
            (char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get type tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_TYPE_TAG,
            hCategoryTag,
            &hTypeTag,
            (char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get unique tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_UNIQUE_TAG,
            hCategoryTag,
            &hUniqueTag,
            (char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get replace tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_REPLACE_TAG,
            hCategoryTag,
            &hReplaceTag,
            (char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Get channels tag
        eResultCode = TAG_eGet(
            CATEGORY_OBJECT_CHANNELS_TAG,
            hCategoryTag,
            &hChannelsTag,
            (const char *)NULL,
            TRUE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        /* Set new values */

        // Set Medium Name value
        eResultCode = TAG_eSetTagValue(
            hMediumNameTag,
            TAG_TYPE_STRING,
            (void *)&psObj->hMediumName,
            sizeof(STRING_OBJECT),
            FALSE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Set Long Name value
        eResultCode = TAG_eSetTagValue(
            hLongNameTag,
            TAG_TYPE_STRING,
            (void *)&psObj->hLongName,
            sizeof(STRING_OBJECT),
            FALSE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Set Type value
        n32Value = (N32)psObj->eCategoryType;

        eResultCode = TAG_eSetTagValue(
            hTypeTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            FALSE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Set Unique value
        n32Value = (N32)psObj->bUniqueItems;

        eResultCode = TAG_eSetTagValue(
            hUniqueTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            FALSE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Set Replace value
        n32Value = (N32)psObj->bReplace;

        eResultCode = TAG_eSetTagValue(
            hReplaceTag,
            TAG_TYPE_INTEGER,
            (void *)&n32Value,
            sizeof(N32),
            FALSE);

        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Commit request
        bCommitMe = TRUE;

        // Initialize iterator data
        sData.bCommitMe = FALSE;
        sData.bSuccess = TRUE;
        sData.hChannelsTag = hChannelsTag;
        sData.hNextSiblingTag = TAG_INVALID_OBJECT;

        eResultCode = TAG_eNumberOfChildren( hChannelsTag, &sData.un32Channels );
        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }
        
        if (sData.un32Channels > 0)
        {
            eResultCode = TAG_eFirstChild( hChannelsTag, &sData.hNextSiblingTag );
            if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // Iterate thru channels
        eOsalCode = OSAL.eLinkedListIterate(
            psObj->hChannelList,
            bSerializationIterator,
            (void *)&sData);

        if( ( (eOsalCode != OSAL_SUCCESS) && (eOsalCode != OSAL_NO_OBJECTS) ) || 
            (sData.bSuccess == FALSE) )
        {
            // Something went wrong
            break;
        }

        // Remove extra tags (if any)
        for( un32Index = 0; un32Index < sData.un32Channels; un32Index++)
        {
            TAG_OBJECT hRemoveMe = sData.hNextSiblingTag;

            eResultCode = TAG_eNextSibling(
                sData.hNextSiblingTag, 
                &sData.hNextSiblingTag
                    );

            if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                sData.bSuccess = FALSE;
                break;
            }

            eResultCode = TAG_eRemove( hRemoveMe, FALSE );
            if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                sData.bSuccess = FALSE;
                break;
            }
        }

        if (sData.bSuccess == FALSE)
        {
            // Failed to remove extra channels
            break;
        }

        // Forward commit request
        bCommitMe |= sData.bCommitMe;

        // Serialization procedure passed successfully
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (TAG_INVALID_OBJECT != hCategoryTag)
        {
            // In case of fail, trying to reset category contents
            TAG_eRemove(hCategoryTag, FALSE);
        }
        bCommitMe = TRUE;
    }

    // Commit requested
    if (bCommitMe == TRUE)
    {
        CM_eCommitChangesToFile();
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CATEGORY_tCreateFromTags
*
*****************************************************************************/
CATEGORY_ID CATEGORY_tCreateFromTags (
    DECODER_OBJECT hDecoder,
    TAG_OBJECT hParentTag
        )
{
    BOOLEAN bOwner = FALSE;
    CHANNEL_OBJECT *phChannel = NULL;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
    STRING_OBJECT hLongName = STRING_INVALID_OBJECT,
                  hMediumName = STRING_INVALID_OBJECT;

    do
    {
        UN32 un32Children;
        size_t tBytesAvail = 0;
        CCACHE_OBJECT hCCache;
        BOOLEAN bUnique, bReplace;
        const char *pacShortName, *pacMediumName, *pacLongName;
        CATEGORY_TYPE_ENUM eCategoryType;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        N32 n32Value, * pn32Value = &n32Value;
        STRING_OBJECT hShortName = STRING_INVALID_OBJECT, *phRefString;
        TAG_OBJECT hCategoryTag, hLongNameTag, hMediumNameTag, hTypeTag, 
                   hUniqueTag, hReplaceTag, hChannelTag;

        // Verify we own the decoder
        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOwner == FALSE)
        {
            break;
        }

        /* Get All the Tags */

        // Get Category Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_NAME,
            hParentTag,
            &hCategoryTag,
            (const char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hCategoryTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Long Name Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_LONG_NAME_TAG,
            hCategoryTag,
            &hLongNameTag,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hLongNameTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Medium Name Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_MED_NAME_TAG,
            hCategoryTag,
            &hMediumNameTag,
            (char *)NULL,
            TRUE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hMediumNameTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Type Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_TYPE_TAG,
            hCategoryTag,
            &hTypeTag,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hTypeTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Unique Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_UNIQUE_TAG,
            hCategoryTag,
            &hUniqueTag,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hUniqueTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Replace Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_REPLACE_TAG,
            hCategoryTag,
            &hReplaceTag,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hReplaceTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        // Get Channels Tag
        eReturnCode = TAG_eGet(
            CATEGORY_OBJECT_CHANNELS_TAG,
            hCategoryTag,
            &hChannelTag,
            (char *)NULL,
            FALSE);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
            (hChannelTag == TAG_INVALID_OBJECT))
        {
            break;
        }

        eReturnCode = TAG_eNumberOfChildren(hChannelTag, &un32Children);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        /* Get All the Tag values */

        // Get Short Name value
        hShortName = TAG_hTagInstanceName(hCategoryTag);
        if (hShortName == STRING_INVALID_OBJECT)
        {
            break;
        }

        pacShortName = STRING.pacCStr(hShortName);

        // Get Long Name value
        phRefString = &hLongName;

        // Get available space size
        tBytesAvail = sizeof(STRING_OBJECT);

        // This method will allocate a new string
        eReturnCode = TAG_eGetTagValue(
            hLongNameTag,
            TAG_TYPE_STRING,
            (void **)&phRefString,
            &tBytesAvail);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Save Long Name
        pacLongName = STRING.pacCStr(hLongName);

        // Get Medium Name value
        phRefString = &hMediumName;

        // This method will allocate a new string
        eReturnCode = TAG_eGetTagValue(
            hMediumNameTag,
            TAG_TYPE_STRING,
            (void **)&phRefString,
            &tBytesAvail);

        if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
            (eReturnCode != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET))
        {
            break;
        }

        // Save Medium Name
        pacMediumName = STRING.pacCStr(hMediumName);

        // Get available space size
        tBytesAvail = sizeof(N32);

        // Get Category Type value
        eReturnCode = TAG_eGetTagValue(
            hTypeTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tBytesAvail
                );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Save Category Type value
        eCategoryType = (CATEGORY_TYPE_ENUM)n32Value;

        // Get available space size
        tBytesAvail = sizeof(N32);

        // Get Unique value
        eReturnCode = TAG_eGetTagValue(
            hUniqueTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tBytesAvail
                );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Save Unique value
        bUnique = (BOOLEAN)n32Value;

        // Get available space size
        tBytesAvail = sizeof(N32);

        // Get Replace value
        eReturnCode = TAG_eGetTagValue(
            hReplaceTag,
            TAG_TYPE_INTEGER,
            (void **)&pn32Value,
            &tBytesAvail
                );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Save Replace value
        bReplace = (BOOLEAN)n32Value;

        /* Create new Category */

        // Get Channel Cache
        hCCache = DECODER_hCCache(hDecoder);

        tCategoryId = CCACHE_tAddCategory(
            hCCache,
            eCategoryType,
            pacLongName,
            pacMediumName,
            pacShortName,
            0,
            bUnique,
            bReplace);

        if (tCategoryId == CATEGORY_INVALID_ID)
        {
            break;
        }

        /* Add channels into category */

        if (un32Children > 0)
        {
            CATEGORY_SERIALIZATION_STRUCT sData;

            // Allocate tags array
            phChannel = (CHANNEL_OBJECT *)
                SMSO_hCreate(CATEGORY_OBJECT_NAME
                    ": Serialization Channels Array",
                    sizeof(CHANNEL_OBJECT) * un32Children,
                    SMS_INVALID_OBJECT,
                    FALSE);

            if (phChannel == NULL)
            {
                break;
            }

            // Initialize iterator struct
            sData.bSuccess = TRUE;
            sData.tIndex = 0;
            sData.hCCache = hCCache;
            sData.phChannel = phChannel;

            // Get all channel handles
            eReturnCode = TAG_eIterateChildren(
                hChannelTag,
                bRestoreIterator,
                (void *)&sData
                    );

            if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) ||
                (sData.bSuccess == FALSE))
            {
                break;
            }

            // Populate all channels into category
            eReturnCode = CATEGORY_eFill(
                hDecoder,
                tCategoryId,
                phChannel,
                sData.tIndex
                    );

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

    } while (FALSE);

    if (hLongName != STRING_INVALID_OBJECT)
    {
        // we're done with this
        STRING_vDestroy(hLongName);
    }

    if (phChannel != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)phChannel);
    }

    return tCategoryId;
}

/*****************************************************************************
*
*   CATEGORY_pacTagName
*
*****************************************************************************/
const char *CATEGORY_pacTagName ( void )
{
    return CATEGORY_OBJECT_NAME;
}

/*****************************************************************************
*
*   CATEGORY_eSort
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CATEGORY_eSort(
    CATEGORY_OBJECT hCategory,
    CATEGORY_SORT_HANDLER n16SortFxn
	    )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);
    if (bOwner == TRUE)
    {
        // De-reference object
        CATEGORY_OBJECT_STRUCT *psObj = 
            (CATEGORY_OBJECT_STRUCT *)hCategory;

        eReturnCode = eSortDirect(psObj, n16SortFxn);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CATEGORY_bBlockNotifications
*
*   Note! This function is non-recursive, it must be used in conjunction with
*   vReleaseNotifications within the same CATEGORY object. So this means you
*   should only call this API to block notifications, do your thing and then
*   release notifications.
*   This is to be used when the caller wishes to make multiple CATEGORY
*   object updates but wants to block notifications until they are all finished.
*
*   Inputs:
*       hCategory - A valid category for which to block (filter) all future
*           category update notifications.
*
*   Returns:
*       BOOLEAN
*
*****************************************************************************/
BOOLEAN CATEGORY_bBlockNotifications(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bLocked = FALSE, bBlocked = FALSE;

    do
    {
        CATEGORY_OBJECT hCategory;

        // Lock object for exclusive access
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Get object handle
        hCategory = hGetHandle(hDecoder, tCategoryId);
        if (hCategory == CATEGORY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CATEGORY_OBJECT_NAME": Failed to get Category Id: %d handle",
                tCategoryId);
            break;
        }

        bBlocked = bBlockNotifications(hCategory);

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bBlocked;
}

/*****************************************************************************
*
*   CATEGORY_vReleaseNotifications
*
*   Processes any pending notifications and restores the CATEGORY object's
*   filter setting before notifications were blocked.
*
*   Inputs:
*       hCategory - A valid category for which to release (un-filter) all future
*           category update notifications which were previously blocked.
*
*   Returns:
*       None.
*
*****************************************************************************/
void CATEGORY_vReleaseNotifications (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bLocked = FALSE;

    do
    {
        CATEGORY_OBJECT hCategory;

        // Verify and check ownership of object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        hCategory = hGetHandle(hDecoder, tCategoryId);
        if (hCategory == CATEGORY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CATEGORY_OBJECT_NAME": Failed to get Category Id: %d handle",
                tCategoryId);
            break;
        }

        vReleaseNotifications(hCategory);

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownersip
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hGetHandle
*
*****************************************************************************/
static CATEGORY_OBJECT hGetHandle (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        )
{
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    CCACHE_OBJECT hCCache;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == FALSE)
    {
        // Error!
        return CATEGORY_INVALID_OBJECT;
    }

    if(tCategoryId == CATEGORY_INVALID_ID )
    {
        // Error!
        return CATEGORY_INVALID_OBJECT;
    }

    // Get the channel cache handle for the associated decoder., Don't worry
    // about locking at this point as the caller has already locked the DECODER
    hCCache = DECODER_hCCache( hDecoder );

    // If the specified decoder has an associated ccache
    if ( hCCache == CCACHE_INVALID_OBJECT )
    {
        // Error!
        return CATEGORY_INVALID_OBJECT;
    }

    hCategory = CCACHE_hCategory(hCCache, &tCategoryId, 0);

    return hCategory;
}

/*****************************************************************************
*
*   n16GetHdlByOffsetIterateFxn
*
*****************************************************************************/
static N16 n16GetHdlByOffsetIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    CATEGORY_GET_HANDLE_BY_OFFSET_STRUCT *psOffsetStruct =
        (CATEGORY_GET_HANDLE_BY_OFFSET_STRUCT *)pvIterateArg;

    // Verify Inputs, we need a valid category, a valid pointer to a counter in
    // pvIterateArg, hChannel could possibly by CHANNEL_INVALID_OBJECT so don't
    // protect for that
    if ( ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pvIterateArg == NULL ) )
    {
        return 0;
    }

    // If this is the item we were looking for
    if ( psOffsetStruct->n16OffsetCntr == 0 )
    {
        // Save the handle back to the structure
        psOffsetStruct->hChannel = hChannel;

        return 0;
    }

    // If we want to go backwards
    if ( psOffsetStruct->n16OffsetCntr < 0 )
    {
        // We need to increase this offset towards 0
        psOffsetStruct->n16OffsetCntr++;

        // Return a negative so we iterate backwards through the list
        return -1;
    }

    // We want to navigate this list in the positive direction

    // We need to decrease this offset towards 0
    psOffsetStruct->n16OffsetCntr--;

    // Return a positive so we iterate forwards through the list
    return 1;
}

/*****************************************************************************
*
*       n16GetIndexByChanIdIterateFxn
*
*       This function is used by the CATEGORY iterator to get the index into the
*       category for the first occurrence of the specified channel, if the
*       channel is present.  The original call to the CATEGORY's iterator should
*       have populated a structure of type CATEGORY_GET_INDEX_BY_ID_STRUCT and
*       cast it to a void pointer. It also should have set the tChannelId member
*       of that structure to the channel id that it wishes to search for.  The
*       pn16Index member of the structure should also be populated with a valid
*       pointer so that the index can be returned to the caller. This function
*       will iterate through the list once, then stop iterating.
*
*       Inputs:
*           hCategory       Category handle to get the search
*           tCurrentIndex   index from the top of the category list
*           hChannel        Handle to the channel object in this position in the
*                           CATEGORY list
*           pvIterateArg    a ptr to a CATEGORY_GET_INDEX_BY_ID_STRUCT that
*                           the user wants the index for.
*
*       Outputs:
*           N16             0 = Stop iterating
*                           1 = Iterate forward one
*
*****************************************************************************/
static N16 n16GetIndexByChanIdIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    CATEGORY_GET_INDEX_BY_ID_STRUCT *psIndexStruct;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Verify Inputs, we need a valid category, a valid pointer to a counter in
    // pvIterateArg, hChannel could possibly by CHANNEL_INVALID_OBJECT so don't
    // protect for that
    if ( ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pvIterateArg == NULL ) )
    {
        return 0;
    }

    // Get the structure pointer
    psIndexStruct = (CATEGORY_GET_INDEX_BY_ID_STRUCT *)pvIterateArg;

    // If we were given a valid channel object
    if ( hChannel != CHANNEL_INVALID_OBJECT )
    {
        // Get the channel id for the object
        tChannelId = CHANNEL.tChannelId( hChannel );
    }

    // If this is the item we want to get the channel id for
    if ( psIndexStruct->tChannelId == tChannelId )
    {
        // Return the index to the caller of the category's iterate function
        psIndexStruct->n16Index = (N16)tCurrentIndex;

        // We found what we were looking for, stop iterating
        return 0;
    }

    // Decrement the size attribute.  When it is zero,
    // we've gone through all the channels in this category
    psIndexStruct->un32CategorySize--;

    if (psIndexStruct->un32CategorySize == 0)
    {
        // We're done
        return 0;
    }

    // Keep moving in a forward direction
    return 1;
}

/*****************************************************************************
*
*       n16GetIndexByServiceIdIterateFxn
*
*       This function is used by the CATEGORY iterator to get the index into the
*       category for the first occurrence of the specified channel, if the
*       channel is present.  The original call to the CATEGORY's iterator shall
*       populate a structure of type CATEGORY_GET_INDEX_BY_ID_STRUCT and cast it
*       to a void pointer. tServiceId member of the structure shall contain SID
*       of the channel to search for.  The pn16Index member of the structure
*       should also be populated with a valid pointer so that the index can be
*       returned to the caller. This function will iterate through the list once,
*       then stop iterating.
*
*       Inputs:
*           hCategory       Category handle to get the search
*           tCurrentIndex   index from the top of the category list
*           hChannel        Handle to the channel object in this position in the
*                           CATEGORY list
*           pvIterateArg    a ptr to a CATEGORY_GET_INDEX_BY_ID_STRUCT that
*                           the user wants the index for.
*
*       Outputs:
*           N16             0 = Stop iterating
*                           1 = Iterate forward one
*
*****************************************************************************/
static N16 n16GetIndexByServiceIdIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void * pvIterateArg
        )
{
    CATEGORY_GET_INDEX_BY_ID_STRUCT *psIndexStruct;
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;

    // Verify Inputs, we need a valid category, a valid pointer to a counter in
    // pvIterateArg, hChannel could possibly by CHANNEL_INVALID_OBJECT so don't
    // protect for that
    if ( ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pvIterateArg == NULL ) )
    {
        return 0;
    }

    // Get the structure pointer
    psIndexStruct = (CATEGORY_GET_INDEX_BY_ID_STRUCT *)pvIterateArg;

    // If we were given a valid channel object
    if ( hChannel != CHANNEL_INVALID_OBJECT )
    {
        // Get the channel id for the object
        tServiceId = CHANNEL.tServiceId( hChannel );
    }

    // If this is the item we want to get the channel id for
    if ( psIndexStruct->tServiceId == tServiceId )
    {
        // Return the index to the caller of the category's iterate function
        psIndexStruct->n16Index = (N16)tCurrentIndex;

        // We found what we were looking for, stop iterating
        return 0;
    }

    // Decrement the size attribute.  When it is zero,
    // we've gone through all the channels in this category
    psIndexStruct->un32CategorySize--;

    if (psIndexStruct->un32CategorySize == 0)
    {
        // We're done
        return 0;
    }

    // Keep moving in a forward direction
    return 1;
}

/*****************************************************************************
*
*   bRemoveCategoryFromChannel
*
*   The purpose of this function is used as a linked list iterator
*   to remove a category from a channel
*
*****************************************************************************/
static BOOLEAN bRemoveCategoryFromChannel(
    void *pvData,
    void *pvArg
        )
{
    // remove the category from the channel
    CHANNEL_vRemoveCategory(
        (CHANNEL_OBJECT)pvData, (CATEGORY_OBJECT)pvArg, FALSE);

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bNotifyCategoryUpdated
*
*   The purpose of this function is used as a linked list iterator
*   to notify update category art from a channel
*
*****************************************************************************/
static BOOLEAN bNotifyCategoryUpdated (
    void *pvData,
    void *pvArg
        )
{
    // Tell the channel to update the app
    // Set event mask & notify
    CHANNEL_vSetEvents((CHANNEL_OBJECT)pvData, CHANNEL_OBJECT_EVENT_CATEGORY);

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   psCreateCategory
*
*   This function performs all the creation and initialization of category
*   objects which is shared between all category types.
*
*****************************************************************************/
static CATEGORY_OBJECT_STRUCT *psCreateCategory (
    CATEGORY_TYPE_ENUM eCategoryType,
    DECODER_OBJECT hDecoder,
    SMS_OBJECT hParent,
    CATEGORY_ID tId,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_OBJECT_STRUCT *psObj =
        (CATEGORY_OBJECT_STRUCT *)CATEGORY_INVALID_OBJECT;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_LL_COMPARE_HANDLER tn16ChannelComparator;
    BOOLEAN bOk;

    // Check the input names
    bOk = bProcessCategoryNames(eCategoryType, 
        &pacLongName, 
        &pacMediumName,
        &pacShortName);
    if (bOk == FALSE)
    {
        // There are input parameter errors!
        return NULL;
    }

    // Find out if the Id provided is in the correct
    // range for this category type
    bOk = CATEGORY_bIdInRange (
                eCategoryType,
                tId );

    // As long as the id is within the valid range for
    // categories of this type we may continue
    if (bOk == FALSE)
    {
         // There are input parameter errors!
        return NULL;
    }

    // Construct a unique name for this category.
    snprintf(&acName[0],
            sizeof(acName),
            CATEGORY_OBJECT_NAME":%u",
            tId);

    // Create an instance of this object
    psObj = (CATEGORY_OBJECT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(CATEGORY_OBJECT_STRUCT),
            hParent,
            FALSE);
    if(psObj == NULL)
    {
        // Error!
        return NULL;
    }

    // Initialize object with inputs
    psObj->tId = tId;
    psObj->eCategoryType = eCategoryType;
    psObj->hDecoder = hDecoder;
    psObj->hChannelArtService = CHANNEL_ART_SERVICE_INVALID_OBJECT;
    psObj->hArt = CHANNEL_ART_INVALID_OBJECT;
    psObj->bUniqueItems = bUniqueItems;
    psObj->bReplace = bReplace;
    psObj->hShortName = STRING_INVALID_OBJECT;
    psObj->hMediumName = STRING_INVALID_OBJECT;
    psObj->hLongName = STRING_INVALID_OBJECT;
    psObj->tFilter = CATEGORY_OBJECT_EVENT_ALL;

    // Initialize iteration attributes
    psObj->sIterationCtrl.hCurrentEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->sIterationCtrl.bChannelRemoved = FALSE;
    psObj->sIterationCtrl.un8ChannelsAddedBefore = 0;

    // Determine which channel comparator should be used
    // for this category
    switch (eCategoryType)
    {
        case CATEGORY_TYPE_BROADCAST:
        {
            tn16ChannelComparator = 
                (OSAL_LL_COMPARE_HANDLER)CHANNEL_n16CompareAlternateIds;
        }
        break;

        case CATEGORY_TYPE_USER:
        case CATEGORY_TYPE_VIRTUAL:
        default:
        {
            tn16ChannelComparator = NULL;
        }
        break;
    }

    // Construct a unique name for this category's channel list.
    snprintf(&acName[0],
            sizeof(acName),
            "CatChanList:%u",
            psObj->tId);

    // Create a channel linked list
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->hChannelList,
        &acName[0],
        tn16ChannelComparator,
        OSAL_LL_OPTION_CIRCULAR
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        CATEGORY_vDestroy((CATEGORY_OBJECT)psObj);
        return NULL;
    }

    // Update new category object by provided names.
    (void) bUpdateNames(psObj, pacLongName, pacMediumName, pacShortName);

    // Initialize asynchronous update configuration
    SMSU_vInitialize(
        &psObj->sEvent,
        psObj,
        CATEGORY_OBJECT_EVENT_INITIAL,
        CATEGORY_OBJECT_EVENT_NONE,
        (SMSAPI_OBJECT_EVENT_CALLBACK)NULL,
        NULL);

    return psObj;
}

/*****************************************************************************
*
*   bProcessCategoryNames
*
*   based on the combinations of input, either
*   1. we don't have sufficient name information to create a category,
*   2. we substitute the long/medium name for the short name or vice versa
*   3. the names are ok as provided to us
*
*  If TRUE is returned, the category can be created.
*  If FALSE is returned it cannot
*
*****************************************************************************/
static BOOLEAN bProcessCategoryNames(
    CATEGORY_TYPE_ENUM eType,
    const char **ppacLongName,
    const char **ppacMediumName,
    const char **ppacShortName
        )
{
    BOOLEAN bNoLongName = TRUE, bNoMedName = TRUE, bNoShortName = TRUE;
    BOOLEAN bOk = TRUE;

    // first, determine if the provided long name is null or blank (0 length)
    if (*ppacLongName != NULL)
    {
        // non-NULL long name pointer provided
        if (strlen(*ppacLongName) > 0)
        {
            // non-blank long name given
            bNoLongName = FALSE;
        }
    }

    // next, determine if the provided medium name is null or blank (0 length)
    if (*ppacMediumName != NULL)
    {
        // non-NULL short name pointer provided
        if (strlen(*ppacMediumName) > 0)
        {
            // non-blank long name given
            bNoMedName = FALSE;
        }
    }

    // last, determine if the provided short name is null or blank (0 length)
    if (*ppacShortName != NULL)
    {
        // non-NULL short name pointer provided
        if (strlen(*ppacShortName) > 0)
        {
            // non-blank long name given
            bNoShortName = FALSE;
        }
    }

    // based on the combinations of input, either
    // 1. we don't have sufficient name information to create a category,
    // 2. we substitute the long or medium name for the short name or vice versa
    // 3. the names are ok as provided to us

    if ( bNoLongName == TRUE )
    {
        // NULL or blank long name provided

        if ( bNoMedName == TRUE )
        {
            // NULL or blank medium name provided

            if ( bNoShortName == TRUE )
            {
                // NULL or blank short name provided

                if (eType != CATEGORY_TYPE_BROADCAST)
                {
                    // reject - all names are invalid.
                    bOk = FALSE;
                }
            }
            else
            {
                // non-blank short name provided
                // set the long and medium names equal to the short name
                *ppacLongName = *ppacShortName;
                *ppacMediumName = *ppacShortName;
            }
        }
        else
        {
            // non-blank medium name provided

            if ( bNoShortName == TRUE )
            {
                // NULL or blank short name provided
                // set the short name equal to the medium name
                *ppacShortName = *ppacMediumName;
            }
            
            // set the long name equal to the medium name
            *ppacLongName = *ppacMediumName;
        }
    }
    else
    {
        // non-blank long name provided - use it

        if ( bNoMedName == TRUE )
        {
            // set the medium name equal to the long name
            *ppacMediumName = *ppacLongName;
        }
        
        if ( bNoShortName == TRUE )
        {
            // set the short name equal to the long name
            *ppacShortName = *ppacLongName;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bUpdateNames
*
*****************************************************************************/
static BOOLEAN bUpdateNames(
    CATEGORY_OBJECT_STRUCT *psObj,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName
        )
{
    BOOLEAN bOk = FALSE;

    if (psObj != (CATEGORY_OBJECT_STRUCT *)NULL)
    {
        if (pacShortName != NULL)
        {
            if (psObj->hShortName == STRING_INVALID_OBJECT)
            {
                // Create object for pacShortName
                psObj->hShortName =
                    STRING_hCreate(
                            (SMS_OBJECT)psObj,
                            pacShortName,
                            strlen(pacShortName),
                            GsRadio.sCategory.un16ShortNameMaxLen);
                if (psObj->hShortName != STRING_INVALID_OBJECT)
                {
                    bOk = TRUE;
                }
            }
            else
            {
                // yes. change the short name
                bOk =
                    STRING.bModifyCStr(psObj->hShortName, pacShortName);
            }
        }

        if (pacMediumName != NULL)
        {
            if (psObj->hMediumName == STRING_INVALID_OBJECT)
            {
                // Create object for pacShortName
                psObj->hMediumName =
                    STRING_hCreate(
                    (SMS_OBJECT)psObj,
                    pacMediumName,
                    strlen(pacMediumName),
                    GsRadio.sCategory.un16MediumNameMaxLen);
                if (psObj->hMediumName != STRING_INVALID_OBJECT)
                {
                    bOk = TRUE;
                }
            }
            else
            {
                // yes. change the short name
                bOk =
                    STRING.bModifyCStr(psObj->hMediumName, pacMediumName);
            }
        }

        if (pacLongName != NULL)
        {
            if (psObj->hLongName == STRING_INVALID_OBJECT)
            {
                // Create object for pacLongName
                psObj->hLongName =
                    STRING_hCreate(
                            (SMS_OBJECT)psObj,
                            pacLongName,
                            strlen(pacLongName),
                            GsRadio.sCategory.un16LongNameMaxLen);
                if (psObj->hLongName != STRING_INVALID_OBJECT)
                {
                    bOk = TRUE;
                }
            }
            else
            {
                BOOLEAN bModified;

                // yes. change the long name
                bModified =
                    STRING.bModifyCStr(psObj->hLongName, pacLongName);
                if (bModified == TRUE)
                {
                    bOk = TRUE;
                }
            }
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   vNotifyRegisteredAndProtectIterator
*
*   This function is used to notify all registered callbacks of a change
*   to a CATEGORY while that CATEGORY is being iterated (via .eIterate).
*   The iteration state is saved before the notification is performed and
*   restored after all notifications have completed.  This allows us to
*   "protect" the current iteration state from the case in which notified
*   objects invoke CATGORY.eIterate, which would clear our current iteration
*   state.
*
*****************************************************************************/
static void vNotifyRegisteredAndProtectIterator (
    CATEGORY_OBJECT_STRUCT *psObj,
    SMSAPI_EVENT_MASK tEventMask
        )
{
    CATEGORY_ITERATION_CTRL_STRUCT sCtrlCopy;

    // Copy the iteration control structure
    sCtrlCopy = psObj->sIterationCtrl;

    // Update the event mask
    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    // Notify all interested parties.  It is
    // possible that CATEGORY.eIterate may be
    // called via SMSU_bNofify, which means our
    // iteration control structure is gonna
    // get altered here...
    SMSU_bNotify(&psObj->sEvent);

    // Restore the state of the iteration
    // control structure
    psObj->sIterationCtrl = sCtrlCopy;

    return;
}

/*****************************************************************************
*
*   bDetermineChanOccurrence
*
*   The purpose of this function is to determine how many times a
*   particular channel id is encountered in a portion of the channel list
*
*****************************************************************************/
static BOOLEAN bDetermineChanOccurrence (
    void *pvObj1,
    void *pvObj2
        )
{
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvObj1;
    CATEGORY_CHAN_OCCURRENCE_STRUCT *psOccurrence =
        (CATEGORY_CHAN_OCCURRENCE_STRUCT *)pvObj2;
    CHANNEL_ID tChannelId;

    // Is the channel argument valid?
    if (pvObj1 == NULL)
    {
        // No -- this isn't an error, we just
        // have an empty space for a channel
        // continue past this entry
        return TRUE;
    }

    // Verify the provided argument is valid
    if (psOccurrence == NULL)
    {
        // Error! Stop
        return FALSE;
    }

    if (psOccurrence->n16Index == psOccurrence->n16Limit)
    {
        // We've reached our limit -
        // stop processing
        return FALSE;
    }

    tChannelId = CHANNEL.tChannelId( hChannel );
    if (tChannelId == psOccurrence->tChannelId)
    {
        // We have found an occurrence of the
        // channel id we're looking for
        psOccurrence->n16Occurrence++;
    }

    psOccurrence->n16Index++;

    return TRUE;
}

/*****************************************************************************
*
*   eRenameLocal
*
*   Assumes that psObj is valid and that we are in the correct context
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRenameLocal(
    CATEGORY_OBJECT_STRUCT *psObj,
    const char *pacNewLongName,
    const char *pacNewMedName,
    const char *pacNewShortName
        )
{
    BOOLEAN bUpdated;

    // Update long and/o short names if caller wants this
    bUpdated = bUpdateNames(psObj, 
        pacNewLongName, pacNewMedName, pacNewShortName);

    // Signal an event only if strings were actually changed
    if (bUpdated == TRUE)
    {
        // Update our event mask and notify our category event listeners
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_NAME);

        // Notify callback if registered and change occurred
        SMSU_bNotify(&psObj->sEvent);

        // At this point all CHANNELs in this category need to be
        // notified that their category name has changed. So we
        // need to iterate the entire list of channels belonging
        // to this category.
        // (maybe the channel should be a listener of the category???)
        OSAL.eLinkedListIterate(
            psObj->hChannelList,
            bNotifyCategoryUpdated,
            NULL
                );
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   bAlwaysInclude
*
*****************************************************************************/
static BOOLEAN bAlwaysInclude (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        )
{
    return TRUE;
}

/*****************************************************************************
*
*   bIteratorShim
*
*****************************************************************************/
static BOOLEAN bIteratorShim (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        )
{
    CATEGORY_ITERATOR_STRUCT *psIterator = (CATEGORY_ITERATOR_STRUCT *)pvArg;
    BOOLEAN bInclude, bContinue = TRUE;

    // For this object, run it's comparator first to decide if it
    // is to be included.
    bInclude = psIterator->bInclude(hCategory, psIterator->pvArg);
    if(bInclude == TRUE)
    {
        // Go ahead...include it
        bContinue = psIterator->bIterator(hCategory, psIterator->pvArg);
    }

    return bContinue; // Keep iterating?
}

/*****************************************************************************
*
*   eInsertAfterChannelLocal
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInsertAfterChannelLocal (
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hNewEntry;

    // Verify this is not a broadcast category
    if ( psObj->eCategoryType == CATEGORY_TYPE_BROADCAST )
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    // Check to make sure we have a valid current entry
    if ( psObj->sIterationCtrl.hCurrentEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Copy the current entry
    hNewEntry = psObj->sIterationCtrl.hCurrentEntry;

    // DO NOT check hChannel against CHANNEL_INVALID_OBJECT because
    // it is perfectly valid to use that value as a reset condition
    // for a member of the category. Add it regardless

    // This category enforces to use each channels only once,
    if (psObj->bUniqueItems == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psObj->hChannelList,
            &hEntry,
            CHANNEL_n16CompareChannelIds,
            (void *)hChannel);

        // Duplicate entry is found
        if (eReturnCode == OSAL_SUCCESS)
        {
            if (psObj->bReplace == TRUE)
            {
                if (hEntry != psObj->sIterationCtrl.hCurrentEntry)
                {
                    // Remove this channel
                    OSAL.eLinkedListRemove(hEntry);
                }
                else
                {
                    // Consider this as success
                    return SMSAPI_RETURN_CODE_SUCCESS;
                }
            }
            else // (psObj->bReplace == FALSE)
            {
                // Duplicate channel should be rejected
                return SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
            }
        }
    }

    // Insert After (this will alter the contents of
    // the provided entry handle, which is why we
    // used a copy instead of psObj->sIterationCtrl.hCurrentEntry
    eReturnCode = OSAL.eLinkedListAddAfterEntry(
                            psObj->hChannelList,
                            &hNewEntry,
                            hChannel );

    // Verify that worked
    if ( (eReturnCode == OSAL_SUCCESS ) &&
         (hNewEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
       )
    {
        if (hChannel != CHANNEL_INVALID_OBJECT)
        {
            // add this category to the channel
            CHANNEL_bAddCategory (hChannel, (CATEGORY_OBJECT)psObj);
        }

        // Notify all registered parties (but don't let them
        // mess up the state of the iterator )
        vNotifyRegisteredAndProtectIterator(
            psObj, CATEGORY_OBJECT_EVENT_CONTENTS );

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eInsertBeforeChannelLocal
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eInsertBeforeChannelLocal(
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hNewEntry;

    // Verify this is not a broadcast category
    if ( psObj->eCategoryType == CATEGORY_TYPE_BROADCAST )
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    // Check to make sure we have a valid current entry
    if ( psObj->sIterationCtrl.hCurrentEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Copy the current entry
    hNewEntry = psObj->sIterationCtrl.hCurrentEntry;

    // DO NOT check hChannel against CHANNEL_INVALID_OBJECT because
    // it is perfectly valid to use that value as a reset condition
    // for a member of the category. Add it regardless

    // This category enforces to use each channels only once,
    if (psObj->bUniqueItems == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psObj->hChannelList,
            &hEntry,
            CHANNEL_n16CompareChannelIds,
            (void *)hChannel);

        if (eReturnCode == OSAL_SUCCESS)
        {
            if (psObj->bReplace == TRUE)
            {
                if (hEntry != psObj->sIterationCtrl.hCurrentEntry)
                {
                    // Remove this channel
                    OSAL.eLinkedListRemove(hEntry);
                }
                else
                {
                    // Consider this as success
                    return SMSAPI_RETURN_CODE_SUCCESS;
                }
            }
            else
            {
                // The channel shall be rejected
                return SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
            }
        }
    }

    // Insert Before
    eReturnCode = OSAL.eLinkedListAddBeforeEntry(
                            psObj->hChannelList,
                            &hNewEntry,
                            hChannel );

    // Verify that worked
    if (( eReturnCode == OSAL_SUCCESS ) &&
        (hNewEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        if (hChannel != CHANNEL_INVALID_OBJECT)
        {
            // add this category to the channel
            CHANNEL_bAddCategory (hChannel, (CATEGORY_OBJECT)psObj);
        }

        // Indicate to the control structure that
        // a channel has been added before the reference
        psObj->sIterationCtrl.un8ChannelsAddedBefore++;

        // Notify all registered parties (but don't let them
        // mess up the state of the iterator )
        vNotifyRegisteredAndProtectIterator(
            psObj, CATEGORY_OBJECT_EVENT_CONTENTS );

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
*
*       eReplaceChannelLocal
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eReplaceChannelLocal (
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        )
{
    CHANNEL_OBJECT hChannelBeingReplaced = CHANNEL_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    CATEGORY_ITERATION_CTRL_STRUCT sCurrentIterationState;

    // Verify this is not a broadcast category
    if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
    {
        return SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
    }

    // Check to make sure we have a valid current entry
    if ( psObj->sIterationCtrl.hCurrentEntry
           == OSAL_INVALID_LINKED_LIST_ENTRY )
    {
        return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
    }

    // Save the current iteration control structure
    sCurrentIterationState = psObj->sIterationCtrl;

    // Grab the handle to the channel that's being changed
    hChannelBeingReplaced = (CHANNEL_OBJECT)
        OSAL.pvLinkedListThis(psObj->sIterationCtrl.hCurrentEntry);

    // This category enforces to use each channels only once,
    if (psObj->bUniqueItems == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psObj->hChannelList,
            &hEntry,
            CHANNEL_n16CompareChannelIds,
            (void *)hChannel);

        if (eReturnCode == OSAL_SUCCESS)
        {
            if (psObj->bReplace == TRUE)
            {
                if (hEntry != psObj->sIterationCtrl.hCurrentEntry)
                {
                    // Remove this channel
                    OSAL.eLinkedListRemove(hEntry);
                }
            }
            else
            {
                // The channel shall be rejected
                return SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
            }
        }
    }

    // This call to LinkedListAdd will actually modify the contents
    // of psObj->sIterationCtrl.hCurrentEntry with the new hChannel information
    eReturnCode = OSAL.eLinkedListReplaceEntry(
                            psObj->hChannelList,
                            psObj->sIterationCtrl.hCurrentEntry,
                            hChannel );

    // Verify that worked
    if ( eReturnCode != OSAL_SUCCESS )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Was the channel we just replaced valid?  If so,
    // we got some more work to do...
    if (hChannelBeingReplaced != CHANNEL_INVALID_OBJECT)
    {
        // We need to see if this was the only entry of
        // this channel id in the category. If so, we
        // need to inform the channel it is no longer
        // a member of this category
        CHANNEL_ID tReplacedId;

        // Get the id of the channel we're replacing
        tReplacedId = CHANNEL.tChannelId(hChannelBeingReplaced);

        // Was that valid?
        if (tReplacedId != CHANNEL_INVALID_ID)
        {
            N16 n16Index;

            // Find any occurrences of this channel
            // in the category. Note: This function utilizes
            // the iterator, which will adjust
            // psObj->sIterationCtrl so we'll have
            // to correct that before we return from this function
            n16Index = CATEGORY_n16GetIndexByChanId(
                                (CATEGORY_OBJECT)psObj,
                                tReplacedId);

            if (n16Index < 0)
            {
                // the neg index indicates that the channel id was not
                // found in the category's channel list, so now we need
                // to remove the category from the channel
                CHANNEL_vRemoveCategory(
                    hChannelBeingReplaced, (CATEGORY_OBJECT)psObj, TRUE);
            }
        }
    }

    // Only update the channel if it is valid
    if (hChannel != CHANNEL_INVALID_OBJECT)
    {
        // add this category to the channel
        CHANNEL_bAddCategory (hChannel, (CATEGORY_OBJECT)psObj);
    }

    // Now, restore the iteration control structure
    psObj->sIterationCtrl = sCurrentIterationState;

    // Notify all registered parties (but don't let them
    // mess up the state of the iterator )
    vNotifyRegisteredAndProtectIterator(
        psObj, CATEGORY_OBJECT_EVENT_CONTENTS );

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*       n16ChannelCompare
*
*****************************************************************************/
static N16 n16ChannelCompare(void *pvObj1, void *pvObj2)
{
    N16 n16Result = -1; // forward
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvObj1;
    CHANNEL_OBJECT hSoughtChannel = (CHANNEL_OBJECT)pvObj2;

    // same channel
    if (hChannel == hSoughtChannel)
    {
        // yup, stop iterating
        n16Result = 0;
    }

    // keep on iteratin'
    return n16Result;
}

/*****************************************************************************
*
*       vChannelRemappedEventCallback
*
*****************************************************************************/
static void vChannelRemappedEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    // we want to know when we got a channel object event, but NOT a service
    // id too. In other words, when an existing cahnnel is remapped, not
    // the creation of a new channel. New channels are added in a sorted
    // manner anyway, so we don't need to sort again.
    if (tEventMask == CHANNEL_OBJECT_EVENT_CHANNEL_ID)
    {
        CATEGORY_OBJECT hCategory =
            (CATEGORY_OBJECT)pvEventCallbackArg;
        BOOLEAN bOwner;

        bOwner = SMSO_bOwner((SMS_OBJECT)hCategory);

        if (bOwner == TRUE)
        {
            CATEGORY_OBJECT_STRUCT *psObj =
                (CATEGORY_OBJECT_STRUCT *)hCategory;

            // only sort if broadcast category changed.
            if (psObj->eCategoryType == CATEGORY_TYPE_BROADCAST)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Sort
                eReturnCode = OSAL.eLinkedListSort (
                    psObj->hChannelList,
                    (OSAL_LL_COMPARE_HANDLER)CHANNEL_n16CompareAlternateIds );

                if ( eReturnCode == OSAL_SUCCESS )
                {
                    // Update the event mask
                    SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

                    // Notify registered parties
                    SMSU_bNotify(&psObj->sEvent);
                }
            }
        }
    }
    return;
}

/*****************************************************************************
*
*   bSerializationIterator
*
*****************************************************************************/
static BOOLEAN bSerializationIterator (
    void * pvData,
    void * pvArg
        )
{
    N32 n32Value;
    TAG_OBJECT hServiceIdTag;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bStop = FALSE, bContinue = TRUE;

    CHANNEL_OBJECT hChannel =
        (CHANNEL_OBJECT)pvData;
    CATEGORY_SERIALIZATION_STRUCT *psData =
        (CATEGORY_SERIALIZATION_STRUCT *)pvArg;

    // Verify inputs
    if ((pvData == NULL) || (pvArg == NULL))
    {
        return bStop;
    }

    if (psData->un32Channels > 0)
    {
        hServiceIdTag = psData->hNextSiblingTag;

        eReturnCode = TAG_eNextSibling(
            psData->hNextSiblingTag, 
            &psData->hNextSiblingTag
                );

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            psData->bSuccess = FALSE;
            return bStop;
        }

        psData->un32Channels--;
    }
    else
    {
        // Add new Channel Tag
        eReturnCode = TAG_eAdd(
            CATEGORY_OBJECT_SERVICE_ID_TAG,
            psData->hChannelsTag,
            &hServiceIdTag,
            (char *)NULL);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            psData->bSuccess = FALSE;
            return bStop;
        }
    }

    // Get Service Id
    n32Value = (N32)CHANNEL.tServiceId(hChannel);

    // Set Service Id to Tag
    eReturnCode = TAG_eSetTagValue(
        hServiceIdTag,
        TAG_TYPE_INTEGER,
        (void *)&n32Value,
        sizeof(N32),
        FALSE);

    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psData->bSuccess = FALSE;
        return bStop;
    }

    // Commit request
    psData->bCommitMe = TRUE;

    return bContinue;
}

/*****************************************************************************
*
*   bRestoreIterator
*
*****************************************************************************/
static BOOLEAN bRestoreIterator (
    TAG_OBJECT hTag,
    void * pvArg
        )
{
    size_t tBytesAvail = sizeof(N32);
    SERVICE_ID tServiceId;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    N32 n32Value, * pn32Value = &n32Value;
    BOOLEAN bStop = FALSE, bContinue = TRUE;
    CATEGORY_SERIALIZATION_STRUCT *psData =
        (CATEGORY_SERIALIZATION_STRUCT *)pvArg;

    // Verify inputs
    if ((hTag == TAG_INVALID_OBJECT) || (pvArg == NULL))
    {
        return bStop;
    }

    // Get Tag value
    eReturnCode = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER, (void **)&pn32Value, &tBytesAvail);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        psData->bSuccess = FALSE;
        return bStop;
    }

    // Assign Service Id value
    tServiceId = (SERVICE_ID)n32Value;

    // Get Channel handle from Channel Cache
    psData->phChannel[psData->tIndex] = CCACHE_hChannelFromIds(psData->hCCache, tServiceId, CHANNEL_INVALID_ID, TRUE);
    if (psData->phChannel[psData->tIndex] == CHANNEL_INVALID_OBJECT)
    {
        psData->bSuccess = FALSE;
        return bStop;
    }

    // Increment index
    psData->tIndex++;

    return bContinue;
}

/*****************************************************************************
*
*   eSortDirect
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSortDirect(
    CATEGORY_OBJECT_STRUCT *psObj,
    CATEGORY_SORT_HANDLER n16SortFxn
	    )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        OSAL_RETURN_CODE_ENUM eOsalCode;

        // Callers may not sort while in the category
        // iteration callback
        if (psObj->sIterationCtrl.hCurrentEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Can't sort while we're iterating!
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        eOsalCode = OSAL.eLinkedListSort(
            psObj->hChannelList,
            (OSAL_LL_COMPARE_HANDLER)n16SortFxn
                );

        if (eOsalCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CATEGORY_OBJECT_NAME": Failed to sort category: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eOsalCode), eOsalCode);
            break;
        }

        // Update the event mask
        SMSU_tUpdate(&psObj->sEvent, CATEGORY_OBJECT_EVENT_CONTENTS);

        // Notify registered parties
        SMSU_bNotify(&psObj->sEvent);

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bBlockNotifications
*
*****************************************************************************/
static BOOLEAN bBlockNotifications(
    CATEGORY_OBJECT hCategory
        )
{
    BOOLEAN bBlocked = FALSE;

    CATEGORY_OBJECT_STRUCT *psObj = 
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // We can only block further category updates (notifications)
    // if our filter holding element is ALL. Otherwise this means
    // someone else has already filtered something in which case
    // this wont need to do anything anyway.
    if (psObj->tFilter == CATEGORY_OBJECT_EVENT_ALL)
    {
        // Record current filter (if any) and filter all further
        // notifications. We record it here so we can restore it later.
        psObj->tFilter =
            SMSU_tFilter(&psObj->sEvent, CATEGORY_OBJECT_EVENT_ALL);

        // Indicate category notifications are blocked
        bBlocked = TRUE;
    }

    return bBlocked;
}

/*****************************************************************************
*
*   vReleaseNotifications
*
*****************************************************************************/
static void vReleaseNotifications(
    CATEGORY_OBJECT hCategory
        )
{
    CATEGORY_OBJECT_STRUCT *psObj = 
        (CATEGORY_OBJECT_STRUCT *)hCategory;

    // Restore previous filter and process any pending notifications.

    // We will only release notifications if we actually have some
    // previously stored filter (other than ALL) otherwise don't
    // do anything.
    if (psObj->tFilter != CATEGORY_OBJECT_EVENT_ALL)
    {
        SMSAPI_EVENT_MASK tRestoreFilter = psObj->tFilter;

        // First thing we do is un-filter everything and record the
        // previous filter mask (should be ALL)
        // This will also return our "filter hold" to ALL
        psObj->tFilter =
            SMSU_tUnFilter(&psObj->sEvent, CATEGORY_OBJECT_EVENT_ALL);

        // Now we filter what was filtered before we got to it.
        SMSU_tFilter(&psObj->sEvent, tRestoreFilter);
    }

    // Process any pending notifications
    SMSU_bNotify(&psObj->sEvent);

    return;
}
