/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CAL_CONTENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "cid_obj.h"
#include "cas.h"
#include "cal_alert.h"
#include "cal_content.h"
#include "_cal_content.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eRemove
*
*   This API is used to remove a CAL_CONTENT obj from its CAL.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM -
*       SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemove(
    CAL_CONTENT_OBJECT hContent
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        CAL_OBJECT hCAL = CAL_INVALID_OBJECT;
        BOOLEAN bLocked;

        bLocked = SMSO_bLock((SMS_OBJECT)hContent, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // All CAL_CONTENT objects belong to a CAL object (their parent)
            hCAL = (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)hContent);
            SMSO_vUnlock((SMS_OBJECT)hContent);
        }

        if(hCAL == CAL_INVALID_OBJECT)
        {
            // Not a valid handle
            return SMSAPI_RETURN_CODE_ERROR;
        }

        // unregister the content
        eReturn = CEML.eRemove(psObj->hCEML);
    }

    return eReturn;
}

/*****************************************************************************
*
*   hCID
*
*   This API is used to retrieve the CID of this CAL_CONTENT obj.
*   Only valid when iterating a CAL.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT.
*
*   Outputs:
*
*   CID_OBJECT
*
*****************************************************************************/
static CID_OBJECT hCID (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    CID_OBJECT hCID = CID_INVALID_OBJECT;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        hCID = CEML.hCID(psObj->hCEML);
    }

    return hCID;
}

/*****************************************************************************
*
*   eSearchForCurrent
*
*   This API is used to search for registered content currently being
*   broadcast
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM -
*       SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSearchForCurrent (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        eReturnCode = CEML.eSearchForCurrent(psObj->hCEML);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eState
*
*   This API is used to retrieve the state of this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT.
*
*   Outputs:
*
*   CAL_CONTENT_STATE_ENUM
*
*****************************************************************************/
static CAL_CONTENT_STATE_ENUM eState (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    CAL_CONTENT_STATE_ENUM eState = CAL_CONTENT_STATE_INVALID;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        eState = psObj->eState;
    }

    return eState;
}

/*****************************************************************************
*
*   pvContentArg
*
*   This API is used to retrieve the content argument for this
*   CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT to enable content.
*
*   Outputs:
*
*   none
*
*****************************************************************************/
static void *pvContentArg (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    void *pvContentArg = NULL;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        pvContentArg = psObj->pvContentArg;
    }

    return pvContentArg;
}

/*****************************************************************************
*
*   tOptions
*
*   This API is used to retrieve the options for this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT to enable content.
*
*   Outputs:
*
*   CAL_CONTENT_OPTIONS
*
*****************************************************************************/
static UN32 un32Options (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    UN32 un32Options = SMS_EVENT_REGISTRATION_OPTION_NONE;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        un32Options = psObj->un32Options;
    }

    return un32Options;
}

/*****************************************************************************
*
*   bEnabled
*
*   This API is used to retrieve the enabled status of this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT.
*
*   Outputs:
*
*   TRUE if Enabled. FALSE if Disabled
*
*****************************************************************************/
static BOOLEAN bEnabled (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid, bEnabled = FALSE;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        bEnabled = psObj->bEnabled;
    }

    return bEnabled;
}

/*****************************************************************************
*
*   eEnable
*
*   This API is used to enable alerts for this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT to enable content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable(
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;

    // verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // invalid object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    psObj->bEnabled = TRUE;
    vUpdateState(psObj, CAL_CONTENT_STATE_SEARCHING);
    printf(CAL_CONTENT_OBJECT_NAME": Content ENABLED %p\n", psObj);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eDisable
*
*   This API is used to disable alerts for this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT to disable content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable(
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // invalid object
        return SMSAPI_RETURN_CODE_ERROR;
    }

    psObj->bEnabled = FALSE;
    vUpdateState(psObj, CAL_CONTENT_STATE_DISABLED);
    printf(CAL_CONTENT_OBJECT_NAME": Content DISABLED %p\n", psObj);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   hArtistText
*
*   This API is used to retrieve the artist for this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT
*
*   Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hArtistText (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    STRING_OBJECT hArtistText = STRING_INVALID_OBJECT;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        hArtistText = psObj->hArtistText;
    }

    return hArtistText;
}

/*****************************************************************************
*
*   hTitleText
*
*   This API is used to retrieve the title for this CAL_CONTENT obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT
*
*   Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hTitleText (
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid;
    STRING_OBJECT hTitleText = STRING_INVALID_OBJECT;

    // Verify SMS Object (CAL_CONTENT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);

    if (bValid == TRUE)
    {
        hTitleText = psObj->hTitleText;
    }

    return hTitleText;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CAL_CONTENT_hCreate
*
* Create a CAL_CONTENT object (CAL_CONTENT).
*
* Inputs:
*
*    pvContentArg - A pointer to an anonymous type (defined by the caller)
*       which will be associated with this registered content and
*       provided to the caller when the content event occurs or when
*       this content is iterated.
*    tOptions - Options provided by the caller about this registration
*       such as initial, end, etc.
*    hArtistText - A handle to a STRING obect containing artist text to be
*       associated with this registered content.
*    hTitleText - A handle to a STRING obect containing title text to be
*       associated with this registered content.
*    hCAL - A handle to a valid Content Alert List for which to
*       register this content with.
*    hCEML - A handle to a valid Content Event Monitor List which belongs
*       to the CAL this content is being registered with .
*
* Outputs:
*
*   A valid CAL_CONTENT_OBJECT on success.
*   CAL_CONTENT_INVALID_OBJECT on error.
*
*****************************************************************************/
CAL_CONTENT_OBJECT CAL_CONTENT_hCreate (
    void *pvContentArg,
    UN32 un32Options,
    STRING_OBJECT hArtistText,
    STRING_OBJECT hTitleText,
    CAL_OBJECT hCAL,
    CEML_OBJECT hCEML
        )
{
    CAL_CONTENT_OBJECT_STRUCT *psObj = NULL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturn;
    STRING_OBJECT hArtistTextCopy, hTitleTextCopy;

    // copy the string so we can later destroy it
    hArtistTextCopy = STRING_hDuplicate(SMS_INVALID_OBJECT, hArtistText);
    if ((hArtistTextCopy == STRING_INVALID_OBJECT) &&
        (hArtistText != STRING_INVALID_OBJECT))
    {
        // failed to create ArtistTextCopy
        return CAL_CONTENT_INVALID_OBJECT;
    }

    hTitleTextCopy = STRING_hDuplicate(SMS_INVALID_OBJECT, hTitleText);
    if ((hTitleTextCopy == STRING_INVALID_OBJECT) &&
        (hTitleText != STRING_INVALID_OBJECT))
    {
        // failed to create TitleTextCopy
        STRING_vDestroy(hArtistTextCopy);
        return CAL_CONTENT_INVALID_OBJECT;
    }

    do
    {
        // Create a CAL_CONTENT object
        psObj = (CAL_CONTENT_OBJECT_STRUCT *)
                    SMSO_hCreate(
                        CAL_CONTENT_OBJECT_NAME":Object",
                        sizeof(CAL_CONTENT_OBJECT_STRUCT),
                        (SMS_OBJECT)hCAL, // Child
                        FALSE); // No Lock (ignored)

        if(psObj == NULL)
        {
            // Error!
            STRING_vDestroy(hArtistTextCopy);
            STRING_vDestroy(hTitleTextCopy);
            break;
        }

        // Create a linked list for alerts

        // Construct an appropriate name for list to be created.
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 CAL_CONTENT_OBJECT_NAME":AL");

        // Create the alert list
        eReturn =
            OSAL.eLinkedListCreate(
                &psObj->hAlertList,
                &acName[0],
                NULL,
                OSAL_LL_OPTION_LINEAR
                    );
        if(eReturn != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // by default we are searching
        psObj->eState = CAL_CONTENT_STATE_SEARCHING;

        // by default we are enabled
        psObj->bEnabled = TRUE;

        psObj->hCEML = hCEML;

        psObj->un32Options = un32Options;
        psObj->hArtistText = hArtistTextCopy;
        psObj->hTitleText = hTitleTextCopy;
        psObj->pvContentArg = pvContentArg;

        printf(CAL_CONTENT_OBJECT_NAME": created content %p\n", psObj);

        // Return CAL object to caller
        return (CAL_CONTENT_OBJECT)psObj;

    } while(0);

    CAL_CONTENT_vDestroy((CAL_CONTENT_OBJECT)psObj);

    return CAL_CONTENT_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vDestroy
*
* Destroys a CAL_CONTENT object
*
* Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT object for which to
*              destroy and release all it's resources.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
void CAL_CONTENT_vDestroy (
    CAL_CONTENT_OBJECT hContent
        )
{
    BOOLEAN bValid;
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // Unable to verify the provided object
        return;
    }

    // Check if the alert list exists.
    if(psObj->hAlertList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list and destroy each content entry
        OSAL.eLinkedListRemoveAll(
            psObj->hAlertList,
            (OSAL_LL_RELEASE_HANDLER)CAL_ALERT_vDestroy
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->hAlertList);
        psObj->hAlertList = OSAL_INVALID_OBJECT_HDL;
    }

    psObj->hCEML = CEML_INVALID_OBJECT;

    if (psObj->hArtistText != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hArtistText);
        psObj->hArtistText = STRING_INVALID_OBJECT;
    }

    if (psObj->hTitleText != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTitleText);
        psObj->hTitleText = STRING_INVALID_OBJECT;
    }

    // Destroy the object itself
    SMSO_vDestroy((SMS_OBJECT)psObj);

    puts(CAL_CONTENT_OBJECT_NAME": destroyed content");

    return;
}

/*****************************************************************************
*
*   CAL_CONTENT_eAddToAlertList
*
* Add an Alert to the Content Object's AlertList
*
* Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT object for which to
*              add the alert.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success. SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_CONTENT_eAddToAlertList (
    CAL_CONTENT_OBJECT hContent,
    CAL_ALERT_OBJECT hAlert
        )
{
    BOOLEAN bValid;
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // invalid object
        return  SMSAPI_RETURN_CODE_ERROR;
    }

    // add the alert to the alert list
    eOsalReturnCode = OSAL.eLinkedListAdd(
                          psObj->hAlertList,
                          OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                          hAlert
                              );

    if(eOsalReturnCode == OSAL_SUCCESS)
    {
        // we're now active, since an alert was just created for us.
        vUpdateState(psObj, CAL_CONTENT_STATE_ACTIVE);
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        puts(CAL_CONTENT_OBJECT_NAME": Added an alert.");
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CAL_CONTENT_hGetAlert
*
* Get an Alert from the Content Object's AlertList
*
* Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT object for which to
*              add the alert.
*   hChannel - the channel which the alert is on
*
* Outputs:
*
*   valid CAL_ALERT_OBJECT on success. CAL_ALERT_INVALID_OBJECT on error.
*
*****************************************************************************/
CAL_ALERT_OBJECT CAL_CONTENT_hGetAlert (
    CAL_CONTENT_OBJECT hContent,
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturn;
    BOOLEAN bValid;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY; // search from start(top) of list
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    CAL_ALERT_OBJECT hAlert;

     // Verify SMS Object
    bValid =
        SMSO_bIsValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // invalid object
        return  CAL_ALERT_INVALID_OBJECT;
    }

    // search the list
    eOsalReturn = OSAL.eLinkedListLinearSearch(
        psObj->hAlertList,
        &hEntry,
        n16CompareChannel,
        (void *)(CHANNEL_OBJECT)hChannel);

    // found it?
    if (eOsalReturn == OSAL_SUCCESS)
    {
        // extract it
        hAlert = (CAL_ALERT_OBJECT)OSAL.pvLinkedListThis(hEntry);
        // return it
        return hAlert;
    }

    // The Alert ain't here!
    return CAL_ALERT_INVALID_OBJECT;
}

/*****************************************************************************
*
*   CAL_CONTENT_eRemoveFromAlertList
*
* Remove an Alert from the Content Object's AlertList
*
* Inputs:
*
*   hContent - A handle to a valid CAL_CONTENT object for which to
*              remove the alert.
*   hChannel - the channel which the alert is on
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success. SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_CONTENT_eRemoveFromAlertList (
    CAL_CONTENT_OBJECT hContent,
    CHANNEL_OBJECT hChannel
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturn;
    BOOLEAN bValid;
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

     // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == FALSE)
    {
        // invalid object
        return  SMSAPI_RETURN_CODE_ERROR;
    }

    // search the list
    eOsalReturn = OSAL.eLinkedListLinearSearch(
        psObj->hAlertList,
        &hEntry,
        n16CompareChannel,
        (void *)(CHANNEL_OBJECT)hChannel);

    // found it?
    if (eOsalReturn == OSAL_SUCCESS)
    {
        OSAL.eLinkedListRemove(hEntry);
    }

    puts(CAL_CONTENT_OBJECT_NAME": Removed an alert");

    vUpdateState(psObj, CAL_CONTENT_STATE_SEARCHING);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   CAL_CONTENT_bLookForChannelInAlertList
*
*****************************************************************************/
BOOLEAN CAL_CONTENT_bLookForChannelInAlertList(
	CAL_CONTENT_OBJECT hContent,
	CHANNEL_OBJECT hChannel
	    )
{
	OSAL_RETURN_CODE_ENUM eOsalReturn;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    CAL_CONTENT_OBJECT_STRUCT *psObj =
        (CAL_CONTENT_OBJECT_STRUCT *)hContent;
    BOOLEAN bValid, bFound = FALSE;

	// Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if(bValid == TRUE)
    {
        eOsalReturn = OSAL.eLinkedListLinearSearch(
			psObj->hAlertList,
			&hEntry,
			n16CompareChannel,
			(void *)(CHANNEL_OBJECT)hChannel);

		// did we find it?
		if (eOsalReturn == OSAL_SUCCESS)
		{
			// found the channel in the list
			bFound = TRUE;
		}
	}

	return bFound;
}

/*****************************************************************************
*
*   CAL_CONTENT_eUpdateContentArg
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_CONTENT_eUpdateContentArg (
    CAL_CONTENT_OBJECT hContent,
    void *pvContentArg
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Verify SMS Object
    bValid =
        SMSO_bValid((SMS_OBJECT)hContent);
    if (bValid != TRUE)
    {
        // invalid object
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }
    else 
    {
        CAL_CONTENT_OBJECT_STRUCT *psObj =
            (CAL_CONTENT_OBJECT_STRUCT *)hContent;
        psObj->pvContentArg = pvContentArg;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    }
    return eReturnCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vUpdateState
*
*****************************************************************************/
static void vUpdateState (
    CAL_CONTENT_OBJECT_STRUCT *psObj,
    CAL_CONTENT_STATE_ENUM eNextState
        )
{
    CAL_CONTENT_STATE_ENUM eCurrentState;

    // Populate with new state information...
    eCurrentState = psObj->eState;

    switch(eCurrentState)
    {
        case CAL_CONTENT_STATE_DISABLED:
        {
            switch(eNextState)
            {
                case CAL_CONTENT_STATE_ACTIVE:
                {
                    // can't transition to from DISABLED to ACTIVE
                }
                break;

                case CAL_CONTENT_STATE_SEARCHING:
                {
                    if (psObj->bEnabled == TRUE)
                    {
                        // Update state
                        psObj->eState = eNextState;
                    }
                }
                break;

                default:
                    // Stay here, ignore
                break;
            }
        }
        break;

        case CAL_CONTENT_STATE_SEARCHING:
        {
            switch(eNextState)
            {
                case CAL_CONTENT_STATE_ACTIVE:
                {
                    // Update state
                    psObj->eState = eNextState;
                }
                break;

                case CAL_CONTENT_STATE_DISABLED:
                {
                    // Update state
                    psObj->eState = eNextState;
                }
                break;

                default:
                    // Stay here, ignore
                break;
            }
        }
        break;

        case CAL_CONTENT_STATE_ACTIVE:
        {
            switch(eNextState)
            {
                case CAL_CONTENT_STATE_DISABLED:
                {
                    // Update state
                    psObj->eState = eNextState;
                }
                break;

                case CAL_CONTENT_STATE_SEARCHING:
                {
                    UN32 un32NumItemsInLL;

                    // see if any alerts are remaining
                    OSAL.eLinkedListItems(psObj->hAlertList, &un32NumItemsInLL);

                    if (un32NumItemsInLL == 0)
                    {
                        // Update state
                        psObj->eState = eNextState;
                    }
                    else
                    {
                        // there must be more alerts active,
                        // so don't switch state
                        printf(CAL_CONTENT_OBJECT_NAME": There are %u items "
                               "left in the alert list\n", un32NumItemsInLL);
                    }
                }
                break;

                default:
                    // Stay here, ignore
                break;
            }
        }
        break;

        case CAL_CONTENT_STATE_INVALID:
        default:
        {
            // Unknown state
            switch(eNextState)
            {
                default:
                    // Stay here
                break;
            }
        }
        break;
    }

    // Process channel update info...
    printf("%s: Content changed state to %s\n",
        CAL_CONTENT_OBJECT_NAME,
        (psObj->eState == CAL_CONTENT_STATE_DISABLED) ?
            MACRO_TO_STRING(CAL_CONTENT_STATE_DISABLED)
      : (psObj->eState == CAL_CONTENT_STATE_ACTIVE) ?
          MACRO_TO_STRING(CAL_CONTENT_STATE_ACTIVE)
      : (psObj->eState == CAL_CONTENT_STATE_SEARCHING) ?
          MACRO_TO_STRING(CAL_CONTENT_STATE_SEARCHING)
      : (psObj->eState == CAL_CONTENT_STATE_INVALID) ?
          MACRO_TO_STRING(CAL_CONTENT_STATE_INVALID)
      : "???"
          );


    return;
}

/*****************************************************************************
*
*   n16CompareChannel
*
*****************************************************************************/
static N16 n16CompareChannel (
    void *pvArg1,
    void *pvArg2
        )
{
    CAL_ALERT_OBJECT hAlert =
        (CAL_ALERT_OBJECT)pvArg1;
    CHANNEL_OBJECT hChannel = (CHANNEL_OBJECT)pvArg2;
    N16 n16Result = N16_MIN;

    if (hAlert != CAL_ALERT_INVALID_OBJECT)
    {
        CHANNEL_OBJECT hThisAlertsChannel;
        hThisAlertsChannel = CAL_ALERT.hChannel(hAlert);

        if (hThisAlertsChannel == hChannel)
        {
            n16Result = 0; // Match!
        }
        else
        {
            n16Result = -1; // No match!
        }
    }

    return n16Result;
}

