/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the BAUDOT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdio.h>
#include <ctype.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "_baudot.h"
#include "_baudot_codes.h"
#include "baudot.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hToString
*
* Allows the caller to create a new STRING object containing ANSI characters
* translated from a string encoded using Baudot characters found within an
* OSAL_BUFFER.
*
*****************************************************************************/
static STRING_OBJECT hToString (
    OSAL_BUFFER_HDL hBuffer,
    BAUDOT_BEHAVIOR_ENUM eProcessBehavior,
    BOOLEAN bStringStartsWithLetters,
    BOOLEAN bStringStartsWithCaps,
    size_t tMaxBaudotSymbolsToRead,
    size_t *ptNumSymbolsFound
        )
{
    // Translate the Baudot characters
    // in hBuffer to an ANSI string
    // contained within a STRING_OBJECT
    return hTranslateBaudotToANSI(
        SMS_INVALID_OBJECT,
        hBuffer, eProcessBehavior,
        bStringStartsWithLetters,
        bStringStartsWithCaps,
        tMaxBaudotSymbolsToRead,
        ptNumSymbolsFound );
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   BAUDOT_hToString
*
* This friend function allows an SMS-internal caller to create a new STRING
* object (with a specified SMS_OBJECT owner) containing ANSI characters
* translated from a string encoded using Baudot characters found within an
* OSAL_BUFFER.
*
*****************************************************************************/
STRING_OBJECT BAUDOT_hToString (
    SMS_OBJECT hStringOwner,
    OSAL_BUFFER_HDL hBuffer,
    BAUDOT_BEHAVIOR_ENUM eProcessBehavior,
    BOOLEAN bStringStartsWithLetters,
    BOOLEAN bStringStartsWithCaps,
    size_t tMaxBaudotSymbolsToRead,
    size_t *ptNumSymbolsFound
        )
{
    // Translate the Baudot characters
    // in hBuffer to an ANSI string
    // contained within a STRING_OBJECT
    return hTranslateBaudotToANSI(
        hStringOwner,
        hBuffer, eProcessBehavior,
        bStringStartsWithLetters,
        bStringStartsWithCaps,
        tMaxBaudotSymbolsToRead,
        ptNumSymbolsFound );
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hTranslateBaudotToANSI
*
*   This function performs all the work of translating a Baudot string into
*   an ANSI string.  Baudot characters are processed from within hBuffer
*   until we've read tMaxBaudotSymbolsToRead, or the END character is found,
*   or all of the contents of hBuffer have been exhausted.
*
* Inputs:
*
*   hStringOwner - An SMS_OBJECT to use as the resultant string's owner.
*   hBuffer - A valid OSAL_BUFFER_HDL containing a string of Baudot
*       characters which may or may not terminate with an END character.
*   eProcessBehavior - Indicates how the parser determines when to
*       stop processing.
*   bStringStartsWithLetters - A flag indicating if the Baudot string
*       found in hBuffer begins with a letter character (as defined by
*       Baudot) or something else.
*   bStringStartsWithCaps - A flag indicating if the Baudot string
*       found in hBuffer begins with a Capital letter
*   tMaxBaudotSymbolsToRead - The maximum number of Baudot charcters
*       the Baudot string may occupy.
*   *ptNumSymbolsFound - A valid pointer to a size_t which is populated
*       by this call with the number of Baudot characters consumed from
*       hBuffer in order to generate the STRING object from the Baudot
*       characters in the message.
*
* Outputs:
*
*   A valid STRING_OBJECT on success, otherwise STRING_INVALID_OBJECT
*
*****************************************************************************/
static STRING_OBJECT hTranslateBaudotToANSI (
    SMS_OBJECT hStringOwner,
    OSAL_BUFFER_HDL hBuffer,
    BAUDOT_BEHAVIOR_ENUM eProcessBehavior,
    BOOLEAN bStringStartsWithLetters,
    BOOLEAN bStringStartsWithCaps,
    size_t tMaxBaudotSymbolsToRead,
    size_t *ptNumSymbolsFound
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    BOOLEAN bCaps = bStringStartsWithCaps,
            bLetters = bStringStartsWithLetters,
            bEscape = FALSE,
            bLF = FALSE,
            bDone = FALSE,
            bUpdateNeeded = FALSE,
            bIncrementWriteIndex = TRUE;
    size_t tWriteIndex = 0;
    BAUDOT_CHAR tCharRead;
    char pacConverted[BAUDOT_BLOCK_SIZE + 1];
    BAUDOT_CHAR tBaudot;
    UN8 un8BitsRead;
    size_t tNumSymbolsFound = 0;

    // Initialize the buffer
    OSAL.bMemSet(&pacConverted[0], 0, sizeof(pacConverted));

    // Check inputs
    if (hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            BAUDOT_NAME": Invalid buffer handle\n");

        return STRING_INVALID_OBJECT;
    }

    // Check inputs
    if ((eProcessBehavior != BAUDOT_BEHAVIOR_PROCESS_TO_MAX_SYMBOLS) &&
        (eProcessBehavior != BAUDOT_BEHAVIOR_PROCESS_TO_END))
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            BAUDOT_NAME": Invalid value given for BAUDOT_BEHAVIOR_ENUM\n");

        return STRING_INVALID_OBJECT;
    }

    // If the caller wants us to process only to a certain number of
    // symbols then they better have given us a good value
    // for tMaxBaudotSymbolsToRead
    if ((eProcessBehavior == BAUDOT_BEHAVIOR_PROCESS_TO_MAX_SYMBOLS) &&
        (tMaxBaudotSymbolsToRead == 0))
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            BAUDOT_NAME": Invalid value given for tMaxBaudotSymbolsToRead\n");

        return STRING_INVALID_OBJECT;
    }

    // Inspect the buffer for size now to ensure
    // there's some data available inside
    if (OSAL.tBufferGetSizeInBits(hBuffer) == 0)
    {
        return STRING_INVALID_OBJECT;
    }

    while (bDone == FALSE)
    {
        if (bLF == FALSE)
        {
            if (tNumSymbolsFound == tMaxBaudotSymbolsToRead)
            {
                if (eProcessBehavior == BAUDOT_BEHAVIOR_PROCESS_TO_MAX_SYMBOLS)
                {
                    // The caller wants us to stop when we've reached
                    // their maximum specified number of symbols.  Don't
                    // worry if we've encountered the END character.
                    break;
                }
                else if (eProcessBehavior == BAUDOT_BEHAVIOR_PROCESS_TO_END)
                {
                    // In this mode, tMaxBaudotSymbolsToRead doesn't
                    // mean anything to us
                }
            }

            // Read the next baudot character
            tCharRead = (BAUDOT_CHAR)0;
            un8BitsRead = OSAL.tBufferReadHeadBits(
                hBuffer, &tCharRead, 0,
                BAUDOT_CHAR_BITWIDTH);

            if (un8BitsRead != BAUDOT_CHAR_BITWIDTH)
            {
                // Not enough data left to read next character
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    BAUDOT_NAME": BAUDOT buffer doesn't appear to be aligned correctly\n");
                break;
            }

            // Keep track of the number of
            // characters we're reading
            tNumSymbolsFound++;
        }

        if (bLF == TRUE)
        {
            // Write out the LF
            pacConverted[tWriteIndex] = 0x0a;
            bIncrementWriteIndex = TRUE;
            bLF = FALSE;
        }
        // Is this a valid baudot character entry index?
        else if (tCharRead < NUM_BAUDOT_ENTRIES)
        {
            // Indicate that we have new data and
            // need to update the string when we get
            // a chance to do so
            bUpdateNeeded = TRUE;

            if (bLetters == TRUE)
            {
                if (bCaps == TRUE)
                {
                    if (bEscape == FALSE)
                    {
                        tBaudot = gasChars[tCharRead].tUppercaseLetter;
                    }
                    else
                    {
                        tBaudot = gasChars[tCharRead].tEscapedUppercase;
                    }
                }
                else
                {
                    if (bEscape == FALSE)
                    {
                        tBaudot = gasChars[tCharRead].tLowercaseLetter;
                    }
                    else
                    {
                        tBaudot = gasChars[tCharRead].tEscapedLowercase;
                    }
                }
            }
            // Using the "Figures" Symbols
            else if (bEscape == FALSE)
            {
                tBaudot = gasChars[tCharRead].tFigure;
            }
            else
            {
                tBaudot = gasChars[tCharRead].tEscapedFigure;
            }

            // Disable the escape flag now
            bEscape = FALSE;

            // Don't go anywhere by default
            bIncrementWriteIndex = FALSE;

            // Is this is a BAUDOT control character?
            switch(tBaudot)
            {
                case BAUDOT_COMMAND_CAPS:
                {
                    bCaps = !bCaps;
                }
                break;

                case BAUDOT_COMMAND_RETURN:
                {
                    // Write out the CR
                    pacConverted[tWriteIndex] = 0x0d;

                    // We also need to write the LF character
                    bLF = TRUE;

                    // We need to move along
                    // in our output buffer
                    bIncrementWriteIndex = TRUE;
                }
                break;

                case BAUDOT_COMMAND_END:
                {
                    bDone = TRUE;
                }
                break;

                case BAUDOT_COMMAND_LF:
                case BAUDOT_COMMAND_FL:
                {
                    bLetters = !bLetters;
                }
                break;

                case BAUDOT_COMMAND_ESC:
                {
                    bEscape = TRUE;
                }
                break;

                default:
                {
                    pacConverted[tWriteIndex] = tBaudot;

                    // We need to move along
                    // in our output buffer
                    bIncrementWriteIndex = TRUE;
                }
                break;
            }
        }
        else if (tCharRead >= NUM_BAUDOT_ENTRIES)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                BAUDOT_NAME": Invalid symbol read from buffer: %u", tCharRead);
        }

        // Have we reached the end of the buffer?
        if (tWriteIndex == (BAUDOT_BLOCK_SIZE - 1))
        {
            // Yes we have.  It's time to
            // append or update the STRING

            // We're taking care of the string update now
            bUpdateNeeded = FALSE;

            // Update the string
            if (bUpdateString(hStringOwner, &hString, &pacConverted[0]) == FALSE)
            {
                break;
            }

            // Reset the write index
            tWriteIndex = 0;

            // Clear the read buffer
            OSAL.bMemSet(&pacConverted[0], 0, BAUDOT_BLOCK_SIZE);
        }
        else if (bIncrementWriteIndex == TRUE)
        {
            // Move on to the next character
            tWriteIndex++;
        }
    }

    // Do we need to perform a final string update?
    if (bUpdateNeeded == TRUE)
    {
        // Yep -- do it now and don't worry about the outcome
        bUpdateString(hStringOwner, &hString, &pacConverted[0]);
    }

    // Does the caller want to know how many bits
    // we've processed and consumed from the buffer?
    if (ptNumSymbolsFound != NULL)
    {
        *ptNumSymbolsFound = tNumSymbolsFound;
    }

    return hString;
}

/*****************************************************************************
*
*   bUpdateString
*
*   This function is called in order to update a string (given by *phString)
*   using the text found in pacNewText.  If no string exists at *phString,
*   one is created.  If the string does already exist, the new text is
*   appended at the end of it.
*
*****************************************************************************/
static BOOLEAN bUpdateString (
    SMS_OBJECT hStringOwner,
    STRING_OBJECT *phString,
    const char *pacNewText
        )
{
    printf(BAUDOT_NAME
        ": Appending ANSI string %s to STRING_OBJECT\n",
        pacNewText);

    // Is this a new string object?
    if (*phString == STRING_INVALID_OBJECT)
    {
        // Create a new one using the provided text
        *phString = STRING_hCreate(
            hStringOwner, pacNewText, strlen(pacNewText), 0);
        if (*phString == STRING_INVALID_OBJECT)
        {
            return FALSE;
        }
    }
    else // We need to append onto a pre-existing string
    {
        size_t tCharsAdded;

        // Append it now!
        tCharsAdded = STRING.tAppendCStr(*phString, pacNewText);

        if (tCharsAdded == 0)
        {
            return FALSE;
        }
    }

    return TRUE;
}
