/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AT_SEEK_CONTENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"

#include "decoder_obj.h"
#include "string_obj.h"

#include "cas.h"
#include "seek.h"
#include "at_seek.h"
#include "at_seek_content.h"
#include "_at_seek_content.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hRegisteredArtistText
*
* This API is used to retrieve the Artist/Title Seek Content's ArtistText.
*
* Inputs:
*
*   hSeekContent    A handle to a valid SEEK_CONTENT_OBJECT for which the
*                   caller wishes to query the RegisteredArtistText
*
* Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hRegisteredArtistText (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hArtistText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Artist/Title Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hArtistText = CAL_CONTENT.hArtistText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hArtistText;
}

/*****************************************************************************
*
*   hRegisteredTitleText
*
* This API is used to retrieve the Artist/Title Seek Content's Title Text.
*
*   hSeekContent    A handle to a valid SEEK_CONTENT_OBJECT for which the
*                   caller wishes to query the Registered Title Text
*
* Outputs:
*
*   A valid STRING_OBJECT or STRING_INVALID_OBJECT.
*
*****************************************************************************/
static STRING_OBJECT hRegisteredTitleText (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    STRING_OBJECT hTitleText = STRING_INVALID_OBJECT;

    // verify the content is associated with the Artist/Title Seek Service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        // get the text. A SEEK_CONTENT_OBJECT is really a CAL_CONTENT_OBJECT
        // so we can just call the appropriate CAL_CONTENT API
        hTitleText = CAL_CONTENT.hTitleText((CAL_CONTENT_OBJECT)hSeekContent);
    }
    return hTitleText;
}

/*****************************************************************************
*
*   eType
*
* This API is used to retrieve the artist/title seek type.
*
* Inputs:
*
*   hSeekContent    A handle to a valid Seek Content for which the
*                   caller wishes to query the content's seek type
*
* Outputs:
*
*   AT_SEEK_ENUM indicating the artist/title seek type
*
*****************************************************************************/
static AT_SEEK_ENUM eType (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{

    SEEK_SERVICE_ENUM eService;
    AT_SEEK_ENUM eType = AT_SEEK_UNKNOWN;

    // verify that this content is from an artist/title seek service
    // and that the caller is the owner of that service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psATSeekContentItem = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psATSeekContentItem != NULL )
        {
            // extract the type
            eType = psATSeekContentItem->eType;
        }
    }
    return eType;
}

/*****************************************************************************
*   bConvertible
*
* This API is used to query if a SEEK_CONTENT_OBJECT can be
* converted to a specified artist/title seek type. This function must be
* called from an iteration callback.
*
* Inputs:
*
*   hSeekContent    A handle to a valid Seek Content.
*   eConvertToType  The seek type the caller wants to know if the
*                   Seek Content can be converted to.
*
* Outputs:
*
*   TRUE if the item can be converted to the specified type. FALSE if it cannot.
*
*****************************************************************************/
static BOOLEAN bConvertible (
    SEEK_CONTENT_OBJECT hSeekContent,
    AT_SEEK_ENUM eConvertToType
        )
{
    BOOLEAN bReturn = FALSE;

    do
    {
        BOOLEAN bRegistered;
        CID_OBJECT hCID;
        AT_SEEK_ENUM eCurrentType;
        SEEK_SERVICE_ENUM eService;
        CAL_OBJECT hSeekList;

        eService = SEEK_CONTENT.eService(hSeekContent);
        // verify that this content is from an artist/title seek service
        // and that the caller is the owner of that service
        if ( eService != SEEK_SERVICE_ARTIST_TITLE)
        {
            bReturn = FALSE;
            break;
        }

        // is the registered item already of the type caller wants
        // to convert to?
        eCurrentType = AT_SEEK_CONTENT.eType(hSeekContent);
        if (eCurrentType == eConvertToType)
        {
            // yes it is
            bReturn = TRUE;
            break;
        }

        // find the cid that would be registered if converted
        hCID = hCIDForConversion(hSeekContent, eConvertToType);
        if (hCID == CID_INVALID_OBJECT)
        {
            // Not a valid handle - must not have sufficient information
            // to convert
            bReturn = FALSE;
            break;
        }

        hSeekList =  (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)hSeekContent);
        if(hSeekList == CAL_INVALID_OBJECT)
        {
            // Not a valid handle
            bReturn = FALSE;
            break;
        }

        // is this cid registered already?
        bRegistered = CAL.bExists(hSeekList, hCID);

        if (bRegistered == TRUE)
        {
            // the cid is already registered therefore we can't convert
            bReturn = FALSE;
            break;
        }

        // conversion is allowable
        bReturn = TRUE;
    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
*   eConvert
*
* This API is used to convert the currently iterated registered content item
* from one type of artist/title seek to another
* This function must be called from an iteration callback.
*
* Inputs:
*
*   hSeekContent    A handle to a valid Seek Content.
*   eConvertToType  The seek type the caller wants to the Seek Content
*                   converted to.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eConvert (
    SEEK_CONTENT_OBJECT hSeekContent,
    AT_SEEK_ENUM eConvertToType
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    STRING_OBJECT hArtistText, hTitleText;
    AT_SEEK_ENUM eOriginalType;
    SEEK_SERVICE_ENUM eService;
    CAL_OBJECT hSeekList;
    AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psConvertedItem = NULL, *psOriginalItem;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psOrig, *psConv = NULL;

    eService = SEEK_CONTENT.eService(hSeekContent);
    // verify that this content is from an artist/title seek service
    // and that the caller is the owner of that service
    if ( eService != SEEK_SERVICE_ARTIST_TITLE)
    {
        return SMSAPI_RETURN_CODE_WRONG_SEEK_SERVICE;
    }

    // get the type this content currently is
    eOriginalType = AT_SEEK_CONTENT.eType(hSeekContent);

    do
    {
        if (eOriginalType == eConvertToType)
        {
            // caller wants to convert to the type this item already is
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            break;
        }

        // get the cid that will be registered for converted content
        hCID = hCIDForConversion(hSeekContent, eConvertToType);
        if (hCID == CID_INVALID_OBJECT)
        {
            // must not have sufficient information to convert
            break;
        }

        // retrieve the original item's service specific content information
        psOrig = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        if (NULL == psOrig)
        {
            break;
        }

        psOriginalItem = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                             &psOrig->uServiceSpecific;

        // Create a item struct for the converted item.
        psConv = SEEK_CONTENT_psCreateRegisteredItem(psOrig->hSeekService, TRUE);
        if (NULL == psConv)
        {
            break;
        }

        psConvertedItem = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                              &psConv->uServiceSpecific;

        // Populate the service specific portion.
        psConvertedItem->hArtistCID = psOriginalItem->hArtistCID;
        psConvertedItem->hTitleCID = psOriginalItem->hTitleCID;
        psConvertedItem->eType = eConvertToType;

        // Get the artist and title text strings.
        hArtistText =
            CAL_CONTENT.hArtistText(hSeekContent);
        hTitleText =
            CAL_CONTENT.hTitleText(hSeekContent);

        // get the seek list this content is a member of
        hSeekList =  (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)hSeekContent);
        if(hSeekList == CAL_INVALID_OBJECT)
        {
            // Not a valid handle - Error!
            break;
        }

        // register the converted item with the CAL
        eReturnCode = CAL.eRegister(
            hSeekList,
            hCID,
            (void *)psConv,
            AT_SEEK_CAL_REGISTRATION_OPTIONS,
            hArtistText,
            hTitleText,
            NULL
                );
        if ( eReturnCode != SMSAPI_RETURN_CODE_SUCCESS )
        {
            // error
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                 AT_SEEK_CONTENT_OBJECT_NAME": couldn't register wth cal.");
            break;
        }

        // Set the original item's cids to invalid. This way the converted
        // item's cids won't be destroyed when the original item is removed.
        psOriginalItem->hArtistCID = CID_INVALID_OBJECT;
        psOriginalItem->hTitleCID = CID_INVALID_OBJECT;

        eReturnCode =
            AT_SEEK_eAddToConfig(psConv,
                hArtistText,
                hTitleText);

        // unregister original content
        if ( eReturnCode == SMSAPI_RETURN_CODE_SUCCESS )
        {
            eReturnCode = SEEK_CONTENT.eRemove(hSeekContent);
            puts(AT_SEEK_CONTENT_OBJECT_NAME": Conversion ok");
            break;
        }

    } while(0);

    // if we weren't successful, but have created resources we need
    // to destroy them
    if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) && (psConv != NULL))
    {
        //This isn't necessary from a code perspective, but it makes some static
        //error checkers happy.
        if (NULL != psConvertedItem)
        {
            // Set the converted item's cids to invalid. This way the original
            // item's cids won't be destroyed when the converted item is destroyed.
            psConvertedItem->hArtistCID = CID_INVALID_OBJECT;
            psConvertedItem->hTitleCID = CID_INVALID_OBJECT;
        }
        SEEK_CONTENT_vDestroyRegisteredItem(psConv);
    }

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   AT_SEEK_CONTENT_vSetServiceSpecificInfo
*
* This function is used to populate the AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT
*
*****************************************************************************/
void AT_SEEK_CONTENT_vSetServiceSpecificInfo(
    void *pvItem,
    CID_OBJECT hArtistCID,
    CID_OBJECT hTitleCID,
    AT_SEEK_ENUM eType,
    SEEK_SERVICE_OBJECT hSeekService
        )
{
    SEEK_SERVICE_ENUM eService;

    eService = SEEK_eService(hSeekService);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSpecificInfo;
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

        psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;
        psATSpecificInfo = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                               &psGenericItem->uServiceSpecific;

        // populate the service specific portion
        psATSpecificInfo->hArtistCID =
            CID.hDuplicate(hArtistCID);
        psATSpecificInfo->hTitleCID =
                CID.hDuplicate(hTitleCID);
        psATSpecificInfo->eType = eType;
    }

    return;
}

/*****************************************************************************
*
*   AT_SEEK_CONTENT_hArtistCID
*
*****************************************************************************/
CID_OBJECT AT_SEEK_CONTENT_hArtistCID (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hArtistCID = CID_INVALID_OBJECT;

    // verify the content is associated with the Artist/Title Seek Service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psATSeekContentItem = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psATSeekContentItem != NULL )
        {
            hArtistCID = psATSeekContentItem->hArtistCID;
        }
    }
    return hArtistCID;
}

/*****************************************************************************
*
*   AT_SEEK_CONTENT_hTitleCID
*
*****************************************************************************/
CID_OBJECT AT_SEEK_CONTENT_hTitleCID (
    SEEK_CONTENT_OBJECT hSeekContent
        )
{
    SEEK_SERVICE_ENUM eService;
    CID_OBJECT hTitleCID = CID_INVALID_OBJECT;

    // verify the content is associated with the Artist/Title Seek Service
    eService = SEEK_CONTENT.eService(hSeekContent);
    if (eService == SEEK_SERVICE_ARTIST_TITLE)
    {
        SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisteredItem;
        AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSeekContentItem;

        psRegisteredItem = SEEK_CONTENT_psRegisteredItem(hSeekContent);
        psATSeekContentItem = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                                  &psRegisteredItem->uServiceSpecific;

        // verify that the pointer to the data isn't NULL
        if ( psATSeekContentItem != NULL )
        {
            hTitleCID =  psATSeekContentItem->hTitleCID;
        }
    }
    return hTitleCID;
}

/*****************************************************************************
*
*    AT_SEEK_CONTENT_pacATSeekTypeText
*
*    This is a local function which simply maps an enumerated type to
*    a textual representation for formatting the enumerated type.
*
*****************************************************************************/
const char *AT_SEEK_CONTENT_pacATSeekTypeText( AT_SEEK_ENUM eType )
{
    const char *pacReturnString;

    // based on the seek state, get the text that describes it
    switch (eType)
    {
        case AT_SEEK_ARTIST:
            pacReturnString = MACRO_TO_STRING(AT_SEEK_ARTIST);
        break;

        case AT_SEEK_TITLE:
            pacReturnString = MACRO_TO_STRING(AT_SEEK_TITLE);
        break;

        case AT_SEEK_UNKNOWN:
        default:
            pacReturnString = MACRO_TO_STRING(AT_SEEK_UNKNOWN);
        break;
    }

    return pacReturnString;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n32FPrintf
*
*    This is used to print out to a file human readable service specific
*    information about the items registered to the seek service
*
*****************************************************************************/
static N32 n32FPrintf (
    void *pvServiceSpecificData,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32NumWritten = 0;
    SEEK_CONTENT_UNION *puServiceSpecificData =
            (SEEK_CONTENT_UNION *)pvServiceSpecificData;

    AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem =
            (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT*)puServiceSpecificData;

    // check input
    if ((pvServiceSpecificData == NULL) || (psFile==NULL))
    {
        return EOF;
    }

    switch(eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            // print out what type of artist / title seek this is
            n32NumWritten +=
                    fprintf(psFile, "%s",
                        AT_SEEK_CONTENT_pacATSeekTypeText(psRegItem->eType));
        }
        break;

        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        default:
        {
            // print out what type of artist / title seek this is
            n32NumWritten +=
                    fprintf(psFile, "%s\n",
                        AT_SEEK_CONTENT_pacATSeekTypeText(psRegItem->eType));

            if (eOutputOption == SMS_OUTPUT_OPTION_GROSS)
            {
                // print out the Artist CID
                n32NumWritten +=
                        fprintf(psFile, "Artist ");
                n32NumWritten +=
                        CID.n32FPrintf(psRegItem->hArtistCID, psFile);

                // print out the Title CID
                n32NumWritten += fprintf(psFile, "Title ");
                n32NumWritten +=
                        CID.n32FPrintf(psRegItem->hTitleCID, psFile);
            }
        }
        break;
    }

    return n32NumWritten;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    void *pvServiceSpecific
        )
{
    // check input
    if (pvServiceSpecific != NULL)
    {
        SEEK_CONTENT_UNION *puServiceSpecific =
            (SEEK_CONTENT_UNION *)pvServiceSpecific;

        AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegItem =
            (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT*)puServiceSpecific;

        // destroy the CIDs
        CID.vDestroy(psRegItem->hArtistCID);
        CID.vDestroy(psRegItem->hTitleCID);

        // uninitialize
        psRegItem->hArtistCID = CID_INVALID_OBJECT;
        psRegItem->hTitleCID = CID_INVALID_OBJECT;

        psRegItem->eType = AT_SEEK_UNKNOWN;
    }
}

/*****************************************************************************
*
*   hCIDForConversion
*
*****************************************************************************/
static CID_OBJECT hCIDForConversion(
    SEEK_CONTENT_OBJECT hSeekContent,
    AT_SEEK_ENUM eConvertToType
        )
{
    CID_OBJECT hCID = CID_INVALID_OBJECT;
    STRING_OBJECT hText = STRING_INVALID_OBJECT;

    do
    {
        BOOLEAN bValid;

        if (eConvertToType == AT_SEEK_TITLE)
        {
            // need text to provide information to user when alert occurs
            // it doesn't make sense to have a title alert, when the title
            // text isn't known
            hText = hRegisteredTitleText(hSeekContent);

            if (hText != STRING_INVALID_OBJECT)
            {
                // title text is known.

                // converting to title seek, so we need to use the title cid
                hCID = AT_SEEK_CONTENT_hTitleCID(hSeekContent);
            }

        }
        else if (eConvertToType == AT_SEEK_ARTIST)
        {
            // need text to provide information to user when alert occurs
            // it doesn't make sense to have a artist alert, when the artist
            // text isn't known
            hText = hRegisteredArtistText(hSeekContent);

            if (hText != STRING_INVALID_OBJECT)
            {
                // artist text is known.

                // converting to artist seek, so we need to use the artist cid
                hCID = AT_SEEK_CONTENT_hArtistCID(hSeekContent);
            }
        }
        else
        {
            // not a valid seek type.
            break;
        }

        // was the cid not valid?
        bValid = SMSO_bIsValid((SMS_OBJECT)hCID);
        if (bValid == FALSE)
        {
            hCID = CID_INVALID_OBJECT;
        }
    } while (FALSE);

    return hCID;
}

/*****************************************************************************
*
*   bSearchForCurrent
*
*****************************************************************************/
static BOOLEAN bSearchForCurrentAfterEnable (
    void *pvServiceSpecific
        )
{
    // we don't want to search for current
    return FALSE;
}
