/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AIR implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "_air_obj.h"
#include "air_obj.h"

#include "sms_api_debug.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eQuality
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eQuality (
    AIR_OBJECT hAir,
    AIR_QUALITY_ENUM *peAirQuality
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    AIR_OBJECT_STRUCT *psObj = (AIR_OBJECT_STRUCT *)hAir;

    do
    {
        if (peAirQuality == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }
        bOwner = SMSO_bOwner((SMS_OBJECT)hAir);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bQualityAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *peAirQuality = psObj->eQuality;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   ePollenCount
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePollenCount (
    AIR_OBJECT hAir,
    UN8 *pun8PollenCount
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    AIR_OBJECT_STRUCT *psObj = (AIR_OBJECT_STRUCT *)hAir;

    do
    {
        if (pun8PollenCount == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hAir);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bPollenAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pun8PollenCount = psObj->sData.un8Value;

    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   ePollenLevel
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePollenLevel (
    AIR_OBJECT hAir,
    AIR_POLLEN_LEVEL_ENUM *pePollenLevel
        )
{
    BOOLEAN bOwner = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;
    AIR_OBJECT_STRUCT *psObj = (AIR_OBJECT_STRUCT *)hAir;

    do
    {
        if (pePollenLevel == NULL)
        {
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hAir);
        if (bOwner == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bPollenAvailable == FALSE)
        {
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }
        eResult = SMSAPI_RETURN_CODE_SUCCESS;
        *pePollenLevel = psObj->sData.eLevel;
    } while (FALSE);

    return eResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AIR_OBJECT hAir,
    FILE *psFile
        )
{
    N32 n32Return = 0;
    AIR_OBJECT_STRUCT *psObj = (AIR_OBJECT_STRUCT *)hAir;
    BOOLEAN bOwner;

    // Determine if we own the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hAir);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }
    // Print AIR information header
    n32Return += fprintf(psFile, "\nAir (Object: 0x%X):\n",
                psObj);

    if (psObj->bQualityAvailable == TRUE)
    {
        const char *pacQuality;

        pacQuality = pacAirQuality(psObj->eQuality);
        n32Return += fprintf(psFile, "\tAir Quality: %s\n",
                   pacQuality);
    }

    if (psObj->bPollenAvailable == TRUE)
    {
        n32Return += fprintf(psFile, "\tPollen Count: %d, Pollen Level: %d\n",
                   psObj->sData.un8Value, psObj->sData.eLevel);
    }

    return n32Return;
}


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   AIR_hCreate
*
*****************************************************************************/
AIR_OBJECT AIR_hCreate(
    SMS_OBJECT hParent,
    BOOLEAN bQualityAvailable,
    AIR_QUALITY_ENUM eAir,
    BOOLEAN bPollenAvailable,
    POLLEN_UNPARSED_DATA tData
        )
{
    AIR_OBJECT_STRUCT *psObj =
        (AIR_OBJECT_STRUCT *)AIR_INVALID_OBJECT;

    // Create an instance of the AIR object
    psObj = (AIR_OBJECT_STRUCT *)
        SMSO_hCreate(
            AIR_OBJECT_NAME,
            sizeof(AIR_OBJECT_STRUCT),
            hParent,
            FALSE
                );

    if(psObj == NULL)
    {
        // Error!
        return AIR_INVALID_OBJECT;
    }

    // Initialize object per inputs
    psObj->bQualityAvailable = bQualityAvailable;
    psObj->bPollenAvailable = bPollenAvailable;
    psObj->eQuality = AIR_QUALITY_NO_DATA;

    if (bQualityAvailable == TRUE)
    {
        psObj->eQuality = eAir;
    }

    if (bPollenAvailable == TRUE)
    {
        psObj->bPollenAvailable = GsWeatherIntf.bProcessPollenData(tData, &psObj->sData);
    }

    return (AIR_OBJECT)psObj;
}

/*****************************************************************************
*
*   AIR_vDestroy
*
*****************************************************************************/
void AIR_vDestroy (
    AIR_OBJECT hAir
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hAir);
    if(bOwner == TRUE)
    {
        // Free object instance
        SMSO_vDestroy((SMS_OBJECT)hAir);
    }
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   pacAirQuality
*
*****************************************************************************/
static const char *pacAirQuality (
    AIR_QUALITY_ENUM eQuality
        )
{
    const char *pacQuality = "Unknown";

    switch (eQuality)
    {
        case AIR_QUALITY_GOOD:
        {
            pacQuality = "Good";
        }
        break;

        case AIR_QUALITY_MODERATE:
        {
            pacQuality = "Moderate";
        }
        break;

        case AIR_QUALITY_UNHEALTHY_FOR_SENS:
        {
            pacQuality = "Unhealthy for sensitive groups";
        }
        break;

        case AIR_QUALITY_UNHEALTHY:
        {
            pacQuality = "Unhealthy";
        }
        break;

        case AIR_QUALITY_VERY_UNHEALTHY:
        {
            pacQuality = "Very Unhealthy";
        }
        break;

        case AIR_QUALITY_DANGEROUS:
        {
            pacQuality = "Dangerous";
        }
        break;

        case AIR_QUALITY_NO_DATA:
        default:
        break;
    }

    return pacQuality;
}
