/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_TILE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "sms_version.h"
#include "string_obj.h"
#include "location_obj.h"
#include "dsrl_entry_obj.h"

#include "agw_tile_obj.h"
#include "_agw_tile_obj.h"
#include "agw_mgr_obj.h"

#ifdef SUPPORT_CUNIT
#include "sms_cunit.h"
#endif

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   eProductType
 *
 *****************************************************************************/
static AGW_PRODUCT_TYPE_ENUM eProductType (
        AGW_TILE_OBJECT hAgwTile
            )
{
    AGW_PRODUCT_TYPE_ENUM eResult = AGW_PRODUCT_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*) hAgwTile;

        eResult = psObj->ePublicProductType;
    }

    return eResult;
}

/*****************************************************************************
 *
 *   tTimeStamp
 *
 *****************************************************************************/
static TIME_T tTimeStamp (
    AGW_TILE_OBJECT hAgwTile
        )
{
    TIME_T tResult = 0;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*) hAgwTile;

        tResult = psObj->tTimeStamp;
    }

    return tResult;

}

/*****************************************************************************
 *
 *   hNorthWest
 *
 *****************************************************************************/
static LOCATION_OBJECT hNorthWest(
    AGW_TILE_OBJECT hAgwTile
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*) hAgwTile;

        hResult = psObj->hNorthWest;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   hSouthEast
 *
 *****************************************************************************/
static LOCATION_OBJECT hSouthEast(
    AGW_TILE_OBJECT hAgwTile
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*) hAgwTile;

        hResult = psObj->hSouthEast;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   hImage
 *
 *****************************************************************************/
static IMAGE_OBJECT hImage(
    AGW_TILE_OBJECT hAgwTile
        )
{
    IMAGE_OBJECT hResult = IMAGE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*) hAgwTile;

        hResult = psObj->hImage;
    }

    return hResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_TILE_OBJECT hAgwTile,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*)hAgwTile;
    N32 n32Return = EOF, n32Tmp;
    AGW_PRODUCT_TYPE_ENUM eProductType;
    const char *pacProductTypeName;
    TIME_T tTimeStamp;
    char acBuffer[OSAL_ASCBUFSIZE];
    BOOLEAN bOwner;

    do
    {
        // Check input
        if (psFile == NULL)
        {
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
        if (bOwner != TRUE)
        {
            break;
        }

        // Gather object attributes
        eProductType = AGW_TILE.eProductType(hAgwTile);
        tTimeStamp = AGW_TILE.tTimeStamp(hAgwTile);
        pacProductTypeName = AGW_MGR_pacGetProductTypeName(eProductType);
        OSAL.ctime_r(&tTimeStamp, acBuffer);

        // Print AGW_MSG information
        n32Tmp = fprintf(psFile,
                        "AGW_TILE Object: %p\n"
                        "\tProduct type = %s (%d)\n"
                        "\tTime Stamp = %s\n",
                        psObj,
                        pacProductTypeName,
                        eProductType,
                        &acBuffer[0]);
        if (n32Tmp <= 0)
        {
            break;
        }
        n32Return = n32Tmp;

        // Upper corner
        n32Tmp = fprintf(psFile, "\tUpper corner: ");
        if (n32Tmp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Tmp;

        n32Tmp = LOCATION.n32FPrintf(psObj->hNorthWest, psFile);
        if (n32Tmp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Tmp;

        // Upper corner
        n32Tmp = fprintf(psFile, "\tLower corner: ");
        if (n32Tmp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Tmp;

        n32Tmp = LOCATION.n32FPrintf(psObj->hSouthEast, psFile);
        if (n32Tmp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Tmp;

        // Image information
        n32Tmp = IMAGE_n32FPrintf(psObj->hImage, psFile);
        if (n32Tmp <= 0)
        {
            n32Return = EOF;
            break;
        }
        n32Return += n32Tmp;
    } while (FALSE);

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   AGW_TILE_hCreate
 *
 *****************************************************************************/
AGW_TILE_OBJECT AGW_TILE_hCreate(
    SMS_OBJECT hParent,
    const char *pacFilePath,
    IMAGE_FORMAT_ENUM eFormat,
    size_t tDescriptorSize
        )
{
    AGW_TILE_OBJECT_STRUCT *psObj = (AGW_TILE_OBJECT_STRUCT*) NULL;
    AGW_TILE_OBJECT hResult = AGW_TILE_INVALID_OBJECT;
    SMS_OBJECT hSMSObject;

    do
    {
        // Check inputs.
        if (pacFilePath == NULL)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        psObj = (AGW_TILE_OBJECT_STRUCT *)
            DSRL_ENTRY_hCreate(
                AGW_TILE_OBJECT_NAME": Obj",
                DSRL_ENTRY_TYPE_AGW_TILE,
                sizeof(AGW_TILE_OBJECT_STRUCT),
                tDescriptorSize,
                hParent,
                FALSE);

        if (psObj == (AGW_TILE_OBJECT_STRUCT*)DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME": failed to create DSRL_ENTRY"
                        );
            break;
        }

        // Init the data
        OSAL.bMemSet(&psObj->sBoundingBox, 0, sizeof(psObj->sBoundingBox));
        psObj->hSouthEast = LOCATION_INVALID_OBJECT;
        psObj->hNorthWest = LOCATION_INVALID_OBJECT;
        psObj->ePublicProductType = AGW_PRODUCT_TYPE_UNKNOWN;

        // Get real SMS_OBJECT from DSRL_ENTRY to use it as parent for
        // the image
        hSMSObject = DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT) psObj);

        // Create image object
        psObj->hImage = IMAGE_hCreate(hSMSObject, pacFilePath,
                            eFormat, &GsAgwTileImageIntf, NULL, FALSE);
        if (psObj->hImage == IMAGE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME": failed to create IMAGE_OBJECT"
                        );
            break;
        }

        // Populate to the caller
        hResult = (AGW_TILE_OBJECT) psObj;
    } while (FALSE);

    if ((hResult == AGW_TILE_INVALID_OBJECT) && (psObj != NULL))
    {
        AGW_TILE_vDestroy((AGW_TILE_OBJECT)psObj);
        psObj = NULL;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_TILE_bUpdate
 *
 *****************************************************************************/
BOOLEAN AGW_TILE_bUpdate(
    AGW_TILE_OBJECT hAgwTile,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_RASTER_PLANE_ENUM ePlane
        )
{
    AGW_TILE_OBJECT_STRUCT *psObj =
            (AGW_TILE_OBJECT_STRUCT*)hAgwTile;
    BOOLEAN bOk = FALSE;
    BOOLEAN bOwner = FALSE;
    AGW_TILE_IMAGE_DATA_STRUCT *psImageData;

    do
    {
        // Check inputs
        if ((hAgwTile == AGW_TILE_INVALID_OBJECT) || (psHeader == NULL))
        {
            break;
        }

        // Verify ownership of this object
        bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
        if (bOwner == FALSE)
        {
            break;
        }

        // Convert private product type input public one
        psObj->ePublicProductType =
                AGW_MGR_eGetPublicProductType(
                        psHeader->eProductType,
                        ePlane
                            );

        // Request image specific data
        psImageData =
            (AGW_TILE_IMAGE_DATA_STRUCT*) IMAGE_pvData(psObj->hImage);
        if (psImageData == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_TILE_OBJECT_NAME": failed to get image specific data");
            break;
        }

        // Update the AGW_TILE image specific data directly
        psImageData->un16Height = psHeader->un16Rows;
        psImageData->un16Width = psHeader->un16Columns;
        psImageData->un8BPP = psHeader->un8PixelDepth;

        // Update header data
        bOk = bUpdateTimeStamp(psObj, psHeader);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME ": failed to update timestamp\n");
            break;
        }

        // Update lower corner location
        bOk = bUpdateLocationObject(psObj, &psObj->hSouthEast,
                psHeader->n16LowerLat,
                psHeader->n16LowerLon
                    );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to update SouthEast location object\n");
            break;
        }

        // Update upper corner location
        bOk = bUpdateLocationObject(psObj, &psObj->hNorthWest,
                psHeader->n16UpperLat,
                psHeader->n16UpperLon
                    );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to update NorthWest location object\n");
            break;
        }

        // Update bounding box
        bOk = bUpdateBoundingBox(psObj);

    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_TILE_vDestroy
 *
 *****************************************************************************/
void AGW_TILE_vDestroy(
    AGW_TILE_OBJECT hAgwTile
        )
{
    BOOLEAN bOwner;

    printf(AGW_TILE_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwTile);

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj =
                (AGW_TILE_OBJECT_STRUCT*)hAgwTile;

        if (psObj->hNorthWest != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hNorthWest);
            psObj->hNorthWest = LOCATION_INVALID_OBJECT;
        }

        if (psObj->hSouthEast != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psObj->hSouthEast);
            psObj->hSouthEast = LOCATION_INVALID_OBJECT;
        }

        if (psObj->hImage != IMAGE_INVALID_OBJECT)
        {
            IMAGE_vDestroy(psObj->hImage);
            psObj->hImage = IMAGE_INVALID_OBJECT;
        }

        DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT)psObj);
    }

    return;
}


/*****************************************************************************
 *
 *   AGW_TILE_psBoundingBox
 *
 *****************************************************************************/
AGW_BOUNDING_BOX_STRUCT *AGW_TILE_psBoundingBox(
    AGW_TILE_OBJECT hAgwTile
        )
{
    BOOLEAN bOwner;
    AGW_BOUNDING_BOX_STRUCT *psResult = NULL;

    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile);
    if (bOwner == TRUE)
    {
        AGW_TILE_OBJECT_STRUCT *psObj;

        psObj = (AGW_TILE_OBJECT_STRUCT*)hAgwTile;

        psResult = &psObj->sBoundingBox;
    }

    return psResult;
}


/*****************************************************************************
 *
 *   AGW_TILE_n16CompareByCenter
 *
 *****************************************************************************/
N16 AGW_TILE_n16CompareByCenter(
    AGW_TILE_OBJECT hAgwTile1,
    AGW_TILE_OBJECT hAgwTile2
        )
{
    N16 n16Result = N16_MIN;
    BOOLEAN bOwner;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile1);
    if (bOwner == TRUE)
    {
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile2);
        if (bOwner == TRUE)
        {
            AGW_TILE_OBJECT_STRUCT *psTile1 =
                (AGW_TILE_OBJECT_STRUCT*)hAgwTile1;
            AGW_TILE_OBJECT_STRUCT *psTile2 =
                (AGW_TILE_OBJECT_STRUCT*)hAgwTile2;

            // Do comparison by LAT first and if they are equal for both
            // objects do LON comparison.
            n16Result = OSAL_FIXED.n16Compare(
                            psTile1->sBoundingBox.hCenterLat,
                            psTile2->sBoundingBox.hCenterLat);
            if (n16Result == 0)
            {
                n16Result = OSAL_FIXED.n16Compare(
                                psTile1->sBoundingBox.hCenterLon,
                                psTile2->sBoundingBox.hCenterLon);
            }
        }
    }

    return n16Result;
}


/*****************************************************************************
 *
 *   AGW_TILE_bEqualByLocation
 *
 *****************************************************************************/
BOOLEAN AGW_TILE_bEqualByLocation(
    AGW_TILE_OBJECT hAgwTile1,
    AGW_TILE_OBJECT hAgwTile2
        )
{
    AGW_TILE_OBJECT_STRUCT *psTile1 =
            (AGW_TILE_OBJECT_STRUCT*)hAgwTile1;
    AGW_TILE_OBJECT_STRUCT *psTile2 =
            (AGW_TILE_OBJECT_STRUCT*)hAgwTile2;
    BOOLEAN bResult = FALSE, bOwner;

    do
    {
        bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile1);
        if (bOwner == FALSE)
        {
            break;
        }

        bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwTile2);
        if (bOwner == FALSE)
        {
            break;
        }

        if ((psTile1->hNorthWest == LOCATION_INVALID_OBJECT) ||
            (psTile2->hNorthWest == LOCATION_INVALID_OBJECT))
        {
            break;
        }

        if ((psTile1->hSouthEast == LOCATION_INVALID_OBJECT) ||
            (psTile2->hSouthEast == LOCATION_INVALID_OBJECT))
        {
            break;
        }

        bResult = LOCATION_bCompare(psTile1->hNorthWest, psTile2->hNorthWest);
        if (bResult == FALSE)
        {
            break;
        }

        bResult = LOCATION_bCompare(psTile1->hSouthEast, psTile2->hSouthEast);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *                            PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hCreateLocationObject
 *
 *****************************************************************************/
static LOCATION_OBJECT hCreateLocationObject(
    AGW_TILE_OBJECT_STRUCT *psObj,
    N16 n16Lat,
    N16 n16Lon
        )
{
    LOCATION_OBJECT hResult = LOCATION_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLon = OSAL_FIXED_INVALID_OBJECT;

    do
    {
        // Check inputs
        if ((AGW_TILE_OBJECT)psObj == AGW_TILE_INVALID_OBJECT)
        {
            break;
        }

        // Compute the LAT
        hLat = AGW_MGR_hCreateFixedFromInteger(n16Lat, AGW_COORDINATE_POW10);
        if (hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to create OSAL_FIXED for LAT (%6d)",
                    n16Lat
                        );
            break;
        }

        // Compute the LON
        hLon = AGW_MGR_hCreateFixedFromInteger(n16Lon, AGW_COORDINATE_POW10);
        if (hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to create OSAL_FIXED for LON (%6d)",
                    n16Lon
                        );
            break;
        }

        // Create location object
        hResult = LOCATION.hCreateForRadius(hLat, hLon, DISTANCE_INVALID_OBJECT);
        if (hResult == LOCATION_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME": failed to create LOCATION object"
                        );
            break;
        }
    } while (FALSE);

    if (hLat != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(hLat);
    }

    if (hLon != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(hLon);
    }

    return hResult;
}


/*****************************************************************************
 *
 *   bUpdateLocationObject
 *
 *****************************************************************************/
static BOOLEAN bUpdateLocationObject(
    AGW_TILE_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT *phLocation,
    N16 n16Lat,
    N16 n16Lon
        )
{
    BOOLEAN bOk = FALSE;

    // Check input
    if ((phLocation != NULL) && (psObj != NULL))
    {
        LOCATION_OBJECT hLocation;

        hLocation = hCreateLocationObject(psObj, n16Lat, n16Lon);

        if (hLocation != LOCATION_INVALID_OBJECT)
        {
            if (*phLocation != LOCATION_INVALID_OBJECT)
            {
                LOCATION.vDestroy(*phLocation);
                *phLocation = LOCATION_INVALID_OBJECT;
            }

            *phLocation = hLocation;
            bOk = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to create location object for update");
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bUpdateTimeStamp
 *
 *****************************************************************************/
static BOOLEAN bUpdateTimeStamp(
    AGW_TILE_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    BOOLEAN bOk = FALSE;

    if ((psObj != NULL) && (psHeader != NULL))
    {
        psObj->tTimeStamp =
                AGW_MGR_tCreateTimeStamp(psHeader, FALSE);

        bOk = TRUE;
    }

    return bOk;
}


/*****************************************************************************
 *
 *   bUpdateBoundingBox
 *
 *****************************************************************************/
static BOOLEAN bUpdateBoundingBox(
    AGW_TILE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_FIXED_OBJECT hWest = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hEast = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hSouth = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hNorth = OSAL_FIXED_INVALID_OBJECT;

    do
    {
        // Check input
        if ((psObj == NULL) ||
            (psObj->hNorthWest == LOCATION_INVALID_OBJECT) ||
            (psObj->hSouthEast == LOCATION_INVALID_OBJECT))
        {
            break;
        }

        bOk = AGW_MGR_bBoundingBoxInit(&psObj->sBoundingBox);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_TILE_OBJECT_NAME
                ": failed to initialize bounding box"
                    );
            break;
        }

        // Gather points from tile corners
        hWest = LOCATION.hLon(psObj->hNorthWest);
        hNorth = LOCATION.hLat(psObj->hNorthWest);
        hEast = LOCATION.hLon(psObj->hSouthEast);
        hSouth = LOCATION.hLat(psObj->hSouthEast);

        // Do calculation
        bOk = AGW_MGR_bBoundingBoxCalculate(&psObj->sBoundingBox,
                                hNorth, hSouth, hWest, hEast, TRUE);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_TILE_OBJECT_NAME
                    ": failed to calculate bounding box"
                        );
            break;
        }

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        AGW_MGR_vPrintBoundingBox(&psObj->sBoundingBox);
#endif
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   bAgwTileImageInitSpecificData
 *
 *****************************************************************************/
static BOOLEAN bAgwTileImageInitSpecificData (
    IMAGE_OBJECT hImage,
    void *pvSpecificData,
    void *pvArg
        )
{
    if (pvSpecificData != NULL)
    {
        AGW_TILE_IMAGE_DATA_STRUCT *psImageData =
            (AGW_TILE_IMAGE_DATA_STRUCT *)pvSpecificData;

        OSAL.bMemSet(psImageData, 0, sizeof(*psImageData));

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   vAgwTileImageUninitSpecificData
 *
 *****************************************************************************/
static void vAgwTileImageUninitSpecificData (
    IMAGE_OBJECT hImage,
    void *pvSpecificData
        )
{
    STRING_OBJECT hFilePath;

    // Remove file from the file system
    hFilePath = IMAGE.hFileName(hImage);
    if (hFilePath != STRING_INVALID_OBJECT)
    {
        N32 n32Result;
        const char *pacFilePath;

        // Get file path
        pacFilePath = STRING.pacCStr(hFilePath);

        printf(AGW_TILE_OBJECT_NAME": removing file %s\n", pacFilePath);

        // Remove tile file since it is not necessary any longer
        n32Result = remove(pacFilePath);
        if (n32Result != 0)
        {
            printf(AGW_TILE_OBJECT_NAME
                ": failed to remove file %s with code %d\n",
                pacFilePath, n32Result
                    );
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_TILE_OBJECT_NAME": failed to get IMAGE file path");
    }


    return;
}

/*****************************************************************************
 *
 *   eAgwTileImageProperty
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAgwTileImageProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    void *pvSpecificData,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    if ((pvSpecificData != NULL) && (psValue != NULL))
    {
        AGW_TILE_IMAGE_DATA_STRUCT *psImageData =
            (AGW_TILE_IMAGE_DATA_STRUCT*)pvSpecificData;

        // Since all controlled properties are UN32
        psValue->eType = IMAGE_PROPERTY_INTERNAL_TYPE_UN32;

        // Process properties the object knows at this point
        switch (eProperty)
        {
            case IMAGE_PROPERTY_INTERNAL_HEIGHT:
            {
                psValue->uData.un32Value = psImageData->un16Height;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_WIDTH:
            {
                psValue->uData.un32Value = psImageData->un16Width;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            case IMAGE_PROPERTY_INTERNAL_BITS_PER_PIXEL:
            {
                psValue->uData.un32Value = psImageData->un8BPP;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            break;
            default:
            {
                eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            }
            break;
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

#ifdef SUPPORT_CUNIT
#include <agw_tile_obj.cunit>
#endif
