/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_STORM_ATTRIBUTES implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "agw_shape_obj.h"
#include "agw_storm_attributes_obj.h"
#include "_agw_storm_attributes_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   hStormID
*
*****************************************************************************/
static STRING_OBJECT hStormID (
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);
    if (psObj != NULL)
    {
        hResult = psObj->sData.hStormID;
    }

    return hResult;
}

/*****************************************************************************
*
*   tObservationTime
*
*****************************************************************************/
static TIME_T tObservationTime (
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    TIME_T tResult = 0;
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);
    if (psObj != NULL)
    {
        tResult = psObj->sData.tObservationTime;
    }

    return tResult;
}

/*****************************************************************************
*
*   eEchoTop
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEchoTop (
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
    OSAL_FIXED_OBJECT hEchoTop
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                            AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj
                                );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sEchoTop,
                                    hEchoTop, TRUE);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   eDirection
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDirection(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
    OSAL_FIXED_OBJECT hDirection
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                            AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj
                                );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sDirection,
                            hDirection, psObj->sData.bDirectionAvailable);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   eSpeed
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSpeed(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
    OSAL_FIXED_OBJECT hSpeed
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sSpeed,
                            hSpeed, psObj->sData.bSpeedAvailable);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   bIsHailProbable
*
*****************************************************************************/
static BOOLEAN bIsHailProbable(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bResult = FALSE;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);

    if (psObj != NULL)
    {
        bResult = psObj->sData.bHailProbable;
    }

    return bResult;
}


/*****************************************************************************
*
*   bIsHailPositive
*
*****************************************************************************/
static BOOLEAN bIsHailPositive(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bResult = FALSE;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);

    if (psObj != NULL)
    {
        bResult = psObj->sData.bHailPositive;
    }

    return bResult;
}

/*****************************************************************************
*
*   bIsMesocyclonic
*
*****************************************************************************/
static BOOLEAN bIsMesocyclonic(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bResult = FALSE;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);

    if (psObj != NULL)
    {
        bResult = psObj->sData.bMesocyclonic;
    }

    return bResult;
}


/*****************************************************************************
*
*   bIsTornadic
*
*****************************************************************************/
static BOOLEAN bIsTornadic(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bResult = FALSE;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);

    if (psObj != NULL)
    {
        bResult = psObj->sData.bTornadic;
    }

    return bResult;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    N32 n32tmp = 0;
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);
    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    n32Return += fprintf(psFile,
        AGW_STORM_ATTRIBUTES_OBJECT_NAME
        ": Storm ID: " );
    n32tmp = STRING.n32FWrite(psObj->sData.hStormID, psFile);
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\n");
    n32Return += fprintf(psFile, "\tDirection: ");
    n32tmp = OSAL_FIXED.n32FPrintf(psObj->sData.sDirection.hValue,
            psFile, FALSE
                );
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return += fprintf(psFile, "\tSpeed: ");
    n32tmp = OSAL_FIXED.n32FPrintf(psObj->sData.sSpeed.hValue,
            psFile, FALSE
                );
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return +=
            fprintf(psFile, "\tObservation time: %3d; Echo top:",
                psObj->sData.tObservationTime
                    );
    n32tmp = OSAL_FIXED.n32FPrintf(psObj->sData.sEchoTop.hValue,
            psFile, FALSE
                );
    n32Return += ( n32tmp > 0 ) ? n32tmp : 0 ;
    n32Return +=
            fprintf(psFile, "\tHailPositive: %u; HailProbable: %u;"
                "Mesocyclonic: %u; Tornadic: %u\n",
                psObj->sData.bHailPositive, psObj->sData.bHailProbable,
                psObj->sData.bMesocyclonic, psObj->sData.bTornadic
                    );

    return n32Return;
}


/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_STORM_ATTRIBUTES_hCreate
 *
 *****************************************************************************/
AGW_STORM_ATTRIBUTES_OBJECT AGW_STORM_ATTRIBUTES_hCreate(
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT hResult = AGW_STORM_ATTRIBUTES_INVALID_OBJECT;
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW STORM ATTRIBUTES object
        hResult = (AGW_STORM_ATTRIBUTES_OBJECT)
                    AGW_SHAPE_hCreate(
                        (void**) &psObj,
                        AGW_STORM_ATTRIBUTES_OBJECT_NAME": Obj",
                        psHeader,
                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES,
                        sizeof(AGW_STORM_ATTRIBUTES_OBJECT_STRUCT),
                        tDescriptorSize,
                        hParent
                            );

        if (hResult == AGW_STORM_ATTRIBUTES_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_ATTRIBUTES_OBJECT_NAME": failed to create\n"
                    );
            break;
        }

        // init
        OSAL.bMemSet(&psObj->sData, 0, sizeof(psObj->sData));

    } while (FALSE);

    return hResult;
}


/*****************************************************************************
 *
 *   AGW_STORM_ATTRIBUTES_bUpdate
 *
 *****************************************************************************/
BOOLEAN AGW_STORM_ATTRIBUTES_bUpdate(
    AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
    AGW_STORM_ATTRIBUTES_DATA_STRUCT *psData
        )
{
    BOOLEAN bOk = FALSE;
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Check input
        if (psData == NULL)
        {
            break;
        }

        AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
                            AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);
        if (psObj == NULL)
        {
            break;
        }

        // Copy data at once
        psObj->sData = *psData;
        psObj->sData.hStormID = STRING_INVALID_OBJECT;

        if (psData->hStormID != STRING_INVALID_OBJECT)
        {
            psObj->sData.hStormID = psData->hStormID;
            psData->hStormID = STRING_INVALID_OBJECT;
        }

        if (psData->bSpeedAvailable == TRUE)
        {
            bOk = AGW_MGR_bCopyFixedTo(&psData->sSpeed, &psObj->sData.sSpeed);
            if (bOk == FALSE)
            {
                break;
            }
        }

        if (psData->bDirectionAvailable == TRUE)
        {
            bOk = AGW_MGR_bCopyFixedTo(&psData->sDirection, &psObj->sData.sDirection);
            if (bOk == FALSE)
            {
                break;
            }
        }

        bOk = AGW_MGR_bCopyFixedTo(&psData->sEchoTop, &psObj->sData.sEchoTop);
        if (bOk == FALSE)
        {
            break;
        }
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != NULL))
    {
        // Clean up fields in case of error
        if (psObj->sData.hStormID != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->sData.hStormID);
        }
        OSAL.bMemSet(&psObj->sData, 0, sizeof(psObj->sData));
    }

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_STORM_ATTRIBUTES_vDestroy
 *
 *****************************************************************************/
void AGW_STORM_ATTRIBUTES_vDestroy(
   AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT_STRUCT *psObj = NULL;

    printf(AGW_STORM_ATTRIBUTES_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwStormAttr
                );

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormAttr,
        AGW_SHAPE_TYPE_STORM_ATTRIBUTES, (void**)&psObj);

    if (psObj != NULL)
    {
        if (psObj->sData.hStormID != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->sData.hStormID);
            psObj->sData.hStormID = STRING_INVALID_OBJECT;
        }
    }

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT) hAgwStormAttr);

    return;
}
