/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_SHAPE implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_entry_obj.h"
#include "location_obj.h"

#include "agw_shape_obj.h"
#include "_agw_shape_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   eProductType
*
*****************************************************************************/
static AGW_PRODUCT_TYPE_ENUM eProductType (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_PRODUCT_TYPE_ENUM eResult = AGW_PRODUCT_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        eResult = psObj->eProductType;
    }

    return eResult;
}

/*****************************************************************************
*
*   eShapeType
*
*****************************************************************************/
static AGW_SHAPE_TYPE_ENUM eShapeType (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_SHAPE_TYPE_ENUM eResult = AGW_SHAPE_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;

        eResult = psObj->eType;
    }

    return eResult;
}

/*****************************************************************************
*
*   tTimeStamp
*
*****************************************************************************/
static TIME_T tTimeStamp (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    TIME_T tResult = 0;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;

        tResult = psObj->tTimeStamp;
    }

    return tResult;
}


/*****************************************************************************
*
*   un16Points
*
*****************************************************************************/
static UN16 un16Points (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    UN16 un16Result = 0;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;

        un16Result = psObj->un16Points;
    }

    return un16Result;
}


/*****************************************************************************
*
*   eIteratePoints
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIteratePoints (
    AGW_SHAPE_OBJECT hAgwShape,
    AGW_SHAPE_POINTS_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
    UN16 un16Point;

    do
    {
        // Check inputs
        if ((bCallback == NULL) || (hAgwShape == AGW_SHAPE_INVALID_OBJECT))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->un16Points > 0)
        {
            AGW_GEO_POINT_STRUCT *psPoint = psObj->pasPoints;
            BOOLEAN bOk = TRUE;

            for (un16Point = 0;
                 (un16Point < psObj->un16Points) && (bOk == TRUE);
                 ++un16Point, ++psPoint)
            {
                if ((psPoint->hLat != OSAL_FIXED_INVALID_OBJECT) &&
                    (psPoint->hLon != OSAL_FIXED_INVALID_OBJECT))
                {
                    bOk = bCallback(psPoint->hLat, psPoint->hLon,
                                        pvCallbackArg
                                            );
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_SHAPE_OBJECT_NAME": incorrect GEO point"
                                );
                }
            }
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eGetPoint
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetPoint (
    AGW_SHAPE_OBJECT hAgwShape,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    UN16 un16Point
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;
    BOOLEAN bOk = FALSE;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;

    do
    {
        // Check inputs
        if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hLon == OSAL_FIXED_INVALID_OBJECT) ||
            (hAgwShape == AGW_SHAPE_INVALID_OBJECT))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner =
            DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->un16Points <= un16Point)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        if ((psObj->pasPoints[un16Point].hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (psObj->pasPoints[un16Point].hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Copy data to user objects
        bOk = OSAL_FIXED.bCopyToMemory(psObj->pasPoints[un16Point].hLat, hLat);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_SHAPE_OBJECT_NAME": failed to copy LAT"
                    );
            break;
        }

        bOk = OSAL_FIXED.bCopyToMemory(psObj->pasPoints[un16Point].hLon, hLon);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_SHAPE_OBJECT_NAME": failed to copy LON"
                    );
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   hFront
*
*****************************************************************************/
static AGW_FRONT_OBJECT hFront (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_FRONT_OBJECT hResult = AGW_FRONT_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_FRONT)
        {
            hResult = (AGW_FRONT_OBJECT)hAgwShape;
        }
    }

    return hResult;
}

/*****************************************************************************
*
*   hStormAttributes
*
*****************************************************************************/
static AGW_STORM_ATTRIBUTES_OBJECT hStormAttributes (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_STORM_ATTRIBUTES_OBJECT hResult = AGW_STORM_ATTRIBUTES_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_STORM_ATTRIBUTES)
        {
            hResult = (AGW_STORM_ATTRIBUTES_OBJECT)hAgwShape;
        }
    }

    return hResult;
}

/*****************************************************************************
*
*   hIsobar
*
*****************************************************************************/
static AGW_ISOBAR_OBJECT hIsobar (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_ISOBAR_OBJECT hResult = AGW_ISOBAR_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_ISOBAR)
        {
            hResult = (AGW_ISOBAR_OBJECT)hAgwShape;
        }
    }
    return hResult;
}

/*****************************************************************************
*
*   hPressureCenter
*
*****************************************************************************/
static AGW_PRESSURE_CENTER_OBJECT hPressureCenter (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_PRESSURE_CENTER_OBJECT hResult = AGW_PRESSURE_CENTER_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_PRESSURE_CENTER)
        {
            hResult = (AGW_PRESSURE_CENTER_OBJECT)hAgwShape;
        }
    }
    return hResult;
}

/*****************************************************************************
*
*   hStormPosition
*
*****************************************************************************/
static AGW_STORM_POSITION_OBJECT hStormPosition (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_STORM_POSITION_OBJECT hResult = AGW_STORM_POSITION_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_STORM_POSITION)
        {
            hResult = (AGW_STORM_POSITION_OBJECT)hAgwShape;
        }
    }
    return hResult;
}

/*****************************************************************************
*
*   hWindRadiiArea
*
*****************************************************************************/
static AGW_WIND_RADII_AREA_OBJECT hWindRadiiArea (
    AGW_SHAPE_OBJECT hAgwShape
        )
{
    AGW_WIND_RADII_AREA_OBJECT hResult = AGW_WIND_RADII_AREA_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify ownership of this object
    bOwner =
        DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == TRUE)
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        if (psObj->eType == AGW_SHAPE_TYPE_WIND_RADII_FIELD)
        {
            hResult = (AGW_WIND_RADII_AREA_OBJECT)hAgwShape;
        }
    }

    return hResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_SHAPE_OBJECT hAgwShape,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    BOOLEAN bOwner;
    N32 n32Return = 0;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*)hAgwShape;
    AGW_GEO_POINT_STRUCT *psPoint = NULL;
    UN16 un16Point = 0;

    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hAgwShape);
    if ((bOwner == FALSE) || (psFile == NULL))
    {
        return EOF;
    }

    n32Return += fprintf(psFile, AGW_SHAPE_OBJECT_NAME
            ": PRODUCT TYPE: %d, SHAPE TYPE: %d\nNUMBER OF POINTS: %d\n",
            psObj->eProductType, psObj->eType, psObj->un16Points
                );

    psPoint = psObj->pasPoints;
    for (un16Point = 0; un16Point < psObj->un16Points;
             ++un16Point, ++psPoint)
    {
        n32Return += fprintf(psFile, "#%03d: ", un16Point);
        n32Return += n32GeoPointFPrintf(psPoint,
                                       psFile, eOutputOption
                                           );
    }

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_SHAPE_hCreate
 *
 *   This function creates DSRL entry with additional space which can be used
 *   by concrete shape implementation. The ppvData in case of success provides
 *   address of the memory right after AGW_SHAPE data itself.
 *
 *   In turn the return value should be used to deal with DSRL & DSRL_ENTRY related
 *   functions.
 *
 *****************************************************************************/
AGW_SHAPE_OBJECT AGW_SHAPE_hCreate(
    void **ppvData,
    const char *pacName,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_SHAPE_TYPE_ENUM eShapeType,
    size_t tSize,
    size_t tDescriptorSize,
    SMS_OBJECT hParent
        )
{
    AGW_SHAPE_OBJECT_STRUCT *psObj =
            (AGW_SHAPE_OBJECT_STRUCT*) AGW_SHAPE_INVALID_OBJECT;

    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Check inputs
    if ((hParent == SMS_INVALID_OBJECT) || (ppvData == NULL))
    {
        return AGW_SHAPE_INVALID_OBJECT;
    }

    // Construct an appropriate name for memory to be allocated
    snprintf(&acName[0], sizeof(acName), AGW_SHAPE_PREFIX":%s", pacName);

    // Create an instance of the AGW Msg object
    psObj = (AGW_SHAPE_OBJECT_STRUCT *)
        DSRL_ENTRY_hCreate(
            &acName[0],
            DSRL_ENTRY_TYPE_AGW_SHAPE,
            sizeof(AGW_SHAPE_OBJECT_STRUCT) + tSize,
            tDescriptorSize,
            hParent,
            FALSE);

    if ((DSRL_ENTRY_OBJECT) psObj == DSRL_ENTRY_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_SHAPE_OBJECT_NAME": failed to create DSRL_ENTRY\n"
                    );
        return AGW_SHAPE_INVALID_OBJECT;
    }

    // Init members for general shape
    psObj->eProductType =
        AGW_MGR_eGetPublicProductType(psHeader->eProductType, AGW_RASTER_PLANE_ALL);
    psObj->tTimeStamp =
        AGW_MGR_tCreateTimeStamp(psHeader, FALSE);
    psObj->eType = eShapeType;
    psObj->psIntf = &GsShapeDefaultIntf;

    // Populate space for concrete shape data
    *ppvData = (void*)(psObj + 1);

    return (AGW_SHAPE_OBJECT)psObj;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_vDestroy
 *
 *****************************************************************************/
void AGW_SHAPE_vDestroy(
   AGW_SHAPE_OBJECT hAgwShape
        )
{
    BOOLEAN bOwner;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*)hAgwShape;

    printf(AGW_SHAPE_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwShape);

    // Verify ownership of this object
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
    if (bOwner == FALSE)
    {
        return;
    }

    vPointsArrayDestroy(psObj);

    DSRL_ENTRY_vDestroy((DSRL_ENTRY_OBJECT) psObj);

    return;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_eIterateLines
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_SHAPE_eIterateLines (
    AGW_SHAPE_OBJECT hShape,
    SHAPE_LINES_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    BOOLEAN bOwner = FALSE;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hShape;
    UN16 un16Point;

    do
    {
        // Check inputs
        if (bCallback == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hShape);
        if (bOwner != TRUE)
        {
            eReturnCode = AGW_RETURN_CODE_NOT_OWNER;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
        if (psObj->un16Points > 1)
        {
            AGW_GEO_POINT_STRUCT *psPoint1 = psObj->pasPoints;
            AGW_GEO_POINT_STRUCT *psPoint2 = psObj->pasPoints + 1;
            BOOLEAN bOk = TRUE;

            for (un16Point = 0;
                 (un16Point < psObj->un16Points - 1) && (bOk == TRUE);
                 ++un16Point, ++psPoint1, ++psPoint2)
            {
                if ((psPoint1->hLat == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint1->hLon == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint2->hLat == OSAL_FIXED_INVALID_OBJECT) ||
                    (psPoint2->hLon == OSAL_FIXED_INVALID_OBJECT))
                {
                    eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
                    bOk = FALSE;
                }
                else
                {
                    // Do the call and if the function returns FALSE just stop
                    // iteration
                    bOk = bCallback(
                            psPoint1->hLat,
                            psPoint1->hLon,
                            psPoint2->hLat,
                            psPoint2->hLon,
                            pvCallbackArg
                                );
                }
            }
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_eHasIntersectionWith
 *
 *   Detects intersection between shape and Location. If at least one
 *   point inside specified location the function returns says YES. In order
 *   to provide checking based on concrete object type the callback interface
 *   is used.
 *
 *   Inputs:
 *      hAgwShape  - object to be verified
 *      psLocDesc  - location and its description to calculate intersection with the shape
 *
 *   Output:
 *       AGW_RETURN_CODE_SUCCESS  - if intersection exists
 *       AGW_RETURN_CODE_NO_INTERSECTION and some other code in failed cases
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_SHAPE_eHasIntersectionWith (
    AGW_SHAPE_OBJECT hAgwShape,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    BOOLEAN bOwner = FALSE;
    AGW_RETURN_CODE_ENUM eReturnValue = AGW_RETURN_CODE_ERROR;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;

    do
    {
        // Check input
        if ((psObj == NULL) || (psLocDesc == NULL) ||
            (psLocDesc->hLocation == LOCATION_INVALID_OBJECT))
        {
            eReturnValue = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            eReturnValue = AGW_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Call the real checking function via interface
        if (psObj->psIntf->eHasIntersection != NULL)
        {
            eReturnValue = psObj->psIntf->eHasIntersection(hAgwShape, psLocDesc);
        }
        else
        {
            eReturnValue = GsShapeDefaultIntf.eHasIntersection(hAgwShape, psLocDesc);
        }
    } while (FALSE);

    return eReturnValue;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_eHasDirectIntersectionWith
 *
 *   Detects intersection between shape and Location using only shape points
 *   regardless concrete object type.
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_SHAPE_eHasDirectIntersectionWith (
    AGW_SHAPE_OBJECT hAgwShape,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    BOOLEAN bOwner = FALSE;
    AGW_RETURN_CODE_ENUM eReturnValue = AGW_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psLocDesc == NULL) ||(psLocDesc->hLocation == LOCATION_INVALID_OBJECT))
        {
            eReturnValue = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)hAgwShape);
        if (bOwner == FALSE)
        {
            eReturnValue = AGW_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Check the intersection through default implementation
        eReturnValue = GsShapeDefaultIntf.eHasIntersection(hAgwShape, psLocDesc);
    } while (FALSE);

    return eReturnValue;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_bUpdatePoints
 *
 *   Prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN AGW_SHAPE_bUpdatePoints(
    AGW_SHAPE_OBJECT hAgwShape,
    UN16 un16Points
        )
{
    BOOLEAN bOk = FALSE, bOwner = FALSE;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*)hAgwShape;

    do
    {
        // Check input
        if (hAgwShape == AGW_SHAPE_INVALID_OBJECT)
        {
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        vPointsArrayDestroy(psObj);
        psObj->un16Points = un16Points;
        bPointsArrayCreate(psObj);

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_SHAPE_bSetPoint
 *
 *****************************************************************************/
BOOLEAN AGW_SHAPE_bSetPoint(
    AGW_SHAPE_OBJECT hAgwShape,
    UN16 un16Index,
    N32 n32Lat,
    N32 n32Lon,
    UN8 un8PowOfDivider
        )
{
    BOOLEAN bOk = FALSE, bOwner = FALSE;
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*)hAgwShape;
    AGW_GEO_POINT_STRUCT *psGeoPoint = NULL;

    do
    {
        // Check inputs
        if ((hAgwShape == AGW_SHAPE_INVALID_OBJECT) ||
            (un16Index >= psObj->un16Points) || (psObj->pasPoints == NULL))
        {
            break;
        }

        // Verify ownership of this object
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Check for point existence
        psGeoPoint = &psObj->pasPoints[un16Index];
        if ((psGeoPoint->hLat != OSAL_FIXED_INVALID_OBJECT) ||
            (psGeoPoint->hLon != OSAL_FIXED_INVALID_OBJECT))
        {
            printf(AGW_SHAPE_OBJECT_NAME": point for index %d already set",
                    un16Index);
            break;
        }

        // Create GEO-point in the memory
        psGeoPoint = psGeoPointCreateInMemory(n32Lat, n32Lon,
                         psGeoPoint, un8PowOfDivider
                             );
        if (psGeoPoint == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_SHAPE_OBJECT_NAME
                    ": failed to create GEO-point in memory"
                        );
            break;
        }

        // Update the borders box
        if (psObj->psPointsDesc != NULL)
        {
            bOk = bUpdateBordersBox(&psObj->psPointsDesc->sBorders, psGeoPoint);
            if (bOk == FALSE)
            {
                printf(AGW_SHAPE_OBJECT_NAME": failed to update the bounding box\n");
            }
        }
        else
        {
            bOk = TRUE;
        }

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_bSetInterface
 *
 *****************************************************************************/
BOOLEAN AGW_SHAPE_bSetInterface(
    AGW_SHAPE_OBJECT hAgwShape,
    const AGW_SHAPE_CALLBACK_INTERFACE_STRUCT *psIntf
        )
{
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
    BOOLEAN bOwner, bResult = FALSE;

    // Check ownership
    bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) hAgwShape);
    if ((bOwner == TRUE) && (psIntf != NULL))
    {
        psObj->psIntf = psIntf;
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_eData
 *
 *   This function provides aux data associated with the shape object.
 *
 *   Inputs:
 *      hAgeShape - the AGW_SHAPE based object
 *      eType - type of the derived object
 *      ppvData - the address of the variable to place the aud data from
 *                the shape object
 *   Output:
 *      AGW_RETURN_CODE_SUCCESS if the shape is exact the same as asked and
 *      the *ppvData has a address of the Shape's AUX data.
 *      Otherwise, the function return specific error code and the *ppvData
 *      will have a NULL inside if the invalid address wasn't a reason for
 *      function failure.
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_SHAPE_eData(
    AGW_SHAPE_OBJECT hAgwShape,
    AGW_SHAPE_TYPE_ENUM eType,
    void **ppvData
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;

    if ((ppvData != NULL) && (hAgwShape != AGW_SHAPE_INVALID_OBJECT))
    {
        AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hAgwShape;
        BOOLEAN bOwner;

        // Prepare output data
        *ppvData = NULL;

        // Check ownership
        bOwner = DSRL_ENTRY_bOwner((DSRL_ENTRY_OBJECT) psObj);
        if (bOwner == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_NOT_OWNER;
        }
        else if (psObj->eType != eType)
        {
            eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
        }
        else
        {
            *ppvData = (void*)(psObj + 1);
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   AGW_SHAPE_ePopulateFixedValue
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_SHAPE_ePopulateFixedValue(
    AGW_FIXED_OBJECT_STRUCT *psSrc,
    OSAL_FIXED_OBJECT hFixedDesc,
    BOOLEAN bTrigger
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    BOOLEAN bOk;

    do
    {
        if ((psSrc == NULL) || (hFixedDesc == OSAL_FIXED_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        if ((bTrigger == FALSE) || (psSrc->hValue == OSAL_FIXED_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_NOT_EXIST;
            break;
        }

        bOk = OSAL_FIXED.bCopyToMemory(psSrc->hValue, hFixedDesc);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_SHAPE_OBJECT_NAME": failed to make a copy of the fixed object"
                    );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *                              PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   eDefaultSinglePointShapeHasIntersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultSinglePointShapeHasIntersection (
    AGW_SHAPE_OBJECT_STRUCT *psObj,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_NO_INTERSECTION;
    AGW_GEO_POINT_STRUCT *psPoint = psObj->pasPoints;
    BOOLEAN bResult;

    printf(AGW_SHAPE_OBJECT_NAME": %s [ENTER]\n", __FUNCTION__);

    bResult = LOCATION.bCoordinatesWithinArea(
                        psLocDesc->hLocation,
                        psPoint->hLat, psPoint->hLon
                            );
    if (bResult == TRUE)
    {
        eResult = AGW_RETURN_CODE_SUCCESS;
    }

    printf(AGW_SHAPE_OBJECT_NAME": %s [LEAVE] %s\n", __FUNCTION__,
                AGW_MGR_pacGetAgwReturnCodeName(eResult));

    return eResult;
}

/*****************************************************************************
 *
 *   eDefaultMultiPointShapeBBHasIntersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultMultiPointShapeBBHasIntersection (
    AGW_SHAPE_OBJECT_STRUCT *psObj,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_NO_INTERSECTION;
    BOOLEAN bResult;

    printf(AGW_SHAPE_OBJECT_NAME": %s [ENTER]\n", __FUNCTION__);

    // Update bounding box if needed
    bResult = bUpdateBoundingBox(psObj);
    if (bResult == TRUE)
    {
        // Check intersection of the bounding boxes
        bResult = AGW_MGR_bBoundingBoxesIntersected(
                        &psObj->psPointsDesc->sBB, &psLocDesc->sBoundingBox
                            );
        if (bResult == TRUE)
        {
            eResult = AGW_RETURN_CODE_SUCCESS;
        }
    }
    else
    {
        eResult = AGW_RETURN_CODE_NOT_EXIST;
    }

    printf(AGW_SHAPE_OBJECT_NAME": %s [LEAVE] %s\n", __FUNCTION__,
                AGW_MGR_pacGetAgwReturnCodeName(eResult));

    return eResult;
}

/*****************************************************************************
 *
 *   eDefaultMultiPointShapePointsHasIntersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultMultiPointShapePointsHasIntersection (
    AGW_SHAPE_OBJECT_STRUCT *psObj,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_NO_INTERSECTION;
    UN16 un16Point = 0;
    AGW_GEO_POINT_STRUCT *psPoint = psObj->pasPoints;
    BOOLEAN bResult = FALSE;

    printf(AGW_SHAPE_OBJECT_NAME": %s [ENTER]\n", __FUNCTION__);

    // Iterate through all points to find out intersection
    for (;(un16Point < psObj->un16Points) && (bResult == FALSE);
        ++un16Point, ++psPoint)
    {
        bResult = LOCATION.bCoordinatesWithinArea(
                            psLocDesc->hLocation,
                            psPoint->hLat, psPoint->hLon
                                );
    }

    if (bResult == TRUE)
    {
        // We've found intersection
        eResult = AGW_RETURN_CODE_SUCCESS;
    }

    printf(AGW_SHAPE_OBJECT_NAME": %s [LEAVE] %s\n", __FUNCTION__,
        AGW_MGR_pacGetAgwReturnCodeName(eResult));

    return eResult;
}

/*****************************************************************************
 *
 *   bDefaultRayIntersectionIterator
 *
 *****************************************************************************/
static BOOLEAN bDefaultRayIntersectionIterator(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        )
{
    AGW_SHAPE_LOCATION_CROSS_ARG_STRUCT *psArg =
        (AGW_SHAPE_LOCATION_CROSS_ARG_STRUCT *)pvArg;
    BOOLEAN bResult;

    bResult = LOCATION_bRayCross (
                hLonBegin, hLatBegin,
                hLonEnd, hLatEnd,
                psArg->hLocation
                    );

    if (bResult == TRUE)
    {
        //odd/even check increasing here
        psArg->un8NumOfRayCrossings++;
    }

    return TRUE; //We should iterate all the lines in order to find a crossing.
}

/*****************************************************************************
 *
 *   bDefaultLineClipIntersectionIterator
 *
 *****************************************************************************/
static BOOLEAN bDefaultLineClipIntersectionIterator(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        )
{
    AGW_SHAPE_LOCATION_CROSS_ARG_STRUCT *psArg =
        (AGW_SHAPE_LOCATION_CROSS_ARG_STRUCT *)pvArg;
    BOOLEAN bResult = TRUE;

    psArg->bCross = LOCATION_bLineClip (
                            hLonBegin,
                            hLatBegin,
                            hLonEnd,
                            hLatEnd,
                            psArg->hLocation);

    if (psArg->bCross == TRUE)
    {
        // If we have found the crossing, there is no need to
        // iterate other lines.
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
 *
 *   eDefaultMultiPointShapeLinesHasIntersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultMultiPointShapeLinesHasIntersection (
    AGW_SHAPE_OBJECT_STRUCT *psObj,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_NO_INTERSECTION;
    AGW_SHAPE_LOCATION_CROSS_ARG_STRUCT sArg;

    printf(AGW_SHAPE_OBJECT_NAME": %s [ENTER]\n", __FUNCTION__);

    do
    {
        // Init the structure
        sArg.bCross = FALSE;
        sArg.hLocation = psLocDesc->hLocation;
        sArg.un8NumOfRayCrossings = 0;

        // The ray crossing applicable only for polygons
        if (psObj->psPointsDesc->bIsClosed == TRUE)
        {
            printf(AGW_SHAPE_OBJECT_NAME":\tray crossing\n");
            // Iterate lines for ray crossing
            eResult = AGW_SHAPE_eIterateLines((AGW_SHAPE_OBJECT) psObj,
                            bDefaultRayIntersectionIterator,
                            (void*)&sArg
                                );
            if (eResult != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_SHAPE_OBJECT_NAME": failed to iterate lines for ray crossing (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eResult)
                        );
                break;
            }

            // Checking if number of crossings is odd
            if ((sArg.un8NumOfRayCrossings % 2) != 0)
            {
                //the location is inside of the polygon.
                sArg.bCross = TRUE;
            }
        }

        // Do the line clipping
        if (sArg.bCross == FALSE)
        {
            printf(AGW_SHAPE_OBJECT_NAME":\tline clipping\n");
            // The location is not inside of the polygon but it is still
            // inside BB area so we should perform an extra check for
            // an area around the current location
            eResult = AGW_SHAPE_eIterateLines(
                            (AGW_SHAPE_OBJECT) psObj,
                            bDefaultLineClipIntersectionIterator,
                            (void*)&sArg);

            if (eResult != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_SHAPE_OBJECT_NAME": failed to iterate lines for line clipping (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eResult)
                        );
                break;
            }
        }

        // Well, check what we have
        eResult = (sArg.bCross == FALSE) ? AGW_RETURN_CODE_NO_INTERSECTION : AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    printf(AGW_SHAPE_OBJECT_NAME": %s [LEAVE] %s\n", __FUNCTION__,
        AGW_MGR_pacGetAgwReturnCodeName(eResult));

    return eResult;
}

/*****************************************************************************
 *
 *   eDefaultMultiPointShapeHasIntersection
 *
 *   This function tries to check the shape and target intersection in a few steps:
 *   1) Check intersection using bounding boxes
 *   2) If the #1 is TRUE check each point for appearance inside the target area
 *   3) If the #1 is TRUE, but the #2 is FALSE check the lines intersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultMultiPointShapeHasIntersection (
    AGW_SHAPE_OBJECT_STRUCT *psObj,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eResult;

    printf(AGW_SHAPE_OBJECT_NAME": %s [ENTER]\n", __FUNCTION__);

    eResult = eDefaultMultiPointShapeBBHasIntersection(psObj, psLocDesc);
    if (eResult == AGW_RETURN_CODE_SUCCESS)
    {
        eResult = eDefaultMultiPointShapePointsHasIntersection(psObj, psLocDesc);
        if (eResult == AGW_RETURN_CODE_NO_INTERSECTION)
        {
            eResult = eDefaultMultiPointShapeLinesHasIntersection(psObj, psLocDesc);
        }
    }

    printf(AGW_SHAPE_OBJECT_NAME": %s [LEAVE] %s\n", __FUNCTION__,
        AGW_MGR_pacGetAgwReturnCodeName(eResult));

    return eResult;
}

/*****************************************************************************
 *
 *   eDefaultHasIntersection
 *
 *****************************************************************************/
static AGW_RETURN_CODE_ENUM eDefaultHasIntersection (
    AGW_SHAPE_OBJECT hShape,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_SHAPE_OBJECT_STRUCT *psObj = (AGW_SHAPE_OBJECT_STRUCT*) hShape;
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_BAD_ARGUMENT;

    if (psObj != NULL)
    {
        printf(AGW_SHAPE_OBJECT_NAME
            ": %s [ENTER] [%d]\n", __FUNCTION__, psObj->eType);

        if (psObj->un16Points == 1)
        {
            eResult = eDefaultSinglePointShapeHasIntersection(psObj, psLocDesc);
        }
        else if (psObj->un16Points > 0)
        {
            eResult = eDefaultMultiPointShapeHasIntersection(psObj, psLocDesc);
        }
        else
        {
            eResult = AGW_RETURN_CODE_NO_INTERSECTION;
        }

        printf(AGW_SHAPE_OBJECT_NAME
            ": %s [LEAVE] [%d]\n\n", __FUNCTION__, psObj->eType);
    }

    return eResult;
}

/*****************************************************************************
 *
 *   bUpdateBordersBox
 *
 *****************************************************************************/
static BOOLEAN bUpdateBordersBox(
    AGW_SHAPE_BORDERS_STRUCT *psBorders,
    AGW_GEO_POINT_STRUCT *psPoint
        )
{
    BOOLEAN bOk = TRUE;

    if (psBorders->bIsValid == FALSE)
    {
        psBorders->hNorth = psPoint->hLat;
        psBorders->hSouth = psPoint->hLat;
        psBorders->hEast  = psPoint->hLon;
        psBorders->hWest  = psPoint->hLon;
        psBorders->bIsValid = TRUE;
    }
    else
    {
        N16 n16Compare;

        // Check new lat
        n16Compare = OSAL_FIXED.n16Compare(psPoint->hLat, psBorders->hNorth);
        if (n16Compare > 0)
        {
            psBorders->hNorth = psPoint->hLat;
        }
        else
        {
            n16Compare = OSAL_FIXED.n16Compare(psPoint->hLat, psBorders->hSouth);
            if (n16Compare < 0)
            {
                psBorders->hSouth = psPoint->hLat;
            }
        }

        // Check new lon
        n16Compare = OSAL_FIXED.n16Compare(psPoint->hLon, psBorders->hWest);
        if (n16Compare < 0)
        {
            psBorders->hWest = psPoint->hLon;
        }
        else
        {
            n16Compare = OSAL_FIXED.n16Compare(psPoint->hLon, psBorders->hEast);
            if (n16Compare > 0)
            {
                psBorders->hEast = psPoint->hLon;
            }
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bUpdateBoundingBox
 *
 *   This function updates the bounding box if it is invalid using corner
 *   coordinates from the borders
 *
 *****************************************************************************/
static BOOLEAN bUpdateBoundingBox(
    AGW_SHAPE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE;

    if (psObj->psPointsDesc != NULL)
    {
        // If box doesn't exist
        if (psObj->psPointsDesc->sBB.bIsValidBox == FALSE)
        {
            AGW_SHAPE_BORDERS_STRUCT *psBorders = &psObj->psPointsDesc->sBorders;
            if (psBorders->bIsValid == TRUE)
            {
                // We need to update since we have borders
                bOk = AGW_MGR_bBoundingBoxCalculate(&psObj->psPointsDesc->sBB,
                                        psBorders->hNorth,
                                        psBorders->hSouth,
                                        psBorders->hWest,
                                        psBorders->hEast,
                                        TRUE
                                            );
                if (bOk == TRUE)
                {
                    N16 n16Compare = 0;
                    AGW_GEO_POINT_STRUCT *psFirst = psObj->pasPoints;
                    AGW_GEO_POINT_STRUCT *psLast = psObj->pasPoints + psObj->un16Points - 1;

                    // Check the first and the last points and if their are
                    // specify the same coordinates consider the shape as
                    // closed polygon
                    n16Compare += OSAL_FIXED.n16Compare(psFirst->hLat, psLast->hLat);
                    n16Compare += OSAL_FIXED.n16Compare(psFirst->hLon, psLast->hLon);
                    psObj->psPointsDesc->bIsClosed = (n16Compare == 0) ? TRUE : FALSE;
                }
            }
        }
        else
        {
            bOk = TRUE;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bPointsArrayCreate
 *
 *****************************************************************************/
static BOOLEAN bPointsArrayCreate (
    AGW_SHAPE_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) || (psObj->pvData != NULL))
        {
            break;
        }

        if (psObj->un16Points > 0)
        {
            SMS_OBJECT hParent;
            size_t tSizeToAllocate = sizeof(AGW_GEO_POINT_STRUCT) * psObj->un16Points;

            // Get the SMS object
            hParent = DSRL_ENTRY_hGetSMSObject((DSRL_ENTRY_OBJECT)psObj);

            if (psObj->un16Points > 1)
            {
                // Since we're going to address more than one point it
                // is good to have the bounding box for them and small
                // additional description
                tSizeToAllocate += sizeof(AGW_SHAPE_POINTS_DESC_STRUCT);
            }

            // Allocate memory
            psObj->pvData = SMSO_hCreate(AGW_SHAPE_OBJECT_NAME":Data",
                                    tSizeToAllocate,
                                    hParent, FALSE
                                        );
            if (psObj->pvData == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_SHAPE_OBJECT_NAME
                    ": failed to create points data array"
                        );
                break;
            }

            // Arrange the buffer
            if (psObj->un16Points == 1)
            {
                psObj->psPointsDesc = NULL;
                psObj->pasPoints = (AGW_GEO_POINT_STRUCT*) psObj->pvData;
            }
            else
            {
                AGW_SHAPE_POINTS_DESC_STRUCT *psPointsDesc;
                psObj->psPointsDesc = (AGW_SHAPE_POINTS_DESC_STRUCT*) psObj->pvData;
                psObj->pasPoints = (AGW_GEO_POINT_STRUCT*) (psObj->psPointsDesc + 1);
                psPointsDesc = psObj->psPointsDesc;

                // Fill in the desc structure (BB)
                bOk = AGW_MGR_bBoundingBoxInit(&psObj->psPointsDesc->sBB);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_SHAPE_OBJECT_NAME
                        ": failed to initialize bounding box"
                            );
                    break;
                }

                // Fill in the desc structure (rest fields)
                psPointsDesc->bIsClosed = FALSE;
                psPointsDesc->sBorders.hEast = OSAL_FIXED_INVALID_OBJECT;
                psPointsDesc->sBorders.hWest = OSAL_FIXED_INVALID_OBJECT;
                psPointsDesc->sBorders.hNorth = OSAL_FIXED_INVALID_OBJECT;
                psPointsDesc->sBorders.hSouth = OSAL_FIXED_INVALID_OBJECT;
            }
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   vPointsArrayDestroy
 *
 *****************************************************************************/
static void vPointsArrayDestroy (
    AGW_SHAPE_OBJECT_STRUCT *psObj
        )
{
    if ((psObj != NULL) && (psObj->pvData != NULL))
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pvData);
        psObj->pvData = NULL;
        psObj->pasPoints = NULL;
        psObj->psPointsDesc = NULL;
    }

    return;
}


/*****************************************************************************
*
*   psGeoPointCreateInMemory
*
*****************************************************************************/
static AGW_GEO_POINT_STRUCT *psGeoPointCreateInMemory (
    N32 n32Lat,
    N32 n32Lon,
    AGW_GEO_POINT_STRUCT *psGeoPoint,
    UN8 un8PowOfDivider
        )
{
    do
    {
        // Check inputs
        if (psGeoPoint == NULL)
        {
            break;
        }

        // LAT
        if (psGeoPoint->hLat == OSAL_FIXED_INVALID_OBJECT)
        {
            psGeoPoint->hLat = AGW_MGR_hCreateFixedFromIntInMem(n32Lat,
                un8PowOfDivider, psGeoPoint->aLatData
                    );
            if (psGeoPoint->hLat == OSAL_FIXED_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_SHAPE_OBJECT_NAME
                        ": failed to create LAT in %p GEO-Point",
                        psGeoPoint
                            );
                break;
            }
        }

        // LON
        if (psGeoPoint->hLon == OSAL_FIXED_INVALID_OBJECT)
        {
            psGeoPoint->hLon = AGW_MGR_hCreateFixedFromIntInMem(n32Lon,
                un8PowOfDivider, psGeoPoint->aLonData
                    );
            if (psGeoPoint->hLon == OSAL_FIXED_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_SHAPE_OBJECT_NAME
                        ": failed to create LON in %p GEO-Point",
                        psGeoPoint
                            );
                break;
            }
        }

        return psGeoPoint;
    } while (FALSE);

    return NULL;
}


/*****************************************************************************
*
*   n32GeoPointFPrintf
*
*****************************************************************************/
static N32 n32GeoPointFPrintf (
    AGW_GEO_POINT_STRUCT *psObj,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutput
        )
{
    N32 n32Result = 0;
    BOOLEAN bTerse = (eOutput == SMS_OUTPUT_OPTION_TERSE) ? TRUE : FALSE;
    n32Result += fprintf(psFile, AGW_SHAPE_OBJECT_NAME"_GEO_POINT:\nLAT: ");
    if (psObj->hLat != OSAL_FIXED_INVALID_OBJECT)
    {
        n32Result += OSAL_FIXED.n32FPrintf(psObj->hLat, psFile, bTerse);
    }
    else
    {
        n32Result += fprintf(psFile, " INVALID\n");
    }

    n32Result += fprintf(psFile, "LON: ");
    if (psObj->hLon != OSAL_FIXED_INVALID_OBJECT)
    {
        n32Result += OSAL_FIXED.n32FPrintf(psObj->hLon, psFile, bTerse);
    }
    else
    {
        n32Result += fprintf(psFile, " INVALID\n");
    }

    return n32Result;
}

