/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:AGW_MGR header file.
 *
 ******************************************************************************/

#ifndef AGW_MGR_OBJ_H_
#define AGW_MGR_OBJ_H_

#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/
/* Coordinate POW10 */
#define AGW_COORDINATE_POW10 (2)
#define AGW_STORM_ATTRIBUTES_COORDINATE_POW10 (3)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef enum agw_file_type_enum
{
    AGW_FILE_TYPE_DEFRAMED = 0,
    AGW_FILE_TYPE_RASTER,
    AGW_FILE_TYPE_MAX
} AGW_FILE_TYPE_ENUM;

// Defines available return codes generated by the AGW Product parser
typedef enum agw_return_code_enum
{
    AGW_RETURN_CODE_SUCCESS = 0, // Success case
    AGW_RETURN_CODE_INVALID_STATE, // Some invalid state detected
    AGW_RETURN_CODE_FAILED_TO_REPLACE, // Replace request is failed
    AGW_RETURN_CODE_ERROR, // General error
    AGW_RETURN_CODE_NOT_OWNER, // Not owner
    AGW_RETURN_CODE_CRC_INVALID, // CRC is invalid
    AGW_RETURN_CODE_PVN_UNSUPPORTED, // PVN or CARID is unsupported
    AGW_RETURN_CODE_BAD_ARGUMENT, // Some bad argument has been passed
    AGW_RETURN_CODE_NOT_ENOUGH_DATA, // Payload has not enough data to process
    AGW_RETURN_CODE_IO_ERROR, // Failed to perform I/O operation
    AGW_RETURN_CODE_NO_MEMORY, // Failed top allocate memory
    AGW_RETURN_CODE_NO_INTERSECTION, // No intersection
    AGW_RETURN_CODE_SHAPE_ALREADY_ATTACHED, // Some data/item exists
    AGW_RETURN_CODE_UNEXPECTED_DATA_TYPE, // Unexpected data type
    AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE, // Unexpected shape type
    AGW_RETURN_CODE_UNSUPPORTED_PRODUCT, // Unsupported product
    AGW_RETURN_CODE_WSI_DECODER_FAILED, // Internal WSI decoder issue
    AGW_RETURN_CODE_DUPLICATED_TILE, // Tile already exists
    AGW_RETURN_CODE_REDUNDANT_TILE, // Tile doesn't exist, but cannot be added
    AGW_RETURN_CODE_NO_STATISTIC_DATA, // Statistic information doesn't received yet
    AGW_RETURN_CODE_FILTERED_OUT, //This product processing is disabled
    AGW_RETURN_CODE_DATA_TIMEOUT, //Data packet is too old
    AGW_RETURN_CODE_NOT_EXIST //Object does not exist
} AGW_RETURN_CODE_ENUM;

// Defines internal product types controlled based on protocol description
typedef enum agw_internal_product_type_enum
{
    AGW_INTERNAL_PRODUCT_TYPE_NOWRAD = 1,
    AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES = 7,
    AGW_INTERNAL_PRODUCT_TYPE_STATISTICS = 23,
    AGW_INTERNAL_PRODUCT_TYPE_CANADIAN_RADAR = 27,
    AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES = 48,
    AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK = 64,
    AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS = 130,
    AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN
} AGW_INTERNAL_PRODUCT_TYPE_ENUM;

// Defines supported data types
typedef enum agw_internal_product_data_type_enum
{
    AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC = 0,
    AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER = 1,
    AGW_INTERNAL_PRODUCT_DATA_TYPE_UNKNOWN
} AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM;

// Defines supported shape types
typedef enum agw_internal_shape_type_enum
{
    AGW_INTERNAL_SHAPE_TYPE_FRONT = 0x00,
    AGW_INTERNAL_SHAPE_TYPE_PRESSURE_CENTER = 0x01,
    AGW_INTERNAL_SHAPE_TYPE_ISOBAR = 0x02,
    AGW_INTERNAL_SHAPE_TYPE_STORM_DESCRIPTION = 0x03,
    AGW_INTERNAL_SHAPE_TYPE_STORM_POSITION = 0x04,
    AGW_INTERNAL_SHAPE_TYPE_WIND_RADII_FIELD = 0x05
} AGW_INTERNAL_SHAPE_TYPE_ENUM;

// Defines protocol based front types
typedef enum agw_internal_front_type_enum
{
    AGW_INTERNAL_FRONT_TYPE_COLD = 0x01,
    AGW_INTERNAL_FRONT_TYPE_WARM = 0x02,
    AGW_INTERNAL_FRONT_TYPE_STATIONARY = 0x03,
    AGW_INTERNAL_FRONT_TYPE_OCCLUDED = 0x04,
    AGW_INTERNAL_FRONT_TYPE_TROUGH = 0x05,
    AGW_INTERNAL_FRONT_TYPE_SQUALL_LINE = 0x06,
    AGW_INTERNAL_FRONT_TYPE_DRY_LINE = 0x07,
    AGW_INTERNAL_FRONT_TYPE_DISSIPATING_COLD = 0x08,
    AGW_INTERNAL_FRONT_TYPE_DISSIPATING_WARM = 0x09,
    AGW_INTERNAL_FRONT_TYPE_DISSIPATING_STATIONARY = 0x0A,
    AGW_INTERNAL_FRONT_TYPE_DISSIPATING_OCCLUDED = 0x0B,
    AGW_INTERNAL_FRONT_TYPE_FORMING_COLD = 0x0C,
    AGW_INTERNAL_FRONT_TYPE_FORMING_WARM = 0x0D,
    AGW_INTERNAL_FRONT_TYPE_FORMING_STATIONARY = 0x0E,
    AGW_INTERNAL_FRONT_TYPE_FORMING_OCCLUDED = 0x0F
} AGW_INTERNAL_FRONT_TYPE_ENUM;

// Defines protocol based pressure types
typedef enum agw_internal_pressure_type_enum
{
    AGW_INTERNAL_PRESSURE_TYPE_UNKNOWN = 0,
    AGW_INTERNAL_PRESSURE_TYPE_HIGH = 1,
    AGW_INTERNAL_PRESSURE_TYPE_LOW = 2
} AGW_INTERNAL_PRESSURE_TYPE_ENUM;

// Defines protocol based storm position types
typedef enum agw_internal_storm_position_type_enum
{
    AGW_INTERNAL_STORM_POSITION_TYPE_UNKNOWN = 0x00,
    AGW_INTERNAL_STORM_POSITION_TYPE_HISTORIC = 0x01,
    AGW_INTERNAL_STORM_POSITION_TYPE_CURRENT = 0x02,
    AGW_INTERNAL_STORM_POSITION_TYPE_FORECAST = 0x03
} AGW_INTERNAL_STORM_POSITION_TYPE_ENUM;

// Defines protocol storm type types
typedef enum agw_internal_storm_type_enum
{
    AGW_INTERNAL_STORM_TYPE_UNKNOWN = 0x00,
    AGW_INTERNAL_STORM_TYPE_TROPICAL_DISTURBANCE = 0x01,
    AGW_INTERNAL_STORM_TYPE_HURRICANE_CAT_1 = 0x02,
    AGW_INTERNAL_STORM_TYPE_HURRICANE_CAT_2 = 0x03,
    AGW_INTERNAL_STORM_TYPE_HURRICANE_CAT_3 = 0x04,
    AGW_INTERNAL_STORM_TYPE_HURRICANE_CAT_4 = 0x05,
    AGW_INTERNAL_STORM_TYPE_HURRICANE_CAT_5 = 0x06,
    AGW_INTERNAL_STORM_TYPE_TROPICAL_STORM = 0x07,
    AGW_INTERNAL_STORM_TYPE_TROPICAL_DEPRESSION = 0x08,
    AGW_INTERNAL_STORM_TYPE_TYPHOON = 0x09,
    AGW_INTERNAL_STORM_TYPE_SUPER_TYPHOON = 0x0A,
    AGW_INTERNAL_STORM_TYPE_TROPICAL_CYCLONE = 0x0B
} AGW_INTERNAL_STORM_TYPE_ENUM;

// Defines raster plane types
typedef enum agw_raster_plane_enum
{
    AGW_RASTER_PLANE_0 = 0,
    AGW_RASTER_PLANE_1,
    AGW_RASTER_PLANE_ALL
} AGW_RASTER_PLANE_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Defines Generic product header
typedef struct agw_product_header_struct
{
    // Product header type
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType;

    // Product type (ID)
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType;

    // Product issue timestamp
    UN8 un8IssueMonth;
    UN8 un8IssueDay;
    UN8 un8IssueHour;
    UN8 un8IssueMinute;
    UN16 un16IssueYear;

    // Product valid timestamp
    UN8 un8ValidMonth;
    UN8 un8ValidDay;
    UN8 un8ValidHour;
    UN8 un8ValidMinute;
    UN16 un16ValidYear;

    // Upper bound for the product
    N16 n16UpperLat; /* MaxLat (-9000 - 9000) */
    N16 n16UpperLon; /* MaxLon  (-18000 - 18000) */

    // Lower bound for the product
    N16 n16LowerLat; /* MinLat (-9000 - 9000) */
    N16 n16LowerLon; /* MinLon (-18000 - 18000) */

    // Raster parameters. This fields are applicable only if
    // eDataType is AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER
    UN16 un16Rows;
    UN16 un16Columns;
    UN8 un8PixelDepth;
    UN8 un8Planes;

} AGW_PRODUCT_HEADER_STRUCT;

// Shape header
typedef struct agw_shape_header_struct
{
    // Number of shapes in this product
    UN16 un16Count;

    // Shape type
    AGW_INTERNAL_SHAPE_TYPE_ENUM eType;

    // Shape ID
    UN16 un16ID;

    // Shape data size in bytes excluding this header
    UN32 un32DataSize;

} AGW_SHAPE_HEADER_STRUCT;

// GEO Point data
typedef struct agw_geo_point_struct
{
    // Lat
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT_DATA aLatData[OSAL_FIXED_OBJECT_SIZE];

    // Lon
    OSAL_FIXED_OBJECT hLon;
    OSAL_FIXED_OBJECT_DATA aLonData[OSAL_FIXED_OBJECT_SIZE];
} AGW_GEO_POINT_STRUCT;

//  Fixed object data
typedef struct agw_fixed_object_struct
{
    OSAL_FIXED_OBJECT hValue;
    OSAL_FIXED_OBJECT_DATA aValueData[OSAL_FIXED_OBJECT_SIZE];
} AGW_FIXED_OBJECT_STRUCT;

typedef struct agw_bounding_box_struct
{
    OSAL_FIXED_OBJECT hHalfHeight;
    OSAL_FIXED_OBJECT hHalfWidth;
    OSAL_FIXED_OBJECT hCenterLat;
    OSAL_FIXED_OBJECT hCenterLon;

    BOOLEAN bIsValidBox;

    OSAL_FIXED_OBJECT_DATA aHalfHeightData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA aHalfWidthData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA aCenterLatData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA aCenterLonData[OSAL_FIXED_OBJECT_SIZE];
} AGW_BOUNDING_BOX_STRUCT;

// Location descriptor
typedef struct agw_location_desc_struct
{
    // Target location
    LOCATION_OBJECT hLocation;

    // Tile's bounding box
    AGW_BOUNDING_BOX_STRUCT sBoundingBox;
} AGW_LOCATION_DESC_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

SMSAPI_RETURN_CODE_ENUM AGW_MGR_eMapToSmsApiReturnCode(
    AGW_RETURN_CODE_ENUM eAgwReturnCode
        );

AGW_PRODUCT_TYPE_ENUM AGW_MGR_eGetPublicProductType(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    AGW_RASTER_PLANE_ENUM ePlane
        );

AGW_FRONT_TYPE_ENUM AGW_MGR_eGetPublicFrontType(
    AGW_INTERNAL_FRONT_TYPE_ENUM eType
        );

AGW_PRESSURE_TYPE_ENUM AGW_MGR_eGetPublicPressureType(
    AGW_INTERNAL_PRESSURE_TYPE_ENUM eType
        );

AGW_SHAPE_TYPE_ENUM AGW_MGR_eGetPublicShapeType(
    AGW_INTERNAL_SHAPE_TYPE_ENUM eType
        );

BOOLEAN AGW_MGR_bCreateTileFilePath (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        );

BOOLEAN AGW_MGR_bGetTileFilePathLen(
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        );

TIME_T AGW_MGR_tCreateTimeStamp(
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    BOOLEAN bShortTimeStamp
        );

const char *AGW_MGR_pacGetProductTypeName(
    AGW_PRODUCT_TYPE_ENUM eProductType
        );

const char *AGW_MGR_pacGetFrontTypeName(
    AGW_FRONT_TYPE_ENUM eFrontType
        );

const char *AGW_MGR_pacGetInternalProductTypeName(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType
        );

const char *AGW_MGR_pacGetAgwReturnCodeName(
    AGW_RETURN_CODE_ENUM eCode
        );

OSAL_FIXED_OBJECT AGW_MGR_hCreateFixedFromInteger(
    N32 n32Value,
    N8 n8Pow10
        );

OSAL_FIXED_OBJECT AGW_MGR_hCreateFixedFromIntInMem(
    N32 n32Value,
    N8 n8Pow10,
    OSAL_FIXED_OBJECT_DATA *ptFixedData
        );

BOOLEAN AGW_MGR_bCopyFixedTo(
    AGW_FIXED_OBJECT_STRUCT *psFrom,
    AGW_FIXED_OBJECT_STRUCT *psTo
        );

BOOLEAN AGW_MGR_bBoundingBoxInit(
    AGW_BOUNDING_BOX_STRUCT *psBox
        );

BOOLEAN AGW_MGR_bBoundingBoxCalculate (
    AGW_BOUNDING_BOX_STRUCT *psBox,
    OSAL_FIXED_OBJECT hNorth,
    OSAL_FIXED_OBJECT hSouth,
    OSAL_FIXED_OBJECT hWest,
    OSAL_FIXED_OBJECT hEast,
    BOOLEAN bCalcCenter
        );

BOOLEAN AGW_MGR_bBoundingBoxesIntersected(
    const AGW_BOUNDING_BOX_STRUCT *psBox1,
    const AGW_BOUNDING_BOX_STRUCT *psBox2
        );

#if (SMS_DEBUG == 1)
void AGW_MGR_vPrintBoundingBox(
    AGW_BOUNDING_BOX_STRUCT *psObj
        );
#endif

#endif /* AGW_MGR_OBJ_H_ */
