/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_MGR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <time.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dataservice_base.h"
#include "dataservice_mgr_impl.h"
#include "string_obj.h"
#include "location_obj.h"
#include "dsrl_obj.h"
#include "dsrl_target_obj.h"
#include "dsrl_entry_obj.h"

#include "agw_mgr_obj.h"
#include "_agw_mgr_obj.h"
#include "agw_tile_obj.h"
#include "agw_shape_obj.h"
#include "agw_front_obj.h"
#include "agw_isobar_obj.h"
#include "agw_pressure_center_obj.h"
#include "agw_storm_attributes_obj.h"
#include "agw_wind_radii_area_obj.h"
#include "agw_storm_position_obj.h"
#include "agw_storm_obj.h"
#include "ds_util.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static AGW_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    const char *pacRasterPath,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        )
{
    AGW_MGR_OBJECT_STRUCT *psObj =
        (AGW_MGR_OBJECT_STRUCT *)AGW_SERVICE_INVALID_OBJECT;

    do
    {
        BOOLEAN bOk;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        DATASERVICE_CREATE_STRUCT sCreate;
        DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

        // grab variable arguments (pop)
        bOk = DATASERVICE_IMPL_bProcessOptions(
            AGW_SUPPORTED_OPTIONS, psOptions, &sOptionValues);
        if (bOk == FALSE)
        {
            // Bad options!
            break;
        }

        // Populate our data service creation structure
        DATASERVICE_IMPL_vInitCreateStruct(&sCreate);
        sCreate.pacSRHDriverName = pacSRHDriverName;
        sCreate.pacServiceObjectName = AGW_MGR_OBJECT_NAME;
        sCreate.tServiceObjectSize = sizeof(AGW_MGR_OBJECT_STRUCT);
        sCreate.tDataID = (DATASERVICE_ID)AGW_DSI;

        // Suggest an OTA buffer size
        sCreate.tSuggestedOTABufferByteSize = AGW_BUFFER_BYTESIZE;

        // Configure the data service's static event attributes
        sCreate.vEventCallback = vEventHandler;
        sCreate.tEventRequestMask = (
            DATASERVICE_EVENT_ALL |
            DATASERVICE_INTERNAL_EVENT_DSRL |
            DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC );

        // Ask the data service manager controller to
        // create our manager object and do everything
        // necessary to create the underlying objects required
        // in order to support this service
        psObj = (AGW_MGR_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hCreateNewService( &sCreate );
        if (psObj == NULL)
        {
            // Can't create the service, fail out!
            break;
        }

        // Initialize asynchronous update configuration
        SMSU_vInitialize(
            &psObj->sEvent,
            psObj,
            DATASERVICE_EVENT_ALL,
            tEventRequestMask,
            (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
            pvAppEventCallbackArg);

        eReturnCode = OSAL.eLinkedListCreate(
                   &psObj->hDSRLList,
                   AGW_MGR_OBJECT_NAME":DSRLList",
                   NULL,
                   OSAL_LL_OPTION_UNIQUE
                       );

        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Initialize facing objects
        bOk = bInitAppFacingObject(psObj, pacRasterPath,
                                   sOptionValues.eImageOutputFormat);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize facing objects");
            break;
        }

        printf(AGW_MGR_OBJECT_NAME": tiles folder is set to %s\n",
            psObj->psAppObj->pacRasterFolder
                );

        // Initializing product handlers
        bOk = bProductHandlersInit(psObj);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize products control structures");
            break;
        }

        // Make the call to init the broadcast-specific parser
        psObj->psParserObj = hAgwLLParserInit((SMS_OBJECT)psObj,
                                    psObj->psAppObj->pacRasterFolder,
                                    psObj->psAppObj->eRasterFileFormat);
        if (psObj->psParserObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize Low-Level Protocol Parser");
            break;
        }

        // The service may now start
        bOk = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psObj);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                                         ": unable to create service");
            break;
        }

        return (AGW_SERVICE_OBJECT)psObj;
    } while (FALSE);

    // Error!
    if (psObj != NULL)
    {
        vUninitObject( psObj, TRUE );
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psObj);
    }

    return AGW_SERVICE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eAddFilter
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddFilter(
    AGW_SERVICE_OBJECT hAgwService,
    AGW_PRODUCT_TYPE_ENUM eProductType
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (eProductType == AGW_PRODUCT_TYPE_UNKNOWN)
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        SMS_EVENT_HDL hEvent;
        AGW_MGR_EVENT_STRUCT *psEvent = (AGW_MGR_EVENT_STRUCT *)NULL;

        // Allocate an event to perform filter update
        hEvent = DATASERVICE_IMPL_hAllocateEvent(
            (DATASERVICE_IMPL_HDL)hAgwService, (void **)&psEvent);

        if (SMS_INVALID_EVENT_HDL != hEvent)
        {
            BOOLEAN bPosted;

            // Indicate this is a filter update event
            psEvent->eEvent = AGW_MGR_EVENT_UPDATE_PRODUCT_FILTER;
            psEvent->uEvent.sUpdateProductFilter.bEnable = TRUE;
            psEvent->uEvent.sUpdateProductFilter.eProductType = eProductType;

            // Post it now
            bPosted = SMSE_bPostEvent(hEvent);

            if (bPosted == TRUE)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eRemoveFilter
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveFilter(
    AGW_SERVICE_OBJECT hAgwService,
    AGW_PRODUCT_TYPE_ENUM eProductType
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (eProductType == AGW_PRODUCT_TYPE_UNKNOWN)
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        SMS_EVENT_HDL hEvent;
        AGW_MGR_EVENT_STRUCT *psEvent = (AGW_MGR_EVENT_STRUCT *)NULL;

        // Allocate an event to perform filter update
        hEvent = DATASERVICE_IMPL_hAllocateEvent(
            (DATASERVICE_IMPL_HDL)hAgwService, (void **)&psEvent);

        if (SMS_INVALID_EVENT_HDL != hEvent)
        {
            BOOLEAN bPosted;

            // Indicate this is a filter update event
            psEvent->eEvent = AGW_MGR_EVENT_UPDATE_PRODUCT_FILTER;
            psEvent->uEvent.sUpdateProductFilter.bEnable = FALSE;
            psEvent->uEvent.sUpdateProductFilter.eProductType = eProductType;

            // Post it now
            bPosted = SMSE_bPostEvent(hEvent);

            if (bPosted == TRUE)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   AGW_MGR_bBoundingBoxInit
*
*   Initializes bounding box object
*
*****************************************************************************/
BOOLEAN AGW_MGR_bBoundingBoxInit(
    AGW_BOUNDING_BOX_STRUCT *psBox
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Make the box invalid
        psBox->bIsValidBox = FALSE;

        // OSAL_FIXED for height
        if (psBox->hHalfHeight == OSAL_FIXED_INVALID_OBJECT)
        {
            psBox->hHalfHeight =
                OSAL_FIXED.hCreateInMemory(0, 0, psBox->aHalfHeightData);
            if (psBox->hHalfHeight == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }
        }

        // OSAL_FIXED for width
        if (psBox->hHalfWidth == OSAL_FIXED_INVALID_OBJECT)
        {
            psBox->hHalfWidth =
                OSAL_FIXED.hCreateInMemory(0, 0, psBox->aHalfWidthData);
            if (psBox->hHalfWidth == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }
        }

        // OSAL_FIXED for center lat
        if (psBox->hCenterLat == OSAL_FIXED_INVALID_OBJECT)
        {
            psBox->hCenterLat =
                OSAL_FIXED.hCreateInMemory(0, 0, psBox->aCenterLatData);
            if (psBox->hCenterLat == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }
        }

        // OSAL_FIXED for center lon
        if (psBox->hCenterLon == OSAL_FIXED_INVALID_OBJECT)
        {
            psBox->hCenterLon =
                OSAL_FIXED.hCreateInMemory(0, 0, psBox->aCenterLonData);
            if (psBox->hCenterLon == OSAL_FIXED_INVALID_OBJECT)
            {
                break;
            }
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   AGW_MGR_bBoundingBoxCalculate
*
*   Calculates bounding box sizes and center point based on provided
*   coordinates.
*
*****************************************************************************/
BOOLEAN AGW_MGR_bBoundingBoxCalculate (
    AGW_BOUNDING_BOX_STRUCT *psBox,
    OSAL_FIXED_OBJECT hNorth,
    OSAL_FIXED_OBJECT hSouth,
    OSAL_FIXED_OBJECT hWest,
    OSAL_FIXED_OBJECT hEast,
    BOOLEAN bCalcCenter
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_FIXED_OBJECT hHalfDenominator = OSAL_FIXED_INVALID_OBJECT,
                      hSemiVar = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA aSemiVarData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA aHalfDenominatorData[OSAL_FIXED_OBJECT_SIZE];

    do
    {
        // Check input
        if ((psBox == NULL) ||
            (hNorth == OSAL_FIXED_INVALID_OBJECT) ||
            (hSouth == OSAL_FIXED_INVALID_OBJECT) ||
            (hWest == OSAL_FIXED_INVALID_OBJECT) ||
            (hEast == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        // Invalidate the passed box since it will be recalculated and in
        // case of error it should remain invalid because it might be in intermediate
        // state instead.
        psBox->bIsValidBox = FALSE;

        // Create temp objects
        hHalfDenominator = OSAL_FIXED.hCreateInMemory(2, 0, aHalfDenominatorData);
        hSemiVar = OSAL_FIXED.hCreateInMemory(0, 0, aSemiVarData);
        if ((hHalfDenominator == OSAL_FIXED_INVALID_OBJECT) || (hSemiVar == OSAL_FIXED_INVALID_OBJECT))
        {
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Calculate half of width (hWest < hEast), hEast < 0, hWest < 0
        eOsalReturnCode = OSAL_FIXED.eSubtract(hEast, hWest, hSemiVar);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate width (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eOsalReturnCode = OSAL_FIXED.eDivide(hSemiVar, hHalfDenominator,
                                             psBox->hHalfWidth);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate width / 2 (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Calculate half of height (hNorth > hSouth), hNorth > 0, hSouth > 0
        eOsalReturnCode = OSAL_FIXED.eSubtract(hNorth, hSouth, hSemiVar);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate height (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        eOsalReturnCode = OSAL_FIXED.eDivide(hSemiVar, hHalfDenominator,
                                             psBox->hHalfHeight);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate height / 2 (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        if (bCalcCenter == TRUE)
        {
            ///////////////////////////////////////////////////////////////////
            // Calculate Center LON
            eOsalReturnCode = OSAL_FIXED.eAdd(hEast, hWest, hSemiVar);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to calculate sum of West and East (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            eOsalReturnCode = OSAL_FIXED.eDivide(hSemiVar, hHalfDenominator,
                                                 psBox->hCenterLon);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to calculate Center LON (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            ///////////////////////////////////////////////////////////////////
            // Calculate Center LAT
            eOsalReturnCode = OSAL_FIXED.eAdd(hNorth, hSouth, hSemiVar);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to calculate sum of North and South (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            eOsalReturnCode = OSAL_FIXED.eDivide(hSemiVar, hHalfDenominator,
                                                 psBox->hCenterLat);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to calculate Center LAT (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }
        }

        // This box is good
        psBox->bIsValidBox = TRUE;
        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   AGW_MGR_eMapToSmsApiReturnCode
*
*   This function is used to map internal AGW return code into appropriate
*   SMS API ones.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM AGW_MGR_eMapToSmsApiReturnCode(
    AGW_RETURN_CODE_ENUM eAgwReturnCode
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_ERROR;

    switch (eAgwReturnCode)
    {
        case AGW_RETURN_CODE_SUCCESS:
            eResult = SMSAPI_RETURN_CODE_SUCCESS;
            break;
        case AGW_RETURN_CODE_BAD_ARGUMENT:
            eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        case AGW_RETURN_CODE_NOT_OWNER:
            eResult = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        case AGW_RETURN_CODE_NO_MEMORY:
            eResult = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        case AGW_RETURN_CODE_NOT_EXIST:
            eResult = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        default:
            // bypass the init value
            break;
    }

    return eResult;
}

/*****************************************************************************
*
*   AGW_MGR_eGetPublicProductType
*
*****************************************************************************/
AGW_PRODUCT_TYPE_ENUM AGW_MGR_eGetPublicProductType(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    AGW_RASTER_PLANE_ENUM ePlane
        )
{
    AGW_PRODUCT_TYPE_ENUM eResult = AGW_PRODUCT_TYPE_UNKNOWN;
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWInternalProductTypeMap) /
                             sizeof(*gsAGWInternalProductTypeMap);
    const AGW_INTERNAL_PRODUCT_TYPE_MAP_STRUCT *psLine = gsAGWInternalProductTypeMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->eInternalProductType == eProductType)
        {
            if (ePlane == AGW_RASTER_PLANE_1)
            {
                eResult = psLine->eAlternativeProductType;
            }
            else
            {
                eResult = psLine->eProductType;
            }
            break;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   AGW_MGR_eGetPublicFrontType
*
*****************************************************************************/
AGW_FRONT_TYPE_ENUM AGW_MGR_eGetPublicFrontType(
    AGW_INTERNAL_FRONT_TYPE_ENUM eType
        )
{
    AGW_FRONT_TYPE_ENUM eResult = AGW_FRONT_TYPE_UNKNOWN;
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWInternalFrontTypeMap) /
                             sizeof(*gsAGWInternalFrontTypeMap);
    const AGW_INTERNAL_ENUM_MAP_STRUCT *psLine = gsAGWInternalFrontTypeMap;

    for (un32CodeLine = 0; un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uInternal.eFrontType == eType)
        {
            eResult = psLine->uExternal.eFrontType;
            break;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   AGW_MGR_eGetPublicPressureType
*
*****************************************************************************/
AGW_PRESSURE_TYPE_ENUM AGW_MGR_eGetPublicPressureType(
    AGW_INTERNAL_PRESSURE_TYPE_ENUM eType
        )
{
    AGW_PRESSURE_TYPE_ENUM eResult = AGW_PRESSURE_TYPE_UNKNOWN;
    UN32 un32CodeLine = 0;
    UN32 un32CodeLineCount = sizeof(gsAGWInternalPressureTypeMap) /
                             sizeof(*gsAGWInternalPressureTypeMap);
    const AGW_INTERNAL_ENUM_MAP_STRUCT *psLine = gsAGWInternalPressureTypeMap;

    for (un32CodeLine = 0; un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uInternal.ePressureType == eType)
        {
            eResult = psLine->uExternal.ePressureType;
            break;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   AGW_MGR_eGetPublicShapeType
*
*****************************************************************************/
AGW_SHAPE_TYPE_ENUM AGW_MGR_eGetPublicShapeType(
    AGW_INTERNAL_SHAPE_TYPE_ENUM eType
        )
{
    AGW_SHAPE_TYPE_ENUM eResult = AGW_SHAPE_TYPE_UNKNOWN;
    UN32 un32CodeLine = 0;
    UN32 un32CodeLineCount = sizeof(gsAGWInternalShapeTypeMap) /
                             sizeof(*gsAGWInternalShapeTypeMap);
    const AGW_INTERNAL_ENUM_MAP_STRUCT *psLine = gsAGWInternalShapeTypeMap;

    for (un32CodeLine = 0; un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uInternal.eShapeType == eType)
        {
            eResult = psLine->uExternal.eShapeType;
            break;
        }
    }

    return eResult;
}
/*****************************************************************************
*
*   AGW_MGR_bCreateTileFilePath
*
*   Generates file path using provided tile path specified by the application,
*   AGW product header, requested file type and folder.
*
******************************************************************************/
BOOLEAN AGW_MGR_bCreateTileFilePath (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    BOOLEAN bOk = FALSE;
    IMAGE_FORMAT_ENUM eFormat;
    size_t tFolderLen, tFilenameLen;
    const char *pacFileExtension;

    do
    {
        // Check input
        if ((pacFilePath == NULL) || (pacBuffer == NULL))
        {
            break;
        }

        // Get image format and the object as well by calling IMAGE API
        eFormat = IMAGE.eFormat(hImage);
        if (eFormat == IMAGE_INVALID_FORMAT)
        {
            break;
        }

        // Get folder path
        tFolderLen = strlen(pacFilePath) + 1;
        if ((tFolderLen >= tBufferSize) || (tBufferSize == 0))
        {
            break;
        }
        strncpy(pacBuffer, pacFilePath, tBufferSize);

        // Calculate product file path
        tFilenameLen =
            tAgwLLGetProductFilePathLen(AGW_FILE_TYPE_RASTER,
                            eFormat, pacFilePath);
        if ((tFilenameLen == 0) || (tFilenameLen > tBufferSize))
        {
            break;
        }

        // Append extension
        pacFileExtension =
            pacAgwLLGetFileExtension(AGW_FILE_TYPE_RASTER, eFormat);

        // Compose the path
        snprintf(pacBuffer + (tFolderLen - 1),
                 tBufferSize - (tFolderLen - 1),
                 "/"AGW_RASTER_FILE_NAME_ID_LABEL"%s",
                 (UN32)hImage,
                 pacFileExtension
                     );

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   AGW_MGR_tCreateTimeStamp
*
*   Converts the  WSI time stamp represented by the product header into
*   POSIX timestamp.
*
*****************************************************************************/
TIME_T AGW_MGR_tCreateTimeStamp(
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    BOOLEAN bShortTimeStamp
        )
{
    TIME_T tTimeStamp = 0;
    struct tm sTimeStamp;

    OSAL.bMemSet(&sTimeStamp, 0, sizeof(sTimeStamp));

    if ((bShortTimeStamp == FALSE) && (psHeader->un16IssueYear != 0))
    {
        sTimeStamp.tm_year = psHeader->un16IssueYear - 1900; /* Adjust the year */
    }
    else
    {
        sTimeStamp.tm_year = 70; /* Start of epoch */
    }
    sTimeStamp.tm_mon = psHeader->un8IssueMonth - 1;
    sTimeStamp.tm_mday = psHeader->un8IssueDay;
    sTimeStamp.tm_hour = psHeader->un8IssueHour;
    sTimeStamp.tm_min = psHeader->un8IssueMinute;
    // No dst in effect for this time
    sTimeStamp.tm_isdst = 0;

    tTimeStamp = OSAL.mktime(&sTimeStamp); //calendar time

    return tTimeStamp;
}

/*****************************************************************************
*
*   AGW_MGR_bGetTileFilePathLen
*
*   Calculates length of the full path to the tile file file which will
*   be generated file type
*
*   Inputs:
*
*      hImage     - IMAGE object for which the path length is calculated
*      pacFilePath - folder with file to be created. it must have trailing
*                     path delimiters symbol.
*      pvSpecificData - AGW_TILE specific data which is kept by  this IMAGE
*                       object
*      ptLength - valid pointer to the path lenght result variable in case
*                 of success
*
*   Outputs:
*      TRUE if the length calculated successfully and
*      FALSE otherwise
*
******************************************************************************/
BOOLEAN AGW_MGR_bGetTileFilePathLen(
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        )
{
    BOOLEAN bResult = FALSE;

    // Check this here since no one else does this.
    if (ptLength != NULL)
    {
        IMAGE_FORMAT_ENUM eFormat;

        // Get image format w/o checking hImage object. All checks will be
        // done inside IMAGE object API and later based on format type value
        eFormat = IMAGE.eFormat(hImage);

        // Calculate the length
        (*ptLength) =
            tAgwLLGetProductFilePathLen(AGW_FILE_TYPE_RASTER,
                eFormat, pacFilePath);

        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
*
*   AGW_MGR_hCreateFixedFromInteger
*
*****************************************************************************/
OSAL_FIXED_OBJECT AGW_MGR_hCreateFixedFromInteger(
    N32 n32Value,
    N8 n8Pow10
        )
{
    OSAL_FIXED_OBJECT hResult = OSAL_FIXED_INVALID_OBJECT;
    UN32 un32FracValue = 0;
    N16 n16Denominator = 1;
    N32 n32WholeValue = 0;
    UN8 un8PowCount;
    UN8 un8TempPow = (UN8)abs(n8Pow10);

    for (un8PowCount = 0; un8PowCount < un8TempPow; ++un8PowCount)
    {
        n16Denominator *= 10;
    }

    if (n8Pow10 > 0)
    {
        un32FracValue = abs(n32Value % n16Denominator);
        n32WholeValue = n32Value / n16Denominator;
        hResult = OSAL_FIXED.hCreate(n32WholeValue, un32FracValue, un8TempPow);
    }
    else
    {
        n32WholeValue = n32Value * n16Denominator;
        hResult = OSAL_FIXED.hCreate(n32WholeValue, un32FracValue, 0);
    }

    return hResult;
}

/*****************************************************************************
*
*   AGW_MGR_hCreateFixedFromIntInMem
*
*****************************************************************************/
OSAL_FIXED_OBJECT AGW_MGR_hCreateFixedFromIntInMem(
    N32 n32Value,
    N8 n8Pow10,
    OSAL_FIXED_OBJECT_DATA *ptFixedData
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_FIXED_OBJECT hTempFixed = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hResult = OSAL_FIXED_INVALID_OBJECT;

    do
    {
        hTempFixed = AGW_MGR_hCreateFixedFromInteger(
            n32Value, n8Pow10
                );
        if (hTempFixed == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create temporary fixed object"
                        );
            break;
        }

        hResult = OSAL_FIXED.hCreateInMemory(0, 0, ptFixedData);
        if (hResult == OSAL_FIXED_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to create fixed in %p memory",
                ptFixedData
            );
            break;
        }
        bOk = OSAL_FIXED.bCopyToMemory(hTempFixed, hResult);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to copy data from temp fixed %p to "
                    "new fixed %p",
                    hTempFixed, hResult
                        );
            break;
        }
    } while (FALSE);

    if (hTempFixed != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(hTempFixed);
    }

    return hResult;
}

/*****************************************************************************
*
*   AGW_MGR_bCopyFixedTo
*
*****************************************************************************/
BOOLEAN AGW_MGR_bCopyFixedTo(
    AGW_FIXED_OBJECT_STRUCT *psFrom,
    AGW_FIXED_OBJECT_STRUCT *psTo
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        // Check input
        if ((psFrom == NULL) || (psTo == NULL))
        {
            break;
        }

        // If the source object represents empty value
        if (psFrom->hValue == OSAL_FIXED_INVALID_OBJECT)
        {
            psTo->hValue = OSAL_FIXED_INVALID_OBJECT;
            bResult = TRUE;
            break;
        }

        // Create object using room inside the TO
        psTo->hValue = OSAL_FIXED.hCreateInMemory(0, 1, psTo->aValueData);
        if (psTo->hValue == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        bResult = OSAL_FIXED.bCopyToMemory(psFrom->hValue, psTo->hValue);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   AGW_MGR_pacGetProductTypeName
*
*   Provides string representation for product type
*
*   Inputs:
*
*       eProductType - product type to know its name
*
*   Outputs:
*
*       String which represents requested type extension.
*
******************************************************************************/
const char *AGW_MGR_pacGetProductTypeName(
    AGW_PRODUCT_TYPE_ENUM eProductType
        )
{
    const char *pacResult = "UNKNOWN";
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWProductTypeNameMap) /
                             sizeof(*gsAGWProductTypeNameMap);
    const AGW_ENUM_TYPE_MAP_STRUCT *psLine = gsAGWProductTypeNameMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uType.eProductType == eProductType)
        {
            pacResult = psLine->pacText;
            break;
        }
    }
    return pacResult;
}

/*****************************************************************************
*
*   AGW_MGR_pacGetFrontTypeName
*
*   Provides string representation for front type
*
*   Inputs:
*
*       eFrontType - front type to know its name
*
*   Outputs:
*
*       String which represents requested type extension.
*
******************************************************************************/
const char *AGW_MGR_pacGetFrontTypeName(
    AGW_FRONT_TYPE_ENUM eFrontType
        )
{
    const char *pacResult = "UNKNOWN";
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWFrontTypeNameMap) /
                             sizeof(*gsAGWFrontTypeNameMap);
    const AGW_ENUM_TYPE_MAP_STRUCT *psLine = gsAGWFrontTypeNameMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uType.eFrontType == eFrontType)
        {
            pacResult = psLine->pacText;
            break;
        }
    }
    return pacResult;
}

/*****************************************************************************
*
*   AGW_MGR_pacGetInternalProductTypeName
*
*   Provides string representation for product type
*
* Inputs:
*
*   eProductType - product type to know its name
*
* Outputs:
*
*   String which represents requested type extension.
*
******************************************************************************/
const char *AGW_MGR_pacGetInternalProductTypeName(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType
        )
{
    const char *pacResult = "UNKNOWN";
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWInternalProductTypeNameMap) /
                             sizeof(*gsAGWInternalProductTypeNameMap);
    const AGW_ENUM_TYPE_MAP_STRUCT *psLine = gsAGWInternalProductTypeNameMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uType.eInternalProductType == eProductType)
        {
            pacResult = psLine->pacText;
            break;
        }
    }
    return pacResult;
}

/*****************************************************************************
*
*   AGW_MGR_pacGetAgwReturnCodeName
*
*****************************************************************************/
const char *AGW_MGR_pacGetAgwReturnCodeName(
    AGW_RETURN_CODE_ENUM eCode
        )
{
    const char *pacResult = "Unknown";
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWReturnCodeNameMap) /
                             sizeof(*gsAGWReturnCodeNameMap);
    const AGW_ENUM_TYPE_MAP_STRUCT *psLine = gsAGWReturnCodeNameMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uType.eReturnCode == eCode)
        {
            pacResult = psLine->pacText;
            break;
        }
    }

    return pacResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)

/*****************************************************************************
*
*   vPrintCacheHeader
*
*****************************************************************************/
static void vPrintCacheHeader(
    AGW_CACHE_HEADER_STRUCT *psCacheHeader
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": Cache header: product - %s, timestamp %d, size %d\n",
           AGW_MGR_pacGetProductTypeName(psCacheHeader->sStatInfo.eType),
           psCacheHeader->un32TimeStamp,
           psCacheHeader->tPayloadSize
                  );
    return;
}

/*****************************************************************************
*
*   vPrintHeader
*
*****************************************************************************/
static void vPrintHeader(
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    printf(AGW_MGR_OBJECT_NAME
           "\tID: %s (%s)\n"
           "\tI:%02d/%02d %02d:%02d, V:%02d/%02d %02d:%02d\n"
           "\tUP(%7d, %7d), LO(%7d, %7d)\n",
           AGW_MGR_pacGetInternalProductTypeName(psHeader->eProductType),
           pacAgwLLGetProductDataTypeName(psHeader->eDataType),
           psHeader->un8IssueMonth, psHeader->un8IssueDay,
           psHeader->un8IssueHour, psHeader->un8IssueMinute,
           psHeader->un8ValidMonth, psHeader->un8ValidDay,
           psHeader->un8ValidHour, psHeader->un8ValidMinute,
           psHeader->n16UpperLat,
           psHeader->n16UpperLon,
           psHeader->n16LowerLat,
           psHeader->n16LowerLon
                  );
    return;
}

/*****************************************************************************
*
*   vPrintStatisticsProduct
*
*****************************************************************************/
static void vPrintStatisticsProduct(
    AGW_STATISTICS_STRUCT *psStatisticProduct
        )
{
    AGW_STATISTICS_ITEM_STRUCT *psProductInfo = NULL;
    UN16 un16Product;

    printf(AGW_MGR_OBJECT_NAME
           ": statistic by year %4d for %d (of %d) products\n",
           psStatisticProduct->un16Year,
           psStatisticProduct->un16NumberOfUsedPID,
           psStatisticProduct->un16NumberPID
               );
    for (un16Product = 0;
         un16Product < psStatisticProduct->un16NumberPID;
         ++un16Product)
    {
        psProductInfo = &psStatisticProduct->aInfo[un16Product];
        if ((psProductInfo->un8Expected > 0) || (psProductInfo->un8Sent > 0))
        {
            printf("\t%03d => %04d of %04d\n",
                    un16Product,
                    psProductInfo->un8Sent,
                    psProductInfo->un8Expected
                        );
        }
    }
    return;
}

/*****************************************************************************
*
*   vPrintStatisticsCollection
*
*****************************************************************************/
static void vPrintStatisticsCollection(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eResult;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry = NULL;
    UN32 un32Items;
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
    char aBuffer[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
#endif
    UN32 un32Item = 0;

    eResult = OSAL.eLinkedListItems(psObj->hStatistics, &un32Items);
    hEntry = OSAL.hLinkedListFirst(psObj->hStatistics, (void**)&psEntry);
    while (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        printf(AGW_MGR_OBJECT_NAME
               ": %02d of %02d => %s\n", un32Item + 1, un32Items,
               OSAL.ctime_r(&psEntry->tTimeStamp,aBuffer));
        ++un32Item;

        hEntry = OSAL.hLinkedListNext(hEntry, (void**)&psEntry);
    }
    return;
}

/*****************************************************************************
*
*   vPrintStormAttribute
*
*****************************************************************************/
//static void vPrintStormAttribute(
//    AGW_STORM_ATTRIBUTE_STRUCT *psObj
//        )
//{
//    printf(AGW_MGR_OBJECT_NAME
//           "TIME(%02d:%02d) COORD(%06d, %07d)"
//           "ECHO(H:%07u, DIR:%03u, SPD:%03u) FLAGS(%02X) ID(%*s)\n",
//           psObj->un8Hour, psObj->un8Minute, psObj->n32Lat, psObj->n32Lon,
//           psObj->un16EchoTop, psObj->un16Direction, psObj->un16Speed,
//           psObj->un8Flags,
//           AGW_STORM_ATTR_UNIQUE_ID_BYTELEN, psObj->aun8ID
//                  );
//    return;
//}

/*****************************************************************************
*
*   vPrintShapeHeader
*
*****************************************************************************/
static void vPrintShapeHeader(
    AGW_SHAPE_HEADER_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
            ": SHAPE HDR (C:%2d, T:%2d, ID: %6d, SIZE:%6d)\n",
            psObj->un16Count, psObj->eType,
            psObj->un16ID, psObj->un32DataSize
                );
    return;
}

#endif /* (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1) */

#if (SMS_DEBUG==1) && (DEBUG_OBJECT == 1)
/*****************************************************************************
*
*   AGW_MGR_vPrintBoundingBox
*
*****************************************************************************/
void AGW_MGR_vPrintBoundingBox(
    AGW_BOUNDING_BOX_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME": BOX \n");
    printf("\tIS VALID: %s\n", (psObj->bIsValidBox == TRUE ? "YES" : "NO"));
    printf("\tCenter LAT: ");
    OSAL_FIXED.n32FPrintf(psObj->hCenterLat, stdout, FALSE);
    printf("\tCenter LON: ");
    OSAL_FIXED.n32FPrintf(psObj->hCenterLon, stdout, FALSE);
    printf("\tHalf Width: ");
    OSAL_FIXED.n32FPrintf(psObj->hHalfWidth, stdout, FALSE);
    printf("\tHalf Heigh: ");
    OSAL_FIXED.n32FPrintf(psObj->hHalfHeight, stdout, FALSE);
    return;
}
#endif

/*****************************************************************************
*
*   hAgwLLParserInit
*
******************************************************************************/
static AGW_PARSER_OBJECT_STRUCT *hAgwLLParserInit (
    SMS_OBJECT hOwner,
    const char *pcTempPath,
    IMAGE_FORMAT_ENUM eImageFormat
        )
{
    AGW_PARSER_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eReturn = OSAL_ERROR;

    do
    {
        // Create an instance of this object
        psObj = (AGW_PARSER_OBJECT_STRUCT *)
            SMSO_hCreate(
                AGW_PARSER_OBJECT_NAME":Obj",
                sizeof(AGW_PARSER_OBJECT_STRUCT),
                SMS_INVALID_OBJECT,
                TRUE
                    );
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Initialize members
        psObj->sInputFile.pacData = NULL;
        psObj->sInputFile.tSize = 0;
        psObj->sInputFile.pFile = NULL;
        psObj->sOutputFile.pacData = NULL;
        psObj->sOutputFile.tSize = 0;
        psObj->sOutputFile.pFile = NULL;
        psObj->sTempBuffer.pacData = NULL;
        psObj->sTempBuffer.tSize = 0;
        psObj->sTempBuffer.pFile = NULL;

        psObj->hPayload = OSAL_INVALID_BUFFER_HDL;

        psObj->eOutputImageFormat = eImageFormat;

        // Request access to ISO 3309 CRC32
        eReturn = OSAL.eGetCRC(
                &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);

        if (eReturn != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to create CRC handler (%s)",
                    OSAL.pacGetReturnCodeName(eReturn));
            break;
        }

        // Creating WSI Decoder
        psObj->hWsiDecoder = GsWsiDecoderIntf.hInit((SMS_OBJECT)psObj, eImageFormat);
        if (psObj->hWsiDecoder == WSI_DECODER_INVALID_OBJECT)
        {
            break;
        }

        // Copy temp file path
        if (pcTempPath != NULL)
        {
            size_t tLength;

            tLength = strlen(pcTempPath);
            psObj->pcTempPath =
                (char *) SMSO_hCreate(
                            AGW_PARSER_OBJECT_NAME":TmpPath",
                            tLength + 1,
                            (SMS_OBJECT)psObj, FALSE );
            if (psObj->pcTempPath != NULL)
            {
                strncpy(psObj->pcTempPath, pcTempPath, tLength + 1);
            }
        }
        else
        {
            psObj->pcTempPath = NULL;
        }

        SMSO_vUnlock((SMS_OBJECT)psObj);

        return psObj;
    } while (FALSE);

    if (psObj != NULL)
    {
        // Error!
        vAgwLLParserUninitParser(psObj);
    }

    return NULL;
}

/*****************************************************************************
*
*   eAgwLLParserValidateAU
*
*   Validates provided AU using PVN, CARID and trailing CRC32
*
* Inputs:
*
*   psObj         - reference to AGW_PARSER_OBJECT_STRUCT for which the function
*                   is called.
*   phPayload     - buffer with Access Unit data
*
* Outputs:
*   Execution error code
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserValidateAU(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    UN8 un8Tmp = 0;
    BOOLEAN bOk = FALSE;

    do
    {
        // Read PVN
        OSAL.tBufferPeekBits(hPayload, &un8Tmp, 0, AGW_AU_PVN_BITLEN,
                AGW_AU_PVN_OFFSET
                    );
        if (un8Tmp != AGW_PVN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": unsupported PVN %x", un8Tmp);
            eReturnCode = AGW_RETURN_CODE_PVN_UNSUPPORTED;
            break;
        }

        // Read Carousel ID
        un8Tmp = 0;
        OSAL.tBufferPeekBits(hPayload, &un8Tmp, 0, AGW_AU_CARID_BITLEN,
                        AGW_AU_CARID_OFFSET
                            );
        if (un8Tmp != AGW_CARID)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": unsupported CARID %x", un8Tmp);
            eReturnCode = AGW_RETURN_CODE_PVN_UNSUPPORTED;
            break;
        }

        // Read and verify CRC
        bOk = DS_UTIL_bIsCRCValid(psObj->hCRC, hPayload, NULL);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": CRC is not valid.");
            eReturnCode = AGW_RETURN_CODE_CRC_INVALID;
            break;
        }

        // Cut the CRC from the buffer
        bOk = DS_UTIL_bCutCRC(hPayload);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": Failed to cut CRC.");
            break;
        }

        // Seek the head position to the WSI product start
        OSAL.tBufferSeekHead(hPayload, AGW_WSI_PRODUCT_BYTEOFFSET);

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserParseHeader
*
*   Parses passed payload data. The payload buffer assumed to be a full
*   AccessUnit for AGW and it can be processed to be converted into DPU.
*
* Inputs:
*
*   psObj         - reference to AGW_PARSER_OBJECT_STRUCT for which the function
*                   is called.
*   hPayload      - buffer with Access Unit data
*   tOffsetInBits - offset in bits to the needed data in the payload
*   psHeader      - filled by the product header information in case of success
*
* Outputs:
*   Execution error code
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserParseHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    size_t tOffsetInBits,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType = AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC;
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType = AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN;
    N16 n16Value = 0;
    BOOLEAN bOk = FALSE;
    size_t tBufferSize = 0;
    size_t tBufferPeekedBits = 0;

    do
    {
        OSAL.bMemSet(psHeader, 0, sizeof(*psHeader));

        // Local variables from the payload
        if (psObj == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Check payload size
        tBufferSize = OSAL.tBufferGetSizeInBits(hPayload) - tOffsetInBits;
        if (tBufferSize < AGW_AU_HEADER_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": not enough data for parsing generic product header"
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        // Read bits header
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload, &eDataType,
                0, AGW_AU_PRODUCT_TYPE_BITLEN,
                AGW_AU_PRODUCT_TYPE_OFFSET + tOffsetInBits
                    );

        bOk = OSAL.bBufferPeekBitsToN16(hPayload, &n16Value,
                AGW_AU_PRODUCT_ID_BITLEN,
                AGW_AU_PRODUCT_ID_OFFSET + tOffsetInBits
                    );
        if (bOk == FALSE)
        {
            break;
        }
        tBufferPeekedBits += AGW_AU_PRODUCT_ID_BITLEN;
        eProductType = (AGW_INTERNAL_PRODUCT_TYPE_ENUM)n16Value;

        psHeader->eDataType = eDataType;
        psHeader->eProductType = eProductType;

        // Filter out not-supported products to prevent WSI Decoder
        // execution.
        bOk = bAgwLLParserIsProductSupported(psHeader->eDataType,
                            psHeader->eProductType
                                );
        if (bOk == FALSE)
        {
#if ((DEBUG_OBJECT == 1) || (SMS_DEBUG == 1))
            printf(AGW_MGR_OBJECT_NAME": unsupported %s of %s data type\n",
                    AGW_MGR_pacGetInternalProductTypeName(psHeader->eProductType),
                    pacAgwLLGetProductDataTypeName(psHeader->eDataType)
                        );
#endif
            eReturnCode = AGW_RETURN_CODE_UNSUPPORTED_PRODUCT;
            break;
        }

        // Read rest fields of the product header
        psHeader->un16IssueYear = 0;
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8IssueMonth), 0, AGW_AU_ISSUE_MONTH_BITLEN,
                AGW_AU_ISSUE_MONTH_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8IssueDay), 0, AGW_AU_ISSUE_DAY_BITLEN,
                AGW_AU_ISSUE_DAY_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8IssueHour), 0, AGW_AU_ISSUE_HOUR_BITLEN,
                AGW_AU_ISSUE_HOUR_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8IssueMinute), 0, AGW_AU_ISSUE_MINUTE_BITLEN,
                AGW_AU_ISSUE_MINUTE_OFFSET + tOffsetInBits
                    );

        psHeader->un16ValidYear = 0;
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8ValidMonth), 0, AGW_AU_VALID_MONTH_BITLEN,
                AGW_AU_VALID_MONTH_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8ValidDay), 0, AGW_AU_VALID_DAY_BITLEN,
                AGW_AU_VALID_DAY_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8ValidHour), 0, AGW_AU_VALID_HOUR_BITLEN,
                AGW_AU_VALID_HOUR_OFFSET + tOffsetInBits
                    );
        tBufferPeekedBits += OSAL.tBufferPeekBits(hPayload,
                &(psHeader->un8ValidMinute), 0, AGW_AU_VALID_MINUTE_BITLEN,
                AGW_AU_ISSUE_MINUTE_OFFSET + tOffsetInBits
                    );

        bOk = OSAL.bBufferPeekBitsToN16(hPayload, &psHeader->n16UpperLat,
                AGW_AU_MAX_LAT_BITLEN,
                AGW_AU_MAX_LAT_OFFSET + tOffsetInBits
                    );
        if (bOk == FALSE)
        {
            break;
        }
        tBufferPeekedBits += AGW_AU_MAX_LAT_BITLEN;

        bOk = OSAL.bBufferPeekBitsToN16(hPayload, &psHeader->n16UpperLon,
                AGW_AU_MIN_LON_BITLEN,
                AGW_AU_MIN_LON_OFFSET + tOffsetInBits
                    );
        if (bOk == FALSE)
        {
            break;
        }
        tBufferPeekedBits += AGW_AU_MIN_LON_BITLEN;

        bOk = OSAL.bBufferPeekBitsToN16(hPayload, &psHeader->n16LowerLat,
                AGW_AU_MIN_LAT_BITLEN,
                AGW_AU_MIN_LAT_OFFSET + tOffsetInBits
                    );
        if (bOk == FALSE)
        {
            break;
        }
        tBufferPeekedBits += AGW_AU_MIN_LAT_BITLEN;

        bOk = OSAL.bBufferPeekBitsToN16(hPayload, &psHeader->n16LowerLon,
                AGW_AU_MAX_LON_BITLEN,
                AGW_AU_MAX_LON_OFFSET + tOffsetInBits
                    );
        if (bOk == FALSE)
        {
            break;
        }
        tBufferPeekedBits += AGW_AU_MAX_LON_BITLEN;

        // Ckeck size of peeked bits using expected value
        if (tBufferPeekedBits != AGW_AU_HEADER_BITLEN)
        {
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserAssignPayload
*
*   Assigns provided payload to PARSER controlled buffer.
*
* Inputs:
*
*   psObj         - reference to AGW_PARSER_OBJECT_STRUCT for which the function
*                   is called.
*   phPayload     - pointer to the buffer with Access Unit data
*
* Outputs:
*   Execution error code
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserAssignPayload(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
    BOOLEAN bOk = TRUE;

    if (phPayload != NULL && *phPayload != OSAL_INVALID_BUFFER_HDL)
    {
        if (psObj->hPayload != OSAL_INVALID_BUFFER_HDL)
        {
            bOk = DATASERVICE_IMPL_bFreeDataPayload(psObj->hPayload);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": failed to release payload");
                eReturnCode = AGW_RETURN_CODE_ERROR;
            }
        }

        if( bOk == TRUE )
        {
            psObj->hPayload = *phPayload;
            *phPayload = OSAL_INVALID_BUFFER_HDL;
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessStatisticsFillUp
*
*   This function fills up the statistics data using protocol layer data.
*   The caller is able to specify the year of the statistics and subsequence
*   of pairs (product id and number of items)
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStatisticsFillUp (
    const AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_STATISTICS_STRUCT *psStatisticsData,
    UN16 un16Year,
    int iNumberPID,
    ...
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;

    if (psStatisticsData != NULL)
    {
        va_list args;

        // Clean up the memory
        OSAL.bMemSet(psStatisticsData, 0, sizeof(*psStatisticsData));

        // Fill in regular fields
        psStatisticsData->sHeader = *psHeader;
        psStatisticsData->sHeader.eDataType =
            AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC;
        psStatisticsData->sHeader.eProductType =
            AGW_INTERNAL_PRODUCT_TYPE_STATISTICS;
        psStatisticsData->un16NumberPID = AGW_STATISTICS_MAX_PRODUCTS;
        psStatisticsData->un16NumberOfUsedPID = 0;
        psStatisticsData->un16Year = un16Year;

        // Reading arguments
        va_start(args, iNumberPID);

        while ((iNumberPID--) > 0)
        {
            AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType;
            BOOLEAN bIsSupported;
            int iProductType, iCount;

            // Read of product type and count with folliwing validation
            iProductType = va_arg(args, int);
            iCount = va_arg(args, int);
            if ((iProductType < 0) || (iProductType > AGW_STATISTICS_ITEMS_MAX))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": incorrect product #%d",
                    iProductType);
                continue;
            }
            if ((iCount <= 0) || (iCount > UN8_MAX))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": incorrect count #%d",
                    iCount);
                continue;
            }

            // Get the product type
            eProductType = (AGW_INTERNAL_PRODUCT_TYPE_ENUM)iProductType;

            // Is product valid?
            bIsSupported =
                bAgwLLParserIsProductSupported(
                    AGW_INTERNAL_PRODUCT_DATA_TYPE_UNKNOWN, eProductType);

            // Update the products array if everything else seems good
            if (bIsSupported == TRUE)
            {
                AGW_STATISTICS_ITEM_STRUCT *psItem =
                    psStatisticsData->aInfo + eProductType;

                // Update counters
                psItem->un8Expected = (UN8)iCount;
                psItem->un8Sent = (UN8)iCount;

                // Update no. of valid products
                ++psStatisticsData->un16NumberOfUsedPID;
            }
        }

        // We've done
        va_end(args);

        // We're ok
        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessStatisticsProduct
*
*   Process the payload ad Product Statistics data
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStatisticsProduct(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_STRUCT *psStatisticsData
        )
{
    OSAL_BUFFER_HDL hBuffer;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_STATISTICS_ITEM_STRUCT *psTile = NULL;
    UN16 un16Tile = 0;
    size_t tBytesRead = 0;
    size_t tHasToRead = 0;

    do
    {
        // Process payload data
        eReturnCode = eAgwLLParserProcessPayload(psObj,
                            &psStatisticsData->sHeader
                                );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process payload (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Processes product header
        eReturnCode = eAgwLLParserProcessProductHeader(psObj,
                        &psStatisticsData->sHeader
                            );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": [%d] failed to process product header (%s)",
                   __LINE__,
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        hBuffer = psObj->hBuffer;

        // Read product data
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psStatisticsData->un16Year,
                AGW_PRODUCT_STAT_YEAR_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psStatisticsData->un16NumberPID,
                AGW_PRODUCT_STAT_NUMBER_PID_BYTELEN
                    );

        if (psStatisticsData->un16NumberPID >= AGW_STATISTICS_MAX_PRODUCTS)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        for (un16Tile = 0; un16Tile < psStatisticsData->un16NumberPID; ++un16Tile)
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

            psTile = &psStatisticsData->aInfo[un16Tile];

            // Read tile info
            tBytesRead += OSAL.tBufferReadHead(hBuffer, &psTile->un8Sent,
                AGW_PRODUCT_STAT_EXPECTED_TILES_SENT_BYTELEN
                    );
            tBytesRead += OSAL.tBufferReadHead(hBuffer, &psTile->un8Expected,
                AGW_PRODUCT_STAT_EXPECTED_NUMBER_TILES_BYTELEN
                    );

            // If the tile data has correct values count it as used.
            if ((psTile->un8Expected > 0) || (psTile->un8Sent > 0))
            {
                ++psStatisticsData->un16NumberOfUsedPID;
            }
        }

        // Check number of read bytes
        tHasToRead = AGW_PRODUCT_STAT_YEAR_BYTELEN +
                     AGW_PRODUCT_STAT_NUMBER_PID_BYTELEN +
                    (AGW_PRODUCT_STAT_EXPECTED_TILES_SENT_BYTELEN +
                     AGW_PRODUCT_STAT_EXPECTED_NUMBER_TILES_BYTELEN) *
                     psStatisticsData->un16NumberPID;
        if (tHasToRead != tBytesRead)
        {
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    // Release internal buffer if exists
    (void)eAgwLLParserCleanupBuffer(psObj);

    (void)eAgwLLParserRemoveOutputFile(psObj);

    return eReturnCode;
}

/*****************************************************************************
*
*   tAgwLLGetObservationTimeStamp
*
*   Returns observation timestamp value from time/day/month/year
*
******************************************************************************/
static TIME_T tAgwLLGetObservationTimeStamp(
    UN8 un8Minute,
    UN8 un8Hour,
    UN8 un8MDay,
    UN8 un8Month,
    UN16 un16Year
        )
{
    struct tm sTime;
    TIME_T tResult = 0;

    OSAL.bMemSet(&sTime, 0, sizeof(sTime));

    sTime.tm_min = un8Minute;
    sTime.tm_hour = un8Hour;
    sTime.tm_mday = un8MDay;
    sTime.tm_mon = un8Month;
    sTime.tm_year = un16Year - 1900; /* Adjust the year */
    sTime.tm_isdst = 0; // No dst in effect for this time

    tResult = OSAL.mktime(&sTime); // calendar time

    return tResult;
}

/*****************************************************************************
*
*   eAgwLLParserProcessStormAttribute
*
*   Process the payload as Storm Attributes
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStormAttribute(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_STORM_ATTRIBUTES_OBJECT hStormAttr,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    size_t tBytesRead = 0;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    BOOLEAN bOk = FALSE;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;

    UN8 un8Hour = 0;
    UN8 un8Minute = 0;
    N32 n32Lat = 0;
    N32 n32Lon = 0;
    UN8 un8Flags = 0;
    UN16 un16Buffer = 0;
    AGW_STORM_ATTRIBUTES_DATA_STRUCT sStormAttrData;
    UN8 aun8ID[AGW_STORM_ATTR_UNIQUE_ID_BYTELEN + 1];

    do
    {
        // Init local structures
        OSAL.bMemSet(aun8ID, 0, sizeof(aun8ID));
        OSAL.bMemSet(&sStormAttrData, 0, sizeof(sStormAttrData));

        // Check input
        if ((hStormAttr == NULL) || (psObj == NULL) || (psHeader == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un8Hour,
                                AGW_STORM_ATTR_OBS_HOUR_BYTELEN
                                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un8Minute,
                                AGW_STORM_ATTR_OBS_MINUTE_BYTELEN
                                    );

        sStormAttrData.tObservationTime = tAgwLLGetObservationTimeStamp(
            un8Minute, un8Hour,
            psHeader->un8IssueDay,
            (UN8)(psHeader->un8IssueMonth - 1),
            psHeader->un16IssueYear
                );

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &n32Lon,
            AGW_STORM_ATTR_LON_BYTELEN
                );

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &n32Lat,
            AGW_STORM_ATTR_LAT_BYTELEN
                );


        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un16Buffer,
            AGW_STORM_ATTR_ECHO_BYTELEN
                );

        sStormAttrData.sEchoTop.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                AGW_STORM_ATTR_ECHO_POW10,
                sStormAttrData.sEchoTop.aValueData
                    );

        un16Buffer = 0;
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un16Buffer,
            AGW_STORM_ATTR_DIR_BYTELEN
                );

        if (un16Buffer != AGW_STORM_ATTR_DIR_NA)
        {
            sStormAttrData.bDirectionAvailable = TRUE;
            sStormAttrData.sDirection.hValue =
                AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                    0,
                    sStormAttrData.sDirection.aValueData
                        );
        }

        un16Buffer = 0;
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un16Buffer,
            AGW_STORM_ATTR_SPEED_BYTELEN
                );

        if (un16Buffer != AGW_STORM_ATTR_SPEED_NA)
        {
            sStormAttrData.bSpeedAvailable = TRUE;
            sStormAttrData.sSpeed.hValue =
                AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                    0,
                    sStormAttrData.sSpeed.aValueData
                        );
        }

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un8Flags,
                                AGW_STORM_ATTR_FLAG_BYTELEN
                                    );

        if (un8Flags & AGW_STORM_ATTR_FLAG_HAIL_PROBABLE)
        {
            sStormAttrData.bHailProbable = TRUE;
        }
        if (un8Flags & AGW_STORM_ATTR_FLAG_HAIL_POSITIVE)
        {
            sStormAttrData.bHailPositive = TRUE;
        }
        if (un8Flags & AGW_STORM_ATTR_FLAG_MESOCYCLONIC)
        {
            sStormAttrData.bMesocyclonic = TRUE;
        }
        if (un8Flags & AGW_STORM_ATTR_FLAG_TORNADIC)
        {
            sStormAttrData.bTornadic = TRUE;
        }

        tBytesRead += OSAL.tBufferReadHead(hBuffer, aun8ID,
                                AGW_STORM_ATTR_UNIQUE_ID_BYTELEN
                                    );

        if (tBytesRead != AGW_STORM_ATTR_BYTESIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": number of bytes read is not correct: %d"
                    " instead of %d (%s)",
                    tBytesRead,
                    AGW_STORM_ATTR_BYTESIZE,
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        sStormAttrData.hStormID = STRING.hCreate((char *)aun8ID, 0);
        if (sStormAttrData.hStormID == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get storm ID (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        bOk = AGW_STORM_ATTRIBUTES_bUpdate(hStormAttr, &sStormAttrData);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hStormAttr, 1);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hStormAttr, 0, n32Lat, n32Lon,
            AGW_STORM_ATTRIBUTES_COORDINATE_POW10
                );
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (sStormAttrData.hStormID != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sStormAttrData.hStormID);
        sStormAttrData.hStormID = STRING_INVALID_OBJECT;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessStormAttributes
*
*   Process the payload as Storm Attributes
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStormAttributes(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    UN16 *pun16AttrCount
        )
{
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    size_t tReadBytes = 0;

    do
    {
        // Check input
        if ((psObj == NULL) || (psHeader == NULL) || (pun16AttrCount == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Process payload
        eReturnCode = eAgwLLParserProcessPayload(psObj, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process payload (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Processes product header
        eReturnCode = eAgwLLParserProcessProductHeader(psObj, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process product header (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        hBuffer = psObj->hBuffer;

        // Read number of attributes
        *pun16AttrCount = 0;
        tReadBytes = OSAL.tBufferReadHead(hBuffer, pun16AttrCount,
            AGW_STORM_ATTR_COUNT_BYTELEN
                );
        if (tReadBytes != AGW_STORM_ATTR_COUNT_BYTELEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to read number of shape types in the buffer"
                          );
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        printf(AGW_MGR_OBJECT_NAME": found %u storm attribute(s)\n",
                *pun16AttrCount);

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserGetShapeTypeCount
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserGetShapeTypeCount(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    UN16 *pun16ShapeCount
        )
{
    size_t tReadBytes;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;

    // Read number of attributes
    *pun16ShapeCount = 0;
    tReadBytes = OSAL.tBufferReadHead(psObj->hBuffer, pun16ShapeCount,
                            AGW_SHAPE_TYPE_COUNT_BYTELEN
                                );
    if (tReadBytes != AGW_SHAPE_TYPE_COUNT_BYTELEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
               ": failed to read number of shape types in the buffer"
                      );
        eReturnCode = AGW_RETURN_CODE_IO_ERROR;
    }

    printf(AGW_MGR_OBJECT_NAME": found %u shape(s)\n",
            *pun16ShapeCount);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessShapeDataBasedProduct
*
*   Process the payload as Shape Based product
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeDataBasedProduct(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psObj == NULL) || (psHeader == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Process payload
        eReturnCode = eAgwLLParserProcessPayload(psObj, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process payload (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Processes product header
        eReturnCode = eAgwLLParserProcessProductHeader(psObj, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process product header (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessShapeHeader
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_SHAPE_HEADER_STRUCT *psShapeHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    UN32 un32Value = 0;
    size_t tReadSize = 0;

    do
    {
        // Check input
        if ((psObj == NULL) || (psShapeHeader == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, psObj->hBuffer);

        hBuffer = psObj->hBuffer;

        OSAL.bMemSet(psShapeHeader, 0, sizeof(*psShapeHeader));

        // Read number of shapes in the product
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &psShapeHeader->un16Count, AGW_SHAPE_COUNT_BYTELEN
                                );

        // Read shape type
        tReadSize += OSAL.tBufferReadHead(hBuffer, &un32Value,
                            AGW_SHAPE_TYPE_BYTELEN
                                );
        psShapeHeader->eType = (AGW_INTERNAL_SHAPE_TYPE_ENUM)un32Value;

        // Read shape identifier
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &psShapeHeader->un16ID, AGW_SHAPE_ID_BYTELEN
                                );

        // Read shape data size
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &psShapeHeader->un32DataSize, AGW_SHAPE_SIZE_BYTELEN
                                );

        // Check all read data
        if (AGW_SHAPE_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Shape header (E: %u, R: %u)",
                    AGW_SHAPE_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessShapeGeoPoint
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeGeoPoint(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    N16 *pn16Lat,
    N16 *pn16Lon
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    size_t tReadSize = 0;

    do
    {
        // Check input
        if ((psObj == NULL) || (pn16Lat == NULL) || (pn16Lon == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        tReadSize += OSAL.tBufferReadHead(hBuffer, pn16Lat,
                                    AGW_SHAPE_GEO_LAT_BYTELEN
                                        );

        tReadSize += OSAL.tBufferReadHead(hBuffer, pn16Lon,
                                    AGW_SHAPE_GEO_LON_BYTELEN
                                        );

        // Check all read data
        if (AGW_SHAPE_GEO_POINT_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Geo-point (E: %u, R: %u)",
                    AGW_SHAPE_GEO_POINT_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessShapeFront
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeFront(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_FRONT_OBJECT hFront
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    AGW_INTERNAL_FRONT_TYPE_ENUM eFrontType;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    UN8 un8Value = 0;
    BOOLEAN bOk = FALSE;
    size_t tReadSize = 0;
    size_t tBufferSize = 0;
    UN16 un16Point = 0;
    UN16 un16Points = 0;
    N16 n16Lat, n16Lon;

    do
    {
        // Check input
        if ((psObj == NULL) || (hFront == AGW_FRONT_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, psObj->hBuffer);

        // Read type
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &un8Value, AGW_SHAPE_FRONT_TYPE_BYTELEN
                                );
        eFrontType = (AGW_INTERNAL_FRONT_TYPE_ENUM)un8Value;

        // Read number of GEO points
        tReadSize += OSAL.tBufferReadHead(hBuffer, &un16Points,
                            AGW_SHAPE_FRONT_NO_OF_POINT_BYTELEN
                                );

        // Check all read data
        if (AGW_SHAPE_FRONT_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Front Header (E: %u, R: %u)",
                    AGW_SHAPE_FRONT_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        bOk = AGW_FRONT_bUpdate(hFront, eFrontType);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update front object"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // Read all geo-points
        tBufferSize = OSAL.tBufferGetSize(hBuffer);
        if (tBufferSize < (AGW_SHAPE_GEO_POINT_BYTESIZE * un16Points))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": there is not enough data to read %u geo-points",
                    un16Points
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }
        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hFront, un16Points);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update shape object"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        for (un16Point = 0; un16Point < un16Points; ++un16Point)
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, psObj->hBuffer);

            eReturnCode = eAgwLLParserProcessShapeGeoPoint(psObj, &n16Lat, &n16Lon);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to extract data for GEO-point"
                            );
                break;
            }

            bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hFront, un16Point,
                n16Lat, n16Lon,
                AGW_COORDINATE_POW10
                    );
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to set shape point (LAT: %7d, LON: %7d)"
                         );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessShapeIsobar
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeIsobar(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_ISOBAR_OBJECT hIsobar
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    BOOLEAN bOk = FALSE;
    size_t tReadSize = 0;
    size_t tBufferSize = 0;
    UN16 un16Point = 0;
    UN16 un16Pressure = 0;
    UN16 un16Points = 0;
    N16 n16Lat, n16Lon;

    do
    {
        // Check input
        if ((psObj == NULL) || (hIsobar == AGW_ISOBAR_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

        // Read pressure
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &un16Pressure,
                        AGW_SHAPE_ISOBAR_PRESSURE_BYTELEN
                                );

        // Read number of GEO points
        tReadSize += OSAL.tBufferReadHead(hBuffer, &un16Points,
                          AGW_SHAPE_ISOBAR_NO_OF_POINT_BYTELEN
                                );

        // Check all read data
        if (AGW_SHAPE_ISOBAR_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Front Header (E: %u, R: %u)",
                    AGW_SHAPE_ISOBAR_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        bOk = AGW_ISOBAR_bUpdate(hIsobar, un16Pressure);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update isobar object"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // Read all geo-points
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize < (AGW_SHAPE_GEO_POINT_BYTESIZE * un16Points))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": there is not data to read %u geo-points",
                    un16Points
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hIsobar, un16Points);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update shape object"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        for (un16Point = 0; un16Point < un16Points; ++un16Point)
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

            eReturnCode = eAgwLLParserProcessShapeGeoPoint(psObj, &n16Lat, &n16Lon);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to extract data for GEO-point"
                            );
                break;
            }

            bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hIsobar, un16Point, n16Lat,
                n16Lon, AGW_COORDINATE_POW10
                    );
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to set shape point (LAT: %7d, LON: %7d)"
                         );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }
        }

    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserProcessShapePressureCenter
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapePressureCenter(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRESSURE_CENTER_OBJECT hPressCntr
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    size_t tReadSize = 0;
    N16 n16Lat, n16Lon;
    UN16 un16Pressure = 0;
    UN8 un8PressureType = 0;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) ||
            (hPressCntr == AGW_PRESSURE_CENTER_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

        // Read pressure type
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        &un8PressureType,
                        AGW_SHAPE_PRESSURE_CENTER_TYPE_BYTELEN
                                );

        // Read pressure value
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Pressure,
                AGW_SHAPE_PRESSURE_CENTER_PRESSURE_BYTELEN
                                );

        // Check all read data
        if (AGW_SHAPE_PRESSURE_CENTER_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Pressure Center Header (E: %u, R: %u)",
                    AGW_SHAPE_PRESSURE_CENTER_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        // Read point
        eReturnCode = eAgwLLParserProcessShapeGeoPoint(psObj, &n16Lat, &n16Lon);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to extract data for GEO-point"
                        );
            break;
        }

        bOk = AGW_PRESSURE_CENTER_bUpdate(hPressCntr,
            (AGW_INTERNAL_PRESSURE_TYPE_ENUM) un8PressureType,
            un16Pressure);

        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }
        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hPressCntr, 1);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }
        bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hPressCntr,
                    0, n16Lat, n16Lon, AGW_COORDINATE_POW10
                        );
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserProcessShapeStormPosition
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeStormPosition(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_STORM_POSITION_OBJECT hStormPos
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    N16 n16Lat, n16Lon;
    size_t tReadSize = 0;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;
    AGW_STORM_POSITION_DATA_STRUCT sStormPosData;
    UN16 un16Buffer = 0;
    UN8 un8Buffer = 0;
    UN16 un16Year = 0;
    UN8 un8Month = 0, un8Day = 0, un8Hour = 0, un8Minute = 0;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) ||
            (hStormPos == AGW_STORM_POSITION_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        hBuffer = psObj->hBuffer;

        OSAL.bMemSet(&sStormPosData, 0, sizeof(sStormPosData));

        // Skip storm id, we have description already
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        NULL,
                        AGW_SHAPE_STORM_POS_UNIQIEID_BYTELEN
                                );

        // Read Seq id value
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &sStormPosData.un16SequenceID,
                AGW_SHAPE_STORM_POS_SEQID_BYTELEN
                                );

        // Read position type value
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &sStormPosData.ePositionType,
                AGW_SHAPE_STORM_POS_TYPE_BYTELEN
                                );

        // Read direction value
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_STORM_POS_DIRECTION_BYTELEN
                                );

        if (un16Buffer != AGW_SHAPE_STORM_POS_NO_DIRECTION)
        {
            sStormPosData.sDirection.hValue =
                AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                    AGW_SHAPE_STORM_POS_DIRECTION_POW10,
                    sStormPosData.sDirection.aValueData
                        );
            sStormPosData.bDirectionAvailable = TRUE;
        }
        else
        {
            sStormPosData.bDirectionAvailable = FALSE;
        }

        // Read speed value
        un16Buffer = 0;
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_STORM_POS_SPEED_BYTELEN
                                );

        if (un16Buffer != AGW_SHAPE_STORM_POS_NO_SPEED)
        {
            sStormPosData.sSpeed.hValue =
                AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                    AGW_SHAPE_STORM_POS_SPEED_POW10,
                    sStormPosData.sSpeed.aValueData
                        );
            sStormPosData.bSpeedAvailable = TRUE;
        }
        else
        {
            sStormPosData.bSpeedAvailable = FALSE;
        }

        // Read max winds value
        un16Buffer = 0;
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_STORM_POS_MAX_WINDS_BYTELEN
                                );

        sStormPosData.sMaxWinds.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                AGW_SHAPE_STORM_POS_MAX_WINDS_POW10,
                sStormPosData.sMaxWinds.aValueData
                    );

        // Read gusts value
        un16Buffer = 0;
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_STORM_POS_GUSTS_BYTELEN
                                );

        sStormPosData.sGusts.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                AGW_SHAPE_STORM_POS_GUSTS_POW10,
                sStormPosData.sGusts.aValueData
                    );

        // Read pressure value
        un16Buffer = 0;
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_STORM_POS_PRESSURE_BYTELEN
                                );

        sStormPosData.sPressure.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                AGW_SHAPE_STORM_POS_PRESSURE_POW10,
                sStormPosData.sPressure.aValueData
                    );

        // Read year
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un8Buffer,
                AGW_SHAPE_STORM_POS_YEAR_BYTELEN
                                );
        un16Year = AGW_SHAPE_STORM_POS_BASIC_YEAR + un8Buffer;

        // Read month (1-12) since the CRC is valid
        // there is an assumption the value lays within
        // the expected range.
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un8Month,
                AGW_SHAPE_STORM_POS_MONTH_BYTELEN
                                );
        // Read day
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un8Day,
                AGW_SHAPE_STORM_POS_DAY_BYTELEN
                                );
        // Read hour
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un8Hour,
                AGW_SHAPE_STORM_POS_HOUR_BYTELEN
                                );
        // Read minute
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un8Minute,
                AGW_SHAPE_STORM_POS_MINUTES_BYTELEN
                                );

        sStormPosData.tTimeStamp = tAgwLLGetObservationTimeStamp(
            un8Minute, un8Hour, un8Day, (UN8)un8Month - 1, un16Year
                );

        //Read storm type
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &sStormPosData.eStormType,
                AGW_SHAPE_STORM_POS_STORM_TYPE_BYTELEN
                                );

        // Check all read data
        if (AGW_SHAPE_STORM_POS_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Storm Position Header"
                    " (E: %u, R: %u)",
                    AGW_SHAPE_STORM_POS_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        // Update the storm position
        bOk = AGW_STORM_POSITION_bUpdate(hStormPos, &sStormPosData);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // Read point
        eReturnCode = eAgwLLParserProcessShapeGeoPoint(psObj, &n16Lat, &n16Lon);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to extract data for GEO-point"
                        );
            break;
        }

        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hStormPos, 1);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }
        bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hStormPos, 0,
                    n16Lat, n16Lon, AGW_COORDINATE_POW10
                        );
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserProcessShapeWindRadiiArea
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeWindRadiiArea(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_WIND_RADII_AREA_OBJECT hWindRd
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    N16 n16Lat, n16Lon;
    AGW_WIND_RADII_AREA_DATA_STRUCT sWindRdData;
    size_t tBufferSize = 0, tReadSize = 0;
    UN8 un8Point = 0;
    UN8 un8Points = 0;
    BOOLEAN bOk = FALSE;
    UN16 un16Buffer = 0;
    OSAL_BUFFER_HDL hBuffer = OSAL_INVALID_BUFFER_HDL;

    do
    {
        // Check input
        if ((psObj == NULL) ||
            (hWindRd == AGW_WIND_RADII_AREA_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Get parent
        hBuffer = psObj->hBuffer;

        OSAL.bMemSet(&sWindRdData, 0, sizeof(sWindRdData));

        // Skip storm id, we have description already
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                        NULL,
                        AGW_SHAPE_WIND_RADII_UNIQUEID_BYTELEN
                                );

        // Read Seq id value
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &sWindRdData.un16SequenceID,
                AGW_SHAPE_WIND_RADII_SEQID_BYTELEN
                                );

        // Read wind speed value
        un16Buffer = 0;
        tReadSize += OSAL.tBufferReadHead(hBuffer,
                &un16Buffer,
                AGW_SHAPE_WIND_RADII_WIND_SPEED_BYTELEN
                                );

        sWindRdData.sWindSpeed.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Buffer,
                AGW_SHAPE_WIND_RADII_SPEED_POW10,
                sWindRdData.sWindSpeed.aValueData
                    );

        // Read number of GEO points
        tReadSize += OSAL.tBufferReadHead(hBuffer, &un8Points,
            AGW_SHAPE_WIND_RADII_NO_OF_POINT_BYTELEN
                );

        // Check all read data
        if (AGW_SHAPE_WIND_RADII_HEADER_BYTESIZE != tReadSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": incorrect read data size for Wind Rd Header (E: %u, R: %u)",
                    AGW_SHAPE_WIND_RADII_HEADER_BYTESIZE, tReadSize
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        bOk = AGW_WIND_RADII_AREA_bUpdate(hWindRd, &sWindRdData);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update wind radii area object"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // Read all geo-points
        tBufferSize = OSAL.tBufferGetSize(hBuffer);
        if (tBufferSize < (AGW_SHAPE_GEO_POINT_BYTESIZE * un8Points))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": there is not data to read %u geo-points",
                    un8Points
                        );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        bOk = AGW_SHAPE_bUpdatePoints((AGW_SHAPE_OBJECT) hWindRd, un8Points);
        if (bOk == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        for (un8Point = 0; un8Point < un8Points; ++un8Point)
        {
            eReturnCode = eAgwLLParserProcessShapeGeoPoint(psObj, &n16Lat, &n16Lon);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to extract data for GEO-point"
                            );
                break;
            }

            bOk = AGW_SHAPE_bSetPoint((AGW_SHAPE_OBJECT) hWindRd, un8Point, n16Lat, n16Lon,
                AGW_COORDINATE_POW10
                    );

            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to set shape point (LAT: %7d, LON: %7d)",
                         n16Lat, n16Lon
                             );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }
        }
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserProcessStormDesc
*
*   The function extracts two strings from the stream and creates for them
*   two STRING objects. In case of success the caller is responsible for
*   releasing that objects.
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStormDesc(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    STRING_OBJECT *phStormID,
    STRING_OBJECT *phStormName,
    UN16 *pun16NoOfPositions
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    size_t tBytesRead = 0, tBytesExpectedRead;
    UN8 un8NameLength = 0;
    OSAL_BUFFER_HDL hBuffer;

    // Check input
    if ((psObj == NULL) || (phStormID == NULL) || (phStormName == NULL) ||
        (pun16NoOfPositions == NULL))
    {
        return AGW_RETURN_CODE_BAD_ARGUMENT;
    }

    do
    {
        hBuffer = psObj->hBuffer;

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &un8NameLength,
            AGW_SHAPE_STORM_NAME_LEN_BYTELEN
                );

        eReturnCode = eAgwLLParserEnsureBufferSize(un8NameLength + 1, &psObj->sTempBuffer);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Read the name
        tBytesRead += OSAL.tBufferReadHead(hBuffer, psObj->sTempBuffer.pacData,
                                            un8NameLength
                                                );
        psObj->sTempBuffer.pacData[un8NameLength] = '\0';

        // Populate the storm name
        *phStormName = STRING.hCreate(psObj->sTempBuffer.pacData, 0);
        if (*phStormName == STRING_INVALID_OBJECT)
        {
            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        eReturnCode = eAgwLLParserEnsureBufferSize(AGW_SHAPE_STORM_UNIQUEID_BYTELEN + 1,
                                                    &psObj->sTempBuffer);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }
        OSAL.bMemSet(psObj->sTempBuffer.pacData, 0,
                            AGW_SHAPE_STORM_UNIQUEID_BYTELEN + 1);
        tBytesRead += OSAL.tBufferReadHead(hBuffer, psObj->sTempBuffer.pacData,
            AGW_SHAPE_STORM_UNIQUEID_BYTELEN
                );

        // Populate the storm id
        *phStormID = STRING.hCreate(psObj->sTempBuffer.pacData, 0);
        if (*phStormID == STRING_INVALID_OBJECT)
        {
            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        // Read no of positions
        tBytesRead += OSAL.tBufferReadHead(hBuffer, (void*)pun16NoOfPositions,
            AGW_SHAPE_STORM_NO_OF_POSITIONS_BYTELEN
                );

        // Check read data
        tBytesExpectedRead = AGW_SHAPE_STORM_NAME_LEN_BYTELEN + un8NameLength +
                             AGW_SHAPE_STORM_UNIQUEID_BYTELEN +
                             AGW_SHAPE_STORM_NO_OF_POSITIONS_BYTELEN;

        if (tBytesRead != tBytesExpectedRead)
        {
            printf(AGW_MGR_OBJECT_NAME": [%d] read %u from %u byte(s)\n",
                __LINE__, tBytesRead, tBytesExpectedRead);
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        // Make sure that all output data released/cleaned up

        if (*phStormID != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(*phStormID);
            *phStormID = STRING_INVALID_OBJECT;
        }

        if (*phStormName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(*phStormName);
            *phStormName = STRING_INVALID_OBJECT;
        }

        if (pun16NoOfPositions != NULL)
        {
            *pun16NoOfPositions = 0;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vAgwLLParserUninitParser
*
******************************************************************************/
static void vAgwLLParserUninitParser (
    AGW_PARSER_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        AGW_RETURN_CODE_ENUM eReturnCode;

        if (GsWsiDecoderIntf.vUninit != NULL)
        {
            GsWsiDecoderIntf.vUninit(psObj->hWsiDecoder);
            psObj->hWsiDecoder = WSI_DECODER_INVALID_OBJECT;
        }

        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            eOsalReturnCode = OSAL.eReleaseCRC(psObj->hCRC);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to release CRC (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
        }

        // Clean up buffers
        eReturnCode = eAgwLLParserCleanupBuffer(psObj);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to cleanup parser buffer (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
        }

        vAgwLLParserBufferDestroy(&psObj->sInputFile);
        vAgwLLParserBufferDestroy(&psObj->sOutputFile);
        vAgwLLParserBufferDestroy(&psObj->sTempBuffer);

        if (psObj->pcTempPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pcTempPath);
            psObj->pcTempPath = NULL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   eAgwLLParserProcessProductHeader
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessProductHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    OSAL_BUFFER_HDL hBuffer = psObj->hBuffer;
    AGW_RETURN_CODE_ENUM eReturnedCode = AGW_RETURN_CODE_ERROR;
    size_t tBytesRead = 0;

    do
    {
        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

        if (OSAL.tBufferGetSize(hBuffer) < AGW_GHEADER_SIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": not enough data for processing product header"
                        );
            eReturnedCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        OSAL.bMemSet(psHeader, 0, sizeof(*psHeader));

        // Read product info
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->eDataType,
                AGW_GHEADER_TYPE_BYTELEN
                    );

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->eProductType,
                AGW_GHEADER_ID_BYTELEN
                    );

        // Read issue date
        psHeader->un16IssueYear = 0;
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8IssueMonth,
                AGW_GHEADER_ISSUE_MONTH_BYTELEN
                    );

        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8IssueDay,
                AGW_GHEADER_ISSUE_DAY_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8IssueHour,
                AGW_GHEADER_ISSUE_HOUR_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8IssueMinute,
                AGW_GHEADER_ISSUE_MINUTE_BYTELEN
                    );

        // Read valid date
        psHeader->un16ValidYear = 0;
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8ValidMonth,
                AGW_GHEADER_VALID_MONTH_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8ValidDay,
                AGW_GHEADER_VALID_DAY_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8ValidHour,
                AGW_GHEADER_VALID_HOUR_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->un8ValidMinute,
                AGW_GHEADER_VALID_MINUTE_BYTELEN
                    );

        // Read geographical product properties
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->n16UpperLat,
                AGW_GHEADER_MAX_LAT_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->n16UpperLon,
                AGW_GHEADER_MIN_LON_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->n16LowerLat,
                AGW_GHEADER_MIN_LAT_BYTELEN
                    );
        tBytesRead += OSAL.tBufferReadHead(hBuffer, &psHeader->n16LowerLon,
                AGW_GHEADER_MAX_LON_BYTELEN
                    );

        // Check read bytes
        if (tBytesRead != AGW_GHEADER_SIZE)
        {
            printf(AGW_PARSER_OBJECT_NAME
                ": read %u from %u expected\n", tBytesRead, AGW_GHEADER_SIZE);
            eReturnedCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        eReturnedCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnedCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessRasterProductHeader
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessRasterProductHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hBuffer = psObj->hBuffer;
    size_t tReadBytes = 0;

    do
    {
        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, hBuffer);

        if (OSAL.tBufferGetSize(hBuffer) < AGW_RHEADER_SIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": not enough data for processing product raster header"
                          );
            eReturnCode = AGW_RETURN_CODE_NOT_ENOUGH_DATA;
            break;
        }

        OSAL.bMemSet(psHeader, 0, sizeof(*psHeader));

        // Read the generic header first
        eReturnCode = eAgwLLParserProcessProductHeader(psObj,
                psHeader
                    );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Verify the header type to make sure we processing raster header
        if (psHeader->eDataType != AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_PARSER_OBJECT_NAME
                ": the payload describes not-raster data");
            eReturnCode = AGW_RETURN_CODE_UNEXPECTED_DATA_TYPE;
            break;
        }

        tReadBytes = OSAL.tBufferReadHead(hBuffer, &psHeader->un16Rows,
            AGW_RHEADER_NUMBER_ROWS_BYTELEN
                );
        tReadBytes += OSAL.tBufferReadHead(hBuffer, &psHeader->un16Columns,
            AGW_RHEADER_NUMBER_COLUMNS_BYTELEN
                );

        tReadBytes += OSAL.tBufferReadHead(hBuffer, &psHeader->un8PixelDepth,
            AGW_RHEADER_PIXEL_DEPTH_BYTELEN
                );
        // Validate pixel depth per protocol
        if (psHeader->un8PixelDepth != AGW_RHEADER_PIXEL_DEPTH_VALUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": incorrect raster pixel depth. %u instead of %d\n",
                psHeader->un8PixelDepth, AGW_RHEADER_PIXEL_DEPTH_VALUE);
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        tReadBytes += OSAL.tBufferReadHead(hBuffer, &psHeader->un8Planes,
            AGW_RHEADER_NUMBER_PLANES_BYTELEN
                );
        if (tReadBytes != AGW_RHEADER_EXTRA_SIZE)
        {
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        printf(AGW_PARSER_OBJECT_NAME": tile %4dx%4d, %2d BPP, %2d plane(s)\n",
               psHeader->un16Columns,
               psHeader->un16Rows,
               psHeader->un8PixelDepth,
               psHeader->un8Planes
                   );

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eAgwLLParserProcessTileDataBasedProduct
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessTileDataBasedProduct(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if ((psObj == NULL) || (psHeader == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Process payload
        eReturnCode = eAgwLLParserProcessPayload(psObj, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process payload (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        // process product header
        eReturnCode = eAgwLLParserProcessRasterProductHeader(psObj,
            psHeader
                );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   "failed to process raster product header (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eAgwLLParserProcessPayload
*
*   Processes the payload data as valid WSI Access Unit.
*   The payload data parsed by WSI decoder and then loaded in to the
*   memory for further processing.
*
*   Since the WSI decoder creates files this function take care about them.
*   In other words, after this function call all resources (including files)
*   will be release (removed).
*
* Inputs:
*
*   psObj         - reference to AGW_PARSER_OBJECT_STRUCT for which the function
*                   is called.
*   psHeader      - filled by the product header information
*
* Outputs:
*   Return code
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserProcessPayload(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    BOOLEAN bSuccess = FALSE;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    size_t tBufferPeekedSize = 0;
    size_t tSize = 0;

    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType = AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC;
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType = AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN;

    do
    {
        // Input check
        if (psObj == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Parse the AU header
        eReturnCode =
            eAgwLLParserParseHeader(psObj, psObj->hPayload, 0, psHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to parse AU header (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
            break;
        }

        // Create path to the temp file from WSI Decoder.
        tSize = tAgwLLGetProductFilePathLen(AGW_FILE_TYPE_DEFRAMED,
                    IMAGE_INVALID_FORMAT, psObj->pcTempPath);
        eReturnCode = eAgwLLParserEnsureBufferSize(tSize, &psObj->sOutputFile);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to prepare output buffer (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                       );
            break;
        }

        bSuccess = bAgwLLCreateProductFilePath(
                psHeader,
                psObj->pcTempPath,
                psObj->sOutputFile.pacData,
                psObj->sOutputFile.tSize
                    );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to create temporary file path");
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // Decodes the product AU to file
        bSuccess = GsWsiDecoderIntf.bDecodePayload(
                        psObj->hWsiDecoder,
                        psObj->hPayload,
                        psObj->sOutputFile.pacData
                            );
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to decode payload data using WSI decoder\n");
            eReturnCode = AGW_RETURN_CODE_WSI_DECODER_FAILED;
            break;
        }

        if (psHeader->eDataType != AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER)
        {
            psObj->sOutputFile.pFile = fopen(psObj->sOutputFile.pacData, "rb");
            if (psObj->sOutputFile.pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                       ": failed to open file '%s' for binary reading\n",
                       psObj->sOutputFile.pacData);
                break;
            }
        }
        else
        {
            // For raster product data we need to read only header part
            // since the rest data will be processed by WSI decode
            // via direct access to the file.
            bSuccess = bAgwLLParserLoadHeaderToBuffer(psObj,
                            psObj->sOutputFile.pacData, AGW_RHEADER_SIZE
                                    );

            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                        ": failed to load data from the file '%s' \n",
                        psObj->sOutputFile.pacData);
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }
        }
        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->sOutputFile.pFile, psObj->hBuffer);

        // Read product type and product ID
        tBufferPeekedSize = OSAL.tBufferPeek(psObj->hBuffer,
            &eDataType, AGW_GHEADER_TYPE_BYTELEN,
            AGW_GHEADER_TYPE_BYTEOFFSET
                );
        if (tBufferPeekedSize != AGW_GHEADER_TYPE_BYTELEN)
        {
            break;
        }

        tBufferPeekedSize = OSAL.tBufferPeek(psObj->hBuffer,
            &eProductType, AGW_GHEADER_ID_BYTELEN,
            AGW_GHEADER_ID_BYTEOFFSET
                );
        if (tBufferPeekedSize != AGW_GHEADER_ID_BYTELEN)
        {
            break;
        }

        // Check to make sure that DPU header and AU header
        // describe the same product
        if ((eDataType != psHeader->eDataType) ||
            (eProductType != psHeader->eProductType))
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vAgwLLParserKeepOutputFile
*
******************************************************************************/
static void vAgwLLParserKeepOutputFile(
    AGW_PARSER_OBJECT_STRUCT *psObj
        )
{
    psObj->bKeepOutputFile = TRUE;

    return;
}

/*****************************************************************************
*
*   eAgwLLParserGenerateTile
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserGenerateTile(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_RASTER_PLANE_ENUM ePlane,
    AGW_TILE_OBJECT hAgwTile
        )
{
    BOOLEAN bSuccess = FALSE;
    IMAGE_OBJECT hImage;
    STRING_OBJECT hImagePath;
    AGW_RETURN_CODE_ENUM eReturnedCode = AGW_RETURN_CODE_ERROR;
    size_t tSize = 0;

    do
    {
        // Create path for deframed WSI File as input data
        tSize = tAgwLLGetProductFilePathLen(AGW_FILE_TYPE_DEFRAMED,
                        IMAGE_INVALID_FORMAT, psObj->pcTempPath);
        eReturnedCode = eAgwLLParserEnsureBufferSize(tSize, &psObj->sInputFile);
        if (eReturnedCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to prepare input buffer (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnedCode)
                       );
            break;
        }

        bSuccess = bAgwLLCreateProductFilePath(
                psHeader,
                psObj->pcTempPath,
                psObj->sInputFile.pacData,
                psObj->sInputFile.tSize
                    );
        if (bSuccess != TRUE)
        {
            break;
        }

        // Get path for raster WSI File as output data
        hImage = AGW_TILE.hImage(hAgwTile);
        if (hImage == IMAGE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME
                    ": failed to get IMAGE object from the AGW_TILE");
            break;
        }
        hImagePath = IMAGE.hFileName(hImage);
        if (hImagePath == STRING_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME
                    ": failed to get image file object from the IMAGE");
            break;
        }
        tSize = STRING.tSize(hImagePath) + 1;

        // Prepare the buffer
        eReturnedCode = eAgwLLParserEnsureBufferSize(tSize, &psObj->sOutputFile);
        if (eReturnedCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME": failed to prepare output buffer");
            break;
        }

        // Copy string data
        STRING.tCopyToCStr(hImagePath, psObj->sOutputFile.pacData, tSize);

        bSuccess = GsWsiDecoderIntf.bSetPlane(
            psObj->hWsiDecoder,
            (UN8)ePlane
                );
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME
                    ": failed to set plane %u",
                    ePlane
                        );
            eReturnedCode = AGW_RETURN_CODE_WSI_DECODER_FAILED;
            break;
        }

        // Generates raster file from the deframed file and put it to the
        // specified location
        bSuccess = GsWsiDecoderIntf.bGenerateRaster(
            psObj->hWsiDecoder,
            psObj->sInputFile.pacData,
            psObj->sOutputFile.pacData
                );
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_PARSER_OBJECT_NAME
                ": failed to generate tile named %s from %s",
                psObj->sInputFile.pacData,
                psObj->sOutputFile.pacData
                    );
            DATASERVICE_IMPL_vLog(AGW_PARSER_OBJECT_NAME
                ": failed to generate tile named %s from %s",
                psObj->sInputFile.pacData,
                psObj->sOutputFile.pacData
                    );
            eReturnedCode = AGW_RETURN_CODE_WSI_DECODER_FAILED;
            break;
        }

        bSuccess = AGW_TILE_bUpdate(hAgwTile, psHeader, ePlane);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME
                    ": failed to update tile information"
                        );
            eReturnedCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnedCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnedCode;
}

/*****************************************************************************
*
*   eAgwLLParserRemoveOutputFile
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserRemoveOutputFile(
    AGW_PARSER_OBJECT_STRUCT *psObj
        )
{
    N32 n32Result = 0;
    AGW_RETURN_CODE_ENUM eReturnedCode = AGW_RETURN_CODE_SUCCESS;

    if (psObj != NULL)
    {
        if (psObj->sOutputFile.pacData != NULL)
        {

            if (psObj->sOutputFile.pFile != NULL)
            {
                fclose(psObj->sOutputFile.pFile);
                psObj->sOutputFile.pFile = NULL;
            }

            // Remove file since it is not necessary any longer
            // in both failed and success cases.
           n32Result = remove(psObj->sOutputFile.pacData);
            if (n32Result != 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to remove file %s with code %d\n",
                    psObj->sOutputFile.pacData, n32Result
                        );
                eReturnedCode = AGW_RETURN_CODE_IO_ERROR;
            }
            SMSO_vDestroy((SMS_OBJECT)psObj->sOutputFile.pacData);
            psObj->sOutputFile.pacData = NULL;
            psObj->sOutputFile.tSize = 0;
        }
        psObj->bKeepOutputFile = FALSE;
    }
    else
    {
        eReturnedCode = AGW_RETURN_CODE_BAD_ARGUMENT;
    }

    return eReturnedCode;
}


/*****************************************************************************
*
*   eAgwLLParserRemoveInputFile
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserRemoveInputFile(
    AGW_PARSER_OBJECT_STRUCT *psObj
        )
{
    N32 n32Result = 0;
    AGW_RETURN_CODE_ENUM eReturnedCode = AGW_RETURN_CODE_SUCCESS;

    if (psObj->sInputFile.pacData != NULL)
    {
        // Remove file since it is not necessary any longer
        // in both failed and success cases.
        n32Result = remove(psObj->sInputFile.pacData);
        if (n32Result != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to remove file %s with code %d\n",
                psObj->sInputFile.pacData, n32Result
                    );
            eReturnedCode = AGW_RETURN_CODE_IO_ERROR;
        }
        SMSO_vDestroy((SMS_OBJECT)psObj->sInputFile.pacData);
        psObj->sInputFile.pacData = NULL;
        psObj->sInputFile.tSize = 0;
    }

    return eReturnedCode;
}


/*****************************************************************************
*
*   bAgwLLParserIsProductSupported
*
*   Says that the passed combination of product type (ID) and product data type
*   is supported or not. In case if UNKNOWN data type is specified the function
*   checks supportability of the product in general.
*
******************************************************************************/
static BOOLEAN bAgwLLParserIsProductSupported(
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType
        )
{
    BOOLEAN bIsSupported = FALSE;
    UN32 un32Line;
    UN32 un32LineCount = sizeof(gsAGWSupportProductSet) / sizeof(*gsAGWSupportProductSet);
    const AGW_SUPPORT_PRODUCT_SET_STRUCT *psLine = gsAGWSupportProductSet;

    for (un32Line = 0; un32Line < un32LineCount; ++un32Line, ++psLine)
    {
        if ((psLine->eProductType == eProductType) &&
            ((psLine->eDataType == eDataType) ||
             (eDataType == AGW_INTERNAL_PRODUCT_DATA_TYPE_UNKNOWN)))
        {
            bIsSupported = TRUE;
            break;
        }
    }

    return bIsSupported;
}

/*****************************************************************************
*
*   bAgwLLParserLoadHeaderToBuffer
*
******************************************************************************/
static BOOLEAN bAgwLLParserLoadHeaderToBuffer(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    const char *aFilePath,
    size_t tReadSize
        )
{
    BOOLEAN bSuccess = FALSE;
    FILE *pFile;
    size_t tReadBytes;
    size_t tRemainBytes;
    size_t tBufferSize;
    size_t tWrittenBytes;

    do
    {
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize > 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PARSER_OBJECT_NAME
                   ": provided payload buffer cannot be reused since "
                   "is has not been fully read out\n");
            break;
        }

        pFile = fopen(aFilePath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to open file '%s' for binary reading\n",
                   aFilePath);
            break;
        }

        tRemainBytes = tReadSize;
        // Read only requested no. of bytes
        do
        {
            tReadBytes = fread(psObj->acBuffer, 1, tRemainBytes, pFile);
            if (tReadBytes > 0)
            {
                tWrittenBytes = OSAL.tBufferWriteTail(psObj->hBuffer,
                                         psObj->acBuffer, tReadBytes
                                              );

                if (tWrittenBytes != tReadBytes)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to write full read from file buffer hBuffer "
                            "to OSAL buffer");
                }
            }
            tRemainBytes -= tReadBytes;
        }
        while (tRemainBytes > 0);

        fclose(pFile);

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   eAgwLLParserEnsureBufferSize
*
*   The function ensures that the passed buffer contains enough space
*   to accommodate requested number of bytes.
*
*   In case if currently allocation buffer is not enough it will be
*   released and new block equal to the passed requested size will be
*   allocated.
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserEnsureBufferSize(
    size_t tRequestedSize,
    AGW_PARSER_BUFFER_STRUCT *psBuffer
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    char *pacBuffer = NULL;

    do
    {
        // Check input
        if ((psBuffer == NULL) || (tRequestedSize <= 0))
        {
            break;
        }

        if (tRequestedSize > psBuffer->tSize)
        {
            if (psBuffer->pacData != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psBuffer->pacData);
                psBuffer->pacData = NULL;
                psBuffer->tSize = 0;

                if (psBuffer->pFile != NULL)
                {
                    fclose(psBuffer->pFile);
                    psBuffer->pFile = NULL;
                }
            }

            pacBuffer = (char*)SMSO_hCreate(AGW_MGR_OBJECT_NAME":TmpBuffer",
                                        tRequestedSize, SMS_INVALID_OBJECT,
                                        FALSE
                                            );
            if (pacBuffer == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                       ": failed to allocate buffer size %d",
                       tRequestedSize
                              );
                eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
                break;
            }

            psBuffer->pacData = pacBuffer;
            psBuffer->tSize = tRequestedSize;
            psBuffer->pFile = NULL;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vAgwLLParserBufferDestroy
*
******************************************************************************/
static void vAgwLLParserBufferDestroy(
    AGW_PARSER_BUFFER_STRUCT *psBuffer
        )
{
    if (psBuffer != NULL)
    {
        if (psBuffer->pacData != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psBuffer->pacData);
            psBuffer->pacData = NULL;
            psBuffer->tSize = 0;
        }

        if (psBuffer->pFile != NULL)
        {
            fclose(psBuffer->pFile);
            psBuffer->pFile = NULL;
        }
    }

    return;
}

/*****************************************************************************
*
*   eAgwLLParserCleanupBuffer
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwLLParserCleanupBuffer(
    AGW_PARSER_OBJECT_STRUCT *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    // Check input
    if (psObj == NULL)
    {
        eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
    }
    else if (psObj->hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        BOOLEAN bOk;

        bOk = DATASERVICE_IMPL_bFreeDataPayload(psObj->hPayload);

        if (bOk == TRUE)
        {
            psObj->hPayload = OSAL_INVALID_BUFFER_HDL;
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to release parser buffer"
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
        }
    }
    else
    {
        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bAgwLLHasDSRLs
*
******************************************************************************/
static BOOLEAN bAgwLLHasDSRLs(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32DSRLItems = 0;
    BOOLEAN bHas = FALSE;

    eReturnCode = OSAL.eLinkedListItems(psObj->hDSRLList, &un32DSRLItems);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
               ": failed to DSRL list size (%s)",
               OSAL.pacGetReturnCodeName(eReturnCode)
                    );
    }
    else if (un32DSRLItems > 0)
    {
        bHas = TRUE;
    }

    return bHas;
}

/*****************************************************************************
*
*   pacAgwLLGetFileExtension
*
******************************************************************************/
static const char *pacAgwLLGetFileExtension (
    AGW_FILE_TYPE_ENUM eFileType,
    IMAGE_FORMAT_ENUM eImageFormat
        )
{
    const char *pacFileExtension = "";

    switch (eFileType)
    {
        case AGW_FILE_TYPE_DEFRAMED:
        {
            pacFileExtension = AGW_FILE_EXTENSION_DEFRAMED;
        }
        break;
        case AGW_FILE_TYPE_RASTER:
        {
            switch (eImageFormat)
            {
                case IMAGE_FORMAT_BMP:
                {
                    pacFileExtension = AGW_FILE_EXTENSION_RASTER_BMP;
                }
                break;
                case IMAGE_FORMAT_PNG:
                {
                    pacFileExtension = AGW_FILE_EXTENSION_RASTER_PNG;
                }
                break;
                case IMAGE_FORMAT_RAW:
                {
                    pacFileExtension = AGW_FILE_EXTENSION_RASTER_RAW;
                }
                break;
                case IMAGE_FORMAT_JPEG:
                case IMAGE_INVALID_FORMAT:
                {
                    // Nothing
                }
                break;
            }
        }
        break;
        case AGW_FILE_TYPE_MAX:
        {
            // Nothing
        }
        break;
    }

    return pacFileExtension;
}

/*****************************************************************************
*
*   bAgwLLCreateProductFilePath
*
*   Generates file path using provided AGW product header and requested file
*   type in the system wide temp folder
*
******************************************************************************/
static BOOLEAN bAgwLLCreateProductFilePath (
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    const char *pcTempPath,
    char *pacBuffer,
    size_t tBufferSize
        )
{
    size_t tFilenameLen = 0;
    size_t tFolderLen = 0;
    const char *pacFileExtension;
    BOOLEAN bOk = FALSE;

    do
    {
        if ((psHeader == NULL) ||
            (pcTempPath == NULL) ||
            (pacBuffer == NULL))
        {
            break;
        }

        // Get folder path
        tFolderLen = strlen(pcTempPath) + 1;
        if ((tFolderLen >= tBufferSize) || (tBufferSize == 0))
        {
            break;
        }
        strncpy(pacBuffer, pcTempPath, tBufferSize);

        tFilenameLen =
            tAgwLLGetProductFilePathLen(AGW_FILE_TYPE_DEFRAMED,
                IMAGE_INVALID_FORMAT, pacBuffer);
        if ((tFilenameLen == 0) || (tFilenameLen > tBufferSize))
        {
            break;
        }

        // Append extension
        pacFileExtension =
            pacAgwLLGetFileExtension(AGW_FILE_TYPE_DEFRAMED,
                IMAGE_INVALID_FORMAT);

        snprintf(pacBuffer + (tFolderLen - 1),
                 tBufferSize - (tFolderLen - 1),
                 "/"AGW_FILE_NAME_ID_LABEL"%s",
                 psHeader->eProductType,
                 psHeader->un8IssueMonth,
                 psHeader->un8IssueDay,
                 psHeader->un8IssueHour,
                 psHeader->un8IssueMinute,
                 psHeader->n16UpperLat,
                 psHeader->n16UpperLon,
                 psHeader->n16LowerLat,
                 psHeader->n16LowerLon,
                 pacFileExtension);
        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   pacAgwLLGetFileExtension
*
*   Calculates length of the full path to the file which will be generated
*   using specified folder and file type
*
*   Inputs:
*       eFileType - file type to defines concrete file extension
*       eImageFormat - image format if the raster file is asked
*       pacFilePath - folder with file to be created. it must have trailing
*                     path delimiters symbol.
******************************************************************************/
static size_t tAgwLLGetProductFilePathLen (
    AGW_FILE_TYPE_ENUM eFileType,
    IMAGE_FORMAT_ENUM eImageFormat,
    const char *pacFilePath
        )
{
    size_t tPathLen = 0;
    const char *pacExt = NULL;

    if (eFileType <= AGW_FILE_TYPE_MAX)
    {
        if (pacFilePath != NULL)
        {
            tPathLen = strlen(pacFilePath);
        }
        tPathLen += 1; /* Path delimiter */

        if (eFileType == AGW_FILE_TYPE_RASTER)
        {
            tPathLen += AGW_RASTER_FILE_NAME_ID_LABEL_LEN;
        }
        else
        {
            tPathLen += AGW_FILE_NAME_ID_LABEL_LEN;
        }

        pacExt = pacAgwLLGetFileExtension(eFileType, eImageFormat);
        if (pacExt != NULL)
        {
            tPathLen += strlen(pacExt);
        }
        tPathLen += 1; /* For training null */
    };

    return tPathLen;
}

#if ((DEBUG_OBJECT == 1) || (SMS_DEBUG == 1))
/*****************************************************************************
*
*   pacAgwLLGetProductDataTypeName
*
*   Provides string representation for product data type
*
*   Inputs:
*       eDataType - product type to know its name
*
*   Outputs:
*       Product type textual representation
*
******************************************************************************/
static const char *pacAgwLLGetProductDataTypeName(
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType
        )
{
    const char *pacResult = "UNKNOWN";
    UN32 un32CodeLine;
    UN32 un32CodeLineCount = sizeof(gsAGWProductDataTypeNameMap) /
                             sizeof(*gsAGWProductDataTypeNameMap);
    const AGW_ENUM_TYPE_MAP_STRUCT *psLine = gsAGWProductDataTypeNameMap;

    for (un32CodeLine = 0;
         un32CodeLine < un32CodeLineCount;
         ++un32CodeLine, ++psLine)
    {
        if (psLine->uType.eInternalDataType == eDataType)
        {
            pacResult = psLine->pacText;
            break;
        }
    }
    return pacResult;
}
#endif

/*****************************************************************************
*
*   vReleaseDSRLLinkNode
*
*****************************************************************************/
static void vReleaseDSRLLinkNode(
    AGW_DSRL_LINK_STRUCT *psDSRLLink
        )
{
    vReleaseTargetLinks(psDSRLLink->hLinkedTargetsList);
    psDSRLLink->hLinkedTargetsList = OSAL_INVALID_OBJECT_HDL;

    SMSO_vDestroy((SMS_OBJECT)psDSRLLink);

    return;
}

/*****************************************************************************
*
*   bIterateTileDescToRemoveDSRLDesc
*
*****************************************************************************/
static BOOLEAN bIterateTileDescToRemoveDSRLDesc(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLdesc
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": removing the msg %p from DSRL %p\n",
           psEntryDesc->uData.hDSRLEntry,
           psDSRLdesc->hDSRL);

    DSRL_vRemoveEntry(psDSRLdesc->hDSRL, psEntryDesc->uData.hDSRLEntry);

    return TRUE;
}

/*****************************************************************************
*
*   bDSRLCreateList
*
*****************************************************************************/
static BOOLEAN bDSRLCreateList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    AGW_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    do
    {
        // Create our DSRL descriptor based
        // on the provided arguments
        psDSRLDesc = psCreateDSRLDesc(psObj, psDSRLArg);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Set finalization function
        bSuccess = DSRL_bSetFinalizeFunction(psDSRLArg->hDSRL,
                            vAgwDSRLEntryFinalizeFunction, NULL
                                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to set DSRL %p finalize function",
                    psDSRLDesc->hDSRL
                        );
            break;
        }

        // Start populating the list
        bSuccess = bBuildAllTargetDescs(psObj, psDSRLDesc,
                psDSRLArg->ahTargetList, psDSRLArg->tNumTargets);

        if (bSuccess == TRUE)
        {
            bSuccess = bBuildDSRL(psObj, psDSRLDesc);
        }

        if (bSuccess == TRUE)
        {
            // Move to ready state in any case as other services doing.
            bDSRLUpdateState(psDSRLDesc->hDSRL, DSRL_STATE_READY);
        }

    } while(FALSE);

    if (bSuccess != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to Build DSRL %p",
                psDSRLArg->hDSRL
                    );

        bDSRLDestroyList(psObj, psDSRLArg);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bDSRLModifyList
*
*****************************************************************************/
static BOOLEAN bDSRLModifyList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    AGW_DSRL_DESC_STRUCT *psDSRLDescToModify = NULL;
    DSRL_STATE_ENUM eOrigState = DSRL_STATE_READY;

    do
    {
        // Extract the DSRL descriptor
        psDSRLDescToModify = (AGW_DSRL_DESC_STRUCT *)
            DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDescToModify == NULL)
        {
            // Error!
            break;
        }

        // Save off the original state
        eOrigState = DSRL.eState(psDSRLDescToModify->hDSRL);

        // Set the DSRL state to updating state
        bDSRLUpdateState(psDSRLDescToModify->hDSRL, DSRL_STATE_UPDATING);

        // Modify the DSRL as the user specified
        switch (psDSRLArg->uAction.sModify.eModifyType)
        {
            case DSRL_MODIFY_OPERATION_ADD:
            {
                bSuccess = bBuildAllTargetDescs(psObj, psDSRLDescToModify,
                        psDSRLArg->ahTargetList, psDSRLArg->tNumTargets);

                if (bSuccess == TRUE)
                {
                    bSuccess = bBuildDSRL(psObj, psDSRLDescToModify);
                }
            }
            break;

            case DSRL_MODIFY_OPERATION_REPLACE:
            {
                bSuccess = bReplaceTargets(psObj, psDSRLDescToModify,  psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVE:
            {
                bSuccess = bRemoveTargets(psDSRLDescToModify,
                        psDSRLArg->ahTargetList, psDSRLArg->tNumTargets);
            }
            break;

            case DSRL_MODIFY_OPERATION_REMOVEALL:
            {
                bSuccess = bRemoveAllTargets(psDSRLDescToModify,  psDSRLArg);
            }
            break;

            case DSRL_MODIFY_OPERATION_INVALID:
            default:
                break;

        }

        if (bSuccess == TRUE)
        {
            bDSRLUpdateState(psDSRLDescToModify->hDSRL, DSRL_STATE_READY);
        }
        else
        {
            bDSRLUpdateState(psDSRLDescToModify->hDSRL  , eOrigState);
        }

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bDSRLDestroyList
*
*****************************************************************************/
static BOOLEAN bDSRLDestroyList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    AGW_DSRL_DESC_STRUCT *psDSRLDesc;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Extract the DSRL descriptor
    psDSRLDesc = (AGW_DSRL_DESC_STRUCT *)
        DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDesc != NULL)
    {
        // Remove this from our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListRemove(psDSRLDesc->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to remove DSLR %p from the Manager's list (%s)",
                psDSRLDesc->hDSRL, OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        // We've done
        psDSRLDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Destroy this list
        vReleaseDSRLDescFull(psDSRLDesc, TRUE);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": DSRL %p is unknown", psDSRLArg->hDSRL);
    }

    return (eReturnCode == OSAL_SUCCESS) ? TRUE : FALSE;
}

/*****************************************************************************
*
*   bDSRLRefreshList
*
*****************************************************************************/
static BOOLEAN bDSRLRefreshList(
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    AGW_DSRL_DESC_STRUCT *psDSRLDescToRefresh;

    // Get the DSRL Desc
    psDSRLDescToRefresh =
       (AGW_DSRL_DESC_STRUCT *) DSRL_pvServiceData(psDSRLArg->hDSRL);
    if (psDSRLDescToRefresh != NULL)
    {
        bDSRLUpdateState(psDSRLDescToRefresh->hDSRL, DSRL_STATE_UPDATING);

        // Populate list
        bSuccess = bBuildDSRL(psObj, psDSRLDescToRefresh);

        if (bSuccess == TRUE)
        {
            // Move back to the READY state if everithing is good so far
            bDSRLUpdateState(psDSRLDescToRefresh->hDSRL, DSRL_STATE_READY);
        }
    }

    return bSuccess;
}


/*****************************************************************************
*
*   bUpdateTargetToRemoveListIterator
*
*****************************************************************************/
static BOOLEAN bUpdateTargetToRemoveListIterator(
    AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc,
    AGW_DSRL_TARGET_ARRAY_ITERATOR *psIterator
        )
{
    LOCATION_OBJECT hTargetLocation;
    DSRL_TARGET_OBJECT hTargetObj;
    BOOLEAN bIsEqual = FALSE;
    size_t tIndex;

    for (tIndex = 0;
         (tIndex < psIterator->psDSRLArg->tNumTargets) && (bIsEqual == FALSE);
         ++tIndex)
    {
        hTargetObj = psIterator->psDSRLArg->ahTargetList[tIndex];

        hTargetLocation = DSRL_TARGET.hLocation(hTargetObj);
        bIsEqual = LOCATION_bCompare(psTargetDesc->hLocation, hTargetLocation);
    }

    if (bIsEqual == FALSE)
    {
        psIterator->psArray->pahTargets[psIterator->psArray->tUsedSize++] =
                        (DSRL_TARGET_OBJECT)psTargetDesc->hLocation;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bReplaceTargets
*
*****************************************************************************/
static BOOLEAN bReplaceTargets (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    BOOLEAN bSuccess = FALSE;
    DSRL_TARGET_OBJECT hTargetObj;
    OSAL_LINKED_LIST_ENTRY hTempTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AGW_DSRL_TARGET_ARRAY_STRUCT sItemsToRemove;
    AGW_DSRL_TARGET_ARRAY_STRUCT sItemsToAdd;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    size_t tIndex = 0;
    UN32 un32Items = 0;

    do
    {
        // Init data
        sItemsToAdd.pahTargets = NULL;
        sItemsToAdd.tUsedSize = 0;
        sItemsToRemove.pahTargets = NULL;
        sItemsToRemove.tUsedSize = 0;

        // Create the data storage
        sItemsToAdd.pahTargets = (DSRL_TARGET_OBJECT*) SMSO_hCreate(
                    AGW_MGR_OBJECT_NAME":TargToAddArr",
                    sizeof(DSRL_TARGET_OBJECT) * psDSRLArg->tNumTargets,
                    SMS_INVALID_OBJECT,
                    FALSE);
        if(sItemsToAdd.pahTargets == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to allocate memory for list");
            break;
        }

        // iterate through our target list
        for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
        {
            hTargetObj = psDSRLArg->ahTargetList[tIndex];
            hTempTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

            // fill in list of targets to add
            eReturnCode = OSAL.eLinkedListLinearSearch(
                   psDSRLDesc->hTargetList,
                   &hTempTargetEntry,
                   (OSAL_LL_COMPARE_HANDLER)n16LinkSearchTargetByLocation,
                   (void*) &hTargetObj);

            if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
            {
                sItemsToAdd.pahTargets[sItemsToAdd.tUsedSize++] =
                    psDSRLArg->ahTargetList[tIndex];

                // Take care of the target since now
                psDSRLArg->ahTargetList[tIndex] = DSRL_TARGET_INVALID_OBJECT;
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to find target (%s)",
                          OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

        }

        eReturnCode = OSAL.eLinkedListItems(psDSRLDesc->hTargetList, &un32Items);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to get number of target items (%s)",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Check if the list was not empty
        if( un32Items > 0 )
        {
            AGW_DSRL_TARGET_ARRAY_ITERATOR sIterator;

            // Initialize iterator
            sIterator.psArray = &sItemsToRemove;
            sIterator.psDSRLArg = psDSRLArg;

            sItemsToRemove.pahTargets = (DSRL_TARGET_OBJECT*)SMSO_hCreate(
                        AGW_MGR_OBJECT_NAME":TargToRemArr",
                        sizeof(DSRL_TARGET_OBJECT) * un32Items,
                        SMS_INVALID_OBJECT,
                        FALSE);

            if (sItemsToRemove.pahTargets == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                             AGW_MGR_OBJECT_NAME
                                             ": failed to allocate memory for list");
                break;
            }

            // fill in list of targets to remove
            eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
                            (OSAL_LL_ITERATOR_HANDLER)bUpdateTargetToRemoveListIterator,
                            (void*)&sIterator);
            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                             AGW_MGR_OBJECT_NAME
                                             ": failed to iterate (%s)",
                                             OSAL.pacGetReturnCodeName(eReturnCode)
                                                 );
                break;
            }
        }

        // If nothing to remove and noting to add just say that is Ok.
        if ((sItemsToAdd.tUsedSize == 0) && (sItemsToRemove.tUsedSize == 0))
        {
            bSuccess = TRUE;
        }
        else
        {
            if (sItemsToAdd.tUsedSize > 0)
            {
                //add targets and change DSRL
                bSuccess = bBuildAllTargetDescs(psObj, psDSRLDesc,
                                sItemsToAdd.pahTargets, sItemsToAdd.tUsedSize
                                    );

                if (bSuccess == TRUE)
                {
                    bSuccess = bBuildDSRL(psObj, psDSRLDesc);
                }
            }

            if (sItemsToRemove.tUsedSize > 0)
            {
                //remove targets and change DSRL
                bSuccess = bRemoveTargets(psDSRLDesc,
                                sItemsToRemove.pahTargets, sItemsToRemove.tUsedSize
                                    );
            }
        }
    } while (FALSE);

    if (sItemsToRemove.pahTargets != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)sItemsToRemove.pahTargets);
        sItemsToRemove.pahTargets = NULL;
    }

    if (sItemsToAdd.pahTargets != NULL)
    {
        // remove lost targets if any
        for (tIndex = 0; tIndex < sItemsToAdd.tUsedSize; tIndex++)
        {
            DSRL_TARGET_vDestroyByType(
                (DSRL_TARGET_OBJECT)sItemsToAdd.pahTargets[tIndex]);
        }

        SMSO_vDestroy((SMS_OBJECT)sItemsToAdd.pahTargets);
        sItemsToAdd.pahTargets = NULL;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bRemoveTargets
*
*****************************************************************************/
static BOOLEAN bRemoveTargets (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
        )
{
    OSAL_LINKED_LIST_ENTRY hDSRLTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AGW_DSRL_TARGET_DESC_STRUCT *psDSRLTargetDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    DSRL_TARGET_OBJECT hTargetObj;
    AGW_DSRL_ENTRY_TARGET_LINKS_ITERATOR sIterator;
    size_t tIndex = 0;

    // iterate through our target list
    for (tIndex = 0; tIndex < tNumTargets; tIndex++)
    {
        hTargetObj = ahTargetList[tIndex];

        sIterator.hDSRL = psDSRLDesc->hDSRL;
        sIterator.hTargetObj = hTargetObj;

        eReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hDSRLEntryList,
            (OSAL_LL_ITERATOR_HANDLER) bRemoveDSRLEntryTargetLinksIterator,
            (void*) &sIterator);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to iterate DSRL Entries list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
        }

        hDSRLTargetEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLDesc->hTargetList,
            &hDSRLTargetEntry,
            (OSAL_LL_COMPARE_HANDLER)n16LinkSearchTargetByLocation,
            &hTargetObj);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psDSRLTargetDesc = (AGW_DSRL_TARGET_DESC_STRUCT*)OSAL.pvLinkedListThis(hDSRLTargetEntry);
            eReturnCode = OSAL.eLinkedListRemove(hDSRLTargetEntry);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to remove target entry (%s)",
                       OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            vReleaseDSRLTargetDesc(psDSRLTargetDesc);
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRemoveAllTargets
*
*****************************************************************************/
static BOOLEAN bRemoveAllTargets (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = FALSE;

    do
    {
        if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psDSRLDesc->hTargetList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLTargetDesc);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to remove all targets. (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode)
                            );
                bSuccess = FALSE;
                break;
            }
        }

        bSuccess = bAgwDSRLUnlinkAllEntries(psDSRLDesc->hDSRLEntryList);

        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to remove all entries. (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   bRemoveDSRLEntryTargetLinksIterator
*
*****************************************************************************/
static BOOLEAN bRemoveDSRLEntryTargetLinksIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc,
    AGW_DSRL_ENTRY_TARGET_LINKS_ITERATOR *psIterator
        )
{
    BOOLEAN bContinue = FALSE;

    do
    {
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        AGW_DSRL_LINK_STRUCT *psDSRLLinkDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32Items = 0;

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLEntryDesc->hDSRLLinksList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16FindLinkedDSRL,
            psIterator->hDSRL );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Passed us an unlinked DSRL handle
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unlinked DSRL handle (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        psDSRLLinkDesc = 
            (AGW_DSRL_LINK_STRUCT*)OSAL.pvLinkedListThis(hEntry);
        hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDSRLLinkDesc->hLinkedTargetsList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)n16LinkSearchTargetByLocation,
            &(psIterator->hTargetObj));

        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to remove target");
                break;
            }

            // Check the list's size
            eReturnCode = OSAL.eLinkedListItems(
                psDSRLLinkDesc->hLinkedTargetsList, 
                &un32Items);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get number of target items (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            if (un32Items == 0)
            {
                //remove entry from DSRL
                // Note: In case of error we can do nothing about that.
                //       that function fails if only DSRL cannot be
                //       update by new state and this very unlikely.
                (void) eAgwDSRLUnlinkEntry(psDSRLEntryDesc);
            }
        }

        bContinue = TRUE;

    } while (FALSE);

    return bContinue;
}

/*****************************************************************************
*
*   n16FindLinkedDSRL
*
*****************************************************************************/
static N16 n16FindLinkedDSRL (
    AGW_DSRL_LINK_STRUCT *psTarget,
    DSRL_OBJECT hDSRL
        )
{
    if ((psTarget == NULL) ||
        (hDSRL == DSRL_INVALID_OBJECT))
    {
        return N16_MIN;
    }

    if (psTarget->psDSRLDesc->hDSRL == hDSRL)
    {
        // Found it!
        return 0;
    }

    // Keep looking
    return 1;
}

/*****************************************************************************
*
*   bBoundingBoxFromLocation
*
*****************************************************************************/
static BOOLEAN bBoundingBoxFromLocation(
    AGW_BOUNDING_BOX_STRUCT *psBox,
    LOCATION_OBJECT hLocation
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_FIXED_OBJECT hNorth = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hSouth = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hWest = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hEast = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hLon = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA aData[4 * OSAL_FIXED_OBJECT_SIZE];

    do
    {
        // Update center coordinates
        hLat = LOCATION.hLat(hLocation);
        hLon = LOCATION.hLon(hLocation);
        if ((hLat == OSAL_FIXED_INVALID_OBJECT) || (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            psBox->bIsValidBox = FALSE;
            bOk = TRUE;
            break;
        }

        // Create OSAL_FIXED to keep coordinates of the bounding box
        hNorth = OSAL_FIXED.hCreateInMemory(0, 0, &aData[0 * OSAL_FIXED_OBJECT_SIZE]);
        hSouth = OSAL_FIXED.hCreateInMemory(0, 0, &aData[1 * OSAL_FIXED_OBJECT_SIZE]);
        hWest = OSAL_FIXED.hCreateInMemory(0, 0, &aData[2 * OSAL_FIXED_OBJECT_SIZE]);
        hEast = OSAL_FIXED.hCreateInMemory(0, 0, &aData[3 * OSAL_FIXED_OBJECT_SIZE]);

        // Calculate coordinates
        bOk = LOCATION_bBottomLeft(hLocation, hSouth, hWest);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get BottomLeft of location %p\n",
                    hLocation);
            break;
        }

        bOk = LOCATION_bTopRight(hLocation, hNorth, hEast);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get TopRight of location %p\n",
                    hLocation);
            break;
        }

        // Do calculation
        bOk = AGW_MGR_bBoundingBoxCalculate(psBox, hNorth, hSouth, hWest, hEast, FALSE);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to calculate bounding box"
                        );
            break;
        }

        // Copy data from location's lat/lon
        bOk = OSAL_FIXED.bCopyToMemory(hLat, psBox->hCenterLat);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to copy center LAT"
                        );
            break;
        }

        bOk = OSAL_FIXED.bCopyToMemory(hLon, psBox->hCenterLon);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to copy center LON"
                        );
            break;
        }

        // Well, this box is good for now
        psBox->bIsValidBox = TRUE;

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        AGW_MGR_vPrintBoundingBox(psBox);
#endif

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bBoundingBoxesIntersectedByDim
*
*   This function is used for detect intersection between two rectangles by one
*   dimension.
*
*   Inputs:
*      hHalfSize1   - the first half size for dimension (always > 0)
*      hHalfSize2   - the second half size for dimension (always > 0)
*      hCoord1      - the first coordinate
*      hCoord2      - the second coordinate
*
*****************************************************************************/
static BOOLEAN bBoundingBoxesIntersectedByDim(
    OSAL_FIXED_OBJECT hHalfSize1,
    OSAL_FIXED_OBJECT hHalfSize2,
    OSAL_FIXED_OBJECT hCoord1,
    OSAL_FIXED_OBJECT hCoord2
        )
{
    N16 n16CompareResult = 0;
    BOOLEAN bResult = TRUE; // Let set TRUE even in failed case
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_FIXED_OBJECT hHalfSizeSum = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT hCoordDiff = OSAL_FIXED_INVALID_OBJECT;
    OSAL_FIXED_OBJECT_DATA aHalfSizeSumData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT_DATA aCoordDiffData[OSAL_FIXED_OBJECT_SIZE];

    do
    {
        ////////////////////////////////////////////////////////////////////////
        // Init variables
        hHalfSizeSum = OSAL_FIXED.hCreateInMemory(0, 0, aHalfSizeSumData);
        hCoordDiff = OSAL_FIXED.hCreateInMemory(0, 0, aCoordDiffData);

        ////////////////////////////////////////////////////////////////////////
        // Calculate sizes sum
        eOsalReturnCode = OSAL_FIXED.eAdd(hHalfSize1, hHalfSize2, hHalfSizeSum);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate sum (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        ////////////////////////////////////////////////////////////////////////
        // Calculate coordinates diff
        // TODO: This should be reworked once the OSAL_FIXED supports ABS
        n16CompareResult = OSAL_FIXED.n16Compare(hCoord1, hCoord2);
        if (n16CompareResult > 0)
        {
            eOsalReturnCode = OSAL_FIXED.eSubtract(hCoord1, hCoord2, hCoordDiff);
        }
        else
        {
            eOsalReturnCode = OSAL_FIXED.eSubtract(hCoord2, hCoord1, hCoordDiff);
        }
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to calculate coordinates diff (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        ////////////////////////////////////////////////////////////////////////
        // Check values
        n16CompareResult = OSAL_FIXED.n16Compare(hHalfSizeSum, hCoordDiff);
        if (n16CompareResult >= 0)
        {
            bResult = TRUE;
        }
        else
        {
            bResult = FALSE;
        }
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   AGW_MGR_bBoundingBoxesIntersected
*
*   This function is used for detect intersection between two rectangles.
*
*****************************************************************************/
BOOLEAN AGW_MGR_bBoundingBoxesIntersected(
    const AGW_BOUNDING_BOX_STRUCT *psBox1,
    const AGW_BOUNDING_BOX_STRUCT *psBox2
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Checks validity of the boxes before real checks
        if ((psBox1->bIsValidBox == FALSE) || (psBox2->bIsValidBox == FALSE))
        {
            break;
        }

        // Check intersection by LAT
        bOk = bBoundingBoxesIntersectedByDim(
                psBox1->hHalfHeight, psBox2->hHalfHeight,
                psBox1->hCenterLat, psBox2->hCenterLat
                    );
        if (bOk == FALSE)
        {
            break;
        }

        // Check intersection by LON
        bOk = bBoundingBoxesIntersectedByDim(
                psBox1->hHalfWidth, psBox2->hHalfWidth,
                psBox1->hCenterLon, psBox2->hCenterLon
                    );

    } while (FALSE);

    return bOk;
}


/*****************************************************************************
*
*   vReleaseTargetLinks
*
*****************************************************************************/
static void vReleaseTargetLinks(
    OSAL_OBJECT_HDL hList
        )
{
    if (hList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        eOsalReturnCode = OSAL.eLinkedListRemoveAll(hList, NULL);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to clear target links"
                " list. (%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
        }

        eOsalReturnCode = OSAL.eLinkedListDelete(hList);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to delete target links"
                " list (%s)", OSAL.pacGetReturnCodeName(eOsalReturnCode));
        }
    }

    return;
}

/*****************************************************************************
*
*   bCreateDSRLEntryTargetLinksIterator
*
*****************************************************************************/
static BOOLEAN bCreateDSRLEntryTargetLinksIterator(
    AGW_DSRL_TARGET_DESC_STRUCT *psDSRLTargetDesc,
    AGW_DSRL_TARGET_TO_BUILD_DSRL_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bIntersected = FALSE;
    AGW_RETURN_CODE_ENUM eReturnCode;
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntry = psArg->psDSRLEntry;

    switch (psDSRLEntry->eType)
    {
        case AGW_DSRL_ENTRY_TILE:
        {
            AGW_BOUNDING_BOX_STRUCT *psBox = NULL;

            psBox = AGW_TILE_psBoundingBox(psDSRLEntry->uData.hTile);
            if (psBox != NULL)
            {
                bIntersected = AGW_MGR_bBoundingBoxesIntersected(
                    psBox,
                    &psDSRLTargetDesc->sBoundingBox
                        );
            }
        }
        break;

        case AGW_DSRL_ENTRY_SHAPE:
        {
            if (psDSRLEntry->uData.hShape != AGW_SHAPE_INVALID_OBJECT)
            {
                eReturnCode = AGW_SHAPE_eHasIntersectionWith(
                    psDSRLEntry->uData.hShape,
                    psDSRLTargetDesc
                        );
                if (eReturnCode == AGW_RETURN_CODE_SUCCESS)
                {
                    bIntersected = TRUE;
                }
            }
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unsupported type (%d)",
                psDSRLEntry->eType
                    );
        }
        break;
    }

    if (bIntersected == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        printf(AGW_MGR_OBJECT_NAME
            ": [%d] bounding boxes are intersected\n",
            __LINE__);

        eOsalReturnCode =
            OSAL.eLinkedListAdd(psArg->hEntries,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void*)psDSRLTargetDesc
                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": Failed to add target "
                "to target links list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
            );
        }
        else
        {
            psArg->bUpdated = TRUE;
        }
    }

    // Continue
    return TRUE;
}

/*****************************************************************************
*
*   hCreateDSRLEntryTargetLinks
*
*****************************************************************************/
static OSAL_OBJECT_HDL hCreateDSRLEntryTargetLinks(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    AGW_DSRL_TARGET_TO_BUILD_DSRL_ITERATOR_STRUCT sArg;

    do
    {
        // Initialization
        sArg.bUpdated = FALSE;
        sArg.hEntries = OSAL_INVALID_OBJECT_HDL;
        sArg.psDSRLEntry = psDSRLEntryDesc;

        // Create the output list
        eOsalReturnCode = OSAL.eLinkedListCreate(&sArg.hEntries,
            AGW_MGR_OBJECT_NAME":LinkItem:TargetsLinks",
            NULL, OSAL_LL_OPTION_NONE
                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                   AGW_MGR_OBJECT_NAME": cannot create target links list"
                       );
            break;
        }

        // Build the list of the targets applicable for that entry
        eOsalReturnCode = OSAL.eLinkedListIterate(psDSRLDesc->hTargetList,
            (OSAL_LL_ITERATOR_HANDLER) bCreateDSRLEntryTargetLinksIterator,
            (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to iterate targets list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        if (sArg.bUpdated == FALSE)
        {
            puts(AGW_MGR_OBJECT_NAME": no targets for this entry");
            break;
        }

        // We're ok
        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        vReleaseTargetLinks(sArg.hEntries);
        sArg.hEntries = OSAL_INVALID_OBJECT_HDL;
    }

    return sArg.hEntries;
}

/*****************************************************************************
*
*   psCreateDSRLDesc
*
*****************************************************************************/
static AGW_DSRL_DESC_STRUCT *psCreateDSRLDesc (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        )
{
    AGW_DSRL_DESC_STRUCT *psDSRLDesc = NULL;
    DSRL_TARGET_TYPE_ENUM eTargetType;
    BOOLEAN bOk = TRUE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    size_t tIndex;

    do
    {
        // Verify all targets are locations
        for (tIndex = 0; tIndex < psDSRLArg->tNumTargets; tIndex++)
        {
            // Get the current type
            eTargetType = DSRL_TARGET.eType(psDSRLArg->ahTargetList[tIndex]);
            if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
            {
                // Only allow location targets in here
                bOk = FALSE;
                break;
            }
        }

        // Did we experience any errors?
        if (bOk == FALSE)
        {
            break;
        }

        // Get the DSRL descriptor
        psDSRLDesc =
            (AGW_DSRL_DESC_STRUCT *)
                DSRL_pvServiceData(psDSRLArg->hDSRL);
        if (psDSRLDesc == NULL)
        {
            // Error!
            break;
        }

        // Create a list of targets associated with this DSRL
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hTargetList,
            AGW_MGR_OBJECT_NAME":DSRLDesc:TargetList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create a list of DSRL entries associated with this DSRL
        eReturnCode = OSAL.eLinkedListCreate(
            &psDSRLDesc->hDSRLEntryList,
            AGW_MGR_OBJECT_NAME":DSRLDesc:DSRLEntryList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_NONE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Add this to our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListAdd(psObj->hDSRLList,
                            &psDSRLDesc->hEntry,
                            (void*) psDSRLArg->hDSRL
                                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                AGW_MGR_OBJECT_NAME
                ": failed to add new DSRL (%s)\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        psDSRLDesc->hDSRL = psDSRLArg->hDSRL;

        // Provide the caller with the
        // target we've created
        return psDSRLDesc;

    } while (FALSE);

    // Destroy this DSRL Desc now
    vReleaseDSRLDescFull(psDSRLDesc, TRUE);

    return NULL;
}

/*****************************************************************************
*
*   bBuildDSRL
*
*****************************************************************************/
static BOOLEAN bBuildDSRL (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bSuccess = TRUE;
    BOOLEAN bBuild = FALSE;

    // Build DSRL for NOWRAD if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_NOWARD)
    {
        bBuild = bNOWRADBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for NOWRAD");
            bSuccess = FALSE;
        }
    }

    // Build DSRL for WIND_MAGNITUDE if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_WIND_MAGNITUDE)
    {
        bBuild = bWindMagnitudeBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for WINDS MAGNITUDE");
            bSuccess = FALSE;
        }
    }

    // Build DSRL for WIND_DIRECTION if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_WIND_DIRECTION)
    {
        bBuild = bWindDirectionBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for WINDS DIRECTION");
            bSuccess = FALSE;
        }
    }

    // Build DSRL for SURFACE FEATURE if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_SURFACE_FEATURES)
    {
        bBuild = bSurfaceFeatureBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for SURFACE FEATURE");
            bSuccess = FALSE;
        }
    }

    // Build DSRL for STORM ATTRIBUTES if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_STORM_ATTRIBUTES)
    {
        bBuild = bStormAttributesBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for STORM ATTRIBUTES");
            bSuccess = FALSE;
        }
    }

    // Build DSRL for STORM TRACK if required
    if (psObj->psAppObj->un32ProductFilter & AGW_PRODUCT_FILTER_STORM_TRACK)
    {
        bBuild = bStormTrackBuildDSRL(psObj, psDSRLDesc);
        if (bBuild == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to build DSRL for STORM TRACK");
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bBuildAllTargetDescs
*
*****************************************************************************/
static BOOLEAN bBuildAllTargetDescs (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    LOCATION_OBJECT *phLocation;
    AGW_DSRL_TARGET_DESC_STRUCT *psCurTargetDesc;
    BOOLEAN bSuccess = TRUE;
    size_t tIndex;

    // iterate through our target list
    for (tIndex = 0; tIndex < tNumTargets; ++tIndex)
    {
        phLocation = &ahTargetList[tIndex];

        psCurTargetDesc = psBuildTargetDesc(psDSRLDesc, phLocation);
        if (psCurTargetDesc == NULL)
        {
            bSuccess = FALSE;
            break;
        }

        // Add the target description to our DSRL
        // Add this to our list of tracked targets for this service
        eReturnCode = OSAL.eLinkedListAdd(
                            psDSRLDesc->hTargetList,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                            psCurTargetDesc
                                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to add target %p to the DSRL %p (%s)",
                psCurTargetDesc, psDSRLDesc->hDSRL,
                OSAL.pacGetReturnCodeName(eReturnCode));
            bSuccess = FALSE;
            break;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   psBuildTargetDesc
*
*****************************************************************************/
static AGW_DSRL_TARGET_DESC_STRUCT* psBuildTargetDesc (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    LOCATION_OBJECT *phLocation
        )
{
    AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc = NULL;
    BOOLEAN bOk = FALSE;

    do
    {
        // Create the target descriptor
        psTargetDesc = (AGW_DSRL_TARGET_DESC_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                AGW_MGR_OBJECT_NAME":TargetDesc",
                sizeof(AGW_DSRL_TARGET_DESC_STRUCT),
                TRUE);
        if (psTargetDesc == NULL)
        {
            // Error!
            break;
        }

        // Initialize bounding box
        bOk = AGW_MGR_bBoundingBoxInit(&psTargetDesc->sBoundingBox);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to initialize bounding box for LOCATION object %p",
                    *phLocation);
            break;
        }

        // Calculate bounding box from the passed location
        bOk = bBoundingBoxFromLocation(&psTargetDesc->sBoundingBox, *phLocation);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to calculate bounding box for LOCATION object %p",
                    *phLocation);
            break;
        }

        // Store LOCATION target for internal needs
        psTargetDesc->hLocation = *phLocation;

        // Take care about the target since now
        *phLocation = LOCATION_INVALID_OBJECT;

        return psTargetDesc;

    } while (FALSE);

    vReleaseDSRLTargetDesc(psTargetDesc);

    return NULL;
}

/*****************************************************************************
*
*   bDSRLUpdateState
*
*****************************************************************************/
static BOOLEAN bDSRLUpdateState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        )
{
    DSRL_STATE_ENUM eDSRLState;

    eDSRLState = DSRL.eState(hDSRL);
    if (eDSRLState == DSRL_STATE_ERROR)
    {
        printf(AGW_MGR_OBJECT_NAME": DSRL %p in error state. Skip it.\n", hDSRL);
    }
    else
    {
        DSRL_vSetState(hDSRL, eState);
    }

    return TRUE;
}

/*****************************************************************************
*
*   eUpdateDSRLListState
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eUpdateDSRLListState(
    OSAL_OBJECT_HDL hDSRLList,
    DSRL_STATE_ENUM eState
        )
{
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    printf(AGW_MGR_OBJECT_NAME": moving DSRL list %p to %d state\n",
            hDSRLList, eState);

    eReturnCode = OSAL.eLinkedListIterate(hDSRLList,
                            (OSAL_LL_ITERATOR_HANDLER)bDSRLUpdateState,
                            (void*)(size_t)eState);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to iterate for changing DSRLs state to %d (%s)\n",
               eState, OSAL.pacGetReturnCodeName(eReturnCode)
                   );
    }
    else
    {
        eResult = AGW_RETURN_CODE_SUCCESS;
    }

    return eResult;
}

/*****************************************************************************
*
*   vReleaseDSRLDescFull
*
*****************************************************************************/
static void vReleaseDSRLDescFull (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bRemoveEntries
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psDSRLDesc == NULL)
    {
        return;
    }

    printf(AGW_MGR_OBJECT_NAME": attempting to destroy DSRL Desc %p\n",
           psDSRLDesc);

    if (psDSRLDesc->hTargetList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode =
            OSAL.eLinkedListRemoveAll(
                psDSRLDesc->hTargetList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLTargetDesc);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to clean up targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode =
            OSAL.eLinkedListDelete(psDSRLDesc->hTargetList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to delete targets list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        psDSRLDesc->hTargetList = OSAL_INVALID_OBJECT_HDL;
    }

    // Take care about entries only if required
    if (psDSRLDesc->hDSRLEntryList != OSAL_INVALID_OBJECT_HDL)
    {
        if (bRemoveEntries == TRUE)
        {
            eReturnCode =
                       OSAL.eLinkedListIterate(
                            psDSRLDesc->hDSRLEntryList,
                            (OSAL_LL_ITERATOR_HANDLER) bIterateTileDescToRemoveDSRLDesc,
                            (void*) psDSRLDesc);
            if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                       ": failed to iterate for removing DSRL entries for %p (%s)\n",
                       psDSRLDesc, OSAL.pacGetReturnCodeName(eReturnCode)
                           );
            }
        }
        else
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": removing DSRL list for %p w/o removing items since "
                    "they have to be remove and this moment\n", psDSRLDesc->hDSRL);

            eReturnCode = OSAL.eLinkedListRemoveAll(
                                    psDSRLDesc->hDSRLEntryList,
                                    NULL
                                        );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                       ": failed to remove all DSRL entries list in for %p (%s)\n",
                       psDSRLDesc->hDSRL, OSAL.pacGetReturnCodeName(eReturnCode)
                           );
            }
        }

        // Delete list
        eReturnCode =
            OSAL.eLinkedListDelete(psDSRLDesc->hDSRLEntryList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to delete DSRL entries list for %d (%s)\n",
                   psDSRLDesc, OSAL.pacGetReturnCodeName(eReturnCode)
                       );
        }

        psDSRLDesc->hDSRLEntryList = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the DSRL and its entries
    if (psDSRLDesc->hDSRL != DSRL_INVALID_OBJECT)
    {
        DSRL_vDestroy(psDSRLDesc->hDSRL);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseDSRLDescWithoutList
*
*****************************************************************************/
static void vReleaseDSRLDescWithoutList (
    DSRL_OBJECT hDSRL
        )
{
    AGW_DSRL_DESC_STRUCT *psDSRLDesc;

    // Extract the descriptor for DSRL
    psDSRLDesc = (AGW_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
    if (psDSRLDesc != NULL)
    {
        vReleaseDSRLDescFull(psDSRLDesc, FALSE);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseDSRLTargetDesc
*
*****************************************************************************/
static void vReleaseDSRLTargetDesc (
    AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        )
{
    printf(AGW_MGR_OBJECT_NAME
            ": attempting to destroy target desc %p\n",
            psTargetDesc);

    if (psTargetDesc != NULL)
    {
        if (psTargetDesc->hLocation != LOCATION_INVALID_OBJECT)
        {
            LOCATION.vDestroy(psTargetDesc->hLocation);
            psTargetDesc->hLocation = LOCATION_INVALID_OBJECT;
        }

        OSAL.vLinkedListMemoryFree((void*)psTargetDesc);
    }

    return;
}

/*****************************************************************************
*
*   vReleaseAgwDSRLEntryData
*
*****************************************************************************/
static void vReleaseAgwDSRLEntryData(DSRL_ENTRY_OBJECT hDSRLEntry)
{
    DSRL_ENTRY_TYPE_ENUM eType = DSRL_ENTRY_TYPE_UNKNOWN;

    do
    {
        if (hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            break;
        }

        eType = DSRL_ENTRY.eType(hDSRLEntry);

        // Release data depending on its content
        switch (eType)
        {
            case DSRL_ENTRY_TYPE_AGW_TILE:
            {
                AGW_TILE_OBJECT hAgwTile;
                hAgwTile = DSRL_ENTRY.hAgwTile(hDSRLEntry);
                // Destroy tile object
                if (hAgwTile != AGW_TILE_INVALID_OBJECT)
                {
                    AGW_TILE_vDestroy(hAgwTile);
                }
            }
            break;
            case DSRL_ENTRY_TYPE_AGW_SHAPE:
            {
                AGW_SHAPE_OBJECT hAgwShape;
                hAgwShape = DSRL_ENTRY.hAgwShape(hDSRLEntry);

                // Destroy shape object
                if (hAgwShape != AGW_SHAPE_INVALID_OBJECT)
                {
                    AGW_SHAPE_TYPE_ENUM eShapeType;
                    eShapeType = AGW_SHAPE.eShapeType(hAgwShape);
                    switch (eShapeType)
                    {
                        case AGW_SHAPE_TYPE_FRONT:
                        {
                            AGW_FRONT_OBJECT hFront;
                            hFront = AGW_SHAPE.hFront(hAgwShape);
                            AGW_FRONT_vDestroy(hFront);
                        }
                        break;
                        case AGW_SHAPE_TYPE_ISOBAR:
                        {
                            AGW_ISOBAR_OBJECT hIsobar;
                            hIsobar = AGW_SHAPE.hIsobar(hAgwShape);
                            AGW_ISOBAR_vDestroy(hIsobar);
                        }
                        break;
                        case AGW_SHAPE_TYPE_PRESSURE_CENTER:
                        {
                            AGW_PRESSURE_CENTER_OBJECT hPressCntr;
                            hPressCntr = AGW_SHAPE.hPressureCenter(hAgwShape);
                            AGW_PRESSURE_CENTER_vDestroy(hPressCntr);
                        }
                        break;
                        case AGW_SHAPE_TYPE_STORM_ATTRIBUTES:
                        {
                            AGW_STORM_ATTRIBUTES_OBJECT hStormAttr;
                            hStormAttr = AGW_SHAPE.hStormAttributes(hAgwShape);
                            AGW_STORM_ATTRIBUTES_vDestroy(hStormAttr);
                        }
                        break;
                        case AGW_SHAPE_TYPE_STORM_POSITION:
                        {
                            AGW_STORM_POSITION_OBJECT hStormPos;
                            hStormPos = AGW_SHAPE.hStormPosition(hAgwShape);
                            AGW_STORM_POSITION_vDestroy(hStormPos);
                        }
                        break;
                        case AGW_SHAPE_TYPE_WIND_RADII_FIELD:
                        {
                            AGW_WIND_RADII_AREA_OBJECT hWindRd;
                            hWindRd = AGW_SHAPE.hWindRadiiArea(hAgwShape);
                            AGW_WIND_RADII_AREA_vDestroy(hWindRd);
                        }
                        break;
                        case AGW_SHAPE_TYPE_UNKNOWN:
                        default:
                        {
                            // Do nothing
                        }
                        break;
                    }
                }
            }
            break;
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": unsupported DSRL Entry type (%d)",
                        eType
                            );
            }
            break;
        }
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vAgwDSRLRemoveEntryFromList
*
*****************************************************************************/
static void vAgwDSRLRemoveEntryFromList (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AGW_DSRL_LINK_STRUCT *psDSRLLink = NULL;

    do
    {
        vAgwLLDSRLDescRemoveEntryFromList(psEntryDesc, psDSRLDesc);

        eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                            psEntryDesc->hDSRLLinksList,
                            &hEntry,
                            (OSAL_LL_COMPARE_HANDLER)n16FindLinkedDSRL,
                            (void*) psDSRLDesc->hDSRL
                                );

        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to find DSRL %p in links list (%s)",
                    psDSRLDesc->hDSRL, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        psDSRLLink = (AGW_DSRL_LINK_STRUCT*)OSAL.pvLinkedListThis(hEntry);

        eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);

        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to remove hDSRLLinkEntry %p (%s)",
                    hEntry, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        vReleaseDSRLLinkNode(psDSRLLink);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vAgwDSRLEntryFinalizeFunction
*
*****************************************************************************/
static void vAgwDSRLEntryFinalizeFunction (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    void *pvFinalizeArg
        )
{
    do
    {
        AGW_DSRL_DESC_STRUCT *psDSRLDesc;
        AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc;

        // Extract the descriptor for DSRL
        psDSRLDesc = (AGW_DSRL_DESC_STRUCT*)DSRL_pvServiceData(hDSRL);
        if (psDSRLDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to extract service data from DSRL %p",
                hDSRL);
            break;
        }

        // Extract the descriptor for DSRL Entry
        psEntryDesc = 
            (AGW_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
                hDSRLEntry);
        if (psEntryDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to extract service data from DSRL Entry %p",
                hDSRLEntry);
            break;
        }

        vAgwDSRLRemoveEntryFromList(psDSRLDesc, psEntryDesc);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vAgwDSRLEntryDescRelease
*
*   This function releases Tile.
*   Please note, it doesn't take care about removing this tile from all
*   referenced DSRL since it suppose to be called only
*   at Manager Uninit routine
*
*   Inputs:
*      psDSRLEntryDesc    - AGW_TILE Descriptor instance for releasing
*      bRemoveAsListEntry - says should the record be removed from the pool
*                           or not
*      bCheckLinks        - says should the connections to other object be
*                           verified or not
*
*****************************************************************************/
static void vAgwDSRLEntryDescRelease(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc,
    BOOLEAN bRemoveAsListEntry
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Check
    if (psDSRLEntryDesc == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME
            ": Invalid input DSRL Entry descriptor");
        return;
    }

    printf(AGW_MGR_OBJECT_NAME
           ": attempting to remove DSRL Entry desc %p (type %d)\n",
           psDSRLEntryDesc, psDSRLEntryDesc->eType);

    // Remove the tile descriptor from the pool list
    if ((bRemoveAsListEntry == TRUE) &&
        (psDSRLEntryDesc->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
    {
        eReturnCode = OSAL.eLinkedListRemove(psDSRLEntryDesc->hEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to remove DSRL Entry Desc %p from pool (%s)\n",
                    psDSRLEntryDesc, OSAL.pacGetReturnCodeName(eReturnCode)
                        );
        }
        psDSRLEntryDesc->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    }

    // Remove DSRL's list
    if (psDSRLEntryDesc->hDSRLLinksList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove targets from target list in DSRL link structure
        eReturnCode = OSAL.eLinkedListRemoveAll(psDSRLEntryDesc->hDSRLLinksList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLLinkNode
                );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to clean up target list"
                        );
        }

        eReturnCode = OSAL.eLinkedListDelete(psDSRLEntryDesc->hDSRLLinksList);
        if (eReturnCode == OSAL_ERROR_LIST_NOT_EMPTY)
        {
            // This error is used to notify about not empty list of DSRL with
            // destroying item. However, during service stop this message
            // appears since there is no need to remove all messages
            // from all DSRL because of destroying those DSLRs.
            // This message could be ignored.
            printf(AGW_MGR_OBJECT_NAME
                    ": Removing non-empty list (ignored)"
                        );
        }
        psDSRLEntryDesc->hDSRLLinksList = OSAL_INVALID_OBJECT_HDL;
    }

    vReleaseAgwDSRLEntryData(psDSRLEntryDesc->uData.hDSRLEntry);

    return;
}

/*****************************************************************************
*
*   vAgwDSRLEntryDescReleaseNoList
*
*****************************************************************************/
static void vAgwDSRLEntryDescReleaseNoList(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc
        )
{
    vAgwDSRLEntryDescRelease(psDSRLEntryDesc, FALSE);
    return;
}

/*****************************************************************************
*
*   vAgwDSRLEntryDescReleaseFull
*
*****************************************************************************/
static void vAgwDSRLEntryDescReleaseFull(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc
        )
{
    vAgwDSRLEntryDescRelease(psDSRLEntryDesc, TRUE);
    return;
}

/*****************************************************************************
*
*   vAgwLLDSRLDescRemoveEntryFromList
*
*   This function removes DSRL entry (instance of AGW_DSRL_ENTRY_DESC_STRUCT)
*   from list of entries in DSRL desc.
*
*****************************************************************************/
static void vAgwLLDSRLDescRemoveEntryFromList(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hDSRLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    eReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hDSRLEntryList,
        &hDSRLEntry, psEntryDesc);
    if (eReturnCode == OSAL_SUCCESS)
    {
        //remove the entry
        eReturnCode = OSAL.eLinkedListRemove(hDSRLEntry);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to remove DSRL entry from list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (eReturnCode != OSAL_NO_OBJECTS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to find DSRL entry in list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return;
}

/*****************************************************************************
*
*   psAgwDSRLEntryDescCreate
*
*   Creates DSRL entry descriptor with internally referenced DSRL_ENTRY which
*   real data object depends on function argument eEntryType
*
*   Inputs:
*       psAppObj -   facing object uses as parent for all created objects
*
*       eEntryType - described data object type:
*           TILE  - AGW_TILE will be created
*           SHAPE - AGW_SHAPE will be created
*
*       psHeader   - WSI Header with product information
*
*   Output:
*       reference to created object or
*       NULL in failed case
*
*****************************************************************************/
static AGW_DSRL_ENTRY_DESC_STRUCT *psAgwDSRLEntryDescCreate(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_DSRL_ENTRY_TYPE_ENUM eEntryType,
    AGW_SHAPE_TYPE_ENUM eShapeType
        )
{
    AGW_DSRL_ENTRY_DESC_STRUCT *psResult = NULL;
    DSRL_ENTRY_OBJECT hResult = DSRL_ENTRY_INVALID_OBJECT;
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        // Check input
        if ((psAppObj == NULL) || (psHeader == NULL) ||
            ((eEntryType == AGW_DSRL_ENTRY_TILE) && (eShapeType != AGW_SHAPE_TYPE_UNKNOWN)) ||
            ((eEntryType == AGW_DSRL_ENTRY_SHAPE) && (eShapeType == AGW_SHAPE_TYPE_UNKNOWN)))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                AGW_MGR_OBJECT_NAME": invalid input parameters");
            break;
        }

        printf(AGW_MGR_OBJECT_NAME": creating DSRL Entry Desc (type %d)\n",
                eEntryType);

        switch (eEntryType)
        {
            case AGW_DSRL_ENTRY_TILE:
            {
                // Create tile object if a header info provided
                hResult =
                    (DSRL_ENTRY_OBJECT) AGW_TILE_hCreate(
                        (SMS_OBJECT)psAppObj,
                        psAppObj->pacRasterFolder, 
                        psAppObj->eRasterFileFormat, 
                        sizeof(AGW_DSRL_ENTRY_DESC_STRUCT));
            }
            break;
            case AGW_DSRL_ENTRY_SHAPE:
            {
                // Create object based on shape type
                switch (eShapeType)
                {
                    case AGW_SHAPE_TYPE_FRONT:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_FRONT_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                sizeof(AGW_DSRL_ENTRY_DESC_STRUCT), 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_ISOBAR:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_ISOBAR_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                sizeof(AGW_DSRL_ENTRY_DESC_STRUCT), 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_PRESSURE_CENTER:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_PRESSURE_CENTER_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                sizeof(AGW_DSRL_ENTRY_DESC_STRUCT), 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_STORM_ATTRIBUTES:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_STORM_ATTRIBUTES_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                sizeof(AGW_DSRL_ENTRY_DESC_STRUCT), 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_STORM_POSITION:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_STORM_POSITION_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                sizeof(AGW_DSRL_ENTRY_DESC_STRUCT), 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_WIND_RADII_FIELD:
                    {
                        hResult =
                            (DSRL_ENTRY_OBJECT) AGW_WIND_RADII_AREA_hCreate(
                                (SMS_OBJECT)psAppObj, 
                                psHeader);
                    }
                    break;
                    case AGW_SHAPE_TYPE_UNKNOWN:
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": unknown shape type");
                    }
                    break;
                }
            }
            break;
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": unsupported type (%d)",
                    psResult->eType);
            }
            break;
        }

        if (hResult == DSRL_ENTRY_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to create DSRL Entry object");
            break;
        }

        psResult = 
            (AGW_DSRL_ENTRY_DESC_STRUCT *)DSRL_ENTRY_pvServiceData(
                hResult);
        if (psResult == NULL)
        {
            // service data doesn't exist, let's stop here
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": DSRL Entry service data doesn't exist");
            break;
        }

        psResult->eType = eEntryType;
        psResult->eState = AGW_DSRL_ENTRY_DESC_STATE_INVALID;
        psResult->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psResult->hDSRLLinksList = OSAL_INVALID_OBJECT_HDL;
        psResult->uData.hTile = AGW_TILE_INVALID_OBJECT;
        psResult->uData.hShape = AGW_SHAPE_INVALID_OBJECT;

        if (psResult->eType == AGW_DSRL_ENTRY_TILE)
        {
            psResult->uData.hTile = (AGW_TILE_OBJECT)hResult;
        }
        else
        {
            psResult->uData.hShape = (AGW_SHAPE_OBJECT)hResult;
        }

        eReturnCode = OSAL.eLinkedListCreate(
                &psResult->hDSRLLinksList,
                AGW_MGR_OBJECT_NAME":DSRLLinksList",
                (OSAL_LL_COMPARE_HANDLER)n16LinkSearchDSRL,
                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                    );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to create list to keep tile owners (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Register the object as a part of DSRL Entry Descs POOL
        eReturnCode = OSAL.eLinkedListAdd(psAppObj->hDSRLEntryDescPool,
                            &psResult->hEntry, (void*) psResult
                                );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to add the DSRL Tile Desc %p to the pool list %p (%s)",
                psResult, psAppObj->hDSRLEntryDescPool,
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Move the record to INITIAL state
        psResult->eState = AGW_DSRL_ENTRY_DESC_STATE_INITIAL;

        // We are OK. Got ahead
        bOk = TRUE;
    } while (FALSE);

    if ((bOk == FALSE) && (psResult != NULL))
    {
        vAgwDSRLEntryDescReleaseFull(psResult);
        psResult = NULL;
    }

    return psResult;
}

/*****************************************************************************
*
*   bAgwDSRLEntryLinkToDSRLDesc
*
*****************************************************************************/
static BOOLEAN bAgwDSRLEntryLinkToDSRLDesc(
        AGW_DSRL_ENTRY_DESC_STRUCT *psObj,
        AGW_DSRL_DESC_STRUCT *psDSRLDesc,
        OSAL_OBJECT_HDL hTargetLinks
            )
{
    BOOLEAN bResult = FALSE;
    AGW_DSRL_LINK_STRUCT *psLinkItem;
    OSAL_LINKED_LIST_ENTRY hFoundLinkItem = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    printf(AGW_MGR_OBJECT_NAME": %s start with parameters psObj"
        " %p, psDSRLDesc %p, hTargetLinks %p\n", __FUNCTION__,
        psObj, psDSRLDesc, hTargetLinks
            );

    do
    {
        if ((psObj == NULL) || (psDSRLDesc == NULL) ||
            (hTargetLinks == OSAL_INVALID_OBJECT_HDL))
        {
            break;
        }

        //Check if link item already exists
        eReturnCode = OSAL.eLinkedListLinearSearch(psObj->hDSRLLinksList,
                             &hFoundLinkItem,
                             (OSAL_LL_COMPARE_HANDLER)n16FindLinkedDSRL,
                             psDSRLDesc->hDSRL
                                 );
        if (eReturnCode == OSAL_SUCCESS)
        {

              psLinkItem = (AGW_DSRL_LINK_STRUCT*)OSAL.pvLinkedListThis(hFoundLinkItem);

              vReleaseTargetLinks(psLinkItem->hLinkedTargetsList);
             psLinkItem->hLinkedTargetsList = hTargetLinks;

             bResult = TRUE;

              break;
        }

        //Creating new entry
        psLinkItem = (AGW_DSRL_LINK_STRUCT *)SMSO_hCreate(
                AGW_MGR_OBJECT_NAME":LinkItem",
                sizeof(AGW_DSRL_LINK_STRUCT),
                (SMS_OBJECT)psObj,
                FALSE
                    );
        if (psLinkItem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                      AGW_MGR_OBJECT_NAME
                      ": failed to create LinkItem"
                          );
            break;
        }

        psLinkItem->psDSRLDesc = psDSRLDesc;
        eReturnCode = OSAL.eLinkedListAdd(
                psObj->hDSRLLinksList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void*) psLinkItem
                    );

        if (eReturnCode != OSAL_SUCCESS)
        {
            if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                puts(AGW_MGR_OBJECT_NAME": DSRL Desc already linked, skipping");
            }

            vReleaseDSRLLinkNode(psLinkItem);
            break;
        }

        //Assigning targets list
        psLinkItem->hLinkedTargetsList = hTargetLinks;
        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eAgwDSRLEntryDescUpdateAsTile
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwDSRLEntryDescUpdateAsTile (
    AGW_DSRL_ENTRY_DESC_STRUCT *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Check type
        if (psObj->eType != AGW_DSRL_ENTRY_TILE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": tile dsrl entry required"
                        );
            break;
        }

        // Move record to UPDATED state
        psObj->eState = AGW_DSRL_ENTRY_DESC_STATE_UPDATED;

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vAgwDSRLUpdateEntryInDSRL
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwDSRLUpdateEntryInDSRL(
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_DSRL_DESC_ACTION_ENUM eAction
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_OBJECT_HDL hTargetLinks = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    DSRL_ADD_REPLACE_RESULT_ENUM eDSRLResult = DSRL_ADD_REPLACE_ERROR;
    DSRL_STATE_ENUM eDSRLState = DSRL_STATE_UNKNOWN;

    do
    {
        // Do not process DSRL in ERROR state
        eDSRLState = DSRL.eState(psDSRLDesc->hDSRL);
        if (eDSRLState == DSRL_STATE_ERROR)
        {
            printf(AGW_MGR_OBJECT_NAME": DSRL %p in error state. Skip it.\n",
                    psDSRLDesc->hDSRL);
            break;
        }

        if (eAction == AGW_DSRL_ENTRY_DESC_ADD)
        {
            hTargetLinks = hCreateDSRLEntryTargetLinks(psEntryDesc, psDSRLDesc);
            if (hTargetLinks == OSAL_INVALID_OBJECT_HDL)
            {
                printf(AGW_MGR_OBJECT_NAME": DSRL entry 0x%p filtered out.\n",
                        psEntryDesc);
                eReturnCode = AGW_RETURN_CODE_SUCCESS;
                break;
            }
        }

        if (psEntryDesc->uData.hDSRLEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            // Stop executing in this case
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to detect DSRL_ENTRY object in entry %p\n",
                   psEntryDesc);
            break;
        }

        // Move DSRL to UPDATING state
        bDSRLUpdateState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);

        // Add tile to the DSRL
        if (eAction == AGW_DSRL_ENTRY_DESC_ADD)
        {
            OSAL_LINKED_LIST_ENTRY hDSRLEntryEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            BOOLEAN bOk = FALSE;

            eOsalReturnCode = OSAL.eLinkedListSearch(psDSRLDesc->hDSRLEntryList,
                &hDSRLEntryEntry, psEntryDesc);
            if ((eOsalReturnCode == OSAL_OBJECT_NOT_FOUND) ||
                (eOsalReturnCode == OSAL_NO_OBJECTS))
            {
                eDSRLResult = DSRL_eAddEntry(psDSRLDesc->hDSRL, psEntryDesc->uData.hDSRLEntry);
                if (eDSRLResult != DSRL_ADD_REPLACE_OK)
                {
                    printf(AGW_MGR_OBJECT_NAME": failed to add DSRL ENTRY %p to DSRL %p (%d)\n",
                            psEntryDesc->uData.hDSRLEntry, psDSRLDesc->hDSRL, eDSRLResult);
                    eReturnCode = AGW_RETURN_CODE_SUCCESS;
                    break;
                }

                // Add entry to the list in DSRL desc
                eOsalReturnCode = OSAL.eLinkedListAdd(
                                    psDSRLDesc->hDSRLEntryList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                    (void*) psEntryDesc
                                        );
                if (eOsalReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to put ENTRY %p to DSRL entries list %p\n",
                            psEntryDesc, psDSRLDesc->hDSRLEntryList);
                    break;
                }
            }

            bOk = bAgwDSRLEntryLinkToDSRLDesc(psEntryDesc, psDSRLDesc, hTargetLinks);
            if (bOk == TRUE)
            {
                hTargetLinks = OSAL_INVALID_OBJECT_HDL;
            }
        }
        else if (eAction == AGW_DSRL_ENTRY_DESC_UPDATE)
        {
            eDSRLResult = DSRL_eReplaceEntry(psDSRLDesc->hDSRL,
                                    psEntryDesc->uData.hDSRLEntry,
                                    psEntryDesc->uData.hDSRLEntry
                                        );
            if (eDSRLResult != DSRL_ADD_REPLACE_OK)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to update DSRL ENTRY %p in DSRL %p",
                        psEntryDesc, psDSRLDesc->hDSRL);
                eReturnCode = AGW_RETURN_CODE_FAILED_TO_REPLACE;
                break;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to perform unsupported action %d for DSRL_ENTRY %p in DSRL %p",
                   eAction, psEntryDesc, psDSRLDesc->hDSRL);
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    vReleaseTargetLinks(hTargetLinks);

    return eReturnCode;
}

/*****************************************************************************
*
*   bAgwDSRLUpdateEntryInDSRLIterator
*
*****************************************************************************/
static BOOLEAN bAgwDSRLByEntryIterator(
    DSRL_OBJECT hDSRL,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    AGW_DSRL_DESC_STRUCT *psDSRLDesc;
    AGW_DSRL_DESC_ACTION_ENUM eAction = AGW_DSRL_ENTRY_DESC_NOTHING;
    OSAL_RETURN_CODE_ENUM eReturnedCode;
    OSAL_LINKED_LIST_ENTRY hDSRLEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bContinue = TRUE;

    // Create the DSRL descriptor
    psDSRLDesc =
        (AGW_DSRL_DESC_STRUCT *) DSRL_pvServiceData(hDSRL);
    if (psDSRLDesc != NULL)
    {
        // Try to define whenever the tile already belongs to the DSRL
        eReturnedCode = 
            OSAL.eLinkedListLinearSearch(
                psEntryDesc->hDSRLLinksList,
                &hDSRLEntry,
                (OSAL_LL_COMPARE_HANDLER)n16FindLinkedDSRL,
                psDSRLDesc->hDSRL);
        if (eReturnedCode == OSAL_SUCCESS)
        {
            // Item exists
            if (psEntryDesc->eState == AGW_DSRL_ENTRY_DESC_STATE_UPDATED)
            {
                // Do anything only if the item has been updated
                eAction = AGW_DSRL_ENTRY_DESC_UPDATE;
            }
        }
        else if ((eReturnedCode == OSAL_NO_OBJECTS) ||
                        (eReturnedCode == OSAL_OBJECT_NOT_FOUND))
        {
            // Items are absent
            eAction = AGW_DSRL_ENTRY_DESC_ADD;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to search DSRL %p (%s)\n",
                psDSRLDesc->hDSRL, OSAL.pacGetReturnCodeName(eReturnedCode));
            bContinue = FALSE;
        }

        // Update DSRL
        if ((bContinue == TRUE) && (eAction != AGW_DSRL_ENTRY_DESC_NOTHING))
        {
            AGW_RETURN_CODE_ENUM eAgwReturnCode;
            eAgwReturnCode = eAgwDSRLUpdateEntryInDSRL(psDSRLDesc, psEntryDesc, eAction);
            if (eAgwReturnCode == AGW_RETURN_CODE_FAILED_TO_REPLACE)
            {
                // The tile, which exists in the DSRL, has been updated
                // but now it is not applicable so we need to remove
                // it from this DSRL.
                vAgwDSRLRemoveEntryFromList(psDSRLDesc, psEntryDesc);
            }
            else if (eAgwReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to put DSRL Entry %p to DSRL Desc %p (%s)",
                    psEntryDesc, psDSRLDesc,
                    AGW_MGR_pacGetAgwReturnCodeName(eAgwReturnCode));
            }
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   bAgwDSRLUpdateEntryInDSRLIterator
*
*****************************************************************************/
static BOOLEAN bAgwDSRLUpdateEntryInDSRLIterator (
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnedCode = OSAL_ERROR;

    if (psEntryDesc->eState == AGW_DSRL_ENTRY_DESC_STATE_REMOVING)
    {
        printf(AGW_MGR_OBJECT_NAME
               ": skip processing tile in REMOVING state\n");
        return TRUE;
    }

    eReturnedCode = OSAL.eLinkedListIterate(psObj->hDSRLList,
                        (OSAL_LL_ITERATOR_HANDLER)bAgwDSRLByEntryIterator,
                        (void*)psEntryDesc
                            );
    if ((eReturnedCode != OSAL_SUCCESS) && (eReturnedCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to iterate (%s)",
            OSAL.pacGetReturnCodeName(eReturnedCode)
                );
    }

    psEntryDesc->eState = AGW_DSRL_ENTRY_DESC_STATE_STABLE;

    return TRUE;
}

/*****************************************************************************
*
*   bAgwDSRLIsExpiredByTimeStamp
*
*****************************************************************************/
static BOOLEAN bAgwDSRLIsExpiredByTimeStamp(
    TIME_T tTimeStamp,
    TIME_T tTimeout
        )
{
    BOOLEAN bResult = FALSE;
    UN32 un32UTCsec = 0;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    /////////////////////////////////////////////////////
    // If we need to make expiration check - do it through
    // gathering current time and comparing it to object's
    // timestamp
    eOsalReturnCode = OSAL.eTimeGet(&un32UTCsec);
    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to current time (%s)",
            OSAL.pacGetReturnCodeName(eOsalReturnCode)
                );
    }
    else
    {
        // Calculate expiration
        bResult = ((UN32)(tTimeStamp + tTimeout) <= un32UTCsec) ? TRUE : FALSE;

#if SMS_DEBUG==1
        if (bResult != TRUE)
        {
            const TIME_T tLeft = tTimeStamp + tTimeout - un32UTCsec;
            printf(AGW_MGR_OBJECT_NAME": %u sec(s)/%u min(s) left to expiration\n",
                tLeft, tLeft / 60
                    );
        }
#endif
    }

    return bResult;
}

/*****************************************************************************
*
*   bAgwDSRLIsExpired
*
*****************************************************************************/
static BOOLEAN bAgwDSRLIsExpired(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bDoExpirationCheck = FALSE;
    TIME_T tExpirationTimeOut = 0;

    do
    {
        TIME_T tProductTimeStamp = 0;

        /////////////////////////////////////////////////////
        // Check the product type for expiration actions
        if (psEntryDesc->eType == AGW_DSRL_ENTRY_TILE)
        {
            AGW_PRODUCT_TYPE_ENUM eProdType;
            eProdType = AGW_TILE.eProductType(psEntryDesc->uData.hTile);

            if (eProdType == AGW_PRODUCT_TYPE_NOWRAD)
            {
#if (AGW_NOWRAD_TIMEOUT_SUPPORT==1)
                tExpirationTimeOut = AGW_NOWRAD_TILE_TIMEOUT_SEC;
                bDoExpirationCheck = TRUE;
#endif
            }
        }

        if (bDoExpirationCheck != TRUE)
        {
            // So, we don't need to make expiration check and
            // pretend that the object has not been expired
            break;
        }

        // Extract timestamp based on entry type
        if (psEntryDesc->eType == AGW_DSRL_ENTRY_TILE)
        {
            tProductTimeStamp = AGW_TILE.tTimeStamp(psEntryDesc->uData.hTile);
        }
        else if (psEntryDesc->eType == AGW_DSRL_ENTRY_SHAPE)
        {
            tProductTimeStamp = AGW_SHAPE.tTimeStamp(psEntryDesc->uData.hShape);
        }
        else
        {
            // Not supported type
            break;
        }

        // Check the timeout
        bOk = bAgwDSRLIsExpiredByTimeStamp(tProductTimeStamp, tExpirationTimeOut);
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bAgwDSRLAddEntry
*
*****************************************************************************/
static BOOLEAN bAgwDSRLAddEntry(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    if (psEntryDesc->eState != AGW_DSRL_ENTRY_DESC_STATE_REMOVING)
    {
        BOOLEAN bExpiredEntry;

        // Well, try to check expiration then
        bExpiredEntry = bAgwDSRLIsExpired(psEntryDesc);
        if (bExpiredEntry == FALSE)
        {
            AGW_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = eAgwDSRLUpdateEntryInDSRL(psDSRLDesc, psEntryDesc,
                                AGW_DSRL_ENTRY_DESC_ADD
                                    );
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to put ENTRY %p to DSRL %p\n",
                        psEntryDesc, psDSRLDesc,
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bAgwDSRLAddEntries
*
*****************************************************************************/
static BOOLEAN bAgwDSRLAddEntries(
    OSAL_OBJECT_HDL hList,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Iterating through all available DSRL Entries to fill in DSLR by
    // corresponded data.
    eOsalReturnCode = OSAL.eLinkedListIterate(hList,
                        (OSAL_LL_ITERATOR_HANDLER) bAgwDSRLAddEntry,
                        (void*) psDSRLDesc
                                );
    if ((eOsalReturnCode != OSAL_SUCCESS) &&
        (eOsalReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed build iterate list %p to update DSRL %p (%s)",
                hList, psDSRLDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
    }
    else
    {
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*   eAgwDSRLEntryDescListReleaseFull
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwDSRLEntryDescListReleaseFull (
    OSAL_OBJECT_HDL hList
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    printf(AGW_MGR_OBJECT_NAME
           ": [%d] attempting to clean up list %p\n",
           __LINE__, hList);

    if (hList == OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
    }
    else
    {
        eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                            hList,
                            (OSAL_LL_RELEASE_HANDLER)vAgwDSRLEntryDescReleaseFull
                                );
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to iterate through the list %p to remove DSRL Entry Desc",
                    hList);
        }
        else
        {
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bAgwDSRLUnlinkEntryIterator
*
*****************************************************************************/
static BOOLEAN bAgwDSRLUnlinkEntryIterator(
    AGW_DSRL_LINK_STRUCT *psDSRLLink,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    AGW_DSRL_DESC_STRUCT *psDSRLDesc = NULL;

    psDSRLDesc = psDSRLLink->psDSRLDesc;
    bDSRLUpdateState(psDSRLDesc->hDSRL, DSRL_STATE_UPDATING);

    DSRL_vRemoveEntry(psDSRLDesc->hDSRL, psEntryDesc->uData.hDSRLEntry);

    return TRUE;
}

/*****************************************************************************
*
*   bAgwDSRLUnlinkEntry
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eAgwDSRLUnlinkEntry(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    eOsalReturnCode = OSAL.eLinkedListIterate(psEntryDesc->hDSRLLinksList,
                            (OSAL_LL_ITERATOR_HANDLER)bAgwDSRLUnlinkEntryIterator,
                            (void*)psEntryDesc
                                );
    if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to iterate (%s)",
            OSAL.pacGetReturnCodeName(eOsalReturnCode)
                );
    }

    eOsalReturnCode = OSAL.eLinkedListRemoveAll(psEntryDesc->hDSRLLinksList,
                            (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLLinkNode
                                );

    if ((eOsalReturnCode != OSAL_SUCCESS) &&
        (eOsalReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
               ": failed to clean up DSRLs for entry %p (%s)",
               psEntryDesc, OSAL.pacGetReturnCodeName(eOsalReturnCode));
    }

    return AGW_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   bAgwDSRLUnlinkAllEntriesIterator
*
*****************************************************************************/
static BOOLEAN bAgwDSRLUnlinkAllEntriesIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    void *pvArg
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = eAgwDSRLUnlinkEntry(psEntryDesc);
    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to unlink DSRL Entry Desc %p "
                "from its DSRL's (%s)",
                psEntryDesc, AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                    );
    }

    return TRUE;
}

/*****************************************************************************
*
*   bAgwDSRLUnlinkAllEntries
*
*****************************************************************************/
static BOOLEAN bAgwDSRLUnlinkAllEntries(
    OSAL_OBJECT_HDL hList
        )
{
    BOOLEAN bOk = TRUE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    // Looking through all records to unlink them
    eOsalReturnCode = OSAL.eLinkedListIterate(hList,
                            (OSAL_LL_ITERATOR_HANDLER) bAgwDSRLUnlinkAllEntriesIterator,
                            NULL
                                );
    if ((eOsalReturnCode != OSAL_SUCCESS) &&
        (eOsalReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to process DSRL Entry Descs list %p (%s)",
                hList,
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
        bOk = FALSE;
    }

    return bOk;
}

/*****************************************************************************
*
*   vDestroyDSRLEntryDescrPool
*
*****************************************************************************/
static void vDestroyDSRLEntryDescrPool(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;

    printf(AGW_MGR_OBJECT_NAME
           ": attempting to destroy DSRL Entries pool\n");

    // Just to clean up all entries in the pool.
    // But in case if something left in the pool at this point it means
    // that some of product control structures during their un-initialization
    // have not released entries controlled by them
    if (psAppObj->hDSRLEntryDescPool != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListRemoveAll(psAppObj->hDSRLEntryDescPool,
                      (OSAL_LL_RELEASE_HANDLER)vAgwDSRLEntryDescReleaseNoList
                            );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to remove all items from message list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
        }

        eReturnCode = OSAL.eLinkedListDelete(psAppObj->hDSRLEntryDescPool);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to delete message list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
        }

        psAppObj->hDSRLEntryDescPool = OSAL_INVALID_OBJECT_HDL;
    }
    return;
}

/*****************************************************************************
*
*   vRemoveDSRLDescrs
*
*****************************************************************************/
static void vRemoveDSRLDescrs(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    puts(AGW_MGR_OBJECT_NAME
           ": attempting to remove DSRL descriptors");

    if (psObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListRemoveAll(
            psObj->hDSRLList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseDSRLDescWithoutList);

        OSAL.eLinkedListDelete(psObj->hDSRLList);
        psObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }
    return;
}

/*****************************************************************************
*
*   n16LinkSearchDSRL
*
*****************************************************************************/
static N16 n16LinkSearchDSRL(AGW_DSRL_LINK_STRUCT *psObj1, AGW_DSRL_LINK_STRUCT *psObj2)
{
    N16 n16Result = -1;

    if ((psObj1 != NULL) && (psObj2 != NULL))
    {
        if (psObj1->psDSRLDesc->hDSRL == psObj2->psDSRLDesc->hDSRL)
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   n16LinkSearchTargetByLocation
*
*****************************************************************************/
static N16 n16LinkSearchTargetByLocation(
        AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc,
        DSRL_TARGET_OBJECT *phTargetEntry
            )
{
    N16 n16Result = -1;

    if ((psTargetDesc != NULL) && (phTargetEntry != NULL))
    {
        BOOLEAN bIsEqual = FALSE;
        DSRL_TARGET_TYPE_ENUM eTargetType;

        eTargetType = DSRL_TARGET.eType(psTargetDesc->hLocation);

        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            return n16Result;
        }

        eTargetType = DSRL_TARGET.eType(*phTargetEntry);

        if (eTargetType != DSRL_TARGET_TYPE_LOCATION)
        {
            return n16Result;
        }

        bIsEqual = LOCATION_bCompare(psTargetDesc->hLocation,
                                     DSRL_TARGET.hLocation(*phTargetEntry));
        if ( bIsEqual == TRUE)
        {
            n16Result = 0;
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*   eProcessProductFilterUpdate
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessProductFilterUpdate(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_APP_OBJECT_STRUCT *psAppObj,
    AGW_PRODUCT_TYPE_ENUM eProductType,
    BOOLEAN bEnable
        )
{
    UN32 un32Mask;
    BOOLEAN bDoUpdate = TRUE;
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_SUCCESS;

    // Get mask for the product
    un32Mask = un32GetMaskForProduct(eProductType);

    if ((bEnable == TRUE) &&
        ((psAppObj->un32ProductFilter & un32Mask) == un32Mask))
    {
        printf(AGW_MGR_OBJECT_NAME": the filter already enabled for %s\n",
                AGW_MGR_pacGetProductTypeName(eProductType)
                    );
        bDoUpdate = FALSE;
    }
    else if ((bEnable == FALSE) &&
             ((psAppObj->un32ProductFilter & un32Mask) != un32Mask))
    {
        printf(AGW_MGR_OBJECT_NAME": the filter already disabled for %s\n",
                AGW_MGR_pacGetProductTypeName(eProductType)
                    );
        bDoUpdate = FALSE;
    }

    if (bDoUpdate == TRUE)
    {
        if (bEnable == TRUE)
        {
            BOOLEAN bSuccess;

            // Special case for WINDS products. These guys can only
            // be enabled together.
            if ((un32Mask & AGW_PRODUCT_FILTER_WIND_MAGNITUDE) ||
                (un32Mask & AGW_PRODUCT_FILTER_WIND_DIRECTION))
            {
                // If at least one of products requested - request both.
                un32Mask |= (AGW_PRODUCT_FILTER_WIND_MAGNITUDE |
                             AGW_PRODUCT_FILTER_WIND_DIRECTION);
            }

            // Apply the mask
            psAppObj->un32ProductFilter |= un32Mask;

            printf(AGW_MGR_OBJECT_NAME": added filter for %s\n",
                    AGW_MGR_pacGetProductTypeName(eProductType));

            bSuccess = bReadCachedData(psObj, un32Mask);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to read cached data for added product");
                eResult = AGW_RETURN_CODE_ERROR;
            }
        }
        else
        {
            // Remove mask from the filter
            psAppObj->un32ProductFilter &= (~un32Mask);

            // TODO: Do we need to update existing DSRL to remove
            //       all data related to this removing product?
            printf(AGW_MGR_OBJECT_NAME": removed filter for %s\n",
                    AGW_MGR_pacGetProductTypeName(eProductType)
                        );
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessExpirationTimeoutIterator
*
*****************************************************************************/
static BOOLEAN bProcessExpirationTimeoutIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_EXPIRATION_ITERATOR_STRUCT *psArg
        )
{
    BOOLEAN bExpired;
    BOOLEAN bContinue = TRUE;

    // Check expiration
    bExpired = bAgwDSRLIsExpired(psEntryDesc);
    if (bExpired == TRUE)
    {
        UN32 un32Items;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Expired!!!
        printf(AGW_MGR_OBJECT_NAME": expired entry has been found for %p\n",
                psEntryDesc
                    );

        // Count the expired item
        ++psArg->un32Counter;

        eOsalReturnCode = OSAL.eLinkedListItems(psEntryDesc->hDSRLLinksList, &un32Items);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to get number of DSRL corresponded to the entry %p",
                psEntryDesc);
            bContinue = FALSE;
        }
        // Does the entry have any referenced DSRLs?
        else if (un32Items > 0)
        {
            AGW_RETURN_CODE_ENUM eReturnCode;

            // We have DSRLs to process
            eReturnCode = eAgwDSRLUnlinkEntry(psEntryDesc);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to unlink entry %p (%s)",
                    psEntryDesc, AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
                bContinue = FALSE;
            }
            else
            {
                ///////////////////////////////////////////////////////////////
                // Perform additional expiration actions per product
                /////////////////////////////////////////////////////
                // Check the product type for expiration actions
                if (psEntryDesc->eType == AGW_DSRL_ENTRY_TILE)
                {
                    AGW_PRODUCT_TYPE_ENUM eProdType;
                    eProdType = AGW_TILE.eProductType(psEntryDesc->uData.hTile);

#if (AGW_NOWRAD_TIMEOUT_SUPPORT==1)
                    if (eProdType == AGW_PRODUCT_TYPE_NOWRAD)
                    {
                        // Well process expired entry as NOWRad product related
                        (void) vNOWRADProcessExpiredEntry(psArg->psObj, psEntryDesc);
                    }
#endif
                }
            }

            // We can do this here since the OSAL LL iteration function
            // is safe for removing picked entry from that callback.
            vAgwDSRLEntryDescReleaseFull(psEntryDesc);
        }
    }

    return bContinue;
}

/*****************************************************************************
*
*   eProcessExpirationTimeout
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessExpirationTimeout(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    AGW_APP_OBJECT_STRUCT *psAppObj;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_EXPIRATION_ITERATOR_STRUCT sIterator;

    do
    {
        // Check input
        if (psObj == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        psAppObj = psObj->psAppObj;

        // Init iterator
        sIterator.psObj = psObj;
        sIterator.un32Counter = 0;

        // Entries validation routine
        eOsalReturnCode = OSAL.eLinkedListIterate(psAppObj->hDSRLEntryDescPool,
            (OSAL_LL_ITERATOR_HANDLER) bProcessExpirationTimeoutIterator,
            (void*)&sIterator);
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to iterate through pool (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        // Move all updated DSRLs into the READY state
        // if at least one record was marked as Expired
        if (sIterator.un32Counter > 0)
        {
            printf(AGW_MGR_OBJECT_NAME": expired %u entry(-ies)\n", sIterator.un32Counter);

            // Move all touched list into the READY state then
            eReturnCode = eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to move all DSRL into READY state (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            }
        }
        else
        {
            puts(AGW_MGR_OBJECT_NAME": no one entry is expired");
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eExpirationTimerInit
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eExpirationTimerInit(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
    AGW_APP_OBJECT_STRUCT *psAppObj;

    printf(AGW_MGR_OBJECT_NAME": Initializing Expiration Timer\n");

    do
    {
        // Check input
        if (psObj == NULL)
        {
            break;
        }

        // Get quick reference
        psAppObj = psObj->psAppObj;

        if (psAppObj->sTimer.hExpirationEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            // Register for a timed event
            psAppObj->sTimer.hExpirationEvent =
                DATASERVICE_IMPL_hRegisterTimedEvent(
                    (DATASERVICE_IMPL_HDL)psObj, (void *)NULL);

            if(psAppObj->sTimer.hExpirationEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to allocate event");
                eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
                break;
            }
        }
        else
        {
            printf(AGW_MGR_OBJECT_NAME": timer already initialized. Just ignore this request.");
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   vExpirationTimerUnInit
*
*****************************************************************************/
static void vExpirationTimerUnInit(
    AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    puts(AGW_MGR_OBJECT_NAME": Uninitializing Expiration Timer");

    // Check input
    if (psAppObj != NULL)
    {
        vExpirationTimerStop(psAppObj);

        // Clear the timed event handle, we're done with it
        psAppObj->sTimer.hExpirationEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;
    }

    return;
}

/*****************************************************************************
*
*   vExpirationTimerStart
*
*****************************************************************************/
static void vExpirationTimerStart (
    AGW_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        )
{
    BOOLEAN bOk;

    printf(AGW_MGR_OBJECT_NAME
            ": starting Expiration timer for %d sec(s)\n",
            un32PeriodInSec);

    // Kick off the event
    bOk = DATASERVICE_IMPL_bSetTimedEvent(
        psAppObj->sTimer.hExpirationEvent,
        FALSE, FALSE,
        un32PeriodInSec);

    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to start timer");
    }
    else
    {
        puts(AGW_MGR_OBJECT_NAME": Starting Expiration timer");
    }

    return;
}

/*****************************************************************************
*
*   vExpirationTimerStop
*
*****************************************************************************/
static void vExpirationTimerStop (
    AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk;

    puts(AGW_MGR_OBJECT_NAME": Stopping expiration timer");

    // Stop the expiration event from occurring
    bOk = DATASERVICE_IMPL_bStopTimedEvent(psAppObj->sTimer.hExpirationEvent);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to stop timer");
    }

    return;
}

/*****************************************************************************
*
*   vExpirationTimerUpdate
*
*****************************************************************************/
static void vExpirationTimerUpdate (
    AGW_MGR_OBJECT_STRUCT *psObj,
    UN32 un32PeriodInSec
        )
{
    do
    {
        AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
        BOOLEAN bHasEntriesToExpire = FALSE;

        puts(AGW_MGR_OBJECT_NAME": Updating Expiration timer state");

        if ((psAppObj == NULL) ||
            (psAppObj->sTimer.hExpirationEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL))
        {
            break;
        }

#if (AGW_NOWRAD_TIMEOUT_SUPPORT == 1)
        // Check NOWRad entries
        bHasEntriesToExpire = bNOWRADHasExpirableEntries(&psObj->sNowradControlStruct);
#endif

        // Make decision regarding timer start up/stop
        if (bHasEntriesToExpire == TRUE)
        {
            vExpirationTimerStart(psAppObj, un32PeriodInSec);
        }
        else
        {
            vExpirationTimerStop(psAppObj);
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bInitAppFacingObject
*
*****************************************************************************/
static BOOLEAN bInitAppFacingObject(
    AGW_MGR_OBJECT_STRUCT *psObj,
    const char *pacRasterPath,
    IMAGE_FORMAT_ENUM eImageFormat
        )
{
    OSAL_RETURN_CODE_ENUM eResultCode = OSAL_ERROR;
    BOOLEAN bOk = FALSE;
    DATASERVICE_DSRL_CONFIG_STRUCT sDSRLConfig;

    do
    {
        // Check input first
        if ((eImageFormat != IMAGE_FORMAT_BMP) &&
            (eImageFormat != IMAGE_FORMAT_PNG) &&
            (eImageFormat != IMAGE_FORMAT_RAW) &&
            (eImageFormat != IMAGE_INVALID_FORMAT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": invalid input");
            break;
        }

        // Begin DSRL config
        DATASERVICE_IMPL_vInitializeDSRLConfig(&sDSRLConfig);

        // Set service type
        sDSRLConfig.eServiceType = DATASERVICE_TYPE_AGW;

        // Set parent object size
        sDSRLConfig.tParentObjectSize = sizeof(AGW_APP_OBJECT_STRUCT);

        // Set the AGW DSLR descriptor size
        sDSRLConfig.tServiceDataSize = sizeof(AGW_DSRL_DESC_STRUCT);

        // Configure the device feature
        sDSRLConfig.bDeviceEnabled = TRUE;
        sDSRLConfig.un32DeviceNotifyDistance = AGW_DEVICE_DISTANCE_THRESHOLD;
        sDSRLConfig.eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_MILES;

        // Create it
        psObj->psAppObj = (AGW_APP_OBJECT_STRUCT *)
            DATASERVICE_IMPL_hConfigureDSRL(
                (DATASERVICE_IMPL_HDL)psObj, &sDSRLConfig);

        if (psObj->psAppObj == NULL)
        {
            break;
        }

        eResultCode = OSAL.eLinkedListCreate(
            &psObj->psAppObj->hDSRLEntryDescPool,
            AGW_MGR_OBJECT_NAME": MsgList",
            NULL,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
        if (eResultCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to create Messages list (%s)\n",
                    OSAL.pacGetReturnCodeName(eResultCode)
                        );
            break;
        }

        // Set default filter
        psObj->psAppObj->un32ProductFilter = AGW_PRODUCT_FILTER_DEFAULT;

        //Allocating block pool
        eResultCode = OSAL.eBlockPoolCreate(&(psObj->psAppObj->hBlockPool),
                AGW_MGR_OBJECT_NAME":BlockPool",
                AGW_BLOCK_POOL_BUF_SIZE,
                AGW_BLOCK_POOL_BUF_NUM,
                OSAL_BLOCK_POOL_OPTION_NONE
                    );
        if (eResultCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create block pool %s",
                    OSAL.pacGetReturnCodeName(eResultCode)
                        );
            break;
        }

        // Correct the output image file format
        if (eImageFormat == IMAGE_INVALID_FORMAT)
        {
            // Since INVALID means the caller hasn't specified the format
            // the service is free to use the default one.
            psObj->psAppObj->eRasterFileFormat =
                AGW_TILE_OUTPUT_IMAGE_FORMAT_DEFAULT;
        }
        else
        {
            psObj->psAppObj->eRasterFileFormat = eImageFormat;
        }

        if (pacRasterPath != NULL)
        {
            size_t tLength;

            tLength = strlen(pacRasterPath);
            psObj->psAppObj->pacRasterFolder =
                    (char *) SMSO_hCreate(
                        AGW_MGR_OBJECT_NAME":RastersFolder",
                        tLength + 1,
                        (SMS_OBJECT)psObj,
                        FALSE
                            );
            if (psObj->psAppObj->pacRasterFolder != NULL)
            {
                strncpy(psObj->psAppObj->pacRasterFolder,
                    pacRasterPath, tLength + 1);
            }
        }
        else
        {
            psObj->psAppObj->pacRasterFolder = NULL;
        }

        bOk = TRUE;
    } while (FALSE);

    if (NULL != psObj->psAppObj)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT) psObj->psAppObj);
    }

    return bOk;
}

/*****************************************************************************
*
*   vUninitAppFacingObject
*
*****************************************************************************/
static void vUninitAppFacingObject(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bLocked;
    AGW_APP_OBJECT_STRUCT *psAppObj;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    // Lock the app object
    bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": Unable to lock app object");
        return;
    }

    printf(AGW_MGR_OBJECT_NAME
           ": attempting to uninit Facing Objects\n");

    psAppObj = psObj->psAppObj;

    if (psAppObj->pacFullyQualifiedDatabaseFilePath != NULL)
    {
        // Remove database path resources
        SMSO_vDestroy( (SMS_OBJECT)
            &psAppObj->pacFullyQualifiedDatabaseFilePath[0]);
        psAppObj->pacFullyQualifiedDatabaseFilePath = NULL;
    }

    if (psAppObj->pacRasterFolder != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psAppObj->pacRasterFolder);
        psAppObj->pacRasterFolder = NULL;
    }

    if (psAppObj->pacAGWServiceFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psAppObj->pacAGWServiceFilePath);
        psAppObj->pacAGWServiceFilePath = NULL;
    }

    if (psAppObj->hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        eOsalReturnCode = OSAL.eBlockPoolDelete(psAppObj->hBlockPool);
        if(eOsalReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to delete block pool %s",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
        }
        psAppObj->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    }

    vExpirationTimerUnInit(psAppObj);

    // Destroy the app object
    DATASERVICE_IMPL_vDestroyDSRLParent((DATASERVICE_IMPL_HDL)psObj, (SMS_OBJECT)psAppObj);

    psObj->psAppObj = NULL;
    return;
}

/*****************************************************************************
*
*   psGetAppFacingObject
*
*****************************************************************************/
static AGW_APP_OBJECT_STRUCT *psGetAppFacingObject(
    AGW_SERVICE_OBJECT hAgwService
        )
{
    AGW_MGR_OBJECT_STRUCT *psObj =
        (AGW_MGR_OBJECT_STRUCT *)hAgwService;
    BOOLEAN bValid, bLocked;

    do
    {
        bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)hAgwService);

        if (bValid == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to validate Facing object");
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psAppObj,
            OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to lock Facing object");
            break;
        }

        return psObj->psAppObj;

    } while (FALSE);

    return NULL;
}

/*****************************************************************************
*
*   bProductHandlersInit
*
*****************************************************************************/
static BOOLEAN bProductHandlersInit(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    do
    {
        // Initializing Statistics Control data
        bOk = bProductStatisticsInit(&psObj->sStatisticsControlStruct);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize Product Statistics Control structure");
            break;
        }

        // Initializing NOWRAD Control data
        bOk = bNOWRADControlInit(&psObj->sNowradControlStruct);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize NOWRAD Control structure");
            break;
        }

        // Initializing Storm Attributes Control data
        bOk = bStormAttributesInit(&psObj->sStormAttrsControlStruct);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize STORM ATTRIBUTES Control structure");
            break;
        }

        // Initializing Storm Track Control data
        bOk = bStormTrackInit(&psObj->sStormTrackControlStruct);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize STORM TRACK Control structure");
            break;
        }

        // Initializing Wind Magnitude Control data
        bOk = bWindMagnitudeInit(&psObj->sWindMagnitudeControlStruct);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize WINDS Magnitude Control structure");
            break;
        }

        // Initializing Wind Direction Control data
        bOk = bWindDirectionInit(&psObj->sWindDirectionControlStruct);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize WINDS Direction Control structure");
            break;
        }

        // Initializing Surface Feature Control data
        bOk = bSurfaceFeatureInit(&psObj->sSurfaceFeatureControlStruct);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to initialize Surface Feature Control structure");
            break;
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bProductControlStructuresUninit
*
*****************************************************************************/
static BOOLEAN bProductHandlersUninit(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;
    BOOLEAN bResult = TRUE;

    bOk = bSurfaceFeatureUninit(&psObj->sSurfaceFeatureControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize SURFACE FEATURE control structure");
        bResult = FALSE;
    }

    bOk = bWindMagnitudeUninit(&psObj->sWindMagnitudeControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize WINDS magnitude control structure");
        bResult = FALSE;
    }

    bOk = bWindDirectionUninit(&psObj->sWindDirectionControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize WINDS direction control structure");
        bResult = FALSE;
    }

    bOk = bStormAttributesUninit(&psObj->sStormAttrsControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize STORM ATTRIBUTES control structure");
        bResult = FALSE;
    }

    bOk = bStormTrackUninit(&psObj->sStormTrackControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize STORM TRACK control structure");
        bResult = FALSE;
    }

    bOk = bNOWRADControlUninit(&psObj->sNowradControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize NOWRAD control structure");
        bResult = FALSE;
    }

    bOk = bProductStatisticsUninit(&psObj->sStatisticsControlStruct);
    if (bOk != TRUE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
               ": failed to uninitialize Product Statistics control structure");
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
*
*   n16ProductStatisticsSortedByTimeStamp
*
*****************************************************************************/
static N16 n16ProductStatisticsSortedByTimeStamp(
    void *pvObj1,
    void *pvObj2
        )
{
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry1 =
            (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *) pvObj1;
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry2 =
            (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *) pvObj2;

    if (psEntry1->tTimeStamp > psEntry2->tTimeStamp)
    {
        return 1;
    }
    else if (psEntry1->tTimeStamp < psEntry2->tTimeStamp)
    {
        return -1;
    }
    else
    {
        return 0;
    }
}

/*****************************************************************************
*
*   n16ProductStatisticsSearchByTimeStamp
*
*****************************************************************************/
static N16 n16ProductStatisticsSearchByTimeStamp(
    void *pvObj1,
    void *pvObj2
        )
{
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry =
            (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *) pvObj1;
    TIME_T tTimeStamp = 0;

    if ((psEntry == NULL) || (pvObj2 == NULL))
    {
        return N16_MIN;
    }

    tTimeStamp = *((TIME_T*)pvObj2);

    if (psEntry->tTimeStamp == tTimeStamp)
    {
        // Found it!
        return 0;
    }

    // Keep looking
    return 1;
}

/*****************************************************************************
*
*   n16ProductStatisticsSearchByShortTimeStamp
*
*****************************************************************************/
static N16 n16ProductStatisticsSearchByShortTimeStamp(
    void *pvObj1,
    void *pvObj2
        )
{
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry =
            (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *) pvObj1;
    TIME_T tTimeStamp = 0;

    if ((psEntry == NULL) || (pvObj2 == NULL))
    {
        return N16_MIN;
    }

    tTimeStamp = *((TIME_T*)pvObj2);

    if (psEntry->tShortTimeStamp == tTimeStamp)
    {
        // Found it!
        return 0;
    }

    // Keep looking
    return 1;
}

/*****************************************************************************
*
*   bProductStatisticsInit
*
*****************************************************************************/
static BOOLEAN bProductStatisticsInit(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eResult;
    BOOLEAN bOk = FALSE;

    do
    {
        // Create statistics keeping list
        eResult = OSAL.eLinkedListCreate(
            &psObj->hStatistics,
            AGW_MGR_OBJECT_NAME": StatList",
            (OSAL_LL_COMPARE_HANDLER) n16ProductStatisticsSortedByTimeStamp,
            OSAL_LL_OPTION_LINEAR
                );
        if (eResult != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create list (%s)",
                    OSAL.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        // Set default value
        psObj->un16Year = 0;

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   eProductStatisticsGetYear
*
*   Gets statistics recently received year value
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProductStatisticsGetYear(
    AGW_STATISTICS_CONTROL_STRUCT *psObj,
    UN16 *pun16Year
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    UN32 un32Items = 0;

    do
    {
        // Check input
        if ((psObj == NULL) || (pun16Year == NULL))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Gets number of items in statistics list
        eOsalReturnCode = OSAL.eLinkedListItems(psObj->hStatistics, &un32Items);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get number of statistics items (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        // Check year availability
        if (un32Items == 0)
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": there are no statistic to find out a year\n");
            break;
        }

        // TODO: Seems like we need to work out better way to get year
        *pun16Year = psObj->un16Year;

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bProductStatisticsGetForProduct
*
*****************************************************************************/
static BOOLEAN bProductStatisticsGetForProduct (
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psStatEntry,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    AGW_STATISTICS_INFO_STRUCT *psInfo
        )
{
    BOOLEAN bFound = FALSE;
    UN16 un16InfoIndex;
    AGW_STATISTICS_ITEM_DESC_STRUCT *psItemDesc;

    // Looking through the list of stat items to find requested product
    psItemDesc = psStatEntry->paInfo;
    for (un16InfoIndex = 0;
         (un16InfoIndex < psStatEntry->un16NumberOfEntries) && (bFound == FALSE);
         ++un16InfoIndex, ++psItemDesc)
    {
        if (psItemDesc->eType == eProductType)
        {
            psInfo->un16Year = psStatEntry->un16Year;
            psInfo->eType = psItemDesc->eType;
            psInfo->un8Expected = psItemDesc->sInfo.un8Expected;
            psInfo->un8Sent = psItemDesc->sInfo.un8Sent;

            // Just found
            bFound = TRUE;
        }
    }

    return bFound;
}

/*****************************************************************************
*
*   bProductStatisticsGet
*
*****************************************************************************/
static BOOLEAN bProductStatisticsGet(
    AGW_STATISTICS_CONTROL_STRUCT *psObj,
    TIME_T tTimeStamp,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    BOOLEAN bUseShortTimeStamp,
    AGW_STATISTICS_INFO_STRUCT *psInfo
        )
{
    OSAL_RETURN_CODE_ENUM eResult;
    BOOLEAN bOk = FALSE;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    eResult = OSAL.eLinkedListLinearSearch(psObj->hStatistics,
                        &hEntry,
                        ((bUseShortTimeStamp == TRUE) ?
                                n16ProductStatisticsSearchByShortTimeStamp :
                                n16ProductStatisticsSearchByTimeStamp),
                        &tTimeStamp
                            );

    switch (eResult)
    {
        case OSAL_SUCCESS:
        {
            AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psStatEntry;

            // The record has been found
            psStatEntry =
                (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT*)
                    OSAL.pvLinkedListThis(hEntry);

            // Looking through the list of stat items to find requested product
            bOk = bProductStatisticsGetForProduct(psStatEntry,
                                                  eProductType, psInfo);
        }
        break;
        case OSAL_OBJECT_NOT_FOUND:
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": [%d] statistics for %s at %d is absent for now\n",
                   __LINE__,
                   AGW_MGR_pacGetInternalProductTypeName(eProductType),
                   tTimeStamp
                          );
        }
        break;
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to search in the list for statistics "
                "for %s at %d (%s)\n",
                AGW_MGR_pacGetInternalProductTypeName(eProductType),
                tTimeStamp,
                OSAL.pacGetReturnCodeName(eResult)
                          );
        }
        break;
    }

    return bOk;
}

/*****************************************************************************
*
*   eProductStatisticsProcess
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProductStatisticsProcess(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode;
    AGW_STATISTICS_STRUCT sStatisticProduct;

    OSAL.bMemSet(&sStatisticProduct, 0, sizeof(sStatisticProduct));

    // Processing the Statistics product data using the Low-Level PARSER
    eReturnCode = eAgwLLParserProcessStatisticsProduct(
                    psObj->psParserObj,
                    &sStatisticProduct
                        );
    if (eReturnCode == AGW_RETURN_CODE_SUCCESS)
    {
        eReturnCode = eProductStatisticsUpdate(psObj, &sStatisticProduct);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                   __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to update product statistics (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
        }

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        vPrintStatisticsProduct(&sStatisticProduct);
#endif
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eProductStatisticsUpdate
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProductStatisticsUpdate (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_STRUCT *psStatisticsProduct
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    BOOLEAN bNeedToRemove = FALSE;
    BOOLEAN bNeedToAdd = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psEntry = NULL;
    AGW_STATISTICS_ITEM_STRUCT *psTile = NULL;
    AGW_STATISTICS_ITEM_DESC_STRUCT *psTileDesc = NULL;
    AGW_STATISTICS_CONTROL_STRUCT *psStatCtrl = &psObj->sStatisticsControlStruct;
    UN16 un16Tile = 0,
         un16RemainTiles = 0;
    UN32 un32Items = 0;
    TIME_T tTimeStamp = 0;

    do
    {
        // Update the Statistics product header by the year which is kept by
        // itself.
        psStatisticsProduct->sHeader.un16IssueYear =
            psStatisticsProduct->un16Year;
        psStatisticsProduct->sHeader.un16ValidYear =
            psStatisticsProduct->un16Year;

        // Here we assume ability of the year value and
        // we able to generate full time stamp (including year)
        tTimeStamp =
            AGW_MGR_tCreateTimeStamp(&psStatisticsProduct->sHeader, FALSE);

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        vPrintStatisticsCollection(psStatCtrl);
#endif

        // Try to find the entry
        eOsalReturnCode = OSAL.eLinkedListLinearSearch(
                                psStatCtrl->hStatistics,
                                &hEntry,
                                n16ProductStatisticsSearchByTimeStamp,
                                (void*) &tTimeStamp
                                    );
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            // Use existing instance since it handles information
            // for signaling data.
            psEntry =
                (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT*)
                   OSAL.pvLinkedListThis(hEntry);
            if (psEntry->un16NumberOfEntries <
                    psStatisticsProduct->un16NumberOfUsedPID)
            {
                // So we need to remove the instance since it provides
                // less space for new data than required.
                bNeedToRemove = TRUE;
                puts(AGW_MGR_OBJECT_NAME
                       ": found item has less space for new statistics "
                       "and need to be removed");
            }
        }
        else if (eOsalReturnCode == OSAL_OBJECT_NOT_FOUND)
        {
            // Checkout the list's size
            un32Items = 0;
            eOsalReturnCode =
                OSAL.eLinkedListItems(psStatCtrl->hStatistics, &un32Items);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to get number of statistics items (%s)",
                       OSAL.pacGetReturnCodeName(eOsalReturnCode));
                break;
            }

            if (un32Items >= AGW_STATISTICS_ITEMS_MAX)
            {
                // We suppose the list is sorted in ascending order
                // and the first item holds the oldest value. So, if
                // the restriction size is reached remove it from the list
                // to stop list growth.
                hEntry = OSAL.hLinkedListFirst(psStatCtrl->hStatistics,
                                                (void**)&psEntry);
                bNeedToRemove = TRUE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to search in the statistics list (%s)",
                   OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        if (bNeedToRemove == TRUE)
        {
            // Release the entry memory
            vProductStatisticsEntryRelease(psEntry);
            // Remove the entry from the list
            eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to remove the first list item (%s)",
                       OSAL.pacGetReturnCodeName(eOsalReturnCode));
            }
            hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psEntry = NULL;
        }

        // Create the entry if it is absent
        if (psEntry == NULL)
        {
            psEntry = psProductStatisticsEntryCreate(
                            psObj->psAppObj,
                            psStatisticsProduct->un16NumberOfUsedPID
                                );
            if (psEntry == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to create statistics entry");
                break;
            }

            bNeedToAdd = TRUE;
        }

        // Fill in the statistics data
        psEntry->tTimeStamp = tTimeStamp;
        psEntry->tShortTimeStamp =
                AGW_MGR_tCreateTimeStamp(&psStatisticsProduct->sHeader, TRUE);
        psEntry->un16Year = psStatisticsProduct->un16Year;
        un16RemainTiles = psStatisticsProduct->un16NumberOfUsedPID;

        psTileDesc = psEntry->paInfo;
        psTile = psStatisticsProduct->aInfo;
        for (un16Tile = 0;
             (un16Tile < psStatisticsProduct->un16NumberPID) &&
                 (un16RemainTiles > 0);
             ++un16Tile, ++psTile)
        {
            if ((psTile->un8Expected > 0) || (psTile->un8Sent > 0))
            {
                // Copy tile data
                psTileDesc->eType = (AGW_INTERNAL_PRODUCT_TYPE_ENUM)un16Tile;
                psTileDesc->sInfo = *psTile;

                // Move to the next desc item place
                ++psTileDesc;
                --un16RemainTiles;
            }
        }

        if (bNeedToAdd == TRUE)
        {
            eOsalReturnCode = OSAL.eLinkedListAdd(psStatCtrl->hStatistics,
                                &hEntry, psEntry);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": unable to add new statistics entry to the list (%s)\n",
                       OSAL.pacGetReturnCodeName(eOsalReturnCode)
                          );
                break;
            }
        }

        // Update statistics year
        psStatCtrl->un16Year = psStatisticsProduct->un16Year;

        // We're ok
        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if ((eReturnCode != AGW_RETURN_CODE_SUCCESS) && (bNeedToAdd == TRUE))
    {
        vProductStatisticsEntryRelease(psEntry);
        psEntry = NULL;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   psProductStatisticsEntryCreate
*
*   Creates the ProductStatistics descriptor for specified number of items.
*
*****************************************************************************/
static AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psProductStatisticsEntryCreate(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    UN16 un16NoOfItems
        )
{
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psStatEntry;

    // Create the entry
    psStatEntry = (AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *)
        SMSO_hCreate(
            AGW_MGR_OBJECT_NAME":StatEntry",
            sizeof(AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT) +
            /* since the one item lays inside
             * AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT */
            sizeof(AGW_STATISTICS_ITEM_DESC_STRUCT) * (un16NoOfItems - 1),
            (SMS_OBJECT)psAppObj,
            FALSE);

    if (psStatEntry != NULL)
    {
        psStatEntry->un16NumberOfEntries = un16NoOfItems;
    }

    return psStatEntry;
}

/*****************************************************************************
*
*   vProductStatisticsEntryRelease
*
*****************************************************************************/
static void vProductStatisticsEntryRelease(
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": releasing statistics object %p\n",
           psObj
                  );
    SMSO_vDestroy((SMS_OBJECT)psObj);
    return;
}

/*****************************************************************************
*
*   bProductStatisticsUninit
*
*****************************************************************************/
static BOOLEAN bProductStatisticsUninit(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        )
{
    BOOLEAN bOk = TRUE;
    OSAL_RETURN_CODE_ENUM eResult;

    printf(AGW_MGR_OBJECT_NAME
            ": attempting to uninit the Statistics product control structure\n");

    eResult = OSAL.eLinkedListRemoveAll(
        psObj->hStatistics,
        (OSAL_LL_RELEASE_HANDLER)vProductStatisticsEntryRelease
            );
    if (eResult != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to remove list items (%s)",
                OSAL.pacGetReturnCodeName(eResult));
    }

    eResult = OSAL.eLinkedListDelete(
        psObj->hStatistics
            );
    if (eResult != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": faile to delete list (%s)",
                OSAL.pacGetReturnCodeName(eResult));
    }

    psObj->hStatistics = OSAL_INVALID_OBJECT_HDL;

    return bOk;
}

/*****************************************************************************
*
*   n16DSRLTileDescSearchByLocation
*
*****************************************************************************/
static N16 n16DSRLTileDescSearchByLocation(
    void *pvObj1,
    void *pvObj2
        )
{
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc1 =
            (AGW_DSRL_ENTRY_DESC_STRUCT*)pvObj1;
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc2 =
            (AGW_DSRL_ENTRY_DESC_STRUCT*)pvObj2;
    BOOLEAN bFound;

    bFound = AGW_TILE_bEqualByLocation(psTileDesc1->uData.hTile,
                                       psTileDesc2->uData.hTile);

    return (bFound == TRUE) ? 0 : 1;
}

/*****************************************************************************
*
*   n16DSRLTileDescSortedByLocation
*
*****************************************************************************/
static N16 n16DSRLTileDescSortedByLocation(
    void *pvObj1,
    void *pvObj2
        )
{
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc1 =
            (AGW_DSRL_ENTRY_DESC_STRUCT*)pvObj1;
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc2 =
            (AGW_DSRL_ENTRY_DESC_STRUCT*)pvObj2;
    N16 n16Result;

    n16Result = AGW_TILE_n16CompareByCenter(psTileDesc1->uData.hTile,
                        psTileDesc2->uData.hTile
                            );
    return n16Result;
}

/*****************************************************************************
*
*   n16NOWRADProcessExpiredEntryInTheListSearcher
*
*****************************************************************************/
static N16 n16NOWRADProcessExpiredEntryInTheListSearcher (
    AGW_DSRL_ENTRY_DESC_STRUCT *psObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryToFind
        )
{
    return (psObj == psEntryToFind) ? 0 : -1;
}

/*****************************************************************************
*
*   bNOWRADProcessExpiredEntryInTheList
*
*   Removes entry passed to the function from the passed list.
*
*   Output:
*       TRUE if function was found and remove or
*       FALSE otherwise.
*
*****************************************************************************/
static BOOLEAN bNOWRADProcessExpiredEntryInTheList(
    OSAL_OBJECT_HDL hList,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    BOOLEAN bRemoved = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Check input
    if (hList == OSAL_INVALID_OBJECT_HDL)
    {
        // Nothing to remove from invalid list
        return bRemoved;
    }

    // Find the entry in the list
    eOsalReturnCode = OSAL.eLinkedListLinearSearch(hList, &hEntry,
                            (OSAL_LL_COMPARE_HANDLER)n16NOWRADProcessExpiredEntryInTheListSearcher,
                            (void*)psEntryDesc
                                );
    if (eOsalReturnCode == OSAL_SUCCESS)
    {
        // Remove entry
        eOsalReturnCode = OSAL.eLinkedListRemove(hEntry);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to remove expired entry from the list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
        }
        else
        {
            bRemoved = TRUE;
        }
    }
    else if (eOsalReturnCode != OSAL_OBJECT_NOT_FOUND)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_MGR_OBJECT_NAME": failed to search the entry in the list to remove expired entry (%s)",
            OSAL.pacGetReturnCodeName(eOsalReturnCode));
    }


    return bRemoved;
}

/*****************************************************************************
*
*   vNOWRADProcessExpiredEntry
*
*****************************************************************************/
static void vNOWRADProcessExpiredEntry(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        )
{
    BOOLEAN bRemoved = FALSE;
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl = &psObj->sNowradControlStruct;
#if DEBUG_OBJECT==1
    const char *pacListName = "UNKNOWN";
#endif

    do
    {
        // Try to remove from the progress set
        bRemoved = bNOWRADProcessExpiredEntryInTheList(psNRCtrl->hProgressSet, psEntryDesc);
        if (bRemoved == TRUE)
        {
#if DEBUG_OBJECT==1
            pacListName = "PROGRESS";
#endif
            break;
        }

        // Try to remove from the full set
        bRemoved = bNOWRADProcessExpiredEntryInTheList(psNRCtrl->hFullSet, psEntryDesc);
        if (bRemoved == TRUE)
        {
#if DEBUG_OBJECT==1
            pacListName = "FULL";
#endif
            break;
        }

        // Try to remove from the removing entries set
        bRemoved = bNOWRADProcessExpiredEntryInTheList(psNRCtrl->hRemovingSet, psEntryDesc);
        if (bRemoved == TRUE)
        {
#if DEBUG_OBJECT==1
            pacListName = "REMOVING";
#endif
            break;
        }

    } while (FALSE);

    if (bRemoved == TRUE)
    {
#if DEBUG_OBJECT==1
        printf(AGW_MGR_OBJECT_NAME": entry %p has been removed from %s set\n",
            pacListName);
#endif
    }

    return;
}

/*****************************************************************************
*
*   bNOWRADHasExpirableEntries
*
*****************************************************************************/
static BOOLEAN bNOWRADHasExpirableEntries(
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl
        )
{
    BOOLEAN bResult = TRUE;
    UN32 un32Items = 0;

    do
    {
        // Check each of list by asking their size in elements.
        // The return values are useless and let's ignore that
        // using only no of utems reported.
        OSAL.eLinkedListItems(psNRCtrl->hFullSet, &un32Items);
        if (un32Items > 0)
        {
            break;
        }

        OSAL.eLinkedListItems(psNRCtrl->hProgressSet, &un32Items);
        if (un32Items > 0)
        {
            break;
        }

        OSAL.eLinkedListItems(psNRCtrl->hRemovingSet, &un32Items);
        if (un32Items > 0)
        {
            break;
        }

        bResult = FALSE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   vNOWRADClearAndDestroyList
*
*****************************************************************************/
static void vNOWRADClearAndDestroyList(
    OSAL_OBJECT_HDL hList
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    AGW_RETURN_CODE_ENUM eReturnCode;

    printf(AGW_MGR_OBJECT_NAME
           ": attempting to clean up and destroy list %p\n", hList);

    eReturnCode = eAgwDSRLEntryDescListReleaseFull(hList);
    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to clean up list (%s)",
                AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                    );
    }

    eOsalReturnCode = OSAL.eLinkedListDelete(hList);
    if (eOsalReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to delete list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
    }

    return;
}

/*****************************************************************************
*
*   bNOWRADMoveListAndCleanIterator
*
*****************************************************************************/
static BOOLEAN bNOWRADMoveListAndCleanIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc,
    AGW_MOVE_AND_CLEAN_LIST_ITERATOR *psIterator
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Move item to the new state
    psTileDesc->eState = psIterator->eState;

    // Add item to another list
    eReturnCode = OSAL.eLinkedListAdd(psIterator->hList,
                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTileDesc);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to move DSRL Tile Desc %p to list %p (%s)",
                psTileDesc, psIterator->hList,
                OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return TRUE;
}

/*****************************************************************************
*
*   vNOWRADMoveListAndClean
*
*   Moves Tiles from one list to another with state changing. After execution
*   source list will be empty
*
*****************************************************************************/
static void vNOWRADMoveListAndClean(
    OSAL_OBJECT_HDL hListFrom,
    OSAL_OBJECT_HDL hListTo,
    AGW_DSRL_DESC_STATE_ENUM eNewState
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    AGW_MOVE_AND_CLEAN_LIST_ITERATOR sIterator;

    // Init the iterator's data
    sIterator.hList = hListTo;
    sIterator.eState = eNewState;

    // Do iteration
    eReturnCode = OSAL.eLinkedListIterate(hListFrom,
                    (OSAL_LL_ITERATOR_HANDLER)bNOWRADMoveListAndCleanIterator,
                    (void*)&sIterator);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to iterate the list %p (%s)",
                hListFrom, OSAL.pacGetReturnCodeName(eReturnCode));
    }

    // Clean up source list
    eReturnCode = OSAL.eLinkedListRemoveAll(hListFrom, NULL);
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to clean up list %p after moving its items to list %p (%s)",
                hListFrom, hListTo,
                OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return;
}

/*****************************************************************************
*
*   bNOWRADTimeStampCounterCallback
*
*****************************************************************************/
static BOOLEAN bNOWRADTimeStampCounterCallback(
    void *pvData,
    void *pvArg
        )
{
    AGW_NOWRAD_TILE_COUNTER_STRUCT *psArg =
        (AGW_NOWRAD_TILE_COUNTER_STRUCT *) pvArg;
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntry =
        (AGW_DSRL_ENTRY_DESC_STRUCT*) pvData;

    TIME_T tTileTimeStamp = AGW_TILE.tTimeStamp(psDSRLEntry->uData.hTile);
    if (tTileTimeStamp == psArg->tTimeStamp)
    {
        ++psArg->tCounter;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bNOWRADIsCurrentSetComplete
*
*****************************************************************************/
static BOOLEAN bNOWRADIsCurrentSetComplete(
    AGW_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN *pbIsSetCompleted
        )
{
    BOOLEAN bOk;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    AGW_STATISTICS_INFO_STRUCT sStatInfo =
        { 0, AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN, 0, 0 };
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl = &psObj->sNowradControlStruct;
    OSAL_OBJECT_HDL hList = psNRCtrl->hProgressSet;
    TIME_T tListTimeStamp = psNRCtrl->tProgressSetTimeStamp;
    UN32 un32NoOfItems;

    do
    {
        *pbIsSetCompleted = FALSE;

        // Request product statistic for particular timestamp
        bOk = bProductStatisticsGet(&psObj->sStatisticsControlStruct,
                                    tListTimeStamp,
                                    AGW_INTERNAL_PRODUCT_TYPE_NOWRAD,
                                    FALSE, &sStatInfo
                                        );
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to get statistics for NOWRAD at %d",
                   tListTimeStamp);
            break;
        }

        // Gets number of tiles in the current set
        eReturnCode = OSAL.eLinkedListItems(hList, &un32NoOfItems);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to get number of received tiles at %d (%s)",
                   psNRCtrl->tProgressSetTimeStamp,
                   OSAL.pacGetReturnCodeName(eReturnCode)
                       );
            bOk = FALSE;
            break;
        }

        // Check quantity
        if (sStatInfo.un8Expected <= un32NoOfItems)
        {
            AGW_NOWRAD_TILE_COUNTER_STRUCT sArg;
            sArg.tCounter = 0;
            sArg.tTimeStamp = tListTimeStamp;

            // At least the number of tiles is correct, however,
            // they might have different time stamps which is
            // violence to the the main trigger for marking
            // the sat as "FULL"
            eReturnCode = OSAL.eLinkedListIterate(hList,
                                    bNOWRADTimeStampCounterCallback, &sArg);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to check tile time stamps (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            else if (sArg.tCounter == sStatInfo.un8Expected)
            {
                // Bingo, we have full set
                bOk = TRUE;
                *pbIsSetCompleted = TRUE;
            }
        }
        else
        {
            // The full set has not been received and it can be
            // marked as not-completed
            *pbIsSetCompleted = FALSE;
            bOk = TRUE;
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bNOWRADControlInit
*
*****************************************************************************/
static BOOLEAN bNOWRADControlInit(
    AGW_NOWRAD_CONTROL_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eResult = OSAL_ERROR;
    BOOLEAN bOk = FALSE;

    psObj->bIsFirstFullSet = FALSE;
    psObj->bIsInProgress = FALSE;
    psObj->tProgressSetTimeStamp = 0;
    psObj->tFullSetTimeStamp = 0;

    do
    {
        // Create PROGRESS list
        eResult = OSAL.eLinkedListCreate(
            &psObj->hProgressSet,
            AGW_MGR_OBJECT_NAME": NR P List",
            (OSAL_LL_COMPARE_HANDLER)n16DSRLTileDescSortedByLocation,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
        if (eResult != OSAL_SUCCESS)
        {
            break;
        }

        // Create FULL list
        eResult = OSAL.eLinkedListCreate(
            &psObj->hFullSet,
            AGW_MGR_OBJECT_NAME": NR F List",
            (OSAL_LL_COMPARE_HANDLER)n16DSRLTileDescSortedByLocation,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
        if (eResult != OSAL_SUCCESS)
        {
            break;
        }

        // Create REMOVED items list
        eResult = OSAL.eLinkedListCreate(
            &psObj->hRemovingSet,
            AGW_MGR_OBJECT_NAME": NR R List",
            (OSAL_LL_COMPARE_HANDLER)n16DSRLTileDescSortedByLocation,
            OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                );
        if (eResult != OSAL_SUCCESS)
        {
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bNOWRADControlUninit
*
*****************************************************************************/
static BOOLEAN bNOWRADControlUninit(
    AGW_NOWRAD_CONTROL_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": attempting to uninit the NOWRAD product control structure\n");

    if (psObj->hFullSet != OSAL_INVALID_OBJECT_HDL)
    {
        vNOWRADClearAndDestroyList(psObj->hFullSet);
        psObj->hFullSet = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hProgressSet != OSAL_INVALID_OBJECT_HDL)
    {
        vNOWRADClearAndDestroyList(psObj->hProgressSet);
        psObj->hProgressSet = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hRemovingSet != OSAL_INVALID_OBJECT_HDL)
    {
        vNOWRADClearAndDestroyList(psObj->hRemovingSet);
        psObj->hRemovingSet = OSAL_INVALID_OBJECT_HDL;
    }

    return TRUE;
}

/*****************************************************************************
*
*   eNOWRADUpdateStatistics
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eNOWRADUpdateStatistics (
    AGW_MGR_OBJECT_STRUCT *psObj,
    const AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        AGW_STATISTICS_STRUCT sStatisticsData;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Time;
        UN16 un16Year;
        struct tm sTime;
        TIME_T tProposedTime;
        TIME_T tTmpTime;

        // Grab data for additional usage
        sStatisticsData.sHeader = *psHeader;

        // Get current time to extract current year
        eOsalReturnCode = OSAL.eTimeGet(&un32Time);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to get current time (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        // Extract current year
        OSAL.bMemSet(&sTime, 0, sizeof(sTime));
        tTmpTime = (TIME_T)un32Time;
        OSAL.gmtime_r((const TIME_T*)&tTmpTime, &sTime);
        un16Year = sTime.tm_year + 1900;

        // In some conditions we may want to use previous year,
        // so let's check new proposed timestamp to make sure we're not
        // going to be far in the future.
        sStatisticsData.sHeader.un16ValidYear = un16Year;
        sStatisticsData.sHeader.un16IssueYear = un16Year;
        tProposedTime =
            AGW_MGR_tCreateTimeStamp(&sStatisticsData.sHeader, FALSE);
        // Check how far we're will be with new proposed year.
        if (tProposedTime >
                (TIME_T)(un32Time + AGW_NOWRAD_THRESHOLD_FOR_FAKE_STATISTICS))
        {
            // Let's use previous year, then
            --un16Year;
        }

        // Prepare statistics data for that product
        eReturnCode =
            eAgwLLParserProcessStatisticsFillUp(
                psHeader, &sStatisticsData, un16Year,
                1, AGW_INTERNAL_PRODUCT_TYPE_NOWRAD, AGW_NOWRAD_TILE_SET_SIZE_MAX);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to prepare fake statistics (%s)",
                AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
            break;
        }

        // Update the statistics
        eReturnCode = eProductStatisticsUpdate(psObj, &sStatisticsData);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to update statistics by own brewed data (%s)",
                AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
            break;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bNOWRADMergeSetIterator
*
*****************************************************************************/
static BOOLEAN bNOWRADMergeSetIterator(
    void *pvData,
    void *pvArg
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc =
            (AGW_DSRL_ENTRY_DESC_STRUCT*)pvData;
    AGW_DSRL_TILE_DESC_LIST_ITERATOR_STRUCT *psIterator =
            (AGW_DSRL_TILE_DESC_LIST_ITERATOR_STRUCT*)pvArg;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Search for the similar raster in the list
    eReturnCode = OSAL.eLinkedListLinearSearch(psIterator->hListTo,
                    &hEntry, n16DSRLTileDescSearchByLocation,
                    psTileDesc
                        );

    switch (eReturnCode)
    {
        case OSAL_OBJECT_NOT_FOUND:
        {
            // The item hasn't been found so need to add it
            // to fill the gap in "to" list
            eReturnCode = OSAL.eLinkedListAdd(
                            psIterator->hListTo,
                            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                            psTileDesc
                                );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                       ": failed to add tile (%s)\n",
                       OSAL.pacGetReturnCodeName(eReturnCode)
                              );
            }
        }
        break;
        case OSAL_SUCCESS:
        {
            // Since the item in the list we may mark the item which
            // we was looking for as REMOVING for further releasing
            psTileDesc->eState = AGW_DSRL_ENTRY_DESC_STATE_REMOVING;
            eReturnCode = OSAL.eLinkedListAdd(psIterator->hListToRemove,
                                  OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTileDesc
                                                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to register tile as removing one (%s)",
                       OSAL.pacGetReturnCodeName(eReturnCode)
                              );
            }
        }
        break;
        default:
        {
            // Any other error code means unexpected situation
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to search tile desc info in the list (%s)",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
        break;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bNOWRADMergeSet
*
*   Add items from the second list if the first one doesn't have them
*
*   Inputs:
*       hListTo   - list to be updated
*       hListFrom - source list of tiles
*       hListToRemove - will keep the tiles in removing state
*
*   Output:
*       TRUE - if operation has been completed successfully and
*       FALSE otherwise
*****************************************************************************/
static BOOLEAN bNOWRADMergeSet(
    OSAL_OBJECT_HDL hListTo,
    OSAL_OBJECT_HDL hListFrom,
    OSAL_OBJECT_HDL hListToRemove
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    AGW_DSRL_TILE_DESC_LIST_ITERATOR_STRUCT sIterator;

    do
    {
        sIterator.hListTo = hListTo;
        sIterator.hListToRemove = hListToRemove;

        eReturnCode = OSAL.eLinkedListIterate(
                            hListFrom,
                            bNOWRADMergeSetIterator,
                            &sIterator
                                );

        if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to iterate through list during merge (%s)\n",
                   OSAL.pacGetReturnCodeName(eReturnCode)
                          );
            break;
        }
        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   vNOWRADSwapSets
*
*****************************************************************************/
static void vNOWRADSwapSets(
    AGW_NOWRAD_CONTROL_STRUCT *psObj
        )
{
    OSAL_OBJECT_HDL hTmpList;

    printf(AGW_MGR_OBJECT_NAME": perform list swapping\n");
    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": perform list swapping\n");

    // Swap lists
    hTmpList = psObj->hFullSet;
    psObj->hFullSet = psObj->hProgressSet;
    psObj->hProgressSet = hTmpList;

    // Swap timestamps
    psObj->tFullSetTimeStamp = psObj->tProgressSetTimeStamp;
    psObj->tProgressSetTimeStamp = 0;

    return;
}

/*****************************************************************************
*
*   bNOWRADBuildDSRL
*
*****************************************************************************/
static BOOLEAN bNOWRADBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl = &psObj->sNowradControlStruct;

    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": building DSRL for NOWRAD\n");
    printf(AGW_MGR_OBJECT_NAME": building DSRL for NOWRAD\n");

    // Looking through all Tiles in the selected list and
    // notify about them each DSRL which id interested in it.
    if (psNRCtrl->bIsFirstFullSet == TRUE)
    {
        hList = psNRCtrl->hFullSet;
    }
    else
    {
        hList = psNRCtrl->hProgressSet;
    }

    // Iterating through all available tiles to fill in DSLR by
    // corresponded data.
    eReturnCode = OSAL.eLinkedListIterate(hList,
                        (OSAL_LL_ITERATOR_HANDLER) bAgwDSRLAddEntry,
                        psDSRLDesc
                                );
    if ((eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed build iterate list %p to update DSRL %p (%s)",
                hList, psDSRLDesc, OSAL.pacGetReturnCodeName(eReturnCode)
                    );
    }
    else
    {
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*   bNOWRADNotifyAppBySet
*
*   Notifies application by set of NOWARAD tiles
*
*   Input:
*      psObj       - reference to the AGW_MGR instance
*      bUseFullSet - requests notification by completed set of tiles. However,
*                    if the manager doesn't have it the Progress one will be used
*
*   Output:
*      TRUE in case of success and
*      FALSE otherwise.
*
*****************************************************************************/
static BOOLEAN bNOWRADNotifyAppBySet(
    AGW_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bUseFullSet
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bHasDSRLs = FALSE;
    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl = &psObj->sNowradControlStruct;

    do
    {
        printf(AGW_MGR_OBJECT_NAME": notify applications by %s set\n",
               (bUseFullSet ? "FULL" : "PROGRESS")
                      );

        DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": notify applications by %s set\n",
            (bUseFullSet ? "FULL" : "PROGRESS")
                 );

        //==================================================================
        // Try to skip processing set if there is no registered DSRLs
        bHasDSRLs = bAgwLLHasDSRLs(psObj);
        if (bHasDSRLs == TRUE)
        {
            //==================================================================
            // Looking tiles in removing state and remove them from involved
            // DSRLs
            bOk = bAgwDSRLUnlinkAllEntries(psNRCtrl->hRemovingSet);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to process Unlink DSRL Entry Descs"
                            );
                break;
            }

            //==================================================================
            // Looking through all tiles in the selected list and
            // notify about them each DSRL which id interested in it.
            if ((psNRCtrl->bIsFirstFullSet == TRUE) && (bUseFullSet == TRUE))
            {
                hList = psNRCtrl->hFullSet;
            }
            else
            {
                hList = psNRCtrl->hProgressSet;
            }

            eOsalReturnCode = OSAL.eLinkedListIterate(hList,
                            (OSAL_LL_ITERATOR_HANDLER)bAgwDSRLUpdateEntryInDSRLIterator,
                            psObj
                                );
            if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed iterate list %p to update DSRLs (%s)",
                        hList, OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            //==================================================================
            // Move all DSRLs from UPDATING state to READY
            // We don't need to worry about error here.
            (void) eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
        }
        else
        {
            printf(AGW_MGR_OBJECT_NAME": there are no DSRLs to notify.\n");
        }

        //==================================================================
        // Permanently remove entries which were marked as removed
        (void)eAgwDSRLEntryDescListReleaseFull(psNRCtrl->hRemovingSet);

        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        // We don't need to case about the error. Even in case of
        // error our function return FASLE
        (void) eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_ERROR);
    }

    return bOk;
}

/*****************************************************************************
*
*   bNOWRADTryToCompleteSet
*
*****************************************************************************/
static BOOLEAN bNOWRADTryToCompleteSet(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bIsCompleted = FALSE;
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl =
            &psObj->sNowradControlStruct;
    do
    {

        bOk = bNOWRADIsCurrentSetComplete(psObj, &bIsCompleted);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get set completion status\n");
            break;
        }

        if (bIsCompleted == TRUE)
        {
            // Move all tiles from full set to removing set
            // since all recent full set of tiles became invalid and we
            // have to notify application about them
            vNOWRADMoveListAndClean(psNRCtrl->hFullSet,
                    psNRCtrl->hRemovingSet, AGW_DSRL_ENTRY_DESC_STATE_REMOVING);

            // Swap sets
            vNOWRADSwapSets(psNRCtrl);

            psNRCtrl->bIsFirstFullSet = TRUE;
            psNRCtrl->bIsInProgress = FALSE;

            bNOWRADNotifyAppBySet(psObj, TRUE);

            printf(AGW_MGR_OBJECT_NAME
                   ": full set of tiles has been detected\n");
        }
        else
        {
            if (psNRCtrl->bIsFirstFullSet != TRUE)
            {
                // Keep notifying until a full set of tiles received
                bOk = bNOWRADNotifyAppBySet(psObj, FALSE);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                           ": failed to notify application\n");
                }
            }
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   eNOWRADPutTileDescToSet
*
*   This function puts a new tile to the tiles list. The list is expected
*   to be unique and the function handles this properly.
*
*   INPUT:
*       hList       - target list
*       ppsTileDesc - the reference to the memory which contains reference
*                     to the real tilde descriptor.
*       bReplace    - says to the function to replace existing tile and
*                     expose exiting one via ppsTileDesc to let caller
*                     handle it
*       hRemovingEntiresList - the list of the entries which are going
*                     to be removed. This list is updated by replaced entry
*                     in case case of bReplace == TRUE.
*
*   OUTPUT:
*       AGW_RETURN_CODE_SUCCESS -
*           if tile is added;
*       AGW_RETURN_CODE_DUPLICATED_TILE -
*           if tile has not been added because similar one already exists.
*       AGW_RETURN_CODE_SUBST_DUPLICATED_TILE -
*           if tile already exists, but caller wished to replace old one by new one.
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eNOWRADPutTileDescToSet(
    OSAL_OBJECT_HDL hList,
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsTileDesc,
    BOOLEAN bReplace,
    OSAL_OBJECT_HDL hRemovingEntiresList
        )
{
    AGW_RETURN_CODE_ENUM eReturn = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    UN32 un32Items = 0;

    do
    {
        // Add item to the ProgressSet
        eReturnCode = OSAL.eLinkedListAdd(hList, &hEntry, *ppsTileDesc);
        if (eReturnCode != OSAL_SUCCESS)
        {
            if (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to add object to list (%s)\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else if (bReplace == TRUE)
            {
                // Keep the entry to delete for future
                AGW_DSRL_ENTRY_DESC_STRUCT *psTileDescToRemove =
                    (AGW_DSRL_ENTRY_DESC_STRUCT*) OSAL.pvLinkedListThis(hEntry);

                // Remove that record
                eReturnCode = OSAL.eLinkedListReplaceEntry(hList, hEntry, *ppsTileDesc);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to remove object from list before replace (%s)\n",
                            OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }
                else
                {
                    // Well need to put replaced record to the list for
                    // removed ones.
                    // The tile is not unique but it was used to substitute already
                    // existing one in the set. So, just remove it in order to be
                    // ensured that application will see the substitution.
                    // This object already removed from the ProgressSet and we just need
                    // to put it into the RemovingSet with appropriate state.

                    // Make sure we wont loose the entry in case of adding error
                    *ppsTileDesc = psTileDescToRemove;
                    psTileDescToRemove->eState = AGW_DSRL_ENTRY_DESC_STATE_REMOVING;

                    eReturnCode = OSAL.eLinkedListAdd(hRemovingEntiresList,
                                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psTileDescToRemove
                                                );
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(
                            gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to add new tile (%s)",
                            OSAL.pacGetReturnCodeName(eReturnCode)
                                );
                        break;
                    }
                }
            }
            else
            {
                printf(AGW_MGR_OBJECT_NAME
                       ": [%d] trying to add some object which already exists\n",
                       __LINE__
                              );
                eReturn = AGW_RETURN_CODE_DUPLICATED_TILE;
                break;
            }
        }
        else
        {
            // Get number of items in the list to make sure we're
            // not out of max available tiles.
            eReturnCode = OSAL.eLinkedListItems(hList, &un32Items);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to get number of list entries (%s)",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            // Check size of the list against maximum allowed
            if (un32Items > AGW_NOWRAD_TILE_SET_SIZE_MAX)
            {
                // The item has been added right now, but it has to be removed since
                // the list size is bigger that acceptable by the requirements
                puts(AGW_MGR_OBJECT_NAME
                    ": there are more items than allowed by the requirement. "
                    "So remove just added one");

                // Remove just added entry from the list
                eReturnCode = OSAL.eLinkedListRemove(hEntry);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to get number of list entries (%s)",
                            OSAL.pacGetReturnCodeName(eReturnCode));
                }

                // Tile not needed in that list
                eReturn = AGW_RETURN_CODE_REDUNDANT_TILE;
                break;
            }
        }

        // Move ownership of the data to the list
        *ppsTileDesc = NULL;

        eReturn = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
    OSAL.eLinkedListItems(hList, &un32Items);

    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": list has %2d item(s)\n", un32Items);
    printf(AGW_MGR_OBJECT_NAME": list has %2d item(s)\n", un32Items);
#endif

    return eReturn;
}


/*****************************************************************************
*
*   eNOWRADProcessGeneratedTile
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eNOWRADProcessGeneratedTile(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsTileDesc,
    TIME_T tTimeStamp)
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    BOOLEAN bSuccess = FALSE;
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl = &psObj->sNowradControlStruct;

    do
    {

        printf (AGW_MGR_OBJECT_NAME
            ": NOWRad Statement: IS_FULL=%s, IN_PROG=%s, TIME_CHANGED=%s\n",
            psNRCtrl->bIsFirstFullSet ? "TRUE" : "FALSE",
            psNRCtrl->bIsInProgress ? "TRUE" : "FALSE",
            (tTimeStamp != psNRCtrl->tProgressSetTimeStamp) ? "TRUE" : "FALSE"
                );

        // Check the date/time changed
        if (tTimeStamp != psNRCtrl->tProgressSetTimeStamp)
        {
            if (psNRCtrl->bIsInProgress == TRUE)
            {
                if (psNRCtrl->bIsFirstFullSet == TRUE)
                {
                    // Merge sets
                    bSuccess = bNOWRADMergeSet(psNRCtrl->hProgressSet,
                                               psNRCtrl->hFullSet,
                                               psNRCtrl->hRemovingSet);
                    if (bSuccess != TRUE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                               ": failed to merge sets"
                                    );
                        break;
                    }

                    // NOTE: At this point we have the following situation:
                    // 1) The ProgressSet has all valid tiles including
                    //    tiles which might be added from the FullSet
                    // 2) The full set has some set of tiles some of them
                    //    in the ProgressSet and rest of them have REMOVING
                    //    state.
                    // 3) The RemovingSet has tiles which are no longer required
                    //    in DSRL and they have REMOVING state.
                    //    The same tiles are situated in the FullSet.

                    // Swap sets
                    vNOWRADSwapSets(psNRCtrl);

                    // NOTE: At this point the FullSet has entity of the recent
                    //       ProgressSet. In turn, the ProgressSet has entity of
                    //       recent FullSet.

                    // NOTE: Just make the ProgressSet empty w/o removing
                    // its items
                    OSAL.eLinkedListRemoveAll(psNRCtrl->hProgressSet, NULL);

                    // Notify application by full set
                    bSuccess = bNOWRADNotifyAppBySet(psObj, TRUE);
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                               ": failed to notify application"
                                    );
                        break;
                    }

                    // Put new tile to the ProgressSet
                    eReturnCode = eNOWRADPutTileDescToSet(psNRCtrl->hProgressSet,
                                        ppsTileDesc, FALSE, OSAL_INVALID_OBJECT_HDL
                                            );
                    if ((eReturnCode == AGW_RETURN_CODE_DUPLICATED_TILE) ||
                        (eReturnCode == AGW_RETURN_CODE_REDUNDANT_TILE))
                    {
                        printf(AGW_MGR_OBJECT_NAME": skipped with %s\n",
                            AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
                        break;
                    }
                    else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                    {
                        // Only print error message for real issue.
                        // Attempt to add duplicated entry is logged only
                        //for debug purposes
                        SMSAPI_DEBUG_vPrintErrorFull(
                            gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to add new tile"
                                );
                        break;
                    }
                    psNRCtrl->tProgressSetTimeStamp = tTimeStamp;

                    psNRCtrl->bIsInProgress = TRUE;

                    bSuccess = bNOWRADTryToCompleteSet(psObj);
                    if (bSuccess != TRUE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                               ": failed to make attempt to complete "
                               "current tiles set");
                        break;
                    }
                }
                else /* psNRCtrl->bIsFirstFullSet == FALSE */
                {
                    psNRCtrl->bIsInProgress = TRUE;

                    // Put new tile to the ProgressSet
                    eReturnCode = eNOWRADPutTileDescToSet(psNRCtrl->hProgressSet,
                                    ppsTileDesc, TRUE, psNRCtrl->hRemovingSet);
                    if ((eReturnCode == AGW_RETURN_CODE_DUPLICATED_TILE) ||
                        (eReturnCode == AGW_RETURN_CODE_REDUNDANT_TILE))
                    {
                        printf(AGW_MGR_OBJECT_NAME": skipped with %s\n",
                            AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
                        break;
                    }
                    else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                    {
                        // Only print error message for real issue.
                        // Attempt to add duplicated entry is logged only
                        // for debug purposes
                        SMSAPI_DEBUG_vPrintErrorFull(
                            gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to add new tile"
                                );
                        break;
                    }

                    psNRCtrl->tProgressSetTimeStamp = tTimeStamp;

                    bSuccess = bNOWRADNotifyAppBySet(psObj, FALSE);
                    if (bSuccess == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                               ": failed to notify application");
                        break;
                    }
                }
            }
            else /* psNRCtrl->bIsInProgress == FALSE */
            {
                // Check does we need to replace the tile or not
                BOOLEAN bReplaceTile = psNRCtrl->bIsFirstFullSet == TRUE ? FALSE : TRUE;
                // Put new tile to the ProgressSet
                eReturnCode = eNOWRADPutTileDescToSet(psNRCtrl->hProgressSet,
                    ppsTileDesc, bReplaceTile, psNRCtrl->hRemovingSet
                        );
                if ((eReturnCode == AGW_RETURN_CODE_DUPLICATED_TILE) ||
                    (eReturnCode == AGW_RETURN_CODE_REDUNDANT_TILE))
                {
                    printf(AGW_MGR_OBJECT_NAME": skipped with %s\n",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
                    break;
                }
                else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    // Only print error message for real issue.
                    // Attempt to add duplicated entry is logged only
                    // for debug purposes
                    SMSAPI_DEBUG_vPrintErrorFull(
                        gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to add new tile"
                            );
                    break;
                }
                psNRCtrl->tProgressSetTimeStamp = tTimeStamp;

                psNRCtrl->bIsInProgress = TRUE;

                bSuccess = bNOWRADTryToCompleteSet(psObj);
                if (bSuccess != TRUE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                           ": failed to make attempt to complete current"
                           " tiles set");
                    break;
                }
            }
        }
        else /* tTimeStamp == psNRCtrl->tProgressSetTimeStamp */
        {
            // Check does we need to replace the tile or not
            BOOLEAN bReplaceTile = psNRCtrl->bIsFirstFullSet == TRUE ? FALSE : TRUE;
            // Put new tile to the ProgressSet
            eReturnCode = eNOWRADPutTileDescToSet(psNRCtrl->hProgressSet,
                ppsTileDesc, bReplaceTile, psNRCtrl->hRemovingSet
                    );
            if ((eReturnCode == AGW_RETURN_CODE_DUPLICATED_TILE) ||
                (eReturnCode == AGW_RETURN_CODE_REDUNDANT_TILE))
            {
                printf(AGW_MGR_OBJECT_NAME": skipped with %s\n",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
                break;
            }
            else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                    // Only print error message for real issue.
                    // Attempt to add duplicated entry is logged only
                    //for debug purposes
                    SMSAPI_DEBUG_vPrintErrorFull(
                        gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to add new tile"
                            );
                break;
            }
            psNRCtrl->tProgressSetTimeStamp = tTimeStamp;

            psNRCtrl->bIsInProgress = TRUE;
            bSuccess = bNOWRADTryToCompleteSet(psObj);
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                       ": failed to make attempt to complete current tiles set");
                break;
            }
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eNOWRADProcess
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eNOWRADProcess(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    TIME_T tTimeStamp = 0;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc = NULL;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    AGW_PRODUCT_HEADER_STRUCT sHeader;
    BOOLEAN bRemoveOutputFile = TRUE;
#if (AGW_NOWRAD_TIMEOUT_SUPPORT == 1)
    BOOLEAN bExpired = FALSE;
#endif

    do
    {
        OSAL.bMemSet(&sHeader, 0, sizeof(sHeader));

        // Request the PARSER to process NOWRAD payload data
        eReturnCode = eAgwLLParserProcessTileDataBasedProduct(
                        psObj->psParserObj,
                        &sHeader
                            );

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to process NOWRAD data via Parser (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
            break;
        }

        // Update the year in the header from the Statistics
        sHeader.un16IssueYear = psStatInfo->un16Year;
        sHeader.un16ValidYear = psStatInfo->un16Year;

        // Check product timestamp
        tTimeStamp = AGW_MGR_tCreateTimeStamp(&sHeader, FALSE);
        if (tTimeStamp <
            psObj->sNowradControlStruct.tFullSetTimeStamp)
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": timestamp from previous set detected. "
                   "The tile has been dropped (%d < %d)\n",
                   tTimeStamp,
                   psObj->sNowradControlStruct.tFullSetTimeStamp
                          );
            DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                ": timestamp from previous set detected. "
                "The tile has been dropped (%d < %d)\n",
                tTimeStamp,
                psObj->sNowradControlStruct.tFullSetTimeStamp
                       );
            break;
        }

        if (tTimeStamp <
            psObj->sNowradControlStruct.tProgressSetTimeStamp)
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": timestamp from previous set detected. "
                   "The tile has been dropped (%d < %d)\n",
                   tTimeStamp,
                   psObj->sNowradControlStruct.tProgressSetTimeStamp
                          );
            DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                ": timestamp from previous set detected. "
                "The tile has been dropped (%d < %d)\n",
                tTimeStamp,
                psObj->sNowradControlStruct.tProgressSetTimeStamp
                       );
            break;
        }

#if (AGW_NOWRAD_TIMEOUT_SUPPORT == 1)
        // Check the product expiration in order to do not process the data
        // at all
        bExpired = bAgwDSRLIsExpiredByTimeStamp(tTimeStamp, AGW_NOWRAD_TILE_TIMEOUT_SEC);
        if (bExpired == TRUE)
        {
#if DEBUG_OBJECT == 1
            OSAL.ctime_r(&tTimeStamp, psAppObj->acBuffer);
            printf(AGW_MGR_OBJECT_NAME": NOWRad time is too old to be processed %s (%u)\n",
                    psAppObj->acBuffer, tTimeStamp);
#endif
            eReturnCode = AGW_RETURN_CODE_DATA_TIMEOUT;
            break;
        }
#endif

        psTileDesc = psAgwDSRLEntryDescCreate(psAppObj,
                                &sHeader,
                                AGW_DSRL_ENTRY_TILE,
                                AGW_SHAPE_TYPE_UNKNOWN
                                    );
        if (psTileDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to create DSRL Tile Desc");
            break;
        }

        // We are ok to process deframed file
        bRemoveOutputFile = FALSE;

        eReturnCode = eAgwLLParserGenerateTile(psObj->psParserObj, &sHeader,
                            AGW_RASTER_PLANE_ALL, psTileDesc->uData.hTile
                                );

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to generate raster (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        eReturnCode = eNOWRADProcessGeneratedTile(psObj, &psTileDesc, tTimeStamp);

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to process generated tile (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (bRemoveOutputFile == TRUE)
    {
        eAgwLLParserRemoveOutputFile(psObj->psParserObj);
    }
    eAgwLLParserRemoveInputFile(psObj->psParserObj);
    eAgwLLParserCleanupBuffer(psObj->psParserObj);

    if (psTileDesc != NULL)
    {
        vAgwDSRLEntryDescReleaseFull(psTileDesc);
        psTileDesc = NULL;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vStormAttributesDataItemDestroy
*
******************************************************************************/
static void vStormAttributesDataItemDestroy (
    AGW_STORM_ATTRIBUTES_DATA_ITEM *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;

    if (psObj != NULL)
    {
        if (psObj->hShapes != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = eAgwDSRLEntryDescListReleaseFull(psObj->hShapes);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up shapes list (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hShapes);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up shapes list (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }
            psObj->hShapes = OSAL_INVALID_OBJECT_HDL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   psStormAttributesDataItemCreate
*
******************************************************************************/
static AGW_STORM_ATTRIBUTES_DATA_ITEM *psStormAttributesDataItemCreate (
     AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_STORM_ATTRIBUTES_DATA_ITEM *psObj = NULL;

    do
    {
        psObj = (AGW_STORM_ATTRIBUTES_DATA_ITEM*)SMSO_hCreate(
                                AGW_MGR_OBJECT_NAME":StrmAttrDataItm",
                                sizeof(AGW_STORM_ATTRIBUTES_DATA_ITEM),
                                (SMS_OBJECT)psAppObj,
                                FALSE
                                    );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create obj"
                        );
            break;
        }

        // Create shapes list
        psObj->hShapes = OSAL_INVALID_OBJECT_HDL;
        eOsalReturnCode = OSAL.eLinkedListCreate(&psObj->hShapes,
            AGW_MGR_OBJECT_NAME":StAttrDtItm:ShpsLst", NULL,
            OSAL_LL_OPTION_LINEAR
                );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME
                ": failed to create Storm Attributes Shapes list"
                    );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        vStormAttributesDataItemDestroy(psObj);
        psObj = NULL;
    }

    return psObj;
}


/*****************************************************************************
*
*   bStormAttributesInit
*
******************************************************************************/
static BOOLEAN bStormAttributesInit (
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    do
    {
        if (psObj == NULL)
        {
            break;
        }
        psObj->psData = NULL;
        bOk = TRUE;
    } while (FALSE);

    return bOk;
}


/*****************************************************************************
*
*   bStormAttributesBuildDSRL
*
*****************************************************************************/
static BOOLEAN bStormAttributesBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = TRUE;
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psSACtrl =
            &psObj->sStormAttrsControlStruct;

    printf(AGW_MGR_OBJECT_NAME": building DSRL for STORM ATTRIBUTES\n");
    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": building DSRL for STORM ATTRIBUTES\n");


    if (psSACtrl->psData != NULL)
    {
        bOk = bAgwDSRLAddEntries(psSACtrl->psData->hShapes, psDSRLDesc);
        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to fill in DSRL %p by Storm Attributes list",
                psDSRLDesc
            );
        }
    }

    return bOk;
}


/*****************************************************************************
*
*   eStormAttributesProcess
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eStormAttributesProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psSACtrl = &psObj->sStormAttrsControlStruct;
    AGW_STORM_ATTRIBUTES_DATA_ITEM **ppsDataItem = &psSACtrl->psData;
    AGW_STORM_ATTRIBUTES_DATA_ITEM *psDataItem = NULL;
    UN16 un16Count = 0;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    AGW_DSRL_ENTRY_DESC_STRUCT *psShapeDesc = NULL;
    BOOLEAN bOk = FALSE, bHasDSRLs = FALSE;
    AGW_PRODUCT_HEADER_STRUCT sHeader;
    TIME_T tTimeStamp = 0;

    do
    {
        eReturnCode = eAgwLLParserProcessStormAttributes(psObj->psParserObj,
                &sHeader, &un16Count
                    );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                   ": failed to process STORM ATTRIBUTES data via Parser (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
            break;
        }

        // Update year from statistics
        sHeader.un16IssueYear = psStatInfo->un16Year;
        sHeader.un16ValidYear = psStatInfo->un16Year;

        tTimeStamp = AGW_MGR_tCreateTimeStamp(&sHeader, FALSE);
        // Is the payload has the same data?
        if (*ppsDataItem != NULL)
        {
            if (tTimeStamp <= (*ppsDataItem)->tTimeStamp)
            {
                printf(AGW_MGR_OBJECT_NAME
                       ": skipped since new data is older (or the same) than existing one\n");
                eReturnCode = AGW_RETURN_CODE_SUCCESS;
                break;
            }
        }

        psDataItem = psStormAttributesDataItemCreate(psAppObj);
        if (psDataItem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Storm Attributes Data Item"
                        );
            break;
        }

        psDataItem->sHeader = sHeader;
        psDataItem->tTimeStamp = tTimeStamp;

        // Initialize return code by success. It will be the same if all
        // parsing process will be completed successfully
        eReturnCode = AGW_RETURN_CODE_SUCCESS;

        // Read all records one-by-one
        while (un16Count-- > 0)
        {
            // Create DSRL entry description
            psShapeDesc = psAgwDSRLEntryDescCreate(psAppObj,
                                        &psDataItem->sHeader,
                                        AGW_DSRL_ENTRY_SHAPE,
                                        AGW_SHAPE_TYPE_STORM_ATTRIBUTES
                                            );
            if (psShapeDesc == NULL)
            {
                eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to create Shape DSRL Desc (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
                break;
            }

            eReturnCode = eAgwLLParserProcessStormAttribute(psObj->psParserObj,
                (AGW_STORM_ATTRIBUTES_OBJECT) psShapeDesc->uData.hShape,
                &psDataItem->sHeader
                    );
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                       ": failed to process Storm Attribute record (%s)",
                       AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                              );
                break;
            }

            // Add item to the list
            eOsalReturnCode = OSAL.eLinkedListAdd(psDataItem->hShapes,
                                        OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psShapeDesc
                                            );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_PARSER_OBJECT_NAME
                       ": failed to add DSRL entry desc instance %p to list %p.",
                       psShapeDesc, psDataItem->hShapes);
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }

            // Say that the record belongs to the list
            psShapeDesc = NULL;
        }


        // Release allocated memory if object has not been put to other storage
        if (psShapeDesc != NULL)
        {
            vAgwDSRLEntryDescReleaseFull(psShapeDesc);
        }

        // Check execution
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Process Application notifications

        //==================================================================
        // Try to skip processing set if there is no registered DSRLs
        bHasDSRLs = bAgwLLHasDSRLs(psObj);
        if (bHasDSRLs == TRUE)
        {
            if ((*ppsDataItem) != NULL)
            {
                //============================================================
                // Looking for shapes in removing state and remove them from
                // involved DSRLs
                bOk = bAgwDSRLUnlinkAllEntries((*ppsDataItem)->hShapes);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to process Removing DSRL Entry Descs (%s)",
                            OSAL.pacGetReturnCodeName(eReturnCode)
                                );
                }
            }

            //==================================================================
            // Looking through all shapes and notify about them each DSRL
            // which is interested in it.
            eOsalReturnCode = OSAL.eLinkedListIterate(psDataItem->hShapes,
                            (OSAL_LL_ITERATOR_HANDLER) bAgwDSRLUpdateEntryInDSRLIterator,
                            psObj
                                );
            if ((eOsalReturnCode != OSAL_SUCCESS) &&
                (eOsalReturnCode != OSAL_NO_OBJECTS) &&
                ((*ppsDataItem) != NULL))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed iterate list 0x%p to update DSRLs (%s)",
                        (*ppsDataItem)->hShapes,
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }

            //==================================================================
            // Move all DSRLs from UPDATING state to READY
            // Note: We don't need to worry about any errors here
            (void)eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
        }

        // Remove previous set
        vStormAttributesDataItemDestroy(*ppsDataItem);

        *ppsDataItem = psDataItem;
        psDataItem = NULL;

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    // Release internal buffer if exists
    eAgwLLParserCleanupBuffer(psObj->psParserObj);

    eAgwLLParserRemoveOutputFile(psObj->psParserObj);

    if (psDataItem != NULL)
    {
        // Something went wrong. Just destroy the data item
        vStormAttributesDataItemDestroy(psDataItem);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bStormAttributesUninit
*
******************************************************************************/
static BOOLEAN bStormAttributesUninit (
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": attempting to uninit the STORM ATTRIBUTES product control structure\n");

    if (psObj->psData != NULL)
    {
        vStormAttributesDataItemDestroy(psObj->psData);
        psObj->psData = NULL;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bWindsInit
*
******************************************************************************/
static BOOLEAN bWindsInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        psObj->psEasternTileDesc = NULL;

        psObj->psWesternTileDesc = NULL;

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bWindMagnitudeInit
*
******************************************************************************/
static BOOLEAN bWindMagnitudeInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    return bWindsInit(psObj);
}

/*****************************************************************************
*
*   bWindDirectionInit
*
******************************************************************************/
static BOOLEAN bWindDirectionInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    return bWindsInit(psObj);
}


/*****************************************************************************
*
*   eWindsTileGenerate
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eWindsTileGenerate (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsTileDesc,
    AGW_RASTER_PLANE_ENUM ePlane
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    TIME_T tTimeStamp = 0, tTileTimeStamp = 0;

    do
    {
        // Check existence of the tile for this area
        if (*ppsTileDesc == NULL)
        {
            // Need to create new tile
            *ppsTileDesc = psAgwDSRLEntryDescCreate(
                psObj->psAppObj, psHeader,
                AGW_DSRL_ENTRY_TILE,
                AGW_SHAPE_TYPE_UNKNOWN
                    );
            if (*ppsTileDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to create TILE Desc");
                break;
            }

            eReturnCode = eAgwLLParserGenerateTile(psObj->psParserObj, 
                                psHeader, ePlane, (*ppsTileDesc)->uData.hTile
                                    );
        }
        else
        {
            // Compare time stamps of current tile and new one
            tTimeStamp = AGW_MGR_tCreateTimeStamp(psHeader, FALSE);
            tTileTimeStamp = AGW_TILE.tTimeStamp((*ppsTileDesc)->uData.hTile);

            if (tTimeStamp > tTileTimeStamp)
            {
                printf(AGW_MGR_OBJECT_NAME
                        ": substitute existing tile by new one\n"
                            );

                eReturnCode = eAgwLLParserGenerateTile(psObj->psParserObj,
                                psHeader, ePlane, 
                                (*ppsTileDesc)->uData.hTile
                                    );
                if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to generate TILE"
                                );
                    break;
                }

                eReturnCode = eAgwDSRLEntryDescUpdateAsTile(*ppsTileDesc);
                if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME": failed to update TILE Desc %p",
                            *ppsTileDesc);
                    break;
                }

            }
            else
            {
                printf(AGW_MGR_OBJECT_NAME
                       ": no need to update since received timestamp %u "
                        "is less or equal to existing %u\n",
                        tTimeStamp, tTileTimeStamp);

                eReturnCode = AGW_RETURN_CODE_DUPLICATED_TILE;
            }
        }
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eWindsProcess
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eWindsProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode;
    AGW_PRODUCT_HEADER_STRUCT sHeader;
    AGW_WINDS_CONTROL_STRUCT *psWindMagnitudeCtrl = &psObj->sWindMagnitudeControlStruct;
    AGW_WINDS_CONTROL_STRUCT *psWindDirectionCtrl = &psObj->sWindDirectionControlStruct;
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsMagnitudeTileDesc = NULL;
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsDirectionTileDesc = NULL;
    BOOLEAN bDoAppWindMagNotification = FALSE;
    BOOLEAN bDoAppWindDirNotification = FALSE;
    BOOLEAN bDSRLUpdate = FALSE;
    BOOLEAN bIsEastern;
    BOOLEAN bRemoveOutputFile = TRUE;

    do
    {
        OSAL.bMemSet(&sHeader, 0, sizeof(sHeader));

        eReturnCode = eAgwLLParserProcessTileDataBasedProduct(psObj->psParserObj, &sHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_PARSER_OBJECT_NAME
                   ": failed to process payload (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        // Update the year in the header from the Statistics
        sHeader.un16IssueYear = psStatInfo->un16Year;
        sHeader.un16ValidYear = psStatInfo->un16Year;

        // Select proper time
        bIsEastern = AGW_MGR_IS_EASTERN_LON(sHeader.n16UpperLon);
        if (bIsEastern == TRUE)
        {
            printf(AGW_MGR_OBJECT_NAME
                    ": Eastern tile detected\n");
            ppsMagnitudeTileDesc = &psWindMagnitudeCtrl->psEasternTileDesc;
            ppsDirectionTileDesc = &psWindDirectionCtrl->psEasternTileDesc;
        }
        else
        {
            printf(AGW_MGR_OBJECT_NAME
                    ": Western tile detected\n");
            ppsMagnitudeTileDesc = &psWindMagnitudeCtrl->psWesternTileDesc;
            ppsDirectionTileDesc = &psWindDirectionCtrl->psWesternTileDesc;
        }

        // Extract Magnitude raster
        eReturnCode = eWindsTileGenerate(psObj, &sHeader, ppsMagnitudeTileDesc,
            AGW_RASTER_PLANE_0
                );

        if (eReturnCode == AGW_RETURN_CODE_SUCCESS)
        {
            bDoAppWindMagNotification = TRUE;
        }
        else if (eReturnCode != AGW_RETURN_CODE_DUPLICATED_TILE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to process wind magnitude tile (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
        }

        // Extract directions raster
        eReturnCode = eWindsTileGenerate(psObj, &sHeader, ppsDirectionTileDesc,
            AGW_RASTER_PLANE_1
                );

        if (eReturnCode == AGW_RETURN_CODE_SUCCESS)
        {
            bDoAppWindDirNotification = TRUE;
        }
        else if (eReturnCode != AGW_RETURN_CODE_DUPLICATED_TILE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to process wind direction tile (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
        }

        // Do application notification if it was requested
        if ((bDoAppWindMagNotification == TRUE) || (bDoAppWindDirNotification == TRUE))
        {
            BOOLEAN bHasDSRLs;

            // Don't need to remove output file because it is not a deframed one
            bRemoveOutputFile = FALSE;

            bHasDSRLs = bAgwLLHasDSRLs(psObj);
            if (bHasDSRLs == FALSE)
            {
                printf(AGW_MGR_OBJECT_NAME
                      ": there are no registered DSLRs to update\n");
            }
            else
            {
                BOOLEAN bOk;

                if (bDoAppWindMagNotification == TRUE)
                {
                    bOk = bAgwDSRLUpdateEntryInDSRLIterator(*ppsMagnitudeTileDesc, psObj);
                    if (bOk == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                                ": failed to update DSRL's by TILE Desc %p for Wind Magnitude",
                                *ppsMagnitudeTileDesc);
                        break;
                    }
                    else
                    {
                        bDSRLUpdate = TRUE;
                    }
                }

                if (bDoAppWindDirNotification == TRUE)
                {
                    bOk = bAgwDSRLUpdateEntryInDSRLIterator(*ppsDirectionTileDesc, psObj);
                    if (bOk == FALSE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                AGW_MGR_OBJECT_NAME
                                ": failed to update DSRL's by TILE Desc %p  for Wind Direction",
                                *ppsDirectionTileDesc);
                        break;
                    }
                    else
                    {
                        bDSRLUpdate = TRUE;
                    }
                }
            }
        }

        if (bDSRLUpdate == TRUE)
        {
            eReturnCode = eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
        }
        else
        {
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    if (bRemoveOutputFile == TRUE)
    {
        eAgwLLParserRemoveOutputFile(psObj->psParserObj);
    }
    eAgwLLParserRemoveInputFile(psObj->psParserObj);
    eAgwLLParserCleanupBuffer(psObj->psParserObj);

    return eReturnCode;
}


/*****************************************************************************
*
*   bWindsBuildDSRL
*
******************************************************************************/
static BOOLEAN bWindsBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    AGW_WINDS_CONTROL_STRUCT *psWindsCtrl
        )
{
    BOOLEAN bOk = FALSE;
    BOOLEAN bAdded = FALSE;

    printf(AGW_MGR_OBJECT_NAME": building DSRL for WINDS\n");
    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": building DSRL for WINDS\n");

    do
    {
        if (psWindsCtrl->psEasternTileDesc != NULL)
        {
            bAdded = bAgwDSRLAddEntry(psWindsCtrl->psEasternTileDesc,
                                        psDSRLDesc);
            if (bAdded == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to put Eastern tile to the DSRL %p",
                        psDSRLDesc->hDSRL);
                break;
            }
        }

        if (psWindsCtrl->psWesternTileDesc != NULL)
        {
            bAdded = bAgwDSRLAddEntry(psWindsCtrl->psWesternTileDesc,
                                        psDSRLDesc);
            if (bAdded == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to put Western tile to the DSRL %p",
                        psDSRLDesc->hDSRL);
                break;
            }
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bWindMagnitudeBuildDSRL
*
******************************************************************************/
static BOOLEAN bWindMagnitudeBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    return bWindsBuildDSRL(psObj,
        psDSRLDesc,
        &psObj->sWindMagnitudeControlStruct
            );
}

/*****************************************************************************
*
*   bWindDirectionBuildDSRL
*
******************************************************************************/
static BOOLEAN bWindDirectionBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    return bWindsBuildDSRL(psObj,
        psDSRLDesc,
        &psObj->sWindDirectionControlStruct
            );
}

/*****************************************************************************
*
*   bWindsUninit
*
******************************************************************************/
static BOOLEAN bWindsUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": attempting to uninit the WINDS product control structure\n");

    if (psObj->psEasternTileDesc != NULL)
    {
        printf(AGW_MGR_OBJECT_NAME": attempting to remove Eastern tile %p\n",
                psObj->psEasternTileDesc);
        vAgwDSRLEntryDescReleaseFull(psObj->psEasternTileDesc);
        psObj->psEasternTileDesc = NULL;
    }

    if (psObj->psWesternTileDesc != NULL)
    {
        printf(AGW_MGR_OBJECT_NAME": attempting to remove Western tile %p\n",
                psObj->psWesternTileDesc);
        vAgwDSRLEntryDescReleaseFull(psObj->psWesternTileDesc);
        psObj->psWesternTileDesc = NULL;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bWindMagnitudeUninit
*
******************************************************************************/
static BOOLEAN bWindMagnitudeUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    return bWindsUninit(psObj);
}

/*****************************************************************************
*
*   bWindDirectionUninit
*
******************************************************************************/
static BOOLEAN bWindDirectionUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        )
{
    return bWindsUninit(psObj);
}

/*****************************************************************************
*
*   vSurfaceFeatureDataItemDestroy
*
******************************************************************************/
static void vSurfaceFeatureDataItemDestroy (
    AGW_SURFACE_FEATURE_DATA_ITEM *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;

    if (psObj != NULL)
    {
        if (psObj->hShapes != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = eAgwDSRLEntryDescListReleaseFull(psObj->hShapes);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up shapes list (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hShapes);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up shapes list (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }
            psObj->hShapes = OSAL_INVALID_OBJECT_HDL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   psSurfaceFeatureDataItemCreate
*
******************************************************************************/
static AGW_SURFACE_FEATURE_DATA_ITEM *psSurfaceFeatureDataItemCreate (
     AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_SURFACE_FEATURE_DATA_ITEM *psObj = NULL;
    do
    {
        psObj = (AGW_SURFACE_FEATURE_DATA_ITEM*)SMSO_hCreate(
                                AGW_MGR_OBJECT_NAME":SrfFtrDataItm",
                                sizeof(AGW_SURFACE_FEATURE_DATA_ITEM),
                                (SMS_OBJECT)psAppObj,
                                FALSE
                                    );
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create obj"
                        );
            break;
        }

        OSAL.bMemSet(psObj, 0, sizeof(*psObj));

        // Create shapes list
        psObj->hShapes = OSAL_INVALID_OBJECT_HDL;
        eOsalReturnCode = OSAL.eLinkedListCreate(&psObj->hShapes,
                                AGW_MGR_OBJECT_NAME":SurFtrShapesList",
                                NULL,
                                OSAL_LL_OPTION_LINEAR
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Surface Feature shapes list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        vSurfaceFeatureDataItemDestroy(psObj);
        psObj = NULL;
    }

    return psObj;
}

/*****************************************************************************
*
*   bSurfaceFeatureInit
*
******************************************************************************/
static BOOLEAN bSurfaceFeatureInit (
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;

    do
    {
        // Check inputs
        if (psObj == NULL)
        {
            break;
        }

        psObj->psEastern = NULL;
        psObj->psWestern = NULL;

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   eProcessFrontShape
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessFrontShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        eReturnCode = eAgwLLParserProcessShapeFront(psParserObj,
                            (AGW_FRONT_OBJECT)psDSRLDesc->uData.hShape
                                );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process front shape (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eProcessIsobarShape
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessIsobarShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        eReturnCode = eAgwLLParserProcessShapeIsobar(
                        psParserObj, (AGW_ISOBAR_OBJECT) psDSRLDesc->uData.hShape
                            );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process isobar shape (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eProcessPressCntrShape
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessPressCntrShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        eReturnCode = eAgwLLParserProcessShapePressureCenter(
                        psParserObj, (AGW_PRESSURE_CENTER_OBJECT) psDSRLDesc->uData.hShape
                            );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process pressure center shape (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eProcessStrmPosShape
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessStrmPosShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc,
    AGW_STORM_OBJECT hStorm
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_STORM_POSITION_OBJECT hStormPosition =
        (AGW_STORM_POSITION_OBJECT)psDSRLDesc->uData.hShape;

    do
    {
        // Process the data through the low-level parser
        eReturnCode = eAgwLLParserProcessShapeStormPosition(psParserObj,
                                hStormPosition
                                    );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process storm position shape (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Attach the position to the storm
        eReturnCode = AGW_STORM_POSITION_eAttachStorm(hStormPosition, hStorm);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to assign storm to position (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }
    } while(FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eProcessWindRdFieldShape
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessWindRdFieldShape (
    AGW_APP_OBJECT_STRUCT *psAppObj,
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_STORM_OBJECT hStorm,
    AGW_DSRL_ENTRY_DESC_STRUCT *psStormPosDesc
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_WIND_RADII_AREA_OBJECT hWindRadiiArea = AGW_WIND_RADII_AREA_INVALID_OBJECT;
    AGW_STORM_POSITION_OBJECT hStormPosition =
        (AGW_STORM_POSITION_OBJECT) psStormPosDesc->uData.hShape;

    do
    {
        // Create the object
        hWindRadiiArea = AGW_WIND_RADII_AREA_hCreate((SMS_OBJECT) psAppObj, psHeader);
        if (hWindRadiiArea == AGW_WIND_RADII_AREA_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": failed to create wind radii object");
            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        eReturnCode = eAgwLLParserProcessShapeWindRadiiArea(
                        psParserObj, hWindRadiiArea
                            );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process wind radii area shape (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = AGW_STORM_POSITION_eAppendWindRadii(
                            hStormPosition, hWindRadiiArea);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to append new wind radii area to the storm position (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }
    } while (FALSE);

    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        AGW_WIND_RADII_AREA_vDestroy(hWindRadiiArea);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eProcessSurfaceFeatureShapes
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessSurfaceFeatureShapes (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psProdHeader,
    AGW_SHAPE_HEADER_STRUCT *psShapeHeader,
    OSAL_OBJECT_HDL hShapeList
        )
{
    AGW_PARSER_OBJECT_STRUCT *psParserObj = psObj->psParserObj;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    UN16 un16Shape = 0;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_DSRL_ENTRY_DESC_STRUCT *psShapeDesc = NULL;
    AGW_SHAPE_TYPE_ENUM eShapeObjectType = AGW_SHAPE_TYPE_UNKNOWN;

    for (un16Shape = 0;
         (un16Shape < psShapeHeader->un16Count) &&
         (eReturnCode == AGW_RETURN_CODE_SUCCESS);
         ++un16Shape)
    {
        // Get shape object type
        eShapeObjectType = AGW_MGR_eGetPublicShapeType(psShapeHeader->eType);
        if (eShapeObjectType == AGW_SHAPE_TYPE_UNKNOWN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unexpected type");
            eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
            break;
        }

        // Create record
        psShapeDesc = psAgwDSRLEntryDescCreate(psAppObj,
                                    psProdHeader,
                                    AGW_DSRL_ENTRY_SHAPE,
                                    eShapeObjectType
                                        );
        if (psShapeDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Shape DSRL Desc"
                        );

            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        switch (psShapeHeader->eType)
        {
            case AGW_INTERNAL_SHAPE_TYPE_FRONT:
            {
                printf(AGW_MGR_OBJECT_NAME": processing front #%d\n",
                        un16Shape
                            );
                eReturnCode = eProcessFrontShape(psParserObj,
                                    psShapeDesc
                                        );
            }
            break;
            case AGW_INTERNAL_SHAPE_TYPE_ISOBAR:
            {
                printf(AGW_MGR_OBJECT_NAME": processing isobar #%d\n",
                        un16Shape
                            );
                eReturnCode = eProcessIsobarShape(psParserObj,
                                    psShapeDesc
                                        );
            }
            break;
            case AGW_INTERNAL_SHAPE_TYPE_PRESSURE_CENTER:
            {
                printf(AGW_MGR_OBJECT_NAME": processing pressure center #%d\n",
                        un16Shape
                            );
                eReturnCode = eProcessPressCntrShape(psParserObj,
                                    psShapeDesc
                                        );
            }
            break;
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": unsupported type %d detected",
                        psShapeHeader->eType
                            );
                eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
            }
            break;
        }
        // Check switch execution
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // Add to the shape list
        eOsalReturnCode = OSAL.eLinkedListAdd(hShapeList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psShapeDesc
                                            );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to add Shape Desc %p to list %p (%s)",
                    psShapeDesc, hShapeList,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        // makes it NULL to say that it is already added to the shapes list
        psShapeDesc = NULL;
    }

    // Clean up stuff if error appeared
    if ((psShapeDesc != NULL) &&
        (eReturnCode != AGW_RETURN_CODE_SUCCESS))
    {
        vAgwDSRLEntryDescReleaseFull(psShapeDesc);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bSurfaceFeatureBuildDSRL
*
*****************************************************************************/
static BOOLEAN bSurfaceFeatureBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = TRUE;
    BOOLEAN bSuccess = FALSE;
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psSFCtrl =
            &psObj->sSurfaceFeatureControlStruct;

    printf(AGW_MGR_OBJECT_NAME": building DSRL for SURFACE FEATURE\n");

    if (psSFCtrl->psEastern != NULL)
    {
        bSuccess = bAgwDSRLAddEntries(psSFCtrl->psEastern->hShapes, psDSRLDesc);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to fill in DSRL %p by Eastern list",
                    psDSRLDesc
                        );
            bOk = FALSE;
        }
    }

    if (psSFCtrl->psWestern != NULL)
    {
        bSuccess = bAgwDSRLAddEntries(psSFCtrl->psWestern->hShapes, psDSRLDesc);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to fill in DSRL %p by Western list",
                    psDSRLDesc
                        );
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   eSurfaceFeatureProcess
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eSurfaceFeatureProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_PARSER_OBJECT_STRUCT *psParserObj = psObj->psParserObj;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psSFCtrl =
            &psObj->sSurfaceFeatureControlStruct;
    AGW_SURFACE_FEATURE_DATA_ITEM **ppsDataItem = NULL;
    AGW_SURFACE_FEATURE_DATA_ITEM *psDataItem = NULL;
    UN16 un16ShapeTypeCount = 0;
    BOOLEAN bIsEastern = FALSE;
    BOOLEAN bHasDSRLs = FALSE;
    BOOLEAN bOk = FALSE;
    TIME_T tTimeStamp = 0;

    AGW_SHAPE_HEADER_STRUCT sShapeHeader;
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    do
    {
        eReturnCode = eAgwLLParserProcessShapeDataBasedProduct(psParserObj,
                            &sHeader
                                );

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process Surface Feature via Parser (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Update year from statistics
        sHeader.un16IssueYear = psStatInfo->un16Year;
        sHeader.un16ValidYear = psStatInfo->un16Year;

        // Detect item location (western or eastern)
        bIsEastern = AGW_MGR_IS_EASTERN_LON(sHeader.n16UpperLon);
        if (bIsEastern == TRUE)
        {
            printf(AGW_MGR_OBJECT_NAME
                    ": Eastern area detected\n");
            ppsDataItem = &psSFCtrl->psEastern;
        }
        else
        {
            printf(AGW_MGR_OBJECT_NAME
                    ": Western area detected\n");
            ppsDataItem = &psSFCtrl->psWestern;
        }

        // Calculate timestamp
        tTimeStamp = AGW_MGR_tCreateTimeStamp(&sHeader, FALSE);

        // Is the payload has the same data?
        if (*ppsDataItem != NULL)
        {
            if (tTimeStamp <= (*ppsDataItem)->tTimeStamp)
            {
                printf(AGW_MGR_OBJECT_NAME
                       ": skipped since new data is older (or the same) than existing one\n");
                eReturnCode = AGW_RETURN_CODE_SUCCESS;
                break;
            }
        }

        // Create temp collection
        psDataItem = psSurfaceFeatureDataItemCreate(psAppObj);
        if (psDataItem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Surface Feature Data Item"
                        );
            break;
        }
        psDataItem->tTimeStamp = tTimeStamp;
        psDataItem->sHeader = sHeader;

        eReturnCode = eAgwLLParserGetShapeTypeCount(psParserObj, &un16ShapeTypeCount);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get shapes count (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Looking through all payload to extract data
        while ((un16ShapeTypeCount > 0) && (eReturnCode == AGW_RETURN_CODE_SUCCESS))
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(psObj->psParserObj->sOutputFile.pFile,
                    psObj->psParserObj->hBuffer);

            // Extract shape header
            eReturnCode = eAgwLLParserProcessShapeHeader(psParserObj, &sShapeHeader);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to process shape header (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
                break;
            }
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
            vPrintShapeHeader(&sShapeHeader);
#endif
            // Process following data based on
            switch (sShapeHeader.eType)
            {
                case AGW_INTERNAL_SHAPE_TYPE_FRONT:
                case AGW_INTERNAL_SHAPE_TYPE_ISOBAR:
                case AGW_INTERNAL_SHAPE_TYPE_PRESSURE_CENTER:
                {
                    eReturnCode = eProcessSurfaceFeatureShapes(psObj,
                                            &(psDataItem->sHeader), &sShapeHeader,
                                                psDataItem->hShapes
                                                    );
                }
                break;
                default:
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": unsupported by Surface Feature shape type detected (%d)",
                            sShapeHeader.eType
                                );
                    eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
                }
                break;
            }

            // Decrease number of remaining products
            --un16ShapeTypeCount;
        }

        // Check execution
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Process Application notifications

        //==================================================================
        // Try to skip processing set if there is no registered DSRLs
        bHasDSRLs = bAgwLLHasDSRLs(psObj);
        if (bHasDSRLs == TRUE)
        {
            if ((*ppsDataItem) != NULL)
            {
                //============================================================
                // Looking for shapes in removing state and remove them from
                // involved DSRLs
                bOk = bAgwDSRLUnlinkAllEntries((*ppsDataItem)->hShapes);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to process Removing DSRL Entry Descs (%s)",
                            OSAL.pacGetReturnCodeName(eReturnCode)
                                );
                }
            }

            //==================================================================
            // Looking through all shapes and notify about them each DSRL
            // which is interested in it.
            eOsalReturnCode = OSAL.eLinkedListIterate(psDataItem->hShapes,
                            (OSAL_LL_ITERATOR_HANDLER) bAgwDSRLUpdateEntryInDSRLIterator,
                            psObj
                                );
            if ((eOsalReturnCode != OSAL_SUCCESS) &&
                (eOsalReturnCode != OSAL_NO_OBJECTS) &&
                ((*ppsDataItem) != NULL))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed iterate list %p to update DSRLs (%s)",
                        (*ppsDataItem)->hShapes,
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }

            //==================================================================
            // Move all DSRLs from UPDATING state to READY
            // Note: We don't need to worry about any errors here
            (void) eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
        }

        // Remove previous set
        vSurfaceFeatureDataItemDestroy(*ppsDataItem);

        *ppsDataItem = psDataItem;
        psDataItem = NULL;

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    // Release internal buffer if exists
    eAgwLLParserCleanupBuffer(psObj->psParserObj);

    eAgwLLParserRemoveOutputFile(psObj->psParserObj);

    if (psDataItem != NULL)
    {
        // Something went wrong. Just destroy the data item
        vSurfaceFeatureDataItemDestroy(psDataItem);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bSurfaceFeatureUninit
*
******************************************************************************/
static BOOLEAN bSurfaceFeatureUninit (
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psObj
        )
{
    printf(AGW_MGR_OBJECT_NAME
           ": attempting to uninit the SURFACE FEATURE product control structure\n");

    if (psObj != NULL)
    {
        if (psObj->psEastern != NULL)
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": attempting to uninit the Eastern data area\n");

            vSurfaceFeatureDataItemDestroy(psObj->psEastern);
            psObj->psEastern = NULL;
        }

        if (psObj->psWestern != NULL)
        {
            printf(AGW_MGR_OBJECT_NAME
                   ": attempting to uninit the Western data area\n");

            vSurfaceFeatureDataItemDestroy(psObj->psWestern);
            psObj->psWestern = NULL;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   vStormTrackDataItemDestroy
*
******************************************************************************/
static void vStormTrackDataItemDestroy (
    AGW_STORM_TRACK_DATA_ITEM *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;

    if (psObj != NULL)
    {
        if (psObj->hStormPositionsList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = eAgwDSRLEntryDescListReleaseFull(psObj->hStormPositionsList);
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up shapes list (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hStormPositionsList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to delete shapes list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psObj->hStormPositionsList = OSAL_INVALID_OBJECT_HDL;
        }

        if (psObj->hStormList != OSAL_INVALID_OBJECT_HDL)
        {
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(psObj->hStormList,
                (OSAL_LL_RELEASE_HANDLER)AGW_STORM_vDestroy
                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to clean up storms list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hStormList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to delete storms list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psObj->hStormList = OSAL_INVALID_OBJECT_HDL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}


/*****************************************************************************
*
*   n16StormSort
*
******************************************************************************/
static N16 n16StormSort(
    AGW_STORM_OBJECT hStorm1,
    AGW_STORM_OBJECT hStorm2
        )
{
    N16 n16Result = 0;
    STRING_OBJECT hStormId1 = AGW_STORM_hID(hStorm1);
    STRING_OBJECT hStormId2 = AGW_STORM_hID(hStorm2);

    n16Result = STRING.n16Compare(hStormId1, hStormId2, TRUE);

    return n16Result;
}


/*****************************************************************************
*
*   psStormTrackDataItemCreate
*
******************************************************************************/
static AGW_STORM_TRACK_DATA_ITEM *psStormTrackDataItemCreate (
     AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_STORM_TRACK_DATA_ITEM *psObj = NULL;

    do
    {
        psObj = (AGW_STORM_TRACK_DATA_ITEM*)SMSO_hCreate(
                                AGW_MGR_OBJECT_NAME":StrTrkDataItm",
                                sizeof(AGW_STORM_TRACK_DATA_ITEM),
                                (SMS_OBJECT)psAppObj,
                                FALSE);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create obj"
                        );
            break;
        }

        // Create shapes list
        eOsalReturnCode = OSAL.eLinkedListCreate(&psObj->hStormPositionsList,
                                AGW_MGR_OBJECT_NAME":StrPosList",
                                NULL,
                                OSAL_LL_OPTION_LINEAR
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Storm Track shapes list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        // Create descriptions list
        eOsalReturnCode = OSAL.eLinkedListCreate(&psObj->hStormList,
                                AGW_MGR_OBJECT_NAME":StormList",
                                (OSAL_LL_COMPARE_HANDLER)n16StormSort,
                                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Storm Track shapes list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    if (bOk == FALSE)
    {
        vStormTrackDataItemDestroy(psObj);
        psObj = NULL;
    }

    return psObj;
}


/*****************************************************************************
*
*   bStormTrackInit
*
******************************************************************************/
static BOOLEAN bStormTrackInit (
    AGW_STORM_TRACK_CONTROL_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;

    // Check inputs
    if (psObj != NULL)
    {
        psObj->psData = NULL;
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*   eProcessStormDesc
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessStormDesc (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STORM_TRACK_DATA_ITEM *psDataItem,
    const AGW_SHAPE_HEADER_STRUCT *psShapeHeader,
    AGW_STORM_OBJECT *phStorm
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    STRING_OBJECT hID = STRING_INVALID_OBJECT;
    STRING_OBJECT hName = STRING_INVALID_OBJECT;
    UN16 un16NoOfStormPositions = 0;

    // Check input
    if (phStorm == NULL)
    {
        return AGW_RETURN_CODE_BAD_ARGUMENT;
    }

    do
    {
        // Try to extract storm description data which will be wrapped
        // by two STRING objects.
        eReturnCode = eAgwLLParserProcessStormDesc(psObj->psParserObj,
                                &hID, &hName, &un16NoOfStormPositions);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process storm description (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        printf(AGW_MGR_OBJECT_NAME": found the %u storm positions\n", un16NoOfStormPositions);

        // Create the storm object to handle that storm. Th
        *phStorm = AGW_STORM_hCreate((SMS_OBJECT)psDataItem, &hID, &hName);
        if (*phStorm == AGW_STORM_INVALID_OBJECT)
        {
            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        // TODO Need to make a search across all known storms at this point
        eOsalReturnCode = OSAL.eLinkedListAdd(psDataItem->hStormList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, *phStorm);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to add storm desc  0x%p to list 0x%p (%s)",
                    *phStorm, psDataItem->hStormList,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            eReturnCode = AGW_RETURN_CODE_IO_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        if (*phStorm != AGW_STORM_INVALID_OBJECT)
        {
            AGW_STORM_vDestroy(*phStorm);
            *phStorm = AGW_STORM_INVALID_OBJECT;
        }
        else
        {
            STRING_vDestroy(hID);
            STRING_vDestroy(hName);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eProcessStormTrackShapes
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessStormTrackShapes (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STORM_TRACK_DATA_ITEM *psDataItem,
    AGW_SHAPE_HEADER_STRUCT *psShapeHeader,
    AGW_STORM_OBJECT hStorm,
    AGW_DSRL_ENTRY_DESC_STRUCT ** ppsStormPosDesc
        )
{
    AGW_PARSER_OBJECT_STRUCT *psParserObj = psObj->psParserObj;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_DSRL_ENTRY_DESC_STRUCT *psShapeDesc = NULL;
    AGW_SHAPE_TYPE_ENUM eShapeObjectType;

    do
    {
        // Get shape object type
        eShapeObjectType = AGW_MGR_eGetPublicShapeType(psShapeHeader->eType);
        if (eShapeObjectType == AGW_SHAPE_TYPE_UNKNOWN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unexpected type");
            eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
            break;
        }

        if (psShapeHeader->eType == AGW_INTERNAL_SHAPE_TYPE_STORM_POSITION)
        {
            // Create record
            psShapeDesc = psAgwDSRLEntryDescCreate(psAppObj,
                                &psDataItem->sHeader,
                                AGW_DSRL_ENTRY_SHAPE,
                                eShapeObjectType
                                    );
            if (psShapeDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Shape DSRL Desc"
                        );

                eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
                break;
            }

            printf(AGW_MGR_OBJECT_NAME": processing storm position \n");
            eReturnCode = eProcessStrmPosShape(psParserObj,
                                psShapeDesc,
                                hStorm
                                    );
            if (eReturnCode == AGW_RETURN_CODE_SUCCESS)
            {
                *ppsStormPosDesc = psShapeDesc;
            }

            // Add to the storm position list
            eOsalReturnCode = OSAL.eLinkedListAdd(psDataItem->hStormPositionsList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psShapeDesc
                                        );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to add Shape Desc %p to list %p (%s)",
                    psShapeDesc, psDataItem->hStormPositionsList,
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
                eReturnCode = AGW_RETURN_CODE_ERROR;
                break;
            }
        }
        else if (psShapeHeader->eType == AGW_INTERNAL_SHAPE_TYPE_WIND_RADII_FIELD)
        {
            printf(AGW_MGR_OBJECT_NAME": processing wind radii field \n");
            eReturnCode = eProcessWindRdFieldShape(psAppObj,
                            psParserObj,
                            &psDataItem->sHeader,
                            hStorm, *ppsStormPosDesc
                                );
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unsupported type %d detected",
                psShapeHeader->eType
                );
            eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
            break;
        }
    } while (FALSE);

    // Clean up stuff if error appeared
    if ((psShapeDesc != NULL) &&
        (eReturnCode != AGW_RETURN_CODE_SUCCESS))
    {
        vAgwDSRLEntryDescReleaseFull(psShapeDesc);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bStormTrackBuildDSRL
*
*****************************************************************************/
static BOOLEAN bStormTrackBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        )
{
    BOOLEAN bOk = TRUE;
    AGW_STORM_TRACK_CONTROL_STRUCT *psSTCtrl = &psObj->sStormTrackControlStruct;

    puts(AGW_MGR_OBJECT_NAME": building DSRL for STORM TRACK");
    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME": building DSRL for STORM TRACK\n");

    if (psSTCtrl->psData != NULL)
    {
        BOOLEAN bSuccess;
        bSuccess = bAgwDSRLAddEntries(psSTCtrl->psData->hStormPositionsList,
                                psDSRLDesc);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to fill in DSRL %p by Storm tracks list",
                    psDSRLDesc
                        );
            bOk = FALSE;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   eStoreDataToCache
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eStoreDataToCache(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo,
    UN32 un32TimeStamp,
    const char *pacFileName
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    FILE *psCacheFile = NULL;
    FILE *psDataFile = NULL;
    N32 n32Result = 0;
    char *pacFilePath = NULL;

    do
    {
        AGW_CACHE_HEADER_STRUCT sCacheHeader;
        BOOLEAN bSuccess = FALSE;
        size_t tElems = 0;
        size_t tWrittenElems = 0;

        printf(AGW_MGR_OBJECT_NAME": storing cache to %s\n",
            psObj->psParserObj->sOutputFile.pacData);

        // open decoded data file
        psDataFile = fopen(psObj->psParserObj->sOutputFile.pacData, "rb");
        if (psDataFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": cannot open data file %s",
                    psObj->psParserObj->sOutputFile.pacData
                        );
            break;
        }

        //filling cache file header
        OSAL.bMemSet(&sCacheHeader, 0, sizeof(sCacheHeader));

        bSuccess = OSAL.bFileSystemGetFileSize(psDataFile,
            &(sCacheHeader.tPayloadSize)
                );
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": cannot get data file size %s",
                    psObj->psParserObj->sOutputFile.pacData
                        );
            break;
        }

        sCacheHeader.un32TimeStamp = un32TimeStamp;
        sCacheHeader.sStatInfo = *psStatInfo;

#if ((SMS_DEBUG == 1) && (DEBUG_OBJECT == 1))
        vPrintCacheHeader(&sCacheHeader);
#endif

        eReturnCode = eGenerateCacheFilePath(psObj->psAppObj, pacFileName,
            &pacFilePath
                );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": cannot generate cache file path"
                        );
            break;
        }

        // create cache data file
        psCacheFile = fopen(pacFilePath, "wb+");
        if (psCacheFile == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": cannot open cache file %s",
                    pacFilePath
                        );
            DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                ": cannot open cache file %s\n", pacFilePath
                    );
            break;
        }

        tElems = fwrite(&sCacheHeader, sizeof(sCacheHeader), 1, psCacheFile);
        if (tElems != 1)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to write cache header to file %s",
                    pacFilePath
                        );
            break;
        }

        // Write data from buffer to file
        do
        {
            tElems = fread(psObj->psAppObj->acBuffer, sizeof(char),
                                    AGW_SHARED_BUFFER_LEN, psDataFile
                                        );
            if (tElems > 0)
            {
                tWrittenElems = fwrite(psObj->psAppObj->acBuffer,
                    sizeof(char), tElems, psCacheFile
                        );
                if (tWrittenElems != tElems)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_PARSER_OBJECT_NAME
                           ": failed to write data to file %s",
                           pacFilePath);
                    bSuccess = FALSE;
                    break;
                }
            }
        }
        while (tElems > 0);
        if (bSuccess == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        fflush(psCacheFile);
        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    if (psDataFile != NULL)
    {
        fclose(psDataFile);
    }

    if (psCacheFile != NULL)
    {
        fclose(psCacheFile);
        // in case of any error, removing cache file
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            n32Result = remove(pacFilePath);
            if (n32Result != 0)
            {
                printf(AGW_MGR_OBJECT_NAME": cannot remove file %s, %d\n",
                    pacFilePath, n32Result
                        );
            }
        }
    }

    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacFilePath);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eStormTrackDataProcess
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eStormTrackDataProcess(
    AGW_MGR_OBJECT_STRUCT *psObj,
    UN16 un16ShapeTypeCount,
    AGW_STORM_TRACK_DATA_ITEM *psDataItem
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_SUCCESS;
    AGW_PARSER_OBJECT_STRUCT *psParserObj = psObj->psParserObj;
    AGW_SHAPE_HEADER_STRUCT sShapeHeader;
    AGW_STORM_OBJECT hStorm = AGW_STORM_INVALID_OBJECT;
    AGW_DSRL_ENTRY_DESC_STRUCT *psStormPos = NULL;

    printf(AGW_MGR_OBJECT_NAME
            ": Processing Storm Track Data defined by b%u shape types\n",
            un16ShapeTypeCount);

    // Looking through all payload to extract data
    while ((un16ShapeTypeCount > 0) && (eReturnCode == AGW_RETURN_CODE_SUCCESS))
    {
        printf(AGW_MGR_OBJECT_NAME": processing shape type #%u\n", un16ShapeTypeCount);

        // Extract shape header
        eReturnCode = eAgwLLParserProcessShapeHeader(psParserObj, &sShapeHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process shape header (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        vPrintShapeHeader(&sShapeHeader);
#endif

        // Fill the buffer with file data
        DATASERVICE_MGR_bFillBufferBlock(psObj->psParserObj->sOutputFile.pFile,
                psParserObj->hBuffer);

        // Process following data based on
        switch (sShapeHeader.eType)
        {
            case AGW_INTERNAL_SHAPE_TYPE_STORM_POSITION:
            case AGW_INTERNAL_SHAPE_TYPE_WIND_RADII_FIELD:
            {
                eReturnCode = eProcessStormTrackShapes(psObj,
                                        psDataItem,
                                        &sShapeHeader,
                                        hStorm,
                                        &psStormPos
                                            );
            }
            break;
            case AGW_INTERNAL_SHAPE_TYPE_STORM_DESCRIPTION:
            {
                // Creates the storm object based on description
                eReturnCode = eProcessStormDesc(psObj,
                                psDataItem, &sShapeHeader,
                                &hStorm
                                    );
            }
            break;
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": unsupported by Storm Track shape type detected (%d)",
                        sShapeHeader.eType
                            );
                eReturnCode = AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE;
            }
            break;
        }

        // Decrease number of remaining products
        --un16ShapeTypeCount;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eStormTrackUpdateDSRL
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eStormTrackUpdateDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STORM_TRACK_DATA_ITEM *psNewDataItem,
    AGW_STORM_TRACK_DATA_ITEM **ppsExistingDataItem
    )
{
    BOOLEAN bHasDSRLs;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        ///////////////////////////////////////////////////////////////////////
        // Process Application notifications

        //==================================================================
        // Try to skip processing set if there is no registered DSRLs
        bHasDSRLs = bAgwLLHasDSRLs(psObj);
        if (bHasDSRLs == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            if ((*ppsExistingDataItem) != NULL)
            {
                BOOLEAN bOk;

                //============================================================
                // Looking for shapes in removing state and remove them from
                // involved DSRLs
                bOk = bAgwDSRLUnlinkAllEntries((*ppsExistingDataItem)->hStormPositionsList);
                if (bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to process Removing DSRL Entry Descs"
                                );
                }
            }

            //==================================================================
            // Looking through all shapes and notify about them each DSRL
            // which is interested in it.
            eOsalReturnCode = OSAL.eLinkedListIterate(psNewDataItem->hStormPositionsList,
                            (OSAL_LL_ITERATOR_HANDLER)bAgwDSRLUpdateEntryInDSRLIterator,
                            psObj
                                );
            if ((eOsalReturnCode != OSAL_SUCCESS) &&
                (eOsalReturnCode != OSAL_NO_OBJECTS))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed iterate list 0x%p to update DSRLs (%s)",
                        psNewDataItem->hStormPositionsList,
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
                break;
            }

            //==================================================================
            // Move all DSRLs from UPDATING state to READY
            eUpdateDSRLListState(psObj->hDSRLList, DSRL_STATE_READY);
        }

        // Remove previous set
        vStormTrackDataItemDestroy(*ppsExistingDataItem);
        *ppsExistingDataItem = psNewDataItem;

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eStormTrackProcess
*
******************************************************************************/
static AGW_RETURN_CODE_ENUM eStormTrackProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    AGW_RETURN_CODE_ENUM eReturnCode;
    AGW_PARSER_OBJECT_STRUCT *psParserObj = psObj->psParserObj;
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    AGW_STORM_TRACK_CONTROL_STRUCT *psSTCtrl =
            &psObj->sStormTrackControlStruct;
    AGW_STORM_TRACK_DATA_ITEM **ppsDataItem = &psSTCtrl->psData;
    AGW_STORM_TRACK_DATA_ITEM *psDataItem = NULL;
    UN16 un16ShapesCount = 0;
    TIME_T tTimeStamp;
    UN32 un32LastTimeStamp = 0;
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    do
    {
        vAgwLLParserKeepOutputFile(psParserObj);

        eReturnCode = eAgwLLParserProcessShapeDataBasedProduct(psParserObj,
                            &sHeader
                                );

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to process Storm Track via Parser (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Update year from statistics
        sHeader.un16IssueYear = psStatInfo->un16Year;
        sHeader.un16ValidYear = psStatInfo->un16Year;

        // Calculate timestamp
        tTimeStamp = AGW_MGR_tCreateTimeStamp(&sHeader, FALSE);

        // Is the payload has the same data?
        if (*ppsDataItem != NULL)
        {
            if (tTimeStamp <= (*ppsDataItem)->tTimeStamp)
            {
                printf(AGW_MGR_OBJECT_NAME
                       ": skipped since new data is older (or the same) than existing one\n");
                eReturnCode = AGW_RETURN_CODE_DATA_TIMEOUT;
                break;
            }
        }

        // Create temp collection
        psDataItem = psStormTrackDataItemCreate(psAppObj);
        if (psDataItem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Storm Track Data Item"
                        );
            break;
        }
        psDataItem->tTimeStamp = tTimeStamp;
        psDataItem->sHeader = sHeader;

        eReturnCode = eAgwLLParserGetShapeTypeCount(psParserObj, &un16ShapesCount);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to get shapes count (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = eStormTrackDataProcess(psObj, un16ShapesCount, psDataItem);
        // Check execution
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to parse storm track shapes (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        eReturnCode = eStormTrackUpdateDSRL(psObj, psDataItem, ppsDataItem);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to update DSRLs (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }
        psDataItem = NULL;

        // Store the data to the NVM with current time
        eOsalReturnCode = OSAL.eTimeGet(&un32LastTimeStamp);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to get current time"
                        );
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        eReturnCode = eStoreDataToCache(psObj,
            psStatInfo,
            un32LastTimeStamp,
            AGW_FILE_NAME_STORM_TRACK_CACHED
                );

        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to store data to file",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
        }
    } while (FALSE);

    // Release internal buffer if exists
    eAgwLLParserCleanupBuffer(psObj->psParserObj);

    eAgwLLParserRemoveOutputFile(psObj->psParserObj);

    if (psDataItem != NULL)
    {
        // Something went wrong. Just destroy the data item
        vStormTrackDataItemDestroy(psDataItem);
    }

    return eReturnCode;
}


/*****************************************************************************
*
*   bStormTrackUninit
*
******************************************************************************/
static BOOLEAN bStormTrackUninit (
    AGW_STORM_TRACK_CONTROL_STRUCT *psObj
        )
{
    puts(AGW_MGR_OBJECT_NAME
           ": attempting to uninit the STORM TRACK product control structure");

    if ((psObj != NULL) && (psObj->psData != NULL))
    {
        puts(AGW_MGR_OBJECT_NAME
               ": attempting to uninit the data item");

        vStormTrackDataItemDestroy(psObj->psData);
        psObj->psData = NULL;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vEventHandler
*
*   This function runs in the context of an SMS resource which has been
*   assigned to this service.
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        )
{
    AGW_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bValid, bStopEvent = FALSE;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    // Get our AGW handle from the callback argument
    psObj = (AGW_MGR_OBJECT_STRUCT *)pvEventCallbackArg;

    // Is this object valid?
    bValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psObj);

    // Only handle events for valid objects...
    if (bValid == TRUE)
    {
        switch( tCurrentEvent )
        {
            // Handle AGW Service events here...

            // State has changed
            case DATASERVICE_EVENT_STATE:
            {
                BOOLEAN bStateChanged;
                DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                    (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;

                // Process the state transition
                bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psObj,
                    psStateChange,
                    &GsAGWStateHandlers,
                    (void *)psObj);

                if (bStateChanged == TRUE)
                {
                    // The state has been updated
                    tEventMask |= DATASERVICE_EVENT_STATE;

                    // Is the service stopped now?
                    if (psStateChange->eCurrentState ==
                            DATASERVICE_STATE_STOPPED)
                    {
                        bStopEvent = TRUE;
                    }
                }
            }
            break;

            // This service has a tile to process
            case DATASERVICE_EVENT_NEW_DATA:
            {
                OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
                BOOLEAN bOk = TRUE;

                // Ensure the payload handle is valid
                // If it isn't, there's a problem with
                // SMS
                if (hPayload == OSAL_INVALID_BUFFER_HDL)
                {
                    bOk = FALSE;

                    // Set the error
                    vSetError( psObj,
                               DATASERVICE_ERROR_CODE_GENERAL);
                }
                else
                {
                    BOOLEAN bPayloadOk;
                    printf(AGW_MGR_OBJECT_NAME": Payload Received (%u)\n",
                        OSAL.tBufferGetSize(hPayload)
                            );

                    bPayloadOk = bProcessPayload(psObj, hPayload);

                    if (bPayloadOk == TRUE)
                    {
                        // Update timer
                        vExpirationTimerUpdate(psObj, AGW_TIMER_PERIOD_SEC);

                        puts(AGW_MGR_OBJECT_NAME
                             ": Message Payload Processed Ok\n");
                    }
                    else
                    {
                        DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                            ": Message Payload failed to process\n");
                    }

                }

                if (bOk != TRUE)
                {
                    // If an error occurred, indicate a state change
                    // to the application
                    tEventMask |= DATASERVICE_EVENT_STATE;
                }
            }
            break;

            // We need to do some work with our DSRLs
            case DATASERVICE_INTERNAL_EVENT_DSRL:
            {
                DSRL_ARG_STRUCT *psDSRLArg =
                    (DSRL_ARG_STRUCT *)pvEventArg;
                AGW_APP_OBJECT_STRUCT *psAppObj;
                BOOLEAN bOk = FALSE;

                printf(AGW_MGR_OBJECT_NAME
                    ": internal event for DSRL\n");

                psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);

                if (psAppObj != NULL)
                {
                    switch (psDSRLArg->eAction)
                    {
                        case DSRL_ACTION_ADD:
                        {
                            puts(AGW_MGR_OBJECT_NAME": DSRL List creating");
                            bOk = bDSRLCreateList(psObj, psDSRLArg);
                            printf(AGW_MGR_OBJECT_NAME": DSRL List creation is %s\n",
                                    bOk ? "OK" : "FAILED");
                        }
                        break;
                        case DSRL_ACTION_MODIFY:
                        {
                            puts(AGW_MGR_OBJECT_NAME": DSRL List modifying");
                            bOk = bDSRLModifyList(psObj, psDSRLArg);
                            printf(AGW_MGR_OBJECT_NAME": DSRL List modification is %s\n",
                                    bOk ? "OK" : "FAILED");
                        }
                        break;
                        case DSRL_ACTION_REMOVE:
                        {
                            puts(AGW_MGR_OBJECT_NAME": DSRL List removing");
                            bOk = bDSRLDestroyList(psObj, psDSRLArg);
                            printf(AGW_MGR_OBJECT_NAME": DSRL List removal is %s\n",
                                    bOk ? "OK" : "FAILED");
                        }
                        break;
                        case DSRL_ACTION_REFRESH:
                        {
                            puts(AGW_MGR_OBJECT_NAME": DSRL List refreshing");
                            bOk = bDSRLRefreshList(psObj, psDSRLArg);
                            printf(AGW_MGR_OBJECT_NAME": DSRL List refresh is %s\n",
                                    bOk ? "OK" : "FAILED");
                        }
                        break;
                        case DSRL_ACTION_INVALID:
                        default:
                        {
                        }
                        break;
                    }

                    // Let's start or stop the timer
                    vExpirationTimerUpdate(psObj, AGW_TIMER_PERIOD_SEC);

                    SMSO_vUnlock((SMS_OBJECT)psAppObj);
                }

                // Go to the error state if the create/modify failed
                // or if we aren't ready
                if (bOk == FALSE)
                {
                    BOOLEAN bLocked = FALSE;

                    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                        ": failed to process DSRL event\n");

                    bLocked = SMSO_bLock((SMS_OBJECT)psDSRLArg->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
                    if (bLocked == TRUE)
                    {
                        DSRL_vSetState(psDSRLArg->hDSRL, DSRL_STATE_ERROR);
                        SMSO_vUnlock((SMS_OBJECT)psDSRLArg->hDSRL);
                    }
                }
            }
            break;

            case DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC:
            {
                AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
                AGW_MGR_EVENT_STRUCT const *psEvent =
                                (AGW_MGR_EVENT_STRUCT const *)pvEventArg;

                puts(AGW_MGR_OBJECT_NAME": event "
                    MACRO_TO_STRING(DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC));

                if (psEvent->eEvent == AGW_MGR_EVENT_UPDATE_PRODUCT_FILTER)
                {
                    AGW_APP_OBJECT_STRUCT *psAppObj;
                    DATASERVICE_STATE_ENUM eServiceState;

                    eServiceState = AGW.eState((AGW_SERVICE_OBJECT) psObj);
                    if ((eServiceState == DATASERVICE_STATE_READY) ||
                        (eServiceState == DATASERVICE_STATE_POI_UPDATES_ONLY))
                    {
                        psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);
                        if (psAppObj != NULL)
                        {
                            eReturnCode =
                                eProcessProductFilterUpdate(psObj, psAppObj,
                                psEvent->uEvent.sUpdateProductFilter.eProductType,
                                psEvent->uEvent.sUpdateProductFilter.bEnable);

                            SMSO_vUnlock((SMS_OBJECT)psAppObj);
                        }
                        else
                        {
                            puts(AGW_MGR_OBJECT_NAME
                                ": failed to get App Facing object");
                            eReturnCode = AGW_RETURN_CODE_NOT_OWNER;
                        }
                    }
                    else
                    {
                        eReturnCode = AGW_RETURN_CODE_INVALID_STATE;
                    }
                }

                if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                        ": failed to process service specific event (%d)\n",
                        eReturnCode);
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to process service specific event (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
                }
            }
            break;

            case DATASERVICE_EVENT_TIMEOUT:
            {
                AGW_APP_OBJECT_STRUCT *psAppObj = NULL;
                AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

                printf(AGW_MGR_OBJECT_NAME": event %s\n",
                    MACRO_TO_STRING(DATASERVICE_EVENT_TIMEOUT)
                        );

                // Need to look through DSRL entries to throw out expired from
                // DSRL's
                do
                {
                    psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);
                    if (psAppObj == NULL)
                    {
                        eReturnCode = AGW_RETURN_CODE_NOT_OWNER;
                        break;
                    }

                    // Look through all entries to find out expired and perform
                    // needed actions to notify the application about that.
                    eReturnCode = eProcessExpirationTimeout(psObj);

                    // Restart expiration timeout
                    vExpirationTimerUpdate(psObj, AGW_TIMER_PERIOD_SEC);

                } while (FALSE);

                if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
                        ": failed to process timer (%d)\n", eReturnCode);
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to process timer (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
                }

                if (psAppObj != NULL)
                {
                    SMSO_vUnlock((SMS_OBJECT) psAppObj);
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": unsupported event"
                        );
            }
            break;
        }

        if (bStopEvent == TRUE)
        {
            // Uninitialize the object, but leave
            // enough to be useful to the callback
            vUninitObject( psObj, FALSE );
        }

        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psObj->sEvent, tEventMask);

        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psObj->sEvent);

        if (bStopEvent == TRUE)
        {
            // Filter out all further agw manager updates
            SMSU_tFilter(&psObj->sEvent, DATASERVICE_EVENT_ALL);

            vDestroyObject(psObj);
        }
    }

    return;
}

/*****************************************************************************
*
*   eProcessPayloadHeader
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eProcessPayloadHeader(
    AGW_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    UN32 un32Filter = 0;

    do
    {
        // Initialize output
        OSAL.bMemSet(psStatInfo, 0, sizeof(*psStatInfo));
        OSAL.bMemSet(psHeader, 0, sizeof(*psHeader));

        // Process payload header by WSI Decoder
        eReturnCode = eAgwLLParserParseHeader(
            psObj->psParserObj,
            *phPayload,
            AGW_WSI_PRODUCT_BITOFFSET,
            psHeader
                );
#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
        vPrintHeader(psHeader);
#endif
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            printf(AGW_MGR_OBJECT_NAME
                ": [%d] failed to parse payload to extract product header (%s)\n",
                __LINE__, AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                    );
            break;
        }

        // Apply filter to the received product to verify its necessity
        un32Filter = un32GetMaskForInternalProduct(psHeader->eProductType);
        if ((un32Filter > 0) &&
            ((psObj->psAppObj->un32ProductFilter & un32Filter) != un32Filter))
        {
            printf(AGW_MGR_OBJECT_NAME
                ": product %s has been filtered out\n",
                AGW_MGR_pacGetInternalProductTypeName(psHeader->eProductType)
                    );
            eReturnCode = AGW_RETURN_CODE_FILTERED_OUT;
            break;
        }

        // Validate the AU
        eReturnCode = eAgwLLParserValidateAU(psObj->psParserObj, *phPayload);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": access unit data is invalid (%s)\n",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Assign payload data to PARSER internal and release STI payload
        // buffer in case of success
        eReturnCode = eAgwLLParserAssignPayload(psObj->psParserObj, phPayload);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to assign payload to parser (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Get statistics
        if (psHeader->eProductType == AGW_INTERNAL_PRODUCT_TYPE_NOWRAD)
        {
            TIME_T tShortTimeStamp;
            BOOLEAN bSuccess;

            tShortTimeStamp = AGW_MGR_tCreateTimeStamp(psHeader, TRUE);
            bSuccess = bProductStatisticsGet(&psObj->sStatisticsControlStruct,
                            tShortTimeStamp, psHeader->eProductType,
                            TRUE, psStatInfo
                                );
            if (bSuccess == FALSE)
            {
                puts(AGW_MGR_OBJECT_NAME
                    ": there is no broadcasted statistics for that packet "
                    "so far. Let's invent our owns special for the NOWRad");

                eReturnCode = eNOWRADUpdateStatistics(psObj, psHeader);
                if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME
                        ": failed to update statistics for the NOWRad (%s)",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode));
                    break;
                }
                // Request the statistics one more time
                bSuccess = bProductStatisticsGet(
                                &psObj->sStatisticsControlStruct,
                                tShortTimeStamp, psHeader->eProductType,
                                TRUE, psStatInfo);
                if (bSuccess == FALSE)
                {
                    eReturnCode = AGW_RETURN_CODE_NO_STATISTIC_DATA;
                    break;
                }
            }
        }
        else if (psHeader->eProductType != AGW_INTERNAL_PRODUCT_TYPE_STATISTICS)
        {
            // For any product except NOWRad we're expecting to get
            // any statistics about the current year.
            eReturnCode = eProductStatisticsGetYear(
                                    &psObj->sStatisticsControlStruct,
                                    &psStatInfo->un16Year
                                        );
            if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
            {
                printf(AGW_MGR_OBJECT_NAME
                    ": Year data did not arrive yet. Skipping\n");
                eReturnCode = AGW_RETURN_CODE_NO_STATISTIC_DATA;
                break;
            }
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bProcessPayload
*
*****************************************************************************/
static BOOLEAN bProcessPayload (
    AGW_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_PRODUCT_HEADER_STRUCT sHeader;
    AGW_STATISTICS_INFO_STRUCT sStatInfo;
    AGW_APP_OBJECT_STRUCT *psAppObj = NULL;
    BOOLEAN bLocked = FALSE;

    OSAL.bMemSet((void*)&sHeader, 0, sizeof(sHeader));
    OSAL.bMemSet((void*)&sStatInfo, 0, sizeof(sStatInfo));

    do
    {
        if (hPayload == OSAL_INVALID_BUFFER_HDL)
        {
            break;
        }

        // Lock facing object to restrict access to data objects
        psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT)psObj);
        if (psAppObj == NULL)
        {
            printf(AGW_MGR_OBJECT_NAME": failed to lock facing object");
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psParserObj, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to lock parser object");
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = eProcessPayloadHeader(psObj, &hPayload,
            &sHeader, &sStatInfo
                );

        if ((eReturnCode == AGW_RETURN_CODE_FILTERED_OUT) ||
            (eReturnCode == AGW_RETURN_CODE_NO_STATISTIC_DATA)||
            (eReturnCode == AGW_RETURN_CODE_UNSUPPORTED_PRODUCT))
        {
            // This is not error case, so just skipping
            break;
        }
        else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to process header (%s)\n",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            break;
        }

        // Create buffer
        psObj->psParserObj->hBuffer = OSAL.hBufferAllocate(
                psAppObj->hBlockPool,
                FALSE, // No read blocking
                FALSE, // No write blocking
                OSAL_BUFFER_ALLOCATE_OPTION_NONE );

        if (psObj->psParserObj->hBuffer == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to create file reader buffer"
                        );
            break;
        }

        // Process payload data depends on product type
        DATASERVICE_IMPL_vLog(AGW_MGR_OBJECT_NAME
            ": start processing product %s\n",
            AGW_MGR_pacGetInternalProductTypeName(sHeader.eProductType));

        switch (sHeader.eProductType)
        {
            case AGW_INTERNAL_PRODUCT_TYPE_STATISTICS:
            {
                eReturnCode = eProductStatisticsProcess(psObj);
            }
            break;
            case AGW_INTERNAL_PRODUCT_TYPE_NOWRAD:
            {
                eReturnCode = eNOWRADProcess(psObj, &sStatInfo);
            }
            break;
            case AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES:
            {
                eReturnCode = eStormAttributesProcess(psObj, &sStatInfo);
            }
            break;
            case AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS:
            {
                eReturnCode = eWindsProcess(psObj, &sStatInfo);
            }
            break;
            case AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES:
            {
                eReturnCode = eSurfaceFeatureProcess(psObj, &sStatInfo);
            }
            break;
            case AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK:
            {
                eReturnCode = eStormTrackProcess(psObj, &sStatInfo);
            }
            break;
            default:
            {
                eReturnCode = AGW_RETURN_CODE_UNSUPPORTED_PRODUCT;
            }
            break;
        }
    } while (FALSE);

    if (psObj->psParserObj != NULL)
    {
        if (psObj->psParserObj->sOutputFile.pFile != NULL)
        {
            fclose(psObj->psParserObj->sOutputFile.pFile);
            psObj->psParserObj->sOutputFile.pFile = NULL;
        }
        if (psObj->psParserObj->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL.eBufferFree(psObj->psParserObj->hBuffer);
            psObj->psParserObj->hBuffer = OSAL_INVALID_BUFFER_HDL;
        }
    }

#if ((DEBUG_OBJECT == 1) || (SMS_DEBUG == 1))
    if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
    {
        printf(AGW_MGR_OBJECT_NAME
               ": failed to process %s [%d] (%s) data (%s)\n",
               AGW_MGR_pacGetInternalProductTypeName(sHeader.eProductType),
               sHeader.eProductType,
               pacAgwLLGetProductDataTypeName(sHeader.eDataType),
               AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                      );
    }
#endif

    if (hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    }

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT)psObj->psParserObj);
    }

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }

    return ((eReturnCode == AGW_RETURN_CODE_SUCCESS)             ||
            (eReturnCode == AGW_RETURN_CODE_FILTERED_OUT)        ||
            (eReturnCode == AGW_RETURN_CODE_NO_STATISTIC_DATA)   ||
            (eReturnCode == AGW_RETURN_CODE_DATA_TIMEOUT)        ||
            (eReturnCode == AGW_RETURN_CODE_REDUNDANT_TILE)      ||
            (eReturnCode == AGW_RETURN_CODE_DUPLICATED_TILE)     ||
            (eReturnCode == AGW_RETURN_CODE_UNSUPPORTED_PRODUCT))
        ? TRUE : FALSE;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    AGW_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        )
{
    AGW_APP_OBJECT_STRUCT *psAppObj = psObj->psAppObj;
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)psAppObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        vRemoveDSRLDescrs(psObj);

        bProductHandlersUninit(psObj);

        vDestroyDSRLEntryDescrPool(psObj);

        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                ": failed to lock Facing object during Un-initialization");
    }

    // Remove any broadcast-specific parsing data
    if (psObj->psParserObj != NULL)
    {
        bLocked = SMSO_bLock((SMS_OBJECT)(psObj->psParserObj), OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            // Make the call to uninit the broadcast-specific parser?
            vAgwLLParserUninitParser(psObj->psParserObj);
            psObj->psParserObj = NULL;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                    ": failed to lock Parsing object during Un-initialization"
                    );
        }
    }

    // Uninit the app object
    vUninitAppFacingObject(psObj);

    if (psObj->hDSRLList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL.eLinkedListDelete(psObj->hDSRLList);
        psObj->hDSRLList = OSAL_INVALID_OBJECT_HDL;
    }

    if (bFullDelete == TRUE)
    {
        vDestroyObject(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    // Destroy the SMS Update Event object
    SMSU_vDestroy(&psObj->sEvent);

    return;
}


/*****************************************************************************
*
*   eLoadFromFile
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eLoadFromFile(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    const char *pacFilePath,
    UN32 un32Timeout,
    OSAL_BUFFER_HDL *phFileData,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        )
{
    FILE *psFile = NULL;
    size_t tElemRead = 0, tBytesWritten = 0;
    AGW_CACHE_HEADER_STRUCT sCacheHeader;
    UN32 un32UTCsec = 0;
    N32 n32Result = 0;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = TRUE;

    do
    {
        OSAL.bMemSet(&sCacheHeader, 0, sizeof(sCacheHeader));

        psFile = fopen(pacFilePath, "rb");
        if (psFile == NULL)
        {
            // File does not exist, skipping
            eReturnCode = AGW_RETURN_CODE_NOT_EXIST;
            break;
        }

        tElemRead = fread(&sCacheHeader, sizeof(sCacheHeader),
            1, psFile
                );
        if (tElemRead != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to read cache header"
                        );
            break;
        }

#if ((SMS_DEBUG == 1) && (DEBUG_OBJECT == 1))
        vPrintCacheHeader(&sCacheHeader);
#endif

        // checking time stamp, has to be < 24 hours old
        eOsalReturnCode = OSAL.eTimeGet(&un32UTCsec);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to get current time"
                        );
            printf("Result: %s\n",
                   OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        if (sCacheHeader.un32TimeStamp <
            (un32UTCsec - un32Timeout))
        {
            //too old file, need to close it and delete
            fclose(psFile);
            psFile = NULL;
            n32Result = remove(pacFilePath);
            if (n32Result != 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to remove old "
                    "cache file %s, %d",
                    pacFilePath, n32Result
                        );
                break;
            }
            eReturnCode = AGW_RETURN_CODE_DATA_TIMEOUT;
            break;
        }

        *psStatInfo = sCacheHeader.sStatInfo;

        *phFileData = OSAL.hBufferAllocate(psAppObj->hBlockPool,
            FALSE, FALSE, OSAL_BUFFER_ALLOCATE_OPTION_NONE
                );

        do
        {
            tElemRead = fread(psAppObj->acBuffer, sizeof(char),
                AGW_SHARED_BUFFER_LEN, psFile
                    );

            if (tElemRead > 0)
            {
                tBytesWritten = OSAL.tBufferWriteTail(*phFileData,
                    psAppObj->acBuffer, tElemRead
                        );
                if (tBytesWritten != tElemRead)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            AGW_MGR_OBJECT_NAME
                            ": failed to write full read from file buffer "
                            "to OSAL buffer"
                                );
                    bSuccess = FALSE;
                    break;
                }
            }
        } while (tElemRead > 0);

        if (bSuccess != TRUE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (psFile != NULL)
    {
        fclose(psFile);
    }

    if ((eReturnCode != AGW_RETURN_CODE_SUCCESS) &&
        (*phFileData != OSAL_INVALID_BUFFER_HDL))
    {
        OSAL.eBufferFree(*phFileData);
        *phFileData = OSAL_INVALID_BUFFER_HDL;
    }

    return eReturnCode;
}


/*****************************************************************************
*
*   eGenerateCacheFilePath
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eGenerateCacheFilePath(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    const char *pacFileName,
    char **ppacFilePath
        )
{
    size_t tPathLen = 0;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        if ((pacFileName == NULL) || (ppacFilePath == NULL))
        {
            break;
        }

        if (psAppObj->pacAGWServiceFilePath == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_INVALID_STATE;
            break;
        }

        tPathLen = strlen(psAppObj->pacAGWServiceFilePath) +
            strlen(pacFileName) + 2;

        *ppacFilePath =
            (char *)SMSO_hCreate(
            AGW_MGR_OBJECT_NAME":FilePath",
            tPathLen + 1,
            (SMS_OBJECT)psAppObj,
            FALSE
                );
        if (*ppacFilePath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to allocate "
                        "memory for file path"
                        );
            eReturnCode = AGW_RETURN_CODE_NO_MEMORY;
            break;
        }

        snprintf(*ppacFilePath, tPathLen, "%s/%s",
            psAppObj->pacAGWServiceFilePath,
            pacFileName
                );

        printf(AGW_MGR_OBJECT_NAME": File path generated %s\n", *ppacFilePath);
        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}


/*****************************************************************************
*
*   eReadStormTrackData
*
*****************************************************************************/
static AGW_RETURN_CODE_ENUM eReadStormTrackData(
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    char *pacStormTrackFilePath = NULL;
    BOOLEAN bLocked = FALSE;
    OSAL_BUFFER_HDL hPayload = OSAL_INVALID_BUFFER_HDL;
    AGW_STORM_TRACK_DATA_ITEM *psDataItem = NULL;

    do
    {
        AGW_PRODUCT_HEADER_STRUCT sHeader;
        AGW_STATISTICS_INFO_STRUCT sStatInfo;
        AGW_STORM_TRACK_DATA_ITEM **ppsDataItem =
            &(psObj->sStormTrackControlStruct.psData);
        TIME_T tTimeStamp = 0;
        UN16 un16ShapeCount = 0;

        // Initialize year field in sStatInfo (warning elimination)
        sStatInfo.un16Year = 0;

        // if some data already loaded, skipping processing
        if (*ppsDataItem != NULL)
        {
            printf(AGW_MGR_OBJECT_NAME": storm track data already exist, "
                "skipping cache reading.");
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
            break;
        }

        eReturnCode = eGenerateCacheFilePath(psObj->psAppObj,
            AGW_FILE_NAME_STORM_TRACK_CACHED,
            &pacStormTrackFilePath
                );
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            if (eReturnCode != AGW_RETURN_CODE_INVALID_STATE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to generate file path"
                    );
            }
            break;
        }

        eReturnCode = eLoadFromFile(psObj->psAppObj,
            pacStormTrackFilePath,
            AGW_STORM_TRACK_CACHE_TIMEOUT,
            &hPayload,
            &sStatInfo
                );
        if (eReturnCode == AGW_RETURN_CODE_DATA_TIMEOUT)
        {
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
            printf(AGW_MGR_OBJECT_NAME": Cache data is too old\n");
            break;
        }
        else if (eReturnCode == AGW_RETURN_CODE_NOT_EXIST)
        {
            eReturnCode = AGW_RETURN_CODE_SUCCESS;
            printf(AGW_MGR_OBJECT_NAME": Cache file not exist\n");
            break;
        }
        else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to load data "
                        "from cache file"
                        );
            break;
        }

        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psParserObj,
            OSAL_OBJ_TIMEOUT_INFINITE
                );

        if (bLocked == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_ERROR;
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to lock parser object"
                        );
            break;
        }

        // Update internal buffer reference
        // Note: We can do it here directly since there is an assumption that
        // under lock of the parser this buffer is empty
        if (psObj->psParserObj->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": buffer already used"
                        );
            break;
        }
        psObj->psParserObj->hBuffer = hPayload;
        hPayload = OSAL_INVALID_BUFFER_HDL;

        // Processes product header
        eReturnCode = eAgwLLParserProcessProductHeader(psObj->psParserObj, &sHeader);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to process product header (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        sHeader.un16IssueYear = sStatInfo.un16Year;
        sHeader.un16ValidYear = sStatInfo.un16Year;
        // Calculate timestamp
        tTimeStamp = AGW_MGR_tCreateTimeStamp(&sHeader, FALSE);
        // Create temp collection
        psDataItem = psStormTrackDataItemCreate(psObj->psAppObj);
        if (psDataItem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to create Storm Track Data Item"
                        );
            break;
        }
        psDataItem->tTimeStamp = tTimeStamp;
        psDataItem->sHeader = sHeader;

        eReturnCode = eAgwLLParserGetShapeTypeCount(psObj->psParserObj, &un16ShapeCount);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, AGW_MGR_OBJECT_NAME
                   ": failed to read shape count (%s)",
                   AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                          );
            break;
        }

        printf(AGW_MGR_OBJECT_NAME": read %d shapes\n", un16ShapeCount);

        eReturnCode = eStormTrackDataProcess(psObj, un16ShapeCount, psDataItem);
        // Check execution
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }

        eReturnCode = eStormTrackUpdateDSRL(psObj, psDataItem, ppsDataItem);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            break;
        }
        psDataItem = NULL;

    } while (FALSE);

    if (pacStormTrackFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)pacStormTrackFilePath);
        pacStormTrackFilePath = NULL;
    }

    if (bLocked == TRUE)
    {
        if ((hPayload == OSAL_INVALID_BUFFER_HDL) &&
            (psObj->psParserObj->hBuffer != OSAL_INVALID_BUFFER_HDL))
        {
            OSAL.eBufferFree(psObj->psParserObj->hBuffer);
            psObj->psParserObj->hBuffer = OSAL_INVALID_BUFFER_HDL;
        }

        // Release internal buffer if exists
        eAgwLLParserCleanupBuffer(psObj->psParserObj);

        SMSO_vUnlock((SMS_OBJECT)psObj->psParserObj);
    }

    if (hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        OSAL.eBufferFree(hPayload);
    }

    if (psDataItem != NULL)
    {
        // Something went wrong. Just destroy the data item
        vStormTrackDataItemDestroy(psDataItem);
    }

    return eReturnCode;
}


/*****************************************************************************
*
*   bReadCachedData
*
*****************************************************************************/
static BOOLEAN bReadCachedData(
    AGW_MGR_OBJECT_STRUCT *psObj,
    UN32 un32ProductsToRead
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
        UN32 un32Filter = 0;

        un32Filter = un32GetMaskForProduct(AGW_PRODUCT_TYPE_STORM_TRACK);
        if ((un32ProductsToRead & un32Filter) == un32Filter)
        {
            eReturnCode = eReadStormTrackData(psObj);
            if ((eReturnCode != AGW_RETURN_CODE_SUCCESS) &&
                (eReturnCode != AGW_RETURN_CODE_INVALID_STATE))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        AGW_MGR_OBJECT_NAME": failed to read storm track data %s",
                        AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                            );
                bResult = FALSE;
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}


/*****************************************************************************
*
*   bBuildServiceFilePath
*
*****************************************************************************/
static BOOLEAN bBuildServiceFilePath(
    AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    size_t tSMSPathLen;
    const char *pacSMSPath;
    size_t tPathLen = 0;
    BOOLEAN bDirExists;
    UN8 un8FileAttributes = 0;
    BOOLEAN bOk = FALSE;

    do
    {
        pacSMSPath = SMS_pacGetPath();
        if (pacSMSPath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to get SMS Path"
                        );
            break;
        }

        tSMSPathLen = strlen(pacSMSPath);

        tPathLen = tSMSPathLen +
                     strlen(AGW_SERVICE_FOLDER) + 2;

        // Now we know the size of the path, so we can now
        // allocate the proper amount of memory
        psAppObj->pacAGWServiceFilePath =
            (char *)SMSO_hCreate(
            AGW_MGR_OBJECT_NAME":ServiceFilePath",
            tPathLen + 1,
            (SMS_OBJECT)psAppObj,
            FALSE
                );

        // Ensure allocation succeeded
        if (psAppObj->pacAGWServiceFilePath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to allocate "
                        "service path memory"
                        );
            break;
        }

        // Construct the full location database filename
        snprintf(psAppObj->pacAGWServiceFilePath,
            tPathLen,
            "%s/%s",
            pacSMSPath,
            AGW_SERVICE_FOLDER
                );

        // Need to create path if it doesn't exist
        // Check to see if our directories exist
        bDirExists = OSAL.bFileSystemGetFileAttributes(
                         psAppObj->pacAGWServiceFilePath, &un8FileAttributes);
        if (bDirExists == FALSE)
        {
            bDirExists = OSAL.bFileSystemMakeDir(psAppObj->pacAGWServiceFilePath);

            if (bDirExists == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME
                    ": failed to make AGW root directory: %s",
                    psAppObj->pacAGWServiceFilePath);
                break;
            }
        }

        printf(AGW_MGR_OBJECT_NAME": service path created: %s\n",
               psAppObj->pacAGWServiceFilePath );

        bOk = TRUE;

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bCheckRasterPath
*
*****************************************************************************/
static BOOLEAN bCheckRasterPath(
    AGW_APP_OBJECT_STRUCT *psAppObj
        )
{
    BOOLEAN bOk = FALSE;
    UN8 un8FileAttrs = 0;

    do
    {
        // Check input
        if (psAppObj == NULL)
        {
            break;
        }

        // Check path existence
        if (psAppObj->pacRasterFolder == NULL)
        {
            // File was not set during initialization
            break;
        }

        // Get attributes
        bOk = OSAL.bFileSystemGetFileAttributes(psAppObj->pacRasterFolder, &un8FileAttrs);
        if (bOk == FALSE)
        {
            // Get folder attributes
            break;
        }

        // Check attributes
        if ((un8FileAttrs & AGW_RASTERS_FOLDER_ATTRS) != AGW_RASTERS_FOLDER_ATTRS)
        {
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bHandleServiceReady
*
*   This function is called when SMS is ready for the service to startup.
*   At this time, the service has a context in which to operate (SMS
*   assigned a resource to us). When this call is made, this service will
*   perform all of its time-intensive startup procedures.
*
*****************************************************************************/
static BOOLEAN bHandleServiceReady (
    AGW_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk = FALSE;
    AGW_APP_OBJECT_STRUCT *psAppObj = NULL;
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;

    do
    {
        /////////////////////////////////////////
        // Validate the hAgwService, then get a pointer to the
        // locked app-facing object
        psAppObj = psGetAppFacingObject((AGW_SERVICE_OBJECT) psObj);
        if (psAppObj == NULL)
        {
            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        /////////////////////////////////////////
        // Init Timer
        eReturnCode = eExpirationTimerInit(psObj);
        if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": failed to init Expiration Timer (%s)",
                    AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                        );
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL );
            break;
        }

        /////////////////////////////////////////
        // Init raster storage
        bOk = bCheckRasterPath(psAppObj);
        if (bOk == FALSE)
        {
            // Error, not sure why this would happen
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_MGR_OBJECT_NAME": unable to use path %s as rasters storage",
                psAppObj->pacRasterFolder
                    );
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        bOk = bBuildServiceFilePath(psAppObj);
        if (bOk == FALSE)
        {
            // Error, not sure why this would happen
            vSetError( psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

        /////////////////////////////////////////
        // Update cached data from the file if needed
        bOk = bReadCachedData(psObj, psAppObj->un32ProductFilter);
        if (bOk == FALSE)
        {
            // Error, not sure why this would happen
            vSetError(psObj, DATASERVICE_ERROR_CODE_GENERAL);
            break;
        }

    } while(FALSE);

    if (psAppObj != NULL)
    {
        SMSO_vUnlock((SMS_OBJECT)psAppObj);
    }
    return bOk;
}

/*****************************************************************************
*
*   un32GetMaskForInternalProduct
*
*****************************************************************************/
static UN32 un32GetMaskForInternalProduct(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType
        )
{
    UN32 un32Result = 0;

    switch (eProductType)
    {
        case AGW_INTERNAL_PRODUCT_TYPE_NOWRAD:
        {
            un32Result = AGW_PRODUCT_FILTER_NOWARD;
        }
        break;
        case AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES:
        {
            un32Result = AGW_PRODUCT_FILTER_STORM_ATTRIBUTES;
        }
        break;
        case AGW_INTERNAL_PRODUCT_TYPE_STATISTICS:
        {
            un32Result = AGW_PRODUCT_FILTER_STATISTICS;
        }
        break;
        case AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES:
        {
            un32Result = AGW_PRODUCT_FILTER_SURFACE_FEATURES;
        }
        break;
        case AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK:
        {
            un32Result = AGW_PRODUCT_FILTER_STORM_TRACK;
        }
        break;
        case AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS:
        {
            un32Result = AGW_PRODUCT_FILTER_WIND_DIRECTION | AGW_PRODUCT_FILTER_WIND_MAGNITUDE;
        }
        break;
        default:
        {
            printf(AGW_MGR_OBJECT_NAME
                    ": [%d] unknown product type %d\n", __LINE__, eProductType
                        );
        }
        break;
    }

    return un32Result;
}

/*****************************************************************************
*
*   un32GetMaskForProduct
*
*****************************************************************************/
static UN32 un32GetMaskForProduct(
    AGW_PRODUCT_TYPE_ENUM eProductType
        )
{
    UN32 un32Result = 0;

    switch (eProductType)
    {
        case AGW_PRODUCT_TYPE_NOWRAD:
        {
            un32Result = AGW_PRODUCT_FILTER_NOWARD;
        }
        break;
        case AGW_PRODUCT_TYPE_STORM_ATTRIBUTES:
        {
            un32Result = AGW_PRODUCT_FILTER_STORM_ATTRIBUTES;
        }
        break;
        case AGW_PRODUCT_TYPE_SURFACE_FEATURES:
        {
            un32Result = AGW_PRODUCT_FILTER_SURFACE_FEATURES;
        }
        break;
        case AGW_PRODUCT_TYPE_STORM_TRACK:
        {
            un32Result = AGW_PRODUCT_FILTER_STORM_TRACK;
        }
        break;
        case AGW_PRODUCT_TYPE_WIND_MAGNITUDE:
        {
            un32Result = AGW_PRODUCT_FILTER_WIND_MAGNITUDE;
        }
        break;
        case AGW_PRODUCT_TYPE_WIND_DIRECTION:
        {
            un32Result = AGW_PRODUCT_FILTER_WIND_DIRECTION;
        }
        break;
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_MGR_OBJECT_NAME": unknown product type %d",
                    eProductType
                        );
        }
        break;
    }

    return un32Result;
}

/*****************************************************************************
*
*   vSetError
*
*****************************************************************************/
static void vSetError (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // Tell the DSM about it
    DATASERVICE_IMPL_vError((DATASERVICE_IMPL_HDL)psObj, eErrorCode);

    return;
}

#ifdef SUPPORT_CUNIT
#include <agw_mgr_obj.cunit>
#endif
