/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_ISOBAR implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "agw_shape_obj.h"
#include "agw_isobar_obj.h"
#include "_agw_isobar_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   ePressure
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePressure (
    AGW_ISOBAR_OBJECT hAgwIsobar,
    OSAL_FIXED_OBJECT hPressure
        )
{
    SMSAPI_RETURN_CODE_ENUM eApiReturnCode;
    AGW_RETURN_CODE_ENUM eReturnCode;
    AGW_ISOBAR_OBJECT_STRUCT *psObj = NULL;

    eReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwIsobar,
                        AGW_SHAPE_TYPE_ISOBAR, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sPressure, hPressure, TRUE);
    }

    eApiReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eReturnCode);

    return eApiReturnCode;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_ISOBAR_OBJECT hAgwIsobar,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    AGW_ISOBAR_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwIsobar,
                        AGW_SHAPE_TYPE_ISOBAR, (void**)&psObj);

    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    n32Return += fprintf(psFile,
                        "SHAPE TYPE: %s\nPRESSURE: ",
                        AGW_ISOBAR_OBJECT_NAME
                            );
    n32Return += OSAL_FIXED.n32FPrintf(psObj->sPressure.hValue,
            psFile, FALSE
                );

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_ISOBAR_hCreate
 *
 *****************************************************************************/
AGW_ISOBAR_OBJECT AGW_ISOBAR_hCreate(
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_ISOBAR_OBJECT hResult = AGW_ISOBAR_INVALID_OBJECT;
    AGW_ISOBAR_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        hResult = (AGW_ISOBAR_OBJECT)
            AGW_SHAPE_hCreate(
                (void**) &psObj,
                AGW_ISOBAR_OBJECT_NAME":Obj",
                psHeader,
                AGW_SHAPE_TYPE_ISOBAR,
                sizeof(AGW_ISOBAR_OBJECT_STRUCT),
                tDescriptorSize,
                hParent);

        if (AGW_ISOBAR_INVALID_OBJECT == hResult)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_ISOBAR_OBJECT_NAME": failed to create"
                        );
            break;
        }
        bOk = TRUE;
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != NULL))
    {
        AGW_ISOBAR_vDestroy(hResult);
        hResult = AGW_ISOBAR_INVALID_OBJECT;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_ISOBAR_bUpdate
 *
 *   Update object data and prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN AGW_ISOBAR_bUpdate(
    AGW_ISOBAR_OBJECT hAgwIsobar,
    UN16 un16Pressure
        )
{
    BOOLEAN bOk = FALSE;
    AGW_ISOBAR_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwIsobar,
                        AGW_SHAPE_TYPE_ISOBAR, (void**)&psObj);
    if (psObj != NULL)
    {
        psObj->sPressure.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Pressure, 0,
                psObj->sPressure.aValueData
                    );
        bOk = TRUE;
    }

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_ISOBAR_vDestroy
 *
 *****************************************************************************/
void AGW_ISOBAR_vDestroy(
   AGW_ISOBAR_OBJECT hAgwIsobar
        )
{
    printf(AGW_ISOBAR_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwIsobar);

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT) hAgwIsobar);

    return;
}

/*****************************************************************************
 *                              PRIVATE FUNCTIONS
 *****************************************************************************/
