/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_FRONT implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "agw_shape_obj.h"
#include "agw_front_obj.h"
#include "_agw_front_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static AGW_FRONT_TYPE_ENUM eType (
    AGW_FRONT_OBJECT hAgwFront
        )
{
    AGW_FRONT_TYPE_ENUM eResult = AGW_FRONT_TYPE_UNKNOWN;
    AGW_FRONT_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwFront,
                        AGW_SHAPE_TYPE_FRONT, (void**)&psObj);
    if (psObj != NULL)
    {
        eResult = psObj->eType;
    }

    return eResult;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_FRONT_OBJECT hAgwFront,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    AGW_FRONT_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwFront,
        AGW_SHAPE_TYPE_FRONT, (void**)&psObj);

    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    switch (eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return += fprintf(psFile,
                        AGW_FRONT_OBJECT_NAME" | %s | %2d\n",
                        AGW_MGR_pacGetFrontTypeName(psObj->eType),
                        psObj->eType
                            );
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            n32Return += fprintf(psFile,
                                 "SHAPE TYPE: %s\nFRONT TYPE: %s\n",
                                 AGW_FRONT_OBJECT_NAME,
                                 AGW_MGR_pacGetFrontTypeName(psObj->eType)
                                    );
        }
        break;
    }

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_FRONT_hCreate
 *
 *****************************************************************************/
AGW_FRONT_OBJECT AGW_FRONT_hCreate(
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_FRONT_OBJECT hResult = AGW_FRONT_INVALID_OBJECT;
    AGW_FRONT_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        hResult = (AGW_FRONT_OBJECT)
            AGW_SHAPE_hCreate(
                (void**)&psObj,
                AGW_FRONT_OBJECT_NAME": Obj",
                psHeader,
                AGW_SHAPE_TYPE_FRONT,
                sizeof(AGW_FRONT_OBJECT_STRUCT),
                tDescriptorSize,
                hParent);

        if (AGW_FRONT_INVALID_OBJECT == hResult)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_FRONT_OBJECT_NAME": failed to create\n"
                        );
            break;
        }

        psObj->eType = AGW_FRONT_TYPE_UNKNOWN;

    } while (FALSE);

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_FRONT_bUpdate
 *
 *   Update object data and prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN AGW_FRONT_bUpdate(
    AGW_FRONT_OBJECT hAgwFront,
    AGW_INTERNAL_FRONT_TYPE_ENUM eFrontType
        )
{
    BOOLEAN bOk = FALSE;
    AGW_FRONT_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // Gets object data
        AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwFront,
                            AGW_SHAPE_TYPE_FRONT, (void**)&psObj);
        if (psObj == NULL)
        {
            break;
        }

        // Get public type based on internal
        psObj->eType = AGW_MGR_eGetPublicFrontType(eFrontType);
        if (psObj->eType == AGW_FRONT_TYPE_UNKNOWN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_FRONT_OBJECT_NAME": incorrect front type (%d)",
                    eFrontType
                        );
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_FRONT_vDestroy
 *
 *****************************************************************************/
void AGW_FRONT_vDestroy(
    AGW_FRONT_OBJECT hAgwFront
        )
{
    printf(AGW_FRONT_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwFront);

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT)hAgwFront);

    return;
}
