/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _WEATHER_MGR_OBJ_H_
#define _WEATHER_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"
#include "sms_api.h"
#include "dataservice_mgr_impl.h"
#include "weather_db_constants.h"
#include "sms_update.h"
#include "sql_interface_obj.h"
#include "weather_interface.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define WEATHER_MGR_OBJECT_NAME "TabWeatherService"

// Constant which defines the size of the buffer used by
// weather for many purposes.
#define WEATHER_SHARED_BUFFER_LEN ( 300 )

// Constant defines the minimum device location
// delta which weather wants to know about
#define WEATHER_DEVICE_DISTANCE_THRESHOLD (10)

#define WEATHER_SUPPORTED_OPTIONS (             \
    (DATASERVICE_OPTIONS_MASK)                  \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      | \
    DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES   \
    )

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Service specific event data structure
typedef struct weather_mgr_event_struct 
{
    BOOLEAN bEnableSkiReports;
} WEATHER_MGR_EVENT_STRUCT;

// Sanity check; please put this in each service implementation
// that uses service-private events.  If this fails,
// you MUST REDUCE the size of the structure!
DSM_IMPL_EVENT_COMPILE_TIME_ASSERT(WEATHER_MGR_EVENT_STRUCT);

typedef struct weather_app_object_struct
{
    // DB Connection Handle, File Paths, and Shared Buffer
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    char acBuffer[WEATHER_SHARED_BUFFER_LEN];

    // A list of active targets (a list of DSRLs for the application)
    // Each entry contains a WEATHER_DSRL_DESC_STRUCT
    OSAL_OBJECT_HDL hDSRLList;

    // A list of weather entries which contain information about weather
    OSAL_OBJECT_HDL hEntriesList;

    // A list of weather entries candidates for removal
    OSAL_OBJECT_HDL hRemovedEntriesList;

    // A list of newly created weather forecast entries
    // After reading the data from cache to entry
    // entry will be removed from this list
    OSAL_OBJECT_HDL hNewForecastEntriesList;

    // A list of newly created ski entries
    // After reading the data from cache to entry
    // entry will be removed from this list
    OSAL_OBJECT_HDL hNewSkiEntriesList;

    // A list of new WEATHER DSRL entries
    OSAL_OBJECT_HDL hNewDSRLEntriesList;

    // A list of entries to add to DSRL
    OSAL_OBJECT_HDL hEntriesToAddList;

    // A list of entries to remove from DSRL
    OSAL_OBJECT_HDL hEntriesToRemoveList;

    // Dummy location object used for dummy msg
    LOCATION_OBJECT hDummyLocation;

    // Dummy msg object used for search
    WEATHER_MSG_OBJECT hDummyMsg;

} WEATHER_APP_OBJECT_STRUCT;

typedef struct weather_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    // like the market filter and the data that feeds the TRAFFIC_MSG
    WEATHER_APP_OBJECT_STRUCT *psAppObj;

    char *pacRefDatabaseDirPath;
    const char *pacCurRefDatabaseFilePath;
    const char *pacNextRefDatabaseFilePath;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Flag indicating if this service will be
    // performing reference database updates
    BOOLEAN bRefDBUpdatesEnabled;

    // For periodic check of old messages
    DATASERVICE_TIMED_EVENT_HDL hCancelEvent;

    //Weather protocol plugin handle
    WEATHER_INTERFACE_OBJECT hInterface;

} WEATHER_MGR_OBJECT_STRUCT;

// Structure used to perform weather database queries
typedef struct weather_db_query_result_struct
{
    BOOLEAN bResultantRows;
    WEATHER_LOCATION_DB_ROW_UNION uDbRow;
} WEATHER_DB_QUERY_RESULT_STRUCT;

//Structure used to filter database entries
typedef struct weather_filter_locations_list_struct
{
    STATE_ID tStateId;
    STRING_OBJECT hFilterString;
    BOOLEAN bWeatherSkiFlag;
    WEATHER_LOCATION_ITERATOR bIterator;
    void *pvArg;
    LOCATION_OBJECT hLocation;
    // this string is owned by location object
    STRING_OBJECT hAreaName;
    // this string is owned by location object
    STRING_OBJECT hStateName;
    // this is state id for hStateName string
    STATE_ID tCurrentStateId;
} WEATHER_FILTER_LOCATIONS_LIST_STRUCT;


  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

//DSRL related structures

// DSRL descriptor structure
typedef struct weather_dsrl_desc_struct
{
    OSAL_OBJECT_HDL hTargetList;
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
    OSAL_OBJECT_HDL hEntriesList;
} WEATHER_DSRL_DESC_STRUCT;

// DSRL entry descriptor structure
typedef struct weather_dsrl_entry_desc_struct {
    // list of DSRL handles to which the entry belongs
    OSAL_OBJECT_HDL hDSRLList;

    // entry handle in the common pool of entries
    OSAL_LINKED_LIST_ENTRY hPoolEntry;
} WEATHER_DSRL_ENTRY_DESC_STRUCT;

// DSRL change entry structure
typedef struct weather_dsrl_change_entry_struct {
    DSRL_ENTRY_OBJECT hDSRLEntry;
    WEATHER_APP_OBJECT_STRUCT *psAppObj;
} WEATHER_DSRL_CHANGE_ENTRY_STRUCT;

// This struct is used as iterator argument to generate entries diff list
typedef struct weather_diff_list_iterator_struct
{
    // list of DSRL entries used for comparison
    OSAL_OBJECT_HDL hSubList;

    // list of DSRL entries not found in hSubList
    OSAL_OBJECT_HDL hDiffList;

    // last found entry. Used to avoid the search from the start of list.
    // Warning: it is assumed that both lists (hSubList and iterated list)
    // are sorted by the same criteria
    OSAL_LINKED_LIST_ENTRY hEntry;
} WEATHER_DIFF_LIST_ITERATOR_STRUCT;

// This struct is used as iterator argument to remove entries
typedef struct weather_dsrl_entries_list_iterator_struct
{
    DSRL_OBJECT hDSRL;
    WEATHER_APP_OBJECT_STRUCT *psAppObj;
} WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT;

// This struct is used as iterator argument to remove
// old items from DSRL entries
typedef struct weather_remove_old_items_iterator_struct
{
    UN32 un32CurrentTime;
    WEATHER_APP_OBJECT_STRUCT *psAppObj;
} WEATHER_REMOVE_OLD_ITEMS_ITERATOR_STRUCT;

typedef struct weather_sql_query_location_struct {
    LOCATION_OBJECT hTargetLocation;
    DISTANCE_OBJECT hMinimumDistance;
    LOCATION_OBJECT hMatchedLocation;
    LOCATION_OBJECT hLocationUnderCheck;
    WEATHER_LOCATION_ROW_STRUCT sLocationRow;
} WEATHER_SQL_QUERY_LOCATION_STRUCT;

typedef struct weather_location_struct {
    LOCATION_OBJECT hLocation;
    STRING_OBJECT hICAO;
} WEATHER_LOCATION_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Public Prototypes */

static WEATHER_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

static SMSAPI_RETURN_CODE_ENUM eEnableSkiReports(
    WEATHER_SERVICE_OBJECT hWeatherService
        );

static SMSAPI_RETURN_CODE_ENUM eDisableSkiReports(
    WEATHER_SERVICE_OBJECT hWeatherService
        );

static SMSAPI_RETURN_CODE_ENUM eIterateLocations(
    WEATHER_SERVICE_OBJECT hWeatherService,
    BOOLEAN bWeatherSkiFlag,
    STRING_OBJECT hSortName,
    STRING_OBJECT hState,
    WEATHER_LOCATION_ITERATOR bIterator,
    void *pvArg
        );

static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

/* Weather manager internal interface function prototypes */

static BOOLEAN bRefDBBank (
    WEATHER_SERVICE_OBJECT hWeatherService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        );

static BOOLEAN bDBUpdateBegin (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        );

static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    UN8 un8DBVer
        );

static BOOLEAN bProcessForecastReport (
    WEATHER_SERVICE_OBJECT hWeatherService,
    LOC_ID tID,
    WEATHER_FORECAST_TYPE_ENUM eType,
    FORECAST_DATA_STRUCT *psForecastData
        );

static BOOLEAN bStartProcessing(
    WEATHER_SERVICE_OBJECT hWeatherService
        );

static BOOLEAN bProcessSkiReport (
    WEATHER_SERVICE_OBJECT hWeatherService,
    LOC_ID tID,
    SKI_DATA_STRUCT *psData
        );

static BOOLEAN bEndProcessing (
    WEATHER_SERVICE_OBJECT hWeatherService
        );

static BOOLEAN bAddLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static BOOLEAN bModifyLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static BOOLEAN bDeleteLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static BOOLEAN bForecastHashRemoved (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        );

static BOOLEAN bForecastHashRepeated (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        );

static BOOLEAN bSkiHashRemoved (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        );

static BOOLEAN bSkiHashRepeated (
    WEATHER_SERVICE_OBJECT hWeatherService,
    WEATHER_HASH_OBJECT hHashEntry
        );

/* Private Function Prototypes */

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static void vUninitObject (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vDestroyObject (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceReady (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static void vSetError (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bHandleMessageReception (
    WEATHER_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bInitAppFacingObject(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static WEATHER_APP_OBJECT_STRUCT *psGetAppFacingObject(
    WEATHER_SERVICE_OBJECT hService
        );

void vUnlockAppFacingObject(
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bPrepareLocationRow(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WEATHER_LOCATION_ROW_STRUCT *psLoc
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessSelectUpdateVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WEATHER_DB_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bDeleteLocFromDB(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBSchemaVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN8 un8SchemaVer
        );

static BOOLEAN bLoadDBContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N16 *pn16DBVer
        );

static BOOLEAN bAddLocToDB(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static BOOLEAN bQueryLocationsFromDB(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    WEATHER_FILTER_LOCATIONS_LIST_STRUCT *psFilterList
        );

static BOOLEAN bCreateList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bModifyList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDeleteList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRefreshList(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static DSRL_ENTRY_OBJECT hSearchEntry(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    LOC_ID tID
        );

static BOOLEAN bEntryChange(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static void vChangeEntryDSRLsState(
    DSRL_ENTRY_OBJECT hEntry,
    DSRL_STATE_ENUM eState
        );

static BOOLEAN bDSRLChangeState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLChangeEntry (
    DSRL_OBJECT hDSRL,
    WEATHER_DSRL_CHANGE_ENTRY_STRUCT *psChangeEntry
        );

static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        );

static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bDSRLFillDiffEntriesLists(
    DSRL_OBJECT hDSRL,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        );

static BOOLEAN bRemoveUnusedEntries(
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bEntryRemoveDSRLLink(
    WEATHER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bUpdateLocationFromRow(
    LOCATION_OBJECT hLocation,
    WEATHER_LOCATION_ROW_STRUCT *psRow
        );

static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WEATHER_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    WEATHER_APP_OBJECT_STRUCT *psAppObj
        );

static void vCheckForItemsToCancel (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static void vRemoveOldItemsFromEntries(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    UN32 un32CurrentTime
        );

static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        );

static BOOLEAN bGetNearestLoc(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_SQL_QUERY_LOCATION_STRUCT *psLocationSQLArg
        );

static void vDestroyWeatherLocData(
    WEATHER_LOCATION_STRUCT *psObj
        );

static void vStartCancelTimer (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static void vStopCancelTimer (
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static void vChangeSkiCondTimestamp(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        );

static LOCATION_OBJECT hCreateLocationFromRow(
    WEATHER_LOCATION_ROW_STRUCT *psRow
        );

static void vChangeForecastTimestamp(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        );

static void vSkipForecastHashEntry(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hHashEntry
        );

static void vSkipSkiCondHashEntry(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    WEATHER_HASH_OBJECT hOldHashEntry
        );

static BOOLEAN bGetLocationFromDB(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_LOCATION_STRUCT *psLoc
        );

static BOOLEAN bGetPredefinedLocFromCoord(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hTargetLocation,
    BOOLEAN bSkiFlag,
    WEATHER_LOCATION_STRUCT *psLoc
        );

static BOOLEAN bGetPredefinedLocFromLocId(
    WEATHER_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hTargetLocation,
    BOOLEAN bSkiFlag,
    WEATHER_LOCATION_STRUCT *psLoc
        );

static BOOLEAN bFillLocationRow(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_LOCATION_ROW_STRUCT *psRow
        );

static BOOLEAN bCheckLocInDB(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbLocExists
        );

static BOOLEAN bExtractLocationFromDB(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WEATHER_FILTER_LOCATIONS_LIST_STRUCT *psFilterList
        );

static BOOLEAN bIsLocInDB(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WEATHER_LOCATION_ROW_STRUCT *psLocationData
        );

static BOOLEAN bLoadDataToEntries(
    WEATHER_MGR_OBJECT_STRUCT *psObj
        );

static N16 n16SortByLocID(
    WEATHER_MSG_OBJECT hWeatherMsg1,
    WEATHER_MSG_OBJECT hWeatherMsg2
        );

static void vDSRLEntryRelease(
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

const WEATHER_MGR_INTERFACE_STRUCT GsWeatherMgr =
{
    /*.bRefDBBank = */bRefDBBank,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.bStartForecastProcessing = */bStartProcessing,
    /*.bProcessForecastReport = */bProcessForecastReport,
    /*.bEndForecastProcessing = */bEndProcessing,
    /*.bStartSkiProcessing = */bStartProcessing,
    /*.bProcessSkiReport = */bProcessSkiReport,
    /*.bEndSkiProcessing = */bEndProcessing,
    /*.bAddLocation = */bAddLocation,
    /*.bModifyLocation = */bModifyLocation,
    /*.bDeleteLocation = */bDeleteLocation,
    /*.bForecastHashRemoved = */bForecastHashRemoved,
    /*.bForecastHashRepeated = */bForecastHashRepeated,
    /*.bSkiHashRemoved = */bSkiHashRemoved,
    /*.bSkiHashRepeated = */bSkiHashRepeated
};

// Global (re-usable) instance of an interface for weather service
const WEATHER_INTERFACE_STRUCT WEATHER =
{
    /*.hStart = */hStart,
    /*.vStop = */(void (*) (WEATHER_SERVICE_OBJECT hWeatherService))DATASERVICE_IMPL_vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (WEATHER_SERVICE_OBJECT hWeatherService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (WEATHER_SERVICE_OBJECT hWeatherService))DATASERVICE_IMPL_eErrorCode,
    /*.eEnableSkiReports =*/eEnableSkiReports,
    /*.eDisableSkiReports =*/eDisableSkiReports,
    /*.eIterateLocations =*/eIterateLocations,
    /*.eGetReferenceDataVersion =*/eGetReferenceDataVersion
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsTabWeatherStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

#endif /* _WEATHER_MGR_OBJ_H_ */
