/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _TRAFFIC_MSG_OBJ_H_
#define _TRAFFIC_MSG_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "alertc_event_obj.h"
#include "traffic_msg_base_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define TRAFFIC_MSG_OBJECT_NAME "TRAFFIC_MSG"

#define MAX_SUPPLEMENTARY_INFO_CODES (12)
#define MAX_QUANTIFIERS (1)
#define MAX_QUANTIFIER_LENGTH (15)

// Used when printing the output text
#define MAX_EVENT_TEXT_OUTPUT_LENGTH (28)

// The number of groups can be in a single message
#define MAX_GROUPS_IN_MESSAGE (5)

// definitions for field lengths in the Alert-C message
#define RESERVED_FIELD_BITLEN           (3)
#define ALERT_C_FLAG_FIELD_BITLEN       (1)
#define FULL_GROUP_FLAG_FIELD_BITLEN    (1)
#define DURATION_FIELD_BITLEN           (3)
#define CONTINUTY_INDEX_FIELD_BITLEN    (3)
#define DIVERSION_ADVISED_FIELD_BITLEN  (1)
#define FIRST_GROUP_FLAG_FIELD_BITLEN   (1)
#define DIRECTION_FLAG_FIELD_BITLEN     (1)
#define SECOND_GROUP_FLAG_FIELD_BITLEN  (1)
#define EXTENT_FIELD_BITLEN             (3)
#define EVENT_CODE_FIELD_BITLEN         (11)
#define POS_CODE_FIELD_BITLEN           (16)
#define GSI_FIELD_BITLEN                (2)
#define FREEFORM_TOTAL_DATA_BITLEN      (28)
#define FREEFORM_LABEL_FIELD_BITLEN     (4)
#define MIN_FREEFORM_DSRL_ENTRY         (FREEFORM_LABEL_FIELD_BITLEN)

#define ALERTC_MESSAGE_TYPE             (0)

// Maximums for the number of values of each free form data type
#define MAX_FREEFORM_CONTROL_CODE_VALUES  (8)
#define MAX_FREEFORM_DURATION_VALUES      (1)
#define MAX_NUM_EXPLICIT_TIMES            (1)

// The range of values for the explicit start/time field
#define EXPLICIT_TIME_15MIN_START         (0)
#define EXPLICIT_TIME_15MIN_STOP          (95)
#define EXPLICIT_TIME_1HOUR_START         (96)
#define EXPLICIT_TIME_1HOUR_STOP          (200)
#define EXPLICIT_TIME_1DAY_START          (201)
#define EXPLICIT_TIME_1DAY_STOP           (231)
#define EXPLICIT_TIME_HALF_MONTH_START    (232)
#define EXPLICIT_TIME_HALF_MONTH_STOP     (255)

#define EXPLICIT_TIME_1DAY_OFFSET         (EXPLICIT_TIME_1HOUR_STOP)

  /**************/
 /** TYPEDEFS **/
/**************/

/*
    Traffic Message Control Structure

    This structure contains the Traffic Message pool
*/

typedef struct traffic_msg_ctrl_struct
{
    // TrafMsg Pool
    OSAL_OBJECT_HDL hPool;

    TRAFFIC_SERVICE_OBJECT hTrafficMgr;

#if DEBUG_OBJECT != 0
    // Statistics
    UN32 un32Available;
    UN32 un32Unavailable;
    UN32 un32Created;
#endif

} TRAFFIC_MSG_CTRL_STRUCT;

// Enum defining the possible values for the control code value
// Associated with ALERTC_FREEFORM_DATA_LABEL_CONTROL
typedef enum alertc_freemform_data_control_value_enum
{
    ALERTC_FREEFORM_DATA_CONTROL_URGENCY_PLUS_1 = 0,
    ALERTC_FREEFORM_DATA_CONTROL_URGENCY_MINUS_1,
    ALERTC_FREEFORM_DATA_CONTROL_DIRECTIONALITY_CHANGED,
    ALERTC_FREEFORM_DATA_CONTROL_PROVISION_INTERCHANGED,
    ALERTC_FREEFORM_DATA_CONTROL_SPOKEN_DURATION_INTERCHANGED,
    ALERTC_FREEFORM_DATA_CONTROL_DIVERSION_BIT_SET,
    ALERTC_FREEFORM_DATA_CONTROL_INCREASE_EXTENT_BY_8,
    ALERTC_FREEFORM_DATA_CONTROL_INCREASE_EXTENT_BY_16,
} ALERTC_FREEFORM_DATA_CONTROL_VALUE_ENUM;

// Data representing a free form data entry
typedef struct alertc_freeform_data_struct
{
    BOOLEAN bInUse;
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel;
    UN16 un16Value;

} ALERTC_FREEFORM_DATA_STRUCT;

// Data structure containing items pertaining to
// the underlying alert-c message
typedef struct alertc_object_struct
{
    /**********************************
     *Alert-C Data Items
     *********************************/
    TRAFFIC_MARKET tMarket;
    TRAFFIC_BSA tBSA;
    TRAFFIC_POS_CODE tPosCode;

    UN8 un8Duration;  // Note, can be modified by freeform data
    BOOLEAN bDiversionAdvised;  // Note, can be modified by freeform data
    TRAFFIC_DIRECTION_ENUM eDirection;
    UN8 un8Extent;  // Note, can be modified by freeform data
    TRAFFIC_EVENT_CODE tEventCode;

    // Multi-group specific data
    UN8 un8MultiGroupCount;

    // List containing the Free form data
    OSAL_OBJECT_HDL hFreeFormDataList;

    // A handle to data containing more details on the event code
    ALERTC_EVENT_OBJECT hEventObj;

} ALERTC_OBJECT_STRUCT;

typedef struct traffic_storage_table_row_struct
{
    UN32 un32Time;
    ALERTC_OBJECT_STRUCT sAlertC;
    TRAFFIC_MSG_CLASS_ENUM eClass;
    UN16 *pun16FreeFormDataBlob;
    UN8 un8FreeFormDataCount;
    UN8  un8BlobSize;

} TRAFFIC_STORAGE_TABLE_ROW_STRUCT;

// Private object structure
typedef struct traffic_msg_object_struct
{
    /**********************************
     *Header items used by TrafMsg Pool
     *********************************/
    OSAL_LINKED_LIST_ENTRY hEntry;
    BOOLEAN bInUse;

    ALERTC_OBJECT_STRUCT sAlertC;

    STRING_OBJECT hTextMod;
    STRING_OBJECT hAdditionalText;

    OSAL_OBJECT_HDL hLocationList;

    UN32 un32TimeStamp;

    BOOLEAN bWrittenToDB;

    SMS_OBJECT hParent;

} TRAFFIC_MSG_OBJECT_STRUCT;

typedef struct traffic_enum_output_struct
{
    UN32 un32Value;
    const char *pcTerseOutput;
    const char *pcFullOutput;
} TRAFFIC_ENUM_OUTPUT_STRUCT;

typedef struct traffic_create_free_form_blob_iterator_struct
{
    UN8 un8FreeFormDataCount;
    UN16 *pun16BlobData;
    size_t tBlobDataSize;

} TRAFFIC_CREATE_FREE_FORM_BLOB_ITERATOR_STRUCT;

typedef enum month_time_value_enum
{
    MONTH_TIME_VALUE_JAN = 0,
    MONTH_TIME_VALUE_FEB,
    MONTH_TIME_VALUE_MAR,
    MONTH_TIME_VALUE_APR,
    MONTH_TIME_VALUE_MAY,
    MONTH_TIME_VALUE_JUN,
    MONTH_TIME_VALUE_JUL,
    MONTH_TIME_VALUE_AUG,
    MONTH_TIME_VALUE_SEP,
    MONTH_TIME_VALUE_OCT,
    MONTH_TIME_VALUE_NOV,
    MONTH_TIME_VALUE_DEC
} MONTH_TIME_VALUE_ENUM;

typedef enum explicit_time_value_enum
{
    EXPLICIT_TIME_VALUE_JAN_15 = 232,
    EXPLICIT_TIME_VALUE_JAN_31,
    EXPLICIT_TIME_VALUE_FEB_15,
    EXPLICIT_TIME_VALUE_FEB_END,
    EXPLICIT_TIME_VALUE_MAR_15,
    EXPLICIT_TIME_VALUE_MAR_31,
    EXPLICIT_TIME_VALUE_APR_15,
    EXPLICIT_TIME_VALUE_APR_30,
    EXPLICIT_TIME_VALUE_MAY_15,
    EXPLICIT_TIME_VALUE_MAY_31,
    EXPLICIT_TIME_VALUE_JUN_15,
    EXPLICIT_TIME_VALUE_JUN_30,
    EXPLICIT_TIME_VALUE_JUL_15,
    EXPLICIT_TIME_VALUE_JUL_31,
    EXPLICIT_TIME_VALUE_AUG_15,
    EXPLICIT_TIME_VALUE_AUG_31,
    EXPLICIT_TIME_VALUE_SEP_15,
    EXPLICIT_TIME_VALUE_SEP_30,
    EXPLICIT_TIME_VALUE_OCT_15,
    EXPLICIT_TIME_VALUE_OCT_31,
    EXPLICIT_TIME_VALUE_NOV_15,
    EXPLICIT_TIME_VALUE_NOV_30,
    EXPLICIT_TIME_VALUE_DEC_15,
    EXPLICIT_TIME_VALUE_DEC_31

} EXPLICIT_TIME_VALUE_ENUM;

  /************/
 /** MACROS **/
/************/

// Time related macros
#define IS_LEAP_YEAR(year) ((year% 4) == 0 && ((year% 100) != 0 || (year%400) == 100))
#define NUM_DAYS_IN_MONTH(month, year)  (gaun8NumDaysInMonth [IS_LEAP_YEAR(year)][month])
#define NUM_DAYS_IN_YEAR(year)          (IS_LEAP_YEAR(year) ? 366 : 365)


  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */
static TRAFFIC_MSG_TYPE_ENUM eType(
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static TRAFFIC_MSG_CLASS_ENUM eClass (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static TRAFFIC_SPEED_ENUM eSpeed (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static TRAFFIC_DURATION_ENUM eDuration (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static TRAFFIC_DIRECTION_ENUM eDirection (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static BOOLEAN bDiversionAdvised (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static TRAFFIC_EVENT_CODE tEventCode (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static STRING_OBJECT hText (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static STRING_OBJECT hAdditionalText (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static UN32 un32StartTime (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static UN32 un32StopTime (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static UN8 un8GetFreeFormData (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 *paun16Values,
    size_t tCount
        );

static UN8 un8NumLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg
        );

static LOCATION_OBJECT hGetLocation (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    UN8 un8Index
        );

static N32 n32FPrintf (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        );

static SMSAPI_RETURN_CODE_ENUM eIterateLocations (
    TRAFFIC_MSG_BASE_OBJECT hTrafficBaseMsg,
    TRAFFIC_MSG_LOCATION_ITERATOR bIterator,
    void *pvIteratorArg
        );

/* Object Private Prototypes */

static STRING_OBJECT hBuildTextWithQuantifiers (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    STRING_OBJECT hText,
    UN16 un16Value
        );

static N16 n16IsLocationEqual (
    const void *pvObj1,
    const void *pvObj2
        );

static ALERTC_EVENT_OBJECT hGetEventInfo (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        );

static TRAFFIC_MSG_OBJECT hFindReusable (
    void
        );

static TRAFFIC_MSG_OBJECT hCreateObject (
    SMS_OBJECT hOwner
        );

static void vDestroyObject (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        );

static void vReleaseFreeFormDataListNode (
    void *pvData
        );

static BOOLEAN bParsePayload (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static size_t tFreeFormDataLength (
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel
        );

static BOOLEAN bParseFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static void vAddFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 un16Value
        );

static ALERTC_FREEFORM_DATA_STRUCT *psCreateFreeFormData(
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateFieldsBasedOnFreeFormData (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
    UN16 un16Value
        );

static N16 n16CompareTrafficMsgPool (
    TRAFFIC_MSG_OBJECT hTrafficMsg1,
    TRAFFIC_MSG_OBJECT hTrafficMsg2
        );

static N16 n16CompareFreeFormData (
    const ALERTC_FREEFORM_DATA_STRUCT *psObj1,
    const ALERTC_FREEFORM_DATA_STRUCT *psObj2
        );

static N16 n16FindTrafficMsgToReuse (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    const void *pvData
        );

static BOOLEAN bMarkFreeFormDataInactive(
    ALERTC_FREEFORM_DATA_STRUCT *psObj,
    const void *pvData
        );

static BOOLEAN bAddFreeFormDataToBlob(
    ALERTC_FREEFORM_DATA_STRUCT *psObj,
    TRAFFIC_CREATE_FREE_FORM_BLOB_ITERATOR_STRUCT *psIteratorArg
        );

static UN32 un32GetStartStopTime (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj,
    ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel
        );

static TRAFFIC_SERVICE_OBJECT hGetTrafficMgr (
    TRAFFIC_MSG_OBJECT_STRUCT *psObj
        );

static N16 n16IsFreeFormDataEqual (
    const void *pvObj1,
    const void *pvObj2
        );

static void vReleaseObject(
    TRAFFIC_MSG_OBJECT hTrafficMsg
        );

static const char *pacTypeText(
    TRAFFIC_MSG_TYPE_ENUM eType,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        );

static const char *pacClassText(
    TRAFFIC_MSG_CLASS_ENUM eClass
        );

static const char *pacDurationText(
    TRAFFIC_DURATION_ENUM eDuration,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        );

static const char *pacDirectionText(
    TRAFFIC_DIRECTION_ENUM eDirection,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        );

static void vSetAddStmtParams (
    ALERTC_OBJECT_STRUCT const *psAlertC,
    UN32 un32UnixTimeSecs,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    UN8 un8FreeFormDataCount,
    void *pvFreeFormData,
    size_t tFreeFormDataSize,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        );

  /***************/
 /** VARIABLES **/
/***************/

// This pointer is a child object of the Traffic Mgr Object
// and is protected from multiple access using that mechanism.
TRAFFIC_MSG_CTRL_STRUCT *gpsTrafficMsgCtrl = (TRAFFIC_MSG_CTRL_STRUCT *)NULL;

const TRAFFIC_ENUM_OUTPUT_STRUCT gasDurationOutput[TRAFFIC_DURATION_COUNT] =
{
    {
        TRAFFIC_DURATION_UNKNOWN,
        "??",
        MACRO_TO_STRING(TRAFFIC_DURATION_UNKNOWN)
    },
    {
        TRAFFIC_DURATION_15_MINUTES,
        "15m",
        MACRO_TO_STRING(TRAFFIC_DURATION_15_MINUTES)
    },
    {
        TRAFFIC_DURATION_30_MINUTES,
        "30m",
        MACRO_TO_STRING(TRAFFIC_DURATION_30_MINUTES)
    },
    {
        TRAFFIC_DURATION_1_HOUR,
        "1h",
        MACRO_TO_STRING(TRAFFIC_DURATION_1_HOUR)
    },
    {
        TRAFFIC_DURATION_2_HOURS,
        "2h",
        MACRO_TO_STRING(TRAFFIC_DURATION_2_HOURS)
    },
    {
        TRAFFIC_DURATION_3_HOURS,
        "3h",
        MACRO_TO_STRING(TRAFFIC_DURATION_3_HOURS)
    },
    {
        TRAFFIC_DURATION_4_HOURS,
        "4h",
        MACRO_TO_STRING(TRAFFIC_DURATION_4_HOURS)
    },
    {
        TRAFFIC_DURATION_REST_OF_DAY,
        "1d",
        MACRO_TO_STRING(TRAFFIC_DURATION_REST_OF_DAY)
    }
};

const TRAFFIC_ENUM_OUTPUT_STRUCT gasMsgClassOutput[TRAFFIC_MSG_CLASS_COUNT] =
{
    {
        TRAFFIC_MSG_CLASS_UNKNOWN,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_UNKNOWN)
    },
    {
        TRAFFIC_MSG_CLASS_LEVEL_OF_SERVICE,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_LEVEL_OF_SERVICE)
    },
    {
        TRAFFIC_MSG_CLASS_EXPECTED_LEVEL_OF_SERVICE,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_EXPECTED_LEVEL_OF_SERVICE)
    },
    {
        TRAFFIC_MSG_CLASS_ACCIDENTS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ACCIDENTS)
    },
    {
        TRAFFIC_MSG_CLASS_INCIDENTS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_INCIDENTS)
    },
    {
        TRAFFIC_MSG_CLASS_CLOSURES_AND_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_CLOSURES_AND_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_ROAD_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ROAD_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_EXIT_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_EXIT_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_ENTRY_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ENTRY_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_TRAFFIC_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_TRAFFIC_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_CARPOOL_INFO,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_CARPOOL_INFO)
    },
    {
        TRAFFIC_MSG_CLASS_ROADWORKS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ROADWORKS)
    },
    {
        TRAFFIC_MSG_CLASS_OBSTRUCTION_HAZARDS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_OBSTRUCTION_HAZARDS)
    },
    {
        TRAFFIC_MSG_CLASS_DANGER_SITUATIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_DANGER_SITUATIONS)
    },
    {
        TRAFFIC_MSG_CLASS_ROAD_CONDITIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ROAD_CONDITIONS)
    },
    {
        TRAFFIC_MSG_CLASS_TEMPERATURES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_TEMPERATURES)
    },
    {
        TRAFFIC_MSG_CLASS_PERCIPITATION_VISIBILITY,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_PERCIPITATION_VISIBILITY)
    },
    {
        TRAFFIC_MSG_CLASS_WIND_AND_AIR_QUALITY,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_WIND_AND_AIR_QUALITY)
    },
    {
        TRAFFIC_MSG_CLASS_ACTIVITIES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_ACTIVITIES)
    },
    {
        TRAFFIC_MSG_CLASS_SECURITY_ALERTS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_SECURITY_ALERTS)
    },
    {
        TRAFFIC_MSG_CLASS_DELAYS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_DELAYS)
    },
    {
        TRAFFIC_MSG_CLASS_CANCELATIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_CANCELATIONS)
    },
    {
        TRAFFIC_MSG_CLASS_TRAVEL_TIME_INFO,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_TRAVEL_TIME_INFO)
    },
    {
        TRAFFIC_MSG_CLASS_DANGEROUS_VEHICLES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_DANGEROUS_VEHICLES)
    },
    {
        TRAFFIC_MSG_CLASS_EXCEPTIONAL_LOAD_VEHICLES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_EXCEPTIONAL_LOAD_VEHICLES)
    },
    {
        TRAFFIC_MSG_CLASS_TRAFFIC_EQUIP_STATUS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_TRAFFIC_EQUIP_STATUS)
    },
    {
        TRAFFIC_MSG_CLASS_SIZE_WEIGHT_LIMITS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_SIZE_WEIGHT_LIMITS)
    },
    {
        TRAFFIC_MSG_CLASS_PARKING_RESTRICTIONS,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_PARKING_RESTRICTIONS)
    },
    {
        TRAFFIC_MSG_CLASS_PARKING,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_PARKING)
    },
    {
        TRAFFIC_MSG_CLASS_REF_TO_AUDIO_BDCAST,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_REF_TO_AUDIO_BDCAST)
    },
    {
        TRAFFIC_MSG_CLASS_SERVICE_MESSAGES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_SERVICE_MESSAGES)
    },
    {
        TRAFFIC_MSG_CLASS_SPECIAL_MESSAGES,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_SPECIAL_MESSAGES)
    },
    {
        TRAFFIC_MSG_CLASS_32,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_32)
    },
    {
        TRAFFIC_MSG_CLASS_33,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_33)
    },
    {
        TRAFFIC_MSG_CLASS_34,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_34)
    },
    {
        TRAFFIC_MSG_CLASS_35,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_35)
    },
    {
        TRAFFIC_MSG_CLASS_36,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_36)
    },
    {
        TRAFFIC_MSG_CLASS_37,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_37)
    },
    {
        TRAFFIC_MSG_CLASS_38,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_38)
    },
    {
        TRAFFIC_MSG_CLASS_39,
        "",
        MACRO_TO_STRING(TRAFFIC_MSG_CLASS_39)
    }
};

static const UN8 gaun8NumDaysInMonth [2][12] =
    {
        { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31},  // Normal
        { 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}   // Leap-year
    };

// Global (re-usable) instance of an interface for this object
const TRAFFIC_MSG_BASE_OBJECT_INTERFACE_STRUCT LEGACY_TRAFFIC_MSG =
{
    /*.eType = */eType,
    /*.eClass = */eClass,
    /*.eSpeed = */eSpeed,
    /*.eDuration = */eDuration,
    /*.eDirection = */eDirection,
    /*.bDiversionAdvised = */bDiversionAdvised,
    /*.tEventCode = */tEventCode,
    /*.hText = */hText,
    /*.hAdditionalText = */hAdditionalText,
    /*.un32StartTime = */un32StartTime,
    /*.un32StopTime = */un32StopTime,
    /*.un8GetFreeFormData = */un8GetFreeFormData,
    /*.un8NumLocations = */un8NumLocations,
    /*.hGetLocation = */hGetLocation,
    /*.n32FPrintf = */n32FPrintf,
    /*.eIterateLocations = */eIterateLocations,
    /*.eCameraDirection = */NULL,
    /*.hImage = */NULL
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _TRAFFIC_MSG_OBJ_H_
