/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _TRAFFIC_MGR_OBJ_H_
#define _TRAFFIC_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "traffic_msg_obj.h"
#include "sql_interface_obj.h"
#include "traffic_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define TRAFFIC_MGR_OBJECT_NAME "TrafficService"

// Constant which defines the size of the buffer used by
// traffic for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define TRAFFIC_SHARED_BUFFER_LEN ( TRAFFIC_MAX_SQL_STRING_LENGTH )

#define UNIQUE_PAYLOAD_TIMEOUT (600)    // 10 Minutes
#define MAX_MSGS_TO_PROCESS (1)

#define CANCEL_MSG_TIMER_PERIOD (60) // One minute
#define MSG_TIMEOUT (900) // Fifteen minutes in seconds

#define WORK_TIME_UNIT_MS (150)

// This define is used to timeout data read from the traffic table storage
#define RESTORED_MSGS_TIMEOUT_SECS (15 * 60) //15 Minutes

// Constants to use for message timeouts
#define TRAFFIC_DURATION_15_MINUTES_AS_SECONDS (15 * 60)
#define TRAFFIC_DURATION_30_MINUTES_AS_SECONDS (15 * 60)
#define TRAFFIC_DURATION_1_HOUR_AS_SECONDS (1 * 60 * 60)
#define TRAFFIC_DURATION_2_HOURS_AS_SECONDS (2 * 60 * 60)
#define TRAFFIC_DURATION_3_HOURS_AS_SECONDS (3 * 60 * 60)
#define TRAFFIC_DURATION_4_HOURS_AS_SECONDS (4 * 60 * 60)
#define TRAFFIC_DURATION_4_HOURS_AS_SECONDS (4 * 60 * 60)
#define TRAFFIC_DURATION_REST_OF_DAY_AS_SECONDS (12 * 60 * 60)
#define TRAFFIC_DURATION_24_HOURS_AS_SECONDS (24 * 60 * 60)

// Constant defines the minimum device location
// delta which traffic wants to know about
#define TRAFFIC_DEVICE_DISTANCE_THRESHOLD (2)

#define TRAFFIC_SUPPORTED_OPTIONS ( \
    (DATASERVICE_OPTIONS_MASK) \
    DATASERVICE_OPTION_NONE \
    )

#define MAX_SQL_BIND_PARAMS 13

  /**************/
 /** TYPEDEFS **/
/**************/
typedef struct target_update_struct
{
    TRAFFIC_MSG_OBJECT hMsg;
    TRAFFIC_MSG_OBJECT hOldMsg;
    DSRL_ENTRY_STATUS_ENUM eStatus;

} TARGET_UPDATE_STRUCT;

typedef struct event_entry_struct
{
    TRAFFIC_MSG_OBJECT hStableMsg;
    TRAFFIC_MSG_OBJECT hUnstableMsg;
    TRAFFIC_DIRECTION_ENUM eDirection;
    TRAFFIC_MSG_CLASS_ENUM eClass;
} EVENT_ENTRY_STRUCT;

typedef struct pos_event_desc_struct
{
    LOCATION_OBJECT hLocation;
    struct pos_event_desc_struct *psPositive;
    struct pos_event_desc_struct *psNegative;
    BOOLEAN bPosFinalLocation;
    BOOLEAN bNegFinalLocation;

    // Linked list containing EVENT_ENTRY_STRUCTs for the
    // active events at this location.
    // Sorted by direction and class
    OSAL_OBJECT_HDL hEventList;

    // List of pointers to DSRL_DESC_STRUCT that are stored
    // in the TRAFFIC_MGR_OBJECT_STRUCT's hDSRLList
    OSAL_OBJECT_HDL hDSRLList;

    // Copy of Position Code value from hLocation
    // Used for fast searching
    TRAFFIC_POS_CODE tPosCode;

} POS_EVENT_DESC_STRUCT;

typedef struct dsrl_desc_struct
{
    DSRL_OBJECT hDSRL;

    DSRL_STATE_ENUM eCurrentRequestState;

    OSAL_OBJECT_HDL hTargetList;

} DSRL_DESC_STRUCT;

typedef struct target_desc_struct
{
    LOCATION_OBJECT hLocation;

    // A list of POS_EVENT_DESC_STRUCT that are stored in
    // the ahTrafficTable
    OSAL_OBJECT_HDL hPosList;

} TARGET_DESC_STRUCT;

typedef struct supplemental_info_struct
{
    ALERTC_SUPPL_INFO_CODE tCode;
    char *pcText;

} SUPPL_INFO_STRUCT;

typedef struct traffic_app_object_struct
{
    // Linked-list containing the Market/BSA filter
    OSAL_OBJECT_HDL hMarketFilterList;

    // Caches of information retrieved from the database
    OSAL_OBJECT_HDL hAlertCEventList;
    OSAL_OBJECT_HDL hSupplInfoList;

    // DB Connection Handles, File Paths, and Shared Buffer
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    char *pacFullyQualifiedRefDatabaseFilePath;
    SQL_INTERFACE_OBJECT hSQLPersistantConnection;
    char *pacFullyQualifiedPersistantDatabaseFilePath;
    char acBuffer[TRAFFIC_SHARED_BUFFER_LEN];

    // Per market array of linked-list containing active
    // traffic information
    // Table should be indexed by using tMarket-1.
    OSAL_OBJECT_HDL ahTrafficTable[NUM_TRAFFIC_MARKETS];

    // AlertC event dummy object is used for search
    ALERTC_EVENT_OBJECT hDummyAlertCEvent;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // Prepared statements
    SQL_PREPARED_STATEMENT_HANDLE hInsertMsgIntoDBStmt;

} TRAFFIC_APP_OBJECT_STRUCT;

typedef struct traffic_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    // like the market filter and the data that feeds the TRAFFIC_MSG
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The callback used to query the application to
    // get information about an Alert-C position.  If
    // this callback is NULL, then the best we can do is
    // is give the application infromation filtered by the
    // Market/BSA filter
    TRAFFIC_POSCODE_LOOKUP_CALLBACK vTrafficPosLookUpCallback;
    void *pvPosLookUpCallbackArg;

    // A list of active targets (a list of DSRLs for the application)
    // Each entry contains a DSRL_DESC_STRUCT
    OSAL_OBJECT_HDL hDSRLList;

    DATASERVICE_TIMED_EVENT_HDL hCancelEvent;

    UN32 un32LastTimeDataRXd;

    // Data structure used for broadcast-specific parsing
    TRAFFIC_INTERFACE_OBJECT hTrafficInterfaceData;

} TRAFFIC_MGR_OBJECT_STRUCT;

// Structure used to perform traffic database queries
typedef struct traffic_db_query_result_struct
{
    BOOLEAN bResultantRows;
    TRAFFIC_DB_ROW_UNION uDbRow;
} TRAFFIC_DB_QUERY_RESULT_STRUCT;

// Argument structure used for processing traffic messages
// from the database storage
typedef struct traffic_storage_restoration_arg_struct
{
    BOOLEAN bResultantRows;
    TRAFFIC_MGR_OBJECT_STRUCT *psMgr;
    TRAFFIC_MARKET tMarket;
    UN32 un32UTCsec;
} TRAFFIC_STORAGE_RESTORATION_ARG_STRUCT;

typedef struct expired_message_iterator_struct
{
    TRAFFIC_MGR_OBJECT_STRUCT *psMgr;
    POS_EVENT_DESC_STRUCT *psPosDesc;
    BOOLEAN bMsgsStillActive;
    UN32 un32UTCsec;

} EXPIRED_MESSAGE_ITERATOR_STRUCT;

typedef struct database_storage_iterator_struct
{
    TRAFFIC_MGR_OBJECT_STRUCT *psMgr;
    TRAFFIC_MARKET tMarket;
    POS_EVENT_DESC_STRUCT *psPosDesc;
    UN32 un32UTCsec;

} DATABASE_STORAGE_ITERATOR_STRUCT;

typedef struct dsrl_remove_target_iterator_struct
{
    POS_EVENT_DESC_STRUCT *psPosDesc;
    BOOLEAN bFound;

} DSRL_REMOVE_TARGET_ITERATOR_STRUCT;

typedef struct traffic_link_pos_desc_iterator_struct
{
    POS_EVENT_DESC_STRUCT *psPosDesc;
    DSRL_DESC_STRUCT *psDSRLDesc;
} TRAFFIC_LINK_POS_DESC_ITERATOR_STRUCT;

typedef struct traffic_link_target_desc_iterator_struct
{
    TARGET_DESC_STRUCT *psTargetDesc;
    DSRL_DESC_STRUCT *psDSRLDesc;
} TRAFFIC_LINK_TARGET_DESC_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static TRAFFIC_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    TRAFFIC_POSCODE_LOOKUP_CALLBACK vTrafficPosLookUpCallback,
    void *pvPosLookUpCallbackArg,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

static SMSAPI_RETURN_CODE_ENUM eAddMarket(
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveMarket(
    TRAFFIC_SERVICE_OBJECT hTrafficService,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        );

/* Object Private Prototypes */

static TRAFFIC_APP_OBJECT_STRUCT *psGetAppFacingObject(
    TRAFFIC_SERVICE_OBJECT hTrafficService
        );

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bCreateList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bModifyList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRefreshList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDestroyList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bInitAppFacingObject(
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vDestroyObject (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static void vReleaseMarketFilterListNode (
    void *pvData
        );

static void vReleasePosDesc (
    void *pvData
        );

static void vReleaseEventDesc (
    EVENT_ENTRY_STRUCT *psEventDesc
        );

static void vReleaseDSRLNode (
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static void vReleaseTargetNode (
    TARGET_DESC_STRUCT *psTargetDesc
        );

static void vReleaseAlertCEventInfo (
    void *pvData
        );

static void vReleaseSupplInfo (
    void *pvData
        );

static BOOLEAN bUpdateFilterForMarket (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket,
    BOOLEAN bAddMarket
        );

static BOOLEAN bProcessPayload (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static void vMarkNationComplete (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessTrafficMessagesFromPayload (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TRAFFIC_LOCID_OBJECT hTrafficLocID,
    size_t tNumAlertCBytes,
    TRAFFIC_MARKET_PROC_STATE_ENUM eMarketProcState
        );

static BOOLEAN bUpdateTrafficTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        );

static BOOLEAN bUpdatePositionsInTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    OSAL_LINKED_LIST_ENTRY *phStartPosEntry
        );

static POS_EVENT_DESC_STRUCT *psCreatePosDesc (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        );

static BOOLEAN bUpdateTrafficMsgsInTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    OSAL_LINKED_LIST_ENTRY hStartPosEntry
        );

static EVENT_ENTRY_STRUCT *psGetEventForPos (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass
        );

static BOOLEAN bHandleCanceledTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        );

static BOOLEAN bHandleNewTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        );

static BOOLEAN bHandleUpdatedTrafficMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_DIRECTION_ENUM eDirection,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EVENT_ENTRY_STRUCT *psHeadEventDesc
        );

static void vHandleOverlappedMessage (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    EVENT_ENTRY_STRUCT *psOverlapedEventDesc
        );

static BOOLEAN bNotifyTargets (
    DSRL_DESC_STRUCT *psDSRLDesc,
    void *pvArg
        );

static void vUpdateTargetList (
    OSAL_OBJECT_HDL hTargetList,
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFFIC_MSG_OBJECT hOldTrafficMsg,
    DSRL_ENTRY_STATUS_ENUM eStatus
        );

static BOOLEAN bUpdateTarget (
    DSRL_DESC_STRUCT *psDSRLDesc,
    TARGET_UPDATE_STRUCT *psUpdate
        );

static BOOLEAN bCheckMarketList (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_LOCID_OBJECT hTrafficLocID
        );

static BOOLEAN bHandleServiceReady (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static N16 n16CompareTrafficLocID (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareDSRLDesc (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareTargetDesc (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareEventDesc (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16ComparePosCode (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareSupplInfoCode (
    SUPPL_INFO_STRUCT *psSupplInfo1,
    SUPPL_INFO_STRUCT *psSupplInfo2
        );

static UN8 un8FindNumLocationsKnown (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    TRAFFIC_POS_CODE *ptPosCode,
    TRAFFIC_MARKET tMarket,
    TRAFFIC_DIRECTION_ENUM eDirection,
    UN8 un8NumLocsReq,
    BOOLEAN *pbFinalLocation
        );

static BOOLEAN bCreateLinksWithDSRLAndTable (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bHandleRemoveTargets (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN *pbSortNeeded
        );

static BOOLEAN bFindPosDescInDSRLDesc(
    DSRL_DESC_STRUCT *psDSRLDesc,
    POS_EVENT_DESC_STRUCT *psPosDesc
        );

static BOOLEAN bFindPosDescInTargetDesc(
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_REMOVE_TARGET_ITERATOR_STRUCT *psPosDescIterator
        );

static BOOLEAN bLinkNewPosDescToDSRLDesc (
    DSRL_DESC_STRUCT *psDSRLDesc,
    POS_EVENT_DESC_STRUCT *psPosDesc
        );

static BOOLEAN bLinkPosDescToTargetDesc (
    TARGET_DESC_STRUCT *psTargetDesc,
    TRAFFIC_LINK_POS_DESC_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bLinkTargetDescToPosDesc (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    TRAFFIC_LINK_TARGET_DESC_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bIterateTargetsForRefresh (
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bIteratePosDescsForRefresh (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bAddEventToTarget (
    EVENT_ENTRY_STRUCT *psCurEventDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bRemoveEventFromTarget (
    EVENT_ENTRY_STRUCT *psCurEventDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bIterateTargetDescToRemoveDSRLDesc (
    TARGET_DESC_STRUCT *psTargetDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bRemoveDSRLDescFromPos (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bShouldLinkLocations (
    LOCATION_OBJECT hLocationDSRL,
    LOCATION_OBJECT hLocationPosDesc
        );

static BOOLEAN bMarkMarketAsComplete (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket
        );

static BOOLEAN bMarkPositionAsComplete (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    DATABASE_STORAGE_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bMarkEventAsComplete (
    EVENT_ENTRY_STRUCT *psCurEventEntry,
    DATABASE_STORAGE_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bRemoveExpiredMessagesFromDatabase (
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj
        );

static void vRestoreAllMarketsFromDatabase (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj
        );

static void vRestoreMarketFromDatabase (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj,
    TRAFFIC_MARKET tMarket
        );

static void vStartCancelTimer (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static void vStopCancelTimer (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static void vCheckForMessagesToCancel (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCheckTimeForMarket (
    POS_EVENT_DESC_STRUCT *psPosDesc,
    EXPIRED_MESSAGE_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bCheckTimeForEvent (
    EVENT_ENTRY_STRUCT *psCurEventEntry,
    EXPIRED_MESSAGE_ITERATOR_STRUCT *psIteratorArg
        );

static UN32 un32GetNextDayStartTimestamp(
    void
        );

static BOOLEAN bCheckTimestampAndDuration (
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    UN32 un32CurrentTime
        );

static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessSelectEventResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessSelectSupplInfoResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessStoredMessages (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static N16 n16SortDSRLByDistance (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    void *pvSortArgs
        );

static void vSetError (
    TRAFFIC_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bEnsureTrafficTableExists (
    TRAFFIC_APP_OBJECT_STRUCT *psAppObj,
    TRAFFIC_MARKET tMarket
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const TRAFFIC_INTERFACE_STRUCT TRAFFIC =
{
    /*.hStart = */hStart,
    /*.vStop = */(void (*) (TRAFFIC_SERVICE_OBJECT hTrafficService))DATASERVICE_IMPL_vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (TRAFFIC_SERVICE_OBJECT hTrafficService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (TRAFFIC_SERVICE_OBJECT hTrafficService))DATASERVICE_IMPL_eErrorCode,
    /*.eAddMarket = */eAddMarket,
    /*.eRemoveMarket = */eRemoveMarket

};

const DATASERVICE_STATE_HANDLERS_STRUCT GsTrafficStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

extern const TRAFFIC_PLUGIN_INTERFACE_STRUCT GsTrafficIntf;

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _TRAFFIC_MGR_OBJ_H_
