/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _TRAFCAM_MGR_OBJ_H_
#define _TRAFCAM_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "trafcam_interface.h"
#include "trafcam_db_constants.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define TRAFCAM_MGR_OBJECT_NAME "TrafficCamerasService"

// Traffic Camera update threshold for device-based DSRL
#define TRAFCAM_DEVICE_DISTANCE_THRESHOLD     (10)

// Supported service startup options
#define TRAFCAM_SUPPORTED_OPTIONS (      \
    (DATASERVICE_OPTIONS_MASK)           \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      \
        )

// Timer period in seconds to check expired images
#define TRAFCAM_TIMER_PERIOD (30)

// special dir to files
#define TRAFCAM_FILES_PATH "/"TRAFCAM_MGR_OBJECT_NAME"/"

// files extension
#define TRAFCAM_FILES_EXT ".jpg"

// buffer used for file operations, currently defined as 1Kb
#define TRAFCAM_MGR_BUFFER_SIZE (1024)

  /**************/
 /** TYPEDEFS **/
/**************/

// Traffic Camera App Facing object
typedef struct trafcam_app_object_struct
{
    // Shared buffer
    char acBuffer[TRAFCAM_MAX_SQL_STRING_LENGTH];

    // DB Connection Handle & File Path for ref database
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    char *pacRefDatabaseFilePath;

    // Dummy object used for search
    TRAFFIC_MSG_OBJECT hDummyMsg;

    // A list of DSRL_ENTRY_OBJECT-TRAFFIC_MSG_OBJECT items
    OSAL_OBJECT_HDL hEntriesList;

    // A list of active targets (a list of DSRLs for the application)
    OSAL_OBJECT_HDL hDSRLList;

    // new entries list
    OSAL_OBJECT_HDL hNewEntriesList;

    // entries to add list
    OSAL_OBJECT_HDL hEntriesToAdd;

    // entries to remove list
    OSAL_OBJECT_HDL hEntriesToRemove;

    // new DSRL filter list
    OSAL_OBJECT_HDL hNewFilter;

} TRAFCAM_APP_OBJECT_STRUCT;

// Traffic Camera manager object Structure
typedef struct trafcam_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The version of the DB contents
    UN16 un16DBContentVersion;

    // Path to reference database(s) location
    char *pacRefDatabaseDirPath;

    // Protocol plugin handle
    TRAFCAM_INTERFACE_OBJECT hInterface;

    // Data filter
    OSAL_OBJECT_HDL hFilter;

    // Event
    DATASERVICE_TIMED_EVENT_HDL hExpireEvent;

    // Path to images dir
    char *pcFilesPath;
    size_t tPathSize;

    // file name buffer
    char *pcFileNameBuffer;
    size_t tFileNameBufferSize;
} TRAFCAM_MGR_OBJECT_STRUCT;

typedef struct trafcam_db_result_struct
{
    TRAFCAM_DB_ROW_UNION uDbRow;
    BOOLEAN bResult;
} TRAFCAM_DB_RESULT_STRUCT;

typedef struct trafcam_db_query_objects_struct
{
    BOOLEAN bResult;
    OSAL_OBJECT_HDL hObjectsList;
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj;
} TRAFCAM_DB_QUERY_OBJECTS_STRUCT;

typedef struct dsrl_desc_struct
{
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
    OSAL_OBJECT_HDL hTargetsList;
    OSAL_OBJECT_HDL hEntriesList;
} TRAFCAM_DSRL_DESC_STRUCT;

typedef struct dsrl_entry_desc_struct
{
    OSAL_LINKED_LIST_ENTRY hPoolEntry;
    OSAL_OBJECT_HDL hDSRLList;
} TRAFCAM_DSRL_ENTRY_DESC_STRUCT;

// This struct is used as iterator argument to generate entries diff list
typedef struct diff_list_iterator_struct
{
    OSAL_OBJECT_HDL hSubList;
    OSAL_OBJECT_HDL hDiffList;
    OSAL_LINKED_LIST_ENTRY hEntry;
} TRAFCAM_DIFF_LIST_ITERATOR_STRUCT;

// This struct is used as iterator argument to generate entries list for target
typedef struct target_list_iterator_struct
{
    TRAFCAM_APP_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hEntriesList;
    OSAL_OBJECT_HDL hTargetsList;
} TRAFCAM_TARGET_LIST_ITERATOR_STRUCT;

typedef struct trafcam_filter_iterator_struct
{
    OSAL_OBJECT_HDL hFilter;
    DSRL_OBJECT hDSRL;
    OSAL_LINKED_LIST_ENTRY hEntry;
    BOOLEAN bChanged;
} TRAFCAM_FILTER_ITERATOR_STRUCT;

typedef struct trafcam_target_message_check_iterator
{
    BOOLEAN bFits;
    LOCATION_OBJECT hLocation;
} TRAFCAM_TARGET_MESSAGE_CHECK_ITERATOR;

typedef struct trafcam_message_timeout_iterator
{
    UN32 un32TimeStamp;
    TRAFCAM_INTERFACE_OBJECT hInterface;
    BOOLEAN bChanged;
} TRAFCAM_MESSAGE_TIMEOUT_ITERATOR;

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static TRAFFIC_CAMERAS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

static BOOLEAN bReportNewImage(
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_INFO_STRUCT *psImageInfo,
    OSAL_BUFFER_HDL hImageBuffer
        );

static BOOLEAN bReportUpdatedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash,
    OSAL_BUFFER_HDL hImageBuffer
        );

static BOOLEAN bReportRemovedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash
        );

static BOOLEAN bReportRepeatedImage (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCameras,
    TRAFCAM_IMAGE_HASH tHash
        );

/* Object Private Prototypes */

static TRAFCAM_APP_OBJECT_STRUCT *psGetAppFacingObject (
    TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService
        );

static void vReleaseAppFacingObject (
    TRAFCAM_APP_OBJECT_STRUCT *psAppObj
        );

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bHandleServiceReady (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitFilesPath(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitAppFacingObject (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vSetError (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection
        );

static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN32 un32SchemaVer
        );

static N16 n16CompareFilterItems(
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter1,
    TRAFCAM_BSA_FILTER_STRUCT *psBSAFilter2
        );

static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    BOOLEAN bUpdateFilter
        );

static BOOLEAN bUpdateGlobalFilter (
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLChangeState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static BOOLEAN bDSRLChangeEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static FILE *psCreateFile(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    TRAFCAM_IMAGE_HASH tHash
        );

static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bEntryRemoveFromDSRLs(
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static void vStartAgeoutTimer(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static void vEntryRemove(
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static BOOLEAN bDSRLsReportEntry(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        );

BOOLEAN bEntryRemoveDSRLLink(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bApplyMessage(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFFIC_MSG_OBJECT hTrafficMsg
        );

static BOOLEAN bCheckForTimeout(
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFCAM_MESSAGE_TIMEOUT_ITERATOR *psIterator
        );

static BOOLEAN bCreateFilterItemsForMarket(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFFIC_MARKET tMarket,
    OSAL_OBJECT_HDL hFilter
        );

static BOOLEAN bDBGetContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N32 *pn32DBVer
        );

static BOOLEAN bDBProcessSelectBSAs(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    TRAFCAM_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bDBProcessSelectDBVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    TRAFCAM_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bDiffListGenerator(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    TRAFCAM_DIFF_LIST_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bDSRLAddEntries(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLAddEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static BOOLEAN bDSRLAddTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        );

static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bDSRLFillDiffEntriesLists(
    DSRL_OBJECT hDSRL,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bDSRLGenerateFilter(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRemoveEntries(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLRemoveEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bEntryCheckForDSRL(
    TRAFFIC_MSG_OBJECT hTrafficMsg,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bFilterRemoveDSRL(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bGenerateDSRLEntriesList(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc,
    OSAL_OBJECT_HDL hEntriesList
        );

static BOOLEAN bGenerateDSRLFilterList(
    TRAFCAM_APP_OBJECT_STRUCT *psObj,
    TRAFCAM_DSRL_DESC_STRUCT *psDSRLDesc,
    OSAL_OBJECT_HDL hNewFilter
        );

static BOOLEAN bGenerateEntriesDiffList (
    OSAL_OBJECT_HDL hMainList,
    OSAL_OBJECT_HDL hSubList,
    OSAL_OBJECT_HDL hDiffList
        );

static BOOLEAN bHandleCreateList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleModifyList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleRefreshList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bIsUpdateNeeded(
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bIterateToAddToGlobal(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bIterateToRemoveFromGlobal(
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem,
    TRAFCAM_FILTER_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bProcessCoordLocation(
    LOCATION_OBJECT hLocation,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bProcessMessage(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessPredefinedLocation(
    TRAFFIC_LOCID_OBJECT hLocID,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bRemoveFromGlobalFilter(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_OBJECT hDSRL,
    BOOLEAN *pbChanged
        );

static BOOLEAN bTargetFilterGenerator(
    DSRL_TARGET_OBJECT hTarget,
    TRAFCAM_TARGET_LIST_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bTargetMessageCheck(
    DSRL_TARGET_OBJECT hTarget,
    TRAFCAM_TARGET_MESSAGE_CHECK_ITERATOR *psIteratorArg
        );

static BOOLEAN bVerifyBufferSize(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    size_t tRequestedSize
        );

static N16 n16CompareByHash (
    TRAFFIC_MSG_OBJECT hTrafficMsg1,
    TRAFFIC_MSG_OBJECT hTrafficMsg2
        );

static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        );

static N16 n16FindTargetEntry(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        );

static TRAFCAM_BSA_FILTER_STRUCT *psCreateFilterItem(
    TRAFFIC_MARKET tMarket,
    TRAFFIC_BSA tBSA
        );

static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    TRAFCAM_APP_OBJECT_STRUCT *psObj
        );

static void vDSRLRemoveTarget(
    DSRL_OBJECT hDSRL,
    DSRL_TARGET_OBJECT hTargetObj
        );

static void vEntryRelease(
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static void vHandleDeleteList(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vProcessTimeout(
    TRAFCAM_MGR_OBJECT_STRUCT *psObj
        );

static void vReleaseGlobalFilterItem (
    TRAFCAM_BSA_FILTER_STRUCT *psFilterItem
        );


  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const TRAFFIC_CAMERAS_INTERFACE_STRUCT TRAFFIC_CAMERAS =
{
    /*.hStart = */       hStart,
    /*.vStop = */        (void (*) (TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService))DATASERVICE_IMPL_vStop,
    /*.eState = */       (DATASERVICE_STATE_ENUM (*) (TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */   (DATASERVICE_ERROR_CODE_ENUM (*) (TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService))DATASERVICE_IMPL_eErrorCode,
    /*.eGetReferenceDataVersion = */eGetReferenceDataVersion
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsTrafcamStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

const TRAFCAM_MGR_INTERFACE_STRUCT GsTrafcamMgr =
{
    /*.bReportNewImage = */bReportNewImage,
    /*.bReportUpdatedImage = */bReportUpdatedImage,
    /*.bReportRepeatedImage = */bReportRepeatedImage,
    /*.bReportRemovedImage = */bReportRemovedImage
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif /* _TRAFCAM_MGR_OBJ_H_ */
