/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
* PRIVATE HEADER
*
******************************************************************************/

/*********************************/
/** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _TEAM_OBJ_H_
#define _TEAM_OBJ_H_

/**************/
/** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "league_obj.h"
#include "sms_api.h"

/***************/
/** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define TEAM_OBJECT_NAME "TEAM"

#define MAX_NUM_MEMBER_LEAGUES (8)

#define CREATE_VERSION_TABLE \
    "create table TeamVersion ( " \
    "TableVersion int,  " \
    "TableComplete int  " \
    "); "

#define INSERT_VERSION_ROW \
    "INSERT INTO TeamVersion VALUES(-1,0);"

#define SELECT_VERSION "select * from TeamVersion;"

#define UPDATE_VERSION \
    "Update TeamVersion set " \
    "TableVersion = %d, "\
    "TableComplete = %d " \
    ";"

#define CREATE_TEAMS_TABLE \
    "create table Teams (  " \
    "TEAMID int, " \
    "Version int, " \
    "Abbr text, " \
    "Name text, " \
    "Nickname text, " \
    "LeagueTiers blob " \
    "); "

// remove obsolete teams from db
#define PRUNE_TEAMS "delete from Teams where (Version != %u)"

// get all teams
#define QUERY_FOR_ALL_TEAMS "select * from Teams order by TEAMID"

#define CREATE_TEAMS_INDEX \
    "create index if not exists teams_id_index ON Teams (TEAMID ASC);"

// retrieve teams with specified id
#define QUERY_TEAM_VERSION "select LeagueTiers," \
    "Version,ROWID from teams where (TEAMID = %u);"

// count teams with a specific table version number
#define COUNT_TEAMS "select count(*) from teams where (version = %u)"

// add a team to the db
#define INSERT_TEAM "insert into Teams "\
    "(TEAMID, Version, Abbr, Name, Nickname, LeagueTiers)"\
    " values ( ?, ?, ?, ?, ?, ?);"

// update a team entry
#define UPDATE_TEAM \
    "Update Teams set " \
    "TEAMID = ?, " \
    "Version = ?, " \
    "Abbr = ?, " \
    "Name = ?, " \
    "Nickname = ?, " \
    "LeagueTiers = ?, " \
    "where (ROWID = ?);"

#define TEAM_MAX_QUERY_LENGTH 500

/**************/
/** TYPEDEFS **/
/**************/

// the following fields are for processing information returned from a db query
typedef enum db_team_fields_enum {
    DB_TEAM_ID = 0,
    DB_TEAM_VERSION,
    DB_TEAM_ABBREV,
    DB_TEAM_NAME,
    DB_TEAM_NICKNAME,
    DB_TEAM_LEAGUES_TIERS,
    DB_TEAM_MAX_FIELDS
} DB_TEAM_FIELDS_ENUM;

typedef struct team_league_tier_struct
{
    UN8 un8LeagueId;
    UN8 un8Tier;
} TEAM_LEAGUE_TIER_STRUCT;

typedef struct team_struct
{
    // Team's abbreviation text
    STRING_OBJECT hAbbreviation;

    // Team's name text
    STRING_OBJECT hName;

    // Team's nickname text
    STRING_OBJECT hNickname;

    // Team's version
    N32 n32Version;
} TEAM_STRUCT;

/* object elements */

typedef struct team_object_struct
{
    // League membership
    LEAGUE_MEMBERSHIP_STRUCT sLeagueMembership;

    // Tier number assigned to each league
    TEAM_LEAGUE_TIER_STRUCT asLeagueTiers[MAX_NUM_MEMBER_LEAGUES];

    // Team's CID ( CID is a constant instance)
    CID_OBJECT hId;

    // Team information
    TEAM_STRUCT *psTeam;

    // Save flag
    BOOLEAN bSaved;

} TEAM_OBJECT_STRUCT;

typedef struct team_query_struct
{
    LEAGUE_OBJECT hLeague;
    UN32 un32LeagueId;
    UN32 un32TeamId;

} TEAM_QUERY_STRUCT;

typedef struct team_ctrl_struct
{
    // A handle to a linked list of available team which maps CIDs to
    // more verbose and abstracted team content to populate
    // an associated CDO with. Note that this list is dynamic and shared
    // amongst all SMS Decoders. Thus access to it must be protected.
    // To enforce this protection this structure is declared as an
    // SMS Object and the built-in locking mechanism is used to
    // limit exclusive access to this element.
    OSAL_OBJECT_HDL hTeamList;
    OSAL_OBJECT_HDL hTeamTrashBin;

    // Sports DB elements
    char *pacFileName;
    SQL_INTERFACE_OBJECT hSQLConnection;
    char acQueryBuffer[TEAM_MAX_QUERY_LENGTH];

    // Current version info
    N32 n32Version;
    BOOLEAN bComplete;

    // CID methods
    CID_ENUM eType;
    CID_OBJECT hWorkingLeagueId;
    CID_OBJECT hWorkingTeamId;

    // Dummy team object
    TEAM_OBJECT_STRUCT sDummyTeam;

} TEAM_CTRL_STRUCT;

typedef struct team_version_match_iterator_struct
{
    N32 n32Version;
    UN32 un32Matching;

} TEAM_VERSION_MATCH_ITERATOR_STRUCT;

typedef struct team_league_match_iterator_struct
{
    BOOLEAN bMatch;
    OSAL_OBJECT_HDL hLeagueList;

} TEAM_LEAGUE_MATCH_ITERATOR_STRUCT;

typedef struct team_print_iterator_struct
{
    FILE *psFile;
    size_t tNumTeams;

} TEAM_PRINT_ITERATOR_STRUCT;

typedef struct team_save_iterator_struct
{
    N32 n32Version;
    size_t tInsertCount;
    size_t tReplacedCount;
    size_t tAlreadyUpToDateCount;

} TEAM_SAVE_ITERATOR_STRUCT;

typedef struct team_version_query_struct
{
    TEAM_LEAGUE_TIER_STRUCT asLeagueTiers[MAX_NUM_MEMBER_LEAGUES];
    N32 n32Version;
    N64 n64RowId;
    BOOLEAN bFound;

} TEAM_VERSION_QUERY_STRUCT;

typedef struct team_row_struct
{
    UN32 un32TeamId;
    UN16 n32Version;
    const char *pacAbbrev;
    const char *pacName;
    const char *pacNickname;
    TEAM_LEAGUE_TIER_STRUCT asLeagueTiers[MAX_NUM_MEMBER_LEAGUES];
    N64 n64RowId;

} TEAM_ROW_STRUCT;

typedef struct team_remove_struct
{
    TEAM_OBJECT hTeam;
    BOOLEAN bSuccess;

} TEAM_REMOVE_STRUCT;

typedef struct team_league_iterator_struct
{
    LEAGUE_OBJECT hLeagueToFind;
    UN8 un8Index;
} TEAM_LEAGUE_ITERATOR_STRUCT;

/************/
/** MACROS **/
/************/

/****************/
/** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

#ifdef SXM_LEGACY_SSP
static TEAM_OBJECT hCreate(
    LEAGUE_ENUM eLeague,
    CID_OBJECT hTeamId,
    const char *pacAbbrev,
    const char *pacName,
    const char *pacNickname
    );
#endif

static STRING_OBJECT hName (
    TEAM_OBJECT hTeam
    );

static STRING_OBJECT hNickname (
    TEAM_OBJECT hTeam
    );

static STRING_OBJECT hAbbreviation (
    TEAM_OBJECT hTeam
    );

static CID_OBJECT hId (
    TEAM_OBJECT hTeam
    );

static BOOLEAN bIterateContent (
    LEAGUE_OBJECT hLeague,
    TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
    );

static N32 n32Version (
    TEAM_OBJECT hTeam
    );

static BOOLEAN bGetTableVersion (
    N32 *pn32Version,
    BOOLEAN *pbComplete
    );

static UN16 un16Id (
    TEAM_OBJECT hTeam
        );

static SMSAPI_RETURN_CODE_ENUM eLeagueMembership (
    TEAM_OBJECT hTeam,
    LEAGUE_MEMBERSHIP_STRUCT *psLeagueMembership
        );

static BOOLEAN bIsSportsFlashEligible (
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam
        );

/* Object Private Prototypes */

static BOOLEAN bCountMatchingVersion (
    TEAM_OBJECT_STRUCT const *psTeamObject,
    TEAM_VERSION_MATCH_ITERATOR_STRUCT *psMatching
    );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bProcessSelectTableVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bQueryTeamVersion (
    UN32 un32TeamId,
    TEAM_LEAGUE_TIER_STRUCT *psLeagueTier,
    N32 *pn32Version,
    N64 *pn64RowId
    );

static BOOLEAN bTeamVersionHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bPrepareTeamColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    TEAM_ROW_STRUCT *psTeamRow
    );

static BOOLEAN bTeamTableHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bPrune(N32 n32Version);

static BOOLEAN bSaveTeam (
    TEAM_OBJECT_STRUCT *psTeamObject,
    TEAM_SAVE_ITERATOR_STRUCT *psTeamSaveIterator
    );

static TEAM_OBJECT_STRUCT *psCreateTeamObject (
    CID_OBJECT hId,
    N32 n32Version,
    char const *pacName,
    char const *pacNickname,
    char const *pacAbbrev,
    size_t tNumLeagues,
    UN8 const *paun8Leagues
        );

static void vDestroyTeamObject (
    TEAM_OBJECT_STRUCT *psTeamObject
    );

static N16 n16Compare (
    TEAM_OBJECT_STRUCT const *psTeamObject1,
    TEAM_OBJECT_STRUCT const *psTeamObject2
    );

static BOOLEAN bSetLeagueMembershipBit (
    LEAGUE_MEMBERSHIP_STRUCT *psLeagueMembership,
    UN8 un8LeagueId
    );

static BOOLEAN bPruneTeamsFromLeague(
    LEAGUE_OBJECT hLeague,
    void *pvArg
        );

/***************/
/** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const TEAM_OBJECT_INTERFACE_STRUCT TEAM =
{
#ifdef SXM_LEGACY_SSP
    /*.hCreate = */hCreate,
#endif
    /*.hName = */hName,
    /*.hNickname = */hNickname,
    /*.hAbbreviation = */hAbbreviation,
    /*.hId = */hId,
    /*.bIterateContent = */bIterateContent,
    /*.n32Version = */n32Version,
    /*.bGetTableVersion = */bGetTableVersion,
    /*.un16Id = */un16Id,
    /*.eLeagueMembership = */eLeagueMembership,
    /*.bIsSportsFlashEligible = */bIsSportsFlashEligible

};

// Team object control structure initialization
static TEAM_CTRL_STRUCT *gpsTeamCtrl = NULL;

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif	// _TEAM_OBJ_H_
