/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _STRING_OBJ_H_
#define _STRING_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "string_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define STRING_OBJECT_NAME "STRING"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef enum string_creator_enum
{
    STRING_CREATOR_SMS,
    STRING_CREATOR_APPLICATION

} STRING_CREATOR_ENUM;

typedef struct string_block_struct
{
    // The capacity allocated for character storage in this block
    size_t tCapacity;

    // The number of byte contained in this string block.
    // Note: the actual string content is placed after this structure
    // in memory.
    size_t tSize;

    // A pointer to store the actual address of the location
    // of a constant string's contents.
    char *pacString;

    // A pointer to the next string block. NULL if this is the last one.
    struct string_block_struct *psNext;

} STRING_BLOCK_STRUCT;

typedef struct string_object_struct
{
    // Object creator (SMS or App)
    STRING_CREATOR_ENUM eCreator;

    // Computed hash code
    N32 n32HashCode;

    // A single-linked list of string blocks (characters) which make up
    // this STRING object's contents. This element must absolutely be the
    // LAST element defined in this structure.
    STRING_BLOCK_STRUCT sBlock;

} STRING_OBJECT_STRUCT;

  /************/
 /** MACROS **/
/************/

// Defines encoded sequence mark symbol
#define STRING_ENCODED_SEQUENCE_MARK '&'

// Defines leng of the ecoded sequence w/o mark
#define STRING_ENCODED_SEQUENCE_LEN 2

// Detects special symbol which should be encoded
#define STRING_IS_SPECIAL_SYMBOL(Symbol)       \
   (                                           \
    (((Symbol >= 0x00) && (Symbol <= 0x1F)) || \
     ((Symbol >= 0x7E) && (Symbol <= 0xA0)) || \
     (Symbol == '>') || (Symbol == '<')     || \
     (Symbol == STRING_ENCODED_SEQUENCE_MARK)  \
    ) ? TRUE : FALSE)

// Defines English alphabet hash formula multiplier
#define STRING_HASH_MULTIPLIER ((UN32)(31))

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static STRING_OBJECT hCreate (
    const char *pacString,
    size_t tMinimumLength
        );

static size_t tCopy (
    STRING_OBJECT hSrcString,
    STRING_OBJECT hDstString
        );

static size_t tCopyToCStr (
    STRING_OBJECT hString,
    char *pacDst,
    size_t tDstSize
        );

static const char *pacCStr (
    STRING_OBJECT hString
        );

static size_t tSize (
    STRING_OBJECT hString
        );

static N16 n16Compare (
    STRING_OBJECT hString1,
    STRING_OBJECT hString2,
    BOOLEAN bBinary
        );

static N16 n16CaseCompare (
    STRING_OBJECT hString1,
    STRING_OBJECT hString2
    );

static N16 n16CompareCStr (
    const char *pacString,
    size_t tLength,
    STRING_OBJECT hString
        );

static N16 n16SubstringCStr (
    STRING_OBJECT hOrigString,
    const char *pacSubString,
    BOOLEAN bCaseInsensitive
        );

static STRING_OBJECT hDuplicate (
    STRING_OBJECT hString
        );

static N32 n32FWrite (
    STRING_OBJECT hString,
    FILE *psFile
        );

static STRING_OBJECT hFRead (
    FILE *psFile
        );

static N32 n32FPrintf (
    STRING_OBJECT hString,
    FILE *psFile
        );

static BOOLEAN bModifyCStr (
    STRING_OBJECT hString,
    const char *pacSrc
        );

static size_t tAppendString (
    STRING_OBJECT hSrcString,
    STRING_OBJECT hDstString
        );

static size_t tAppendCStr (
    STRING_OBJECT hString,
    const char *pacSrc
        );

static void vDestroy (
    STRING_OBJECT hString
        );

/* Object Private Prototypes */

static N16 n16LocalStringCompare(
    STRING_OBJECT hString1,
    STRING_OBJECT hString2,
    BOOLEAN bBinary,
    BOOLEAN bCaseSensitive
        );

static const char *pacCreatorText(
    STRING_CREATOR_ENUM eCreator
        );

static N32 n32ComputeHashCode (
    const STRING_BLOCK_STRUCT *psBlock
        );

static void vStatistics (
    const STRING_BLOCK_STRUCT *psBlock,
    size_t *ptSize,
    size_t *ptCapacity,
    size_t *ptNumBlocks
        );

static BOOLEAN bCopyCStrLocal (
    STRING_OBJECT hString,
    const char *pacSrc,
    size_t *ptCopied,
    BOOLEAN bOverwrite
        );

static size_t tCopyLocal (
    STRING_OBJECT hDstString,
    STRING_OBJECT hSrcString,
    BOOLEAN bOverwrite
        );

static BOOLEAN bHasSpecialSymbols (
    const STRING_BLOCK_STRUCT *psBlock
        );

static signed char cHexDigitToChar (
    const char cHexDigit
        );

static void vDestroyBlocks (
    STRING_OBJECT_STRUCT *psObj,
    STRING_BLOCK_STRUCT *psStartBlock
        );

static char cDecodeSymbol(
    const char *pcSrcStr
        );
  /***************/
 /** VARIABLES **/
/***************/

// Local block instance for handling NULL STRINGs
const STRING_BLOCK_STRUCT gsEmptyBlock =
{
    /*.tCapacity = */0,
    /*.tSize = */0,
    /*.pacString = */NULL,
    /*.psNext = */NULL
};

// Global (re-usable) instance of an interface for this object
const STRING_OBJECT_INTERFACE_STRUCT STRING =
{
    /*.hCreate = */hCreate,
    /*.tCopy = */tCopy,
    /*.tCopyToCStr = */tCopyToCStr,
    /*.pacCStr = */pacCStr,
    /*.tSize = */tSize,
    /*.n16Compare = */n16Compare,
    /*.n16CaseCompare = */n16CaseCompare,
    /*.n16CompareCStr = */n16CompareCStr,
    /*.n16SubstringCStr = */n16SubstringCStr,
    /*.hDuplicate = */hDuplicate,
    /*.n32FWrite = */n32FWrite,
    /*.hFRead = */hFRead,
    /*.n32FPrintf = */n32FPrintf,
    /*.bModifyCStr = */bModifyCStr,
    /*.tAppendString = */tAppendString,
    /*.tAppendCStr = */tAppendCStr,
    /*.vDestroy = */vDestroy

};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _STRING_OBJ_H_
