/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SPORTS_SERVICE_MGR_OBJ_H_
#define _SPORTS_SERVICE_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "sms.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sports_service_mgr_obj.h"
#include "sports_service_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

typedef struct sports_service_table_class_info_struct
{
    UN8 un8IsValid;
    UN8 un8SaveDur;
    UN8 un8SaveRule;
    UN8 un8ShutdownRule;
    UN32 un32ReceiptDur;
} SPORTS_SERVICE_TABLE_CLASS_INFO_STRUCT;

typedef enum ss_save_rule_enum {
    SS_SAVE_RULE_KEEP_ALL = 0,
    SS_SAVE_RULE_KEEP_ONE
} SS_SAVE_RULE_ENUM;

typedef enum ss_shutdown_rule_enum {
    SS_SHUTDOWN_RULE_DROP_CURRENT_DAY = 0,
    SS_SHUTDOWN_RULE_NONE
} SS_SHUTDOWN_RULE_ENUM;

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

#define SCRATCH_PAD_INCREMENT_SIZE 1024*16
#define MAX_SCRATCH_PAD_INCREMENTS 16
// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define SS_SUPPORTED_OPTIONS ( \
    (DATASERVICE_OPTIONS_MASK) \
    DATASERVICE_OPTION_NONE \
    )

#define SS_MAX_AFFILIATE_ID 1023
#define SS_MAX_AFFILIATE_COUNT (SS_MAX_AFFILIATE_ID + 1)

#define SS_MAX_TCLASS_ID 15
#define SS_MAX_TCLASS_COUNT (SS_MAX_TCLASS_ID + 1)

#define SS_MAX_SPORT_ID 255
#define SS_MAX_SPORT_COUNT (SS_MAX_SPORT_ID + 1)

// Results in bit field stack variable with this many bits.
#define SS_MAX_LABEL_TYPE (255)
#define SS_MAX_LABEL_COUNT (SS_MAX_LABEL_TYPE + 1)

// Limit the affiliate hierarchy to 4 levels.
#define SS_MAX_AFFILIATE_DELIMITER_COUNT (3)

#define SS_SECONDS_PER_HOUR (3600)
#define SS_SECONDS_PER_DAY (24*SS_SECONDS_PER_HOUR)
#define SS_TWO_HOURS_IN_SECONDS (2*SS_SECONDS_PER_HOUR)

#define SS_BACKGROUND_EVENT_INTERVAL_SECONDS (60*10)

// Use this for testing to accelerate operations.
//#define SS_BACKGROUND_EVENT_INTERVAL_SECONDS (60*1)

#define SS_BACKGROUND_EXECUTE_PRUNE_BIAS (60*30)

/* Object name prefix for objects */
#define SPORTS_SERVICE_MGR_OBJECT_NAME "SportsService"

/* Sports Service Directory Name */
#define SPORTS_SERVICE_DIRECTORY_NAME "SportsService"

// Constant which defines the size of the buffer used by
// sports service for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define SPORTS_SERVICE_SHARED_BUFFER_LEN \
    ( SPORTS_SERVICE_MAX_SQL_STRING_LENGTH )

/* XML output constants */
#define SS_TAG_SPORT_LIST_OPEN                     "<sl>"
#define SS_TAG_SPORT_LIST_CLOSE                    "</sl>"

#define SS_TAG_SPORT_LIST_ENTRY_OPEN             "<sle>"
#define SS_TAG_SPORT_LIST_ENTRY_CLOSE            "</sle>"

#define SS_TAG_AFFILIATE_LIST_OPEN                 "<afl>"
#define SS_TAG_AFFILIATE_LIST_CLOSE                "</afl>"

#define SS_TAG_AFFILIATE_OPEN                     "<af>"
#define SS_TAG_AFFILIATE_CLOSE                    "</af>"

#define SS_TAG_NAME_OPEN                         "<nm>"
#define SS_TAG_NAME_CLOSE                        "</nm>"

#define SS_TAG_LEAGUE_ID_OPEN                     "<lid>"
#define SS_TAG_LEAGUE_ID_CLOSE                    "</lid>"

#define SS_TAG_SPORT_ID_OPEN                     "<sid>"
#define SS_TAG_SPORT_ID_CLOSE                    "</sid>"

#define SS_TAG_INFORMATION_CLASS_OPEN             "<ic>"
#define SS_TAG_INFORMATION_CLASS_CLOSE            "</ic>"

#define SS_TAG_EPOCH_OPEN                         "<ep>"
#define SS_TAG_EPOCH_CLOSE                        "</ep>"

#define SS_TAG_SEASON_STATUS_OPEN                 "<ss>"
#define SS_TAG_SEASON_STATUS_CLOSE                "</ss>"

#define SS_TAG_INFORMATION_CLASS_LIST_OPEN         "<icl>"
#define SS_TAG_INFORMATION_CLASS_LIST_CLOSE        "</icl>"

#define SS_TAG_TABLE_LIST_OPEN                     "<tl>"
#define SS_TAG_TABLE_LIST_CLOSE                    "</tl>"

#define SS_TAG_TABLE_LIST_ENTRY_OPEN             "<tle>"
#define SS_TAG_TABLE_LIST_ENTRY_CLOSE            "</tle>"

#define SS_TAG_TABLE_HEADER_OPEN                 "<thead>"
#define SS_TAG_TABLE_HEADER_CLOSE                "</thead>"
#define SS_TAG_TABLE_HEADER_OPEN_MAX_SIZE    \
    sizeof(SS_TAG_TABLE_HEADER_OPEN) + \
    sizeof(SS_TAG_ID_OPEN) + \
    sizeof(SS_TAG_ID_CLOSE) + \
    12

#define SS_TAG_TABLE_ROW_OPEN                     "<r>"
#define SS_TAG_TABLE_ROW_CLOSE                    "</r>"

#define SS_TAG_TABLE_ROW_DATA_OPEN                 "<d>"
#define SS_TAG_TABLE_ROW_DATA_CLOSE                "</d>"

#define SS_TAG_TABLE_OPEN                         "<table>"
#define SS_TAG_TABLE_CLOSE                        "</table>"

#define SS_TAG_COLUMN_DESC_OPEN                 "<cd>"
#define SS_TAG_COLUMN_DESC_CLOSE                 "</cd>"

#define SS_TAG_ID_OPEN                          "<id>"
#define SS_TAG_ID_CLOSE                         "</id>"

#define SS_TAG_LABEL_OPEN                       "<lb>"
#define SS_TAG_LABEL_CLOSE                      "</lb>"

#define SS_TAG_PRIORITY_OPEN                    "<pr>"
#define SS_TAG_PRIORITY_CLOSE                   "</pr>"

#define SS_TAG_DMOD_OPEN                        "<dm>"
#define SS_TAG_DMOD_CLOSE                       "</dm>"

#define SS_TAG_BINARY_DATA_OPEN                 "<bd>"
#define SS_TAG_BINARY_DATA_CLOSE                "</bd>"

#define SS_TAG_RECEIPT_TIME_OPEN                "<rt>"
#define SS_TAG_RECEIPT_TIME_CLOSE               "</rt>"

#define SS_TAG_HEADER_KEY_OPEN                  "<hk>"
#define SS_TAG_HEADER_KEY_CLOSE                 "</hk>"

#define SS_TAG_LBSIZE_OPEN                      "<ls>"
#define SS_TAG_LBSIZE_CLOSE                     "</ls>"

#define SS_TAG_OVERRIDE_LIST_OPEN               "<ol>"
#define SS_TAG_OVERRIDE_LIST_CLOSE              "</ol>"

#define SS_TAG_OVERRIDE_LIST_ENTRY_OPEN         "<ole>"
#define SS_TAG_OVERRIDE_LIST_ENTRY_CLOSE        "</ole>"

// If 0 only the sports listed in the supported sports array are supported
#define SS_SUPPORT_ALL_SPORTS (1)

// All published sports as of 2011_10_06.
static UN8 gaun8SupportedSport[] =
{
//    1
        0,1,2,3,4,5,6,7,8
};

// Supported table classes
#if 1
// All classes per pvn1
static UN32 gun32SupportedTableClass =  0
                            | (1<<INFO_CLASS_HEAD_TO_HEAD_EVENT)
                            | (1<<INFO_CLASS_RANKED_LIST_EVENT)
                            | (1<<INFO_CLASS_SEASON_SUMMARY)
                            | (1<<INFO_CLASS_NEWS_ITEM)
                            | (1<<INFO_CLASS_PARTICIPANT_STATISTICS)
                            | (1<<INFO_CLASS_PARTICIPANT_INDEX)
                            | (1<<INFO_CLASS_EVENT_DESCRIPTION);
#else
// All classes per published sports and priority 0 labels as of 2011_10_06.
static UN32 gun32SupportedTableClass =  0
                            | (1<<INFO_CLASS_HEAD_TO_HEAD_EVENT)
                            | (1<<INFO_CLASS_RANKED_LIST_EVENT)
//                            | (1<<INFO_CLASS_SEASON_SUMMARY)
                            | (1<<INFO_CLASS_NEWS_ITEM)
                            | (1<<INFO_CLASS_PARTICIPANT_STATISTICS)
                            | (1<<INFO_CLASS_PARTICIPANT_INDEX);
//                            | (1<<INFO_CLASS_EVENT_DESCRIPTION);
#endif

//#define SS_RECEIPT_KEEP_DURATION_SEC (60*5)
#define SS_RECEIPT_KEEP_DURATION_SEC (24*SS_SECONDS_PER_HOUR)

// Traits of all table classes.
static SPORTS_SERVICE_TABLE_CLASS_INFO_STRUCT gsClassInfo[SS_MAX_TCLASS_COUNT] =
{
//  {Valid, SaveDur, SaveRule, ShutDownRule, ReceiptDur,}
    {1, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_DROP_CURRENT_DAY, SS_RECEIPT_KEEP_DURATION_SEC},   //INFO_CLASS_HEAD_TO_HEAD_EVENT
    {1, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_DROP_CURRENT_DAY, SS_RECEIPT_KEEP_DURATION_SEC},   //INFO_CLASS_RANKED_LIST_EVENT
    {1, 2, SS_SAVE_RULE_KEEP_ONE, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //INFO_CLASS_SEASON_SUMMARY
    {1, 2, SS_SAVE_RULE_KEEP_ONE, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //INFO_CLASS_NEWS_ITEM

    {1, 2, SS_SAVE_RULE_KEEP_ONE, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //INFO_CLASS_PARTICIPANT_STATISTICS
    {1, 2, SS_SAVE_RULE_KEEP_ONE, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //INFO_CLASS_PARTICIPANT_INDEX
    {1, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_DROP_CURRENT_DAY, SS_RECEIPT_KEEP_DURATION_SEC},   //INFO_CLASS_EVENT_DESCRIPTION
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED

    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED

    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
    {0, 8, SS_SAVE_RULE_KEEP_ALL, SS_SHUTDOWN_RULE_NONE, SS_RECEIPT_KEEP_DURATION_SEC},               //UNUSED
};

// Make 0xff to support all 8 priorities.
#define SS_SUPPORTED_PRIORITY_INITIALIZER (0xff)

// All supported label priorities.
static UN32 gun32SupportedLabelPriority = SS_SUPPORTED_PRIORITY_INITIALIZER
                            | (1<<0);
//                            | (1<<1);

// If 0 only the labels listed in the supported labels array are supported.
// Otherwise all pvn1 defined labels are supported.
#define SS_SUPPORT_ALL_LABELS (1)
#define SS_PVN1_LAST_DEFINED_LABEL_TYPE (0x25)

COMPILE_TIME_ASSERT(SS_PVN1_LAST_DEFINED_LABEL_TYPE <= SS_MAX_LABEL_TYPE,
    PVN1_Last_Label_Defined_exceeds_max_label_id );

// All priority 0 labels for all sports as of 2011_10_06.
// You must always support labels 0x0, 0x1, 0x1c, 0x25.  They require special
// parsing.  They cannot be skipped over generically.   0x1c is the only special
// parsing label not used for priority 0 but is included here.
static UN8 gaun8SupportedLabel[] =
{
        0x00,
        0x01,
        0x02,
        0x03,
        0x04,
        0x05,
        0x06,
        0x08,
        0x09,
        0x0a,
        0x0e,
        0x16,
        0x17,
        0x18,
        0x19,
        0x1c,   // required to properly parse.  Not used by priority 0
        0x1e,
        0x1f,
        0x24,
        0x25,
};

static const char *gapcDbBuildStrings[SS_BUILD_DB_STATEMENT_COUNT] =
{
    SS_BUILD_DB_CREATE_TABLE_1,
    SS_BUILD_DB_CREATE_TABLE_2,
    SS_BUILD_DB_CREATE_TABLE_3,
    SS_BUILD_DB_CREATE_TABLE_4,
    SS_BUILD_DB_CREATE_TABLE_5,
    SS_BUILD_DB_CREATE_TABLE_6,
    SS_BUILD_DB_CREATE_TABLE_7,
    SS_BUILD_DB_CREATE_TABLE_8,
    SS_BUILD_DB_CREATE_TABLE_9,
    SS_BUILD_DB_CREATE_TABLE_10,
    SS_BUILD_DB_SET_VERSION,
    SS_BUILD_DB_CREATE_INDEX_1,
    SS_BUILD_DB_CREATE_INDEX_2,
    SS_BUILD_DB_CREATE_INDEX_3,
    SS_BUILD_DB_CREATE_INDEX_4,
    SS_BUILD_DB_CREATE_INDEX_5
};

  /************/
 /** MACROS **/
/************/
// round up the the next highest 8, then divide by 8.
#define SS_BYTES_FOR_BITS(n_) (((n_) + 7)>>3)
#define SS_BITS_FOR_BYTES(n_) ((n_)<<3)

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */
typedef struct sports_service_mgr_config_struct
{
    // holds current config information as
    // retrieved from the stream (CarId = 0)
    // If version or crc do not match the OTA stream then we
    // must discard the database and re-acquire.

    // System config version
    UN8 un8ConfigVersion;

    // Crc of OTA SC_Config data (for detecting change)
    UN32 un32ConfigCrc;

    // Helper param for the db query to get the values.
    BOOLEAN bSuccess;
} SPORTS_SERVICE_MGR_CONFIG_STRUCT;

typedef struct sports_service_af_config_struct
{
    // holds current affiliate information as
    // retrieved from the stream (CarId = 1)
    // If version or crc do not match the OTA stream then we
    // must discard the data and re-acquire.

    // System config version
    UN8 un8AfCfgVersion;

    // Crc of OTA SC_Config data (for detecting change)
    UN32 un32AfConfigCrc;

    // Helper param for the db query to get the values.
    BOOLEAN bSuccess;
} SPORTS_SERVICE_AF_CONFIG_STRUCT;

typedef struct sports_service_af_update_struct
{
    UN8 aun8AfRemoveBitfield[SS_BYTES_FOR_BITS(SS_MAX_AFFILIATE_COUNT)];
    UN8 aun8AfNameChangeBitfield[SS_BYTES_FOR_BITS(SS_MAX_AFFILIATE_COUNT)];
    BOOLEAN bNewAfAdded;
} SPORTS_SERVICE_AF_UPDATE_STRUCT;

typedef struct sports_service_prune_ctrl_struct
{
    UN32 un32Epoch;
    BOOLEAN bRetryData;
    BOOLEAN bRetryTdef;
} SPORTS_SERVICE_PRUNE_CTRL_STRUCT;

typedef struct sports_service_table_list_entry_struct
{
    TABLE_ID tInstId;
    UN32 un32SeasonStatus;
    UN32 un32Epoch;
    UN32 un32ReceiptTime;
    UN32 un32TdefKey;
    BOOLEAN bTitlePresent;
    char acTitle[SS_MAX_TABLE_TITLE_SYMBOL_COUNT];
} SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT;

typedef struct sports_service_mgr_object_struct
{
    UN32 un32DbConfigVersion;

    // System Config info
    SPORTS_SERVICE_MGR_CONFIG_STRUCT sSC_Config;

    // Affiliate config info
    SPORTS_SERVICE_AF_CONFIG_STRUCT sAF_Config;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The handle to the sports service interface
    SPORTS_SERVICE_INTERFACE_OBJECT hInterface;

    // DBMS connection handle
    SQL_INTERFACE_OBJECT hSQLConnection;

    // Sports Service File Paths
    char *pacFullyQualifiedDatabaseFilePath;

    // Buffer used for application SQL query commands
    char acAppQueryBuf[SPORTS_SERVICE_SHARED_BUFFER_LEN+1];

    // Buffer used for internal SQL commands.
    char acInternalSqlBuf[SPORTS_SERVICE_SHARED_BUFFER_LEN+1];

    // Flag indicating if the manager
    // is currently in an SQL transaction
    BOOLEAN bInTransaction;

    // Semaphore to control access to the read side of the db
    OSAL_OBJECT_HDL hDbReadSem;

    // Flag to indicate the data set is in use
    BOOLEAN bDataSetInUse;

    // Flag to indicate we are processing monitors
    BOOLEAN bMonitorProcessing;

    // Flag to lock out user data set access during a config update.
    BOOLEAN bLockOutDueToConfigUpdate;

    // List of all attached monitors.
    OSAL_OBJECT_HDL hMonitorList;

    // Semaphore to control access to monitor list.
    OSAL_OBJECT_HDL hMonitorListSem;

    // Structure to help update the affiliate table.
    SPORTS_SERVICE_AF_UPDATE_STRUCT sAfUpdate;

    // Structure for all info of known table classes.
    SPORTS_SERVICE_TABLE_CLASS_INFO_STRUCT asClassInfo[SS_MAX_TCLASS_COUNT];

    // Periodic event for general usage
    DATASERVICE_TIMED_EVENT_HDL hBackgroundEvent;

    // Structure for managing db pruning of stale data and structure.
    SPORTS_SERVICE_PRUNE_CTRL_STRUCT sPruneCtrl;

    // Structures for what is supported.  Initialized from constants. Intended
    // to be overwritten by user config API if one is added in the future.
    UN8 aun8SupportedSport[SS_BYTES_FOR_BITS(SS_MAX_SPORT_COUNT)];
    UN8 aun8SupportedLabel[SS_BYTES_FOR_BITS(SS_MAX_LABEL_COUNT)];
    UN32 un32SupportedTableClass;
    UN32 un32SupportedLabelPriority;

    // Page size of the current persistent db.
    BOOLEAN bHavePersistentDbPageSize;
    UN32 un32PageSize;

    char *pacScratchPad;
    UN32 un32ScratchPadSize;

} SPORTS_SERVICE_MGR_OBJECT_STRUCT;

// Structure used to iterate the affiliate list.
typedef struct affiliate_change_iterator_struct
{
    AFFILIATE_ID tAffiliateId;
    SPORTS_MONITOR_EVENT_MASK tChangeBits;

} AFFILIATE_CHANGE_ITERATOR_STRUCT;

// Structure used to manage formatting the results of a sql query
typedef struct sports_service_query_result_struct
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj;
    UN32 un32CarryOver;
    UN8 *pun8ColumnMap;
    UN32 un32ColumnMapSize;
    UN32 un32ExpectedColumnCount;
    char *pacResultBuffer;
    N32 n32ResultBufferSize;
    char *pacCursor;
    UN32 un32RowCount;
    UN32 un32RequiredSize;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
    UN32 *pun32RequiredSize;
    char *pacScratchPad;
    UN32 un32ScratchPadSize;
    UN32 un32LastRowStartPos;

} SPORTS_SERVICE_QUERY_RESULT_STRUCT;

typedef struct sports_service_instance_id_query_result_struct
{
    N64  n64InstanceId;
    UN32 un32Crc;
    UN32 un32DefKey;
    UN8  un8TableClass;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT;

typedef struct sports_service_afid_query_result_struct
{
    UN16 un16AfId;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_AFID_QUERY_RESULT_STRUCT;

typedef struct sports_service_tref_remove_struct
{
    UN32 un32TdefKey;
    UN32 un32AfId;
    N64 n64IntanceId;

} SPORTS_SERVICE_TREF_REMOVE_STRUCT;

typedef struct sports_service_tref_remove_query_struct
{
    SPORTS_SERVICE_TREF_REMOVE_STRUCT *psRemoveInfo;
    UN32 un32RowCount;
    UN32 un32MaxRowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_TREF_REMOVE_QUERY_STRUCT;

typedef struct sports_service_un32_pragma_result_struct
{
    UN32 un32Value;
    UN32 un32RowCount;
    BOOLEAN bSuccess;
} SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT;

typedef struct sports_service_cstring_pragma_result_struct
{
    char *pacBuffer;
    UN32 un32BufferSize;
    UN32 un32RowCount;
    BOOLEAN bSuccess;
} SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT;

typedef struct sports_service_row_id_query_result_struct
{
    N64 n64LastRowId;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
} SPORTS_SERVICE_ROW_ID_QUERY_RESULT_STRUCT;

/* General typedefs */

typedef struct sports_service_mgr_default_by_type_struct
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bCategory;
    BOOLEAN bEventAdded;

} SPORTS_SERVICE_MGR_DEFAULT_BY_TYPE_STRUCT;

typedef struct sports_service_all_afid_query_result_struct
{
    UN8* pun8BitField;
    UN32 un32BitFieldSize;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_ALL_AFID_QUERY_RESULT_STRUCT;

typedef struct sports_service_af_name_change_query_result_struct
{
    STRING_OBJECT hSportAffiliation;
    BOOLEAN bNameChanged;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_AF_NAME_CHANGE_QUERY_RESULT_STRUCT;

typedef struct sports_service_tdef_query_result_struct
{
    OSAL_OBJECT_HDL hAddList;
    OSAL_OBJECT_HDL hRemoveList;
    N32 n32CurrentTdefId;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT;

typedef struct sports_service_column_cull_query_struct
{
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj;
    UN8* pun8ColumnMap;
    UN32 un32ColumnMapSize;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;

} SPORTS_SERVICE_COLUMN_CULL_QUERY_STRUCT;

typedef struct sports_service_table_list_query_struct
{
    OSAL_OBJECT_HDL hTableList;
    UN32 un32RowCount;
    SMSAPI_RETURN_CODE_ENUM eSmsApiCode;
} SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT;

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SPORTS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

static void vStop (
    SPORTS_SERVICE_OBJECT hSportsDataService
        );

static SMSAPI_RETURN_CODE_ENUM eUseDataSet (
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_ACCESS_CALLBACK vCallback,
    void *pvCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eGetSportList (
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcDest,
    UN32 *pun32DestSize
          );

static SMSAPI_RETURN_CODE_ENUM eGetRootAffiliateList (
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetAffiliateChildren (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetAffiliateInfoClassList (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetAffiliateTableList (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    INFORMATION_CLASS tInfoClass,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetAffiliate (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetTable (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        );

static SMSAPI_RETURN_CODE_ENUM eGetTableHeader (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize
        );

static SMSAPI_RETURN_CODE_ENUM eGetTableByColumnMatch (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 tColumnId,
    char *pcValue,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        );

static SMSAPI_RETURN_CODE_ENUM eGetTableByColumnRange (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 un32ColumnId,
    UN32 un32Value1,
    UN32 un32Value2,
    SORT_TYPE_ENUM eSortType,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        );

static SMSAPI_RETURN_CODE_ENUM eGetTableByTeamPresence (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    UN32 un32ColumnOneId,
    UN32 un32ColumnTwoId,
    UN32 un32TeamId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        );

static SMSAPI_RETURN_CODE_ENUM eGetReferencedTable (
    SPORTS_SERVICE_OBJECT hSportsService,
    REFERENCED_TABLE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize,
    UN32 un32RowOffset
        );

static SMSAPI_RETURN_CODE_ENUM eGetReferencedTableHeader (
    SPORTS_SERVICE_OBJECT hSportsService,
    REFERENCED_TABLE_ID tId,
    char *pcDest,
    UN32 *pun32DestSize
        );

static CID_OBJECT hCreateLeagueCid( UN32 un32LeagueId );

static CID_OBJECT hCreateTeamCid( UN32 un32LeagueId, UN32 un32TeamId );

/* Object Private Prototypes */
static BOOLEAN bBeginTableUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    BOOLEAN bIsPrimaryTable,
    char *acSavePointName
        );

static BOOLEAN bEndTableUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        );

static BOOLEAN bBeginRowUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        );

static BOOLEAN bEndRowUpdate(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction
        );

static BOOLEAN bUpdateColumnInt(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    int iColumn,
    UN8 un8LPriority,
    UN8 un8LType,
    int iValue
        );

static BOOLEAN bUpdateColumnString(
    SPORTS_SERVICE_OBJECT hSportsService,
    SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
    int iColumn,
    UN8 un8LPriority,
    UN8 un8LType,
    STRING_OBJECT hString
        );

static BOOLEAN bUnsafeUpdateTrefRelationships (
                SPORTS_SERVICE_OBJECT hSportsService,
                UN32 un32TdefKey,
                UN8 un8LabelId,
                TABLE_ID tChildInstanceId,
                N64 n64RowId,
                TABLE_ID tParentInstanceId,
                UN8 un8LPriority
        );

static BOOLEAN bUnsafeAddLabelOverride(
        SPORTS_SERVICE_OBJECT hSportsService,
        TABLE_ID tTableId,
        UN32 un32LabelId,
        UN32 un32Override
        );

static BOOLEAN bDbgDumpTable(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcTableName
        );

static BOOLEAN bDbgExecuteSqlCmd(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *SQLStatement
        );

static BOOLEAN bDbgDumpAllTableNames(
    SPORTS_SERVICE_OBJECT hSportsService
        );

static BOOLEAN bDbgDumpDefs(
    SPORTS_SERVICE_OBJECT hSportsService,
    char *pcDefInfo
        );

static BOOLEAN bStartServiceConfigUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer,
        UN32 un32Crc
        );

static BOOLEAN bEndServiceConfigUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        BOOLEAN bUpdateOk,
        UN8 un8CfgVer,
        UN32 un32Crc
        );

static BOOLEAN bIsCorrectServiceConfigVersion(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer
        );

static void vDebugSetServiceConfigVersion(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN8 un8CfgVer
        );

static BOOLEAN bAddSportsNameToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        STRING_OBJECT hSportsName,
        UN8 un8SportId
        );

static BOOLEAN bAddTdefSportMappingToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8SportId
        );

static BOOLEAN bShouldKeepTdef(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TdefClass
        );

static BOOLEAN bUnsafeGetParentAffiliateIdFromString(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        STRING_OBJECT hSportAffiliation,
        UN8 un8SportID,
        BOOLEAN *bIsChildAffiliate,
        UN16 *pUn16AfId
        );

static BOOLEAN bProcessGetParentAfIdResult(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_AFID_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bAddTableDefToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_HIGHBAND_TDEF_INFO_STRUCT *TdefInfo,
        UN16 un16TableID
        );

static BOOLEAN bDeleteTableDefFromDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TabVer,
        BOOLEAN bEraseUnderlyingData
        );

static BOOLEAN bUnsafeDeleteTableDefFromDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        UN16 un16TableID,
        UN8 un8TabVer,
        BOOLEAN bDropTable,
        BOOLEAN bDeleteInstanceEntries
        );

static BOOLEAN bStartSportAffiliationUpdate(
          SPORTS_SERVICE_OBJECT hSportsService,
         UN8 un8AdVer,
         UN32 un32CRC);

static BOOLEAN bUnsafeAddSportsAffiliationToDB(
        SPORTS_SERVICE_OBJECT hSportsService,
        STRING_OBJECT hSportAffiliation,
        UN16 un16AffiliateId,
        BOOLEAN bPresBit,
        UN8 un8GDRef,
        UN8 un8SportID
        );

static BOOLEAN bEndSportAffiliationUpdate(
        SPORTS_SERVICE_OBJECT hSportsService,
        BOOLEAN bUpdateOk,
        UN8 un8AdVer,
        UN32 un32CRC);

static void vReleasePayload(
        OSAL_BUFFER_HDL hPayload
        );

/* General */
static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bHandleServiceReady (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceStopped (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vSaveMemoryDb (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static DATASERVICE_ERROR_CODE_ENUM eHandleNewDataEvent (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static void vUninitObject (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bBuildDBFilePath (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUnsafeLoadSportsConfigDataFromDB(
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN *pbError
        );

static void vSetError (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

/* Query helper functions */

static SMSAPI_RETURN_CODE_ENUM eCheckDataSetUsage (
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pcDest,
        UN32 *pun32DestSize );

static BOOLEAN bPerformQuery (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcOpenTag,
    char *pcCloseTag,
    char *pcQueryString,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *ptResult,
    SQL_QUERY_RESULT_HANDLER tRowHandler );

static BOOLEAN bExecuteCommandWithString(
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacCommandString,
        char *pacTemplateString );

static BOOLEAN bPrepareCommandWithString (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    char *pacTemplateString );

static BOOLEAN bUnsafeGetDbPageSize(
        SQL_INTERFACE_OBJECT hSQLConnection,
        UN32 *pun32PageSize );

static BOOLEAN bUnsafeSetDbPageSize(
        SQL_INTERFACE_OBJECT hSQLConnection,
        char *pacBuffer,
        UN32 un32BufferSize,
        UN32 un32PageSize );

static BOOLEAN bUnsafeSetDbJournalMode(
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pacBuffer,
    UN32 un32BufferSize,
    char *pcMode,
    char *pcDb );

/* Database Query Processors */

static BOOLEAN bProcessSelectSportList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectAffiliateList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectAffiliateInfoClassList (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectTableHeaderFixed (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectTableHeaderVariable (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectTable (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessGetInstanceIdResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessGetTdefKeyResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bDbgProcessSelectAllAndDump (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessRowCountHandler(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bProcessRemoveTrefInstanceTables(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_TREF_REMOVE_QUERY_STRUCT *psResult
        );

static BOOLEAN bUnsafeRemoveInstanceTables(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *pcGetRemoveInfoSql,
        const char *pcRemoveSql,
        BOOLEAN bUseKeyTwice,
        N64 n64KeyValue
        );

static BOOLEAN bUnsafeGetLastRowId(
        SPORTS_SERVICE_OBJECT hSportsService,
        N64 *n64RowId
        );

static BOOLEAN bProcessGetSCVerResult(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bProcessGetAFVerResult(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bProcessCStringPragma(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT *psResult
            );

static BOOLEAN bProcessUn32Pragma(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT *psResult
            );

static BOOLEAN bProcessGetAllAfIds (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_ALL_AFID_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessAffiliateNameChange (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SPORTS_SERVICE_AF_NAME_CHANGE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bDropRemovedAffiliates(
        SPORTS_SERVICE_OBJECT hSportsService
        );

static BOOLEAN bUnsafeDropAllDataInstanceTables(
        SPORTS_SERVICE_OBJECT hSportsService
        );

static BOOLEAN bUnsafeClearTable(
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pacName
        );

static BOOLEAN bUnsafeLoadTdefs(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_INTERFACE_OBJECT hSportsHbInterface
        );

static BOOLEAN bUnsafeGetTdefLists(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        OSAL_OBJECT_HDL hAddList,
        OSAL_OBJECT_HDL hRemoveList
        );

static BOOLEAN bProcessGetTdefLists(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bUnsafeLoadTdefHandler(
        SPORTS_SERVICE_TDEF_ADD_STRUCT *psTdefAdd,
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bProcessMakeLabelList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TDEF_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bUnsafeRemoveUnusedTdefHandler(
        void *pvTdefKey,
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bExecuteAgeOut(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        const char *pcRowCountSql,
        const char *pcGetRemoveInfoSql,
        UN32 un32Class,
        UN32 un32Epoch,
        UN32 un32ReceiptTime,
        BOOLEAN bUseReceiptTime,
        BOOLEAN bNotify,
        N32 n32MaxDropCount
        );

static BOOLEAN bExecuteSaveRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        BOOLEAN bNotify,
        N32 n32MaxDropCount
        );

static void vExecuteShutdownRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bUnsafeExecuteReplacementRules(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance
        );

static void vHandleTimeoutEvent (
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
        void *pvEventArg
        );

static BOOLEAN bPruneTdefs(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static void vIntitializeSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bUnsafeGetTableQueryInfo(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        char *pcTdefSql,
        UN32* pun32TdefKey,
        UN32* pun32ColumnCount,
        UN8* pun8ColumnMap,
        UN32 un32ColumMapSize
        );

static BOOLEAN bProcessColumnPriorityCull(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_COLUMN_CULL_QUERY_STRUCT *psResult
        );

static BOOLEAN bUnsafeBuildDb(
        SQL_INTERFACE_OBJECT hDb
        );

static DATASERVICE_ERROR_CODE_ENUM eLoadPersistentDb (
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vUnsafeUpdateTableReceiptTime(
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance,
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction,
        BOOLEAN bIsPrimaryTable
        );

static BOOLEAN bProcessGetLabelOverrides (
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessGetTableList (
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
        );

static BOOLEAN bUnsafeGetTableListHandler(
        SPORTS_SERVICE_TABLE_LIST_ENTRY_STRUCT *psEntry,
        SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static SQL_INTERFACE_OBJECT hCreateMemoryDb(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj,
        BOOLEAN bBuildDb
        );

static BOOLEAN bProcessMakeTableList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
       );

static BOOLEAN bUnsafeDropTable(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TdefKey
        );

static BOOLEAN bProcessMakeTableDropList(
        SQL_QUERY_COLUMN_STRUCT *psColumn,
        N32 n32NumberOfColumns,
        SPORTS_SERVICE_TABLE_LIST_QUERY_STRUCT *psResult
        );

static SMSAPI_RETURN_CODE_ENUM eGetTableHeaderCommon (
    SPORTS_SERVICE_OBJECT hSportsService,
    AFFILIATE_ID tAffiliateId,
    TABLE_ID tTableId,
    char *pcDest,
    UN32 *pun32DestSize,
    BOOLEAN bIsPrimaryTable
        );

static BOOLEAN bUnsafeRemoveTdefTables(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TdefKey
        );

/* Monitor Functions */
static BOOLEAN bCreateMonitorList ( SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj );

static void vDestroyMonitorList ( SPORTS_SERVICE_MGR_OBJECT_STRUCT *psObj );

static void vProcessMonitorChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bProcessMonitorChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        );

static void vNotifySportListChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bNotifySportListChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        );

static void vNotifyAffiliateExistanceChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj
        );

static BOOLEAN bNotifyAffiliateExistanceChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    void *pvArg
        );

static void vNotifyAffiliateChange (
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
    AFFILIATE_ID tAffiliateId,
    SPORTS_MONITOR_EVENT_MASK tChangeBits
        );

static BOOLEAN bNotifyAffiliateChangeHandler (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_CHANGE_ITERATOR_STRUCT *psHelper
        );

static BOOLEAN bUnsafeSetDbSavepoint(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        );

static BOOLEAN bUnsafeReleaseDbSavepoint(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        );

static BOOLEAN bUnsafeRollbackDb(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName
        );

static void vEndDataTableInstance(
        SPORTS_SERVICE_OBJECT hSportsService,
        const char *acSavePointName,
        BOOLEAN bSuccess,
        SPORTS_SERVICE_DATA_INSTANCE_STRUCT *psTableInstance
        );

static BOOLEAN bBitfieldGet(
        UN8* pun8Bitfield,
        UN32 un32BitLength,
        UN32 un32Bit
        );

static void vBitfieldSet(
        UN8* pun8Bitfield,
        UN32 un32BitLength,
        UN32 un32Bit,
        BOOLEAN bVal
        );

static BOOLEAN bBitfieldAnySet(
        UN8* pun8Bitfield,
        UN32 un32BitLength
        );

/***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const SPORTS_SERVICE_INTERFACE_STRUCT SPORTS_SERVICE =
{
    /*.hStart = */hStart,
    /*.vStop = */ vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (SPORTS_SERVICE_OBJECT hSportsService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (SPORTS_SERVICE_OBJECT hSportsService))DATASERVICE_IMPL_eErrorCode,
    /*.eUseDataSet = */eUseDataSet,
    /*.eGetSportList = */eGetSportList,
    /*.eGetRootAffiliateList = */eGetRootAffiliateList,
    /*.eGetAffiliateChildren = */eGetAffiliateChildren,
    /*.eGetAffiliateInfoClassList = */eGetAffiliateInfoClassList,
    /*.eGetAffilaiteTableList = */eGetAffiliateTableList,
    /*.eGetAffiliate = */eGetAffiliate,
    /*.eGetTable = */eGetTable,
    /*.eGetTableHeader = */eGetTableHeader,
    /*.eGetTableByColumnMatch = */eGetTableByColumnMatch,
    /*.eGetTableByColumnRange = */eGetTableByColumnRange,
    /*.eGetTableByTeamPresence = */eGetTableByTeamPresence,
    /*.eGetReferencedTable = */eGetReferencedTable,
    /*.eGetReferencedTableHeader = */eGetReferencedTableHeader,
    /*.hCreateLeagueCid = */hCreateLeagueCid,
    /*.hCreateTeamCid = */hCreateTeamCid
};

// Manager interface for plugins
const SPORTS_SERVICE_MGR_INTERFACE_STRUCT GsSportsServiceMgrIntf =
{
    /*bBeginTableUpdate=*/bBeginTableUpdate,
    /*bEndTableUpdate=*/bEndTableUpdate,
    /*BeginRowUpdate=*/bBeginRowUpdate,
    /*bEndRowUpdate=*/bEndRowUpdate,
    /*bGetLastRowId=*/bUnsafeGetLastRowId,
    /*bUpdateColumnInt=*/bUpdateColumnInt,
    /*bUpdateColumnString=*/bUpdateColumnString,
    /*bDbgDumpTable=*/bDbgDumpTable,
    /*bDbgExecuteSqlCmd=*/bDbgExecuteSqlCmd,
    /*bDbgDumpDefs=*/bDbgDumpDefs,
    /*bStartServiceConfigUpdate=*/bStartServiceConfigUpdate,
    /*bEndServiceConfigUpdate=*/bEndServiceConfigUpdate,
    /*un8GetServiceConfigVersion=*/bIsCorrectServiceConfigVersion,
    /*bAddSportsNameToDB=*/bAddSportsNameToDB,
    /*bAddTdefSportMappingToDB=*/bAddTdefSportMappingToDB,
    /*bShouldKeepTdef=*/bShouldKeepTdef,
    /*bAddTableDefToDB=*/bAddTableDefToDB,
    /*bDeleteTableDefFromDB=*/bDeleteTableDefFromDB,
    /*bDbgDumpAllTableNames=*/bDbgDumpAllTableNames,
    /*vDebugSetServiceConfigVersion=*/vDebugSetServiceConfigVersion,
    /*bStartSportAffiliationUpdate=*/bStartSportAffiliationUpdate,
    /*bAddSportsAffiliationToDB=*/bUnsafeAddSportsAffiliationToDB,
    /*bEndSportAffiliationUpdate=*/bEndSportAffiliationUpdate,
    /*bUpdateTrefRelationships=*/bUnsafeUpdateTrefRelationships,
    /*vEndDataTableInstance=*/vEndDataTableInstance,
    /*vReleasePayload=*/vReleasePayload,
    /*bUnsafeLoadTdefs=*/bUnsafeLoadTdefs,
    /*bAddLabelOverride=*/bUnsafeAddLabelOverride
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsSportsStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */(DATASERVICE_STATE_HANDLER)bHandleServiceStopped,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

static BOOLEAN bSnprintf( char *buf, size_t count, const char *format, ... );

static void vEndProcessBuffer(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult
        );

static SMSAPI_RETURN_CODE_ENUM eSports_Service_snprintf(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult,
    const char *pcFormat,
    ...
        );

__INLINE__ void vInitializeQueryStruct(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult,
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
    char *pcDest,
    UN32 *pun32DestSize );

__INLINE__ void vInitializeInstanceIdQueryStruct(
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *ptResult
        );

__INLINE__ void vInitializeUn32PragmaQueryStruct(
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT *ptResult
        );

__INLINE__ void vInitializeCstringPragmaQueryStruct(
    SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT *ptResult,
    char *pcDest,
    UN32 un32DestSize );

__INLINE__ BOOLEAN bIsAffiliateValid( AFFILIATE_ID tAffiliateId );

__INLINE__ BOOLEAN bSnprintfOk(
    N32 n32Written,
    N32 n32CurrentSize );

__INLINE__ BOOLEAN bIsSportSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32SportId );

__INLINE__ BOOLEAN bIsTableClassSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TableClass );

__INLINE__ BOOLEAN bIsLabelPrioritySupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32Priority );

__INLINE__ BOOLEAN bIsLabelSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32LabelType );

__INLINE__ UN64 un64TimeUpMs(void);

/**********************/
/** INLINE FUNCTIONS **/
/*********************/

__INLINE__ void vInitializeQueryStruct(
    SPORTS_SERVICE_QUERY_RESULT_STRUCT *psResult,
    SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
    char *pcDest,
    UN32 *pun32DestSize )
{
    psResult->psSportsObj = psSportsObj;
    psResult->pacResultBuffer = pcDest;
    psResult->n32ResultBufferSize = *pun32DestSize;
    psResult->pacCursor = psSportsObj->pacScratchPad;
    psResult->un32RowCount = 0;
    psResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    psResult->un32RequiredSize = *pun32DestSize;
    psResult->pun32RequiredSize = pun32DestSize;
    psResult->pacScratchPad = psSportsObj->pacScratchPad;
    psResult->un32ScratchPadSize = psSportsObj->un32ScratchPadSize;
    psResult->un32LastRowStartPos = 0;
    return;
}

__INLINE__ void vInitializeInstanceIdQueryStruct(
    SPORTS_SERVICE_INSTANCE_ID_QUERY_RESULT_STRUCT *ptResult
        )
{
    ptResult->n64InstanceId = 0;
    ptResult->un32RowCount = 0;
    ptResult->eSmsApiCode = SMSAPI_RETURN_CODE_SUCCESS;
    return;
}

__INLINE__ void vInitializeUn32PragmaQueryStruct(
    SPORTS_SERVICE_UN32_PRAGMA_RESULT_STRUCT *ptResult
        )
{
    ptResult->un32RowCount = 0;
    ptResult->bSuccess = TRUE;
    return;
}

__INLINE__ void vInitializeCstringPragmaQueryStruct(
    SPORTS_SERVICE_CSTRING_PRAGMA_RESULT_STRUCT *ptResult,
    char *pcDest,
    UN32 un32DestSize )
{
    ptResult->pacBuffer = pcDest;
    ptResult->un32BufferSize = un32DestSize;
    ptResult->un32RowCount = 0;
    ptResult->bSuccess = TRUE;
    return;
}

__INLINE__ BOOLEAN bIsAffiliateValid( AFFILIATE_ID tAffiliateId )
{
    return ( tAffiliateId <= SS_MAX_AFFILIATE_ID );
}

__INLINE__ BOOLEAN bSnprintfOk(
    N32 n32Written,
    N32 n32CurrentSize )
{
    if( (n32Written < 0) || (n32Written >= n32CurrentSize) )
    {
        return FALSE;
    }
    return TRUE;
}

__INLINE__ BOOLEAN bIsSportSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32SportId )
{
    if( un32SportId > SS_MAX_SPORT_ID )
    {
        return FALSE;
    }

    return bBitfieldGet(
            psSportsObj->aun8SupportedSport,
            SS_MAX_SPORT_COUNT,
            un32SportId );
}

__INLINE__ BOOLEAN bIsTableClassSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32TableClass )
{
    if( psSportsObj->un32SupportedTableClass & (1<<un32TableClass) )
    {
        return TRUE;
    }

    return FALSE;
}

__INLINE__ BOOLEAN bIsLabelPrioritySupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32Priority )
{
    if( psSportsObj->un32SupportedLabelPriority & (1<<un32Priority) )
    {
        return TRUE;
    }

    return FALSE;
}

__INLINE__ BOOLEAN bIsLabelSupported(
        SPORTS_SERVICE_MGR_OBJECT_STRUCT *psSportsObj,
        UN32 un32LabelType )
{
    if( un32LabelType > SS_MAX_LABEL_TYPE )
    {
        return FALSE;
    }

    return bBitfieldGet(
            psSportsObj->aun8SupportedLabel,
            SS_MAX_LABEL_COUNT,
            un32LabelType );
}

__INLINE__ UN64 un64TimeUpMs(void)
{
    UN32 un32Sec;
    UN16 un16Msec;
    OSAL.vTimeUp(&un32Sec, &un16Msec);
    return un32Sec *1000 + un16Msec;
}

void vReleaseBindings(
        SPORTS_SERVICE_TABLE_TRANSACTION_STRUCT *psTransaction );

#endif    // _SPORTS_SERVICE_MGR_OBJ_H_
