/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SPORTS_OBJ_H_
#define _SPORTS_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define SPORTS_CONTENT_OBJECT_NAME "SPORTS"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

// Basic sports list entry structure
typedef struct sport_entry_struct
{
    // CID for this sports type
    CID_OBJECT hId;

    // Sport text & info
    SPORT_STRUCT const *psSport;

} SPORT_ENTRY_STRUCT;

typedef struct sports_team_iterator_struct
{
    TEAM_OBJECT hTeam;
    UN8 un8CurrentIndex;
    UN8 un8StopIndex;
    N32 n32Return;
    FILE *psFile;

} SPORTS_TEAM_ITERATOR_STRUCT;

typedef struct sports_team_cid_iterator_struct
{
    // A pointer to a found flag
    BOOLEAN *pbHasId;

    // The CID to find
    CID_OBJECT hId;

    // League to associate Teams with
    LEAGUE_OBJECT hLeague;

} SPORTS_TEAM_CID_ITERATOR_STRUCT;

// An iterator structure used to contain the callers call back
// function and argument when iterating the sports list.
typedef struct sports_iterator_struct
{
    SPORTS_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback;
    void *pvContentIteratorCallbackArg;

} SPORTS_ITERATOR_STRUCT;

typedef struct sports_ctrl_struct
{
    // A handle to a linked list of available sports which maps CIDs to
    // more verbose and abstracted sports content to populate
    // an associated CDO with. Note that this list is dynamic and shared
    // amongst all SMS Decoders. Thus access to it must be protected.
    // To enforce this protection this structure is declared as an
    // SMS Object and the built-in locking mechanism is used to
    // limit exclusive access to this element.
    OSAL_OBJECT_HDL hSportsList;

} SPORTS_CTRL_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SPORTS_ENUM eSubType (
    CD_OBJECT hCDO
        );

static CID_OBJECT hId (
    CD_OBJECT hCDO
        );

static LEAGUE_OBJECT hLeague (
    CD_OBJECT hCDO
        );

static UN8 un8Teams (
    CD_OBJECT hCDO
        );

static TEAM_OBJECT hTeam (
    CD_OBJECT hCDO,
    UN8 un8Index
        );

static UN8 un8Cars (
    CD_OBJECT hCDO
        );

static CAR_OBJECT hCar (
    CD_OBJECT hCDO,
    UN8 un8Index
        );

static SPORTS_BROADCAST_ENUM eBroadcast (
    CD_OBJECT hCDO,
    UN8 *pun8Index
        );

static BOOLEAN bIterateContent (
    SPORTS_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        );

/* Object Private Prototypes */

static BOOLEAN bInitialize ( void );

static void vUnInitialize ( void );

static void vUnInit (
    SPORTS_OBJECT_STRUCT *psObj
        );

static N16 n16Equal (
    const SPORTS_OBJECT_STRUCT *psObj1,
    const SPORTS_OBJECT_STRUCT *psObj2
        );

static N16 n16Compare (
    const SPORTS_OBJECT_STRUCT *psObj1,
    const SPORTS_OBJECT_STRUCT *psObj2
        );

static N32 n32FPrintf (
    const SPORTS_OBJECT_STRUCT *psObj,
    FILE *psFile
        );

static BOOLEAN bHasId (
    const SPORTS_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
        );

static void vDestroySportEntry (
    SPORT_ENTRY_STRUCT const *psSportEntry
        );

static void vRemoveTeam (
    TEAM_OBJECT hTeam
        );

static N16 n16CompareSportNames (
    SPORT_ENTRY_STRUCT const *psSportEntry1,
    SPORT_ENTRY_STRUCT const *psSportEntry2
        );

static N16 n16IsSportEqualToId (
    SPORT_ENTRY_STRUCT const *psSportEntry,
    CID_OBJECT hCID
        );

static BOOLEAN bTeamIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_ITERATOR_STRUCT *psTeamIterator
        );

static BOOLEAN bTeamRemoveIterator (
    TEAM_OBJECT hTeam,
    CD_OBJECT hCDO
        );

#if SMS_DEBUG == 1
static BOOLEAN bPrintTeamIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_ITERATOR_STRUCT *psTeamIterator
        );
#endif

static BOOLEAN bIterator (
    SPORT_ENTRY_STRUCT const *psSportEntry,
    SPORTS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bTeamIdIterator (
    TEAM_OBJECT hTeam,
    SPORTS_TEAM_CID_ITERATOR_STRUCT *psTeamIterator
        );

static TEAM_OBJECT hTeamLocal (
    SPORTS_OBJECT_STRUCT *psObj,
    UN8 un8Index
        );

static UN8 un8TeamsLocal (
    SPORTS_OBJECT_STRUCT *psObj
        );

static SPORT_ENTRY_STRUCT const *psCreateSportEntry (
    CID_OBJECT hId,
    SPORT_STRUCT const *psSport,
    BOOLEAN bCopySport
        );

  /***************/
 /** VARIABLES **/
/***************/

// Sports object control structure initialization
static SPORTS_CTRL_STRUCT *gpsSportsCtrl = NULL;

// Default sport
static const SPORT_STRUCT gsDefaultSport =
{
    SPORTS_UNKNOWN,
    "?",
    "Unknown"
};

// Default sports
static const SPORTS_OBJECT_STRUCT gsDefaultSports =
{
    CID_INVALID_OBJECT,
    NULL,
    LEAGUE_INVALID_OBJECT,
    OSAL_INVALID_OBJECT_HDL,
    SPORTS_BROADCAST_UNKNOWN,
    0
};

// Local instance of an interface for the CDO object
static const CDO_INTERFACE_STRUCT gsSportsInterface =
{
    /*.bInitialize = */bInitialize,
    /*.vUnInitialize = */vUnInitialize,
    /*.vUnInit = */(void (*)(void *))vUnInit,
    /*.n16Equal = */(N16 (*)(const void *, const void *))n16Equal,
    /*.n16Compare = */(N16 (*)(const void *, const void *))n16Compare,
    /*.n32FPrintf = */(N32 (*)(const void *, FILE *))n32FPrintf,
    /*.bHasId = */(BOOLEAN (*)(const void *, CID_OBJECT))bHasId
};

// SPORTS CDO information definition
const CDO_INFO_STRUCT GsSportsInfo =
{
    /*.eType = */CDO_SPORTS,
    /*.pacTypeText = */MACRO_TO_STRING(CDO_SPORTS),
    /*.pacDescription = */"Sports",
    /*.pvDefaultObjData = */(void*)&gsDefaultSports,
    /*.psInterface = */&gsSportsInterface
};

// Global (re-usable) instance of an interface for this object
const SPORTS_CONTENT_OBJECT_INTERFACE_STRUCT SPORTS =
{
    /*.eSubType = */eSubType,
    /*.hId = */hId,
    /*.hLeague = */hLeague,
    /*.un8Teams = */un8Teams,
    /*.hTeam = */hTeam,
    /*.un8Cars = */un8Cars,
    /*.hCar = */hCar,
    /*.eBroadcast = */eBroadcast,
    /*.bIterateContent = */bIterateContent
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _SPORTS_OBJ_H_
