/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:SPORTS_FLASH private header file.
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SPORTS_FLASH_OBJ_H_
#define _SPORTS_FLASH_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define SPORTS_FLASH_OBJECT_NAME "SPORTS_FLASH"

#define SPORTS_FLASH_EVENT_TIMER_DURATION_MS (5 * 60 * 1000)

// Max number of favorites teams possible to add
#define SPORTS_FLASH_TEAMS_COUNT_MAX (128)

// SXI version 3.0 and upper support Sports Flash feature
#define SPORTS_FLASH_MIN_SXI_VERSION_REQUIRED \
    MODULE_PROTOCOL_VERSION_PACK(3, 0, 0)

/* Sports Flash Favorite Teams List tag name */
#define SPORTS_FLASH_TAG_NAME_FAVORITE_TEAMS "FavoriteTeams"
/* Sports Flash Team tag names */
#define SPORTS_FLASH_TAG_NAME_TEAM "Team"
#define SPORTS_FLASH_TAG_NAME_LEAGUE_CID "league_id"
#define SPORTS_FLASH_TAG_NAME_TEAM_CID "team_id"
/* Sports Flash Enable Notifications tag name */
#define SPORTS_FLASH_TAG_NAME_NOTIFICATIONS_ENABLED "NotificationEnabled"

static const UN8 aun8GamePriorityHomeTeam[] = {2, 1, 3, 4};
static const UN8 aun8GamePriorityAwayTeam[] = {2, 3, 1, 4};
static const UN8 aun8GamePriorityBothTeams[] = {3, 1, 2, 4};

  /**************/
 /** TYPEDEFS **/
/**************/

//  The SPORTS_FLASH_SERVICE structure
typedef struct sports_flash_object_struct
{
    /* THINGS PROVIDED BY APPLICATION */

    /* THINGS TO INITIALIZE ONCE */

    // The decoder in which this service operates
    DECODER_OBJECT hDecoder;

    // The LL used to store the favorite teams
    OSAL_OBJECT_HDL hTeamsList;

    // The LL used to store the games
    OSAL_OBJECT_HDL hGamesList;

    // The LL used to store the games
    OSAL_OBJECT_HDL hEventsList;

    // Game Events Callback handler
    SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback;
    void *pvGameEventCallbackArg;

    // Sports Flash Tag Handle
    TAG_OBJECT hTag;

    // Max Sports Flash game monitors count
    UN8 un8MaxSportsFlash;

    /* THINGS TO RE-INITIALIZE ON DEMAND or RESET */
    SPORTS_FLASH_EVENT_ID tCurrentFlashEventID;

    OSAL_OBJECT_HDL hEventTimer;

    // Flash Events Callback handler
    SPORTS_FLASH_FLASH_EVENT_CALLBACK vFlashEventCallback;
    void *pvFlashEventCallbackArg;

    // Teams monitor status
    BOOLEAN bSeekMonitorEnabled;

    // Games monitor status
    BOOLEAN bGamesMonitorEnabled;

    // Notification status
    BOOLEAN bNotificationEnabled;

} SPORTS_FLASH_OBJECT_STRUCT;

// This structure is used to handle Favorite Team data.
typedef struct sports_flash_favorite_team_struct
{
    TAG_OBJECT hTag;
    CID_OBJECT hLeagueId;
    CID_OBJECT hTeamId;
} SPORTS_FLASH_FAVORITE_TEAM_STRUCT;

// This structure is used to handle Flash Event data
typedef struct sports_flash_flash_event_struct
{
    CHANNEL_ID tChannelID;
    SPORTS_FLASH_EVENT_ID tFlashEventID;
} SPORTS_FLASH_FLASH_EVENT_STRUCT;

typedef struct sports_flash_teams_iterator_struct
{
    SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK bIteratorCallback;
    void *pvTeamsIteratorCallbackArg;

} SPORTS_FLASH_TEAMS_ITERATOR_STRUCT;

typedef struct sports_flash_events_iterator_struct
{
    SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK bIteratorCallback;
    void *pvEventsIteratorCallbackArg;

} SPORTS_FLASH_EVENTS_ITERATOR_STRUCT;

typedef struct sports_flash_team_convert_iterator_struct
{
    UN32 un32Idx;
    UN32 *paun32Teams;

} SPORTS_FLASH_TEAM_CONVERT_ITERATOR_STRUCT;

typedef struct sports_flash_team_load_iterator_struct
{
    TAG_OBJECT hParentTag;
    size_t tBuffSize;
    char *pacBuff;
    SPORTS_FLASH_OBJECT_STRUCT *psObj;
    BOOLEAN bNeedUpdate;

} SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SMSAPI_RETURN_CODE_ENUM eStart (
    DECODER_OBJECT hDecoder,
    UN32 *pun32MaxSportsFlash,
    SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
    void *pvGameEventCallbackArg
        );

static void vStop (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eAddFavoriteTeam (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveFavoriteTeam (
    DECODER_OBJECT hDecoder,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveAllFavoriteTeams (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eIterateFavoriteTeams (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK bTeamsIteratorCallback,
    void *pvTeamsIteratorCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eIterateCurrentGames (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_GAMES_ITERATOR_CALLBACK bGamesIteratorCallback,
    void *pvGamesIteratorCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eSetGamesMonitor (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID *ptChannelIDList,
    UN16 un16Count,
    SPORTS_FLASH_FLASH_EVENT_CALLBACK vFlashEventCallback,
    void *pvFlashEventCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eIterateCurrentFlashEvents (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK bEventsIteratorCallback,
    void *pvEventsIteratorCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM ePlayFlashEvent (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventID
        );

static SMSAPI_RETURN_CODE_ENUM eAbortFlashEvent (
    DECODER_OBJECT hDecoder
        );

static SPORTS_FLASH_EVENT_ID tCurrentFlashEventID (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eEnableNotification (
    DECODER_OBJECT hDecoder,
    BOOLEAN bEnable
        );

static SMSAPI_RETURN_CODE_ENUM eIsNotificationEnabled (
    DECODER_OBJECT hDecoder,
    BOOLEAN *pbEnable
        );

/* Object Private Prototypes */

static BOOLEAN bHandleUpdateTeams (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleSetGamesMonitor (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_SET_GAMES_MONITOR_STRUCT *psSetGamesMonitor
        );

static BOOLEAN bHandlePlayFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT *psFlashEvent
        );

static BOOLEAN bHandleAbortFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleRemainFlashEvent (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_FLASH_EVENT_STRUCT *psFlashEvent
        );

static N16 n16ComparePriority(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame1,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame2
        );

static UN8 un8FindPriority(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psInfo
        );

static SMSAPI_RETURN_CODE_ENUM eCreateSportsFlash (
    DECODER_OBJECT hDecoder,
    UN8 un8MaxSportsFlash,
    SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
    void *pvGameEventCallbackArg
        );

static void vDestroySportsFlash (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCreateTag (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRestoreSportsFlash (
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bTeamsLoadIterator(
    TAG_OBJECT hTag,
    SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT *psIterator
        );

static CID_OBJECT hLoadCID(
    TAG_OBJECT hTag,
    SPORTS_FLASH_TEAM_LOAD_ITERATOR_STRUCT *psIterator
        );

static void vSaveTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam
        );

static void vDestroyFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psFavoriteTeam
        );

static void vDestroyGame(
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        );

static OSAL_LINKED_LIST_ENTRY hFindInTeamsList (
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId
        );

static N16 n16CompareFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam1,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam2
        );

static N16 n16CompareGameInfo(
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psObj1,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psObj2
        );

static N16 n16CompareFlashEvent(
    SPORTS_FLASH_FLASH_EVENT_STRUCT *psObj1,
    SPORTS_FLASH_FLASH_EVENT_STRUCT *psObj2
        );

static BOOLEAN bTeamsIterator(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT const *psObj,
    SPORTS_FLASH_TEAMS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bTeamConvertIterator(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT const *psObj,
    SPORTS_FLASH_TEAM_CONVERT_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bFlashEventsIterator(
    SPORTS_FLASH_FLASH_EVENT_STRUCT const *psObj,
    SPORTS_FLASH_EVENTS_ITERATOR_STRUCT *psIterator
        );

static void vFlashEventTimerCallback(
    OSAL_OBJECT_HDL hTimer,
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static void vUninitGameEventInfo(
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        );

static BOOLEAN bInitGameInfoEvent(
    const SPORTS_FLASH_EVENT_GAME_EVENT_STRUCT *psGameEvent,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psEventInfo
        );

static void vReleaseFavoriteTeam(
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam
        );

static BOOLEAN bHandleStopEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleAddTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_TEAM_STRUCT *psTeam
        );

static BOOLEAN bHandleRemoveTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_TEAM_STRUCT *psTeam
        );

static BOOLEAN bCheckGameEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame
        );

static BOOLEAN bRemoveGamesByTeam(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY hGameLLEntry,
    SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGame,
    SPORTS_FLASH_FAVORITE_TEAM_STRUCT *psTeam
        );

static BOOLEAN bHandleRemoveAllTeams(
    SPORTS_FLASH_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleEnableNotificationEvent(
    SPORTS_FLASH_OBJECT_STRUCT *psObj,
    const SPORTS_FLASH_ACTION_ENABLE_NOTIFICATION_STRUCT *psEnableNotification
        );

static void vCleanupEvent(
    const SMS_EVENT_SPORTS_FLASH_STRUCT *psEventData
        );

/***************/
/** VARIABLES **/
/***************/

// Sports Flash Object Defaults
static const SPORTS_FLASH_OBJECT_STRUCT gsObjectDefaults =
{
    /* THINGS PROVIDED BY APPLICATION */

    /* THINGS TO INITIALIZE ONCE */

    // The decoder in which this service operates
    DECODER_INVALID_OBJECT,

    // Lists
    OSAL_INVALID_OBJECT_HDL,
    OSAL_INVALID_OBJECT_HDL,
    OSAL_INVALID_OBJECT_HDL,

    // Game Events Callback handler
    NULL,
    NULL,

    // Sports Flash  Tag handle
    TAG_INVALID_OBJECT,

    // Max Sports Flash game monitors count
    0,

    /* THINGS TO RE-INITIALIZE ON DEMAND or RESET */
    SPORTS_FLASH_INVALID_EVENT_ID,

    // Timer
    OSAL_INVALID_OBJECT_HDL,

    // Flash Events Callback handler
    NULL,
    NULL,

    // Teams monitor status
    FALSE,

    // Games monitor status
    FALSE,

    // Notification status
    TRUE
};

// Global instance of an interface for this object
const SPORTS_FLASH_OBJECT_INTERFACE_STRUCT SPORTS_FLASH =
{
    /*.eStart = */eStart,
    /*.vStop = */vStop,
    /*.eAddFavoriteTeam = */eAddFavoriteTeam,
    /*.eRemoveFavoriteTeam = */eRemoveFavoriteTeam,
    /*.eRemoveAllFavoriteTeams = */eRemoveAllFavoriteTeams,
    /*.eIterateFavoriteTeams = */eIterateFavoriteTeams,
    /*.eIterateCurrentGames = */eIterateCurrentGames,
    /*.eSetGamesMonitor = */eSetGamesMonitor,
    /*.eIterateCurrentFlashEvents = */eIterateCurrentFlashEvents,
    /*.ePlayFlashEvent = */ePlayFlashEvent,
    /*.eAbortFlashEvent = */eAbortFlashEvent,
    /*.tCurrentFlashEventID = */tCurrentFlashEventID,
    /*.eEnableNotification = */eEnableNotification,
    /*.bIsNotificationEnabled = */eIsNotificationEnabled
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif /* _SPORTS_FLASH_OBJ_H_ */
