/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SMS_EVENT_H_
#define _SMS_EVENT_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// If logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define SMS_EVENT_NAME "SMSE"

// Event log size
#define SMS_EVENT_LOG_SIZE  ((128 * 1024) - 1)  // 128k is a good size for now.

// Event internal flags
#define SMSE_EVENT_FLAGS_NONE            (0x00)
#define SMSE_EVENT_FLAGS_STATIC          (0x01) // Event is static
                                                // (should never be released)
#define SMSE_EVENT_FLAGS_REALLOCATED     (0x02) // Event has been re-allocated
#define SMSE_EVENT_FLAGS_FREE            (0x04) // Event should be freed and not processed

// Logging constants
#define SMSE_EVENT_LOGGING_RATE_SECS (60)
#define SMSE_EVENT_LOGGING_BANNER \
    "\n*************** SMS Event Queue Statistics ***************\n"

#define SMSE_EVENT_LOGGING_END \
      "**********************************************************\n"

// The maximum number of queues that will
// be tracked for logging purposes
#define SMSE_EVENT_LOGGING_MAX_QUEUE_TRACKING (2)

  /**************/
 /** TYPEDEFS **/
/**************/

typedef UN8 SMS_EVENT_FLAGS;

typedef SMS_OBJECT (*SMSE_CREATE_QUEUE) (
    SMS_OBJECT hParent,
    const char *pacName,
    UN32 un32EventQueueSize
        );

typedef void (*SMSE_DESTROY_QUEUE) (
    SMS_OBJECT hQueueData
        );

typedef void * (*SMSE_QUEUE_GET_MSG) (
    SMS_OBJECT hQueueData,
    N32 n32Timeout
        );

typedef OSAL_OBJECT_HDL (*SMSE_GET_QUEUE_FOR_EVENT) (
    SMS_OBJECT hQueueData,
    EVENT_OPTIONS_TYPE tEventOptions
        );

typedef BOOLEAN (*SMSE_POST_EVENT) (
    SMS_OBJECT hQueueData,
    const void *pvMessage,
    UN32 un32MessageSize,
    BOOLEAN *pbAlreadyPresent
        );

typedef BOOLEAN (*SMSE_ITERATE_QUEUE_CALLBACK)(
    OSAL_OBJECT_HDL hEventQueue,
    const char *pacQueueName,
    UN8 un8QueueIndex,
    void *pvCallbackArg
        );

typedef void (*SMSE_ITERATE_QUEUES) (
    SMS_OBJECT hQueueData,
    SMSE_ITERATE_QUEUE_CALLBACK bCallback,
    void *pvCallbackArg
        );

typedef struct sms_event_queue_interface
{
    // Create the queue
    SMSE_CREATE_QUEUE hCreate;

    // Destroy the queue
    SMSE_DESTROY_QUEUE vDestroy;

    // Allocate a message from the queue
    SMSE_QUEUE_GET_MSG pvGetMsg;

    // Get the queue handle
    SMSE_GET_QUEUE_FOR_EVENT hGetQueue;

    // Post to the queue
    SMSE_POST_EVENT bPost;

    // Iterate the queues in this interface
    SMSE_ITERATE_QUEUES vIterateQueues;

} SMS_EVENT_QUEUE_INTERFACE;

typedef struct sms_event_time_struct
{
    UN32 un32Seconds;
    UN16 un16MilliSeconds;

} SMS_EVENT_TIME_STRUCT;

#if SMS_LOGGING == 1
typedef struct sms_event_logging_struct
{
    // Log object for event logging
    OSAL_OBJECT_HDL hLog;

    // Time stamping info
    SMS_EVENT_TIME_STRUCT sStart;
    SMS_EVENT_TIME_STRUCT sEnd;

    // Number of events handled
    UN32 un32NumberOfEvents;

    // Maximum number of msec any one event handling episode
    // incurred. Tracks the maximum execution time of handling one event.
    UN32 un32MaxExecutionMsec;

    // Flag indicating if message has been generated for this event
    BOOLEAN bEventMessagePresent;

    // Last time we printed queue info
    UN32 un32LastQueueLogDump;

    // The last number reported for message
    // drop count
    UN32 aun32LastMessageDropCount[SMSE_EVENT_LOGGING_MAX_QUEUE_TRACKING];

} SMS_EVENT_LOGGING_STRUCT;

#endif /* SMS_LOGGING == 1 */

/*
    SMS Event Structure

    Basic structure for all SMS events which can be posted.

*/

typedef struct sms_event_struct
{
    // Owning event handler
    struct sms_event_handler_struct *psEventHdlr;

    // Generic event type (event defined)
    SMS_EVENT_TYPE_ENUM eType;

    // Internal Event flags (see above)
    SMS_EVENT_FLAGS tFlags;

    // Mutex used to perform synchronous events
    OSAL_OBJECT_HDL hSynchronousEventMutex;

    // Event data (event defined)
    SMS_EVENT_DATA_UNION uData;

} SMS_EVENT_STRUCT;

typedef struct sms_event_control_struct
{
    // Current event being handled
    SMS_EVENT_STRUCT *psCurrentEvent;

#if SMS_LOGGING == 1

    // A logging structure to hold event information when
    // event logging is enabled.
    SMS_EVENT_LOGGING_STRUCT sLogging;

#endif /* SMS_LOGGING == 1 */

} SMS_EVENT_CONTROL_STRUCT;

/*
    SMS Event Handler Structure
*/

typedef struct sms_event_handler_struct
{
    // Queue Interface structure & data
    SMS_EVENT_QUEUE_INTERFACE const *psQueueInterface;
    SMS_OBJECT hQueueData;

    // Event Control structure
    SMS_EVENT_CONTROL_STRUCT sCtrl;

    // Event handler object. This is the SMS object which
    // consumes the events.
    SMS_OBJECT hSMS;

    // ID of the task running this event handler
    OSAL_TASK_ID tTaskId;

    // Event handler function. This is the event handler
    // functions which actually processes and handles events.
    SMS_OBJECT_EVENT_HANDLER_PROTOTYPE vEventHandler;

} SMS_EVENT_HANDLER_STRUCT;

typedef struct sms_event_single_queue_struct {

    OSAL_OBJECT_HDL hEventQueue;

} SMS_EVENT_SINGLE_QUEUE_STRUCT;

typedef struct sms_event_dual_queue_struct {

    OSAL_OBJECT_HDL hSignalSem;
    OSAL_OBJECT_HDL hEventQueue1;
    OSAL_OBJECT_HDL hEventQueue2;

} SMS_EVENT_DUAL_QUEUE_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

static SMS_EVENT_RESULT_ENUM eEventHdlr (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_STRUCT *psEvent
        );

static void vFreeEvent (
    SMS_EVENT_STRUCT *psEvent
        );

static SMS_EVENT_HDL hReAllocateEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_TYPE_ENUM eEvent,
    SMS_EVENT_DATA_UNION **ppuData
        );

#if SMS_LOGGING == 1

static void vLogStartOfEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr,
    SMS_EVENT_STRUCT *psEvent
        );

static void vLogEndOfEvent (
    SMS_EVENT_HANDLER_STRUCT *psEventHdlr
        );

static BOOLEAN bIterateQueuesForLog (
    OSAL_OBJECT_HDL hEventQueue,
    const char *pacQueueName,
    UN8 un8QueueIndex,
    SMS_EVENT_LOGGING_STRUCT *psLogging
        );

static const char *pacEventTypeText(
    SMS_EVENT_TYPE_ENUM eType
        );

#endif /* SMS_LOGGING == 1 */

/* INTERFACE FUNCTIONS FOR SINGLE QUEUE */

static SMS_OBJECT hCreateQueue (
    SMS_OBJECT hParent,
    const char *pacName,
    UN32 un32EventQueueSize
        );

static void vDestroyQueue (
    SMS_OBJECT hQueueData
        );

static OSAL_OBJECT_HDL hQueueForEvent (
    SMS_OBJECT hQueueData,
    EVENT_OPTIONS_TYPE tEventOptions
        );

static void *pvGetMsg (
    SMS_OBJECT hQueueData,
    N32 n32Timeout
        );

static BOOLEAN bPostEvent (
    SMS_OBJECT hQueueData,
    const void *pvMessage,
    UN32 un32MessageSize,
    BOOLEAN *pbAlreadyPresent
        );

static void vIterateQueue (
    SMS_OBJECT hQueueData,
    SMSE_ITERATE_QUEUE_CALLBACK bCallback,
    void *pvCallbackArg
        );

/* INTERFACE FUNCTIONS FOR DUAL QUEUE */

static SMS_OBJECT hCreateDualQueue (
    SMS_OBJECT hParent,
    const char *pacName,
    UN32 un32EventQueueSize
        );

static void vDestroyDualQueue (
    SMS_OBJECT hQueueData
        );

static OSAL_OBJECT_HDL hQueueForEventDualQueue (
    SMS_OBJECT hQueueData,
    EVENT_OPTIONS_TYPE tEventOptions
        );

static void *pvGetMsgDualQueue (
    SMS_OBJECT hQueueData,
    N32 n32Timeout
        );

static BOOLEAN bPostEventDualQueue (
    SMS_OBJECT hQueueData,
    const void *pvMessage,
    UN32 un32MessageSize,
    BOOLEAN *pbAlreadyPresent
        );

static void vIterateDualQueue (
    SMS_OBJECT hQueueData,
    SMSE_ITERATE_QUEUE_CALLBACK bCallback,
    void *pvCallbackArg
        );

  /***************/
 /** VARIABLES **/
/***************/

SMS_EVENT_QUEUE_INTERFACE const sSingleQueue =
{
    /*.hCreate = */hCreateQueue,
    /*.vDestroy = */vDestroyQueue,
    /*.pvGetMsg = */pvGetMsg,
    /*.hGetQueue = */hQueueForEvent,
    /*.bPost = */bPostEvent,
    /*.vIterateQueues = */vIterateQueue
};

SMS_EVENT_QUEUE_INTERFACE const sDualQueue =
{
    /*.hCreate = */hCreateDualQueue,
    /*.vDestroy = */vDestroyDualQueue,
    /*.pvGetMsg = */pvGetMsgDualQueue,
    /*.hGetQueue = */hQueueForEventDualQueue,
    /*.bPost = */bPostEventDualQueue,
    /*.vIterateQueues = */vIterateDualQueue
};
  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _SMS_EVENT_H_
