/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _REPORT_OBJ_H_
#define _REPORT_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "tag_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#if DEBUG_OBJECT == 0

#ifdef vPrintMarkets
#undef vPrintMarkets
#endif
#define vPrintMarkets(a)

#endif

/* Object name prefix for objects */
#define REPORT_OBJECT_NAME "REPORT"

#define MARKET_CONFIRMED ((DECODER_OBJECT)(size_t)(0-1))  // All F's
#define MARKET_NOT_SEEN (DECODER_INVALID_OBJECT)

// tag name definitions
#define MARKET_LIST_COMPLETE_TAG     "MarketListComplete"
#define MARKETS_TAG                  "Markets"
#define MARKETS_RADIO_SPECIFIC       "MarketsRadioSpecific"
#define MARKET_TAG                   "Mkt"
#define MARKET_NAME_TAG              "Name"
#define MARKET_CID_TAG               "CID"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct report_ctrl_struct
{
    // A handle to a linked list of available markets which maps CIDs to
    // more verbose and abstracted market content to populate
    // an associated CDO with. Note that this list is dynamic and shared
    // amongst all SMS Decoders. Thus access to it must be protected.
    // To enforce this protection this structure is declared as an
    // SMS Object and the built-in locking mechanism is used to
    // limit exclusive access to this element.
    OSAL_OBJECT_HDL hMarketList;

    // A flag to indicate when we have a completed market list
    // This occurs when each market has been seen twice by the same DECODER.
    BOOLEAN bMarketListComplete;

    TAG_OBJECT hMarketListCompleteTag;
    TAG_OBJECT hMarketsTag;
    TAG_OBJECT hRadioSpecificData;

    MARKET_CID_CREATOR hCreateMarketCid;

    UN16 un16ConfirmedMarkets;

} REPORT_CTRL_STRUCT;

// A report entry structure
typedef struct report_entry_struct
{
    // Report object (re-usable)
    REPORT_OBJECT_STRUCT sReport;

    // only used when we are determining if the market list is complete
    DECODER_OBJECT hDecoderHdl;

} REPORT_ENTRY_STRUCT;

// An iterator structure used to contain the callers call back
// function and argument when iterating the market list.
typedef struct report_iterator_struct
{
    REPORT_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback;
    void *pvContentIteratorCallbackArg;

} REPORT_ITERATOR_STRUCT;

// An iterator structure used to print the market list.
typedef struct report_print_iterator_struct
{
    FILE *psFile;
    UN32 un32NumMarkets;
    UN32 un32NumMarketsSeen;
    UN32 un32NumMarketsConfirmed;

} REPORT_PRINT_ITERATOR_STRUCT;

typedef struct report_mkt_tag_iterator_struct
{
    TAG_OBJECT hParentTag;
    size_t tBuffSize;
    char *pacBuff;

} REPORT_MKT_TAG_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/
#define REPORT_MARKET_LIST_COMPLETE_TO_N32(x) (N32)((x == TRUE) ? 1 : 0)

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static const char *pacId (
    CD_OBJECT hCDO
        );

static const char *pacName (
    CD_OBJECT hCDO
        );

static CID_OBJECT hId (
    CD_OBJECT hCDO
        );

static BOOLEAN bIterateContent (
    REPORT_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
        );

static BOOLEAN bContentComplete ( void );

/* Object Private Prototypes */

static BOOLEAN bInitialize ( void );

static void vUnInitialize ( void );

static void vUnInit (
    REPORT_OBJECT_STRUCT *psObj
        );

static N16 n16Equal (
    const REPORT_OBJECT_STRUCT *psObj1,
    const REPORT_OBJECT_STRUCT *psObj2
        );

static N16 n16Compare (
    const REPORT_OBJECT_STRUCT *psObj1,
    const REPORT_OBJECT_STRUCT *psObj2
        );

static N32 n32FPrintf (
    const REPORT_OBJECT_STRUCT *psObj,
    FILE *psFile
        );

static BOOLEAN bHasId (
    const REPORT_OBJECT_STRUCT *psObj,
    CID_OBJECT hId
        );

REPORT_ENTRY_STRUCT *psCreateReportEntry (
    CID_OBJECT hId,
    const MARKET_STRUCT *psMarket,
    BOOLEAN bCopyMarket
        );

static void vDestroyReportEntry (
    REPORT_ENTRY_STRUCT *psReportEntry
        );

static N16 n16MarketEqualToId (
    const REPORT_ENTRY_STRUCT *psObj,
    CID_OBJECT hCID
        );

static N16 n16CompareReportNames (
    const REPORT_ENTRY_STRUCT *psReport1,
    const REPORT_ENTRY_STRUCT *psReport2
        );

static BOOLEAN bIterator (
    REPORT_ENTRY_STRUCT *psObj,
    REPORT_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bInvalidate (
    REPORT_ENTRY_STRUCT *psObj,
    REPORT_ITERATOR_STRUCT *psIterator
        );

static void vSaveMarkets (
    void
        );

static BOOLEAN bWriteMktTag (
    const char *pacId,
    const char *pacName,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        );

static BOOLEAN bReadMktTag (
    TAG_OBJECT hTag,
    void *pvArg
        );

static REPORT_ENTRY_STRUCT *psCreateReportEntryFromTagValue (
    CID_OBJECT hId,
    STRING_OBJECT hName,
    STRING_OBJECT hMktId
        );

static void vLoadMarket (
    CID_OBJECT hId,
    STRING_OBJECT hName,
    STRING_OBJECT hMktId
        );

static BOOLEAN bRemovedMktTag (
    TAG_OBJECT hTag,
    void *pvArg
        );

#if DEBUG_OBJECT == 1

static BOOLEAN bPrint (
    REPORT_ENTRY_STRUCT *psReportEntry,
    REPORT_PRINT_ITERATOR_STRUCT *psIterator
        );

static void vPrintMarkets (
    FILE *psFile
        );

#endif

  /***************/
 /** VARIABLES **/
/***************/

// Report object control structure initialization
static REPORT_CTRL_STRUCT *gpsReportCtrl = NULL;

// A default market structure to use for new (discovered) CIDs
static const MARKET_STRUCT gsDefaultMarket =
{
    "?",
    "Unknown"
};

// An initialized default report object structure
static const REPORT_OBJECT_STRUCT gsDefaultReport =
{
    CID_INVALID_OBJECT,
    NULL
};

// Global instance of an interface for the CDO object
static const CDO_INTERFACE_STRUCT gsReportInterface =
{
    /*.bInitialize = */bInitialize,
    /*.vUnInitialize = */vUnInitialize,
    /*.vUnInit = */(void (*)(void *))vUnInit,
    /*.n16Equal = */(N16 (*)(const void *, const void *))n16Equal,
    /*.n16Compare = */(N16 (*)(const void *, const void *))n16Compare,
    /*.n32FPrintf = */(N32 (*)(const void *, FILE *))n32FPrintf,
    /*.bHasId = */(BOOLEAN (*)(const void *, CID_OBJECT))bHasId
};

// REPORT CDO information definition
const CDO_INFO_STRUCT GsReportInfo =
{
    /*.eType = */CDO_REPORT,
    /*.pacTypeText = */MACRO_TO_STRING(CDO_REPORT),
    /*.pacDescription = */"Traffic and Weather Reports",
    /*.pvDefaultObjData = */(void*)&gsDefaultReport,
    /*.psInterface = */&gsReportInterface
};

// Global (re-usable) instance of an interface for this object
const REPORT_OBJECT_INTERFACE_STRUCT REPORT =
{
    /*.pacId = */pacId,
    /*.pacName = */pacName,
    /*.hId = */hId,
    /*.bIterateContent = */bIterateContent,
    /*.bContentComplete = */bContentComplete,
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _REPORT_OBJ_H_
