/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _PRESETS_OBJ_H_
#define _PRESETS_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"

#include "ccache.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define PRESETS_OBJECT_NAME                   "PRESETS"
/* Featured Favorites Band Capacity */
#define PRESETS_FEATURED_BAND_CAPACITY        "Capacity"
/* Featured Favorites Band Purpose */
#define PRESETS_FEATURED_BAND_PURPOSE         "Purpose"
/* Featured Favorites Number of Bands Limit */
#define PRESETS_FEATURED_BAND_LIMIT           "Limit"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

// Enumeration which indicates the
// current state of each preset
typedef enum preset_state_enum
{
    PRESET_STATE_UNCHANGED = 0,
    PRESET_STATE_ADDED,
    PRESET_STATE_UPDATED,
    PRESET_STATE_REMOVED

} PRESET_STATE_ENUM;

// Presets Notification Callback Shim Struct
typedef struct presets_callback_shim_struct
{
    // Original callback
    PRESETS_OBJECT_EVENT_CALLBACK vEventCallback;

    // Original event callback argument
    void *pvEventCallbackArg;

    // Band handle
    PRESET_BAND_OBJECT hBand;

} PRESETS_CALLBACK_SHIM_STRUCT;

// A structure which is used to
// maintain all relevant data about
// a preset
typedef struct preset_struct
{
    // This preset's owning band
    PRESET_BAND_OBJECT hBand;

    // This preset's name
    STRING_OBJECT hName;

    // This preset's service Id
    SERVICE_ID tServiceId;

    // This preset's channel handle
    // (only used to deal with channel events)
    CHANNEL_OBJECT hChannel;

    // The preset's owning Band Id --
    // used to order the presets in
    // the master preset list
    size_t tBandId;

    // Where this preset occurs in
    // it's owning band
    size_t tPresetIndex;

    // This preset's entry handle
    // in the master presets list
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The current state of this preset
    PRESET_STATE_ENUM eState;

} PRESET_STRUCT;

typedef struct presets_object_struct
{
    // The decoder in which this service operates
    DECODER_OBJECT hDecoder;

    // List of bands which are a part of this service
    OSAL_OBJECT_HDL hBands;

    // Featured Favorites Bank ID Pool
    OSAL_OBJECT_HDL hID;

    // The Id which will be assigned to the
    // next band added to this service
    size_t tNextBandId;

    // "Current" band handle
    OSAL_LINKED_LIST_ENTRY hCurrentBand;

    // List of all presets in this service
    // (ordered by channel id)
    OSAL_OBJECT_HDL hPresets;

    // Id of the Presets service's category
    CATEGORY_ID tCategoryId;

    // Current Band Sync Category
    CATEGORY_ID tCategorySyncId;

    // Long / Short names provided to this service
    STRING_OBJECT hLongName;
    STRING_OBJECT hShortName;

    // Flag indicating if the service is performing
    // an update on the presets virtual category
    BOOLEAN bUpdatingCategory;

    // Flag indicating if the service needs
    // to do a complete rebuild of the category
    BOOLEAN bRebuildCategory;

    // Tag object
    TAG_OBJECT hTag;

    // Presets Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // SMS Callback shim
    PRESETS_CALLBACK_SHIM_STRUCT sCallbackShim;

    // Featured Favorites targeting information (how many banks and
    // presets should be processed)
	FAVORITES_RECEIVER_INFO_STRUCT sCapabilities;

    BOOLEAN bFeaturedFavoritesEnabled;

    // Flag blocking notifications from channel updates
    BOOLEAN bBlockNotifications;

} PRESETS_OBJECT_STRUCT;

// Structure used to describe each band
// which is a part of the presets service
typedef struct presets_band_descriptor_struct
{
    // Band object handle
    PRESET_BAND_OBJECT hBand;

    // ID for this band (used to order presets)
    size_t tBandId;

    // Indicate that this band was loaded from broadcast
	PRESET_BAND_TYPE_ENUM eType;

	// Signifying the relative ordering of the
    // Featured Favorites banks provided by a service
    // Unique index for the received featured favorites
	FAVORITES_BAND_ORDER tOrder;

    // Unique ID
	FAVORITES_BAND_ID tId;

    // Pointer to memory allocated for this
    // band's presets
    PRESET_STRUCT *pasPresets;

    // This band's capacity
    size_t tCapacity;

    //State of Band
    BOOLEAN bRemoved;

} PRESETS_BAND_DESCRIPTOR_STRUCT;

// Structure used to aid in updating
// the presets virtual category
typedef struct presets_category_update_struct
{
    // Flag indicating if an error was encountered
    BOOLEAN bError;

    // Flag indicating if an update has completed
    BOOLEAN bUpdateComplete;

    // The first band in the band list
    OSAL_LINKED_LIST_ENTRY hFirstBand;

    // The band currently being iterated
    OSAL_LINKED_LIST_ENTRY hCurrentBand;

    // The current band's descriptor
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc;

    // The current preset index being iterated
    size_t tPresetIndex;

    // Used to determine when to stop iterating
    CATEGORY_CHANNEL_INDEX tLastChannelIndex;

    // Flag indicating if a channel has been removed
    BOOLEAN bChanRemoved;

} PRESETS_CATEGORY_UPDATE_STRUCT;

// Structure used to aid in updating
// channel preset handles
typedef struct presets_handle_update_struct
{
    // CCache used to access the channels
    CCACHE_OBJECT hCCache;

    // The last channel Id encountered
    CHANNEL_ID tLastChannelId;

    // Flag indicating if we are
    // clearing handles
    BOOLEAN bClearHandles;

} PRESETS_HANDLE_UPDATE_STRUCT;

// Structure used to learn about the
// occurrence of a channel in the presets list
typedef struct presets_channel_occurrence_struct
{
    CHANNEL_ID tChannelId;
	SERVICE_ID tServiceId;
    PRESET_STRUCT *psFirstOccurrence;
    size_t tNumOccurrences;

} PRESETS_CHANNEL_OCCURRENCE_STRUCT;

// Structure used to aid featured favorites banks iteration for removal
typedef struct presets_favorites_remove_iterator_struct
{
    size_t tBandID;
    PRESETS_OBJECT hPresets;

} PRESETS_FAVORITES_REMOVE_ITERATOR_STRUCT;

// Structure used to update presets from channel event
typedef struct presets_update_from_channel_event_iterator_struct
{
    CHANNEL_OBJECT hChannel;
    PRESETS_OBJECT hPresets;

} PRESETS_UPDATE_FROM_CHANNEL_EVENT_ITERATOR_STRUCT;

// Structure used to aid in printing presets
typedef struct presets_print_struct
{
    // Tracks the number of characters
    // written throughout iteration of bands
    N32 n32CharsWritten;

    // File to print to
    FILE *psFile;

    // Format string each print
    // should utilize
    const char *pacFormat;

    // Presets service
    PRESETS_OBJECT hPresets;

} PRESETS_PRINT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SMSAPI_RETURN_CODE_ENUM eStart (
    PRESETS_OBJECT *phPresets,
    DECODER_OBJECT hDecoder,
    const char *pacLongName,
    const char *pacShortName
        );

static SMSAPI_RETURN_CODE_ENUM eStop (
    PRESETS_OBJECT hPresets
        );

static CATEGORY_ID tCategoryId (
    PRESETS_OBJECT hPresets
        );

static size_t tNumBands (
    PRESETS_OBJECT hPresets
        );

static STRING_OBJECT hShortName (
    PRESETS_OBJECT hPresets
        );

static STRING_OBJECT hLongName (
    PRESETS_OBJECT hPresets
        );

static SMSAPI_RETURN_CODE_ENUM eIterate (
    PRESETS_OBJECT hPresets,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        );

static SMSAPI_RETURN_CODE_ENUM eSetCurrentBand (
    PRESET_BAND_OBJECT hBand
        );

static PRESET_BAND_OBJECT hCurrentBand (
    PRESETS_OBJECT hPresets
        );

static PRESET_BAND_OBJECT hPreviousBand (
    PRESETS_OBJECT hPresets
        );

static PRESET_BAND_OBJECT hNextBand (
    PRESETS_OBJECT hPresets
        );

static N32 n32FPrintf (
    PRESETS_OBJECT hPresets,
    FILE *psFile
        );

static SMSAPI_RETURN_CODE_ENUM eNotifyOnChange (
    PRESETS_OBJECT hPresets,
    PRESETS_OBJECT_EVENT_CALLBACK vCallback,
    void *pvCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eFeaturedFavoritesStart (
    PRESETS_OBJECT hPresets,
    FAVORITES_BAND_CAPACITY tBandCapacity,
    FAVORITES_BAND_NUMBER tBandsNumber,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        );

static SMSAPI_RETURN_CODE_ENUM eFeaturedFavoritesStop (
    PRESETS_OBJECT hPresets,
    BOOLEAN bRemoveBands
        );

/* Object Private Prototypes */

static BOOLEAN bRegisterCategory (
    PRESETS_OBJECT_STRUCT *psObj
        );

static void vRemoveCategory (
    PRESETS_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRebuildCategory (
    PRESETS_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateCategory (
    PRESETS_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCompleteCategoryUpdate (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_CATEGORY_UPDATE_STRUCT *psUpdate
        );

static OSAL_LINKED_LIST_ENTRY hCreateBandDescriptor (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand,
    size_t tBandCapacity,
    size_t tBandId
        );

static BOOLEAN bAddPresetsForBand (
    PRESETS_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY hBandEntry
        );

static void vClearBandPresets (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc,
    BOOLEAN bClearNames,
    BOOLEAN bRemovePresets
        );

static BOOLEAN bRemoveBandFromService (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc
        );

static N16 n16IterateCategoryForUpdate (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static N16 n16RemoveAllChannels(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static PRESET_STRUCT *psGetPreset (
    PRESET_BAND_OBJECT hBand,
    OSAL_LINKED_LIST_ENTRY hBandEntry,
    size_t tPresetIndex
        );

static SMSAPI_RETURN_CODE_ENUM eIterateCommon (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        );

static BOOLEAN bUpdateChannelPresetHdls (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_STRUCT *psPreset,
    SERVICE_ID tPrevServiceId
        );

static void vChannelEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vEventCallbackShim(
    PRESETS_OBJECT hPresets,
    SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

SMSAPI_RETURN_CODE_ENUM  eRestoreFromConfigFile(
    PRESETS_OBJECT_STRUCT *psObj,
    BOOLEAN *pbAlreadyInConfig
        );

SMSAPI_RETURN_CODE_ENUM eUpdateCurrentBandTags(
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_OBJECT hBand
        );

static void vLoadReceiverCapabilities (
    PRESETS_OBJECT_STRUCT *psObj
        );

static void vSaveReceiverCapabilities (
    PRESETS_OBJECT_STRUCT *psObj
        );

static void vClearBandSequenceNumbers (
    PRESETS_OBJECT_STRUCT *psObj
        );

static void vRemoveExtraBands (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator
        );

static void vUpdateBandsCapacity (
    PRESETS_OBJECT_STRUCT *psObj,
    PRESET_BAND_SEARCH_FOR_TAG_ITERATOR_STRUCT *psIterator
        );

// Linked List Related

static N16 n16CompareBands (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16SortPresetsByServiceId (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16SortUserAndFeaturedFavorites (
    void *pvObj1,
    void *pvObj2
        );

static void vDestroyBandDescriptor (
    void *pvData
        );

static BOOLEAN bDiscoverChannelOccurrence (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdatePresetsFromChannelEvent (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bClearCurrentBandTags(
    TAG_OBJECT hTag,
    void *pvArg
        );

static N16 n16CompareFFBankIDs(
    void *pvData1,
    void *pvData2
        );

static BOOLEAN bIsUpdateNeeded(
    PRESETS_OBJECT_STRUCT *psObj
        );

static void vInitiateUpdate(
    PRESETS_OBJECT_STRUCT *psObj
        );

static PRESET_BAND_OBJECT hBrowseBands(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    SMSAPI_DIRECTION_ENUM eDirection
        );

static BOOLEAN bPresetsNotifyIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psBandDesc,
    PRESETS_OBJECT_STRUCT *psObj
        );

static BOOLEAN bSetAllFFBanksRemovedIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        );

static BOOLEAN bRemoveAllSelectedFFBanksIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        );

static BOOLEAN bRemoveFirstSelectedFFBankIterator(
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        );

static BOOLEAN bRemoveAllFavorites (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bSetSelectedFFBankRemovedIterator (
    PRESETS_BAND_DESCRIPTOR_STRUCT *psEntry,
    void *pvArg
        );

static BOOLEAN bSearchForFavoritesBandIterator (
    void *pvEntry,
    void *pvArg
        );

static BOOLEAN bClearBandSeqNumberIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bAddRemovePresetsIterator (
    void *pvData,
    void *pvArg
        );

#if SMS_DEBUG == 1

static BOOLEAN bPrintBands (
    void *pvData,
    void *pvArg
        );

#endif // SMS_DEBUG==1

  /***************/
 /** VARIABLES **/
/***************/

const PRESET_STRUCT gsEmptySlot =
{
    /*hBand =*/ PRESET_BAND_INVALID_OBJECT,
    /*hName =*/ STRING_INVALID_OBJECT,
    /*tServiceId =*/ SERVICE_INVALID_ID,
    /*hChannel =*/ CHANNEL_INVALID_OBJECT,
    /*tBandId =*/ 0,
    /*tPresetIndex =*/ 0,
    /*hEntry =*/ OSAL_INVALID_LINKED_LIST_ENTRY,
    /*eState =*/ PRESET_STATE_UNCHANGED

};

// Global instance of an interface for this object
const PRESETS_OBJECT_INTERFACE_STRUCT PRESETS =
{
    /*.eStart = */eStart,
    /*.eStop = */eStop,
    /*.tCategoryId = */tCategoryId,
    /*.tNumBands = */tNumBands,
    /*.hShortName = */hShortName,
    /*.hLongName = */hLongName,
    /*.eIterate = */eIterate,
    /*.eSetCurrentBand = */eSetCurrentBand,
    /*.hCurrentBand = */hCurrentBand,
    /*.hPreviousBand = */hPreviousBand,
    /*.hNextBand = */hNextBand,
    /*.n32FPrintf = */n32FPrintf,
    /*.eNotifyOnChange = */eNotifyOnChange,
    /*.eFeaturedFavoritesStart = */eFeaturedFavoritesStart,
    /*.eFeaturedFavoritesStop = */eFeaturedFavoritesStop
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _PRESETS_OBJ_H_
