/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _PRESET_BAND_OBJ_H_
#define _PRESET_BAND_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define PRESET_BAND_OBJECT_NAME "BAND"

// These definitions are names used for band related tags
#define PRESET_BAND_CAPACITY        "Capacity"
#define PRESET_BAND_CURRENT         "Current"

#define PRESET_BAND_LOADED          "Loaded"
#define PRESET_BAND_REMOVED         "Removed"

#define PRESET_ORDER                "Order"
#define PRESET_ID                   "ID"
#define PRESET_SEQ                  "Sequence"
#define PRESET_LONG_NAME            "LongName"
#define PRESET_VERBOSE_NAME         "VerboseName"
#define PRESET_DESCRIPTION          "Description"
#define PRESET_PURPOSE              "Purpose"

#define PRESET_NAME                 "Name" // the name assigned to the preset or band

// These definitions are names used for preset related tags
#define PRESET_TAG_NAME             "Preset" // the parent tag of a preset
#define PRESET_SERVICE_ID           "ServiceId"
#define PRESET_INDEX                "Index"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct preset_band_object_struct
{
    // Name of this BAND
    STRING_OBJECT hName;

    // Number of presets this band may store
    size_t tPresetCapacity;

    // Where this band resides in the preset service
    OSAL_LINKED_LIST_ENTRY hEntry;

    TAG_OBJECT hTag;

    // Indicate that this band was loaded from broadcast
    PRESET_BAND_TYPE_ENUM eType;

    // THE FEATURED FAVORITES BANKS FIELDS

    // Signifying the relative ordering of the
    // Featured Favorites banks provided by a service
    // Unique index for the received featured favorites
    FAVORITES_BAND_ORDER tOrder;

    // Unique ID
    FAVORITES_BAND_ID tId;

    // cycles from 0 ... 15, 0 ... each time an existing
    // bank is modified (identified by Bank ID)
    FAVORITES_BAND_SEQUENCE tSequence;

    // Optional Long Title of the Bank
    STRING_OBJECT hLongName;

    // Optional Verbose Title of the Bank
    STRING_OBJECT hVerboseName;

    // Optional string filed contains a description of the bank
    STRING_OBJECT hDescription;

    // Indicates a purpose of a bank
    FAVORITES_BAND_PURPOSE_MASK tPurpose;

    // State of Band
    BOOLEAN bRemoved;

} PRESET_BAND_OBJECT_STRUCT;

typedef struct preset_band_tag_iterator_struct
{
    PRESETS_OBJECT hPresets;
    OSAL_OBJECT_HDL hLL;
    PRESET_BAND_OBJECT hBand;
    BOOLEAN bPresets;

} PRESET_BAND_TAG_ITERATOR_STRUCT;

typedef struct preset_search_for_tag_iterator_struct
{
    size_t tIndex;
    TAG_OBJECT hPresetTag;
} PRESET_SEARCH_FOR_TAG_ITERATOR_STRUCT;

typedef struct presets_restore_struct
{
    PRESET_BAND_OBJECT hBand;
    OSAL_LINKED_LIST_ENTRY hEntry;
    size_t tPresetIndex;
    SERVICE_ID tServiceId;
    char *pacPresetName;

} PRESETS_RESTORE_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static PRESET_BAND_OBJECT hCreate (
    PRESETS_OBJECT hPresets,
    size_t tCapacity,
    const char *pacName
        );

static SMSAPI_RETURN_CODE_ENUM eDestroyPublic (
    PRESET_BAND_OBJECT hBand
        );

static STRING_OBJECT hName (
    PRESET_BAND_OBJECT hBand
        );

static STRING_OBJECT hLongName (
    PRESET_BAND_OBJECT hBand
        );

static SMSAPI_RETURN_CODE_ENUM eSetLongName (
    PRESET_BAND_OBJECT hBand,
    char *pacLongName
        );

static STRING_OBJECT hVerboseName (
    PRESET_BAND_OBJECT hBand
        );

static SMSAPI_RETURN_CODE_ENUM eSetVerboseName (
    PRESET_BAND_OBJECT hBand,
    char *pacVerboseName
        );

static STRING_OBJECT hDescription (
    PRESET_BAND_OBJECT hBand
        );

static SMSAPI_RETURN_CODE_ENUM eSetDescription (
    PRESET_BAND_OBJECT hBand,
    char *pacDescription
        );

static size_t tCapacity (
    PRESET_BAND_OBJECT hBand
        );

static PRESET_BAND_TYPE_ENUM eType (
    PRESET_BAND_OBJECT hBand
        );

static CHANNEL_ID tGetPreset (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex
        );

static STRING_OBJECT hGetPresetName (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex
        );

static SMSAPI_RETURN_CODE_ENUM eClearAllPresetsPublic (
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames
        );

static SMSAPI_RETURN_CODE_ENUM eSetPresetPublic (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eIterate (
    PRESET_BAND_OBJECT hBand,
    PRESETS_ITERATOR_HANDLER n16Iterator,
    void *pvIterateArg
        );

static SMSAPI_RETURN_CODE_ENUM eTune (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    BOOLEAN bLockOverride
        );

static N32 n32FPrintf (
    PRESET_BAND_OBJECT hBand,
    FILE *psFile
        );

/* Object Private Prototypes */

static PRESET_BAND_OBJECT hCreateBandLocal(
    PRESETS_OBJECT hPresets,
    size_t tCapacity,
    PRESET_BAND_PRIVATE_ATTRS_STRUCT *psAttrs,
    TAG_OBJECT hBandTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetPresetLocal(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName
        );

static SMSAPI_RETURN_CODE_ENUM eSavePresetToConfigFile(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex,
    PRESET_UPDATE_TYPE_ENUM eUpdateType,
    SERVICE_ID tServiceId,
    const char *pacName,
    BOOLEAN bCommit
        );

static STRING_OBJECT hGetBandName(
    TAG_OBJECT hBandTag
        );

static size_t tGetCapacity(
    TAG_OBJECT hBandTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetBandCapacity(
    TAG_OBJECT hBandTag,
    size_t tCapacity
        );

static BOOLEAN bGetLoaded(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetLoaded(
    TAG_OBJECT hParentTag,
    BOOLEAN bLoaded
        );

static BOOLEAN bGetRemoved(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetRemoved(
    TAG_OBJECT hParentTag,
    BOOLEAN bRemoved
        );

static FAVORITES_BAND_ORDER tGetOrder(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetOrder(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_ORDER tOrder
        );

static FAVORITES_BAND_ID tGetId(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetId(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_ID tId
        );

static FAVORITES_BAND_SEQUENCE tGetSequence(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetSequence(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_SEQUENCE tSequence
        );

static FAVORITES_BAND_PURPOSE_MASK tGetPurpose(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetPurpose(
    TAG_OBJECT hParentTag,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        );

static STRING_OBJECT hGetLongName(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetLongNameTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        );

static STRING_OBJECT hGetVerboseName(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetVerboseNameTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        );

static STRING_OBJECT hGetDescription(
    TAG_OBJECT hParentTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetDescriptionTag (
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        );

static char *pacGetPresetName(
    TAG_OBJECT hPresetTag
        );

static BOOLEAN bGetPresetIndex(
    TAG_OBJECT hPresetTag,
    size_t *tIndex
        );

static SMSAPI_RETURN_CODE_ENUM eClearAllPresets (
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bClearNames
        );

static SMSAPI_RETURN_CODE_ENUM eSetPreset (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eSetPresetName (
    PRESET_BAND_OBJECT hBand,
    size_t tPresetIndex,
    const char *pacName
        );

static SERVICE_ID tGetPresetServiceId(
    TAG_OBJECT hPresetTag
        );

static SMSAPI_RETURN_CODE_ENUM eSetNameTag(
    TAG_OBJECT hParentTag,
    STRING_OBJECT hString
        );

static BOOLEAN bGetCurrent(
    TAG_OBJECT bBandTag
        );

static BOOLEAN bUpdatePresets (
    PRESET_BAND_OBJECT hBand,
    SERVICE_ID const *ptArrangement,
    size_t tSIDCount,
    FAVORITES_BAND_CAPACITY tCapacity
        );

static BOOLEAN bUpdateOrder (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_ORDER tOrder
        );

static BOOLEAN bUpdatePurpose (
    PRESET_BAND_OBJECT hBand,
    FAVORITES_BAND_PURPOSE_MASK tPurpose
        );

static BOOLEAN bUpdateShortName (
    PRESET_BAND_OBJECT hBand,
    char const * pacShortName
        );

static BOOLEAN bUpdateLongName (
    PRESET_BAND_OBJECT hBand,
    char const * pacLongName
        );

static BOOLEAN bUpdateVerboseName (
    PRESET_BAND_OBJECT hBand,
    char const * pacVerboseName
        );

static BOOLEAN bUpdateDescription (
    PRESET_BAND_OBJECT hBand,
    char const * pacDescription
        );

static BOOLEAN bPresetBandTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        );

static BOOLEAN bSearchForPresetTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        );

static TAG_OBJECT hGetPresetTagByIndex(
    PRESET_BAND_OBJECT_STRUCT *psObj,
    size_t tPresetIndex
        );

static BOOLEAN bSetThePresets(
    void *pvData,
    void *pvArg
        );

static void vRemovePresetRestoreStruct(
    void *pvData
        );

static void vDestroyBandLocal(
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    BOOLEAN bNotify
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global instance of an interface for this object
const PRESET_BAND_OBJECT_INTERFACE_STRUCT BAND =
{
    /*.hCreate = */hCreate,
    /*.eDestroy = */eDestroyPublic,
    /*.hName = */hName,
    /*.hLongName = */hLongName,
    /*.eSetLongName = */eSetLongName,
    /*.hVerboseName = */hVerboseName,
    /*.eSetVerboseName = */eSetVerboseName,
    /*.hDescription = */hDescription,
    /*.eSetDescription = */eSetDescription,
    /*.eType = */eType,
    /*.tCapacity = */tCapacity,
    /*.eSetPreset = */eSetPresetPublic,
    /*.tGetPreset = */tGetPreset,
    /*.eSetPresetName = */eSetPresetName,
    /*.hGetPresetName = */hGetPresetName,
    /*.eClearAllPresets = */eClearAllPresetsPublic,
    /*.eIterate = */eIterate,
    /*.eTune = */eTune,
    /*.n32FPrintf = */n32FPrintf
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _PRESET_BAND_OBJ_H_
