/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _MOVIES_MGR_OBJ_H_
#define _MOVIES_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "db_util.h"
#include "theater_obj.h"
#include "movies_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define MOVIES_MGR_OBJECT_NAME "MovieService"

// Constant which defines the size of the buffer used by
// traffic for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define MOVIES_SHARED_BUFFER_LEN ( MOVIES_MAX_SQL_STRING_LENGTH )

// Constant defines the minimum device location
// delta which movies wants to know about
#define MOVIES_DEVICE_DISTANCE_THRESHOLD (3)

#define MOVIES_SUPPORTED_OPTIONS (              \
    (DATASERVICE_OPTIONS_MASK)                  \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      | \
    DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES   \
    )

// Service-private ids for ratings & ratings systems
#define MOVIE_RSYS_MPAA (0)
#define MOVIE_RSYS_CAN1 (1)
#define MOVIE_RSYS_CAN2 (2)
#define MOVIE_RSYS_QUEBEC (3)
#define MOVIE_RSYS_MEX (4)

#define MOVIE_RCODE_MPAA_UNRATED (0)
#define MOVIE_RCODE_MPAA_G (1)
#define MOVIE_RCODE_MPAA_PG (2)
#define MOVIE_RCODE_MPAA_PG13 (3)
#define MOVIE_RCODE_MPAA_R (4)
#define MOVIE_RCODE_MPAA_NC17 (5)

#define MOVIE_RCODE_CAN1_UNRATED (0)
#define MOVIE_RCODE_CAN1_G (1)
#define MOVIE_RCODE_CAN1_PG (2)
#define MOVIE_RCODE_CAN1_14A (3)
#define MOVIE_RCODE_CAN1_18A (4)
#define MOVIE_RCODE_CAN1_R (5)

#define MOVIE_RCODE_CAN2_UNRATED (0)
#define MOVIE_RCODE_CAN2_G (1)
#define MOVIE_RCODE_CAN2_PG (2)
#define MOVIE_RCODE_CAN2_14 (3)
#define MOVIE_RCODE_CAN2_18 (4)
#define MOVIE_RCODE_CAN2_E (5)
#define MOVIE_RCODE_CAN2_R (6)

#define MOVIE_RCODE_QUEBEC_NONCLASS (0)
#define MOVIE_RCODE_QUEBEC_G (1)
#define MOVIE_RCODE_QUEBEC_13 (2)
#define MOVIE_RCODE_QUEBEC_16 (3)
#define MOVIE_RCODE_QUEBEC_18 (4)

#define MOVIE_RCODE_MEX_SC (0)
#define MOVIE_RCODE_MEX_AA (1)
#define MOVIE_RCODE_MEX_A (2)
#define MOVIE_RCODE_MEX_B (3)
#define MOVIE_RCODE_MEX_B15 (4)
#define MOVIE_RCODE_MEX_C (5)
#define MOVIE_RCODE_MEX_D (6)

  /**************/
 /** TYPEDEFS **/
/**************/

typedef enum movie_ota_update_state_enum
{
    MOVIE_OTA_UPDATE_STATE_INITIAL,
    MOVIE_OTA_UPDATE_STATE_LOOKING_FOR_UPDATE,
    MOVIE_OTA_UPDATE_STATE_IN_PROGRESS,
    MOVIE_OTA_UPDATE_STATE_COMPLETE,
    MOVIE_OTA_UPDATE_STATE_STABLE,
    MOVIE_OTA_UPDATE_STATE_ERROR

} MOVIE_OTA_UPDATE_STATE_ENUM;

typedef struct movie_description_control_struct
{
    // The list of movie descriptions
    OSAL_OBJECT_HDL hDescriptions;

    // Version that we're currently collected
    MOVIE_VERSION tVersion;

    // Time at which update completed
    UN32 un32UTCsec;

    // State of the times data
    MOVIE_OTA_UPDATE_STATE_ENUM eState;

} MOVIE_DESCRIPTION_CONTROL_STRUCT;

typedef struct movie_times_struct
{
    // The list of times entries
    OSAL_OBJECT_HDL hTimes;

    // Version that we're currently collecting
    MOVIE_VERSION tVersion;

    // Time at which update completed
    UN32 un32UTCsec;

    // State of the times data
    MOVIE_OTA_UPDATE_STATE_ENUM eState;

    // Reusable object for message processing
    THEATER_TIMES_OBJECT hScratchTimes;

} MOVIE_TIMES_CONTROL_STRUCT;

typedef struct movie_theater_times_entry_struct
{
    // the ID for this theater in hTheaterList that
    // is populated after we verify the theater exists during
    // OTA data processing
    THEATER_ID tTheaterID;

    THEATER_TIMES_OBJECT hShowTimes;

} MOVIE_THEATER_TIMES_ENTRY_STRUCT;

typedef struct movie_theater_entry_struct
{
    // the entry for this theater in hTheaterList
    OSAL_LINKED_LIST_ENTRY hEntry;

    // List of MOVIES_DSRL_DESC_STRUCT
    OSAL_OBJECT_HDL hDSRLList;

    THEATER_OBJECT hTheater;
    THEATER_ID tID;

} MOVIE_THEATER_ENTRY_STRUCT;

typedef struct movies_app_object_struct
{
    // Structure which groups movie time data
    MOVIE_TIMES_CONTROL_STRUCT *psCurrentTimes;

    // Structure which groups movie description data
    MOVIE_DESCRIPTION_CONTROL_STRUCT *psCurrentMovies;

    // List of ratings
    OSAL_OBJECT_HDL hRatingsList;

    // List of theaters
    OSAL_OBJECT_HDL hTheaterList;

    // "Dummy" movie object
    MOVIE_OBJECT hDummyMovie;

} MOVIES_APP_OBJECT_STRUCT;

typedef struct movies_mgr_object_struct
{
    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Data Expire event
    DATASERVICE_TIMED_EVENT_HDL hDataExpireEvent;

    // Application facing object
    MOVIES_APP_OBJECT_STRUCT *psAppObj;

    // Flag indicating if this service will be
    // performing reference database updates
    BOOLEAN bRefDBUpdatesEnabled;

    // List of MOVIES_DSRL_DESC_STRUCT
    OSAL_OBJECT_HDL hDSRLList;

    // Data structure used for broadcast-specific parsing
    MOVIES_INTERFACE_OBJECT hMoviesInterfaceData;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MOVIES_MAX_SQL_BIND_PARAMS];

    // DB Connection Handles, File Paths, and Shared Buffer
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    char *pacRefDatabaseDirPath;
    const char *pacCurRefDatabaseFilePath;
    const char *pacNextRefDatabaseFilePath;

    SQL_INTERFACE_OBJECT hSQLPersistConnection;

    SQL_PREPARED_STATEMENT_HANDLE hInsertTheaterTimesStmt;
    SQL_PREPARED_STATEMENT_HANDLE hInsertDescriptionStmt;
    SQL_PREPARED_STATEMENT_HANDLE hClearMoviesStmt;

    char acBuffer[MOVIES_MAX_SQL_STRING_LENGTH];

    MOVIE_DESCRIPTION_CONTROL_STRUCT *psInProgressMovies;

    MOVIE_TIMES_CONTROL_STRUCT *psInProgressTimes;

    MOVIE_VERSION tBaselineVersion;

    // Scratch memory used to read/write movie ratings
    MOVIE_RATING_STRUCT asRatings[MOVIES_MAX_MOVIE_RATINGS];

    // Memory used to read/write blob data for theater times
    UN16 aun16TimesBlobData[THEATER_TIMES_BLOB_MAX_NUM_UN16];

} MOVIES_MGR_OBJECT_STRUCT;

typedef struct movie_ratings_entry_struct
{
    MOVIE_RATING_SYSTEM_ENUM eSystem;
    MOVIE_RATING_SYS_CODE tSysCode;
    STRING_OBJECT hSystemText;
    MOVIE_RATING_CODE tRatingCode;
    STRING_OBJECT hRatingText;
    MOVIE_RATING_ENUM eRating;

} MOVIE_RATINGS_ENTRY_STRUCT;

typedef struct movies_dsrl_target_desc_struct
{
    // The list of state IDs that are represented
    // by this target.  This data is gathered from the
    // database as theaters are read in
    OSAL_OBJECT_HDL hStateList;

    // A list of THIDs that are associated
    // with this target
    OSAL_OBJECT_HDL hTheaterList;

    LOCATION_OBJECT hLocation;

} MOVIES_DSRL_TARGET_DESC_STRUCT;

typedef struct movies_dsrl_desc_struct
{
    // This descriptor's entry in the list of DSRL descriptors
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The list of MOVIES_DSRL_TARGET_DESC_STRUCTs
    // that define the multiple targets associated with
    // this DSRL
    OSAL_OBJECT_HDL hTargetList;

    // Target information
    DSRL_OBJECT hDSRL;

} MOVIES_DSRL_DESC_STRUCT;

typedef struct movies_dsrl_remove_target_iterator_struct
{
    DSRL_OBJECT hTargetDSRL;
    OSAL_LINKED_LIST_ENTRY hFoundEntry;

} MOVIES_DSRL_REMOVE_TARGET_ITERATOR_STRUCT;

typedef struct movies_target_iterator_struct
{
    BOOLEAN bFound;
    STATE_ID tStateID;

} MOVIES_TARGET_ITERATOR_STRUCT;

typedef struct movies_build_dsrl_iterator_struct
{
    MOVIES_MGR_OBJECT_STRUCT *psMgr;
    BOOLEAN bSuccess;
    BOOLEAN bRefresh;
    BOOLEAN bDSRLUpdated;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc;

} MOVIES_BUILD_DSRL_ITERATOR_STRUCT;

typedef struct movies_replace_dsrl_iterator_struct
{
    THEATER_OBJECT hTheater;
    BOOLEAN bDSRLUpdated;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc;

} MOVIES_REPLACE_DSRL_ITERATOR_STRUCT;

typedef struct movies_desc_update_iterator_struct
{
    MOVIE_OBJECT hUpdatedMovie;
    BOOLEAN bInTheater;

} MOVIES_DESC_UPDATE_ITERATOR_STRUCT;

typedef struct movies_db_write_iterator_struct
{
    BOOLEAN bError;
    MOVIES_MGR_OBJECT_STRUCT *psMgr;
    MOVIE_OBJECT hMovie;

} MOVIES_DB_WRITE_ITERATOR_STRUCT;

typedef struct movies_theater_dsrl_entry_struct
{
    MOVIES_DSRL_DESC_STRUCT *psDesc;
    BOOLEAN bInDSRL;
    BOOLEAN bMayDestroy;
    OSAL_LINKED_LIST_ENTRY hEntry;

} MOVIES_THEATER_DSRL_ENTRY_STRUCT;

typedef struct movies_db_query_build_theater_struct
{
    MOVIES_MGR_OBJECT_STRUCT *psMgr;
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc;
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget;
    STATE_ID tCurrentStateID;
    BOOLEAN bRefresh;
    BOOLEAN *pbDSRLUpdated;

} MOVIES_DB_QUERY_BUILD_STATION_STRUCT;

// Structure used to perform database queries
typedef struct movies_db_query_result_struct
{
    BOOLEAN bResultantRows;
    MOVIES_DB_ROW_UNION uDbRow;

} MOVIES_DB_QUERY_RESULT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static MOVIES_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

static SMSAPI_RETURN_CODE_ENUM eIterate (
    MOVIES_SERVICE_OBJECT hMoviesService,
    MOVIES_ITERATOR bIterator,
    void *pvIteratorArg
        );

static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

/* Object Private Prototypes */

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static MOVIES_APP_OBJECT_STRUCT *psGetAppFacingObject(
    MOVIES_SERVICE_OBJECT hMoviesService
        );

static BOOLEAN bHandleServiceReady (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static MOVIE_DESCRIPTION_CONTROL_STRUCT *psBuildMovieDescStruct (
    SMS_OBJECT hOwner
        );

static void vFreeMovieDescStruct (
    MOVIE_DESCRIPTION_CONTROL_STRUCT *psDesc
        );

static MOVIE_TIMES_CONTROL_STRUCT *psBuildMovieTimesStruct (
    SMS_OBJECT hOwner
        );

static void vFreeMovieTimesStruct (
    MOVIE_TIMES_CONTROL_STRUCT *psTimes
        );

static void vDestroyTheaterTimesEntry (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimesEntry
        );

static BOOLEAN bInitAppFacingObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vRemoveMoviesAsExpired (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_APP_OBJECT_STRUCT *psAppObj
        );

static void vDestroyObject (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleCreateList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static MOVIES_DSRL_DESC_STRUCT *psCreateDSRLDesc (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vReleaseDSRLDesc (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static void vReleaseDSRLTargetDesc (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        );

static BOOLEAN bHandleModifyList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleAddTargetList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleReplaceTargetList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleRefreshList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleRemoveTargets (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bClearDSRLEntryInTheaterEntry(
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        );

static BOOLEAN bFindTheaterEntryInTargetDesc(
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_REMOVE_TARGET_ITERATOR_STRUCT *psIterator
        );

static void vHandleDeleteList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRemoveDSRLDescFromTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bPrepareTheaterEntriesForReplace (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bPostProcessTheaterEntriesForReplace (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bIterateTheaterEntryForDescriptionUpdate (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIE_OBJECT hUpdatedMovie
        );

static BOOLEAN bIterateMoviesForDescriptionUpdate (
    THEATER_OBJECT hTheater,
    MOVIE_OBJECT hMovie,
    MOVIES_DESC_UPDATE_ITERATOR_STRUCT *psUpdate
        );

static BOOLEAN bIterateDSRLEntryForDescriptionUpdate (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    THEATER_OBJECT hTheater
        );

static void vRemoveAllDSRLEntriesFromTheaterEntry (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry
        );

static BOOLEAN bRemoveDSRLEntryFromTheaterEntry (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bPrepareDSRLEntryForReplace (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bPostProcessDSRLEntryForReplace (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIES_REPLACE_DSRL_ITERATOR_STRUCT *psReplace
        );

static MOVIES_DSRL_DESC_STRUCT *psCreateDSRLDesc (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static MOVIES_DSRL_TARGET_DESC_STRUCT *psBuildTargetDesc (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    LOCATION_OBJECT hLocation
        );

static BOOLEAN bBuildTargetsForDSRL (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bBuildDSRL (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bMoveToReady,
    BOOLEAN bRefresh
        );

static BOOLEAN bIterateTargetsForRefresh (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    void *pvUnused
        );

static BOOLEAN bLoadTheatersForDSRL (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    MOVIES_BUILD_DSRL_ITERATOR_STRUCT *psBuildDSRL
        );

static BOOLEAN bProcessSelectTheatersAndBuildObjects (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MOVIES_DB_QUERY_BUILD_STATION_STRUCT *psBuild
        );

static void vFreeTheaterRowObjects(
    THEATER_ROW_STRUCT *psTheaterRow
        );

static MOVIE_THEATER_TIMES_ENTRY_STRUCT *psFindTheaterTimes (
    MOVIES_APP_OBJECT_STRUCT *psAppObj,
    OSAL_OBJECT_HDL hTimesList,
    THEATER_ID tID
        );

static BOOLEAN bAddTheater (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    THEATER_OBJECT hTheater,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    BOOLEAN bRefresh,
    BOOLEAN *pbAddedToDSRL
        );

static void vFinalizeDSRLEntry (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static void vDestroyTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        );

static void vPruneTheaters(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPruneTheatersIterator (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    void *pvArg
        );

static BOOLEAN bProcessPayload (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessDescription (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION tVersion
        );

static BOOLEAN bProcessTimes (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION tVersion
        );

static BOOLEAN bAddShowTimes(
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hTimesList,
    THEATER_TIMES_OBJECT hShowTimes,
    THEATER_ID tTHID
        );

static void vAttemptToMoveToCurrent (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static void vMoveOTADataToComplete (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateTheatersWithTimes (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bLinkTimesWithTheaters (
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry,
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateTheaterEntryInDSRL (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    MOVIE_THEATER_ENTRY_STRUCT *psTheaterEntry
        );

static BOOLEAN bIsStateInDSRLs (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    STATE_ID tStateID
        );

static BOOLEAN bIsStateInDSRLIterator (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    MOVIES_TARGET_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bIsStateInTargetIterator (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psTarget,
    MOVIES_TARGET_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bIsTHIDInList (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    THEATER_ID tTheaterIDToFind,
    OSAL_LINKED_LIST_ENTRY *phEntry
        );

static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    MOVIE_VERSION *ptBaselineVersion
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bLoadRatingsTable (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessSelectRatingsTableResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bLoadPersistentData (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessControlTableResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bProcessMovieDescFromDB (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bProcessMovieTimesFromDB (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static void vStartAgeoutTimer (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static void vStopAgeoutTimer (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bSaveTimesToDatabase (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bWriteTimesToDB (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimesEntry,
    MOVIES_DB_WRITE_ITERATOR_STRUCT *psIteratorArg
        );

static BOOLEAN bSaveDescToDatabase (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bWriteMovieDescToDB (
    MOVIE_OBJECT hMovie,
    MOVIES_DB_WRITE_ITERATOR_STRUCT *psIteratorArg
        );

static void vSaveControlDataToDatabase(
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bAddRatingsEntry (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow
        );

static void vDestroyRatingsEntry (
    MOVIE_RATINGS_ENTRY_STRUCT *psEntry
        );

static void vUpdateAllDSRLStates (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DSRL_STATE_ENUM eState
        );

static void vSendAllUpdatingDSRLsToError (
    MOVIES_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateAllDSRLStateIterator (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_STATE_ENUM eState
        );

static BOOLEAN bErrorAllUpdatingDSRLs (
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc,
    void *pvUnused
        );

static N16 n16CompareTargetDesc (
    MOVIES_DSRL_TARGET_DESC_STRUCT *psObj1,
    MOVIES_DSRL_TARGET_DESC_STRUCT *psObj2
        );

static N16 n16FindDSRLInTheater (
    MOVIES_THEATER_DSRL_ENTRY_STRUCT *psDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static N16 n16CompareTheaterEntry (
    MOVIE_THEATER_ENTRY_STRUCT *psObj1,
    MOVIE_THEATER_ENTRY_STRUCT *psObj2
        );

static N16 n16CompareTheaterTimes (
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimes1,
    MOVIE_THEATER_TIMES_ENTRY_STRUCT *psTimes2
        );

static N16 n16CompareRatingsEntry (
    MOVIE_RATINGS_ENTRY_STRUCT *psObj1,
    MOVIE_RATINGS_ENTRY_STRUCT *psObj2
        );

static N16 n16SortDSRLByDistance (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    MOVIES_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bDeleteTheaterFromDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ID tID
        );

static BOOLEAN bInsertTheaterIntoDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow
        );

static BOOLEAN bUpdateTheaterInDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    THEATER_ROW_STRUCT *psTheaterRow,
    BOOLEAN bAmenitiesUpdated
        );

static BOOLEAN bCheckTheaterInDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    THEATER_ROW_STRUCT  *psTheaterRow
        );

static BOOLEAN bCopyTheaterRowValues (
    THEATER_ROW_STRUCT *psFromTheaterRow,
    THEATER_ROW_STRUCT *psToTheaterRow,
    BOOLEAN bAmenitiesUpdated
        );

static BOOLEAN bPrepareTheaterColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    THEATER_ROW_STRUCT *psTheaterRow
        );

static BOOLEAN bPrepareRatingsColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow
        );

static void vSetError (
    MOVIES_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bReadTheaterFromDB (
    THEATER_ROW_STRUCT *psTheaterRow,
    SQL_QUERY_COLUMN_STRUCT *psColumn
        );

static void vSetInsertDescriptionStmtParams (
    MOVIE_ID tID,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psName,
    UN8 un8RunTime,
    STRING_OBJECT hActors,
    MOVIE_MULTI_LANG_FIELD_STRUCT *psSynopsis,
    size_t tNumRatings,
    MOVIE_RATING_STRUCT *pasRatings,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const MOVIES_INTERFACE_STRUCT MOVIES =
{
    /*.hStart = */hStart,
    /*.vStop = */(void (*) (MOVIES_SERVICE_OBJECT hMoviesService))DATASERVICE_IMPL_vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (MOVIES_SERVICE_OBJECT hMoviesService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (MOVIES_SERVICE_OBJECT hMoviesService))DATASERVICE_IMPL_eErrorCode,
    /*.eIterate = */eIterate,
    /*.eGetReferenceDataVersion = */eGetReferenceDataVersion
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsMoviesStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};


  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _MOVIES_MGR_OBJ_H_
