/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _MODULE_OBJ_H_
#define _MODULE_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "string_obj.h"
#include "sms_update.h"
#include "tag_obj.h"
#include "radio.h"

#include "module_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define MODULE_OBJECT_NAME "MODULE"

// tag names
#define GMT_OFFSET_TAG   "GmtOffset"

// default tag values
#define GMT_OFFSET_DEFAULT     (-300)

#define MODULE_TRUE_TEXT       "TRUE"
#define MODULE_FALSE_TEXT      "FALSE"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef enum module_decoder_status_enum
{
    MODULE_DECODER_STATUS_ACTIVE=0,
    MODULE_DECODER_STATUS_RESETTING,
    MODULE_DECODER_STATUS_STOPPING,
    MODULE_DECODER_STATUS_APP_MODE_REMOVAL,
    MODULE_DECODER_STATUS_UNKNOWN

} MODULE_DECODER_STATUS_ENUM;

typedef struct module_package_struct
{
    // A mutex which provides a way to
    // signal a waiter that new data has arrived.
    OSAL_OBJECT_HDL hMutex;

    // The number of bytes (size of) this package
    size_t tNumBytes;

    // The pointer to it's data
    UN8 const *pun8Data;

} MODULE_PACKAGE_STRUCT;

typedef struct module_data_cache_struct
{
    // Sirius ID / ESN constant string object
    STRING_OBJECT hESN;

    // Activation Package Response
    MODULE_PACKAGE_STRUCT sPackage;

} MODULE_DATA_CACHE_STRUCT;

typedef struct module_support_struct
{
    // IR Support
    BOOLEAN bIRSupported;
    // Advanced IR Support
    BOOLEAN bAdvancedIRSupported;
} MODULE_SUPPORT_STRUCT;

typedef struct module_object_struct
{
    /* THINGS PROVIDED BY APPLICATION */

    // SRM this object is attached to
    SRM_OBJECT hSRM;
    const char *pacSRMName;

    /* THINGS TO INITIALIZE ONCE */

    // Module object name
    const char *pacObjectName;

    // Module Id
    MODULE_ID tId;

    // Module Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // Services task for this module
    SMS_TASK_HANDLE hServicesTask;

    // Linked list of decoder descriptors associated with this Module
    OSAL_OBJECT_HDL hDecoderList;

    // Radio Initialized
    BOOLEAN bRadioInitialized;

    // Device to use
    FILE *psDevice;

    // STI Connection Hdl
    STI_HDL hSTI;

    /* THINGS TO RE-INITIALIZE ON DEMAND */

    // Object's operational mode mask
    SMS_MODE_MASK tCurrentModes;

    // Object Error Codes
    MODULE_ERROR_CODE_ENUM eErrorCode;

    // State that the object is in
    MODULE_STATE_ENUM eState;

    // Module Config Cache
    MODULE_DATA_CACHE_STRUCT sCache;

    // config manager tag
    TAG_OBJECT hTag;

    STRING_OBJECT hTrueString;
    STRING_OBJECT hFalseString;

    // Object which contains radio specific data
    RADIO_PRIVATE_DATA_OBJECT hRadioData;

    // flag to indicate that vUninitObject will need to be called when handling
    // an event in the STOPPED state
    BOOLEAN bUninitialized;

    // Features support
    MODULE_SUPPORT_STRUCT sSupport;

    // Module version object
    MODULE_VERSION_OBJECT hModuleVersion;

    UN8 un8UpdateProgress;
    SMSAPI_RETURN_CODE_ENUM eFWUpdateFileLoadError;

    MODULE_EVENT_DATA_STRUCT sEventData;

    // Time Update Notification Handle
    OSAL_TIME_NOTIFICATION_OBJECT hTimeNotificationHandle;

    // Object reference counter
    UN16 un16RefCounter;

} MODULE_OBJECT_STRUCT;

typedef struct module_decoder_dispatcher_struct
{
    // Event type to dispatch
    SMS_EVENT_TYPE_ENUM eType;

    // Argument to dispatch
    void *pvEventArg;

    // Dispatcher callback
    DISPATCH_DECODER_EVENT_CALLBACK bDispatch;

    // Flag to dispatch to only DECODERs with Eng.Data enabled
    BOOLEAN bOnlyIfEngineeringDataRequested;

} MODULE_DECODER_DISPATCHER_STRUCT;

typedef struct module_decoder_entry_struct
{
    DECODER_OBJECT hDecoder;
    BOOLEAN bValid;
    BOOLEAN bEngineeringDataEnabled;

} MODULE_DECODER_ENTRY_STRUCT;

typedef struct module_decoder_in_reset_struct
{
    DECODER_OBJECT hDecoder;
    BOOLEAN bAllInReset;

} MODULE_DECODER_IN_RESET_STRUCT;

typedef struct module_decoder_enable_engineering_data
{
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification;
    DECODER_OBJECT hDecoder;
    UN8 un8NumEngineeringDataEnabled;
    BOOLEAN bDisableAll;

} MODULE_DECODER_ENABLE_ENGINEERING_DATA;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static MODULE_OBJECT hGet (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_EVENT_MASK tEventRequestMask,
    MODULE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vRelease (
    MODULE_OBJECT hModule
        );

static MODULE_STATE_ENUM eState (
    MODULE_OBJECT hModule
        );

static MODULE_ERROR_CODE_ENUM eErrorCode (
    MODULE_OBJECT hModule
        );

static MODULE_EVENT_MASK tEventMask (
    MODULE_OBJECT hModule
        );

static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        );

static BOOLEAN bUpdate (
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tMask
        );

static N16 n16GmtOffsetGet (
    MODULE_OBJECT hModule
        );

static SMSAPI_RETURN_CODE_ENUM eGmtOffsetSet (
    MODULE_OBJECT hModule,
    N16 n16GmtOffset
        );

static BOOLEAN bDaylightSavingsTimeObservedGet (
    MODULE_OBJECT hModule
        );

static SMSAPI_RETURN_CODE_ENUM eDaylightSavingsTimeObservedSet (
    MODULE_OBJECT hModule,
    BOOLEAN bDaylightSavingsTimeObserved
        );

static STRING_OBJECT hESN (
    MODULE_OBJECT hModule
        );

static SMSAPI_RETURN_CODE_ENUM eSubscription (
    MODULE_OBJECT hModule,
    MODULE_SUBSTATUS_ENUM *peStatus,
    MODULE_SUBSTATUS_REASON_CODE *ptReasonCode,
    UN32 *pun32UTCSuspendDate,
    STRING_OBJECT hReasonText,
    STRING_OBJECT hPhoneNumber,
    BOOLEAN *pbIsAudioSubscribed
        );

static SMSAPI_RETURN_CODE_ENUM ePackage (
    MODULE_OBJECT hModule,
    MODULE_PACKGE_CMD_ENUM eCmd,
    UN8 un8PackageIndex,
    UN8 *pun8Resp,
    size_t tRespSize,
    size_t *ptBytesRxd
        );

/* Object Private Prototypes */

static BOOLEAN bSetTime (
    MODULE_OBJECT_STRUCT const *psObj,
    SMS_EVENT_MODULE_SET_TIME_STRUCT const *psSetTime
        );

static MODULE_OBJECT_STRUCT *psCreateObject (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_ID tId,
    MODULE_EVENT_MASK tEventRequestMask,
    MODULE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static  void vDestroyObject (
    MODULE_OBJECT_STRUCT *psObj
        );

static N16 n16CompareDecoders (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareDecoderName (
    void *pvArg1,
    void *pvArg2
        );

static BOOLEAN bResetDecoderIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bAllInResetIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bReleaseDecoderIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bInitializeObject (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vUpdateState (
    MODULE_OBJECT_STRUCT *psObj,
    MODULE_STATE_ENUM eNextState
        );

static MODULE_STATE_ENUM eTransitionToErrorState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher,
    MODULE_ERROR_CODE_ENUM eErrorCode
        );

static MODULE_STATE_ENUM eTransitionToInitialState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static MODULE_STATE_ENUM eTransitionToReadyState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static MODULE_STATE_ENUM eTransitionToStoppedState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static MODULE_STATE_ENUM eTransitionToUpdatingState (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static void vEventHandler (
    MODULE_OBJECT hModule,
    SMS_EVENT_HDL hEvent
        );

static BOOLEAN bDecoderIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bDispatchToAll (
    void *pvDescriptor,
    SMS_EVENT_TYPE_ENUM eType,
    void *pvEventArg
        );

static void vAssignToDataServices (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vSetError (
    MODULE_OBJECT_STRUCT *psObj,
    MODULE_ERROR_CODE_ENUM eErrorCode
        );

static void vGetTimeParams (
    MODULE_OBJECT_STRUCT *psObj
        );

static SMSAPI_RETURN_CODE_ENUM eFirmwareUpdate (
    MODULE_OBJECT hModule,
    const char *pcFirmwareFileName
        );

static SMSAPI_RETURN_CODE_ENUM eUpdateProgress (
    MODULE_OBJECT hModule,
    UN8 *pun8UpdateProgress
        );

static SMSAPI_RETURN_CODE_ENUM eEventData (
    MODULE_OBJECT hModule,
    MODULE_EVENT_DATA_STRUCT *psEventData
        );

static BOOLEAN bProcessModuleEventMaskChange (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_EVENT_MODULE_EVENT_MASK_STRUCT const *psModuleEventMask
        );

static SMSAPI_RETURN_CODE_ENUM eProcessModuleEngineeringDataChange (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_EVENT_MODULE_ENGDATA_STRUCT const *psModModifyEngData
        );

static size_t tCopyPackageData (
    MODULE_OBJECT_STRUCT *psModule,
    UN32 un32Timeout,
    UN8 *pun8Resp,
    size_t tRespSize
        );

static void vTimeUpdateNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        );

static BOOLEAN bAssociateWithDecoder (
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bReleaseParentSRM (
    MODULE_OBJECT_STRUCT *psObj
        );

/* Shutdown management */

static BOOLEAN bObjectBusy (
    MODULE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bTryStop (
    MODULE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitiateObjectRelease (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static BOOLEAN bReleaseDecoders (
    MODULE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPostFinalStop (
    MODULE_OBJECT_STRUCT *psObj
        );

/* Event handlers */

static void vHandleInitEvent (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vHandleRadioReadyEvent (
    MODULE_OBJECT_STRUCT *psObj,
    STI_HDL hSTI
        );

static void vHandleRadioReleasedEvent (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vHandleAssociateEvent (
    MODULE_OBJECT_STRUCT *psObj,
    const SMS_EVENT_MODULE_ASSOCIATE_STRUCT *psAssociate
        );

static void vHandleUnassociateEvent (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vHandleReleaseEvent (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleAppInitiatedRelease (
    MODULE_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleGeneralRelease (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vHandleStopEvent (
    MODULE_OBJECT_STRUCT *psObj
        );

static void vHandleRemoveDecoderEvent (
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

static void vHandleResetEvent (
    MODULE_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

  /***************/
 /** VARIABLES **/
/***************/

// MODULE Object Defaults
static const MODULE_OBJECT_STRUCT gsObjectDefaults =
{
    // SRM this object is attached to
    SRM_INVALID_OBJECT,
    NULL,

    // Module object name
    NULL,

    // Module Id
    -1,

    // Module Update Control Structure
    {
        // Object update masks
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,

        // Object update callback list
        OSAL_INVALID_OBJECT_HDL,
        OSAL_INVALID_LINKED_LIST_ENTRY,

        // Object being updated
        NULL,

        // In-active callbacks
        FALSE

    },

    // Event Handler
    SMS_INVALID_EVENT_HANDLER,

    // Services task for this module
    SMS_INVALID_TASK_HANDLE,

    // Linked list of decoder descriptors associated with this Module
    OSAL_INVALID_OBJECT_HDL,

    // Radio Initialized
    FALSE,

    // Device to use
    NULL,

    // STI-Handle
    STI_INVALID_HDL,

    // Object's operational mode mask
    SMS_MODE_NONE,

    // Object Error Codes
    MODULE_ERROR_CODE_NONE,

    // State that the object is in
    MODULE_STATE_INITIAL,

    // Module Config Cache
    {
        // Sirius ID / ESN constant string object
        STRING_INVALID_OBJECT,

        // Package data
        {
            OSAL_INVALID_OBJECT_HDL,
            0,
            NULL
        }
    },

    // Tag object
    TAG_INVALID_OBJECT,

    // hTrueString
    NULL,

    //hFalseString
    NULL,

    // hRadioData - Pointer to contain radio specific data
    RADIO_PRIVATE_DATA_INVALID_OBJECT,

    // bUninitialized
    FALSE,

    // Supported Features
    {
        FALSE,  // IR
        FALSE   // Advanced IR
    },

    //hModuleVersion
    MODULE_VERSION_INVALID_OBJECT,

    //un8UpdateProgress
    100,

    //eFWUpdateFileLoadError
    SMSAPI_RETURN_CODE_SUCCESS,

    //sEventData
    { {0,0,0,0,0,0,0,0,0,0} },

    // hTimeNotificationHandle
    OSAL_TIME_NOTIFICATION_INVALID_OBJECT,

    // un16RefCounter
    0
};

static const MODULE_EVENT_DATA_STRUCT gsDefaultEventData =
{
    // aun16Data[10]
    {
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0,
        (UN16)0
    }
};

static const SMS_EVENT_MODULE_EVENT_STRUCT gsDefaultModuleEventData =
{
     (MODULE_EVENT_MASK)0,
     (MODULE_OBJECT_EVENT_CALLBACK)NULL,
     NULL
};

// MODULE Object configuration for SMS
static const SMS_TASK_CONFIGURATION_STRUCT gsModuleTaskConfiguration =
{
    /*.pacName = */"MODULE",
    /*.un32StackSize = */8192, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_MEDIUM,
    /*.un32Options = */OSAL_TASK_OPTION_DEBUG_OUTPUT,
    /*.un16ReportingInterval = */30, // seconds
    /*.un32EventQueueSize = */256, // events  //Bosch ID#21: Module object queue size increased
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_NONE
};

// Global (re-usable) instance of an interface for this object
const MODULE_OBJECT_INTERFACE_STRUCT MODULE =
{
    /*.hGet = */hGet,
    /*.vRelease = */vRelease,
    /*.eState = */eState,
    /*.eErrorCode = */eErrorCode,
    /*.tEventMask = */tEventMask,
    /*.eModifyRegisteredEventMask = */eModifyRegisteredEventMask,
    /*.bUpdate = */bUpdate,
    /*.n16GmtOffsetGet = */n16GmtOffsetGet,
    /*.eGmtOffsetSet = */eGmtOffsetSet,
    /*.bDaylightSavingsTimeObservedGet = */bDaylightSavingsTimeObservedGet,
    /*.eDaylightSavingsTimeObservedSet = */eDaylightSavingsTimeObservedSet,
    /*.hESN = */hESN,
    /*.eSubscription = */eSubscription,
    /*.eFirmwareUpdate =*/eFirmwareUpdate,
    /*.eUpdateProgresse =*/eUpdateProgress,
    /*.eEventData =*/eEventData,
    /*.ePackage =*/ePackage
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _MODULE_OBJ_H_
