/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _LOCATION_OBJ_H_
#define _LOCATION_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "location_obj.h"
#include "tag_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Additional debugging of location coordinates
// in true floating point representation
#if DEBUG_OBJECT == 1
#ifndef DEBUG_LOCATION_COORDINATES
#define DEBUG_LOCATION_COORDINATES 1
#endif
#else
#define DEBUG_LOCATION_COORDINATES 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define LOCATION_OBJECT_NAME "Location"

// LAT/LON range for North America as defined in the Fuel & Movie
// bitstream protocol specs.
#define LOCATION_MIN_LATITUDE  (0)
#define LOCATION_MAX_LATITUDE  (90)
#define LOCATION_MIN_LONGITUDE (-178)
#define LOCATION_MAX_LONGITUDE (-30)

// Bearings to determine TopRight and BottomLeft
#define LOCATION_TOP_RIGHT_BEARING   (45)
#define LOCATION_BOTTOM_LEFT_BEARING (225)
#define LOCATION_TOP_BEARING         (0)
#define LOCATION_RIGHT_BEARING       (90)
#define LOCATION_BOTTOM_BEARING      (180)
#define LOCATION_LEFT_BEARING        (270)

  /**************/
 /** TYPEDEFS **/
/**************/
typedef struct state_name_struct
{
    const char *pcName;
    const char *pcAbbv;

} STATE_NAME_STRUCT;

/* Private object elements */

// Types of locations available
typedef enum location_position_type_enum
{
    LOCATION_POSITION_TYPE_STATIC,
    LOCATION_POSITION_TYPE_DEVICE,
    LOCATION_POSITION_TYPE_INVALID

} LOCATION_POSITION_TYPE_ENUM;

// Rectangular area properties
typedef struct location_object_properties_rectangle_struct
{
    DISTANCE_OBJECT hHeight;
    DISTANCE_OBJECT hWidth;

} LOCATION_OBJECT_PROPERTIES_RECTANGLE_STRUCT;

// Circular area properties
typedef struct location_object_properties_circle_struct
{
    DISTANCE_OBJECT hRadius;

} LOCATION_OBJECT_PROPERTIES_CIRCLE_STRUCT;

// Area-specific properties
typedef union location_object_properties_union
{
    LOCATION_OBJECT_PROPERTIES_CIRCLE_STRUCT sCircle;
    LOCATION_OBJECT_PROPERTIES_RECTANGLE_STRUCT sRectangle;

} LOCATION_OBJECT_PROPERTIES_UNION;

// Precalculated osal fixed values
typedef struct location_object_precalc_osal_fixed_struct
{
    /* Associated coordinate information */
    OSAL_FIXED_OBJECT  hLatInRadian;
    OSAL_FIXED_OBJECT  hLonInRadian;
    BOOLEAN bIsLatLonInitialized;
    
    OSAL_FIXED_OBJECT  hSinLat;
    OSAL_FIXED_OBJECT  hCosLat;
    BOOLEAN bIsSinCosLatInitialized;

} LOCATION_OBJECT_PRECALC_OSAL_FIXED_STRUCT;

// Global LOCATION coordinate limits
typedef struct location_object_coord_limits_struct
{
    OSAL_FIXED_OBJECT  hMinLat;
    OSAL_FIXED_OBJECT  hMaxLat;
    OSAL_FIXED_OBJECT  hMinLon;
    OSAL_FIXED_OBJECT  hMaxLon;
    OSAL_FIXED_OBJECT_DATA atData[OSAL_FIXED_OBJECT_SIZE * 4];
    BOOLEAN bIsMinMaxCoordinatesInitialized;

} LOCATION_OBJECT_COORD_LIMITS_STRUCT;

typedef struct location_object_struct
{
    /* Associated coordinate information */
    OSAL_FIXED_OBJECT  hLat;
    OSAL_FIXED_OBJECT_DATA atLatData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT  hLon;
    OSAL_FIXED_OBJECT_DATA atLonData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT  hBotLeftLat;
    OSAL_FIXED_OBJECT_DATA atBotLeftLatData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT  hBotLeftLon;
    OSAL_FIXED_OBJECT_DATA atBotLeftLonData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT  hTopRightLat;
    OSAL_FIXED_OBJECT_DATA atTopRightLatData[OSAL_FIXED_OBJECT_SIZE];
    OSAL_FIXED_OBJECT  hTopRightLon;
    OSAL_FIXED_OBJECT_DATA atTopRightLonData[OSAL_FIXED_OBJECT_SIZE];

    // Associated Location ID
    LOCID_OBJECT hLocID;

    // STRING_OBJECT based descriptive attributes
    LOCATION_ATTRIBUTE_STRUCT sDescriptiveAttribs;

    // Object type
    LOCATION_POSITION_TYPE_ENUM ePositionType;

    // Area-specific properties
    LOCATION_OBJECT_PROPERTIES_UNION uProperties;

    // Interface pointer
    const struct location_type_interface_struct *psIntf;

    // Does this location contain some area or it is just a point
    BOOLEAN bIsPoint;

    // Precalculated math values
    LOCATION_OBJECT_PRECALC_OSAL_FIXED_STRUCT sPrecalcFixedValues;

} LOCATION_OBJECT_STRUCT;

typedef struct location_tag_search_struct
{
    LOC_ID tID;
    TAG_OBJECT hTag;

} LOCATION_TAG_SEARCH_STRUCT;

typedef struct location_type_interface_struct
{
    // Area type
    LOCATION_TYPE_ENUM eType;

    BOOLEAN (*bInitializeObject) (
        LOCATION_OBJECT hLocation,
        N32 n32Lat,
        UN8 un8LatBits,
        N32 n32Lon,
        UN8 un8LonBits,
        LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
        LOCATION_POSITION_TYPE_ENUM ePosType,
        BOOLEAN bPerformDeepCopy,
        BOOLEAN bIsLocID
           );

    void (*vDestroyProperties) (
        LOCATION_OBJECT hLocation
           );

    BOOLEAN (*bCoordinatesWithinArea) (
        const LOCATION_OBJECT hLocation,
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon
            );

    BOOLEAN (*bContains) (
        LOCATION_OBJECT hLocationContainer,
        LOCATION_OBJECT hLocationContained,
        DISTANCE_OBJECT hThreshold
            );

    BOOLEAN (*bCompare) (
        LOCATION_OBJECT hLocation1,
        LOCATION_OBJECT hLocation2
            );

    BOOLEAN (*bUpdateCoordinates) (
        LOCATION_OBJECT hLocation,
        N32 n32Lat,
        UN8 un8LatBits,
        N32 n32Lon,
        UN8 un8LonBits
            );

    N32 (*n32FWrite) (
        const LOCATION_OBJECT hLocation,
        FILE *psFile
            );

    LOCATION_OBJECT (*hFRead) (
        FILE *psFile
            );

}  LOCATION_TYPE_INTERFACE_STRUCT;


  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static LOCATION_OBJECT hCreateForRadius (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon,
    DISTANCE_OBJECT hRadius
        );

static LOCATION_OBJECT hCreateForLocID (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon,
    LOCID_OBJECT    hLocID
        );

static LOCATION_OBJECT hCreateRectangle (
    OSAL_FIXED_OBJECT  hSouthernLat,
    OSAL_FIXED_OBJECT  hWesternLon,
    DISTANCE_OBJECT hDistanceLat,
    DISTANCE_OBJECT hDistanceLon
        );

static LOCATION_OBJECT hDuplicate (
    LOCATION_OBJECT hLocation
        );

static LOCATION_TYPE_ENUM eType(
    LOCATION_OBJECT hLocation
        );

static OSAL_FIXED_OBJECT hLat (
    LOCATION_OBJECT hLocation
        );

static OSAL_FIXED_OBJECT hLon (
    LOCATION_OBJECT hLocation
        );

static DISTANCE_OBJECT hRadius (
    LOCATION_OBJECT hLocation
        );

static DISTANCE_OBJECT hHeight (
    LOCATION_OBJECT hLocation
        );

static DISTANCE_OBJECT hWidth (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hDescription (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hStreetNum (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hStreetName (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hCity (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hState (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hZipCode (
    LOCATION_OBJECT hLocation
        );

static STRING_OBJECT hPhone (
    LOCATION_OBJECT hLocation
        );

static LOCID_OBJECT hLocID (
    LOCATION_OBJECT hLocation
        );

static DISTANCE_OBJECT hDistance (
    LOCATION_OBJECT hLocationFrom,
    LOCATION_OBJECT hLocationTo
        );

static LOCATION_OBJECT hLocationAtDistance (
    LOCATION_OBJECT hLocationFrom,
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hBearing
        );

static BOOLEAN bCoordinatesWithinArea (
    LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static LOCATION_OBJECT hClosest (
    LOCATION_OBJECT hTarget,
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        );

static N32 n32FWrite (
    LOCATION_OBJECT hLocation,
    FILE *psFile
        );

static LOCATION_OBJECT hFRead (
    FILE *psFile
        );

static N32 n32FPrintf (
    LOCATION_OBJECT hLocation,
    FILE *psFile
        );

static void vDestroy (
    LOCATION_OBJECT hLocation
        );

static SMSAPI_RETURN_CODE_ENUM eIsPoint (
    LOCATION_OBJECT hLocation,
    BOOLEAN *pbIsPoint
        );

/* Object Private Prototypes */

static BOOLEAN bCheckLatLonRange (
    OSAL_FIXED_OBJECT  hLat,
    OSAL_FIXED_OBJECT  hLon
        );

static BOOLEAN bComputeDistanceVector (
    LOCATION_OBJECT_STRUCT *psObj,
    OSAL_FIXED_OBJECT hLatTo,
    OSAL_FIXED_OBJECT hLonTo,
    OSAL_FIXED_OBJECT hResultantVector
        );

static BOOLEAN bCalcDestCoordinatesFromLocation(
    LOCATION_OBJECT_STRUCT *psObj,
    DISTANCE_OBJECT hDistance,
    OSAL_FIXED_OBJECT hBearingInDeg,
    OSAL_FIXED_OBJECT hLatTo,
    OSAL_FIXED_OBJECT hLonTo
        );

static LOCATION_OBJECT_STRUCT *psCreateLocation (
    SMS_OBJECT hOwner,
    LOCID_OBJECT hLocID,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs,
    BOOLEAN bPerformDeepCopy,
    LOCATION_TYPE_ENUM eLocationAreaType
        );

static void vUpdateDescriptiveAttribute (
    STRING_OBJECT *phOrigField,
    STRING_OBJECT hNewField,
    BOOLEAN bBlindOverwrite,
    BOOLEAN bPerformDeepCopy
        );

static BOOLEAN bFindLocationTag (
    TAG_OBJECT hTag,
    LOCATION_TAG_SEARCH_STRUCT *psSearch
        );

static N32 n32ComputeCode (
    OSAL_FIXED_OBJECT hCLineX,
    OSAL_FIXED_OBJECT hCLineY,
    OSAL_FIXED_OBJECT hBotLeftX,
    OSAL_FIXED_OBJECT hBotLeftY,
    OSAL_FIXED_OBJECT hTopRightX,
    OSAL_FIXED_OBJECT hTopRightY
        );

static BOOLEAN bPointInsideRectangle (
    LOCATION_OBJECT_STRUCT *psObj,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static void vLocationInitCommonPart (
    LOCATION_OBJECT_STRUCT *psObj,
    BOOLEAN bPerformDeepCopy,
    LOCID_OBJECT hLocID,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs
        );

static N32 n32WriteCommonPart (
    const LOCATION_OBJECT_STRUCT *psObject,
    FILE *psFile
        );

static void vFReadCommonPart (
    FILE *psFile,
    N32 *n32Lat,
    N32 *n32Lon,
    UN8 *un8LatBits,
    UN8 *un8LonBits,
    LOCATION_ATTRIBUTE_STRUCT *sAttrs,
    LOCID_OBJECT *hLocID
        );

static BOOLEAN bCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        );

static void vUpdateCoordinatesCommonPart (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        );

static LOCATION_OBJECT_STRUCT *psCreateObject (
    SMS_OBJECT hOwner,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    LOCATION_TYPE_ENUM eType,
    LOCID_OBJECT hLocID,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_ATTRIBUTE_STRUCT *psDescriptiveAttrs,
    BOOLEAN bPerformDeepCopy
        );

static BOOLEAN bRectangleInitializeObject (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    BOOLEAN bPerformDeepCopy,
    BOOLEAN bIsLocID
        );

static void vRectangleDestroyProperties (
    const LOCATION_OBJECT hLocation
        );

static BOOLEAN bRectangleCoordinatesWithinArea (
    const LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static N32 n32RectangleFWrite (
    const LOCATION_OBJECT hLocation,
    FILE *psFile
        );

static LOCATION_OBJECT hRectangleFRead (
    FILE *psFile
        );

static BOOLEAN bRectangleContains (
    LOCATION_OBJECT hLocationContainer,
    LOCATION_OBJECT hLocationContained,
    DISTANCE_OBJECT hThreshold
        );

static BOOLEAN bRectangleCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        );

static BOOLEAN bRectangleUpdateCoordinates (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        );

static BOOLEAN bCircleInitializeObject (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits,
    LOCATION_OBJECT_PROPERTIES_UNION *puProperties,
    LOCATION_POSITION_TYPE_ENUM ePosType,
    BOOLEAN bPerformDeepCopy,
    BOOLEAN bIsLocID
        );

static void vCircleDestroyProperties (
    const LOCATION_OBJECT hLocation
    );

static BOOLEAN bCircleCoordinatesWithinArea (
    const LOCATION_OBJECT hLocation,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static N32 n32CircleFWrite (
    const LOCATION_OBJECT hLocation,
    FILE *psFile
        );

static LOCATION_OBJECT hCircleFRead (
    FILE *psFile
        );

static BOOLEAN bCircleContains (
    LOCATION_OBJECT hLocationContainer,
    LOCATION_OBJECT hLocationContained,
    DISTANCE_OBJECT hThreshold
        );

static BOOLEAN bCircleCompare (
    LOCATION_OBJECT hLocation1,
    LOCATION_OBJECT hLocation2
        );

static BOOLEAN bCircleUpdateCoordinates (
    LOCATION_OBJECT hLocation,
    N32 n32Lat,
    UN8 un8LatBits,
    N32 n32Lon,
    UN8 un8LonBits
        );

static BOOLEAN bCalculateHalfwayShift (
    OSAL_FIXED_OBJECT hBaseLat,
    OSAL_FIXED_OBJECT hBaseLon,
    DISTANCE_OBJECT hWidth,
    DISTANCE_OBJECT hHeight,
    N32 n32FirstBearing,
    N32 n32SecondBearing,
    OSAL_FIXED_OBJECT hTargetLat,
    OSAL_FIXED_OBJECT hTargetLon
        );

static BOOLEAN bCalculateLongitudeAtDistance (
    OSAL_FIXED_OBJECT hBaseLat,
    OSAL_FIXED_OBJECT hBaseLon,
    DISTANCE_OBJECT hDeltaLon,
    OSAL_FIXED_OBJECT hTargetLon
        );

static BOOLEAN bPrecalculateLatLonRadians(
    LOCATION_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPrecalculateLatSinCos(
    LOCATION_OBJECT_STRUCT *psObj
        );

static void vCreateFixedConstants( void );

#if DEBUG_LOCATION_COORDINATES == 1
static float LOCATION_DEBUG_fFixed2Float(OSAL_FIXED_OBJECT hFixed);
#endif


  /***************/
 /** VARIABLES **/
/***************/

static LOCATION_OBJECT_COORD_LIMITS_STRUCT GsLocationCoordLimits = 
{
    /*.hMinLat = */ OSAL_FIXED_INVALID_OBJECT,
    /*.hMaxLat = */ OSAL_FIXED_INVALID_OBJECT,
    /*.hMinLon = */ OSAL_FIXED_INVALID_OBJECT,
    /*.hMaxLon = */ OSAL_FIXED_INVALID_OBJECT,
    /*.atData = */ {0,0,0,0,0,0,0,0},
    /*.bIsMinMaxCoordinatesInitialized = */ FALSE

};

static const LOCATION_TYPE_INTERFACE_STRUCT GsLocationRectangleIntf =
{
    // Object Area Type
    /*.eType = */LOCATION_TYPE_RECTANGLE,

    // Object Data Create/Destroy Methods
    /*.bInitializeObject = */bRectangleInitializeObject,
    /*.vDestroyProperties = */vRectangleDestroyProperties,

    // Object Data Manipulation Methods
    /*.bCoordinatesWithinArea = */bRectangleCoordinatesWithinArea,
    /*.bContains = */bRectangleContains,
    /*.bCompare = */bRectangleCompare,
    /*.bUpdateCoordinates = */bRectangleUpdateCoordinates,

    // Object Data I/O Methods
    /*.n32FWrite = */n32RectangleFWrite,
    /*.hFRead = */hRectangleFRead
};

static const LOCATION_TYPE_INTERFACE_STRUCT GsLocationCircleIntf =
{
    // Object Area Type
    /*.eType = */LOCATION_TYPE_CIRCLE,

    // Object Data Create/Destroy Methods
    /*.bInitializeObject = */bCircleInitializeObject,
    /*.vDestroyProperties = */vCircleDestroyProperties,

    // Object Data Manipulation Methods
    /*.bCoordinatesWithinArea = */bCircleCoordinatesWithinArea,
    /*.bContains = */bCircleContains,
    /*.bCompare = */bCircleCompare,
    /*.bUpdateCoordinates = */bCircleUpdateCoordinates,

    // Object Data I/O Methods
    /*.n32FWrite = */n32CircleFWrite,
    /*.hFRead = */hCircleFRead
};

// Global (re-usable) instance of an interface for this object
const LOCATION_OBJECT_INTERFACE_STRUCT LOCATION =
{
    /*.hCreateForRadius = */hCreateForRadius,
    /*.hCreateForLocID = */hCreateForLocID,
    /*.hCreateRectangle = */hCreateRectangle,
    /*.hDuplicate = */hDuplicate,
    /*.eType = */eType,
    /*.hLat = */hLat,
    /*.hLon = */hLon,
    /*.hRadius = */hRadius,
    /*.hHeight = */hHeight,
    /*.hWidth = */hWidth,
    /*.hDescription = */hDescription,
    /*.hStreetNum = */hStreetNum,
    /*.hStreetName = */hStreetName,
    /*.hCity = */hCity,
    /*.hState = */hState,
    /*.hZipCode = */hZipCode,
    /*.hPhone = */hPhone,
    /*.hLocID = */hLocID,
    /*.hDistance = */hDistance,
    /*.hLocationAtDistance = */hLocationAtDistance,
    /*.bCoordinatesWithinArea = */bCoordinatesWithinArea,
    /*.hClosest = */hClosest,
    /*.n32FWrite = */n32FWrite,
    /*.hFRead = */hFRead,
    /*.n32FPrintf = */n32FPrintf,
    /*.vDestroy = */vDestroy,
    /*.eIsPoint = */eIsPoint
};

static const STATE_NAME_STRUCT gasStateName[] =
{
    { "RESERVED", "RSV"},           // 0
    { "Alabama", "AL"},             // 1
    { "Alaska", "AK"},
    { "Arizona", "AZ"},
    { "Arkansas", "AR"},
    { "California", "CA"},
    { "Colorado", "CO"},
    { "Connecticut", "CT"},
    { "Delaware", "DE"},
    { "Florida", "FL"},
    { "Georgia", "GA"},             // 10
    { "Hawaii", "HI"},
    { "Idaho", "ID"},
    { "Illinois", "IL"},
    { "Indiana", "IN"},
    { "Iowa", "IA"},
    { "Kansas", "KS"},
    { "Kentucky", "KY"},
    { "Louisiana", "LA"},
    { "Maine", "ME"},
    { "Maryland", "MD"},            // 20
    { "Massachusetts", "MA"},
    { "Michigan", "MI"},
    { "Minnesota", "MN"},
    { "Mississippi", "MS"},
    { "Missouri", "MO"},
    { "Montana", "MT"},
    { "Nebraska", "NE"},
    { "Nevada", "NV"},
    { "New Hampshire", "NH"},
    { "New Jersey", "NJ"},          // 30
    { "New Mexico", "NM"},
    { "New York", "NY"},
    { "North Carolina", "NC"},
    { "North Dakota", "ND"},
    { "Ohio", "OH"},
    { "Oklahoma", "OK"},
    { "Oregon", "OR"},
    { "Pennsylvania", "PA"},
    { "Rhode Island", "RI"},
    { "South Carolina", "SC"},      // 40
    { "South Dakota", "SD"},
    { "Tennessee", "TN"},
    { "Texas", "TX"},
    { "Utah", "UT"},
    { "Vermont", "VT"},
    { "Virginia", "VA"},
    { "Washington", "WA"},
    { "West Virginia", "WV"},
    { "Wisconsin", "WI"},
    { "Wyoming", "WY"},             // 50
    { "Washington, D.C.", "DC"},    // 51
    { "Puerto Rico ", "PR"},        // 52
    { "Alberta", "AB"},
    { "British Columbia", "BC"},
    { "Manitoba", "MB"},
    { "New Brunswick", "NB"},
    { "Newfoundland/Labrador", "NL"},
    { "Nova Scotia", "NS"},
    { "Ontario", "ON"},
    { "Prince Edward Island", "PE"},// 60
    { "Quebec", "QC"},
    { "Saskatchewan", "SK"},
    { "Northwest Territories", "NT"},
    { "Nunavut", "NU"},
    { "Yukon Territories", "YT"},
    { "Aguascalientes", "AGU"},
    { "Baja California", "BCN"},
    { "Baja California Sur", "BCS"},
    { "Campeche", "CAM"},
    { "Chiapas", "CHP"},            // 70
    { "Chihuahua", "CHH"},
    { "Coahuila", "COA"},
    { "Colima", "COL"},
    { "Durango", "DUR"},
    { "Guanajuato", "GUA"},
    { "Guerrero", "GRO"},
    { "Hidalgo", "HID"},
    { "Jalisco", "JAL"},
    { "Mexico State", "MEX"},
    { "Michoacán", "MIC"},          // 80
    { "Morelos", "MOR"},
    { "Nayarit", "NAY"},
    { "Nuevo León", "NLE"},
    { "Oaxaca", "OAX"},
    { "Puebla", "PUE"},
    { "Querétaro", "QUE"},
    { "Quintana Roo", "ROO"},
    { "San Luis Potosí", "SLP"},
    { "Sinaloa", "SIN"},
    { "Sonora", "SON"},             // 90
    { "Tabasco", "TAB"},
    { "Tamaulipas", "TAM"},
    { "Tlaxcala", "TLA"},
    { "Veracruz", "VER"},
    { "Yucatán", "YUC"},
    { "Zacatecas", "ZAC"}               // 96
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _LOCATION_OBJ_H_
