/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _FUEL_PRICES__H_
#define _FUEL_PRICES__H_

  /**************/
 /** INCLUDES **/
/**************/

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

#include "fuel_interface.h"
#include "fuel_db_constants.h"

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define FUEL_PRICE_NAME "FuelService"
#define FUEL_CAN_PRICE_NAME "CanFuelService"

// Well known fuel types
#define REGULAR_FUEL_NAME "Regular"
#define REGULAR_FUEL_ID (0)
#define MIDRANGE_FUEL_NAME "Mid-Range"
#define MIDRANGE_FUEL_ID (1)
#define PREMIUM_FUEL_NAME "Premium"
#define PREMIUM_FUEL_ID (2)
#define DIESEL_FUEL_NAME "Diesel"
#define DIESEL_FUEL_ID (3)

// Text-specific definitions
#define FUEL_PRICE_TEXT_INITIAL_VALUE (-1)

// General
#define FUEL_PRICES_BUFFER_INCREASE_MARGIN (20)
#define FUEL_PRICES_AGE_MAX_IN_SECS (432000) // 5 days

  /**************/
 /** TYPEDEFS **/
/**************/

/**************************************************/
/** Fuel-price specific DB definitions typedefs **/
/************************************************/

/* US Fuel Database File Name */
#define FUEL_DATABASE_FOLDER "fuel"
#define FUEL_REF_DATABASE_FILENAMEA "fuel_ref.db"
#define FUEL_REF_DATABASE_FILENAMEB "fuel_ref.b.db"
#define FUEL_PERSIST_DATABASE_FILENAME "fuel.db"

/* Database Version */
#define FUEL_DATABASE_FILE_VERSION (7)

/* Canadian Fuel Database File Name */
#define FUEL_CAN_DATABASE_FOLDER "canfuel"
#define FUEL_CAN_REF_DATABASE_FILENAMEA "canfuel_ref.db"
#define FUEL_CAN_REF_DATABASE_FILENAMEB "canfuel_ref.b.db"
#define FUEL_CAN_PERSIST_DATABASE_FILENAME "canfuel.db"

/* Canadian Database Version */
#define FUEL_CAN_DATABASE_FILE_VERSION (7)

// Version table
typedef enum fuel_price_db_version_types_enum {
    FUEL_PRICE_DB_VERSION_TYPE_DB = 0,
    FUEL_PRICE_DB_VERSION_TYPE_TEXT,
    FUEL_PRICE_DB_VERSION_TYPE_CONTENTS,
    FUEL_PRICE_DB_VERSION_MAX_TYPES

} FUEL_PRICE_DB_VERSION_TYPES_ENUM;

// Enumeration specifying all the available fields
// in the db_version table
typedef enum fuel_price_db_version_fields_enum {
    FUEL_PRICE_DB_VERSION_TYPE = 0,
    FUEL_PRICE_DB_VERSION_VER,
    FUEL_PRICE_DB_VERSION_MAX_FIELDS

} FUEL_PRICE_DB_VERSION_FIELDS_ENUM;

// Structure which specifies all attributes
// of a single database version row that is
// available within the fuel database
typedef struct fuel_price_version_row_struct
{
    FUEL_PRICE_DB_VERSION_TYPES_ENUM eType; // The type of version data
    N32 n32Version;                   // The version of that data

} FUEL_PRICE_VERSION_ROW_STRUCT;

// Text Table

#define FUEL_CREATE_TEXT_TABLE \
    "create table "TEXT_TABLE_NAME" ( " \
        "ID tinyint, isBrand tinyint, " \
        "svcText varchar, " \
        "primary key (ID, isBrand) " \
        ");"

// Insert a new text row
#define FUEL_INSERT_TEXT_ROW \
    "insert into "TEXT_TABLE_NAME" " \
    "values(?, ?, ?);"

// Enumeration specifying all the available fields
// in the text table
typedef enum fuel_price_db_text_fields_enum {
    FUEL_PRICE_DB_TEXT_ID = 0,
    FUEL_PRICE_DB_TEXT_BRAND,
    FUEL_PRICE_DB_TEXT_VALUE,
    FUEL_PRICE_DB_TEXT_MAX_FIELDS

} FUEL_PRICE_DB_TEXT_FIELDS_ENUM;

// Station Table

// Enumeration specifying all the available fields
// in the station tables
typedef enum fuel_price_db_station_fields_enum {
    FUEL_PRICE_DB_STATION_ID = 0,
    FUEL_PRICE_DB_STATION_BRAND,
    FUEL_PRICE_DB_STATION_NAME,
    FUEL_PRICE_DB_STATION_ADDR,
    FUEL_PRICE_DB_STATION_CITY,
    FUEL_PRICE_DB_STATION_STATE,
    FUEL_PRICE_DB_STATION_ZIP,
    FUEL_PRICE_DB_STATION_PHONE,
    FUEL_PRICE_DB_STATION_LAT,
    FUEL_PRICE_DB_STATION_LON,
    FUEL_PRICE_DB_STATION_AMENITIES,
    FUEL_PRICE_DB_STATION_MAX_FIELDS

} FUEL_PRICE_DB_STATION_FIELDS_ENUM;

// Price Table

#define FUEL_INSERT_UPDATE_PRICE_ROW \
    "insert or replace into "PRICE_TABLE_NAME" " \
    "values(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

#define FUEL_CREATE_PRICES_TABLE \
    "create table "PRICE_TABLE_NAME"( " \
        "regionID int, " \
        "StationID int, " \
        "brand varchar, " \
        "name varchar, addr varchar, " \
        "city varchar, state int, " \
        "ZIP varchar, phone int, " \
        "lat int, lon int, " \
        "amenities int, " \
        "priceAgeInEpochSeconds int, " \
        "numPrices int, " \
        "priceList blob, " \
        "primary key (regionID, stationID)" \
        ");"

// Enumeration specifying all the available fields
// in the price table
typedef enum fuel_price_db_price_fields_enum {
    FUEL_PRICE_DB_PRICE_REGION_ID = 0,
    FUEL_PRICE_DB_PRICE_STATION_ID,
    FUEL_PRICE_DB_PRICE_BRAND,
    FUEL_PRICE_DB_PRICE_NAME,
    FUEL_PRICE_DB_PRICE_ADDR,
    FUEL_PRICE_DB_PRICE_CITY,
    FUEL_PRICE_DB_PRICE_STATE,
    FUEL_PRICE_DB_PRICE_ZIP,
    FUEL_PRICE_DB_PRICE_PHONE,
    FUEL_PRICE_DB_PRICE_LAT,
    FUEL_PRICE_DB_PRICE_LON,
    FUEL_PRICE_DB_PRICE_AMENITIES,
    FUEL_PRICE_DB_PRICE_AGE_EPOCH_SECONDS,
    FUEL_PRICE_DB_PRICE_NUM_PRICES,
    FUEL_PRICE_DB_PRICE_PRICE_LIST,
    FUEL_PRICE_DB_PRICE_MAX_FIELDS

} FUEL_PRICE_DB_PRICE_FIELDS_ENUM;

typedef struct fuel_price_ver_ctrl_struct
{
    // The expected DB version
    N16 n16DBVersion;

    // The version of the DB contents
    N16 n16DBContentVersion;

    // The version of the text table
    N16 n16TextVer;

} FUEL_PRICE_VER_CTRL_STRUCT;

typedef struct fuel_price_db_obj_struct
{
    FUEL_SERVICE_OBJECT hFuelService;

    BOOLEAN bInTransaction;

    // A structure which groups all the
    // versions tracked by this service
    FUEL_PRICE_VER_CTRL_STRUCT sVerCtrl;

    // DB Connection Handle for ref database
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    const char *pacRefDatabaseDirPath;
    const char *pacCurRefDatabaseFilePath;
    const char *pacNextRefDatabaseFilePath;

    // OTA Interface
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface;

    // DB Connection Handle for persist database
    SQL_INTERFACE_OBJECT hSQLPersistConnection;

    // Work buffer and the buffer's current size
    char *pacBuffer;
    size_t tBufferSize;

} FUEL_PRICE_DB_OBJ_STRUCT;

typedef struct fuel_price_db_result_struct
{
    FUEL_PRICE_DB_OBJ_STRUCT *psObj;
    BOOLEAN bSuccess;

} FUEL_PRICE_DB_RESULT_STRUCT;

typedef struct fuel_price_db_version_result_struct
{
    FUEL_PRICE_VER_CTRL_STRUCT *psVerCtrl;
    BOOLEAN bSuccess;

} FUEL_PRICE_DB_VERSION_RESULT_STRUCT;

typedef struct fuel_price_db_load_station_struct
{
    FUEL_REGION tRegion;
    FUEL_STATION_ID tStation;
    FUEL_PRICE_DB_OBJ_STRUCT *psObj;
    BOOLEAN bSuccess;
    size_t tNumObjectsCreated;

} FUEL_PRICE_DB_LOAD_STATION_STRUCT;

// Structure used to perform fuel price database queries
typedef struct fuel_db_price_query_result_struct
{
    BOOLEAN bSuccess;
    UN32 un32CurrentTime;
    UN32 un32PriceAgeLimit;
    FUEL_PRICE_DB_OBJ_STRUCT *psObj;

} FUEL_DB_PRICE_QUERY_RESULT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* DB Interface Functions */

static FUEL_DB_INTERFACE_OBJECT hConnect (
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    const char *pacRefDBDirPath,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        );

static void vDisconnect (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        );

static BOOLEAN bCanProcessPriceMessage (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8TextVersion
        );

static FUEL_TYPE_ENUM eMatchFuelType (
    UN8 un8TextId
        );

static size_t tLoadStations (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        );

static BOOLEAN bLoadPrices (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        );

static BOOLEAN bTextQuery (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8NewTextVersion
       );

static BOOLEAN bUpdateTextTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin,
    UN8 un8NewTextVersion
        );

static BOOLEAN bUpdateTextEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        );

static BOOLEAN bUpdateLogoEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bDelete,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        );

static void vUpdatePriceTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin
        );

static BOOLEAN bUpdatePriceEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_STATION_OBJECT hStation
        );

static UN16 un16DBContentVer (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        );

static BOOLEAN bRefDBBank (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        );

static BOOLEAN bDBUpdateBegin (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize
        );

static BOOLEAN bDBUpdateEnd (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize,
   UN16 un16NewDBVersion
       );

static BOOLEAN bRegionUpdateBegin (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    FUEL_REGION tRegionID
        );

static BOOLEAN bRegionUpdateEnd (
    SQL_INTERFACE_OBJECT hSQLConnection
        );

static BOOLEAN bStationUpdate (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char **ppcSQLCommandBuffer,
   size_t *ptBufferSize,
   FUEL_STATION_ROW_STRUCT *psStationRow,
   FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
   BOOLEAN bAmenitiesUpdated
       );

static BOOLEAN bPriceAgeFilter (
    UN32 un32PriceAgeInEpochSeconds,
    UN32 un32CurrentTime
        );

// Text functions

static BOOLEAN bInitializeWellKnownFuelTypes (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
       );

// Price functions

static BOOLEAN bFlushPrices (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bAgeoutPrices (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static void vShiftPriceEntries(
    FUEL_PRICE_ENTRY_STRUCT *pasPrices,
    size_t tNumPrices,
    size_t tCurIndex
        );

// Station functions

static BOOLEAN bUpdateStationDBEntry (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char **ppacBuffer,
    size_t *ptBufferSize,
    FUEL_STATION_ROW_STRUCT *psStation,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    BOOLEAN bAmenitiesUpdated
        );

static BOOLEAN bBuildStationLoadQuery (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegion
        );

// General purpose functions

static BOOLEAN bLoadTables (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bLoadTextTable (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bLoadRegionTable (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bExpandBuffer (
    char **ppacBuffer,
    size_t *ptBufferSize,
    size_t tNewSizeRequired,
    BOOLEAN bCopy
        );

static BOOLEAN bFormatTextField (
    char **ppacBuffer,
    size_t *ptBufferSize,
    size_t *ptBufferIndex
        );

static void vStartTransaction (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );

static void vEndTransaction (
    FUEL_PRICE_DB_OBJ_STRUCT *psObj
        );


/* SQL callback functions */

// Writers

static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bPrepareTextColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        );

static BOOLEAN bPrepareRegionColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_REGION *ptRegionID
        );

static BOOLEAN bPreparePriceColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT *psDBUpdate
        );

// Readers

static BOOLEAN bVerifyAndLoadDBVersionFields (
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_PRICE_VER_CTRL_STRUCT *psVerCtrl
        );

static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_VERSION_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectText (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectRegionExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbRegionExists
        );

static BOOLEAN bProcessSelectRegions (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectStations (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_PRICE_DB_LOAD_STATION_STRUCT *psBuild
        );

static BOOLEAN bProcessSelectStationExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbStationExists
        );

static BOOLEAN bProcessSelectPrices (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    FUEL_DB_PRICE_QUERY_RESULT_STRUCT *psResult
        );

static DATASERVICE_ERROR_CODE_ENUM eRegularGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    const FUEL_DB_INTERFACE_STRUCT *psInterface,
    size_t tVersionBitLen,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

  /***************/
 /** VARIABLES **/
/***************/

// Manager interface for plugins
const FUEL_DB_INTERFACE_STRUCT GsPriceDBIntf =
{
    /*.pacFolder = */ FUEL_DATABASE_FOLDER,
    /*.pacRefDBNameA = */ FUEL_REF_DATABASE_FILENAMEA,
    /*.pacRefDBNameB = */ FUEL_REF_DATABASE_FILENAMEB,
    /*.n16DBVersion = */ FUEL_DATABASE_FILE_VERSION,
    /*.pacPerstDBName = */ FUEL_PERSIST_DATABASE_FILENAME,
    /*.hConnect = */hConnect,
    /*.vDisconnect = */vDisconnect,
    /*.bCanProcessUpdate = */bCanProcessPriceMessage,
    /*.eMatchFuelType = */eMatchFuelType,
    /*.tLoadStations = */tLoadStations,
    /*.bLoadPrices = */bLoadPrices,
    /*.bTextQuery = */bTextQuery,
    /*.bUpdateTextTable = */bUpdateTextTable,
    /*.bUpdateTextEntry = */bUpdateTextEntry,
    /*.bUpdateLogoEntry = */bUpdateLogoEntry,
    /*.vUpdatePriceTable = */vUpdatePriceTable,
    /*.bUpdatePriceEntry = */bUpdatePriceEntry,
    /*.un16DBContentVer = */un16DBContentVer,
    /*.bRefDBBank = */bRefDBBank,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.bRegionUpdateBegin = */bRegionUpdateBegin,
    /*.bRegionUpdateEnd = */bRegionUpdateEnd,
    /*.bStationUpdate = */bStationUpdate
};

// Manager interface for plugins
const FUEL_DB_INTERFACE_STRUCT GsCanadianPriceDBIntf =
{
    /*.pacFolder = */ FUEL_CAN_DATABASE_FOLDER,
    /*.pacRefDBNameA = */ FUEL_CAN_REF_DATABASE_FILENAMEA,
    /*.pacRefDBNameB = */ FUEL_CAN_REF_DATABASE_FILENAMEB,
    /*.n16DBVersion = */ FUEL_CAN_DATABASE_FILE_VERSION,
    /*.pacPersistDBName = */ FUEL_CAN_PERSIST_DATABASE_FILENAME,
    /*.hConnect = */hConnect,
    /*.vDisconnect = */vDisconnect,
    /*.bCanProcessUpdate = */bCanProcessPriceMessage,
    /*.eMatchFuelType = */eMatchFuelType,
    /*.tLoadStations = */tLoadStations,
    /*.bLoadPrices = */bLoadPrices,
    /*.bTextQuery = */bTextQuery,
    /*.bUpdateTextTable = */bUpdateTextTable,
    /*.bUpdateTextEntry = */bUpdateTextEntry,
    /*.bUpdateLogoEntry = */bUpdateLogoEntry,
    /*.vUpdatePriceTable = */vUpdatePriceTable,
    /*.bUpdatePriceEntry = */bUpdatePriceEntry,
    /*.un16DBContentVer = */un16DBContentVer,
    /*.bRefDBBank = */bRefDBBank,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.bRegionUpdateBegin = */bRegionUpdateBegin,
    /*.bRegionUpdateEnd = */bRegionUpdateEnd,
    /*.bStationUpdate = */bStationUpdate
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _FUEL_PRICES__H_
