/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _FUEL_MGR_OBJ_H_
#define _FUEL_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "fuel_interface.h"
#include "fuel_price_service.h"
#include "ev_charging_service.h"
#include "image_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define FUEL_MGR_OBJECT_NAME "FuelManager"

// General definitions
#define FUEL_MAX_ZIP_CHARS (15)
#define FUEL_AGE_MAX_IN_SECS (432000) // 5 days
#define FUEL_PRICE_HASH_LIFETIME (2700) // 45 minutes
#define FUEL_PROCESS_TARGET_TIMEOUT_SECONDS (1)
#define FUEL_AREA_CODE_MULTIPLIER (1000)
#define FUEL_EXCHANGE_MULTIPLIER (10000UL)
#define FUEL_AREA_CODE_DIVISOR (10000000UL)
#define FUEL_PHONE_CHAR_ARRAY_SIZE (14)

// Constant defines the minimum device location
// delta which fuel wants to know about
#define FUEL_DEVICE_DISTANCE_THRESHOLD (2)

// Bearing used to create DB query strings
#define FUEL_UPPER_RIGHT_BEARING (45)

// Background radius
#define FUEL_BACKGROUND_PROCESSING_RADIUS_IN_MILES (37)

// Logo file info
#define FUEL_PNG_FILE_EXTENSION ".png"
#define FUEL_FILE_NAME_ID_LABEL "I%03u_T%1u"
#define FUEL_FILE_NAME_ID_LABEL_LEN (7)

#define FUEL_SUPPORTED_OPTIONS (         \
    (DATASERVICE_OPTIONS_MASK)           \
    DATASERVICE_OPTION_REFERENCE_DB_PATH \
    )

  /**************/
 /** TYPEDEFS **/
/**************/

// This structure provides a grouping
// for the defining attributes of a DSRL target
typedef struct fuel_target_desc_struct
{
    // This target's entry in the
    // target list
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The current location object
    // being processed
    OSAL_LINKED_LIST_ENTRY hCurrentLocation;
    BOOLEAN bCurrentLocationIsCoordBased;

    // The current region being processed
    FUEL_REGION tCurrentRegion;
    BOOLEAN bCurrentRegionInDB;
    FUEL_STATION_ID tCurrentStation;

    // The target(s) which come from DSRL events
    DSRL_OBJECT hDSRL;
    BOOLEAN bDSRLLockedForAgeout;
    BOOLEAN bFavorite;
    BOOLEAN bProcessPending;
    BOOLEAN bSerializeDSRL;
    OSAL_OBJECT_HDL hLocations;

    // The manager for this target
    struct fuel_mgr_object_struct *psObj;

} FUEL_TARGET_DESC_STRUCT;

// This structure provides a grouping
// for the attributes used for general
// work performed on a target / service level
typedef struct fuel_mgr_work_ctrl_struct
{
    // A structure that we'll use as
    // the owner of every station object
    struct fuel_mgr_station_owner_struct *psStationOwner;

    // The list of fuel stations in use
    // by DSRLs
    OSAL_OBJECT_HDL hStations;

    // The application-chosen fuel price sort method
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod;

    // Timer used to ensure old data is aged-out
    DATASERVICE_TIMED_EVENT_HDL hProcessEvent;

    // The size of the background location
    DISTANCE_OBJECT hBackgroundRadius;

    FUEL_TARGET_DESC_STRUCT *psCurrentTarget;
    BOOLEAN bLoadingPrices;

    FUEL_STATION_OBJECT hDummyStation;

} FUEL_MGR_WORK_CTRL_STRUCT;

// This structure groups the defining attributes
// of a region with how it is being used
typedef struct fuel_region_descriptor_struct
{
    FUEL_REGION tRegionId;
    size_t tTotalUsageCount;
    size_t tLocationUsageCount;

    OSAL_OBJECT_HDL hPriceUpdates;

} FUEL_REGION_DESCRIPTOR_STRUCT;

// This structure provides a grouping
// for the attributes used for process
// OTA messaging
typedef struct fuel_mgr_ota_ctrl_struct
{
    // The current region / station being processed for
    // price data, if any
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion;

    // A flag indicating if OTA messaging is currently
    // updating a DSRL
    BOOLEAN bDSRL;

    // Flag indicating if we're currently
    // in the middle of a transaction
    BOOLEAN bInTransaction;

    // Current age for a price group (UTC)
    UN32 un32UTCsec;

    // The text version we're using
    // to handle OTA messages
    UN8 un8TextVer;

} FUEL_MGR_OTA_CTRL_STRUCT;

// This structure provides a grouping
// for the attributes used for ageing-out
// prices
typedef struct fuel_mgr_ageout_ctrl_struct
{
    // Data Expire event
    DATASERVICE_TIMED_EVENT_HDL hDataExpireEvent;

    // The oldest data age currently in DSRLs
    UN32 un32OldestPriceData;

    // The oldest data age we've come across
    // while processing the DB or OTA messages
    UN32 un32OldestProcessedPriceData;

} FUEL_MGR_AGEOUT_CTRL_STRUCT;

/* Private object elements */
typedef struct fuel_mgr_object_struct
{
    // The service we're managing
    DATASERVICE_TYPE_ENUM eServiceType;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The interface structures / objects we're currently using
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface;
    FUEL_OTA_INTERFACE_OBJECT hOTAInterface;
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface;
    FUEL_DB_INTERFACE_OBJECT hDBInterface;

    // A list of active targets (a list of DSRLs for the application)
    // Each entry contains a TARGET_DESC_STRUCT
    OSAL_OBJECT_HDL hTargets;

    // A structure which groups all the
    // work-unit related attributes & state
    FUEL_MGR_WORK_CTRL_STRUCT sWorkCtrl;

    // A structure which groups all the state
    // variables used for OTA message processing
    FUEL_MGR_OTA_CTRL_STRUCT sOTACtrl;

    // A structure which groups all the
    // data ageout related attributes
    FUEL_MGR_AGEOUT_CTRL_STRUCT sAgeoutCtrl;

    // The list of regions
    OSAL_OBJECT_HDL hRegions;

    // Location of the reference db
    char *pacRefDatabaseDirPath;

} FUEL_MGR_OBJECT_STRUCT;

// Structure which is used as the
// station owner object
typedef struct fuel_mgr_station_owner_struct
{
    // Know how to find the fuel manager
    FUEL_MGR_OBJECT_STRUCT *psObj;

    // In-memory copy of the fuel text table
    OSAL_OBJECT_HDL hTextEntries;

    // Directory in which images are found
    const char *pacLogoFileDir;
    const char *pacBaselineLogoFileDir;
    char *pacLogoFilePath;
    size_t tLogoFilePathLen;

    // The list of logos being managed by this service
    OSAL_OBJECT_HDL hLogos;

} FUEL_MGR_STATION_OWNER_STRUCT;

// Structure used to store locations used in targets
typedef struct fuel_target_location_desc_struct
{
    LOCATION_OBJECT hLocation;

    OSAL_OBJECT_HDL hRegionsOfInterest;

    OSAL_LINKED_LIST_ENTRY hCurrentRegion;
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The "background" processing area
    LOCATION_OBJECT hBackground;
    BOOLEAN bBackgroundLocationUpdated;

} FUEL_TARGET_LOCATION_DESC_STRUCT;

// Structure which maps a fuel station
// to a target & DSRL (if necessary)
typedef struct fuel_station_entry_struct
{
    BOOLEAN bInDSRL;
    BOOLEAN bBackgroundOnly;

    FUEL_TARGET_DESC_STRUCT *psTarget;
    FUEL_STATION_OBJECT hStation;

} FUEL_STATION_ENTRY_STRUCT;

// Ths is the structure that is attached to 
// each DSRL_ENTRY object.
typedef struct fuel_station_dsrl_entry_struct
{
    // The entry handle for the first entry
    // for a station (if the station has multiple
    // entries in the cache, this is the first
    // entry in the series; if the station only
    // has a single entry in the cache this is
    // the handle of that entry)
    OSAL_LINKED_LIST_ENTRY hFirstStationEntry;

} FUEL_STATION_DSRL_ENTRY_STRUCT;

// Structure which maps a fuel logo row
// to an image object
typedef struct fuel_logo_entry_struct
{
    OSAL_LINKED_LIST_ENTRY hEntry;
    FUEL_LOGO_ROW_STRUCT sLogoRow;
    IMAGE_OBJECT hLogo;

} FUEL_LOGO_ENTRY_STRUCT;

// Structure used to indicate how a region
// is of interest to the manager / a target
typedef struct fuel_region_interest_entry_struct
{
    FUEL_REGION tRegionID;
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion;
    BOOLEAN bBackgroundOnly;
    OSAL_LINKED_LIST_ENTRY hEntry;

} FUEL_REGION_INTEREST_ENTRY_STRUCT;

// Structure used when flushing stations for price ageout
typedef struct fuel_station_price_ageout_struct
{
    FUEL_MGR_OBJECT_STRUCT *psObj;
    UN32 un32PriceExpireAge;
    UN32 *pun32OldestPriceAfterFlush;
    FUEL_STATION_OBJECT hLastStation;
    BOOLEAN bLastStationUpdated;

} FUEL_STATION_PRICE_AGEOUT_STRUCT;

// Structure used to aid in the processing
// of location updates for a DSRL
typedef struct fuel_target_location_update_struct
{
    FUEL_TARGET_DESC_STRUCT *psTarget;
    BOOLEAN bDSRLUpdated;
} FUEL_TARGET_LOCATION_UPDATE_STRUCT;

// Structure used to track / filter price messages
typedef struct fuel_region_price_update_msg_struct
{
    UN32 un32Hash;
    UN32 un32TimeStamp;
    OSAL_LINKED_LIST_ENTRY hEntry;
} FUEL_REGION_PRICE_UPDATE_MSG_STRUCT;

// Structure used to help processing of
// price message filter entries
typedef struct fuel_price_hash_iterator_struct
{
    UN32 un32CurrentTime;
    UN32 un32Hash;
    BOOLEAN bFiltered;
    BOOLEAN bFound;
} FUEL_PRICE_HASH_ITERATOR_STRUCT;

// Structure used to add a region of
// interest based on a location
typedef struct fuel_region_interest_struct
{
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation;
    BOOLEAN bBackgroundOnly;
    BOOLEAN bSuccess;

} FUEL_REGION_INTEREST_STRUCT;

// Structure used to aid in the use of FUEL.eFuelTypes
typedef struct fuel_text_iterator_struct
{
    FUEL_TYPE_ITERATOR bIterator;
    void *pvIteratorArg;

    // The in-use DB interface structure
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface;

} FUEL_TEXT_ITERATOR_STRUCT;

// Structure used to aid in refresh operations
typedef struct fuel_refresh_dsrl_struct
{
    FUEL_TARGET_DESC_STRUCT *psTarget;
    BOOLEAN bFlush;
    BOOLEAN bDSRLUpdated;

} FUEL_REFRESH_DSRL_STRUCT;

// Structure used to handle processing of a
// new station found during an RFD update
typedef struct fuel_new_station_struct
{
    FUEL_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bAddedToCache;

} FUEL_NEW_STATION_STRUCT;

// Structure used to update the cache
// when a DSRL entry is removed from a DSRL
typedef struct fuel_dsrl_entry_remove_struct
{
    FUEL_TARGET_DESC_STRUCT *psTarget;
    FUEL_STATION_OBJECT hStation;

} FUEL_DSRL_ENTRY_REMOVE_STRUCT;

// Structure used to represent specific data for the FUEL
// service inside IMAGE object
typedef struct fuel_image_data_struct
{
    UN16 un16LogoId;
    UN8 un8LogoVer;
    FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType;
} FUEL_IMAGE_DATA_STRUCT;

// The fuel manager defines its own iterator
// function to operate on each cache entry
// for a particular station.  That iterator
// function uses this type to define the 
// callback used along with the function
typedef BOOLEAN (*STATION_ITERATOR_CALLBACK) (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT hStation
        );

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SMSAPI_RETURN_CODE_ENUM eFuelTypes (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TYPE_ITERATOR bIterator,
    void *pvIteratorArg
        );

/* Object Private Prototypes */

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bHandleServiceReady (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessMessage (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bHandleAgeout (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vDestroyObject(
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

/* Manager Interface APIs */

static BOOLEAN bRegionUpdateBegin (
    FUEL_SERVICE_OBJECT hFuelService,
    UN32 un32MsgId,
    FUEL_REGION tRegion,
    UN8 un8TextVersion
        );

static BOOLEAN bPriceUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_ID tStationId,
    FUEL_PRICE_ROW_STRUCT *psPriceRow
       );

static BOOLEAN bPositionUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    UN16 un16ReportTime,
    FUEL_STATION_ID tStationId,
    FUEL_POSITION_UPDATE_STRUCT *pasPositions,
    size_t tNumPositions
        );

static void vRegionUpdateComplete (
    FUEL_SERVICE_OBJECT hFuelService
       );

static BOOLEAN bTextUpdateBegin (
    FUEL_SERVICE_OBJECT hFuelService,
    UN8 un8NewTextVersion
        );

static BOOLEAN bTextUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TEXT_ROW_STRUCT *psTextRow
       );

static void vTextUpdateEnd (
    FUEL_SERVICE_OBJECT hFuelService,
    UN8 un8NewTextVersion
        );

static BOOLEAN bLogoUpdate (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_LOGO_ROW_STRUCT *psLogoRow,
    BOOLEAN bDelete,
    OSAL_BUFFER_HDL hLogoData
        );

static BOOLEAN bAddRegionOfInterest (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegion,
    void *pvArg
        );

static BOOLEAN bEnableLogoSupport (
    FUEL_SERVICE_OBJECT hFuelService,
    const char *pacServicePath,
    const char *pacBaselinePath
        );

static BOOLEAN bUpdateTextEntry (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_TEXT_ROW_STRUCT *psTextRow
       );

static BOOLEAN bUpdateLogoEntry (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
       );

static BOOLEAN bPopulateRegionTable (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tID
        );

static BOOLEAN bLocationFilter (
    FUEL_SERVICE_OBJECT hFuelService,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    BOOLEAN *pbCacheOnly
        );

static FUEL_STATION_OBJECT hGetStationFromCache (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID,
    BOOLEAN *pbStationNeeded
        );

static BOOLEAN bAddFuelPricesToStation (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_OBJECT hStation,
    FUEL_PRICE_ROW_STRUCT *psPriceRow,
    UN8 un8TextVer
        );

static BOOLEAN bPutStationInService (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_OBJECT hStation,
    BOOLEAN bCacheOnly
        );

static BOOLEAN bIsRegionNeeded (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_REGION tRegion,
    UN8 un8TextVersion
        );

/* Text functions */

static STRING_OBJECT hGetShortText (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bBrandText,
    UN8 un8TextId
        );

static BOOLEAN bGetTextPair (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bBrandText,
    UN8 un8TextId,
    STRING_OBJECT *phShortText,
    STRING_OBJECT *phLongText
        );

/* Logo functions */

static BOOLEAN bBuildLogoDirPath (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    const char *pacLogoPath,
    const char *pacBaselineLogoPath
        );

static BOOLEAN bImageInitSpecificData(
    IMAGE_OBJECT hImage,
    void *pvSpecificData,
    void *pvArg
        );

static BOOLEAN bImageFilenameLenBasedOnFormat (
    IMAGE_FORMAT_ENUM eFormat,
    const char *pacFilePath,
    size_t *ptLength
        );

static BOOLEAN bImageFilenameLen (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *psSpecificData,
    size_t *ptLength
        );

static BOOLEAN bImageFilenameCreateBasedOnFormat (
    IMAGE_FORMAT_ENUM eFormat,
    const char *pacFilePath,
    FUEL_IMAGE_DATA_STRUCT *psImageData,
    char *pacBuffer,
    size_t tBufferSize
        );

static BOOLEAN bImageFilenameCreate (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        );

static BOOLEAN bWriteLogoFile (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    OSAL_BUFFER_HDL hLogoData,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        );

static BOOLEAN bDeleteLogo (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    FUEL_LOGO_ENTRY_STRUCT *psLogoEntry
        );

static FUEL_LOGO_ENTRY_STRUCT *psGetLogoEntry (
    FUEL_MGR_STATION_OWNER_STRUCT *psOwner,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        );

/* Region functions */

static FUEL_REGION_DESCRIPTOR_STRUCT *psGetRegion (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_REGION tRegionID
        );

/* Price functions */

static BOOLEAN bFilterPriceMessage (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_REGION_DESCRIPTOR_STRUCT *psPriceRegion,
    UN32 un32PriceMessageHash,
    UN32 un32CurrentTime
        );

static void vCloseOutUpdateGroup (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bDSRL,
    FUEL_STATION_OBJECT hStation,
    BOOLEAN bAllGroupsComplete
        );

static STRING_OBJECT hFuelTextCallback (
    UN8 un8FuelType,
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static FUEL_TYPE_ENUM eFuelTypeCallback (
    UN8 un8FuelType,
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

/* Station functions */

static FUEL_STATION_OBJECT hCreateStationObject (
    FUEL_SERVICE_OBJECT hFuelService,
    FUEL_STATION_ROW_STRUCT *psStation
        );

static BOOLEAN bAddStationToCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_STATION_OBJECT hStation,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN bInDSRL,
    BOOLEAN bBackgroundOnly
        );

static FUEL_STATION_OBJECT hFindStationInCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN *pbStationFoundInProvidedTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStationId
        );

static BOOLEAN bIterateStationEntries (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_STATION_OBJECT hStation,
    STATION_ITERATOR_CALLBACK bIterator
       );

static BOOLEAN bPruneCache (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static void vDestroyCache (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

/* Target / DSRL functions */

static BOOLEAN bHandleCreateList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleModifyList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleRefreshList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vHandleDeleteList (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static FUEL_TARGET_DESC_STRUCT *psCreateTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vDestroyTarget (
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bUpdateTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bExpandTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bReduceTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bReduceCacheForLocationChange (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bUpdateDSRLForLocationChange (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bUpdateDSRLForRefresh (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bVerifyTargetList (
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bAddLocationsToTarget (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT  *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bCreateBackgroundLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg,
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        );

static BOOLEAN bUpdateTargetLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    DSRL_ARG_STRUCT *psDSRLArg,
    BOOLEAN *pbBackgroundUpdatedInThisCall,
    FUEL_TARGET_LOCATION_DESC_STRUCT **ppsLocation
        );

static BOOLEAN bVerifyQueryResultLocation (
    FUEL_SERVICE_OBJECT hFuelService,
    LOC_ID tLocId,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static BOOLEAN bStationMeetsTargetLocationCriteria (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN bUseBackground,
    LOC_ID tLocID,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        );

static BOOLEAN bMoveToNextLocation (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bInitDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bProcessTarget (
    FUEL_TARGET_DESC_STRUCT *psTarget
            );

static BOOLEAN bAgeoutPrices (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    UN32 un32PriceExpireAge,
    UN32 *pun32OldestPriceAfterFlush
        );

static BOOLEAN bLoadStationIntoDSRLFromCache (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStationId
        );

static BOOLEAN bAddToDSRL (
    DSRL_OBJECT hDSRL,
    FUEL_STATION_OBJECT hStation
        );

/* Linked list related functions*/

static void vDestroyTargetLocation (
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        );

static BOOLEAN bIterateTargetLocationsForReduce (
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bIteratePriceMsgs (
    FUEL_REGION_PRICE_UPDATE_MSG_STRUCT *psPriceUpdate,
    FUEL_PRICE_HASH_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bIterateTargetsToProcess (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    BOOLEAN *pbSuccess
        );

static BOOLEAN bIterateTargetsForDSRLReady (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    void *pvUnused
        );

static BOOLEAN bIterateTargetsForDSRLAccess (
    FUEL_TARGET_DESC_STRUCT *psTarget,
    void *pvArg
        );

static N16 n16CompareTextEntries (
    FUEL_TEXT_ROW_STRUCT *psText1,
    FUEL_TEXT_ROW_STRUCT *psText2
        );

static N16 n16CompareLogoEntries (
    FUEL_LOGO_ENTRY_STRUCT *psLogo1,
    FUEL_LOGO_ENTRY_STRUCT *psLogo2
        );

static BOOLEAN bIterateTextEntries (
    FUEL_TEXT_ROW_STRUCT *psText,
    void *pvArg
        );

static void vReleaseTextEntry (
    FUEL_TEXT_ROW_STRUCT *psText
        );

static void vReleaseLogoEntry (
    FUEL_LOGO_ENTRY_STRUCT *psLogo
        );

static N16 n16CompareRegionsOfInterestById (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry1,
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry2
        );

static BOOLEAN bRemoveOldRegionsOfInterest (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry,
    BOOLEAN *pbRemoveAll
        );

static void vDestroyRegionOfInterest (
    FUEL_REGION_INTEREST_ENTRY_STRUCT *psRegionEntry
        );

static N16 n16CompareRegions (
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion1,
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion2
        );

static void vReleaseRegionEntry (
    FUEL_REGION_DESCRIPTOR_STRUCT *psRegion
        );

static N16 n16CompareStationEntry (
    FUEL_STATION_ENTRY_STRUCT *psEntry1,
    FUEL_STATION_ENTRY_STRUCT *psEntry2
        );

static BOOLEAN bUpdateDSRLForNewStation (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT hStation
        );

static BOOLEAN bUpdateDSRLForStationChange (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT hStation
        );

static BOOLEAN bIterateCacheForBackgroundChange(
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bIterateCacheForLocationUpdate (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_TARGET_LOCATION_UPDATE_STRUCT *psUpdate
        );

static BOOLEAN bIterateCacheForDSRLRefresh (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_REFRESH_DSRL_STRUCT *psRefresh
        );

static BOOLEAN bIterateCacheForAgeout (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_PRICE_AGEOUT_STRUCT *psAgeout
        );

static BOOLEAN bIterateCacheForDestroy (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    FUEL_STATION_OBJECT *phLastStation
        );

static BOOLEAN bIterateCacheForLogoUpdate (
    FUEL_STATION_ENTRY_STRUCT *psEntry,
    UN16 *pun16LogoId
        );

static void vReleaseStations(
    FUEL_STATION_ENTRY_STRUCT *psEntry
        );

static BOOLEAN bEntryExists (
    OSAL_LINKED_LIST_ENTRY hStartingEntry,
    FUEL_STATION_ENTRY_STRUCT *psEntry
        );

static FUEL_STATION_ENTRY_STRUCT *psCacheEntryFromDSRLEntry (
    OSAL_LINKED_LIST_ENTRY hStartingEntry,
    FUEL_STATION_OBJECT hStation,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static BOOLEAN bLoadPricesForFavoriteStationID (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_REGION tRegion,
    FUEL_STATION_ID tStation
        );

/* DSRL callbacks */

static N16 n16SortDSRLByPrice (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static void vDSRLFinalizeStationDelete (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

/* Timer Related */

static void vStartAgeoutTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static void vStopAgeoutTimer (
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static void vStartProcessTimer(
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

static void vStopProcessTimer(
    FUEL_MGR_OBJECT_STRUCT *psObj
        );

/* General Functions */

static BOOLEAN bLoadStationsForDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget,
    FUEL_TARGET_LOCATION_DESC_STRUCT *psLocation
        );

static BOOLEAN bLoadPricesForDSRL (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    FUEL_TARGET_DESC_STRUCT *psTarget
        );

static void vSetError (
    FUEL_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for the fuel service
const FUEL_INTERFACE_STRUCT FUEL =
{
    /*.hStart = */FUEL_PRICES_hStart,
    /*.hEVStart = */EV_CHARGING_hStart,
    /*.hCanadianStart = */ CANFUEL_PRICES_hStart,
    /*.vStop = */(void (*) (FUEL_SERVICE_OBJECT hFuelService))DATASERVICE_IMPL_vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (FUEL_SERVICE_OBJECT hFuelService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (FUEL_SERVICE_OBJECT hFuelService))DATASERVICE_IMPL_eErrorCode,
    /*.eFuelTypes = */eFuelTypes,
    /*.eGetReferenceDataVersion = */FUEL_PRICES_eGetReferenceDataVersion,
    /*.eEVGetReferenceDataVersion = */EV_CHARGING_eGetReferenceDataVersion,
    /*.eCanadianGetReferenceDataVersion = */ CANFUEL_PRICES_eGetReferenceDataVersion
};

// Manager interface for plugins
const FUEL_MGR_INTERFACE_STRUCT GsFuelMgrIntf =
{
    /*.bRegionUpdateBegin = */bRegionUpdateBegin,
    /*.bPriceUpdate = */bPriceUpdate,
    /*.bPositionUpdate = */bPositionUpdate,
    /*.vRegionUpdateComplete = */vRegionUpdateComplete,
    /*.bTextUpdateBegin = */bTextUpdateBegin,
    /*.bTextUpdate = */bTextUpdate,
    /*.vTextUpdateEnd = */vTextUpdateEnd,
    /*.bLogoUpdate = */bLogoUpdate,
    /*.bAddRegionOfInterest = */bAddRegionOfInterest,
    /*.bEnableLogoSupport = */bEnableLogoSupport,
    /*.bReportTextFromDB = */bUpdateTextEntry,
    /*.bReportLogoFromDB = */bUpdateLogoEntry,
    /*.bReportRegionFromDB = */bPopulateRegionTable,
    /*.bBackgroundLocationFilter = */bLocationFilter,
    /*.bVerifyQueryResultLocation = */bVerifyQueryResultLocation,
    /*.hGetStation = */hGetStationFromCache,
    /*.hCreateStation = */hCreateStationObject,
    /*.bAddFuelPricesToStation = */bAddFuelPricesToStation,
    /*.bPutStationInService = */bPutStationInService,
    /*.bIsRegionNeeded = */bIsRegionNeeded
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsFuelStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

// IMAGE object specific interface implementation
static const IMAGE_INTF_STRUCT GsFuelImageIntf =
{
    /*.tSpecificDataSize = */ sizeof(FUEL_IMAGE_DATA_STRUCT),
    /*.bInitSpecificData = */ bImageInitSpecificData,
    /*.bCreationPostStep = */ NULL,
    /*.vUninitSpecificData = */ NULL,
    /*.bCalculateImageFileNameLen = */ bImageFilenameLen,
    /*.bCreateImageFileName = */ bImageFilenameCreate,
    /*.eProperty = */ NULL,
    /*.eUpdateProperty = */ NULL
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _FUEL_MGR_OBJ_H_
