/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _EV_CHARGING__H_
#define _EV_CHARGING__H_

  /**************/
 /** INCLUDES **/
/**************/

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

#include "fuel_interface.h"
#include "fuel_db_constants.h"

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define EV_CHARGING_NAME "EVChargingService"

// Well known charge types
#define LEVEL1_120V_SNAME "Level 1"
#define LEVEL1_120V_LNAME "Level 1 120 V"
#define LEVEL1_120V_ID (1)
#define LEVEL2_240V_SNAME "Level 2"
#define LEVEL2_240V_LNAME "Level 2 240 V"
#define LEVEL2_240V_ID (2)
#define DC_SNAME "DC Charge"
#define DC_LNAME "DC Fast Charge"
#define DC_ID (3)

// Text-specific definitions
#define EV_CHARGING_TEXT_INITIAL_DB_VER (-1)

// General
#define EV_CHARGING_BUFFER_INCREASE_MARGIN (20)
#define EV_CHARGING_AGE_MAX_IN_SECS (432000) // 5 days

  /**************/
 /** TYPEDEFS **/
/**************/

/****************************************************/
/** EV Charging specific DB defintions & typedefs **/
/**************************************************/

/* Database File Name */
#define EV_CHARGING_DATABASE_FOLDER "evcharging"
#define EV_CHARGING_BASELINE_FOLDER "baseline/"
#define EV_CHARGING_REF_DATABASE_FILENAMEA "evcharging_ref.db"
#define EV_CHARGING_REF_DATABASE_FILENAMEB "evcharging_ref.b.db"
#define EV_CHARGING_PERSIST_DATABASE_FILENAME "evcharging.db"

/* Database Version */
#define EV_CHARGING_DATABASE_FILE_VERSION (2)

// Version table
typedef enum ev_charging_db_version_types_enum {
    EV_CHARGING_DB_VERSION_TYPE_DB = 0,
    EV_CHARGING_DB_VERSION_TYPE_CONTENTS,
    EV_CHARGING_DB_VERSION_MAX_TYPES

} EV_CHARGING_DB_VERSION_TYPES_ENUM;

// Enumeration specifying all the available fields
// in the db_version table
typedef enum ev_charging_db_version_fields_enum {
    EV_CHARGING_DB_VERSION_TYPE = 0,
    EV_CHARGING_DB_VERSION_VER,
    EV_CHARGING_DB_VERSION_MAX_FIELDS

} EV_CHARGING_DB_VERSION_FIELDS_ENUM;

// Structure which specifies all attributes
// of a single database version row that is
// available within the EV database
typedef struct ev_charging_version_row_struct
{
    EV_CHARGING_DB_VERSION_TYPES_ENUM eType; // The type of version data
    N32 n32Version;                   // The version of that data

} EV_CHARGING_VERSION_ROW_STRUCT;

// Text Table

// Insert a new text row
#define EV_CHARGING_INSERT_TEXT_ROW \
    "insert or replace into "TEXT_TABLE_NAME" " \
    "values(?, ?, ?);"

// Enumeration specifying all the available fields
// in the text table
typedef enum ev_charging_db_text_fields_enum {
    EV_CHARGING_DB_TEXT_ID = 0,
    EV_CHARGING_DB_SHORT_TEXT,
    EV_CHARGING_DB_LONG_TEXT,
    EV_CHARGING_DB_TEXT_MAX_FIELDS

} EV_CHARGING_DB_TEXT_FIELDS_ENUM;

// Station Table

// Enumeration specifying all the available fields
// in the station tables
typedef enum ev_charging_db_station_fields_enum {
    EV_CHARGING_DB_STATION_ID = 0,
    EV_CHARGING_DB_STATION_BRAND,
    EV_CHARGING_DB_STATION_LOGO,
    EV_CHARGING_DB_STATION_NAME,
    EV_CHARGING_DB_STATION_DESC,
    EV_CHARGING_DB_STATION_ADDR,
    EV_CHARGING_DB_STATION_CITY,
    EV_CHARGING_DB_STATION_STATE,
    EV_CHARGING_DB_STATION_ZIP,
    EV_CHARGING_DB_STATION_PHONE,
    EV_CHARGING_DB_STATION_CHARGE_TYPE_COUNT,
    EV_CHARGING_DB_STATION_CHARGE_TYPES,
    EV_CHARGING_DB_STATION_CHARGE_NUMS,
    EV_CHARGING_DB_STATION_LAT,
    EV_CHARGING_DB_STATION_LON,
    EV_CHARGING_DB_STATION_AMENITIES,
    EV_CHARGING_DB_STATION_MAX_FIELDS

} EV_CHARGING_DB_STATION_FIELDS_ENUM;

// Price Table

#define EV_CHARGING_INSERT_UPDATE_PRICE_ROW \
    "insert or replace into "PRICE_TABLE_NAME" " \
    "values(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

#define EV_CHARGING_CREATE_PRICES_TABLE \
    "create table "PRICE_TABLE_NAME"( " \
        "regionID int, " \
        "StationID int, " \
        "brand varchar, " \
        "logoID int," \
        "name varchar, description varchar, addr varchar, " \
        "city varchar, state int, " \
        "ZIP varchar, phone int, " \
        "chargeTypeCount int, "\
        "chargeTypes blob, " \
        "chargeNums blob, " \
        "lat int, lon int, " \
        "amenities int, " \
        "priceAgeInEpochSeconds int, " \
        "primary key (regionID, stationID)" \
        ");"

#define EV_CHARGING_CREATE_STATION_TABLE \
    "create table "STATION_TABLE_NAME"%u ( " \
        "StationID int, " \
        "brand varchar, " \
        "logoID int," \
        "name varchar, description varchar, addr varchar, " \
        "city varchar, state int, " \
        "ZIP varchar, phone int, " \
        "chargeTypeCount int, "\
        "chargeTypes blob, " \
        "chargeNums blob, " \
        "lat int, lon int, " \
        "amenities int, " \
        "primary key (stationID)" \
        ");"

#define EV_CHARGING_INSERT_UPDATE_STATION_ROW \
    "insert or replace into "STATION_TABLE_NAME"%u " \
    "values(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);"

// Enumeration specifying all the available fields
// in the price table
typedef enum ev_charging_db_price_fields_enum {
    EV_CHARGING_DB_PRICE_REGION_ID = 0,
    EV_CHARGING_DB_PRICE_STATION_ID,
    EV_CHARGING_DB_PRICE_BRAND,
    EV_CHARGING_DB_PRICE_LOGO,
    EV_CHARGING_DB_PRICE_NAME,
    EV_CHARGING_DB_PRICE_DESC,
    EV_CHARGING_DB_PRICE_ADDR,
    EV_CHARGING_DB_PRICE_CITY,
    EV_CHARGING_DB_PRICE_STATE,
    EV_CHARGING_DB_PRICE_ZIP,
    EV_CHARGING_DB_PRICE_PHONE,
    EV_CHARGING_DB_PRICE_CHARGE_TYPE_COUNT,
    EV_CHARGING_DB_PRICE_CHARGE_TYPES,
    EV_CHARGING_DB_PRICE_CHARGE_NUMS,
    EV_CHARGING_DB_PRICE_LAT,
    EV_CHARGING_DB_PRICE_LON,
    EV_CHARGING_DB_PRICE_AMENITIES,
    EV_CHARGING_DB_PRICE_AGE_EPOCH_SECONDS,
    EV_CHARGING_DB_PRICE_MAX_FIELDS

} EV_CHARGING_DB_PRICE_FIELDS_ENUM;

// Logo Table

#define EV_CHARGING_CREATE_LOGO_TABLE \
    "create table "LOGO_TABLE_NAME" ( " \
        "ID int, " \
        "type int, " \
        "tablever int, " \
        "logover int, " \
        "baseline tinyint, "\
        "primary key (ID, type) " \
        ");"

#define EV_CHARGING_INSERT_UPDATE_LOGO \
    "insert or replace into "LOGO_TABLE_NAME" " \
    "values(?, ?, ?, ?, ?);"

#define EV_CHARGING_REMOVE_LOGO \
    "delete from "LOGO_TABLE_NAME \
    " where ID=%3u and type=%1u;"

#define EV_CHARGING_SELECT_ALL_LOGOS \
    "select * from "LOGO_TABLE_NAME";"

typedef enum ev_charging_db_logo_fields_enum
{
    EV_CHARGING_DB_LOGO_ID = 0,
    EV_CHARGING_DB_LOGO_TYPE,
    EV_CHARGING_DB_LOGO_TABLE_VER,
    EV_CHARGING_DB_LOGO_LOGO_VER,
    EV_CHARGING_DB_LOGO_BASELINE,
    EV_CHARGING_DB_LOGO_MAX_FIELDS

} EV_CHARGING_DB_LOGO_FIELDS_ENUM;

typedef struct ev_charging_ver_ctrl_struct
{
    // The version of the DB contents
    N16 n16DBContentVersion;

} EV_CHARGING_VER_CTRL_STRUCT;

typedef struct ev_charging_db_obj_struct
{
    FUEL_SERVICE_OBJECT hFuelService;

    BOOLEAN bInTransaction;

    // A structure which groups all the
    // versions tracked by this service
    EV_CHARGING_VER_CTRL_STRUCT sVerCtrl;

    // DB Connection Handle for ref database
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    const char *pacRefDatabaseDirPath;
    const char *pacCurRefDatabaseFilePath;
    const char *pacNextRefDatabaseFilePath;

    // DB Connection Handle for persist database
    SQL_INTERFACE_OBJECT hSQLPersistConnection;

    // Work buffer
    char acBuffer[FUEL_MAX_SQL_STRING_LENGTH];

} EV_CHARGING_DB_OBJ_STRUCT;

typedef struct ev_charging_db_result_struct
{
    EV_CHARGING_DB_OBJ_STRUCT *psObj;
    BOOLEAN bSuccess;

} EV_CHARGING_DB_RESULT_STRUCT;

typedef struct ev_charging_logo_copy_result_struct
{
    SQL_INTERFACE_OBJECT hWriteConnection;
    BOOLEAN bSuccess;

} EV_CHARGING_LOGO_COPY_RESULT_STRUCT;

typedef struct ev_charging_db_version_result_struct
{
    EV_CHARGING_VER_CTRL_STRUCT *psVerCtrl;
    BOOLEAN bSuccess;

} EV_CHARGING_DB_VERSION_RESULT_STRUCT;


typedef struct ev_charging_db_load_station_struct
{
    FUEL_REGION tRegion;
    FUEL_STATION_ID tStation;
    EV_CHARGING_DB_OBJ_STRUCT *psObj;
    BOOLEAN bSuccess;
    size_t tNumObjectsCreated;

} EV_CHARGING_DB_LOAD_STATION_STRUCT;

// Structure used to perform ev charging database queries
typedef struct ev_db_price_query_result_struct
{
    BOOLEAN bSuccess;
    UN32 un32UTCsec;
    UN32 un32PriceAgeLimit;
    EV_CHARGING_DB_OBJ_STRUCT *psObj;

} EV_DB_PRICE_QUERY_RESULT_STRUCT;

typedef struct ev_db_station_update_struct
{
    FUEL_STATION_ROW_STRUCT *psStationRow;
    EV_CHARGING_DB_STATION_FIELDS_ENUM aeFields[EV_CHARGING_DB_STATION_MAX_FIELDS];
    UN8 un8NumFields;
    UN8 aun8RawRefuelingTypeData[FUEL_REFUELING_TYPES_ARRAY_SIZE];

} EV_DB_STATION_UPDATE_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* DB Interface Functions */

static FUEL_DB_INTERFACE_OBJECT hConnect (
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    const char *pacRefDBDirPath,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        );

static void vDisconnect (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        );

static BOOLEAN bCanProcessUpdate (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8TextVersion
        );

static FUEL_TYPE_ENUM eMatchFuelType (
    UN8 un8TextId
        );

static size_t tLoadStations (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        );

static BOOLEAN bLoadPrices (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_REGION tRegionID,
    FUEL_STATION_ID tStationID
        );

static BOOLEAN bTextQuery (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    UN8 un8NewTextVersion
       );

static BOOLEAN bUpdateTextTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin,
    UN8 un8NewTextVersion
        );

static BOOLEAN bUpdateTextEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    SQL_INTERFACE_OBJECT hSQLConnection,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        );

static BOOLEAN bUpdateLogoEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bDelete,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        );

static void vUpdatePriceTable (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    BOOLEAN bBegin
        );

static BOOLEAN bUpdatePriceEntry (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    FUEL_STATION_OBJECT hStation
        );

static UN16 un16DBContentVer (
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        );

static BOOLEAN bRefDBBank (
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        );

static BOOLEAN bDBUpdateBegin (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize
        );

static BOOLEAN bDBUpdateEnd (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char *pcSQLCommandBuffer,
   size_t tBufferSize,
   UN16 un16NewDBVersion
       );

static BOOLEAN bRegionUpdateBegin (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pcSQLCommandBuffer,
    size_t tBufferSize,
    FUEL_REGION tRegionID
        );

static BOOLEAN bRegionUpdateEnd (
    SQL_INTERFACE_OBJECT hSQLConnection
        );

static BOOLEAN bStationUpdate (
   SQL_INTERFACE_OBJECT hSQLConnection,
   char **ppcSQLCommandBuffer,
   size_t *ptBufferSize,
   FUEL_STATION_ROW_STRUCT *psStationRow,
   FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
   BOOLEAN bAmenitiesUpdated
       );

static BOOLEAN bPriceAgeFilter (
    UN32 un32PriceAgeInEpochSeconds,
    UN32 un32CurrentTime
        );

// Text functions

static BOOLEAN bInitializeWellKnownFuelTypes (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
       );

// Price functions

static BOOLEAN bFlushPrices (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bAgeoutPrices (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

// Station functions

static BOOLEAN bUpdateStationDBEntry (
    SQL_INTERFACE_OBJECT hSQLConnection,
    char *pacBuffer,
    size_t tBufferSize,
    FUEL_STATION_ROW_STRUCT *psStation,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    BOOLEAN bAmenitiesUpdated
        );

static BOOLEAN bPrepareUpdateCommand (
    char *pacBuffer,
    size_t tBufferSize,
    EV_DB_STATION_UPDATE_STRUCT *psUpdate,
    BOOLEAN bNewStation,
    BOOLEAN bAmenitiesUpdated
        );

static BOOLEAN bBuildStationLoadQuery (
    EV_CHARGING_DB_OBJ_STRUCT *psObj,
    LOCATION_OBJECT hLocation,
    FUEL_REGION tRegion
        );

// General purpose functions

static BOOLEAN bLoadTables (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bLoadTextTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bLoadRegionTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bLoadLogoTable (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static void vStartTransaction (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static void vEndTransaction (
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );


/* SQL callback functions */

// Writers

static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    EV_CHARGING_DB_OBJ_STRUCT *psObj
        );

static BOOLEAN bPrepareTextColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_TEXT_ROW_STRUCT *psTextRow
        );

static BOOLEAN bPrepareLogoColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_LOGO_ROW_STRUCT *psLogoRow
        );

static BOOLEAN bPrepareRegionColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_REGION *ptRegionID
        );

static BOOLEAN bPreparePriceColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    FUEL_DB_UPDATE_PRICE_TABLE_STRUCT *psDBUpdate
        );

static BOOLEAN bPrepareStationColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    EV_DB_STATION_UPDATE_STRUCT *psUpdate
        );

// Readers

static BOOLEAN bVerifyAndLoadDBVersionFields (
    SQL_INTERFACE_OBJECT hSQLConnection,
    EV_CHARGING_VER_CTRL_STRUCT *psVerCtrl
        );

static N32 n32ExtractDataVersion (
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_VERSION_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectText (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectLogos (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectLogosIntoPersistDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_LOGO_COPY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectRegionExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbRegionExists
        );

static BOOLEAN bProcessSelectRegions (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectStations (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_CHARGING_DB_LOAD_STATION_STRUCT *psBuild
        );

static BOOLEAN bProcessSelectStationExists (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbStationExists
        );

static BOOLEAN bProcessSelectPrices (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    EV_DB_PRICE_QUERY_RESULT_STRUCT *psResult
        );

  /***************/
 /** VARIABLES **/
/***************/

// Manager interface for plugins
const FUEL_DB_INTERFACE_STRUCT GsEVChargingDBIntf =
{
    /*.pacFolder = */ NULL,
    /*.pacRefDBNameA = */ NULL,
    /*.pacRefDBNameB = */ NULL,
    /*.n16DBVersion = */ 0,
    /*.pacPersistDBName = */ NULL,
    /*.hConnect = */hConnect,
    /*.vDisconnect = */vDisconnect,
    /*.bCanProcessUpdate = */bCanProcessUpdate,
    /*.eMatchFuelType = */eMatchFuelType,
    /*.tLoadStations = */tLoadStations,
    /*.bLoadPrices = */bLoadPrices,
    /*.bTextQuery = */bTextQuery,
    /*.bUpdateTextTable = */bUpdateTextTable,
    /*.bUpdateTextEntry = */bUpdateTextEntry,
    /*.bUpdateLogoEntry = */bUpdateLogoEntry,
    /*.vUpdatePriceTable = */vUpdatePriceTable,
    /*.bUpdatePriceEntry = */bUpdatePriceEntry,
    /*.un16DBContentVer = */un16DBContentVer,
    /*.bRefDBBank = */bRefDBBank,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.bRegionUpdateBegin = */bRegionUpdateBegin,
    /*.bRegionUpdateEnd = */bRegionUpdateEnd,
    /*.bStationUpdate = */bStationUpdate
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _EV_CHARGING__H_
