/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _EPG_MGR_OBJ_H_
#define _EPG_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "sms_task.h"
#include "srm_obj.h"

#include "dataservice_mgr_impl.h"
#include "epg_interface.h"
#include "eprogram_obj.h"
#include "epg_cache_obj.h"
#include "epg_mgr_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/
// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define EPG_MGR_API_DBG_PREFIX                  "EPG MGR API"
#define EPG_MGR_DBG_PREFIX                      "EPG MGR"
#define EPG_LOADER_DBG_PREFIX                   "EPG LOADER"

/* EPG directory name */
#define EPG_DIRECTORY_NAME                      "epg"

/* EPG objects names */
#define EPG_MGR_OBJECT_NAME                     "EPG:MGR"

#define EPG_LOADER_OBJECT_NAME                  "EPG:LOADER"

#define EPG_SCHEDULE_ROOT_OBJECT_NAME           EPG_MGR_OBJECT_NAME":ScheduleRoot"
#define EPG_SCHEDULE_CHANNELS_LIST_NAME         EPG_SCHEDULE_ROOT_OBJECT_NAME":ChannelsLL"
#define EPG_SCHEDULE_TOPICS_LIST_NAME           EPG_SCHEDULE_ROOT_OBJECT_NAME":TopicsLL"
#define EPG_SCHEDULE_STATES_ARRAY_NAME          EPG_SCHEDULE_ROOT_OBJECT_NAME":SegStates"

#define EPG_PROC_DESC_OBJECT_NAME               EPG_MGR_OBJECT_NAME":ProcDescriptor"
#define EPG_GRID_DESC_OBJECT_NAME               EPG_PROC_DESC_OBJECT_NAME":GridDescArray"
#define EPG_TEXT_DESC_OBJECT_NAME               EPG_PROC_DESC_OBJECT_NAME":TextDescArray"
#define EPG_YESTERDAY_TEXT_DESC_OBJECT_NAME     EPG_PROC_DESC_OBJECT_NAME":YesterdayTextDescArray"

#define EPG_OWNER_OBJECT_NAME                   EPG_MGR_OBJECT_NAME":ScheduleGrid"
#define EPG_DIRECTORY_PATH_OBJECT_NAME          EPG_OWNER_OBJECT_NAME":DirPath"

#define EPG_FILE_PATH_OBJECT_NAME               "EPG:FilePathStr"
#define EPG_TMP_PAYLOAD_BUF_OBJECT_NAME         "EPG:TmpAuPayload"

#define EPG_SUPPORTED_OPTIONS (\
    (DATASERVICE_OPTIONS_MASK) \
    DATASERVICE_OPTION_NONE \
    )

/* EPG Loader events masks */
#define EPG_LOADER_EVENT_NONE                   (0x0000)
#define EPG_LOADER_EVENT_PROCESSING_FINISHED    (0x0001)
#define EPG_LOADER_EVENT_PROCESSING_CANCELLED   (0x0002)
#define EPG_LOADER_EVENT_PROCESSING_ERROR       (0x0004)

#define EPG_LOADER_EVENT_ALL (\
        EPG_LOADER_EVENT_PROCESSING_FINISHED | \
        EPG_LOADER_EVENT_PROCESSING_CANCELLED | \
        EPG_LOADER_EVENT_PROCESSING_ERROR)

/* Compressed text file name format constants */

#define EPG_FILE_NAME_LEN_MAX           (15)
#define EPG_GRID_FILE_NAME_LABEL        "epg_grid_%1u"
#define EPG_TEXT_FILE_NAME_LABEL        "epg_text_%1u"
#define EPG_CACHE_FILE_NAME_LABEL       "epg_cache_%1u"
#define EPG_MIDNIGHT_FILE_NAME_LABEL    "epg_midnight"

#define EPG_FILE_EXT_LEN_MAX            (4)
#define EPG_BIN_FILE_EXT                ".bin"
#define EPG_TMP_FILE_EXT                ".tmp"

#define EPG_GRID_BIN_FILE_NAME_LABEL        EPG_GRID_FILE_NAME_LABEL EPG_BIN_FILE_EXT
#define EPG_GRID_TMP_FILE_NAME_LABEL        EPG_GRID_FILE_NAME_LABEL EPG_TMP_FILE_EXT

#define EPG_TEXT_BIN_FILE_NAME_LABEL        EPG_TEXT_FILE_NAME_LABEL EPG_BIN_FILE_EXT
#define EPG_TEXT_TMP_FILE_NAME_LABEL        EPG_TEXT_FILE_NAME_LABEL EPG_TMP_FILE_EXT

#define EPG_CACHE_BIN_FILE_NAME_LABEL       EPG_CACHE_FILE_NAME_LABEL EPG_BIN_FILE_EXT
#define EPG_CACHE_TMP_FILE_NAME_LABEL       EPG_CACHE_FILE_NAME_LABEL EPG_TMP_FILE_EXT

#define EPG_MIDNIGHT_BIN_FILE_NAME_LABEL    EPG_MIDNIGHT_FILE_NAME_LABEL EPG_BIN_FILE_EXT
#define EPG_MIDNIGHT_TMP_FILE_NAME_LABEL    EPG_MIDNIGHT_FILE_NAME_LABEL EPG_TMP_FILE_EXT

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef enum epg_loader_state_enum
{
    EPG_LOADER_STATE_UNINITIALIZED,
    EPG_LOADER_STATE_IDLE,
    EPG_LOADER_STATE_PROCESSING

} EPG_LOADER_STATE_ENUM;

// Typedef given a specific value due to
// use in the service specific event mask.
typedef enum epg_loader_result_enum
{
    EPG_LOADER_RESULT_PROCESSING_FINISHED = 0,
    EPG_LOADER_RESULT_PROCESSING_CANCELLED,
    EPG_LOADER_RESULT_PROCESSING_ERROR,
    EPG_LOADER_RESULT_UNKNOWN

} EPG_LOADER_RESULT_ENUM;

typedef enum epg_sub_update_type_enum
{
    // Service is started
    EPG_SUB_UPDATE_TYPE_START,

    // Service is stopping
    EPG_SUB_UPDATE_TYPE_STOP,

    // Data change
    EPG_SUB_UPDATE_TYPE_PEM_CHANGE,

    EPG_SUB_UPDATE_INVALID_TYPE

} EPG_SUB_UPDATE_TYPE_ENUM;

typedef enum epg_file_type_enum
{
    EPG_FILE_GRID_BIN,
    EPG_FILE_GRID_TMP,
    EPG_FILE_TEXT_BIN,
    EPG_FILE_TEXT_TMP,
    EPG_FILE_CACHE_BIN,
    EPG_FILE_CACHE_TMP,
    EPG_FILE_MIDNIGHT_BIN,
    EPG_FILE_MIDNIGHT_TMP

} EPG_FILE_TYPE_ENUM;

/* Grid data releated definitions */

typedef struct epg_grid_file_header_struct
{
    UN16    un16Epoch;
    UN8     un8Version;
    UN32    un32AusCount;

} EPG_GRID_FILE_HEADER_STRUCT;

typedef struct epg_text_file_header_struct
{
    UN16    un16Epoch;
    UN8     un8Version;
    UN16    un16StringsCount;
    UN32    un32DecompressedSize;
    UN32    un32CompressedSize;

} EPG_TEXT_FILE_HEADER_STRUCT;

typedef struct epg_midnight_file_header_struct
{
    UN16    un16Epoch;
    UN8     un8Version;
    UN8     un8ProgramSize;
    UN32    un32NumPrograms;

} EPG_MIDNIGHT_FILE_HEADER_STRUCT;

typedef struct epg_full_schedule_root
{
    OSAL_OBJECT_HDL hEpgChannels; // Full list of channels
    OSAL_OBJECT_HDL hFullTopicsList; // Full list of all topics mentioned in this schedule
    EPG_TEXT_CACHE_STRUCT *psTextCache; // Strings cache //TODO: Create on demand?

    EPG_SEG_STATE_ENUM *aeSegState; // Dynamic array (size = un8NumSegmentsTotal)
    UN8 un8NumSegmentsTotal;
    UN8 un8NumSegmentsCollected;

    EPG_SCHEDULE_VERSION_STRUCT sVersion;
    SCHEDULE_STATE_ENUM eState;

} EPG_FULL_SCHEDULE_ROOT;

typedef struct epg_processing_descriptor_struct
{
    // Grid related processing information
    UN8 un8CurrentSegment;
    EPG_CHANNEL_OBJECT_STRUCT *psCurrentChannel;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule;
    TIME_T tCumulativeTime;

    // Maximum number of segments
    UN8 un8NumSegmentsMax;

    // Grid and Text data files related information
    EPG_GRID_FILE_HEADER_STRUCT *asGridFileDesc; // Dynamic array (size = un8NumSegmentsMax)
    EPG_TEXT_FILE_HEADER_STRUCT *asTextFileDesc; // Dynamic array (size = un8NumSegmentsMax)

    // Midnight files related information
    EPG_MIDNIGHT_FILE_HEADER_STRUCT sMidnightFileDesc;
    BOOLEAN *abYesterdayTextSaved; // Dynamic array (size = un8NumSegmentsMax)

} EPG_PROCESSING_DESCRIPTOR_STRUCT;

typedef struct epg_owner_struct
{
    /* EPG directory path */
    char *pacEpgDirectoryPath;

    /* Collected schedule data*/
    EPG_FULL_SCHEDULE_ROOT *psCurrentSchedule;

    /* Descriptor required during processing of incoming schedule*/
    EPG_PROCESSING_DESCRIPTOR_STRUCT *psProcessingDescriptor;

} EPG_OWNER_STRUCT;

/* Data used by the loader task */
typedef struct epg_mgr_loader_control_struct
{
    // EPG MGR Handle
    EPG_SERVICE_OBJECT hEpgService;

    /* EPG Loader task handle */
    /* Created by the service task; destroyed by the loader task itself. */
    SMS_TASK_HANDLE hLoaderTask;

    /* Task's Event Handler used to send messages to the task's handler function */
    /* Created and destroyed by the loader task itself. */
    SMS_EVENT_HANDLER hEventHandler;

    /* Event to send notification to the task's initiator */
    /* Created and destroyed by the service task. */
    SMSU_EVENT_STRUCT sLoaderEvent;

    /* Set in the service task to inform the loader task to cancel current processing */
    BOOLEAN bCancel;

} EPG_MGR_LOADER_CONTROL_STRUCT;

typedef struct epg_mgr_object_struct
{
    /* Event to send notification to the EPG service's subscribers */
    SMSU_EVENT_STRUCT sEvent;

    /* Maximum number of segments to be received.
       Initialized once on EPG MGR creation. */
    UN8 un8NumSegmentsToRecieve;

    /* EPG Interface object used with EPG Parser API */
    EPG_PARSER_INTERFACE_OBJECT hEpgParserObject;

    /* Object to hold EPG Schedule data to expose.
       Accessed via psGetAppFacingObject() function. */
    EPG_OWNER_STRUCT *psEpgOwner;

    /* Data used by the EPG loader task */
    /* Created by the service task; destroyed by the loader task. */
    EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader;

    /* FOLLOWING MEMBERS MUST BE ACCESSED UNDER EPG_MGR_OBJECT LOCK */

    /* Flag used to indicate the service is currently stopping
       Required for synchronization of DSM (manager) and EPG_MGR
       (schedule loader) tasks termination. */
    BOOLEAN bStopping;
    /* Current state of EPG Loader task */
    EPG_LOADER_STATE_ENUM eEpgLoaderState;
    /* Indicates initial schedule is loaded on service startup */
    BOOLEAN bInitialEpgLoaded;

} EPG_MGR_OBJECT_STRUCT;

/* This struct is used to pass event type and pointer to EPG_MGR object
   to subscribed decoders iterator */
typedef struct epg_sub_update_struct
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj;
    EPG_SUB_UPDATE_TYPE_ENUM eUpdateType;
    BOOLEAN bSuccess;

} EPG_SUB_UPDATE_STRUCT;

/* Service Private Event types */
typedef enum epg_mgr_event_enum
{
    EPG_MGR_STOP_PREPARE_EVENT,
    EPG_MGR_LOADER_EVENT,
    EPG_MGR_INVALID_EVENT

} EPG_MGR_EVENT_ENUM;

// Structure for the epg loader result event
typedef struct epg_mgr_loader_event_struct
{
    EPG_LOADER_RESULT_ENUM eLoaderResult;

} EPG_MGR_LOADER_EVENT_STRUCT;

// Union of all epg-private event structures
typedef union epg_mgr_event_union
{
    EPG_MGR_LOADER_EVENT_STRUCT sLoader;

} EPG_MGR_EVENT_UNION;

// Top-level ePG event struct
typedef struct epg_mgr_event_struct
{
    EPG_MGR_EVENT_ENUM eEvent;
    EPG_MGR_EVENT_UNION uEvent;
} EPG_MGR_EVENT_STRUCT;

// Sanity check; please put this in each service implementation
// that uses service-private events.  If this fails,
// you MUST REDUCE the size of the structure!
DSM_IMPL_EVENT_COMPILE_TIME_ASSERT(EPG_MGR_EVENT_STRUCT);

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static EPG_SERVICE_OBJECT hStart (
        const char *pacSRHDriverName,
        UN8 un8NumSegmentsForRecieve,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

static void vStop (
    EPG_SERVICE_OBJECT hEpgService
            );

static SMSAPI_RETURN_CODE_ENUM eIteratePrograms (
        EPG_SERVICE_OBJECT hEpgService,
        PROGRAM_ITERATOR_CALLBACK bEpgIterator,
        void *pvIteratorArg,
        EPG_FILTER_STRUCT *psEpgFilter
            );

static SMSAPI_RETURN_CODE_ENUM eScheduleStates (
        EPG_SERVICE_OBJECT hEpgService,
        SCHEDULE_STATE_STRUCT *psCurrentState,
        SCHEDULE_STATE_STRUCT *psInProcessState
            );

static SMSAPI_RETURN_CODE_ENUM eEnableCheckVersionsBySegments (
        EPG_SERVICE_OBJECT hEpgService,
        BOOLEAN bEnable
            );

/* EPG Parser Client interface implementation */

static BOOLEAN bEpgClientProcessParserEvent (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bEpgClientProcessCompressedTextPayload (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            );

static void vEpgClientSaveAuData (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            );

static void vEpgClientSegmentGridCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            );

static void vEpgClientSegmentTextCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber,
        UN16 un16StringsCount,
        UN32 un32DecompressedFileSize
            );

static void vEpgClientWholeSegmentCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            );

static void vEpgClientGetVersion (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            );

static void vEpgClientVersionChanged (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psNewVersion
            );

static void vEpgClientRestartGridProcessing (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            );

static void vEpgClientRestartTextProcessing (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            );

/* Object Private Prototypes */

/* General */

static void vUninitEpgMgrObject (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static BOOLEAN bIsEpgInitialized (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static BOOLEAN bPrepareForInitialProcessing (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static EPG_OWNER_STRUCT *psCreateEpgOwner (
        void
            );

static void vDestroyEpgOwner (
        EPG_OWNER_STRUCT *psEpgOwner
            );

static BOOLEAN bBuildEpgFilePath (
        EPG_OWNER_STRUCT *psEpgOwner
            );

static EPG_OWNER_STRUCT *psGetAppFacingObject (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static char *pcGetFilePath (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_FILE_TYPE_ENUM eFileType,
        UN8 un8SegNum
            );

static BOOLEAN bSaveCompressedTextPartToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_AU_INFO_STRUCT *psAuData
            );

static BOOLEAN bSaveGridPayloadPartToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_AU_INFO_STRUCT *psAuData
            );

static BOOLEAN bCopyEpgFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_FILE_TYPE_ENUM eSrcFileType,
        UN8 un8SrcSegNum,
        EPG_FILE_TYPE_ENUM eDstFileType,
        UN8 un8DstSegNum,
        BOOLEAN bRename
            );

static BOOLEAN bMoveProcessingFilesToReady (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegmentNumber
            );

static SMSAPI_RETURN_CODE_ENUM eScheduleState (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        SCHEDULE_STATE_STRUCT *psState
            );

static BOOLEAN bLoadStoredSegmentGridVersion (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegNum,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            );

static EPG_RETURN_CODES_ENUM eLoadStoredSchedule (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static EPG_RETURN_CODES_ENUM eLoadStoredSegmentGridData (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        UN8 un8SegNum
            );

static EPG_RETURN_CODES_ENUM eLoadStoredSegmentTextData (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion,
        UN8 un8SegNum
            );

static BOOLEAN bCreateMidnightEventCopy (
        EPG_OWNER_STRUCT *psEpgOwner,
        SERVICE_ID tServiceId,
        TIME_T tEventEndTime
            );

PROG_EVENT_STRUCT *psGetMidnightEventCopy (
        EPG_OWNER_STRUCT *psEpgOwner,
        SERVICE_ID tServiceId,
        TIME_T tEventEndTime
            );

static BOOLEAN bCopyYesterdayTextFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegNumCopyFrom,
        UN8 *pun8SegNumCopiedTo
            );

static BOOLEAN bSaveMidnightProgramToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        PROG_EVENT_STRUCT *psProgramEvent
            );

static BOOLEAN bSaveTopicToFile (
        TOPIC_OBJECT hTopic,
        FILE *pFile
            );

static BOOLEAN bLoadMidnightPrograms (
        EPG_OWNER_STRUCT *psEpgOwner
            );

static BOOLEAN bLoadProgramEventsFromFile (
        FILE *pFile,
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        UN32 un32NumProgramsToRead
            );

static BOOLEAN bReadTopicsFromFile (
        FILE *pFile,
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        UN8 un8NumTopicsToRead
            );

static BOOLEAN bMoveProcessingMidnightFileToReady (
        EPG_OWNER_STRUCT *psEpgOwner
            );

static void vMoveReadyTextFilesToCache (
        EPG_OWNER_STRUCT *psEpgOwner
            );

/* Events handlers */

static void vEventHandler (
        DATASERVICE_MGR_OBJECT hDataService,
        DATASERVICE_EVENT_MASK tCurrentEvent,
        void *pvEventArg,
        void *pvEventCallbackArg
            );

static void vHandleServiceStopRequest (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static void vInitiateServiceStop (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static BOOLEAN bHandleServiceReady (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static void vHandleDecoderSubscribed (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        DECODER_OBJECT hDecoder
            );

static void vHandleDecoderUnsubscribed (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        DECODER_OBJECT hDecoder
            );

static BOOLEAN bHandleMessageReception (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        OSAL_BUFFER_HDL hPayload
            );

static BOOLEAN bHandleEpgLoaderReport (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_LOADER_RESULT_ENUM eLoaderResult
            );

static BOOLEAN bSendUpdate (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        DECODER_OBJECT hDecoder,
        EPG_SUB_UPDATE_TYPE_ENUM eUpdateType
            );

static BOOLEAN bSendEpgUpdateEvent (
        DECODER_OBJECT hDecoder,
        SMS_EVENT_EPG_STRUCT *psUpdateData
            );

static BOOLEAN bAttachChannelsEpgData (
        DECODER_OBJECT hDecoder,
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

BOOLEAN bDetachChannelsEpgData (
        DECODER_OBJECT hDecoder,
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

/* Linked List Iterators / Searchers */

static BOOLEAN bIterateSubscribedForEvent (
        DECODER_OBJECT hDecoder,
        void *pvArg
            );

static BOOLEAN bEpgChannelsIterator (
        EPG_CHANNEL_OBJECT hEpgChannel,
        EPG_EVENTS_ITERATOR_STRUCT *psIterator
            );

static BOOLEAN bProgramsIterator (
        PROGRAM_OBJECT hProgEvent,
        EPG_EVENTS_ITERATOR_STRUCT *psIterator
            );

/* EPG schedule loader task functions */

/* Functions called in the main task context: */
static BOOLEAN bEpgLoaderTaskInitialize (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static void vEpgLoaderTaskUninitialize (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static BOOLEAN bEpgLoaderTaskStart (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static void vEpgLoaderTaskCancel (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

/* Functions called in the loader task context: */
static void vEpgLoaderEventCallback (
    void *pvObj,
    SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vEpgLoaderTaskEventHandler (
        SMS_OBJECT hEpgLoader,
        SMS_EVENT_HDL hEvent
            );

static void vEpgTaskLoadInitialSchedule (
        EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader
            );

static BOOLEAN bEpgLoadingCancellationCheckCallback (
        void *pvCallbackArg
            );

/* Data processing */

static UN32 un32CalculateEventDuration (
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        TIME_T tStartTime
            );

static BOOLEAN bInitScheduleProcessing (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8NumSegments
            );

static void vUninitScheduleProcessing (
        EPG_OWNER_STRUCT *psEpgOwner
            );

static EPG_FULL_SCHEDULE_ROOT *psCreateEpgSchedule (
        SMS_OBJECT hParent,
        UN8 un8NumSegmentsTotal
            );

static void vCleanEpgSchedule (
        EPG_FULL_SCHEDULE_ROOT *psSchedule
            );

static void vDestroyEpgSchedule (
        EPG_FULL_SCHEDULE_ROOT *psSchedule
            );

static EPG_PROCESSING_DESCRIPTOR_STRUCT *psCreateProcessingDescriptor (
        SMS_OBJECT hParent,
        UN8 un8NumSegmentsMax
            );

static void vCleanGridDescriptors (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static void vCleanTextDescriptors (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static void vDestroyProcessingDescriptor (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static EPG_CHANNEL_OBJECT_STRUCT *psGetEpgChannel (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        SERVICE_ID tServiceId,
        BOOLEAN bCreate
            );

static BOOLEAN bProcessParserEvent (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bUnwrapParserEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static BOOLEAN bUnwrapDurationEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static BOOLEAN bUnwrapModifyEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static BOOLEAN bUnwrapAddEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static BOOLEAN bUnwrapFullEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            );

static PROG_EVENT_STRUCT *psCreateNewProgramEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        TIME_T tStartTime,
        SMS_OBJECT hParent
            );

static BOOLEAN bSaveTopicInfo (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        TOPIC_ID tTopicId,
        char *pcTopicName
            );

static BOOLEAN bAddExtraProgramInstances (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_CHANNEL_OBJECT_STRUCT *psChannel,
        PROG_EVENT_STRUCT *psProgramEvent,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

static BOOLEAN bCopyTopics (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        TOPIC_ID *atTopicId,
        UN8 un8NumTopics
            );

static void vHandleReceivedScheduleCompletion (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            );

static void vHandleLoadedScheduleCompletion (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        BOOLEAN bUnbrokenSchedule
            );

  /***************/
 /** VARIABLES **/
/***************/

// EPG interface struct
static EPG_CLIENT_INTERFACE_STRUCT gsEpgClientInterface =
{
    /*.hClientObj = */ EPG_CLIENT_INTERFACE_INVALID_OBJECT,
    /*.bProcessParserEvent = */ bEpgClientProcessParserEvent,
    /*.bProcessCompressedTextPayload = */ bEpgClientProcessCompressedTextPayload,
    /*.vSaveAuData*/ vEpgClientSaveAuData,
    /*.vSegmentGridCompleted = */ vEpgClientSegmentGridCompleted,
    /*.vSegmentTextCompleted = */ vEpgClientSegmentTextCompleted,
    /*.vWholeSegmentCompleted = */ vEpgClientWholeSegmentCompleted,
    /*.vGetVersion = */ vEpgClientGetVersion,
    /*.vVersionChanged = */ vEpgClientVersionChanged,
    /*.vRestartGridProcessing = */ vEpgClientRestartGridProcessing,
    /*.vRestartTextProcessing = */ vEpgClientRestartTextProcessing
};

// EPG task config struct
static const SMS_TASK_CONFIGURATION_STRUCT gsEPGMTaskConfiguration =
{
    /*.pacName = */ "EPG_LOADER_TASK",
    /*.un32StackSize = */ 32768, // bytes
    /*.ePriority = */ OSAL_TASK_PRIORITY_MEDIUM,
    /*.un32Options = */ OSAL_TASK_OPTION_DEBUG_OUTPUT,
    /*.un16ReportingInterval = */ 0, // seconds
    /*.un32EventQueueSize = */ 8, // events
    /*.un32EventHandlerOptions = */ SMS_EVENT_HANDLER_OPTION_NONE
};

// Global (re-usable) instance of an interface for this object
const EPG_INTERFACE_STRUCT EPG =
{
    /*.hStart = */hStart,
    /*.vStop = */vStop,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (EPG_SERVICE_OBJECT hEpgService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (EPG_SERVICE_OBJECT hEpgService))DATASERVICE_IMPL_eErrorCode,
    /*.eIteratePrograms = */eIteratePrograms,
    /*.eScheduleStates = */eScheduleStates,
    /*.eCheckVersionsBySegments = */eEnableCheckVersionsBySegments
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsEPGStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */(DATASERVICE_STATE_HANDLER)DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */DATASERVICE_IMPL_bGenericStateHandler
};

#endif  // _EPG_MGR_OBJ_H_
