/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _DSRL_H_
#define _DSRL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "dsrl_obj.h"
#include "dataservice_mgr_impl.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define DSRL_OBJECT_NAME "DSRL"

/* State-dependent attribute constants */
#define DSRL_NUM_STATE_DEPENDENT_ATTRIBS (1)
#define DSRL_TARGET_UPDATE_ATTRIB_INDEX (0)
  /**************/
 /** TYPEDEFS **/
/**************/

// Attributes of this type are managed automatically
// by the DSRL as a part of the state transition logic
typedef struct dsrl_state_dependent_attrib_struct
{
    // State in which this attribute is
    // to be reported
    DSRL_STATE_ENUM eReportingState;

    // Value used as initializer for this value
    BOOLEAN bInitializerValue;

    // Value to report for this attribute
    BOOLEAN bValue;

    // Value staged until reporting
    // state has been entered
    BOOLEAN bStagedValue;

} DSRL_STATE_DEPENDENT_ATTRIB_STRUCT;

//  The main Data Service Result List (DSRL) structure
typedef struct dsrl_object_struct
{
    /************************************
     *  Application Provided Properties
     ************************************/

    // The maximum capacity the DSRL will maintain
    // in view of public APIs
    size_t tReportedCapacity;

    // The capacity of the DSRL as it deals
    // with adds / removes and the like
    size_t tWorkingCapacity;

    // Default sort function / arg provided by SMS
    DSRL_SORT_FUNCTION n16DefaultSort;
    void *pvDefaultSortArgs;

    // Application provided sort and filter functions
    DSRL_SORT_FUNCTION n16Sort;
    void *pvSortArgs;
    BOOLEAN bSortUsesDeviceLocation;
    DSRL_FILTER_FUNCTION bFilterItem;
    void *pvFilterArgs;

    // The callback and arguments to call when the contents of
    // this list change
    DSRL_CALLBACK vCallback;
    void *pvCallbackArgs;

    // The service associated with this DSRL object
    DATASERVICE_IMPL_HDL hService;

    /************************************
     *  Manager Provided Properties
     ************************************/

    // Service specified function to deal with the removal
    // of objects from the DSRL
    DSRL_FINALIZE_FUNCTION vFinalize;
    void *pvFinalizeArgs;

    // Function that is called just before the handle is stored in the DSRL
    // Might be used to do something complex where you use a stack object
    // to see if it *can* be added to the DSRL, then you provide a heap object
    // when you know that it will be added.  Prevent a few mallocs.
    DSRL_COMPLETE_ADD_FUNCTION hCompleteAdd;
    void *pvCompleteAddArgs;

    /************************************
     *  Instance Properties
     ************************************/

    // The list of active data service items for this DSRL
    OSAL_OBJECT_HDL hObjectList;

    // The list of the objects removed from the DSRL
    OSAL_OBJECT_HDL hRemovedObjList;

    // "Unique" ID for each entry
    DSRL_ENTRY_ID tCurrentID;

    // The type of data service data this list is managing
    DATASERVICE_TYPE_ENUM eType;

    // Type of DSRL we have here
    DSRL_TYPE_ENUM eDSRLType;

    // Flag indicates if the DSRL is being shut down
    BOOLEAN bStopping;

    // The state of the DSRL
    DSRL_STATE_ENUM eState;

    // The device location instance
    // associated with this DSRL
    LOCATION_OBJECT hDeviceLocation;

    // The service-specific descriptor pointer
    void *pvServiceData;

    // State dependent attributes include:
    // Targets Updated Flag: This indicates if the targets
    // for this DSRL have been updated for the current state transition
    DSRL_STATE_DEPENDENT_ATTRIB_STRUCT asStateAttribs[DSRL_NUM_STATE_DEPENDENT_ATTRIBS];

} DSRL_OBJECT_STRUCT;

// An iterator structure used to contain the callers call back
// function and argument when iterating the content list.
typedef struct dsrl_iterator_struct
{
    // The DSRL we are iterating
    DSRL_OBJECT hDSRL;

    // Callback information
    DSRL_ITERATOR_CALLBACK bIterator;
    void *pvIteratorArg;

    // A pointer to the current entry being iterated.
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;

    // Flag which indicates if we
    // should prune after iterating the list
    BOOLEAN bPrune;

    // Stores the indication from the iterator
    // callback which tells us if we need
    // to continue iterating or not
    BOOLEAN bContinue;

} DSRL_ITERATOR_STRUCT;

typedef struct dsrl_entry_struct
{
    DSRL_ENTRY_ID tID;

    DSRL_ENTRY_STATUS_ENUM eStatus;

    DSRL_ENTRY_OBJECT hObject;

    // The DSRL in which this entry is a member
    DSRL_OBJECT_STRUCT *psDSRL;

    // An entry handle which points to itself
    OSAL_LINKED_LIST_ENTRY hEntry;

} DSRL_ENTRY_STRUCT;

  /************/
 /** MACROS **/
/************/

#define DSRL_COMPUTE_WORKING_CAPACITY(tCapacity) \
    (((tCapacity) < (SIZE_T_MAX / 2)) ? ((tCapacity) * 2) : (tCapacity))

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static DSRL_OBJECT hCreate (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
        );

static DSRL_OBJECT hCreateForDeviceLocation (
    DATASERVICE_MGR_OBJECT hDataServiceMgr,
    DISTANCE_OBJECT hRadiusOfInterest,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArg,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArg,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArg
        );

static DSRL_OBJECT hCreateFavorites (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
        );

static SMSAPI_RETURN_CODE_ENUM eModifyTargets (
    DSRL_OBJECT hDSRL,
    DSRL_MODIFY_OPERATION_ENUM eOperation,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets
        );

static SMSAPI_RETURN_CODE_ENUM eModifyCapacity (
    DSRL_OBJECT hDSRL,
    size_t tCapacity
        );

static SMSAPI_RETURN_CODE_ENUM eModifyFilter (
    DSRL_OBJECT hDSRL,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs
        );

static SMSAPI_RETURN_CODE_ENUM eModifySort (
    DSRL_OBJECT hDSRL,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs
        );

static SMSAPI_RETURN_CODE_ENUM eModifyDeviceRadius (
    DSRL_OBJECT hDSRL,
    DISTANCE_OBJECT hRadius
        );

static void vDestroy (
    DSRL_OBJECT hDSRL
        );

static DATASERVICE_MGR_OBJECT hDataServiceMgr (
    DSRL_OBJECT hDSRL
        );

static LOCATION_OBJECT hDeviceLocation (
    DSRL_OBJECT hDSRL
        );

static DATASERVICE_TYPE_ENUM eType (
    DSRL_OBJECT hDSRL
        );

static DSRL_STATE_ENUM eState (
    DSRL_OBJECT hDSRL
        );

static size_t tCapacity (
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bTargetsUpdated (
    DSRL_OBJECT hDSRL
        );

static SMSAPI_RETURN_CODE_ENUM eIterate (
    DSRL_OBJECT hDSRL,
    DSRL_ITERATOR_CALLBACK bEntryIterator,
    void *pvIteratorArg
        );

static SMSAPI_RETURN_CODE_ENUM eExclusiveAccess (
   DSRL_OBJECT hDSRL,
   DSRL_EXCLUSIVE_ACCESS_CALLBACK vAccessCallback,
   void *pvAccessArg
      );

/* Object Private Prototypes */

static DSRL_OBJECT_STRUCT *psCreate (
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_TYPE_ENUM eType,
    size_t tCapacity,
    DSRL_SORT_FUNCTION n16Sort,
    void *pvSortArgs,
    DSRL_FILTER_FUNCTION bFilterItem,
    void *pvFilterArgs,
    DSRL_CALLBACK vCallback,
    void *pvCallbackArgs
        );

static BOOLEAN bPostTargetArgEvent (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_ACTION_ENUM eAction,
    DSRL_MODIFY_OPERATION_ENUM eOperation,
    DSRL_TARGET_OBJECT *phTargetList,
    size_t tNumTargets
        );

static SMSAPI_RETURN_CODE_ENUM eIterateLocal (
    DSRL_OBJECT_STRUCT *psObj,
    BOOLEAN bEnforceReady,
    DSRL_ITERATOR_CALLBACK bEntryIterator,
    void *pvIteratorArg
        );

static DSRL_ADD_REPLACE_RESULT_ENUM eTestDSRLEntryData (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_ENTRY_OBJECT hObject,
    BOOLEAN bTestingNewEntry,
    DSRL_ENTRY_STRUCT **ppsEntryToRemove
        );

static void vPrepareForReady (
    DSRL_OBJECT_STRUCT *psObj
        );

static void vFullSort (
    DSRL_OBJECT_STRUCT *psObj
        );

static void vInitializeStateAttributes (
    DSRL_OBJECT_STRUCT *psObj
        );

static void vUpdateStateAttributes (
    DSRL_OBJECT_STRUCT *psObj,
    DSRL_STATE_ENUM eLastState
        );

static BOOLEAN bApplyNewFilter (
    DSRL_OBJECT_STRUCT *psObj
        );

static BOOLEAN bResultEntryIterator  (
    DSRL_ENTRY_STRUCT *psResultEntry,
    DSRL_ITERATOR_STRUCT *psIterator
        );

static N16 n16SortShim (
    DSRL_ENTRY_STRUCT *psDSRLEntry1,
    DSRL_ENTRY_STRUCT *psDSRLEntry2
        );

static void vPruneResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        );

static void vRemoveResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        );

static void vRemoveAllResultEntries (
    DSRL_ENTRY_STRUCT *psResultEntry
        );

static void vDestroyResultEntry (
    DSRL_ENTRY_STRUCT *psResultEntry
        );

static BOOLEAN bApplyNewFilterToEntry (
    DSRL_ENTRY_STRUCT *psResultEntry,
    BOOLEAN *pbSuccess
        );

static BOOLEAN bRemovedEntriesIterator(
    DSRL_ENTRY_STRUCT* psResultEntry,
    DSRL_ITERATOR_STRUCT *psIterator
        );

static N16 n16CompareEntryId (
    DSRL_ENTRY_STRUCT *psListEntry1,
    DSRL_ENTRY_STRUCT *psListEntry2
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global instance of an interface for this object
const DSRL_OBJECT_INTERFACE_STRUCT DSRL =
{
    /*.hCreate = */hCreate,
    /*.hCreateForDeviceLocation = */hCreateForDeviceLocation,
    /*.hCreateFavorites = */hCreateFavorites,
    /*.eModifyTargets = */eModifyTargets,
    /*.eModifyCapacity = */eModifyCapacity,
    /*.eModifyFilter = */eModifyFilter,
    /*.eModifySort = */eModifySort,
    /*.eModifyDeviceRadius = */eModifyDeviceRadius,
    /*.vDestroy = */vDestroy,
    /*.hDataServiceMgr = */hDataServiceMgr,
    /*.hDeviceLocation = */hDeviceLocation,
    /*.eType = */eType,
    /*.eState = */eState,
    /*.tCapacity = */tCapacity,
    /*.bTargetsUpdated = */bTargetsUpdated,
    /*.eIterate = */eIterate,
    /*.eExclusiveAccess = */eExclusiveAccess
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _DSRL_H_
