/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CME_H_
#define _CME_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "cid_obj.h"
#include "sms_api.h"
#include "cme.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CME_OBJECT_NAME "CME"

// Event flags (extension of options) to indicate each content
// registration's enable/disable configuration.
#define CONTENT_STATE_DISABLED      (0x80000000)
#define CONTENT_STATE_ENABLED       (0x40000000)

// Event state flags to indicate event condition (extension of options)
#define CONTENT_STATE_INITIAL (0x20000000)
#define CONTENT_STATE_END     (0x10000000)
#define CONTENT_STATE_FINAL   (0x08000000)

#define CONTENT_STATE_NONE    (0x00000000)
#define CONTENT_STATE_MASK ( \
    CONTENT_STATE_DISABLED | \
    CONTENT_STATE_ENABLED | \
    CONTENT_STATE_INITIAL | \
    CONTENT_STATE_END | \
    CONTENT_STATE_FINAL \
        )

// mapping the content events to a priority so that the active event list
// can be sorted such that we get the higher priority events first
// higher number indicates higher priority
#define CONTENT_STATE_FINAL_PRIORITY (N16)(3)
#define CONTENT_STATE_END_PRIORITY (N16)(2)
#define CONTENT_STATE_INITIAL_PRIORITY (N16)(1)

// Update Event Timeout
#define CME_UPDATE_EVENT_TIMEOUT  (2000)

  /**************/
 /** TYPEDEFS **/
/**************/

// This is the Content Monitoring Engine's object definition.
// An instance of this object exists for each DECODER that needs one.
typedef struct cme_object_struct
{
    // Master content list (all CIDs being monitored for are contained
    // in this list).
    OSAL_OBJECT_HDL hMasterContentList;

    // An array of linked list entries, one for each CID type.
    // This makes for easier CID group searching by type. Instead of
    // searching the entire list from beginning to end, we can instead
    // start our search from the entry provided in this array which indexes
    // us immediately to the correct CID group within the list. I call these
    // "cheats" later on in the code.
    OSAL_LINKED_LIST_ENTRY ahEntry[NUM_CID_TYPES];

    // A pending event list, any events in here have occured but
    // action has not been taken yet (callback has not been called).
    OSAL_OBJECT_HDL hPendingEventList;

    // Flag to indicate if global content monitoring is enabled or disabled.
    BOOLEAN bEnabled;

    // Current iterated event (when iterating). Updated while iterating
    // events within the content list.
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;

    // Update Event Timer
    OSAL_OBJECT_HDL hUpdateEventTimer;

    // Update Timer Status
    BOOLEAN bUpdateTimerActive;

    // Update Event Handle
    SMS_EVENT_HDL hUpdateEvent;

    // A special flag which prevents update timer to be started.
    // This particulary helpfull when removing CME events.
    BOOLEAN bBlockUpdateTimer;

} CME_OBJECT_STRUCT;

// Each registered content entry has an instance of this structure
// allocated and populated for it.
typedef struct cme_entry_struct
{
    // An entry handle which points to itself
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The CHANNEL object which contains the content this
    // event is pending for.
    CHANNEL_OBJECT hChannel;

    // State flags used to determine the current state of this
    // entry in our list.
    UN32 un32StateFlags;

    // The caller provided content information
    CONTENT_REGISTRATION_STRUCT sContent;

} CME_ENTRY_STRUCT;

// An iterator structure used to contain the callers call back
// function and argument when iterating the content list.
typedef struct cme_iterator_struct
{
    // Decoder object this CME belongs to
    DECODER_OBJECT hDecoder;

    // Callback information
    CME_ITERATOR_CALLBACK bIterator;
    void *pvIteratorArg;

    // A pointer to the current entry being iterated.
    OSAL_LINKED_LIST_ENTRY *phCurrentEntry;

} CME_ITERATOR_STRUCT;

// An iterator structure used to scan the channel cache for
// a particular CID.
typedef struct cme_cache_iterator_struct
{
    // CID to look for
    CID_OBJECT hId;

    // CME object handle
    CME_OBJECT hCME;

    CME_ENTRY_STRUCT *psEntry;

} CME_CACHE_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

/* Object Private Prototypes */

static N16 n16CompareContent (
    CME_ENTRY_STRUCT *psEntry1,
    CME_ENTRY_STRUCT *psEntry2
        );

static N16 n16Equal (
    CME_ENTRY_STRUCT *psEntry,
    CID_OBJECT hCID
        );

static CME_ENTRY_STRUCT *psCreateContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CID_OBJECT hCID,
    UN32 un32Options,
    CME_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vDestroyContentEntry (
    CME_ENTRY_STRUCT *psEntry
        );

static BOOLEAN bAddContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CME_ENTRY_STRUCT *psEntry
        );

static BOOLEAN bRemoveContentEntry (
    CME_OBJECT_STRUCT *psObj,
    CME_ENTRY_STRUCT *psEntry
        );

static BOOLEAN bIterator (
    CME_ENTRY_STRUCT *psEntry,
    CME_ITERATOR_STRUCT *psIterator
        );

static SMSAPI_RETURN_CODE_ENUM eAccessObjectFromDecoder(
    DECODER_OBJECT hDecoder,
    CME_OBJECT_STRUCT **ppsObj,
    CME_ENTRY_STRUCT **ppsEntry
        );

static SMSAPI_RETURN_CODE_ENUM eLock (
    DECODER_OBJECT hDecoder,
    CME_OBJECT_STRUCT **ppsObj
        );

static void vUnlock (
    DECODER_OBJECT hDecoder
        );

static void vTriggerEvent (
    CME_ENTRY_STRUCT *psEntry
        );

static void vUpdateContent (
    CME_OBJECT hCME,
    CHANNEL_OBJECT hChannel,
    CID_OBJECT hCID,
    BOOLEAN bNewContent
        );

#if SMS_DEBUG == 1
static BOOLEAN bPrintContent (
    CME_ENTRY_STRUCT *psEntry,
    FILE *psFile
        );
#endif /* SMS_DEBUG == 1 */

static void vGetHandles (
    CD_OBJECT hCDO,
    CME_OBJECT *phCME,
    CHANNEL_OBJECT *phChannel
        );

static SMSAPI_RETURN_CODE_ENUM eVerifyOptions (
    UN32 *pun32Options
        );

static BOOLEAN bChannelIterator (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

static void vUpdateSpecificContent (
    CME_OBJECT hCME,
    CHANNEL_OBJECT hChannel,
    CME_ENTRY_STRUCT *psEntry,
    BOOLEAN *pbContentRemoved
        );

static N16 n16Sort (
    void *psEvent1,
    void *psEvent2
        );

static N16 n16MapEvent (
    UN32 un32StateFlags
        );

void vPostUpdateEvent(
    OSAL_OBJECT_HDL hTimer, 
    void *pvArg
        );

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _CME_H_
