/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

#ifndef _CM_H_
#define _CM_H_

#include "standard.h"
#include "osal.h"

#include "sms_task.h"

#include "tag_cm_interface.h"
#include "tag_obj.h"
#include "cm.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define BUFF_SIZE 128

/* Object name prefix for objects */
#define CM_OBJECT_NAME "CM"

// These definitions are used to represent a BOOLEAN value in text form.
#define CM_TRUE_TEXT              "TRUE"
#define CM_FALSE_TEXT             "FALSE"

#define CM_FILENAME_NAME          "File Name"
#define CM_SECOND_FILENAME_NAME   "2nd File Name"
#define CM_INITIALIZER_FILENAME_NAME  "Initializer File Name"

#define SECONDARY_CONFIG_FILE_TOKEN ".new"

#define CHECKSUM_TAG_NAME "CMChecksum"
#define CHECKSUM_TAG_OPEN \
            TAG_BEGIN_DELIMITER\
            CHECKSUM_TAG_NAME\
            TAG_END_DELIMITER
#define CHECKSUM_TAG_CLOSE \
            TAG_BEGIN_DELIMITER \
            "/" \
            CHECKSUM_TAG_NAME\
            TAG_END_DELIMITER

#define INITIAL_SHARED_BUFF_SIZE  (512)
#define SHARED_BUFFER_SIZE_EXTRA_INCREMENT (128)

#define READ_BUFF_SIZE  (64 + 1) // Add one to allow null termination

#define CM_DEFAULT_TOP_TAG_NAME "Config"

// the amount of time we will hold off a commit
// each time a request to commit is made we will hold
// off the commit for this period of time
#define CM_COMMIT_TIMEOUT (200) // msec
// Successive calls to commit will hold off committing
// the file. If these calls keep coming at a fast enough
// rate, the commit timer may never have a chance to timeout
// and will keep being reset. This could lead to too much time
// passing between the first request to commit and the actual commit.
// A loss of power could then lead to too much data lost. The
// threshold is the maximum time allowed from the a request to commit
// until it is committed.
#define CM_COMMIT_THRESHOLD (1000) // msec

#define CM_CHECKSUM_BUF_LENGTH 6 // 5 for UN16 decimal digits, 1 for '0'

#define WRITE_LITERAL(file, literal) \
    fwrite((literal), sizeof(char), sizeof(literal)/sizeof(char)-1, (file))

/**************/
 /** TYPEDEFS **/
/**************/

typedef enum cm_chksum_status
{
    CM_CHKSUM_BLANK,
    CM_CHKSUM_PRESENT,
    CM_CHKSUM_NOT_PRESENT

} CM_CHKSUM_STATUS;

typedef enum cm_read_file_state
{
    CM_STATE_SEARCHING_FOR_BEGIN_DELIMITER,
    CM_STATE_SEARCHING_FOR_CLOSE_DELIMITER,
    CM_STATE_SEARCHING_FOR_END_DELIMITER,
    CM_STATE_READ_COMPLETE

} CM_READ_FILE_STATE;

typedef enum cm_config_source
{
    CM_CONFIG_SOURCE_NOT_APPLICABLE,
    CM_CONFIG_SOURCE_EXISTING_SMS_CFG,
    CM_CONFIG_SOURCE_NEW_MINIMAL_SMS_CFG,
    CM_CONFIG_SOURCE_CONFIG_INITIALIZER
} CM_CONFIG_SOURCE_ENUM;

typedef struct cm_process_struct
{
    char *pacStart;
    char *pacEnd;
    char *pacCurrentLocation;
    char *pacReadBufferEnd;
    TAG_OBJECT hCurrentTag;

} CM_PROCESS_STRUCT;


/* Private object elements */

typedef struct cm_object_struct
{
    // config file name
    char *pacConfigFileName;
    char *pacSecondaryConfigFileName;
    char *pacConfigInitializerFileName;

    // the top tag
    TAG_OBJECT hTopTag;

    char *pacCMReadBuffer;
    char *pacCMBuffer;
    size_t tCMBufferSize;
    char *pacCMBufferPtr;
    STRING_OBJECT hValueString;
    STRING_OBJECT hTrueString;
    STRING_OBJECT hFalseString;

    // flag to indicate that there are pending changes
    BOOLEAN bPendingChanges;

    // Removed Tag List
    OSAL_OBJECT_HDL hRemovedTagList;

    CM_READ_FILE_STATE eReadState;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // Services task for the CM
    SMS_TASK_HANDLE hServicesTask;

    // since the actual initialization is done in the CM task
    // now, instead of the calling task, we need to store
    // some a return code to indicate success or, it not,
    // the reason (bad config file, etc.)
    SMSAPI_RETURN_CODE_ENUM eInitResult;
    // likewise, we have to store an indication of what we're
    // using as the configuration file source.
    CM_CONFIG_SOURCE_ENUM eConfigSource;

    // handle to the timer used to hold off commits
    OSAL_OBJECT_HDL hTimerObj;
    // cumulative amount of time a commit has been held off
    UN32 un32TimeCommitHeldOff;
    // pre-allocated timeout event
    SMS_EVENT_HDL hTimeOutEvent;

} CM_OBJECT_STRUCT;

typedef struct cm_object * CM_OBJECT;
#define CM_INVALID_OBJECT (CM_OBJECT)NULL

  /************/
 /** MACROS **/
/************/
#define WHITESPACE(x) \
       (   ((x) == ' ')  || ((x) == '\r') \
        || ((x) == '\n') || ((x) == '\t') \
        || ((x) == '\0')  \
       )

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */


/* Object Private Prototypes */

static BOOLEAN bSetFilenames(
    CM_OBJECT_STRUCT *psObj,
    const char *pacCfgFileNameInput,
    const char *pacCfgFilePathInput,
    const char *pacCfgInitializerFileNameInput,
    const char *pacCfgInitializerFilePathInput
        );

static void vUnsetFilenames(
    CM_OBJECT_STRUCT *psObj
        );

static void vCreateDefaultTopTag(
    CM_OBJECT_STRUCT *psObj
        );

static SMSAPI_RETURN_CODE_ENUM eReadCfgFile(
    const char *pacConfigFileName,
    CM_OBJECT_STRUCT *psObj
        );

static SMSAPI_RETURN_CODE_ENUM eProcessChunk(
    CM_OBJECT_STRUCT *psObj,
    size_t tChunkSize
        );

static BOOLEAN bTrimWhitespace(
    char **ppacStart,
    char **ppacEnd
        );

static BOOLEAN bReadChecksum(
    FILE *psFile,
    char *pacBuffer,
    size_t tBufferSize,
    size_t *ptContentLength,
    UN16 *pun16Checksum,
    BOOLEAN *pbChecksumExists
        );

static BOOLEAN bComputeChecksum(
    FILE *psFile,
    char *pacBuffer,
    size_t tBufferSize,
    size_t tContentLength,
    UN16 *pun16Checksum
        );

static BOOLEAN bStampChecksum(
    FILE *psFile,
    UN16 un16Checksum
        );

static BOOLEAN bAppendBuffer(
    CM_OBJECT_STRUCT *psObj,
    char *pacStart,
    char *pacEnd
        );

static void vResetBuffer( void );

static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForBeginDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess
      );

static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForCloseDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT hCurrentTag,
    BOOLEAN *pbClose
      );

static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForEndDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT *phCurrentTag,
    BOOLEAN bClose
      );

static SMSAPI_RETURN_CODE_ENUM eTagEndDelimiterNotFound(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess
        );

static SMSAPI_RETURN_CODE_ENUM eTagEndDelimiterFound(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT *phCurrentTag,
    BOOLEAN bClose
        );

static void vInitializeObject(
    CM_OBJECT_STRUCT *psObj
        );

static void vEventHandler(
    CM_OBJECT_STRUCT *psObj,
    SMS_EVENT_HDL hEvent
        );

static void vUninitialize( void );

static void vTimerHandler(
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static SMSAPI_RETURN_CODE_ENUM eCommitChangesToFile( void );

static SMSAPI_RETURN_CODE_ENUM eCheckConfigFile (
    const char *pacFileName,
    CM_OBJECT_STRUCT *psObj,
    CM_CONFIG_SOURCE_ENUM eConfigSource
        );

  /***************/
 /** VARIABLES **/
/***************/

// locally global instance.
CM_OBJECT ghCM = CM_INVALID_OBJECT;

// CM Object task configuration for SMS
static const SMS_TASK_CONFIGURATION_STRUCT gsCMTaskConfiguration =
{
    /*.pacName = */"CM",
    /*.un32StackSize = */8192, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_LOW,
    /*.un32Options = */OSAL_TASK_OPTION_NONE,
    /*.un16ReportingInterval = */30, // seconds
    /*.un32EventQueueSize = */3, // events
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_NONE
};
  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _CM_H_

