/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CID_OBJ_H_
#define _CID_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "cid_obj.h"
#include "cid_string.h"
#include "cid_integer.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CID_OBJECT_NAME "CID"

/* CID Flags and Masks */
// Creator
#define CID_CREATOR                     ((UN8) 0x03 << 0)
#define CID_CREATOR_INVALID             ((UN8) 0x00 << 0)
#define CID_CREATOR_APPLICATION         ((UN8) 0x01 << 0)
#define CID_CREATOR_SMS                 ((UN8) 0x02 << 0)
#define CID_CREATOR_UNASSIGNED          ((UN8) 0x03 << 0)
// Initial value
#define CID_FLAGS_INITIAL (CID_CREATOR_INVALID)

// Minimum Text Field Sizes
// Note: This is only a minimum specification which is used when the
// object data is allocated to prevent frequent reallocations due to
// string expansion. It does not 'limit' the text size as it can be
// of arbitrary length. A minimum is specified to simply avoid
// frequent reallocations of memory for the artist, title and composer
// text.
#define CID_ARTIST_TEXT_MINIMUM_SIZE      (36)
#define CID_TITLE_TEXT_MINIMUM_SIZE       (36)
#define CID_ALBUM_TEXT_MINIMUM_SIZE       (36)
#define CID_COMPOSER_TEXT_MINIMUM_SIZE    (36)
#define CID_CONTENTINFO_TEXT_MINIMUM_SIZE (36)

#define CID_TYPE_VALUE_DELIMITER_STRING ","

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef enum cid_sub_pool_rearrangement_action_enum
{
    CID_SUB_POOL_REARRANGEMENT_ACTION_ADD,
    CID_SUB_POOL_REARRANGEMENT_ACTION_REUSE,
    CID_SUB_POOL_REARRANGEMENT_ACTION_RECYCLE
} CID_SUB_POOL_REARRANGEMENT_ACTION_ENUM;

typedef struct cid_equivalent_cid_struct
{
    BOOLEAN bUseCid;
    CID_OBJECT hCid;

    struct cid_equivalent_cid_struct *psNext;

} CID_EQUIVALENT_CID_STRUCT;

/*
 * CID Object Structure
 * Each CID consists of the following structure.
 */
typedef struct cid_object_struct
{
    // CID Flags
    UN8 un8Flags;

    // Entry handle for this CID in the CID pool
    OSAL_LINKED_LIST_ENTRY hEntry;

    // The Object Data (type depends on eType)
    const void *pvObjectData;

    // Interface to manipulate this CID based on type
    const CID_TYPE_INTERFACE_STRUCT *psIntf;

    CID_EQUIVALENT_CID_STRUCT *psEquivalent;

} CID_OBJECT_STRUCT;

/*
 * CID Object Item
 */
typedef struct cid_sub_pool_item_struct
{
    // Reference to the CID object wrapped by the item
    CID_OBJECT_STRUCT *psObj;
} CID_SUB_POOL_ITEM_STRUCT;

/*
 * CID Sub Pool Structure
 */
typedef struct cid_sub_pool_struct
{
    // Interface to manipulate this CID based on type
    const CID_TYPE_INTERFACE_STRUCT *psIntf;

    // List of CID_SUB_POOL_ITEM_STRUCT
    OSAL_OBJECT_HDL hPool;

#if DEBUG_OBJECT != 0
    // Statistics
    UN32 un32Available;
    UN32 un32Unavailable;
#endif
} CID_SUB_POOL_STRUCT;

/*
 *  CID Pool Structure
 *
 */
typedef struct cid_pool_struct
{
    // List of CID_SUB_POOL_STRUCT
    CID_SUB_POOL_STRUCT *apsSubPool[NUM_CID_TYPES];
} CID_POOL_STRUCT;

/*
    CID Control Structure

    This structure contains the CID pool and other objects
    which are required to maintain the CID pool and related objects.
*/

typedef struct cid_ctrl_struct
{
    // CID Type List
    const CID_TYPE_INTERFACE_STRUCT *apsTypes[NUM_CID_TYPES];

} CID_CTRL_STRUCT;

// this union is used to determine the longest cid type text string
typedef union cid_type_text
{
    const char acCID_PID_Text[sizeof(MACRO_TO_STRING(CID_PID))];
    const char acCID_AID_Text[sizeof(MACRO_TO_STRING(CID_AID))];
    const char acCID_ARTISTTEXT_Text[sizeof(MACRO_TO_STRING(CID_ARTIST_TEXT))];
    const char acCID_TITLETEXT_Text[sizeof(MACRO_TO_STRING(CID_TITLE_TEXT))];
    const char acCID_SXIARTISTID_Text[sizeof(MACRO_TO_STRING(CID_SXI_ARTIST_ID))];
    const char acCID_SXISONGID_Text[sizeof(MACRO_TO_STRING(CID_SXI_SONG_ID))];
    const char acCID_SXIMARKETID_Text[sizeof(MACRO_TO_STRING(CID_SXI_MARKET_ID))];
    const char acCID_SXISPORTSID_Text[sizeof(MACRO_TO_STRING(CID_SXI_SPORTS_ID))];
    const char acCID_SONGTAGID_Text[sizeof(MACRO_TO_STRING(CID_SONG_TAG_ID))];
    const char acCID_COMPOSERTEXT_Text[sizeof(MACRO_TO_STRING(CID_COMPOSER_TEXT))];
    const char acCID_ALBUMNAME_Text[sizeof(MACRO_TO_STRING(CID_ALBUM_NAME_TEXT))];
    const char acCID_CONTENTINFOTEXT_Text[sizeof(MACRO_TO_STRING(CID_CONTENTINFO_TEXT))];
    const char acCID_CIDLEAGUETEAMID_Text[sizeof(MACRO_TO_STRING(CID_LEAGUE_TEAM_ID))];
    const char acCID_INVALID_Text[sizeof(MACRO_TO_STRING(CID_INVALID))];

} CID_TYPE_TEXT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static void vDestroy (
    CID_OBJECT hCID
        );

static N16 n16Equal (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2
        );

static N16 n16Compare (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2
        );

static CID_OBJECT hDuplicate (
    CID_OBJECT hCID
        );

static BOOLEAN bSetEquivalent (
    CID_OBJECT hCID,
    CID_OBJECT hEquivalentCID
        );

static N32 n32FWrite (
    CID_OBJECT hCID,
    FILE *psFile
        );

static CID_OBJECT hFRead (
    FILE *psFile
        );

static N32 n32FPrintf (
    CID_OBJECT hCID,
    FILE *psFile
        );

/* Object Private Prototypes */

static CID_SUB_POOL_STRUCT *psSubPoolCreate (
    CID_POOL_STRUCT *psPool,
    CID_ENUM eType
        );

static void vSubPoolDestroy (
    CID_SUB_POOL_STRUCT *psSPool
        );

static void vSubPoolItemDestroy (
    CID_SUB_POOL_ITEM_STRUCT *psItem
        );

static BOOLEAN bSubPoolReArrange(
    CID_SUB_POOL_STRUCT *psSPool,
    CID_SUB_POOL_ITEM_STRUCT *psItem,
    CID_SUB_POOL_REARRANGEMENT_ACTION_ENUM eAction
        );

static N16 n16CompareObjects (
    CID_OBJECT hCID1,
    CID_OBJECT hCID2,
    BOOLEAN bBinary
        );

static BOOLEAN bMatchWithEquivalentCids (
    CID_OBJECT_STRUCT *psObj1,
    CID_OBJECT_STRUCT *psObj2
        );

static BOOLEAN bDuplicateEquivalentCids (
    CID_OBJECT_STRUCT *psSrcObj,
    CID_OBJECT_STRUCT *psDstObj
        );

static N32 n32FWriteEquivalentCids (
    CID_OBJECT_STRUCT *psObj,
    FILE *psFile
        );

static BOOLEAN bFReadEquivalentCids (
    CID_OBJECT_STRUCT *psObj,
    FILE *psFile
        );

static CID_OBJECT_STRUCT *psFindReusable (
    CID_POOL_STRUCT *psCidPool,
    CID_ENUM eType,
    CID_SUB_POOL_STRUCT **ppsSPool
        );

static CID_OBJECT_STRUCT *psCreateObject (
    CID_POOL_STRUCT *psCidPool,
    CID_ENUM eType,
    CID_SUB_POOL_STRUCT *psSPool
        );

static void vRecycle (
    CID_OBJECT_STRUCT *psObj
        );

static void vDestroyObject (
    CID_OBJECT_STRUCT *psObj
        );

static const char *pacTypeText(
    CID_ENUM eType,
    size_t *ptLength
        );

static CID_ENUM eTypeFromText (
    const char *pacText
        );

  /***************/
 /** VARIABLES **/
/***************/

// This pointer is a child object of SMS and is protected
// from multiple access using that mechanism.
CID_CTRL_STRUCT *gpsCidCtrl = (CID_CTRL_STRUCT *)NULL;

// Global structure defining the CID Interface for Artist Text
static const CID_TYPE_INTERFACE_STRUCT gsArtistTextIntf =
{
    // CID type
    /*.eType = */CID_ARTIST_TEXT,

    // Init/Unit methods
    /*.bInitialize = */NULL,
    /*.vUnInitialize = */NULL,

    // CID Input Data Parse Method
    /*.bParse = */NULL,

    // Object Data Methods
    /*.sObjData = */{

        // Minimum Expected Size (if known, otherwise 0)
        /*.tMinSize = */CID_ARTIST_TEXT_MINIMUM_SIZE,

        // Object Data Create/Destroy Methods
        /*.pvCreate = */CIDSTR_pvCreate,
        /*.vDestroy = */CIDSTR_vDestroy,

        // Object Data Manipulation Methods
        /*.pvDuplicate = */CIDSTR_pvDuplicate,
        /*.bModify = */CIDSTR_bModify,
        /*.bCopy = */CIDSTR_bCopy,
        /*.tSize = */CIDSTR_tSize,
        /*.n16Compare = */CIDSTR_n16Compare,

        // Object Data I/O Methods
        /*.n32FWrite = */CIDSTR_n32FWrite,
        /*.n32FWriteToMemory = */CIDSTR_n32FWriteToMemory,
        /*.pvRead = */CIDSTR_pvRead,
        /*.bReadFromMemory = */CIDSTR_bReadFromMemory,
        /*.n32FPrintf = */CIDSTR_n32FPrintf,
        /*.n32GetValue = */CIDSTR_n32GetValue
    }
};

// Global structure defining the CID Interface for Title Text
static const CID_TYPE_INTERFACE_STRUCT gsTitleTextIntf =
{
    // CID type
    /*.eType = */CID_TITLE_TEXT,

    // Init/Unit methods
    /*.bInitialize = */NULL,
    /*.vUnInitialize = */NULL,

    // CID Input Data Parse Method
    /*.bParse = */NULL,

    // Object Data Methods
    /*.sObjData = */{

        // Minimum Expected Size (if known, otherwise 0)
        /*.tMinSize = */CID_TITLE_TEXT_MINIMUM_SIZE,

        // Object Data Create/Destroy Methods
        /*.pvCreate = */CIDSTR_pvCreate,
        /*.vDestroy = */CIDSTR_vDestroy,

        // Object Data Manipulation Methods
        /*.pvDuplicate = */CIDSTR_pvDuplicate,
        /*.bModify = */CIDSTR_bModify,
        /*.bCopy = */CIDSTR_bCopy,
        /*.tSize = */CIDSTR_tSize,
        /*.n16Compare = */CIDSTR_n16Compare,

        // Object Data I/O Methods
        /*.n32FWrite = */CIDSTR_n32FWrite,
        /*.n32FWriteToMemory = */CIDSTR_n32FWriteToMemory,
        /*.pvRead = */CIDSTR_pvRead,
        /*.bReadFromMemory = */CIDSTR_bReadFromMemory,
        /*.n32FPrintf = */CIDSTR_n32FPrintf,
        /*.n32GetValue = */CIDSTR_n32GetValue
    }
};

// Global structure defining the CID Interface for Composer Text
static const CID_TYPE_INTERFACE_STRUCT gsComposerTextIntf =
{
    // CID type
    /*.eType = */CID_COMPOSER_TEXT,

    // Init/Unit methods
    /*.bInitialize = */NULL,
    /*.vUnInitialize = */NULL,

    // CID Input Data Parse Method
    /*.bParse = */NULL,

    // Object Data Methods
    /*.sObjData = */
    {
        // Minimum Expected Size (if known, otherwise 0)
        /*.tMinSize = */CID_COMPOSER_TEXT_MINIMUM_SIZE,

        // Object Data Create/Destroy Methods
        /*.pvCreate = */CIDSTR_pvCreate,
        /*.vDestroy = */CIDSTR_vDestroy,

        // Object Data Manipulation Methods
        /*.pvDuplicate = */CIDSTR_pvDuplicate,
        /*.bModify = */CIDSTR_bModify,
        /*.bCopy = */CIDSTR_bCopy,
        /*.tSize = */CIDSTR_tSize,
        /*.n16Compare = */CIDSTR_n16Compare,

        // Object Data I/O Methods
        /*.n32FWrite = */CIDSTR_n32FWrite,
        /*.n32FWriteToMemory = */CIDSTR_n32FWriteToMemory,
        /*.pvRead = */CIDSTR_pvRead,
        /*.bReadFromMemory = */CIDSTR_bReadFromMemory,
        /*.n32FPrintf = */CIDSTR_n32FPrintf,
        /*.n32GetValue = */CIDSTR_n32GetValue
    }
};

// Global structure defining the CID Interface for Title Text
static const CID_TYPE_INTERFACE_STRUCT gsAlbumNameIntf =
{
    // CID type
    /*.eType = */CID_ALBUM_NAME_TEXT,

    // Init/Unit methods
    /*.bInitialize = */NULL,
    /*.vUnInitialize = */NULL,

    // CID Input Data Parse Method
    /*.bParse = */NULL,

    // Object Data Methods
    /*.sObjData = */{

        // Minimum Expected Size (if known, otherwise 0)
        /*.tMinSize = */CID_ALBUM_TEXT_MINIMUM_SIZE,

        // Object Data Create/Destroy Methods
        /*.pvCreate = */CIDSTR_pvCreate,
        /*.vDestroy = */CIDSTR_vDestroy,

        // Object Data Manipulation Methods
        /*.pvDuplicate = */CIDSTR_pvDuplicate,
        /*.bModify = */CIDSTR_bModify,
        /*.bCopy = */CIDSTR_bCopy,
        /*.tSize = */CIDSTR_tSize,
        /*.n16Compare = */CIDSTR_n16Compare,

        // Object Data I/O Methods
        /*.n32FWrite = */CIDSTR_n32FWrite,
        /*.n32FWriteToMemory = */CIDSTR_n32FWriteToMemory,
        /*.pvRead = */CIDSTR_pvRead,
        /*.bReadFromMemory = */CIDSTR_bReadFromMemory,
        /*.n32FPrintf = */CIDSTR_n32FPrintf,
        /*.n32GetValue = */CIDSTR_n32GetValue
    }
};

// Global structure defining the CID Interface for ContentInfo Text
static const CID_TYPE_INTERFACE_STRUCT gsContentInfoTextIntf =
{
    // CID type
    /*.eType = */CID_CONTENTINFO_TEXT,

    // Init/Unit methods
    /*.bInitialize = */NULL,
    /*.vUnInitialize = */NULL,

    // CID Input Data Parse Method
    /*.bParse = */NULL,

    // Object Data Methods
    /*.sObjData = */{

        // Minimum Expected Size (if known, otherwise 0)
        /*.tMinSize = */CID_CONTENTINFO_TEXT_MINIMUM_SIZE,

        // Object Data Create/Destroy Methods
        /*.pvCreate = */CIDSTR_pvCreate,
        /*.vDestroy = */CIDSTR_vDestroy,

        // Object Data Manipulation Methods
        /*.pvDuplicate = */CIDSTR_pvDuplicate,
        /*.bModify = */CIDSTR_bModify,
        /*.bCopy = */CIDSTR_bCopy,
        /*.tSize = */CIDSTR_tSize,
        /*.n16Compare = */CIDSTR_n16Compare,

        // Object Data I/O Methods
        /*.n32FWrite = */CIDSTR_n32FWrite,
        /*.n32FWriteToMemory = */CIDSTR_n32FWriteToMemory,
        /*.pvRead = */CIDSTR_pvRead,
        /*.bReadFromMemory = */CIDSTR_bReadFromMemory,
        /*.n32FPrintf = */CIDSTR_n32FPrintf,
        /*.n32GetValue = */CIDSTR_n32GetValue
    }
};

// Global (re-usable) instance of an interface for this object
const CID_OBJECT_INTERFACE_STRUCT CID =
{
    /*.vDestroy = */vDestroy,
    /*.n16Equal = */n16Equal,
    /*.n16Compare = */n16Compare,
    /*.hDuplicate = */hDuplicate,
    /*.bSetEquivalent = */bSetEquivalent,
    /*.n32FWrite = */n32FWrite,
    /*.hFRead = */hFRead,
    /*.n32FPrintf = */n32FPrintf
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _CID_OBJ_H_
