/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CHANNEL_ART_OBJ_H_
#define _CHANNEL_ART_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "sms_api.h"
#include "sms_update.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"
#include "image_obj.h"
#include "channel_art_obj.h"

/* Object name prefix for objects */
#define CHANNEL_ART_OBJECT_NAME "Channel Art"

#define CHANNEL_ART_MAX_ASSOC_TYPES (CHANNEL_ART_IMAGE_ASSOC_TYPE_UNKNOWN)

#define CHANNEL_ART_IMAGE_FILE_READ_MODE "rb"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct channel_art_image_descriptor_struct
{
    // Handle to the object which actually created
    // the Image referenced in the handle below
    CHANNEL_ART_OBJECT hSourceObject;

    // The version of the link to the image
    BOOLEAN bAssocVerConfirmed;
    UN8 un8AssociationVersion;
    BOOLEAN bSecondaryAvailable;
    BOOLEAN bHighPriority;

    // The IMAGE object abstracts all image attributes
    IMAGE_OBJECT hImage;
    IMAGE_OBJECT hSecondaryImage;

} CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT;

typedef struct channel_art_table_entry_struct
{
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psStatic;
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psContent;

} CHANNEL_ART_TABLE_ENTRY_STRUCT;

typedef struct channel_art_object_struct
{
    // The art entries
    CHANNEL_ART_TABLE_ENTRY_STRUCT sLogoEntry;
    CHANNEL_ART_TABLE_ENTRY_STRUCT sBackgroundEntry;
    CHANNEL_ART_TABLE_ENTRY_STRUCT sAlbumEntry;

    // Path at which all images are located
    const char *pacFilePath;

    // A mask used to track which images
    // are available to the application
    // at any given time
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableMask;

    // The object's version
    CHANNEL_ART_VERSION tVersion;

} CHANNEL_ART_OBJECT_STRUCT;

typedef struct channel_art_assoc_type_iterator_struct
{
    UN8 un8NumTypesToReport;
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM aeTypesToReport[CHANNEL_ART_MAX_ASSOC_TYPES];

} CHANNEL_ART_ASSOC_TYPE_ITERATOR_STRUCT;

typedef struct channel_art_update_image_iterator_struct
{
    BOOLEAN bUpdated;
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psUpdatedAttribs;

} CHANNEL_ART_UPDATE_IMAGE_ITERATOR_STRUCT;

typedef struct channel_art_image_data_struct
{
    UN16 un16ImageId;
    UN16 un16ImageVer;
    CHANNEL_ART_IMAGETYPE_ENUM eImageType;
    size_t tFileSizeInBytes;
    IMAGE_BACKGROUND_STRUCT sBackground;
    STRING_OBJECT hCaption;
} CHANNEL_ART_IMAGE_DATA_STRUCT;

typedef BOOLEAN (*ART_DESC_ITERATOR) (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc,
    BOOLEAN bStatic,
    void *pvCallbackArg
        );

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

/* Object Private Prototypes */

static CHANNEL_ART_OBJECT hCreateChannelArtObject(
    SMS_OBJECT hOwner,
    const char *pacFilePath
        );

static BOOLEAN bGetImageAttributes (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eImageAssocType,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageCopied
        );

static void vUpdateEventForImageType (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bAdd,
    BOOLEAN bUpdateAvailable
        );

static BOOLEAN bCopyArtEntry (
    CHANNEL_ART_OBJECT_STRUCT *psSrcObj,
    CHANNEL_ART_OBJECT_STRUCT *psDstObj,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN *pbImageCopied
        );

static BOOLEAN bClearArtEntry (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bContentArtOnly
        );

static BOOLEAN bSetChannelImage (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    BOOLEAN *pbImageUpdated
        );

static BOOLEAN bSetAlbumImage (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    STRING_OBJECT *phCaption,
    BOOLEAN *pbImageUpdated
        );

static CHANNEL_ART_TABLE_ENTRY_STRUCT *psFindEntry (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN *pbSecondaryImageRequested
        );

static CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psFindImageDescByTypeAndId (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN8 un8ImageType
        );

static CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psFindImageDesc (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    BOOLEAN bContentAssociation,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    BOOLEAN bCreate
        );

static void vIterateMatchingDescriptors (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN8 un8ImageType,
    ART_DESC_ITERATOR bIterator,
    void *pvArg
        );

static BOOLEAN bReportAssocTypeIterator (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc,
    BOOLEAN bStatic,
    void *pvArg
        );

static BOOLEAN bUpdateImageIterator (
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGE_DESCRIPTOR_STRUCT *psDesc,
    BOOLEAN bStatic,
    void *pvArg
        );

static IMAGE_OBJECT hCreateImage(
    CHANNEL_ART_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttribs,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    CHANNEL_ART_BACKGROUND_OPTION_MASK tBackgroundOptions
        );

static BOOLEAN bImageInitSpecificData(
    IMAGE_OBJECT hImage,
    void *pvSpecificData,
    void *pvArg
        );

static void vImageUninitSpecificData (
    IMAGE_OBJECT hImage,
    void *pvSpecificData
        );

static BOOLEAN bImageCalculateFileNameLen (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    size_t *ptLength
        );

static BOOLEAN bImageCreateImageFileName (
    IMAGE_OBJECT hImage,
    const char *pacFilePath,
    void *pvSpecificData,
    char *pacBuffer,
    size_t tBufferSize
        );

static SMSAPI_RETURN_CODE_ENUM eImageProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    void *pvSpecificData,
    IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue
        );

static SMSAPI_RETURN_CODE_ENUM eImageUpdateProperty (
    IMAGE_OBJECT hImage,
    IMAGE_PROPERTY_INTERNAL_ENUM eProperty,
    void *pvSpecificData,
    const IMAGE_PROPERTY_INTERNAL_VALUE_STRUCT *psValue,
    BOOLEAN *pbUpdated
        );

static BOOLEAN bImageDetermineFileSize (
    IMAGE_OBJECT hImage,
    void *pvSpecificData
        );

  /***************/
 /** VARIABLES **/
/***************/

// IMAGE object specific interface implementation
static const IMAGE_INTF_STRUCT GsChannelArtImageIntf =
{
    /*.tSpecificDataSize = */ sizeof(CHANNEL_ART_IMAGE_DATA_STRUCT),
    /*.bInitSpecificData = */ bImageInitSpecificData,
    /*.bCreationPostStep = */ bImageDetermineFileSize,
    /*.vUninitSpecificData = */ vImageUninitSpecificData,
    /*.bCalculateImageFileNameLen = */ bImageCalculateFileNameLen,
    /*.bCreateImageFileName = */ bImageCreateImageFileName,
    /*.eProperty = */ eImageProperty,
    /*.eUpdateProperty = */ eImageUpdateProperty
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _CHANNEL_ART_OBJ_H_
