/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CCACHE_H_
#define _CCACHE_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "sms_update.h"
#include "sms_api.h"
#include "decoder_obj.h"
#include "module_obj.h"

#include "dataservice_mgr_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CCACHE_OBJECT_NAME "CCACHE"

/* ACO File Name */
#define CCACHE_OBJECT_ACO_FILE_NAME "aco.bin"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

// this ccache channel struct contains ccache items that are channel related
typedef struct ccache_channel_struct
{
    // Channel list (ordered by SID)
    OSAL_OBJECT_HDL hChannelListByServiceId;

    // CCache Channel List (ordered by navigation style)
    // Currently this is by channel id (increasing)
    OSAL_OBJECT_HDL hChannelListByChannelId;

    // Current channel list used for navigation
    OSAL_OBJECT_HDL hCurrentNavigationChannelList;

    // "All Channels" notification list
    OSAL_OBJECT_HDL hAllChanNotificationList;

    // The Appliation's registerd channel notification callback and the current
    // channel being checked for notification status
    OSAL_LINKED_LIST_ENTRY hCurChanEntry;
    CHANNEL_OBJECT_EVENT_CALLBACK vAppNotifierCallback;
    void *pvAppNotifierCallbackArg;

} CCACHE_CHANNEL_STRUCT;

// this ccache category struct contains ccache items that are category related
typedef struct ccache_category_struct
{
    // Category list
    OSAL_OBJECT_HDL hCategoryList;

    // Number of invalid or 'ghost' categories in this list
    N16 n16NumInvalidCategories;

    // Number of Valid Derived Categories
    N16 n16NumDerivedCategories;

    // "All Categories" notification list
    OSAL_OBJECT_HDL hAllCatNotificationList;

    // The Appliation's registerd category notification callback and the current
    // category being checked for notification status
    OSAL_LINKED_LIST_ENTRY hCurCatEntry;
    CATEGORY_OBJECT_EVENT_CALLBACK vAppNotifierCallback;
    void *pvAppNotifierCallbackArg;

} CCACHE_CATEGORY_STRUCT;

// the ccache object structure
typedef struct ccache_object_struct
{
    // Decoder object this channel cache belongs to
    DECODER_OBJECT hDecoder;

    // A handle to this channel cache's CID pool
    CID_POOL hCidPool;

    // Channel Struct
    CCACHE_CHANNEL_STRUCT sChannel;

    // Category Struct
    CCACHE_CATEGORY_STRUCT sCategory;

    // Channel Art Service Handle
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService;

    // CCache Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Dummy category, used for searching
    CATEGORY_OBJECT hDummyCategory;

    // Dummy channel, used for searching
    CHANNEL_OBJECT *phDummyChannel;

    // Dummy channel, used to substitute unsubscribed information
    CHANNEL_OBJECT hUnsubscribedChannel;

    // Dummy channel, used to substitute subscription alert information
    CHANNEL_OBJECT hSubscriptionAlertChannel;

    // CCache availability flag. FALSE is being updated
    BOOLEAN bAvailable;

} CCACHE_OBJECT_STRUCT;

// A ccache channel iterator shim structure
typedef struct ccache_channel_iterator_handler_struct
{
    CCACHE_CHANNEL_ITERATOR_HANDLER bIterator;
    void *pvArg;

} CCACHE_CHANNEL_ITERATOR_HANDLER_STRUCT;

// A ccache category iterator shim structure
typedef struct ccache_category_iterator_handler_struct
{
    CCACHE_CATEGORY_ITERATOR_HANDLER bIterator;
    void *pvArg;

} CCACHE_CATEGORY_ITERATOR_HANDLER_STRUCT;

// Structure which aids in the determination
// of a category Id by an offset or an offset
// by a category Id
typedef struct ccache_derived_category_offset_struct
{
    // Targets to located
    CATEGORY_ID tTargetCategory;
    N16 n16TargetOffset;

    // Offset of the current derived category
    N16 n16DerivedOffset;

    // Search mode flag
    BOOLEAN bSearchByOffset;

    // Success flag
    BOOLEAN bFound;

} CCACHE_DERIVED_CATEGORY_OFFSET_STRUCT;

typedef struct ccache_all_chan_notification_struct
{
    CHANNEL_EVENT_MASK tEventRequestMask;
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback;
    void *pvEventCallbackArg;
    BOOLEAN bImmediateNotify;

} CCACHE_ALL_CHAN_NOTIFICATION_STRUCT;

typedef struct ccache_all_cat_notification_struct
{
    CATEGORY_EVENT_MASK tEventRequestMask;
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback;
    void *pvEventCallbackArg;
    BOOLEAN bImmediateNotify;

} CCACHE_ALL_CAT_NOTIFICATION_STRUCT;

typedef struct ccache_commit_struct
{
    BOOLEAN bSuccess;
    UN32 un32Channels;
    UN8 *pun8Block;

} CCACHE_COMMIT_STRUCT;

// ACO file storage structure
typedef struct ccache_aco_storage_struct
{
    SERVICE_ID tSID;
    CHANNEL_ACO tACO;

} CCACHE_ACO_STORAGE_STRUCT;

typedef struct ccache_chan_add_notifier_struct
{
    CHANNEL_OBJECT hChannel;
    BOOLEAN bImmediateNotify;
} CCACHE_CHAN_ADD_NOTIFIER_STRUCT;
typedef struct ccache_art_update_struct
{
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService;
    CCACHE_UPDATE_TYPE_ENUM eUpdateType;
} CCACHE_ART_UPDATE_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

/* Object Private Prototypes */

static CHANNEL_OBJECT hCreateChannel (
    CCACHE_OBJECT_STRUCT *psObj,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bNotify
        );

static CATEGORY_OBJECT hCreateBroadcastCategory (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId
        );

static CATEGORY_OBJECT hCreateDerivedCategory (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_TYPE_ENUM eCategoryType,
    CATEGORY_ID tCategoryId,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        );

static BOOLEAN bUpdateChannelLocal (
    CCACHE_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        );

static CATEGORY_OBJECT hFindCategoryLocal (
    CCACHE_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId
        );

static BOOLEAN bAddNotifierToChannelIterator (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify,
    CCACHE_CHAN_ADD_NOTIFIER_STRUCT *psData
        );

static BOOLEAN bRemoveNotifierFromChannelIterator (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify,
    CHANNEL_OBJECT hChannel
        );

static BOOLEAN bUpdateChannelNotificationsIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bAddNotifierToCategoryIterator (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify,
    CATEGORY_OBJECT hCategory
        );

static BOOLEAN bRemoveNotifierFromCategoryIterator (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify,
    CATEGORY_OBJECT hCategory
        );

static BOOLEAN bUpdateCategoryNotificationsIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bReleaseChannelIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bChannelIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bCategoryIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bRemoveCategoryNotifierIterator(
    void *pvData,
    void *pvArg
        );

static BOOLEAN bRemoveChannelNotifierIterator(
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdateAllChannelArt (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bSetChannelEvent (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdateAllCategoryArt (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdateChannelEpgData (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bDerivedCategoryOffsetIterator (
    void *pvData,
    void *pvArg
        );

static void vDestroyChannel (
    void *pvData
        );

static void vDestroyCategory (
    void *pvData
        );

static void vDestroyChannelNotifier (
    void *pvData
        );

static void vDestroyCategoryNotifier (
    void *pvData
        );

static BOOLEAN bIsAvailable (
    CCACHE_OBJECT hCCache
        );

static void vRemapFSM (
    CCACHE_OBJECT_STRUCT *psObj,
    SMS_EVENT_CCACHE_REMAP_STRUCT const *psInRemap
        );

static void vPrune (
    CCACHE_OBJECT_STRUCT *psObj
        );

static N16 n16CompareAllChannelNotifers (
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify1,
    CCACHE_ALL_CHAN_NOTIFICATION_STRUCT *psNotify2
        );

static N16 n16CompareAllCategoryNotifers (
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify1,
    CCACHE_ALL_CAT_NOTIFICATION_STRUCT *psNotify2
        );

static N16 n16CountNumInvalidCategories(
    CCACHE_OBJECT_STRUCT *psObj
        );

static CHANNEL_OBJECT hChannelFromIds (
    CCACHE_OBJECT_STRUCT *psObj,
    OSAL_LINKED_LIST_ENTRY *phEntry,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        );

static void vStoreACOData(
    CCACHE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bStoreACODataIterator ( 
    void *pvData, 
    void *pvArg 
        );

static void vRestoreACOData(
    CCACHE_OBJECT_STRUCT *psObj
        );

static const char *pacGetACOPath( 
    CCACHE_OBJECT_STRUCT *psObj 
        );

static BOOLEAN bCalculateISO3309CRC32(
    OSAL_BUFFER_HDL hBuffer, 
    OSAL_CRC_RESULT *ptCRC
        );

  /***************/
 /** VARIABLES **/
/***************/

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _CCACHE_H_
