/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _AGW_MGR_OBJ_H_
#define _AGW_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "wsi_platform_port.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// This define is used to enable/disable tiles timeout
// support for the NOWRad product.
#ifndef AGW_NOWRAD_TIMEOUT_SUPPORT
#define AGW_NOWRAD_TIMEOUT_SUPPORT (1)
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define AGW_MGR_OBJECT_NAME "GFxWeatherService"

// Constant which defines the size of the buffer used by
// AGW service for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define AGW_SHARED_BUFFER_LEN ( 100 )

/* Max message sizes (from SX-9845-0035)
*   Section 3.3 says:
*       The maximum size of any Access Unit is 100K bytes.
*
*   In addition, let's make sure we have enough memory for
*   a few queued messages -- let's say 2.
*/

// Defines size of the block in the AGW PARSER block pool
#define AGW_BLOCK_POOL_BUF_SIZE (1024)
// Defines number of blocks allocated for AGW PARSER
#define AGW_BLOCK_POOL_BUF_NUM (100 * 2) // As per the comment above

#define AGW_BUFFER_BYTESIZE \
    (AGW_BLOCK_POOL_BUF_SIZE * AGW_BLOCK_POOL_BUF_NUM)

#define AGW_SUPPORTED_OPTIONS ( \
    (DATASERVICE_OPTIONS_MASK) \
    DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT \
    )

// Default output image format for AGW_TILE's images
#ifndef NO_SUPPORT_PNG
#define AGW_TILE_OUTPUT_IMAGE_FORMAT_DEFAULT (IMAGE_FORMAT_PNG)
#else
// Choose the BMP since in case of PNG support absence the service start
// w/ no explicit option to set the format will fail.
#define AGW_TILE_OUTPUT_IMAGE_FORMAT_DEFAULT (IMAGE_FORMAT_BMP)
#endif

// This constant is used to restrict number of statistics items are
// stores in the same time
#define AGW_STATISTICS_ITEMS_MAX (15)

// Defines supported file extensions
#define AGW_FILE_EXTENSION_RASTER_BMP ".bmp"
#define AGW_FILE_EXTENSION_RASTER_PNG ".png"
#define AGW_FILE_EXTENSION_RASTER_RAW ".raw"
#define AGW_FILE_EXTENSION_DEFRAMED   ".deframed"

// These constants are used to create filenames via snprintf
#define AGW_FILE_NAME_ID_LABEL "P%03d_%02u%02u%02u%02u_%05d_%06d_%05d_%06d"
#define AGW_FILE_NAME_ID_LABEL_LEN (39)

#define AGW_RASTER_FILE_NAME_ID_LABEL "%08x"
#define AGW_RASTER_FILE_NAME_ID_LABEL_LEN (8)

#define AGW_SERVICE_FOLDER "agw"
#define AGW_RASTERS_FOLDER_ATTRS ((UN8)(OSAL_FILE_ATTR_DIRECTORY | OSAL_FILE_ATTR_WRITE))

#define AGW_FILE_NAME_STORM_TRACK_CACHED "storm_track.cch"
#define AGW_STORM_TRACK_CACHE_TIMEOUT (60 * 60 * 24)  //24 hours for storm track product cache timeout

// 15 minutes to decide about tile expiration
#define AGW_NOWRAD_TILE_TIMEOUT_SEC (15 * 60)

// AGW data evaluate timeout (1 minute)
#define AGW_TIMER_PERIOD_SEC ((UN32)(1 * 60))

// Defines day threshold in seconds (12 hours)
#define AGW_NOWRAD_THRESHOLD_FOR_FAKE_STATISTICS (12 * 60 * 60)

// Default/max number of supported NOWRad object in one set
#define AGW_NOWRAD_TILE_SET_SIZE_MAX (32)

#define AGW_PARSER_OBJECT_NAME AGW_MGR_OBJECT_NAME"_PARSER"

// Defines Graphical Weather DSI/DMI
#define AGW_DSI                 (DSI)(442)
#define AGW_PVN                 (1)
#define AGW_CARID               (0)
#define AGW_CRC_INVERSION_VALUE (0xFFFFFFFFL)

/* Location coordinates fractional bits */
#define AGW_COORD_FRACTIONAL_BITS (16)

// Constant defines the minimum device location
// delta which weather wants to know about
#define AGW_DEVICE_DISTANCE_THRESHOLD (10)

/* AGW Data Service Protocol related constants */

/** Constants for parsing the high-band headers **/

/** Product header in Access Unit format **/
#define AGW_AU_PVN_BITLEN (4)
#define AGW_AU_PVN_OFFSET (0)

#define AGW_AU_CARID_BITLEN (3)
#define AGW_AU_CARID_OFFSET (AGW_AU_PVN_BITLEN)

#define AGW_AU_HDR_RESERVER_BITLEN (1)

#define AGW_WSI_PRODUCT_BYTEOFFSET (1)
#define AGW_WSI_PRODUCT_BITOFFSET (AGW_WSI_PRODUCT_BYTEOFFSET * 8)

#define AGW_AU_PRODUCT_TYPE_BITLEN (1)
#define AGW_AU_PRODUCT_TYPE_OFFSET (0)

#define AGW_AU_PRODUCT_ID_BITLEN (9)
#define AGW_AU_PRODUCT_ID_OFFSET (AGW_AU_PRODUCT_TYPE_OFFSET + \
                                  AGW_AU_PRODUCT_TYPE_BITLEN)

#define AGW_AU_ISSUE_MONTH_BITLEN (4)
#define AGW_AU_ISSUE_MONTH_OFFSET (AGW_AU_PRODUCT_ID_OFFSET + \
                                   AGW_AU_PRODUCT_ID_BITLEN )

#define AGW_AU_ISSUE_DAY_BITLEN (5)
#define AGW_AU_ISSUE_DAY_OFFSET (AGW_AU_ISSUE_MONTH_OFFSET + \
                                 AGW_AU_ISSUE_MONTH_BITLEN )

#define AGW_AU_ISSUE_HOUR_BITLEN (5)
#define AGW_AU_ISSUE_HOUR_OFFSET (AGW_AU_ISSUE_DAY_OFFSET + \
                                  AGW_AU_ISSUE_DAY_BITLEN )

#define AGW_AU_ISSUE_MINUTE_BITLEN (6)
#define AGW_AU_ISSUE_MINUTE_OFFSET (AGW_AU_ISSUE_HOUR_OFFSET + \
                                    AGW_AU_ISSUE_HOUR_BITLEN )

#define AGW_AU_VALID_MONTH_BITLEN (4)
#define AGW_AU_VALID_MONTH_OFFSET (AGW_AU_ISSUE_MINUTE_OFFSET + \
                                   AGW_AU_ISSUE_MINUTE_BITLEN )

#define AGW_AU_VALID_DAY_BITLEN (5)
#define AGW_AU_VALID_DAY_OFFSET (AGW_AU_VALID_MONTH_OFFSET + \
                                 AGW_AU_VALID_MONTH_BITLEN )

#define AGW_AU_VALID_HOUR_BITLEN (5)
#define AGW_AU_VALID_HOUR_OFFSET (AGW_AU_VALID_DAY_OFFSET + \
                                  AGW_AU_VALID_DAY_BITLEN )

#define AGW_AU_VALID_MINUTE_BITLEN (6)
#define AGW_AU_VALID_MINUTE_OFFSET (AGW_AU_VALID_HOUR_OFFSET + \
                                    AGW_AU_VALID_HOUR_BITLEN )

#define AGW_AU_MAX_LAT_BITLEN (15)
#define AGW_AU_MAX_LAT_OFFSET (AGW_AU_VALID_MINUTE_OFFSET + \
                               AGW_AU_VALID_MINUTE_BITLEN )

#define AGW_AU_MIN_LON_BITLEN (16)
#define AGW_AU_MIN_LON_OFFSET (AGW_AU_MAX_LAT_OFFSET + \
                               AGW_AU_MAX_LAT_BITLEN )

#define AGW_AU_MIN_LAT_BITLEN (15)
#define AGW_AU_MIN_LAT_OFFSET (AGW_AU_MIN_LON_OFFSET + \
                               AGW_AU_MIN_LON_BITLEN )

#define AGW_AU_MAX_LON_BITLEN (16)
#define AGW_AU_MAX_LON_OFFSET (AGW_AU_MIN_LAT_OFFSET + \
                               AGW_AU_MIN_LAT_BITLEN )

#define AGW_AU_HEADER_BYTELEN (14)
#define AGW_AU_HEADER_BITLEN (AGW_AU_HEADER_BYTELEN * 8)

/** Data Product Unit Header (Sec 4.3.1) **/
#define AGW_DPU_PRODUCT_ID_BYTELEN (2) /* LOHI - Little Endian */
#define AGW_DPU_INSTANCE_ID_BYTELEN (2)
#define AGW_DPU_SEQ_NUMBER_BYTELEN (2)
#define AGW_DPU_LAST_DPU_BYTELEN (1)

/** Product Headers ( Sec 5.2) **/
/*** Generic Product Header (Sec 5.2.1) ***/
#define AGW_GHEADER_TYPE_BYTELEN (1)
#define AGW_GHEADER_TYPE_BYTEOFFSET (0)

#define AGW_GHEADER_ID_BYTELEN (2)
#define AGW_GHEADER_ID_BYTEOFFSET (AGW_GHEADER_TYPE_BYTELEN)

#define AGW_GHEADER_ISSUE_MONTH_BYTELEN (1)
#define AGW_GHEADER_ISSUE_DAY_BYTELEN (1)
#define AGW_GHEADER_ISSUE_HOUR_BYTELEN (1)
#define AGW_GHEADER_ISSUE_MINUTE_BYTELEN (1)
#define AGW_GHEADER_VALID_MONTH_BYTELEN (1)
#define AGW_GHEADER_VALID_DAY_BYTELEN (1)
#define AGW_GHEADER_VALID_HOUR_BYTELEN (1)
#define AGW_GHEADER_VALID_MINUTE_BYTELEN (1)
#define AGW_GHEADER_MAX_LAT_BYTELEN (2)
#define AGW_GHEADER_MAX_LON_BYTELEN (2)
#define AGW_GHEADER_MIN_LAT_BYTELEN (2)
#define AGW_GHEADER_MIN_LON_BYTELEN (2)
#define AGW_GHEADER_SIZE (AGW_GHEADER_TYPE_BYTELEN + \
        AGW_GHEADER_ID_BYTELEN + \
        AGW_GHEADER_ISSUE_MONTH_BYTELEN + \
        AGW_GHEADER_ISSUE_DAY_BYTELEN + \
        AGW_GHEADER_ISSUE_HOUR_BYTELEN + \
        AGW_GHEADER_ISSUE_MINUTE_BYTELEN + \
        AGW_GHEADER_VALID_MONTH_BYTELEN + \
        AGW_GHEADER_VALID_DAY_BYTELEN + \
        AGW_GHEADER_VALID_HOUR_BYTELEN + \
        AGW_GHEADER_VALID_MINUTE_BYTELEN + \
        AGW_GHEADER_MAX_LAT_BYTELEN + \
        AGW_GHEADER_MAX_LON_BYTELEN + \
        AGW_GHEADER_MIN_LAT_BYTELEN + \
        AGW_GHEADER_MIN_LON_BYTELEN)

/*** Raster Product Header (Sec 5.2.2) ***/
#define AGW_RHEADER_NUMBER_ROWS_BYTELEN (2)
#define AGW_RHEADER_NUMBER_COLUMNS_BYTELEN (2)
#define AGW_RHEADER_PIXEL_DEPTH_BYTELEN (1)
#define AGW_RHEADER_NUMBER_PLANES_BYTELEN (1)
#define AGW_RHEADER_PRECISION_MULTIPLIER_BYTELEN (1)
#define AGW_RHEADER_OFFSET_MULTIPLIER_BYTELEN (2)
#define AGW_RHEADER_SIZE (AGW_GHEADER_SIZE + \
        AGW_RHEADER_NUMBER_ROWS_BYTELEN + \
        AGW_RHEADER_NUMBER_COLUMNS_BYTELEN + \
        AGW_RHEADER_PIXEL_DEPTH_BYTELEN + \
        AGW_RHEADER_NUMBER_PLANES_BYTELEN)
#define AGW_RHEADER_EXTRA_SIZE (AGW_RHEADER_SIZE - AGW_GHEADER_SIZE)
#define AGW_RHEADER_PIXEL_DEPTH_VALUE (8)

/** WSI NOWrad (Sec 5.3) **/

/** Storm Attributes (Sec 5.4) */
#define AGW_STORM_ATTR_COUNT_BYTELEN (2u)
#define AGW_STORM_ATTR_COUNT_BITLEN (AGW_STORM_ATTR_COUNT_BYTELEN * 8u)
#define AGW_STORM_ATTR_OBS_HOUR_BYTELEN (1u)
#define AGW_STORM_ATTR_OBS_MINUTE_BYTELEN (1u)
#define AGW_STORM_ATTR_LON_BYTELEN (4u)
#define AGW_STORM_ATTR_LON_BITLEN (AGW_STORM_ATTR_LON_BYTELEN * 8u)
#define AGW_STORM_ATTR_LAT_BYTELEN (4u)
#define AGW_STORM_ATTR_LAT_BITLEN (AGW_STORM_ATTR_LAT_BYTELEN * 8u)
#define AGW_STORM_ATTR_ECHO_BYTELEN (2u)
#define AGW_STORM_ATTR_ECHO_BITLEN (AGW_STORM_ATTR_ECHO_BYTELEN * 8u)
#define AGW_STORM_ATTR_DIR_BYTELEN (2u)
#define AGW_STORM_ATTR_DIR_BITLEN (AGW_STORM_ATTR_DIR_BYTELEN * 8u)
#define AGW_STORM_ATTR_SPEED_BYTELEN (2u)
#define AGW_STORM_ATTR_SPEED_BITLEN (AGW_STORM_ATTR_SPEED_BYTELEN * 8u)
#define AGW_STORM_ATTR_FLAG_BYTELEN (1u)
#define AGW_STORM_ATTR_UNIQUE_ID_BYTELEN (8u)
#define AGW_STORM_ATTR_BYTESIZE \
                            (AGW_STORM_ATTR_OBS_HOUR_BYTELEN + \
                             AGW_STORM_ATTR_OBS_MINUTE_BYTELEN + \
                             AGW_STORM_ATTR_LON_BYTELEN + \
                             AGW_STORM_ATTR_LAT_BYTELEN + \
                             AGW_STORM_ATTR_ECHO_BYTELEN + \
                             AGW_STORM_ATTR_DIR_BYTELEN + \
                             AGW_STORM_ATTR_SPEED_BYTELEN + \
                             AGW_STORM_ATTR_FLAG_BYTELEN + \
                             AGW_STORM_ATTR_UNIQUE_ID_BYTELEN )

#define AGW_STORM_ATTR_FLAG_HAIL_PROBABLE (0x01)
#define AGW_STORM_ATTR_FLAG_HAIL_POSITIVE (0x02)
#define AGW_STORM_ATTR_FLAG_MESOCYCLONIC  (0x04)
#define AGW_STORM_ATTR_FLAG_TORNADIC      (0x08)

#define AGW_STORM_ATTR_DIR_NA              (400)
#define AGW_STORM_ATTR_SPEED_NA            (255)
#define AGW_STORM_ATTR_ECHO_POW10           (-2)


/** Product Statistics (Sec 5.6) **/
#define AGW_PRODUCT_STAT_YEAR_BYTELEN (2)
#define AGW_PRODUCT_STAT_NUMBER_PID_BYTELEN (2)
#define AGW_PRODUCT_STAT_EXPECTED_NUMBER_TILES_BYTELEN (1)
#define AGW_PRODUCT_STAT_EXPECTED_TILES_SENT_BYTELEN (1)

/** Shape Data Products (Sec 5.7) **/
#define AGW_SHAPE_TYPE_COUNT_BYTELEN (2)
#define AGW_SHAPE_TYPE_COUNT_BITLEN (AGW_SHAPE_TYPE_COUNT_BYTELEN * 8)
#define AGW_SHAPE_COUNT_BYTELEN (2)
#define AGW_SHAPE_COUNT_BITLEN (AGW_SHAPE_COUNT_BYTELEN * 8)
#define AGW_SHAPE_TYPE_BYTELEN (4)
#define AGW_SHAPE_TYPE_BITLEN (AGW_SHAPE_TYPE_BYTELEN * 8)
#define AGW_SHAPE_ID_BYTELEN (2)
#define AGW_SHAPE_ID_BITLEN (AGW_SHAPE_ID_BYTELEN * 8)
#define AGW_SHAPE_SIZE_BYTELEN (4)
#define AGW_SHAPE_SIZE_BITLEN (AGW_SHAPE_SIZE_BYTELEN * 8)
#define AGW_SHAPE_HEADER_BYTESIZE (AGW_SHAPE_COUNT_BYTELEN + \
                                   AGW_SHAPE_TYPE_BYTELEN + \
                                   AGW_SHAPE_ID_BYTELEN + \
                                   AGW_SHAPE_SIZE_BYTELEN)

#define AGW_SHAPE_GEO_LAT_BYTELEN (2u)
#define AGW_SHAPE_GEO_LAT_BITLEN (AGW_SHAPE_GEO_LAT_BYTELEN * 8u)
#define AGW_SHAPE_GEO_LON_BYTELEN (2u)
#define AGW_SHAPE_GEO_LON_BITLEN (AGW_SHAPE_GEO_LON_BYTELEN * 8u)
#define AGW_SHAPE_GEO_POINT_BYTESIZE (AGW_SHAPE_GEO_LAT_BYTELEN + \
                                      AGW_SHAPE_GEO_LON_BYTELEN )

#define AGW_SHAPE_FRONT_TYPE_BYTELEN (1)
#define AGW_SHAPE_FRONT_NO_OF_POINT_BYTELEN (2)
#define AGW_SHAPE_FRONT_HEADER_BYTESIZE (AGW_SHAPE_FRONT_TYPE_BYTELEN + \
                                         AGW_SHAPE_FRONT_NO_OF_POINT_BYTELEN )

#define AGW_SHAPE_ISOBAR_PRESSURE_BYTELEN (2)
#define AGW_SHAPE_ISOBAR_NO_OF_POINT_BYTELEN (2)
#define AGW_SHAPE_ISOBAR_HEADER_BYTESIZE (AGW_SHAPE_ISOBAR_PRESSURE_BYTELEN + \
                                          AGW_SHAPE_ISOBAR_NO_OF_POINT_BYTELEN )

#define AGW_SHAPE_PRESSURE_CENTER_TYPE_BYTELEN (1)
#define AGW_SHAPE_PRESSURE_CENTER_PRESSURE_BYTELEN (2)
#define AGW_SHAPE_PRESSURE_CENTER_HEADER_BYTESIZE \
                                (AGW_SHAPE_PRESSURE_CENTER_TYPE_BYTELEN + \
                                 AGW_SHAPE_PRESSURE_CENTER_PRESSURE_BYTELEN )

#define AGW_SHAPE_STORM_NAME_LEN_BYTELEN (1)
#define AGW_SHAPE_STORM_UNIQUEID_BYTELEN (8)
#define AGW_SHAPE_STORM_NO_OF_POSITIONS_BYTELEN (2)

#define AGW_SHAPE_STORM_POS_UNIQIEID_BYTELEN (8)
#define AGW_SHAPE_STORM_POS_SEQID_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_TYPE_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_DIRECTION_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_SPEED_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_MAX_WINDS_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_GUSTS_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_PRESSURE_BYTELEN (2)
#define AGW_SHAPE_STORM_POS_YEAR_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_MONTH_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_DAY_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_HOUR_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_MINUTES_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_STORM_TYPE_BYTELEN (1)
#define AGW_SHAPE_STORM_POS_HEADER_BYTESIZE \
                                (AGW_SHAPE_STORM_POS_UNIQIEID_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_SEQID_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_TYPE_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_DIRECTION_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_SPEED_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_MAX_WINDS_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_GUSTS_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_PRESSURE_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_YEAR_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_MONTH_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_DAY_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_HOUR_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_MINUTES_BYTELEN + \
                                 AGW_SHAPE_STORM_POS_STORM_TYPE_BYTELEN)

#define AGW_SHAPE_STORM_POS_POINT_BYTELEN (AGW_SHAPE_GEO_LAT_BYTELEN + \
                                           AGW_SHAPE_GEO_LON_BYTELEN)

#define AGW_SHAPE_STORM_POS_NO_DIRECTION (9990)
#define AGW_SHAPE_STORM_POS_DIRECTION_POW10 (1)

#define AGW_SHAPE_STORM_POS_NO_SPEED (9990)
#define AGW_SHAPE_STORM_POS_SPEED_POW10 (1)

#define AGW_SHAPE_STORM_POS_MAX_WINDS_POW10 (1)
#define AGW_SHAPE_STORM_POS_GUSTS_POW10 (1)
#define AGW_SHAPE_STORM_POS_PRESSURE_POW10 (1)

#define AGW_SHAPE_STORM_POS_BASIC_YEAR (2000)

#define AGW_SHAPE_WIND_RADII_UNIQUEID_BYTELEN (8)
#define AGW_SHAPE_WIND_RADII_SEQID_BYTELEN (2)
#define AGW_SHAPE_WIND_RADII_WIND_SPEED_BYTELEN (2)
#define AGW_SHAPE_WIND_RADII_NO_OF_POINT_BYTELEN (1)

#define AGW_SHAPE_WIND_RADII_HEADER_BYTESIZE \
                                (AGW_SHAPE_WIND_RADII_UNIQUEID_BYTELEN + \
                                 AGW_SHAPE_WIND_RADII_SEQID_BYTELEN + \
                                 AGW_SHAPE_WIND_RADII_WIND_SPEED_BYTELEN + \
                                 AGW_SHAPE_WIND_RADII_NO_OF_POINT_BYTELEN)

#define AGW_SHAPE_WIND_RADII_SPEED_POW10 (1)


// Provides max possible number of info items in single statistics
// message
#define AGW_STATISTICS_MAX_PRODUCTS (210)

/* Defines mask values for each product for filtration purposes */
#define AGW_PRODUCT_FILTER_NOWARD                (1 << 0)
#define AGW_PRODUCT_FILTER_STORM_ATTRIBUTES      (1 << 1)
#define AGW_PRODUCT_FILTER_STATISTICS            (1 << 2)
#define AGW_PRODUCT_FILTER_SURFACE_FEATURES      (1 << 3)
#define AGW_PRODUCT_FILTER_STORM_TRACK           (1 << 4)
#define AGW_PRODUCT_FILTER_WIND_MAGNITUDE        (1 << 5)
#define AGW_PRODUCT_FILTER_WIND_DIRECTION        (1 << 6)

#define AGW_PRODUCT_FILTER_ALL ( \
                    AGW_PRODUCT_FILTER_STATISTICS | \
                    AGW_PRODUCT_FILTER_NOWARD | \
                    AGW_PRODUCT_FILTER_WIND_MAGNITUDE | \
                    AGW_PRODUCT_FILTER_WIND_DIRECTION | \
                    AGW_PRODUCT_FILTER_SURFACE_FEATURES | \
                    AGW_PRODUCT_FILTER_STORM_ATTRIBUTES | \
                    AGW_PRODUCT_FILTER_STORM_TRACK )

#define AGW_PRODUCT_FILTER_DEFAULT ( \
                    AGW_PRODUCT_FILTER_STATISTICS)

  /**************/
 /** TYPEDEFS **/
/**************/

// Tiles merging iterator
typedef struct agw_dsrl_tile_desc_list_iterator_struct {
    // Destination list
    OSAL_OBJECT_HDL hListTo;
    // List of items for removing
    OSAL_OBJECT_HDL hListToRemove;
} AGW_DSRL_TILE_DESC_LIST_ITERATOR_STRUCT;

/***** AGW DSRL *****/
// Defined state of the DSRL desc object
typedef enum agw_dsrl_desc_state_enum
{
    AGW_DSRL_ENTRY_DESC_STATE_INVALID = 0,
    AGW_DSRL_ENTRY_DESC_STATE_INITIAL,
    AGW_DSRL_ENTRY_DESC_STATE_UPDATED,
    AGW_DSRL_ENTRY_DESC_STATE_STABLE,
    AGW_DSRL_ENTRY_DESC_STATE_REMOVING,
    AGW_DSRL_ENTRY_DESC_STATE_UNUSED
} AGW_DSRL_DESC_STATE_ENUM;

typedef enum agw_dsrl_desc_action_enum
{
    AGW_DSRL_ENTRY_DESC_ADD = 0,
    AGW_DSRL_ENTRY_DESC_REMOVE,
    AGW_DSRL_ENTRY_DESC_UPDATE,
    AGW_DSRL_ENTRY_DESC_NOTHING
} AGW_DSRL_DESC_ACTION_ENUM;

typedef enum agw_dsrl_entry_type_enum
{
    AGW_DSRL_ENTRY_TILE = 0,
    AGW_DSRL_ENTRY_SHAPE
} AGW_DSRL_ENTRY_TYPE_ENUM;

// DSRL Message Descriptor
typedef struct agw_dsrl_entry_desc_struct
{
    union
    {
        // General DSRL ENTRY which is used for operations
        // if pure DSRL_ENTRY_OBJECT expected instead of
        // concrete one.
        DSRL_ENTRY_OBJECT hDSRLEntry;

        // Tile DSLR_ENTRY
        AGW_TILE_OBJECT hTile;

        // Shape DSLR_ENTRY
        AGW_SHAPE_OBJECT hShape;
    } uData;

    // Desc state
    AGW_DSRL_DESC_STATE_ENUM eState;

    // Entry type
    AGW_DSRL_ENTRY_TYPE_ENUM eType;

    // List of DSRLs where the message is.
    OSAL_OBJECT_HDL hDSRLLinksList;

    // Keeps reference to the record in the messages pool
    OSAL_LINKED_LIST_ENTRY hEntry;
} AGW_DSRL_ENTRY_DESC_STRUCT;

// DSRL Target description
typedef AGW_LOCATION_DESC_STRUCT AGW_DSRL_TARGET_DESC_STRUCT;

// DSRL Description structure
typedef struct agw_dsrl_desc_struct {
    // Reference to the described DSRL
    DSRL_OBJECT hDSRL;

    // Reference to the object within the manager's DSRL list
    OSAL_LINKED_LIST_ENTRY hEntry;

    // List of targets per the DSRL
    OSAL_OBJECT_HDL hTargetList;

    // List of DSRL entries (AGW_DSRL_ENTRY_DESC_STRUCT structures)
    OSAL_OBJECT_HDL hDSRLEntryList;

} AGW_DSRL_DESC_STRUCT;

//List of targets related to the certain AGW_DSRL_ENTRY_DESC_STRUCT structure
typedef struct agw_dsrl_link_struct {
    AGW_DSRL_DESC_STRUCT *psDSRLDesc;
    OSAL_OBJECT_HDL hLinkedTargetsList;
} AGW_DSRL_LINK_STRUCT;

/***** AGW STATISTICS *****/
typedef struct agw_statistics_item_struct
{
    // Number of expected tiles per product set
    UN8 un8Expected;
    // Number of sent tiles per product set
    UN8 un8Sent;
} AGW_STATISTICS_ITEM_STRUCT;

typedef struct agw_statistics_struct
{
    // Generic header
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    // Statistics year
    UN16 un16Year;

    // Number of statistic PID's
    UN16 un16NumberPID;

    // Number of used PID's
    UN16 un16NumberOfUsedPID;

    // Whole set of statistics
    AGW_STATISTICS_ITEM_STRUCT aInfo[AGW_STATISTICS_MAX_PRODUCTS];
} AGW_STATISTICS_STRUCT;

// Statistics description
typedef struct agw_statistics_item_desc_struct {
    // Product type with statistic
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eType;

    // Tile information
    AGW_STATISTICS_ITEM_STRUCT sInfo;
} AGW_STATISTICS_ITEM_DESC_STRUCT;

// Statistics searching result
typedef struct agw_statistis_info_struct {
    // Year
    UN16 un16Year;

    // Product type with statistic
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eType;

    // Number of expected tiles per product set
    UN8 un8Expected;

    // Number of sent tiles per product set
    UN8 un8Sent;
} AGW_STATISTICS_INFO_STRUCT;

typedef struct agw_statistics_item_desc_list_entry_struct {
    // Entry timestamp
    TIME_T tTimeStamp;

    // Entry timestamp excluding the year
    TIME_T tShortTimeStamp;

    // Entry year
    UN16 un16Year;

    // Number of tiles
    UN16 un16NumberOfEntries;

    // Tiles statistics. Size of the array is defined
    // by the un16NumberOfEntries and the size of the
    // structure instance has to be calculated using this
    // approach.
    AGW_STATISTICS_ITEM_DESC_STRUCT paInfo[1];
} AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT;

// Defines the Product Statistics data control structure
typedef struct agw_statisctics_control_struct
{
    // Keeps list of statistics for the all product with in 60 unique
    // updates
    OSAL_OBJECT_HDL hStatistics;

    // Keeps recently received year
    UN16 un16Year;

} AGW_STATISTICS_CONTROL_STRUCT;

/***** AGW NOWRAD *****/
// Defines the NOWRAD data control structure
typedef struct agw_nowrad_control_struct {
    BOOLEAN bIsFirstFullSet;

    BOOLEAN bIsInProgress;

    // Keeps timestamp for currently processing set of products
    TIME_T tFullSetTimeStamp;

    // Keeps full set of statistics
    OSAL_OBJECT_HDL hFullSet;

    // Keeps timestamp for currently processing set of products
    TIME_T tProgressSetTimeStamp;

    // Keeps full set of tiles in progress
    OSAL_OBJECT_HDL hProgressSet;

    // Keeps tiles in removing set
    OSAL_OBJECT_HDL hRemovingSet;
} AGW_NOWRAD_CONTROL_STRUCT;

// Declares structure for tiles counter with some
// required timestamp
typedef struct agw_nowrad_tiles_counter_struct
{
    // Keeps required time stamp to count
    TIME_T tTimeStamp;

    // Keeps number of tiles with the time stamp specified above
    size_t tCounter;
} AGW_NOWRAD_TILE_COUNTER_STRUCT;

/***** AGW STORM ATTRIBUTES *****/
typedef struct agw_storm_attributes_data_item
{
    // WSI Generic header
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    // Time stamp
    TIME_T tTimeStamp;

    // Keeps list of storm attribute shapes
    OSAL_OBJECT_HDL hShapes;
} AGW_STORM_ATTRIBUTES_DATA_ITEM;

// Defines the Storm Attributes data control structure
typedef struct agw_storm_attributes_control_struct
{
    AGW_STORM_ATTRIBUTES_DATA_ITEM *psData;
} AGW_STORM_ATTRIBUTES_CONTROL_STRUCT;

/***** AGW WIND MAGNITUDE & DIRECTION CONTROL STRUCT *****/
typedef struct agw_winds_control_struct
{
    // Eastern tile
    AGW_DSRL_ENTRY_DESC_STRUCT *psEasternTileDesc;

    // Western tile
    AGW_DSRL_ENTRY_DESC_STRUCT *psWesternTileDesc;
} AGW_WINDS_CONTROL_STRUCT;

/***** SURFACE FEATURE FORECAST *****/
typedef struct agw_surface_feature_data_item
{
    // Looks like we need it twice
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    // Keeps all create shapes
    OSAL_OBJECT_HDL hShapes;

    // Time stamp
    TIME_T tTimeStamp;
} AGW_SURFACE_FEATURE_DATA_ITEM;

typedef struct agw_surface_feature_control_struct
{
    // Eastern part
    AGW_SURFACE_FEATURE_DATA_ITEM *psEastern;

    // Western path
    AGW_SURFACE_FEATURE_DATA_ITEM *psWestern;

} AGW_SURFACE_FEATURE_CONTROL_STRUCT;

/***** STORM TRACK FORECAST *****/
typedef struct agw_storm_track_data_item
{
    // Looks like we need it twice
    AGW_PRODUCT_HEADER_STRUCT sHeader;

    // Time stamp
    TIME_T tTimeStamp;

    // Keeps all created storm positions
    OSAL_OBJECT_HDL hStormPositionsList;

    // Keeps all known storms to structurize the shape
    // kept by hShapes
    OSAL_OBJECT_HDL hStormList;
} AGW_STORM_TRACK_DATA_ITEM;

typedef struct agw_storm_track_control_struct
{
    // Current storm track data
    AGW_STORM_TRACK_DATA_ITEM *psData;
} AGW_STORM_TRACK_CONTROL_STRUCT;

/***** AGW MANAGER *****/
typedef struct agw_parser_buffer_struct
{
    // Buffer
    char *pacData;
    // Buffer Size
    size_t tSize;
    // File ID
    FILE *pFile;
} AGW_PARSER_BUFFER_STRUCT;

/* Private object elements */
typedef struct agw_parser_object_struct
{
    // WSI Decoder instance
    WSI_DECODER_OBJECT hWsiDecoder;

    // Shared buffer
    char acBuffer[AGW_SHARED_BUFFER_LEN];

    // Buffer for some input file path
    AGW_PARSER_BUFFER_STRUCT sInputFile;

    // Buffer for some output file path
    AGW_PARSER_BUFFER_STRUCT sOutputFile;

    // Buffer for some internal needs
    AGW_PARSER_BUFFER_STRUCT sTempBuffer;

    // Temporary payload buffer
    OSAL_BUFFER_HDL hPayload;

    // ISO 3309 CRC32 Handler
    OSAL_OBJECT_HDL hCRC;

    // Path for any temp parser files
    char *pcTempPath;

    // Output image format
    IMAGE_FORMAT_ENUM eOutputImageFormat;

    // Flag which sets state for output file: to keep or not
    BOOLEAN bKeepOutputFile;

    // Temporary buffer
    OSAL_BUFFER_HDL hBuffer;
} AGW_PARSER_OBJECT_STRUCT;

// Timer control structure
typedef struct agw_timer_control_struct
{
    DATASERVICE_TIMED_EVENT_HDL hExpirationEvent;

} AGW_TIMER_CONTROL_STRUCT;

// AGW Facing object
typedef struct agw_app_object_struct
{
    // Handle SQL Connection
    SQL_INTERFACE_OBJECT hSQLConnection;

    // Path to the database file
    char *pacFullyQualifiedDatabaseFilePath;

    // Path to AGW Service home folder
    char *pacAGWServiceFilePath;

    // Keeps folder for rasters generated
    char *pacRasterFolder;

    // Keep output image file format
    IMAGE_FORMAT_ENUM eRasterFileFormat;

    // Shared buffer
    char acBuffer[AGW_SHARED_BUFFER_LEN];

    // Keeps products filter mask
    UN32 un32ProductFilter;

    // A list of TILE descriptors
    OSAL_OBJECT_HDL hDSRLEntryDescPool;

    //Block pool handle
    OSAL_OBJECT_HDL hBlockPool;

    // Declares timer control structure to support timeouts
    AGW_TIMER_CONTROL_STRUCT sTimer;
} AGW_APP_OBJECT_STRUCT;

// AGW Manager object Structure
typedef struct agw_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    // like the market filter and the data that feeds the AGW_MSG
    AGW_APP_OBJECT_STRUCT *psAppObj;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // A list of active targets (a list of DSRLs for the application)
    // Each entry contains a DSRL_OBJECT
    OSAL_OBJECT_HDL hDSRLList;

    // Data structure used for broadcast-specific parsing
    AGW_PARSER_OBJECT_STRUCT *psParserObj;

    // Product Statistics Control structure
    AGW_STATISTICS_CONTROL_STRUCT sStatisticsControlStruct;

    // NOWRAD Control structure
    AGW_NOWRAD_CONTROL_STRUCT sNowradControlStruct;

    // Storm Attributes control structure
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT sStormAttrsControlStruct;

    // Wind magnitude control structure
    AGW_WINDS_CONTROL_STRUCT sWindMagnitudeControlStruct;

    // Wind direction control structure
    AGW_WINDS_CONTROL_STRUCT sWindDirectionControlStruct;

    // Surface Feature control structure
    AGW_SURFACE_FEATURE_CONTROL_STRUCT sSurfaceFeatureControlStruct;

    // Storm track control structure
    AGW_STORM_TRACK_CONTROL_STRUCT sStormTrackControlStruct;
} AGW_MGR_OBJECT_STRUCT;

// Expiration iterator data struct
typedef struct agw_expitration_iterator_struct
{
    // Keeps number of expire entries
    UN32 un32Counter;

    // Keeps manager reference
    AGW_MGR_OBJECT_STRUCT *psObj;
} AGW_EXPIRATION_ITERATOR_STRUCT;

// Tile move and clean iterator
typedef struct agw_move_and_clean_list_iterator
{

    // Some list to operate with
    OSAL_OBJECT_HDL hList;

    // State for entry
    AGW_DSRL_DESC_STATE_ENUM eState;

} AGW_MOVE_AND_CLEAN_LIST_ITERATOR;

// Target list iterator structure
typedef struct agw_dsrl_target_to_build_dsrl_iterator_struct
{
    // List of the entries
    OSAL_OBJECT_HDL hEntries;

    // Indicates that at least one record was added
    BOOLEAN bUpdated;

    // Evaluating DSRL Entry
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntry;
} AGW_DSRL_TARGET_TO_BUILD_DSRL_ITERATOR_STRUCT;

// Array described structure
typedef struct agw_dsrl_target_array_struct
{
    // Array
    DSRL_TARGET_OBJECT *pahTargets;

    // Number of used bytes
    size_t tUsedSize;
} AGW_DSRL_TARGET_ARRAY_STRUCT;

typedef struct agw_dsrl_target_array_iterator
{
    // Array descriptor
    AGW_DSRL_TARGET_ARRAY_STRUCT * psArray;

    // DSRL event argument
    DSRL_ARG_STRUCT *psDSRLArg;
} AGW_DSRL_TARGET_ARRAY_ITERATOR;

typedef struct agw_dsrl_entry_target_links_iterator
{
    // DSRL object
    DSRL_OBJECT hDSRL;
    // Target object
    DSRL_TARGET_OBJECT hTargetObj;
} AGW_DSRL_ENTRY_TARGET_LINKS_ITERATOR;

// AGW enum type mapping structure.
typedef struct agw_enum_type_map_struct
{
    union
    {
        // Code
        AGW_RETURN_CODE_ENUM eReturnCode;
        // Product type
        AGW_PRODUCT_TYPE_ENUM eProductType;
        // Product type
        AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eInternalDataType;
        // Internal product type
        AGW_INTERNAL_PRODUCT_TYPE_ENUM eInternalProductType;
        // Front type
        AGW_FRONT_TYPE_ENUM eFrontType;
        // Internal Front type
        AGW_INTERNAL_FRONT_TYPE_ENUM eInternalFrontType;
        // Pressure type
        AGW_PRESSURE_TYPE_ENUM ePressureType;
        // Internal Pressure type
        AGW_INTERNAL_PRESSURE_TYPE_ENUM eInternalPressureType;

    } uType;

    // Name
    const char *pacText;

} AGW_ENUM_TYPE_MAP_STRUCT;

// AGW internal enum type to public enum
typedef struct agw_internal_enum_map_struct
{
    union
    {
        // Internal front type
        AGW_INTERNAL_FRONT_TYPE_ENUM eFrontType;

        // Internal pressure type
        AGW_INTERNAL_PRESSURE_TYPE_ENUM ePressureType;

        // Internal shape type
        AGW_INTERNAL_SHAPE_TYPE_ENUM eShapeType;

    } uInternal;

    union
    {
        // Front type
        AGW_FRONT_TYPE_ENUM eFrontType;

        // Pressure type
        AGW_PRESSURE_TYPE_ENUM ePressureType;

        // Shape type
        AGW_SHAPE_TYPE_ENUM eShapeType;

    } uExternal;

} AGW_INTERNAL_ENUM_MAP_STRUCT;


// AGW internal product enum type to public product enum
typedef struct agw_internal_product_type_map_struct
{
    // Internal product type
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eInternalProductType;

    // Product type
    AGW_PRODUCT_TYPE_ENUM eProductType;

    // Alternative product type
    AGW_PRODUCT_TYPE_ENUM eAlternativeProductType;
} AGW_INTERNAL_PRODUCT_TYPE_MAP_STRUCT;

// AGW Supported products set structure
typedef struct agw_support_product_set_struct
{
    // Product type
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType;

    // Corresponded product data type
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType;
} AGW_SUPPORT_PRODUCT_SET_STRUCT;


// AGW cache file header struct
typedef struct agw_cache_header_struct
{
    size_t tPayloadSize;
    UN32 un32TimeStamp;
    AGW_STATISTICS_INFO_STRUCT sStatInfo;
} AGW_CACHE_HEADER_STRUCT;

/* Service Private Event types */
typedef enum agw_mgr_event_enum
{
    AGW_MGR_EVENT_UPDATE_PRODUCT_FILTER

} AGW_MGR_EVENT_ENUM;

typedef union agw_mgr_event_union
{
    struct
    {
        AGW_PRODUCT_TYPE_ENUM eProductType;
        BOOLEAN bEnable;
    } sUpdateProductFilter;

} AGW_MGR_EVENT_UNION;

typedef struct agw_mgr_event_struct
{
    AGW_MGR_EVENT_ENUM eEvent;
    AGW_MGR_EVENT_UNION uEvent;

} AGW_MGR_EVENT_STRUCT;

  /************/
 /** MACROS **/
/************/

// Defines belonging of the passes longitude to the eastern part of US
// FIXME: Need to work out better way to do this
#define AGW_MGR_IS_EASTERN_LON(_n16Lon) \
    ((_n16Lon) > (-110 * 100)) ? TRUE : FALSE;

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static AGW_SERVICE_OBJECT hStart(
   const char *pacSRHDriverName,
   const char *pacRasterPath,
   DATASERVICE_EVENT_MASK tEventRequestMask,
   DATASERVICE_EVENT_CALLBACK vEventCallback,
   void *pvEventCallbackArg,
   DATASERVICE_OPTIONS_STRUCT const *psOptions
      );

static SMSAPI_RETURN_CODE_ENUM eAddFilter(
    AGW_SERVICE_OBJECT hAgwService,
    AGW_PRODUCT_TYPE_ENUM eProductType
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveFilter(
    AGW_SERVICE_OBJECT hAgwService,
    AGW_PRODUCT_TYPE_ENUM eProductType
        );

/* Object Private Prototypes */

static void vEventHandler(
   DATASERVICE_MGR_OBJECT hDataService,
   DATASERVICE_EVENT_MASK tCurrentEvent,
   void *pvEventArg,
   void *pvEventCallbackArg
      );

static BOOLEAN bDSRLCreateList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLModifyList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLDestroyList (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRefreshList(
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bReplaceTargets (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRemoveTargets (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
        );

static BOOLEAN bRemoveAllTargets (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static N16 n16FindLinkedDSRL (
    AGW_DSRL_LINK_STRUCT *psTarget,
    DSRL_OBJECT hDSRL
        );

static AGW_DSRL_DESC_STRUCT *psCreateDSRLDesc (
    AGW_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bBuildDSRL (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bBuildAllTargetDescs(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
            );

static AGW_DSRL_TARGET_DESC_STRUCT *psBuildTargetDesc (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    LOCATION_OBJECT *phLocation
        );

static BOOLEAN bDSRLUpdateState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static AGW_RETURN_CODE_ENUM eUpdateDSRLListState(
    OSAL_OBJECT_HDL hDSRLList,
    DSRL_STATE_ENUM eState
        );

static void vReleaseDSRLDescFull (
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bRemoveEntries
        );

static void vReleaseDSRLDescWithoutList (
    DSRL_OBJECT hDSRL
        );

static void vAgwDSRLEntryFinalizeFunction (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    void *pvFinalizeArg
        );

static void vAgwDSRLEntryDescReleaseFull(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc
        );

static void vAgwLLDSRLDescRemoveEntryFromList(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static AGW_RETURN_CODE_ENUM eAgwDSRLUnlinkEntry(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        );

static void vReleaseDSRLTargetDesc (
    AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        );

/* -- TIMER -- */
static AGW_RETURN_CODE_ENUM eExpirationTimerInit(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static void vExpirationTimerUnInit(
    AGW_APP_OBJECT_STRUCT *psAppObj
        );

static void vExpirationTimerStart (
    AGW_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        );

static void vExpirationTimerStop (
    AGW_APP_OBJECT_STRUCT *psAppObj
        );

static void vExpirationTimerUpdate (
    AGW_MGR_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        );

/** APP FACING **/
static BOOLEAN bInitAppFacingObject(
    AGW_MGR_OBJECT_STRUCT *psObj,
    const char *pacRasterPath,
    IMAGE_FORMAT_ENUM eImageFormat
        );

static void vUninitAppFacingObject(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static AGW_APP_OBJECT_STRUCT *psGetAppFacingObject(
    AGW_SERVICE_OBJECT hAgwService
        );

static BOOLEAN bProcessExpirationTimeoutIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc,
    AGW_EXPIRATION_ITERATOR_STRUCT *psArg
        );

static AGW_RETURN_CODE_ENUM eProcessExpirationTimeout(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRemoveDSRLEntryTargetLinksIterator(
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLEntryDesc,
    AGW_DSRL_ENTRY_TARGET_LINKS_ITERATOR *psIterator
        );

static BOOLEAN bProductHandlersInit(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProductHandlersUninit(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProcessPayload(
   AGW_MGR_OBJECT_STRUCT *psObj,
   OSAL_BUFFER_HDL hPayload
      );

static BOOLEAN bHandleServiceReady(
   AGW_MGR_OBJECT_STRUCT *psObj
      );

static void vUninitObject(
    AGW_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vDestroyObject (
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static void vSetError(
   AGW_MGR_OBJECT_STRUCT *psObj,
   DATASERVICE_ERROR_CODE_ENUM eErrorCode
      );

static UN32 un32GetMaskForProduct(
    AGW_PRODUCT_TYPE_ENUM eProductType
        );

static UN32 un32GetMaskForInternalProduct(
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType
        );

static N16 n16LinkSearchDSRL(
    AGW_DSRL_LINK_STRUCT *psObj1,
    AGW_DSRL_LINK_STRUCT *psObj2
      );

static N16 n16LinkSearchTargetByLocation(
        AGW_DSRL_TARGET_DESC_STRUCT *psTargetDesc,
        DSRL_TARGET_OBJECT *phTargetEntry
            );

static AGW_RETURN_CODE_ENUM eProcessProductFilterUpdate(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_APP_OBJECT_STRUCT *psAppObj,
    AGW_PRODUCT_TYPE_ENUM eProductType,
    BOOLEAN bEnable
        );

static void vReleaseTargetLinks(
    OSAL_OBJECT_HDL hList
        );

static BOOLEAN bReadCachedData(
    AGW_MGR_OBJECT_STRUCT *psObj,
    UN32 un32ProductsToRead
        );

static AGW_RETURN_CODE_ENUM eGenerateCacheFilePath(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    const char *pacFileName,
    char **ppacFilePath
        );

/** PRODUCT STATISTICS */

static BOOLEAN bProductStatisticsInit(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        );

static BOOLEAN bProductStatisticsGetForProduct (
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psStatEntry,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    AGW_STATISTICS_INFO_STRUCT *psInfo
        );

static BOOLEAN bProductStatisticsGet(
    AGW_STATISTICS_CONTROL_STRUCT *psObj,
    TIME_T tTimeStamp,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eProductType,
    BOOLEAN bUseShortTimeStamp,
    AGW_STATISTICS_INFO_STRUCT *psInfo
        );

static AGW_RETURN_CODE_ENUM eProductStatisticsProcess(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static AGW_RETURN_CODE_ENUM eProductStatisticsUpdate(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_STRUCT *psStatisticsProduct
        );

static AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psProductStatisticsEntryCreate(
    AGW_APP_OBJECT_STRUCT *psAppObj,
    UN16 un16NoOfItems
        );

static void vProductStatisticsEntryRelease(
    AGW_STATISTICS_ITEM_DESC_LIST_ENTRY_STRUCT *psObj
        );

static BOOLEAN bProductStatisticsUninit(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        );

/** NOWRAD */
static BOOLEAN bNOWRADControlInit(
   AGW_NOWRAD_CONTROL_STRUCT *psObj
      );

static AGW_RETURN_CODE_ENUM eNOWRADProcess(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        );

static BOOLEAN bNOWRADControlUninit(
   AGW_NOWRAD_CONTROL_STRUCT *psObj
      );

static AGW_RETURN_CODE_ENUM eNOWRADUpdateStatistics (
    AGW_MGR_OBJECT_STRUCT *psObj,
    const AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static BOOLEAN bNOWRADMergeSetIterator(
    void *pvData,
    void *pvArg
        );

static void vNOWRADProcessExpiredEntry(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psEntryDesc
        );

static BOOLEAN bNOWRADHasExpirableEntries(
    AGW_NOWRAD_CONTROL_STRUCT *psNRCtrl
        );

static BOOLEAN bAgwDSRLAddEntry(
    AGW_DSRL_ENTRY_DESC_STRUCT *psTileDesc,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bAgwDSRLAddEntries(
    OSAL_OBJECT_HDL hList,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bAgwDSRLUnlinkAllEntries(
    OSAL_OBJECT_HDL hList
        );

static BOOLEAN bNOWRADBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static AGW_RETURN_CODE_ENUM eNOWRADPutTileDescToSet(
    OSAL_OBJECT_HDL hList,
    AGW_DSRL_ENTRY_DESC_STRUCT **ppsTileDesc,
    BOOLEAN bReplace,
    OSAL_OBJECT_HDL hRemovingEntiresList
        );

/** STORM ATTRIBUTES */

static BOOLEAN bStormAttributesInit (
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psObj
        );

static BOOLEAN bStormAttributesBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static AGW_RETURN_CODE_ENUM eStormAttributesProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        );

static BOOLEAN bStormAttributesUninit (
    AGW_STORM_ATTRIBUTES_CONTROL_STRUCT *psObj
        );

/** WINDS */

static BOOLEAN bWindsInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

static BOOLEAN bWindDirectionInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

static BOOLEAN bWindMagnitudeInit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

static AGW_RETURN_CODE_ENUM eWindsProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        );

static BOOLEAN bWindsBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc,
    AGW_WINDS_CONTROL_STRUCT *psWindCtrl
        );

static BOOLEAN bWindDirectionBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bWindMagnitudeBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bWindsUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

static BOOLEAN bWindDirectionUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

static BOOLEAN bWindMagnitudeUninit(
    AGW_WINDS_CONTROL_STRUCT *psObj
        );

/** SURFACE FEATURE */
static BOOLEAN bSurfaceFeatureInit (
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psObj
        );

static AGW_RETURN_CODE_ENUM eSurfaceFeatureProcess (
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_INFO_STRUCT *psStatInfo
        );

static AGW_RETURN_CODE_ENUM eProcessFrontShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        );

static AGW_RETURN_CODE_ENUM eProcessIsobarShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        );

static AGW_RETURN_CODE_ENUM eProcessPressCntrShape (
    AGW_PARSER_OBJECT_STRUCT *psParserObj,
    AGW_DSRL_ENTRY_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bSurfaceFeatureBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bSurfaceFeatureUninit (
    AGW_SURFACE_FEATURE_CONTROL_STRUCT *psObj
        );

/** STORM TRACK **/

static BOOLEAN bStormTrackInit (
    AGW_STORM_TRACK_CONTROL_STRUCT *psObj
        );

static BOOLEAN bStormTrackBuildDSRL(
    AGW_MGR_OBJECT_STRUCT *psObj,
    AGW_DSRL_DESC_STRUCT *psDSRLDesc
        );

static BOOLEAN bStormTrackUninit (
    AGW_STORM_TRACK_CONTROL_STRUCT *psObj
        );

/** UTILITIE FUNCTION **/
static BOOLEAN bAgwLLHasDSRLs(
    AGW_MGR_OBJECT_STRUCT *psObj
        );

static const char *pacAgwLLGetFileExtension (
    AGW_FILE_TYPE_ENUM eFileType,
    IMAGE_FORMAT_ENUM eImageFormat
        );

static BOOLEAN bAgwLLCreateProductFilePath (
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    const char *pcTempPath,
    char *pacBuffer,
    size_t tBufferSize
        );

static size_t tAgwLLGetProductFilePathLen (
    AGW_FILE_TYPE_ENUM eFileType,
    IMAGE_FORMAT_ENUM eImageFormat,
    const char *pacFilePath
        );

#if ((DEBUG_OBJECT == 1) || (SMS_DEBUG == 1))
static const char *pacAgwLLGetProductDataTypeName(
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType
        );
#endif

/** LOW LEVEL PARSING CODE */
static AGW_PARSER_OBJECT_STRUCT *hAgwLLParserInit (
    SMS_OBJECT hOwner,
    const char *pcTempPath,
    IMAGE_FORMAT_ENUM eImageFormat
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserParseHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    size_t tOffsetInBits,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static void vAgwLLParserBufferDestroy(
    AGW_PARSER_BUFFER_STRUCT *psBuffer
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserEnsureBufferSize(
    size_t tRequestedSize,
    AGW_PARSER_BUFFER_STRUCT *psBuffer
        );


static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStatisticsProduct(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_STATISTICS_STRUCT *psStatisticsData
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStormAttribute(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_STORM_ATTRIBUTES_OBJECT hStormAttr,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStormAttributes(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    UN16 *pun16AttrCount
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeDataBasedProduct(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_SHAPE_HEADER_STRUCT *psShapeHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeGeoPoint(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    N16 *pn16Lat,
    N16 *pn16Lon
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeFront(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_FRONT_OBJECT hFront
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessShapeIsobar(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_ISOBAR_OBJECT hIsobar
        );

static void vAgwLLParserUninitParser (
    AGW_PARSER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bAgwLLParserLoadHeaderToBuffer(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    const char *aFilePath,
    size_t tReadSize
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessPayload(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessProductHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessRasterProductHeader(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserGenerateTile(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_RASTER_PLANE_ENUM ePlane,
    AGW_TILE_OBJECT hAgwTile
        );

static BOOLEAN bAgwLLParserIsProductSupported(
    AGW_INTERNAL_PRODUCT_DATA_TYPE_ENUM eDataType,
    AGW_INTERNAL_PRODUCT_TYPE_ENUM eType
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserAssignPayload(
    AGW_PARSER_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserProcessStatisticsFillUp (
    const AGW_PRODUCT_HEADER_STRUCT *psHeader,
    AGW_STATISTICS_STRUCT *psStatisticsData,
    UN16 un16Year,
    int iNumberPID,
    ...
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserCleanupBuffer(
    AGW_PARSER_OBJECT_STRUCT *psObj
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserRemoveInputFile(
    AGW_PARSER_OBJECT_STRUCT *psObj
        );

static AGW_RETURN_CODE_ENUM eAgwLLParserRemoveOutputFile(
    AGW_PARSER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCheckRasterPath(
    AGW_APP_OBJECT_STRUCT *psAppObj
        );


#if (SMS_DEBUG == 1) && (DEBUG_OBJECT == 1)
static void vPrintCacheHeader(
    AGW_CACHE_HEADER_STRUCT *psCacheHeader
        );

static void vPrintHeader(
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        );

static void vPrintStatisticsProduct(
    AGW_STATISTICS_STRUCT *psStatisticProduct
        );

static void vPrintStatisticsCollection(
    AGW_STATISTICS_CONTROL_STRUCT *psObj
        );

static void vPrintShapeHeader(
    AGW_SHAPE_HEADER_STRUCT *psObj
        );
#endif

  /***************/
 /** VARIABLES **/
/***************/

// AGW Return codes map
static const AGW_ENUM_TYPE_MAP_STRUCT gsAGWReturnCodeNameMap[] = {
    { {AGW_RETURN_CODE_SUCCESS},
            MACRO_TO_STRING(AGW_RETURN_CODE_SUCCESS)},
    { {AGW_RETURN_CODE_INVALID_STATE},
            MACRO_TO_STRING(AGW_RETURN_CODE_INVALID_STATE)},
    { {AGW_RETURN_CODE_ERROR},
            MACRO_TO_STRING(AGW_RETURN_CODE_ERROR)},
    { {AGW_RETURN_CODE_FAILED_TO_REPLACE},
            MACRO_TO_STRING(AGW_RETURN_CODE_FAILED_TO_REPLACE)},
    { {AGW_RETURN_CODE_NOT_OWNER},
            MACRO_TO_STRING(AGW_RETURN_CODE_NOT_OWNER)},
    { {AGW_RETURN_CODE_CRC_INVALID},
            MACRO_TO_STRING(AGW_RETURN_CODE_CRC_INVALID)},
    { {AGW_RETURN_CODE_PVN_UNSUPPORTED},
            MACRO_TO_STRING(AGW_RETURN_CODE_PVN_UNSUPPORTED)},
    { {AGW_RETURN_CODE_BAD_ARGUMENT},
            MACRO_TO_STRING(AGW_RETURN_CODE_BAD_ARGUMENT)},
    { {AGW_RETURN_CODE_NOT_ENOUGH_DATA},
            MACRO_TO_STRING(AGW_RETURN_CODE_NOT_ENOUGH_DATA)},
    { {AGW_RETURN_CODE_NO_INTERSECTION},
            MACRO_TO_STRING(AGW_RETURN_CODE_NO_INTERSECTION)},
    { {AGW_RETURN_CODE_IO_ERROR},
            MACRO_TO_STRING(AGW_RETURN_CODE_IO_ERROR)},
    { {AGW_RETURN_CODE_NO_MEMORY},
            MACRO_TO_STRING(AGW_RETURN_CODE_NO_MEMORY)},
    { {AGW_RETURN_CODE_SHAPE_ALREADY_ATTACHED},
            MACRO_TO_STRING(AGW_RETURN_CODE_SHAPE_ALREADY_ATTACHED)},
    { {AGW_RETURN_CODE_UNEXPECTED_DATA_TYPE},
            MACRO_TO_STRING(AGW_RETURN_CODE_UNEXPECTED_DATA_TYPE)},
    { {AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE},
            MACRO_TO_STRING(AGW_RETURN_CODE_UNEXPECTED_SHAPE_TYPE) },
    { {AGW_RETURN_CODE_UNSUPPORTED_PRODUCT},
            MACRO_TO_STRING(AGW_RETURN_CODE_UNSUPPORTED_PRODUCT)},
    { {AGW_RETURN_CODE_WSI_DECODER_FAILED},
            MACRO_TO_STRING(AGW_RETURN_CODE_WSI_DECODER_FAILED)},
    { {AGW_RETURN_CODE_DUPLICATED_TILE},
            MACRO_TO_STRING(AGW_RETURN_CODE_DUPLICATED_TILE)},
    { {AGW_RETURN_CODE_REDUNDANT_TILE},
            MACRO_TO_STRING(AGW_RETURN_CODE_REDUNDANT_TILE)},
    { {AGW_RETURN_CODE_NO_STATISTIC_DATA},
            MACRO_TO_STRING(AGW_RETURN_CODE_NO_STATISTIC_DATA)},
    { {AGW_RETURN_CODE_FILTERED_OUT},
            MACRO_TO_STRING(AGW_RETURN_CODE_FILTERED_OUT)},
    { {AGW_RETURN_CODE_DATA_TIMEOUT},
            MACRO_TO_STRING(AGW_RETURN_CODE_DATA_TIMEOUT)},
    { {AGW_RETURN_CODE_NOT_EXIST},
            MACRO_TO_STRING(AGW_RETURN_CODE_NOT_EXIST)}
};

// AGW Internal Product types map to public (API)
static const AGW_ENUM_TYPE_MAP_STRUCT gsAGWInternalProductTypeNameMap[] = {
    { {AGW_INTERNAL_PRODUCT_TYPE_NOWRAD},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_NOWRAD) },
    { {AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES) },
    { {AGW_INTERNAL_PRODUCT_TYPE_STATISTICS},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_STATISTICS) },
    { {AGW_INTERNAL_PRODUCT_TYPE_CANADIAN_RADAR},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_CANADIAN_RADAR) },
    { {AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES) },
    { {AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK) },
    { {AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS) },
    { {AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_UNKNOWN) }
};

// AGW Product types map
static const AGW_ENUM_TYPE_MAP_STRUCT gsAGWProductTypeNameMap[] = {
    { {AGW_PRODUCT_TYPE_NOWRAD},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_NOWRAD) },
    { {AGW_PRODUCT_TYPE_STORM_ATTRIBUTES},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_STORM_ATTRIBUTES) },
    { {AGW_PRODUCT_TYPE_SURFACE_FEATURES},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_SURFACE_FEATURES) },
    { {AGW_PRODUCT_TYPE_STORM_TRACK},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_STORM_TRACK) },
    { {AGW_PRODUCT_TYPE_WIND_MAGNITUDE},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_WIND_MAGNITUDE) },
    { {AGW_PRODUCT_TYPE_WIND_DIRECTION},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_WIND_DIRECTION) },
    { {AGW_PRODUCT_TYPE_UNKNOWN},
            MACRO_TO_STRING(AGW_PRODUCT_TYPE_UNKNOWN) }
};

// AGW Front types map
static const AGW_ENUM_TYPE_MAP_STRUCT gsAGWFrontTypeNameMap[] = {
    { {AGW_FRONT_TYPE_COLD},
            MACRO_TO_STRING(AGW_FRONT_TYPE_COLD) },
    { {AGW_FRONT_TYPE_WARM},
            MACRO_TO_STRING(AGW_FRONT_TYPE_WARM) },
    { {AGW_FRONT_TYPE_STATIONARY},
            MACRO_TO_STRING(AGW_FRONT_TYPE_STATIONARY) },
    { {AGW_FRONT_TYPE_OCCLUDED},
            MACRO_TO_STRING(AGW_FRONT_TYPE_OCCLUDED) },
    { {AGW_FRONT_TYPE_TROUGH},
            MACRO_TO_STRING(AGW_FRONT_TYPE_TROUGH) },
    { {AGW_FRONT_TYPE_SQUALL_LINE},
            MACRO_TO_STRING(AGW_FRONT_TYPE_SQUALL_LINE) },
    { {AGW_FRONT_TYPE_DRY_LINE},
            MACRO_TO_STRING(AGW_FRONT_TYPE_DRY_LINE) },
    { {AGW_FRONT_TYPE_DISSIPATING_COLD},
            MACRO_TO_STRING(AGW_FRONT_TYPE_DISSIPATING_COLD) },
    { {AGW_FRONT_TYPE_DISSIPATING_WARM},
            MACRO_TO_STRING(AGW_FRONT_TYPE_DISSIPATING_WARM) },
    { {AGW_FRONT_TYPE_DISSIPATING_STATIONARY},
            MACRO_TO_STRING(AGW_FRONT_TYPE_DISSIPATING_STATIONARY) },
    { {AGW_FRONT_TYPE_DISSIPATING_OCCLUDED},
            MACRO_TO_STRING(AGW_FRONT_TYPE_DISSIPATING_OCCLUDED) },
    { {AGW_FRONT_TYPE_FORMING_COLD},
            MACRO_TO_STRING(AGW_FRONT_TYPE_FORMING_COLD) },
    { {AGW_FRONT_TYPE_FORMING_WARM},
            MACRO_TO_STRING(AGW_FRONT_TYPE_FORMING_WARM) },
    { {AGW_FRONT_TYPE_FORMING_STATIONARY},
            MACRO_TO_STRING(AGW_FRONT_TYPE_FORMING_STATIONARY) },
    { {AGW_FRONT_TYPE_FORMING_OCCLUDED},
            MACRO_TO_STRING(AGW_FRONT_TYPE_FORMING_OCCLUDED) },
    { {AGW_FRONT_TYPE_UNKNOWN},
            MACRO_TO_STRING(AGW_FRONT_TYPE_UNKNOWN) }
};

// AGW Product Data types map
static const AGW_ENUM_TYPE_MAP_STRUCT gsAGWProductDataTypeNameMap[] = {
    { {AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC) },
    { {AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER) },
    { {AGW_INTERNAL_PRODUCT_DATA_TYPE_UNKNOWN},
            MACRO_TO_STRING(AGW_INTERNAL_PRODUCT_DATA_TYPE_UNKNOWN) }
};

// AGW Internal Product Type map to public (API)
static const AGW_INTERNAL_PRODUCT_TYPE_MAP_STRUCT gsAGWInternalProductTypeMap[] =
{
    {
        AGW_INTERNAL_PRODUCT_TYPE_NOWRAD,
        AGW_PRODUCT_TYPE_NOWRAD,
        AGW_PRODUCT_TYPE_UNKNOWN
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES,
        AGW_PRODUCT_TYPE_STORM_ATTRIBUTES,
        AGW_PRODUCT_TYPE_UNKNOWN
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_STATISTICS,
        AGW_PRODUCT_TYPE_UNKNOWN,
        AGW_PRODUCT_TYPE_UNKNOWN
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES,
        AGW_PRODUCT_TYPE_SURFACE_FEATURES,
        AGW_PRODUCT_TYPE_UNKNOWN
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK,
        AGW_PRODUCT_TYPE_STORM_TRACK,
        AGW_PRODUCT_TYPE_UNKNOWN
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS,
        AGW_PRODUCT_TYPE_WIND_MAGNITUDE,
        AGW_PRODUCT_TYPE_WIND_DIRECTION,
    },
    {
        AGW_INTERNAL_PRODUCT_TYPE_UNKNOWN,
        AGW_PRODUCT_TYPE_UNKNOWN,
        AGW_PRODUCT_TYPE_UNKNOWN
    }
};

// AGW Internal Shape Type map to public (API)
static const AGW_INTERNAL_ENUM_MAP_STRUCT gsAGWInternalShapeTypeMap[] =
{
    {{AGW_INTERNAL_SHAPE_TYPE_FRONT}, {AGW_SHAPE_TYPE_FRONT}},
    {{AGW_INTERNAL_SHAPE_TYPE_ISOBAR}, {AGW_SHAPE_TYPE_ISOBAR}},
    {{AGW_INTERNAL_SHAPE_TYPE_PRESSURE_CENTER}, {AGW_SHAPE_TYPE_PRESSURE_CENTER}},
    {{AGW_INTERNAL_SHAPE_TYPE_STORM_POSITION}, {AGW_SHAPE_TYPE_STORM_POSITION}},
    {{AGW_INTERNAL_SHAPE_TYPE_WIND_RADII_FIELD}, {AGW_SHAPE_TYPE_WIND_RADII_FIELD}},
    {{AGW_INTERNAL_SHAPE_TYPE_STORM_DESCRIPTION}, {AGW_SHAPE_TYPE_UNKNOWN}},
};

// AGW Internal Front Type map to public (API)
static const AGW_INTERNAL_ENUM_MAP_STRUCT gsAGWInternalFrontTypeMap[] =
{
    { {AGW_INTERNAL_FRONT_TYPE_COLD}, {AGW_FRONT_TYPE_COLD} },
    { {AGW_INTERNAL_FRONT_TYPE_WARM}, {AGW_FRONT_TYPE_WARM} },
    { {AGW_INTERNAL_FRONT_TYPE_STATIONARY}, {AGW_FRONT_TYPE_STATIONARY} },
    { {AGW_INTERNAL_FRONT_TYPE_OCCLUDED}, {AGW_FRONT_TYPE_OCCLUDED} },
    { {AGW_INTERNAL_FRONT_TYPE_TROUGH}, {AGW_FRONT_TYPE_TROUGH} },
    { {AGW_INTERNAL_FRONT_TYPE_SQUALL_LINE}, {AGW_FRONT_TYPE_SQUALL_LINE} },
    { {AGW_INTERNAL_FRONT_TYPE_DRY_LINE}, {AGW_FRONT_TYPE_DRY_LINE} },
    { {AGW_INTERNAL_FRONT_TYPE_DISSIPATING_COLD}, {AGW_FRONT_TYPE_DISSIPATING_COLD} },
    { {AGW_INTERNAL_FRONT_TYPE_DISSIPATING_WARM}, {AGW_FRONT_TYPE_DISSIPATING_WARM} },
    { {AGW_INTERNAL_FRONT_TYPE_DISSIPATING_STATIONARY}, {AGW_FRONT_TYPE_DISSIPATING_STATIONARY} },
    { {AGW_INTERNAL_FRONT_TYPE_DISSIPATING_OCCLUDED}, {AGW_FRONT_TYPE_DISSIPATING_OCCLUDED} },
    { {AGW_INTERNAL_FRONT_TYPE_FORMING_COLD}, {AGW_FRONT_TYPE_FORMING_COLD} },
    { {AGW_INTERNAL_FRONT_TYPE_FORMING_WARM}, {AGW_FRONT_TYPE_FORMING_WARM} },
    { {AGW_INTERNAL_FRONT_TYPE_FORMING_STATIONARY}, {AGW_FRONT_TYPE_FORMING_STATIONARY} },
    { {AGW_INTERNAL_FRONT_TYPE_FORMING_OCCLUDED}, {AGW_FRONT_TYPE_FORMING_OCCLUDED} }
};

// AGW Internal Pressure Type map to public (API)
static const AGW_INTERNAL_ENUM_MAP_STRUCT gsAGWInternalPressureTypeMap[] =
{
    { {AGW_INTERNAL_PRESSURE_TYPE_UNKNOWN}, {AGW_PRESSURE_TYPE_UNKNOWN} },
    { {AGW_INTERNAL_PRESSURE_TYPE_HIGH}, {AGW_PRESSURE_TYPE_HIGH} },
    { {AGW_INTERNAL_PRESSURE_TYPE_LOW}, {AGW_PRESSURE_TYPE_LOW} }
};

// AGW Supported products set
static const AGW_SUPPORT_PRODUCT_SET_STRUCT gsAGWSupportProductSet[] = {
    { AGW_INTERNAL_PRODUCT_TYPE_NOWRAD,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER },
    { AGW_INTERNAL_PRODUCT_TYPE_STATISTICS,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC },
    { AGW_INTERNAL_PRODUCT_TYPE_STORM_ATTRIBUTES,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC },
    { AGW_INTERNAL_PRODUCT_TYPE_STORM_TRACK,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC },
    { AGW_INTERNAL_PRODUCT_TYPE_FORECAST_WINDS,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_RASTER },
    { AGW_INTERNAL_PRODUCT_TYPE_SURFACE_FEATURES,
            AGW_INTERNAL_PRODUCT_DATA_TYPE_GENERIC }
};

// Global (re-usable) instance of an interface for this object
const AGW_INTERFACE_STRUCT AGW = {
   /*.hStart = */        hStart,
   /*.vStop = */         (void (*) (AGW_SERVICE_OBJECT hAgwService))DATASERVICE_IMPL_vStop,
   /*.eState = */        (DATASERVICE_STATE_ENUM (*) (AGW_SERVICE_OBJECT hAgwService))DATASERVICE_IMPL_eState,
   /*.eErrorCode = */    (DATASERVICE_ERROR_CODE_ENUM (*) (AGW_SERVICE_OBJECT hAgwService))DATASERVICE_IMPL_eErrorCode,
   /*.eAddFilter = */    eAddFilter,
   /*.eRemoveFilter = */ eRemoveFilter
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsAGWStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */DATASERVICE_IMPL_bGenericStateHandler
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _AGW_MGR_OBJ_H_
